describe('DeducedMonoFuture Factory', function() {
    // --- SETUP ---
    function $stateParamsMock() { return {}; }

    beforeEach(module('dataiku.logger'));
    beforeEach(module('dataiku.constants'));
    beforeEach(module('dataiku.services', function($provide) {
        $provide.value('$stateParams', $stateParamsMock);
    }));
    beforeEach(module('dataiku.recipes'));
    beforeEach(module('dataiku.shaker'));
    beforeEach(module('dataiku.mock'));

    let $q, DeducedMonoFuture, $rootScope, DataikuAPI;

    beforeEach(inject(function(_$q_, _DeducedMonoFuture_, _$rootScope_, _DataikuAPI_) {
        $q = _$q_;
        DeducedMonoFuture = _DeducedMonoFuture_;
        $rootScope = _$rootScope_;
        DataikuAPI = _DataikuAPI_;
    }));

    // Helper function to create a mock promise with .success() and .error() methods
    function createMockHttpPromise(qPromise) {
        // Start with the real promise to have a working .then() and .finally()
        const mockPromise = qPromise;

        mockPromise.success = function(callback) {
            mockPromise.then(function(response) {
                callback(response);
            });
            return mockPromise; // Return for chaining
        };

        mockPromise.error = function(callback) {
            mockPromise.then(null, function(error) {
                callback(error);
            });
            return mockPromise; // Return for chaining
        };

        return mockPromise;
    }

    // --- TESTS ---

    it('should handle a single immediate success', () => {
        // Arrange
        // FIX: Wrap the standard promise with our helper
        const realPromise = $q.when({ hasResult: true, data: 'SUCCESS' });
        const mockPromise = createMockHttpPromise(realPromise);
        const apiFunc = jasmine.createSpy('apiFunc').and.returnValue(mockPromise);

        const future = DeducedMonoFuture().wrap(apiFunc);
        const successCallback = jasmine.createSpy('successCallback');

        // Act
        future('arg1').success(successCallback);
        $rootScope.$apply();

        // Assert
        expect(apiFunc).toHaveBeenCalledWith('arg1');
        expect(successCallback).toHaveBeenCalledWith({ hasResult: true, data: 'SUCCESS' });
    });

    it('should queue a second request and execute it after the first completes', () => {
        // Arrange
        const apiFunc = jasmine.createSpy('apiFunc');

        // FIX: Wrap the deferred promises with our helper
        const firstCallDeferred = $q.defer();
        const firstMockPromise = createMockHttpPromise(firstCallDeferred.promise);
        apiFunc.withArgs('first').and.returnValue(firstMockPromise);

        const secondCallDeferred = $q.defer();
        const secondMockPromise = createMockHttpPromise(secondCallDeferred.promise);
        apiFunc.withArgs('second').and.returnValue(secondMockPromise);

        const future = DeducedMonoFuture().wrap(apiFunc);
        const successCallback1 = jasmine.createSpy('successCallback1');
        const successCallback2 = jasmine.createSpy('successCallback2');

        // Act
        future('first').success(successCallback1);
        future('second').success(successCallback2);
        $rootScope.$apply();

        // Assert (before completion)
        expect(apiFunc).toHaveBeenCalledTimes(1);
        expect(apiFunc).toHaveBeenCalledWith('first');

        // Act (complete first call)
        firstCallDeferred.resolve({ hasResult: true, data: 'FIRST' });
        $rootScope.$apply();

        // Assert (after first completion)
        expect(successCallback1).toHaveBeenCalledWith({ hasResult: true, data: 'FIRST' });
        expect(apiFunc).toHaveBeenCalledTimes(2);
        expect(apiFunc).toHaveBeenCalledWith('second');

        // Act (complete second call)
        secondCallDeferred.resolve({ hasResult: true, data: 'SECOND' });
        $rootScope.$apply();

        // Assert (final)
        expect(successCallback2).toHaveBeenCalledWith({ hasResult: true, data: 'SECOND' });
    });

    it('should cancel a pending request if a new one is made', () => {
        // Arrange
        const apiFunc = jasmine.createSpy('apiFunc');

        // Deferred promises for the calls that will actually execute
        const firstCallDeferred = $q.defer();
        apiFunc.withArgs('first').and.returnValue(createMockHttpPromise(firstCallDeferred.promise));

        const thirdCallDeferred = $q.defer();
        apiFunc.withArgs('third').and.returnValue(createMockHttpPromise(thirdCallDeferred.promise));

        const future = DeducedMonoFuture().wrap(apiFunc);
        const successCallback1 = jasmine.createSpy('successCallback1');
        const errorCallback2 = jasmine.createSpy('errorCallback2');
        const successCallback3 = jasmine.createSpy('successCallback3');

        // Act: Make three calls in succession while the first is busy
        future('first').success(successCallback1);
        future('second').error(errorCallback2); // We expect this call to be canceled and error out
        future('third').success(successCallback3);
        $rootScope.$apply();

        // Assert: Check initial state
        // Only the first API call should be made
        expect(apiFunc).toHaveBeenCalledTimes(1);
        expect(apiFunc).toHaveBeenCalledWith('first');
        // The second call should have been immediately rejected because the third superseded it
        expect(errorCallback2).toHaveBeenCalledTimes(1);
        expect(errorCallback2.calls.mostRecent().args[0].data.reason).toBe('Superseded by new call');
        expect(successCallback1).not.toHaveBeenCalled();
        expect(successCallback3).not.toHaveBeenCalled();

        // Act: Complete the first call
        firstCallDeferred.resolve({ hasResult: true, data: 'FIRST' });
        $rootScope.$apply();

        // Assert: Check state after first call completes
        // The first call's success callback should be triggered
        expect(successCallback1).toHaveBeenCalledWith({ hasResult: true, data: 'FIRST' });
        // The third (and now only pending) call should be executed
        expect(apiFunc).toHaveBeenCalledTimes(2);
        expect(apiFunc).toHaveBeenCalledWith('third');
        // The API call for 'second' should never have been made
        expect(apiFunc).not.toHaveBeenCalledWith('second');

        // Act: Complete the third call
        thirdCallDeferred.resolve({ hasResult: true, data: 'THIRD' });
        $rootScope.$apply();

        // Assert: Final state
        // The third call's success callback should be triggered
        expect(successCallback3).toHaveBeenCalledWith({ hasResult: true, data: 'THIRD' });
    });
});