/* global describe, beforeEach, module, inject, it, expect, jasmine */

describe('MonoFuturePool', function() {

    // --- SETUP ---
    function $stateParamsMock() { return {}; }

    beforeEach(module('dataiku.logger'));
    beforeEach(module('dataiku.constants'));
    beforeEach(module('dataiku.services', function($provide) {
        $provide.value('$stateParams', $stateParamsMock);
    }));
    beforeEach(module('dataiku.recipes'));
    beforeEach(module('dataiku.shaker'));
    beforeEach(module('dataiku.mock'));


    let $q, MonoFuturePool, $rootScope, clock, apiCallsLog, $timeout, DataikuAPI, $flushPendingTasks;

    // Small fake async tool to help mock a timed series of async stuff
    function Clock() {
        this.currentTime = 0;
        this.actions = []; // {when, what}[]
    }

    Clock.prototype.schedule = function(when, what) {
        expect(when).toBeGreaterThan(this.currentTime);
        this.actions.push({when, what});
    };

    Clock.prototype.goto = function(to, flushTimeout) {
        expect(to).toBeGreaterThan(this.currentTime);
        for( ; this.currentTime <= to && this.actions.length > 0; this.currentTime++ ) {
            if(flushTimeout) $flushPendingTasks(1000);
            this.todo = this.actions.filter(action => action.when === this.currentTime);
            this.actions = this.actions.filter(action => action.when > this.currentTime);
            this.todo.forEach(action => action.what());
            $rootScope.$apply();
        }
        this.currentTime = to;
    };

    Clock.prototype.flush = function(flushTimeout) {
        while(this.actions.length)
            this.goto(this.actions[0].when, flushTimeout);
    };



    beforeEach(inject(function(_$q_, _$timeout_, _MonoFuturePool_, _$rootScope_, _DataikuAPI_, _$flushPendingTasks_) {
        $q = _$q_;
        MonoFuturePool = _MonoFuturePool_;
        $rootScope = _$rootScope_;
        $timeout = _$timeout_;
        $flushPendingTasks = _$flushPendingTasks_;
        DataikuAPI = _DataikuAPI_
        clock = new Clock();
        apiCallsLog = [];
    }));

    function log(str) {
        apiCallsLog.push(`t=${clock.currentTime}: ${str}`)
    }

    // Helper function to create a mock promise with .success() and .error() methods
    function createMockHttpPromise(qPromise) {
        // Start with the real promise to have a working .then() and .finally()
        const mockPromise = qPromise;

        mockPromise.success = function(callback) {
            mockPromise.then(function(response) {
                callback(response);
            });
            return mockPromise; // Return for chaining
        };

        mockPromise.error = function(callback) {
            mockPromise.then(null, function(error) {
                callback(error);
            });
            return mockPromise; // Return for chaining
        };

        return mockPromise;
    }

    // mock the return value of apiFunc(param) so that it's a promise that returns data when the fake clock hits the time of call + delay
    function mockApiCall(apiFunc, param, data, duration) {
        const callDeferred = $q.defer();
        apiFunc.withArgs(param).and.callFake((params) => {
            log(`call ${params}`);
            clock.schedule(clock.currentTime + duration, () => callDeferred.resolve({ hasResult: true, data }));
            return createMockHttpPromise(callDeferred.promise)
        });
    }

    function mockApiError(apiFunc, param, duration) {
        const callDeferred = $q.defer();
        apiFunc.withArgs(param).and.callFake((params) => {
            log(`call ${params}`);
            clock.schedule(clock.currentTime + duration, () => callDeferred.reject('api call error'));
            return createMockHttpPromise(callDeferred.promise)
        });
    }

    function mockLongTask(apiFunc, param, finalPayload, duration) {
        apiFunc.withArgs(param).and.callFake((params) => {
            const callDeferred = $q.defer();
            const callTime = clock.currentTime;

            log(`call ${params}`);
            callDeferred.resolve({ hasResult: false, jobId: param });

            DataikuAPI.futures.getUpdate.withArgs(param).and.callFake(() => {
                const updateDeferred = $q.defer();
                if(clock.currentTime >= callTime + duration) {
                    updateDeferred.resolve(finalPayload);
                } else {
                    updateDeferred.resolve({ hasResult: false, id: param, progress: clock.currentTime - callTime });
                }
                return createMockHttpPromise(updateDeferred.promise)
            });

            clock.schedule(callTime + duration, () => {}); // just to have flush last long enough

            return createMockHttpPromise(callDeferred.promise)
        });
    }

    function mockLongTaskSuccess(apiFunc, param, data, duration) {
        mockLongTask(apiFunc, param, { hasResult: true, result: data }, duration);
    }

    function mockLongTaskError(apiFunc, param, data, duration) {
        mockLongTask(apiFunc, param, $q.reject({ hasResult: true, result: data }), duration);
    }

    function mockLongTaskAborted(apiFunc, param, data, duration) {
        mockLongTask(apiFunc, param, { hasResult: true, result: data, aborted: true }, duration);
    }

    const logSuccess = ({data}) => apiCallsLog.push(`t=${clock.currentTime}: get ${data}`);
    const logTaskResult = ({result}) => apiCallsLog.push(`t=${clock.currentTime}: result ${result}`);
    const logTaskUpdate = ({id, progress}) => apiCallsLog.push(`t=${clock.currentTime}: update ${id} ${progress}`);
    const logError = (err) => apiCallsLog.push(`t=${clock.currentTime}: error ${err}`);


    // --- TESTS ---
    describe('with a single MonoFuture & createMonoFuture', () => {
        it('should handle a single success', () => {
            // Arrange
            const monoFuture = MonoFuturePool().createMonoFuture();
            const apiFunc = jasmine.createSpy('apiFunc');
            const future = monoFuture.wrap(apiFunc);
            mockApiCall(apiFunc, 'arg1', 'SUCCESS', 1);

            // Act
            future('arg1').success(logSuccess);
            clock.flush();

            // Assert
            expect(apiCallsLog).toEqual(jasmine.arrayWithExactContents([
                't=0: call arg1',
                't=1: get SUCCESS'
            ]));
        });

        it('should queue a second request and execute it after the first completes', () => {
            // Arrange
            const apiFunc = jasmine.createSpy('apiFunc');
            mockApiCall(apiFunc, 'first', 'FIRST', 1);
            mockApiCall(apiFunc, 'second', 'SECOND', 2);
            const monoFuture = MonoFuturePool().createMonoFuture();
            const future = monoFuture.wrap(apiFunc);

            // Act
            future('first').success(logSuccess);
            future('second').success(logSuccess);
            clock.flush();

            // Assert
            expect(apiCallsLog).toEqual(jasmine.arrayWithExactContents([
                't=0: call first',
                't=1: get FIRST',
                't=1: call second',
                't=3: get SECOND'
            ]));
        });

        it('should cancel a waiting request if a new one is made', () => {
            // Arrange
            const apiFunc = jasmine.createSpy('apiFunc');
            mockApiCall(apiFunc, 'first', 'FIRST', 1);
            mockApiCall(apiFunc, 'third', 'THIRD', 3);
            const monoFuture = MonoFuturePool().createMonoFuture();
            const future = monoFuture.wrap(apiFunc);

            // Act
            future('first').success(logSuccess);
            future('second').success(logSuccess);
            future('third').success(logSuccess);
            clock.flush();

            // Assert
            expect(apiCallsLog).toEqual(jasmine.arrayWithExactContents([
                't=0: call first',
                't=1: get FIRST',
                't=1: call third',
                't=4: get THIRD'
            ]));
        });
    });

    describe('with a multiple MonoFutures', () => {
        it('should queue a query on a second mono-future', () => {
            // Arrange
            const apiFunc1 = jasmine.createSpy('apiFunc1');
            const apiFunc2 = jasmine.createSpy('apiFunc2');
            const monoFuturePool = MonoFuturePool();
            const future1 = monoFuturePool.createMonoFuture().wrap(apiFunc1);
            const future2 = monoFuturePool.createMonoFuture().wrap(apiFunc2);
            mockApiCall(apiFunc1, 'first', 'FIRST', 1);
            mockApiCall(apiFunc2, 'second', 'SECOND', 3);

            // Act
            future1('first').success(logSuccess);
            future2('second').success(logSuccess);
            clock.flush();

            // Assert
            expect(apiCallsLog).toEqual(jasmine.arrayWithExactContents([
                't=0: call first',
                't=1: get FIRST',
                't=1: call second',
                't=4: get SECOND'
            ]));
        });

        it('multiple waiting queries should be debounced', () => {
            // Arrange
            const apiFunc1 = jasmine.createSpy('apiFunc1');
            const apiFunc2 = jasmine.createSpy('apiFunc2');
            const monoFuturePool = MonoFuturePool();
            const future1 = monoFuturePool.createMonoFuture().wrap(apiFunc1);
            const future2 = monoFuturePool.createMonoFuture().wrap(apiFunc2);
            mockApiCall(apiFunc1, 'a1', 'A1', 3);
            mockApiCall(apiFunc2, 'b1', 'B1', 3);
            mockApiCall(apiFunc2, 'b2', 'B2', 3);

            // Act
            future1('a1').success(logSuccess);
            future2('b1').success(logSuccess);
            clock.goto(1);
            future2('b2').success(logSuccess);
            clock.flush();

            // Assert
            expect(apiCallsLog).toEqual(jasmine.arrayWithExactContents([
                't=0: call a1',
                't=3: get A1',
                't=3: call b2',
                't=6: get B2'
            ]));
        });

        it('multiple waiting queries should be debounced - even using different wraps', () => {
            // Arrange
            const apiFunc1 = jasmine.createSpy('apiFunc1');
            const apiFunc2 = jasmine.createSpy('apiFunc2');
            const apiFunc2b = jasmine.createSpy('apiFunc2b');
            const monoFuturePool = MonoFuturePool();
            const future1 = monoFuturePool.createMonoFuture().wrap(apiFunc1);
            const monoFuture2 = monoFuturePool.createMonoFuture();
            const future2 = monoFuture2.wrap(apiFunc2);
            const future2b = monoFuture2.wrap(apiFunc2b);
            mockApiCall(apiFunc1, 'a1', 'A1', 3);
            mockApiCall(apiFunc2, 'b1', 'B1', 3);
            mockApiCall(apiFunc2b, 'c2', 'C2', 3);

            // Act
            future1('a1').success(logSuccess);
            future2('b1').success(logSuccess);
            clock.goto(1);
            future2b('c2').success(logSuccess);
            clock.flush();

            // Assert
            expect(apiCallsLog).toEqual(jasmine.arrayWithExactContents([
                't=0: call a1',
                't=3: get A1',
                't=3: call c2',
                't=6: get C2'
            ]));
        });
    });

    it('should work with many MonoFutures', () => {
        // Arrange
        const monoFuturePool = MonoFuturePool();
        const logExpectation = [];

        for (let i = 1; i < 20 ; i++) {
            const apiFunc = jasmine.createSpy('apiFunc'+i);
            const future = monoFuturePool.createMonoFuture().wrap(apiFunc);
            mockApiCall(apiFunc, 'aaa'+i, 'AAA'+i, 2);
            clock.schedule(i, () => future('aaa'+i).success(logSuccess));

            logExpectation.push(`t=${2*i-1}: call aaa${i}`)
            logExpectation.push(`t=${2*i + 1}: get AAA${i}`)
        }
        clock.flush();

        // Assert
        expect(apiCallsLog).toEqual(jasmine.arrayWithExactContents(logExpectation));
    });

    describe('abort', () => {
        // emphasis on ability to reuse other futures is because not accidentally locking the full poll on aborted monoFuture was not so easy

        it('on pending should cancel query and allow to reuse the mono-future it later', () => {
            // Arrange
            const monoFuture = MonoFuturePool().createMonoFuture()
            const apiFunc = jasmine.createSpy('apiFunc');
            const future = monoFuture.wrap(apiFunc);
            mockApiCall(apiFunc, 'a', 'A', 3);
            mockApiCall(apiFunc, 'b', 'B', 3);
            mockApiCall(apiFunc, 'c', 'C', 3);

            // Act
            future('a').success(logSuccess);
            future('b').success(logSuccess); // this one should be waiting until abort, then aborted without even starting
            clock.goto(1);
            monoFuture.abort();
            clock.goto(2);
            future('c').success(logSuccess);
            clock.flush();

            // Assert
            expect(apiCallsLog).toEqual(jasmine.arrayWithExactContents([
                't=0: call a',
                't=2: call c',
                't=5: get C'
            ]));
        });

        it('on pending should trigger the next waiting promise', () => {
            // Arrange
            const monoFuturePool = MonoFuturePool();
            const monoFuture1 = monoFuturePool.createMonoFuture();
            const monoFuture2 = monoFuturePool.createMonoFuture();
            const apiFunc1 = jasmine.createSpy('apiFunc1');
            const future1 = monoFuture1.wrap(apiFunc1);
            const apiFunc2 = jasmine.createSpy('apiFunc1');
            const future2 = monoFuture2.wrap(apiFunc2);
            mockApiCall(apiFunc1, 'a', 'A', 3);
            mockApiCall(apiFunc2, 'b', 'B', 3);

            // Act
            future1('a').success(logSuccess);
            future2('b').success(logSuccess);
            clock.goto(1);
            monoFuture1.abort();
            clock.flush();

            // Assert
            expect(apiCallsLog).toEqual(jasmine.arrayWithExactContents([
                't=0: call a',
                't=1: call b',
                't=4: get B'
            ]));
        });


        it('on waiting should drop it and allow to use it later', () => {
            // Arrange
            const monoFuturePool = MonoFuturePool();
            const monoFuture1 = monoFuturePool.createMonoFuture();
            const monoFuture2 = monoFuturePool.createMonoFuture();
            const apiFunc1 = jasmine.createSpy('apiFunc1');
            const future1 = monoFuture1.wrap(apiFunc1);
            const apiFunc2 = jasmine.createSpy('apiFunc1');
            const future2 = monoFuture2.wrap(apiFunc2);
            mockApiCall(apiFunc1, 'a', 'A', 3);
            mockApiCall(apiFunc2, 'b', 'B', 3);
            mockApiCall(apiFunc2, 'c', 'C', 3);

            // Act
            future1('a').success(logSuccess);
            future2('b').success(logSuccess);
            clock.goto(1);
            monoFuture2.abort();
            clock.goto(2);
            future2('c').success(logSuccess);
            clock.flush();

            // Assert
            expect(apiCallsLog).toEqual(jasmine.arrayWithExactContents([
                't=0: call a',
                't=3: get A',
                't=3: call c',
                't=6: get C'
            ]));
        });
    });

    describe('monoFuture destroy', () => {
        // emphasis on ability to reuse other futures because not accidentally locking the full poll on destroyed monoFuture was not so easy

        it('on pending should cancel query and not to reuse the mono-future it later - but does not lock out other futures', () => {
            // Arrange
            const monoFuturePool = MonoFuturePool();
            const monoFuture1 = monoFuturePool.createMonoFuture();
            const monoFuture2 = monoFuturePool.createMonoFuture();
            const apiFunc1 = jasmine.createSpy('apiFunc1');
            const future1 = monoFuture1.wrap(apiFunc1);
            const apiFunc2 = jasmine.createSpy('apiFunc1');
            const future2 = monoFuture2.wrap(apiFunc2);
            mockApiCall(apiFunc1, 'a1', 'A1', 3);
            mockApiCall(apiFunc1, 'b1', 'B1', 3);
            mockApiCall(apiFunc2, 'a2', 'A2', 3);

            // Act
            future1('a1').success(logSuccess);
            future1('b1').success(logSuccess); // this one should be waiting until destroy, then dropped without even starting
            clock.goto(1);
            monoFuture1.destroy();
            $timeout.flush();
            clock.goto(10);
            future1('c1').success(logSuccess);
            future2('a2').success(logSuccess);
            clock.flush();

            // Assert
            expect(apiCallsLog).toEqual(jasmine.arrayWithExactContents([
                't=0: call a1',
                't=10: call a2',
                't=13: get A2'
            ]));
        });

        it('on pending should trigger the next waiting promise', () => {
            // Arrange
            const monoFuturePool = MonoFuturePool();
            const monoFuture1 = monoFuturePool.createMonoFuture();
            const monoFuture2 = monoFuturePool.createMonoFuture();
            const apiFunc1 = jasmine.createSpy('apiFunc1');
            const future1 = monoFuture1.wrap(apiFunc1);
            const apiFunc2 = jasmine.createSpy('apiFunc1');
            const future2 = monoFuture2.wrap(apiFunc2);
            mockApiCall(apiFunc1, 'a', 'A', 3);
            mockApiCall(apiFunc2, 'b', 'B', 3);

            // Act
            future1('a').success(logSuccess);
            future2('b').success(logSuccess);
            clock.goto(1);
            monoFuture1.destroy();
            $timeout.flush();
            clock.flush();

            // Assert
            expect(apiCallsLog).toEqual(jasmine.arrayWithExactContents([
                't=0: call a',
                't=1: call b',
                't=4: get B'
            ]));
        });


        it('on waiting should drop it, not allow to use it later, but let other monoFuture proceed and does not lock out other monoFutures', () => {
            // Arrange
            const monoFuturePool = MonoFuturePool();
            const monoFuture1 = monoFuturePool.createMonoFuture();
            const monoFuture2 = monoFuturePool.createMonoFuture();
            const apiFunc1 = jasmine.createSpy('apiFunc1');
            const future1 = monoFuture1.wrap(apiFunc1);
            const apiFunc2 = jasmine.createSpy('apiFunc1');
            const future2 = monoFuture2.wrap(apiFunc2);
            mockApiCall(apiFunc1, 'a1', 'A1', 3);
            mockApiCall(apiFunc2, 'b2', 'B2', 3);
            mockApiCall(apiFunc2, 'c2', 'C2', 3);
            mockApiCall(apiFunc1, 'd1', 'D1', 3);

            // Act
            future1('a1').success(logSuccess);
            future2('b2').success(logSuccess);
            clock.goto(1);
            monoFuture2.destroy();
            $timeout.flush();
            clock.goto(2);
            future2('c2').success(logSuccess);
            future1('d1').success(logSuccess);
            clock.flush();

            // Assert
            expect(apiCallsLog).toEqual(jasmine.arrayWithExactContents([
                't=0: call a1',
                't=3: get A1',
                't=3: call d1',
                't=6: get D1'
            ]));
        });
    });

    describe('monoFuture auto-destroy', () => {
        let poolScope, futureScope1, futureScope2, monoFuturePool, monoFuture1, monoFuture2, future1, future2;
        beforeEach(() => {
            poolScope = $rootScope.$new();
            futureScope1 = poolScope.$new();
            futureScope2 = poolScope.$new();

            monoFuturePool = MonoFuturePool(poolScope);
            monoFuture1 = monoFuturePool.createMonoFuture(futureScope1);
            monoFuture2 = monoFuturePool.createMonoFuture(futureScope2);

            const apiFunc1 = jasmine.createSpy('apiFunc1');
            future1 = monoFuture1.wrap(apiFunc1);
            const apiFunc2 = jasmine.createSpy('apiFunc1');
            future2 = monoFuture2.wrap(apiFunc2);

            mockApiCall(apiFunc1, 'a1', 'A1', 3);
            mockApiCall(apiFunc1, 'b1', 'B1', 3);
            mockApiCall(apiFunc2, 'a2', 'A2', 3);
            mockApiCall(apiFunc2, 'b2', 'B2', 3);
        })

        it('on single future scope destroy should destroy just the right mono-future and let others be', () => {

            // Act
            future1('a1').success(logSuccess);
            clock.goto(1);
            futureScope1.$destroy();
            $timeout.flush();
            clock.goto(10);
            future1('b1').success(logSuccess);
            future2('b2').success(logSuccess);
            clock.flush();

            // Assert
            expect(apiCallsLog).toEqual(jasmine.arrayWithExactContents([
                't=0: call a1',
                't=10: call b2',
                't=13: get B2'
            ]));
        });

        it('on pool scope should abort all pending / waiting calls', () => {

            // Act
            future1('a1').success(logSuccess);
            future2('a2').success(logSuccess);
            clock.goto(1);
            poolScope.$destroy();
            $timeout.flush();
            clock.flush();

            // Assert
            expect(apiCallsLog).toEqual(jasmine.arrayWithExactContents([
                't=0: call a1'
            ]));
        });
    });

    describe('Api call error behavior', () => {

        it('on single future scope destroy should destroy just the right mono-future and let others be', () => {
            // Arrange
            const monoFuturePool = MonoFuturePool();
            const monoFuture1 = monoFuturePool.createMonoFuture();
            const monoFuture2 = monoFuturePool.createMonoFuture();
            const apiFunc1 = jasmine.createSpy('apiFunc1');
            const future1 = monoFuture1.wrap(apiFunc1);
            const apiFunc2 = jasmine.createSpy('apiFunc1');
            const future2 = monoFuture2.wrap(apiFunc2);
            mockApiError(apiFunc1, 'a1', 3);
            mockApiCall(apiFunc2, 'a2', 'A2', 3);

            // Act
            future1('a1').error(logError);
            future2('a2').success(logSuccess);
            clock.flush();

            // Assert
            expect(apiCallsLog).toEqual(jasmine.arrayWithExactContents([
                't=0: call a1',
                't=3: error api call error',
                't=3: call a2',
                't=6: get A2'
            ]));
        });
    });


    describe('Update tracking behavior', () => {
        beforeEach(() => {
            DataikuAPI.futures = {
                getUpdate: jasmine.createSpy('getUpdate'),
                abort: jasmine.createSpy('getUpdate').and.callFake((id) => log('abort '+id)),
            };
        });

        it('should track long future and get its result', () => {
            // Arrange
            const monoFuturePool = MonoFuturePool();
            const monoFuture = monoFuturePool.createMonoFuture();
            const apiFunc = jasmine.createSpy('apiFunc');
            const future = monoFuture.wrap(apiFunc);
            mockLongTaskSuccess(apiFunc, 'a1', 'A1', 2);

            // Act
            future('a1')
                .update(logTaskUpdate)
                .success(logTaskResult);
            clock.flush(true);

            // Assert
            expect(apiCallsLog).toEqual(jasmine.arrayWithExactContents([
                't=0: call a1',
                't=0: update a1 0',
                't=0: update a1 0',
                't=1: update a1 1',
                't=2: result A1'
            ]));
        });

        it('should reject if long task fails', () => {
            // Arrange
            const monoFuturePool = MonoFuturePool();
            const monoFuture = monoFuturePool.createMonoFuture();
            const apiFunc = jasmine.createSpy('apiFunc');
            const future = monoFuture.wrap(apiFunc);
            mockLongTaskError(apiFunc, 'a1', 'A1', 5);

            // Act
            future('a1').error(logTaskResult);
            clock.flush(true);

            // Assert
            expect(apiCallsLog).toEqual(jasmine.arrayWithExactContents([
                't=0: call a1',
                't=5: result A1'
            ]));
        });

        it('should reject if long task is aborted (backend side)', () => {
            // Arrange
            const monoFuturePool = MonoFuturePool();
            const monoFuture = monoFuturePool.createMonoFuture();
            const apiFunc = jasmine.createSpy('apiFunc');
            const future = monoFuture.wrap(apiFunc);
            mockLongTaskAborted(apiFunc, 'a1', 'A1', 5);

            // Act
            future('a1').error(logTaskResult);
            clock.flush(true);

            // Assert
            expect(apiCallsLog).toEqual(jasmine.arrayWithExactContents([
                't=0: call a1',
                't=5: result A1'
            ]));
        });

        it('should abort future if long task is aborted (frontend side)', () => {
            // Arrange
            const monoFuturePool = MonoFuturePool();
            const monoFuture = monoFuturePool.createMonoFuture();
            const apiFunc = jasmine.createSpy('apiFunc');
            const future = monoFuture.wrap(apiFunc);
            mockLongTaskSuccess(apiFunc, 'a1', 'A1', 5);

            // Act
            future('a1').error(logTaskResult);
            clock.schedule(3, () => monoFuture.abort());
            clock.flush(true);

            // Assert
            expect(apiCallsLog).toEqual(jasmine.arrayWithExactContents([
                't=0: call a1',
                't=3: abort a1'
            ]));
        });


        it('multiple calls should wait the long task completion', () => {
            // Arrange
            const monoFuturePool = MonoFuturePool();
            const monoFuture1 = monoFuturePool.createMonoFuture();
            const apiFunc1 = jasmine.createSpy('apiFunc1');
            const future1 = monoFuture1.wrap(apiFunc1);
            mockLongTaskSuccess(apiFunc1, 'a1', 'A1', 5);
            const monoFuture2 = monoFuturePool.createMonoFuture();
            const apiFunc2 = jasmine.createSpy('apiFunc2');
            const future2 = monoFuture2.wrap(apiFunc2);
            mockLongTaskSuccess(apiFunc2, 'a2', 'A2', 5);

            // Act
            future1('a1').success(logTaskResult);
            future2('a2').success(logTaskResult);
            clock.flush(true);

            // Assert
            expect(apiCallsLog).toEqual(jasmine.arrayWithExactContents([
                't=0: call a1',
                't=5: result A1',
                't=5: call a2',
                't=10: result A2'
            ]));
        });

        it('long task error / abort should allow next task to proceed', () => {
            // Arrange
            const monoFuturePool = MonoFuturePool();
            const monoFuture1 = monoFuturePool.createMonoFuture();
            const apiFunc1 = jasmine.createSpy('apiFunc1');
            const future1 = monoFuture1.wrap(apiFunc1);
            mockLongTaskError(apiFunc1, 'a1', 'A1', 5);

            const monoFuture2 = monoFuturePool.createMonoFuture();
            const apiFunc2 = jasmine.createSpy('apiFunc2');
            const future2 = monoFuture2.wrap(apiFunc2);
            mockLongTaskAborted(apiFunc2, 'a2', 'A2', 5);

            const monoFuture3 = monoFuturePool.createMonoFuture();
            const apiFunc3 = jasmine.createSpy('apiFunc3');
            const future3 = monoFuture3.wrap(apiFunc3);
            mockLongTaskSuccess(apiFunc3, 'a3', 'A3', 5);

            const monoFuture4 = monoFuturePool.createMonoFuture();
            const apiFunc4 = jasmine.createSpy('apiFunc4');
            const future4 = monoFuture4.wrap(apiFunc4);
            mockLongTaskSuccess(apiFunc4, 'a4', 'A4', 5);

            // Act
            future1('a1').error(logTaskResult);
            future2('a2').error(logTaskResult);
            future3('a3').success(logTaskResult);
            future4('a4').success(logTaskResult);
            clock.schedule(13, () => monoFuture3.abort());
            clock.flush(true);

            // Assert
            expect(apiCallsLog).toEqual(jasmine.arrayWithExactContents([
                't=0: call a1',
                't=5: result A1',
                't=5: call a2',
                't=10: result A2',
                't=10: call a3',
                't=13: abort a3',
                't=13: call a4',
                't=18: result A4',
            ]));
        });


    });

});
