from abc import ABCMeta
from abc import abstractmethod
import logging

from six import add_metaclass

from dataiku.base.utils import safe_unicode_str
from dataiku.core import doctor_constants
from dataiku.core.dku_logging import LogLevelContext


@add_metaclass(ABCMeta)
class PredictorAdapterForEmu(object):
    """
    Adapter to make DSS predictors compatible with the EMU engine.
    Motivation: EMU's counterfactuals engine needs a sklearn-compatible classifier
        implementing `predict(self, X)` and `predict_proba(self, X)` correctly.
        In particular, X is supposed to be a numpy array (not a pd.DataFrame) that
        didn't go through the doctor's preprocessing. So, simply passing the predictor's
        _clf attribute to the engine cannot work because _clf is expecting preprocessed
        features. Moreover, in order to use KMeans, EMU requires a preprocess function to
        turn the pd.DataFrame into a preprocessed numpy array.
    """
    def __init__(self, model_handler):
        self.predictor = model_handler.get_predictor()

    def preprocess(self, df):
        with LogLevelContext(logging.CRITICAL, doctor_constants.PREPROCESSING_RELATED_LOGGER_NAMES):
            final_df = self.predictor.preformat(df)
            X, _, _ = self.predictor.preprocess(final_df)
        return X

    @abstractmethod
    def predict(self, df):
        pass


class ClassifierAdapterForEmu(PredictorAdapterForEmu):
    """For counterfactuals."""
    def __init__(self, model_handler):
        super(ClassifierAdapterForEmu, self).__init__(model_handler)
        self.target_map = model_handler.get_target_map()
        self.inv_target_map = model_handler.get_inv_map()

    def predict(self, df):
        # retrieve the "processed" targets, e.g. classes replaced by 0, 1, ...
        y = self.predictor.predict(df)["prediction"].apply(lambda x: self.target_map[x])
        return y.values

    def predict_proba(self, df):
        predictions = self.predictor.predict(df)
        proba_columns = [u"proba_{}".format(safe_unicode_str(self.inv_target_map[i]))
                         for i in range(len(self.inv_target_map))]
        probas = predictions.loc[:, proba_columns].values
        return probas


class RegressorAdapterForEmu(PredictorAdapterForEmu):
    """For outcome optimization."""
    def predict(self, df):
        y = self.predictor.predict(df)["prediction"]
        return y.values
