#! /bin/bash

# This is script is based on resources/container-exec/kubernetes/aws-ecr-prepush.sh
# with two main modifs, so it works to deploy api endpoints from Dataiku Cloud
# - we do not use docker
# - we create two repositories, one for the "intermediate" code-env image, one for the "final" endpoint image

REPOSITORY=$1
REGION=$2
ENDPOINT_IMAGE=$3
CODEENV_IMAGE=$4


# if repository is like 000000000000.dkr.ecr.us-east-1.amazonaws.com/project
# then repository name for aws ecr command is project/image-name
ENDPOINT_NAME=$(echo "$REPOSITORY"/"$ENDPOINT_IMAGE" | sed 's_^[^/]*/__')
CODEENV_NAME=$(echo "$REPOSITORY"/"$CODEENV_IMAGE" | sed 's_^[^/]*/__')
REPOSITORY_ROOT=$(echo "$REPOSITORY"/"$ENDPOINT_IMAGE" | sed 's_^\([^/]*\)/.*$_\1_')
echo "INFO: On Dataiku Cloud. Login in to AWS ECR, and making sure repositories '$ENDPOINT_NAME' and '$CODEENV_NAME' exists on ECR" >&2

if [ -z $(command -v aws) ]; then
    echo 'ERROR: required `aws` binary is not in PATH' >&2
    exit 1
fi

REGISTRY_PASSWORD=$(aws ecr get-login-password --region $REGION)
if [[ $? -ne 0 ]]; then
    echo 'WARNING: could not get password' >&2
else
    crane auth login --username AWS --password "$REGISTRY_PASSWORD" $REPOSITORY_ROOT
    if [[ $? -ne 0 ]]; then
        echo 'WARNING: could not log in' >&2
    else
        echo 'INFO: Logged in to AWS ECR.' >&2
    fi
fi

check_and_create_repo () {
    NAME=$1
    A=$(aws ecr describe-repositories --repository-name "$NAME" --region $REGION 2>&1)
    if [[ $? -ne 0 ]]; then
        if echo $A | grep RepositoryNotFoundException; then
            echo "INFO: repository '$NAME' does not exist, creating" >&2
            aws ecr create-repository --repository-name "$NAME" --region $REGION
            if (( $? )); then
                echo "ERROR: could not create repository '$NAME'" >&2
                exit 2
            else
                echo "INFO: created repository '$NAME'" >&2
            fi
        else
            echo "ERROR: could not check repository '$NAME':" >&2
            echo "$A" >&2
            exit 2
        fi
    else
        echo "INFO: repository '$NAME' already exist"
    fi
}

check_and_create_repo "$ENDPOINT_NAME"
check_and_create_repo "$CODEENV_NAME"
