DROP VIEW artifacts_with_external_fields_and_attachments;

CREATE VIEW artifacts_with_external_fields_and_attachments AS

SELECT
    artifacts.artifact_id,
    artifacts.artifact_blueprint_id,
    artifacts.artifact_version_id,
    artifacts.artifact_json AS artifact_json_no_relationships,
    -- building the whole artifact JSON including basic fields, references, backreferences, and attachments (references and uploaded files)
    (
        artifacts.artifact_json

            -------------------------------------------------- build the final fields map in the artifact JSON
            || jsonb_build_object(
                'fields',

                ----------------------------------------------------------- first, need to add the existing fields (internal fields)
                artifacts.artifact_json #> '{fields}'

                ---------------------------------------------------- add the references
                || COALESCE(references_relationships_for_artifact.relationships_object, jsonb_build_object())

                -------------------------- add the back references
                || COALESCE(
                    (
                        CASE -- fallback on empty object if there are no backref field definition for this artifact
                            WHEN backref_references_definitions.backref_references_definitions IS NULL
                            THEN NULL
                            ELSE backreferences_relationships_for_artifact.relationships_object
                        END
                    ),
                    jsonb_build_object()
                )

                ---------------------------------------------------- add the uploaded files
                || COALESCE(uploaded_files_relationships_for_artifact.relationships_object, jsonb_build_object())

                ---------------------------------------------------- add the timeseries
                || COALESCE(time_series_relationships_for_artifact.relationships_object, jsonb_build_object())
            )

            ------------------------------------------- add attachments map in the artifact JSON
            || jsonb_build_object(
                'attachments',
                jsonb_build_object(
                    -------------------------------------- reference attachments
                    'references', COALESCE(attachments_references_relationships_for_artifact.relationships_array, jsonb_build_array()),
                    -------------------------------------- uploaded files attachments
                    'files', COALESCE(attachments_uploaded_files_relationships_for_artifact.relationships_array, jsonb_build_array())
                )
            )
    ) AS artifact_json
FROM
    artifacts artifacts

    LEFT JOIN (
        SELECT
            blueprint_id,
            version_id,
            jsonb_object_agg(field_id, references_list) AS backref_references_definitions
        FROM (
            SELECT
                blueprint_id,
                version_id,
                field_id,
                jsonb_agg(
                    jsonb_build_object(
                        'backref_reference_bp_id', backref_reference_bp_id,
                        'backref_reference_field_id', backref_reference_field_id
                    )
                ) AS references_list
            FROM (
                SELECT
                    blueprint_versions.blueprint_version_blueprint_id AS blueprint_id,
                    blueprint_versions.blueprint_version_version_id AS version_id,
                    fields_definitions.key AS field_id,
                    backref_reference_entry.value #>> '{blueprintId}' AS backref_reference_bp_id,
                    backref_reference_entry.value #>> '{fieldId}' AS backref_reference_field_id
                FROM
                    blueprint_versions blueprint_versions,
                    jsonb_each(blueprint_versions.blueprint_version_json #> '{fieldDefinitions}') AS fields_definitions,
                    jsonb_array_elements(fields_definitions.value #> '{references}') AS backref_reference_entry
                WHERE
                    fields_definitions.value #>> '{fieldType}' = 'REFERENCE'
                    AND fields_definitions.value #>> '{sourceType}' = 'COMPUTE'
            ) backref_references_flatten
            GROUP BY blueprint_id, version_id, field_id
        ) backref_references_per_field
        GROUP BY blueprint_id, version_id
    ) backref_references_definitions
    ON
        backref_references_definitions.blueprint_id = artifacts.artifact_blueprint_id
        AND backref_references_definitions.version_id = artifacts.artifact_version_id

    LEFT JOIN LATERAL (
        SELECT
            jsonb_object_agg(
                elem.field_id,
                (
                    CASE elem.field_type
                        WHEN 'plain'
                        THEN elem.field_value #> '{0}'
                        ELSE elem.field_value
                    END
                )
            ) AS relationships_object
        FROM (
            SELECT
                rel.reference_relationship_json #>> '{relationshipData, fieldId}' AS field_id,
                MIN(rel.reference_relationship_json #>> '{relationshipData, details, type}') AS field_type, -- OK to take only the MIN of field type while grouping by field ID: all field type (plain or list) values should be the same for the same field
                jsonb_agg(
                    rel.reference_relationship_artifact_id_to
                    ORDER BY rel.reference_relationship_json #> '{relationshipData, details, index}' ASC
                ) AS field_value
            FROM reference_relationships rel
            WHERE
                rel.reference_relationship_artifact_id_from = artifacts.artifact_id
                AND rel.reference_relationship_json #>> '{relationshipData, type}' = 'field'
                AND rel.reference_relationship_json #>> '{relationshipData, fieldId}' IS NOT NULL
            GROUP BY rel.reference_relationship_json #>> '{relationshipData, fieldId}'
        ) elem
    ) references_relationships_for_artifact ON TRUE


    LEFT JOIN LATERAL (
        SELECT
            jsonb_object_agg(
                backrefs.field_id,
                backrefs.artifact_id_from_list
            ) AS relationships_object
        FROM (
            SELECT
                field_id,
                COALESCE(
                    jsonb_agg(DISTINCT defs.artifact_id_from),
                    jsonb_build_array()
                ) AS artifact_id_from_list
            FROM (
                SELECT
                    rel.reference_relationship_artifact_id_from AS artifact_id_from,
                    backref_references.field_ids_list AS field_ids_list

                FROM (
                    SELECT
                        jsonb_agg(backref_references_per_field.key) AS field_ids_list,
                        backref_references_flatten #>> '{backref_reference_bp_id}' AS backref_reference_bp_id,
                        backref_references_flatten #>> '{backref_reference_field_id}' AS backref_reference_field_id
                    FROM
                        jsonb_each(backref_references_definitions.backref_references_definitions) AS backref_references_per_field,
                        jsonb_array_elements(backref_references_per_field.value) AS backref_references_flatten
                    GROUP BY
                        backref_references_flatten #>> '{backref_reference_bp_id}',
                        backref_references_flatten #>> '{backref_reference_field_id}'
                ) backref_references,

                LATERAL (
                    SELECT rel.reference_relationship_artifact_id_from
                    FROM reference_relationships rel
                    WHERE
                        rel.reference_relationship_artifact_id_to = artifacts.artifact_id
                        AND rel.reference_relationship_json #>> '{relationshipData, fieldId}' = backref_references.backref_reference_field_id
                ) rel,

                LATERAL (
                    SELECT 1
                    FROM artifacts artifact_from
                    WHERE
                        artifact_from.artifact_id = rel.reference_relationship_artifact_id_from
                        AND artifact_from.artifact_blueprint_id = backref_references.backref_reference_bp_id
                ) artifact_from
            ) defs,
            jsonb_array_elements_text(defs.field_ids_list) AS field_id
            GROUP BY field_id
        ) backrefs
    ) backreferences_relationships_for_artifact ON TRUE

    LEFT JOIN LATERAL (
        SELECT
            jsonb_object_agg(
                elem.field_id,
                (
                    CASE elem.field_type
                        WHEN 'plain'
                        THEN elem.field_value #> '{0}'
                        ELSE elem.field_value
                    END
                )
            ) AS relationships_object
        FROM (
            SELECT
                rel.uploaded_file_relationship_json #>> '{relationshipData, fieldId}' AS field_id,
                MIN(rel.uploaded_file_relationship_json #>> '{relationshipData, details, type}') AS field_type, -- OK to take only the MIN of field type while grouping by field ID: all field type (plain or list) values should be the same for the same field
                jsonb_agg(
                    rel.uploaded_file_relationship_uploaded_file_id
                    ORDER BY rel.uploaded_file_relationship_json #> '{relationshipData, details, index}' ASC
                ) AS field_value
            FROM uploaded_file_relationships rel
            WHERE
                rel.uploaded_file_relationship_artifact_id = artifacts.artifact_id
                AND rel.uploaded_file_relationship_json #>> '{relationshipData, type}' = 'field'
                AND rel.uploaded_file_relationship_json #>> '{relationshipData, fieldId}' IS NOT NULL
            GROUP BY rel.uploaded_file_relationship_json #>> '{relationshipData, fieldId}'
        ) elem
    ) uploaded_files_relationships_for_artifact ON TRUE

    LEFT JOIN LATERAL (
        SELECT
            jsonb_object_agg(
                elem.field_id,
                (
                    CASE elem.field_type
                        WHEN 'plain'
                        THEN elem.field_value #> '{0}'
                        ELSE elem.field_value
                    END
                )
            ) AS relationships_object
        FROM (
            SELECT
                rel.time_series_relationship_json #>> '{relationshipData, fieldId}' AS field_id,
                MIN(rel.time_series_relationship_json #>> '{relationshipData, details, type}') AS field_type, -- OK to take only the MIN of field type while grouping by field ID: all field type (plain or list) values should be the same for the same field
                jsonb_agg(
                    rel.time_series_relationship_time_series_id
                    ORDER BY rel.time_series_relationship_json #> '{relationshipData, details, index}' ASC
                ) AS field_value
            FROM time_series_relationships rel
            WHERE
                rel.time_series_relationship_artifact_id = artifacts.artifact_id
                AND rel.time_series_relationship_json #>> '{relationshipData, type}' = 'field'
                AND rel.time_series_relationship_json #>> '{relationshipData, fieldId}' IS NOT NULL
            GROUP BY rel.time_series_relationship_json #>> '{relationshipData, fieldId}'
        ) elem
    ) time_series_relationships_for_artifact ON TRUE

    LEFT JOIN LATERAL (
        SELECT
            jsonb_agg(
                rel.reference_relationship_artifact_id_to
                ORDER BY rel.reference_relationship_json #> '{relationshipData, index}' ASC
            ) AS relationships_array
        FROM reference_relationships rel
        WHERE
            rel.reference_relationship_artifact_id_from = artifacts.artifact_id
            AND rel.reference_relationship_json #>> '{relationshipData, type}' = 'attachment'
    ) attachments_references_relationships_for_artifact ON TRUE

    LEFT JOIN LATERAL (
        SELECT
            jsonb_agg(
                rel.uploaded_file_relationship_uploaded_file_id
                ORDER BY rel.uploaded_file_relationship_json #> '{relationshipData, index}' ASC
            ) AS relationships_array
        FROM uploaded_file_relationships rel
        WHERE
            rel.uploaded_file_relationship_artifact_id = artifacts.artifact_id
            AND rel.uploaded_file_relationship_json #>> '{relationshipData, type}' = 'attachment'
    ) attachments_uploaded_files_relationships_for_artifact ON TRUE;
