/*
 * Decompiled with CFR 0.152.
 */
package com.dataiku.dip.datasets.cassandra;

import com.dataiku.dip.connections.CassandraUtil;
import com.dataiku.dip.connections.ConnectionsDAO;
import com.dataiku.dip.coremodel.Dataset;
import com.dataiku.dip.coremodel.SchemaColumn;
import com.dataiku.dip.dataflow.jobrunner.JobContext;
import com.dataiku.dip.datalayer.ColumnFactory;
import com.dataiku.dip.datalayer.Row;
import com.dataiku.dip.datasets.cassandra.AbstractCassandraDatasetHandler;
import com.dataiku.dip.exceptions.DKUSecurityException;
import com.dataiku.dip.output.Output;
import com.dataiku.dip.output.OutputWriter;
import com.dataiku.dip.partitioning.Partition;
import com.dataiku.dip.utils.DKULogger;
import com.dataiku.dip.utils.ErrorContext;
import com.datastax.driver.core.BoundStatement;
import com.datastax.driver.core.ColumnDefinitions;
import com.datastax.driver.core.PreparedStatement;
import com.datastax.driver.core.ResultSet;
import com.datastax.driver.core.ResultSetFuture;
import com.datastax.driver.core.Statement;
import com.datastax.driver.core.TableMetadata;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import java.io.IOException;
import java.util.List;
import java.util.Random;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Semaphore;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

public class CassandraOutput
implements Output {
    private AbstractCassandraDatasetHandler handler;
    private Dataset dataset;
    private Partition targetPartition;
    private ConnectionsDAO connectionsDAO;
    private AbstractCassandraDatasetHandler.Config config;
    protected static DKULogger logger = DKULogger.getLogger((String)"dku.cassandra");

    public CassandraOutput(AbstractCassandraDatasetHandler handler, Dataset dataset, Partition targetPartition, ConnectionsDAO connectionsDAO) {
        assert (dataset.getType().equals(AbstractCassandraDatasetHandler.META.getType()));
        this.handler = handler;
        this.dataset = dataset;
        this.config = dataset.getParamsAs(AbstractCassandraDatasetHandler.Config.class);
        this.targetPartition = targetPartition;
        this.connectionsDAO = connectionsDAO;
        if (dataset.getPartitioningSchema().isPartitioned()) {
            assert (this.config.partitioningColumn != null);
            if (dataset.isManaged()) assert (this.config.partitionSpread >= 1);
        }
    }

    public OutputWriter getWriter(Output.WriteMode mode) throws IOException {
        if (mode == Output.WriteMode.APPEND) {
            throw new IllegalArgumentException("APPEND mode is not supported for Cassandra");
        }
        try {
            return new CassandraOutputWriter(this.connectionsDAO);
        }
        catch (DKUSecurityException e) {
            throw new IOException("Failed to create writer", e);
        }
    }

    public class CassandraOutputWriter
    extends OutputWriter {
        private CassandraUtil.ClusterConnection dbConn;
        private ColumnFactory cf;
        private List<SchemaColumn> schemaColumns;
        private SchemaColumn partitioningColumn;
        private int partitionSpread;
        private Random random;
        private PreparedStatement ps;
        private long totalRows = 0L;
        private AtomicLong insertedRows = new AtomicLong(0L);
        private AtomicLong failedRows = new AtomicLong(0L);
        private boolean deleteBeforeWrite = true;
        private long maxPrintErrors = 100L;
        private int maxQueries = 500;
        private Semaphore availableQueries = new Semaphore(this.maxQueries);
        private ExecutorService asyncQueryExecutor = Executors.newSingleThreadExecutor();
        private FutureCallback<ResultSet> asyncQueryHandler = new FutureCallback<ResultSet>(){

            public void onSuccess(ResultSet result) {
                CassandraOutputWriter.this.insertedRows.incrementAndGet();
                CassandraOutputWriter.this.availableQueries.release();
            }

            public void onFailure(Throwable t) {
                long failed = CassandraOutputWriter.this.failedRows.incrementAndGet();
                CassandraOutputWriter.this.availableQueries.release();
                if (failed <= CassandraOutputWriter.this.maxPrintErrors) {
                    logger.warn((Object)"Failed to insert Cassandra row", t);
                    if (failed == CassandraOutputWriter.this.maxPrintErrors) {
                        logger.warn((Object)"Too many errors, stop reporting");
                    }
                }
            }
        };

        public CassandraOutputWriter(ConnectionsDAO connectionsDAO) throws IOException, DKUSecurityException {
            logger.info((Object)("Connecting to cluster, connection=" + CassandraOutput.this.config.connection));
            this.dbConn = CassandraUtil.acquireConnection(CassandraOutput.this.handler.authCtx, CassandraOutput.this.config.connection);
            if (CassandraOutput.this.dataset.getPartitioningSchema().isPartitioned()) {
                if (CassandraOutput.this.dataset.isManaged()) {
                    this.partitionSpread = CassandraOutput.this.config.partitionSpread;
                    if (this.partitionSpread > 0) {
                        this.random = new Random();
                    }
                } else {
                    this.deleteBeforeWrite = CassandraOutput.this.config.deleteBeforeWrite;
                }
            }
            final JobContext jobContext = JobContext.getCurrentJobContext();
            this.asyncQueryExecutor = Executors.newSingleThreadExecutor(new ThreadFactory(){

                @Override
                public Thread newThread(final Runnable r) {
                    return new Thread(){

                        @Override
                        public void run() {
                            Thread.currentThread().setName("CASSANDRA-" + Thread.currentThread().getId());
                            JobContext.setJobContext(jobContext);
                            r.run();
                        }
                    };
                }
            });
        }

        public void init(ColumnFactory cf) {
            this.cf = cf;
            if (CassandraOutput.this.dataset.getSchema() == null || CassandraOutput.this.dataset.getSchema().getColumns().size() == 0) {
                throw ErrorContext.iaef((String)"Dataset %s has no schema, cannot write", (Object)CassandraOutput.this.dataset.getName(), (Object[])new Object[0]);
            }
            this.schemaColumns = CassandraOutput.this.dataset.getSchema().getColumns();
            if (CassandraOutput.this.dataset.isManaged()) {
                logger.info((Object)"Checking table");
                TableMetadata tableMeta = this.dbConn.getKeyspaceMetadata().getTable(CassandraUtil.quote(CassandraOutput.this.config.table));
                if (tableMeta != null) {
                    try {
                        CassandraUtil.checkManagedTableSchema(tableMeta, CassandraOutput.this.dataset.getSchema());
                    }
                    catch (IllegalArgumentException e) {
                        logger.warn((Object)("table schema mismatch: " + e.getMessage()));
                        if (CassandraOutput.this.config.partitioningColumn != null) {
                            throw new Error("table schema mismatch on partitioned dataset: " + CassandraOutput.this.config.table);
                        }
                        logger.info((Object)"dropping table");
                        this.dbConn.getSession().execute("DROP TABLE " + CassandraUtil.quote(CassandraOutput.this.config.table));
                        tableMeta = null;
                    }
                }
                if (tableMeta == null) {
                    logger.info((Object)("creating table: " + CassandraOutput.this.config.table));
                    String cql = CassandraOutput.this.handler.createTableCommand(false);
                    logger.debug((Object)("Sending cql: " + cql));
                    this.dbConn.getSession().execute(cql);
                    this.deleteBeforeWrite = false;
                }
            }
            StringBuilder sb1 = new StringBuilder();
            StringBuilder sb2 = new StringBuilder();
            if (CassandraOutput.this.dataset.isManaged()) {
                sb1.append(CassandraUtil.quote("_dssId"));
                sb2.append("now()");
            }
            for (SchemaColumn sc : this.schemaColumns) {
                String colName;
                if (sb2.length() > 0) {
                    sb1.append(',');
                    sb2.append(',');
                }
                if ((colName = sc.getName()).equals(CassandraOutput.this.config.partitioningColumn)) {
                    this.partitioningColumn = sc;
                }
                sb1.append(CassandraUtil.quote(colName));
                sb2.append('?');
            }
            if (this.partitionSpread > 0) {
                sb1.append(',');
                sb1.append(CassandraUtil.quote("_dssSpread"));
                sb2.append(',');
                sb2.append('?');
            }
            String insertCQL = "INSERT INTO " + CassandraUtil.quote(CassandraOutput.this.config.table) + "(" + String.valueOf(sb1) + ") VALUES (" + String.valueOf(sb2) + ")";
            logger.debug((Object)("Preparing cql: " + insertCQL));
            this.ps = this.dbConn.prepare(insertCQL);
            ColumnDefinitions colDefs = this.ps.getVariables();
            if (colDefs.size() != this.schemaColumns.size() + (this.partitionSpread > 0 ? 1 : 0)) {
                throw ErrorContext.iaef((String)"prepared query column number %d does not match schema column number %d", (Object)colDefs.size(), (Object[])new Object[]{this.schemaColumns.size()});
            }
            if (this.deleteBeforeWrite) {
                if (CassandraOutput.this.config.partitioningColumn == null) {
                    logger.info((Object)("Deleting table contents: " + CassandraOutput.this.config.table));
                    cql = "TRUNCATE " + CassandraUtil.quote(CassandraOutput.this.config.table);
                    logger.debug((Object)("Sending cql: " + cql));
                    this.dbConn.getSession().execute(cql);
                } else {
                    logger.info((Object)("Deleting partition contents for table " + CassandraOutput.this.config.table + " partition=" + CassandraOutput.this.targetPartition.id()));
                    cql = "DELETE FROM " + CassandraUtil.quote(CassandraOutput.this.config.table) + " WHERE " + CassandraUtil.partitionFilterTerm(CassandraOutput.this.config.partitioningColumn, this.partitionSpread);
                    logger.debug((Object)("Preparing cql: " + cql));
                    BoundStatement st1 = this.dbConn.prepare(cql).bind();
                    CassandraUtil.setCassandraValue(st1, 0, CassandraOutput.this.targetPartition.id());
                    this.dbConn.getSession().execute((Statement)st1);
                }
            }
        }

        public void emitRow(Row row) throws Exception {
            ++this.totalRows;
            BoundStatement st2 = this.ps.bind();
            try {
                for (int i = 0; i < this.schemaColumns.size(); ++i) {
                    String val;
                    SchemaColumn sc = this.schemaColumns.get(i);
                    String string = val = sc == this.partitioningColumn ? CassandraOutput.this.targetPartition.id() : row.get(this.cf.column(sc.getName()));
                    if (val != null) {
                        CassandraUtil.setCassandraValue(st2, i, val);
                        continue;
                    }
                    st2.setToNull(i);
                }
            }
            catch (Exception e) {
                long failed = this.failedRows.incrementAndGet();
                if (failed <= this.maxPrintErrors) {
                    logger.warn((Object)"Failed to build Cassandra row", (Throwable)e);
                    if (failed == this.maxPrintErrors) {
                        logger.warn((Object)"Too many errors, stop reporting");
                    }
                }
                return;
            }
            if (this.partitionSpread > 0) {
                st2.setInt(this.schemaColumns.size(), this.random.nextInt(this.partitionSpread));
            }
            this.availableQueries.acquire();
            ResultSetFuture future = this.dbConn.getSession().executeAsync((Statement)st2);
            Futures.addCallback((ListenableFuture)future, this.asyncQueryHandler, (Executor)this.asyncQueryExecutor);
            if (this.totalRows % 10000L == 0L) {
                logger.info((Object)("Processed " + this.totalRows + " rows, appended=" + this.insertedRows.get() + " errors=" + this.failedRows.get()));
            }
        }

        public void lastRowEmitted() throws Exception {
            logger.info((Object)("Done processing " + this.totalRows + " rows, waiting for pending queries to complete"));
            this.availableQueries.acquire(this.maxQueries);
            this.asyncQueryExecutor.shutdown();
            this.asyncQueryExecutor.awaitTermination(Long.MAX_VALUE, TimeUnit.SECONDS);
            logger.info((Object)("Processed " + this.totalRows + " rows, appended=" + this.insertedRows.get() + " errors=" + this.failedRows.get()));
            if (this.failedRows.get() > 0L) {
                logger.warn((Object)("Errors were encountered on " + this.failedRows.get() + " rows"));
            }
            CassandraUtil.releaseConnection(this.dbConn);
        }

        public long writtenBytes() throws IOException {
            return 0L;
        }

        public void cancel() throws Exception {
            logger.info((Object)"Aborting");
            this.asyncQueryExecutor.shutdownNow();
            this.asyncQueryExecutor.awaitTermination(1L, TimeUnit.SECONDS);
            logger.warn((Object)("Aborted: processed " + this.totalRows + " rows, appended=" + this.insertedRows.get() + " errors=" + this.failedRows.get()));
            CassandraUtil.releaseConnection(this.dbConn);
        }
    }
}

