(function(){
'use strict';

const app = angular.module('dataiku.code-studios', [])
    .constant('readmeFileMeta', {zone: 'workspace_root', file: 'CodeStudio_README.md'});

app.controller("CodeStudiosCommonController", function($scope, $state, $rootScope, $stateParams, $q, $controller, TopNav, CodeStudiosService, LoggerProvider, WT1,
                                                    DataikuAPI, FutureWatcher, Dialogs, ActivityIndicator, TAIL_STATUS, DKUtils, StateUtils, PluginsService,
                                                    ActiveProjectKey, FutureProgressModal, CreateModalFromTemplate) {

    $scope.hooks = $scope.hooks || {};

    $scope.restart = function(codeStudioObjectId) {
        const deferred = $q.defer();
        DataikuAPI.codeStudioObjects.restart($stateParams.projectKey, codeStudioObjectId)
            .then(deferred.resolve)
            .catch((data, status, headers) => {
                setErrorInScope.bind($scope)(data, status, headers);
                deferred.reject({data: data, status: status, headers: headers});
            });
        return deferred.promise;
    };
    
    $scope.restartWaitAndShowError = function(codeStudioObjectId) {
        const deferred = $q.defer();
        $scope.restart(codeStudioObjectId)
            .then(function(data) {
                FutureProgressModal.show($scope, data.data, "Start Code Studio", undefined, 'static', 'false', true).then(function(result) {
                    if (result) {
                        Dialogs.infoMessagesDisplayOnly($scope, "Start", result.messages, result.futureLog, undefined, 'static', false).then(() => {
                            if (result.messages.maxSeverity == 'ERROR') {
                                deferred.reject("Failed to start");
                            } else {
                                deferred.resolve("ok");
                            }
                        });
                    } else {
                        deferred.resolve("no message"); // probably an abort
                    }
                }, () => deferred.reject("failed during start"));
            }, () => deferred.reject("failed at start"));
        return deferred.promise;
    };

    $scope.doStop = function(codeStudioObject){
        WT1.event("code_studio_action_stop", {type:codeStudioObject.templateId});
        const deferred = $q.defer();
        DataikuAPI.codeStudioObjects.stop($stateParams.projectKey, codeStudioObject.id)
            .then(deferred.resolve, deferred.reject)
            .catch(function(data, status, headers) {
                setErrorInScope.bind($scope)(data, status, headers);
                deferred.reject({data: data, status: status, headers: headers});
            });
        return deferred.promise;
    }

    function doForcePullEverything(codeStudioObject) {
        WT1.event("code_studio_action_syncstop", {type:codeStudioObject.templateId});
        let deferred = $q.defer();
        DataikuAPI.codeStudioObjects.getState($stateParams.projectKey, codeStudioObject.id).success(function(result) {
            let syncedZones = result.syncedZones;
            const promises = syncedZones.map(z => {
                return DataikuAPI.codeStudioObjects.pullBundle($stateParams.projectKey, codeStudioObject.id, z.id, true);
            });
            $q.all(promises)
                .then(responses => {
                    deferred.resolve(responses);
                }, error => {
                    ActivityIndicator.error("Synchronization failed");
                    deferred.reject(error);
                });
        });
        return deferred.promise;
    }

    $scope.checkFilesAndStop = function(codeStudioObject, connectedUsers){
        const deferred = $q.defer();
        let connectedUsersMessage;
        if (connectedUsers)
        {
            let subject;
            if (connectedUsers.length > 0) {
                subject = connectedUsers.every(u => u.login === $rootScope.appConfig.login)
                    ? 'You are'
                    : connectedUsers.length + ' user' + (connectedUsers.length>1 ? 's are' : ' is');
            } else {
                subject = 'Nobody';
            }
            connectedUsersMessage = subject + ' currently using the Code Studio "' + sanitize(codeStudioObject.name) + '".';
        }
        DataikuAPI.codeStudioObjects.getChanges($stateParams.projectKey, codeStudioObject.id)
            .then(response => {
                function hasChanges(name) {
                    return ((response.data[name] || {}).added || []).length + ((response.data[name] || {}).deleted || []).length + ((response.data[name] || {}).modified || []).length > 0;
                };
                if (Object.keys(response.data).some(hasChanges)) {
                    // some changes somewhere
                    CreateModalFromTemplate("/templates/code-studios/code-studio-confirm-stop-modal.html", $scope, null, function(newScope) {
                        newScope.setResult(codeStudioObject, response.data);
                        newScope.connectedUsersMessage = connectedUsersMessage;
                        newScope.codeStudioName = codeStudioObject.name;
                        newScope.syncCommand = () => doForcePullEverything(codeStudioObject);
                        newScope.stopCommand = () => $scope.doStop(codeStudioObject).then(deferred.resolve, deferred.reject);
                        newScope.cancelCommand = () => deferred.reject();
                    })
                } else if (connectedUsersMessage) {
                    Dialogs.confirmInfoMessages($scope, 'Confirm Code Studio Stop', {}, connectedUsersMessage, false)
                        .then(() => $scope.doStop(codeStudioObject).then(deferred.resolve, deferred.reject), deferred.reject);
                } else {
                    Dialogs.confirm($scope, 'Stop Code Studio', 'Are you sure you want to stop the Code Studio "' + codeStudioObject.name + '" ?')
                        .then(() => $scope.doStop(codeStudioObject).then(deferred.resolve, deferred.reject), deferred.reject);
                }
            })
            .catch(function(data, status, headers) {
                setErrorInScope.bind($scope)(data, status, headers);
                deferred.reject({data: data, status: status, headers: headers});
            });

        return deferred.promise;
    };

    $scope.checkUsersAndStop = function(codeStudioObject) {
        var deferred = $q.defer();
        DataikuAPI.codeStudioObjects.getCurrentUsage(codeStudioObject.projectKey, codeStudioObject.id)
            .then(response => {
                const users = response.data.users;
                $scope.checkFilesAndStop(codeStudioObject, users).then(deferred.resolve, deferred.reject);
            }, deferred.reject);
        return deferred.promise;
    };
    
    $scope.renameObjectAndSave = function(newName) {
        $scope.codeStudioObject.name = newName;
        return DataikuAPI.codeStudioObjects.save($scope.codeStudioObject);
    };
    
    $scope.showCreateCodeStudioWebappModal = function(codeStudioId, codeStudioName, codeStudioTemplateId) {
        WT1.event("code_studio_action_publish", { type: codeStudioTemplateId });
        DataikuAPI.codeStudioTemplates.getFullInfo(codeStudioTemplateId).success(function(data) {
            if (data.codeStudioTemplate.type === "block_based") {
                let availablePorts = data.availablePorts;
                if (availablePorts !== undefined && availablePorts.length === 0) {
                    Dialogs.error($scope, "Cannot expose webapp", "The Code Studio's template must define at least one entrypoint, and activate it for Webapps");
                } else {
                    DataikuAPI.codeStudioObjects.getRunInfo($stateParams.projectKey, codeStudioId).success(function(result) {
                        const syncZones = result.syncedZones.map(z => {
                            return {
                                zone: z,
                                name: z.id,
                                selected: true
                            };
                        });
                        CreateModalFromTemplate('/templates/code-studios/new-code-studio-webapp-modal.html', $scope, null, function(modalScope) {
                            modalScope.withSyncOption = syncZones.length > 0; //no sync zones if the runtime isn't running
                            modalScope.app.name = codeStudioName || ('Code Studio ' + codeStudioId);
                            modalScope.app.params.exposedPort = availablePorts[0];
                            modalScope.app.params.codeStudioId = codeStudioId;
                            modalScope.availablePorts = availablePorts;
                            modalScope.app.syncBeforePublish = modalScope.withSyncOption;
                            modalScope.syncCommand = () => $scope.syncFiles({
                                id: codeStudioId,
                                projectKey: $stateParams.projectKey,
                                templateId: data.codeStudioTemplate.id
                            }, syncZones);
                        }).then(function(webApp) {
                            $state.go("projects.project.webapps.webapp.view", {projectKey : $stateParams.projectKey, webAppId: webApp.id});
                        });
                    }).error(setErrorInScope.bind($scope));
                }
            } else {
                Dialogs.error($scope, "Cannot expose webapp", "This Code Studio's template type can't be publish as a Webapp");
            }
        }).error(setErrorInScope.bind($scope));
    };

    $scope.sendCommand = function(projectKey, codeStudioObjectId, commandType, commandParams, commandBehavior) {
        let deferred = $q.defer();
        DataikuAPI.codeStudioObjects.issueCommand(projectKey, codeStudioObjectId, commandType, commandParams, commandBehavior).success(function(result) {
            if (commandBehavior == 'WAIT_FOR_RETURN_VALUE' || commandBehavior == 'WAIT_FOR_RETURN_VALUE_AND_REFRESH_VIEW') {
                FutureProgressModal.show($scope, result, "Run action", undefined, 'static', 'false', true).then(data => {
                    if (data.hasResult && data.result == null) {
                        deferred.reject("aborted");
                        return; // that was an abort
                    }
                    if (commandBehavior == 'WAIT_FOR_RETURN_VALUE_AND_REFRESH_VIEW' && $scope.hooks.refreshViews) {
                        $scope.hooks.refreshViews();
                    }
                    CreateModalFromTemplate("/templates/code-studios/code-studio-command-result-modal.html", $scope, null, function(newScope) {
                        newScope.result = data;
                    });
                    deferred.resolve(data);
                }, () => {deferred.reject("failed");});
            } else if (commandBehavior == 'WAIT_FOR_RETURN_VALUE_IN_BACKGROUND') {
                if (result && result.jobId && !result.hasResult) {
                    FutureWatcher.watchJobId(result.jobId).success(deferred.resolve).error(deferred.reject);
                } else {
                    deferred.resolve(result);
                }
            } else {
                // we don't really care
                deferred.resolve("enqueued");
            }
        }).catch(function(data, status, headers) {
            setErrorInScope.bind($scope)(data, status, headers);
            deferred.reject({data: data, status: status, headers: headers});
        });
        return deferred.promise;
    };

    $scope.doPull = function(projectKey, codeStudioObjectId, selectedOptions) {
        let deferred = $q.defer();
        const promises = selectedOptions.map(opt => {
            return DataikuAPI.codeStudioObjects.pullBundle(projectKey, codeStudioObjectId, opt.name, true);
        });
        //if we are in the kubikle runtime page, we have to handle the UI state
        //that we don't have from the kubikles list page
        if ($scope.uiState) {
            delete $scope.uiState['pullSyncOK'];
            $scope.uiState.syncing = 'pull';
            $q.all(promises)
                .then(responses => {
                    $scope.uiState['pullSyncOK'] = true;
                    setTimeout(() => delete $scope.uiState['pullSyncOK'], 2000);
                    $scope.uiState['pullSyncLog'] = `Last sync ${(new Date()).toLocaleTimeString()}`;
                    deferred.resolve(responses);
                }, error => {
                    ActivityIndicator.error("Synchronization failed");
                    $scope.uiState['pullSyncOK'] = false;
                    $scope.uiState['pullSyncLog'] = `Last sync ${(new Date()).toLocaleTimeString()} failed: ${error.message}`;
                    deferred.reject(error);
                }).finally(() => { $scope.uiState.syncing = null; });
        } else {
            $q.all(promises).then(deferred.resolve, deferred.reject);
        }
        return deferred.promise;
    }

    $scope.doPush = function(projectKey, codeStudioObjectId, selectedOptions) {
        let deferred = $q.defer();
        const promises = selectedOptions.map(opt => {
            return $scope.sendCommand(projectKey, codeStudioObjectId, 'push_bundle_to_code_studio', {zone:opt.name}, "FIRE_AND_FORGET");
        });
        //if we are in the kubikle runtime page, we have to handle the UI state
        //that we don't have from the kubikles list page
        if ($scope.uiState) {
            delete $scope.uiState['pushSyncOK'];
            $scope.uiState.syncing = 'push';
            $q.all(promises)
                .then(responses => {
                    $scope.uiState['pushSyncOK'] = true;
                    setTimeout(() => delete $scope.uiState['pushSyncOK'], 2000);
                    $scope.uiState['pushSyncLog'] = `Last Code Studio reset ${(new Date()).toLocaleTimeString()}`;
                    deferred.resolve(responses);
                }, error => {
                    ActivityIndicator.error("Synchronization failed");
                    $scope.uiState['pushSyncOK'] = false;
                    $scope.uiState['pushSyncLog'] = `Last Code Studio reset ${(new Date()).toLocaleTimeString()} failed: ${error.message}`;
                    deferred.reject(error);
                }).finally(() => { $scope.uiState.syncing = null; });
        } else {
            $q.all(promises).then(deferred.resolve, deferred.reject);
        }
        return deferred.promise;
    }

    $scope.makeSyncEventPayload = function(codeStudioObject, selectedOptions){
        const eventPayload = { type: codeStudioObject.templateId };
        selectedOptions.forEach(o => {
            eventPayload[o.zone ? o.zone.zone : o.name.replace(/_[^_]+$/, '')] = true;
        });
        return eventPayload;
    }

    $scope.syncFiles = function(codeStudioObject, syncOptions) {
        const deferred = $q.defer();
        const selectedOptions = syncOptions.filter(opt => opt.selected);
        if (selectedOptions.length == 0) {
            // easy. nothing to do.
            deferred.resolve();
        } else {
            const eventPayload = $scope.makeSyncEventPayload(codeStudioObject, selectedOptions);
            WT1.event("code_studio_action_sync", eventPayload);
            const zones = selectedOptions.map(o => o.name);
            DataikuAPI.codeStudioObjects.checkConflicts(codeStudioObject.projectKey, codeStudioObject.id, zones).success(function(result) {
                function hasConflict(name) {
                    return ((result[name] || {}).added || []).length + ((result[name] || {}).deleted || []).length + ((result[name] || {}).modified || []).length + ((result[name] || {}).ignored || []).length > 0;
                }
                if (zones.some(hasConflict)) {
                    // some conflict somewhere
                    CreateModalFromTemplate("/templates/code-studios/code-studio-pull-files-conflicts-modal.html", $scope, null, function(newScope) {
                        newScope.setResult(result, selectedOptions);
                        newScope.eraseCommand = () => $scope.doPull(codeStudioObject.projectKey, codeStudioObject.id, selectedOptions).then(deferred.resolve, deferred.reject);
                        newScope.forgetCommand = () => $scope.doPush(codeStudioObject.projectKey, codeStudioObject.id, selectedOptions).then(deferred.resolve, deferred.reject);
                    });
                } else {
                    // no conflict, happy.
                    $scope.doPull(codeStudioObject.projectKey, codeStudioObject.id, selectedOptions).then(deferred.resolve, deferred.reject);
                }
            }).error(setErrorInScope.bind($scope));
        }
        return deferred.promise;
    };

    $scope.runOutdatedTemplate = function(state){
        return state && 'RUNNING' === state.state && state.lastTemplateBuilt > state.lastStateChange;
    };

});

app.controller("NewCodeStudioModalController", function($scope, $window, $state, $stateParams, $controller, DataikuAPI, WT1, FutureWatcher, SpinnerService,
                                                     PluginsService, CodeStudiosService, PluginConfigUtils, Dialogs) {
    $scope.newCodeStudio = {
        templateId: null,
        name: '',
        params: {}
    };
    
    $scope.uiState = {};
    
    WT1.event("code_studio_action_new");

    function refreshSelectedType() {
        if ($scope.codeStudioTypes) {
            $scope.selectedType = $scope.codeStudioTypes.filter(t => t.templateId == $scope.newCodeStudio.templateId)[0];
        }
    }
    function refreshFilteredTypes() {
        let query = ($scope.uiState.codeStudioTypeQuery || '').toLowerCase();
        function filterMatch(t) {
            if (t.templateId && t.templateId.toLowerCase().indexOf(query) >= 0) return true;
            if (t.label && t.label.toLowerCase().indexOf(query) >= 0) return true;
            if (t.shortDesc && t.shortDesc.toLowerCase().indexOf(query) >= 0) return true;
            return false;
        }
        $scope.filteredCodeStudioTypes = $scope.codeStudioTypes.filter(t => filterMatch(t));
    }

    $scope.codeStudioTypes = [];
    $scope.codeStudioTypeDescriptions = [];
    DataikuAPI.codeStudioObjects.listTypes().success(function(data){
        $scope.codeStudioTypes.push(...data);
        $scope.codeStudioTypeDescriptions.push(...(data.map(t => t.shortDesc)));
        if ($scope.codeStudioTypes && $scope.codeStudioTypes.length > 0) {
            $scope.newCodeStudio.templateId = $scope.newCodeStudio.templateId || $scope.codeStudioTypes[0].templateId;
        }
        refreshSelectedType();
        refreshFilteredTypes();
    }).error(setErrorInScope.bind($scope));

    $scope.searchFilter = '';
    
    $scope.$watch('newCodeStudio.templateId', refreshSelectedType);
    $scope.$watch('uiState.codeStudioTypeQuery', refreshFilteredTypes);

    $scope.create = function(){
        WT1.event("code_studio_action_create", { type: $scope.newCodeStudio.templateId });

        $scope.isCreating = true;
        DataikuAPI.codeStudioObjects.create($stateParams.projectKey, $scope.newCodeStudio.name, $scope.newCodeStudio.templateId, $scope.newCodeStudio.template, $scope.newCodeStudio.config)
        .success(function(result) {
            $scope.isCreating = false;
            $scope.newCodeStudio.id = result.codeStudio.id;
            if (result.messages.anyMessage) {
                Dialogs.infoMessagesDisplayOnly($scope, "Creation", result.messages, undefined, undefined, 'static', false).then(() => {
                    $scope.resolveModal($scope.newCodeStudio);
                });
            } else {
                $scope.resolveModal($scope.newCodeStudio);
            }
        }).error(function(data, status, headers, config, statusText, xhrStatus) {
            $scope.isCreating = false;
            setErrorInScope.bind($scope)(data, status, headers, config, statusText, xhrStatus);
        });
    };
});

app.controller("NewCodeStudioWebappModalController", function($scope, $window, $state, $stateParams, $controller, DataikuAPI, WT1, FutureWatcher, SpinnerService,
                                                           PluginsService, CodeStudiosService, PluginConfigUtils) {
    $scope.app = {
        name: '',
        params: {}
    };

    $scope.searchFilter = '';

    $scope.create = function () {
        const publish = () => {
            $scope.isCreating = true;
            DataikuAPI.webapps.create($stateParams.projectKey, $scope.app.name, 'CODE_STUDIO_AS_WEBAPP', {}, {}, $scope.app.params)
                .success(function (result) {
                    $scope.app.id = result.webAppId;
                    if (result.backendState && !result.backendState.hasResult) { // There is a backend still starting, wait for it
                        SpinnerService.lockOnPromise(FutureWatcher.watchJobId(result.backendState.jobId)
                            .success(function (data) {
                                $scope.app.backendReadyOrNoBackend = true;
                                $scope.resolveModal($scope.app);
                            }).error(function (data, status, headers) {
                                $scope.app.backendReadyOrNoBackend = false;
                                $scope.resolveModal($scope.app);
                            }));
                    } else {
                        // No backend, nothing to wait for => dubious in this case
                        $scope.resolveModal($scope.app);
                    }
                }).error(function (data, status, headers, config, statusText, xhrStatus) {
                    $scope.isCreating = false;
                    setErrorInScope.bind($scope)(data, status, headers, config, statusText, xhrStatus);
                });
        }
        if ($scope.app.syncBeforePublish) {
            $scope
                .syncCommand()
                .then(() => publish());
        } else {
            publish();
        }
    };
});


app.controller("CodeStudiosListController", function($scope, $controller, $stateParams, DataikuAPI, $state, $q, TopNav, Fn, $filter, Dialogs, WT1, StateUtils,
                                                  CodeStudiosService, readmeFileMeta) {
    
    $controller('_TaggableObjectsListPageCommon', {$scope: $scope});
    $controller("CodeStudiosCommonController", {$scope: $scope});

    $scope.listHeads = DataikuAPI.codeStudioObjects.listHeads;

    $scope.sortBy = [
        { value: 'name', label: 'Name' },
        { value: 'uiState.state', label: 'Status' },
        { value: '-uiState.lastStateChange', label: 'State change' }
    ];
    
    $scope.selection = $.extend({
        filterQuery: {
            userQuery: '',
            tags: [],
            interest: {
                starred: '',
            },
            uiState: {
                state: ''
            }
        },
        filterParams: {
            userQueryTargets: ["name","tags","uiState.state","uiState.runAsUser", "owner"]
        },
        orderQuery: "name",
        orderReversed: false,
        filteredSelectedObjects: []
    }, $scope.selection || {});

    $scope.toggleRunningFilter = function() {
        $scope.selection.filterQuery.uiState.state = $scope.selection.filterQuery.uiState.state ? '' : 'RUNNING';
    }

    $scope.sortCookieKey = 'code-studios';
    $scope.maxItems = 20;

    TopNav.setLocation(TopNav.TOP_NOTEBOOKS, 'code-studios', TopNav.TABS_NONE, null);
    TopNav.setNoItem();

    $scope.runningCount = 0;
    $scope.listHeadHook = {
        list: items => {
            $scope.runningCount = items.filter(i => i.uiState.state === 'RUNNING').length;
            const needRefresh = items.some(i => ['STARTING', 'STOPPING'].indexOf(i.uiState.state) >= 0);
            if (needRefresh) {
                $scope.refreshTimer = setTimeout(() => $scope.list(), 5000);
            }
        }
    };
    $scope.list();

    $scope.newCodeStudio = function() {
        $scope.showCreateCodeStudioModal(null, null, null, "");
    };
    
    $scope.duplicateCodeStudio = function(codeStudio) {
        CodeStudiosService.duplicate($scope, codeStudio);
    };

    $scope.changeOwner = function(codeStudio) {
        CodeStudiosService.changeOwner($scope, codeStudio);
    };

    $scope.stopCodeStudio = function(codeStudioObject){
        $scope.checkUsersAndStop(codeStudioObject).then($scope.list, $scope.list);
    };

    $scope.startCodeStudio = function(codeStudioObject) {
        WT1.event("code_studio_action_start", {type:codeStudioObject.templateId});
        const openCodeStudioParams =  {
            projectKey: $stateParams.projectKey,
            codeStudioObjectId: codeStudioObject.id,
            codeStudioName: codeStudioObject.name
        }
        if (codeStudioObject.uiState.lastStateChange === 0) {
            Object.assign(openCodeStudioParams, readmeFileMeta)
        }
        const openAfterStart = () => {
            $state.go(
                "projects.project.code-studios.code-studio.view",
                openCodeStudioParams
            );
        };
        $scope.restartWaitAndShowError(codeStudioObject.id).then(openAfterStart, $scope.list);
    };

    $scope.restartCodeStudio = function(codeStudioObject){
        $scope.checkUsersAndStop(codeStudioObject).then(() => $scope.startCodeStudio(codeStudioObject), $scope.list);
    };

    $scope.canMassStartCodeStudios = function(codeStudios){
        return codeStudios.some(k => k.uiState.state !== 'RUNNING');
    };

    $scope.canMassStopCodeStudios = function(codeStudios){
        return codeStudios.some(k => k.uiState.state === 'RUNNING');
    };

    $scope.startAllCodeStudios = function(codeStudios){
        WT1.event("code_studio_action_startall", {types:codeStudios.map(k => k.templateId)});
        const stoppedCodeStudios = codeStudios.filter(k => k.uiState.state !== 'RUNNING');
        if (stoppedCodeStudios.length > 0) {
            Promise
                .all(stoppedCodeStudios.map(async k => $scope.restart(k.id)))
                .then(() => refreshList(), () => refreshList());
        }
    };

    $scope.stopAllCodeStudios = function(codeStudios){
        WT1.event("code_studio_action_stopall", {types:codeStudios.map(k => k.templateId)});
        const runningCodeStudios = codeStudios.filter(k => k.uiState.state === 'RUNNING');
        if (runningCodeStudios.length > 1) {
            Dialogs.confirmInfoMessages($scope, 'Confirm ' + runningCodeStudios.length +' Code Studios Stop', {}, "You will disconnect current users of these Code Studios (if any), and any un-synchronised file will be lost.", false)
                .then(() => {
                    Promise
                        .all(runningCodeStudios.map(async k => $scope.doStop(k)))
                        .then(() => refreshList(), () => refreshList());
                });
        } else if (runningCodeStudios.length === 1) {
            $scope.stopCodeStudio(runningCodeStudios[0]);
        }
    };

    function refreshList(attemptsLeft = 8) {
        return $scope.list().then(() => {
            if (!$scope.listItems.some(cs => cs.uiState.state === 'STARTING' || cs.uiState.state === 'STOPPING')) {
                return new Promise(resolve => setTimeout(resolve, 200))
                    .then(() => refreshList(attemptsLeft -1));
            }
        }) 
    }

    $scope.gotoLogs = function(codeStudioObjectId, tab){
        $state.go('projects.project.code-studios.code-studio.logs', {
            codeStudioObjectId: codeStudioObjectId,
            tab: tab
        });
    };

});


app.controller("CodeStudioCoreController", function($scope, $rootScope, $stateParams, $filter, $state, $controller, $timeout, $q, FutureWatcher, WT1,
                                                 DKUtils, DataikuAPI, TopNav, CodeStudiosService, Assert, $interval, FutureProgressModal, Dialogs,
                                                 CreateModalFromTemplate, ActivityIndicator, APIXHRService, readmeFileMeta) {

    $controller("CodeStudiosCommonController", {$scope: $scope});
    
    TopNav.setItem(TopNav.ITEM_CODE_STUDIO, $stateParams.codeStudioObjectId);
    
    $scope.appConfig = $rootScope.appConfig;
    $scope.$rootScope = $rootScope;
    
    $scope.uiState = {actions:[], pulls: [], pushs: []};

    $scope.getWorkspaceIfied = function(location) {
        const workspace = '/home/dataiku/workspace';
        if ((location||'').startsWith(workspace)) {
            return '<workspace>' + location.substring(workspace.length);
        } else {
            return location;
        }
    };

    $scope.syncOptions = [];
    $scope.refreshSyncOptions = function(syncZones) {
        let oldSelected = $scope.syncOptions.filter(o => o.selected).map(o => o.name);
        $scope.syncOptions = (syncZones || []).map(z => {
            let o = {};
             if (z.zone == 'recipes') {
                o.label = () => "Recipes";
            } else {
                o.label = () => "Files in " + $scope.getWorkspaceIfied(z.pathInContainer);
            }
            o.zone = z;
            o.name = z.id;
            o.selected = oldSelected[o.name];
            if (o.selected == undefined) o.selected = true;
            return o;
        });
    };

    $scope.isAny = (list, k) => list.find(item => {
        const value = item[k];
        return typeof(value) === 'function'
            ? value()
            : value;
    });

    let wt1EventsToSendWhenSummaryIsAvailable = [];

    function getSummary() {
        return DataikuAPI.codeStudioObjects.getSummary($stateParams.projectKey, $stateParams.codeStudioObjectId).success(function(data) {
            $scope.codeStudioObject = data.object;
            $scope.templateWasDeleted = data.templateWasDeleted;
            $scope.codeStudioDesc = data.desc;
            $scope.templateType = data.templateType;
            $scope.timeline = data.timeline;
            $scope.interest = data.interest;
            $scope.hooks.origCodeStudio = angular.copy($scope.codeStudio);
            
            let itemForTopNav = angular.copy($scope.codeStudioObject);
            itemForTopNav.icon = (data.desc || {}).icon;

            TopNav.setItem(TopNav.ITEM_CODE_STUDIO, $stateParams.codeStudioObjectId, itemForTopNav);
            TopNav.setPageTitle($scope.codeStudioObject.name + " - Code Studio");
            
            wt1EventsToSendWhenSummaryIsAvailable.forEach(e => WT1.event(e.eventType, e.eventPayloadFunc($scope.codeStudioObject)));
            wt1EventsToSendWhenSummaryIsAvailable = [];
        }).error(setErrorInScope.bind($scope));
    }

    const getSummaryResult = getSummary();
    
    $scope.sendEventWhenSummaryIsAvailable = function(eventType, eventPayloadFunc) {
        if ($scope.codeStudioObject) {
            WT1.event(eventType, eventPayloadFunc($scope.codeStudioObject));
        } else {
            wt1EventsToSendWhenSummaryIsAvailable.push({eventType:eventType, eventPayloadFunc:eventPayloadFunc});
        }
    };

    $scope.isDirty = function() {
        return !angular.equals($scope.codeStudio, $scope.hooks.origCodeStudio);
    };

    $scope.saveCodeStudio = function(commitMessage, forceRestartBackend) {
        WT1.event("code_studio_save", {type: $scope.codeStudioObject.templateId});
        var deferred = $q.defer();

        DataikuAPI.codeStudioObjects.save($scope.codeStudioObject, commitMessage).success(function(result) {
            $scope.hooks.origCodeStudio = angular.copy($scope.codeStudioObject);
            deferred.resolve($scope.codeStudioObject);
        }).error(function(...args) {
            setErrorInScope.bind($scope)(...args);
            deferred.reject("failed");
        });

        return deferred.promise;
    };

    $scope.getBackendLogURL = function(app) {
        return DataikuAPI.codeStudioObjects.getBackendLogURL(app.projectKey, app.id);
    };

    $scope.duplicateCodeStudio = function() {
        $scope.saveCodeStudio().then(function() {
            CodeStudiosService.duplicate($scope, $scope.codeStudioObject);
        });
    };

    $scope.changeOwner = function() {
        $scope.saveCodeStudio().then(function() {
            CodeStudiosService.changeOwner($scope, $scope.codeStudioObject);
        });
    };
    
    $scope.sendCustomAction = function(action) {
        WT1.event("code_studio_action_customcommand", {type:$scope.codeStudioObject.templateId, command:action.name});
        $scope.sendCommand($stateParams.projectKey, $stateParams.codeStudioObjectId, action.name, {}, action.behavior);
    };

    $scope.startCodeStudio = async function() {
        if (!$scope.codeStudioObject) {
            await getSummaryResult;
        }
        WT1.event("code_studio_action_start", {type:$scope.codeStudioObject.templateId});
        $scope.codeStudioState.state = 'STARTING';
        let startCallback = $scope.refreshState;
        if ((!$stateParams.file || !$stateParams.zone) && $scope.codeStudioState.lastStateChange === 0) {
            startCallback = function() {
                $state.go(
                    'projects.project.code-studios.code-studio.view',
                    {projectKey: $stateParams.projectKey, codeStudioObjectId: $stateParams.codeStudioObjectId, ...readmeFileMeta}
                ).then($scope.refreshState);
            }
        }
        $scope.restartWaitAndShowError($scope.codeStudioObject.id).then(startCallback, $scope.refreshState);
    };

    $scope.stopCodeStudio = function() {
        $scope
            .checkFilesAndStop($scope.codeStudioObject)
            .then(() => $scope.refreshState())
            .catch(() => {}); // user cancelled the modal
    };

    $scope.restartCodeStudio = function(){
        $scope
            .checkFilesAndStop($scope.codeStudioObject)
            .then(() => {
                $scope.refreshState();
                $scope.startCodeStudio();
            })
            .catch(() => {}); // user cancelled the modal
    };

    $scope.doSyncFiles = function(){
        $scope.syncFiles($scope.codeStudioObject, $scope.syncOptions);
    };

    $scope.resetFiles = function(event) {
        event.preventDefault();
        event.stopPropagation();
        const selectedOptions = $scope.syncOptions.filter(opt => opt.selected);
        const eventPayload = $scope.makeSyncEventPayload($scope.codeStudioObject, selectedOptions);
        WT1.event("code_studio_action_reset", eventPayload);
        if (selectedOptions.length == 0) return; // easy. nothing to do.
        $scope.doPush($scope.codeStudioObject.projectKey, $scope.codeStudioObject.id, selectedOptions);
    };

    let cancelRefresh = null;
    const stopAutoRefresh = function() {
        if (cancelRefresh) {
            $timeout.cancel(cancelRefresh);
            cancelRefresh = null;
        }
    };

    const prepareForEdit = async function() {
        if ($scope.zoneToPrepareForEdit && $scope.codeStudioState) {
            if ($scope.codeStudioState.state === 'RUNNING') {
                $scope.refreshSyncOptions($scope.codeStudioState.syncedZones);
                if (!$scope.codeStudioObject) {
                    await getSummaryResult;
                }
                const options = $scope.syncOptions.filter(option => option.zone.zone === $scope.zoneToPrepareForEdit)
                delete $scope.zoneToPrepareForEdit;
                await $scope.syncFiles($scope.codeStudioObject, options);
            }
        }
    }

    $scope.setZoneToPrepareForEdit = function(zoneToPrepare) {
        $scope.zoneToPrepareForEdit = zoneToPrepare;
    }

    $scope.refreshState = function() {
        DataikuAPI.codeStudioObjects.getState($stateParams.projectKey, $stateParams.codeStudioObjectId).success(function(result) {
            $scope.codeStudioState = result;
            $scope.uiState.pushOptions = [];
            $scope.uiState.actions = result.customCommands;
            $scope.codeStudioState._exposed = result.exposed;
            $scope.codeStudioState.exposed = result.exposed.filter(e => e.exposeHtml);
            $scope.uiState.activeTab = $scope.uiState.activeTab || ($scope.codeStudioState.exposed[0] || {}).label;
            prepareForEdit();
            if ($scope.codeStudioState.state == 'STOPPED') {
                // not running and state isn't going to change magically, so we don't need to refresh
                return;
            }
            $scope.refreshSyncOptions($scope.codeStudioState.syncedZones);
            let needRefresh = false;
            needRefresh |= $scope.codeStudioState.state != 'RUNNING'; // because if it's STOPPING or STARTING then we're in a transient state
            if ($scope.codeStudioState.exposed == null || $scope.codeStudioState.exposed.length == 0) { // because the port expositions just haven't started yet
                // check if we actually expect something to expose some html
                needRefresh |= $scope.codeStudioState.exposableCount > $scope.codeStudioState._exposed.length; // all the current exposed have exposeHtml == false, but there are some more
            } else {
                needRefresh |= !$scope.codeStudioState.exposed.every(e => e.url);
            }
            if (needRefresh) {
                // do more refreshing
                stopAutoRefresh();
                cancelRefresh = $timeout($scope.refreshState, 2000);
            }
        }).error(setErrorInScope.bind($scope));
        $scope.$broadcast("refreshState"); // so that sub-controllers can also refresh their things
    };
    $scope.refreshState();

    $scope.refreshExposedView = function(exposed, exposedIndex, doNotRefreshIfUnchanged) {
        if ($scope.codeStudioState.exposed[exposedIndex].url !== exposed.url || !doNotRefreshIfUnchanged) {
            const newUrl =  new URL(exposed.url, window.location.origin);
            newUrl.searchParams.set('r', Date.now());
            const newUrlPath = newUrl.pathname + newUrl.search;
            if (doNotRefreshIfUnchanged) {
                APIXHRService("GET", exposed.url)
                    .success(() => $scope.codeStudioState.exposed[exposedIndex].url = newUrlPath)
                    .error(() => {
                        if (_.get($scope, "$rootScope.$state.params.file", 0) !== 0 && _.get($scope, "$rootScope.$state.params.zone", 0) !== 0) {
                            ActivityIndicator.warning("Failed to open file '" + $scope.$rootScope.$state.params.file + "'");
                        }
                    });
            } else {
                DataikuAPI.codeStudioObjects.tryRestartServer($stateParams.projectKey, $stateParams.codeStudioObjectId, exposed.label).error(
                    setErrorInScope.bind($scope)
                ).finally(function() {
                    $scope.codeStudioState.exposed[exposedIndex].url = newUrlPath;}
                );
            }
        }
    };

    $scope.refreshCurrentView = function(event){
        WT1.event("code_studio_action_refresh", {type: $scope.codeStudioObject.templateId});
        if (event.metaKey || event.ctrlKey) {
            //we refresh state + all views
            $scope.refreshState();
        } else if ($scope.uiState.activeTab) {
            //we refresh only current view
            const idx = $scope.codeStudioState.exposed.findIndex(e => e.label === $scope.uiState.activeTab);
            if (idx !== -1) $scope.refreshExposedView($scope.codeStudioState.exposed[idx], idx);
        }
    };

    $scope.openStandaloneView = function($event){
        if ($scope.uiState.activeTab) {
            const tab = $scope.codeStudioState.exposed.find(e => e.label === $scope.uiState.activeTab);
            if (tab) {
                const url = `/dip/api/code-studio-objects/standalone-view/${$stateParams.projectKey}/${$stateParams.codeStudioObjectId}/${tab.exposedPort}/`;
                window.open(url, '_blank');
            }
        }
    };

    $scope.$on("$destroy", function() {
        stopAutoRefresh();
    });
});

app.controller("CodeStudioViewController", function($scope, $rootScope, $stateParams, $filter, $state, $controller, $timeout, $q, DKUtils, DataikuAPI, TopNav, CodeStudiosService, Assert, $interval) {

    TopNav.setLocation(TopNav.TOP_NOTEBOOKS, "code-studios", TopNav.TABS_CODE_STUDIO, "view");
    
    $scope.sendEventWhenSummaryIsAvailable("code_studio_view_viewed", k => {return {type: k.templateId}});
    
    $scope.canStillExpectExposedHtml = function() {
        if ($scope.codeStudioState == null) return true; // no state fetched yet
        if ($scope.codeStudioState.exposed == null || $scope.codeStudioState.exposed.length == 0) { // the port expositions just haven't started yet
            // check if we actually expect something to expose some html
            return $scope.codeStudioState.exposableCount > $scope.codeStudioState._exposed.length; // all the current exposed have exposeHtml == false, but there are some more
        } else {
            // already some html exposed, no need to wait for more
            return false;
        }
    };

    let cancelRefreshState = null;

    function checkState(){
        DataikuAPI.codeStudioObjects.getState($stateParams.projectKey, $stateParams.codeStudioObjectId).success(function(result) {
            if( result.state != $scope.codeStudioState.state ) {
                $scope.codeStudioState.state = result.state;
            }
            if( result.lastTemplateBuilt != $scope.codeStudioState.lastTemplateBuilt ) {
                $scope.codeStudioState.lastTemplateBuilt = result.lastTemplateBuilt;
            }
            if (!$scope.runOutdatedTemplate(result)) scheduleCheckState();
        });
    }

    function scheduleCheckState() {
        cancelRefreshState = $timeout(() => {
            cancelRefreshState = null;
            checkState();
        }, 60 * 1000); //check every 60s
    }
    scheduleCheckState();

    $scope.$on("$destroy", function() {
        if (cancelRefreshState) $timeout.cancel(cancelRefreshState);
    });

    $scope.refreshViewWithFile = function() {
        DataikuAPI.codeStudioObjects.openFile($stateParams.projectKey, $stateParams.codeStudioObjectId, $stateParams.zone, $stateParams.file).success(function(result) {
            result.exposed.filter(e => e.exposeHtml).forEach($scope.refreshExposedViewWithFile);
        }).error(setErrorInScope.bind($scope));
    }

    $scope.refreshExposedViewWithFile = function(exposed, exposedIndex) {
        $scope.refreshExposedView(exposed, exposedIndex, true);
    }

    $scope.onIframeLoaded = function(element) {
        if ($scope.shouldLoadfile && $stateParams.zone && $stateParams.file) {
            $scope.refreshViewWithFile();
            $scope.shouldLoadfile = false;
        }
    }

    $scope.doPrepareForEdit = function() {
        if ($stateParams.zone && $stateParams.file) {
            $scope.shouldLoadfile = true;
            $scope.setZoneToPrepareForEdit($stateParams.zone);
            DataikuAPI.codeStudioObjects.getState($stateParams.projectKey, $stateParams.codeStudioObjectId).success(function(result) {
                if (result.state === 'STOPPED') {
                    $scope.startCodeStudio();
                }
            }).error(setErrorInScope.bind($scope));
        }
    }

    $scope.doPrepareForEdit();
});

app.controller("PullCodeStudioConflictsModalController", function($scope, $window, $state, $stateParams, $controller, DataikuAPI, WT1, FutureWatcher, SpinnerService, PluginsService, CodeStudiosService, PluginConfigUtils, ActivityIndicator) {
    $scope.setResult = function(result, zones) {
        $scope.zones = zones;
        function mergeLists(name, result) {
            let conflicts = result[name] || {};
            return (conflicts.added || []).concat(conflicts.modified || []).concat(conflicts.deleted || []).sort();
        }
        function getAuthors(name, result) {
            let conflicts = (result[name] || {});
            return conflicts.authors || [];
        }
        $scope.files = {};
        $scope.authors = {};
        $scope.ignored = {};
        let eventPayload = { type: $scope.codeStudioObject.templateId };
        $scope.hasConflicts = false;
        $scope.hasIgnoredFiles = false;
        $scope.zones.forEach(z => {
            $scope.files[z.name] = mergeLists(z.name, result);
            $scope.hasConflicts |= $scope.files[z.name].length > 0;
            $scope.authors[z.name] = getAuthors(z.name, result);
            $scope.ignored[z.name] = (result[z.name] || {}).ignored || [];
            $scope.hasIgnoredFiles |= $scope.ignored[z.name].length > 0;
            eventPayload[z.zone.zone + '_files'] = $scope.files[z.name].length;
        });
        WT1.event("code_studio_action_conflicts", eventPayload);
    };
    
    function syncDone(result) {
        ActivityIndicator.success("Synchronization successful");
        $scope.dismiss();
    }
    $scope.cancel = function() {
        WT1.event("code_studio_action_conflicts_cancel");
        $scope.dismiss();
    };
    $scope.erase = function() {
        WT1.event("code_studio_action_conflicts_erase");
        let c = $scope.eraseCommand();
        if (c.success != null) {
            // single http promise
            c.success(syncDone).error(setErrorInScope.bind($scope));
        } else {
            // multiple promises
            c.then(syncDone, () => { /* Error, don't close */ });
        }
    };
    $scope.forget = function() {
        WT1.event("code_studio_action_conflicts_forget");
        let c = $scope.forgetCommand();
        if (c.success != null) {
            // single http promise
            c.success(syncDone).error(setErrorInScope.bind($scope));
        } else {
            // multiple promises
            c.then(syncDone, () => { /* Error, don't close */ });
        }
    };
});

app.controller("StopCodeStudioCheckChangesModalController", function($scope, $window, $state, $stateParams, $controller, DataikuAPI, WT1, FutureWatcher, SpinnerService, PluginsService, CodeStudiosService, PluginConfigUtils, ActivityIndicator) {
    $scope.setResult = function(codeStudioObject, zones) {
        $scope.zonePaths = Object.keys(zones)
        $scope.zones = zones;
        $scope.isStopping = false;
        // Remove added or modified folders which themselves have modified files (so if we notify /foo/bar/baz, we don't notify /foo/bar/)
        function mergeLists(name, zones) {
            const changes = zones[name] || {};
            const results = (changes.added || []).concat(changes.modified || []).concat(changes.deleted || []).concat(changes.ignored || [])
                .sort()
                .reverse()
                .reduce((deduped, change) => {
                    if (!deduped.length || !deduped[deduped.length-1].startsWith(change + "/"))
                        deduped.push(change);
                    return deduped;
                }, [])
                .sort();
            return results;

        };
        $scope.files = {};
        let eventPayload = { type: codeStudioObject.templateId };
        $scope.zonePaths.forEach(z => {
            $scope.files[z] = mergeLists(z, zones);
            eventPayload[$scope.zonePaths + '_files'] = $scope.files[z].length;
        });
        WT1.event("code_studio_action_changes", eventPayload);
    };
    function stopDone(result, message) {
        ActivityIndicator.success(message);
        $scope.dismiss();
    };
    $scope.cancel = function() {
        WT1.event("code_studio_action_conflicts_cancel");
        $scope.cancelCommand();
        $scope.dismiss();
    };
    $scope.sync = function() {
        $scope.isStopping = true;
        WT1.event("code_studio_action_conflicts_erase");
        $scope
            .syncCommand()
            .then(() => {
                $scope
                    .stopCommand()
                    .then(result => stopDone(result, "Synchronization successful"))
                    .catch(() => $scope.isStopping = false);
            }).catch(() => $scope.isStopping = false);
    };
    $scope.forget = function() {
        $scope.isStopping = true;
        WT1.event("code_studio_action_conflicts_forget");
        $scope
            .stopCommand()
            .then(result => stopDone(result, "Discard successful"))
            .catch(() => $scope.isStopping = false);
    };
});


app.controller("CodeStudioLogsController", function($scope, $rootScope, $stateParams, $filter, $state, $controller, $timeout, $q, FutureWatcher, WT1, DKUtils,
                                                    DataikuAPI, TopNav, CodeStudiosService, Assert, $interval, CreateModalFromTemplate, FutureProgressModal, 
                                                    Logger, ActivityIndicator) {

    TopNav.setLocation(TopNav.TOP_NOTEBOOKS, "code-studios", TopNav.TABS_CODE_STUDIO, "logs");
    
    $scope.sendEventWhenSummaryIsAvailable("code_studio_logs_viewed", k => {return {type: k.templateId}});
    
    $scope.uiState.active = $stateParams.tab || $scope.uiState.active || 'main_log';
    $scope.uiState.files = {path:'.', exists:false, children:[], preview:{}};
    
    $scope.startCodeStudio = function() {
        WT1.event("code_studio_action_start", {type:$scope.codeStudioObject.templateId});
        $scope.restartWaitAndShowError($scope.codeStudioObject.id).then($scope.refreshRunInfo, $scope.refreshRunInfo);
    };
    
    $scope.getLogURL = function() {
        return DataikuAPI.codeStudioObjects.getLogURL($stateParams.projectKey, $stateParams.codeStudioObjectId);
    };
    
    $scope.refreshRunInfo = function() {
        DataikuAPI.codeStudioObjects.getRunInfo($stateParams.projectKey, $stateParams.codeStudioObjectId).success(function(result) {
            $scope.codeStudioRunInfo = result;
            $scope.uiState.actions = result.customCommands;
            $scope.refreshSyncOptions($scope.codeStudioRunInfo.syncedZones);
        }).error(setErrorInScope.bind($scope));
    };
    $scope.refreshRunInfo();
    
    $scope.$on("refreshState", $scope.refreshRunInfo);

    $scope.refreshLogs = function() {
        if (!$scope.codeStudioRunInfo.jobId) return;
        WT1.event("code_studio_action_refresh", {type: $scope.codeStudioObject.templateId});
        DataikuAPI.futures.getUpdate($scope.codeStudioRunInfo.jobId).success(function(result) {
            $scope.codeStudioRunInfo.logTail = result.log;
        }).error(function(data, status, headers, config, statusText, xhrStatus) {
            // check if it's not a "this CodeStudio has been restarted" issue
            if ((data.message || '').indexOf("JobID not found :") >= 0) {
                Logger.info("Code Studio appears to have been restarted");
                $scope.refreshRunInfo();
            } else {
                setErrorInScope.bind($scope)(data, status, headers, config, statusText, xhrStatus);
            }
        });
    };

    $scope.downloadDiagnosis = function() {
        ActivityIndicator.success("Preparing Code Studio diagnosis ...");
        downloadURL(DataikuAPI.codeStudioObjects.getDiagnosisURL($stateParams.projectKey, $stateParams.codeStudioObjectId));
    };

    $scope.setLogTab = function(tabname) {
        $scope.uiState.active = tabname;
        if (tabname === 'files') {
            $scope.$broadcast('repaintFatTable');
        }
    };

    $scope.filesOrder = 'name';
    $scope.sortFiles = function(data){
        //we sort recursively files by $scope.filesOrder
        //directories first, then files
        if (data && data.children) {
            const sortFn = (a,b) => a[$scope.filesOrder].toLocaleLowerCase().localeCompare(b[$scope.filesOrder].toLocaleLowerCase());
            let files = [], dirs = [];
            data.children.forEach(f => {
                if (f.isDirectory)
                    dirs.push($scope.sortFiles(f));
                else
                    files.push(f);
            });
            files.sort(sortFn);
            dirs.sort(sortFn);
            data.children = dirs.concat(files);
        }
        return data;
    };

    // inspection of files in the container
    $scope.refreshFileList = function() {
        $scope.uiState.files.error = null;
        if (!$scope.uiState.files.path || $scope.codeStudioRunInfo == null || ['STARTING', 'RUNNING', 'STOPPING'].indexOf($scope.codeStudioRunInfo.state) < 0) {
            // not running
            $scope.uiState.files.exists = false;
            $scope.uiState.files.children = [];
        } else {
            $scope.isRefreshingFileList = true;
            $scope.isLongRefreshing = false;
            $timeout(function() {$scope.isLongRefreshing=true;}, 1000);
            DataikuAPI.codeStudioObjects.listContainerFiles($stateParams.projectKey, $stateParams.codeStudioObjectId, $scope.uiState.files.path).success(function(result) {
                $scope.uiState.files = $scope.sortFiles(result);
                $scope.$broadcast('repaintFatTable');
            }).error(function(data, status, headers) {
                $scope.uiState.files.error = getErrorDetails(data, status, headers)
            }).finally(function() {
                $scope.isRefreshingFileList = false;
            });
        }
    };
    $scope.$watch('codeStudioRunInfo.state', $scope.refreshFileList);
    $scope.$watch('uiState.files.path', $scope.refreshFileList);

    $scope.changeFilePath = function(path) {
        $scope.uiState.files.path = path;
    };
    
    $scope.clickOnFileItem = function(item) {
        let itemPath = $scope.uiState.files.path;
        if (!itemPath.endsWith('/')) {
            itemPath = itemPath + '/';
        }
        itemPath = itemPath + item.name;
        if (item.isDirectory) {
            $scope.uiState.files.path = itemPath; // the $watch will do the rest
        } else if (item.isLink) {
            // TODO resolve the link in the agent
        } else {
            $scope.uiState.files.preview = null;
            DataikuAPI.codeStudioObjects.previewContainerFile($stateParams.projectKey, $stateParams.codeStudioObjectId, itemPath).success(function(result) {
                $scope.uiState.files.preview = result;
            }).error(setErrorInScope.bind($scope));
        }
    };
});

app.component('filePathSelector', {
    bindings: {
        path: '=',
        goto: '&'
    },
    templateUrl: ' /templates/code-studios/file-path-selector.html',
    controller: ['$scope', '$timeout',
        function ($scope, $timeout) {
            const $ctrl = this;
            $ctrl.chunks = [];
            let buildChunksFromPath = function() {
                if ($ctrl.path == null) return;
                let path = $ctrl.path;
                let chunks = path.split('/');
                if (chunks.length >= 2) {
                    if (chunks[chunks.length - 1] == '') {
                        chunks.splice(chunks.length - 1, 1);
                    }
                } else {
                    chunks = ['']; // at root
                }
                $ctrl.chunks = chunks.map((x, i) => { return {name:x, path:chunks.slice(0, i+1).join('/') || '/'}; });
            };
            $scope.$watch('$ctrl.path', buildChunksFromPath);
            
            $ctrl.editedPath = '';
            $ctrl.editing = false;
            $ctrl.startEdit = function($event) {
                $ctrl.editedPath = $ctrl.path;
                $ctrl.editing = true;
                $timeout(function() {
                    $('input.code-studio__file-path-edit-field').first().focus();
                });
            };
            $ctrl.cancelEdit = function($event) {
                $ctrl.editing = false;
            };
            $ctrl.commitEdit = function($event) {
                $ctrl.path = $ctrl.editedPath;
                $ctrl.editing = false;
            };
        }
    ]
});


app.controller("CodeStudioPageRightColumnActions", async function($controller, $scope, $q, $rootScope, $stateParams, ActiveProjectKey, DataikuAPI, WT1) {

    $controller('_TaggableObjectPageRightColumnActions', {$scope: $scope});
    $controller("CodeStudioCoreController", {$scope: $scope});

    $scope.codeStudioFullInfo = (await DataikuAPI.codeStudioObjects.getFullInfo(ActiveProjectKey.get(), $stateParams.codeStudioObjectId)).data;
    $scope.codeStudioObject = $scope.codeStudioFullInfo.codeStudioObject;
    $scope.codeStudioObject.nodeType = 'CODE_STUDIO';
    $scope.codeStudioObject.interest = $scope.codeStudioFullInfo.interest;


    $scope.selection = {
        selectedObject : $scope.codeStudioObject,
        confirmedItem : $scope.codeStudioObject
    };

    $scope.updateUserInterests = function() {
        DataikuAPI.interests.getForObject($rootScope.appConfig.login, "CODE_STUDIO", ActiveProjectKey.get(), $scope.codeStudioObject.id)
            .success(function(data){
                $scope.selection.selectedObject.interest = data;
            })
            .error(setErrorInScope.bind($scope));
    }

    const interestsListener = $rootScope.$on('userInterestsUpdated', $scope.updateUserInterests);

    $scope.$on("$destroy", function() {
        interestsListener();
    });
});

app.directive('codeStudioRightColumnSummary', function(DataikuAPI, $stateParams, GlobalProjectActions, QuickView, $controller, ActivityIndicator, $rootScope){
    return {
        templateUrl :' /templates/code-studios/right-column-summary.html',
        link : function($scope, element, attrs) {
            $controller("CodeStudiosCommonController", {$scope: $scope});
            $controller('_TaggableObjectsMassActions', {$scope: $scope});
            $controller('_TaggableObjectsCapabilities', {$scope: $scope});

            $scope.QuickView = QuickView;

            /* Auto save when summary is modified */
            $scope.$on("objectSummaryEdited", function(){
                DataikuAPI.codeStudioObjects.saveMetadata($scope.codeStudioObject).success(function(data) {
                    ActivityIndicator.success("Saved");
                }).error(setErrorInScope.bind($scope));
            });

            $scope.refreshData = function() {
                $scope.codeStudioFullInfo = { codeStudioObject: $scope.selection.selectedObject }; // temporary incomplete data
                DataikuAPI.codeStudioObjects.getFullInfo($scope.selection.selectedObject.projectKey, $scope.selection.selectedObject.id).success(function(data) {
                    if (!$scope.selection.selectedObject
                        || $scope.selection.selectedObject.id != data.codeStudioObject.id
                        || $scope.selection.selectedObject.projectKey != data.codeStudioObject.projectKey) {
                        return; //too late!
                    }
                    DataikuAPI.codeStudioObjects.getCurrentUsage($scope.selection.selectedObject.projectKey, $scope.selection.selectedObject.id).success(function(usage) {
                        if (!$scope.selection.selectedObject
                            || $scope.selection.selectedObject.id != data.codeStudioObject.id
                            || $scope.selection.selectedObject.projectKey != data.codeStudioObject.projectKey) {
                            return; //too late!
                        }
                        data.usage = usage;
                    }).error(setErrorInScope.bind($scope));
                    $scope.codeStudioFullInfo = data;
                    $scope.codeStudioObject = data.codeStudioObject;
                }).error(setErrorInScope.bind($scope));
            };

            $scope.$watch("selection.confirmedItem", function(nv, ov) {
                if (!nv) return;
                $scope.refreshData();
            });
        }
    }
});

app.controller("CodeStudioHistoryController", function($scope, $stateParams, TopNav) {
    TopNav.setLocation(TopNav.TOP_NOTEBOOKS, "code-studios", null, "history");
});

app.controller("CodeStudioActionsController", function($scope, $stateParams, TopNav, CreateModalFromTemplate, WT1) {
    TopNav.setLocation(TopNav.TOP_NOTEBOOKS, "code-studios", null, "actions");

    $scope.runCommand = function() {
        CreateModalFromTemplate("/templates/code-studios/code-studio-run-command-modal.html", $scope, null, function(newScope) {
            newScope.commandParams = {
                useShell: true,
                commandString: "",
                commandArray: []
            };
            newScope.actionBehavior = 'WAIT_FOR_RETURN_VALUE';

            newScope.run = function() {
                WT1.event("code_studio_action_customcommand", {type:$scope.codeStudioObject.templateId, command:"run_command_line"});
                $scope.sendCommand($stateParams.projectKey, $stateParams.codeStudioObjectId, "run_command_line", newScope.commandParams, newScope.actionBehavior);
                newScope.dismiss();
            };
        });
    };
});

app.controller("CodeStudioFilesController", function($scope, $stateParams, TopNav, CreateModalFromTemplate, WT1) {
    TopNav.setLocation(TopNav.TOP_NOTEBOOKS, "code-studios", TopNav.TABS_CODE_STUDIO, "files");
    
    $scope.sendEventWhenSummaryIsAvailable("code_studio_files_viewed", k => {return {type: k.templateId}});
    
    $scope.uiState.activeZone = $stateParams.tab || $scope.uiState.activeZone || 'versioned';

    $scope.setFilesTab = function(tabname) {
        $scope.uiState.activeZone = tabname;
    };

    $scope.codeStudioVersionedEmptyCta = {
        title: "No file on this Code Studio.",
        text: "Create your own libraries or helpers. The contents are accessible only to this Code Studio instance.",
        btnAction: "create",
        btnLabel: "Create your first Code Studio file"
    }

    $scope.codeStudioResourcesEmptyCta = {
        title: "No file on this Code Studio.",
        text: "Create your own resources. The contents are accessible only to this Code Studio instance.",
        btnAction: "create",
        btnLabel: "Create your first Code Studio resource"
    }

});

app.service('CodeStudiosService', function($rootScope, DataikuAPI, CreateModalFromTemplate, WT1, ActivityIndicator, $state, $stateParams, 
                                           $filter, localStorageService) {
    const svc = this;

    svc.duplicate = function(scope, codeStudio) {
        CreateModalFromTemplate("/templates/code-studios/duplicate-code-studio-modal.html", scope, null, function(newScope) {
            newScope.newCodeStudio = {
                name: "Copy of " + codeStudio.name
            };
        
            newScope.duplicate = function() {
                WT1.event("code_studio_copy", {type: codeStudio.templateId});
                return DataikuAPI.codeStudioObjects.copy(codeStudio.projectKey, codeStudio.id, newScope.newCodeStudio.name)
                .success(function(data) {
                    $state.go('projects.project.code-studios.code-studio.view', {projectKey: data.projectKey, codeStudioObjectId: data.id});
                })
                .error(setErrorInScope.bind(newScope))
            };
        });
    };

    svc.changeOwner = function(scope, codeStudio) {
        CreateModalFromTemplate("/templates/code-studios/code-studio-change-owner-modal.html", scope, null, function(newScope) {
            DataikuAPI.security.listUsers(codeStudio.projectKey).success(function(data) {
                newScope.allUsers = data.sort((a, b) => a.displayName.localeCompare(b.displayName));
                newScope.allUsersLogin = data.map(user => '@' + user.login);
            }).error(setErrorInScope.bind(newScope));

            newScope.name = codeStudio.name;
            newScope.newOwner = codeStudio.owner;
            newScope.currentOwner = codeStudio.owner;

            newScope.changeOwner = function() {
                WT1.event("code_studio_owner_change", {type: codeStudio.templateId});
                return DataikuAPI.codeStudioObjects.changeOwner(codeStudio.projectKey, codeStudio.id, newScope.newOwner)
                .success(function() {
                    ActivityIndicator.success("Saved");
                    $state.reload();
                })
                .error(setErrorInScope.bind(newScope))
            };
        });
    };

    const CSIO_FAVORITE_EDITOR_KEY = 'dataiku.code-studio.favorite-editor';

    svc.editFileInCodeStudio = function(scope, zone, filepath) {
        CreateModalFromTemplate("templates/code-studios/edit-in-code-studio-box.html", scope, null, function(newScope){
            newScope.filepath = filepath;
            newScope.goToCsDashboard = function() {
                $state.go(
                    "projects.project.code-studios.list"
                )
            }
            DataikuAPI.codeStudioObjects.listHeads($stateParams.projectKey).success(function(data) {
                newScope.codeStudioObjects = data.items.filter(e => !e.templateWasDeleted || e.uiState.state !== 'STOPPED');
            }).error(setErrorInScope.bind(scope));
            newScope.selectedCodeStudioId = localStorageService.get(CSIO_FAVORITE_EDITOR_KEY);

            newScope.go = function() {
                localStorageService.set(CSIO_FAVORITE_EDITOR_KEY, newScope.selectedCodeStudioId);
                const selectedCodeStudio = newScope.codeStudioObjects.find(e => e.id === newScope.selectedCodeStudioId);
                $state.go(
                    "projects.project.code-studios.code-studio.view",
                    {
                        projectKey: $stateParams.projectKey,
                        codeStudioObjectId: selectedCodeStudio.id,
                        codeStudioName: $filter('slugify')(selectedCodeStudio.name),
                        zone: zone,
                        file: filepath
                    }
                );
            }
        });
    }
});


})();
