(function(){
'use strict';

const app = angular.module('dataiku.ml.explainability');

app.service('OverridesExtraColumnsService', function(ModelDataUtils, MLTaskDesignUtils) {

    // Prediction is not selectable in binary classification because it does not make sense, i.e. it is almost always possible to get
    // the same outcome without prediction in the rule (unless when you revert the prediction, which is dumb), and it
    // would considerably complexify the results.

    function _getExtraColumns (isBinaryClassification, isClassification, uncertaintySettings) {
        let extraColumns = [];
        // User cannot select prediction for binary classification tasks, but it can for the rest of cases
        if (!isBinaryClassification) {
            extraColumns.push({
                name: "prediction",
                type: isClassification ? "string" : "double"
            });
        }
        // Uncertainty is only applicable for classificaiton tasks as 1 - max(probas)
        if (isClassification) {
            extraColumns.push({
                name: "prediction_uncertainty",
                type: "double"
            });
        }
        if (uncertaintySettings && uncertaintySettings.predictionIntervalsEnabled) {
            extraColumns = extraColumns.concat([
                {
                    name: "prediction_interval_size",
                    type: "double"
                },
                {
                    name: "prediction_interval_relative_size",
                    type: "double"
                },
                {
                    name: "prediction_interval_lower",
                    type: "double"
                },
                {
                    name: "prediction_interval_upper",
                    type: "double"
                }
            ]);
        }
        return extraColumns;
    }

    function getExtraColumnsFromModelData(modelData) {
        return _getExtraColumns(ModelDataUtils.isBinaryClassification(modelData),
                                ModelDataUtils.isClassification(modelData),
                                ModelDataUtils.getUncertaintySettings(modelData));
    }

    function getExtraColumnsFromMLTaskDesign(mltaskDesign) {
        return _getExtraColumns(MLTaskDesignUtils.isBinaryClassification(mltaskDesign),
                                MLTaskDesignUtils.isClassification(mltaskDesign),
                                MLTaskDesignUtils.getUncertaintySettings(mltaskDesign));
    }

    return {
        getExtraColumnsFromModelData,
        getExtraColumnsFromMLTaskDesign
    };
});

app.constant('OverridesMetricsView', {
    SANKEY: 'SANKEY',
    TABLE: 'TABLE'
});

app.controller('OverridesMetricsController', function($scope, $stateParams, OverridesMetricsView, ModelDataUtils) {
    $scope.OverridesMetricsView = OverridesMetricsView;
    $scope.ModelDataUtils = ModelDataUtils;
    $scope.activeView = OverridesMetricsView.SANKEY;
    $scope.inDashboard = $stateParams.dashboardId || $stateParams.insightId;
});

app.component('overridesMetricsViewSelector', {
    bindings: { activeView: '=' },
    templateUrl: '/templates/ml/prediction-model/overrides/metrics_view_selector.html',
    controller: function($scope, OverridesMetricsView) {
        $scope.OverridesMetricsView = OverridesMetricsView;
    }
});

app.service('OverridesMetricsService', function(BinaryClassificationModelsService, ModelDataUtils) {
    return {
        getNbTotalRows: (modelData, evaluation) => {
            if (evaluation) {
                return evaluation.nbEvaluationRows;
            }
            else if (modelData.trainInfo.fullRows) {
                return modelData.trainInfo.fullRows;
            }
            return modelData.trainInfo.testRows;
        },
        getOverridesMetrics: (modelData) => {
            if (ModelDataUtils.isBinaryClassification(modelData)) {
                let currentCutData = BinaryClassificationModelsService.findCutData(modelData.perf, modelData.userMeta['activeClassifierThreshold']);
                if (currentCutData !== undefined && modelData.perf.perCutData.overridesMetrics) {
                    return modelData.perf.perCutData.overridesMetrics[currentCutData.index].perOverride;
                }
            } else {
                return modelData.perf.metrics && modelData.perf.metrics.overridesMetrics.perOverride;
            }
        }
    };
});

app.component('overridesMetricsSankey', {
    bindings: {
        modelData: '<',
        evaluation: '<',
    },
    templateUrl: "/templates/ml/prediction-model/overrides/metrics_sankey.html",
    controller: function($scope, $filter, OverridesMetricsService, ModelDataUtils) {
        const $ctrl = this;

        function getOverridesMetrics() {
            return OverridesMetricsService.getOverridesMetrics($ctrl.modelData);
        }

        function getNiceOverrideDescription(overrideIndex) {
            const override = $ctrl.modelData.overridesParams.overrides[overrideIndex];
            const name = $filter('escapeHtml')(override.name);
            return `'${name}': ` + '<code class="overrides-metrics__tooltip-code">' + $filter('stripHtml')($filter('filterNiceRepr')(override.filter)) + '</code>';
        }

        function getChartStyle() {
            const nbOverrides = getOverridesMetrics().length;
            return {
                height: `${250 + 5 * nbOverrides}px`,
                minWidth: `${150 + 180 * nbOverrides}px` // if too many overrides, add a scrollbar
            };
        }

        function getChartOptions() {
            const BASE_OPTIONS = {
                type: 'sankey',
                orient: 'horizontal',
                draggable: false,
                nodeAlign: 'left',
                label: { position: 'left' },
                layoutIterations: 0,
                nodeGap: 16,
                top: 16,
                right: 16,
                bottom: 16,
                left: 16
            };

            // Colors
            const MATCHING_COLOR = '#76B8FD';
            const NOT_MATCHING_COLOR = '#BBBBBB';
            const OVERRIDE_COLOR = '#2D86FB';
            const KEPT_COLOR = '#666666';

            // Node params
            const MATCHING_NODE_PARAMS = {
                itemStyle: { color: MATCHING_COLOR },
                label: { show: true }
            };
            const NOT_MATCHING_NODE_PARAMS = {
                itemStyle: { color: NOT_MATCHING_COLOR },
                label: { show: false, formatter: '' }
            };
            const LAST_NOT_MATCHING_NODE_PARAMS = {
                itemStyle: { color: NOT_MATCHING_COLOR },
                label: { show: true }
            };
            const OVERRIDDEN_NODE_PARAMS = {
                itemStyle: { color: OVERRIDE_COLOR },
                label: { formatter: 'Overridden' }
            };
            const KEPT_NODE_PARAMS = {
                itemStyle: { color: KEPT_COLOR },
                label: { formatter: 'Kept' }
            };
            const ORIGINAL_PREDICTIONS_NODE_PARAMS = {
                itemStyle: { color: NOT_MATCHING_COLOR },
                label: { show: true, rotate: 90, position: 'inside', align: 'middle' }
            };

            // Link params
            const MATCHING_LINK_PARAMS = {
                lineStyle: { opacity: 0.5, color: MATCHING_COLOR }
            };
            const NOT_MATCHING_LINK_PARAMS = {
                lineStyle: { opacity: 0.5, color: NOT_MATCHING_COLOR }
            };
            const OVERRIDDEN_LINK_PARAMS = {
                lineStyle: { opacity: 0.5, color: OVERRIDE_COLOR }
            };
            const KEPT_LINK_PARAMS = {
                lineStyle: { opacity: 0.3, color: KEPT_COLOR }
            };

            // Names
            const ORIGINAL_PREDICTIONS_NODE_NAME = "Original predictions";
            const NOT_MATCHING_ANY_NODE_NAME = "Didn't match any override";
            const getNotMatchingNodeName = index => `Didn't match override #${index}`;
            const getMatchingNodeName = index => `Matched override #${index}`;
            const getKeptNodeName = index => `Kept by #${index}`;
            const getOverriddenNodeName = index => `Overridden by #${index}`;

            // Tooltips
            const getMatchingTooltip = (value, index) => `<strong>${value}</strong> rows <strong>matched</strong> ${getNiceOverrideDescription(index)}`;
            const getNotMatchingAnyTooltip = (value) => `<strong>${value}</strong> rows <strong>didn't match</strong> any overrides`;
            const getKeptTooltip = (value, index) => `<strong>${value}</strong> rows had the right prediction before matching ${getNiceOverrideDescription(index)}`;
            const getOverriddenTooltip = (value, index) => `<strong>${value}</strong> rows were <strong>overridden</strong> by ${getNiceOverrideDescription(index)}`;
            const getNotMatchingTooltip = (value, index, previousValue) => {
                const tooltipContent = `<strong>${value}</strong> rows <strong>didn't match</strong> ${getNiceOverrideDescription(index)}`;
                if (index > 0) {
                    return `Out of the ${previousValue} rows that didn't match any prior overrides, ` + tooltipContent;
                }
                return tooltipContent;
            };

            const overrides = getOverridesMetrics();
            const nodes = [];
            const links = [];
            nodes.push({ name: ORIGINAL_PREDICTIONS_NODE_NAME, ...ORIGINAL_PREDICTIONS_NODE_PARAMS });
            let remainingRows = OverridesMetricsService.getNbTotalRows($ctrl.modelData, $ctrl.evaluation);

            // Adding the (matched | didn't match) nodes and links
            for (let i = 0; i < overrides.length; i++) {
                const isFirstOverride = i === 0;
                const override = overrides[i];
                const source = isFirstOverride ? ORIGINAL_PREDICTIONS_NODE_NAME : getNotMatchingNodeName(i);

                remainingRows -= override.nbMatchingRows;
                if (remainingRows > 0) {
                    const target = getNotMatchingNodeName(i + 1);
                    const value = remainingRows;
                    const tooltipContent = getNotMatchingTooltip(value, i, remainingRows + override.nbMatchingRows);
                    nodes.push({ name: target, tooltipContent, ...NOT_MATCHING_NODE_PARAMS });
                    links.push({ value, target, source, tooltipContent, ...NOT_MATCHING_LINK_PARAMS });
                }
                if (override.nbMatchingRows > 0) {
                    const target = getMatchingNodeName(i + 1);
                    const value = override.nbMatchingRows;
                    const tooltipContent = getMatchingTooltip(value, i);
                    nodes.push({ name: target, tooltipContent, ...MATCHING_NODE_PARAMS });
                    links.push({ value, source, target, tooltipContent, ...MATCHING_LINK_PARAMS });
                }
            }

            // Adding the last node for rows that didn't match any override
            if (remainingRows > 0) {
                const source = getNotMatchingNodeName(overrides.length);
                const target = NOT_MATCHING_ANY_NODE_NAME;
                const value = remainingRows;
                const tooltipContent = getNotMatchingAnyTooltip(value);
                nodes.push({ name: target, tooltipContent, ...LAST_NOT_MATCHING_NODE_PARAMS });
                links.push({ value, target, source, tooltipContent, ...NOT_MATCHING_LINK_PARAMS });
            }

            // Adding the (overridden | kept) nodes and links
            // Not merging this for loop with the previous one because order in which nodes are added is important
            for (let i = 0; i < overrides.length; i++) {
                const override = overrides[i];
                const source = getMatchingNodeName(i + 1);
                if (override.nbMatchingRows - override.nbChangedRows > 0) {
                    const target = getKeptNodeName(i + 1);
                    const value = override.nbMatchingRows - override.nbChangedRows;
                    const tooltipContent = getKeptTooltip(value, i);
                    nodes.push({ name: target, tooltipContent, ...KEPT_NODE_PARAMS });
                    links.push({ value, source, target, tooltipContent, ...KEPT_LINK_PARAMS });
                }
                if (override.nbChangedRows > 0) {
                    const target = getOverriddenNodeName(i + 1);
                    const value = override.nbChangedRows;
                    const tooltipContent = getOverriddenTooltip(value, i);
                    nodes.push({ name: target, tooltipContent, ...OVERRIDDEN_NODE_PARAMS });
                    links.push({ value, source, target, tooltipContent, ...OVERRIDDEN_LINK_PARAMS });
                }
            }
            return { series: [{ nodes, links, ...BASE_OPTIONS }] };
        }

        const updateOptions = () => {
            $scope.options = getChartOptions();
            $scope.chartStyle = getChartStyle();
        };

        $ctrl.onChartInit = chartHandle => {
            chartHandle.on('mousemove', params => {
                if (!params.data.tooltipContent) {
                    return;
                }
                $scope.showTooltip = true;
                $scope.content = params.data.tooltipContent;
                const container = angular.element(".overrides-metrics__sankey-container");
                const tooltip = angular.element(".overrides-metrics__tooltip");
                // vertical position
                const totalScrollFromTop = container.parents().toArray().reduce((total, e) => total + $(e).scrollTop(), 0);
                const contentY = params.event.offsetY - totalScrollFromTop;
                // horizontal position
                let contentX = params.event.offsetX - container.scrollLeft() - (tooltip.outerWidth() / 2);
                // make sure that the tooltip doesn't overflow
                const overflowRight = contentX + tooltip.outerWidth() - container.width();
                const overflowLeft = -contentX;
                if (overflowRight > 0) {
                    contentX -= overflowRight;
                } else if (overflowLeft > 0) {
                    contentX += overflowLeft;
                }
                $scope.contentX = contentX;
                $scope.contentY = contentY;
                $scope.arrowX = params.event.offsetX - container.scrollLeft();
                $scope.$apply();
            });
            chartHandle.on('mouseout', () => {
                $scope.showTooltip = false;
                $scope.$apply();
            });
        };

        $ctrl.$onInit = () => {
            if (ModelDataUtils.isBinaryClassification($ctrl.modelData)) {
                const thresholdUpdateChecker = ModelDataUtils.createThresholdUpdateChecker();
                $ctrl.$doCheck = () => thresholdUpdateChecker.executeIfUpdated($ctrl.modelData, updateOptions);
            }
            updateOptions();
        };
    }
});

app.component("overridesMetricsTable", {
    bindings: {
        modelData: '<',
        evaluation: '<'
    },
    templateUrl: "/templates/ml/prediction-model/overrides/metrics_table.html",
    controller: function($scope, OverridesMetricsService, ModelDataUtils) {
        const $ctrl = this;

        const ALREADY_MATCHED_COLOR = '#666666';
        const MATCHING_COLOR = '#76B8FD';
        const NOT_MATCHING_COLOR = '#BBBBBB';
        const OUTCOME_CHANGED_COLOR = '#2D86FB';
        const OUTCOME_KEPT_COLOR = '#BBBBBB';

        $scope.matchStatusRegions = {}; // already-matched / matched / non-matching
        $scope.matchedRegions = {}; // matched
        $scope.outcomeStatusRegions = {}; // overridden / matched but kept

        $scope.anyMatch = {}; // true if at least one row matched

        function updateRegions() {
            /* For each override, set the '*Regions' variables in the scope to use the 'multi-region-bar' components */
            const overridesMetrics = OverridesMetricsService.getOverridesMetrics($ctrl.modelData);
            const nbTotalRows = OverridesMetricsService.getNbTotalRows($ctrl.modelData, $ctrl.evaluation);
            $scope.overridesResult = {};
            let nbAlreadyMatchedRows = 0;
            for (const overrideMetrics of overridesMetrics) {
                const name = overrideMetrics.name;
                const nbMatchingRows = overrideMetrics.nbMatchingRows;
                const nbChangedRows = overrideMetrics.nbChangedRows;
                const nbNonMatchingRows = nbTotalRows - nbMatchingRows - nbAlreadyMatchedRows;
                const nbKeptRows = nbMatchingRows - nbChangedRows;

                $scope.anyMatch[name] = nbMatchingRows > 0;
                $scope.matchStatusRegions[name] = [
                    {size: nbAlreadyMatchedRows, color: ALREADY_MATCHED_COLOR},
                    {size: nbMatchingRows, color: MATCHING_COLOR},
                    {size: nbNonMatchingRows, color: NOT_MATCHING_COLOR}
                ];
                $scope.matchedRegions[name] = [
                    {size: nbMatchingRows, color:MATCHING_COLOR}
                ];
                $scope.outcomeStatusRegions[name] = [
                    {size: nbChangedRows, color: OUTCOME_CHANGED_COLOR},
                    {size: nbKeptRows, color: OUTCOME_KEPT_COLOR}
                ];

                nbAlreadyMatchedRows += nbMatchingRows;
            }
        }

        $ctrl.$onInit = () => {
            $ctrl.hasOverrides = ModelDataUtils.hasOverrides($ctrl.modelData);
            if (ModelDataUtils.isBinaryClassification($ctrl.modelData)) {
                const thresholdUpdateChecker = ModelDataUtils.createThresholdUpdateChecker();
                $ctrl.$doCheck = () => thresholdUpdateChecker.executeIfUpdated($ctrl.modelData, updateRegions);
            }
            updateRegions();
        };
    }
});

app.component("overridesCreateNewModel", {
    bindings: {
        modelData: "<",
    },
    // For now we do not display anything if not on analysis report or saved model
    template: `
    <saved-model-overrides-create-new-model ng-if="$ctrl.isSavedModel"
                                            fmi="$ctrl.fmi"
                                            model-data="$ctrl.modelData">
    </saved-model-overrides-create-new-model>
    <analysis-overrides-create-new-model ng-if="$ctrl.isAnalysis"
                                         fmi="$ctrl.fmi"
                                         model-data="$ctrl.modelData">
    </analysis-overrides-create-new-model>`,
    controller: function(FullModelLikeIdUtils) {
        const $ctrl = this;

        $ctrl.$onInit = function() {
            $ctrl.fmi = $ctrl.modelData.fullModelId;
            $ctrl.isAnalysis = FullModelLikeIdUtils.isAnalysis($ctrl.fmi);
            $ctrl.isSavedModel = FullModelLikeIdUtils.isSavedModel($ctrl.fmi);
        };
    }
});

app.component("savedModelOverridesCreateNewModel", {
    bindings: {
        modelData: "<",
        fmi: "<"
    },
    template: `
    <overrides-create-new-model-button model-data="$ctrl.modelData"
                                       fmi="$ctrl.fmi"
                                       on-model-creation-callback="$ctrl.onModelCreationCallback"
                                       modal-title="$ctrl.modalTitle"
                                       modal-description="A new model version will be created to apply the newly defined overrides. The existing model version will remain available.">
    </overrides-create-new-model-button>
`,
    controller: function($scope, $state, DataikuAPI, FutureProgressModal, ModelDataUtils) {
        const $ctrl = this;
        $ctrl.$onInit = () => {
            $ctrl.modalTitle = `Create new version with ${ModelDataUtils.hasOverrides($ctrl.modelData) ?  "different overrides" : "newly defined overrides"}`;
        };
        $ctrl.onModelCreationCallback = function(newModelName, newOverridesParams, dismissModalCallback) {
            DataikuAPI.savedmodels.prediction.createModelVersionWithDifferentOverrides($ctrl.fmi, newModelName, newOverridesParams).success(function(data) {
                const hasOverrides = ModelDataUtils.hasOverrides($ctrl.modelData);
                const progressModalTitle = hasOverrides ? "Creating new version with different overrides": "Creating new version with overrides";
                FutureProgressModal.show($scope.$new(), data, progressModalTitle).then(function(newFmi) {
                    if (newFmi) { // No result would mean abort
                        dismissModalCallback();
                        $state.go('projects.project.savedmodels.savedmodel.prediction.report', {fullModelId: newFmi})
                    }
                }).catch(setErrorInScope.bind($scope));
            }).error(setErrorInScope.bind($scope));
        };
    }
});

app.component("analysisOverridesCreateNewModel", {
    bindings: {
        modelData: "<",
        fmi: "<"
    },
    template: `
  <overrides-create-new-model-button model-data="$ctrl.modelData"
                                     fmi="$ctrl.fmi"
                                     create-disabled-message="$ctrl.createDisabledMessage"
                                     on-model-creation-callback="$ctrl.onModelCreationCallback" 
                                     modal-title="$ctrl.modalTitle"
                                     modal-description="A new modeling session will be created to apply the newly defined overrides. The existing model will remain available in the existing session.">
  </overrides-create-new-model-button>
`,
    controller: function($scope, $state, $stateParams, $interval, DataikuAPI, ModelDataUtils) {
        const $ctrl = this;

        // Using a very simple poll logic to fetch whether taks is currently working or not, to decide on activating the create new model button.
        // Because train might be launched anytime while visiting widget page, we just fetch the info every 5 seconds.
        function pollTaskStatus() {
            DataikuAPI.analysis.pml.getLightTaskStatus($stateParams.projectKey, $stateParams.analysisId, $stateParams.mlTaskId).success(function(data) {
                $ctrl.isTaskWorking = data && (data.guessing || data.training);
            }).error(setErrorInScope.bind($scope));
        }

        let cancelPoll;
        function startPoll() {
            cancelPoll = $interval(pollTaskStatus, 5000);
        }

        $ctrl.onModelCreationCallback = function(newModelName, newOverridesParams, dismissModalCallback) {
            DataikuAPI.analysis.pml.createNewModelWithDifferentOverrides($ctrl.fmi, newModelName, newOverridesParams).success(function() {
                dismissModalCallback();
                $state.go('projects.project.analyses.analysis.ml.predmltask.list.results.sessions');
            }).error(setErrorInScope.bind($scope));
        };

        $ctrl.createDisabledMessage = function() {
            if ($ctrl.isTaskWorking) {
                return "Cannot create a new model during an ongoing training.";
            }
            return null;
        };

        $ctrl.$onInit = function() {
            $ctrl.isTaskWorking = false;
            $ctrl.modalTitle = `Create new model with ${ModelDataUtils.hasOverrides($ctrl.modelData) ? "different" : "newly defined"} overrides`;
            startPoll();
        };

        $ctrl.$onDestroy = function() {
            $interval.cancel(cancelPoll);
        };
    }
});

app.component("overridesCreateNewModelButton", {
    bindings: {
        fmi: "<",
        modelData: "<",
        onModelCreationCallback: "<",
        modalTitle: "<",
        modalDescription: "@",
        createDisabledMessage: "<?"
    },
    templateUrl: "/templates/ml/prediction-model/overrides/create_new_model.html",
    controller: function($scope, CreateModalFromTemplate, DataikuAPI, MlParamsWithFilterService, ModelDataUtils, OverridesExtraColumnsService) {
        const $ctrl = this;

        function getAuthorizedColumnsForOverrides() {
            const authorizedColumnsSchema = angular.copy($ctrl.modelData.splitDesc.schema);
            authorizedColumnsSchema.columns.splice(authorizedColumnsSchema.columns.map(column => column.name).indexOf($ctrl.modelData.coreParams.target_variable), 1);
            MlParamsWithFilterService.enrichPostScriptSchemaWithExtraColumns(authorizedColumnsSchema, OverridesExtraColumnsService.getExtraColumnsFromModelData($ctrl.modelData));
            return authorizedColumnsSchema;
        }

        function getSuggestedNameForNewVersion(currentName) {
            const regexMatch = currentName.match(/(?<baseName>.*) - new Overrides( \((?<nbDuplicates>\d+)\))?$/);
            if (!regexMatch) {
                // Example: 'MyModelName' --> 'MyModelName - new Overrides'
                return currentName + " - new Overrides";
            }
            // Example: 'MyModelName - new Overrides' --> 'MyModelName - new Overrides (2)'
            // Example: 'MyModelName - new Overrides (50)' --> 'MyModelName - new Overrides (51)'
            const baseName = regexMatch.groups.baseName;
            const nbDuplicates = regexMatch.groups.nbDuplicates ? (Number(regexMatch.groups.nbDuplicates) + 1) : 2;
            return `${baseName} - new Overrides (${nbDuplicates})`;
        }

        $ctrl.$onInit = () => {
            $ctrl.hasOverrides = ModelDataUtils.hasOverrides($ctrl.modelData);
        };

        $ctrl.openCreateNewModelModal = function () {
            CreateModalFromTemplate("/templates/ml/prediction-model/overrides/new_model_modal.html", $scope, null, function(newScope) {
                // Current overrides params might be undefined for old models
                newScope.newOverridesParams = $ctrl.modelData.overridesParams ? angular.copy($ctrl.modelData.overridesParams): {overrides: []};
                newScope.overridesChanged = () => !angular.equals($ctrl.modelData.overridesParams, newScope.newOverridesParams);
                newScope.isClassification = ModelDataUtils.isClassification($ctrl.modelData);

                newScope.newModelName = getSuggestedNameForNewVersion($ctrl.modelData.userMeta.name);
                newScope.postScriptFeaturesSchema = getAuthorizedColumnsForOverrides();
                newScope.classes = $ctrl.modelData.classes;

                newScope.confirm = function() {
                    $ctrl.onModelCreationCallback(newScope.newModelName, newScope.newOverridesParams, () => newScope.dismiss());
                };

                newScope.cancel = function() {
                    newScope.dismiss();
                };

                // This API call will only work if there is predicted data for the corresponding model. This should always be the case for
                // models that support overrides, both on Analysis & Saved Model.
                const targetColumnAnalysisCallback = () => DataikuAPI.analysis.predicted.detailedColumnAnalysis($ctrl.fmi, $ctrl.modelData.trainedWithScript, $ctrl.modelData.coreParams.target_variable, 50);

                newScope.addNewMlOverride = MlParamsWithFilterService.createAddNewOverrideParamCallback(
                    newScope.newOverridesParams.overrides,
                    ModelDataUtils.isClassification($ctrl.modelData),
                    targetColumnAnalysisCallback,
                    $ctrl.modelData.classes
                );
            });
        }
    }
});

app.component('overriddenBadgeBase', {
    /**
     * WARNING:
     * When this component is used, any function in the transcluded part will be executed in `overriddenBadgeBase`'s context.
     * That's why we had to set `$scope.WhatIfFormattingService = WhatIfFormattingService;` in the controller.
     * This is because of the way that `dkuInlinePopover` leverages transclusion.
     */
    template: `
        <span ng-if="$ctrl.shouldDisplay"
              class="overridden-prediction-badge cursor-pointer"
              dku-inline-popover
              placement="bottom"
              container="body">
            <label>
                <i class="overridden-prediction-badge__icon icon-dku-override {{ !$ctrl.iconOnly ? 'mleft4' : '' }}"></i>
                <span class="overridden-prediction-badge__text" ng-show="!$ctrl.iconOnly">{{ $ctrl.textInBadge }}</span>
            </label>
            <content>
                <ng-transclude ng-show="!$ctrl.hideTranscludedContent"></ng-transclude>
                <hr ng-show="!$ctrl.hideTranscludedContent" class="mtop8 mbot8">
                Applied rule: <strong>{{ getAppliedOverride().name }}</strong>
                <code class="overrides-metrics__tooltip-code">{{ getAppliedOverride().filter | filterNiceRepr | stripHtml }}</code>
            </content>
        </span>
        <span ng-if="!$ctrl.shouldDisplay" class="overridden-prediction-badge-placeholder"></span>
    `,
    bindings: {
        modelData: '<',
        overrideInfo: '<',
        textInBadge: '<?',
        shouldDisplay: '<',
        iconOnly: '=?',
        hideTranscludedContent: '<?'
    },
    restrict: 'E',
    transclude: true,
    controller: function ($scope, ModelDataUtils, WhatIfFormattingService) {
        const $ctrl = this;

        $scope.ModelDataUtils = ModelDataUtils;
        $scope.WhatIfFormattingService = WhatIfFormattingService;

        $scope.getAppliedOverride = () => $ctrl.modelData.overridesParams.overrides.find(e => $ctrl.overrideInfo && $ctrl.overrideInfo.appliedRule === e.name);
    }
});

app.component('overriddenPredictionBadge', {
    template: `
        <overridden-badge-base ng-if="ModelDataUtils.isMulticlass($ctrl.modelData)"
                               model-data="$ctrl.modelData"
                               override-info="$ctrl.overrideInfo"
                               text-in-badge="'was ' + getFormattedPredictionForMulticlass() + ' before override'"
                               should-display="$ctrl.overrideInfo.ruleMatched && ($ctrl.overrideInfo.predictionChanged || !$ctrl.iconOnly)"
                               icon-only="$ctrl.iconOnly">
            Before overrides:
            <ul>
                <li ng-repeat="clazz in $ctrl.modelData.classes">{{ clazz }}: <span class="interactive-scoring__prediction">{{ WhatIfFormattingService.formatProba($ctrl.overrideInfo.rawResult.probabilities[clazz]) }}%</span></li>
            </ul>
        </overridden-badge-base>
        <overridden-badge-base ng-if="ModelDataUtils.isBinaryClassification($ctrl.modelData)"
                               model-data="$ctrl.modelData"
                               override-info="$ctrl.overrideInfo"
                               text-in-badge="'was &lsquo;' + WhatIfFormattingService.formatPrediction($ctrl.modelData, $ctrl.overrideInfo.rawResult.prediction) + '&rsquo; before override'"
                               should-display="$ctrl.overrideInfo.ruleMatched && $ctrl.overrideInfo.predictionChanged"
                               icon-only="$ctrl.iconOnly"
                               hide-transcluded-content="!$ctrl.iconOnly">
            ...was <span class="interactive-scoring__prediction">'{{ WhatIfFormattingService.formatPrediction($ctrl.modelData, $ctrl.overrideInfo.rawResult.prediction) }}'</span> before overrides
        </overridden-badge-base>
        <overridden-badge-base ng-if="ModelDataUtils.isRegression($ctrl.modelData)"
                               model-data="$ctrl.modelData"
                               override-info="$ctrl.overrideInfo"
                               text-in-badge="'was ' + WhatIfFormattingService.formatPrediction($ctrl.modelData, $ctrl.overrideInfo.rawResult.prediction) + ' before override'"
                               should-display="$ctrl.overrideInfo.ruleMatched && $ctrl.overrideInfo.predictionChanged"
                               icon-only="$ctrl.iconOnly"
                               hide-transcluded-content="!$ctrl.iconOnly">
            ...was <span class="interactive-scoring__prediction">{{ WhatIfFormattingService.formatPrediction($ctrl.modelData, $ctrl.overrideInfo.rawResult.prediction) }}</span> before overrides
        </overridden-badge-base>
    `,
    bindings: {
        overrideInfo: '<',
        modelData: '<',
        iconOnly: '=?'
    },
    controller: function($scope, WhatIfFormattingService, ModelDataUtils) {
        const $ctrl = this;

        $scope.ModelDataUtils = ModelDataUtils;
        $scope.WhatIfFormattingService = WhatIfFormattingService;

        $scope.getFormattedPredictionForMulticlass = () => {
            if (!$ctrl.overrideInfo || !$ctrl.overrideInfo.rawResult) {
                return;
            }
            const proba = WhatIfFormattingService.formatProba($ctrl.overrideInfo.rawResult.probabilities[$ctrl.overrideInfo.rawResult.prediction]) + '%';
            if ($ctrl.overrideInfo.predictionChanged) {
                return `'${WhatIfFormattingService.formatPrediction($ctrl.modelData, $ctrl.overrideInfo.rawResult.prediction)}' — ${proba}`;
            }
            return proba;
        };
    }
});

app.component('overriddenProbaBadge', {
    template: `
        <overridden-badge-base ng-if="ModelDataUtils.isMulticlass($ctrl.modelData)"
                               model-data="$ctrl.modelData"
                               override-info="$ctrl.overrideInfo"
                               should-display="$ctrl.overrideInfo.ruleMatched"
                               icon-only="$ctrl.iconOnly">
            Before overrides:
            <ul>
                <li ng-repeat="clazz in $ctrl.modelData.classes">{{ clazz }}: <span class="interactive-scoring__prediction">{{ WhatIfFormattingService.formatProba($ctrl.overrideInfo.rawResult.probabilities[clazz]) }}%</span></li>
            </ul>
        </overridden-badge-base>
        <overridden-badge-base ng-if="ModelDataUtils.isBinaryClassification($ctrl.modelData)"
                               model-data="$ctrl.modelData"
                               override-info="$ctrl.overrideInfo"
                               text-in-badge="'was ' + WhatIfFormattingService.formatProba($ctrl.overrideInfo.rawResult.probabilities[$ctrl.modelData.classes[1]]) + '% before override'"
                               should-display="$ctrl.overrideInfo.ruleMatched"
                               icon-only="$ctrl.iconOnly"
                               hide-transcluded-content="!$ctrl.iconOnly">
            ...was <span class="interactive-scoring__prediction">{{ WhatIfFormattingService.formatProba($ctrl.overrideInfo.rawResult.probabilities[$ctrl.modelData.classes[1]]) }}%</span> before overrides
        </overridden-badge-base>
    `,
    bindings: {
        overrideInfo: '<',
        modelData: '<',
        iconOnly: '=?'
    },
    controller: function($scope, ModelDataUtils, WhatIfFormattingService) {
        $scope.ModelDataUtils = ModelDataUtils;
        $scope.WhatIfFormattingService = WhatIfFormattingService;
    }
});

})();
