(function() {
    'use strict';

    var app = angular.module('dataiku.recipes');

    const getIconType = function (type) {
        switch (type) {
            case 'LEFT': return "icon-dku-left-join";
            case 'LEFT_ANTI': return "icon-dku-left-unmatched-join";
            case 'INNER': return "icon-dku-inner-join";
            case 'FULL': return "icon-dku-outer-join";
            case 'RIGHT': return "icon-dku-right-join";
            case 'RIGHT_ANTI': return "icon-dku-right-unmatched-join";
            case 'CROSS': return "icon-dku-cross-join";
            case 'ADVANCED': return "icon-dku-advanced-join";
        }
    };

    const getRealColor = function (index) {
        // Grab color from visual-recipes (and color-variable.less)
        switch (index%6) {
            case 0: return "#28A9DD";
            case 1: return "#29AF5D";
            case 2: return "#8541AA";
            case 3: return "#F44336";
            case 4: return "#4785A4";
            case 5: return "#F28C38";
        }
    }

    app.controller("JoinRecipeCreationController", function($scope, Fn, $stateParams, DataikuAPI, $controller, translate) {
        $scope.recipeType = "join";
        $controller("SingleOutputDatasetRecipeCreationController", {$scope:$scope});

        $scope.autosetName = function() {
            if ($scope.io.inputDataset) {
                var niceInputName = $scope.io.inputDataset.replace(/[A-Z]*\./,"");
                $scope.maybeSetNewDatasetName(niceInputName + "_joined");
            }
        };

        $scope.getCreationSettings = function () {
            return {virtualInputs: [$scope.io.inputDataset, $scope.io.inputDataset2]};
        };

        var superFormIsValid = $scope.formIsValid;
        $scope.formIsValid = function() {
            return !!(superFormIsValid() &&
                $scope.io.inputDataset2 && $scope.activeSchema2 && $scope.activeSchema2.columns && $scope.activeSchema2.columns.length
            );
        };
        $scope.showOutputPane = function() {
            return !!($scope.io.inputDataset && $scope.io.inputDataset2);
        };
    });


    app.controller("JoinRecipeController", function ($scope, $timeout, $controller, $q, $stateParams, DataikuAPI, DKUtils, Dialogs,
                   PartitionDeps, CreateModalFromTemplate, RecipesUtils, Logger, DatasetUtils, RecipeComputablesService, WT1, JoinDisplayNamesService, translate) {
        $scope.joinRecipeType = 'REGULAR';
        $scope.distanceUnits = {
            'KILOMETER': {name: 'Kilometers', simbol: 'km'},
            'METER': {name: 'Meters', simbol: 'm'},
            'MILE': {name: 'Miles', simbol: 'mi'},
            'YARD': {name: 'Yards', simbol: 'yd'},
            'FOOT': {name: 'Feet', simbol: 'ft'},
            'NAUTICAL_MILE': {name: 'Nautical miles', simbol: 'nmi'},
        }
        var visualCtrl = $controller('VisualRecipeEditorController', {$scope: $scope}); //Controller inheritance

        let contextProjectKey = $scope.context && $scope.context.projectKey ? $scope.context.projectKey:$stateParams.projectKey;
        /****** computed columns ********/
        function computedColumnListUpdated(computedColumns) {
            $scope.params.computedColumns = angular.copy(computedColumns);
            $scope.updateRecipeStatusLater();
        }

        const originalSaveHook = $scope.hooks.save;
        $scope.hooks.save = function () {
            // fuzzy and geojoin recipes inherit that hook, but have their own sendRecipeParamsToWT1 method, so we skip them
            if($scope.joinRecipeType === 'REGULAR') {
                sendRecipeParamsToWT1();
            }
            if(originalSaveHook) {
                return originalSaveHook();
            }
        };

        $scope.showDropDown = function() {
            return $('#join-condition-modal').parent().length == 0;
        }

        $scope.getJoinTypeName = function (type) {
            return JoinDisplayNamesService.getJoinTypeName(type);
        };

        $scope.getIconType = getIconType;
        $scope.getRealColor = getRealColor;

        $scope.getDescriptionType = function (type) {
            switch (type) {
                case 'LEFT': return translate("JOIN_RECIPE.JOIN_DESCRITION.LEFT_JOIN", "Keep all rows of the left dataset and add information from the right dataset");
                case 'INNER': return translate("JOIN_RECIPE.JOIN_DESCRITION.INNER_JOIN", "Keep matches and drop rows without match from both datasets");
                case 'FULL': return translate("JOIN_RECIPE.JOIN_DESCRITION.FULL_JOIN", "Keep all matches and keep rows without match from both datasets");
                case 'RIGHT': return translate("JOIN_RECIPE.JOIN_DESCRITION.RIGHT_JOIN", "Keep all matches and keep rows without match from the right dataset");
                case 'LEFT_ANTI': return translate("JOIN_RECIPE.JOIN_DESCRITION.LEFT_ANTI_JOIN", "Keep rows of the left dataset without match from the right");
                case 'RIGHT_ANTI': return translate("JOIN_RECIPE.JOIN_DESCRITION.RIGHT_ANTI_JOIN", "Keep rows of the right dataset without match from the left");
                case 'CROSS': return translate("JOIN_RECIPE.JOIN_DESCRITION.CARTESIAN_PRODUCT", "Cartesian product : match all rows of the left dataset with all rows of the right dataset");
                case 'ADVANCED': return translate("JOIN_RECIPE.JOIN_DESCRITION.ADVANCED_JOIN", "Custom options for rows selection and deduplication");
            }
        };

        const conditionsNumOperand =[
            {label: "=", value: "EQ", enableIfCanNonEquiJoin: true, description: translate("JOIN_RECIPE.JOIN.CONDITIONS_MODAL.OPERATORS.EQUALITY", "Equality")},
            {label: "~", value: "WITHIN_RANGE", enableIfCanNonEquiJoin: false, description:translate("JOIN_RECIPE.JOIN.CONDITIONS_MODAL.OPERATORS.MATCH_ALL", "Match all values within range")},
            {label: "~", value: "K_NEAREST", enableIfCanNonEquiJoin: false, description:translate("JOIN_RECIPE.JOIN.CONDITIONS_MODAL.OPERATORS.MATCH_NEAREST", "Match the nearest value(s)")},
            {label: "<", value: "LT", enableIfCanNonEquiJoin: true, description:translate("JOIN_RECIPE.JOIN.CONDITIONS_MODAL.OPERATORS.STRICTLY_INFERIOR", "Strictly inferior")},
            {label: "<=", value: "LTE", enableIfCanNonEquiJoin: true, description:translate("JOIN_RECIPE.JOIN.CONDITIONS_MODAL.OPERATORS.INFERIOR_EQUAL", "Inferior or equal")},
            {label: ">", value: "GT", enableIfCanNonEquiJoin: true, description:translate("JOIN_RECIPE.JOIN.CONDITIONS_MODAL.OPERATORS.STRICTLY_SUPERIOR", "Strictly superior")},
            {label: ">=", value: "GTE", enableIfCanNonEquiJoin: true, description:translate("JOIN_RECIPE.JOIN.CONDITIONS_MODAL.OPERATORS.SUPERIOR_EQUAL", "Superior or equal")},
            {label: "!=", value: "NE", enableIfCanNonEquiJoin: true, description:translate("JOIN_RECIPE.JOIN.CONDITIONS_MODAL.OPERATORS.DIFFERENT", "Different")},
        ];

        const conditionsDateOperand =[
            {label: "=", value: "EQ", enableIfCanNonEquiJoin: true, description: translate("JOIN_RECIPE.JOIN.CONDITIONS_MODAL.OPERATORS.EQUALITY", "Equality")},
            {label: "~", value: "WITHIN_RANGE", enableIfCanNonEquiJoin: false, description:translate("JOIN_RECIPE.JOIN.CONDITIONS_MODAL.OPERATORS.MATCH_ALL_DATES", "Match all dates within range")},
            {label: "~", value: "K_NEAREST", enableIfCanNonEquiJoin: false, description:translate("JOIN_RECIPE.JOIN.CONDITIONS_MODAL.OPERATORS.MATCH_NEAREST_DATES", "Match the nearest date(s)")},
            {label: "<", value: "LT", enableIfCanNonEquiJoin: true, description:translate("JOIN_RECIPE.JOIN.CONDITIONS_MODAL.OPERATORS.STRICTLY_BEFORE", "Strictly before")},
            {label: "<=", value: "LTE", enableIfCanNonEquiJoin: true, description:translate("JOIN_RECIPE.JOIN.CONDITIONS_MODAL.OPERATORS.BEFORE_EQUAL", "Before or equal")},
            {label: ">", value: "GT", enableIfCanNonEquiJoin: true, description:translate("JOIN_RECIPE.JOIN.CONDITIONS_MODAL.OPERATORS.STRICTLY_AFTER", "Strictly after")},
            {label: ">=", value: "GTE", enableIfCanNonEquiJoin: true, description:translate("JOIN_RECIPE.JOIN.CONDITIONS_MODAL.OPERATORS.AFTER_EQUAL", "After or equal")},
            {label: "!=", value: "NE", enableIfCanNonEquiJoin: true, description:translate("JOIN_RECIPE.JOIN.CONDITIONS_MODAL.OPERATORS.DIFFERENT", "Different")},
        ];

        const conditionsStringOperand =[
            {label: "=", value: "EQ", enableIfCanNonEquiJoin: true, description: translate("JOIN_RECIPE.JOIN.CONDITIONS_MODAL.OPERATORS.EQUALITY", "Equality")},
            {label: "~", value: "CONTAINS", enableIfCanNonEquiJoin: true, description:translate("JOIN_RECIPE.JOIN.CONDITIONS_MODAL.OPERATORS.CONTAINS", "Contains")},
            {label: "~", value: "STARTS_WITH", enableIfCanNonEquiJoin: true, description:translate("JOIN_RECIPE.JOIN.CONDITIONS_MODAL.OPERATORS.STARTS_WITH", "Starts with")},
            {label: "<", value: "LT", enableIfCanNonEquiJoin: true, description:translate("JOIN_RECIPE.JOIN.CONDITIONS_MODAL.OPERATORS.STRICTLY_BEFORE_AZ", "Is strictly before (A-Z)")},
            {label: "<=", value: "LTE", enableIfCanNonEquiJoin: true, description:translate("JOIN_RECIPE.JOIN.CONDITIONS_MODAL.OPERATORS.BEFORE_EQUAL_AZ", "Is before or equal (A-Z)")},
            {label: ">", value: "GT", enableIfCanNonEquiJoin: true, description:translate("JOIN_RECIPE.JOIN.CONDITIONS_MODAL.OPERATORS.STRICTLY_AFTER_AZ", "Is strictly after (A-Z)")},
            {label: ">=", value: "GTE", enableIfCanNonEquiJoin: true, description:translate("JOIN_RECIPE.JOIN.CONDITIONS_MODAL.OPERATORS.AFTER_EQUAL_AZ", "Is after or equal (A-Z)")},
            {label: "!=", value: "NE", enableIfCanNonEquiJoin: true, description:translate("JOIN_RECIPE.JOIN.CONDITIONS_MODAL.OPERATORS.IS_DIFFERENT", "Is different")},
        ];

        $scope.outputColumnsSelectionModes = [
            ['AUTO_NON_CONFLICTING', translate("JOIN_RECIPE.SELECTED_COLUMNS.SELECT_ALL_NON_CONFLICTING", "Select all non-conflicting columns")],
            ['ALL', translate("JOIN_RECIPE.JOIN.SELECTED_COLUMNS.SELECT_ALL", "Select all columns")],
            ['MANUAL', translate("JOIN_RECIPE.JOIN.SELECTED_COLUMNS.MANUALLY_SELECT", "Manually select columns")]
        ];

        $scope.outputColumnsSelectionModeDesc = [
            translate("JOIN_RECIPE.JOIN.SELECTED_COLUMNS.SELECT_ALL_NON_CONFLICTING_DESCRIPTION", "Select all columns that don’t conflict with a similarly named column from an earlier dataset in the join."),
            translate("JOIN_RECIPE.JOIN.SELECTED_COLUMNS.SELECT_ALL_DESCRIPTION", "Select all columns from this dataset. Conflicts may occur, and can be mitigated with a prefix."),
            translate("JOIN_RECIPE.JOIN.SELECTED_COLUMNS.MANUALLY_SELECT_DESCRIPTION", "Explicitly select columns.")
        ];

        function updateDescription(list){
            return list.map(element => {
                const updatedElement = {...element};
                updatedElement.description = 
                    !(element.enableIfCanNonEquiJoin || $scope.canNonEquiJoin())
                        ? 'Not available in the database'
                        : updatedElement.description;
                return updatedElement;
            });
        }

        $scope.$watch('recipeStatus.selectedEngine.canNonEquiJoin', () => {
            $scope.conditionsNumOperand = updateDescription(conditionsNumOperand);
            $scope.conditionsDateOperand = updateDescription(conditionsDateOperand);
            $scope.conditionsStringOperand = updateDescription(conditionsStringOperand);
        })

        $scope.getOperandDescriptionWithoutFilter = function(possibleSelection) {
            return possibleSelection
                .map(x => x.description);
        }


        $scope.isConditionExpanded = (condition, current) => {
            return current && current.condition && current.condition === condition;
        };

        $scope.getJoinColumns = function (virtualInputIndex){
            // to be overridden in child controllers if needed
            return $scope.getColumnsWithComputed(virtualInputIndex);
        }

        $scope.getColumnType = function (datasetVirtualIndex, columnName) {
            const col = $scope.getJoinColumns(datasetVirtualIndex).find(col => col.name === columnName);
            return col && col.type; // col might be undefined if input schema changed & un-existing column is still selected
        }

        $scope.joinColumnsExist = function (join){
            return $scope.getJoinColumns(join.table1).length > 0 &&
            $scope.getJoinColumns(join.table2).length > 0;
        }

        $scope.getColumnsWithComputed = function(virtualInputIndex) {
            if (!$scope.uiState.columnsWithComputed || !$scope.uiState.columnsWithComputed[virtualInputIndex]) {
                const datasetName = $scope.getDatasetName(virtualInputIndex);
                const columns = angular.copy($scope.getColumns(datasetName));
                const hasVirtualInputs = $scope.params && $scope.params.virtualInputs;
                if (hasVirtualInputs) {
                    const virtualInput = $scope.params.virtualInputs[virtualInputIndex];
                    if (virtualInput) {
                        const computedColumns = virtualInput.computedColumns;
                        if (computedColumns) {
                            for (let i = 0; i < computedColumns.length; i++) {
                                columns.push({
                                    name: computedColumns[i].name,
                                    type: computedColumns[i].type,
                                    timestampNoTzAsDate: false,
                                    maxLength: -1
                                });
                            }
                        }
                    }
                }
                $scope.uiState.columnsWithComputed = $scope.uiState.columnsWithComputed || {};
                $scope.uiState.columnsWithComputed[virtualInputIndex] = columns;
            }
            return $scope.uiState.columnsWithComputed[virtualInputIndex];
        };

        $scope.getColumnWithComputed = function(datasetVirtualIndex, name) {
            return $scope.getColumnsWithComputed(datasetVirtualIndex).filter(function(col){return col.name===name})[0];
        };

        /* callback given to the computed columns module */
        $scope.onComputedColumnListUpdate = computedColumnListUpdated;

        var savePayloadAsIsForDirtyness = true;
        $scope.hooks.getPayloadData = function () {
            if (!$scope.params) return;
            if (savePayloadAsIsForDirtyness) {
                savePayloadAsIsForDirtyness = false;
            } else {
                $scope.params.selectedColumns = $scope.getSelectedColumns();
            }
            // cleanup : - null values for alias
            var clean = angular.copy($scope.params);
            (clean.selectedColumns || []).forEach(function(c) {if (c.alias == null) delete c.alias;});
            return angular.toJson(clean);
        };

        var applyEngineLimitations = function(){
            if ($scope.params.joins) {
                var eng = $scope.recipeStatus.selectedEngine;
                if (eng != null && eng.canDeduplicateJoinMatches === false) {
                    $scope.params.joins.forEach(function(join){
                        if (join.rightLimit != null && join.rightLimit.enabled) {
                            Logger.warn("Deactivate rightLimit (deduplicate join matches) because of engine");
                            join.rightLimit.enabled = false;
                        }
                    });
                }
                $scope.params.joins.forEach(function(join){
                    if (join.rightLimit != null && join.rightLimit.enabled && $scope.hasNonSymmetricConditions(join)) {
                        Logger.warn("Deactivate rightLimit because (deduplicate join matches) of non equi-join");
                        join.rightLimit.enabled = false;
                    }
                });
            }
        };

        var removeUnusedInputs = function() {
            if (!$scope.params.virtualInputs) return;
            var usedIndices = [];
            $scope.params.virtualInputs.forEach(function(vi){
                if (usedIndices.indexOf(vi.index) < 0) {
                    usedIndices.push(vi.index);
                }
            });
            var newIndices = {};
            for(var i=0;i<$scope.recipe.inputs.main.items.length;i++) {
                newIndices[i] = usedIndices.filter(function(k) {return k < i;}).length;
            }
            $scope.recipe.inputs.main.items = $scope.recipe.inputs.main.items.filter(function(input, idx) {
                return usedIndices.indexOf(idx) >= 0;
            });
            $scope.params.virtualInputs.forEach(function(vi) {vi.index = newIndices[vi.index];});
        };

        $scope.onInputReplaced = function(replacement, virtualIndex) {
            var inputNames = RecipesUtils.getInputsForRole($scope.recipe, "main").map(function(input){return input.ref});
            $scope.params.virtualInputs[virtualIndex] = {
                ...$scope.params.virtualInputs[virtualIndex],
                index: inputNames.indexOf(replacement.name),
                originLabel: replacement.name
            };
            removeUnusedInputs();
            DatasetUtils.updateRecipeComputables($scope, $scope.recipe, $stateParams.projectKey, contextProjectKey)
                    .then(_ => $scope.resyncSchemas());
        }

        $scope.getAvailableReplacementDatasets = function(datasets) {
            return DatasetUtils.setInputDatasetsUsability(datasets, $scope.recipe, $scope.outputDatasetName);
        }

        var updateCodeMirrorUI = function(){
            $('.CodeMirror').each(function(idx, el){
                el.CodeMirror.refresh();
            });
        };

        $scope.hooks.updateRecipeStatus = function(forceUpdate, exactPlan) {
            var payload = $scope.hooks.getPayloadData();
            if (!payload) return $q.reject("payload not ready");
            var deferred = $q.defer();
            $scope.updateRecipeStatusBase(forceUpdate, payload, {reallyNeedsExecutionPlan: exactPlan, exactPlan: exactPlan}).then(function() {
                // $scope.recipeStatus should have been set by updateRecipeStatusBase
                if (!$scope.recipeStatus) return deferred.reject();
                if ($scope.recipeStatus.outputSchema) {
                    $scope.params.postFilter = $scope.params.postFilter || {};
                    $scope.params.postFilter.$status = $scope.params.postFilter.$status || {};
                    $scope.params.postFilter.$status.schema = $scope.recipeStatus.outputSchema;
                }

                // Build the multi-output infos for the output step UI
                // Only do it if sqlWithExecutionPlanList is undefined as recipe status may be cached, and $scope.recipeStatus.sql may have been overridden by the sql of the output selected in the output tab.
                // we don't want to override the main output sql with it, and only initialize it if it's not a fresh server result
                if (!$scope.recipeStatus.sqlWithExecutionPlanList) {
                    $scope.recipeStatus.sqlWithExecutionPlanList = [{
                        outputName: $scope.outputDatasetName + translate("JOIN_RECIPE.OUTPUT.MAIN_JOIN_OUTPUT", " - (main join output)"),
                        sql: $scope.recipeStatus.sql,
                        executionPlan: $scope.recipeStatus.executionPlan,
                        schema: $scope.recipeStatus.outputSchema,
                    }];
                    if ($scope.recipeStatus.leftUnmatchedStatus && $scope.recipeStatus.leftUnmatchedStatus.sql) {
                        $scope.recipeStatus.sqlWithExecutionPlanList.push({
                            ...$scope.recipeStatus.leftUnmatchedStatus,
                            outputName: unmatchedJoin.getCurrentOutputForRole(unmatchedJoin.ROLE_UNMATCHED_LEFT) + translate("JOIN_RECIPE.OUTPUT.LEFT_UNMATCHED_DATA", " - (left unmatched data)"), 
                        });
                    }
                    if ($scope.recipeStatus.rightUnmatchedStatus && $scope.recipeStatus.rightUnmatchedStatus.sql) {
                        $scope.recipeStatus.sqlWithExecutionPlanList.push({
                            ...$scope.recipeStatus.rightUnmatchedStatus,
                            outputName: unmatchedJoin.getCurrentOutputForRole(unmatchedJoin.ROLE_UNMATCHED_RIGHT) + translate("JOIN_RECIPE.OUTPUT.RIGHT_UNMATCHED_DATA", " - (right unmatched data)"), 
                        });
                    }
                }
                // this will update the infos and unselect the selected unmatched output if it is removed
                $scope.selectOutputForSql($scope.selectedOutputName);

                applyEngineLimitations();
                DKUtils.reflowLater();
                $timeout(updateCodeMirrorUI);
                deferred.resolve($scope.recipeStatus);
            });
            return deferred.promise;
        };
        $scope.getSuggestionsFunction = DataikuAPI.flow.recipes.join.getSuggestions;

        $scope.getJoinSuggestions = function() {
            var payload = $scope.hooks.getPayloadData();
            var recipeSerialized = angular.copy($scope.recipe);
            PartitionDeps.prepareRecipeForSerialize(recipeSerialized);
            return $scope.getSuggestionsFunction($stateParams.projectKey, recipeSerialized, payload)
                .success(function(suggestions) {
                    var lastJoin = $scope.params.joins[$scope.params.joins.length - 1];
                    if (suggestions.length > 0 ) {
                        // select everything available, let the user clean up later if he wants to
                        suggestions.forEach(function(condition) {condition.selected = true;});
                        $scope.addConditions(lastJoin, suggestions);
                        $scope.hooks.updateRecipeStatus();
                    }
                })
                .error(setErrorInScope.bind($scope));
        };

        $scope.canNonEquiJoin = function() {
            return $scope.recipeStatus && $scope.recipeStatus.selectedEngine != null && $scope.recipeStatus.selectedEngine.canNonEquiJoin;
        };

        $scope.getTitleForJoinModal = function(join) {
            if (join && join.table1Index != null) {
                return translate("JOIN_RECIPE.DATASET.ADD_DATASET_MODAL.ADD_A_DATASET", "Add a dataset to join with " + $scope.getDatasetName(join.table1Index), { dataset: $scope.getDatasetName(join.table1Index)});
            } else {
                return translate("JOIN_RECIPE.DATASET.ADD_DATASET_MODAL.ADD_INPUT_DATASET", "Add an input dataset");
            }
        }

        $scope.showNewJoinModal = function(index) {
            $scope.newDatasetIndex = index;
            CreateModalFromTemplate("/templates/recipes/visual-recipes-fragments/join-modal.html", $scope);
        };

        $scope.showJoinEditModal = function(join, tab) {
            //check if the modal is already shown
            if (!$('#join-condition-modal').parent().hasClass('in') && join.type !== 'CROSS') {
                var newScope = $scope.$new();
                newScope.join = join;
                newScope.current = {};
                newScope.current.tab = tab || 'conditions'
                newScope.current.condition = null; //no selected condition when the modal is created
                newScope.showConditionRemove = true;

                CreateModalFromTemplate("/templates/recipes/visual-recipes-fragments/join-edit-modal.html", newScope, $scope.joinRecipeType === 'FUZZY' ? "FuzzyJoinEditController" : "JoinEditController", (scope, el) => {
                    $timeout(() => {
                            scope.joinBlockBodyEl = el[0].getElementsByClassName('modal-body')[0];
                        }
                    );
                });
            }
        };

        $scope.getSelectedColumns = function() {
            var outputSchema = [];
            if (!$scope.uiState || !$scope.uiState.selectedColumns) return;
            $scope.uiState.selectedColumns.forEach(function(datasetColumns, tableIndex) {
                datasetColumns.forEach(function(column) {
                    if(column.selected) {
                        outputSchema.push({
                            name: column.name,
                            alias: column.alias,
                            table: tableIndex,
                            type: column.type
                        });
                    }
                })
            });
            return outputSchema;
        };

        $scope.addDataset = function(datasetName) {
            if (RecipesUtils.getInput($scope.recipe, "main", datasetName) == null) {
                RecipesUtils.addInput($scope.recipe, "main", datasetName);
            }
            const inputNames = RecipesUtils.getInputsForRole($scope.recipe, "main").map(function(input){return input.ref});
            const inputDesc = {
                index: inputNames.indexOf(datasetName),
                outputColumnsSelectionMode: "AUTO_NON_CONFLICTING",
                preFilter: {}
            };
            $scope.params.virtualInputs.push(inputDesc);
            $scope.uiState.currentStep = 'join';
        }

        $scope.autoSelectNonConflictingColumnsForManualMode = function(virtualInputIndex) {
            const inputDesc = $scope.params.virtualInputs[virtualInputIndex];
            $scope.uiState.columnsWithComputed = undefined;
            //Auto select columns that do not conflict
            $scope.params.selectedColumns = $scope.params.selectedColumns || [];
            var selectedNames = $scope.params.selectedColumns.map(function(col){
                return $scope.getColumnOutputName(inputDesc, col).toLowerCase();
            });

            var selectColumn = function(columnName) {
                $scope.params.selectedColumns.push({
                    name: columnName,
                    table: $scope.params.virtualInputs.length-1
                });
            }

            var datasetName = $scope.outputDatasetName;
            var excludedColumnNames = [];
            if ($scope.computablesMap && datasetName && $scope.computablesMap[datasetName] && $scope.computablesMap[datasetName].dataset) {
                var dataset = $scope.computablesMap[datasetName].dataset;
                if (dataset.type == 'HDFS') {
                    if (dataset.partitioning && dataset.partitioning.dimensions.length > 0) {
                        dataset.partitioning.dimensions.forEach(function(p) {excludedColumnNames.push(p.name);});
                    }
                }
            }

            $scope.getColumnsWithComputed(virtualInputIndex).forEach(function(column) {
                var outputName = $scope.getColumnOutputName(inputDesc, {
                    name: column.name
                });
                if (selectedNames.indexOf(outputName.toLowerCase()) < 0 && !excludedColumnNames.includes(column.name)) { //no conflict
                    selectColumn(column.name);
                }
            });

            createColumnList();
        }

        var removeDatasets = function(indices) {
            indices.sort().reverse();
            indices.forEach(function(index) {
                $scope.params.joins = $scope.params.joins.filter(function(join) {
                    return join.table1 != index && join.table2 != index;
                });
                $scope.params.joins.forEach(function(join) {
                    if (join.table1 > index) {
                        join.table1--;
                        join.on.forEach(function(condition){
                            condition.column1.table--;
                        })
                    }
                    if (join.table2 > index) {
                        join.table2--;
                        join.on.forEach(function(condition){
                            condition.column2.table--;
                        })
                    }
                });
                $scope.params.selectedColumns = $scope.params.selectedColumns.filter(function(column) {
                    return column.table != index;
                });
                $scope.params.selectedColumns.forEach(function(column) {
                    if(column.table > index) {
                        column.table--;
                    }
                });

                var datasetName = $scope.getDatasetNameFromRecipeInputIndex($scope.params.virtualInputs[index].index);
                var numberOfUses = $scope.params.virtualInputs.filter(function(table) {
                    return table.name == datasetName;
                }).length;
                if (numberOfUses == 1) {
                    RecipesUtils.removeInput($scope.recipe, "main", datasetName);
                    $scope.params.virtualInputs.forEach(function(vi) {
                        if (vi.index > index) {
                            vi.index--;
                        }
                    });
                }

                $scope.params.virtualInputs.splice(index, 1);
                $scope.uiState.selectedColumns.splice(index, 1);
            });

            $scope.hooks.updateRecipeStatus();

            if ($scope.params.virtualInputs.length == 0) {
                $scope.showNewJoinModal();
            }
        };

        var getDependantDatasets = function(index) {
            var dependantDatasets = [];
            for(var i = 0; i < $scope.params.joins.length; i++) {
                var join = $scope.params.joins[i];
                if (join.table1 == index) {
                    dependantDatasets.push(join.table2);
                    dependantDatasets = dependantDatasets.concat(getDependantDatasets(join.table2))
                }
            }
            return dependantDatasets;
        };

        $scope.removeDataset = function(index) {
            var datasetsToBeRemoved = getDependantDatasets(index);
            datasetsToBeRemoved.push(index);
            if (datasetsToBeRemoved.length == 1) {
                removeDatasets(datasetsToBeRemoved);
            } else {
                var datasetList = datasetsToBeRemoved.map(function(index) {
                    return $scope.getDatasetNameFromRecipeInputIndex($scope.params.virtualInputs[index].index);
                })
                Dialogs.confirm($scope,
                    translate("JOIN_RECIPE.JOIN.REMOVE_DATASETS_MODAL.REMOVE_DATASETS", "Remove datasets"),
                    translate("JOIN_RECIPE.JOIN.REMOVE_DATASETS_MODAL.FOLLOWING_DATASETS_REMOVED", "The following datasets will be removed from the recipe:")+
                    '<ul><li>'+datasetList.join('</li><li>')+'</li></ul>'
                )
                .then(function() {
                     removeDatasets(datasetsToBeRemoved);
                });
            }
        };

        // gets the dataset name from the index within the virtual inputs
        $scope.getDatasetName = function(virtualIndex) {
            var dataset = $scope.params.virtualInputs[virtualIndex];
            return $scope.getDatasetNameFromRecipeInputIndex(dataset.index);
        };

        // gets the dataset name from the index within the recipe's inputs
        $scope.getDatasetNameFromRecipeInputIndex = function(index) {
            var input = $scope.recipe.inputs.main.items[index];
            return input ? input.ref : "";
        };

        var createColumnList = function() {
            var selectedColumns = (($scope.params || {}).virtualInputs || []).map(function() {return {};});
            if ($scope.params.selectedColumns) {
                $scope.params.selectedColumns.forEach(function(column) {
                    selectedColumns[column.table][column.name] = column.alias || null;
                });
            }

            var columnList = [];
            selectedColumns.forEach(function(selectedColumn, index) {
                const inputColumns = $scope.getColumnsWithComputed(index).map(function(column) {
                    var alias = selectedColumn[column.name];
                    return {
                        name: column.name,
                        type: column.type,
                        maxLength: column.maxLength,
                        selected: alias !== undefined,
                        alias: alias
                    }
                });
                columnList.push(inputColumns);
            });
            $scope.uiState.selectedColumns = columnList;
        };

        $scope.resyncSchemas = function() {
            $scope.uiState.columnsWithComputed = undefined;
            $scope.uiState.joinableColumns = undefined;
            // regenerated selected columns (drop columns that don't exist anymore)
            createColumnList();

            // remove join conditions if the columns do not exist anymore
            $scope.params.joins.forEach(function(join){
                const columns1 = ($scope.getColumnsWithComputed(join.table1)||[]).map(function(col){return col.name});
                const columns2 = ($scope.getColumnsWithComputed(join.table2)||[]).map(function(col){return col.name});
                join.on = join.on.filter(function(cond){
                    if (!columns1 || !columns2 || columns1.indexOf(cond.column1.name) < 0 || columns2.indexOf(cond.column2.name) < 0) {
                        return false;
                    }
                    return true;
                })
            });
        };

        $scope.getColumnList = function(index) {
            return $scope.uiState.selectedColumns[index];
        };

        $scope.getColumnOutputName = function(inputDesc, column) { //TODO compute on server
            if (column.alias) {
                return column.alias;
            } else if (inputDesc.prefix) {
                return inputDesc.prefix + '_' + column.name;
            } else {
                return column.name;
            }
        };

        $scope.hasNonSymmetricConditions = function(join) {
            if (!join || !join.on) {
                return false;
            }
            var asymetricConditions = ['K_NEAREST', 'K_NEAREST_INFERIOR'];
            for (var i = 0; i < join.on.length; ++i) {
                if (asymetricConditions.indexOf(join.on[i].type) >= 0) {
                    return true;
                }
            }
            return false;
        };

        $scope.addEmptyCondition = function(join, current) {
            var newCondition = {
                column1: {
                    table: join.table1,
                    name: $scope.getColumnsWithComputed(join.table1)[0].name
                },
                column2: {
                    table: join.table2,
                    name: $scope.getColumnsWithComputed(join.table2)[0].name
                },
                type: 'EQ'
            };
            join.on = join.on || [];
            join.on.push(newCondition);
            if (current) {
                current.condition = newCondition;
            }
        };

        $scope.addConditions = function(join, conditions) {
            conditions.forEach(function(condition){
                if (condition.selected) {
                    delete condition.selected;
                    join.on.push(condition);
                }
            });
            $scope.updateRecipeStatusLater(0);
        };

        $scope.removeCondition = function(scope, join, condition) {
            if ( scope.current != null && scope.current.condition == condition ) {
                scope.current.condition = null;
            }
            var index = join.on.indexOf(condition);
            join.on.splice(index, 1);
            $scope.hooks.updateRecipeStatus();
        };

        $scope.removeAllConditions = function(scope, join) {
            if ( scope.current != null ) {
                scope.current.condition = null;
            }
            join.on = [];
            $scope.hooks.updateRecipeStatus();
        };

        $scope.range = function(n) {
            return Array.apply(null, Array(n)).map(function(_, i) {return i;});
        };

        $scope.getConditionString = function(condition) {
            var col1 = condition.column1.name,
                col2 = condition.column2.name,
                dataset1 = $scope.getDatasetNameFromRecipeInputIndex($scope.params.virtualInputs[condition.column1.table].index),
                dataset2 = $scope.getDatasetNameFromRecipeInputIndex($scope.params.virtualInputs[condition.column2.table].index);
            switch(condition.type) {
                case 'EQ':
                    return dataset1+'.'+col1+' = '+dataset2+'.'+col2;
                case 'WITHIN_RANGE':
                    return 'abs('+dataset2+'.'+col2+' - '+dataset1+'.'+col1+') < '+condition.maxDistance;
                case 'K_NEAREST':
                    return dataset2+'.'+col2+' is the nearest match for '+dataset1+'.'+col1+(condition.strict ? '(strict)' : '');
                case 'K_NEAREST_INFERIOR':
                    return dataset2+'.'+col2+' is the nearest match before '+dataset1+'.'+col1+(condition.strict ? '(strict)' : '');
                case 'CONTAINS':
                    return dataset1+'.'+col1+' contains '+dataset2+'.'+col2;
                case 'STARTS_WITH':
                    return dataset1+'.'+col1+' contains '+dataset2+'.'+col2;
                case 'LTE':
                    return dataset1+'.'+col1+' is before '+dataset2+'.'+col2;
                case 'GTE':
                    return dataset1+'.'+col1+' is after '+dataset2+'.'+col2;
                case 'NE':
                    return dataset1+'.'+col1+' different from '+dataset2+'.'+col2;
            }
        };

        $scope.onFilterUpdate = function(filterDesc) {
            $scope.updateRecipeStatusLater();
        };

        $scope.listColumnsForCumstomColumnsEditor = function(){
            return $scope.getSelectedColumns().map(function(c){
                const inputDesc = $scope.params.virtualInputs[c.table];
                return $scope.getColumnOutputName(inputDesc, c);
            });
        };

        $scope.$watch("params.postFilter.expression", $scope.updateRecipeStatusLater);
        $scope.$watch("params.postFilter.enabled", $scope.updateRecipeStatusLater);
        $scope.$watch("params.virtualInputs", $scope.updateRecipeStatusLater, true);

        $scope.isRelativeDistance = function (condition) {
            return angular.isNumber(condition.fuzzyMatchDesc.relativeTo);
        };

        $scope.getMatchingTypeDescription = function(condition) {
            if ($scope.joinRecipeType === 'FUZZY') {
                return $scope.joinDistanceTypes[condition.fuzzyMatchDesc.distanceType];
            }
             else if ($scope.joinRecipeType === 'GEO') {
                if (['DWITHIN', 'BEYOND'].includes(condition.type)) {
                    return $scope.joinDistanceTypes[condition.type];
                } else {
                    return null;
                }
            }
        }

        $scope.getMatchingTypeSymbol = function(condition) {
            function round(threshold, multiplier) {
                return Math.round((threshold * (multiplier || 1)) * 10 ** 12) / 10 ** 12;
            }

            if ($scope.joinRecipeType === 'FUZZY') {
                const fuzzyMatchDesc = condition.fuzzyMatchDesc;
                let threshold = fuzzyMatchDesc.threshold;
                if (!fuzzyMatchDesc || !angular.isNumber(threshold)) return '?';
                if (fuzzyMatchDesc.distanceType === 'EXACT') return '=';
                if ($scope.isRelativeDistance(condition)) {
//                  Rounding to avoid long decimal tail after floating point math operations e.g. 0.072*100=7.199999999999999
                    return `${round(threshold, 100)} %`;
                } else {
                    return threshold.toString();
                }
            } else if ($scope.joinRecipeType === 'GEO') {
                if (condition.type === "DWITHIN" || condition.type === "BEYOND") {
                    return `${round(condition.threshold)} ${$scope.distanceUnits[condition.unit].simbol}`
                } else {
                    return $scope.joinDistanceTypes[condition.type].toLowerCase();
                }
            } else {
                return JoinDisplayNamesService.getMatchingTypeName(condition.type);
            }
        };

        $scope.getDatasetColorClass = function(datasetIndex) {
            return 'dataset-color-'+(datasetIndex%6);
        };

        function onScriptChanged(nv, ov) {
            if (nv) {
                $scope.params = JSON.parse($scope.script.data);
                $scope.params.computedColumns = $scope.params.computedColumns || [];
                $scope.uiState.computedColumns = angular.copy($scope.params.computedColumns);
                $scope.uiState.columnsWithComputed = undefined;
                savePayloadAsIsForDirtyness = true;
                visualCtrl.saveServerParams(); //keep for dirtyness detection
                createColumnList();
                $scope.hooks.updateRecipeStatus();
                DKUtils.reflowLater();
                let joins = $scope.params.joins;
                // Condition to know if the recipe is new, set default values in join conditions and selected columns if true
                if (joins && joins.length == 1 && joins[0].type == "LEFT" && joins[0].on.length == 0) {
                    $scope.getJoinSuggestions();
                    $scope.autoSelectNonConflictingColumnsForManualMode(joins[0].table2);
                }
            }
        };

        $scope.$watchCollection("recipe.outputs.main.items", function() {
            var outputs = RecipesUtils.getOutputsForRole($scope.recipe, "main");
            if (outputs.length == 1) {
                $scope.outputDatasetName = outputs[0].ref;
            }
            $scope.updateRecipeStatusLater();
        });

        $scope.$watchCollection("params.virtualInputs", removeUnusedInputs);
        $scope.$watch("params.virtualInputs", function() {
            $scope.uiState.columnsWithComputed = undefined;
            DatasetUtils.updateRecipeComputables($scope, $scope.recipe, $stateParams.projectKey, contextProjectKey)
                    .then(_ => createColumnList());
        }, true);

        $scope.hooks.onRecipeLoaded = function() {
            Logger.info("On Recipe Loaded");
            $scope.$watch("script.data", onScriptChanged);
            // the onScriptChanged will be called because adding a $watch on the scope triggers an 'initialization' run
        };

        $scope.specificControllerLoadedDeferred.resolve();

        $scope.params = $scope.params || {};
        $scope.enableAutoFixup();
        $scope.uiState = {
            currentStep: 'join',
            computedColumns: []
        };

        /* UNMATCHED JOIN CODE */
        const unmatchedJoin = {
            ROLE_UNMATCHED_LEFT: 'unmatchedLeft',
            ROLE_UNMATCHED_RIGHT: 'unmatchedRight',
            roles: ['unmatchedLeft', 'unmatchedRight'],
            unmatchedOutputModes: [{'id': 'DROP', 'label': translate("JOIN_RECIPE.JOIN.DROP_UNMATCHED", "Drop unmatched rows")}, {'id': 'SAVE', 'label': translate("JOIN_RECIPE.JOIN.SEND_UNMATCHED", "Send unmatched rows to other output dataset(s)")}],
            savedUnmatchedOutputs: {},
            selectUnmatchedOutputMode: (mode) => {
                switch (mode.id) {
                    case 'DROP':
                        unmatchedJoin.roles.forEach((role) => {
                            unmatchedJoin.saveAndClearUnmatchedOutputsForRole(role);

                        });
                        break;
                    case 'SAVE':
                        unmatchedJoin.restoreUnmatchedOutputs();
                        break
                    default:
                        throw 'Unsupported unmatch output mode ' + mode.id;
                }
            },
            saveAndClearUnmatchedOutputsForRole: (role) => {
                const currentOutput = unmatchedJoin.getCurrentOutputForRole(role);
                if (currentOutput) {
                    unmatchedJoin.savedUnmatchedOutputs[role] = currentOutput;
                }
                unmatchedJoin.setOutputForRole(role, undefined);
            },
            restoreUnmatchedOutputs: () => {
                 unmatchedJoin.roles.forEach((role) => {
                    if (unmatchedJoin.canUseRole(role)) {
                        const savedOutput = unmatchedJoin.savedUnmatchedOutputs[role];
                        if (savedOutput) {
                            const currentOutput = unmatchedJoin.getCurrentOutputForRole(role);
                            if (!currentOutput) {
                                unmatchedJoin.setOutputForRole(role, savedOutput);
                            }
                        }
                    } else {
                        unmatchedJoin.saveAndClearUnmatchedOutputsForRole(role);
                    }
                });
            },
            hasAnyUnmatchedOutput: () => {
                return unmatchedJoin.getCurrentOutputForRole(unmatchedJoin.ROLE_UNMATCHED_LEFT) !== undefined
                    || unmatchedJoin.getCurrentOutputForRole(unmatchedJoin.ROLE_UNMATCHED_RIGHT) !== undefined;
            },
            shouldShowUnmatchedOutputSection: () => {
                return $scope.joinRecipeType === 'REGULAR';
            },
            hasMultipleUnmatchedOutput: () => {
                return $scope.params.joins[0].type === 'INNER';
            },
            canUseUnmatchedOutputs: () => {
                return $scope.params.joins // ensure params are loaded
                    && $scope.params.joins.length === 1
                    && ['INNER', 'RIGHT', 'LEFT'].includes($scope.params.joins[0].type);
            },
            getShowUnmatchedDisabledReason: (join) => { // assumes canUseUnmatchedOutputs() is false
                if ($scope.params.joins.length > 1) {
                    if(join === $scope.params.joins[0]) return 'several-joins';
                    else return 'hidden';
                } else {
                    return 'not-supported-join-type';
                }
            },
            canUseRole(role) {
                if (role === unmatchedJoin.ROLE_UNMATCHED_LEFT) {
                    return $scope.params.joins && $scope.params.joins.length === 1 &&
                        ['INNER', 'RIGHT'].includes($scope.params.joins[0].type)
                } else {
                    return $scope.params.joins && $scope.params.joins.length === 1 &&
                        ['INNER', 'LEFT'].includes($scope.params.joins[0].type)
                }
            },
            getCurrentOutputForRole: (role) => { // may return undefined if no output is set
                const output = RecipesUtils.getOutputsForRole($scope.recipe, role)[0];
                return output && output.ref;
            },
            getCurrentOutputTypeForRole(role) { // may return undefined if no output is set
                const outputRef = unmatchedJoin.getCurrentOutputForRole(role)
                return outputRef && $scope.computablesMap[outputRef].datasetType;
            },
            getInputDatasetNameForRole: (role) => $scope.getDatasetName(
                role === unmatchedJoin.ROLE_UNMATCHED_LEFT ? $scope.params.joins[0].table1 : $scope.params.joins[0].table2
            ),
            setOutputForRole(role, ref) { // ref is allowed to be undefined to remove the output
                RecipesUtils.removeOutputsForRole($scope.recipe, role);
                if (ref) {
                    RecipesUtils.addOutput($scope.recipe, role, ref);
                }
                $scope.updateRecipeStatusLater();
            },
            openNewOutputModalForRole(role) {
                CreateModalFromTemplate("/templates/recipes/io/output-selection-modal.html", $scope, null, function (modalScope) {
                    $controller("_RecipeOutputNewManagedBehavior", {$scope: modalScope}); // adds some magic function required by the new output modal
                    modalScope.setErrorInTopScope = () => setErrorInScope.bind($scope); // _RecipeOutputNewManagedBehavior expects this to bubble up errors TODO display it somewhere
                    modalScope.formIsValid = () => { // used to decide if the validation button is enabled or not
                        return modalScope.io.newOutputTypeRadio == 'select' && modalScope.io && modalScope.io.existingOutputDataset
                            || modalScope.io.newOutputTypeRadio == 'create' && modalScope.newOutputDataset && modalScope.newOutputDataset.name && modalScope.isDatasetNameUnique(modalScope.newOutputDataset.name);
                    };

                    // init some values required by the modal
                    DatasetUtils.listDatasetsUsabilityInAndOut($stateParams.projectKey, $scope.recipe.type).then(function(data) {
                        const thisRecipeOutputs = RecipesUtils.getFlatOutputsList($scope.recipe).map(o => o.ref);
                        // we cannot trust the computable.alreadyUsedAsOutputOf for current recipe because there might be unsaved changes to outputs
                        const alreadyInThisRecipeOutput = (computable) => thisRecipeOutputs.includes(computable.smartName);
                        const alreadyInOtherRecipeOutput = (computable) => computable.alreadyUsedAsOutputOf && computable.alreadyUsedAsOutputOf !== $scope.recipe.name;
                        
                        return data[1].filter((computable) => computable.usableAsOutput[role].usable
                                && !alreadyInOtherRecipeOutput(computable)
                                && !alreadyInThisRecipeOutput(computable)
                        );
                    }).then(val => modalScope.availableOutputDatasets = val);
                    modalScope.getManagedDatasetOptions(role).then(modalScope.setupManagedDatasetOptions);
                    modalScope.singleOutputRole = {name: role, arity: "UNARY", acceptsDataset: true};

                    modalScope.ok = function(dismissModalCallback) {
                        return $q.resolve()
                        .then(() => {
                            if (modalScope.io.newOutputTypeRadio == 'select') {
                                return modalScope.io.existingOutputDataset;
                            } else {
                                // createAndUseNewOutputDataset function doesn't return a promise, but calls acceptEdit with the added dataset once everything is done
                                return $q((resolve) => {
                                    modalScope.acceptEdit = (dataset) => resolve(dataset.name);
                                    modalScope.createAndUseNewOutputDataset(false);
                                });
                            }
                        }).then((newOutputName) => {
                            unmatchedJoin.setOutputForRole(role, newOutputName)
                            dismissModalCallback();
                        });
                    };
                });

            },
            removeOutputForRole(role) {
                unmatchedJoin.savedUnmatchedOutputs[role] = undefined;
                unmatchedJoin.setOutputForRole(role, undefined);
            }
            
        }

        $scope.unmatchedJoin = unmatchedJoin;

        // auto-set unmatched mode and output(s) when recipe loads if there is any output defined
        $scope.$watch(() => $scope.params.joins, (nv) => {
            if(nv) {
                if (!unmatchedJoin.unmatchedOutputMode) {
                    const mode = unmatchedJoin.hasAnyUnmatchedOutput() ? 'SAVE' : 'DROP';
                    unmatchedJoin.unmatchedOutputMode = unmatchedJoin.unmatchedOutputModes.find(_ => _.id === mode);
                }
                unmatchedJoin.restoreUnmatchedOutputs();
            }
        });

        // auto-link / unlink unmatched outputs when user changes join type.
        // Called on join type change
        $scope.hooks.updateJoinUnmatchedOutputs = () => {
            unmatchedJoin.restoreUnmatchedOutputs();
        };

        function sendRecipeParamsToWT1() {
            $scope.tryRecipeWT1Event('regular-join-params', () => {
                function createFilterWT1Entry(filter) {
                    if (!filter){
                        return undefined;
                    }
                    return {
                        enabled: filter.enabled,
                        mode: filter.enabled ? filter.uiData && filter.uiData.mode : undefined,
                        distinct: filter.distinct,
                    };
                }

                const leftUnmatched = unmatchedJoin.getCurrentOutputForRole(unmatchedJoin.ROLE_UNMATCHED_LEFT);
                const rightUnmatched = unmatchedJoin.getCurrentOutputForRole(unmatchedJoin.ROLE_UNMATCHED_RIGHT);
                const engine = $scope.recipeStatus && $scope.recipeStatus.selectedEngine && $scope.recipeStatus.selectedEngine.label;
                const recipeData = {
                    "joins": $scope.params.joins.map(join => {
                        return {
                            "engine": engine,
                            "type": join.type,
                            "conditionsMode": join.conditionsMode,
                            "on": join.on.map(on => {
                                    const column1 = $scope.getColumnWithComputed(on.column1.table, on.column1.name);
                                    const column2 = $scope.getColumnWithComputed(on.column2.table, on.column2.name);
                                    return {
                                        conditionsMode: on.conditionsMode,
                                        type: on.type,
                                        column1: column1 ? column1.type : null,
                                        column2: column2 ? column2.type : null,
                                    };
                                }
                            )
                        }
                    }),
                    unmatchedOutput: {
                        left: {
                            enabled: leftUnmatched !== undefined,
                            output: leftUnmatched ? md5(leftUnmatched) : undefined
                        },
                        right: {
                            enabled: rightUnmatched !== undefined,
                            output: rightUnmatched ? md5(rightUnmatched) : undefined
                        }
                    },
                    "postFilter": createFilterWT1Entry($scope.params.postFilter),
                    "preFilters": $scope.params.virtualInputs.map(i => createFilterWT1Entry(i.preFilter))
                };
                return _.mapValues(recipeData, (x) => JSON.stringify(x));
            }, 'Failed to report join params');
        }

        $scope.isAntiJoin = function() {
            if ($scope.params.joins.length == 1) {
                const joinType = $scope.params.joins[0].type;
                if (joinType === 'LEFT_ANTI' || joinType === 'RIGHT_ANTI') {
                    return true;
                }
            }
            return false;
        };

        $scope.areInputColumnsSelectable = function(virtualIndex) {
            if (!$scope.isAntiJoin()) {
                return true;
            }
            const joinType = $scope.params.joins[0].type;
            return (joinType === 'LEFT_ANTI' && virtualIndex === 0) || (joinType === 'RIGHT_ANTI' && virtualIndex === 1);
        };

        $scope.getAntiJoinCannotSelectInfo = function(index) {
            const left = translate("JOIN_RECIPE.JOIN.SELECTED_COLUMNS.LEFT", "left");
            const right = translate("JOIN_RECIPE.JOIN.SELECTED_COLUMNS.RIGHT", "right");

            const first = index === 0 ? left : right;
            const second = index === 0 ? right : left;
            return translate("JOIN_RECIPE.JOIN.SELECTED_COLUMNS.CANNOT_SELECT_ANTI_JOIN", 
                "Cannot select columns from " + first + " dataset with a " + second + " anti join.",
                {firstDataset : first, secondDataset: second});
        }

    });

    app.component('joinTypeIcon', {
        bindings: {
            type: '<',
            leftTableIdx: '<',
            rightTableIdx: '<',
        },
        // do not break it in several lines of it will break the icons for some reason.
        template : `<i class="{{$ctrl.icon}}"><span class="path1" style="color: {{$ctrl.leftColor}};"></span><span class="path2" style="color: {{$ctrl.rightColor}};"></span><span class="path3"></span><span class="path4"></span><span class="path5"></span></i>
        `,
        controller: function() {
            const $ctrl = this;

            $ctrl.$onChanges = () => {
                $ctrl.icon = getIconType($ctrl.type);
                $ctrl.leftColor = getRealColor($ctrl.leftTableIdx);
                $ctrl.rightColor = getRealColor($ctrl.rightTableIdx);
            }
        }
    });

    app.controller("NewJoinController", function ($scope, $stateParams, DatasetUtils) {
        $scope.params.virtualInputs = $scope.params.virtualInputs || [];
        $scope.creation = !$scope.params.virtualInputs || !$scope.params.virtualInputs.length;
        $scope.newJoin = {
            table1Index: $scope.newDatasetIndex
        };

        $scope.joinIsValid = function() {
            return $scope.creation
                ? !!($scope.newJoin.dataset1 && $scope.newJoin.dataset2)
                : !! $scope.newJoin.dataset2;
        };

        $scope.addJoin = function() {
            if ($scope.creation) {
                $scope.newJoin.table1Index = 0;
                $scope.addDataset($scope.newJoin.dataset1);
            }

            let contextProjectKey = $scope.context && $scope.context.projectKey ? $scope.context.projectKey:$stateParams.projectKey;
            DatasetUtils.updateDatasetInComputablesMap($scope, $scope.newJoin.dataset2, $stateParams.projectKey, contextProjectKey)
            .then(() => {
                if (!$scope.dataset2IsValid($scope.newJoin.dataset2)) {
                    return;
                }
                $scope.newJoin.table2Index = $scope.params.virtualInputs.length;

                $scope.addDataset($scope.newJoin.dataset2);

                var join = {
                    table1: $scope.newJoin.table1Index,
                    table2: $scope.newJoin.table2Index,
                    type: 'LEFT',
                    conditionsMode : 'AND',
                    on: [],
                    outerJoinOnTheLeft: true, // just for ADVANCED join type
                    rightLimit: {}
                };
                $scope.params.joins = $scope.params.joins || [];
                $scope.params.joins.push(join);
                $scope.dismiss();
                $scope.getJoinSuggestions();

                $scope.autoSelectNonConflictingColumnsForManualMode($scope.newJoin.table2Index);
            });
        };

        $scope.dataset2IsValid = function(datasetName) {
            if (!datasetName) {
                return false;
            }
            const computable = $scope.computablesMap[datasetName];
            if (!computable) {
                $scope.error = 'Dataset '+datasetName+' does not seem to exist, try reloading the page.';
                return false;
            }
            if (!computable.dataset) {
                $scope.error = datasetName+' is not a dataset';
                return false;
            }
            if (!computable.dataset.schema || !computable.dataset.schema.columns.length) {
                $scope.error = 'Dataset '+datasetName+' has an empty schema';
                return false;
            }
            return true;
        };

        $scope.$on('$destroy', function() {
            $scope.updateRecipeStatusLater(0);
        });

        DatasetUtils.listDatasetsUsabilityInAndOut($stateParams.projectKey, "join").then(function (data) {
            $scope.availableInputDatasets = DatasetUtils.setInputDatasetsUsability(data[0], $scope.recipe, $scope.outputDatasetName);
        });
    });

    /*
    Controller for join edit modal
    */
    app.controller("JoinEditController", function ($scope, CodeMirrorSettingService, DatasetTypesService) {
        $scope.uiState = $scope.uiState || {};

        if (($scope.join.on.length == 0) && (!$scope.inFuzzy)){
            $scope.addEmptyCondition($scope.join);
            $scope.current.condition = $scope.join.on[0];
        }

        //TODO @join add $right, $left in autocompletion
        $scope.sqlEditorOptions = CodeMirrorSettingService.get('text/x-sql');
        $scope.sqlEditorOptions.autofocus = true;

        // getColumn: 1 or 2
        $scope.getColumn = function (condition, columnIdx) {
            const col = !columnIdx || columnIdx === 1 ? 'column1' : 'column2';
            return $scope.getColumnWithComputed(condition[col].table, condition[col].name);
        };

        $scope.hasStringOperand = function(condition, columnIdx) {
            const col = $scope.getColumn(condition, columnIdx);
            return col && col.type === 'string';
        };

        $scope.hasNumOperand = function(condition, columnIdx) {
            const col = $scope.getColumn(condition, columnIdx);
            return col && ['tinyint', 'smallint', 'int', 'bigint', 'float', 'double'].includes(col.type);
        };

        $scope.hasDateOperand = function(condition, columnIdx) {
            const col = $scope.getColumn(condition, columnIdx);
            condition.dateDiffUnit = condition.dateDiffUnit || "SECOND";
            return col && DatasetTypesService.isTemporalType(col.type);
        };

        $scope.hasGeoOperand = function(condition, columnIdx) {
            const col = $scope.getColumn(condition, columnIdx);
            return col && col.type === 'geopoint';
        };

        $scope.setJoinType = function(join, type) {
            join.type = type;
        };

        /* on operand change, make sure the condition type makes sense, if not fall back to = condition */
        var updateOperandType = function() {
            var condition = $scope.current.condition;
            if (!condition) {
                return;
            }
            var numOrDateJoinType = ['EQ', 'K_NEAREST', 'K_NEAREST_INFERIOR', 'WITHIN_RANGE', 'LTE', 'GTE', 'NE'];
            var stringJoinType = ['EQ', 'CONTAINS', 'STARTS_WITH', 'LTE', 'GTE', 'NE'];
            if (($scope.hasNumOperand(condition) || $scope.hasDateOperand(condition)) && numOrDateJoinType.indexOf(condition.type) < 0) {
                condition.type = 'EQ';
            } else if ($scope.hasStringOperand(condition) && stringJoinType.indexOf(condition.type) < 0) {
                condition.type = 'EQ';
            }
        };
        if ($scope.joinRecipeType !== 'FUZZY') {
            $scope.$watch('current.condition.column1.name', updateOperandType);
            $scope.$watch('current.condition.column2.name', updateOperandType);
        }
        $scope.$on('$destroy', function() {
            $scope.updateRecipeStatusLater(0);
        });
    });

    app = angular.module('dataiku.directives.widgets');

    app.directive('irregularJoinConditionSettings', function () {
        return {
            templateUrl: 'templates/recipes/fragments/join-condition-settings.html',
            link: function ($scope, element, attrs) {
                $scope.model = $scope.$eval(attrs['ngModel'])
                $scope.$watch('[current.condition.column1.name, current.condition.column2.name]', function (nv, ov) {
                    if (nv !== ov) {
                        $scope.guessDistanceType($scope.current.condition);
                    }
                });
                if ($scope.joinRecipeType === 'FUZZY') {
                    $scope.$watch('isRelativeDistance(current.condition)', function (nv, ov) {
                        if (angular.isDefined(nv) && angular.isDefined(ov) && nv !== ov) {
                            $scope.setInitialThreshold($scope.current.condition);
                        }
                    });
                }
            }
        };
    });

    /*
    this directive creates an element representing a join between two datasets
    */
    app.directive('joinBlock', function() {
        return {
            restrict: 'EA',
            scope: true,
            templateUrl: '/templates/recipes/fragments/join-block.html',
            link : function(scope, element, attrs) {
                scope.onConditionClicked = function (join, condition) {
                    if (attrs.onConditionClicked) {
                        if (!scope.current || scope.current.condition !== condition) {
                            var newScope = scope.$new();
                            newScope.join = join;
                            newScope.condition = condition;
                            newScope.$eval(attrs.onConditionClicked);
                        } else {
                            scope.current.condition = null;
                        }
                    }
                };
            }
        };
    });

    app.directive('joinBlockDropdownJoin', function(CreateModalFromTemplate, translate) {
        return {
            restrict: 'EA',
            scope: true,
            templateUrl: '/templates/recipes/fragments/join-block-dropdown-join.html',
            link : function(scope, element, attrs) {
                scope.getJoinTypes = function() {
                    if (scope.joinRecipeType === 'FUZZY') {
                        return ['LEFT', 'INNER', 'FULL', 'RIGHT'];
                    } else if (scope.joinRecipeType === 'GEO') {
                        return ['LEFT', 'INNER', 'FULL', 'RIGHT', 'CROSS'];
                    } else {
                        return ['LEFT', 'INNER', 'FULL', 'RIGHT', 'LEFT_ANTI', 'RIGHT_ANTI', 'CROSS', 'ADVANCED'];
                    }
                };
                scope.joinTypes = scope.getJoinTypes();

                scope.shouldDisableJoinType = (type) => {
                    switch (type) {
                        case 'FULL':
                            return scope.recipeStatus && scope.recipeStatus.selectedEngine && !scope.recipeStatus.selectedEngine.canFullOuterJoin;
                        case 'LEFT_ANTI':
                        case 'RIGHT_ANTI':
                            return scope.params && scope.params.joins && scope.params.joins.length > 1;
                        default:
                            return false;
                    }
                };
                
                scope.getDisabledText = (type) => {
                    switch (type) {
                        case 'FULL':
                            return translate("JOIN_RECIPE.JOIN.NOT_AVAILABLE_ENGINE", "Not available with this engine");
                        case 'LEFT_ANTI':
                        case 'RIGHT_ANTI':
                            return translate("JOIN_RECIPE.JOIN.NOT_AVAILABLE_MORE_JOIN", "Not available when performing more than one join");
                        default:
                            return '';
                    }
                };

                scope.setJoinType = function(join, type) {
                    join.type = type;
                    if (type === 'ADVANCED') {
                        scope.showAdvancedModal();
                    }
                    if(scope.hooks.updateJoinUnmatchedOutputs) {
                        scope.hooks.updateJoinUnmatchedOutputs();
                    }
                    scope.hooks.updateRecipeStatus();
                };

                scope.showAdvancedModal = function() {
                    // set settings to default if required
                    scope.join.rightLimit = scope.join.rightLimit || {decisionColumn: {}};
                    const rl = scope.join.rightLimit;
                    rl.maxMatches = rl.maxMatches === undefined || rl.maxMatches === null ? 1 : rl.maxMatches;
                    rl.enabled = (rl.enabled === true || rl.enabled === false) ? rl.enabled : false;
                    rl.type = rl.type || 'KEEP_LARGEST';

                    CreateModalFromTemplate("/templates/recipes/visual-recipes-fragments/join-advanced-modal.html", scope);
                }

                scope.updateDecisionColumn = function() {
                    if (scope.join.rightLimit) {
                        scope.join.rightLimit.decisionColumn = {
                            name: scope.uiState.decisionColumnName,
                            table: scope.join.table2,
                        };
                        scope.hooks.updateRecipeStatus();
                    }
                };

                scope.isSafari = function() {
                    var ua = navigator.userAgent.toLowerCase();
                    if (ua.indexOf('safari') != -1) {
                      if (ua.indexOf('chrome') > -1) {
                        return false;
                      } else {
                        return true; // Safari
                      }
                    } else {
                        return false;
                    }
                }

                scope.getCSSStyle = function(join) {
                    if (scope.isSafari()) {
                        return "";
                    } else {
                        return "background-image: linear-gradient(to right, "+ scope.getRealColor(join.table1) + " 50%,  " + scope.getRealColor(join.table2) + " 50%);" +
                                "background-clip: text;" +
                                "-webkit-background-clip: text;" +
                                "-moz-background-clip: text;" +
                                "-webkit-text-fill-color: transparent;" +
                                "color: transparent;" +
                                "display: inline;"
                    }
                };
            }
        }
    });
    app.directive('joinBlockEmpty', function() {
        return {
            restrict: 'EA',
            scope: true,
            templateUrl: '/templates/recipes/fragments/join-block-empty.html',
            link : function(scope, element, attrs) {
                scope.onConditionClicked = function (join, condition) {
                    if (attrs.onConditionClicked) {
                        if (!scope.current || scope.current.condition !== condition) {
                            var newScope = scope.$new();
                            newScope.join = join;
                            newScope.condition = condition;
                            newScope.$eval(attrs.onConditionClicked);
                        } else {
                            scope.current.condition = null;
                        }
                    }
                };
            }
        };
    });

    /*
    Widget to select columns from input dataset and edit their output names
    */
    app.directive('selectedColumnsEditor', function($timeout, translate) {
        return {
            restrict: 'EA',
            scope: true,
            link : function(scope, element, attrs) {
                var getColumns = function() {
                    return scope.$eval(attrs.columns);
                };

                scope.getExpectedFinalColumnName = function(name) {
                    if (scope.input && scope.input.prefix) {
                        return scope.input.prefix + '_' + name;
                    } else {
                        return name;
                    }
                }

                scope.editColumnAlias = function(columnIndex, column) {
                    column.$newName = column ? scope.getColumnOutputName(scope.input, column) : '';
                    scope.currentEditedColumn = column;
                    $timeout(function(){$('.alias-editor', element).get(columnIndex).focus();});
                };

                scope.validateColumnEdition = function() {
                    var col = scope.currentEditedColumn;
                    col.alias = col.$newName || null;
                    var expected = scope.getExpectedFinalColumnName(col ? col.name : '');
                    if (col && col.alias == expected) {
                        delete col.alias;
                    }
                    scope.currentEditedColumn = null;
                    scope.hooks.updateRecipeStatus();
                };

                scope.cancelColumnEdition = function() {
                    scope.currentEditedColumn = null;
                };

                scope.onBlurColumnEdition = function() {
                    if (!scope.currentEditedColumn) {
                        scope.cancelColumnEdition();
                    } else {
                        scope.validateColumnEdition();
                    }
                }

                scope.deleteColumnAlias = function(column) {
                    delete column.alias;
                    scope.currentEditedColumn = null;
                    scope.hooks.updateRecipeStatus();
                }

                scope.keyDownOnAliasBox = function(event) {
                   if (event.keyCode == 13){  // enter
                        scope.validateColumnEdition();
                   } else if (event.keyCode == 27){  // esc
                        scope.cancelColumnEdition();
                   }
                };

                scope.updateSelectAll = function() {
                    $.each(getColumns(), function(idx, column) {
                        column.selected = scope.selected.all;
                    });
                    scope.selected.any = scope.selected.all;
                    scope.hooks.updateRecipeStatus();
                };

                var updateGlobalSelectionStatus = function() {
                    var all = true, any = false;
                    $.each(getColumns(), function(idx, column) {
                        if (column.selected) {
                            any = true;
                        } else {
                            all = false;
                        }
                    });
                    scope.selected = {
                        all: all, any: any
                    };
                };

                scope.onSelectionChange = function() {
                    updateGlobalSelectionStatus();
                    scope.hooks.updateRecipeStatus();
                };

                scope.hasDuplicates = function (datasetIndex, column) {
                    if (!scope.recipeStatus || !scope.recipeStatus.selectedColumns || !scope.recipeStatus.selectedColumns.duplicates)
                        return false;
                    if (!column.selected)
                        return false;
                    var duplicates = scope.recipeStatus.selectedColumns.duplicates;
                    for (var i = 0; i < duplicates.length; ++i) {
                        var duplicate = duplicates[i];
                        if ((duplicate.dataset1 == datasetIndex && duplicate.column1 == column.name)
                            || (duplicate.dataset2 == datasetIndex && duplicate.column2 == column.name)) {
                            return true;
                        }
                    }
                    return false;
                };

                updateGlobalSelectionStatus();
            }
        };
    });
})();
