(function() {
    'use strict';

    angular.module('dataiku.charts').component('kpiHolder', {
        templateUrl: '/static/dataiku/js/simple_report/components/kpi-holder/kpi-holder.component.html',
        bindings: {
            chartDef: '<',
            theme: '<',
            data: '<',
            loadedCallback: '&'
        },

        controller: function($scope, $element, $timeout, ChartDataWrapperFactory, ChartFormatting, ChartLabels, ChartsStaticData, ConditionalFormattingOptions, CHART_TYPES) {
            const ctrl = this;
            const VALUE_SELECTOR = 'kpi-holder-measure__value';
            const LABEL_SELECTOR = 'kpi-holder-measure__label';
            const MEASURE_SELECTOR = 'kpi-holder-measure';
            const ROUNDING = 0.9;
            const VALUE_HEIGHT_OVERFLOW = 0.17;
            const MIN_FONT_BEFORE_HIDDING_LABEL = 30;
            const MAX_MARGINS = 4;
            let valuePadding = 0;

            ctrl.el = $element[0];

            const init = (theme = ctrl.theme) => {
                // init can be triggered for a non-kpi chart because of the way we handle displays.
                if (!ctrl.data || !ctrl.chartDef || ctrl.chartDef.type !== CHART_TYPES.KPI) {
                    return;
                }
                const chartData = ChartDataWrapperFactory.chartTensorDataWrapper(ctrl.data);
                ctrl.chartDef.colorMode = 'COLOR_GROUPS';

                // index of measure group colors are based of in data.aggregations
                let basedOnMeasureIndexCounter = ctrl.chartDef.genericMeasures.length - 1;
                ctrl.colorGroupByMeasureId = ctrl.chartDef.colorGroups ? ctrl.chartDef.colorGroups
                    .filter(({ appliedColumns }) => !!appliedColumns)
                    .reduce((acc, { appliedColumns, colorMeasure, rules }) => {
                        let groupColorMeasure;

                        if (colorMeasure && colorMeasure.length > 0) {
                            basedOnMeasureIndexCounter += 1;
                            groupColorMeasure = colorMeasure[0];
                        }

                        appliedColumns.forEach(column => (
                            acc[ConditionalFormattingOptions.getMeasureId(column)] = {
                                rules,
                                basedOnMeasure: groupColorMeasure,
                                basedOnMeasureIndex: groupColorMeasure ? basedOnMeasureIndexCounter : -1
                            }
                        ));
                        return acc;
                    }, {}) : []
                ;


                ctrl.colorRulesClasses = ctrl.chartDef.genericMeasures.map((m, i) => {
                    const colorGroup = ctrl.colorGroupByMeasureId[ConditionalFormattingOptions.getMeasureId(m)];
                    if (!colorGroup) {
                        return ConditionalFormattingOptions.getColorRuleClass('', [], m, theme);
                    }

                    const measureRules = colorGroup.rules;

                    // use "base on another column" value if defined, if not use the value of the column itself
                    const basedOnMeasureIndex = colorGroup.basedOnMeasureIndex >= 0 ? colorGroup.basedOnMeasureIndex : i;
                    const ruleBaseValue = chartData.getAggrExtent(basedOnMeasureIndex)[0];

                    const colorMeasure = colorGroup.basedOnMeasure || m;

                    return ConditionalFormattingOptions.getColorRuleClass(ruleBaseValue, measureRules, colorMeasure, theme);
                });

                ctrl.multipleKPIs = ctrl.chartDef.genericMeasures.length > 1;
                if (ctrl.multipleKPIs) {
                    valuePadding = MAX_MARGINS * 2; // 2 sides
                }

                // sanitize all dom measures
                for (let i = 0; i < ctrl.chartDef.genericMeasures.length; i++) {
                    ctrl.sanitizeKpi(i);
                }

                // take all dimensions generated by flexbox and thank him for his loyal services
                const dimensions = [];
                for (let i = 0; i < ctrl.chartDef.genericMeasures.length; i++) {
                    dimensions[i] = ctrl.getKpiBoxSize(i);
                }

                // replace the sanitized kpi value and label if displayed & find the best fitting fontSize for all kpi and keep the minimum one
                const computeMinValue = (hideAllLabels = false) => {
                    let minValueFontSize = Number.MAX_VALUE;
                    for (let i = 0; i < ctrl.chartDef.genericMeasures.length; i++) {
                        const hideLabel = hideAllLabels || !ctrl.chartDef.genericMeasures[i].showDisplayLabel;
                        minValueFontSize = Math.min(minValueFontSize, ctrl.applyKpiValue(i, dimensions[i], hideLabel));
                    }
                    return minValueFontSize;
                };

                // first compute the minFontSize we can display with labels
                let minValueFontSize = computeMinValue();
                const hideAllLabels = minValueFontSize <= MIN_FONT_BEFORE_HIDDING_LABEL;

                // if we must hide labels, recompute the new minFontSize, as it can now be greater
                if (hideAllLabels) {
                    minValueFontSize = computeMinValue(true);
                }

                for (let i = 0; i < ctrl.chartDef.genericMeasures.length; i++) {
                    ctrl.applyKpiLabel(i, dimensions[i], hideAllLabels);
                }

                // apply the minimum fontSize found to all kpis
                for (let i = 0; i < ctrl.chartDef.genericMeasures.length; i++) {
                    ctrl.applyFontSize(
                        ctrl.el.getElementsByClassName(VALUE_SELECTOR)[i],
                        ctrl.chartDef.genericMeasures[i].kpiValueFontSizeMode === 'RESPONSIVE' ? minValueFontSize * ctrl.chartDef.genericMeasures[i].responsiveTextAreaFill / 100 : ctrl.chartDef.genericMeasures[i].kpiValueFontSize
                    );
                }

                ctrl.loadedCallback();
            };

            /*
             * Re-init on resize
             * Add a timeout to make sure the scope is ready, as now chartDef.genericMeasures can be modified by Angular forms in formatting pane
             */
            $scope.$on('window-resized-kpi', (_, { theme }) => $timeout(() => init(theme), 0));

            /*
             * On first init wait for the dom to be rendered
             * https://stackoverflow.com/questions/18646756
             */
            $scope.$watch('$viewContentLoaded', () => $timeout(() => init(), 0));

            /**
             * Sanitize the dom by setting default values and minimal font-size.
             * This step is necessary to render a default dom skeletton and
             * allow flexbox to render symmetrical boxes for each measure.
             *
             * A default value like a whitespace must be set for the value and label
             * as the flexbox rendering is different with and without.
             *
             * As we keep a default font-size for the label, the latter is not concerned by sanitization.
             *
             * @param {number} index of the measure in dom
             */
            this.sanitizeKpi = function(index) {

                // sanitize container
                const container = ctrl.el.getElementsByClassName(MEASURE_SELECTOR)[index];
                if (ctrl.multipleKPIs) {
                    container.style.margin = `${MAX_MARGINS}px`;
                    container.style.padding = `${MAX_MARGINS}px`;
                }

                // sanitize value div
                const valueDom = ctrl.el.getElementsByClassName(VALUE_SELECTOR)[index];
                valueDom.textContent = ' ';
                valueDom.style.fontSize = '1px';
                valueDom.style.lineHeight = '1px';
                valueDom.style.width = '';

                // sanitize label div
                const labelDom = ctrl.el.getElementsByClassName(LABEL_SELECTOR)[index];
                labelDom.style.display = '';
                labelDom.textContent = ' ';
                labelDom.style.width = '';
            };

            this.applyKpiValue = function(index, containerDim, hidelabel) {
                const chartData = ChartDataWrapperFactory.chartTensorDataWrapper(ctrl.data);
                const kpiValueDom = ctrl.el.getElementsByClassName(VALUE_SELECTOR)[index];
                kpiValueDom.textContent = ChartFormatting.getForIsolatedNumber(ctrl.chartDef.genericMeasures[index])(chartData.getAggrExtent(index)[0]);
                const labelFontSize = hidelabel ? 0 : this.getRawLabelFontSize(index);
                return ctrl.fitKpiValueToContainer(kpiValueDom, containerDim, labelFontSize);
            };

            this.applyKpiLabel = function(index, containerDim, hideAllLabels) {
                const measure = ctrl.chartDef.genericMeasures[index];
                const kpiLabelDom = ctrl.el.getElementsByClassName(LABEL_SELECTOR)[index];

                if (!measure.showDisplayLabel || hideAllLabels) {
                    kpiLabelDom.style.display = 'none';
                    return;
                }

                const kpiValueDom = ctrl.el.getElementsByClassName(VALUE_SELECTOR)[index];
                const boxWidth = Math.ceil(containerDim.width * (ROUNDING + 0.05)) + 'px'; // hack to make sure we don't ellipsis on responsive mode

                kpiLabelDom.style.width = boxWidth;
                kpiValueDom.style.width = boxWidth;
                kpiLabelDom.style.display = '';

                if (measure.labelPosition === ChartsStaticData.LABEL_POSITIONS.BOTTOM.value) {
                    kpiLabelDom.parentNode.insertBefore(kpiValueDom, kpiLabelDom);
                } else {
                    kpiValueDom.parentNode.insertBefore(kpiLabelDom, kpiValueDom);
                }

                kpiLabelDom.textContent = ChartLabels.getLongMeasureLabel(ctrl.chartDef.genericMeasures[index]);
            };

            this.getKpiBoxSize = function(index) {
                const kpiDiv = ctrl.el.getElementsByClassName(MEASURE_SELECTOR)[index];
                return { height: kpiDiv.clientHeight - valuePadding, width: kpiDiv.clientWidth - valuePadding };
            };

            /**
             * Perform a binary search to find the best fontSize fit.
             * O(log(n))
             * @param {HTMLElement} kpiValue the dom of the displayed value
             * @param {{width: number, height: number}} containerDimensions original dimension where the span must fit
             */
            this.fitKpiValueToContainer = function(kpiValue, containerDimensions, labelFontSize) {
                let left = 0;
                let right = 500;
                let mid = 0;
                while (left <= right) {
                    mid = (right + left) >> 1;
                    ctrl.applyFontSize(kpiValue, mid);
                    const kpiSpanClientRect = kpiValue.getBoundingClientRect();
                    if (kpiSpanClientRect.width <= containerDimensions.width * ROUNDING && kpiSpanClientRect.height <= containerDimensions.height * ROUNDING - labelFontSize) {
                        left = mid + 1;
                    } else {
                        right = mid - 1;
                    }
                }

                return mid;
            };

            this.applyFontSize = function(kpiValue, size) {
                /*
                 * some characters like ',' or 'p' are overflowing span bounding rect
                 * we add an extra margin to lineHeight to remedy this
                 */
                kpiValue.style.fontSize = size + 'px';
                kpiValue.style.lineHeight = Math.round(size + size * VALUE_HEIGHT_OVERFLOW) + 'px';
            };

            this.getAlignmentClass = (index) => {
                const measure = this.chartDef.genericMeasures[index];
                if (measure) {
                    switch (measure.kpiTextAlign) {
                        case 'LEFT':
                            return 'kpi-holder-measure--left';
                        case 'RIGHT':
                            return 'kpi-holder-measure--right';
                        default:
                            return 'kpi-holder-measure--center';
                    }
                }
            };

            this.getRawValueFontSize = (index) => {
                return this.chartDef.genericMeasures && this.chartDef.genericMeasures.length && this.chartDef.genericMeasures[index] && this.chartDef.genericMeasures[index].kpiValueFontSize;
            };

            this.getRawLabelFontSize = (index) => {
                return this.chartDef.genericMeasures && this.chartDef.genericMeasures.length && this.chartDef.genericMeasures[index] && this.chartDef.genericMeasures[index].labelTextFormatting && this.chartDef.genericMeasures[index].labelTextFormatting.fontSize;
            };

            this.getValueFontSize = (index) => {
                return `${this.getRawValueFontSize(index)}px`;
            };

            this.getLabelFontSize = (index) => {
                return `${this.getRawLabelFontSize(index)}px`;
            };

            this.getLabelFontColor = (index) => {
                const hasMeasureConditionalFormatting = this.chartDef.genericMeasures && this.chartDef.genericMeasures.length && ctrl.colorGroupByMeasureId && ctrl.colorGroupByMeasureId[ConditionalFormattingOptions.getMeasureId(this.chartDef.genericMeasures[index])];
                return !hasMeasureConditionalFormatting && this.chartDef.genericMeasures && this.chartDef.genericMeasures.length && this.chartDef.genericMeasures[index] && this.chartDef.genericMeasures[index].labelTextFormatting && this.chartDef.genericMeasures[index].labelTextFormatting.fontColor;
            };
        }
    });
})();
