/*
 * Decompiled with CFR 0.152.
 */
package com.dataiku.dip.analysis.ml.hf;

import com.dataiku.dip.DKUApp;
import com.dataiku.dip.analysis.ml.hf.HuggingFaceClient;
import com.dataiku.dip.analysis.ml.hf.HuggingFaceClientFactory;
import com.dataiku.dip.analysis.ml.hf.ModelCacheDataProvider;
import com.dataiku.dip.connections.ConnectionWithBasicCredential;
import com.dataiku.dip.connections.ConnectionsDAO;
import com.dataiku.dip.connections.HuggingFaceLocalConnection;
import com.dataiku.dip.coremodel.InfoMessage;
import com.dataiku.dip.exceptions.DKUSecurityException;
import com.dataiku.dip.exceptions.UnauthorizedException;
import com.dataiku.dip.futures.FutureAborter;
import com.dataiku.dip.partitioning.PartitioningUtils;
import com.dataiku.dip.security.AuthCtx;
import com.dataiku.dip.security.DSSAuthCtx;
import com.dataiku.dip.security.model.ICredentialsService;
import com.dataiku.dip.transactions.fs.NativeFS;
import com.dataiku.dip.transactions.fs.RelFile;
import com.dataiku.dip.transactions.fs.ZipWriteFS;
import com.dataiku.dip.transactions.fs.ifaces.ReadOnlyFS;
import com.dataiku.dip.transactions.fs.ifaces.ReadWriteFS;
import com.dataiku.dip.transactions.fs.utils.FSUtils;
import com.dataiku.dip.transactions.fs.utils.RelFileFilter;
import com.dataiku.dip.utils.AutoCloseableLock;
import com.dataiku.dip.utils.DKULogger;
import com.dataiku.dip.utils.ImmutableValueObject;
import com.dataiku.dip.utils.NamedLock;
import com.dataiku.dss.shadelib.org.apache.commons.io.FileUtils;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import javax.annotation.Nullable;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Service
public class ModelCacheService {
    @Autowired
    ConnectionsDAO connectionsDAO;
    private static final String METADATA_FILENAME = "model_metadata.json";
    private static final String ADAPTER_LIBRARY_NAME = "peft";
    private static final DKULogger logger = DKULogger.getLogger((String)"dku.modelcache");
    private final HuggingFaceClientFactory clientFactory;
    private final ModelCacheDataProvider cacheDataProvider;
    private static final Set<String> isoLanguages = Collections.unmodifiableSet(new HashSet<String>(Arrays.asList(Locale.getISOLanguages())));

    @Autowired
    public ModelCacheService(HuggingFaceClientFactory clientFactory, ModelCacheDataProvider cacheDataProvider) {
        this.clientFactory = clientFactory;
        this.cacheDataProvider = cacheDataProvider;
    }

    public ModelMetadata getModelMetadata_Check(String modelKey, AuthCtx authCtx, @Nullable String connectionName) throws IOException, DKUSecurityException {
        String hfApiKey = this.getHfApiKeyFromConnectionName_Check(authCtx, connectionName);
        return this.getModelMetadata(new ModelStorageDefinition(modelKey), hfApiKey);
    }

    private ModelMetadata getModelMetadata(ModelStorageDefinition modelDefinition, String hfApiKey) throws IOException {
        try (AutoCloseableLock lock = NamedLock.acquire((String)modelDefinition.getLockName());){
            NativeFS cacheData = this.cacheDataProvider.getCacheData();
            RelFile metadataFile = ModelCacheService.metadataFile(modelDefinition);
            if (cacheData.exists(metadataFile)) {
                logger.info((Object)("Retrieving " + modelDefinition.toPrettyMessageFormat() + " metadata from cache."));
                ModelMetadata modelMetadata = (ModelMetadata)cacheData.readObject(metadataFile, ModelMetadata.class);
                return modelMetadata;
            }
        }
        logger.info((Object)("Retrieving " + modelDefinition.toPrettyMessageFormat() + " metadata from HuggingFace."));
        return this.buildMetadataFromHF(modelDefinition, hfApiKey);
    }

    public String getLocalModelPath(String modelKey) throws IOException {
        ModelStorageDefinition modelStorageDefinition = new ModelStorageDefinition(modelKey);
        this.throwIfModelNotDownloaded(modelStorageDefinition);
        return this.cacheDataProvider.getCacheData().resolve(ModelCacheService.modelRootFolder(modelStorageDefinition)).getPath();
    }

    public String getBaseModelKey(String modelKey) throws IOException {
        ModelStorageDefinition modelStorageDefinition = new ModelStorageDefinition(modelKey);
        this.throwIfModelNotDownloaded(modelStorageDefinition);
        ModelMetadata metadata = this.getModelMetadata(ModelCacheService.metadataFile(ModelCacheService.modelRootFolder(modelStorageDefinition)));
        if (metadata.baseModelDefinition == null) {
            return null;
        }
        return metadata.baseModelDefinition.getModelName();
    }

    private String getHfApiKeyFromConnectionName(AuthCtx authCtx, @Nullable String connectionName) throws IOException, DKUSecurityException {
        if (connectionName == null) {
            return null;
        }
        HuggingFaceLocalConnection hfConnection = this.connectionsDAO.getMandatoryConnectionAs(authCtx, connectionName, HuggingFaceLocalConnection.class);
        ICredentialsService.BasicCredential creds = hfConnection.getFullyResolvedCredentials_fsLike(new ConnectionWithBasicCredential.CredentialResolutionContext(DSSAuthCtx.newNone(), null), ICredentialsService.BasicCredential.class);
        return creds.password;
    }

    private String getHfApiKeyFromConnectionName_Check(AuthCtx authCtx, @Nullable String connectionName) throws IOException, DKUSecurityException {
        if (connectionName == null) {
            return null;
        }
        HuggingFaceLocalConnection hfConnection = this.connectionsDAO.getMandatoryConnectionAs(authCtx, connectionName, HuggingFaceLocalConnection.class);
        if (!hfConnection.isFreelyUsableBy(authCtx)) {
            throw new UnauthorizedException("You may not use this connection", "denied");
        }
        ICredentialsService.BasicCredential creds = hfConnection.getFullyResolvedCredentials_fsLike(new ConnectionWithBasicCredential.CredentialResolutionContext(DSSAuthCtx.newNone(), null), ICredentialsService.BasicCredential.class);
        return creds.password;
    }

    public void downloadEnabledModel(String modelKey, AuthCtx authCtx, @Nullable String connectionName) throws IOException, DKUSecurityException {
        boolean canDownloadModel = false;
        if (authCtx.isAdmin()) {
            canDownloadModel = true;
        } else if (connectionName != null) {
            canDownloadModel = this.canDownloadModelInConnection(modelKey, authCtx, connectionName);
        } else {
            List<ConnectionsDAO.ConnectionTypeAndName> hfConnections = this.connectionsDAO.getConnectionTypesAndNames("HuggingFaceLocal", null, false);
            for (ConnectionsDAO.ConnectionTypeAndName connectionTypeAndName : hfConnections) {
                if (!this.canDownloadModelInConnection(modelKey, authCtx, connectionTypeAndName.name)) continue;
                canDownloadModel = true;
                break;
            }
        }
        if (!canDownloadModel) {
            throw new DKUSecurityException("The requested model is not set within a connection please ask your admin to activate it to download it");
        }
        ModelStorageDefinition modelStorageDefinition = new ModelStorageDefinition(modelKey);
        String hfApiKey = this.getHfApiKeyFromConnectionName_Check(null, connectionName);
        this.downloadModelIfNeeded(modelStorageDefinition, authCtx, hfApiKey);
    }

    private boolean canDownloadModelInConnection(String modelKey, AuthCtx authCtx, String connectionName) throws IOException, DKUSecurityException {
        HuggingFaceLocalConnection hfConnection = this.connectionsDAO.getMandatoryConnectionAs(authCtx, connectionName, HuggingFaceLocalConnection.class);
        return hfConnection.isModelEnabled(modelKey);
    }

    public void downloadModelIfNeeded(String modelKey, AuthCtx authCtx, @Nullable String connectionName) throws IOException, DKUSecurityException {
        ModelStorageDefinition modelStorageDefinition = new ModelStorageDefinition(modelKey);
        String hfApiKey = this.getHfApiKeyFromConnectionName(authCtx, connectionName);
        this.downloadModelIfNeeded(modelStorageDefinition, authCtx, hfApiKey);
    }

    public void downloadModelIfNeeded_Check(String modelKey, AuthCtx authCtx, @Nullable String connectionName) throws IOException, DKUSecurityException {
        ModelStorageDefinition modelStorageDefinition = new ModelStorageDefinition(modelKey);
        String hfApiKey = this.getHfApiKeyFromConnectionName_Check(authCtx, connectionName);
        this.downloadModelIfNeeded(modelStorageDefinition, authCtx, hfApiKey);
    }

    private void downloadModelIfNeeded(ModelStorageDefinition modelDefinition, AuthCtx authCtx, @Nullable String hfApiKey) throws IOException {
        try (AutoCloseableLock lock = NamedLock.acquire((String)modelDefinition.getLockName());){
            NativeFS cacheData = this.cacheDataProvider.getCacheData();
            if (this.modelIsDownloaded(modelDefinition)) {
                logger.info((Object)(modelDefinition.toPrettyMessageFormat() + " already available."));
                this.downloadBaseModelIfNeeded(modelDefinition, authCtx, hfApiKey);
                return;
            }
            try (HuggingFaceClient client = this.clientFactory.getClient(hfApiKey);
                 FutureAborter.AutoCloseableAbortHook abortHook = FutureAborter.pushAutoCloseableHook(client::close);){
                if (!client.modelExists(modelDefinition)) {
                    throw new ModelNotFoundException("Model: " + modelDefinition.getModelName() + " not found.");
                }
                ModelMetadata metadata = this.buildMetadataFromHF(modelDefinition, hfApiKey);
                metadata.downloadedBy = authCtx.getIdentifier();
                metadata.downloadDate = System.currentTimeMillis();
                RelFile dstModelFilesFolder = ModelCacheService.modelFilesFolder(modelDefinition);
                logger.info((Object)("User '" + metadata.downloadedBy + "' initiated download of weights from '" + metadata.url + "' to '" + cacheData.resolve(dstModelFilesFolder).getPath() + "'."));
                int parallelism = DKUApp.getParams().getIntParam("dku.ml.modelCache.downloadModelParallelism", Integer.valueOf(8));
                FSUtils.newRecursiveSync().parallelism(parallelism).maxRetry(1).from(client.getModelRepo(modelDefinition.getModelName(), modelDefinition.getRevision())).to((ReadWriteFS)cacheData, dstModelFilesFolder).filter((RelFileFilter)new HuggingFaceClient.HuggingFaceWeightsFilter()).run();
                cacheData.writeObject(ModelCacheService.metadataFile(modelDefinition), (Object)metadata);
                logger.info((Object)("Completed download of model: " + metadata.modelDefinition.key));
                this.downloadBaseModelIfNeeded(modelDefinition, authCtx, hfApiKey);
            }
        }
    }

    private void downloadBaseModelIfNeeded(ModelStorageDefinition modelDefinition, AuthCtx authCtx, @Nullable String hfApiKey) throws IOException {
        ModelStorageDefinition modelStorageDefinition = new ModelStorageDefinition(modelDefinition.getModelKey());
        this.throwIfModelNotDownloaded(modelStorageDefinition);
        ModelMetadata metadata = this.getModelMetadata(ModelCacheService.metadataFile(ModelCacheService.modelRootFolder(modelStorageDefinition)));
        if (ADAPTER_LIBRARY_NAME.equals(metadata.libraryName)) {
            logger.info((Object)("Model is an adapter. Downloading base model if needed: " + metadata.baseModelDefinition.toPrettyMessageFormat()));
            this.downloadModelIfNeeded(metadata.baseModelDefinition, authCtx, hfApiKey);
        }
    }

    public boolean modelIsDownloaded(ModelStorageDefinition modelDefinition) throws IOException {
        NativeFS cacheData = this.cacheDataProvider.getCacheData();
        return cacheData.exists(ModelCacheService.metadataFile(modelDefinition));
    }

    public void updateModelLastUsage(ModelStorageDefinition modelStorageDefinition, AuthCtx authCtx) throws IOException {
        try (AutoCloseableLock lock = NamedLock.acquire((String)modelStorageDefinition.getLockName());){
            NativeFS cacheData = this.cacheDataProvider.getCacheData();
            ModelMetadata metadata = (ModelMetadata)cacheData.readObject(ModelCacheService.metadataFile(modelStorageDefinition), ModelMetadata.class);
            metadata.lastUsedBy = authCtx.getIdentifier();
            metadata.lastDssUsage = System.currentTimeMillis();
            cacheData.writeObject(ModelCacheService.metadataFile(modelStorageDefinition), (Object)metadata);
        }
    }

    public ModelCacheDetails getCacheDetails() throws IOException {
        NativeFS cacheData = this.cacheDataProvider.getCacheData();
        ModelCacheDetails cacheDetails = new ModelCacheDetails();
        cacheDetails.modelCount = 0L;
        cacheDetails.totalSizeInBytes = FSUtils.calculateSize((ReadOnlyFS)cacheData, (RelFile)RelFile.root());
        cacheDetails.totalSizeToDisplay = FileUtils.byteCountToDisplaySize((long)cacheDetails.totalSizeInBytes);
        cacheDetails.models = new ArrayList<ModelDetails>();
        for (RelFile modelFolder : cacheData.listFiles("/")) {
            RelFile modelMetadataFile = ModelCacheService.metadataFile(modelFolder);
            if (!cacheData.exists(modelMetadataFile)) continue;
            ModelMetadata modelMetadata = this.getModelMetadata(modelMetadataFile);
            cacheDetails.models.add(modelMetadata.getDisplayDetails());
            ++cacheDetails.modelCount;
        }
        return cacheDetails;
    }

    public void clear() throws IOException {
        NativeFS cacheData = this.cacheDataProvider.getCacheData();
        for (RelFile modelFolder : cacheData.listFiles("/")) {
            cacheData.deleteDirectory(modelFolder);
        }
    }

    public void deleteModel(ModelStorageDefinition modelDefinition) throws IOException {
        if (!this.modelIsDownloaded(modelDefinition)) {
            throw new IllegalArgumentException("No model '" + modelDefinition.getModelKey() + "' to delete.");
        }
        NativeFS cacheData = this.cacheDataProvider.getCacheData();
        if (cacheData.deleteDirectory(modelDefinition.getFolderName())) {
            logger.info((Object)("Deleted: " + cacheData.resolve(new RelFile(new String[]{modelDefinition.getFolderName()})).getAbsolutePath()));
        } else {
            logger.info((Object)("Failed to delete: " + cacheData.resolve(new RelFile(new String[]{modelDefinition.getFolderName()})).getAbsolutePath()));
        }
    }

    public void zipModelToStream(ModelStorageDefinition modelDefinition, OutputStream outputStream) throws IOException {
        try (AutoCloseableLock lock = NamedLock.acquire((String)modelDefinition.getLockName());
             ZipWriteFS zipWriteFS = new ZipWriteFS(outputStream, 0);){
            NativeFS cacheData = this.cacheDataProvider.getCacheData();
            RelFile modelFolder = ModelCacheService.modelRootFolder(modelDefinition);
            FSUtils.newRecursiveCopy().from((ReadOnlyFS)cacheData, modelFolder).to((ReadWriteFS)zipWriteFS, modelFolder).run();
        }
    }

    public void importModel(AuthCtx authCtx, File modelToImport, ModelImportResult importResult) {
        try {
            NativeFS extractedTmpModelDir = NativeFS.from((File)modelToImport).skipFileTypeCheck(false).atomicWrite(true).build();
            for (RelFile modelFolder : extractedTmpModelDir.listFiles("/")) {
                this.importSingleModel(NativeFS.from((File)extractedTmpModelDir.resolve(modelFolder)).skipFileTypeCheck(false).atomicWrite(true).build(), importResult.messages, authCtx);
            }
        }
        catch (Exception e) {
            importResult.messages.withError((InfoMessage.MessageCode)ModelImportCode.ERR_MODEL_UNKNOWN, "Unexpected error while importing archive " + e.getMessage());
            logger.error((Object)"Error occurred while importing archive", (Throwable)e);
        }
    }

    private void importSingleModel(NativeFS modelToImportFolder, InfoMessage.InfoMessages messages, AuthCtx authCtx) throws IOException {
        if (!modelToImportFolder.exists(METADATA_FILENAME)) {
            String message = "Folder: '" + modelToImportFolder.getRoot().getPath() + "' contains no model metadata, skipping.";
            logger.error((Object)message);
            messages.withError((InfoMessage.MessageCode)ModelImportCode.ERR_MODEL_MISSING_METADATA, message);
            return;
        }
        ModelMetadata modelMetadata = (ModelMetadata)modelToImportFolder.readObject(METADATA_FILENAME, ModelMetadata.class);
        try (AutoCloseableLock lock = NamedLock.acquire((String)modelMetadata.modelDefinition.getLockName());){
            if (this.modelIsDownloaded(modelMetadata.modelDefinition)) {
                String message = "Model '" + modelMetadata.modelDefinition.getModelName() + "' already in cache. Not importing this model.";
                logger.info((Object)message);
                messages.withWarning((InfoMessage.MessageCode)ModelImportCode.WARN_MODEL_EXISTING_MODEL, message);
                return;
            }
            File dstDir = this.cacheDataProvider.getCacheData().resolve(new RelFile(new String[]{modelMetadata.modelDefinition.getFolderName()}));
            FileUtils.moveDirectory((File)modelToImportFolder.getRoot(), (File)dstDir);
            messages.withInfo((InfoMessage.MessageCode)ModelImportCode.INFO_MODEL_IMPORTED, "Imported '" + modelMetadata.modelDefinition.getModelName() + "' successfully.");
            NativeFS cacheData = this.cacheDataProvider.getCacheData();
            ModelMetadata metadata = (ModelMetadata)cacheData.readObject(ModelCacheService.metadataFile(modelMetadata.modelDefinition), ModelMetadata.class);
            metadata.downloadedBy = authCtx.getIdentifier();
            metadata.downloadDate = System.currentTimeMillis();
            metadata.lastUsedBy = authCtx.getIdentifier();
            metadata.lastDssUsage = System.currentTimeMillis();
            metadata.manuallyImported = true;
            cacheData.writeObject(ModelCacheService.metadataFile(modelMetadata.modelDefinition), (Object)metadata);
        }
    }

    private ModelMetadata buildMetadataFromHF(ModelStorageDefinition modelStorageDefinition, @Nullable String hfApiKey) throws IOException {
        try (HuggingFaceClient client = this.clientFactory.getClient(hfApiKey);){
            HuggingFaceClient.HFResponseModelDetails hfDetails = client.getModelDetails(modelStorageDefinition);
            long totalSize = FSUtils.calculateSize((ReadOnlyFS)client.getModelRepo(modelStorageDefinition.getModelName(), modelStorageDefinition.getRevision()), (RelFile)RelFile.root(), (RelFileFilter)new HuggingFaceClient.HuggingFaceWeightsFilter());
            HuggingFaceClient.HFCommitInfo commitInfo = client.getCommitInfos(modelStorageDefinition.getModelName(), modelStorageDefinition.getRevision());
            ModelMetadata metadata = new ModelMetadata();
            metadata.version = 0;
            metadata.libraryName = hfDetails.library_name;
            metadata.pipelineName = hfDetails.pipeline_tag;
            metadata.modelDefinition = modelStorageDefinition;
            metadata.tags = hfDetails.tags;
            metadata.sizeInBytes = totalSize;
            metadata.url = client.getModelUrl(modelStorageDefinition.getModelName(), modelStorageDefinition.getRevision());
            metadata.lastModified = commitInfo.date;
            metadata.commitHash = commitInfo.id;
            metadata.taggedLanguages = ModelCacheService.getTaggedLanguages(metadata.tags);
            String baseModelKey = client.getBaseModelKey(modelStorageDefinition.getModelName(), modelStorageDefinition.getRevision());
            if (baseModelKey != null) {
                metadata.baseModelDefinition = new ModelStorageDefinition(baseModelKey);
            }
            ModelMetadata modelMetadata = metadata;
            return modelMetadata;
        }
    }

    private void throwIfModelNotDownloaded(ModelStorageDefinition modelStorageDefinition) throws IOException {
        if (!this.modelIsDownloaded(modelStorageDefinition)) {
            throw new ModelNotFoundException("Model " + modelStorageDefinition.toPrettyMessageFormat() + " not downloaded.");
        }
    }

    private static RelFile modelRootFolder(ModelStorageDefinition modelStorageDefinition) {
        return new RelFile(new String[]{modelStorageDefinition.getFolderName()});
    }

    private static RelFile metadataFile(ModelStorageDefinition modelStorageDefinition) {
        return new RelFile(new String[]{modelStorageDefinition.getFolderName(), METADATA_FILENAME});
    }

    private static RelFile metadataFile(RelFile modelRootFolder) {
        return new RelFile(modelRootFolder, new String[]{METADATA_FILENAME});
    }

    private ModelMetadata getModelMetadata(RelFile metadataFile) throws IOException {
        NativeFS cacheData = this.cacheDataProvider.getCacheData();
        return (ModelMetadata)cacheData.readObject(metadataFile, ModelMetadata.class);
    }

    private static RelFile modelFilesFolder(ModelStorageDefinition modelStorageDefinition) {
        return new RelFile(new String[]{modelStorageDefinition.getFolderName(), "model"});
    }

    private static Set<String> getTaggedLanguages(List<String> tags) {
        HashSet<String> taggedLanguages = new HashSet<String>();
        for (String tag : tags) {
            if (!isoLanguages.contains(tag)) continue;
            taggedLanguages.add(tag);
        }
        return taggedLanguages;
    }

    public static class ModelStorageDefinition
    extends ImmutableValueObject {
        private static final String KEY_SEPARATOR = "@";
        private static final String DEFAULT_REVISION = "main";
        final String key;

        public ModelStorageDefinition(String modelKey) {
            if (StringUtils.isBlank((String)modelKey)) {
                throw new BadModelKeyException("blank model key");
            }
            this.key = "hf@" + modelKey;
        }

        public String toPrettyMessageFormat() {
            return "model '" + this.getModelName() + "'; revision '" + this.getRevision() + "'; model key '" + this.getModelKey() + "'";
        }

        public String getModelKey() {
            String[] chunks = this.key.split(KEY_SEPARATOR, 2);
            if (chunks.length < 2) {
                throw new RuntimeException(this.getKeyFormatError());
            }
            return chunks[1];
        }

        public String getLockName() {
            return "analysis.ml.model.cache." + this.key;
        }

        public String getModelName() {
            String[] chunks = this.key.split(KEY_SEPARATOR, 3);
            if (chunks.length < 2) {
                throw new RuntimeException(this.getKeyFormatError());
            }
            return chunks[1];
        }

        private String getKeyFormatError() {
            return "Model key " + this.key + " is incorrect. It must be hf@<model>(@<revision>) instead";
        }

        public String getRevision() {
            String[] modelIdentifiers = this.key.split(KEY_SEPARATOR, 3);
            if (modelIdentifiers.length == 3) {
                return modelIdentifiers[2];
            }
            return DEFAULT_REVISION;
        }

        public String getFolderName() {
            return PartitioningUtils.encode(this.getModelName(), true) + (String)(this.hasRevision() ? KEY_SEPARATOR + PartitioningUtils.encode(this.getRevision(), true) : "");
        }

        public boolean hasRevision() {
            return this.key.split(KEY_SEPARATOR, 3).length == 3;
        }
    }

    public static class ModelMetadata {
        public int version;
        public ModelStorageDefinition modelDefinition;
        public ModelStorageDefinition baseModelDefinition;
        public String libraryName;
        public long sizeInBytes;
        public String pipelineName;
        public List<String> tags;
        public Set<String> taggedLanguages;
        public String url;
        public String lastModified;
        public String commitHash;
        public String downloadedBy;
        public long downloadDate;
        public String lastUsedBy;
        public long lastDssUsage;
        public boolean manuallyImported;

        ModelDetails getDisplayDetails() {
            ModelDetails modelDetails = new ModelDetails();
            modelDetails.modelKey = this.modelDefinition.getModelKey();
            modelDetails.sizeToDisplay = FileUtils.byteCountToDisplaySize((long)this.sizeInBytes);
            return modelDetails;
        }
    }

    public static class ModelNotFoundException
    extends IllegalArgumentException {
        public ModelNotFoundException(String message) {
            super(message);
        }
    }

    public static class ModelCacheDetails {
        long totalSizeInBytes;
        String totalSizeToDisplay;
        long modelCount;
        List<ModelDetails> models;
    }

    public static class ModelDetails {
        String modelKey;
        String sizeToDisplay;
    }

    public static class ModelImportResult {
        public InfoMessage.InfoMessages messages = new InfoMessage.InfoMessages();
    }

    public static enum ModelImportCode implements InfoMessage.MessageCode
    {
        INFO_MODEL_IMPORTED("Import succeeded", InfoMessage.FixabilityCategory.IRRELEVANT),
        WARN_MODEL_EXISTING_MODEL("Model already exists", InfoMessage.FixabilityCategory.IRRELEVANT),
        ERR_MODEL_MISSING_METADATA("Missing metadata file", InfoMessage.FixabilityCategory.IRRELEVANT),
        ERR_MODEL_UNKNOWN("Unknown error while importing archive", InfoMessage.FixabilityCategory.UNKNOWN);

        private final String title;
        private final InfoMessage.FixabilityCategory fixability;

        private ModelImportCode(String title, InfoMessage.FixabilityCategory fixability) {
            this.title = title;
            this.fixability = fixability;
        }

        public String getCode() {
            return this.name();
        }

        public String getCodeTitle() {
            return this.title;
        }

        public InfoMessage.FixabilityCategory getFixability() {
            return this.fixability;
        }
    }

    public static class BadModelKeyException
    extends IllegalArgumentException {
        public BadModelKeyException(String message) {
            super(message);
        }
    }
}

