from dataiku.core import base, flow, metrics, dkuio, default_project_key
from dataiku import Dataset
import json, logging

logger = logging.getLogger(__name__)

try:
    import pandas as pd
except:
    logger.warning("Pandas not available, certain model_evaluation_store methods are unavailable")


from dataiku.core.intercom import jek_or_backend_json_call, jek_or_backend_stream_call, backend_json_call


class ModelEvaluationStore(base.Computable):
    """
    This is a handle to interact with a model evaluation store.

    Note: this class is also available as :class:`dataiku.core.model_evaluation_store.ModelEvaluationStore`
    """

    def __init__(self, lookup, project_key=None, ignore_flow=False):
        """Obtains a handle for a named model evaluation store.

        :param str lookup: name or identifier of the model evaluation store
        :param str project_key: project key of the model evaluation store, if it is not in the current project (defaults to **None**)
        :param bool ignore_flow: set to false to use the current flow to retrieve the Model Evaluation Store, otherwise it will parse the lookup param
            to retrieve it (defaults to **False**)
        """
        self.lookup = lookup
        self.path = None
        self.info = None
        self.ignore_flow = ignore_flow

        if flow.FLOW is not None and ignore_flow == False:
            self._init_data_from_flow(obj_type="Model evaluation store", project_key=project_key)

        else:
            if "." not in lookup:
                self.project_key = project_key or default_project_key()
                self.short_name = lookup
                self.name = self.project_key + "." + lookup
            else:
                self.project_key = lookup.split(".")[0]
                self.short_name = lookup.split(".")[1]
                self.name = lookup
                #except:
                #    raise Exception("Model evaluation store %s is specified with a relative name, "
                #                    "but no default project was found. Please use complete name" % id)

    def _repr_html_(self,):
        s = "ModelEvaluationStore[   <b>%s</b>   ]" % (self.name)
        return s

    def get_info(self, sensitive_info=False):
        """
        Gets information about the location and settings of this model evaluation store.

        :param bool sensitive_info: flag for sensitive information such as the Model Evaluation Store absolute path (defaults to **False**)

        :rtype: dict
        """
        if self.info is None:
            self.info = jek_or_backend_json_call("model-evaluation-stores/get-info", {
                "projectKey": self.project_key,
                "lookup" : self.short_name,
                "sensitiveInfo" : sensitive_info
            })
        return self.info["info"]

    def get_path(self):
        """
        Gets the filesystem path of this model evaluation store.

        :rtype: str
        """
        return self.get_info()["path"]

    def get_id(self):
        """
        Gets the id of this model evaluation store.

        :rtype: str
        """
        return self.get_info()["id"]

    def get_name(self):
        """
        Gets the name of this model evaluation store.

        :rtype: str
        """
        return self.get_info()["name"]
        
    def list_runs(self):
        """
        Gets the list of runs of this model evaluation store.

        :rtype: list of :class:`dataiku.core.model_evaluation_store.ModelEvaluation`
        """
        return jek_or_backend_json_call("model-evaluation-stores/list-runs", {
                "projectKey": self.project_key,
                "id" : self.get_id()
            })
            
    def get_evaluation(self, evaluation_id):
        """
        Gets a model evaluation from the store based on its id.

        :param str evaluation_id: the id of the model evaluation to retrieve
        :return: a :class:`dataiku.core.model_evaluation_store.ModelEvaluation` handle on the model evaluation
        """
        return ModelEvaluation(self, evaluation_id)

    # ################################### Metrics #############################

    def get_last_metric_values(self):
        """
        Gets the set of last values of the metrics on this folder.

        :return: a :class:`dataiku.core.ComputedMetrics` object
        """
        return metrics.ComputedMetrics(backend_json_call("metrics/modelevaluationstores/get-last-values", data = {
            "projectKey": self.project_key,
            "mesId" : self.get_id(),
        }))

    def get_metric_history(self, metric_lookup):
        """
        Gets the set of all values a given metric took on this folder.

        :param metric_lookup: metric name or unique identifier
        :rtype: dict
        """
        return backend_json_call("metrics/modelevaluationstores/get-metric-history", data = {
            "projectKey": self.project_key,
            "mesId" : self.get_id(),
            "metricLookup" : metric_lookup if isinstance(metric_lookup, str) or isinstance(metric_lookup, unicode) else json.dumps(metric_lookup)
        })


# one evaluation in a store
class ModelEvaluation(object):
    """
    This is a handle to interact with a model evaluation from a model evaluation store.
    """

    def __init__(self, store, evaluation_id):
        """Obtains a handle for a model evaluation from a given model evaluation store and with a specific model evaluation id.

        :param store: model evaluation store that owns the model evaluation
        :type store: a :class:`dataiku.core.model_evaluation_store.ModelEvaluationStore`
        :param str evaluation_id: id of the model evaluation
        """
        self.store = store
        self.evaluation_id = evaluation_id
        self.sample_schema = None
        self.preparation_steps = None
        self.preparation_requested_output_schema = None
        self.preparation_context_project_key = None

    def _repr_html_(self,):
        s = "ModelEvaluation[   <b>%s %s</b>   ]" % (self.store.name, self.evaluation_id)
        return s

    def set_preparation_steps(self, steps, requested_output_schema, context_project_key=None):
        """
        Sets the preparation steps of the input dataset in a model evaluation.

        :param dict steps: the steps of the preparation
        :param dict requested_output_schema: the schema of the prepared input dataset as a list of objects like this one:
            ``{ 'type': 'string', 'name': 'foo', 'maxLength': 1000 }``
        :param str context_project_key: a different project key to use instead of the current project key, because the preparation steps can live
            in a different project than the dataset (defaults to **None**)
        """
        self.preparation_steps = steps
        self.preparation_requested_output_schema = requested_output_schema
        self.preparation_context_project_key = context_project_key

    def get_schema(self):
        """
        Gets the schema of the sample used for this model evaluation.
        There is more information for the map, array and object types.

        :rtype: list of dict
        :return: a schema as a list of objects like this one:
            ``{ 'type': 'string', 'name': 'foo', 'maxLength': 1000 }``
        """
        if self.sample_schema is None:
            self.sample_schema = jek_or_backend_json_call("model-evaluation-stores/get-sample-schema", data={
                "projectKey": self.store.project_key,
                "id": self.store.get_id(),
                "evaluationId": self.evaluation_id
            }, err_msg='Unable to fetch schema for %s %s' % (self.store.name, self.evaluation_id))

        return self.sample_schema

    def _stream(self):
        preparation_steps = self.preparation_steps
        preparation_requested_output_schema = self.preparation_requested_output_schema
        preparation_context_project_key = self.preparation_context_project_key,
        if preparation_steps is None or preparation_requested_output_schema is None:
            logging.info("Read sample without script")
            preparation_steps = []
            preparation_requested_output_schema = self.get_schema()
            
        data = {
            "projectKey": self.store.project_key,
            "id": self.store.get_id(),
            "evaluationId": self.evaluation_id,
            "script" :  json.dumps({ "steps" : preparation_steps }),
            "contextProjectKey": preparation_context_project_key,
            "requestedOutputSchema" : json.dumps(preparation_requested_output_schema)
        }

        return jek_or_backend_stream_call("model-evaluation-stores/stream-prepared-sample", data=data, err_msg="Failed to read prepared data")

    def _get_dataframe_schema(self,
                              columns=None,
                              parse_dates=True,
                              infer_with_pandas=False,
                              bool_as_str=False):

        if self.preparation_steps is not None:
            schema_to_use = self.preparation_requested_output_schema
        else:
            schema_to_use = self.get_schema()
        return Dataset.get_dataframe_schema_st(schema_to_use.get("columns", []), columns, parse_dates, infer_with_pandas, bool_as_str)

    def get_dataframe(self,
                      columns=None,
                      infer_with_pandas=True,
                      parse_dates=True,
                      bool_as_str=False,
                      float_precision=None):
        """Reads the sample in the run as a Pandas dataframe.

        Pandas dataframes are fully in-memory, so you need to make
        sure that your dataset will fit in RAM before using this.

        Inconsistent sampling parameter raise ValueError.

        Note about encoding:

        * Column labels are "unicode" objects
        * When a column is of string type, the content is made of utf-8 encoded "str" objects

        :param columns: the columns with information on type, names, etc.
            e.g. ``{ 'type': 'string', 'name': 'foo', 'maxLength': 1000 }`` (defaults to **None**)
        :type columns: list of dict
        :param bool infer_with_pandas: uses the types detected by pandas rather than the types from the dataset schema as detected in DSS (defaults to **True**)
        :param bool parse_dates: parses date columns in DSS's dataset schema (defaults to **True**)
        :param bool bool_as_str: leaves boolean values as strings (defaults to **False**)
        :param str float_precision: float precision for pandas read_table (defaults to **None**)
        :return: a :class:`pandas.Dataframe` representing the sample used in the evaluation
        """
        (names, dtypes, parse_date_columns) = self._get_dataframe_schema(
            columns=columns,
            parse_dates=parse_dates,
            infer_with_pandas=infer_with_pandas,
            bool_as_str=bool_as_str) # see df_from_split_desc
        with self._stream() as dku_output:
            return pd.read_table(dku_output,
                                 names=names,
                                 dtype=dtypes,
                                 header=None,
                                 sep='\t',
                                 doublequote=True,
                                 quotechar='"',
                                 parse_dates=parse_date_columns,
                                 float_precision=float_precision)

    def iter_dataframes_forced_types(self,
                        names, dtypes, parse_date_columns,
                        sampling=None, # ignored (for the moment)
                        chunksize=10000,
                        float_precision=None):
        """Reads the model evaluation sample as Pandas dataframes by chunks of fixed size with forced types.

        Returns a generator over pandas dataframes.

        Useful is the sample doesn't fit in RAM.

        :param names: names of the columns of the dataset
        :type names: list of str
        :param dtypes: data types of the columns
        :type dtypes: list of str or object
        :param bool parse_date_columns: parses date columns in DSS's dataset schema
        :param sampling: ignored at the moment (defaults to **None**)
        :param int chunksize: the size of the dataframes yielded by the iterator (defaults to **10000**)
        :param str float_precision: float precision for pandas read_table (defaults to **None**)
        :return: a generator of :class:`pandas.Dataframe`
        """
        with self._stream() as dku_output:
            df_it = pd.read_table(
                dku_output,
                dtype=dtypes,
                names=names,
                low_memory=True,
                header=None,
                sep='\t',
                doublequote=True,
                chunksize=chunksize,
                iterator=True,
                parse_dates=parse_date_columns,
                float_precision=float_precision)
            logging.info("Starting dataframes iterator")
            for df in df_it:
                yield df

    def iter_dataframes(self,
                        chunksize=10000,
                        infer_with_pandas=True,
                        parse_dates=True,
                        columns=None,
                        bool_as_str=False,
                        float_precision=None):
        """Read the model evaluation sample to Pandas dataframes by chunks of fixed size.

        Returns a generator over pandas dataframes.

        Useful is the sample doesn't fit in RAM.

        :param int chunksize: the size of the dataframes yielded by the iterator (defaults to **10000**)
        :param bool infer_with_pandas: uses the types detected by pandas rather than the dataset schema as detected in DSS (defaults to **True**)
        :param bool parse_dates: parses date columns in DSS's dataset schema (defaults to **True**)
        :param columns: columns of the dataset as dict with names and dtypes (defaults to **None**)
        :type columns: list of dict
        :param bool bool_as_str: leaves boolean values as strings (defaults to **False**)
        :param str float_precision: float precision for pandas read_table. For more information on this parameter, please check pandas documentation:
            https://pandas.pydata.org/docs/reference/api/pandas.read_table.html (defaults to **None**)
        :return: a generator of :class:`pandas.Dataframe`
        """
        
        (names, dtypes, parse_date_columns) = self._get_dataframe_schema(
            columns=columns,
            parse_dates=parse_dates,
            infer_with_pandas=infer_with_pandas,
            bool_as_str=bool_as_str) # see df_from_split_desc
        with self._stream() as dku_output:
            df_it = pd.read_table(
                dku_output,
                dtype=dtypes,
                names=names,
                low_memory=True,
                header=None,
                sep='\t',
                doublequote=True,
                chunksize=chunksize,
                iterator=True,
                parse_dates=parse_date_columns,
                float_precision=float_precision)
            logging.info("Starting dataframes iterator")
            for df in df_it:
                yield df

