import json
import logging
import os
import shutil
from typing import TYPE_CHECKING

from dataiku.core.vector_stores.lifecycle.base import (
    create_langchain_vector_store,
    download_to_folder,
    generate_random_name,
    get_folder_path_to_load_kb
)
from dataiku.core.vector_stores.data.writer import VectorStoreWriter

if TYPE_CHECKING:
    from langchain_core.vectorstores import VectorStore
    from dataiku.llm.types import RetrievableKnowledge

logger = logging.getLogger(__name__)


def load_into_isolated_folder(
        project_key: str,
        kb_id: str,
        version: str,
        use_latest_settings=False
) -> 'VectorStoreIsolatedFolder':
    """
    Loads the vector store files on disk.
    For local vector stores, downloads metadata files as well as data files.
    For remote vector stores, only downloads metadata files.

    ..note:
        The downloaded folder is an isolated copy. Multiple calls with same
        arguments will download the same folder multiple times.

    :param str project_key: The knowledge bank project key.
    :param str kb_id: The knowledge bank identifier.
    :param str version: The knowledge bank version.
    :return: the path to the local folder
    :rtype: str
    """
    # append random name to the folder name to make it unique
    folder_name = "{}-{}-{}-{}".format(
        project_key, kb_id, version,
        generate_random_name(8)
    )

    root_folder_path = get_folder_path_to_load_kb(isolated=True)
    folder_path = os.path.join(root_folder_path, folder_name)

    logger.debug("load folder {}".format(folder_path))
    download_to_folder(
        project_key, kb_id, version, folder_path,
        use_latest_settings
    )
    kb_full_id = "{}.{}".format(project_key, kb_id)
    return VectorStoreIsolatedFolder(folder_path, project_key, kb_full_id)


class VectorStoreIsolatedFolder:
    """
    A handle on the vector store isolated folder.

    .. important::
        Do not create this class directly, use
        :meth:`dataiku.KnowledgeBank.load_into_isolated_folder()`
    """

    def __init__(self, folder_path: str, project_key: str, kb_full_id: str):
        self._folder_path = folder_path
        self._project_key = project_key
        self._kb_full_id = kb_full_id

    def __repr__(self):
        return "{}(path={})".format(
            self.__class__.__name__,
            self._folder_path
        )

    @property
    def folder_path(self) -> str:
        return self._folder_path

    def get_kb_desc(self) -> "RetrievableKnowledge":
        # read the kb settings from the folder
        kb_file_path = os.path.join(self.folder_path, "kb.json")
        with open(kb_file_path, "r") as kb_file:
            return json.load(kb_file)

    def get_writer(self) -> VectorStoreWriter:
        return VectorStoreWriter(self._project_key, self._kb_full_id, self)

    def remove(self):
        """
        Removes this folder from the disk.
        """
        if os.path.exists(self.folder_path):
            logger.info("remove isolated folder {}".format(self.folder_path))
            shutil.rmtree(self.folder_path, ignore_errors=True)

    def create_langchain_vectorstore(self, **vectorstore_kwargs) -> 'VectorStore':
        """
        Creates a Langchain Vectorstore object backed by this folder.

        :rtype: :class:`langchain_core.vectorstores.VectorStore`
        """
        return create_langchain_vector_store(self.folder_path, **vectorstore_kwargs)
