(function () {
    "use strict";

    angular.module("dataiku.aiExplanations").factory("AIExplanationService", AIExplanationService);

    function AIExplanationService(
        $q,
        DataikuAPI,
        FutureWatcher,
        WT1,
        translate,
        LocalStorage,
        $rootScope
    ) {
        function explainProject(projectSummary, options, onInitialResponseReceived) {
            return explainStuff(
                DataikuAPI.flow.ai.startExplainProject.bind(null, projectSummary.projectKey, options),
                onInitialResponseReceived,
                "explain-project"
            );
        }

        function explainFlowZone(flowZone, options, onInitialResponseReceived) {
            return explainStuff(
                DataikuAPI.flow.ai.startExplainFlowZone.bind(
                    null,
                    flowZone.projectKey,
                    flowZone.cleanId,
                    options
                ),
                onInitialResponseReceived,
                "explain-flow-zone"
            );
        }

        function explainDataset(dataset, options, onInitialResponseReceived) {
            return explainStuff(
                DataikuAPI.flow.ai.startExplainDataset.bind(null, dataset.projectKey, dataset.name, options),
                onInitialResponseReceived,
                "explain-dataset"
            );
        }

        function explainObject(objectType, object, options, onInitialResponseReceived) {
            let fn;
            switch (objectType) {
                case "PROJECT":
                    fn = explainProject;
                    break;
                case "FLOW_ZONE":
                    fn = explainFlowZone;
                    break;
                case "DATASET":
                    fn = explainDataset;
                    break;
                default:
                    throw new Error("Unsupported object type");
            }
            return fn(object, options, onInitialResponseReceived);
        }

        function explainCode(language, code, options, onInitialResponseReceived) {
            return explainStuff(
                DataikuAPI.code.ai.startExplainCode.bind(null, language, code, options),
                onInitialResponseReceived,
                "explain-code"
            );
        }

        function explainStuff(startExplainFn, onInitialResponseReceived, wt1EventTypePrefix) {
            return $q(function(resolve, reject) {
                function success(data) {
                    resolve(data);
                    WT1.event(`${wt1EventTypePrefix}-request-successful`, {
                        aborted: data.aborted,
                        aiServer: $rootScope.appConfig.isUsingLocalAiAssitant && $rootScope.appConfig.isUsingLocalAiAssitant.aiExplanations ? "webapp" : "default"
                    });
                }

                function fail(data, status, headers) {
                    reject({ data, status, headers });
                    WT1.event(`${wt1EventTypePrefix}-request-failed`,
                      {
                          reason: getErrorDetails(data, status, headers).message,
                          aiServer: $rootScope.appConfig.isUsingLocalAiAssitant && $rootScope.appConfig.isUsingLocalAiAssitant.aiExplanations ? "webapp" : "default"
                      });
                }

                startExplainFn().success(function(initialResponse) {
                    if (initialResponse.hasResponse) {
                        success(initialResponse);
                    } else {
                        if (onInitialResponseReceived) {
                            onInitialResponseReceived(initialResponse);
                        }
                        FutureWatcher.watchJobId(initialResponse.jobId)
                            .success(success)
                            .error(fail);
                    }
                })
                .error(fail);
            });
        }

        function abortExplain(jobId) {
            return $q(function(resolve, reject) {
                DataikuAPI.futures.abort(jobId)
                .success(function() {
                    resolve();
                    WT1.event("explanation-request-aborted", { aiServer: $rootScope.appConfig.isUsingLocalAiAssitant && $rootScope.appConfig.isUsingLocalAiAssitant.aiExplanations ? "webapp" : "default" });
                })
                .error(function(data, status, headers) {
                    reject({ data, status, headers })
                    WT1.event("explaination-request-abort-failed", { aiServer: $rootScope.appConfig.isUsingLocalAiAssitant && $rootScope.appConfig.isUsingLocalAiAssitant.aiExplanations ? "webapp" : "default" });
                });
            });
        }

        function getObjectNotExplainableReason(objectType) {
            switch (objectType) {
                case "PROJECT":
                case "FLOW_ZONE":
                    return "";
                default:
                    return translate("AI_EXPLANATION.GENERATE.REASONS.DEFAULT", "A description can only be generated for a project or a Flow zone");
            }
        }

        function createAvailableOptions() {
            return {
                language: {
                    dutch: translate("GLOBAL.LANGUAGE.DUTCH", "Dutch"),
                    english: translate("GLOBAL.LANGUAGE.ENGLISH", "English"),
                    french: translate("GLOBAL.LANGUAGE.FRENCH", "French"),
                    german: translate("GLOBAL.LANGUAGE.GERMAN", "German"),
                    japanese: translate("GLOBAL.LANGUAGE.JAPANESE", "Japanese"),
                    portuguese: translate("GLOBAL.LANGUAGE.PORTUGUESE", "Portuguese"),
                    spanish: translate("GLOBAL.LANGUAGE.SPANISH", "Spanish"),
                },
                purpose: {
                    GENERIC: "Generic",
                    TECHNICAL: "Technical explanation",
                    BUSINESS_ORIENTED: "Business oriented explanation of the outcome",
                    EXECUTIVE: "Executive overview",
                },
                verbosity: {
                    LOW: "Short",
                    MEDIUM: "Medium",
                    HIGH: "Long",
                },
            };
        }

        const LOCAL_STORAGE_KEY = "dss.aiExplanations.explanationOptions";

        function loadOptions() {
            const defaultOptions = {
                language: "english",
                purpose: "GENERIC",
                verbosity: "MEDIUM",
            };
            const storedOptions = LocalStorage.get(LOCAL_STORAGE_KEY);
            if (typeof storedOptions !== "object") {
                return defaultOptions;
            }
            const availableOptions = createAvailableOptions();
            return Object.fromEntries(Object.keys(defaultOptions).map(
                function(optionKey) {
                    return [
                        optionKey,
                        storedOptions[optionKey] in availableOptions[optionKey]
                            ? storedOptions[optionKey]
                            : defaultOptions[optionKey]
                    ];
                }
            ));
        }

        function storeOptions(options) {
            LocalStorage.set(LOCAL_STORAGE_KEY, options);
        }

        function getDefaultEmptyMessage(messageType) {
            switch (messageType) {
                case "AI_SECTION":
                    return translate("MODALS.GENERATE_METADATA.AI_SECTION_MESSAGE", 'Configure your parameters and click "GENERATE".');
                case "EXISTING_SECTION":
                    return translate("MODALS.GENERATE_METADATA.EXISTING_SECTION_MESSAGE", "No description available");
                case "AI_DISABLED_SECTION":
                    return translate("MODALS.GENERATE_METADATA.AI_DISABLED_SECTION_MESSAGE", "Enter your description");
                default: 
                    return "";
            }
        }

        return {
            explainProject,
            explainFlowZone,
            explainObject,
            explainCode,
            abortExplain,
            getObjectNotExplainableReason,
            createAvailableOptions,
            loadOptions,
            storeOptions,
            getDefaultEmptyMessage
        };
    }
})();
