(function () {
    'use strict';

    var app = angular.module('dataiku.connections', ['dataiku.credentials']);

    /* Base controller used both for ConnectionsList and hive indexing (as a lot of layout and actions are common) */
    app.controller("_ConnectionsListBaseController", function ($scope, $rootScope, TopNav, DataikuAPI, Dialogs, $timeout, $filter, $state, ConnectionUtils, CreateModalFromTemplate) {
        $scope.noTags = true;
        $scope.noStar = true;
        $scope.noWatch = true;
        $scope.canIndexConnections = true;
        $scope.canCreateConnection = $state.is('admin.connections.list');
        $scope.noDelete = !$state.is('admin.connections.list');
        $scope.selection = $.extend({
            filterQuery: {
                userQuery: '',
                type: []
            },
            filterParams: {
                userQueryTargets: ["name", "type"],
                exactMatch: ["type"]
            },
            orderQuery: "creationTag.lastModifiedOn",
            orderReversed: false,
        }, $scope.selection || {});
        $scope.sortBy = [
            {value: 'name', label: 'Name'},
            {value: 'type', label: 'Type'},
            {value: 'creationTag.lastModifiedOn', label: 'Creation date'}
        ];
        $scope.uiState = $scope.uiState || {};
        $scope.uiState.newConnectionQuery = '';
        $scope.connectionGroups = buildConnectionList();

        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        let pollPromise;

        function processRunningJobsRequest(data) {
            let doRunningJobsRemain = false;

            $scope.listItems.forEach(conn => {
                if (conn.name in data) {
                    conn.indexingMetadata = data[conn.name];
                    if (conn.indexingMetadata.currentJobId) {
                        doRunningJobsRemain = true;
                    }
                } else if (conn.indexingMetadata && conn.metadata.currentJobId) {
                    delete conn.indexingMetadata.currentJobId;
                }
            });
            return doRunningJobsRemain;
        }

        $scope.pollRunningJobs = function () {
            $timeout.cancel(pollPromise);
            DataikuAPI.admin.connections.listRunningJobs().success(function (data) {
                let doRunningJobsRemain = processRunningJobsRequest(data);
                if (doRunningJobsRemain) {
                    pollPromise = $timeout(() => {
                        $scope.pollRunningJobs();
                    }, 5000);
                }
            });
        }

        /** Order connections by their nice connection type rather than the raw type */
        $scope.niceConnectionTypeComparator = (a,b) => {
            return $filter("connectionTypeToNameForList")(a.value).localeCompare($filter("connectionTypeToNameForList")(b.value));
        }

        /** Adds or removes the specified connection type from the filter depending on whether it's already there */
        $scope.setConnectionTypeFilter = (connectionType) => {
            const index = $scope.selection.filterQuery.type.indexOf(connectionType);
            if (index > -1) {
                $scope.selection.filterQuery.type.splice(index, 1);;
            } else {
                $scope.selection.filterQuery.type.push(connectionType);
            }
        }

        $scope.isIndexable = connection => ConnectionUtils.isIndexable(connection);

        $scope.list = function () {
            var func = $state.is('admin.connections.hiveindexing') ? DataikuAPI.admin.connections.listHiveVirtual : DataikuAPI.admin.connections.list;
            func().success(function (data) {
                $scope.connections = data;
                $scope.listItems = $.map(data, function (v, k) {
                    return v;
                });
                $scope.connectionTypes = [...new Set($scope.listItems.map(it => it.type))];
                $scope.canIndexAllConnections = $scope.listItems.length > 0;
                if ($scope.listItems.find(c => c.indexingMetadata && c.indexingMetadata.currentJobId)) {
                    $scope.pollRunningJobs();
                }
            }).error(setErrorInScope.bind($scope));
        };
        $scope.$on("$destroy", function () {
            pollPromise && $timeout.cancel(pollPromise);
        });

        $scope.$on("indexConnectionEvent", (event, connectionName) => {
            createIndexConnectionsModal([connectionName]);
        });

        $scope.list();

        $scope.deleteSelected = function (name) {
            var selectedConnectionsNames = $scope.selection.selectedObjects.map(function (c) {
                return c.name;
            });
            Dialogs.confirm($scope, 'Connection deletion', 'Are you sure you want to delete this connection ?').then(function () {
                DataikuAPI.admin.connections.delete(selectedConnectionsNames).success(function (data) {
                    $scope.list();
                }).error(setErrorInScope.bind($scope));
            });
        };

        let createIndexConnectionsModal = function (connectionNames) {
            const newScope = $scope.$new();
            newScope.selectedConnections = connectionNames;
            CreateModalFromTemplate("/templates/admin/index-connections-modal.html", newScope, 'IndexConnectionsModalController');
        };

        $scope.indexSelectedConnections = function () {
            createIndexConnectionsModal($scope.selection.selectedObjects.map(function (c) {
                return c.name;
            }));
        };

        $scope.isIndexationRunning = function () {
            return $scope.listItems && $scope.listItems.find(c => {
                    return c && c.indexingMetadata && c.indexingMetadata.currentJobId;
                });
        };

        $scope.abortIndexation = function () {
            DataikuAPI.admin.connections.abortIndexation()
                .success(function (data) {
                    processRunningJobsRequest(data);
                })
                .error(setErrorInScope.bind($scope));
        };

        $scope.indexAllConnections = function () {
            CreateModalFromTemplate("/templates/admin/index-connections-modal.html", $scope, 'IndexConnectionsModalController');
        };

        $scope.focusNewConnectionFilter = function() {
            $timeout(() => {
                document.querySelector('#new-connection-query').focus();
            })
        }

        $scope.clearNewConnectionQuery = function($event) {
            $scope.uiState.newConnectionQuery = '';
            $scope.focusNewConnectionFilter();
            $event.stopPropagation();
        }

        $scope.filterNewConnectionGroups = function(connectionGroup) {
            return connectionGroup.connections.some(connection => connectionFound(connection, $scope.uiState.newConnectionQuery));
        }

        $scope.filterNewConnections = function(connection) {
            return connectionFound(connection, $scope.uiState.newConnectionQuery);
        }

        function connectionFound(connection, query) {
            query = query?.toLowerCase() || '';
            return (connection.title || '').toLowerCase().includes(query) || (connection.type || '').toLowerCase().includes(query) || (connection.category || '').toLowerCase().includes(query);
        }

        function buildConnectionList() {
            const categories = [
                {
                    title: 'SQL Databases',
                    connections: [
                        { type: 'Snowflake' },
                        { type: 'Databricks' },
                        { type: 'Redshift', title: 'Amazon Redshift' },
                        { type: 'BigQuery', title: 'Google BigQuery' },
                        { type: 'Synapse', title: 'Azure Synapse' },
                        { type: 'FabricWarehouse', title: 'MS Fabric Warehouse' },
                        { type: 'PostgreSQL' },
                        { type: 'MySQL' },
                        { type: 'SQLServer', title: 'MS SQL Server' },
                        { type: 'Oracle' },
                        { type: 'Teradata' },
                        ...($rootScope.featureFlagEnabled('lakebase') ? [{ type: 'DatabricksLakebase', title: 'Databricks Lakebase' }] : []),
                        { type: 'AlloyDB', title: 'Google AlloyDB' },
                        { type: 'Athena' },
                        { type: 'Greenplum' },
                        { type: 'Vertica' },
                        { type: 'SAPHANA' },
                        { type: 'Netezza', title: 'IBM Netezza' },
                        { type: 'Trino'},
                        { type: 'TreasureData' },
                        { type: 'Denodo' },
                        ...($rootScope.featureFlagEnabled('kdbplus') ? [{ type: 'KDBPlus', title: 'KDB+' }] : []),
                        { type: 'JDBC', title: 'Other SQL databases' },
                    ]
                }, {
                    title: 'File-based',
                    connections: [
                        ...($scope.appConfig.canAccessCloudDataikerAdminCapabilities && $scope.appConfig.admin ? [{ type: 'Filesystem', title: 'Server Filesystem' }] : []),
                        { type: 'FTP' },
                        { type: 'SSH', title: 'SCP/SFTP' },
                    ]
                }, {
                    title: 'Cloud Storages & Hadoop',
                    connections: [
                        { type: 'EC2', title: 'Amazon S3' },
                        { type: 'Azure', title: 'Azure Blob Storage' },
                        { type: 'GCS', title: 'Google Cloud Storage' },
                        ...($scope.appConfig.canAccessCloudDataikerAdminCapabilities ? [{ type: 'HDFS', title: 'Hadoop HDFS' }] : []),
                        { type: 'SharePointOnline', title: 'SharePoint Online' },
                    ]
                }, {
                    title: 'NoSQL',
                    connections: [
                        { type: 'MongoDB' },
                        { type: 'Cassandra' },
                        { type: 'ElasticSearch' },
                        ...($rootScope.featureFlagEnabled('DynamoDB') ? [{ type: 'DynamoDB' }] : []),
                    ]
                }, {
                    title: 'Managed Model Deployment Infrastructures',
                    connections: [
                        { type: 'SageMaker', title: 'Amazon SageMaker' },
                        { type: 'AzureML', title: 'Azure Machine Learning' },
                        { type: 'VertexAIModelDeployment', title: 'Google Vertex AI' },
                        { type: 'DatabricksModelDeployment', title: 'Databricks Managed Model Deployment' },
                    ]
                }, {
                    title: 'LLM Mesh',
                    connections: [
                        { type: 'OpenAI' },
                        { type: 'AzureOpenAI', title: 'Azure OpenAI' },
                        { type: 'AzureLLM', title: 'Azure LLM' },
                        { type: 'Cohere' },
                        { type: 'Anthropic' },
                        { type: 'VertexAILLM', title: 'Vertex Generative AI' },
                        { type: 'Bedrock', title: 'AWS Bedrock' },
                        { type: 'DatabricksLLM', title: 'Databricks Mosaic AI' },
                        { type: 'SnowflakeCortex', title: 'Snowflake Cortex' },
                        { type: 'MistralAI' },
                        { type: 'HuggingFaceLocal', title: 'Hugging Face' },
                        { type: 'SageMaker-GenericLLM', title: 'SageMaker LLM' },
                        { type: 'StabilityAI' },
                        { type: 'CustomLLM', title: 'Custom LLM' },
                        { type: 'Pinecone' },
                        { type: 'AzureAISearch', title: 'Azure AI Search' },
                        { type: 'NVIDIA-NIM', title: 'NVIDIA NIM' },
                    ]
                }, {
                    title: 'Other',
                    connections: [
                        { type: 'Twitter' },
                        ...($scope.appConfig.streamingEnabled ? [{ type: 'Kafka' }] : []),
                        ...($scope.appConfig.streamingEnabled ? [{ type: 'SQS' }] : []),
                    ]
                }
            ];

            categories.forEach(({ title, connections }) => connections.forEach(connection => connection.category = title));

            return categories;
        }
    });

    app.controller("ConnectionsController", function ($scope, $controller, TopNav, DataikuAPI, Dialogs, $timeout, $stateParams, ConnectionUtils, CreateModalFromTemplate) {
        $controller("_ConnectionsListBaseController", { $scope: $scope });
    });

    app.controller("ConnectionsHiveIndexingController", function ($scope, $controller, TopNav, DataikuAPI, Dialogs, $timeout, $stateParams, ConnectionUtils, CreateModalFromTemplate) {
        $controller("_ConnectionsListBaseController", { $scope: $scope });
    });

    app.controller("IndexConnectionsModalController", function ($scope, $state, $stateParams, TopNav, DataikuAPI, $timeout, FutureProgressModal, Dialogs) {
        $scope.indexationMode = 'scan';
        $scope.loading = true;

        DataikuAPI.admin.connections.listProcessableConnections($scope.type, $scope.selectedConnections).success(function (response) {
            $timeout(() => {
                $scope.loading = false;
                $scope.processableConnections = response;
                $scope.$digest();
            })
        }).error(setErrorInScope.bind($scope));

        $scope.canStartIndexation = function () {
            return $scope.processableConnections && (
                    $scope.indexationMode === 'index' && $scope.processableConnections.indexableConnections.length ||
                    $scope.indexationMode === 'scan' && $scope.processableConnections.scannableConnections.length
                );
        };
        $scope.startIndexation = function () {
            let indexationFunction;
            let connectionsToProcess;

            if ($scope.indexationMode === 'index') {
                indexationFunction = DataikuAPI.admin.connections.index;
                connectionsToProcess = $scope.processableConnections.indexableConnections;
            } else if ($scope.indexationMode === 'scan') {
                indexationFunction = DataikuAPI.admin.connections.scan;
                connectionsToProcess = $scope.processableConnections.scannableConnections;
            }

            indexationFunction(connectionsToProcess).success(function (data) { // NOSONAR: OK this call is indeed a method call.
                var parentScope = $scope.$parent.$parent;
                $scope.pollRunningJobs();
                FutureProgressModal.show(parentScope, data, "Indexing", newScope => newScope.tellToCloseWindow = true).then(function(result){
                    if (result) {
                        Dialogs.infoMessagesDisplayOnly(parentScope, "Indexing result", result);
                    }
                });
                $scope.dismiss();
            }).error(setErrorInScope.bind($scope));
        };
    });

    app.controller("ConnectionController", function ($scope, $rootScope, $state, $stateParams, Assert, TopNav, DataikuAPI, SqlConnectionNamespaceService) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.connectionParamsForm = {};

        $scope.availableCatalogs = null;
        $scope.availableSchemasMap = {};

        // Angular guys do not want to support the validation of an Angular input inside an AngularJS form, so handling validity through booleans.
        // See https://github.com/angular/angular/issues/9213.
        $scope.areAdvancedPropertiesInvalid = false;
        $scope.areAdvancedConnectionPropertiesInvalid = false;
        $scope.areJobLabelsInvalid = false;

        $scope.setAdvancedPropertiesValidity = function(isValid) {
            $scope.areAdvancedPropertiesInvalid = !isValid;
        };

        $scope.setAdvancedConnectionPropertiesValidity = function(isValid) {
            $scope.areAdvancedConnectionPropertiesInvalid = !isValid;
        };

        $scope.setJobLabelsValidity = function(isValid) {
            $scope.areJobLabelsInvalid = !isValid;
        };

        $scope.isConnectionParamsFormInvalid = function() {
            return $scope.connectionParamsForm.$invalid
                || $scope.areAdvancedPropertiesInvalid
                || $scope.areAdvancedConnectionPropertiesInvalid
                || $scope.areJobLabelsInvalid;
        };

        DataikuAPI.admin.connections.list().success(function (data) {
            $scope.connections = data;
        }).error(setErrorInScope.bind($scope));

        DataikuAPI.security.listGroups(false).success(function (data) {
            if (data) {
                data.sort();
            }
            $scope.allGroups = data;
        });

        function $canHaveProxy(connectionType) {
            return $scope.appConfig.hasGlobalProxy &&
                ['ElasticSearch', 'HTTP', 'FTP', 'EC2', 'GCS', 'Twitter', 'BigQuery', 'Snowflake', 'Azure', 'SAPHANA', 'SQS', 'Databricks', 'AzureML', 'VertexAIModelDeployment', 'SageMaker', 'DatabricksModelDeployment', 'SharePointOnline'].indexOf(connectionType) >= 0;
        }

        $scope.isFsProviderizable = function (t) {
            return ['Filesystem', 'FTP', 'SSH', 'HDFS', 'Azure', 'GCS', 'EC2', 'SharePointOnline', 'DatabricksVolume'].indexOf(t) >= 0;
        };

        $scope.canBeUsedAsVectorStore = function (t) {
            return ['ElasticSearch', 'AzureAISearch', 'Pinecone', 'GCS'].indexOf(t) >= 0;
        };

        $scope.getKBDisabledReason = function() {
            if (!$scope.connection.allowWrite){
                $scope.connection.allowKnowledgeBanks = false; // not necessary since it's already done in the allowWrite watcher but for consistency
                return "Write permissions on the connection are required to use knowledge banks.";
            }
            // No incompatibility found
            return null;
        }

        function isNonDataConnection(connectionType) {
            const managedModelDeploymentInfra = ['SageMaker','AzureML','VertexAIModelDeployment','DatabricksModelDeployment'];
            const llmConnections = ["Anthropic", "AzureOpenAI", "Bedrock", "Cohere", "CustomLLM", "DatabricksLLM", "HuggingFaceInferenceAPI", "HuggingFaceLocal",
                 "MosaicML", "NVIDIA-NIM", "OpenAI", "SageMaker-GenericLLM", "SnowflakeCortex", "VertexAILLMConnection"]; // keep in sync w/ connection inheriting from AbstractLLMConnection (connections/AbstractLLMConnection.java)
            return llmConnections.includes(connectionType) || managedModelDeploymentInfra.includes(connectionType);
        }

        Assert.trueish($stateParams.connectionName || $stateParams.type, "no $stateParams.connectionName and no $stateParams.type");
        if ($stateParams.connectionName) {
            $scope.creation = false;
            DataikuAPI.admin.connections.get($stateParams.connectionName).success(function (data) {
                $scope.savedConnection = angular.copy(data);
                $scope.connection = data;
                $scope.connection.$canHaveProxy = $canHaveProxy(data.type);
                $scope.loadDone = true;
                SqlConnectionNamespaceService.setTooltips($scope, data.type);
                $scope.isPersonalConnection = data.owner != null;
                if($scope.isPersonalConnection) {
                    $scope.usableBy = {
                        "policy": data.usableBy === "ALLOWED" && _.isEmpty(data.allowedGroups) ? "CREATOR" : data.usableBy
                    };
                } else {
                    $scope.usableBy = {
                        "policy": data.usableBy
                    };
                }
            }).error(setErrorInScope.bind($scope));
        } else if ($stateParams.type) {
            $scope.creation = true;
            $scope.savedConnection = null;
            $scope.isPersonalConnection = !$rootScope.appConfig.admin;
            $scope.usableBy = {
                "policy": ($scope.isPersonalConnection) ? "CREATOR" : "ALL"
            };
            $scope.connection = {
                "type": $stateParams.type,
                "params": {
                    namingRule: {}
                },
                "credentialsMode": "GLOBAL",
                "allowMirror": ($stateParams.type == "Vertica" || $stateParams.type == "ElasticSearch"),
                "usableBy": "ALL",
                "$canHaveProxy": $canHaveProxy($stateParams.type),
                "useGlobalProxy": $stateParams.type == 'ElasticSearch' ? false : $canHaveProxy($stateParams.type)
            };

            /* Per connection defaults */
            if ($scope.connection.type == "BigQuery") {
                $scope.connection.params.properties = [
                    { "name": "Timeout", "value": 180, "secret": false }
                ]
                $scope.connection.params.driverMode = "DRIVERLESS";
                $scope.connection.params.authType = "KEYPAIR";
                $scope.connection.params.forbidPartitionsWriteToNonPartitionedTable = true;
            } else if ($scope.connection.type == "Redshift") {
                $scope.connection.params.driverMode = "MANAGED_LEGACY_POSTGRESQL";
                $scope.connection.params.redshiftAuthenticationMode = "USER_PASSWORD";
            } else if ($scope.connection.type == "Greenplum") {
                $scope.connection.params.driverMode = "MANAGED_LEGACY_POSTGRESQL";
            } else if ($scope.connection.type == "PostgreSQL" || $scope.connection.type === "AlloyDB"
                    || $scope.connection.type == "Snowflake" ||  $scope.connection.type == "Databricks" ||  $scope.connection.type == "DatabricksLakebase") {
                $scope.connection.params.driverMode = "MANAGED";
            } else if ($scope.connection.type == "Trino") {
                $scope.connection.params.driverMode = "MANAGED";
                $scope.connection.params.authType = "PASSWORD";
            } else if ($scope.connection.type == "Teradata") {
                $scope.connection.params.properties = [
                    { "name": "CHARSET", "value": "UTF8", "secret": false }
                ]
            } else if ($scope.connection.type == "SSH") {
                $scope.connection.params.port = 22;
            } else if ($scope.connection.type == "HDFS") {
                $scope.connection.params.hiveSynchronizationMode = 'KEEP_IN_SYNC';
            } else if ($scope.connection.type == "EC2") {
                $scope.connection.params.credentialsMode = "KEYPAIR";
                $scope.connection.params.switchToRegionFromBucket = true;
            } else if ($scope.connection.type == "Synapse") {
                // The first option is here to keep compatibility with SQLServer. It should not be used.
                $scope.connection.params.azureDWH = true;
                $scope.connection.params.autocommitMode = true;
            } else if ($scope.connection.type == "GCS") {
                $scope.connection.params.authType = "KEYPAIR";
            } else if ($scope.connection.type == "FabricWarehouse") {
                $scope.connection.params.grantType = "AUTHORIZATION_CODE";
                $scope.connection.params["azureOAuthLoginEnabled"] = true;
                $scope.connection.params["kerberosLoginEnabled"] = false;
                $scope.connection.credentialsMode = "GLOBAL";
                $scope.connection.params["port"] = 1433;
                $scope.connection.params["refreshTokenRotation"] = false;
            } else if ($scope.connection.type == "Oracle") {
                $scope.connection.params.maxIdentifierSize = 30;
            } else if ($scope.connection.type == "Vertica") {
                $scope.connection.params.usePkce = true;
            } else if ($scope.connection.type == "TreasureData") {
                $scope.connection.params.region = "US";
                $scope.connection.params.db = "td-presto";
                $scope.connection.params.driverMode = "MANAGED";
            }

            $scope.connection.allowManagedFolders = $scope.isFsProviderizable($scope.connection.type);
            $scope.connection.allowKnowledgeBanks = $scope.canBeUsedAsVectorStore($scope.connection.type);
            $scope.connection.allowManagedDatasets = $scope.connection.allowWrite = !isNonDataConnection($scope.connection.type);

            $scope.loadDone = true;
        }
        $scope.customFieldsMap = $rootScope.appConfig.customFieldsMap["CONNECTION"];

        $scope.isConnectionNameUnique = function (v) {
            if (v == null) return true;
            if ($scope.connections == null) return true;
            return !$scope.connections.hasOwnProperty(v);
        };

        $scope.isConnectionNameValid = function () {
            return $scope.connection && $scope.connection.name && $scope.connection.name.length;
        };

        function usableByToConnection(connection, usableByPolicy) {
            const tmpConnection = angular.copy(connection);
            switch (usableByPolicy) {
                case "CREATOR":
                    tmpConnection.usableBy = "ALLOWED";
                    tmpConnection.allowedGroups = [];
                    break;
                default:
                    tmpConnection.usableBy = usableByPolicy;
            }

            return tmpConnection;
        }

        $scope.connectionDirty = function () {
            return !angular.equals(usableByToConnection($scope.connection, $scope.usableBy.policy), $scope.savedConnection);
        };

        $scope.saveConnection = function () {
            if ($scope.isConnectionParamsFormInvalid()) {
                return;
            }
            if ($scope.connection.params.regionOrEndpoint != null && $scope.connection.params.regionOrEndpoint.toLowerCase().startsWith('https://')) {
                $scope.connection.params.switchToRegionFromBucket = false;
            }
            const tmpConnection = usableByToConnection($scope.connection, $scope.usableBy.policy);
            DataikuAPI.admin.connections.save(tmpConnection, $scope.creation).success(function (data) {
                $scope.savedConnection = tmpConnection;
                // reset available schemas everytime to permit the user to refresh schemas if needed.
                // ideally we should only refresh it if basic params, advanced params or credentials have changed but it is a bit complicated to check
                $scope.availableCatalogs = null;
                $scope.availableSchemasMap = {};
                $state.transitionTo("admin.connections.edit", {connectionName: $scope.connection.name});
            }).error(setErrorInScope.bind($scope));
        };

        $scope.$watch('connection.allowWrite', function (a) {
            if (!a && $scope.connection) {
                $scope.connection.allowManagedDatasets = false;
                $scope.connection.allowManagedFolders = false;
                $scope.connection.allowKnowledgeBanks = false;
            }
        });

        $scope.canSwitchToRegionFromBucket = function(endpoint) {
            return !endpoint || !endpoint.toLowerCase().startsWith('https://');
        };

        $scope.fetchCatalogs = function(connectionName, origin, connectionType, inputBtnName) {
            SqlConnectionNamespaceService.listSqlCatalogs(connectionName, $scope, origin, connectionType, inputBtnName);
        };

        $scope.fetchSchemas = function(connectionName, catalog, origin, connectionType, inputBtnName) {
            SqlConnectionNamespaceService.listSqlSchemas(connectionName, $scope, catalog, origin, connectionType, inputBtnName);
        };

        $scope.$on('$destroy', function() {
            SqlConnectionNamespaceService.abortListSqlSchemas($scope);
            SqlConnectionNamespaceService.abortListSqlCatalogs($scope);
        });

    });

    app.controller("SQLConnectionController", function ($scope, $controller, DataikuAPI, $timeout, $rootScope, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.supportsWriteSQLComment = function(type) {
            return ["Snowflake", "Databricks", "PostgreSQL", "Oracle", "Redshift", "MySQL", "BigQuery"].includes(type)
        }

        $scope.supportsWriteSQLCommentInCreateTableStatement = (type) => {
            return ["Snowflake", "BigQuery", "Databricks", "MySQL"].includes(type)
        }

        if (!$scope.connection.params.properties) {
            $scope.connection.params.properties = [];
        }

        if ($scope.creation) {
            const dp = $rootScope.appConfig && $rootScope.appConfig.namingRulesSettings && $rootScope.appConfig.namingRulesSettings.defaultPrefixForTables ?
                    $rootScope.appConfig.namingRulesSettings.defaultPrefixForTables : "${projectKey}_";
            $scope.connection.params.namingRule.tableNameDatasetNamePrefix = dp;
            $scope.connection.params.namingRule.writeDescriptionsAsSQLComment = $scope.supportsWriteSQLCommentInCreateTableStatement($scope.connection.type);
        }

        if (!$scope.connection.params.dkuProperties) {
            $scope.connection.params.dkuProperties = [];
        }

        $scope.warnings = {
            noVariableInTable: false,
        };

        if (!$scope.connection.customBasicConnectionCredentialProviderParams) {
            $scope.connection.customBasicConnectionCredentialProviderParams = [];
        }

        $scope.checkForHttpInHostUrl = (host) => host && (host.startsWith('http://') || host.startsWith('https://'));

        // Force teradata timezone to GMT on connection creation
        if ($scope.connection.type === "Teradata" && $scope.creation) {
            $scope.connection.params.defaultAssumedTzForUnknownTz = "GMT";
            $scope.connection.params.defaultAssumedDbTzForUnknownTz = "GMT";
        }

        $scope.$watch("connection.params", function (nv, ov) {
            $scope.warnings.noVariableInTable = false;
            // Snowflake and BigQuery don't support global Oauth yet ch63879
            if ($scope.connection.type=="BigQuery") {
                if (nv.authType=="OAUTH") {
                    $scope.connection.credentialsMode = "PER_USER";
                } else if (nv.authType=="KEYPAIR") {
                    $scope.connection.credentialsMode = "GLOBAL";
                }
            }
            if ($scope.connection.type=="Databricks" && nv.authType=="OAUTH2_APP" && $scope.connection.params.authType != $scope.savedConnection.params.authType) {
                $scope.connection.credentialsMode = "PER_USER";
            }
            if ($scope.connection.type=="Trino" && nv.authType=="OAUTH2" && $scope.connection.params.authType != $scope.savedConnection.params.authType) {
                $scope.connection.credentialsMode = "PER_USER";
            }
            if (!nv) return;
            if (!$scope.connection.allowManagedDatasets) return;

            if ((!nv.namingRule.tableNameDatasetNamePrefix || nv.namingRule.tableNameDatasetNamePrefix.indexOf("${") == -1) &&
                (!nv.namingRule.tableNameDatasetNameSuffix || nv.namingRule.tableNameDatasetNameSuffix.indexOf("${") == -1) &&
                (!nv.namingRule.schemaName || nv.namingRule.schemaName.indexOf("${") == -1)) {
                $scope.warnings.noVariableInTable = true;
            }
        }, true);
        $scope.$watch("connection.credentialsMode", function (nv, ov) {
            if ($scope.connection.type=="BigQuery") {
                if (nv=="GLOBAL") {
                    if ($scope.connection.params.authType == "OAUTH") {
                        $scope.connection.params.authType = "KEYPAIR";
                    }
                } else if (nv=="PER_USER") {
                    $scope.connection.params.authType = "OAUTH";
                }
            }
        });


        $scope.uiState = {};

        $scope.testConnection = function () {
            if (!$scope.connectionParamsForm || $scope.connectionParamsForm.$valid) {
                DataikuAPI.admin.connections.testSQL($scope.connection, null).success(function (data) {
                    $scope.testResult = data;
                }).error(setErrorInScope.bind($scope));
            }
        };

        $scope.getCatalog = function () {
            if ($scope.connection.type === "BigQuery") {
                return $scope.connection.params.projectId;
            }
            if ($scope.connection.type === "Snowflake") {
                return $scope.connection.params.db;
            }
            if ($scope.connection.type === "Databricks") {
                return $scope.connection.params.defaultCatalog;
            }
            if ($scope.connection.type === "SQLServer") {
                return $scope.connection.params.db;
            }
            if ($scope.connection.type === "MySQL") {
                return $scope.connection.params.db;
            }
            return null;
        }

        $scope.$watch("connection", function (nv, ov) {
            if (nv != null) {
                if (!$scope.connection.params.properties) {
                    $scope.connection.params.properties = [];
                }
            }
        });

        $scope.warnAboutSearchPath = function () {
            if ($scope.connection.params.schemaSearchPath) {
                if ($scope.connection.params.namingRule.schemaName) {
                    return false;
                }
                if ($scope.connection.params.schemaSearchPath.indexOf(',public,') > 0) { // NOSONAR: OK to ignore 0 index.
                    return false;
                }
                if ($scope.connection.params.schemaSearchPath.endsWith(',public')) {
                    return false;
                }
                if ($scope.connection.params.schemaSearchPath.startsWith('public,')) {
                    return false;
                }
                if ($scope.connection.params.schemaSearchPath == 'public') {
                    return false;
                }
                return true;
            } else {
                // no schema search path => don't care
                return false;
            }
        };

        $scope.connectionHasConceptOfDefaultCatalogAndSchema = function() {
            if (!$scope.connection) return false;

            /* Keep in sync with SQLUtils.resolveCatalogFromConnectionDefault / SQLUtils.resolveSchemaFromConnectionDefault */
            return ["MySQL", "Snowflake", "BigQuery", "SQLServer", "Databricks"].indexOf($scope.connection.type) >= 0;
        }

         $scope.dialects = [
            {"value":"","label":"Default"},
            {"value":"MySQLDialect","label":"MySQL < 8.0"},
            {"value":"MySQL8Dialect","label":"MySQL >= 8.0"},
            {"value":"PostgreSQLDialect","label":"PostgreSQL"},
            {"value":"OracleSQLDialect","label":"Oracle"},
            {"value":"SQLServerSQLDialect","label":"SQL Server"},
            {"value":"SynapseSQLDialect","label":"Azure Synapse"},
            {"value":"FabricWarehouseSQLDialect","label":"MS Fabric Warehouse"},
            {"value":"GreenplumSQLDialect","label":"Greenplum < 5.0"},
            {"value":"Greenplum5SQLDialect","label":"Greenplum >= 5.0"},
            {"value":"TeradataSQLDialect","label":"Teradata"},
            {"value":"VerticaSQLDialect","label":"Vertica"},
            {"value":"RedshiftSQLDialect","label":"Redshift"},
            {"value":"SybaseIQSQLDialect","label":"Sybase IQ"},
            {"value":"AsterDataSQLDialect","label":"Aster Data"},
            {"value":"NetezzaSQLDialect","label":"IBM Netezza"},
            {"value":"BigQuerySQLDialect","label":"Google BigQuery"},
            {"value":"SAPHANASQLDialect","label":"SAP HANA"},
            {"value":"ExasolSQLDialect","label":"Exasol"},
            {"value":"SnowflakeSQLDialect","label":"Snowflake"},
            {"value":"DatabricksSQLDialect","label":"Databricks"},
            {"value":"DB2SQLDialect","label":"IBM DB2"},
            {"value":"H2SQLDialect","label":"H2 < 2.0"},
            {"value":"H2V2SQLDialect","label":"H2 >= 2.0"},
            {"value":"ImpalaSQLDialect","label":"Impala"},
            {"value":"HiveSQLDialect","label":"Hive"},
            {"value":"PrestoSQLDialect","label":"Presto"},
            {"value":"TrinoSQLDialect","label":"Trino"},
            {"value":"AthenaSQLDialect","label":"Athena"},
            {"value":"SparkSQLDialect","label":"SparkSQL (via JDBC)"},
            {"value":"SqreamSQLDialect","label":"SQream"},
            {"value":"YellowbrickSQLDialect","label":"Yellowbrick"},
            {"value":"DremioSQLDialect","label":"Dremio"},
            {"value":"DenodoSQLDialect","label":"Denodo"}
        ];
        if ($rootScope.featureFlagEnabled("kdbplus")) {
            $scope.dialects.push({"value":"KDBSQLDialect","label":"KDB+"});
        }
        $rootScope.appConfig.customDialects.forEach(function(d) {
            $scope.dialects.push({"value":d.dialectType, "label":d.desc.meta.label || d.id})
        });
    });

    app.controller("PostgreSQLConnectionController", function ($scope, $controller, DataikuAPI) {
        $controller('SQLConnectionController', {$scope: $scope});

        $scope.testConnection = function () {
            if (!$scope.connectionParamsForm || $scope.connectionParamsForm.$valid) {
                $scope.testing = true;
                DataikuAPI.admin.connections.testPostgreSQL($scope.connection).success(function (data) {
                    $scope.testing = false;
                    $scope.testResult = data;
                }).error(function (a, b, c) {
                    $scope.testing = false;
                    setErrorInScope.bind($scope)(a, b, c)
                });
            }
        }
    });

    app.controller("FilesystemConnectionController", function ($scope, $controller, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.notTestable = true;

        if (!$scope.connection.params.dkuProperties) {
            $scope.connection.params.dkuProperties = [];
        }
    });

    app.controller("HDFSConnectionController", function ($scope, $controller, $rootScope, TopNav, $stateParams, DataikuAPI, FutureProgressModal, Dialogs) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.isExtraHadoopConfigurationInvalid = false;

        $scope.setExtraHadoopConfigurationValidity = function(isValid) {
            $scope.isExtraHadoopConfigurationInvalid = !isValid;
        }

        // Overriding Connection Common
        $scope.isConnectionParamsFormInvalid = function() {
            return $scope.connectionParamsForm.$invalid || $scope.isExtraHadoopConfigurationInvalid || $scope.areAdvancedConnectionPropertiesInvalid;
        }

        $scope.notTestable = true;

        if ($scope.creation) {
            const dpp = $rootScope.appConfig && $rootScope.appConfig.namingRulesSettings && $rootScope.appConfig.namingRulesSettings.defaultPrefixForPath ?
                                $rootScope.appConfig.namingRulesSettings.defaultPrefixForPath : "${projectKey}/";
            const dpt = $rootScope.appConfig && $rootScope.appConfig.namingRulesSettings && $rootScope.appConfig.namingRulesSettings.defaultPrefixForTables ?
                                $rootScope.appConfig.namingRulesSettings.defaultPrefixForTables : "${projectKey}_";
            $scope.connection.params.namingRule.hdfsPathDatasetNamePrefix = dpp;
            $scope.connection.params.namingRule.tableNameDatasetNamePrefix = dpt;
        }

        if (!$scope.connection.customBasicConnectionCredentialProviderParams) {
            $scope.connection.customBasicConnectionCredentialProviderParams = [];
        }

        if (!$scope.connection.params.dkuProperties) {
            $scope.connection.params.dkuProperties = [];
        }

        $scope.warnings = {
            noVariableInPath: false,
            noVariableInHive: false,
        };

        $scope.$watch("connection.params", function (nv, ov) {
            $scope.warnings.noVariableInPath = false;
            $scope.warnings.noVariableInHive = false;

            if (!nv) return;
            if (!$scope.connection.allowManagedDatasets) return;


            if ((!nv.namingRule.hdfsPathDatasetNamePrefix || nv.namingRule.hdfsPathDatasetNamePrefix.indexOf("${") == -1) &&
                (!nv.namingRule.hdfsPathDatasetNameSuffix || nv.namingRule.hdfsPathDatasetNameSuffix.indexOf("${") == -1)) {
                $scope.warnings.noVariableInPath = true;
            }

            if ((!nv.namingRule.tableNameDatasetNamePrefix || nv.namingRule.tableNameDatasetNamePrefix.indexOf("${") == -1) &&
                (!nv.namingRule.tableNameDatasetNameSuffix || nv.namingRule.tableNameDatasetNameSuffix.indexOf("${") == -1) &&
                (!nv.namingRule.hiveDatabaseName || nv.namingRule.hiveDatabaseName.indexOf("${") == -1)) {
                $scope.warnings.noVariableInHive = true;
            }
        }, true);

        $scope.resyncPermissions = function () {
            DataikuAPI.admin.connections.hdfs.resyncPermissions($stateParams.connectionName).success(function (data) {
                FutureProgressModal.show($scope, data, "Permissions update").then(function (result) {
                    Dialogs.infoMessagesDisplayOnly($scope, "Update result", result);
                })
            }).error(setErrorInScope.bind($scope));
        }

        $scope.resyncRootPermissions = function () {
            DataikuAPI.admin.connections.hdfs.resyncRootPermissions($stateParams.connectionName).success(function (data) {
                FutureProgressModal.show($scope, data, "Permissions update");
            }).error(setErrorInScope.bind($scope));
        }

        DataikuAPI.projects.list().success(function (data) {
            $scope.projectsList = data;
            if (data.length) {
                $scope.massImportTargetProjectKey = data[0].projectKey;
                $scope.massImportTargetProjectName = data[0].name;
            }
            $scope.$watch("massImportTargetProjectKey", function () {
                var filteredProjects = $scope.projectsList.filter(function (project) {
                    return project.projectKey == $scope.massImportTargetProjectKey;
                });
                if (filteredProjects && filteredProjects.length) {
                    $scope.massImportTargetProjectName = filteredProjects[0].name;
                } else {
                    $scope.massImportTargetProjectName = null;
                }
            })

        }).error(setErrorInScope.bind($scope));
    });


    app.controller("EC2ConnectionController", function ($scope, $controller, DataikuAPI, TopNav, $rootScope) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.testConnection = function () {
            if (!$scope.connectionParamsForm || $scope.connectionParamsForm.$valid) {
                $scope.testing = true;
                if ($scope.connection.params.regionOrEndpoint != null && $scope.connection.params.regionOrEndpoint.toLowerCase().startsWith('https://')) {
                    $scope.connection.params.switchToRegionFromBucket = false;
                }
                DataikuAPI.admin.connections.testEC2($scope.connection).success(function (data) {
                    $scope.testing = false;
                    $scope.testResult = data;
                }).error(function (a, b, c) {
                    $scope.testing = false;
                    setErrorInScope.bind($scope)(a, b, c)
                });
            }
        }

        if ($scope.creation) {
            $scope.connection.params["defaultManagedPath"] = "/dataiku";
            /* On this connection, null prefix defaults to ${projectKey}/, we don't set it explicitly if not in the naming rule settings */
            if ($rootScope.appConfig && $rootScope.appConfig.namingRulesSettings && $rootScope.appConfig.namingRulesSettings.defaultPrefixForPath) {
                $scope.connection.params.namingRule.pathDatasetNamePrefix = $rootScope.appConfig.namingRulesSettings.defaultPrefixForPath;
            }
        }
        if (!$scope.connection.params["hdfsInterface"]) {
            $scope.connection.params["hdfsInterface"] = "S3A";  // Default value
        }
        if (!$scope.connection.params.customAWSCredentialsProviderParams) {
            $scope.connection.params.customAWSCredentialsProviderParams = [];
        }
        if (!$scope.connection.customBasicConnectionCredentialProviderParams) {
            $scope.connection.customBasicConnectionCredentialProviderParams = [];
        }
        if (!$scope.connection.params.dkuProperties) {
            $scope.connection.params.dkuProperties = [];
        }
    });

    app.controller("DatabricksVolumeConnectionController", function ($scope, $controller, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.testConnection = function () {
            if (!$scope.connectionParamsForm || $scope.connectionParamsForm.$valid) {
                $scope.testing = true;
                DataikuAPI.admin.connections.testDatabricksVolume($scope.connection).success(function (data) {
                    $scope.testing = false;
                    $scope.testResult = data;
                }).error(function (a, b, c) {
                    $scope.testing = false;
                    setErrorInScope.bind($scope)(a, b, c)
                });
            }
        }

        if ($scope.creation) {
            $scope.connection.params["defaultManagedPath"] = "/dataiku";
        }
        if (!$scope.connection.params.dkuProperties) {
            $scope.connection.params.dkuProperties = [];
        }
    });

    app.controller("SageMakerConnectionController", function ($scope, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.testConnection = function () {
            $scope.testing = true;
            $scope.testResult = null;
            if ($scope.connection.params.regionOrEndpoint != null && $scope.connection.params.regionOrEndpoint.toLowerCase().startsWith('https://')) {
                $scope.connection.params.switchToRegionFromBucket = false;
            }
            DataikuAPI.admin.connections.testSageMaker($scope.connection).success(function (data) {
                $scope.testing = false;
                $scope.testResult = data;
            }).error(function (a, b, c) {
                $scope.testing = false;
                setErrorInScope.bind($scope)(a, b, c)
            });
        }

        if ($scope.creation) {
            $scope.connection.params.credentialsMode = "STS_ASSUME_ROLE";
        }

    });

    app.controller("GCSConnectionController", function ($scope, $controller, $rootScope, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.testConnection = function () {
            if (!$scope.connectionParamsForm || $scope.connectionParamsForm.$valid) {
                $scope.testing = true;
                DataikuAPI.admin.connections.testGCS($scope.connection).success(function (data) {
                    $scope.testing = false;
                    $scope.testResult = data;
                }).error(function (a, b, c) {
                    $scope.testing = false;
                    setErrorInScope.bind($scope)(a, b, c)
                });
            }
        }

        $scope.$watch("connection.params", function (nv, ov) {
            // GCS doesn't support global Oauth yet
            if (nv.authType=="OAUTH") {
               $scope.connection.credentialsMode = "PER_USER";
            } else {
               $scope.connection.credentialsMode = "GLOBAL";
            }
        }, true);

        if ($scope.creation) {
            $scope.connection.params["defaultManagedPath"] = "/dataiku";
            /* On this connection, null prefix defaults to ${projectKey}/, we don't set it explicitly if not in the naming rule settings */
            if ($rootScope.appConfig && $rootScope.appConfig.namingRulesSettings && $rootScope.appConfig.namingRulesSettings.defaultPrefixForPath) {
                $scope.connection.params.namingRule.pathDatasetNamePrefix = $rootScope.appConfig.namingRulesSettings.defaultPrefixForPath;
            }
        }

        if (!$scope.connection.params.dkuProperties) {
            $scope.connection.params.dkuProperties = [];
        }
    });

    app.controller("SharePointOnlineConnectionController", function ($scope, $controller, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testSharePointOnline($scope.connection).success(function (data) {
                $scope.testing = false;
                $scope.testResult = data;
            }).error(function (a, b, c) {
                $scope.testing = false;
                setErrorInScope.bind($scope)(a, b, c)
            });
        }

        if ($scope.creation) {
            $scope.connection.params = {};
            $scope.connection.params["defaultManagedPath"] = "/dataiku/";
            $scope.connection.params.authType = "OAUTH2_APP";
            $scope.connection.credentialsMode = "PER_USER";
            $scope.connection.params.credentialsMode = "PER_USER";
            $scope.connection.allowManagedDatasets = true;
            $scope.connection.allowManagedFolders = true;
            $scope.connection.allowWrite = true;
            $scope.connection.params.scopes = "User.Read Files.ReadWrite.All Sites.ReadWrite.All Sites.Manage.All offline_access";
            $scope.connection.params.authorizationEndpoint = "https://login.microsoftonline.com/common/oauth2/v2.0/authorize";
            $scope.connection.params.tokenEndpoint = "https://login.microsoftonline.com/common/oauth2/v2.0/token";
        }

        if (!$scope.connection.params.properties) {
            $scope.connection.params.properties = [];
        }

        if (!$scope.connection.params.dkuProperties) {
            $scope.connection.params.dkuProperties = [];
        }

        if (!$scope.connection.credentialsMode) {
            $scope.connection.credentialsMode = "PER_USER";
        }
        if ($scope.connection.authType=="PASSWORD") {
            $scope.connection.credentialsMode = "GLOBAL";
        }


        $scope.$watch("connection.credentialsMode", function (nv, ov) {
            if (!nv || nv === ov) return;

            if (nv === "GLOBAL") {
                $scope.connection.params.scopes = "https://graph.microsoft.com/.default";
            } else {
                $scope.connection.params.scopes = "User.Read Files.ReadWrite.All Sites.ReadWrite.All Sites.Manage.All offline_access";
            }
        }, true);
    });

    app.controller("VertexAIModelDeploymentConnectionController", function ($scope, $controller, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.testConnection = function () {
            $scope.testing = true;
            $scope.testResult = null;
            DataikuAPI.admin.connections.testVertexAIModelDeployment($scope.connection).success(function (data) {
                $scope.testing = false;
                $scope.testResult = data;
            }).error(function (a, b, c) {
                $scope.testing = false;
                setErrorInScope.bind($scope)(a, b, c)
            });
        }

        $scope.$watch("connection.params", function (nv, ov) {
            // GCS doesn't support global Oauth yet
            if (nv.authType=="OAUTH") {
               $scope.connection.credentialsMode = "PER_USER";
            } else {
               $scope.connection.credentialsMode = "GLOBAL";
            }
        }, true);

        if ($scope.creation) {
            $scope.connection.params.authType = "KEYPAIR";
            $scope.connection.params.dkuProperties = [];
        }
    });

    app.controller("DatabricksModelDeploymentConnectionController", function ($scope, $controller, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.testConnection = function () {
            $scope.testing = true;
            $scope.testResult = null;
            DataikuAPI.admin.connections.testDatabricksModelDeployment($scope.connection).success(function (data) {
                $scope.testing = false;
                $scope.testResult = data;
            }).error(function (a, b, c) {
                $scope.testing = false;
                setErrorInScope.bind($scope)(a, b, c)
            });
        }

        $scope.$watch("connection.params", function (nv) {
            if (nv.authType=="PERSONAL_ACCESS_TOKEN") {
               $scope.connection.credentialsMode = "GLOBAL";
            }
        }, true);

        if ($scope.creation) {
            $scope.connection.params.authType = "PERSONAL_ACCESS_TOKEN";
            $scope.connection.params.dkuProperties = [];
        }
    });

    app.controller("AzureConnectionController", function ($scope, $controller, $rootScope, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.testConnection = function () {
            if (!$scope.connectionParamsForm || $scope.connectionParamsForm.$valid) {
                $scope.testing = true;
                DataikuAPI.admin.connections.testAzure($scope.connection).success(function (data) {
                    $scope.testing = false;
                    $scope.testResult = data;
                }).error(function (a, b, c) {
                    $scope.testing = false;
                    setErrorInScope.bind($scope)(a, b, c)
                });
            }
        }

        if ($scope.creation) {
            $scope.connection.params["defaultManagedPath"] = "/dataiku";
            $scope.connection.params["defaultManagedContainer"] = "dataiku";
            $scope.connection.params["useSSL"] = true;
            $scope.connection.params["authType"] = "SHARED_KEY";

            /* On this connection, null prefix defaults to ${projectKey}/, we don't set it explicitly if not in the naming rule settings */
            if ($rootScope.appConfig && $rootScope.appConfig.namingRulesSettings && $rootScope.appConfig.namingRulesSettings.defaultPrefixForPath) {
                $scope.connection.params.namingRule.pathDatasetNamePrefix = $rootScope.appConfig.namingRulesSettings.defaultPrefixForPath;
            }
        }

        if (!$scope.connection.params.dkuProperties) {
            $scope.connection.params.dkuProperties = [];
        }
    });


    app.controller("AzureMLConnectionController", function ($scope, $controller, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.testConnection = function () {
            $scope.testing = true;
            $scope.testResult = null;
            DataikuAPI.admin.connections.testAzureML($scope.connection).success(function (data) {
                $scope.testing = false;
                $scope.testResult = data;
            }).error(function (a, b, c) {
                $scope.testing = false;
                setErrorInScope.bind($scope)(a, b, c)
            });
        }

        if ($scope.creation) {
            $scope.connection.params["authType"] = "OAUTH2_APP";
        }

        if (!$scope.connection.params.dkuProperties) {
            $scope.connection.params.dkuProperties = [];
        }
    });


    app.controller("ElasticSearchConnectionController", function ($scope, $controller, DataikuAPI, TopNav, $rootScope) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.availableAuthTypes = [
            ['NONE', 'None', 'No authentication needed', null],
            ['PASSWORD', 'Simple', 'Provide a user and a password', null],
            ['OAUTH2_APP', 'OAuth', 'Use an authentication service supporting OAuth v2.0', null],
            ['AWS_KEYPAIR', 'AWS keypair', 'AccessId + SecretId. Supports only OpenSearch.', 'KEYPAIR'],
            ['AWS_ENVIRONMENT', 'AWS Environment', 'Use credentials from environment variables, or ~/.aws/credentials file, or instance profile. Supports only OpenSearch.', 'ENVIRONMENT'],
            ['AWS_STS', 'AWS STS with AssumeRole', 'Assume a role, with master credentials coming from the environment. Supports only OpenSearch.', 'STS_ASSUME_ROLE'],
            ['AWS_CUSTOM', 'AWS custom provider', 'Use a third-party authentication provider. Supports only OpenSearch.','CUSTOM_PROVIDER']];
        $scope.availableAuthTypesDesc = $scope.availableAuthTypes.map((x) => x[2]);
        $scope.availableAWSAuthTypesDesc = $scope.availableAuthTypes.reduce((acc, x) => {acc[x[0]] = x[3]; return acc }, {});

        $scope.availableAWSServiceTypes = [
            ['OPENSEARCH_SERVERLESS', 'OpenSearch Serverless', 'You are connecting to a AWS serverless instance of OpenSearch.'],
            ['OPENSEARCH_HOSTED', 'Managed OpenSearch', 'You are connecting to a managed OpenSearch instance hosted on AWS.']];
        $scope.availableAWSServiceTypesDesc = $scope.availableAWSServiceTypes.map((x) => x[2]);

        $scope.connectionParamsForm = {};
        if ($scope.creation) {
            $scope.connection.params["host"] = "localhost";
            $scope.connection.params["port"] = 9200;
            $scope.connection.params["dialect"] = 'ES_7';
            $scope.connection.params["connectionLimit"] = 8;
            $scope.connection.params["oauth"] = {};
            $scope.connection.params["aws"] = {"service": "OPENSEARCH_SERVERLESS"};
            $scope.connection.params["authType"] = "NONE";

            const dp = $rootScope.appConfig && $rootScope.appConfig.namingRulesSettings && $rootScope.appConfig.namingRulesSettings.defaultPrefixForTables ?
                                $rootScope.appConfig.namingRulesSettings.defaultPrefixForTables : "${projectKey}_";

            $scope.connection.params.namingRule.indexNameDatasetNamePrefix = dp;
        }

        $scope.isConnectionParamsFormInvalid = function() {
            return $scope.connectionParamsForm.$invalid;
        }

        $scope.checkForHttpInHostUrl = (host) => host && (host.startsWith('http://') || host.startsWith('https://'));

        $scope.testConnection = function () {
            if (!$scope.connectionParamsForm || $scope.connectionParamsForm.$valid) {
                DataikuAPI.admin.connections.testElasticSearch($scope.connection, null).success(function (data) {
                    if (data.dialect && data.dialect !== $scope.connection.params.dialect) {
                        data.dialectMismatch = true;
                    }
                    $scope.testResult = data;
                }).error(setErrorInScope.bind($scope));
            }
        };

        $scope.warnings = {
            noVariableInIndex: false,
        };

        $scope.isIndexNameAllowed = function (v) {
            if (!v) {
                return true;
            }
            // See https://www.elastic.co/guide/en/elasticsearch/reference/current/regexp-syntax.html#regexp-reserved-characters for reserved characters
            return v.match(/^(?:[^.?+*|{}[\]()"\\#@&<>~ ]|\$\{[^}]*\})+$/) !== null;
        };

        $scope.getKBDisabledReason = function() {

            if($scope.connection.params.dialect !== "ES_7") {
                $scope.connection.allowKnowledgeBanks = false;
                return "Knowledge banks require at least ElasticSearch version 7 or OpenSearch.";
            }
            if($scope.connection.params.authType == "OAUTH2_APP") {
                $scope.connection.allowKnowledgeBanks = false;
                return "Knowledge banks are not supported with OAuth authentication.";
            }
            if (!$scope.connection.allowWrite){
                $scope.connection.allowKnowledgeBanks = false; // not necessary since it's already done in the allowWrite watcher but for consistency
                return "Write permissions on the connection are required to use knowledge banks.";
            }
            // No incompatibility found
            return null;
        }

        $scope.fixDialectMismatch = function() {
            if($scope.testResult && $scope.testResult.dialect != $scope.connection.params.dialect) {
                $scope.connection.params.dialect = $scope.testResult.dialect;
                $scope.testConnection();
            }
        };

        $scope.$watch("connection.params", function (nv, ov) {
            $scope.warnings.noVariableInIndex = false;
            $scope.connection.params.aws['credentialsMode'] = $scope.availableAWSAuthTypesDesc[$scope.connection.params.authType];
            if ((!nv.namingRule.indexNameDatasetNamePrefix || nv.namingRule.indexNameDatasetNamePrefix.match(/^.*\$\{[^}]*\}.*$/) === null) &&
                (!nv.namingRule.indexNameDatasetNameSuffix || nv.namingRule.indexNameDatasetNameSuffix.match(/^.*\$\{[^}]*\}.*$/) === null)) {
                $scope.warnings.noVariableInIndex = true;
            }
        }, true);
    });

    app.controller("TwitterConnectionController", function ($scope, $controller, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.connection.allowWrite = false;
        $scope.connection.allowManagedDatasets = false;
        $scope.connection.allowMirror = false;

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testTwitter($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        }

        $scope.clearForm = function () {
            $scope.connection.params.api_key = "";
            $scope.connection.params.api_secret = "";
            $scope.connection.params.token_key = "";
            $scope.connection.params.token_secret = "";
            $scope.verified = false;
        }

        DataikuAPI.connections.getTwitterConfig().success(function (data) {
            var activeConnection = data.connection;
            $scope.isActive = ($scope.connection.name === activeConnection);
            $scope.isRunning = (data.running.length > 0);
        }).error(function () {
            setErrorInScope.bind($scope);
            $scope.isActive = false;
        });

        $scope.setActiveConnection = function (name) {
            DataikuAPI.admin.connections.setActiveTwitterConnection(name).success(function () {
                $scope.isActive = true;
            }).error(setErrorInScope.bind($scope));
        }

        DataikuAPI.connections.getNames('Twitter').success(function (data) {
            $scope.displaySetActive = (data.length > 1);
        }).error(setErrorInScope.bind($scope));
    });

    app.controller("_LLMConnectionController", function($scope, DataikuAPI, $state) {
        $scope.openAiConnections = [];
        $scope.huggingFaceLocalConnections = [];
        if ($scope.connection.params.imageAuditManagedFolderRef) {
            const chunks = $scope.connection.params.imageAuditManagedFolderRef.split('.');
            $scope.imageStorage = {
                projectKey: chunks[0],
                managedFolderId: chunks[1],
            };
        } else {
            $scope.imageStorage = {
                projectKey: null,
                managedFolderId: null,
            };
        }

        $scope.showFineTuningSettings = function() {
            return $scope.appConfig.licensedFeatures.advancedLLMMeshAllowed;
        }

        DataikuAPI.pretrainedModels.listAvailableConnectionLLMs("GENERIC_COMPLETION").success(function(data){
            $scope.availableCompletionLLMs = data["identifiers"];
        }).error(setErrorInScope.bind($scope));


        $scope.showFineTuningSettingsIfConnectionAllowsIt = function(allowFinetuning) {
            return $scope.showFineTuningSettings() && allowFinetuning;
        }

        $scope.$watch("imageStorage.projectKey", function(nv, ov) {
            if (nv !== ov) {
                // 1 - set the folder id to null in case the new project got a managed folder with the same name,
                //     forcing the user to select a managed folder for the newly selected project
                // 2 - setting managedFolderId to an empty string '' will trigger the other watch below,
                //     hence invalidating the folder ref, forcing the user to select a new folder to get rid of the warning
                $scope.imageStorage.managedFolderId = '';
                $scope.connection.params.imageAuditManagedFolderRef = $scope.imageStorage.projectKey + ".";
             }
        });

        $scope.$watch("imageStorage.managedFolderId", function(nv, ov) {
            if (nv !== ov) {
                $scope.connection.params.imageAuditManagedFolderRef = $scope.imageStorage.projectKey + "." + $scope.imageStorage.managedFolderId;
            }
        });

        const deregister = $scope.$watch("connection.params", function(nv) {
            if (!nv) return;

            if ($scope.creation) {
                $scope.connection.params.auditingMode = "METADATA_ONLY";

                $scope.connection.params.guardrailsPipelineSettings = {"guardrails":[]};

                $scope.connection.params.cachingEnabled = false;
                $scope.connection.params.embeddingsCachingEnabled = true;

                $scope.connection.params.networkSettings = {
                    queryTimeoutMS: 60000,
                    maxRetries: 3,
                    initialRetryDelayMS: 3000,
                    retryDelayScalingFactor: 2.0,
                }
            }

            deregister();
        });
    });

    // Keep in sync with CustomOpenAIModelType (dip/connections/OpenAIConnection.java)
    app.constant("OpenAiModelTypes", [
        { rawValue: 'COMPLETION_CHAT', displayName: 'Chat completion' },
        { rawValue: 'COMPLETION_CHAT_MULTIMODAL', displayName: 'Chat completion (multimodal)' },
        { rawValue: 'COMPLETION_CHAT_NO_SYSTEM_PROMPT', displayName: 'Chat completion (no system message)' },
        { rawValue: 'COMPLETION_SIMPLE', displayName: 'Simple completion (legacy)' },
        { rawValue: 'TEXT_EMBEDDING_EXTRACTION', displayName: 'Embedding' },
        { rawValue: 'IMAGE_GENERATION', displayName: 'Image Generation' } // Can't really hide this under FF due to usage of app.constant :/
    ]);

    // Keep in sync with OpenAIAPI (dip/connections/OpenAIConnection.java)
    app.constant('OpenAIAPIModes', [
        {rawValue: 'CHAT_COMPLETIONS', displayName: 'Chat Completions API'},
        {rawValue: 'RESPONSES', displayName: 'Responses API'},
    ]);

    // Keep in sync with CustomOpenAIModelType (dip/connections/OpenAIConnection.java:OpenAIConnectionParams.OpenAIMaxTokensAPIMode)
    // See SC-217279
    app.constant('OpenAiMaxTokensAPIModes', [
        { rawValue: 'AUTO', displayName: 'Auto' },
        { rawValue: 'MODERN', displayName: 'Modern', description: 'Use max_completion_tokens parameter' },
        { rawValue: 'LEGACY', displayName: 'Legacy', description: 'Use max_tokens parameter (deprecated by OpenAI)' },
    ]);

    // Keep in sync with AzureOpenAIConnection (dip/connections/OpenAIConnection.java:AzureOpenAIConnection.AzureOpenAIMaxTokensAPIMode)
    app.constant('AzureOpenAiMaxTokensAPIModes', [
        { rawValue: 'MODERN', displayName: 'Modern (gpt5*, gpt4*, o*)', description: 'Use max_completion_tokens parameter' },
        { rawValue: 'LEGACY', displayName: 'Legacy (gpt3* or earlier)', description: 'Use max_tokens parameter' },
    ]);

    // Keep in sync with OpenAIImageHandling (dip/connections/OpenAIImageHandling.java)
    app.constant('OpenAIImageHandlingModes', [
        { rawValue: 'DALL_E_3', displayName: 'Dall-E 3' },
        { rawValue: 'GPT_IMAGE_1', displayName: 'GPT Image 1' },
    ]);

    app.controller("OpenAiConnectionController", function ($scope, $controller, TopNav, DataikuAPI, OpenAiModelTypes, OpenAiMaxTokensAPIModes, OpenAIImageHandlingModes, OpenAIAPIModes) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        if ($scope.creation) {
            $scope.connection.params.allowFinetuning = false;
            $scope.connection.params["allowGPT5"] = true;
            $scope.connection.params["allowGPT5Mini"] = true;
            $scope.connection.params["allowGPT5Nano"] = true;
            $scope.connection.params["allowGPT5Chat"] = false;
            $scope.connection.params["allowGPT41"] = false;
            $scope.connection.params["allowGPT41Mini"] = false;
            $scope.connection.params["allowGPT41Nano"] = false;
            $scope.connection.params["allowGPT4oMini"] = false;
            $scope.connection.params["allowGPT35Turbo"] = false;
            $scope.connection.params["allowO3"] = false;
            $scope.connection.params["allowO4Mini"] = false;
            $scope.connection.params["allowEmbedding3Small"] = true;
            $scope.connection.params.maxParallelism = 8;
            $scope.connection.params.maxTokensAPIMode = 'AUTO';
        }

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testOpenAi($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        };

        $scope.modelAPIModes = OpenAIAPIModes;
        $scope.modelTypes = OpenAiModelTypes;
        $scope.maxTokensAPIModes = OpenAiMaxTokensAPIModes;
        $scope.imageHandlingModes = OpenAIImageHandlingModes;
    });

    app.controller("AzureOpenAiConnectionController", function ($scope, $controller, TopNav, DataikuAPI, OpenAiModelTypes, AzureOpenAiMaxTokensAPIModes, OpenAIImageHandlingModes, OpenAIAPIModes) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        if ($scope.creation) {
            $scope.connection.params.allowFinetuning = false;
            $scope.connection.params["availableDeployments"] = [];
            $scope.connection.params.maxParallelism = 8;
        }

        $scope.azureResourceURLFormat = "https://RESOURCE_NAME.openai.azure.com/openai"

        DataikuAPI.admin.connections.list().success(function(data) {
            $scope.azureMLConnections = Object.values(data).filter(c => c.type === 'AzureML').map(c => c.name);
        });

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testAzureOpenAi($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        };

        $scope.deploymentTypes = OpenAiModelTypes;
        $scope.modelAPIModes = OpenAIAPIModes;
        $scope.imageHandlingModes = OpenAIImageHandlingModes;

        $scope.usePromptAndCompletionCosts = function(deployment) {
            return ['COMPLETION_CHAT', 'COMPLETION_CHAT_MULTIMODAL', 'COMPLETION_CHAT_NO_SYSTEM_PROMPT', 'COMPLETION_SIMPLE'].includes(deployment.deploymentType)
        };

        $scope.useEmbeddingCosts = function(deployment) {
            return 'TEXT_EMBEDDING_EXTRACTION' === deployment.deploymentType;
        };

        const chatModels = [
            { rawValue: '', displayName: 'Custom pricing' },
            { rawValue: 'gpt-5', displayName: 'GPT 5', promptCost: .00125, completionCost: .010 },
            { rawValue: 'gpt-5-mini', displayName: 'GPT 5 mini', promptCost: .00025, completionCost: .002 },
            { rawValue: 'gpt-5-nano', displayName: 'GPT 5 nano', promptCost: .00005, completionCost: .0004 },
            { rawValue: 'gpt-5-chat', displayName: 'GPT 5 Chat', promptCost: .00125, completionCost: .010 },
            { rawValue: 'o4-mini', displayName: 'o4-mini', promptCost: .0011, completionCost: 0.0044 },
            { rawValue: 'o3', displayName: 'o3', promptCost: .01, completionCost: 0.04 },
            { rawValue: 'o3-mini', displayName: 'o3-mini', promptCost: .0011, completionCost: .0044 },
            { rawValue: 'o1', displayName: 'o1', promptCost: .015, completionCost: .06 },
            { rawValue: 'gpt-4.1', displayName: 'GPT 4.1', promptCost: .002, completionCost: .008 },
            { rawValue: 'gpt-4.1-mini', displayName: 'GPT 4.1-mini', promptCost: .0004, completionCost: .0016 },
            { rawValue: 'gpt-4.1-nano', displayName: 'GPT 4.1-nano', promptCost: .0001, completionCost: .0004 },
            { rawValue: 'gpt-4o', displayName: 'GPT 4o', promptCost: .0025, completionCost: .01 },
            { rawValue: 'gpt-4o-mini', displayName: 'GPT 4o-mini', promptCost: .00015, completionCost: .0006 },
            { rawValue: 'gpt-4', displayName: 'GPT 4', promptCost: .03, completionCost: .06 },
            { rawValue: 'gpt-4-32k', displayName: 'GPT 4 (large context)', promptCost: .06, completionCost: .12 },
            { rawValue: 'gpt-3.5-turbo', displayName: 'GPT 3.5 Chat Turbo', promptCost: .0015, completionCost: .002 },
            { rawValue: 'gpt-3.5-turbo-16k', displayName: 'GPT 3.5 Turbo - large context' , promptCost: .003, completionCost: .004 }
        ];

        const simpleCompletionModels = [
            { rawValue: 'text-davinci-003', displayName: 'GPT 3 Text Davinci (text-davinci-003)', promptCost: .02, completionCost: .02 },
            { rawValue: 'babbage-002', displayName: 'GPT 3 Text Curie (text-curie-001)', promptCost: 0.002, completionCost: .002 },
            { rawValue: 'babbage-002', displayName: 'GPT 3 Text Babbage (text-babbage-001)', promptCost: 0.002, completionCost: .0005 },
            { rawValue: 'babbage-002', displayName: 'GPT 3 Text Ada (text-ada-001)', promptCost: 0.002, completionCost: .0004 },
            { rawValue: '', displayName: 'Custom pricing' },
        ];

        const embeddingModels = [
            { rawValue: 'text-embedding-ada-002', displayName: 'Embedding (Ada 002)', embeddingCost: 0.0001 },
            { rawValue: 'text-embedding-3-small', displayName: 'Embedding (v3 Small)', embeddingCost: 0.00002 },
            { rawValue: 'text-embedding-3-large', displayName: 'Embedding (v3 Large)', embeddingCost: 0.00013 },
            { rawValue: '', displayName: 'Custom pricing' },
        ];

        $scope.getOpenAIModels = function(deploymentType) {
            switch (deploymentType) {
            case 'COMPLETION_CHAT':
            case 'COMPLETION_CHAT_MULTIMODAL':
            case 'COMPLETION_CHAT_NO_SYSTEM_PROMPT':
                return chatModels;
            case 'COMPLETION_SIMPLE':
                return simpleCompletionModels;
            case 'TEXT_EMBEDDING_EXTRACTION':
                return embeddingModels;
            case 'IMAGE_GENERATION':
                return [];
            }
            // should not happen
            return [{ rawValue: '', displayName: 'Custom pricing' }];
        };

        $scope.onDeploymentTypeChange = function(deployment) {
            if (deployment.deploymentType !== 'IMAGE_GENERATION') {
                deployment.imageHandlingMode = null;
            } else if (!deployment.imageHandlingMode) {
                deployment.imageHandlingMode = 'DALL_E_3';
            }
            const availableModelsForPricing = $scope.getOpenAIModels(deployment.deploymentType);
            const openAIModel = availableModelsForPricing.find(model => model.rawValue === deployment.underlyingModelName);
            if (openAIModel) {
                deployment.underlyingModelName = openAIModel.rawValue;
            } else {
                deployment.underlyingModelName = '';
            }
        };

        $scope.onModelPricingChange = function(deployment) {
            const availableModelsForPricing = $scope.getOpenAIModels(deployment.deploymentType);
            const openAIModel = (availableModelsForPricing.find(model => model.rawValue === deployment.underlyingModelName) || {});
            deployment.promptCost = openAIModel.promptCost;
            deployment.completionCost = openAIModel.completionCost;
            deployment.embeddingCost = openAIModel.embeddingCost;
        };

        $scope.connection.params.availableDeployments.forEach($scope.onDeploymentTypeChange);
        $scope.maxTokensAPIModes = AzureOpenAiMaxTokensAPIModes;
    });

    app.controller("NvidiaNimConnectionController", function ($scope, $controller, TopNav, DataikuAPI) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        // An entry is called a deployment here because NIM models need to be deployed
        if ($scope.creation) {
            $scope.connection.params.allowFinetuning = false;
            $scope.connection.params["availableDeployments"] = [];
            $scope.connection.params.maxParallelism = 8;
        }

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testNvidiaNim($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        };

        // see src/main/java/com/dataiku/dip/connections/NvidiaNimConnection.java for the below mappings
        const openAiV1ChatCompletions = { rawValue: 'OPENAI_V1_CHAT_COMPLETIONS', displayName: 'OpenAI Chat completions', apiPath: 'v1/chat/completions' };
        const openAiV1Responses = { rawValue: 'OPENAI_V1_RESPONSES', displayName: 'OpenAI Responses', apiPath: 'v1/responses' };
        const openAiV1EmbeddingsApi = { rawValue: 'OPENAI_V1_EMBEDDINGS', displayName: 'OpenAI Embeddings', apiPath: 'v1/embeddings' };
        $scope.apiPathsById = Object.assign({}, ...[openAiV1ChatCompletions, openAiV1Responses, openAiV1EmbeddingsApi].map((type) => ({[type.rawValue]: type.apiPath})));

        $scope.modelTypes = [
            { rawValue: 'CHAT_COMPLETIONS', displayName: 'Chat completions', supportedApis: [ openAiV1ChatCompletions, openAiV1Responses ] },
            { rawValue: 'EMBEDDINGS', displayName: 'Embeddings', supportedApis: [ openAiV1EmbeddingsApi ] },
        ];
        $scope.apiByModelType = Object.assign({}, ...$scope.modelTypes.map((type) => ({[type.rawValue]: type.supportedApis})));

        $scope.onModelTypeChange = function(deployment) {
            // If the model type changed then we update the list of allowed API it can use
            // and if the newly selected model type does not support the previously selected API we reset it
            let allowedApis =  $scope.apiByModelType[deployment.modelType];
            if (allowedApis && !allowedApis.some(api => api.rawValue === deployment.api)) {
                deployment.api = allowedApis[0].rawValue;
            }
        };
        $scope.connection.params.availableDeployments.forEach($scope.onModelTypeChange);
    });

    app.controller("AzureLLMConnectionController", function ($scope, $controller, TopNav, DataikuAPI, OpenAiModelTypes) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        $scope.uiState = {
            azureMLDeplConnections: null,
        };

        if ($scope.creation) {
            $scope.connection.params = {
                defaultKey: null,
                maxParallelism: 8,
                customModels:  [{
                    id: null,
                    displayName: null,
                    modelType: "COMPLETION_CHAT",
                    targetURI: null,
                    key: null,
                    promptCost: null,
                    completionCost: null,
                    embeddingCost: null
                }]
            }
        }

        $scope.modelTypes = OpenAiModelTypes.filter(mt => ['COMPLETION_CHAT', 'COMPLETION_SIMPLE', 'COMPLETION_CHAT_MULTIMODAL', 'COMPLETION_CHAT_NO_SYSTEM_PROMPT', 'TEXT_EMBEDDING_EXTRACTION'].includes(mt.rawValue));

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testAzureMLGenericLLM($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        };

        $scope.usePromptAndCompletionCosts = function(model) {
            return ['COMPLETION_CHAT', 'COMPLETION_CHAT_MULTIMODAL','COMPLETION_SIMPLE'].includes(model.modelType)
        };

        $scope.useEmbeddingCosts = function(model) {
            return ['TEXT_EMBEDDING_EXTRACTION'].includes(model.modelType);
        };
    });

    app.controller("CohereConnectionController", function ($scope, $controller, TopNav, DataikuAPI) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        if ($scope.creation) {
            $scope.connection.params["allowCohereCommand"] = false;
            $scope.connection.params["allowCohereCommandLight"] = false;
            $scope.connection.params["allowCohereCommandR"] = true;
            $scope.connection.params["allowCohereCommandRPlus"] = true;
            $scope.connection.params.maxParallelism = 1;
        }

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testCohere($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        }
    });

    app.controller("StabilityAIConnectionController", function ($scope, $controller, TopNav, DataikuAPI) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        $scope.notTestable = true;

        if ($scope.creation) {
            $scope.connection.params.maxParallelism = 1;
        }

        $scope.testConnection = function () {
            // TODO @llm-img
        }
    });

    app.controller("MistralAiConnectionController", function ($scope, $controller, TopNav, DataikuAPI) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        if ($scope.creation) {
            $scope.connection.params["allowMistralSmall"] = true;
            $scope.connection.params["allowMistralMedium"] = true;
            $scope.connection.params["allowMistralLarge"] = true;

            $scope.connection.params["allowMistralEmbed"] = true;
            $scope.connection.params.maxParallelism = 4;
        }

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testMistralAi($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        };
    });

    app.controller("AnthropicConnectionController", function ($scope, $controller, TopNav, DataikuAPI) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        if ($scope.creation) {
            $scope.connection.params["allowClaudeV3Opus"] = false;
            $scope.connection.params["allowClaudeV4Opus"] = false;
            $scope.connection.params["allowClaudeV41Opus"] = true;
            $scope.connection.params["allowClaudeV35Sonnet"] = false;
            $scope.connection.params["allowClaudeV35SonnetV2"] = false;
            $scope.connection.params["allowClaudeV37Sonnet"] = false;
            $scope.connection.params["allowClaudeV4Sonnet"] = false;
            $scope.connection.params["allowClaudeV45Sonnet"] = true;
            $scope.connection.params["allowClaudeV3Haiku"] = false;
            $scope.connection.params["allowClaudeV35Haiku"] = false;
            $scope.connection.params["allowClaudeV45Haiku"] = true;
            $scope.connection.params.maxParallelism = 2;
        }

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testAnthropic($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        }
    });

    app.service('BedrockConnectionService', function() {
        // Keep in sync with the backend CustomBedrockModelType (dip/connections/BedrockConnection.java)
        this.BedrockModelTypes = [
            { rawValue: 'TEXT_MODEL', displayName: 'Chat completion' },
            { rawValue: 'MULTIMODAL_MODEL', displayName: 'Chat completion (multimodal)' },
            { rawValue: 'TEXT_EMBEDDING_EXTRACTION', displayName: 'Text embedding' },
            { rawValue: 'IMAGE_GENERATION', displayName: 'Image generation' },
            { rawValue: 'TEXT_IMAGE_EMBEDDING_EXTRACTION', displayName: 'Multimodal embedding' },
        ];

        // Keep in sync with the backend GenericLLMHandling (dip/llm/online/sagemakergeneric/GenericLLMHandling.java)
        // Commented out the modes that are not applicable for Bedrock
        this.BedrockHandlingModes = {
            TEXT_MODEL: [
                { rawValue: 'GENERIC_CONVERSE', displayName: 'Generic Converse API Model' },
                { rawValue: 'AMAZON_NOVA', displayName: 'Amazon Nova' },
                { rawValue: 'AMAZON_TITAN', displayName: 'Amazon Titan' },
                { rawValue: 'ANTHROPIC_CLAUDE_CHAT', displayName: 'Anthropic Claude Chat' },
                { rawValue: 'ANTHROPIC_CLAUDE', displayName: 'Anthropic Claude (legacy completion API)' },
                { rawValue: 'AI21_J2', displayName: 'AI21 Jurassic 2' },
                //{ rawValue: 'AI21_SUMMARIZE', displayName: 'AI21 Summarize'},
                { rawValue: 'COHERE_COMMAND_CHAT', displayName: 'Cohere Command Chat' },
                { rawValue: 'COHERE_COMMAND', displayName: 'Cohere Command (completion API)' },
                //{ rawValue: 'HUGGING_FACE', displayName: 'Hugging Face'},
                { rawValue: 'META_LLAMA_2_BEDROCK', displayName: 'Meta Llama 2' },
                { rawValue: 'META_LLAMA_3_BEDROCK', displayName: 'Meta Llama 3' },
                //{ rawValue: 'META_LLAMA_2_SAGEMAKER', displayName: 'Meta Llama 2'},
                { rawValue: 'MISTRAL_AI_CHAT', displayName: 'MistralAI Chat' },
                { rawValue: 'MISTRAL_AI', displayName: 'MistralAI (text completion API)' },
                //{ rawValue: 'FULLY_CUSTOM', displayName: 'Fully Custom Handling'},
            ],
            MULTIMODAL_MODEL: [
                { rawValue: 'GENERIC_CONVERSE', displayName: 'Generic Converse API Model' },
                { rawValue: 'AMAZON_NOVA', displayName: 'Amazon Nova' },
                { rawValue: 'ANTHROPIC_CLAUDE_CHAT', displayName: 'Anthropic Claude Chat' },
            ],
            TEXT_EMBEDDING_EXTRACTION: [
                { rawValue: 'AMAZON_TITAN_TEXT_EMBEDDING', displayName: 'Text embedding Amazon Titan' },
                { rawValue: 'COHERE_EMBED', displayName: 'Text embedding Cohere' },
            ],
            TEXT_IMAGE_EMBEDDING_EXTRACTION: [
               { rawValue: "AMAZON_TITAN_TEXT_IMAGE_EMBEDDING", displayName: "Multimodal embedding Amazon Titan" },
            ],
            IMAGE_GENERATION: [
                { rawValue: "AMAZON_TITAN", displayName: "Amazon Titan Image Generator G1" },
                { rawValue: "STABILITYAI_STABLE_DIFFUSION_10", displayName: "Stability AI SDXL 1.0" },
                { rawValue: "STABILITYAI_STABLE_IMAGE_CORE", displayName: "Stability AI Stable Image Core" },
                { rawValue: "STABILITYAI_STABLE_DIFFUSION_3", displayName: "Stability AI SD3" },
                { rawValue: "STABILITYAI_STABLE_IMAGE_ULTRA", displayName: "Stability AI Stable Image Ultra" },
            ]
        }

    })

    app.controller("BedrockConnectionController", function($scope, $controller, TopNav, DataikuAPI, BedrockConnectionService) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        if ($scope.creation) {
            $scope.connection.params.allowFinetuning = false;

            /* AWS - Completion */
            $scope.connection.params.allowAWSNovaPro = true;
            $scope.connection.params.allowAWSNovaLite = false;
            $scope.connection.params.allowAWSNovaMicro = false;
            $scope.connection.params.allowAWSTitanTextPremierV1 = false;
            $scope.connection.params.allowAWSTitanTextLiteV1 = false;
            $scope.connection.params.allowAWSTitanTextExpressV1 = false;
            $scope.connection.params.allowAWSTitanLarge = false;

            /* Anthropic */
            $scope.connection.params.allowAnthropicClaude45Haiku = true;
            $scope.connection.params.allowAnthropicClaude45Sonnet = true;
            $scope.connection.params.allowAnthropicClaude41Opus = true;
            $scope.connection.params.allowAnthropicClaude4Sonnet = false;
            $scope.connection.params.allowAnthropicClaude4Opus = false;
            $scope.connection.params.allowAnthropicClaude37Sonnet = false;
            $scope.connection.params.allowAnthropicClaude35SonnetV2 = false;
            $scope.connection.params.allowAnthropicClaude35Sonnet = false;
            $scope.connection.params.allowAnthropicClaude3Sonnet = false;
            $scope.connection.params.allowAnthropicClaude35Haiku = false;
            $scope.connection.params.allowAnthropicClaude3Haiku = false;
            $scope.connection.params.allowAnthropicClaude3Opus = false;

            /* AI21 */
            $scope.connection.params.allowAI21Jurassic2Ultra = false;
            $scope.connection.params.allowAI21Jurassic2Mid = false;

            /* Cohere - Completion */
            $scope.connection.params.allowCohereCommandRPlus = true;
            $scope.connection.params.allowCohereCommandR = false;

            /* Meta */
            $scope.connection.params.allowMetaLlama33_70BInstruct = true;
            $scope.connection.params.allowMetaLlama31_8BInstruct = false;
            $scope.connection.params.allowMetaLlama31_70BInstruct = true;
            $scope.connection.params.allowMetaLlama31_405BInstruct = false;
            $scope.connection.params.allowMetaLlama38BInstruct = false;
            $scope.connection.params.allowMetaLlama370BInstruct = false;

            /* Mistral */
            $scope.connection.params.allowMistral7BInstruct = false;
            $scope.connection.params.allowMixtral8X7BInstruct = false;
            $scope.connection.params.allowMistralSmall = false;
            $scope.connection.params.allowMistralLarge = false;
            $scope.connection.params.allowMistralLarge2 = true;

            /* AWS - Embedding */
            $scope.connection.params.allowAWSTitanEmbedTextV2 = true;
            $scope.connection.params.allowAWSTitanEmbedText = false;
            $scope.connection.params.allowAWSTitanMultimodalEmbedV1 = false;

            /* Cohere - Embedding */
            $scope.connection.params.allowCohereEmbedEnglish = false;
            $scope.connection.params.allowCohereEmbedMultilingual = false;

            /* AWS - Image generation */
            $scope.connection.params.allowAWSTitanImageGeneratorV1 = false;

            /* DeepSeek */
            $scope.connection.params.allowDeepSeekR1 = false;

            $scope.connection.params.inferenceProfile = null;
            $scope.connection.params.maxParallelism = 8;
            $scope.connection.params.useBedrockGuardrail = false;
            $scope.connection.params.guardrailIdentifier = null;
            $scope.connection.params.guardrailVersion = null;
        }

        $scope.s3Connections = [];
        DataikuAPI.admin.connections.list().success(function(data) {
            $scope.s3Connections = Object.values(data).filter(c => c.type === 'EC2').map(c => c.name);
        });

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testBedrock($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        };

        $scope.modelTypes = BedrockConnectionService.BedrockModelTypes;
        $scope.handlingModes = BedrockConnectionService.BedrockHandlingModes;
        $scope.TEXT_MODEL = 'TEXT_MODEL';
        $scope.MULTIMODAL_MODEL = 'MULTIMODAL_MODEL';

        $scope.customModelTypeChanged = function(customModel) {
            if (!customModel.modelType) return;

            customModel.handlingMode = undefined;
        };

        $scope.customModelHandlingChanged = function(customModel) {
            if (!customModel.handlingMode) return;

            if (['GENERIC_CONVERSE', 'AMAZON_NOVA'].includes(customModel.handlingMode)) {
                customModel.useConverseAPI = true;
            }
        };
    });

    app.controller("MosaicMLConnectionController", function ($scope, $controller, TopNav, DataikuAPI) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        if ($scope.creation) {
            $scope.connection.params.allowMPT7BInstruct = true;
            $scope.connection.params.allowMPT30BInstruct = true;
            $scope.connection.params.allowLLAMA270BChat = true;
            $scope.connection.params.maxParallelism = 1;
        }

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testMosaicML($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        }
    });

    app.controller("SageMakerGenericLLMConnectionController", function ($scope, $controller, TopNav, DataikuAPI) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        // Keep in sync with CustomSageMakerModelType (dip/connections/SageMakerGenericLLMConnection/java)
        $scope.modelTypes =  [
            { rawValue: 'TEXT_COMPLETION', displayName: 'Text completion' },
            { rawValue: 'SUMMARIZATION', displayName: 'Summarization' },
            { rawValue: "TEXT_EMBEDDING", displayName: "Text embedding" }
    ];

        const modelHandlingModesCompletionSummarization = [
            ['AI21_J2', 'AI21 Jurassic 2'],
            ['AI21_SUMMARIZE', 'AI21 Summarize'],
            ['COHERE_COMMAND', 'Cohere Command'],
            ['HUGGING_FACE', 'Hugging Face'],
            ['META_LLAMA_2_SAGEMAKER', 'Meta Llama 2'],
            ['FULLY_CUSTOM', 'Fully Custom Handling']
        ];

        const modelHandlingModesTextEmbeddings = [
            ['COHERE_EMBED', 'Cohere Embed']
        ]

        $scope.modelHandlingModes = {
            'TEXT_COMPLETION': modelHandlingModesCompletionSummarization,
            'SUMMARIZATION':  modelHandlingModesCompletionSummarization,
            'TEXT_EMBEDDING': modelHandlingModesTextEmbeddings
        };

        $scope.useFullyCustomHandling = function() {
            if (!$scope.connection || !$scope.connection.params || !$scope.connection.params.sageMakerModel) return false;
            return $scope.connection.params.sageMakerModel.handling === 'FULLY_CUSTOM';
        }

        if ($scope.creation) {
            $scope.connection.params.maxParallelism = 8;
            $scope.connection.params.sageMakerModel= {
                friendlyNameShort: "Custom SageMaker LLM Endpoint",
                modelType: 'TEXT_COMPLETION',
                customHeaders: [],
                handling: null
            }
            $scope.connection.params.customQuery =
`{
    "prompt": __PROMPT__,
    "parameters": {
        "top_k": __TOPK__,
        "top_p": __TOPP__,
        "temperature": __TEMPERATURE__,
        "max_tokens": __MAX_TOKENS__,
        "frequency_penalty": __FREQUENCY_PENALTY__,
        "presence_penalty": __PRESENCE_PENALTY__,
        "logit_bias": __LOGIT_BIAS__
    }
}`;
            $scope.connection.params.responseJsonPath = "$.response";
        }

        $scope.$watch("connection.params.sageMakerModel.modelType", function(nv, ov) {
            if (ov && (nv !== ov)) {
                $scope.connection.params.sageMakerModel.handling = null;
            }
        });

        $scope.sageMakerConnections = [];
        DataikuAPI.admin.connections.list().success(function(data) {
            $scope.sageMakerConnections = Object.values(data).filter(c => c.type === 'SageMaker').map(c => c.name);
            if ($scope.sageMakerConnections.length == 1) {
                $scope.connection.params.sageMakerConnection = $scope.sageMakerConnections[0];
            }
        });

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testSageMakerGenericLLM($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        }
    });

    // Keep in sync with VertexModelType (dip/connections/VertexAILLMConnection.java)
     app.constant("VertexModelTypes", [
        { rawValue: 'GEMINI_CHAT', displayName: 'Chat Completion' },
        { rawValue: 'TEXT_EMBEDDING_EXTRACTION', displayName: 'Embedding' },
        { rawValue: 'TEXT_IMAGE_EMBEDDING_EXTRACTION', displayName: 'Embedding multimodal' },
        { rawValue: 'IMAGE_GENERATION', displayName: 'Image Generation' }  // Can't really hide this under FF due to usage of app.constant :/
    ]);
    app.controller("VertexAILLMConnectionController", function ($scope, $controller, TopNav, DataikuAPI, VertexModelTypes) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        if ($scope.creation) {
            $scope.connection.params.allowGeminiFlash25 = true;
            $scope.connection.params.allowGeminiPro25 = true;
            $scope.connection.params.allowGeminiFlashLite25 = true;
            $scope.connection.params.allowGeminiFlash20 = false;
            $scope.connection.params.allowGeminiFlashLite20 = false;
            $scope.connection.params.allowGeminiFlash20Exp = false;


            $scope.connection.params.allowGeminiTextEmb = true;
            $scope.connection.params.allowTextEmb005 = true;
            $scope.connection.params.allowTextEmb = false;
            $scope.connection.params.allowTextMultilangEmb = true;
            $scope.connection.params.allowMultimodalEmb = true;

            $scope.connection.params.allowImagen3 = false;
            $scope.connection.params.allowImagen3Fast = false;

            $scope.connection.params.authType = "KEYPAIR";
            $scope.connection.params.dkuProperties = [];
            $scope.connection.params.maxParallelism = 8;
        }

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testVertexAILLM($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        }

        $scope.$watch("connection.params", function (nv, ov) {
            // GCS doesn't support global Oauth yet
            if (nv.authType=="OAUTH") {
               $scope.connection.credentialsMode = "PER_USER";
            } else {
               $scope.connection.credentialsMode = "GLOBAL";
            }
        }, true);

        $scope.modelTypes = VertexModelTypes;

    });


 // Keep in sync with DatabricksLLMModelType (dip/connections/DatabricksLLMConnection.java)
    app.constant("DatabricksLLMModelTypes", [
        { rawValue: 'CHAT', displayName: 'Chat Completion' },
        { rawValue: 'TEXT_EMBEDDING', displayName: 'Embedding' }
    ]);
    app.controller("DatabricksLlmConnectionController", function ($scope, $controller, TopNav, DataikuAPI, DatabricksLLMModelTypes) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        if ($scope.creation) {
            $scope.connection.params.allowClaude3_7_Sonnet= true;
            $scope.connection.params.allowLlama4_Maverick = false;
            $scope.connection.params.allowLlama3_3_70BChat = true;
            $scope.connection.params.allowLlama3_1_405BChat = false;
            $scope.connection.params.allowBGELargeEn = true;

            $scope.connection.params.maxParallelism = 1;
        }

        $scope.databricksModelDeplConnections = [];
        DataikuAPI.admin.connections.list().success(function(data) {
            $scope.databricksModelDeplConnections = Object.values(data).filter(c => c.type === 'DatabricksModelDeployment').map(c => c.name);
        }).error(setErrorInScope.bind($scope));

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testDatabricksLLM($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        }

        $scope.modelTypes = DatabricksLLMModelTypes;
    });

// Keep in sync with VertexModelType (dip/connections/SnowflakeCortexLLMConnection.java)
    app.constant("SnowflakeCortexLLMModelTypes", [
        { rawValue: 'CHAT_COMPLETION', displayName: 'Chat Completion' },
        { rawValue: 'TEXT_EMBEDDING', displayName: 'Embedding' }
    ]);
    app.controller("SnowflakeCortexConnectionController", function ($scope, $controller, TopNav, DataikuAPI, SnowflakeCortexLLMModelTypes) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        $scope.uiState = $scope.uiState || {};

        if ($scope.creation) {
            $scope.connection.params.allowLlama33_70BChat = true;
            $scope.connection.params.allowLlama31_70BChat = false;
            $scope.connection.params.allowLlama32_3BChat = true;
            $scope.connection.params.allowLlama2_70BChat = false;
            $scope.connection.params.allowMixtral8x7B = false;
            $scope.connection.params.allowMistral_7B = false;
            $scope.connection.params.allowMistral_Large2 = true;
            $scope.connection.params.allowMistral_Large = false;
            $scope.connection.params.allowGemma_7B = true;
            $scope.connection.params.allowSnowflakeArctic = true;
            $scope.connection.params.allowDeepSeekR1 = true;
            $scope.connection.params.allowClaude35Sonnet = true;
            $scope.connection.params.allowLlama4_Maverick = true;

            $scope.connection.params.allowSnowflakeArcticEmbedM = true;
            $scope.connection.params.allowE5BaseV2 = true;
            $scope.connection.params.allowNVEmbedQA4 = true;
            $scope.connection.params.maxParallelism = 8;
        }

        function computeSelectedConnection() {
            if ($scope.connection.params.snowflakeConnection) {
                $scope.uiState.selectedSnowflakeConnection = $scope.snowflakeConnections.find(sc => sc.name == $scope.connection.params.snowflakeConnection);
            } else {
                $scope.uiState.selectedSnowflakeConnection = null;
            }
        }

        $scope.snowflakeConnections = [];
        DataikuAPI.admin.connections.list().success(function(data) {
            $scope.snowflakeConnections = Object.values(data).filter(c => c.type === 'Snowflake');
            $scope.snowflakeConnectionNames = $scope.snowflakeConnections.map(c => c.name);
            computeSelectedConnection();
        }).error(setErrorInScope.bind($scope));

        $scope.$watch("connection.params.snowflakeConnection", () => {
            computeSelectedConnection();
        })

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testSnowflakeCortex($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        }

        $scope.modelTypes = SnowflakeCortexLLMModelTypes;
    });

    app.service('HuggingFaceLocalConnectionService', function() {
        const GUARDED_MODEL_PRESETS = [
            "LLAMA_2_7B_CHAT",
            "LLAMA_2_13B_CHAT",
            "LLAMA_3_8B_INSTRUCT",
            "LLAMA_GUARD2",
            "LLAMA_GUARD3_1B",
            "LLAMA_GUARD3_8B",
            "PROMPT_GUARD",
            "MISTRAL_7B_INSTRUCT",
            "MISTRAL_7B_INSTRUCT_V2",
            "MISTRAL_7B_INSTRUCT_V3",
            "MISTRAL_NEMO_12B_INSTRUCT",
            "MIXTRAL_8X7B_INSTRUCT",
            "LLAMA_3_1_8B_INSTRUCT",
            "LLAMA_3_1_70B_INSTRUCT",
            "GEMMA_2B_INSTRUCT",
            "GEMMA_7B_INSTRUCT",
            "GEMMA_2_2B_INSTRUCT",
            "GEMMA_2_9B_INSTRUCT",
            "LLAMA_3_2_3B_INSTRUCT",
            "LLAMA_3_2_11B_VISION_INSTRUCT",
            "LLAMA_3_3_70B_INSTRUCT",
        ];

        this.isGuardedModel = (presetId) => {
            return GUARDED_MODEL_PRESETS.includes(presetId);
        }

        const HF_PURPOSES = {
            GENERIC_COMPLETION: {
                categoryName: "Text generation models",
                handlingModes: [
                    { rawValue: "TEXT_GENERATION_LLAMA_2", displayName: "Llama 2/3 model" },
                    { rawValue: "TEXT_GENERATION_DEEPSEEK", display: "Deepseek model" },
                    { rawValue: "TEXT_GENERATION_DOLLY", displayName: "Dolly model" },
                    { rawValue: "TEXT_GENERATION_GPT", displayName: "GPT model" },
                    { rawValue: "TEXT_GENERATION_MISTRAL", displayName: "Mistral model" },
                    { rawValue: "TEXT_GENERATION_QWEN", display: "Qwen model" },
                    { rawValue: "TEXT_GENERATION_ZEPHYR", displayName: "Zephyr model" },
                    { rawValue: "TEXT_GENERATION_FALCON", displayName: "Falcon model" },
                    { rawValue: "TEXT_GENERATION_MPT", displayName: "MPT model" },
                    { rawValue: "TEXT_GENERATION_GEMMA", displayName: "Gemma model" },
                    { rawValue: "TEXT_GENERATION_PHI_3", displayName: "Phi-3 model" },
                ],
            },
            TEXT_EMBEDDING_EXTRACTION: {
                categoryName: "Text embedding models",
                handlingModes: [
                    { rawValue: "TEXT_EMBEDDING", displayName: "Text embedding" },
                ],
            },
            IMAGE_GENERATION: {
                categoryName: "Image generation models",
                handlingModes: [
                    { rawValue: "IMAGE_GENERATION_DIFFUSION", displayName: "Image generation model" },
                ],
            },

            IMAGE_EMBEDDING_EXTRACTION: {
                categoryName: "Image embedding models",
                handlingModes: [
                    { rawValue: "IMAGE_EMBEDDING", displayName: "Image embedding" },
                ],
            },
            TOXICITY_DETECTION: {
                categoryName: "Toxicity detection models",
                handlingModes: [
                    { rawValue: "TEXT_CLASSIFICATION_TOXICITY", displayName: "Toxicity detection (Bert models)" },
                    { rawValue: "TEXT_GENERATION_LLAMA_GUARD", displayName: "Toxicity detection (Llama Guard models)" },
                ],
            },
            PROMPT_INJECTION_DETECTION: {
                categoryName: "Prompt injection detection models",
                handlingModes: [
                    { rawValue: "TEXT_CLASSIFICATION_PROMPT_INJECTION", displayName: "Prompt injection" },
                ],
            },
            SENTIMENT_ANALYSIS: {
                categoryName: "Text classification models: Sentiment analysis",
                handlingModes: [
                    { rawValue: "TEXT_CLASSIFICATION_SENTIMENT", displayName: "Sentiment analysis" },
                ],
            },
            EMOTION_ANALYSIS: {
                categoryName: "Text classification models: Emotion analysis",
                handlingModes: [
                    { rawValue: "TEXT_CLASSIFICATION_EMOTIONS", displayName: "Emotion analysis" },
                ],
            },
            CLASSIFICATION_WITH_OTHER_MODEL_PROVIDED_CLASSES: {
                categoryName: "Text classification models: Other use cases",
                handlingModes: [
                    { rawValue: "TEXT_CLASSIFICATION_OTHER", displayName: "Text classification (other use cases)" },
                ],
            },
            CLASSIFICATION_WITH_USER_PROVIDED_CLASSES: {
                categoryName: "Text classification models: Custom classes (aka zero-shot)",
                handlingModes: [
                    { rawValue: "ZSC_GENERIC", displayName: "Zero-shot classification" },
                ],
            },
            SUMMARIZATION: {
                categoryName: "Text summarization models",
                handlingModes: [
                    { rawValue: "SUMMARIZATION_ROBERTA", displayName: "Summarization (Roberta models)" },
                    { rawValue: "SUMMARIZATION_GENERIC", displayName: "Summarization (generic)" },
                ],
            },
        };

        this.getPossibleHandlingModes = (purpose) => {
            return HF_PURPOSES[purpose]['handlingModes'];
        }

        this.getPurposes = () => {
            return HF_PURPOSES;
        }

        this.getNoPresetCustomModel = function(purpose) {
            const handlingMode = this.getPossibleHandlingModes(purpose)[0].rawValue;
            return { displayName: '', huggingFaceId: '', id: '', handlingMode: handlingMode, quantizationMode: 'NONE', enabled: true, containerSelection: {containerMode: 'INHERIT'} };
        }
    });

    app.controller("HuggingFaceLocalConnectionController", function ($scope, $state, $controller, $interval, TopNav, DataikuAPI, HuggingFaceLocalConnectionService) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        $scope.notTestable = true;

        if ($scope.creation) {
            $scope.connection.params.useDSSModelCache = true;
            $scope.connection.params.enableReserveCapacity = true;
            $scope.connection.params.allowFinetuning = false;
            $scope.connection.params.models = [];
            $scope.connection.params.containerSelection = {containerMode: "NONE"};
        }

        DataikuAPI.admin.getGeneralSettings().then(function({data}) {
            $scope.globalMaxKernels = data.generativeAISettings.huggingFaceLocalSettings.maxConcurrentKernels
        }).catch(setErrorInScope.bind($scope));

        DataikuAPI.admin.clusters.listAccessible('KUBERNETES').success(function(data){
            $scope.k8sClusters = [{id:undefined, name:'Inherit instance default'}].concat(data);
        }).error(setErrorInScope.bind($scope));

        $scope.useInternalCodeEnvLabel = "Use internal code env";

        DataikuAPI.codeenvs.listNames('PYTHON').then(function({data}) {
            $scope.codeEnvItemsListWithDefault = [{"label": $scope.useInternalCodeEnvLabel, "value": undefined}].concat(data.map(codeEnv => ({"label": codeEnv, "value": codeEnv})));
        }).catch(setErrorInScope.bind($scope));

        let hfLocalInternalCodeEnvChecked = false;
        DataikuAPI.codeenvs.checkDSSInternalCodeEnv("HUGGINGFACE_LOCAL_CODE_ENV").then(function({data}) {
            if (Object.keys(data).length > 0) {
                $scope.hfLocalInternalCodeEnv = data.value;
                if ($scope.creation && $scope.hfLocalCodeEnv != null) {
                    $scope.connection.params.codeEnvName = $scope.hfLocalInternalCodeEnv.envName;
                }
            }
            hfLocalInternalCodeEnvChecked = true;
        }).catch(setErrorInScope.bind($scope));

        $scope.hfLocalInternalCodeEnvExists = function() {
            return hfLocalInternalCodeEnvChecked && $scope.hfLocalInternalCodeEnv != null;
        }

        $scope.hfLocalCodeEnvIsInternal = function () {
            return $scope.connection.params.codeEnvName == null || (
                $scope.hfLocalInternalCodeEnvExists() && $scope.connection.params.codeEnvName == $scope.hfLocalInternalCodeEnv.envName
            );
        }

        $scope.showHFLocalCodeEnvWarning = function () {
            return hfLocalInternalCodeEnvChecked && (
                (!$scope.hfLocalInternalCodeEnvExists()) || (!$scope.hfLocalCodeEnvIsInternal())
            );
        }

        $scope.internalCodeEnvsHRef = function() {
            if ($scope.appConfig.isAutomation) {
                return $state.href("admin.codeenvs-automation.internal");
            } else {
                return $state.href("admin.codeenvs-design.internal");
            }
        }

        $scope.fixupNullishForDirtynessCheck = function() {
            // if the connection.params.codeEnvName is unset in the backend, it will be initialized as non-set (~undefined) in the frontend
            // when selecting "Use internal code env", the code env selector forces this value to null
            // we need to fix this to avoid the dirtyness check failing due to null !== undefined
            if ($scope.connection.params.codeEnvName == null) {
                $scope.connection.params.codeEnvName = undefined;
            }
        }

        $scope.presetModels = null;
        $scope.facets = null;
        $scope.presets = null;
        $scope.facetFamilyDescriptions = [];
        DataikuAPI.admin.connections.listHuggingFacePresets().success(function(config) {
            $scope.presets = config.presets;
            $scope.presetModels = config.presets.map(p => p.model);
            // add default preset models for new connections
            if ($scope.creation) {
                for (const preset of config.presets) {
                    if (preset.includeInNewConnections) {
                        const newModel = _.cloneDeep(preset.model)
                        if (HuggingFaceLocalConnectionService.isGuardedModel(preset.id)) {
                            // apiKey is always blank initially on new connection creation
                            newModel.enabled = false;
                        }
                        $scope.connection.params.models.push(newModel);
                    }
                }
            }

            $scope.facets = config.facets;
        }).catch(setErrorInScope.bind($scope));

        // Used in "connection-name-test-sav.html"
        $scope.shouldShowExplicitUnsavedWarning = true;

        $scope.hasAtLeastOneReservedCapacity = function() {
            return $scope.connection.params.models.some(model => $scope.isValidHFModel(model) && model.enabled && model.minKernelCount > 0)
        };

        $scope.isValidHFModel = function(model) {
            return model.id && model.huggingFaceId && model.handlingMode
        };

        $scope.updateStatus = function() {
            if (!$scope.creation && $scope.connection.name) {
                DataikuAPI.admin.connections.getHfKernelStatus($scope.connection.name).success(function(status) {
                    $scope.kernelsStatus = status;
                }).catch(setErrorInScope.bind($scope));
            }
        }

        $scope.kernelsStatus = null;
        $scope.updateStatus(); // First call
        const cancelStatusUpdate = $interval(() => {
            $scope.updateStatus();
        }, 5000);

        $scope.$on('$destroy', () => {
            if (cancelStatusUpdate) {
                $interval.cancel(cancelStatusUpdate);
            }
        });

        checkChangesBeforeLeaving($scope, $scope.connectionDirty);
    });

    app.controller("HuggingFaceAddModelFromPresetController", function ($scope, Debounce, TaggingService) {
        $scope.filteredPresets = $scope.presets;
        $scope.selectedPreset = null;
        $scope.filterFacets = angular.copy($scope.facets);
        $scope.uiState = $scope.uiState || {};
        $scope.uiState.facetFilter = '';

        const facetFilterNames = {
            family: 'All families',
            mainUsagePurpose: 'All model types'
        };
        for (const [facetKey, facet] of Object.entries($scope.filterFacets)) {
            facet.values.unshift({name: facetFilterNames[facetKey], id: ''});
        }
        if ($scope.filterFacets || $scope.filterFacets['family']) {
            $scope.facetFamilyDescriptions = ($scope.filterFacets['family'].values || []).map(facet => facet.description);
        }

        $scope.filterPresets = () => {
            $scope.filteredPresets = $scope.presets.filter(preset => {
                for (const [facetKey, facetValue] of Object.entries($scope.selectedFacets)) {
                    if (typeof facetValue === "string" && facetValue !== '' && 
                        (preset.facets[facetKey] === undefined || !preset.facets[facetKey].includes(facetValue))) {
                        // corresponds to model types and model families
                        return false; // facet mismatch, filtering out
                    }
                    if (Array.isArray(facetValue) && facetValue.length > 0) {
                        // corresponds to model tags
                        if (preset.facets[facetKey] === undefined) {
                            return false; // preset has no corresponding facet, filtering out
                        }
                        for (const facetValueItem of facetValue) {
                            if (!preset.facets[facetKey].includes(facetValueItem)) {
                                return false; // facet mismatch, filtering out
                            }
                        }
                    }

                }
                // $scope.uiState.facetFilter is the text search in the top right corner of the modal
                return preset.model && (preset.model.displayName || '').toLowerCase().includes($scope.uiState.facetFilter.toLowerCase());
            });
        }
        $scope.filterPresets();

        $scope.mainPresetModelPurpose = (preset) => {
            return $scope.filterFacets["mainUsagePurpose"].values.filter(v => preset.facets.mainUsagePurpose && v.id === preset.facets.mainUsagePurpose[0])[0];
        };

        $scope.selectPreset = (preset) => {
            $scope.selectedPreset = preset;
        };

        $scope.addModel = (preset) => {
            $scope.appendModel(preset.model, preset.facets['mainUsagePurpose'][0]);
            $scope.resolveModal();
        };

        $scope.resetFilters = () => {
            $scope.filteredPresets = $scope.presets;
            $scope.uiState.facetFilter = '';
            $scope.selectedFacets = {
                family: '',
                tags: [],
                mainUsagePurpose: ''
            };
        };

        $scope.isFiltering = () => {
            return $scope.filteredPresets.length !== $scope.presets.length;
        }

        $scope.selectTag = function(tagId) {
            if ($scope.selectedFacets['tags'].includes(tagId)) { return; }
            $scope.selectedFacets['tags'].push(tagId);
            $scope.filterPresets();
        };
        $scope.unSelectTag = function(tagId) {
            if (!$scope.selectedFacets['tags'].includes(tagId)) { return; }
            $scope.selectedFacets['tags'] = $scope.selectedFacets['tags'].filter(element => element !== tagId);
            $scope.filterPresets();
        };
        $scope.tagColor = TaggingService.getTagColor;

        $scope.$watch('uiState.facetFilter', Debounce().withDelay(100, 200).wrap($scope.filterPresets));
    });

    app.component("hfModelsTable", {
        templateUrl: "/templates/admin/connection-huggingface-local-models-table.html",
        bindings: {
            models: '<',
            kernelsStatus: '<',
            presetModels: '<',
            apiKey: '<',
            supportsLlmFineTuning: '<',
            connectionName: '<',
            presets: '<',
            facets: '<',
            updateStatus: '<',
        },
        controller: function($scope, HuggingFaceLocalConnectionService, CreateModalFromTemplate, Dialogs, $sce) {
            const kernelsStatusWrapper = {kernelsStatus: this.kernelsStatus};

            this.purposes = HuggingFaceLocalConnectionService.getPurposes();

            this.$onChanges = () => {
                kernelsStatusWrapper.kernelsStatus = this.kernelsStatus;
            };

            this.isHandlingModeSupportedForPurpose = (purpose, handlingMode) => {
                const handlingModes = HuggingFaceLocalConnectionService.getPossibleHandlingModes(purpose);
                const supportedHandlingModes = handlingModes.map(mode => mode.rawValue);
                return supportedHandlingModes.includes(handlingMode);
            }

            this.addCustomModel = function(purpose) {
                const newModel = HuggingFaceLocalConnectionService.getNoPresetCustomModel(purpose);
                this.openEditModal(newModel, purpose, "Edit new model", "Add");
            };

            this.appendModel = (model, purpose) => {
                const clonedModel = _.cloneDeep(model);
                this.openEditModal(clonedModel, purpose, "Edit new model", "Add");
            };

            this.addModelFromPresets = function (usagePurpose = null) {
                const newScope = $scope.$new();
                newScope.presets = this.presets;
                newScope.facets = this.facets;
                newScope.appendModel = this.appendModel;
                
                newScope.tagsDescription = {};
                const allTagNamesMap = new Map();
                for (const tag of this.facets.tags.values) {
                    newScope.tagsDescription[tag.id] = tag.description;
                    allTagNamesMap.set(tag.id, tag.name);
                }
                function getTagNames(tagIds) {
                    /**
                     * get tag display name from a list of tagIds.
                     * if no tagName is found, fallsback to tagId as a display name
                     */
                    const tagNamesMap = new Map();
                    tagIds.forEach(tagId => tagNamesMap.set(tagId, allTagNamesMap.get(tagId)));  // map preset tag ids to corresponding name if any, or null
                    
                    const tagNames = [];
                    allTagNamesMap.forEach((tagName, tagId) => tagNamesMap.has(tagId) ? tagNames.push(tagName) : null);  // add tags from global list in correct order
                    tagNamesMap.forEach((tagName, tagId) => tagName ?? tagNames.push(tagId));  // add extra tags that do not match any of the global list

                    return tagNames;
                }
                // Add tag names to the presets  
                newScope.presets.forEach((preset) => preset.tagNames = getTagNames(preset.facets.tags ?? []));

                newScope.selectedFacets = {
                    family: '',
                    tags: [],
                    mainUsagePurpose: usagePurpose
                };

                CreateModalFromTemplate("/templates/admin/connection-huggingface-local-add-model.html", newScope, 'HuggingFaceAddModelFromPresetController');
            };

            this.delete = (idx) => {
                const dialogScope = $scope.$new();
                const dialogModels = this.models;
                Dialogs.confirmAlert(dialogScope, "Delete model", "Are you sure you want to delete this model?").then(function() {
                    dialogModels.splice(idx, 1);
                }, function() {
                    // Dialog closed
                });
            };

            this._getCustomModelWarningMessages = (customModel, checkDuplicatedModelId) => {
                const warnings = [];
                if (checkDuplicatedModelId(customModel)) {
                    warnings.push("Duplicate model id.");
                }
                if (HuggingFaceLocalConnectionService.isGuardedModel(customModel.presetId) && !this.apiKey && customModel.enabled) {
                    warnings.push("This model is gated and cannot be used without an access token.");
                }
                if (warnings.length == 0) {
                    return null;
                }
                if (warnings.length == 1) {
                    return warnings[0];
                }
                return warnings.map(w => "• " + w).join("<br>");
            }

            this.getCustomModelWarningMessages = (customModel) => {
                return this._getCustomModelWarningMessages(customModel, (model) => {
                    const sameIdModels = this.models.filter(m => m.id == model.id);
                    return sameIdModels.length > 1;
                })
            };

            this.getHumanStatus = (modelId) => {
                if (this.kernelsStatus) {
                    const modelStatus = this.kernelsStatus['kernels'].filter(kernel => kernel.modelId === modelId);

                    if (modelStatus.some(kernel => kernel.state === "READY")) {
                        return $sce.trustAsHtml('<span style="color: green">Running</span>');
                    }

                    const deadKernels = modelStatus.filter(k => k.state === "DEAD");
                    let lastDeadKernel = null;
                    if (deadKernels && deadKernels.length > 0) {
                        lastDeadKernel = deadKernels.reduce((prev, current) => current.diedAtTime >= prev.diedAtTime ? current : prev);
                    }
                    if (lastDeadKernel && lastDeadKernel.deathReason && lastDeadKernel.deathReason.includes("FAIL")) {
                        return $sce.trustAsHtml('<span style="color: red">Error</span>');
                    }

                    if (modelStatus.some(kernel => kernel.state === "STARTING")) {
                        return $sce.trustAsHtml('<span style="color: orange">Starting</span>');
                    }

                    if (modelStatus.some(kernel => ["SENTENCED", "DYING"].includes(kernel.state))) {
                        return $sce.trustAsHtml('<span style="color: orange">Stopping</span>');
                    }

                    if (lastDeadKernel) {
                        return `Stopped ${moment(lastDeadKernel.diedAtTime).fromNow()}`;
                    }

                    return "&mdash;"
                }
            };

            this.getEnrichedLLMId = (model) => {
                return `huggingfacelocal:${this.connectionName}:${model.id}`
            }

            this.openStatusModal = (model) => {
                const newScope = $scope.$new();
                newScope.model = model;
                newScope.connectionName = this.connectionName;
                newScope.kernelsStatusWrapper = kernelsStatusWrapper;
                newScope.updateStatus = this.updateStatus;
                CreateModalFromTemplate("/templates/admin/connection-huggingface-local-custom-status-modal.html", newScope, "HuggingFaceModelStatusController");
            };

            this.openEditModal = (model, purpose, titleText = "Edit model", confirmText = "Ok") => {
                const newScope = $scope.$new();
                newScope.model = model;
                newScope.models = this.models;
                newScope.purpose = purpose;
                newScope.presetModels = this.presetModels;
                newScope.supportsLlmFineTuning = this.supportsLlmFineTuning;
                newScope.titleText = titleText;
                newScope.confirmText = confirmText;
                newScope._getCustomModelWarningMessages = this._getCustomModelWarningMessages;
                CreateModalFromTemplate("/templates/admin/connection-huggingface-local-edit-modal.html", newScope, "HuggingFaceModelEditController");
            }
        },
    });

    app.controller("HuggingFaceModelEditController", function($scope, HuggingFaceLocalConnectionService, Dialogs, HuggingFaceKernelPoolConstants, DataikuAPI) {
        const NULL_PRESET_ID = "NONE";
        $scope.HuggingFaceKernelPoolConstants = HuggingFaceKernelPoolConstants;
        $scope.quantizationModes = [
            { rawValue: "NONE", displayName: "None (recommended)" },
            { rawValue: "Q_8BIT", displayName: "8 bit" },
            { rawValue: "Q_4BIT", displayName: "4 bit" },
        ];
        $scope.enforceEagerModes = [
            { rawValue: "AUTO", displayName: "Auto (recommended)" },
            { rawValue: true, displayName: "Enforce eager mode" },
            { rawValue: false, displayName: "Enable CUDA graph" },
        ];
        $scope.trustRemoteCodeModes = [
            { rawValue: "AUTO", displayName: "Auto (recommended)" },
            { rawValue: true, displayName: "Yes" },
            { rawValue: false, displayName: "No" },
        ];
        $scope.enableExpertParallelismModes = [
            { rawValue: "AUTO", displayName: "Auto (recommended)" },
            { rawValue: true, displayName: "Yes" },
            { rawValue: false, displayName: "No" },
        ];

        $scope.supportedHandlingModes = HuggingFaceLocalConnectionService.getPossibleHandlingModes($scope.purpose);
        const supportedHandlingModesRawValues = $scope.supportedHandlingModes.map(mode => mode.rawValue);
        const supportedPresetModels = _.cloneDeep(($scope.presetModels || []).filter(model => supportedHandlingModesRawValues.includes(model.handlingMode)));

        $scope.tempModel = _.cloneDeep($scope.model);

        function applyDefaultValuesForEmptyFields() {
            $scope.tempModel.enforceEager = $scope.tempModel.enforceEager ?? "AUTO";
            $scope.tempModel.trustRemoteCode = $scope.tempModel.trustRemoteCode ?? "AUTO";
            $scope.tempModel.enableExpertParallelism = $scope.tempModel.enableExpertParallelism ?? "AUTO";
            $scope.tempModel.presetId = $scope.tempModel.presetId ?? NULL_PRESET_ID;
        }
        applyDefaultValuesForEmptyFields();

        // deprecated preset ids are considered as no preset id
        const isDeprecatedPresetId = !supportedPresetModels.some(model => model.presetId === $scope.tempModel.presetId);
        const nonePresetId = isDeprecatedPresetId ? $scope.tempModel.presetId : NULL_PRESET_ID;
        $scope.presetIdOptions = [
            { presetId: nonePresetId, displayName: "None" },
            ...supportedPresetModels
        ];

        $scope.checkDuplicatedModelId = () => {
            const sameIdModels = $scope.models.filter(m => m.id == $scope.tempModel.id);

            // If we're editing a model (and not adding a new one), and the id is the same as the original model (we didn't change it),
            // we check if there is ANOTHER one with the same id. Otherwise, we check if there is 1 or more with the same id
            if ($scope.models.includes($scope.model) && $scope.model.id === $scope.tempModel.id) {
                return sameIdModels.length > 1;
            };

            return sameIdModels.length > 0;
        };

        $scope.getCustomModelWarningMessages = () => {
            return $scope._getCustomModelWarningMessages($scope.tempModel, $scope.checkDuplicatedModelId)
        };

        $scope.applyPreset = function() {
            const presetModel = supportedPresetModels.find(o => o.presetId === $scope.tempModel.presetId);
            const newModel = _.cloneDeep(presetModel ?? HuggingFaceLocalConnectionService.getNoPresetCustomModel($scope.purpose));
            // refresh the "None" presetId option which may have changed for models that used to have a deprecated presetId
            $scope.presetIdOptions = [
                { presetId: NULL_PRESET_ID, displayName: "None" },
                ...supportedPresetModels
            ];

            const settingsToPreserve = [
                "enabled",
                "containerSelection",
                "cudaVisibleDevices",
                "minKernelCount",
                "maxKernelCount",
                "autoscalingTimeWindowSeconds",
                "autoscalingTargetRequestsPerKernel"
            ];
            settingsToPreserve.forEach(k => newModel[k] = $scope.tempModel[k]);

            angular.copy(newModel, $scope.tempModel);
            applyDefaultValuesForEmptyFields();
        };

        $scope.hasInferenceSettings = function() {
            return ['GENERIC_COMPLETION', 'TEXT_EMBEDDING_EXTRACTION'].includes($scope.purpose) || $scope.tempModel.handlingMode === 'IMAGE_GENERATION_DIFFUSION';
        };

        $scope.resetModel = function() {
            const inferenceText = $scope.hasInferenceSettings() ? ' and inference' : '';
            Dialogs.confirm($scope, 'Reset model', `Are you sure you want to reset this model ? It will overwrite the model${inferenceText} settings.`).then(function () {
                $scope.applyPreset();
            }, function () {
                // Dialog closed
            });
        };

        $scope.fixupNullValuesForTextInput = function(attr) {
            $scope.tempModel[attr] = $scope.tempModel[attr] ?? '';
        };

        $scope.saveModel = function() {
            if (!$scope.models.includes($scope.model)) {
                $scope.models.push($scope.model);
            }
            if ($scope.tempModel.dtype === "") $scope.tempModel.dtype = undefined;
            if ($scope.tempModel.enforceEager === "AUTO") $scope.tempModel.enforceEager = undefined;
            if ($scope.tempModel.trustRemoteCode === "AUTO") $scope.tempModel.trustRemoteCode = undefined;
            if ($scope.tempModel.enableExpertParallelism === "AUTO") $scope.tempModel.enableExpertParallelism = undefined;
            if ($scope.tempModel.presetId === NULL_PRESET_ID) $scope.tempModel.presetId = undefined;

            // fixup nullish values for connection dirtyness check
            Object.keys($scope.tempModel).forEach(k => $scope.tempModel[k] = $scope.tempModel[k] ?? undefined);

            angular.copy($scope.tempModel, $scope.model);
        }

        DataikuAPI.admin.getGeneralSettings().then(function({data}) {
            $scope.globalKernelIdleTTLSeconds = data.generativeAISettings.huggingFaceLocalSettings.kernelIdleTTLSeconds;
        }).catch(setErrorInScope.bind($scope));
    });

    app.controller("HuggingFaceModelStatusController", function ($scope, DataikuAPI) {
        $scope.getStateColor = (kernel) => {
            switch (kernel.state) {
                case "READY": return "green";
                case "STARTING":
                case "SENTENCED":
                case "DYING": return "orange";
                case "DEAD": {
                    if (kernel.deathReason && kernel.deathReason.includes("FAIL")) {
                        return "red";
                    } else if (kernel.deathError) {
                        return "orange";
                    } else {
                        return "inherit";
                    }
                }
            }
        };

        $scope.getStateText = (kernel) => {
            let humanReadableDeathReason = null;
            let failedMessage = null;
            switch (kernel.deathReason) {
                // See KernelPool.java#DeathReason enum
                case "STRATEGY": {
                    humanReadableDeathReason = "because it isn't needed anymore";
                    break;
                }
                case "PER_KERNEL_MAX_LIMIT": {
                    humanReadableDeathReason = "to comply with the per-model max limit";
                    break;
                }
                case "GLOBAL_MAX_LIMIT": {
                    humanReadableDeathReason = "to comply with the global model instance limit";
                    break;
                }
                case "ROOM_FOR_RESERVED_CAPACITY": {
                    humanReadableDeathReason = "to make room for reserved capacity for another model instance";
                    break;
                }
                case "ROOM_FOR_REQUEST": {
                    humanReadableDeathReason = "to make room for a request that had no model instance running";
                    break;
                }
                case "OUTDATED": {
                    humanReadableDeathReason = "because model settings have changed";
                    break;
                }
                case "USER_REQUEST": {
                    humanReadableDeathReason = "because it was requested to stop by a user";
                    break;
                }
                case "DEBUG": {
                    humanReadableDeathReason = "for debugging reasons";
                    break;
                }
                case "FAIL_START":
                    failedMessage = "Model instance failed to start. Check the logs for more information.";
                    break;
                case "FAIL_RUNNING": {
                    failedMessage = "Model instance failed while it was running. Check the logs for more information.";
                    break;
                }
                case "UNKNOWN":
                default:
                    humanReadableDeathReason = "for an unknown reason";
            }

            switch (kernel.state) {
                case "STARTING": {
                    return `Model instance is starting (${moment(kernel.startingAtTime).fromNow()})`;
                }
                case "READY": {
                    if (kernel.nbActiveRequests > 0) {
                        return `Processing ${kernel.nbActiveRequests} requests (started ${moment(kernel.readyAtTime).fromNow()})`;
                    }
                    else {
                        return `Ready to accept requests (started ${moment(kernel.readyAtTime).fromNow()})`;
                    }
                }
                case "SENTENCED": {
                    return (failedMessage ?? `Model instance is scheduled to shutdown ${humanReadableDeathReason}, after completing the ${kernel.nbActiveRequests} ongoing requests`) + ` (${moment(kernel.sentencedAtTime).fromNow()})`;
                }
                case "DYING": {
                    return (failedMessage ?? `Model instance is shutting down ${humanReadableDeathReason}`)+ ` (${moment(kernel.sentencedAtTime).fromNow()})`;
                }
                case "DEAD": {
                    if (kernel.deathError) {
                        return (failedMessage ?? `Model instance should have been stopped ${humanReadableDeathReason}, but an issue occurred during shutdown`) + ` (${moment(kernel.diedAtTime).fromNow()})`;
                    } else {
                        return (failedMessage ?? `Model instance was stopped ${humanReadableDeathReason}`) + ` (${moment(kernel.diedAtTime).fromNow()})`;
                    }
                };
                default:
                    return kernel.state;
            }
        };

        $scope.stoppedKernels = [];

        $scope.killKernel = (kernel) => {
            if ($scope.stopShouldBeDisabled(kernel)) return;

            DataikuAPI.admin.connections.killHFKernel(kernel.id).then(() => {
                $scope.updateStatus();
            });

            $scope.stoppedKernels.push(kernel.id);
        }

        $scope.stopShouldBeDisabled = (kernel) => {
            return $scope.stoppedKernels.includes(kernel.id) || (kernel.state != "STARTING" && kernel.state != "READY");
        }

        $scope.logs = {};
        this.fetchedStoppedKernelLogs = [];
        const updateLogs = () => {
            if ($scope.modelKernelsStatus) {
                $scope.modelKernelsStatus.forEach(kernel => {
                    if (kernel.state !== "DEAD" || !this.fetchedStoppedKernelLogs.includes(kernel.id)) {
                        DataikuAPI.admin.connections.getHfKernelLogs(kernel.id).success(function(data) {
                            $scope.logs[kernel.id] = data;
                        }).catch(setErrorInScope.bind($scope));

                        if (kernel.state === "DEAD") {
                            this.fetchedStoppedKernelLogs.push(kernel.id)
                        }
                    }
                });
            }
        }

        $scope.$watch('kernelsStatusWrapper.kernelsStatus', (status) => {
            $scope.modelKernelsStatus = status['kernels'].filter(kernel => kernel.modelId === $scope.model.id);
            $scope.runningKernelsStatus = _.sortBy($scope.modelKernelsStatus.filter(k => k.state !== "DEAD"), k => k.startingAtTime).reverse();
            $scope.deadKernelsStatus = _.sortBy($scope.modelKernelsStatus.filter(k => k.state === "DEAD"), k => k.diedAtTime).reverse();
            $scope.graveyardTimeout = status['graveyardTimeoutInS'];
            updateLogs();
        })
    });

    app.controller("HuggingFaceInferenceAPIConnectionController", function ($scope, $controller, TopNav, DataikuAPI) { // TODO @llm :implement HF API inference
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        if ($scope.creation) {
            $scope.connection.params.maxParallelism = 2;
        }

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testHuggingFace($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        }
    });

    app.controller("CustomLLMConnectionController", function ($scope, $controller, $rootScope, TopNav, DataikuAPI, PluginConfigUtils) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");
        $controller("_LLMConnectionController", {$scope});

        $scope._allCustomLLMPlugins = [...$scope.appConfig.customPythonLLMs, ...$scope.appConfig.customJavaLLMs];

        const idToType = {}
        $scope.appConfig.customPythonLLMs.forEach(llm => {
            idToType[llm.ownerPluginId] = "python"
        });
        $scope.appConfig.customJavaLLMs.forEach(llm => {
            idToType[llm.ownerPluginId] = "java"
        });

        $scope.customLLMPlugins = Array.from(new Set($scope._allCustomLLMPlugins.map(llm => llm.ownerPluginId)))
            .map(pluginID => $scope.appConfig.loadedPlugins.find(plugin => plugin.id === pluginID))
            .filter(plugin => plugin) // remove any `undefined` entries from plugins that could not be found
            .map(plugin => {
                plugin['type'] = idToType[plugin['id']] ?? "unknown";
                return plugin;
            })
            .sort((a,b) => a.label.localeCompare(b.label));

        $scope.selectedPlugin = $scope.customLLMPlugins.find(plugin => plugin.id === $scope.connection.params.pluginID);

        const cachedLLMDefinitions = {};

        $scope.$watch("selectedPlugin", function(nv, ov) {
            if (!nv) {
                $scope.connection.params.pluginID = "";
                $scope.pluginDesc = {};
                $scope.llmOptions = [];
            } else {
                $scope.connection.params.pluginID = $scope.selectedPlugin.id;
                $scope.pluginDesc = $scope.selectedPlugin;
                $scope.llmOptions = $scope._allCustomLLMPlugins
                                        .filter(llm => llm.ownerPluginId === nv.id)
                                        .sort((a,b) => a.desc.meta.label.localeCompare(b.desc.meta.label));
            }
            cachedLLMDefinitions[ov && ov.id] = $scope.connection.params.models;
            $scope.connection.params.models = cachedLLMDefinitions[nv && nv.id] || [];
            for (const model of $scope.connection.params.models) {
                $scope.onModelTypeChanged(model);
            }
        });

        let previousType;
        $scope.onModelTypeChanged = function(model) {
            model.$cachedCustomConfig = model.$cachedCustomConfig || {};
            model.$cachedCustomConfig[previousType || model.type] = angular.copy(model.customConfig);
            previousType = model.type

            if (model.$cachedCustomConfig[model.type]) {
                // Retrieve cached custom config for the newly selected type, if any
                model.customConfig = model.$cachedCustomConfig[model.type];
            }

            const loadedDesc = $scope.llmOptions.find(llm => llm.llmType == model.type);
            if (loadedDesc) {
                model.$desc = angular.copy(loadedDesc.desc);

                for (let paramName in model.customConfig) {  // Remove previous params that are not in the new model type
                    if (!model.$desc.params.some(x => x.name === paramName)) {
                        delete model.customConfig[paramName];
                    }
                }

                // This method sets default value for the relevant fields (based on the desc of the plugin) in model.customConfig, only if the fields are
                // already not defined
                PluginConfigUtils.setDefaultValues(model.$desc.params, model.customConfig);
            }
        };

        $scope.addModel = function() {
            DataikuAPI.humanId.create($scope.connection.params.models.map(model => model.id)).then(function({data}) {
                $scope.connection.params.models.push({
                    id: data.id,
                    capability: 'TEXT_COMPLETION',
                    type: undefined,
                    customConfig: {}
                });
            });
        };

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testCustomLLM($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(setErrorInScope.bind($scope))
                ["finally"](function () {
                $scope.testing = false;
            });
        }

        // Keep in sync with the backend Capability class (dip/connections/CustomLLMConnection.java)
        $scope.capabilities = [
            ['TEXT_COMPLETION', 'Chat completion'],
            ['TEXT_COMPLETION_MULTIMODAL', 'Chat completion (multimodal)'],
            ['TEXT_EMBEDDING', 'Text embedding'],
            ['IMAGE_GENERATION', 'Image generation'],
            ['TEXT_IMAGE_EMBEDDING_EXTRACTION', 'Multimodal embedding'],
        ];
    });

    app.controller("PineconeConnectionController", function ($scope, TopNav, DataikuAPI) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        if ($scope.creation) {
            $scope.connection.params["version"] = "POST_APRIL_2024";
        }

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testPinecone($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(
                setErrorInScope.bind($scope)
            ).finally(function () {
                $scope.testing = false;
            });
        };
    });

    app.controller("AzureAISearchConnectionController", function ($scope, TopNav, DataikuAPI) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.azureResourceURLFormat = "https://RESOURCE_NAME.search.windows.net"

        $scope.testConnection = function () {
            $scope.testing = true;
            DataikuAPI.admin.connections.testAzureAISearch($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(
                setErrorInScope.bind($scope)
            ).finally(function () {
                $scope.testing = false;
            });
        };
    });

    app.controller("KafkaConnectionController", function ($scope, $controller, DataikuAPI, TopNav, CodeMirrorSettingService, FeatureFlagsService) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.codeMirrorSettingService = CodeMirrorSettingService;

        $scope.securityModes = [
                                        {id:'NONE', label:'No security protocol'},
                                        {id:'KERBEROS', label:'Kerberos'},
                                        {id:'SASL', label:'Generic Sasl'},
                                        {id:'CUSTOM', label:'Custom (using properties)'}
                                    ];

        $scope.testConnection = function () {
            $scope.testing = true;
            $scope.testResult = null;
            DataikuAPI.admin.connections.testKafka($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(
                setErrorInScope.bind($scope)
            ).finally(function () {
                $scope.testing = false;
            });
        };

        $scope.testKsql = function() {
            $scope.testingKsql = true;
            $scope.testKsqlResult = null;
            DataikuAPI.admin.connections.testKsql($scope.connection).success(function (data) {
                $scope.testKsqlResult = data;
            }).error(
                setErrorInScope.bind($scope)
            ).finally(function () {
                $scope.testingKsql = false;
            });
        };

        $scope.showKsqlSettings = function() {
            return FeatureFlagsService.featureFlagEnabled('ignoreKsqlDeprecation');
        };
    });

    app.controller("SQSConnectionController", function ($scope, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.testConnection = function () {
            $scope.testing = true;
            $scope.testResult = null;
            DataikuAPI.admin.connections.testSQS($scope.connection).success(function (data) {
                $scope.testResult = data;
            }).error(
                setErrorInScope.bind($scope)
            ).finally(function () {
                $scope.testing = false;
            });
        }
    });


    app.controller("MongoDBConnectionController", function ($scope, $controller, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.checkForHttpInHostUrl = (host) => host && (host.startsWith('http://') || host.startsWith('https://'));

        if ($scope.creation) {
            $scope.connection.params["useURI"] = false;
            $scope.connection.params["uri"] = "mongodb://HOST:27017/DB";
        }

        var sequenceId = 0;
        $scope.testConnection = function () {
            if (!$scope.connectionParamsForm || $scope.connectionParamsForm.$valid) {
                $scope.testing = true;
                $scope.testResult = null;
                DataikuAPI.admin.connections.testMongoDB($scope.connection, ++sequenceId).success(function (data) {
                    if (data.sequenceId != sequenceId) {
                        // Too late! Another call was triggered
                        return;
                    }
                    $scope.testing = false;
                    $scope.testResult = data;
                }).error(setErrorInScope.bind($scope));
            }
        };

        // TODO - test on arrival - connection form not valid soon enough ???
    });

    app.controller("DynamoDBConnectionController", function ($scope, $controller, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        if ($scope.creation) {
            $scope.connection.params["regionOrEndpoint"] = "eu-west-3";
            $scope.connection.params["mode"] = "WEBSERVICE";
            $scope.connection.params["port"] = 8000;
            $scope.connection.params["hostname"] = "localhost";
            $scope.connection.params["rwCapacityMode"] = "ON_DEMAND";
            $scope.connection.params["readCapacity"] = 1;
            $scope.connection.params["writeCapacity"] = 1;
        }
        var sequenceId = 0;
        $scope.testConnection = function () {
            if (!$scope.connectionParamsForm || $scope.connectionParamsForm.$valid) {
                $scope.testing = true;
                $scope.testResult = null;
                DataikuAPI.admin.connections.testDynamoDB($scope.connection, ++sequenceId).success(function (data) {
                    if (data.sequenceId != sequenceId) {
                        // Too late! Another call was triggered
                        return;
                    }
                    $scope.testing = false;
                    $scope.testResult = data;
                }).error(setErrorInScope.bind($scope));
            }
        };
     });


    app.controller("CassandraConnectionController", function ($scope, $controller, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        addDatasetUniquenessCheck($scope, DataikuAPI);

        $scope.checkForHttpInHostsUrl = (hosts) => hosts && hosts.split(',').some(host => host.startsWith('http://') || host.startsWith('https://'));

        $scope.testConnection = function () {
            if (!$scope.connectionParamsForm || $scope.connectionParamsForm.$valid) {
                $scope.testing = true;
                $scope.testResult = null;
                DataikuAPI.admin.connections.testCassandra($scope.connection).success(function (data) {
                    $scope.testing = false;
                    $scope.testResult = data;
                }).error(setErrorInScope.bind($scope));
            }
        };

        if (!$scope.connection.customBasicConnectionCredentialProviderParams) {
            $scope.connection.customBasicConnectionCredentialProviderParams = [];
        }

        // TODO - test on arrival
    });

    app.controller("FTPConnectionController", function ($scope, $controller, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        if ($scope.creation) {
            $scope.connection.params.passive = true;
            $scope.connection.allowManagedDatasets = false;
        }
        $scope.connection.allowMirror = false;

        $scope.notTestable = true;

        if (!$scope.connection.params.dkuProperties) {
            $scope.connection.params.dkuProperties = [];
        }
    });

    app.controller("SSHConnectionController", function ($scope, $controller, DataikuAPI, TopNav) {
        TopNav.setLocation(TopNav.DSS_HOME, "administration");

        $scope.connection.allowMirror = false;
        $scope.notTestable = true;

        if ($scope.creation) {
            $scope.connection.allowManagedDatasets = false;
            $scope.connection.params["usePublicKey"] = false;
        }

        if (!$scope.connection.params.dkuProperties) {
            $scope.connection.params.dkuProperties = [];
        }
    });

    app.component('credentialsErrorHandler', {
        templateUrl: '/templates/admin/credentials-error-handler.html',
        bindings: {
            error: '<',
            short: '<?'
        },
        controller: function($scope, ActivityIndicator, CredentialDialogs, DataikuAPI) {
            const $ctrl = this;
            $scope.error = $ctrl.error;
            $scope.$watch('$ctrl.error', function(error) {
                $scope.short = $ctrl.short ? $ctrl.short : false;
                $scope.error = $ctrl.error;
                if(error && error.code && error.payload && (error.payload.connectionName || error.payload.pluginId)) {
                    DataikuAPI.profile
                        .listConnectionCredentials()
                        .then(({data: {credentials}}) => {
                            $scope.credential = credentials.find(credential =>
                                (error.payload.connectionName && error.payload.connectionName === credential.connection) ||
                                (error.payload.pluginId && credential.pluginCredentialRequestInfo &&
                                    error.payload.pluginId === credential.pluginCredentialRequestInfo.pluginId &&
                                    error.payload.paramSetId === credential.pluginCredentialRequestInfo.paramSetId &&
                                    error.payload.presetId === credential.pluginCredentialRequestInfo.presetId &&
                                    error.payload.paramName === credential.pluginCredentialRequestInfo.paramName
                                )
                            );
                            if($scope.credentials) {
                                $scope.plugin = $scope.credential.pluginCredentialRequestInfo;
                            }
                        });
                } else {
                    $scope.credential = null;
                }
            });

            $scope.needConnection = function() {
                return $scope.credential &&
                    ($scope.credential.type === 'OAUTH_REFRESH_TOKEN' ||
                    $scope.credential.type === 'AZURE_OAUTH_DEVICECODE');
            }

            $scope.enterCredential = function() {
                CredentialDialogs.enterCredential($scope, $scope.credential)
                    .then(function(redirect) {
                    if(!redirect) {
                        ActivityIndicator.success("Credential saved");
                        $scope.credentialEntered = true;
                    }
                });
            }

            $scope.isConnectionCredential = function() {
                return $ctrl.error && $ctrl.error.payload && $ctrl.error.payload.connectionName;
            }
        }
    });

}());
