(function(){
'use strict';


var app = angular.module('dataiku.admin.maintenance', []);


app.constant("TAIL_STATUS", {
    DEBUG: 0,
    INFO: 1,
    WARNING: 2,
    ERROR: 3
});


app.controller("AdminScheduledTasksController", function($scope, $rootScope, $state, DataikuAPI, ActivityIndicator, TopNav){
	TopNav.setLocation(TopNav.DSS_HOME, "administration");
	$scope.refresh = function() {
        DataikuAPI.admin.scheduledTasks.getStatus().success(function(data){
            $scope.status = data;
        }).error(setErrorInScope.bind($scope));
    };

    $scope.fireTask = function(task){
        DataikuAPI.admin.scheduledTasks.fire(task.jobGroup, task.jobName).success(function(data){
            ActivityIndicator.success("Task fired");
            $scope.refresh();
        }).error(setErrorInScope.bind($scope));
    };

    $scope.refresh();
});


app.controller("AdminMaintenanceInfoController", function($scope, DataikuAPI, TopNav) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");

    DataikuAPI.admin.getInstanceInfo().success(function(data){
        $scope.data = data;
    });
});


app.controller("AdminProfilingController", function($scope, DataikuAPI, TopNav) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");

    $scope.uiState = {};

    $scope.loadProfilesList = function() {
        $scope.uiState.loadingProfilesList = true;
        DataikuAPI.admin.profiling.listProfiles().success(function(data) {
            $scope.uiState.loadingProfilesList = false;
            $scope.profiles = data;
        });
    };

    $scope.downloadProfile = function(profile) {
        downloadURL(DataikuAPI.admin.profiling.downloadProfile(profile.id));
    };

    $scope.deleteProfile = function(profile) {
        DataikuAPI.admin.profiling.deleteProfile(profile.id).success(function() {
            $scope.loadProfilesList();
        });
    };

    $scope.refreshProfilerSettings = function() {
        DataikuAPI.admin.profiling.getConfig().success(function(profilerSettings) {
            $scope.profilerSettings = profilerSettings;
            $scope.originalProfilerSettings = angular.copy(profilerSettings);
        }).error(setErrorInScope.bind($scope));
    }

    $scope.saveProfilerSettings = function() {
        return DataikuAPI.admin.profiling.saveConfig($scope.profilerSettings).success(function() {
            $scope.refreshProfilerSettings();
            $scope.loadProfilesList();
        });
    };

    $scope.isDirty = function() {
        return $scope.originalProfilerSettings 
            && $scope.profilerSettings
            && !angular.equals($scope.originalProfilerSettings, $scope.profilerSettings);
    };

    $scope.loadProfilesList();
    $scope.refreshProfilerSettings();
});


app.controller("AdminLogsController", function($scope, $state, $rootScope, $window, $timeout,
               Logs, Diagnostics, DataikuAPI, ActivityIndicator, TopNav, TAIL_STATUS) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");
    //save JS data to file
    var saveData = (function () {
        var a = document.createElement("a");
        a.style.display = "none";
        document.body.appendChild(a);
        return function (data, fileName) {
            var blob = new Blob([data], {type: "octet/stream"}),
                url = window.URL.createObjectURL(blob);
            a.href = url;
            a.download = fileName;
            a.click();

            //give Firefox time...
            setTimeout(function(){
                window.URL.revokeObjectURL(url);
            }, 1000);
        };
    }());

    $scope.uiState = {
        active:  'logs',
        dState: null
    };

    $scope.TAIL_STATUS = TAIL_STATUS;

    $scope.loadLogsList = function() {
        $scope.uiState.loadingLogsList = true;
        Logs.list().success(function(data) {
            $scope.uiState.loadingLogsList = false;
            $scope.logs = data;
        });
    };

    $scope.loadLog = function(log) {
        $scope.uiState.currentLog = log;
        $scope.uiState.loadingLog = log;
        Logs.cat(log.name).success(function(data) {
            $scope.uiState.loadingLog = null;
            $scope.logData = data;
            $scope.logDataHTML = smartLogTailToHTML(data.tail, false);
            $timeout(function(){
                var content = $('.log-container .scrollable')[0];
                content.scrollTop = content.scrollHeight;
            })
        });
    };

    $scope.reloadLog = function() {
        if ($scope.uiState.currentLog) {
            $scope.loadLog($scope.uiState.currentLog);
        }
    };

    $scope.downloadExtract = function() {
        var text = $scope.logData.tail.lines.join('\n');
        var filename = 'extractof_'+$scope.uiState.currentLog.name;
        saveData(text, filename);
    };

    $scope.downloadCurrentLogFile = function() {
        if ($scope.uiState.currentLog) {
            Logs.download($scope.uiState.currentLog.name);
        }
    };


    $scope.downloadBackendLog = function() {
        Logs.download("backend.log");
    };


    $scope.downloadAllLogFiles = function() {
        Logs.downloadAll();
    };
    $scope.loadLogsList();

});

app.controller("AdminDiagnosticsController", function($scope, $state, $rootScope, $window, $timeout,
               Logs, Diagnostics, DataikuAPI, ActivityIndicator, TopNav, TAIL_STATUS, FutureProgressModal) {

    $scope.now = new Date().getTime()

    $scope.options = {
        includeConfigDir: true,
        includeBackendStacks: true,
        includeDockerImagesListing: true,
        includeFullLogs: false,
        includeFullDataDirListing: true
    };

    $scope.getLatestDiagnosis = function () {
        Diagnostics.getLatest(function(data) {
            if (data.exists) {
                $scope.latestDiagnosis = data;
            }
        });
    }

    $scope.downloadLatestDiagnosis = function () {
        Diagnostics.downLoadLatest();
    }

    $scope.runDiagnosis = function() {
        DataikuAPI.admin.diagnostics.run($scope.options).success(function(data) {
            FutureProgressModal.show($scope, data, "Running diagnosis...", null, 'static', false, true).then(function(result) {
                if (result) {
                    $scope.downloadLatestDiagnosis();
                }
                $scope.getLatestDiagnosis();
                    
            })
        }).error(setErrorInScope.bind($scope));
    }

    $scope.getLatestDiagnosis();
});

app.controller("AdminSanityCheckController", function($scope, $sce, DataikuAPI, TopNav, FutureProgressModal, WT1) {
    TopNav.setLocation(TopNav.DSS_HOME, "administration");

    const getLevel = function(message) {
        if(message.isFatal) return 0;
        if(message.severity === 'ERROR') return 1;
        if(message.severity === 'WARNING') return 2;
        if(message.severity === 'INFO') return 3;
        return 4;
    }

    const sortMessages = function (a, b) {
        return getLevel(a) !== getLevel(b) ?
            getLevel(a) - getLevel(b) :
            a.title.localeCompare(b.title);
    };

    const sortCodes = function (a, b) {
        return a.label.localeCompare(b.label);
    };

    $scope.now = new Date().getTime()
    $scope.exclusions = [];

    DataikuAPI.admin.sanityCheck.getRunningJob().success(function(data) {
        FutureProgressModal.showPeekOnlyIfRunning($scope, data.jobId, "Running sanity check...").then(function(result) {
            DataikuAPI.admin.sanityCheck.getLatestRun().success(function(result) {
                result.analysisResult.messages.sort(sortMessages)
                $scope.latestSanityCheck = result;
            }).error(setErrorInScope.bind($scope));
        })
    }).error(setErrorInScope.bind($scope));

    $scope.getCodes = function(customOptionsCallback) {
        Promise.all([DataikuAPI.admin.sanityCheck.getCodes(), DataikuAPI.admin.sanityCheck.getExclusions()])
            .then(([{data: codes}, {data: exclusions}]) => {
                codes.filter(code => exclusions.includes(code.value)).forEach(code => code.selected = true);
                codes.sort(sortCodes);
                customOptionsCallback(codes);
            }).catch(setErrorInScope.bind($scope));
    };

    $scope.runSanityCheck = function() {
        WT1.event('instance-sanity-check-run', {exclusions: $scope.exclusions});
        DataikuAPI.admin.sanityCheck.run($scope.exclusions).success(function(data) {
            FutureProgressModal.show($scope, data, "Running sanity check...", null, 'static', false, true).then(function(result) {
                result.messages.sort(sortMessages)
                $scope.latestSanityCheck = {
                    analysisResult: result,
                    epochTimestamp: new Date().getTime()
                 };
            })
        }).error(setErrorInScope.bind($scope));
    }

    $scope.resultContainsMessages = function() {
        return _.get($scope, "latestSanityCheck.analysisResult.messages", []).length > 0;
    }

    $scope.hasResults = function() {
        return _.get($scope, "latestSanityCheck.epochTimestamp", 0) !== 0;
    }

    $scope.downloadLatestAnalysis = function() {
        let data = {};
        angular.copy($scope.latestSanityCheck, data);
        delete data.analysisResult["futureLog"];

        const file = new Blob([JSON.stringify(data)], {type: 'application/json'});
        let a = document.createElement('a');
        a.href = URL.createObjectURL(file);
        a.download = "sanity_check.json";
        a.click();
    }
});

})();