/* global clippy */
(function() {
'use strict';

const app = angular.module('dataiku.controllers', ['dataiku.services', 'dataiku.filters', 'dataiku.markdown', 'dataiku.shared']);



app.controller('DataikuController', function($cacheFactory, $filter, $http, $injector, $state, $location, $modal, $rootScope,
       $route, $scope, $controller, $stateParams, $templateCache, $timeout, $exceptionHandler, $translate, translate,
       PreloadedTranslationTables, Assert, Dialogs, ActivityIndicator, FlowToolsLoader, Discussions,
       BackendReportsService, Breadcrumb, Throttle, CachedAPICalls,
       CreateModalFromTemplate, DataikuAPI, localStorageService, ContextualMenu,
       LoggerProvider, Notification, TopNav, WebSocketService, TrackingService, WT1, FullstorySupport,
       Markdown, GrelMode, RMarkdownMode, //Not used but included here to force load
       TaggingService, ProjectFolderContext,
       ExportUtils, ErrorReporting, StateUtils, SmartId, IntercomSupport, RecipeDescService, MessengerUtils,
       AlationCatalogChooserService, CodeMirrorSettingService, UserImageUrl, ProjectStatusService, HomePageContextService,
       CatalogItemService, Debounce, FeatureFlagsService, DetectUtils, FullScreenService, $httpParamSerializer, TaggableObjectsUtils, CatalogUtils,
       RequestCenterService, TopbarDrawersService, TOPBAR_DRAWER_IDS, OpalsService, StateObserverService, DatasetTypesService, BuiltinMapBackgrounds) {

    $rootScope.DataikuAPI = DataikuAPI;
    $rootScope.$state = $state;
    $rootScope.translate = translate;
    $scope.isFullScreen = FullScreenService.isFullScreen;

    TopNav.setLocation(TopNav.DSS_HOME);

    // Since the controller is not properly declared, it's not possible to use "Logger" directly (see angular-instantiable.js)
    const Logger = LoggerProvider.getLogger('DataikuController');
    Logger.info("Starting DSS load");

    window.APIErrorLogger = LoggerProvider.getLogger("api.errors");

    $rootScope.wl = {
        productShortName: "DSS",
        productLongName: "Dataiku DSS"
    }
    const dssMinorVersion = "14";
    $rootScope.dssMinorVersion = dssMinorVersion;
    $rootScope.versionDocRoot = `https://doc.dataiku.com/dss/${dssMinorVersion}/`;
    $rootScope.apiDocRoot = `https://doc.dataiku.com/dss/api/${dssMinorVersion}/`;
    $rootScope.academyRootUrl = "https://academy.dataiku.com/";
    $rootScope.learnRootUrl = "https://www.dataiku.com/learn/";
    $rootScope.kbaseRootUrl = `https://knowledge.dataiku.com/${dssMinorVersion}/`;

    // change the bootstrap-dropdown keydown listener so that it leaves mat-menus alone
    $(document).off('keydown.dropdown.data-api');
    $(document).on('keydown.dropdown.data-api', '[data-toggle=dropdown], [role=menu]:not(.mat-menu-panel, .mat-mdc-menu-panel)', $.fn.dropdown.Constructor.prototype.keydown)

    $controller("DatasetsCommon", {$scope: $scope});

    $rootScope.isTemporalType = DatasetTypesService.isTemporalType;
    $rootScope.isDateOnly = DatasetTypesService.isDateOnly;

    function userAvatar(userLogin, size) {
        if (!userLogin) return "";
        const imageUrl = UserImageUrl(userLogin, size);
        const sizeClass = size ? "size-" + sanitize(size) : "size-fit";
        return `<img class="user-avatar ${sizeClass}" src="${imageUrl}" />`;
    }

    function dssObjectLink(objectType, projectKey, objectId, innerHTML) {
        var link = StateUtils.href.dssObject(objectType, objectId, projectKey);
        return '<a href="'+link+'" class="link-std">'+innerHTML+'</a>';
    }

    function pluginStoreLink(pluginId, innerHTML) {
        var link = StateUtils.href.pluginStore(pluginId);
        return '<a href="'+link+'" class="link-std">'+innerHTML+'</a>';
    }

    function pluginSummaryLink(pluginId, innerHTML) {
        var link = StateUtils.href.pluginSummary(pluginId);
        return '<a href="'+link+'" class="link-std">'+innerHTML+'</a>';
    }

    function codeEnvLink(envName, envLang, innerHTML) {
        const link = StateUtils.href.codeEnvEdit(envName, envLang);
        return link ? `<a href="${link}" class="link-std">${innerHTML}</a>` : innerHTML;
    }

    function projectLink(projectKey, innerHTML) {
        var link = StateUtils.href.project(projectKey);
        return '<a href="'+link+'" class="link-std">'+innerHTML+'</a>';
    }

    function appLink(appId, innerHTML) {
        var link = StateUtils.href.app(appId);
        return '<a href="'+link+'" class="link-std">'+innerHTML+'</a>';
    }

    function dssObjectLabel(objectName) {
        return '<span>' + objectName + '</span>';
    }

    function userLink(userLogin, innerHTML) {
        return '<a href="/profile/'+escape(userLogin)+'/" class="link-std">'+ innerHTML + '</a>';
    }

    $scope.$on("$stateChangeStart", function(event, toState, toParams, fromState, fromParams) {
        Logger.debug('State: '+((fromState && fromState.name)?fromState.name:'Unknown') + ' -> '+ ((toState && toState.name)?toState.name:'Unknown'), toParams);
    });

    // Check for unsaved changes in the page before leaving:
    window.addEventListener("beforeunload", function (event) {
        try {
            if (typeof window.dssHasDirtyThings == "function" && window.dssHasDirtyThings()) {
                var msg = 'Unsaved changes will be lost';
                event.returnValue = msg; //this string will not be displayed anyway
                return msg;
            }
        } catch (e){
            Logger.error("Failed to compute dirtiness. Let it go.", e);
        }
    });

    $scope.reflow = {};
    $scope.$on('reflow',function() {
        $scope.reflow = {};
    });

    Notification.registerEvent('websocket-status-changed',function(evt,data) {
        $scope.wsFail = false;
        $("body").removeClass("ws-disconnected");
        if(data.code == WebSocketService.ERROR_CODE.CONNECTION_FAILED) {
            $scope.wsFail = true;
        } else if(data.code == WebSocketService.ERROR_CODE.CONNECTION_LOST) {
            $("body").addClass("ws-disconnected");
        }
    });

    $rootScope.$watch("promptRefreshFromVersionId", function (newValue, oldValue) {
        if (newValue && oldValue !== newValue && $rootScope.promptRefresh) {
            // Create a modal for the user that prompts them to refresh their page.
            // This modal cannot be dismissed by clicking outside
            Dialogs.confirmImportant(
                $scope,
                "New DSS Version",
                `A new version of DSS is available and it is recommended that you refresh your browser tab.<br>
                You can also dismiss this window and refresh later at your convenience.</br>
                Would you like to refresh now?`)
                .then(() => {
                    window.location.reload();
            }, (resolution) => {
                    if (resolution === "Cancelled") {
                        // User clicked on Cancel, dismiss the modal and don't show it back.
                        $rootScope.promptRefresh = false;
                    } else {
                        // Modal was destroyed without confirmation from the user, trigger it back.
                        $rootScope.promptRefreshFromVersionId = undefined;
                        $timeout(() => {
                            $rootScope.promptRefreshFromVersionId = newValue;
                        });
                    }
            });
        }
    });

    $scope.closeContextualMenus = function(){
        ContextualMenu.prototype.closeAny();
    };

    $scope.reconnectWebSocket = function() {
        WebSocketService.connect();
    };

    $scope.sendOfflineQueues = function() {
        DataikuAPI.internal.sendOfflineQueues();
    };
    $scope.slowFailAllBackendCalls = function(params) {
        DataikuAPI.internal.slowFail(params);
    };
    $scope.sendDigests = function() {
        DataikuAPI.internal.sendDigests();
    };
    $scope.buildUsageSummaryReports = function () {
        DataikuAPI.internal.buildUsageSummaryReports();
    };

    /* Put some stuff in the global scopes */
    $rootScope.$stateParams = $stateParams;
    $rootScope.StateUtils = StateUtils;
    $rootScope.SmartId = SmartId;
    $rootScope.dismissUpgradeProfile = localStorageService.get("dismissUpgradeProfile");
    $scope.$state = $state;
    $scope.sanitize = sanitize;
    $scope.JSON = JSON;
    $scope.$route = $route;
    $scope.Object = Object;
    $scope.pendingRequests = $http.pendingRequests;
    $rootScope.spinnerPosition = undefined;
    $scope.isTouchDevice = isTouchDevice();
    $scope.TopbarDrawersService = TopbarDrawersService;
    $scope.TOPBAR_DRAWER_IDS = TOPBAR_DRAWER_IDS;

    Breadcrumb.set([])

    $timeout(function() {
        $('.selectpicker').selectpicker();
        $('[data-toggle=dropdown]').dropdown();
     },10);

    /* Some global state management */
    $scope.$on('$stateChangeSuccess', function(e, toState){
        WT1.setSessionParam("currentState", toState.name);
        WT1.event("state-changed");
        $rootScope.$broadcast("dismissModals");
        $rootScope.$broadcast("dismissPopovers");
    });

    /* *************** Global login / config management ***************** */

    $scope.isSAASAuth = function() {
        return $rootScope.appConfig && $rootScope.appConfig.saasAuth;
    };

    $scope.onConfigurationLoaded = function() {
        IntercomSupport.activate();
        FullstorySupport.activate();
        Assert.inScope($rootScope, 'appConfig');
        if(!$rootScope.appConfig.unattendedMode) {
            StateObserverService.init();
        }
        $rootScope.wl = $rootScope.appConfig.whiteLabeling;

        if ($rootScope.wl.referenceDocRootUrl) {
            $rootScope.versionDocRoot = $rootScope.wl.referenceDocRootUrl;
        }
        if ($rootScope.wl.apiDocRootUrl) {
            $rootScope.apiDocRoot = $rootScope.wl.apiDocRootUrl;
        }
        if ($rootScope.wl.academyRootUrl) {
            $rootScope.academyRootUrl = $rootScope.wl.academyRootUrl;
        }
        if ($rootScope.wl.learnRootUrl) {
            $rootScope.learnRootUrl = $rootScope.wl.learnRootUrl;
        }

        if ($rootScope.appConfig.loggedIn) {
            WebSocketService.connect();
            $scope.countNotifications();
            updateRequestCount();
            // Temporary stuff ... Just in case it remained here...
            WT1.delVisitorParam("tutorial-project");
            WT1.delVisitorParam("tutorial-id");
            WT1.configure();
            ErrorReporting.configure();
            TrackingService.configurePingTracking();
            if ($rootScope.appConfig.customJS) {
                function evalCustomJS() {
                    try {
                        eval($rootScope.appConfig.customJS); //NOSONAR
                    } catch (e){
                        $exceptionHandler(e);
                    }
                }
                evalCustomJS();
            }

            if ($rootScope.appConfig.loadedPlugins) {
                $rootScope.appConfig.loadedPlugins.forEach(function(pluginDesc) {
                    if (!pluginDesc.customJSSnippets) {
                        return;
                    }
                    function evalCustomJSSnippet(snippet) { //Keep a named function to easily spot custom js in stacks
                        try {
                            eval(snippet); //NOSONAR
                        } catch (e) {
                            $exceptionHandler(e);
                        }
                    }
                    pluginDesc.customJSSnippets.forEach(evalCustomJSSnippet);
                });
            }

            if ($rootScope.appConfig.theme) {
                $scope.setTheme($rootScope.appConfig.theme);
            }

            // For consumption only profiles, check if a profile upgrade request is pending
            if ($rootScope.mayRequestProfileUpgrade && $rootScope.mayRequestProfileUpgrade()) {
                RequestCenterService.checkPendingProfileRequestStatus($rootScope);
            }

            /** Additional license info */
            $rootScope.addLicInfo = {};
            $rootScope.addLicInfo.sparkLicensed = $rootScope.appConfig.licensedFeatures && $rootScope.appConfig.licensedFeatures.sparkAllowed || $rootScope.appConfig.ceEntrepriseTrial;
            $rootScope.addLicInfo.hiveLicensed = !$rootScope.appConfig.community;
            $rootScope.addLicInfo.pigLicensed = !$rootScope.appConfig.community;
            $rootScope.addLicInfo.impalaLicensed = !$rootScope.appConfig.community;
            $rootScope.addLicInfo.containersLicensed = !$rootScope.appConfig.community;

            if ($rootScope.appConfig.alationSettings.enabled) {
                AlationCatalogChooserService.install();
            }
        } else {
            /* Still configure WT1 for push login state event */
            WT1.configure();
        }
        if (window.devInstance) {
            Mousetrap.bind("@ r r", function(){
                $templateCache.removeAll();
                $cacheFactory.get("$http").removeAll();
                $state.go($state.current, $stateParams, {reload:true, inherit: false, notify: true});
            })
            Mousetrap.bind("@ c c", function(){
                $templateCache.removeAll();
                $cacheFactory.get("$http").removeAll();
            })
        }
    };

    function getParameters(location = window.location) {
        let params = {};

        const urlParams = new URLSearchParams(location.search);
        let iterator = urlParams.entries();
        let result = iterator.next();

        while( !result.done ) {
            params[result.value[0]] = result.value[1];
            result = iterator.next();
        }

        location.hash.substring(1).split('&').forEach(kv => {
            kv = kv.trim();
            if (kv) {
                const [k, v] = kv.split('=');
                params[k] = v;
            }
        });
        return params;
    }

    BuiltinMapBackgrounds.bindOnTranslateChangeSuccess()

    DataikuAPI.getConfiguration().success(function(data) {

        // on safari 15.x cookies are sometimes not set before success is invoked
        // this led to other requests to fail with a 403 because they are triggered by this success without cookie properly set.
        // the timeout solution is fragile but it works
        // see https://app.shortcut.com/dataiku/story/88346/connection-on-safari-is-buggy
        $timeout(()=> {
            $rootScope.appConfig = data;
            $scope.appConfig = data;
            window.dkuAppConfig = data;

            if(data.loggedIn) {
                if (data.userSettings && data.userSettings.uiLanguage) {
                    if (data.translations) {
                        PreloadedTranslationTables.add(data.userSettings.uiLanguage, data.translations);
                    }
                    $translate.use(data.userSettings.uiLanguage);
                } else if (data.userSettings) {
                    data.userSettings.uiLanguage = "en"; // Use English if no language is selected.
                }
                RecipeDescService.load($scope);
                CachedAPICalls.notifyLoggedIn();
            }

            var ac = data;
            WT1.event("studio-open", {
                loggedIn : ac.loggedIn,
                installId: ac.installId,
                version: ac.version,
                deploymentMode: ac.deploymentMode,
                cloudStacksCloud: ac.cloudStacksCloud,
                launcherVersion: ac.launcherVersion,
                vmInfo: ac.vmInfo,
                wslInfo: ac.wslInfo,
                hadoopVersion: ac.hadoopVersion,
                hasNodeName: !!ac.nodeName,
                hasExternalURL: !!ac.dssExternalURL,
                themeId: ac.theme && ac.theme.id,

                hadoopEnabled: ac.hadoopEnabled,
                hiveEnabled: ac.hiveEnabled,
                impalaEnabled: ac.impalaEnabled,
                pigEnabled: ac.pigEnabled,
                twitterEnabled: ac.twitterEnabled,
                rEnabled: ac.rEnabled,
                legacyH2OEnabled: ac.h2oEnabled,
                impersonationEnabled: ac.impersonationEnabled,
                sparkEnabled: ac.sparkEnabled,
                pluginDevExplicitCommit: ac.pluginDevExplicitCommit,
                pluginDevGitMode: ac.pluginDevGitMode,
                alationEnabled: !!ac.alationSettings && ac.alationSettings.enabled,
                anonRegistrationAllowed: ac.anonRegistrationAllowed,
                gitMode : ac.gitMode,

                nbProjectStatus: ac.projectStatusList && ac.projectStatusList.length || 0,

                plugins: ac.loadedPlugins.map(x => x.id).join(","),
                customCodeRecipes: ac.customCodeRecipes && ac.customCodeRecipes.length || 0,
                customDatasets: ac.customDatasets && ac.customDatasets.length || 0,
                customDialects: ac.customDialects && ac.customDialects.length || 0,
                customExporters: ac.customExporters && ac.customExporters.length || 0,
                customFSProviders: ac.customFSProviders && ac.customFSProviders.length || 0,
                customJavaFormats: ac.customJavaFormats && ac.customJavaFormats.length || 0,
                customJythonProcessors: ac.customJythonProcessors && ac.customJythonProcessors.length || 0,
                customPythonChecks: ac.customPythonChecks && ac.customPythonChecks.length || 0,
                customPythonFormats: ac.customPythonFormats && ac.customPythonFormats.length || 0,
                customPythonPluginSteps: ac.customPythonPluginSteps && ac.customPythonPluginSteps.length || 0,
                customPythonPluginTriggers: ac.customPythonPluginTriggers && ac.customPythonPluginTriggers.length || 0,
                customPythonProbes: ac.customPythonProbes && ac.customPythonProbes.length || 0,
                customRunnables: ac.customRunnables && ac.customRunnables.length || 0,
                customWebApps: ac.customWebApps && ac.customWebApps.length || 0,
                customSQLProbes: ac.customSQLProbes && ac.customSQLProbes.length || 0,
                nbHomeMessages: ac.homeMessages && ac.homeMessages.length || 0, // TODO @homepage cleanup when removing feature flag
                isOpalsEnabled: !!ac.opalsEnabled,

                noneProfileStartBehavior: ac.noneUsersCallToActionBehavior,
                allowRequestAccessWithStartedTrial: ac.allowRequestAccessWithStartedTrial,
                requiresExternalWelcomeEmail: !ac.emailChannelId || !(ac.welcomeEmailSettings && ac.welcomeEmailSettings.enabled),
            });

            if (ac && ac.version && ac.version.product_version && ac.version.product_version.includes('dev') && !window.localStorage.forceRollbar) {
                // Disable WT1 reporting for dev kits
                window.devInstance = true;
            }

            if (!$scope.appConfig.loggedIn) {
                /* Don't redirect to login access to the login or logout page */
                if ($location.path().indexOf("/login/openid-redirect-uri/") === 0) {
                    const params = getParameters(window.location);

                    if (params["error"]) {
                        Logger.info("An error happened during the authentication.");
                        $state.transitionTo("sso-error", {error : "Identity provider error: " + params["error"], errorDescription: params["error_description"]});
                        return;
                    }

                    var code = params["code"];
                    var state = params["state"];

                    let getStateFromLocalStorage = localStorageService.get("openid-state");
                    localStorageService.remove("openid-state");
                    if (getStateFromLocalStorage != state) {
                        Logger.info("An error happened during the authentication. The State is not matching the initial one");
                        $state.transitionTo("sso-error", {error : "DSS error ", errorDescription: "Initial state value not matching request state param. This could be an attack attempt, please contact your administrator."});
                        return;
                    }

                    DataikuAPI.exchangeAuthorizationCode(code, state).success(function(data){
                        const redirectTo = localStorageService.get("postSSOLoginRedirect");
                        const search = localStorageService.get("postSSOLoginSearch") || "";
                        if (redirectTo) {
                            Logger.info("There is a post-SSO login redirect, following it", redirectTo + "?" + search);
                            localStorageService.remove("postSSOLoginRedirect");
                            localStorageService.remove("postSSOLoginSearch");
                            const url = new URL(window.location.href);
                            url.search = search;
                            url.pathname = redirectTo;  // Only follow redirects to a local path, not to another site
                            window.location = url.href;
                            return;
                        }
                    }).error(function(error) {
                        Logger.info("An error happened during the authentication.");
                        $state.transitionTo("sso-error", {error : "DSS error " + (error.code ? error.code: "") , errorDescription: error.detailedMessage});
                    });

                    return;
                } else if ($location.path().indexOf("/login/") === 0 || $location.path().indexOf("/login") === 0 || $location.path().indexOf("/logged-out") === 0 || $location.path().indexOf("/sso-error") === 0) {
                    return;
                } else if ($scope.appConfig.licensingMode == 'NONE') {
                    Logger.info("Not logged in, but registration flow active, not redirecting");
                } else if ($scope.isSAASAuth() && !$scope.appConfig.saasAccess.loggedIn) {
                    Logger.info("You are not logged in, redirecting you ...");
                    window.location = $scope.appConfig.saasUserURL + "/login/?redirectTo=" + window.location + "&search=" + $httpParamSerializer($location.search());
                } else if ($scope.isSAASAuth() && $scope.appConfig.saasAccess.loggedIn) {
                    Logger.info("logged in but no SAAS access");
                } else if ($scope.appConfig.noLoginMode) {
                    Logger.info("Not logged in, but no-login-mode enabled, getting an access token");
                    DataikuAPI.noLoginLogin().success(_ => location.reload());
                } else if ($scope.appConfig.ssoLoginEnabled) {
                    const params = getParameters();
                    if (params["error"]) {
                        Logger.info("An error happened during the authentication.");
                        // Redirect to login
                        $state.transitionTo("sso-error", {error : params["error"], errorDescription: params["error_description"]});
                        return;
                    }
                    const setPostSSOLogin = () => {
                        let path = '/', search = '';
                        if ('redirectTo' in params) {
                            [path, ...search] = params.redirectTo.split('?');
                            search = search.join('?');
                        } else if ($location.path()) {
                            path = $location.path();
                            search = $httpParamSerializer($location.search());
                        }
                        Logger.info("Setting a post-SSO redirect to", path, search);
                        localStorageService.set("postSSOLoginRedirect", path);
                        localStorageService.set("postSSOLoginSearch", search);
                        let redirectTo = path;
                        if (search) redirectTo += '?' + search;
                        return redirectTo;
                    };
                    if ($scope.appConfig.ssoProtocol == "SAML") {
                        const redirectTo = setPostSSOLogin();
                        DataikuAPI.getSAMLRedirectURL(redirectTo).success(function(data){
                            Logger.info("SAML redirect url: " + data.url);
                            window.location = data.url;
                        }).error(function(error) {
                            Logger.info("An error happened during the authentication.");
                            $state.transitionTo("sso-error", {error : "DSS error ", errorDescription: error.message ? error.message : "Please contact your administrator." });
                        });
                    } else if ($scope.appConfig.ssoProtocol == "OPENID") {
                        setPostSSOLogin();
                        DataikuAPI.getOpenIDRedirectURL().success(function(data){
                            localStorageService.set("openid-state", data.state);
                            window.location = data.url;
                        }).error(function(error) {
                            Logger.info("An error happened during the authentication.");
                            $state.transitionTo("sso-error", {error : "DSS error ", errorDescription: error.message? error.message : error});
                        });
                    } else if ($scope.appConfig.ssoProtocol == "SPNEGO") {
                        Logger.info("SPNEGO mode, redirecting to login URL");
                        window.location = "/dip/api/spnego-login";
                    }
                } else {
                    Logger.info("You are not logged in, redirecting you ...");
                    const params = getParameters();
                    // Redirect to login
                    if ('redirectTo' in params) {
                        const [path, ...search] = params.redirectTo.split('?');
                        $state.transitionTo("login", {redirectTo: path, search: search.join('?')});
                    } else {
                        $state.transitionTo("login", {redirectTo: $location.path(), search: $httpParamSerializer($location.search())});
                    }
                }
            } else if (!$scope.appConfig.unattendedMode &&  // When launched by puppeteer and automated tools, do not display the licence warning
                        $scope.appConfig.licensing.expired &&
                        $scope.appConfig.loggedIn &&
                        $scope.appConfig.admin) {
                const last = localStorageService.get("licenseExpired");
                if (!last || last <= Date.now() - 24 * 3600000) {
                    Dialogs.ack($scope,
                        "License expired!",
                        "Your DSS license expired on " + (new Date($scope.appConfig.licensing.expiresOn)).toLocaleString()
                    );
                    localStorageService.set("licenseExpired", Date.now());
                }
            } else if (!$scope.appConfig.unattendedMode && // When launched by puppeteer and automated tools, do not display the licence warning
                    $scope.appConfig.licensing.expiresOn &&  // 0 on Free Edition
                    $scope.appConfig.licensing.expiresOn <= Date.now() + 7 * 24 * 3600000 &&
                    $scope.appConfig.loggedIn &&
                    $scope.appConfig.admin) {
                const last = localStorageService.get("licenseExpiring");
                if (!last || last <= Date.now() - 24 * 3600000) {
                    Dialogs.ack($scope,
                        "License expires soon!",
                        "Your DSS license expires on " + (new Date($scope.appConfig.licensing.expiresOn)).toLocaleString()
                    );
                    localStorageService.set("licenseExpiring", Date.now());
                }
            } else {
                const redirectTo = localStorageService.get("postSSOLoginRedirect");
                const search = localStorageService.get("postSSOLoginSearch") || "";
                if (redirectTo) {
                    Logger.info("There is a post-SSO login redirect, following it", redirectTo + "?" + search);
                    localStorageService.remove("postSSOLoginRedirect");
                    localStorageService.remove("postSSOLoginSearch");
                    const url = new URL(window.location.href);
                    url.search = search;
                    url.pathname = redirectTo;  // Only follow redirects to a local path, not to another site
                    if (window.location.href !== url.href) {
                        Logger.info("redirect from", window.location.href, "to", url.href);
                        window.location = url.href;
                    } else {
                        Logger.info("no redirect needed");
                    }
                    return;
                }

                /* None user --> redirects to /home for the call to action */
                if ($scope.appConfig.userProfile && $scope.appConfig.userProfile.profile == 'NONE' && $location.path().indexOf("/home/") !== 0) {
                    Logger.info("NONE profile directly accessing a page, redirecting to home");
                    // TODO @homepage cleanup one migration done
                    if($rootScope.featureFlagEnabled('homepageRedesign')) {
                        $state.go('homeV2.homepage', null, {location: "replace"});
                    } else {
                        $state.go('home', null, {location: "replace"});
                    }
                }
            }
            $scope.onConfigurationLoaded();
        });
    }).error(setErrorInScope.bind($scope));

    $scope.isDSSAdmin = function(permission) {
        return $scope.appConfig && $scope.appConfig.loggedIn && $scope.appConfig.admin;
    };
    $rootScope.isDSSAdmin = $scope.isDSSAdmin;
    $rootScope.mayRequestProfileUpgrade = () => {
        const consumptionOnlyProfiles = ["READER", "EXPLORER", "AI_CONSUMER", "AI_ACCESS_USER", "GOVERNANCE_MANAGER"];
        const weirdlyForbidden = $scope.isDSSAdmin();
        return !weirdlyForbidden && $rootScope.appConfig && $rootScope.appConfig.userProfile && consumptionOnlyProfiles.includes($rootScope.appConfig.userProfile.profile) && !$rootScope.dismissUpgradeProfile
        && !($rootScope.appConfig.trialStatus && $rootScope.appConfig.trialStatus.valid);
    }

    $rootScope.sendWT1RequestUpgradeProfileShow = function() {
        WT1.tryEvent("request-upgrade-profile-show", () => {
            return { origin: "error-pop-up" }
        });
    }

    $rootScope.sendWT1RequestUpgradeProfileOpenModal = function() {
        WT1.tryEvent("request-upgrade-profile-open-modal", () => {
            return { origin: "error-pop-up" }
        });
    }

    $rootScope.openProfileUpgradeModal = function(fromErrorPopup) {
        if (fromErrorPopup) {
            $scope.sendWT1RequestUpgradeProfileOpenModal();
        }
        return CreateModalFromTemplate("/templates/request-profile-upgrade-modal.html", $rootScope, null, function(newScope) {
            newScope.profileUpgradeText = "Your current user profiles gives you limited access to Dataiku capabilities (mainly viewing content, but not modifying it).";
            newScope.textArea = { requestMessage: "" };
            RequestCenterService.checkPendingProfileRequestStatus($rootScope);
            newScope.requestProfileUpgrade = function(requestMessage) {
                DataikuAPI.requests.createProfileUpgradeRequest(requestMessage).success((data) => {
                    RequestCenterService.WT1Events.onRequestSent("PROFILE", null, null, requestMessage, data.id);
                    ActivityIndicator.success("Profile upgrade request sent!", 5000);
                    newScope.dismiss();
                    $rootScope.hasPendingProfileRequest = true;
                }).error(setErrorInScope.bind(newScope));
            };
        });
    };

    // TODO @homepage should be deletable once homepage is migrated
    $scope.canWriteInProjectFolder = function() {
        let currentFolder = ProjectFolderContext.getCurrentProjectFolder();
        return ProjectFolderContext.getCurrentProjectFolderId() && (currentFolder || {}).id == ProjectFolderContext.getCurrentProjectFolderId() ? (currentFolder && currentFolder.canWriteContents) : $rootScope.appConfig.globalPermissions.mayWriteInRootProjectFolder;
    };
    $rootScope.canWriteInProjectFolder = $scope.canWriteInProjectFolder;

    $scope.isPluginDeveloper = function() {
        return $scope.appConfig && $scope.appConfig.loggedIn && ($scope.appConfig.admin || $scope.appConfig.globalPermissions.mayDevelopPlugins);
    };

    $scope.isCodeStudioTemplateDeveloper = function() {
        return $scope.appConfig && $scope.appConfig.loggedIn && ($scope.appConfig.admin || $scope.appConfig.globalPermissions.mayCreateCodeStudioTemplates || $scope.appConfig.globalPermissions.mayManageCodeStudioTemplates);
    };

    $scope.isLibFolderEditor = function() {
        return $scope.appConfig && $scope.appConfig.loggedIn && ($scope.appConfig.admin || $scope.appConfig.globalPermissions.mayEditLibFolders);
    };

    $scope.mayWriteSafeCode = function() {
        return $scope.appConfig && $scope.appConfig.loggedIn &&
        ($scope.appConfig.admin || $scope.appConfig.globalPermissions.mayWriteUnsafeCode || ($scope.appConfig.impersonationEnabled && $scope.appConfig.globalPermissions.mayWriteSafeCode));
    };

    $scope.mayWriteUnsafeCode = function() {
        return $scope.appConfig && $scope.appConfig.loggedIn &&
        ($scope.appConfig.admin || $scope.appConfig.globalPermissions.mayWriteUnsafeCode);
    };

    $scope.mayCreateActiveWebContent = function() {
        return $scope.appConfig && $scope.appConfig.loggedIn &&
        ($scope.appConfig.admin || $scope.appConfig.globalPermissions.mayCreateActiveWebContent);
    };

    $rootScope.mayCreateCodeEnvs = function() {
        return $scope.appConfig && $scope.appConfig.loggedIn &&
        ($scope.appConfig.admin || $scope.appConfig.globalPermissions.mayCreateCodeEnvs || $scope.appConfig.globalPermissions.mayManageCodeEnvs);
    }

    $scope.canSeeAdminMenu = function() {
        if (!$scope.appConfig || !$scope.appConfig.loggedIn) {
            return false;
        }
        /* Because of code envs, anybody must be allowed to access the administration screen
         * (almost anybody will have at least "use") */
        return true;
    };

    $scope.adminMenuSref = function() {
        if (!$scope.appConfig || !$scope.appConfig.dkuCloudLaunchpadUrl) {
            return 'admin.home';
        }

        if ($scope.appConfig.isAutomation) {
            return 'admin.codeenvs-automation.list';
        }

        return 'admin.codeenvs-design.list'
    };

    $scope.openRequestTrialModal = function(){
        CreateModalFromTemplate("/templates/request-trial-modal.html", $scope);
    };

    $scope.logout = function(){
        if ($scope.isSAASAuth()) {
            window.location = $scope.appConfig.saasUserURL + "/logout/";
        } else {
            DataikuAPI.logout().success(function(data) {
                // Violent redirect to avoid keeping a cached appConfig
                if ($scope.appConfig && $scope.appConfig.postLogoutBehavior == "CUSTOM_URL") {
                    window.location = $scope.appConfig.postLogoutCustomURL;
                } else {
                    window.location = "/logged-out";
                }
            });
        }
    };

    /* ********************* Keyboard shortcuts handling ******************* */
    $scope.keyboardsModal = { shown : false };
    $rootScope.showKeyboardShortcuts = function() {
        if (!$scope.keyboardsModal.shown) {
            $scope.closeContextualMenus();
            $scope.keyboardsModal.shown = true;
            CreateModalFromTemplate("/templates/shortcuts.html", $scope, null, function(newScope) {
                newScope.$on("$destroy", function(){ $scope.keyboardsModal.shown = false});
            });
        }
    }

    $rootScope.showAdminContactInfo = function(){
        CreateModalFromTemplate("/templates/dialogs/admin-contact.html", $scope);
    }

    const originalStopCallback = Mousetrap.prototype.stopCallback;
    Mousetrap.prototype.stopCallback = function(e, element) {
        // if the element or any of its ancestors has the class "prevent-mousetrap" then deactivate all keyboard shortcuts
        if (element.closest('.prevent-mousetrap')) {
            return true;
        }
        return originalStopCallback.apply(this, arguments)
    }

    Mousetrap.bind("?", function() {
        $scope.showKeyboardShortcuts();
        $scope.$apply();
    });

    Mousetrap.bind(": q", function() {
        window.location = "about:blank"
    })

    var goToView = function(viewRoute) {
        return function() {
            if ($stateParams.projectKey) {
                let zoneId = $scope.getDestZone();
                $state.go(viewRoute, {projectKey : $stateParams.projectKey, zoneId : zoneId}, {reload: true});
            }
        }
    };

    $scope.reloadPluginConfiguration = function() {
        // reload config & descriptors for smoother plugin development
        if ($rootScope.appConfig && $rootScope.appConfig.loggedIn) {
            // no point if you're not already logged in
            DataikuAPI.plugindev.reloadAll().success(function(data) {
                DataikuAPI.getConfiguration().success(function(data) {
                    $rootScope.appConfig = data;
                    $scope.appConfig = data;
                    window.dkuAppConfig = data;
                    if ( CachedAPICalls != null ) {
                        // reload recipe types
                        RecipeDescService.load($scope);
                    }
                });
            });
        }
    };

    $scope.reloadAppConfig = function() {
        // reload config after modifying the admin settings for example
        if ($rootScope.appConfig && $rootScope.appConfig.loggedIn) {
            // no point if you're not already logged in
            DataikuAPI.getConfiguration().success(function(data) {
                $rootScope.appConfig = data;
                $scope.appConfig = data;
                window.dkuAppConfig = data;
            });
        }
    };

    Mousetrap.bind("@ r c d", $scope.reloadPluginConfiguration);
    Mousetrap.bind("g c", function() { $state.go("catalog.items", {}, {reload: true});  });
    Mousetrap.bind("g f", goToView("projects.project.flow"));
    Mousetrap.bind("g n", goToView("projects.project.notebooks.list"));
    Mousetrap.bind("g d", goToView("projects.project.datasets.list"));
    Mousetrap.bind("g r", goToView("projects.project.recipes.list"));
    Mousetrap.bind("g a", goToView("projects.project.analyses.list"));
    Mousetrap.bind("g p", goToView("projects.project.dashboards.list"));
    Mousetrap.bind("g i", goToView("projects.project.dashboards.insights.list"));
    Mousetrap.bind("g j", goToView("projects.project.jobs.list"));
    Mousetrap.bind("g q", goToView("projects.project.data-quality.current-status"));
    Mousetrap.bind("g w", goToView("projects.project.wiki"));
    Mousetrap.bind("g l", goToView("projects.project.libedition.versioned"));
    Mousetrap.bind("g h", function() { $rootScope.showHelp(null); });

    Mousetrap.bind("0 0 7", goToView("projects.project.agenttools.list"));

    Mousetrap.bind("c h a m p i o n s", function() {
        $(".master-nav").addClass("master-nav-champions");
        $(".icon-dkubird").replaceWith("<img class='champions'>");
        $(".dku-icon-dataiku-24").replaceWith("<img class='champions not-hover'>");
        $(".champions").attr("src","/static/dataiku/images/coq.png");
    });

    Mousetrap.bind("c f o m o d e", function() {
        $(".pivot-chart").css("filter", "grayscale(1) blur(3px)")
    });

    Mousetrap.bind("@ r b", function(){
        if ($state.current.name.startsWith("projects.project.webapps.webapp")){
            DataikuAPI.webapps.restartBackend({"projectKey":$stateParams.projectKey, "id":$stateParams.webAppId})
        }
    })

    Mousetrap.bind("s e u m", function() {
        $(".master-nav").addClass("master-nav-seum");
    });

    Mousetrap.bind("k i t t y", function() {
        CreateModalFromTemplate("/templates/kitty.html", $scope, null, function(newScope) {
            $timeout(function(){
                $rootScope.$broadcast("reflow")
            }, 0);
        });
    });

    Mousetrap.bind("r o c k e t", function() {
        if ($state.current.name.startsWith('projectdeployer')) {
            const sentences = [
                {msg: "I'm stepping through the door"},
                {error: true, msg: "Your circuit's dead, there's something wrong"},
                {error: true, msg:"Can you hear me, Major Tom?"},
                {msg: "Now it's time to leave the capsule if you dare"}
            ];

            CreateModalFromTemplate("/templates/rocket.html", $scope, null, function(newScope) {
                $scope.leaveMessage = sentences[Math.floor(Math.random()*sentences.length)];
                $timeout(function(){
                    $rootScope.$broadcast("reflow")
                }, 0);
            });
        }
    });

    Mousetrap.bind("p u p p y", function() {
        CreateModalFromTemplate("/templates/puppy.html", $scope, null, function(newScope) {
            $timeout(function(){
                $rootScope.$broadcast("reflow")
            }, 0);
        });
    });

    Mousetrap.bind("c o m i c", function() {
        var rnd = Math.floor(Math.random()*3);
        var font = ["'Comic Neue'", "cursive", "fantasy"][rnd];

        $("head").append($("<link rel='stylesheet' type='text/css' href='https://dku-assets.s3.amazonaws.com/comicneue/comicneue.css'>"));
        $("head").append($("<style>div, button, p, span, a, input, textarea { font-family: " + font + " !important;}</style>"));
    });

    function setColor(x) {
        $("body").css("background-color", x);
        $("#flow-graph").css("background-color", x);
    }

    Mousetrap.bind("s a r c e l l e", function() {
        if (Math.random() > 0.75) {
           setColor("#045067");
        } else {
            setColor("#2AB1AC");
        }
    });

    Mousetrap.bind("p i n k", function() {
            $("body").css("background-color", "pink");
    });

    Mousetrap.bind("p u r p l e", function() {
            $("body").css("background-color", "purple");
    });

    Mousetrap.bind("& @ &", function(){
        CreateModalFromTemplate("/templates/debugging-tools.html", $scope, "DebuggingToolsController");
    });

    Mousetrap.bind("@ t", function() {
        if ($scope.appConfig && $scope.appConfig.loggedIn && $scope.appConfig.admin) {
            CreateModalFromTemplate("/templates/translation-tools.html", $scope, "TranslationToolsController");
        }
    });

    function trollMe() {
        $("i").addClass("icon-spin");
        $(".avatar20").addClass("icon-spin");
        $(".avatar32").addClass("icon-spin");
        $(".avatar").addClass("icon-spin");
    }

    function untrollMe() {
        $("i").removeClass("icon-spin");
        $(".avatar20").removeClass("icon-spin");
        $(".avatar32").removeClass("icon-spin");
        $(".avatar").removeClass("icon-spin");
    }

    window.showNativeNotification = function(txt, tag, onclick, user) {
        if ("document.hasFocus", document.hasFocus()) return; // Only display notification when the user is not on the page
        if (window.Notification.permission === "default") {
            // User did not choose the notifications type yet, ask (but don't wait for the answer to display in-window notification)
            window.Notification.requestPermission(function (permission) {
                WT1.event("allow-browser-notification", {permission : permission});
            });
        } else if (window.Notification.permission === "granted") {
            // User native browser Notifications
            var options = {
                icon: UserImageUrl(user || $rootScope.appConfig.login, 200),
                dir: "ltr",
                tag: tag,
                renotify: false,
                silent: true
            };
            var notification = new window.Notification(txt, options);

            notification.onclick = (function(onclick) {return function () {
                window.focus();
                if (onclick) onclick();
                this.close();
            };})(onclick);

            var timeout = setTimeout((function(n){return function(){ n.close()}; }(notification)), 5000);// native notifications have no custom timeout
        }
    }

    Notification.registerEvent('spinnee-troll',function() {
        trollMe();
    });

    Notification.registerEvent('spinnee-untroll',function() {
        untrollMe();
    });

    Mousetrap.bind("w h e e e", function() {
        // Troll others
        Notification.broadcastToOtherSessions('spinnee-troll',{lol:"salut"});

        // Untroll me
        untrollMe();
    });

    Mousetrap.bind("w h o o o", function() {
        // Untroll everyone
        Notification.broadcastToFrontends('spinnee-untroll',{lol:"salut"});
    })

    function fallingBird() {
        $(".dku-icon-dataiku-24").css("visibility", "hidden");
        var falling = $("<i class='icon-dkubird falling-bird' />");
        $("body").append(falling);
        $("body").append($("<audio src='/static/dataiku/css/rifle.mp3' autoplay/>"));
        window.setTimeout(function(){falling.css("top", "105%")}, 10);
    }

    Mousetrap.bind("p a n", fallingBird);
    Mousetrap.bind("b a n g", fallingBird);

    Mousetrap.bind("s o n i a", function () {
        $("body").append($("<audio src='/static/dataiku/css/sonia.mp3' autoplay/>"));
    });

    Mousetrap.bind("d r w h o", function() {
        if (!$state.current.name.startsWith('projects.project.savedmodels.savedmodel') && !$state.current.name.startsWith('projects.project.analyses.analysis.ml.predmltask')) return;
        CreateModalFromTemplate("/templates/drwho.html", $scope, null, function(newScope) {
            $timeout(function(){
                $rootScope.$broadcast("reflow")
            }, 0);
        });
    });

    Mousetrap.bind("f u r y r o a d", function() {
        if ($state.current.name.startsWith('apideployer')) {
            $('head').append($('<style>@keyframes furyroad{from{left:0%;}to{left:100%;}}</style>'));
            $('body').append($('<div style="position: fixed; height: 400px; width: 600px; bottom: 0; animation-name: furyroad; animation-duration: 4s; animation-timing-function: ease-in-out; animation-iteration-count: infinite;"><svg viewBox="0 0 600 400"><circle cx="145" cy="295" r="30" fill="#6b6c69" stroke="#000"/><circle cx="145" cy="295" r="20" fill="#907354" stroke="#000"/><circle cx="145" cy="295" r="5" fill="#6b6c69" stroke="#000"/><circle cx="395" cy="295" r="30" fill="#6b6c69" stroke="#000"/><circle cx="395" cy="295" r="20" fill="#907354" stroke="#000"/><circle cx="395" cy="295" r="5" fill="#6b6c69" stroke="#000"/><path d="m100 300v-100q0-20 20-20h100q30 0 50 20l10 10q20 20 50 20l100-10q10 0 10 10v70h-10v-10q-7-13-20-20h-30q-13 7-20 20v10h-180v-10q-7-13-20-20h-30q-13 7-20 20v10h-10z" fill="#907354" stroke="#000"/><path d="m110 260q35-20 70 0l40 20q20 10 40 10" fill="transparent" stroke="#000"/><path d="m110 230v-30q0-10 10-10h50v40h-60z" fill="#ba9e9a" stroke="#000"/><path d="m180 230v-40h50q20 5 40 25v15h-90z" fill="#ba9e9a" stroke="#000"/><path d="m422 240l20-10q5-5 10 0v30q-5 5-10 0l-20-10q-5-5 0-10z" fill="#7d4842" stroke="#000"/><path d="m100 290l-80-40v-15l80 50v5z" fill="#6b6c69" stroke="#000"/><path d="m100 280l-80-60v-20l80 75v5z" fill="#6b6c69" stroke="#000"/><path d="m418 220l4-200h-2v-10h6v10h-2l-4 200h-2z" fill="#444" stroke="#000"/><path d="m422 220l10-200h-2v-10h6v10h-2l-10 200h-2z" fill="#444" stroke="#000"/><path d="m120 180l-30-50 40 50z" fill="#7d4842" stroke="#000"/><path d="m140 180l10-40v40z" fill="#7d4842" stroke="#000"/><path d="m170 180l-10-80 20 80z" fill="#7d4842" stroke="#000"/><path d="m190 180l-20-50 30 50z" fill="#7d4842" stroke="#000"/><path d="m210 180l40-50-30 50z" fill="#7d4842" stroke="#000"/><path d="m220 180l10-90v90z" fill="#7d4842" stroke="#000"/><path d="m130 180l-30-100 40 100z" fill="#7d4842" stroke="#000"/><path d="m155 180l-10-90 20 90z" fill="#7d4842" stroke="#000"/><path d="m180 180l20-80-10 80z" fill="#7d4842" stroke="#000"/><path d="m195 180v-120l10 120z" fill="#7d4842" stroke="#000"/><path d="m205 180l10-50v50z" fill="#7d4842" stroke="#000"/><path d="m330 230l-20-50 30 49z" fill="#7d4842" stroke="#000"/><path d="m352 228l10-40v38z" fill="#7d4842" stroke="#000"/><path d="m370 226l-10-80 20 79z" fill="#7d4842" stroke="#000"/><path d="m388 224l25-50-15 49z" fill="#7d4842" stroke="#000"/><path d="m410 222l-20-120 30 118z" fill="#7d4842" stroke="#000"/><path d="m420 220l30-90-20 90z" fill="#7d4842" stroke="#000"/><path d="m340 229l10-90v89z" fill="#7d4842" stroke="#000"/><path d="m358 227l50-120-40 119z" fill="#7d4842" stroke="#000"/></svg></div>'));
        }
    });

    Mousetrap.bind("n e y m a r", function() {
        $('head').append($('<style>@keyframes neymar{from{left:0%;transform:rotate(0deg);}to{left:100%;transform:rotate(4000deg);}}</style>'));
        const el = $('.icon-dkubird');
        el.css('position', 'absolute').css('color', 'yellow').css('background', '#184bad').css('top', '12px').css('animation-name', 'neymar').css('animation-duration', '10s').css('animation-timing-function', 'linear').css('animation-iteration-count', 'infinite');
    });

    Mousetrap.bind("b i g b r o t h e r", function() {
        $('body').append($('<div style="position: absolute;z-index: 2147483647000;bottom: 0;left: 0;right: 0;width: 100%;height: 415px;background: white;box-shadow: #121212 2px 2px 14px 2px;"><div style="margin: 20px auto;width: 450px;"><h4>This website uses quotes <i class="icon-eye-open"/></h4><p style=" line-height: 1.6em; font-size: 1.2em;">“He thought of the telescreen with its never-sleeping ear. They could spy upon you night and day, but if you kept your head you could still outwit them. With all their cleverness they had never mastered the secret of finding out what another human being was thinking. . . . Facts, at any rate, could not be kept hidden. They could be tracked down by inquiry, they could be squeezed out of you by torture. But if the object was not to stay alive but to stay human, what difference did it ultimately make? They could not alter your feelings; for that matter you could not alter them yourself, even if you wanted to. They could lay bare in the utmost detail everything that you had done or said or thought; but the inner heart, whose workings were mysterious even to yourself, remained impregnable.”</p><p class="" style="text-align: right;font-weight: bold;">― George Orwell, 1984</p><div><button class="btn btn--primary pull-right">2 + 2 = 5</button></div></div></div>'));
        $('body').append($('<div style="position: absolute;z-index: 2147483647000;top: 0;left: 0;right: 0;width: 100%;height: 150px;background: white;box-shadow: #121212 2px 2px 14px 2px;"><div style="margin: 20px auto;width: 450px;"><div class="alert alert-danger">Warning angry cookies</div><p style="height: 45px; background: repeat-x url(data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAADIAAAAyCAYAAAAeP4ixAAAAAXNSR0IArs4c6QAAAARnQU1BAACxjwv8YQUAAAAJcEhZcwAADsMAAA7DAcdvqGQAAAYWSURBVGhD7VntU1RVGHf6UP0FNVOyb6z7cu/de++CgiE0vGqZExIKgiAi6iTCIqCiIqDZjFZDGjiZTpqAmIqShhNmmk5NHypzaqaZBLIPOdVk04uwCAVyOr/b2W2XvazLyq5+2N/Mb/ZwznOfl3vOec5zD9MiiCCCCCKI4EGBwSA+pjVwxTZBPinyUp9phjigN/CjOgM/ptVz+UzMB0aj8ZH0uNiugrS4XvzibzYUXkTprUkiJ102m8WhlVnpzvb6ZeTqAQe52bmZDHfXk1lyzC2tlrcycR/MluXNrWvT7nTXPENaStPuPGW317Ch8CDKyEfbaACiYB9qqS0cc56tI6PnG3wYbeRHtFrto+wxH8BxBOAKJEGyb2RDoYfeKCylMzD4uiNn9FZXrWoALibFxfXrdNan2aM+cC+tjPiesC4ti0ncIfLy8Jf7ylUdH8/j24rGOIv0g8HAaZiK+w+LRW5ImhU/eOP4RlWnJ2Jj+eJ/jDOEv5ia+wujUSiw0Zm4cWJyQYDnG1cTOotXmCo3BEGwPJ8Y+/WChJnfmM02M+sOHTQamwF74qv9gS2n8XxrQx4xzxD3MnVuJNrl3e9VzyOnq+cStFl36IDs1OhYNKrmpCf72qpIc8Ui5dezH4GYTLY3mDo3TKb/Z8RiEU2sWwE2fWKMvCNekpJZ171BOSdoiu3v2urlnBod2SnK23Vkp3r1d+0qIZLNfpmpDAgJNIj9q1PHMufM7GVd9waRky+10nPC07GJiAA6q+aRchqQZ/+fZ2oxI0M4/ZnauyJe4pNpEH3JsfIZ1hU89HrhcZzYA2fvPhvg90eqSFNFNrneXu0zVlecOWyxSMeY6vACtdOqrLnO8U6h9Dj5UhHZu24xuXa40mtsIuJlJM+Od9JzaA9V/dB/FsIEgbd3oHbydAhBVCxOJwdfTFOWUUN+Cjm7a4WXzET8jdZfC1OTnDwn9Wr03NLp06UnOY57mJ78Wo1eKIiKMj/BTE8tUMVePeCdcju2F5G3aRCojVzckJvmJeOPIx82kO5XV5L8+an9AicN0gr5jo2XBhckJ962mMW+yeyjgEE3qBNv0dMRLKfOqrnegeR4Z6lg2Vi+6G+zyTYoWOUzRqOtiR6in5hM4jXmTvDA98RQt3dVe+2dSlKfl+IOAkHVF833ksE5glRc/kIqTQC+G98ff+6oIZ0vFytnD2YOPjB3godaIGDXzhV0OWWQ9XQmGpY/R/44vcVrvMmRTVwnNtqeY5Ph7Q/qpiYQLK2bnZtUjfhjX1u1e0aut69XlQmEsA0fmDvBQ22zh4NX3iojX1DiU0Hi7T3MneAh8FLHkTrv9BsOtm0tVKoB/PK8fJy5Ezy0Wm55SVbGgJoxEGv48OZ8hWirybgIxw5uXEIO1ixR2moy47kiM31Ap+OKmDvBAzkdJcpEBeOWwmdJW1m6QrTVZMDfaTIoy0oh75ZnKCxdmDxhMN8eWkfOvbaKwCbdH8NTdq7QovHj9rplqkUjspYrDaOtJgNiJo46MtyyR2kwh2ryVGV3rM4iw+fqSWttwRg9Ry4wN+4dUQYucZYc60RpMt4oZqFlbbrCWj8zAqcDCeQ7Wre9v7NEWaaxUoxTY+DnMDemBjarfKm5MndkvGEYbNmylL49/3sE5wyWEwIA1/hZWuCeipwRnrN/xMxPHTQaUY9P3UBvTtSobHa60Q9tyvMJApXx9pULlfbn+8qI2SQ5tVqzjpmfWhgMXJ7ABXf54I9IBPUlmeQnqhe3M7Chi7bmMLOhAf2ursN10I/HNqg6NVni7WMmMEPQCd2wwcyFFlaztB3XQnBCzblA2NNaRXatySanaHGIv6ELOs1maRszEx7oosVcWm47mypzVYtKTyIJ/EIr2s+aS8mb6/PIK6XZuH0kuCvGs7sdOSPQFfLlNBGQAASrdDFGsg8i56vdASNb7a1aQtrqCsmnzWvcmxyyeAYplu6JCyHb2JOBJppPoMviIioAlBSoj3CR9+upTcobB9FGH8aKM9P6ISvQww7PMjUPDlBKoC6iQZ2gn669rn/0gGhLvNyDMciE5HM2gggiiCCCCILCtGn/AgQAuiyJeko6AAAAAElFTkSuQmCC)"></p></div></div>'));
    });

    Mousetrap.bind("r m space - r f space /", function() {
        if ($state.current.name.startsWith('project-list')) {
            const e=$("<div style=\"position:absolute;padding:4px;width:500px;height:350px;top:50%;left:50%;transform:translate(-50%,-50%);background:#000;color:#fff;font-size:10px;font-family:Monaco,'SF Mono',Consolas,Console;z-index:999999999999;line-height:12px;font-weight:lighter;border-radius:6px;border-top:solid 20px #dedede;box-shadow:0 10px 20px rgba(0,0,0,.19),0 6px 6px rgba(0,0,0,.23)\"></div>"),i=$('<div style="overflow-y:auto;height:100%"></div>');e.append($('<div style="position:absolute;border:solid 6px #ff6158; border-radius:6px;top:-16px;left:7px"></div><div style="position:absolute;border:solid 6px #ffbd2e; border-radius:6px;top:-16px;left:26px"></div><div style="position:absolute;border:solid 6px #27c940; border-radius:6px;top:-16px;left:45px"></div>')),e.append(i),$("body").append(e);const h=$("<div>MBP-DKU:~ dataiku$ rm -rf /</div>");function l(){const o="rm: cannot remove `"+(1==Math.floor(2*Math.random())+1?"/etc/"+function(){const o=Math.floor(6*Math.random())+3;let t="";for(let e=0;e<o;e++)t+="abcdefghijklmnopqrstuvwxyz"[Math.floor(25*Math.random())];return t}()+".d":"/proc/"+(Math.floor(10235*Math.random())+255))+"': Permissions denied",t=$("<div>"+o+"</div>");i.append(t),t[0].scrollIntoView(),Math.floor(3*Math.random())===0&&$(".project-folder, .project").first().remove(),$timeout(l,Math.floor(50*Math.pow(10,Math.random()+.1))+2)}i.append(h),l(); //NOSONAR
        }
    });

    Mousetrap.bind("m a s s e", function() {
        let items = [
            { title: 'A', desc: "Qu'il n'est pas encore arrivé à Toronto" },
            { title: 'B', desc: "Qu'il est supposé arriver à Toronto, mais qu'on l'attend toujours" },
            { title: 'C', desc: "Qu'est-ce qu'il fout ce maudit pancake tabernacle\u00A0?" },
            { title: 'D', desc: "La réponse D" }
        ];
        Dialogs.select($scope, "Qui veut gagner de l'argent en masse\u00A0?",
            "Lorsqu'un Pancake prend l'avion à destination de Toronto, et qui s'en va faire une escale technique à St Claude, qui c'est qu'on va dire de ce pancake là ?",
            items,
        ).then(function(bafouille) {
            Dialogs.confirmSimple($scope, "C'est votre ultime bafouille\u00A0?", true).then(function() {
                 Dialogs.ack($scope, bafouille === items[2] ? "Bravo Gui" : "Vous auriez dû prendre le super moit-moit");
            });
        });
    });

    Mousetrap.bind("u n d e r space t h e space h o o d", () => {
        $('#dku-under-the-hood-dku').remove();
        $('.right-panel__content').append($('<div id="dku-under-the-hood-dku" style="position: absolute;z-index: 2147483647000;bottom: -151px;right: 0;width: 200px;height: 151px;background: white;transition: bottom 2s ease;-webkit-transition: bottom 2s ease;-moz-transition: bottom 2s ease;"><img style="width: 200px;height: 151px;" src="data:image/jpeg;base64,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"></div>'));
        setTimeout(() => $('#dku-under-the-hood-dku').css('bottom', '0px'), 200);
    });

    // Begining Clippy.js
    var clippyjs_agent;
    function create_clippyjs_agent(name_agent, callback) {
        name_agent = typeof name_agent !== 'undefined' ? name_agent : 'Clippy';
        clippy.load(name_agent, function(agent) {
            clippyjs_agent = agent;
            clippyjs_agent.show();
            //clippyjs_agent.moveTo(window.innerWidth-200,window.innerHeight-200);
            if (callback && typeof(callback) === "function") {
                callback();
            }
        });
    }

    Mousetrap.bind("c l i p p y", function() {
        if (typeof clippyjs_agent == 'undefined') {
            // First call : initializing
            clippyjs_agent = null;
            $("head").append($("<link rel='stylesheet' type='text/css' href='https://dku-assets.s3.amazonaws.com/clippy-js/clippy.css'>"));
            $("head").append($("<style>#loader-clippy{position:fixed;top:0px;left:0px;width:100%}#loader-clippy>div{margin:10em auto;"
                             +"font-size:10px;position:relative;text-indent:-9999em;border-top:1.1em solid rgba(255,255,255,.2);"
                             +"border-right:1.1em solid rgba(255,255,255,.2);border-bottom:1.1em solid rgba(255,255,255,.2);"
                             +"border-left:1.1em solid #ffc324;-webkit-transform:translateZ(0);transform:translateZ(0);"
                             +"-webkit-animation:load8 1.1s infinite linear;animation:load8 1.1s infinite linear;border-radius:50%;"
                             +"width:10em;height:10em}@-webkit-keyframes load8{0%{transform:rotate(0deg)}100%{transform:rotate(360deg)}}"
                             +"@keyframes load8{0%{transform:rotate(0deg)}100%{transform:rotate(360deg)}}"
                             +".clippy,.clippy-balloon{z-index:4000 !important}</style>"));
            $("body").append($("<div id='loader-clippy'><div></div></div>"));
            $("body").append($("<script type='text/javascript' src='https://dku-assets.s3.amazonaws.com/clippy-js/clippy.min.js'>"));
            setTimeout(function(){
                create_clippyjs_agent('Clippy', function(){
                    $("#loader-clippy").remove();
                });
            }, 1500);
        }
        else if (clippyjs_agent !== null) {
            // After first call : switching agent
            clippyjs_agent.hide(true, function() {
                $('.clippy').remove();
                $('.clippy-balloon').remove();
                if (clippyjs_agent.path.indexOf('Clippy') > -1) {
                    create_clippyjs_agent('Links', function() {
                        clippyjs_agent.play('GetWizardy');
                    });
                } else if (clippyjs_agent.path.indexOf('Links') > -1) {
                    create_clippyjs_agent('Merlin');
                } else {
                    create_clippyjs_agent('Clippy');
                }
            });
        }
    });

    Notification.registerEvent("job-state-change", function(evt, message) {
        if (message.state == "RUNNING" && typeof clippyjs_agent != 'undefined') {
            clippyjs_agent.play('Writing');
        } else if (message.state == "DONE" && typeof clippyjs_agent != 'undefined') {
            clippyjs_agent.play('Congratulate');
        } else if (message.state == "FAILED" && typeof clippyjs_agent != 'undefined') {
            clippyjs_agent.play('Alert');
        }
    });
    // End Clippy.js

    Mousetrap.bind("m s g a l l", function() {
        if ($rootScope.appConfig.admin === true) {
            var msg = window.prompt("What message do you want to send to all users?");
            Notification.broadcastToFrontends('msg-all', {msg:msg, user:$rootScope.appConfig.user});
        }
        else {
            alert("You must be admin to send a message."); // NOSONAR: OK to use alert for this message
        }
    });
    Notification.registerEvent('msg-all',function(evt, data) {
        MessengerUtils.post({
          message: "<div><b>Message from " + userLink(data.user.login, sanitize(data.user.displayName)) + ":</b><br>"+sanitize(data.msg)+"</div>",
          icon: userAvatar(data.user.login),
          hideAfter: 120,
          showCloseButton: true,
          id: 'msg-all-'+data.msg,
          type: 'no-severity'
        });
    });

    Mousetrap.bind("h a d o o p", function() {
        $("body").append($("<audio src='/static/dataiku/css/hadoop.mp3' autoplay/>"));
    });

    Mousetrap.bind("p i g", function() {
        $("body").append($("<audio src='/static/dataiku/css/pig.mp3' autoplay/>"));
    });

    Mousetrap.bind("h i v e", function() {
        $("body").append($("<audio src='/static/dataiku/css/hive.mp3' autoplay/>"));
    });

    Mousetrap.bind("p y t h o n", function() {
        $("body").append($("<audio src='/static/dataiku/css/python.mp3' autoplay/>"));
    });

    Mousetrap.bind("ctrl+e", function() {
        $rootScope.appConfig.easterEggs = true;
    });

    Mousetrap.bind("m i n i n g", function() {
        var elt = $("<div class='modal-container'><div id='minesweeper' class='modal modal3 dku-modal'><div class='modal-header'>"+
            "<h4><button type='button' class='close' data-dismiss='modal' aria-hidden='true'>&times;</button>"+
            "This is not data mining</h4></div><iframe  style='height: 500px;width: 98%' "+
            "src='http://www.chezpoor.com/minesweeper/minecore.html' /></div></div>");
        elt.modal("show");
    });

     Mousetrap.bind("k a t t a r s h i a n s", function() {
        var elt = $("<div class='modal-container'><div id='kattarshians' class='modal modal3 dku-modal relative-modal-90-90'>"+
            "  <iframe  style='width: 100%; height: 100%' "+
            "src='http://nutiminn.is/kattarshians/' /></div></div>");
        elt.modal("show");
    });

    Mousetrap.bind('up up down down left right left right b a enter', function() {
        CreateModalFromTemplate("/templates/infinity.html", $scope);
    });

    Mousetrap.bind("g e l l y", function(){
        window.setInterval(function(){
            d3.selectAll("g.node,g.edge").transition().duration(600).ease("elastic").attr("transform", function(d, i) {
                var x = 70 * Math.random() - 35;
                var y = 70 * Math.random() - 35;
                return "translate(" + x + " , " + y + ")";
            });
        }, 600)
    });

    Notification.registerEvent('discussions-wizz',function() {
        var elt = $('.discussions-widget-popover, .right-panel--opened .right-panel__content');
        elt.effect('shake');
    });

    Mousetrap.bind("w i z z", function() {
        Notification.broadcastToOtherSessions('discussions-wizz',{lol:"kikoo"});
    });

    Mousetrap.bind("l e a k s", function() {
        if ($state.current.name.startsWith('projects.project.wiki')) {
            $('.wiki-article-content.wiki-article-body-main').append($('<div style="position: absolute; top: 0; left: 0; right: 0; bottom: 0;"><div class="water"><div class="drop"></div><div class="drop"></div><div class="drop"></div><div class="drop"></div></div><svg version="1.1" xmlns="http://www.w3.org/2000/svg"><defs><filter id="goo"><feGaussianBlur in="SourceGraphic" result="blur" stdDeviation="12"/><feColorMatrix in="blur" mode="matrix" values="1 0 0 0 0 0 1 0 0 0 0 0 1 0 0 0 0 0 18 -7" result="goo"/></filter></defs></svg><style>.water{background:#083a44;width:200%;height:19%;position:absolute;bottom:-11%;left:-50%;-webkit-filter:url(#goo);filter:url(#goo)}.drop,.water::before{background:inherit;position:absolute;bottom:598%}.water::before{content:"";width:100%;height:100%}.drop{width:64px;height:64px;left:50%;border-radius:0 50% 50%;-webkit-transform:translateX(-50%) rotate(45deg);transform:translateX(-50%) rotate(45deg);-webkit-animation:drop 2s ease-in infinite;animation:drop 2s ease-in infinite}.drop:nth-child(1){width:64px;height:64px;-webkit-animation-delay:.1875s;animation-delay:.1875s}.drop:nth-child(2){width:51.2px;height:51.2px;-webkit-animation-delay:375ms;animation-delay:375ms}.drop:nth-child(3){width:38.4px;height:38.4px;-webkit-animation-delay:.5625s;animation-delay:.5625s}.drop:nth-child(4){width:25.6px;height:25.6px;-webkit-animation-delay:.75s;animation-delay:.75s}@-webkit-keyframes drop{0%{bottom:598%}100%,50%{bottom:0}}@keyframes drop{0%{bottom:598%}100%,50%{bottom:0}}</style></div>'));
        }
    });

    Mousetrap.bind("w h a t i f", function() {
        $(".master-nav__tile a div div").text("Hasta la vista, baby");
        setInterval(() => {
            $("body").effect('shake');
            $("a,span,p,button,option,h1,h2,h3,h4").not(":has(*:not(i)").each((i, x) => {
                if (/[a-zA-Z0-9]/.test($(x).text())) {
                    const newText = $(x).text().split('').sort(_ => 0.5 - Math.random()).join('');
                    $(x).text(newText);
                }
            })
        }, 5000);
    });

    $scope.win = function() {
         Notification.publishToFrontend("achievement-unlocked", {achievementId : 'LOL'});
    };

    /* ********************* Various notification stuff ******************* */

    Notification.registerEvent("login", function(evt, message) {
        if ($rootScope.appConfig && $rootScope.appConfig.login === message.user) {
            return; // Ignore event as it is about the current user
        }
        MessengerUtils.post({
          message: '<span>' + userLink(message.user, sanitize(message.userDisplayName)) + " just connected</span>",
          icon: userAvatar(message.user),
          hideAfter: 5,
          showCloseButton: true,
          id: message.user+'connected',
          type: 'no-severity'
        });
    });

    Notification.registerEvent("logout", function(evt, message) {
        if ($rootScope.appConfig && $rootScope.appConfig.login === message.user) {
            return; // Ignore event as it is about the current user
        }
        MessengerUtils.post({
          message: '<span>' + userLink(message.user, sanitize(message.userDisplayName)) + " just disconnected</span>",
          icon: userAvatar(message.user),
          hideAfter: 5,
          showCloseButton: true,
          id: message.user+'disconnected',
          type: 'no-severity'
        });
    });


    Notification.registerEvent("access-request", function(_, evt) {
        if ($rootScope.appConfig && ($rootScope.appConfig.login === evt.request.requesterLogin || !$rootScope.appConfig.userSettings.frontendNotifications.requestAccess)) {
            return; // Ignore event as it is about the current user or if option is disabled
        }

        let message = "<span>" + userLink(evt.request.requesterLogin, sanitize(evt.details.userDisplayName)) + " requested ";

        if (evt.objectType == "PROJECT") {
            message += " access to project " + projectLink(evt.projectKey, evt.details.objectDisplayName) + "</span>";
        } else if (evt.objectType == "APP") {
            message += " to execute application " + appLink(evt.objectId, evt.details.objectDisplayName) + "</span>";
        } else {
            message += " to share " + TaggableObjectsUtils.humanReadableObjectType(evt.objectType) + " "
                + dssObjectLink(evt.objectType, evt.projectKey, evt.objectId, evt.details.objectDisplayName);
        }

        if (evt.request.requestMessage && evt.request.requestMessage !== "") {
            message += '<span class="messenger-comment">' + sanitize(evt.request.requestMessage) + '</span>'
        }

        MessengerUtils.post({
          message: message,
          icon: userAvatar(evt.request.requesterLogin),
          hideAfter: 5,
          showCloseButton: true,
          id: evt.request.requestId + '-access-request',
          type: 'no-severity',
          actions: {
              gotoRequest: {
                  label: "See Request",
                  action: () => StateUtils.go.inboxRequest(evt.request.requestId)
              }
          }
        });
    });

    Notification.registerEvent("instance-access-request", function(_, evt) {
        if ($rootScope.appConfig && ($rootScope.appConfig.login === evt.requesterLogin || !$rootScope.appConfig.userSettings.frontendNotifications.requestAccess)) {
            return; // Ignore event as it is about the current user or if option is disabled
        }

        let message = "<span>" + userLink(evt.requesterLogin, sanitize(evt.details.authorDisplayName)) + " requested access to this " + $rootScope.wl.productLongName + " instance";

        if (evt.requestMessage && evt.requestMessage !== "") {
            message += '<span class="messenger-comment">' + evt.requestMessage + '</span>'
        }

        MessengerUtils.post({
          message: message,
          icon: userAvatar(evt.requesterLogin),
          hideAfter: 5,
          showCloseButton: true,
          id: evt.requestId + '-instance-access-request',
          type: 'no-severity',
          actions: {
              gotoRequest: {
                  label: "See Request",
                  action: () => StateUtils.go.inboxRequest(evt.requestId)
              }
          }
        });
    });

    Notification.registerEvent("profile-upgrade-request", function(_, evt) {
        if ($rootScope.appConfig && ($rootScope.appConfig.login === evt.requesterLogin || !$rootScope.appConfig.userSettings.frontendNotifications.requestAccess)) {
            return; // Ignore event as it is about the current user or if option is disabled
        }

        let message = "<span>" + userLink(evt.requesterLogin, sanitize(evt.details.authorDisplayName)) + " requested to upgrade their profile";

        if (evt.requestMessage && evt.requestMessage !== "") {
            message += '<span class="messenger-comment">' + evt.requestMessage + '</span>'
        }

        MessengerUtils.post({
          message: message,
          icon: userAvatar(evt.requesterLogin),
          hideAfter: 5,
          showCloseButton: true,
          id: evt.requestId + '-profile-upgrade-request',
          type: 'no-severity',
          actions: {
              gotoRequest: {
                  label: "See Request",
                  action: () => StateUtils.go.inboxRequest(evt.requestId)
              }
          }
        });
    });

    Notification.registerEvent("profile-upgrade-approved", function(_, evt) {
        if ($rootScope.appConfig && $rootScope.appConfig.login !== evt.requesterLogin) {
            return; // Ignore event as it is not about the current user
        }

        MessengerUtils.post({
          message: "<span>Your profile has been upgraded to a " + evt.selectedUserProfile +  " profile</span>",
          icon: userAvatar(evt.requesterLogin),
          hideAfter: 5,
          showCloseButton: true,
          id: evt.requesterLogin + '-profile-upgrade-approved',
          type: 'no-severity',
          actions: {}
        });
    });

    Notification.registerEvent("plugin-request", function(_, evt) {
        if ($rootScope.appConfig && ($rootScope.appConfig.login === evt.requesterLogin || !$rootScope.appConfig.userSettings.frontendNotifications.requestAccess)) {
            return; // Ignore event as it is about the current user or if option is disabled
        }

        let message = "<span>" + userLink(evt.requesterLogin, sanitize(evt.details.userDisplayName)) + " requested ";
        let actionType = evt.requestType === "INSTALL_PLUGIN" ? "installation" : "update";
        message += " " + actionType + " of the plugin " + pluginStoreLink(evt.objectId, evt.details.objectDisplayName) + "</span>";

        if (evt.requestMessage && evt.requestMessage !== "") {
            message += '<span class="messenger-comment">' + sanitize(evt.requestMessage) + '</span>'
        }

        MessengerUtils.post({
          message: message,
          icon: userAvatar(evt.requesterLogin),
          hideAfter: 5,
          showCloseButton: true,
          id: evt.requestId + '-plugin-request',
          type: 'no-severity',
          actions: {
              gotoRequest: {
                  label: "See Request",
                  action: () => StateUtils.go.inboxRequest(evt.requestId)
              }
          }
        });
    });

    Notification.registerEvent("code-env-request", function(_, evt) {
        Notification.publishToFrontend("update-pending-requests");
        if ($rootScope.appConfig && ($rootScope.appConfig.login === evt.requesterLogin || !$rootScope.appConfig.userSettings.frontendNotifications.requestAccess)) {
            return; // Ignore event as it is about the current user or if option is disabled
        }

        const user = userLink(evt.requesterLogin, sanitize(evt.details.userDisplayName));
        const action = (evt.requestType === "INSTALL_CODE_ENV") ? "installation" : "update";
        const commentHtml = (evt.requestMessage && evt.requestMessage !== "") ? '<span class="messenger-comment">' + sanitize(evt.requestMessage) + '</span>' : "";
        const message = `<span>${user} requested ${action} of the code env ` + sanitize(evt.objectId) + `</span>${commentHtml}`;

        MessengerUtils.post({
          message: message,
          icon: userAvatar(evt.requesterLogin),
          hideAfter: 5,
          showCloseButton: true,
          id: evt.requestId + '-code-env-request',
          type: 'no-severity',
          actions: {
              gotoRequest: {
                  label: "See Request",
                  action: () => StateUtils.go.inboxRequest(evt.requestId)
              }
          }
        });
    });

    Notification.registerEvent("access-granted", function(_, evt) {
        if ($rootScope.appConfig && ($rootScope.appConfig.login != evt.request.requesterLogin || !$rootScope.appConfig.userSettings.frontendNotifications.requestAccess)) {
            return; // Ignore event as it is not about the current user or if option is disabled
        }

        let message = "<span>" + "You have been granted";
        let objectLabel;

        if (evt.objectType == "PROJECT") {
            message += " access to a project: "
                + dssObjectLabel(evt.details.objectDisplayName)
                + "</span>";
            objectLabel = evt.objectType;
        } else if (evt.objectType == "APP") {
            message += " rights to execute an application: "
                + dssObjectLabel(evt.details.objectDisplayName)
                + "</span>";
            objectLabel = evt.objectType;
        } else {
            message += " access to a"
                + (evt.objectType == "MODEL_EVALUATION_STORE" ? "n " : " ")
                + TaggableObjectsUtils.humanReadableObjectType(evt.objectType)
                + ": "
                + dssObjectLink(evt.objectType, evt.projectKey, evt.objectId, evt.details.objectDisplayName)
                + "</span>";
            objectLabel = TaggableObjectsUtils.humanReadableObjectType(evt.objectType);
        }

        MessengerUtils.post({
          message: message,
          icon: userAvatar(evt.request.requesterLogin),
          hideAfter: 5,
          showCloseButton: true,
          id: evt.request.requesterLogin + '-access-granted',
          type: 'no-severity',
          actions: {
              gotoRequest: {
                  label: "Open " + objectLabel,
                  action: () => StateUtils.goWithOptions({reload: true}).dssObject(evt.objectType, evt.objectId, evt.projectKey)
              }
          }
        });
    });

    Notification.registerEvent("plugin-request-granted", function(_, evt) {
        if ($rootScope.appConfig && ($rootScope.appConfig.login !== evt.requesterLogin || !$rootScope.appConfig.userSettings.frontendNotifications.requestAccess)) {
            return; // Ignore event as it is not about the current user or if option is disabled
        }

        let requestAction = evt.requestType === "INSTALL_PLUGIN" ? "installed" : "updated"
        let message = "<span>" + "Your requested plugin " + pluginSummaryLink(evt.objectId, evt.details.objectDisplayName) + " has been " + requestAction + "</span>";
        if (evt.requestMessage && evt.requestMessage !== "") {
            message += '<span class="messenger-comment">' + sanitize(evt.requestMessage) + '</span>'
        }

        MessengerUtils.post({
          message: message,
          icon: userAvatar(evt.requesterLogin),
          hideAfter: 5,
          showCloseButton: true,
          id: evt.requesterLogin + '-plugin-request-granted',
          type: 'no-severity',
          actions: {}
        });
    });

    Notification.registerEvent("code-env-request-granted", function(_, evt) {
        Notification.publishToFrontend("update-pending-requests");
        if ($rootScope.appConfig && ($rootScope.appConfig.login !== evt.requesterLogin || !$rootScope.appConfig.userSettings.frontendNotifications.requestAccess)) {
            return; // Ignore event as it is not about the current user or if option is disabled
        }

        const codeEnvName = sanitize((evt.details && evt.details.objectDisplayName) || evt.objectId);
        const commentHtml = (evt.requestMessage && evt.requestMessage !== "") ? '<span class="messenger-comment">' + sanitize(evt.requestMessage) + '</span>' : "";
        const codeEnvLinkHtml = codeEnvLink(codeEnvName, evt.envLang, codeEnvName);
        let message = `<span>Your requested code env ${codeEnvLinkHtml} has been installed</span>${commentHtml}`;
        if(evt.targetName){
            message = `<span>Your code env request has been processed ${codeEnvLinkHtml}</span>${commentHtml}`;
        }

        MessengerUtils.post({
          message: message,
          icon: userAvatar(evt.requesterLogin),
          hideAfter: 5,
          showCloseButton: true,
          id: evt.requesterLogin + '-code-env-request-granted',
          type: 'no-severity',
          actions: {}
        });
    });

    Notification.registerEvent("job-state-change", function(evt, message) {
        if (!displayUserTaskNotification(message)) return;

        // for job state changes coming from scenarios, only notify if corresponding user setting is enabled
        if (message.triggeredFrom === "SCHEDULER" && !$rootScope.appConfig.userSettings.frontendNotifications.scenarioRun) return;

        // If we are on the recipe page and the result panel is open, the info is already displayed
        try {
            var resultPanel = angular.element(".recipe-editor-job-result, .recipe-settings-floating-result");
            if (resultPanel.length && resultPanel.scope().isJobRunning()) {
                if (resultPanel.scope().startedJob.jobId == message.jobId) {
                    return;
                }
            }
        } catch (e) {
            Logger.error("Failed to check if user is on running recipe page.", e);
        }


        var initiatedByCurrentUser = $rootScope.appConfig.login == message.initiator;

        function jobLink(innerHTML) {
            var link = StateUtils.href.job(message.projectKey, message.jobId);
            return '<a href="'+link+'" class="link-std">'+innerHTML+'</a>';
        }

        function goToLogs() {
            return StateUtils.go.job(message.projectKey, message.jobId);
        }

        function goToFirstOutput() {
            var output = message.outputs[0];
            if (!output) {
                return;
            }
            if (!output.type) {
                throw new Error("Job output type not specified: "+ angular.toJson(output));
            }
            return StateUtils.go.dssObject(output.type, output.targetDataset, output.targetDatasetProjectKey);
        }

        var gotoJob = (function(message){ return function() {
            StateUtils.go.job(message.projectKey, message.jobId);
        }})(message);

        var triggerLabel = '<i class="icon-play"/> ';
        if (message.triggeredFrom == 'SCHEDULER') {
            triggerLabel = '<i class="icon-calendar" title="Scheduled in a scenario"/> ';
        } else if (message.triggeredFrom == 'API') {
            triggerLabel = '<i class="icon-code" title="Triggered from API"/> ';
        }

        var userLabel = initiatedByCurrentUser ? '' :
            userLink(message.initiator,
                        userAvatar(message.initiator, 20)
                        + '<span class="messenger-initiator">'
                        + sanitize(message.initiatorDisplayName || message.initiator)
                        + '</span>'
                    ) + '<br />';
        var jobLabel = jobLink(sanitize(message.humanReadableJobDesc));

        if (message.state == "DONE") {
            var warnLevel = message.warningsCount ? 'warning' : 'success';
            window.showNativeNotification("Job completed", message.jobId, gotoJob, message.initiator);
            if (!message.warningsCount) {
                MessengerUtils.post({
                    message: '<div>' + userLabel + 'Job completed<br/>' + jobLabel + '</div>',
                    icon: triggerLabel,
                    type: warnLevel + ' current-user',
                    id: message.jobId,
                    hideAfter: 5,
                    actions: {
                        target: {
                            label: "View",
                            action: goToFirstOutput
                        },
                        logs: {
                            label: "Logs",
                            action: goToLogs
                        }
                    }
                });
            }
        } else if (message.state == "PENDING") {
             MessengerUtils.post({
                 message: '<div>' + userLabel + 'Job pending<br/>' + jobLabel + '</div>',
                 icon: triggerLabel,
                 type: 'current-user',
                 hideAfter: 5,
                 id: message.jobId,
                 showCloseButton: true,
                 actions: {}
             });
        }  else if (message.state == "STARTING") {
            MessengerUtils.post({
                message: '<div>' + userLabel + 'Job starting<br/>' + jobLabel + '</div>',
                icon: triggerLabel,
                type: 'current-user',
                hideAfter: 5,
                id: message.jobId,
                showCloseButton: true,
                actions: {
                    logs: {
                        label: "Logs",
                        action: goToLogs
                    }
                }
            });
        } else if(message.state == "FAILED" || message.state == "ABORTED") {
            window.showNativeNotification("Job " + message.state.toLowerCase(), message.jobId, gotoJob, message.initiator);
            MessengerUtils.post({
                message: '<div>' + userLabel + 'Job ' + message.state.toLowerCase() + '<br/>' + jobLabel + '</div>',
                icon: triggerLabel,
                type: 'error current-user',
                showCloseButton: true,
                id: message.jobId,
                hideAfter: 5,
                actions: {
                    logs: {
                        label: "Logs",
                        action: goToLogs
                    }
                }
            });
        }
    });


    function mlTaskLink(mlTaskType, projectKey, analysisId, mlTaskId, innerHTML) {
        var href = StateUtils.href.mlTask(analysisId, projectKey, mlTaskType, mlTaskId);
        return '<a href="'+href+'" class="link-std">'+innerHTML+'</a>';
    }

    function displayUserTaskNotification(evt) {
        // For scenario-state-change, we can have multiple initiator
        // others event still use .initiator
        const initiatedByCurrentUser = evt.initiator === $rootScope.appConfig.login || (evt.initiators ? evt.initiators.includes($rootScope.appConfig.login) : false);
        var otherUsersTasks = $rootScope.appConfig.userSettings.frontendNotifications.otherUsersTasks;
        return initiatedByCurrentUser || otherUsersTasks;
    }

    Notification.registerEvent("mltask-state-change", function(evt, message) {
        if (!displayUserTaskNotification(message)) return;

        // If we are on the mltask page, the info is already displayed
        try {
            var x = 'projects.project.analyses.analysis.ml.';
            if (
                ($state.current.name.startsWith(x+'predmltask.list.results') || $state.current.name.startsWith(x+'clustmltask.list.results'))
                && message.projectKey == $stateParams.projectKey
                && message.taskId == $stateParams.mlTaskId
                ) {
                return;
            }
        } catch (e) {
            Logger.error("Failed to check if user is on running mlTask page.", e);
        }

        MessengerUtils.post({
            message: mlTaskLink(message.taskType, message.projectKey, message.analysisId, message.taskId, sanitize(message.name + ": training done")),
            icon: '<i class="icon-dku-nav_analysis" />',
            type: 'success',
            showCloseButton: true
        });
    });

    Notification.registerEvent("workspace-invitation-sent", (_, message) => handleInvitationSentEvent(message));

    Notification.registerEvent("project-invitation-sent", (_, message) => handleInvitationSentEvent(message));

    const handleInvitationSentEvent = (message) => {
        if ($rootScope.appConfig && $rootScope.appConfig.login !== message.user) {
            return; // Ignore event as it is not triggered by the current user
        }
        if (message && message.result && message && message.result.infoMessages) {
            message.result.infoMessages.messages.forEach((message) => {
                MessengerUtils.post({
                    message: '<b> ' + message.title + '</b><div>' + message.details + '</div>',
                    type: 'error',
                    icon: '<i class="dku-icon-mail-24"/>',
                    hideAfter: 5,
                    showCloseButton: true
                });
            });
        }
        if (message && message.result && message && message.result.sentEmails) {
            message.result.sentEmails.forEach((email) => {
                MessengerUtils.post({
                    message: `Invitation email sent to ${email}`,
                    type: 'success',
                    icon: '<i class="dku-icon-mail-24"/>',
                    hideAfter: 5,
                    showCloseButton: true
                });
            });
        }
    }

    Notification.registerEvent("timeline-item", function(evt, message) {
        if (message.item.action == "COMMENT") {
            MessengerUtils.post({
                message: userLink(message.item.user, sanitize(message.item.details.userDisplayName))
                    + " commented on "
                    + dssObjectLink(message.item.objectType, message.item.projectKey, message.item.objectId, sanitize(message.item.details.objectDisplayName))
                    + ":"
                    + '<span class="messenger-comment">'
                    + sanitize(message.item.details.text.substr(0,400))
                    + (message.item.details.text.length > 400 ? '[...]' : '')
                    + '</span>'
                    ,
                icon: userAvatar(message.item.user),
                type: 'no-severity',
                hideAfter: 5,
                showCloseButton: true
            });
        } else if (message.item.action == "EDIT_COLLABORATIVE_METADATA") {
            Assert.trueish(message.item.details.doneTasks != null, 'no done tasks');
            var tasks = "";
            for (var i = 0; i < message.item.details.doneTasks.length; ++i) {
                tasks += '<i class="icon-ok" /> ' + sanitize(message.item.details.doneTasks[i]);
            }

            MessengerUtils.post({
                message: userLink(message.item.user, sanitize(message.item.details.userDisplayName))
                    + " completed a task on "
                    + dssObjectLink(message.item.objectType, message.item.projectKey, message.item.objectId, sanitize(message.item.details.objectDisplayName))
                    + ":"
                    + '<span class="messenger-comment">'
                    + tasks
                    + '</span>'
                    ,
                icon: userAvatar(message.item.user),
                type: 'no-severity',
                hideAfter: 5,
                showCloseButton: true
            });
        }
    });

    Notification.registerEvent("commit-mention", function(evt, message) {
        MessengerUtils.post({
            message: userLink(message.author, sanitize(message.details.authorDisplayName || message.author))
                + " mentioned you in commit: "
                + '<span class="messenger-comment">'
                + sanitize(message.message.substr(0,400))
                + (message.message.length > 400 ? '[...]' : '')
                + '</span>'
                ,
            icon: userAvatar(message.author),
            type: 'no-severity',
            showCloseButton: true
        });
    });

    Notification.registerEvent("interest-added", function(evt, message) {
        MessengerUtils.post({
            message: '<i class="icon-star interests-star active"></i>'
                + userLink(message.user, sanitize(message.details.userDisplayName))
                + ' starred '
                + dssObjectLink(message.objectType, message.projectKey, message.objectId, sanitize(message.details.objectDisplayName)),
            icon: userAvatar(message.user),
            id: message.user+'starred'+message.details.objectDisplayName,
            type: 'no-severity',
            showCloseButton: true
        });
    });

    Notification.registerEvent("scenario-run-failed-check-logs", function(evt, message) {
        if (!displayUserTaskNotification(message) || !$rootScope.appConfig.userSettings.frontendNotifications.scenarioRun) return;
        MessengerUtils.post({
          message: "Failed to run scenario " + message.projectKey+"."+message.scenarioId + ": " + message.message + ".\nPlease check logs",
          icon: '<i class="icon-calendar"/>',
          type: "error",
          id: "ScenarioState"+ message.scenarioId,
          showCloseButton: true
        });
    });

    Notification.registerEvent("scenario-state-change", function(evt, message) {
        if (!$rootScope.appConfig.userSettings.frontendNotifications.scenarioRun) return;
        function goToLogs() {
            return StateUtils.go.scenario(message.scenarioId, message.projectKey);
        }
        if (!displayUserTaskNotification(message)) return;
        var msg = {
                DONE: 'finished',
                DONE_WITH_WARNING: 'finished',
                RUNNING: 'started',
                FAILED: 'failed',
                ABORTED: 'aborted'
            };
        var isSuccess = ['DONE'].indexOf(message.state) >= 0;
        var isWarning = ['DONE_WITH_WARNING'].indexOf(message.state) >= 0;
        var isError = ['FAILED', 'ABORTED'].indexOf(message.state) >= 0;
        var triggerLabel = '<i class="icon-calendar" /> ';
        var actions = {};
        if (isSuccess || isWarning || isError) {
            actions.logs = {label: "Logs",action: goToLogs};
        }
        MessengerUtils.post({
          message: "Scenario " + msg[message.state] + "<br/>" + dssObjectLink('SCENARIO', message.projectKey, message.scenarioId, message.scenarioName),
          icon: triggerLabel,
          id: "ScenarioState"+ message.scenarioId,
          type: isSuccess ? 'success' : (isError ? 'error' : (isWarning ? 'warning' : '')),
          showCloseButton: true,
          actions : actions
        });
    });

    $scope.showAbout = function() {
        $scope.closeContextualMenus();
        CreateModalFromTemplate("/templates/about-dss.html", $scope, null, function(modalScope){
            modalScope.currentYear = new Date().getFullYear();
        });
    };

    $scope.showAboutPartitioning = function() {
        $scope.closeContextualMenus();
         CreateModalFromTemplate("/templates/about-partitioning.html", $scope);
    };

    $rootScope.showFeedbackModal = function() {
        $scope.closeContextualMenus();
        CreateModalFromTemplate("/templates/widgets/topbar_drawers/feedback-modal.html", $scope, 'FeedbackController');
    };

    $scope.HOME_STATE_HOMEPAGE = $rootScope.featureFlagEnabled('homepageRedesign') ? 'homeV2.homepage' : 'home';
    $scope.PREVIOUS_HOME_STATE = $scope.HOME_STATE_HOMEPAGE;
    $scope.PREVIOUS_HOME_STATE_PARAMS = undefined;

    $scope.redirectHome = () => {
        $state.go($scope.HOME_STATE_HOMEPAGE, {}, { reload: true });
    };

    $scope.isOnProjectHome = () => $state.includes('projects.project.home.summary');

    $scope.isOnHome = () => {
        const currentState = $state.current.name;
        return currentState.startsWith('homeV2') || currentState === 'home';
    }

    $scope.homeButtonTooltip = () => {
        if($scope.isOnHome()) return '';
        return 'Home';
    }

    $scope.freshWidgetInitDone = false;

    $rootScope.showHelp = function(opalsPage) {
        OpalsService.isEnabled().then(function(isOpalsEnabled) {
            if (isOpalsEnabled) {
                if (opalsPage === null) {
                    TopbarDrawersService.getDrawer(TOPBAR_DRAWER_IDS.OPALS_HELP).show();
                } else {
                    OpalsService.navigateToAndShowDrawer(opalsPage ? opalsPage : OpalsService.PAGES.HELP);
                }
                return;
            }
            function showSupportWidget() {
                if (window.dkuAppConfig && window.dkuAppConfig.offlineFrontend){
                    ActivityIndicator.error("Offline mode - Support widget not available");
                    return;
                }
                if (!$scope.freshWidgetInitDone) {
                    const version = $rootScope.dssMinorVersion;
                    const instanceId = $rootScope.appConfig.dipInstanceId;
                    if (!window.devInstance) {
                        window.FreshWidget.init("", {
                            "queryString": "&widgetType=popup&helpdesk_ticket[custom_field][dss_version_112979]=" + version
                                + "&helpdesk_ticket[custom_field][dss_instance_112979]=" + instanceId,
                            "widgetType": "popup", "buttonType": "text", "buttonText": "Support",
                            "buttonColor": "white", "buttonBg": "#006063", "alignment": "4",
                            "submitThanks" : "Thanks for your message. We'll get in touch very soon",
                            "offset": "-1500px", "formHeight": "500px",
                            "url": "https://dataiku.freshdesk.com",
                            "loadOnEvent" : "immediate"
                        } );
                    } else {
                        ActivityIndicator.error("Support widget not available on a dev instance");
                        return;
                    }
                    $scope.freshWidgetInitDone = true;
                }
                window.FreshWidget.show();
            }
            let wt1EventParams;
            if ($rootScope.wl) {
                wt1EventParams = {
                    wlReferenceDocRootUrlSet: !!$rootScope.wl.referenceDocRootUrl,
                    wlApiDocRootUrlSet: !!$rootScope.wl.apiDocRootUrl,
                    wlAcademyRootUrlSet: !!$rootScope.wl.academyRootUrl,
                    wlLearnRootUrlSet: !!$rootScope.wl.learnRootUrl,
                    wlGetHelpModalTitleSet: !!$rootScope.wl.getHelpModalTitle,
                    wlGetHelpModalTextSet: !!$rootScope.wl.getHelpModalText,
                };
            } else {
                wt1EventParams = {};
            }
            if ($rootScope.appConfig) {
                wt1EventParams.admin = !!$rootScope.appConfig.admin;
                wt1EventParams.studioAdminContactSet = !!$rootScope.appConfig.studioAdminContact;
                wt1EventParams.communityEdition = !!$rootScope.appConfig.communityEdition;
            }
            wt1EventParams.intercomEnabled = !!$rootScope.intercomEnabled;
            WT1.event("help-open-get-help", wt1EventParams);
            CreateModalFromTemplate("/templates/widgets/topbar_drawers/get-help-modal.html", $scope, null, function(newScope) {
                newScope.openSupport = function() {
                    newScope.dismiss();
                    showSupportWidget();
                };
                newScope.openIntercom = function() {
                    newScope.dismiss();
                    $scope.forceShowIntercom();
                };
                OpalsService.getHelpCenterUrl().then(function(url) {
                    newScope.helpCenterUrl = url;
                });
            });
        });
    };

    const getZoomedZoneCtxKey = () => {
        return `dku.flow.zoneId.${$stateParams.projectKey}`;
    }

    $scope.getDestZone = function() {
        if ($stateParams.zoneId || $state.current.name.includes('projects.project.home.summary')) {
            localStorageService.remove(getZoomedZoneCtxKey());
            return null;
        }
        return localStorageService.get(getZoomedZoneCtxKey());
    }

    $scope.$on("$stateChangeSuccess", function(_event, to, toParams, from, fromParams) {
        if (to.pageTitle) {
            TopNav.setPageTitle(to.pageTitle(toParams));
        }
        if (to.name.includes('projects.project.flow')) {
            localStorageService.set(getZoomedZoneCtxKey(), toParams.zoneId);
        }
        if ($rootScope.featureFlagEnabled('homepageRedesign') && (from.name.includes('homeV2') || from.name === 'home')) {
            $scope.PREVIOUS_HOME_STATE = from.name;
            $scope.PREVIOUS_HOME_STATE_PARAMS = fromParams;
        }
    });


    /* ******************* Persistent notifications handling ****************** */

    $scope.pnotifications = {};

    $scope.countNotifications = function() {
        DataikuAPI.notifications.count().success(function(data) {
            $scope.pnotifications.totalUnread = data.totalUnread;
            $rootScope.totalUnreadNotifications = data.totalUnread;
            TopNav.refreshPageTitle();
        });
    };

    Notification.registerEvent("update-notifications-count", function(evt, message) {
        // Manytimes the sent event contains the totalUnread value, no need to fetch it again
        if (message.totalUnread == -1) {
            $scope.countNotifications();
        } else {
            $scope.pnotifications.totalUnread = message.totalUnread;
            $rootScope.totalUnreadNotifications = message.totalUnread;
            TopNav.refreshPageTitle();
        }
    });

    $rootScope.discussionsUnreadStatus = $rootScope.discussionsUnreadStatus || {};
    Notification.registerEvent("discussions-unread-full-ids-changed", function(evt, message) {
        let newFIDs = angular.copy(message.unreadFullIds || []);
        $rootScope.discussionsUnreadStatus.unreadFullIds = newFIDs;
    });

    function updateRequestCount() {
        RequestCenterService.getAllRequests().then(result => {
            $scope.pendingAccessRequests = RequestCenterService.numberOfPendingRequests(result);
            $scope.hasAccessRequests = RequestCenterService.hasRequests(result);
        })
    }

    $scope.pendingAccessRequests = 0;
    Notification.registerEvent("update-pending-requests", function() {
       updateRequestCount();
    });

    $scope.hasUnreadThings = function() {
        return $scope.pnotifications.totalUnread || ($rootScope.discussionsUnreadStatus.unreadFullIds || []).length;
    };


    /* ******************* Exports handling ****************** */

    Notification.registerEvent("export-state-change", function(evt, message) {
        if (!displayUserTaskNotification(message)) {
            return;
        }
        var txt = null;
        var type = null;
        if (message.status.state == 'DONE') {
            var lowerDescription = message.status.inputDescription.description ? message.status.inputDescription.description.toLowerCase() : '';
            const datasetName = sanitize(message.status.inputDescription.name);
            const datasetProjectKey = sanitize(message.status.inputDescription.projectKey);
            if (lowerDescription.startsWith("dataset")) {
                txt = 'Export done: dataset <a href="/projects/'+datasetProjectKey+'/datasets/'+datasetName+'/explore/" class="link-std">'+ datasetName + '</a>';
            } else if (lowerDescription.startsWith("apply shaker")) {
                txt = 'Export done: apply shaker on <a href="/projects/'+datasetProjectKey+'/datasets/'+datasetName+'/explore/" class="link-std">'+ datasetName + '</a>';
            } else {
                txt = sanitize("Export done: "+ message.status.inputDescription.name);
            }
            type = 'success';
        } else if (message.status.state == 'FAILED'){
            txt = sanitize("Export failed : "+ message.status.inputDescription.name);
            type = 'error';
        }
        if (txt) {
            MessengerUtils.post({
                 message: txt,
                 icon: '<i class="icon-download-alt"></i>',
                 type: type,
                 hideAfter: 5,
                 showCloseButton: true
             });
        }
    });

    /* ********* Global actions we want in every scope ********************* */

    /* Open the modal for exporting a dataset */
    $scope.exportDataset = function(projectKey, datasetName, overrideFeatures) {
        ExportUtils.exportDataset($scope, projectKey, datasetName, $stateParams.projectKey, overrideFeatures);
    };

    $scope.datasetSmartHRef = function(smartName, subState) {
        if (!smartName) return;
        if(!subState) {
            subState = 'explore';
        }
        if (smartName.indexOf(".") > 0) {
            var chunks = smartName.split(".");
            return $state.href("projects.project.datasets.dataset."+subState, {projectKey :chunks[0], datasetName : chunks[1]})
        } else {
            return $state.href("projects.project.datasets.dataset."+subState, {projectKey :$stateParams.projectKey, datasetName :smartName})
        }
    };

    /* Shortcut : put the service in the scope so we can use it directly in templates */
    $scope.WT1Event = function(type, params) {
        WT1.event(type, params);
    };

    $scope.setSpinnerPosition = function(position){
        $rootScope.spinnerPosition = position;
    };

    $scope.setTheme = function(theme) {
        if (theme) {
            var uri = $scope.getThemeUri(theme);
            var cssUri = uri + "theme.css";
            $("#theme-stylesheet").remove();
            $("head").append('<link id="theme-stylesheet" rel="stylesheet" type="text/css" href="'+cssUri+'">');
            if (!theme.isUnitedColorBg) {
                let imgUri;
                if (theme.background.startsWith("http") || theme.background.startsWith("/")) {
                    imgUri = theme.background;
                } else {
                    imgUri = uri + theme.background;
                }
                $("#root-dom-element").css("background-image","url("+imgUri+")");
            } else {
                $("#root-dom-element").css("background-image","none");
            }
        } else {
            $("#theme-stylesheet").remove();
            $("#root-dom-element").css("background-image","none");
        }
        /* Update or revert favicon */
        var faviconLink = $("head").find("link[rel='shortcut icon']");
        if (theme && theme.favicon) {
            var faviconUri = uri + theme.favicon + '?t=' + new Date().getTime();
            faviconLink.attr("href", faviconUri);
        } else {
            faviconLink.attr("href", "/favicon.ico?v=3");
        }
    };

    $scope.getThemeUri = function(theme) {
        var uri;
        switch (theme.origin) {
            case "BUILTIN":
                uri = "/themes/builtin/" + theme.id + "/";
                break;
            case "PLUGIN":
                uri = "/plugins/" + theme.pluginId + "/resource/themes/" + theme.id + "/";
                break;
            case "USER":
                uri = "/themes/user/"  + theme.id + "/";
                break;
        }
        return uri;
    };

    // refactor status color handling out into a service for easier usage in separated scopes
    $scope.getProjectStatusColor = function(status) {
        return ProjectStatusService.getProjectStatusColor(status);
    };


        $scope.codeMirrorSettingService = CodeMirrorSettingService;

    /* ******************* Top nav override ****************** */
    $scope.onEnterSecondNav = function(type){
        TopNav.setOverrideLeftType(type);
    };

    $scope.onLeaveSecondNav = function(){
        TopNav.setOverrideLeftType(null);
    };

    Logger.info("DSS loaded");

    $scope.closeNavMenu = function (triggerId) {
        let trigger = document.querySelector('#' + triggerId);
        trigger.classList.add('js-blurred');
        let blurListener = trigger.addEventListener('mouseout', function() {
            trigger.classList.remove('js-blurred');
            trigger.removeEventListener('mouseout', blurListener);
        })
    };

    /* ******************* Global Finder ****************** */
    Mousetrap(document.body).bind("mod+shift+f", e => {
        const inCodeMirror = $(e.target).closest('.CodeMirror').length > 0;
        if (inCodeMirror === false || $scope.isMacOS) {
            e.preventDefault();
            $scope.openGlobalFinder();
        }
    });

    $scope.isMacOS = DetectUtils.getOS() === "macos"; //also used in shortcuts.html
    $scope.modKey = $scope.isMacOS ? "⌘" : "Ctrl"; //also used in shortcuts.html
    $scope.altKey = $scope.isMacOS ? "⌥" : "Alt"; //also used in shortcuts.html
    $scope.globalFinderShortcut = $scope.isMacOS ? `${$scope.modKey}⇧F` : `${$scope.modKey}+Shift+F`;
    $scope.globalFinderLocalStorageKey = "global-search__last-searches";
    $scope.globalFinderTabStorageKey = "global-search__last-tabId";


    $scope.initQuery = (filter = '') => {
        $scope.globalfinder = Object.assign({}, $scope.globalfinder, { q: filter, results: [], hits: [], searchResultsAnswers: {hits: []}, searchResultsDoc: {hits: []}, searchResultsLearn: {hits: []} });
        $scope.initLastSearches();
    };

    $scope.initLastSearches = () => {
        let searches = localStorageService.get($scope.globalFinderLocalStorageKey);
        if (searches === null || Array.isArray(searches) === false) {
            searches = [];
        }
        let data = searches.map((q, index) => ({$idx: index, _type: 'search', _category: 'recent', _source: {name: q.query, query: q.query, time: q.time, tabId: q.tabId }, icon: 'global-finder-modal__search-type-icon--smaller--nomargin icon-time'}));
        $scope.initial.index = 1;
        data.unshift({_type: 'search-separator', selectable: false, _source: {name: translate('GLOBAL_SEARCH.RECENT_SEARCHES', 'Recent searches'), query: '' }});
        $scope.globalfinder.results = data;
        $scope.globalfinder.allData = [...data];
    };

    $scope.globalFinderModal = null;

    $scope.openGlobalFinder = (filter = '') => {
        if ($scope.globalFinderModal !== null) { // Forbid two modal at the same time
            return;
        }
        if (!$rootScope.appConfig.userProfile || $rootScope.appConfig.userProfile.profile === "NONE") {
            return;
        }
        const filterPattern = '(\\w+:(".+"|[^ ]+))';
        $scope.initial = { index: 0 };
        $scope.initQuery(filter);

        $scope.globalFinderModal = CreateModalFromTemplate("/templates/global-finder-modal.html", $scope, null, newScope => {

            const projectNames = {};
            DataikuAPI.projects.list(true)
                .success(function (response) {
                    angular.forEach(response, function (project) {
                        projectNames[project.projectKey] = project.name;
                    })
                })
                .error(setErrorInScope.bind($scope));

            const users = {};
            DataikuAPI.security.listUsers()
                .success(function (response) {
                    angular.forEach(response, function (user) {
                        users[user.login] = user.displayName;
                    })
                })
                .error(setErrorInScope.bind($scope));

            newScope.inProject = !!newScope.$stateParams.projectKey;

            newScope.getHelp = function () {
                newScope.dismiss();
                $scope.showHelp();
            };

            newScope.helpIntegrationEnabled = () => newScope.wl.contextualHelpSearchEnabled && newScope.appConfig.helpIntegrationEnabled;

            newScope.updateLastSearches = (newSearch) => {
                if (newSearch === '') {
                    return;
                }

                let searches = localStorageService.get($scope.globalFinderLocalStorageKey);
                if (searches === null || Array.isArray(searches) === false) {
                    searches = [];
                }
                let index = searches.map(s => s.query.toLowerCase()).indexOf(newSearch.toLowerCase());
                if (index !== -1) {
                    searches.splice(index, 1);
                }
                let savedSearch = {
                    query: newSearch,
                    time: Date.now()
                };
                searches.unshift(savedSearch);

                if (searches.length > 5) {
                    searches.splice(5);
                }
                localStorageService.set($scope.globalFinderLocalStorageKey, searches)
            };

            newScope.onSmartInputChange = () => {
                newScope.initial.index = 0;
                newScope.triggerSearch();
            };

            newScope.triggerSearch = function (shouldHandleLoading = true) {
                if (shouldHandleLoading) {
                    newScope.globalfinder.searching = true;
                    newScope.globalfinder.results = [];
                    newScope.globalfinder.allData = [];
                }
                newScope.globalfinder.trimmedQuery = newScope.trimQuery(newScope.globalfinder.q);
                if (newScope.globalfinder.trimmedQuery === '') {
                    newScope.globalfinder.searching = false;
                }
                newScope.debouncedTriggerSearch();
            };

            newScope.debouncedTriggerSearch = Debounce().withDelay(200, 200).wrap(function () {
                let query = newScope.globalfinder.q;
                if (query === "") {
                    $scope.initQuery('');
                    return;
                }

                if (newScope.helpIntegrationEnabled()) {
                    newScope.globalfinder.search(query);
                }

                DataikuAPI.globalfinder.search(query, 15, newScope.$stateParams.projectKey)
                    .success(data => {
                        newScope.globalfinder.hits = data.hits.map(hit => Object.assign({url: newScope.getLink(hit)}, hit));
                        newScope.globalfinder.aggregations = data.aggregations;
                        newScope.buildResult();
                    })
                    .error(() => {
                        newScope.globalfinder.hits = [];
                        newScope.buildResult();
                    });

                newScope.focusSearchInput();
            });

            newScope.trimQuery = query => {
                return query.replace(new RegExp(filterPattern, 'g'), '').trim().replace(/\s{2,}/g, ' ').trim();
            };

            newScope.emptySearch = () => {
                newScope.globalfinder.q = "";
                newScope.triggerSearch();
                newScope.focusSearchInput();
            };

            newScope.focusSearchInput = () => {
                const searchInputs = document.getElementsByClassName("global-finder-modal__search-input");
                if (searchInputs && searchInputs.length > 0) {
                    searchInputs[0].focus();
                }
            };

            newScope.hasFilter = filter => newScope.globalfinder.q.includes(filter);

            newScope.formatItemName = item => {
                if (!item) {
                    return '';
                }
                if (item._type === 'discussion') {
                    return item._source.discussions && item._source.discussions.length && item._source.discussions[0].topic ? item._source.discussions[0].topic : "Unnamed discussion";
                }
                if (item._type === 'chart') {
                    return item._source.name || "Untitled chart"; // TODO : change done near release. We should factorize the untitled behavior to all types
                }
                return item._source.name;
            };

            newScope.formatItemPath = item => {
                if (!item) {
                    return '';
                }
                if (newScope.isNavigation(item)) {
                    return item._source.path.replace(new RegExp(" > " + item._source.name + "$","g"), "");
                }
                if (item._type === 'discussion') {
                    return item._source.projectName + " > " + item._source.objectName;
                }
                if (item._type === 'project') {
                    return item._source.shortDesc;
                }
                return item._source.projectName;
            };

            newScope.formatItemHelpType = item => {
                if (item && item._type) {
                    return $filter('capitalize')(item._type.replace(/_/g, " "));
                }
                return "Help";
            };


            newScope.chunkSize = () => 3;
            if (newScope.globalfinder === undefined){
                newScope.globalfinder = {};
            }
            newScope.globalfinder.searchResults = {hits: []};

            newScope.globalfinder.search = (query = newScope.globalfinder.q) => {
                const trimmedQuery = newScope.trimQuery(query);
                // Google search does not accept empty query
                if (!trimmedQuery) {
                    return;
                }
                return DataikuAPI.help.search({query: trimmedQuery,pageSize: 3})
                    .success(newScope.onResult("searchResults"))
                    .error(onError(newScope.globalfinder, "searchResults", {hits: []}));
            }

            const onError = (globalfinder, fieldName, defaultValue = []) => () => {
                globalfinder[fieldName] = defaultValue;
                newScope.buildResult();
            };

            newScope.onResult = fieldName => content => {
                newScope.globalfinder[fieldName] = Object.assign({hits: (content.results ? content.results : []).map(result => Object.assign({title:result.document.derivedStructData.title, url: result.document.derivedStructData.link, isHelp: true, isExternalUrl: true, _type: newScope.getItemType(result), _id:result.id}))});
                newScope.buildResult();
            };

            newScope.getItemType = result => {
                                const type = result && result.document && result.document.derivedStructData && result.document.derivedStructData.pagemap && result.document.derivedStructData.pagemap.metatags ? result.document.derivedStructData.pagemap.metatags.map(metatag => metatag["og:site_name"]).find(site => site) : undefined;
                if (type === undefined && result.document.derivedStructData.displayLink) {
                    return  result.document.derivedStructData.displayLink.match("^[^.]*")[0];
                }
                return type;
            };

            const nameMatch = (hit, query) => {
                if (!hit) {
                    return false;
                }
                if (hit._source.name) {
                    return hit._source.name.toLowerCase() === query.toLowerCase();
                }
                else if (hit._source.objectName) {
                    return hit._source.objectName.toLowerCase() === query.toLowerCase();
                }
                return false;
            };

            newScope.mergeResult = () => {
                const helpData = newScope.helpIntegrationEnabled() ? newScope.globalfinder.searchResults.hits : [];
                const initialData = {_type: 'search', _source: { name: `Search ${newScope.globalfinder.q}`, query:newScope.globalfinder.q }, icon: "global-finder-modal__search-type-icon icon-dku-search"};
                const hits = newScope.globalfinder.hits.map(hit => {
                    if($rootScope.appConfig.governURL) {
                        hit.isExternalUrl = hit.url.includes(':governURL:');
                        if (hit.isExternalUrl) {
                            hit.url = hit.url.replace(':governURL:', $rootScope.appConfig.governURL);
                        }
                    }
                    if (hit.url && newScope.$stateParams.projectKey) {
                        hit.url = hit.url.replace(':projectKey:', newScope.$stateParams.projectKey);
                    }
                    return hit;
                });

                const firstResults = hits.length > 0 && nameMatch(hits[0], newScope.globalfinder.trimmedQuery)
                    ? [hits[0], initialData]
                    : [initialData, hits[0]];

                newScope.globalfinder.results = firstResults.concat(hits.slice(1)).concat(helpData).filter(el => el);
                newScope.globalfinder.results.forEach((val, index) => val.$idx = index);
                newScope.globalfinder.allData = [...newScope.globalfinder.results];
                newScope.globalfinder.searching = false;
            };

            newScope.buildResult = Debounce().withDelay(100, 100).wrap(newScope.mergeResult);

            newScope.clickItem = item => {
                if (item === null || item === undefined) {
                    return;
                }
                WT1.event("global-finder-item-open", {
                    type: item._type,
                    id: item._id || item.objectID,
                    currentTab: '', filters: [] // used to contain more specific info when we had the advanced modal. kept for consistency of the WT1 data even if not really useful anymore.
                });
                if (item._type === 'search') {
                    newScope.updateLastSearches(newScope.globalfinder.q);
                    $state.go('catalog.items', {hash: CatalogUtils.getHash(item._source.query, {})});
                    return;
                }
                if (newScope.simulateClick(item)) {
                    newScope.focusSearchInput(); // reset activeElement after a click
                }
            };

            newScope.openItem = item => {
                if (item && item.url) {
                    const aElem = document.querySelector(`.global-finder-modal__line a[href="${item.url}"]`);
                    if (aElem) {
                        $timeout(() => {
                            aElem.focus();
                            aElem.click();
                        });
                    }
                } else {
                    newScope.clickItem(item);
                }
            };

            newScope.simulateClick = item => {
                const itemURL = new URL(item.isExternalUrl ? item.url : `${window.location.protocol}//${window.location.host}${item.url}`);
                // The redirect behavior is different if the user is already on the desired page
                if (window.location.pathname === itemURL.pathname) {
                    newScope.dismiss();
                    if (itemURL.hash) {
                        window.location.hash = itemURL.hash;
                        document.getElementById(itemURL.hash.substr(1)).scrollIntoView();
                    }
                    return false;
                }
                return true;
            };

            newScope.selectItem = item => {
                if (item.selectable === false || item.$idx === newScope.selected.index) {
                    return;
                }
                newScope.selectIndex(item.$idx);
            };
        });

        $scope.globalFinderModal.catch(() => {
            $scope.globalFinderModal = null
        });
    };

    $scope.isItemSelectable = item => item.selectable === undefined || item.selectable === true;
    $scope.isNavigation = item => item && item._type === 'page';

    $scope.itemToIcon = (item, inList, size) => {
        if (!item) {
            return;
        }
        if (item.isHelp) {
            return 'icon-dku-help';
        }
        if (item._type === 'page') {
            return 'icon-list'
        }
        return CatalogItemService.itemToIcon(item._type, item._source, inList, size);
    };

    $scope.itemToColor = item => {
        if (!item) {
            return;
        }
        if (item._type === 'page') {
            return 'navigation';
        }
        if (item.isHelp) {
            return 'home';
        }
        return CatalogItemService.itemToColor(item._type, item._source);
    };

    $scope.disableItemIcon = item => {
        if (item && item._source && item._source.closed) {
            return 'global-finder-modal__search-type-icon--disabled';
        }
    };

    $scope.getLink = (item, discussionId) => {
        if (!item || item.isHelp) {
            return;
        }
        if (item._type === 'page') {
            return item._source.url;
        }
        return CatalogItemService.getLink(item._type, item._source, discussionId);
    };
});


app.controller('RequestEETrialController', function ($scope, $state, Assert, DataikuAPI, DataikuCloudAPI, $rootScope) {
    Assert.inScope($scope, 'appConfig');
    Assert.trueish($scope.appConfig.licensing.community, 'not a free edition');
    $scope.request = {
        state : "initial"
    };

    $scope.request.updatedEmailAddress = $scope.appConfig.licensing.ceRegistrationEmail;

    $scope.sendRequest = function() {
        DataikuCloudAPI.community.requestEETrial(
                $scope.appConfig.licensing.ceInstanceId,
                $scope.request.updatedEmailAddress).success(function(data){

            $scope.trialRequestResponse = data;
            if (data.granted) {
                $scope.request.state = "granted";
            } else {
                $scope.request.state = "denied";
            }
        }).error(setErrorInScope.bind($scope));
    };
});


app.controller('RegisterController', function ($scope, $state, Assert, DataikuAPI, DataikuCloudAPI, $rootScope) {
    Assert.inScope($scope, 'appConfig');

    $scope.register = {
        state: 'welcome',
        wantEETrial: false,
        step: '1'
    };
    $scope.newAccount = {
        newsletter: true
    };
    $scope.existingAccount = {};
    $scope.existingKey = {};

    function fetchWebConfig() {
        DataikuCloudAPI.getWebConfig().then(function(id) {
            $scope.webVisitorId = id;
        });
         DataikuCloudAPI.getNewWebConfig().then(function(data) {
            $scope.webVisitorLocalId = data.visitor_id;
            $scope.webVisitorHSId = data.hs_id;
        });
    }
    fetchWebConfig();

    $scope.logMeIn = function() {
        window.location = '/';
    };

    $scope.switchStep = function(state, step) {
        $scope.register.state = state;
        $scope.register.step = step;
    }

    $scope.$watch("register.mode", function(nv, ov) {
        $scope.fatalAPIError = null;
    });

    function setCEThanks(data) {
        if (!data.trialRequestResponse) {
            $scope.register.state = "thanks-ce";
        } else if (data.trialRequestResponse.granted) {
            $scope.register.state = "thanks-ee-trial-granted";
        } else if (!data.trialRequestResponse.granted) {
            $scope.register.state = "thanks-ee-trial-denied";
        }
    }

    $scope.registerNewAccount = function() {
        Assert.trueish($rootScope.appConfig.saasManagerURL, 'Not a saas instance');

        DataikuCloudAPI.community.register(
            $scope.newAccount.firstName, $scope.newAccount.lastName,
            $scope.newAccount.company, $scope.newAccount.persona,
            $scope.newAccount.userEmail,
            $scope.newAccount.newsletter,
            $scope.register.wantEETrial,
            $rootScope.appConfig.version.product_version,
            $scope.webVisitorId, $scope.webVisitorLocalId, $scope.webVisitorHSId,
            $rootScope.appConfig.registrationChannel
        ).success(function(data) {
            /* Write the received license */
            DataikuAPI.registration.initialRegisterCommunity(
                $scope.newAccount.firstName, $scope.newAccount.lastName,
                $scope.newAccount.userEmail,
                data.instanceId, data.license).success(function(data2) {

                $scope.register.registrationResult = data;
                $scope.register.loginInfo = data2;
                if (data.trialRequestResponse && data.trialRequestResponse.granted) {
                    $scope.switchStep('enter-trial-license', 3);
                } else {
                    setCEThanks($scope.register.registrationResult);
                }
            }).error(setErrorInScope.bind($scope));
        }).error(setErrorInScope.bind($scope));
    };

    $scope.registerNoAccount = function () {
        Assert.trueish($rootScope.appConfig.saasManagerURL, 'not a saas instance');

        var firstName = "Unknown";
        var lastName = "Unknown";
        var company = "Unknown";
        var ts = new Date().getTime();
        var userEmail = $scope.webVisitorId + "-" + ts +  "@unknownvisitor.no";
        var password = "Unknown";

        DataikuCloudAPI.community.registerNewAccount(
            firstName, lastName,
            company, userEmail,
            password, 0,
            false,
            $rootScope.appConfig.version.product_version,
            $scope.webVisitorId, $scope.webVisitorLocalId, $scope.webVisitorHSId,
            $rootScope.appConfig.registrationChannel
        ).success(function(data) {
                /* Write the received license */
                DataikuAPI.registration.initialRegisterCommunity(
                    $scope.newAccount.firstName, $scope.newAccount.lastName,
                    $scope.newAccount.userEmail,
                    data.instanceId, data.license).success(function(data2) {

                        $scope.register.registrationResult = data;
                        $scope.register.loginInfo = data2;
                        setCEThanks(data);

                    }).error(setErrorInScope.bind($scope));
            }).error(setErrorInScope.bind($scope));
    };

    $scope.registerExistingAccount = function() {
        Assert.trueish($rootScope.appConfig.saasManagerURL, 'not a saas instance');

        DataikuCloudAPI.community.registerExistingAccount(
            $scope.existingAccount.userEmail,
            $scope.existingAccount.password,
            $scope.register.wantEETrial,
            $rootScope.appConfig.version.product_version,
            $scope.webVisitorId, $scope.webVisitorLocalId, $scope.webVisitorHSId,
            $rootScope.appConfig.registrationChannel
        ).success(function(data) {
            DataikuAPI.registration.initialRegisterCommunity(
                data.firstName, data.lastName,
                $scope.existingAccount.userEmail,
                data.instanceId, data.license).success(function(data2) {

                $scope.register.registrationResult = data;
                $scope.register.loginInfo = data2;
                setCEThanks(data);

            }).error(setErrorInScope.bind($scope));
        }).error(setErrorInScope.bind($scope));
    };

    $scope.setLicense = function() {
        DataikuAPI.registration.initialRegisterLicensed($scope.existingKey.license).success(function(data) {
            $scope.register.loginInfo = data;
            $scope.register.state = "thanks-license";
        }).error(setErrorInScope.bind($scope));
    };
});


app.controller('RenewLicenseController', function($scope, $state, Assert, DataikuAPI, DataikuCloudAPI, $rootScope) {
    Assert.inScope($scope, 'appConfig');
    $scope.existingKey = {}

    $scope.logMeIn = function(){
        window.location = '/';
    };

    $scope.setLicense = function() {
        DataikuAPI.registration.renewExpiredLicense($scope.existingKey.license).success(function(data) {
            $scope.registrationSuccessful = {}
        }).error(setErrorInScope.bind($scope));
    };
});

app.controller("FeedbackController", function($scope, WT1, OpalsService){
    $scope.feedbackContent = {
        comment: '',
        email: $scope.appConfig.user && $scope.appConfig.user.email ? $scope.appConfig.user.email : ''
    }
    $scope.finished = false;

    $scope.sendFeedback = function() {
        $scope.finished = true;
        WT1.event("dss-feedback", $scope.feedbackContent);
    }

    OpalsService.isEnabled().then(function(isOpalsEnabled) {
        $scope.isOpalsEnabled = isOpalsEnabled;
    });
});

app.controller("NoneUserHomeController", function($rootScope, $scope, $state, DataikuAPI, RequestCenterService, CreateModalFromTemplate, WT1){

    $scope.message = "";

    $scope.startTrialFromNoneUser = function() {
        WT1.event("none-user-trial-start");
        DataikuAPI.security.startTrialFromNoneUser().success((data) => {
            WT1.event("none-user-trial-start-success");
            // Violent redirect to avoid keeping a cached appConfig
            window.location = "/";
        }).error(function(a,b,c) {
            setErrorInScope.bind($scope)(a,b,c);
            WT1.event("none-user-trial-start-failed");
        });
    }

    $scope.requestLicenseFromNoneUser = (message) => {
        DataikuAPI.requests.createInstanceAccessRequest(message).success((data) => {
            RequestCenterService.WT1Events.onRequestSent("INSTANCE", null, null, message, data.id);
            $scope.state = "pending-request";
        }).error(setErrorInScope.bind($scope));
    };

    $scope.logoutNoneUser = () => {
        DataikuAPI.logout().success(function(data) {
            // Violent redirect to avoid keeping a cached appConfig
            window.location = "/";
        }).error(setErrorInScope.bind($scope));
    };

    const fetchLatestRequest = () => {
        DataikuAPI.requests.getLatestRequestForCurrentUser("", "INSTANCE", "")
        .then(response => {
            $scope.hasPreviousRequest = response.data.status === "PENDING";
        }, error => {
            $scope.hasPreviousRequest = false;
        }).finally(() => {
            if ($scope.hasPreviousRequest) {
                $scope.state = "pending-request";
            } else if ($scope.appConfig.noneUsersCallToActionBehavior === 'ALLOW_REQUEST_ACCESS') {
                $scope.state = "request-access";
            } else if ($scope.appConfig.noneUsersCallToActionBehavior === 'ALLOW_START_TRIAL' && $scope.appConfig.allowRequestAccessWithStartedTrial && hasTrialToken && $scope.appConfig.trialStatus.expired) {
                $scope.state = "request-access";
            }
        });
    };

    const hasTrialToken = $scope.appConfig.trialStatus && $scope.appConfig.trialStatus.exists;
    if (hasTrialToken && $scope.appConfig.trialStatus.illegal) {
        $scope.state = "illegal-token";
    } else if (!hasTrialToken && $scope.appConfig.noneUsersCallToActionBehavior === 'ALLOW_START_TRIAL') {
        $scope.state = "start-trial";
    } else {
        $scope.state = "message-only";
        fetchLatestRequest();
    }
});

app.controller('LoginController', function($scope, $state, $location, DataikuAPI, TopNav, LoggerProvider, LocalStorage) {

    const Logger = LoggerProvider.getLogger('LoginController');
    TopNav.setLocation(TopNav.LOGIN, "login");
    $scope.ssoEnabled = false;

    DataikuAPI.getConfiguration().success(function(data) {
        $scope.ssoEnabled = data.ssoLoginEnabled;
    });

    var lic = $scope.appConfig.licensing;
    $scope.communityLook = lic.community && !lic.ceEntrepriseTrial;
    if (lic.ceEntrepriseTrialUntil > Date.now()) {
        $scope.daysLeft = Math.floor((lic.ceEntrepriseTrialUntil - Date.now()) / (24 * 3600 * 1000));
    }

    $scope.submit = function() {
        var formLogin = $("input[name=login]").val(),
            formPassword = $("input[name=password]").val();

        $scope.loginFailed = false;
        $scope.loginErrorMessage = '';
        DataikuAPI.login(formLogin, formPassword).success(function(data) {
            const redirectTo = $state.params.redirectTo;
            const search = $state.params.search || "";
            if (redirectTo) {
                Logger.info("GO " + redirectTo);
                // ui-router does not seem to manage changes of $location.url ... It just does not do anything
                // And since I have a URL, I can't use transitionTo
                // SO I have to reload. It sucks
                const url = new URL(window.location.href);
                url.search = search;
                url.pathname = redirectTo;  // Only follow redirects to a local path, not to another site
                window.location = url.href;
            } else {
                // I also do it here to ensure that we reload appConfig
                window.location = "/";
            }
            LocalStorage.set("dss.hasUsedDashboardFilters", false);
        }).error(function(data, status, headers) {
            $scope.loginFailed = true;
            if (data.errorType) {
                $scope.loginError = getErrorDetails(data, status, headers)
            } else {
                $scope.loginErrorMessage = data;
            }
        });
    };

    if ($scope.appConfig.loggedIn && !$scope.appConfig.noLoginMode) {
        // it's confusing to leave people on a blank login screen when they are actually logged in
        $scope.redirectHome();
    }

    $scope.redirectToSSO = function() {
        window.location = "/";
    }
});

app.controller('SSOErrorController', function($scope, $state, TopNav) {

    TopNav.setLocation(TopNav.LOGIN, "login");

    $scope.error = $state.params.error;
    $scope.errorDescription = $state.params.errorDescription;

    if ($scope.appConfig.loggedIn && !$scope.appConfig.noLoginMode) {
        // it's confusing to leave people on a blank login screen when they are actually logged in
        $scope.redirectHome();
    }
});

// You MUST call setCurrentProjectFolderId with the appropriate folderId in the link method of CreateModal...
app.controller('NewLearningProjectController', function($scope, $rootScope, Assert, DataikuAPI, $state, WT1, CreateModalFromTemplate, ListFilter, ProjectFolderContext, ProjectFolderService, PathUtils, PromiseService, translate) {

    $scope.facets = [
        { "id": "type", "label": "Type", values: [], $open: true, $allChecked: true }
    ];
    $scope.query = { q:"" };
    $scope.filteredProjects = [];
    $scope.selectedProject = null;

    $scope.targetProjectFolder = null; // used in the project folder service
    $scope.defaultProjectFolderId = null;

    $scope.setCurrentProjectFolderId = (folderId) => {
        ProjectFolderService.getDefaultFolderForNewProject(folderId).then((folder) => {
            $scope.targetProjectFolder = folder;
            $scope.defaultProjectFolderId = folder.id;
        }).catch(setErrorInScope.bind($scope));
    }

    $scope.browse = folderIds => {
        // Use last id in path
        $scope.destination = PathUtils.makeNLNT(folderIds).split('/').pop();
        // browse-path expects a success-error promise so we need to wrap with qToHttp for now (catch() does not return a monkey-patched promise)
        return PromiseService.qToHttp(ProjectFolderService.getBrowseNode($scope.destination).catch(setErrorInScope.bind($scope)));
    };

    $scope.getProjectFolderName = item => item.name;
    $scope.canSelectFolder = item => item.canWriteContents;

    function getFacet(facetId) {
        return $scope.facets.find(facet => facet["id"] === facetId);
    }

    function populateSections() {
        if (!$scope.tutorialsList) return;

        $scope.tutorialsList.items.forEach(function(project) {
            if (project.archiveType !== 'FETCH' ) {
                // get the image from the backend for builtin tutorials. Remote tutorials have
                // to provide the image themselves, so we can use their imageURL directly
                project.imageURL = '/dip/api/image/get-tutorial-thumbnail?tutorialId=' + project.id;
            }
            addSectionsToFacet(project.sectionName, "type");
        });
    }

    function addSectionsToFacet(sectionName, facetId) {
        let values = getFacet(facetId).values;
        if (!values.some(value => value.label === sectionName)) {
            values.push({ label: sectionName, $checked: false });
        }
    }

    DataikuAPI.projects.listTutorials().success(function(data) {
        $scope.tutorialsList = data;
        $scope.tutorialsList.items = $scope.tutorialsList.items
            .filter(d => d.type === 'SAMPLE' || d.type === 'TUTORIAL')
            .map(d => {
                // patch tutorials
                if (d.type === 'SAMPLE') {
                    d.sectionName = 'Sample';
                } else if (d.sectionName === "Core Designer / Basics") {
                    // Fix the section name so that old tutorials are displayed in the new category
                    d.sectionName = 'Core Designer';
                }
                return d;
            });

        $scope.filteredProjects = $scope.tutorialsList.items;

        populateSections();
    }).error(setErrorInScope.bind($scope));

    $scope.$watch("query", () => {
        updateFilteredProjects();
    }, true);

    function matchesFacet(facetName, sectionName) {
        const facet = getFacet(facetName);
        if (facet.$allChecked) {
            return true;
        }
        const selectedValues = facet.values
            .filter(facetItem => facetItem.$checked)
            .map(facetItem => facetItem.label);
        return selectedValues.includes(sectionName);

    }

    function updateFilteredProjects() {
        if (!$scope.tutorialsList || !$scope.tutorialsList.items) return;

        $scope.filteredProjects = ListFilter
            .filter($scope.tutorialsList.items || [], $scope.query.q)
            .filter(tutorial => matchesFacet("type", tutorial.sectionName)
            );
    }

    $scope.setSelectedProject = function(project) {
        return $scope.selectedProject = project;
    }

    $scope.onFacetItemClicked = function(facet) {
        facet.$allChecked = facet.values.every(facetItem => !facetItem.$checked);
        updateFilteredProjects();
    }

    $scope.onFacetAllItemClicked = function(facet) {
        if (facet.$allChecked) {
            facet.values.forEach(facetItem => facetItem.$checked = false);
        }
        updateFilteredProjects();
    }

    $scope.start = function(project, targetProjectFolder, defaultProjectFolderId) {
        Assert.trueish(project.id, 'No tutorial id');

        // note to self: pass the parent scope of this modal's scope as the download modal's scope's parent, since
        // we're going to dismiss this modal right now (and if the download modal was using this scope, then it would be
        // created non functional...)
        CreateModalFromTemplate("/templates/projects/tutorial-download.html", $scope.attachDownloadTo, null, function(newScope) {
            newScope.tutorialIdToInstall = project.id;
            newScope.tutorialType = project.type;
            newScope.tutorialDescription = project.description;
            newScope.tutorialName = project.name;
            newScope.targetProjectFolder = targetProjectFolder;
            newScope.defaultProjectFolderId = defaultProjectFolderId;
        });

        $scope.dismiss();
    };

    $scope.installButtonTooltip = function(selectedProject, targetProjectFolder) {
        if (selectedProject === null) {
            return translate("HOME.PROJECTS.NEW.TUTORIALS.HELP", "Select a project to install");
        }

        if (!$scope.mayCreateProjectsFromTemplates()) {
            return translate("HOME.PROJECTS.NEW.TUTORIALS.NO_PERM.HELP", "You do not have permission to create projects using templates");
        }

        if (!targetProjectFolder.canWriteContents) {
            return translate("HOME.NEW_PROJECT.FOLDER.CANNOT_WRITE.HELP", "You cannot write to this folder");
        }

        return "";
    };

    $scope.mayCreateProjectsFromTemplates = function() {
        return $rootScope.appConfig && $rootScope.appConfig.globalPermissions.mayCreateProjectsFromTemplates;
    };

});

// You MUST call setCurrentProjectFolderId with the appropriate folderId in the link method of CreateModal...
app.controller('NewSolutionProjectController', function($sce, $scope, $rootScope, Assert, DataikuAPI, $state, WT1, CreateModalFromTemplate, ListFilter, ProjectFolderContext, ProjectFolderService, PathUtils, PromiseService, translate, $window) {

    $scope.facets = [
        { "id": "industry", "label": "Industry", values: [], $open: true, $allChecked: true },
        { "id": "lineOfBusiness", "label": "Line of Business", values: [], $open: true, $allChecked: true },
        { "id": "type", "label": "Type", values: [], $open: true, $allChecked: true }
    ];
    $scope.query = { q:"" };
    $scope.filteredSolutions = [];
    $scope.selectedSolution = null;

    $scope.targetProjectFolder = null; // used in the project folder service
    $scope.defaultProjectFolderId = null;

    $scope.browse = folderIds => {
        // Use last id in path
        $scope.destination = PathUtils.makeNLNT(folderIds).split('/').pop();
        // browse-path expects a success-error promise so we need to wrap with qToHttp for now (catch() does not return a monkey-patched promise)
        return PromiseService.qToHttp(ProjectFolderService.getBrowseNode($scope.destination).catch(setErrorInScope.bind($scope)));
    };

    $scope.getProjectFolderName = item => item.name;
    $scope.canSelectFolder = item => item.canWriteContents;

    function getFacet(facetId) {
        return $scope.facets.find(facet => facet["id"] === facetId);
    }

    function addSections(solution) {
        addSectionsToFacet(solution.solutionCategories, "industry");
        addSectionsToFacet(solution.solutionLineOfBusiness, "lineOfBusiness");
        addSectionsToFacet(solution.solutionType, "type");
    }

    function addSectionsToFacet(items, facetId) {
        if (items) {
            items.forEach((sectionName) => {
                let values = getFacet(facetId).values;
                if (!values.some(value => value.label === sectionName)) {
                    values.push({ label: sectionName, $checked: false });
                }
            });
        }
    }

    function matchesFacet(facetName, facetValues) {
        const facet = getFacet(facetName);
        if (facet.$allChecked) {
            return true;
        }
        const selectedValues = facet.values
            .filter(facetItem => facetItem.$checked)
            .map(facetItem => facetItem.label);
        return Array.isArray(facetValues) && facetValues.some(item => selectedValues.includes(item));

    }

    function updateFilteredSolutions() {
        $scope.filteredSolutions = ListFilter
            .filter($scope.solutions || [], $scope.query.q)
            .filter(solution => matchesFacet("industry", solution.solutionCategories)
                && matchesFacet("lineOfBusiness", solution.solutionLineOfBusiness)
                && matchesFacet("type", solution.solutionType)
            );
    }

    $scope.setSelectedSolution = function(solution) {
        WT1.tryEvent("solution-browsed", () => {
            return {
                tutorialId: solution.id
            };
        })
        return $scope.selectedSolution = solution;
    }

    $scope.onFacetItemClicked = function(facet) {
        facet.$allChecked = facet.values.every(facetItem => !facetItem.$checked);
        updateFilteredSolutions();
    }

    $scope.onFacetAllItemClicked = function(facet) {
        if (facet.$allChecked) {
            facet.values.forEach(facetItem => facetItem.$checked = false);
        }
        updateFilteredSolutions();
    }

    $scope.$watch("query", () => {
        updateFilteredSolutions();
    }, true);

    $scope.openDocumentationUrl = function(solution) {
        WT1.tryEvent("solution-documentation-opened", () => {
             return {
                 tutorialId: solution.id
             };
        });
        $window.open(solution.documentationURL, '_blank');
    };

    $scope.start = function(solution, targetProjectFolder, defaultProjectFolderId) {
        Assert.trueish(solution.id, 'No tutorial id');

        WT1.tryEvent("tutorial-project-creating", () => {
            return {
                tutorialId: solution.id
            };
        });

        // note to self: pass the parent scope of this modal's scope as the download modal's scope's parent, since
        // we're going to dismiss this modal right now (and if the download modal was using this scope, then it would be
        // created non functional...)
        CreateModalFromTemplate("/templates/projects/solution-download.html", $scope.attachDownloadTo, null, function(newScope) {
            newScope.tutorialIdToInstall = solution.id;
            newScope.tutorialType = "INDUSTRY_SOLUTION";
            newScope.documentationURL = solution.documentationURL;
            newScope.tutorialName = solution.name;
            newScope.targetProjectFolder = targetProjectFolder;
            newScope.defaultProjectFolderId = defaultProjectFolderId;
        });

        $scope.dismiss();
    };

    DataikuAPI.projects.listTutorials().success(function(data){
        $scope.tutorialsList = data;

        if ($scope.tutorialsList && $scope.tutorialsList.items) {
            $scope.solutions = $scope.tutorialsList.items.filter(tutorial => tutorial.type === 'INDUSTRY_SOLUTION');
            $scope.solutions.forEach(function(solution) {
                if (solution.archiveType !== 'FETCH') {
                    // get the image from the backend for builtin tutorials. Remote tutorials have
                    // to provide the image themselves, so we can use their imageURL directly
                    solution.imageURL = '/dip/api/image/get-tutorial-thumbnail?tutorialId=' + solution.id;
                }

                // Check conditions to show the flags
                solution.showFlag = solution.flagRelease === "New Solution" || solution.flagRelease === "Upgrade Available";
                if (solution.flagMaxDate) {
                    solution.isBeforeMaxDate = new Date() < new Date(solution.flagMaxDate);
                }
                $scope.getLabel = function(project) {
                    switch (project.flagRelease) {
                        case "New Solution":
                            return "New";
                        case "Upgrade Available":
                            return "Updated";
                        default:
                            return ""; // Default label if neither condition is met
                    }
                };

                solution.previewURL = $sce.trustAsResourceUrl(solution.previewURL);

                addSections(solution);
            });
        } else {
            $scope.solutions = [];
        }
        $scope.filteredSolutions = $scope.solutions;
    }).error(setErrorInScope.bind($scope));

    $scope.installButtonTooltip = function(selectedSolution, targetProjectFolder) {
        if (selectedSolution === null) {
            return translate("HOME.NEW_SOLUTION.INSTALL_BUTTON.NONE_SELECTED.HELP", "Select a Dataiku Solution to install");
        }

        if (!$scope.mayCreateProjectsFromTemplates()) {
            return translate("HOME.NEW_SOLUTION.INSTALL_BUTTON.NO_PERM.HELP", "You do not have permission to create projects using templates");
        }

        if (!targetProjectFolder.canWriteContents) {
            return translate("HOME.NEW_PROJECT.FOLDER.CANNOT_WRITE.HELP", "You cannot write to this folder");
        }

        return "";
    };

    $scope.mayCreateProjectsFromTemplates = function() {
        return $rootScope.appConfig && $rootScope.appConfig.globalPermissions.mayCreateProjectsFromTemplates;
    };

    $scope.setCurrentProjectFolderId = (folderId) => {
        if ($scope.mayCreateProjectsFromTemplates()) {
            ProjectFolderService.getDefaultFolderForNewProject(folderId).then((folder) => {
                $scope.targetProjectFolder = folder;
                $scope.defaultProjectFolderId = folder.id;
            }).catch(setErrorInScope.bind($scope));
        }
    }

});

app.controller('TutorialDownloadController', function($scope, $rootScope, QuestionnaireService, PageSpecificTourService, DataikuAPI, MonoFuture, Fn, WT1, $state, ProjectFolderService, OpalsService) {
    $scope.state = "NOT_STARTED";

    function go(){
        MonoFuture($scope).wrap(DataikuAPI.projects.createTutorial)($scope.tutorialIdToInstall, $scope.tutorialType, $scope.targetProjectFolder.id).success(function(data) {
            $scope.state = data.result.success ? "DONE" : "FAILED";
            $scope.stateShown = null;
            if (!data.result.success) {
                $scope.failure = {
                    aborted : data.aborted,
                    message : data.result.installationError.detailedMessage
                }
                WT1.tryEvent("tutorial-project-creation-failed", () => {
                    return {
                        tutorialId : $scope.tutorialIdToInstall
                    };
                });
            } else {
                $scope.needsGoingToTutorial = true;
                $scope.projectKey = data.result.projectKey;
                WT1.tryEvent("tutorial-project-created", () => {
                    return {
                        tutorialId : $scope.tutorialIdToInstall,
                        isInSuggestedFolder: $scope.targetProjectFolder.id === $scope.defaultProjectFolderId,
                        isInRoot: ProjectFolderService.isInRoot($scope.targetProjectFolder.id),
                        isUnderSandbox: ProjectFolderService.isUnderSandbox($scope.targetProjectFolder)
                    };
                });
            }

            $scope.installingFuture = null;
        }).update(function(data) {
            $scope.stateShown = data.progress != null && data.progress.states != null && data.progress.states.length > 0 ?
                                                                    data.progress.states[data.progress.states.length - 1] : null;
            $scope.state = "RUNNING";
            $scope.installingFuture = data;
        }).error(function (data, status, headers) {
            $scope.state = "FAILED";
            if ( data.aborted) {
                $scope.failure = {
                    aborted : data.aborted,
                    message : "Aborted"
                }
            } else if (data.hasResult) {
                $scope.failure = {
                    aborted : data.aborted,
                    message : data.result.errorMessage
                }
            } else {
                $scope.failure = {
                    aborted : data.aborted,
                    message : "Unexpected error"
                }
            }
            $scope.installingFuture = null;
        });
    }

    $scope.abort = function() {
        $scope.state = "FAILED";
        $scope.failure = {
            message : "Aborted"
        }
        DataikuAPI.futures.abort($scope.installingFuture.jobId);
    };

    $scope.closeAndGo = function(){
        $scope.dismiss();
        $state.go("projects.project.home.regular", {projectKey : $scope.projectKey});
    }

    $scope.$on("$destroy", function(){
        if ($scope.state == "RUNNING") {
            $scope.abort();
        }
    });

    /* Wait for the plugin id to start */
    $scope.$watch("tutorialIdToInstall", Fn.doIfNv(go));
});

app.controller('BusinessSolutionDownloadController', function($scope, DataikuAPI, MonoFuture, Fn, WT1, $state, $window, StateUtils, Assert, $rootScope, Notification, RequestCenterService, ProjectFolderService) {
    $scope.state = "NOT_STARTED";
    $scope.phase = "READY_TO_DOWNLOAD";
    $scope.downloadData = {};
    $scope.importSettings = {
        remapping : {
            connections : []
        },
    };

    $scope.isDSSAdmin = $rootScope.isDSSAdmin();

    function download(){
        $scope.phase = "DOWNLOADING";

        MonoFuture($scope).wrap(DataikuAPI.projects.downloadTutorial)($scope.tutorialIdToInstall, $scope.tutorialType).success(function(data) {
            $scope.state = data.result.success ? "DONE" : "FAILED";
            $scope.stateShown = null;

            if (!data.result.success) {
                $scope.failure = {
                    aborted : data.aborted,
                    message : data.result.downloadError.detailedMessage
                }
                WT1.tryEvent("tutorial-project-creation-failed", () => {
                    return {
                        tutorialId : $scope.tutorialIdToInstall
                    };
                });
            } else {
                $scope.downloadResult = data.result;
                $scope.prepareImport();
            }
            $scope.installingFuture = null;
        }).update(function(data) {
            $scope.stateShown = data.progress != null && data.progress.states != null && data.progress.states.length > 0 ?
                                                                    data.progress.states[data.progress.states.length - 1] : null;
            $scope.state = "RUNNING";
            $scope.installingFuture = data;
        }).error(handleJobError);
    };

    function handleJobError(data, status, headers) {
        $scope.state = "FAILED";
        if ( data.aborted) {
           $scope.failure = {
                aborted : data.aborted,
                message : "Aborted"
           }
        } else if (data.hasResult) {
           $scope.failure = {
                aborted : data.aborted,
                message : data.result.errorMessage
           }
        } else {
           $scope.failure = {
                aborted : data.aborted,
                message : "Unexpected error"
           }
        }
        $scope.installingFuture = null;
    }

    $scope.prepareImport = function() {
        $scope.phase = "PREPARE_IMPORT";
        $scope.state = "RUNNING";

        DataikuAPI.projects.prepareImportTutorial($scope.downloadResult.importId)
            .success(function(data) {
                $scope.prepareResponse = data;
                $scope.state = "DONE";
                if($scope.prepareResponse.success){
                    $scope.startImport();
                }else if($scope.prepareResponse.prepareError){
                    $scope.state = "FAILED";
                    $scope.failure = {
                        aborted : false,
                        message : $scope.prepareResponse.prepareError.detailedMessage
                    }
                }else{
                    $scope.updatePreviousRequests();
                    Notification.registerEvent("plugin-request-granted", function(_, evt) {
                        removePluginWarnings(evt.objectId);
                    });
                    Notification.registerEvent("plugin-request", function(_, evt) {
                        const requestedPluginInstall = $scope.prepareResponse.notInstalledPlugins.find(element => {
                            return element.id === evt.objectId;
                        });
                        if(requestedPluginInstall && $rootScope.appConfig.login === evt.requesterLogin){
                            requestedPluginInstall.hasPreviousRequest = true;
                        }
                        const requestedPluginUpdate = $scope.prepareResponse.notUpToDatePlugins.find(element => {
                            return element.id === evt.objectId;
                        });
                        if(requestedPluginUpdate && $rootScope.appConfig.login === evt.requesterLogin){
                            requestedPluginUpdate.hasPreviousRequest = true;
                        }
                    });
                    Notification.registerEvent("plugin-changed", function(_, evt) {
                        if(evt.action !== 'INSTALLED'){
                            return;
                        }
                        removePluginWarnings(evt.pluginId);
                    });
                    Notification.registerEvent("code-env-request-granted", function(_, evt) {
                        removeCodeEnvErrors(evt.objectId);
                    });
                    Notification.registerEvent("code-env-request", function(_, evt) {
                        const requestedCodeEnv = $scope.prepareResponse.missingCodeEnvs.find(element => {
                            return element.id === evt.objectId;
                        });
                        if(requestedCodeEnv && $rootScope.appConfig.login === evt.requesterLogin){
                            requestedCodeEnv.hasPreviousRequest = true;
                        }
                    });
                    Notification.registerEvent("code-env-changed", function(_, evt) {
                        if(evt.action == 'CREATED'){
                            removeCodeEnvErrors(evt.envName);
                        }
                    });
                }
            }).error(setErrorInScope.bind($scope));
    }

    function removePluginWarnings(pluginId){
        const requestedPluginInstall = $scope.prepareResponse.notInstalledPlugins.find(element => {
            return element.id === pluginId;
        });
        if(requestedPluginInstall){
            requestedPluginInstall.hasPreviousRequest = false;
            let index = $scope.prepareResponse.notInstalledPlugins.indexOf(requestedPluginInstall);
            if (index !== -1) {
                $scope.prepareResponse.notInstalledPlugins.splice(index, 1);
            }
        }
        const requestedPluginUpdate = $scope.prepareResponse.notUpToDatePlugins.find(element => {
            return element.id === pluginId;
        });
        if(requestedPluginUpdate){
            requestedPluginUpdate.hasPreviousRequest = false;
            let index = $scope.prepareResponse.notUpToDatePlugins.indexOf(requestedPluginUpdate);
            if (index !== -1) {
                $scope.prepareResponse.notUpToDatePlugins.splice(index, 1);
            }
        }
    }

    function removeCodeEnvErrors(codeEnvName){
        const codeEnv = $scope.prepareResponse.missingCodeEnvs.find(element => {
            return element.envName === codeEnvName;
        });
        if(codeEnv){
            codeEnv.hasPreviousRequest = false;
            let index = $scope.prepareResponse.missingCodeEnvs.indexOf(codeEnv);
            if (index !== -1) {
                $scope.prepareResponse.missingCodeEnvs.splice(index, 1);
            }
        }
    }

    $scope.updatePreviousRequests = function(){
        for(let p of $scope.prepareResponse.notInstalledPlugins){
            updatePreviousPluginRequest(p);
        }
        for(let p of $scope.prepareResponse.notUpToDatePlugins){
            updatePreviousPluginRequest(p);
        }
        for(let p of $scope.prepareResponse.missingCodeEnvs){
            updatePreviousCodeEnvRequest(p);
        }
    }

    function updatePreviousPluginRequest(pluginRequirement){
        DataikuAPI.requests.getLatestRequestForCurrentUser(pluginRequirement.id, "PLUGIN", "")
        .then(response => {
            if (response.data.status === "PENDING") {
                pluginRequirement.hasPreviousRequest = true;
                pluginRequirement.latestRequest = response.data;
            } else {
                pluginRequirement.hasPreviousRequest = false;
                pluginRequirement.latestRequest = {};
            }
        }, error => {
            if(error.status === 404){
                pluginRequirement.hasPreviousRequest = false;
                pluginRequirement.latestRequest = {};
            } else {
                setErrorInScope.bind($scope)(error);
            }
        });
    }

    function updatePreviousCodeEnvRequest(codeEnv){
        DataikuAPI.requests.getLatestRequestForCurrentUser(codeEnv.envName, "CODE_ENV", "")
        .then(response => {
            if (response.data.status === "PENDING") {
                codeEnv.hasPreviousRequest = true;
                codeEnv.latestRequest = response.data;
            } else {
                codeEnv.hasPreviousRequest = false;
                codeEnv.latestRequest = {};
            }
        }, error => {
            if(error.status === 404){
                codeEnv.hasPreviousRequest = false;
                codeEnv.latestRequest = {};
            } else {
                setErrorInScope.bind($scope)(error);
            }
        });
    }

    $scope.startImport = function() {
        Assert.trueish($scope.phase === "PREPARE_IMPORT" && $scope.state === "DONE", 'not ready to import');
        Assert.trueish($scope.downloadResult.importId, 'no download id');

        $scope.phase = "IMPORTING";
        $scope.state = "RUNNING";

        resetErrorInScope($scope);

        MonoFuture($scope).wrap(DataikuAPI.projects.startImportTutorial)($scope.downloadResult.importId, $scope.tutorialIdToInstall, $scope.downloadResult.projectKey, $scope.targetProjectFolder.id).success(function(data) {
            $scope.state = data.result.success ? "DONE" : "FAILED";
            $scope.stateShown = null;
            if (!data.result.success) {
                $scope.failure = {
                    aborted : data.aborted,
                    message : data.result.installationError.detailedMessage
                }
                WT1.tryEvent("tutorial-project-creation-failed", () => {
                    return {
                        tutorialId : $scope.tutorialIdToInstall
                    };
                });
            } else {
                $scope.projectKey = data.result.projectKey;
                WT1.tryEvent("tutorial-project-created", () => {
                    return {
                        tutorialId : $scope.tutorialIdToInstall,
                        isInSuggestedFolder: $scope.targetProjectFolder.id === $scope.defaultProjectFolderId,
                        isInRoot: ProjectFolderService.isInRoot($scope.targetProjectFolder.id),
                        isUnderSandbox: ProjectFolderService.isUnderSandbox($scope.targetProjectFolder)
                    };
                });
            }

            $scope.installingFuture = null;
        }).update(function(data) {
            $scope.stateShown = data.progress != null && data.progress.states != null && data.progress.states.length > 0 ?
                                                                    data.progress.states[data.progress.states.length - 1] : null;
            $scope.state = "RUNNING";
            $scope.installingFuture = data;
        }).error(handleJobError);
    }

    $scope.abort = function() {
        $scope.state = "FAILED";
        $scope.failure = {
            message : "Aborted"
        }
        DataikuAPI.futures.abort($scope.installingFuture.jobId);
    };

    $scope.closeAndGo = function(){
        $scope.dismiss();
        $state.go("projects.project.home.regular", {projectKey : $scope.projectKey})
    }

    $scope.openPluginInstall = function(pluginId){
        $window.open(StateUtils.href.pluginStore(pluginId), '_blank');
    }

    $scope.openCodeEnvCreate = function(codeEnv){
        DataikuAPI.admin.codeenvs.design.prepareDraftFromProjectImport(codeEnv.envLang, codeEnv.envName, JSON.stringify(codeEnv.desc), codeEnv.specCondaEnvironment, codeEnv.specPackageList).success((data) => {
             $window.open(StateUtils.href.codeEnvCreation(data.config.id), '_blank');
        }).error(setErrorInScope.bind($scope));
    }

    $scope.sendInstallPluginRequest = function(pluginRequirement){
        $scope.sendPluginRequest(pluginRequirement, 'INSTALL_PLUGIN');
    }

    $scope.sendCreateCodeEnvRequest = function(codeEnv){
        let requestMessage = "Required code-env to install the industry solution: " + $scope.tutorialName;
        DataikuAPI.requests.createCodeEnvRequest("INSTALL_CODE_ENV", codeEnv.envName, codeEnv.envLang, JSON.stringify(codeEnv.desc) ,codeEnv.specPackageList, codeEnv.specCondaEnvironment, requestMessage)
        .success(function(data){
            codeEnv.hasPreviousRequest = true;
            RequestCenterService.WT1Events.onRequestSent("CODE_ENV", null, codeEnv.envName, requestMessage, data.id);
        }).error(() => {
            setErrorInScope.bind($scope);
        });
    }

    $scope.sendUpdatePluginRequest = function(pluginRequirement){
        $scope.sendPluginRequest(pluginRequirement, 'UPDATE_PLUGIN');
    }

    $scope.sendPluginRequest = function(pluginRequirement, requestType){
        let requestTypeLabel = requestType === "INSTALL_PLUGIN" ? "install" : "update";
        let requestMessage = "Required plugin to install the Dataiku Solution: " + $scope.tutorialName;
        DataikuAPI.requests.createPluginRequest(requestType, pluginRequirement.id, requestMessage)
        .success((data) => {
            pluginRequirement.hasPreviousRequest = true;
            RequestCenterService.WT1Events.onRequestSent("PLUGIN", null, pluginRequirement.id, requestMessage, data.id);
        }).error(() => {
            setErrorInScope.bind($scope);
        });
    }

    $scope.hasErrors = function(){
        return $scope.prepareResponse
                && $scope.prepareResponse.missingCodeEnvs.length > 0;
    }
    $scope.hasWarnings = function(){
        return $scope.prepareResponse
                && ($scope.prepareResponse.notInstalledPlugins.length > 0 ||
                    $scope.prepareResponse.notUpToDatePlugins.length > 0 ||
                    $scope.prepareResponse.notExistingPlugins.length > 0);
    }

    $scope.$on("$destroy", function(){
        if ($scope.state == "RUNNING") {
            $scope.abort();
        }
    });

    /* Wait for the plugin id to start */
    $scope.$watch("tutorialIdToInstall", Fn.doIfNv(download));
});

// You MUST call setCurrentProjectFolderId with the appropriate folderId in the link method of CreateModal...
app.controller('NewProjectController', function($scope, DataikuAPI, $state, $stateParams, WT1, ProjectFolderContext, PromiseService, ProjectFolderService, PathUtils) {

    $scope.modalTabState = { active: "create" };
    $scope.newProject = {};
    $scope.uniq = true;

    $scope.targetProjectFolder = null; // used in the project folder service
    $scope.defaultProjectFolderId = null;

    $scope.setCurrentProjectFolderId = (folderId) => {
        ProjectFolderService.getDefaultFolderForNewProject(folderId).then((folder) => {
            $scope.targetProjectFolder = folder;
            $scope.defaultProjectFolderId = folder.id;
        }).catch(setErrorInScope.bind($scope));
    }

    $scope.getProjectFolderName = item => item.name;
    $scope.canSelectFolder = item => item.canWriteContents;

    DataikuAPI.projects.listAllKeys()
        .then(function(resp) { $scope.allProjectKeys = resp.data; })
        .catch(setErrorInScope.bind($scope));

    function isProjectKeyUnique(value) {
        return !$scope.allProjectKeys || $scope.allProjectKeys.indexOf(value) < 0;
    };

    $scope.$watch("newProject.projectKey", function(nv, ov) {
        $scope.uniq = !nv || isProjectKeyUnique(nv);
    });

    const isNum = (v) => v && !isNaN(v);
    $scope.projectKeyIsNum = () => isNum($scope.newProject.projectKey);

    $scope.$watch("newProject.name", function(nv, ov) {
        if (!nv) return;
        let slug = nv.toUpperCase().replace(/\W+/g, ""),
            cur = isNum(slug) ? 'P' + slug : slug,
            i = 0;
        while (!isProjectKeyUnique(cur)) {
            cur = slug + "_" + (++i);
        }
        $scope.newProject.projectKey = cur;
    });

    $scope.create = function() {
        DataikuAPI.projects.create($scope.newProject.projectKey, $scope.newProject.name, $scope.targetProjectFolder.id)
            .success(function(data) {
                $scope.dismiss();
                $state.transitionTo("projects.project.home.regular", {projectKey : $scope.newProject.projectKey});
            }).error(setErrorInScope.bind($scope));
        WT1.tryEvent("project-create", () => {
            return {
                hashedProjectId: md5($scope.newProject.projectKey),
                isInSuggestedFolder: $scope.targetProjectFolder.id === $scope.defaultProjectFolderId,
                isInRoot: ProjectFolderService.isInRoot($scope.targetProjectFolder.id),
                isUnderSandbox: ProjectFolderService.isUnderSandbox($scope.targetProjectFolder)
            };
        });
    };

    $scope.browse = folderIds => {
        // Use last id in path
        $scope.destination = PathUtils.makeNLNT(folderIds).split('/').pop();
        // browse-path expects a success-error promise so we need to wrap with qToHttp for now (catch() does not return a monkey-patched promise)
        return PromiseService.qToHttp(ProjectFolderService.getBrowseNode($scope.destination).catch(setErrorInScope.bind($scope)));
    };
});


// Re-render dku-bs-select every time the source connection to update connection sorting
app.directive("importProjectRemappingForm", function() {
    return {
        scope: false,
        link: function($scope, $elt) {
            $scope.$watch("conn.source", function(nv) {
                if (!nv) return;
                $elt.find('select[dku-bs-select]').selectpicker('refresh');
            })
            $scope.$watch("codeEnv.source", function(nv) {
                if (!nv) return;
                $elt.find('select[dku-bs-select]').selectpicker('refresh');
            })
        }
    }
});

// You MUST call setCurrentProjectFolderId with the appropriate folderId in the link method of CreateModal...
app.controller('ImportProjectController', function($scope, Assert, DataikuAPI, $state, $filter,
               FutureWatcher, ProgressStackMessageBuilder, CreateModalFromTemplate, Dialogs, $timeout, Fn, WT1, ProjectFolderContext, PromiseService, ProjectFolderService, PathUtils) {
    // get the list, don't get it from the home (in case the call to populate the home is too slow)
    DataikuAPI.projects.listAllKeys()
    .success(function(data) { $scope.allProjectKeys = data; })
    .error(setErrorInScope.bind($scope));

    $scope.importData = {}
    $scope.importSettings = {
        remapping : {
            connections : []
        },
        targetProjectFolderId: null
    };
    $scope.uiState = {
        targetProjectFolder: null
    }

    $scope.setCurrentProjectFolderId = (folderId) => {
        ProjectFolderService.getDefaultFolderForNewProject(folderId).then((folder) => {
            $scope.uiState.targetProjectFolder = folder;
            $scope.importSettings.targetProjectFolderId = folder.id;
            $scope.defaultProjectFolderId = folder.id;
        }).catch(setErrorInScope.bind($scope));
    }

    $scope.getProjectFolderName = item => item.name;
    $scope.canSelectFolder = item => item.canWriteContents;

    $scope.browse = folderIds => {
        // Use last id in path
        $scope.destination = PathUtils.makeNLNT(folderIds).split('/').pop();
        // browse-path expects a success-error promise so we need to wrap with qToHttp for now (catch() does not return a monkey-patched promise)
        return PromiseService.qToHttp(ProjectFolderService.getBrowseNode($scope.destination).catch(setErrorInScope.bind($scope)));
    };

    $scope.phase = "READY_TO_UPLOAD";
    $scope.prepare = {enabled: false};

    $scope.selectFilter = function(selected) {
        return function(connection) {
            return !connection.mapped || connection.name == selected;
        }
    };

    $scope.updateSelect = function(e) {
        // Nothing to do
    };

    $scope.connComparator = function(sourceCon) {
        var
            source = $scope.findConnection(
            $scope.prepareResponse.usedConnections, sourceCon);
        var sourceType = source && source.type;
        return function(connection) {
            const niceConnectionType = $filter('connectionTypeToNameForList')(connection.type);
            if (connection.type == sourceType) {
                return "AAAAA" + niceConnectionType + "." + connection.name;
            } else {
                return "ZZZZZ" + niceConnectionType + "." + connection.name;
            }
        };
    };

    $scope.codeEnvComparator = function(sourceCodeEnv) {
        var
            source = $scope.findCodeEnv(
            $scope.prepareResponse.usedCodeEnvs, sourceCodeEnv);
        var sourceEnvLang = source && source.envLang;
        return function(codeEnv) {
            if (codeEnv.envLang == sourceEnvLang) {
                return "AAAAA" + codeEnv.envLang + "." + codeEnv.name;
            } else {
                return "ZZZZZ" + codeEnv.envLang + "." + codeEnv.name;
            }
        };
    };

    $scope.findConnection = function(connections, connection) {
        return Array.dkuFindFn(connections, function(c) { return c.name == connection });
    };

    $scope.findCodeEnv = function(codeEnvs, codeEnvName) {
        return Array.dkuFindFn(codeEnvs, function(c) { return c.envName == codeEnvName });
    };

    var abortHook = null;
    $scope.attemptImport = function(){
        Assert.trueish($scope.phase == "READY_TO_IMPORT", 'not ready to import');
        Assert.trueish($scope.uploadResult.id, 'no upload id');

        $scope.phase = "IMPORTING";

        resetErrorInScope($scope);
        DataikuAPI.projects.startImport($scope.uploadResult.id, $scope.importSettings).success(function(initialResponse){
            abortHook = function() {
                DataikuAPI.futures.abort(initialResponse.jobId).error(setErrorInScope.bind($scope));
            };
            FutureWatcher.watchJobId(initialResponse.jobId)
                .success(function(data) {
                    abortHook = null;
                    $scope.futureResponse = null;
                    $scope.importResponse = data.result;
                    if ($scope.importResponse && $scope.importResponse.success){
                        var p2 = $scope.$parent.$parent;
                        $scope.dismiss();
                        Dialogs.infoMessagesDisplayOnly(p2, "Import report", $scope.importResponse).then(function(){
                            $state.transitionTo("projects.project.home.regular", {projectKey : $scope.importResponse.usedProjectKey});
                        });
                    } else {
                        $scope.phase = "READY_TO_IMPORT";

                        // fetch the new manifest in case the migration added some stuff
                        $scope.prepare.enabled = true;
                        prepareImport();
                    }

                }).update(function(data){
                    $scope.percentage =  ProgressStackMessageBuilder.getPercentage(data.progress);
                    $scope.futureResponse = data;
                    $scope.stateLabels = ProgressStackMessageBuilder.build($scope.futureResponse.progress, true);
                }).error(function(data, status, headers) {
                    abortHook = null;
                    $scope.futureResponse = null;
                    $scope.importResponse = null;
                    $scope.phase = "READY_TO_IMPORT";
                    setErrorInScope.bind($scope)(data, status, headers);
                    $timeout(checkProjectKey);
                });
        }).error(function(a,b,c){
            $scope.phase = 'READY_TO_IMPORT';
            setErrorInScope.bind($scope)(a,b,c);
            $scope.importResponse = null;
            $timeout(checkProjectKey);
        });
        WT1.tryEvent("project-import", () => {
            return {
                displayAdvancedOptions : $scope.prepare.enabled,
                nbRemappings : $scope.importSettings.remapping ? $scope.importSettings.remapping.connections ? $scope.importSettings.remapping.connections.length : 0 : 0,
                isInSuggestedFolder: $scope.uiState.targetProjectFolder.id === $scope.defaultProjectFolderId,
                isInRoot: ProjectFolderService.isInRoot($scope.uiState.targetProjectFolder.id),
                isUnderSandbox: ProjectFolderService.isUnderSandbox($scope.uiState.targetProjectFolder),
            };
        });
    }

    $scope.$on("$destroy", function() {
        // cancel import if modal dismissed
        if (abortHook) abortHook();
    });

    $scope.startImport = function(){

        $scope.phase = "UPLOADING";

        DataikuAPI.projects.uploadForImport($scope.importData.file, function(e){
            if (e.lengthComputable) {
                $scope.$apply(function () {
                    $scope.uploadProgress = Math.round(e.loaded * 100 / e.total);
                });
            }
        }).then(function (data) {
            $scope.uploadResult = JSON.parse(data);
            prepareImport();

        }).catch((error) => {
            $scope.phase = '';
            setErrorInScope2.call($scope, error);
        });
    };

    function prepareImport() {
        DataikuAPI.projects.prepareImport($scope.uploadResult.id, $scope.importSettings)
            .success(function(data) {
                $scope.prepareResponse = data;

                $scope.usedConnections = data.usedConnections.map(Fn.prop('name'));
                $scope.usedCodeEnvs = data.usedCodeEnvs.map(Fn.prop('envName'));
                $scope.usedContainerExecConfs = data.usedContainerExecConfs.map(Fn.prop('name'));
                $scope.availableCodeEnvs = [{envLang:'PYTHON', envName:'Builtin', builtin:true}, {envLang:'R', envName:'Builtin', builtin:true}].concat($scope.prepareResponse.availableCodeEnvs);
                $scope.$watch("importSettings.targetProjectKey", checkProjectKey);

                $scope.phase = "READY_TO_IMPORT";
                if (!$scope.prepare.enabled) $scope.attemptImport();
            }).error(setErrorInScope.bind($scope));
    }

    $scope.refreshConnections = function() {
        $scope.prepare.enabled = true;
        prepareImport();
    };

    $scope.refreshCodeEnvs = function() {
        $scope.prepare.enabled = true;
        prepareImport();
    };

    $scope.refreshContainerExecConfs = function() {
        $scope.prepare.enabled = true;
        prepareImport();
    };

    function checkProjectKey(nv) {
        if ($scope.phase != "READY_TO_IMPORT") return;

        var unique;
        if(!$scope.importSettings.targetProjectKey) {
            unique = $scope.allProjectKeys.indexOf($scope.prepareResponse.originalProjectKey) == -1;
        } else {
            unique = $scope.allProjectKeys.indexOf($scope.importSettings.targetProjectKey.toUpperCase().replace(/\W+/g, "")) == -1;
        }
        $scope.importProjectForm.projectKey.$dirty = true;
        $scope.importProjectForm.projectKey.$setValidity("unique", unique);
    }
});

// You MUST call setCurrentProjectFolderId with the appropriate folderId in the link method of CreateModal...
app.controller('DuplicateProjectController', function($scope, DataikuAPI, FutureWatcher, ProgressStackMessageBuilder, WT1, PathUtils, $filter, $state, PromiseService, ProjectFolderService, ProjectFolderContext, $q, $window) {
    $scope.hasPartitionedDataset = false;
    $scope.uniq = true;
    $scope.dupProject = {
        projectKey: "COPY_OF_" + $scope.projectSummary.projectKey,
        name: "Copy of " + $scope.projectSummary.name
    };
    $scope.dupOptions = {
        exportAnalysisModels: true,
        exportSavedModels: true,
        exportGitRepository: $scope.projectSummary.canExportGitRepository,
        exportInsightsData: true,
        duplicationMode: 'UPLOADS_ONLY',
        exportUploads: true,
        exportEditableDatasets: true,
        exportAllInputDatasets: true,
        exportAllInputManagedFolders: true,
        exportAllDatasets: false,
        exportManagedFolders: false,
        exportProjectResources: false,
        exportNotebooksWithOutputs: true,
        exportPromptStudioHistories: true,
        targetProjectFolderId: ProjectFolderContext.getCurrentProjectFolderId() || "",
    };
    $scope.phase = 'READY_TO_DUPLICATE';

    $scope.uiState = {
        showAdvancedOptions: false,
        canExportGitRepository: $scope.projectSummary.canExportGitRepository
    };

    DataikuAPI.projects.listAllKeys()
        .success(function(data) {
            $scope.allProjectKeys = data;
            $scope.$watch("dupProject.name", function(nv, ov) {
                if (!nv) return;
                var slug = nv.toUpperCase().replace(/\W+/g, ""),
                    cur = slug,
                    i = 0;
                while (!isProjectKeyUnique(cur)) {
                    cur = slug + "_" + (++i);
                }
                $scope.dupProject.projectKey = cur;
            });
        })
        .error(setErrorInScope.bind($scope));

    $scope.uiState = {
        targetProjectFolder: null,
        defaultProjectFolderId: null
    };

    $scope.setCurrentProjectFolderId = (folderId) => {
        $scope.dupOptions.targetProjectFolderId = folderId;
        ProjectFolderService.getDefaultFolderForNewProject($scope.dupOptions.targetProjectFolderId).then((folder) => {
            $scope.uiState.targetProjectFolder = folder;
            $scope.uiState.defaultProjectFolderId = folder.id;
            $scope.dupOptions.targetProjectFolderId = folder.id;
        }).catch(setErrorInScope.bind($scope));
    }

    function isProjectKeyUnique(value) {
        return !$scope.allProjectKeys || $scope.allProjectKeys.indexOf(value) < 0;
    }

    $scope.uniq = isProjectKeyUnique($scope.dupProject.projectKey);

    $scope.$watch("dupProject.projectKey", function(nv, ov) {
        $scope.uniq = !nv || isProjectKeyUnique(nv);
    });

    $scope.connComparator = function(sourceCon) {
        var
            source = $scope.findConnection(
            $scope.prepareResponse.usedConnections, sourceCon);
        var sourceType = source && source.type;
        return function(connection) {
            /**
             * Returns the order in which the available connections will be displayed in the selector
             * - high up in the list if the connection types are compatible (so starting with "AAAAA")
             * - last in the list if they are not (starting with "ZZZZZ")
             */
             const niceConnectionType = $filter('connectionTypeToNameForList')(connection.type);
            if (connection.type == sourceType) {
                return "AAAAA" + niceConnectionType + "." + connection.name;
            } else {
                return "ZZZZZ" + niceConnectionType + "." + connection.name;
            }
        };
    };

    $scope.findConnection = function(connections, connection) {
        return Array.dkuFindFn(connections, function(c) { return c.name == connection });
    };

    $scope.refreshConnections = function(projectKey) {
        DataikuAPI.projects.getProjectDatasets(
            projectKey
        ).then(function(initialResponse){
            $scope.prepareResponse = $scope.prepareResponse ? $scope.prepareResponse : {};
            $scope.prepareResponse.usedConnections = [];
            $scope.usedConnections = [];
            for (let requiredConnection in initialResponse.data.requiredConnections) {
                $scope.prepareResponse.usedConnections.push(initialResponse.data.requiredConnections[requiredConnection]);
                $scope.usedConnections.push(initialResponse.data.requiredConnections[requiredConnection].name);
            }
            $scope.hasPartitionedDataset = initialResponse.data.hasPartitionedDataset;
        });

        DataikuAPI.projects.prepareImport('', '')
            .then(function(response) {
                $scope.prepareResponse = $scope.prepareResponse ? $scope.prepareResponse : {};
                $scope.prepareResponse.availableConnections = [];
                response.data.availableConnections.forEach(function(availableConnection){
                    $scope.prepareResponse.availableConnections.push(availableConnection);
                });
            })
    };

    $scope.setDuplicationMode = function(mode) {
        $scope.dupOptions.duplicationMode = mode;
    };

    var abortHook = null;

    $scope.gotoResult = function() {
        $scope.dismiss();
        const targetProjectFolderId = $scope.dupOptions.targetProjectFolderId;
        // When duplicating a project from the project summary page we redirect to the new duplicated project summary page
        // Else we redirect to the projects folder homepage where the project has been duplicated to
        if ($state.includes('projects.project')) {
            $state.go('projects.project.home.regular', { projectKey: $scope.dupProject.projectKey })
        } else {
            $state.go('homeV2.projects.folder', { folderId: targetProjectFolderId }, { reload: true })
        }
    };

    $scope.duplicate = function() {
        $scope.phase = 'DUPLICATING';
        $scope.dupOptions.targetProjectKey = $scope.dupProject.projectKey;
        $scope.dupOptions.targetProjectName = $scope.dupProject.name;
        DataikuAPI.projects.startProjectDuplication(
            $scope.projectSummary.projectKey,
            $scope.dupOptions
        ).success(function(initialResponse){
            abortHook = function() {
                DataikuAPI.futures.abort(initialResponse.jobId).error(setErrorInScope.bind($scope));
            };
            FutureWatcher.watchJobId(initialResponse.jobId).success(function(data){
                abortHook = null;
                $scope.futureResponse = null;
                $scope.duplicateResponse = data.result;
                if (!data.aborted && (data.result.success || data.result.messages == null || data.result.messages.length == 0)) {
                    $scope.gotoResult();
                } else if ((data.result.warning || data.result.error) && !data.result.fatal) {
                    $scope.phase = "SHOW_WARNINGS";
                } else {
                    $scope.phase = "READY_TO_DUPLICATE";
                }
            }).update(function(data){
                $scope.percentage = ProgressStackMessageBuilder.getPercentage(data.progress);
                $scope.futureResponse = data;
                $scope.stateLabels = ProgressStackMessageBuilder.build($scope.futureResponse.progress, true);
            }).error(function(data, status, headers) {
                abortHook = null;
                $scope.futureResponse = null;
                $scope.duplicateResponse = null;
                $scope.phase = "READY_TO_DUPLICATE";
                setErrorInScope.bind($scope)(data, status, headers);
            })
        }).error(function(a,b,c){
            $scope.phase = 'READY_TO_DUPLICATE';
            setErrorInScope.bind($scope)(a,b,c);
            $scope.duplicateResponse = null;
        });
        WT1.tryEvent("project-duplicate", () => {
            return {
                duplicationMode: $scope.dupOptions.duplicationMode,
                exportAnalysisModels: $scope.dupOptions.exportAnalysisModels,
                exportSavedModels: $scope.dupOptions.exportSavedModels,
                exportModelEvaluationStores: $scope.dupOptions.exportModelEvaluationStores,
                exportProjectResources: $scope.dupOptions.exportProjectResources,
                exportGitRepository: $scope.dupOptions.exportGitRepository,
                exportInsightsData: $scope.dupOptions.exportInsightsData,
                isInSuggestedFolder: ($scope.uiState.targetProjectFolder.id === $scope.uiState.defaultProjectFolderId),
                isInRoot: ProjectFolderService.isInRoot($scope.uiState.targetProjectFolder.id),
                isUnderSandbox: ProjectFolderService.isUnderSandbox($scope.uiState.targetProjectFolder),
                nbRemappings: $scope.dupOptions.remapping ? $scope.dupOptions.remapping.connections ? $scope.dupOptions.remapping.connections.length : 0 : 0
            };
        });
    };

    $scope.$on("$destroy", function() {
        // cancel import if modal dismissed
        if (abortHook) abortHook();
    });

    $scope.refreshConnections($scope.projectSummary.projectKey);

    $scope.browse = folderIds => {
        // Use last id in path
        $scope.destination = PathUtils.makeNLNT(folderIds).split('/').pop();
        // browse-path expects a success-error promise so we need to wrap with qToHttp for now (catch() does not return a monkey-patched promise)
        return PromiseService.qToHttp(ProjectFolderService.getBrowseNode($scope.destination).catch(setErrorInScope.bind($scope)));
    };

    $scope.canSelect = item => item.canWriteContents;

    $scope.getProjectFolderName = item => item.name;
});

app.controller('DebuggingToolsController', function($scope, DataikuAPI, $state, $stateParams, Dialogs, $window, OpalsService) {
    $scope.$state = $state;
    $scope.uiState = {};
    $scope.fakeFutureTypes = [];
    $scope.fakeFutureTypes.push({"name":"export from dataset", "payloadClassName":"com.dataiku.dip.export.LocalExportFutureThread", "payloadMethodName":"buildFuturePayload"});
    $scope.fakeFutureTypes.push({"name":"sql query in notebook", "payloadClassName":"com.dataiku.dip.server.services.SQLNotebooksService", "payloadMethodName":"buildFuturePayload"});
    $scope.fakeFutureTypes.push({"name":"sample building", "payloadClassName":"com.dataiku.dip.shaker.SampleBuilder", "payloadMethodName":"buildFuturePayload"});
    $scope.killBackend = function(){
        DataikuAPI.internal.debugKillBackend();
    }
    $scope.getBackendStacks = function(){
        DataikuAPI.internal.debugGetBackendStacks().success(function(data){
            $scope.retdata = data;
        })
    }
    $scope.restartAllHTMLBackends = function(){
        DataikuAPI.internal.restartAllHTMLBackends().success(function(data){
            $scope.retdata = data;
        })
    }
    $scope.runScenarioTriggers = function(){
        DataikuAPI.internal.runScenarioTriggers().success(function(data){
            $scope.retdata = data;
        })
    }
    $scope.insertFakeFuture = function(){
        var f = $scope.uiState.fakeFutureType;
        DataikuAPI.internal.fakeFuture($stateParams.projectKey, f.payloadClassName, f.payloadMethodName, false).success(function(data){
            $scope.retdata = data;
            $scope.uiState.fakeFutureType = null;
        })
    }
    $scope.getTriggerQueueingInfo = function(){
        DataikuAPI.internal.getTriggerQueueingInfo().success(function(data){
            $scope.retdata = data;
        })
    }
    $scope.resyncProjectFolders = () => {
        DataikuAPI.internal.resyncProjectFolders();
    };
    $scope.clearScenarioReportsCaches = function () {
        DataikuAPI.internal.clearScenarioReportsCaches();
    }
    $scope.invalidateEDACaches = function () {
        DataikuAPI.internal.invalidateEDACaches();
    }
    $scope.invalidateDriftCaches = function () {
        DataikuAPI.internal.invalidateDriftCaches();
    }
    $scope.killLLMRequests = function () {
        DataikuAPI.internal.killLLMRequests();
    }
    $scope.killLLMKernels = function () {
        DataikuAPI.internal.killLLMKernels();
    }
    $scope.dumpLLMMesh = function () {
        $window.open(DataikuAPI.internal.dumpLLMMeshURL());
    }
    $scope.dumpLLMCostLimitCounters = function () {
        $window.open(DataikuAPI.internal.dumpLLMCostLimitCountersURL());
    }
    $scope.clearLLMCostLimitCounters = function () {
        DataikuAPI.internal.clearLLMCostLimitCounters();
    }
    $scope.massGenerateLogs = function () {
        DataikuAPI.internal.massGenerateLogs().success(function(data){
            $scope.retdata = data;
        });
    }
    $scope.invalidateConfigCache = function () {
         var options = {type: 'text'};
         Dialogs.prompt($scope, "Invalidate cache", "Path to invalidate", "", options)
                .then(function(path) {
                    DataikuAPI.admin.invalidateConfigCache(path);
                 });
    }
    $scope.invalidateInMemoryAPIKeys = function () {
        DataikuAPI.internal.invalidateInMemoryAPIKeys();
    }
    $scope.listInMemoryAPIKeys = function () {
        DataikuAPI.internal.listInMemoryAPIKeys().success(function(data){
            $scope.retdata = data;
        });
    }
    $scope.reloadTutorials = function () {
         DataikuAPI.internal.reloadTutorials();
    }

    $scope.toggleQaSelectors = function () {
        window.toggleQaSelectors();
    }

    const emptyQuestionnaire = {
        skippedQuestionnaire: false,
        finishedQuestionnaire: false,
        finishedOnboardingChoice: false,
        answers: []
    };

    $scope.resetInProductOnboarding = function () {
        $scope.resetInProductOnboardingNoOpals();
        OpalsService.setLocalStorage('onboardingQuestionnaire', JSON.stringify(emptyQuestionnaire));
        OpalsService.setLocalStorage('prepareTourCompleted', "false");
        OpalsService.setLocalStorage('flowTourCompleted', "false");
        OpalsService.setLocalStorage('exploreTourCompleted', "false");
    }

    $scope.resetInProductOnboardingNoOpals = function () {
        DataikuAPI.profile.setQuestionnaire(emptyQuestionnaire).error(setErrorInScope.bind($scope));
        DataikuAPI.profile.updatePageSpecificTourSettings({}).error(setErrorInScope.bind($scope)); // reset the pageSpecificTour settings to default values by sending an empty object
    }

});

app.controller('TranslationToolsController', function($scope, DataikuAPI, $rootScope, $translate) {
    $scope.dirty = false;
    $scope.dirtyRows = new Map();
    $scope.location = "frontend";
    $scope.language = $rootScope.appConfig.userSettings.uiLanguage || "en";
    $scope.translationTable = [];
    $scope.translationReferences = {};

    function confirmChanges() {
        return confirm("This will discard any unsaved changes. Are you sure?");
    }

    function buildTranslationTable(translations) {
        const translationTable = [];
        const allKeys = new Set(Object.keys(translations)).union(new Set(Object.keys($scope.translationReferences)))
        for (const key of [...allKeys].sort()) {
            translationTable.push({
                key: key,
                reference: $scope.translationReferences[key],
                value: translations[key],
                originalValue: translations[key]
            });
        }
        $scope.translationTable = translationTable;
        $scope.dirty = false;
        $scope.dirtyRows.clear();
    }

    function loadTranslationTable() {
        DataikuAPI.translations.get($scope.location, $scope.language)
            .success(data => buildTranslationTable(data.translations))
            .error(setErrorInScope.bind($scope));
    }

    $scope.markAsDirty = function(row) {
        if (row.value === row.originalValue) {
            $scope.dirtyRows.delete(row.key);
        } else {
            $scope.dirtyRows.set(row.key, row);
        }
        $scope.dirty = ($scope.dirtyRows.size !== 0);
    }
    $scope.save = function() {
        const translations = {};
        for (const translation of $scope.translationTable) {
            translations[translation["key"]] = translation["value"];
        }
        DataikuAPI.translations.save($scope.location, $scope.language, { "translations": translations }).success(function() {
            $scope.dirtyRows.forEach(row => row["originalValue"] = row["value"]);
            $scope.dirty = false;
            $scope.dirtyRows.clear();
            if ($scope.location === "frontend" && $rootScope.appConfig.userSettings.uiLanguage === $scope.language) {
                $translate.refresh();
            }
        }).error(setErrorInScope.bind($scope));
    }
    $scope.upload = function() {
        const uploadFile = $('<input type="file" id="fileUpload" accept=".json" />');
        uploadFile.on("change", () => {
            if (uploadFile[0].files.length > 0) {
                const reader = new FileReader();
                reader.onload = event => {
                    DataikuAPI.translations.save($scope.location, $scope.language, event.target.result, true).success(() => {
                        DataikuAPI.translations.get($scope.location, $scope.language)
                            .success(function(data) {
                                buildTranslationTable(data.translations);
                                if ($scope.location === "frontend" && $rootScope.appConfig.userSettings.uiLanguage === $scope.language) {
                                    $translate.refresh();
                                }
                            }).error(setErrorInScope.bind($scope));
                    }).error(setErrorInScope.bind($scope));
                };
                reader.readAsDataURL(uploadFile[0].files[0]);
            }
        });
        uploadFile.click();
    }
    $scope.onLocationChange = function(ov) {
        if ($scope.dirty && !confirmChanges()) {
            $scope.location = ov;
        }
    }
    $scope.onLanguageChange = function(ov) {
        if ($scope.dirty && !confirmChanges()) {
            $scope.language = ov;
        }
    }
    $scope.onClose = function() {
        if (!$scope.dirty || confirmChanges()) {
            $scope.dismiss();
        }
    }
    $scope.$watch("location", function(nv, ov) {
        if (ov && nv !== ov) {
            DataikuAPI.translations.get($scope.location, "en").success(function(data) {
                $scope.translationReferences = data.translations;
                loadTranslationTable();
            }).error(setErrorInScope.bind($scope));
        }
    });
    $scope.$watch("language", function(nv, ov) {
        if (ov && nv !== ov) {
            loadTranslationTable();
        }
    });

    // Load the initial translation table
    DataikuAPI.translations.get($scope.location, "en").success(function(data) {
        $scope.translationReferences = data.translations;
        loadTranslationTable();
    }).error(setErrorInScope.bind($scope));
});

app.controller("NameFolderCommonController", $scope => {
    $scope.isNameValid = (nameFormInput, isPristineOk) => {
        if (!nameFormInput) {
            return false;
        }
        const name = nameFormInput.$viewValue;
        const isPristine = nameFormInput.$pristine;
        const hasName = name && name.length > 0;
        return (isPristineOk && isPristine) || hasName;
    };
});

app.directive("dkuShow", function($timeout) {
    return {
        scope: false,
        link: function(scope, elem, attrs) {
            let showTimer;
            let delay = parseInt(attrs.delay);
            delay = angular.isNumber(delay) ? delay : 200;

            scope.$watch(attrs.dkuShow, newVal => {
              newVal ? showSpinner() : hideSpinner();
            });

            const showSpinner = () => {
              if (showTimer) {
                  return;
              }
              showTimer = $timeout(showElement.bind(this, true), delay);
            }

            const hideSpinner = () =>  {
              if (showTimer) {
                $timeout.cancel(showTimer);
              }
              showTimer = null;
              showElement(false);
            }

            const showElement = (show) => {
              show ? elem.css({display:''}) : elem.css({display:'none'});
            }
        }
    }
});
}());
