(function() {
    'use strict';

    const DEFAULT_BACKGROUND_COLOR = '#ffffff';

    angular.module('dataiku.dashboards').component('dashboardSettingsPanel', {
        templateUrl: '/static/dataiku/js/dashboards/components/dashboard-settings-panel/dashboard-settings-panel.component.html',
        bindings: {
            // Inputs
            activeView: '<', // DASHBOARD_SETTINGS_PANEL_VIEWS
            dashboard: '<', // Dashboard
            page: '<', // DashboardPage
            currentPageIndex: '<', // number
            selectedTiles: '<', // Tile
            selectedInsights: '<', // Insight,
            canModerateDashboards: '<', // boolean
            theme: '<', //DSSVisualizationTheme
            // Outputs
            openCopySlideModal: '&', // ({ $pageIndex: number }) => void
            removePage: '&', // ({ $pageIndex: number }) => void
            toggleDashboardSettings: '&' // () => void
        },
        controller: function($scope, $timeout, WT1, DashboardFilters, DashboardUtils, DASHBOARD_SETTINGS_PANEL_VIEWS, translate, TileUtils, ColorUtils, DSSVisualizationThemeUtils, DefaultDSSVisualizationTheme) {
            const ctrl = this;

            ctrl.themeColors = ColorUtils.getThemeColorsWithBlackWhite(ctrl.theme);
            ctrl.colors = ColorUtils.generateThemePaletteColors(DefaultDSSVisualizationTheme.colors, ctrl.themeColors.length > 0).foregroundColors;
            ctrl.TileUtils = TileUtils;
            ctrl.translate = translate;
            ctrl.resizeDashboard = DashboardUtils.resizeDashboard;

            ctrl.DASHBOARD_SETTINGS_PANEL_VIEWS = DASHBOARD_SETTINGS_PANEL_VIEWS;
            ctrl.page = { 'backgroundColor': DEFAULT_BACKGROUND_COLOR };
            ctrl.canCrossFilter = false;
            ctrl.isSingleTileSelected = false;
            ctrl.isSingleGroupTileSelected = false;
            ctrl.areMultipleTilesSelected = false;
            ctrl.shouldShowResetTileSpacing = shouldShowResetTileSpacing();

            let needsResize = false;

            ctrl.$onChanges = (changes) => {
                if ((changes.dashboard || changes.currentPageIndex) && ctrl.dashboard && ctrl.currentPageIndex != null) {
                    ctrl.page = ctrl.dashboard.pages[ctrl.currentPageIndex];
                    if (ctrl.page) {
                        ctrl.page.backgroundColor = ctrl.page.backgroundColor || DEFAULT_BACKGROUND_COLOR;
                        ctrl.canCrossFilter = !!DashboardFilters.findFiltersTileInPage(ctrl.page);
                    }
                }
                if (changes.theme && changes.theme.currentValue && (!ctrl.themeColors || !_.isEqual(ctrl.themeColors, ctrl.dashboard.theme.colors))) {
                    ctrl.themeColors = ColorUtils.getThemeColorsWithBlackWhite(changes.theme.currentValue);
                    ctrl.colors = ColorUtils.generateThemePaletteColors(changes.theme.currentValue.colors, ctrl.themeColors.length > 0).foregroundColors;
                }
                if (changes.selectedTiles) {
                    updateTileSelectionFlags(ctrl.selectedTiles);
                }

                // The template might have changed before the blur event is triggered, which would prevent the resize from happening if the flag is not checked.
                if (needsResize) {
                    needsResize = false;
                    DashboardUtils.resizeDashboard();
                }
            };

            ctrl.enableCrossFiltersChanged = function(dashboardId, pageId, isCrossFilteringEnabled) {
                WT1.event('dashboard-cross-filtering-enabled-change', {
                    dashboardId,
                    pageId,
                    isCrossFilteringEnabled
                });
            };

            ctrl.setTitleFontColor = (color) => {
                $timeout(() => ctrl.page.titleFontColor = color);
            };

            ctrl.setTitleAlignment = (alignment) => {
                $timeout(() => ctrl.page.titleAlign = alignment);
            };

            ctrl.shouldShowResetTitle = () => {
                const theme = (ctrl.theme ? ctrl.theme: DefaultDSSVisualizationTheme);
                return ctrl.page && (
                    (ctrl.page.titleFontColor !== theme.pageTitleFormatting.fontColor)
                    || (ctrl.page.titleFontSize !== theme.pageTitleFormatting.fontSize)
                    || (ctrl.page.titleAlign !== theme.pageTitleAlignment)
                );
            };

            ctrl.resetTitle = () => {
                const theme = (ctrl.theme ? ctrl.theme: DefaultDSSVisualizationTheme);
                ctrl.page.titleFontColor = theme.pageTitleFormatting.fontColor;
                ctrl.page.titleFontSize = theme.pageTitleFormatting.fontSize;
                ctrl.page.titleAlign = theme.pageTitleAlignment;

                $timeout(() => $scope.$apply());
            };

            ctrl.setBackgroundColor = (color) => {
                $timeout(() => ctrl.page.backgroundColor = color);
            };

            ctrl.resetBackgroundColor = () => {
                $timeout(() => ctrl.page.backgroundColor = ctrl.theme ? ctrl.theme.backgroundColor : DEFAULT_BACKGROUND_COLOR);
            };

            ctrl.shouldShowResetBackgroundColorButton = () => {
                return ctrl.page && ctrl.page.backgroundColor !== (ctrl.theme ? ctrl.theme.backgroundColor : DEFAULT_BACKGROUND_COLOR);
            };

            ctrl.getPageTitle = function() {
                return ctrl.page && ctrl.page.title ? ctrl.page.title : 'Page ' + (ctrl.currentPageIndex + 1);
            };

            ctrl.changePageVisibility = function() {
                if (ctrl.page) {
                    ctrl.page.show = !ctrl.page.show;
                }
            };

            ctrl.handleTileSpacingChange = () => {
                ctrl.shouldShowResetTileSpacing = shouldShowResetTileSpacing();
                needsResize = true;
            };

            ctrl.overrideFormattingWithTheme = function(theme) {
                $timeout(() => {
                    const currentDashboardCopy = angular.copy(ctrl.dashboard);
                    const newTheme = angular.copy(theme);
                    ctrl.dashboard.theme = newTheme;
                    DSSVisualizationThemeUtils.applyToDashboard(ctrl.dashboard, newTheme);
                    DashboardUtils.resizeDashboard();

                    DSSVisualizationThemeUtils.showThemeAppliedSnackbar(
                        ctrl.dashboard,
                        currentDashboardCopy,
                        () => DashboardUtils.resizeDashboard()
                    );
                });
            };

            ctrl.handleTileSpacingBlur = () => {
                needsResize = false;
                DashboardUtils.resizeDashboard();
            };

            ctrl.toggleAutoStackUp = () => {
                WT1.event('dashboard-auto-stack-up-toggled-from-page-settings', {
                    autoStackUp: ctrl.dashboard.autoStackUp
                });
                if (!ctrl.dashboard.autoStackUp) {
                    return;
                }
                const gridsToUpdate = [];

                /*
                 * When disabling vertival gaps we need to update the tile positions
                 * of pages that are not the current ones
                 */
                ctrl.dashboard.pages.forEach((page, index) => {
                    if (ctrl.currentPageIndex === index) {
                        return;
                    }
                    gridsToUpdate.push({ grid: page.grid, columnNumber: DashboardUtils.getColumnNumber() });
                    page.grid.tiles.forEach(tile => {
                        if (TileUtils.isGroupTile(tile) && tile.grid != null && tile.grid.tiles != null) {
                            gridsToUpdate.push({ grid: tile.grid, columnNumber: tile.box.width });
                        }
                    });
                });

                gridsToUpdate.forEach(tiles => removeGridVerticalGaps(tiles));
            };

            function removeGridVerticalGaps({ grid, columnNumber }) {
                const items = grid.tiles.map(tile => ({
                    x: tile.box.left >= 0 ? tile.box.left : null,
                    y: tile.box.top >= 0 ? tile.box.top : null,
                    w: tile.box.width,
                    h: tile.box.height,
                    tile
                }));
                // eslint-disable-next-line
                new GridList(items, {
                    lanes: columnNumber,
                    direction: 'vertical',
                    allowDirectionalGaps: true
                }).toggleGapMode(false);

                grid.tiles = items.map(item => {
                    item.tile.box.left = item.x;
                    item.tile.box.top = item.y;
                    return item.tile;
                });
            }

            ctrl.resetTileSpacing = () => {
                ctrl.dashboard.tileSpacing = (ctrl.theme ? ctrl.theme.tileSpacing : DefaultDSSVisualizationTheme.tileSpacing);
                $timeout(() => $scope.$apply()).then(() => ctrl.handleTileSpacingChange());
            };

            function updateTileSelectionFlags(selectedTiles) {
                ctrl.isSingleTileSelected = selectedTiles && selectedTiles.length === 1;
                ctrl.isSingleGroupTileSelected = ctrl.isSingleTileSelected && ctrl.TileUtils.isGroupTile(selectedTiles[0]);
                ctrl.areMultipleTilesSelected = selectedTiles && selectedTiles.length > 1;
            }

            function shouldShowResetTileSpacing() {
                return ctrl.dashboard && ctrl.dashboard.tileSpacing !== (ctrl.theme ? ctrl.theme.tileSpacing : DefaultDSSVisualizationTheme.tileSpacing);
            };
        }
    });
})();
