(function(){
'use strict';

const app = angular.module('dataiku.modelcomparisons', []);

const LINE_GRAPH_SYMBOL = "circle";
const LINE_GRAPH_SYMBOL_SIZE = 3;
const LINE_GRAPH_EMPHASIS_BORDER_WIDTH = 2;
const LINE_GRAPH_ITEM_BORDER_WIDTH = 20; // trick : big, transparent border to make hover easier
const LINE_GRAPH_LINE_WIDTH = 1;

function getStdSeriesLine(color, name, data) {
    return {
        type: 'line',
        showSymbol:false,
        symbol: LINE_GRAPH_SYMBOL,
        symbolSize: LINE_GRAPH_SYMBOL_SIZE,
        smooth: false,
        lineStyle: {
            type: 'solid',
            width: LINE_GRAPH_LINE_WIDTH,
            color
        },
        emphasis: {
            focus: 'series',
            itemStyle: {
                borderWidth: LINE_GRAPH_EMPHASIS_BORDER_WIDTH,
                borderColor: color
            }
        },
        itemStyle: {
            color,
            borderWidth: LINE_GRAPH_ITEM_BORDER_WIDTH,
            borderColor: 'transparent'
        },
        name,
        data
    };
}

function makeItemAndSetReference(comparables, makeItemFunc) {
    let referenceItem = null;
    const items = comparables.map((comparable) => {
        const item = makeItemFunc(comparable, comparable.displayInfo.color);
        if (comparable.displayInfo.isChampion) {
            referenceItem = item;
        }
        return item;
    });
    return [items, referenceItem];
}

function isComparableWeighted(comparable) {
    return comparable.details && comparable.details.coreParams && !!comparable.details.coreParams.weight
        && (comparable.details.coreParams.weight.weightMethod == "SAMPLE_WEIGHT"
            || comparable.details.coreParams.weight.weightMethod == "CLASS_AND_SAMPLE_WEIGHT");
}

function isExternalModelType(comparable){
    return comparable.modelType === "EXTERNAL";
}

function getLegendMarker(color) {
    return `<span style="display:inline-block;margin-right:5px;border-radius:10px;width:10px;height:10px;background-color:${sanitizeColor(color)};"></span>`;
}

function findSeriesIndexByName(series, name) {
    return series.map(s => s.name).indexOf(name);
}

// Colors in ModelComparison are always represented by their hex color codes (this isn't a general purpose color sanitization utility)
function sanitizeColor(color) {
    if((''+color).match(/^#[a-f0-9]{6}$/i)) {
        return color;
    }
    // Fallback to black
    return '#000000';
}

app.controller("modelComparisonController", function($scope, $controller, $rootScope, DataikuAPI, $stateParams,
    ActiveProjectKey, TopNav, $q, CreateModalFromTemplate, PMLFilteringService, PMLSettings, LLMEvaluationMetrics, $filter,
    ModelComparisonHelpers, FullModelLikeIdUtils, StateUtils, ModelEvaluationUtils, $timeout, StringUtils, ComparablesService) {

    $controller("ModelEvaluationsMetricsHandlingCommon", {$scope, PMLFilteringService, PMLSettings});

    $scope.modelTaskTypes = PMLSettings.task.predictionTypes.filter(type => type.classical || type.forecast || type.causal);
    if ($rootScope.appConfig.licensedFeatures.advancedLLMMeshAllowed) {
        $scope.modelTaskTypes = $scope.modelTaskTypes.concat(LLMEvaluationMetrics.llmModelTaskTypes);
    }

    $scope.uiState = {};
    const stateNameParts = $scope.$state.current.name.split(".");
    $scope.uiState.comparisonPane = stateNameParts[stateNameParts.length-1] || "summary";
    $scope.uiState.notFoundFullIds = [];

    $scope.uiState.colors = window.dkuColorPalettes.discrete[0].colors.filter((x,idx) => idx%2 === 0);



    let resetSavedSettings = function() {
        $scope.savedSettings = angular.copy($scope.modelComparison);
    }

    $scope.setColorPaletteFromComparedModels = function(nv) {
        if (!nv) {
            $scope.uiState.colors = [];
            $scope.uiState.allColors = [];
            return;
        }
        // This is a safety net: ensure colors are real colors (no backend-side validation)
        // Colors must always be escaped anyway when they are inserted into any HTML fragment
        const newColors = nv.filter(cm => cm.isSelected).map(cm => sanitizeColor(cm.color));
        // do not update the array if equal to the new value, so that we do not trigger
        // a change and redraw for noting
        if (!angular.equals($scope.uiState.colors, newColors)) {
            $scope.uiState.colors = newColors;
        }
        const newAllColors = nv.map(cm => sanitizeColor(cm.color));
        // do not update the array if equal to the new value, so that we do not trigger
        // a change and redraw for noting
        if (!angular.equals($scope.uiState.allColors, newAllColors)) {
            $scope.uiState.allColors = newAllColors;
        }
    }


    $q.all([
        DataikuAPI.modelevaluationstores.listWithAccessible($stateParams.projectKey),
        DataikuAPI.savedmodels.listWithAccessible($stateParams.projectKey),
        DataikuAPI.analysis.listHeads($stateParams.projectKey, true),
        DataikuAPI.modelcomparisons.getFullInfo(ActiveProjectKey.get(), $stateParams.modelComparisonId)
    ]).then(function(data) {
        $scope.storeList = data[0].data;
        $scope.modelList = data[1].data;
        $scope.analysesWithHeads = data[2].data;
        $scope.modelComparisonFullInfo = data[3].data;
        $scope.modelComparison = $scope.modelComparisonFullInfo.modelComparison;
        $scope.meLikesCategories = melCategoriesFormodelTaskType($scope.modelComparison.modelTaskType);
        ComparablesService.fetchComparables($stateParams.projectKey, $scope.modelComparison.modelTaskType)
        .then(ret => {
            $scope.comparables = ret.comparableItems;
            $scope.infoMessages = ret.infoMessages;
        });




        $scope.modelTaskTypeName = $scope.modelTaskTypes.find(pt => pt.type == $scope.modelComparison.modelTaskType).fullName;
        $scope.isLLM = LLMEvaluationMetrics.llmModelTaskTypes.map(t => t.type).includes($scope.modelComparison.modelTaskType);

        if ($scope.modelComparison.comparedModels && $scope.modelComparison.comparedModels.length) {
            ModelComparisonHelpers.adjustComparableModelsColors($scope.modelComparison.comparedModels);
            $scope.setColorPaletteFromComparedModels($scope.modelComparison.comparedModels);
        }

        const DEFAULT_X_LABEL = "model:date";
        const DEFAULT_Y_LABELS = ["model:algorithm", "evaluation:date", "evaluationDataset:dataset-name"];
                if (!$scope.modelComparison.displayParams.sortColumn) {
            $scope.modelComparison.displayParams.sortColumn = "";
        }
        if (!$scope.modelComparison.displayParams.xLabel) {
            $scope.modelComparison.displayParams.xLabel = DEFAULT_X_LABEL;
            $scope.modelComparison.displayParams.yLabels = DEFAULT_Y_LABELS;
        }

        if (!$scope.modelComparison.displayParams.graphStyle) {
            $scope.modelComparison.displayParams.graphStyle = "BAR";
        }

        resetSavedSettings()
        TopNav.setItem(TopNav.ITEM_MODEL_COMPARISON, $stateParams.modelComparisonId, {name: $scope.modelComparison.displayName});
    });

    $scope.refreshTimeline = function() {
        DataikuAPI.timelines.getForObject(ActiveProjectKey.get(), "MODEL_COMPARISON", $stateParams.modelComparisonId)
        .success(function(data){
            $scope.objectTimeline = data;
        })
        .error(setErrorInScope.bind($scope));
    };


    $scope.savedSettings = {}

    $scope.dirtySettings = function() {
        return !angular.equals($scope.savedSettings, $scope.modelComparison);
    }

    $scope.save = function() {
        DataikuAPI.modelcomparisons.save($scope.modelComparison)
        .success(() => {
            resetSavedSettings();
        })
        .error(setErrorInScope.bind($scope));
    }

    $scope.$on("saveIfDirty", () => {
        if ($scope.dirtySettings()) {
            $scope.save()
        }
    });

    $scope.uiState.allComparables = [];
    $scope.uiState.comparables = [];

    $scope.computeRefs = function() {
        if (!$scope.uiState.allComparables  || !$scope.modelComparison || !$scope.modelComparison.comparedModels
            || $scope.uiState.allComparables.length != $scope.modelComparison.comparedModels.length) {
            return;
        }

        $scope.metricParamsList = [];
        const newRefs = $scope.modelComparison.comparedModels.map((cm, index) => {
            const comparable = $scope.uiState.allComparables[index];
            let ret;
            if (comparable) {
                ret = ModelEvaluationUtils.makeRefDisplayItemFromComparable(comparable, $scope.storeList, $scope.modelList, $scope.analysesWithHeads);
                ret.metrics = comparable.metrics;
                if (!$scope.isLLM) {
                    $scope.metricParamsList.push(comparable.metricParams);
                }
            } else {
                ret = {
                    ref: {
                        fullId: cm.refId
                    },
                    metrics: [],
                    labels: new Map()
                };
            }
            ret.refStr = comparable?comparable.displayInfo.displayName:cm.refId;
            ret.graphStr = comparable?comparable.displayInfo.championedNameNoSymbol:cm.refId;
            ret.isChampion = cm.isChampion;
            ret.isDisabled = !cm.isSelected;
            return ret;
        });

        // do not update the array if equal to the new value, so that we do not trigger
        // a change and redraw for noting
        if (!angular.equals($scope.uiState.refs, newRefs)) {
            $scope.uiState.refs = newRefs;

            if ($scope.uiState.refs.length) {
                $scope.refreshMetrics($scope.modelComparison.modelTaskType, $scope.getAllCustomMetricNames());

                if ($scope.modelComparison.displayParams.pinnedMetrics.length) {
                    const possibleMetricCodes = $scope.possibleMetrics.map(pm => pm[0]);
                    $scope.modelComparison.displayParams.pinnedMetrics =
                    $scope.modelComparison.displayParams.pinnedMetrics.filter(pm => possibleMetricCodes.includes(pm));
                }
            }
        }
    }

    $scope.handleLoadedComparables = function(loaded, comparedModelsCopy, allComparablesCopy) {
        const resMap = new Map();
        [...loaded,...allComparablesCopy].forEach(cm => {
            if (cm) resMap.set(cm.ref.fullId, cm);
        });
        $scope.uiState.allComparables = comparedModelsCopy.map(
            cm => resMap.get(cm.refId)
        );
        const alreadyUsedNames = [];
        $scope.uiState.allComparables.forEach((cmp,i) =>
            {
                if (!cmp) return;
                cmp.displayInfo = comparedModelsCopy[i];
                cmp.displayInfo.displayName = cmp.userMeta.name;
                const uniqueName = StringUtils.transmogrify(
                    cmp.displayInfo.displayName, alreadyUsedNames,
                    (count, name) => `${name} (${count})`
                );
                alreadyUsedNames.push(uniqueName);
                cmp.displayInfo.championedNameNoSymbol = uniqueName;
                cmp.displayInfo.championedName = (cmp.displayInfo.isChampion?ModelComparisonHelpers.getChampionSymbol()+' ':'') + cmp.displayInfo.championedNameNoSymbol;
                // we can force the model type to prediction here because LLM saved models can't be compared, only LLM evaluations can
                cmp.displayInfo.href = StateUtils.href.fullModelLikeId(cmp.displayInfo.refId, 'PREDICTION', $scope.isLLM);
                let subtitle = undefined;

                if (cmp.details && cmp.details.userMeta.labels && (cmp.ref.modelType === "MODEL_EVALUATION")) {
                    const label = cmp.details.userMeta.labels.find(l => l.key === "model:name");
                    if (label && (cmp.displayInfo.displayName != label.value))  {
                        subtitle = label.value;
                    }
                }
                cmp.displayInfo.subtitle = subtitle;
            });
        $scope.uiState.allComparables.forEach(cm => {
            if (!cm || !cm.details || !cm.details.perf) return;
            const pcd = cm.details.perf.perCutData;
            const costMatrixWeights = cm.metricParams.costMatrixWeights;
            if (pcd && costMatrixWeights) {
                pcd.cmg = pcd.cut.map(function(c, i) {
                    return (costMatrixWeights.fnGain * pcd.fn[i] + costMatrixWeights.tnGain * pcd.tn[i] +
                        costMatrixWeights.fpGain * pcd.fp[i] + costMatrixWeights.tpGain * pcd.tp[i]
                        ) / (pcd.fn[i] + pcd.tn[i] +  pcd.fp[i] + pcd.tp[i]);
                });
            }
        });
    }

    $scope.loadComparables = function() {
        var deferred = $q.defer();
        const comparedModelsCopy = angular.copy($scope.modelComparison.comparedModels);
        const allComparablesCopy = angular.copy($scope.uiState.allComparables);
        let comparedIds = new Set(comparedModelsCopy.filter(cm => !!cm).map(cm => cm.refId))
        let comparableIds = new Set(allComparablesCopy.filter(cm => !!cm).map(cm => cm.ref.fullId))
        let toLoad = new Set([...comparedIds].filter(x => !comparableIds.has(x)));
        $scope.uiState.notFoundFullIds = [];
        if (toLoad.size) {
            const loadRequest = [...toLoad];
            DataikuAPI.modelcomparisons.getModelsDetails(loadRequest).then(function(data) {
                $scope.uiState.notFoundFullIds = _.zip(loadRequest, data.data).filter(([_, resp]) => !resp).map(([req,_]) => req);
                $scope.handleLoadedComparables(data.data, comparedModelsCopy, allComparablesCopy);
                deferred.resolve($scope.uiState.allComparables);
            });
        } else {
            $scope.handleLoadedComparables([], comparedModelsCopy, allComparablesCopy);
            deferred.resolve($scope.uiState.allComparables);
        }
        return deferred.promise;
    };

    $scope.$watch("uiState.allComparables", $scope.computeRefs, true);
    $scope.$watch("modelComparison.comparedModels", $scope.computeRefs, true);

    $scope.$watch("uiState.allModelEvaluationInfos", $scope.computeRefs, true);
    $scope.$watch("uiState.allComparables", $scope.computeRefs, true);
    $scope.$watch("modelComparison.comparedModels", (nv, ov) => {
        if (nv && !nv.length) {
            $scope.uiState.comparisonPane = 'summary';
            StateUtils.go.modelComparison($scope.modelComparison.id, $scope.modelComparison.projectKey);
        }

        $scope.computeRefs(nv,ov);
    }, true);

    $scope.canCompare = (tabName) => {
        if (!$scope.uiState.allComparables || !$scope.uiState.allComparables.length) {
            return false;
        }
        const modelTaskType = $scope.modelComparison.modelTaskType;
        switch (tabName) {
            case 'summary':
                return true;
            case 'decision_chart':
            case 'lift_chart':
                return modelTaskType === "BINARY_CLASSIFICATION";
            case 'calibration_curve':
            case 'roc_curve':
            case 'pr_curve':
            case 'density_chart':
                return ["BINARY_CLASSIFICATION", "MULTICLASS"].includes(modelTaskType);
            case 'scatter_plot':
                return modelTaskType === "REGRESSION";
            case 'ts_resampling':
                return modelTaskType === "TIMESERIES_FORECAST";
            case 'uplift_charts':
                return ["CAUSAL_BINARY_CLASSIFICATION", "CAUSAL_REGRESSION"].includes(modelTaskType);
            case 'feature_importance':
                return !$scope.isLLM && !["CAUSAL_BINARY_CLASSIFICATION", "CAUSAL_REGRESSION", "TIMESERIES_FORECAST"].includes(modelTaskType);
            case 'features':
            case 'algorithm':
            case 'training_information':
                return !$scope.isLLM;
            case 'row_by_row_analysis':
                return $scope.isLLM;
            default:
                return false;
        }
    }

    $scope.editSelection = function() {
        CreateModalFromTemplate("/templates/modelcomparisons/modals/edit-compared-models-selection-modal.html", $scope, "EditComparedModelsSelectionController", function(newScope) {
            newScope.init($scope.modelComparison.modelTaskType, $scope.modelTaskTypeName, $scope.modelComparison.comparedModels, $scope.modelComparison.displayParams,
                $scope.uiState.allComparables, $scope.uiState.missings, $scope.uiState.refs);
        }).then(function([newComparedModels, additionalComparedModels, newDisplayParams]) {
            const renamedModels = newComparedModels.filter(cm => !!cm.originalName);
            if (renamedModels.length) {
                const queries = renamedModels.map(cm => {
                    if (FullModelLikeIdUtils.isAnalysis(cm.refId) || FullModelLikeIdUtils.isSavedModel(cm.refId)) {
                        const comparable = $scope.uiState.allComparables.filter(cmp => cmp.ref.fullId === cm.refId)[0];
                        comparable.userMeta.name = cm.displayName;
                        return DataikuAPI.ml.saveModelUserMeta(cm.refId, comparable.userMeta);
                    } else if (FullModelLikeIdUtils.isModelEvaluation(cm.refId)) {
                        const comparable = $scope.uiState.allComparables.filter(cmp => cmp.ref.fullId === cm.refId)[0];
                        comparable.userMeta.name = cm.displayName;
                        return DataikuAPI.modelevaluations.saveEvaluationUserMeta(cm.refId, comparable.userMeta);
                    }
                });
                $q.all(queries)
                  .then(() => {
                      $scope.modelComparison.comparedModels = newComparedModels;
                      $scope.modelComparison.displayParams = newDisplayParams;
                      renamedModels.forEach(rm => rm.originalName = undefined);
                      if (additionalComparedModels && additionalComparedModels.length) {
                          for (const additionalComparedModel of additionalComparedModels) {
                              if (additionalComparedModel.evaluationMetric && !$scope.modelComparison.displayParams.pinnedMetrics.includes(additionalComparedModel.evaluationMetric)) {
                                  $scope.modelComparison.displayParams.pinnedMetrics.push(additionalComparedModel.evaluationMetric);
                              }
                          }
                      }
                      $scope.loadComparables();
                })
            } else {
                $scope.modelComparison.comparedModels = newComparedModels;
                $scope.modelComparison.displayParams = newDisplayParams;
                if (additionalComparedModels && additionalComparedModels.length) {
                    for (const additionalComparedModel of additionalComparedModels) {
                        if (additionalComparedModel.evaluationMetric && !$scope.modelComparison.displayParams.pinnedMetrics.includes(additionalComparedModel.evaluationMetric)) {
                            $scope.modelComparison.displayParams.pinnedMetrics.push(additionalComparedModel.evaluationMetric);
                        }
                    }
                    $scope.loadComparables();
                }
            }
        });
    }

    function filterSelectComparables() {
        if (!$scope.modelComparison || !$scope.modelComparison.comparedModels
            || !$scope.uiState.allComparables || !$scope.uiState.allComparables.length
             || $scope.uiState.allComparables.length != $scope.modelComparison.comparedModels.length) {
            return;
        }

        const newComparables = [];
        $scope.modelComparison.comparedModels
            .filter(
                cm => !$scope.uiState.notFoundFullIds.includes(cm.refId)
            )
            .forEach((cm) => {
                if (cm.isSelected) {
                    const matches = $scope.uiState.allComparables.filter(cmp => cmp?cmp.ref.fullId === cm.refId:false);
                    if (matches && matches.length) {
                        newComparables.push(matches[0]);
                    }
                }
            });
        $scope.uiState.comparables = newComparables;


    }

    $scope.addMELikes = function(initialState, comparables=null) {
        CreateModalFromTemplate("/templates/modelcomparisons/modals/add-melikes-modal/add-melikes-modal.html", $scope, "AddMELikesToModelComparisonController", function(newScope) {
            newScope.init($scope.modelComparison.comparedModels, $scope.modelComparison.modelTaskType, $scope.modelTaskTypeName, initialState, comparables);
        }).then(function(additionalComparedModels) {
            for (const additionalComparedModel of additionalComparedModels) {
                if (additionalComparedModel.evaluationMetric && !$scope.modelComparison.displayParams.pinnedMetrics.includes(additionalComparedModel.evaluationMetric)) {
                    $scope.modelComparison.displayParams.pinnedMetrics.push(additionalComparedModel.evaluationMetric);
                }
            }
            $scope.modelComparison.comparedModels.push(...additionalComparedModels);
            ModelComparisonHelpers.adjustComparableModelsColors($scope.modelComparison.comparedModels);
            $scope.onModelComparisonChange();
        });
    }

    $scope.onComparedModelsChange = function(nv, ov) {
        filterSelectComparables(nv);
        if (nv && ov && $scope.$root.projectSummary.canWriteProjectContent && !angular.equals(nv, ov)) {
            $timeout($scope.save,0);
        }
        $scope.setColorPaletteFromComparedModels(nv);
    }

    function unselectUnavailableComparables() {
        $scope.modelComparison.comparedModels.forEach(cm => {
            if ($scope.uiState.notFoundFullIds.includes(cm.refId)) {
                cm.isSelected = false;
            }
        });
    }

    $scope.$watchCollection("uiState.allComparables", filterSelectComparables);
    $scope.$watchCollection("uiState.notFoundFullIds", (nv, ov) => {
        if (!nv || !$scope.modelComparison || !$scope.modelComparison.comparedModels) {
            return;
        }
        unselectUnavailableComparables();
        filterSelectComparables(nv, ov);
    });

    $scope.onModelComparisonChange = function () {
        if (!$scope.modelComparison) {
            $scope.uiState.comparables = [];
        } else {
            if ($scope.uiState.comparables) $scope.loadComparables();
        }
    }

   $scope.$watch("modelComparison.comparedModels",(nv, ov) => {$scope.onModelComparisonChange(); $scope.onComparedModelsChange(nv, ov);}, true)

    $scope.computeMissings = function(nv) {
        if (!nv || !nv.length) {
            $scope.uiState.missings = [];
            return;
        }
        $scope.uiState.missings = $scope.modelComparison.comparedModels.filter(
            cm => $scope.uiState.notFoundFullIds.includes(cm.refId)
        );
    }

    $scope.$watch("uiState.notFoundFullIds", $scope.computeMissings);


    $scope.removeNotFound = function() {
        $scope.modelComparison.comparedModels = $scope.modelComparison.comparedModels.filter(
            cm => !$scope.uiState.notFoundFullIds.includes(cm.refId)
        );
    }

    $scope.selectCategory = function(category) {
        $scope.uiState.selectedCategory = category;
        $scope.addMELikes(category.state, $scope.comparables);
    }

    $scope.comparables = {}
});


app.factory('TreeViewSelectorUtils', function() {
    /*
        Helper for constructing a tree from a list of items. Ordering of items is preserved.

        items: {
            groups: string[], // Path of this item in the hierarchy
            payload: T        // Item payload
        }

        leafFn: T=>leaf node  // Function to the leaf node from item's payload
    */
    function treeify(items, leafFn) {
        const itemsWithIndex = items.map((item, index) => ({ ...item, index }));
        const groups = _.groupBy(itemsWithIndex, item => item.groups[0] || '');
        const nodes = [];
        const selectedNodes = [];
        const expandedNodes = [];

        for(const groupName in groups) {
            if(!groupName) {
                for(const item of groups[groupName]) {
                    const leaf = {...leafFn(item.payload), ordering: item.index}
                    nodes.push(leaf);
                    if(leaf.preselected) {
                        selectedNodes.push(leaf);
                    }
                }
            } else {
                const newItems = groups[groupName].map(item => ({payload: item.payload, groups: item.groups.slice(1)}))
                const [childrenNodes, selectedChildrenNodes, expandedChildNodes] = treeify(newItems, leafFn);
                for(const node of selectedChildrenNodes) {
                    selectedNodes.push(node);
                }
                const firstItemIndexWithinGroup = _.min(groups[groupName].map(item => item.index));
                const newNode = {
                    type: 'parent',
                    title: groupName,
                    groups: [],
                    children: childrenNodes,
                    ordering: firstItemIndexWithinGroup
                };
                nodes.push(newNode);
                if(expandedChildNodes.length > 0 || childrenNodes.some(n => n.type == 'leaf' && n.preexpanded)) {
                    expandedNodes.push(newNode);
                }
                for(const node of expandedChildNodes) {
                    expandedNodes.push(node);
                }
            }
        }
        const sortedNodes = _.sortBy(nodes, node => node.ordering);
        return [sortedNodes, selectedNodes, expandedNodes];
    }

    return { treeify };
})
app.component('treeViewSelector', {
    template: `
        <table ng-if="$ctrl.rows" class="tree-view-selector__table">
            <tr>
                <th ng-repeat="column in $ctrl.columns" class="tree-view-selector__head">
                    {{ column }}
                </th>
            </tr>
            <tr ng-repeat="row in $ctrl.rows"
                class="tree-view-selector__row"
                ng-click="$ctrl.toggleRow(row)"
                ng-class="{
                    'tree-view-selector__row--parent': row.node.type == 'parent',
                    'tree-view-selector__row--leaf': row.node.type == 'leaf',
                    'tree-view-selector__row--disabled': row.allDisabled
                }"
            >
                <td
                    ng-if="row.node.type == 'parent'"
                    colspan="{{ $ctrl.columns.length }}"
                    class="tree-view-selector__parent"
                    >
                    <div ng-style="{'padding-left': (row.level * 30 + 10) + 'px' }" class="faic">
                        <input
                            type="checkbox"
                            ng-if="$ctrl.multiselect"
                            ng-disabled="row.allDisabled"
                            class="tree-view-selector__checkbox"
                            ng-model="row.allSelected"
                            ng-change="$ctrl.toggleSelection(row.node, row.allSelected)"
                            dku-indeterminate="row.onlySomeSelected" stop-propagation>
                        <i class="tree-view-selector__chevron mright4" ng-class="{
                            'dku-icon-chevron-down-12': row.expanded,
                            'dku-icon-chevron-right-12': !row.expanded
                        }"></i>
                        <span data-qa-chevron-title="{{row.node.title}}">{{ row.node.title }}</span>
                    </div>
                </td>
                <td
                    ng-if="row.node.type == 'leaf'"
                    ng-repeat="column in $ctrl.columns"
                    class="tree-view-selector__cell"
                    title="{{row.message}}">
                    <div ng-style="{'padding-left': $index == 0 ? (row.level * 30 + 10) + 'px' : 0 }">
                        <input
                            type="checkbox"
                            ng-if="$ctrl.multiselect && $index == 0"
                            ng-disabled="row.allDisabled"
                            class="tree-view-selector__checkbox"
                            ng-model="row.allSelected"
                            ng-change="$ctrl.toggleSelection(row.node, !row.allSelected)"
                            data-qa-checkbox-id="{{row.node.payload.mli.fullId}}"
                        >
                        <input
                            type="radio"
                            ng-if="!$ctrl.multiselect && $index == 0"
                            ng-disabled="row.allDisabled"
                            class="tree-view-selector__input"
                            ng-model="row.allSelected"
                            ng-value="true"
                            ng-click="$ctrl.toggleSelection(row.node, !row.allSelected)"
                            data-qa-radio-btn-id="{{row.node.payload.mli.fullId}}"
                        >
                        <span>{{ row.node.data[$index] }}</span>
                    </div>
                </td>
            </tr>
        </table>
    `,
    bindings: {
        /*
            type Leaf = {
                type: 'leaf';
                disabled?: boolean, // cannot be selected (or unselected if already selected)
                data: string[];  // content of each column
            }
            type Parent = {
                type: 'parent',
                title: string,
                children: Node[]
            }
            type Node = Leaf | Parent;
            in: treeData: Node | Node[];   // tree data
            in: columns: string[];  // column names
            in: multiselect: boolean // single selection mode (defaults to true)
            in/out: selectedLeafs: Leaf[];  // selected leaves (aka "selected MEs")
            in/out: expandedNodes: Node[];  // expanded nodes (everything collapsed if empty)
        */
        treeData: '<',
        columns: '<',
        multiselect: '<?',
        selectedLeafs: '=',
        expandedNodes: '='
    },
    controller: function() {
        const $ctrl = this;

        function flattenLeafs(node) {
            return node.type == 'parent' ? node.children.flatMap(n => flattenLeafs(n)) : [node];
        }

        $ctrl.toggleSelection = function(node, select) {
            if(select) {
                if($ctrl.multiselect) {
                    $ctrl.selectedLeafs = _.uniq([...($ctrl.selectedLeafs || []), ...flattenLeafs(node).filter(n => !n.disabled)]);
                } else if(node.type == 'leaf' && !node.disabled) {
                    $ctrl.selectedLeafs = [node];
                }
            } else {
                if($ctrl.multiselect) {
                    $ctrl.selectedLeafs = _.difference($ctrl.selectedLeafs || [], flattenLeafs(node).filter(n => !n.disabled));
                } else {
                    // Can't un-select (like a radio button)
                }
            }
            refresh();
        }

        $ctrl.toggleExpansion = function(node, expand) {
            if(expand) {
                const nodesToExpand = [];
                while(node && node.type == 'parent') {
                    nodesToExpand.push(node);
                    if(node.children.length == 1) {
                        // Also expand child nodes recursively if they have no siblings
                        node = node.children[0];
                    } else {
                        node = null;
                    }
                }
                $ctrl.expandedNodes = _.uniq(($ctrl.expandedNodes || []).concat(nodesToExpand));
            } else {
                $ctrl.expandedNodes = _.difference($ctrl.expandedNodes || [], [node]);
            }
            refresh();
        }

        $ctrl.toggleRow = function(row) {
            if(row.node.type == 'parent') {
                $ctrl.toggleExpansion(row.node, !row.expanded)
            }
            if(row.node.type == 'leaf' && !row.node.disabled) {
                $ctrl.toggleSelection(row.node, !row.allSelected)
            }
        }

        function refresh() {
            if(!$ctrl.treeData) {
                return;
            }
            const selectedLeafs = $ctrl.selectedLeafs || [];
            const expandedNodes = $ctrl.expandedNodes || [];

            const rows = [];
            function visitTree(node, level, populate) {
                const expanded = expandedNodes.includes(node);
                const row = { node, level, expanded };
                if(populate) {
                    rows.push(row);
                }
                if(node.type == 'parent') {
                    let nbSelected = 0;
                    let nbTotal = 0;
                    let nbDisabled = 0;
                    for(let child of node.children) {
                        const subTreeStats = visitTree(child, level + 1, populate && expanded);
                        nbSelected += subTreeStats.nbSelected;
                        nbTotal += subTreeStats.nbTotal;
                        nbDisabled += subTreeStats.nbDisabled;
                    }
                    row.allSelected = nbSelected > 0 && nbSelected == nbTotal;
                    row.onlySomeSelected = nbSelected > 0 && nbSelected < nbTotal;
                    row.allDisabled = nbDisabled > 0 && nbDisabled == nbTotal;
                    return { nbTotal, nbSelected, nbDisabled };
                }
                if(node.type == 'leaf') {
                    row.allSelected = selectedLeafs.includes(node);
                    row.onlySomeSelected = false;
                    row.allDisabled = !!node.disabled;
                    row.message = node.message;
                    return {
                        nbSelected: row.allSelected? 1 : 0,
                        nbTotal: 1,
                        nbDisabled: row.allDisabled ? 1 : 0
                    };
                }
            }
            [$ctrl.treeData].flat().forEach(n => visitTree(n, 0, true));
            $ctrl.rows = rows;
        };

        $ctrl.$onChanges = () => refresh();
    }
});

app.component('tabularDataComparator', {
    template: `
        <table ng-if="$ctrl.sections && $ctrl.comparableItems" class="table tabular-data-comparator">
            <thead>
                <tr>
                    <th class="tabular-data-comparator__top-left-corner"></th>
                    <td ng-repeat="item in $ctrl.comparableItems track by $index" 
                        ng-style="{'border-top': item.columnColor ? '6px solid '+ item.columnColor : ''}"
                        class="tabular-data-comparator__header-cell"
                        ng-class="{'tabular-data-comparator__header-cell--reference': item == $ctrl.referenceItem}">
                        <ng-include src="$ctrl.colHeaderTemplate"></ng-include>
                    </td>
                </tr>
            </thead>
            <tbody ng-repeat="section in $ctrl.sections" class="tabular-data-comparator__section">
                <tr ng-if="section.title && $ctrl.sections.length > 1" 
                    class="tabular-data-comparator__section-row">
                    <th class="tabular-data-comparator__section-title-cell" 
                        colspan="{{ $ctrl.comparableItems.length + 1 }}">
                        {{ section.title }}
                    </th>
                </tr>
                <tr ng-repeat="row in section.rows" class="tabular-data-comparator__row">
                    <th class="tabular-data-comparator__label"
                        ng-class="{
                            'tabular-data-comparator__label--highlighted': $ctrl.highlightDifferences && !$ctrl.referenceItem && !row.allEquals
                        }">
                        <ng-include src="$ctrl.rowHeaderTemplate"></ng-include>
                    </th>
                    <td ng-repeat="cell in row.cells"
                        class="tabular-data-comparator__cell" 
                        ng-class="{
                            'tabular-data-comparator__cell--highlighted': $ctrl.highlightDifferences && cell.isHighlighted,
                            'tabular-data-comparator__cell--reference': cell.isReference
                        }"
                    >
                        <ng-include src="$ctrl.cellTemplate"></ng-include>
                    </td>
                </tr>
            </tbody>
        </table>
    `,
    bindings: {
        /*
            comparableItems: {
                columnLabel: string,
                color?: string,
                rows: {
                    // Used to pair rows within the section (via ==)
                    rowIndexKey: string,
                    // Used to detect differences (via deep equality)
                    // (null & empty string are always ignored)
                    rowValueKey: any,
                    // Title of the section (optional)
                    sectionTitle?: string,
                    ...
                }[]
            }[]
        */
        comparableItems: '<',
        // Must be one of 'comparableItems' or null
        referenceItem: '=',
        // Enable or disable sorting of rows (by rowIndexKey)
        sortRows: '<',
        // Templates used to display cells, col headers and row headers
        cellTemplate: '<',
        colHeaderTemplate: '<',
        rowHeaderTemplate: '<',
        highlightDifferences: '<'
    },
    controller: function($scope, $filter, ModelComparisonHelpers, EmbeddingService) {
        const $ctrl = this;
        function getModelTaskType() {
            const firstComparable = $ctrl.comparableItems && ($ctrl.comparableItems[0] || {}).comparable;
            return firstComparable.modelTaskType;
        }

        $scope.getTargetRoleName = function() {
            const isCausalPrediction = ['CAUSAL_BINARY_CLASSIFICATION', 'CAUSAL_REGRESSION'].includes(getModelTaskType());
            return $filter("capitalize")($filter("targetRoleName")(isCausalPrediction));
        };

        function isEmpty(value) {
            return value == null || value === '';
        }

        $scope.championSymbol = ModelComparisonHelpers.getChampionSymbol();
        $scope.embeddingMetadataFetcher = EmbeddingService.metadataFetcher(setErrorInScope.bind($scope));

        $ctrl.redraw = () => {
            if(!$ctrl.comparableItems) {
                $ctrl.rows = null;
                return;
            }

            const sectionTitles = _.uniq($ctrl.comparableItems.flatMap(c => c.rows.map(r => r.sectionTitle)));
            const sections = [];

            for(const sectionTitle of sectionTitles) {
                let rowsIndexKeys = _.uniqBy($ctrl.comparableItems.flatMap(c => {
                    return c.rows.filter(r => r.sectionTitle == sectionTitle)
                        .map(function(r) { return { rowIndexKey: r.rowIndexKey, rowSortKey: r.rowSortKey } });
                }), rowKeys => rowKeys.rowIndexKey);

                if($ctrl.sortRows) {
                    rowsIndexKeys.sort((a, b) => {
                        if (!angular.isUndefined(a.rowSortKey) && !angular.isUndefined(b.rowSortKey)) {
                            return a.rowSortKey - b.rowSortKey;
                        }
                        return a.rowIndexKey.toLowerCase().localeCompare(b.rowIndexKey.toLowerCase());
                    });
                }

                const rows = rowsIndexKeys.map(rowKeys => ({ key: rowKeys.rowIndexKey }));

                for(let row of rows) {
                    const valueKeys = $ctrl.comparableItems
                        .map(c => c.rows.find(c => c.sectionTitle == sectionTitle && c.rowIndexKey == row.key))
                        .map(c => c && c.rowValueKey)
                        .filter(v => !isEmpty(v));
                    row.allEquals = valueKeys.every(v => angular.equals(v, valueKeys[0]));

                    const referenceCell = $ctrl.referenceItem
                        && $ctrl.referenceItem.rows.find(r => r.sectionTitle == sectionTitle && r.rowIndexKey == row.key)

                    row.cells = $ctrl.comparableItems.map(c => {
                        const data = c.rows.find(r => r.sectionTitle == sectionTitle && r.rowIndexKey== row.key);
                        const isReference = c == $ctrl.referenceItem;
                        const differentThanRef = (referenceCell && !isEmpty(referenceCell.rowValueKey) && !data)
                                                ||
                                                (!referenceCell && data && !isEmpty(data.rowValueKey))
                                                ||
                                                (referenceCell && data
                                                && !isEmpty(data.rowValueKey) && !isEmpty(referenceCell.rowValueKey)
                                                && !angular.equals(data.rowValueKey, referenceCell.rowValueKey)
                                                );
                        const isHighlighted = ($ctrl.referenceItem && !isReference && differentThanRef)
                            || ($ctrl.referenceItem && isReference && !row.allEquals)
                            || (!$ctrl.referenceItem && !row.allEquals);
                        const tooltip = (c.comparable.modelType === 'EXTERNAL') ? {
                            toggle: "tooltip",
                            title: "These values are inferred by DSS. They do not reflect how the features were actually handled by the model",
                            container: "body"
                        } : {};
                        return { data, isHighlighted, isReference, tooltip };
                    });
                }
                sections.push({
                    title: sectionTitle,
                    rows: rows
                });
            }
            $ctrl.sections = sections;
        }

        $ctrl.$onChanges = () => $ctrl.redraw();
    }
})

app.component("modelComparisonCalibrationCurveComparator", {
    templateUrl: '/templates/modelcomparisons/comparison-tabs/calibration_curve.html',
    bindings: {
        comparables: '<',
        colors: '<'
    },
    controller: function(ModelComparisonHelpers, PMLSettings, $filter, StateUtils) {
        const $ctrl = this;

        $ctrl.$onInit = function() {
            $ctrl.StateUtils = StateUtils;

            $ctrl.selectedClass = null;

            $ctrl.championSymbol = ModelComparisonHelpers.getChampionSymbol();
        }

        $ctrl.$onChanges = function () {
            $ctrl.classes = ModelComparisonHelpers.listOneVsAllClassesAmongComparables($ctrl.comparables);
            if($ctrl.classes.length > 0) {
                $ctrl.selectedClass = $ctrl.classes[0].mainClass;
            }
            $ctrl.updateCalibrationLosses();
            $ctrl.updateChart();
        }

        $ctrl.updateCalibrationLosses = function() {
            $ctrl.calibrationLossesMap = $ctrl.comparables.reduce((mapAccumulator, comparable) => {
                const perf = comparable.details.perf;
                if (perf) {
                    if (comparable.details.coreParams.prediction_type === 'MULTICLASS') {
                        mapAccumulator[comparable.ref.fullId] = perf.oneVsAllCalibrationLoss[$ctrl.selectedClass];
                    } else {
                        mapAccumulator[comparable.ref.fullId] = perf.tiMetrics.calibrationLoss;
                    }
                }
                return mapAccumulator;
            }, {});

        }

        $ctrl.updateChart = function() {
            $ctrl.chartOptions = buildChartOptions();
        }

        $ctrl.getCalibrationLoss = function(comparable) {
            const perf = comparable.details.perf;
            if (perf) {
                if(comparable.details.coreParams.prediction_type === 'BINARY_CLASSIFICATION' && perf && perf.tiMetrics) {
                    return perf.tiMetrics.calibrationLoss;
                } else if(comparable.details.coreParams.prediction_type === 'MULTICLASS' && $ctrl.selectedClass && perf.oneVsAllCalibrationLoss
                    && perf.oneVsAllCalibrationLoss[$ctrl.selectedClass]) {
                    return perf.oneVsAllCalibrationLoss[$ctrl.selectedClass];
                }
            }
            return "-";
        }

        $ctrl.getCalibrationMethodDisplayName = function(calibrationMethodId) {
            const calibrationMethod = PMLSettings.task.calibrationMethods.find((method) => method[0] === calibrationMethodId);
            return calibrationMethod ? calibrationMethod[1] : calibrationMethodId;
        }

        function buildChartOptions() {
            const series = [];
            const lossMap = new Map();

            const seriesWeighted = new Map();
            $ctrl.comparables.forEach((comparable, idx) => {
                const perf = comparable.details.perf;
                if(!perf) {
                    return;
                }
                const calibrationLoss = $filter('nicePrecision')($ctrl.calibrationLossesMap[comparable.ref.fullId],4);
                const name = comparable.displayInfo.championedName;
                lossMap.set(name, calibrationLoss);
                let curveData;
                // Binary classification
                if(comparable.details.coreParams.prediction_type === 'BINARY_CLASSIFICATION' && perf.calibrationData
                    && ModelComparisonHelpers.listOneVsAllClasses(comparable)[0].mainClass == $ctrl.selectedClass) {
                        curveData = perf.calibrationData;
                }
                // Multiclass classification
                else if(comparable.details.coreParams.prediction_type === 'MULTICLASS' && perf.oneVsAllCalibrationCurves) {
                    curveData = perf.oneVsAllCalibrationCurves[$ctrl.selectedClass];
                }

                // 'curveData' can be undefined if classes don't exactly match
                if(curveData) {
                    const data = curveData.map(d => [d.x, d.y, d.n]);
                    const weighted = isComparableWeighted(comparable)?' (Weighted)':'';
                    seriesWeighted.set(name, weighted);
                    series.push(Object.assign({}, getStdSeriesLine($ctrl.colors[idx], name, data),{
                        tooltip: {
                            formatter: (param) => {
                                return  `${param.marker}<b>${sanitize(comparable.displayInfo.displayName)}${sanitize(weighted)}</b><br>`
                                    + 'Frequency of positives: ' + sanitize(Math.round(param.data[1] * 100)) + '%<br>'
                                    + 'Average probability of predicted positive: ' + sanitize(Math.round(param.data[0] * 100)) + '%<br>'
                                    + 'Test elements in bin: ' + sanitize(param.data[2]) + '<br>';
                            }
                        }
                    }));
                }
            });

            const perfectModelName = 'Perfectly calibrated model';
            seriesWeighted.set(perfectModelName,'');
            series.push({
                name: 'Perfectly calibrated model',
                type: 'line',
                itemStyle: {
                    color: 'blue'
                },
                lineStyle: {
                    color: 'blue',
                    width: 1,
                    type: 'dashed'
                },
                data: [[0, 0], [1, 1]]
            })

             const LEGEND_WIDTH = 180;

            return {
                legend: {
                    type: 'scroll',
                    show: true,
                    icon: LINE_GRAPH_SYMBOL,
                    orient: 'vertical',
                    x: 'right',
                    y: 'center',
                    textStyle: {
                        lineOverflow: 'truncate',
                        overflow: 'truncate',
                        width: LEGEND_WIDTH
                    },
                    tooltip: {
                        show: true,
                        trigger: 'item',
                        formatter: (params) => {
                            const seriesIndex = findSeriesIndexByName(series,params.name);
                            const marker = getLegendMarker($ctrl.colors[seriesIndex])
                            return `${marker}${sanitize(params.name)}${sanitize(seriesWeighted.get(params.name))}<br/><b>Loss:</b> ${sanitize(lossMap.get(params.name))}`;
                        }
                    }
                },
                animation: false,
                xAxis: {
                    type: 'value',
                    min: 0,
                    max: 1,
                    name: 'Average of Predicted Probability for Positive Class',
                    nameLocation: 'middle',
                    nameGap: 20,
                    splitNumber: 10,
                    axisLabel: {
                        formatter: function(value) {
                            const numValue = parseFloat(value)*100;
                            return sanitize(`${numValue}%`);
                        }
                    }
                },
                yAxis: {
                    type: 'value',
                    name: 'Frequency of positive class',
                    nameLocation: 'middle',
                    min: 0,
                    max: 1,
                    nameGap: 40,
                    splitNumber: 10
                },
                tooltip: {
                    confine: true,
                    trigger: 'item',
                    formatter: (params) => {
                        let ret = `<b>${sanitize($filter('nicePrecision')(params.value[0],2)*100)}%</b><br/>`
                        ret += `${params.marker}${sanitize(params.seriesName)}: <b>${sanitize($filter('nicePrecision')(params.value[1],2)*100)}%</b>`;
                        return ret;
                    },
                    rich: {
                        bold: {
                            fontWeight: 'bold'
                        }
                    }
                },
                series,
                grid: {
                    right: `${LEGEND_WIDTH+40}px`
                },
                axisPointer: {
                    show: true,
                    snap: true,
                    lineStyle: {
                        type: "dashed"
                    },
                    triggerTooltip: false
                }
            };
        }



    }
});

app.service('ModelComparisonHelpers', function(FullModelLikeIdUtils, DataikuAPI, $q, $filter, CustomMetricIDService, PMLSettings, PMLFilteringService) {
    // Returns a Promise<String[]>
    function lookupEvaluationsOfSavedModelVersions(projectKey, modelTaskType, smvFullIds) {
        return DataikuAPI.modelcomparisons.browseComparables(projectKey, 'FROM_MES', modelTaskType, smvFullIds)
            .then(({data})=> data.comparableModelItems.map(me => me.mli.fullId));
    }

    // Returns a Promise<String[]>
    function lookupSavedModelVersionsFromTheirEvaluations(meFullIds) {
        return DataikuAPI.modelcomparisons.getModelsDetails(meFullIds)
            .then((data)=> data.data.map(cm => cm.details? cm.details.fullModelId : null))
            .then(ids => _.compact(ids));
    }

    function getChampionDataElem(value) {
        return {
            label: {
                show: true,
                position: 'inside',
                formatter: getChampionSymbol(),
            },
            value
        };
    }

    function getChampionBarElem() {
        return {
            show: true,
            position: 'insideTop',
            formatter: getChampionSymbol(),
        };
    }

    function getChampionSymbol() {
        return '🏅';
    }

    function listOneVsAllClasses(comparable) {
        const target_remapping = comparable.details.preprocessing.target_remapping;
        const allClasses = (target_remapping || []).map(tr => tr.sourceValue);
        return (target_remapping || []).flatMap(tr => {
            // In binary, we only show class1 VS class0
            if(comparable.details.coreParams.prediction_type === 'BINARY_CLASSIFICATION' && tr.mappedValue == 0) {
                return [];
            }
            const otherClasses = allClasses.filter(c => c != tr.sourceValue);
            return [{
                // Label showed to user
                label: tr.sourceValue + ' VS ' + allClasses.filter(c => c != tr.sourceValue).join(', '),
                // Deduplication key
                key: JSON.stringify([tr.sourceValue, otherClasses]),
                // The considered class
                mainClass: tr.sourceValue,
                // The other classes
                otherClasses: otherClasses
            }]
        });
    }

    function listOneVsAllClassesAmongComparables(comparables) {
        return _.uniqBy(comparables.flatMap(comparable => listOneVsAllClasses(comparable)), item => item.key);
    }

    function getOriginStyle(item) {
        const id = item.refId ? item.refId : item.ref.fullId;
        if (FullModelLikeIdUtils.isAnalysis(id)) {
            return "icon analysis icon-dku-nav_analysis universe-color";
        } else if (FullModelLikeIdUtils.isSavedModel(id)) {
            const computedIcon = $filter("savedModelSubtypeToIcon")(item.taskType, item.backendType, item.modelTaskType, item.savedModelType, item.proxyModelProtocol, 24);
            return "icon universe-color saved-model " + computedIcon;
        } else if (FullModelLikeIdUtils.isModelEvaluation(id)) {
            return "icon icon-model-evaluation-store universe-color model-evaluation-store";
        } else {
            return "";
        }
    }

    function getOriginTooltip(item) {
        if (FullModelLikeIdUtils.isAnalysis(item.ref.fullId)) {
            return item.mlTaskName;
        } else if (FullModelLikeIdUtils.isSavedModel(item.ref.fullId)) {
            return item.smName;
        } else if (FullModelLikeIdUtils.isModelEvaluation(item.ref.fullId)) {
            return item.storeName;
        } else {
            return "";
        }
    }

    const dkuPalette = window.dkuColorPalettes.discrete[0].colors.filter((x,idx) => idx%2 === 0);
    function getPalette() {
        return dkuPalette;
    }

    function adjustComparableModelsColors(comparedModels) {
        const colorPaletteUsages = new Array(dkuPalette.length).fill(0);
        comparedModels.forEach(cm => {
            const curColorPaletteIdx = dkuPalette.indexOf(cm.color);
            if (curColorPaletteIdx >= 0) {
                colorPaletteUsages[curColorPaletteIdx]++;
            }
        });

        comparedModels.forEach(cm => {
            if (!cm.color) {
                const minUsages = Math.min(...colorPaletteUsages);
                const newColorIdx = colorPaletteUsages.indexOf(minUsages);
                colorPaletteUsages[newColorIdx]++;
                cm.color = dkuPalette[newColorIdx];
            }
        });
    }

    function addPinnedMetricsInComparisonTables(comparable, pinnedMetrics, rows) {
        if(comparable && comparable.details && comparable.details.perf) {
            for(const pinnedMetric of pinnedMetrics || []) {
                let value;
                try {
                    // May fail if the metric doesn't exist on this ME-like
                    value = PMLFilteringService.getMetricValueFromModelWithCurrentCut(comparable.details, pinnedMetric);
                } catch (e) {
                    value = null;
                }
                if(value != null) {
                    let label;
                    if (CustomMetricIDService.checkMetricIsCustom(pinnedMetric)) {
                        label = CustomMetricIDService.getCustomMetricName(pinnedMetric);
                    } else {
                        label = PMLSettings.names.evaluationMetrics[pinnedMetric];
                    }
                    rows.push({
                        rowIndexKey: label,
                        sectionTitle: 'Performance metrics',
                        // value can be a string in some cases, format otherwise
                        value: typeof(value) == 'string' ? value : $filter('mlMetricFormat')(value, pinnedMetric, 3),
                        rowValueKey: value
                    });
                    if (pinnedMetric === 'CUMULATIVE_LIFT') {
                        const liftPoint = comparable.details.modeling && comparable.details.modeling.metrics && comparable.details.modeling.metrics.liftPoint;
                        rows.push({
                            rowIndexKey: "Lift point",
                            sectionTitle: 'Performance metrics',
                            value: (liftPoint * 100) + '%',
                            rowValueKey: value
                        });
                    }
                    if (pinnedMetric === 'NET_UPLIFT') {
                        const netUpliftPoint = comparable.details.modeling && comparable.details.modeling.metrics && comparable.details.modeling.metrics.netUpliftPoint;
                        rows.push({
                            rowIndexKey: "Net uplift point",
                            sectionTitle: 'Performance metrics',
                            value: (netUpliftPoint * 100) + '%',
                            rowValueKey: value
                        });
                    }
                    if (pinnedMetric === 'COST_MATRIX') {
                        const cmg = comparable.details.modeling && comparable.details.modeling.metrics && comparable.details.modeling.metrics.costMatrixWeights;
                        const userFriendlyCmg = Object.entries(cmg).map(function(entry) {
                            const [ name, value ] = entry;
                            if (name.startsWith("tp")) return "True positive: " + value;
                            if (name.startsWith("tn")) return "True negative: " + value;
                            if (name.startsWith("fp")) return "False positive: " + value;
                            if (name.startsWith("fn")) return "False negative: " + value;
                        });
                        rows.push({
                            rowIndexKey: "Cost Matrix weights",
                            sectionTitle: 'Performance metrics',
                            value: userFriendlyCmg.join(" | "),
                            rowValueKey: cmg
                        });
                    }
                }
            }
        }
    }

    return { getChampionDataElem, getChampionBarElem, getChampionSymbol, listOneVsAllClasses,
        listOneVsAllClassesAmongComparables, getOriginStyle, getOriginTooltip,
        lookupEvaluationsOfSavedModelVersions, lookupSavedModelVersionsFromTheirEvaluations,
        adjustComparableModelsColors, getPalette, addPinnedMetricsInComparisonTables };
});

app.component("modelComparisonTrainingInformationComparator", {
    templateUrl: "/templates/modelcomparisons/comparison-tabs/training_information.html",
    bindings: {
        comparables: '<',
        colors: '<',
    },
    controller: function(PMLCVParamsUtils, $filter) {
        const $ctrl = this;

        function makeStepItem(comparable, color) {
            const rows = [];
            if(comparable.details.trainInfo && comparable.details.trainInfo.progress && comparable.details.trainInfo.progress.top_level_done) {
                for(let step of comparable.details.trainInfo.progress.top_level_done) {
                    rows.push({
                        rowIndexKey: step.name,
                        value: step.time,
                        rowValueKey: step.time,
                        sectionTitle: 'Steps'
                    });
                }

                rows.push({
                    rowIndexKey: 'Preprocessing time',
                    value: comparable.details.trainInfo.preprocessingTime,
                    rowValueKey: comparable.details.trainInfo.preprocessingTime,
                    sectionTitle: 'Global'
                });

                rows.push({
                    rowIndexKey: 'Train time',
                    value: comparable.details.trainInfo.trainingTime,
                    rowValueKey: comparable.details.trainInfo.trainingTime,
                    sectionTitle: 'Global'
                });
            }

            return {
                columnColor: color,
                comparable: comparable,
                rows
            }
        }

        function makeCustomTrainTestSplitItem(comparable, color) {
            const rows = [];
            const interval = comparable.details.coreParams.customTrainTestSplit && comparable.details.coreParams.customTrainTestIntervals && comparable.details.coreParams.customTrainTestIntervals[0] || undefined;
            rows.push({
                rowIndexKey: "Train start",
                value: interval && interval["train"][0] || "N/A",
                rowValueKey: interval && interval["train"][0] || "N/A"
            });
            rows.push({
                rowIndexKey: "Train end",
                value: interval && interval["train"][1] || "N/A",
                rowValueKey: interval && interval["train"][1] || "N/A"
            });
            rows.push({
                rowIndexKey: "Test start",
                value: interval && interval["test"][0] || "N/A",
                rowValueKey: interval && interval["test"][0] || "N/A"
            });
            rows.push({
                rowIndexKey: "Test end",
                value: interval && interval["test"][1] || "N/A",
                rowValueKey: interval && interval["test"][1] || "N/A",
            });
            return {
                columnColor: color,
                comparable: comparable,
                rows
            }
        }

        function makeTrainValidationPolicyItem(comparable, color) {
            const rows = [];
            if(comparable.details.splitDesc) {
                const cvParams = PMLCVParamsUtils.makeCVParams(comparable.details.splitDesc.params, comparable.details.coreParams, comparable.details.modeling.grid_search_params);
                for(let param of cvParams) {
                    rows.push({
                        rowIndexKey: $filter("capitalize")(param[0]),
                        value: param[1],
                        rowValueKey: param[1]
                    });
                }
            }
            return {
                columnColor: color,
                comparable: comparable,
                rows
            }
        }

        function makeTrainTestSetsItem(comparable, color) {
            const rows = [];

            if(comparable.details.splitDesc && comparable.details.trainInfo) {
                rows.push({
                    rowIndexKey: 'Generated on',
                    value: comparable.details.splitDesc.generationDate,
                    rowValueKey: comparable.details.splitDesc.generationDate,
                    isDate: true
                });

                if(comparable.details.trainInfo.fullRows) {
                    rows.push({
                        rowIndexKey: 'Rows',
                        value: comparable.details.trainInfo.fullRows,
                        rowValueKey: comparable.details.trainInfo.fullRows,
                    });
                } else {
                    rows.push({
                        rowIndexKey: 'Train set rows',
                        value: comparable.details.trainInfo.trainRows,
                        rowValueKey: comparable.details.trainInfo.trainRows,
                    });

                    rows.push({
                        rowIndexKey: 'Test set rows',
                        value: comparable.details.trainInfo.testRows,
                        rowValueKey: comparable.details.trainInfo.trainRows,
                    });
                }
            }

            return {
                columnColor: color,
                comparable: comparable,
                rows
            }
        }

        $ctrl.$onChanges = function () {
            [$ctrl.stepItems, $ctrl.stepItemsRef] = makeItemAndSetReference($ctrl.comparables, makeStepItem);
            [$ctrl.trainValidationPolicyItems, $ctrl.trainValidationPolicyItemsRef] = makeItemAndSetReference($ctrl.comparables, makeTrainValidationPolicyItem);
            if ($ctrl.comparables.some(c => c.details.coreParams.customTrainTestSplit)) {
                [$ctrl.customTrainTestSplitItems, $ctrl.customTrainTestSplitItemsRef] = makeItemAndSetReference($ctrl.comparables, makeCustomTrainTestSplitItem);
            } else {
                [$ctrl.customTrainTestSplitItems, $ctrl.customTrainTestSplitItemsRef] = [[], []];
            }
            [$ctrl.trainTestSetsItems, $ctrl.trainTestSetsItemsRef] = makeItemAndSetReference($ctrl.comparables, makeTrainTestSetsItem);
        }
    }
})

app.component("modelComparisonAlgorithmComparator", {
    templateUrl: '/templates/modelcomparisons/comparison-tabs/algorithm.html',
    bindings: {
        pinnedMetrics: '<',
        comparables: '<',
        colors: '<'
    },
    controller: function(PMLSettings, TimeseriesForecastingUtils, ModelComparisonHelpers, $filter) {
        const $ctrl = this;

        function makeTrainingData(comparable, color) {
            const rows = [];

            // If the model is external, it means we have no training data and we should not invent nor use default values.
            if(comparable.modelType == "EXTERNAL") {
                return {
                    columnColor: color,
                    comparable: comparable,
                    rows
                };
            }

            if(comparable.details.splitDesc) {
                const value = comparable.details.splitDesc.trainRows > 0 ? comparable.details.splitDesc.trainRows : comparable.details.splitDesc.fullRows;

                if(value > 0) {
                    rows.push({
                        rowIndexKey: 'Rows (before preprocessing)',
                        value: value,
                        rowValueKey: value
                    });
                }
            }

            if(comparable.details.iperf && comparable.details.iperf.modelInputNRows) {
                rows.push({
                    rowIndexKey: 'Rows (after preprocessing)',
                    value: comparable.details.iperf.modelInputNRows,
                    rowValueKey: comparable.details.iperf.modelInputNRows
                });
            }

            if(comparable.details.splitDesc) {
                const value = comparable.details.splitDesc.schema.columns.length;

                rows.push({
                    rowIndexKey: 'Columns (before preprocessing)',
                    value: value,
                    rowValueKey: value
                });
            }

            if(comparable.details.iperf && comparable.details.iperf.modelInputNCols) {
                const value = comparable.details.iperf.modelInputNCols;

                rows.push({
                    rowIndexKey: 'Columns (after preprocessing)',
                    value: value,
                    rowValueKey: value
                });
            }

            if(comparable.details.iperf) {
                const value = comparable.details.iperf.modelInputIsSparse ? 'sparse' : 'dense';

                rows.push({
                    rowIndexKey: 'Matrix type',
                    value: value,
                    rowValueKey: value
                });
            }

            if(comparable.details.coreParams) {
                if(isComparableWeighted(comparable)) {
                    const coreParams = comparable.details.coreParams;
                    rows.push({
                        rowIndexKey: 'Sample weights variable',
                        value: coreParams.weight.sampleWeightVariable,
                        rowValueKey: coreParams.weight.sampleWeightVariable
                    });
                }
            }

            if(comparable.details.iperf && comparable.details.iperf.modelInputMemory) {
                rows.push({
                    rowIndexKey: 'Estimated memory usage',
                    isFileSize: true,
                    value: comparable.details.iperf.modelInputMemory,
                    rowValueKey: comparable.details.iperf.modelInputMemory
                });
            }

            return {
                columnColor: color,
                comparable: comparable,
                rows
            };
        }

        function makeHPItems(comparable, color) {
            let rows = [];

            ModelComparisonHelpers.addPinnedMetricsInComparisonTables(comparable, $ctrl.pinnedMetrics, rows);

            if(comparable && comparable.details && comparable.details.coreParams && comparable.details.coreParams.predictionLength) {
                const coreParams = comparable.details.coreParams;
                const forecastHorizon = coreParams.predictionLength * coreParams.timestepParams.numberOfTimeunits;
                rows.push({
                    rowIndexKey: 'Forecast horizon',
                    sectionTitle: 'Forecast horizon',
                    value: TimeseriesForecastingUtils.prettyTimeSteps(forecastHorizon, coreParams.timestepParams.timeunit)
                });
            }

            if(comparable
                && comparable.details
                && comparable.details.actualParams
                && comparable.details.actualParams.resolved) {
                    const resolved = comparable.details.actualParams.resolved;

                if (resolved.causal_method === 'META_LEARNER') {
                    rows.push({
                        rowIndexKey: 'Meta-learner',
                        sectionTitle: 'Model',
                        value: $filter('niceConst')(resolved.meta_learner, "-"),
                        rowValueKey: resolved.meta_learner
                    });
                }

                // Add algorithm & HP-search related infos
                const algorithm = resolved.algorithm;
                rows.push({
                    rowIndexKey: resolved.causal_method === 'META_LEARNER' ? 'Base learner' : 'Algorithm',
                    sectionTitle: 'Model',
                    value: $filter('niceConst')(algorithm),
                    rowValueKey: algorithm
                });

                // Add HP-search infos if HP search indeed occurred, aka grid size > 0
                const modeling = comparable.details.modeling;
                if(modeling && comparable.details.iperf && comparable.details.iperf.gridSize) {
                    if(['TIME_SERIES_SINGLE_SPLIT', 'SHUFFLE'].includes(modeling.grid_search_params.mode)) {
                        const value = 'Simple train/validation split';
                        rows.push({
                            rowIndexKey: 'Mode',
                            rowSortKey: 1,
                            sectionTitle: 'Hyperparameter search',
                            value,
                            rowValueKey: value
                        });
                    }
                    if(['KFOLD', 'TIME_SERIES_KFOLD'].includes(modeling.grid_search_params.mode)) {
                        const value = modeling.grid_search_params.nFolds + '-fold cross-test';
                        rows.push({
                            rowIndexKey: 'Mode',
                            rowSortKey: 1,
                            sectionTitle: 'Hyperparameter search',
                            value,
                            rowValueKey: value
                        });
                    }

                    if(modeling.grid_search_params.mode == 'CUSTOM') {
                        const value = 'Custom code';
                        rows.push({
                            rowIndexKey: 'Mode',
                            rowSortKey: 1,
                            sectionTitle: 'Hyperparameter search',
                            value,
                            rowValueKey: value
                        });
                    }

                    if(['TIME_SERIES_SINGLE_SPLIT', 'SHUFFLE'].includes(modeling.grid_search_params.mode)) {
                        const value = modeling.grid_search_params.splitRatio;
                        rows.push({
                            rowIndexKey: 'Split ratio',
                            rowSortKey: 2,
                            sectionTitle: 'Hyperparameter search',
                            value,
                            rowValueKey: value
                        });
                    }

                    if (comparable.details.coreParams.prediction_type === 'TIMESERIES_FORECAST' && comparable.details.splitDesc.params.kfold
                            && modeling.grid_search_params.mode === 'TIME_SERIES_KFOLD') {
                        const value = modeling.grid_search_params.foldOffset ? 'Yes' : 'No';
                        rows.push({
                            rowIndexKey: 'Fold offset',
                            rowSortKey: 2,
                            sectionTitle: 'Hyperparameter search',
                            value,
                            rowValueKey: value
                        });
                    }

                    if (comparable.details.coreParams.prediction_type === 'TIMESERIES_FORECAST' && modeling.grid_search_params.mode === 'TIME_SERIES_KFOLD') {
                        const value = comparable.details.modeling.grid_search_params.equalDurationFolds ? 'Yes' : 'No';
                        rows.push({
                            rowIndexKey: 'Equal duration train set folds',
                            rowSortKey: 3,
                            sectionTitle: 'Hyperparameter search',
                            value,
                            rowValueKey: value
                        });
                    }

                    if(modeling.grid_search_params.mode != 'CUSTOM'
                        && ['BINARY_CLASSIFICATION', 'MULTICLASS', 'CAUSAL_BINARY_CLASSIFICATION'].includes(comparable.details.coreParams.prediction_type)) {
                        const value = modeling.grid_search_params.stratified ? 'Yes' : 'No';
                        rows.push({
                            rowIndexKey: 'Stratified',
                            rowSortKey: 3,
                            sectionTitle: 'Hyperparameter search',
                            value,
                            rowValueKey: value
                        });
                    }

                    if(modeling.grid_search_params.mode == 'KFOLD') {
                        const valueGrouped = modeling.grid_search_params.grouped ? 'Yes' : 'No';
                        rows.push({
                            rowIndexKey: 'Grouped',
                            rowSortKey: 4,
                            sectionTitle: 'Hyperparameter search',
                            value: valueGrouped,
                            rowValueKey: valueGrouped
                        });

                        if (modeling.grid_search_params.grouped) {
                            const valueGroupColumnName = modeling.grid_search_params.groupColumnName;
                            rows.push({
                                rowIndexKey: 'Group column',
                                rowSortKey: 5,
                                sectionTitle: 'Hyperparameter search',
                                value: valueGroupColumnName,
                                rowValueKey: valueGroupColumnName
                            });
                        }
                    }
                }

                if (modeling) {
                    // LightGBM and XGBoost early-stopping is disabled in the last run of HP search
                    // so resolved params are always False and 0, so we get them from modeling instead
                    if (['XGBOOST_CLASSIFICATION', 'XGBOOST_REGRESSION'].includes(modeling.algorithm)) {
                        resolved.xgboost.enable_early_stopping = modeling.xgboost_grid.enable_early_stopping;
                        resolved.xgboost.early_stopping_rounds = modeling.xgboost_grid.early_stopping_rounds;
                    } else if (['LIGHTGBM_CLASSIFICATION', 'LIGHTGBM_REGRESSION'].includes(modeling.algorithm)) {
                        const grid = modeling.algorithm === 'LIGHTGBM_CLASSIFICATION' ? 'lightgbm_classification_grid': 'lightgbm_regression_grid';
                        resolved.lightgbm.early_stopping = modeling[grid].early_stopping;
                        resolved.lightgbm.early_stopping_rounds = modeling[grid].early_stopping_rounds;
                    }
                }

                // Add all hyperparameters
                rows = rows.concat(
                    Object.keys(resolved)
                    .filter(k => typeof resolved[k] == 'object')
                    .flatMap(algorithmKey => {
                        const hyperparams = resolved[algorithmKey];
                        return Object.keys(hyperparams).filter(_ =>  typeof hyperparams[_] !== 'object').map(hpKey => ({
                            rowIndexKey: PMLSettings.hpPrettyName(algorithmKey, hpKey),
                            value: hyperparams[hpKey],
                            rowValueKey: hyperparams[hpKey],
                            sectionTitle: 'Hyperparameters',
                        }))
                    })
                );
            }
            return {
                columnColor: color,
                comparable: comparable,
                rows
            }
        }

        $ctrl.$onChanges = function () {
            [$ctrl.hpItems, $ctrl.hpItemsRef] = makeItemAndSetReference($ctrl.comparables, makeHPItems);
            [$ctrl.trainingDataItems, $ctrl.trainingDataItemsRef] = makeItemAndSetReference($ctrl.comparables, makeTrainingData);
        }
    }
});

app.component("modelComparisonTimeseriesResamplingComparator", {
    templateUrl: '/templates/modelcomparisons/comparison-tabs/resampling.html',
    bindings: {
        comparables: '<',
        colors: '<',
        pinnedMetrics: '<'
    },
    controller: function(TimeseriesForecastingUtils) {
        const $ctrl = this;

        function makeTimeSeriesResamplingItem(comparable, color) {
            const rows = [];

            if(comparable && comparable.details) {
                const timestepParams = comparable.details.coreParams && comparable.details.coreParams.timestepParams;
                if (timestepParams) {
                    const timeStep = TimeseriesForecastingUtils.prettyTimeSteps(timestepParams.numberOfTimeunits, timestepParams.timeunit);
                    rows.push({
                        sectionTitle: 'Resampling time step',
                        rowIndexKey: 'Time step',
                        value: timeStep,
                        rowValueKey: timeStep,
                    });

                    if (timestepParams.timeunit === 'WEEK') {
                        rows.push({
                            sectionTitle: 'Resampling time step',
                            rowIndexKey: 'Day of week',
                            value: getDayLabels(timestepParams.endOfWeekDay-1),
                            rowValueKey: timestepParams.endOfWeekDay,
                        });
                    } else if (['MONTH', 'QUARTER', 'HALF_YEAR', 'YEAR'].includes(timestepParams.timeunit)) {
                        const timestepDate = TimeseriesForecastingUtils.prettySelectedDate(timestepParams.timeunit, timestepParams.monthlyAlignment, timestepParams.unitAlignment);
                        rows.push({
                            sectionTitle: 'Resampling time step',
                            rowIndexKey: 'Selected day',
                            value: timestepDate,
                            rowValueKey: timestepDate,
                        });
                    }
                }

                const timeseriesSampling = comparable.details.preprocessing && comparable.details.preprocessing.timeseriesSampling;
                if (timeseriesSampling) {
                    const numericalInterpolateMethod = TimeseriesForecastingUtils.TIMESERIES_IMPUTE_METHODS.find(
                        obj => obj.value === timeseriesSampling.numericalInterpolateMethod
                    );
                    const numericalExtrapolateMethod = TimeseriesForecastingUtils.TIMESERIES_IMPUTE_METHODS.find(
                        obj => obj.value === timeseriesSampling.numericalExtrapolateMethod
                    );
                    const categoricalImputeMethod = TimeseriesForecastingUtils.TIMESERIES_IMPUTE_METHODS.find(
                        obj => obj.value === timeseriesSampling.categoricalImputeMethod
                    );
                    rows.push({
                        sectionTitle: 'Time series resampling',
                        rowIndexKey: 'Numerical interpolation',
                        rowSortKey: 0,
                        value: numericalInterpolateMethod.displayName,
                        rowValueKey: numericalInterpolateMethod.displayName,
                    });
                    if(numericalInterpolateMethod.value === 'CONSTANT') {
                        rows.push({
                            sectionTitle: 'Time series resampling',
                            rowSortKey: 1,
                            rowIndexKey: 'Numerical interpolation constant',
                            value: timeseriesSampling.numericalInterpolateConstantValue,
                            rowValueKey: timeseriesSampling.numericalInterpolateConstantValue,
                        });
                    }
                    rows.push({
                        sectionTitle: 'Time series resampling',
                        rowSortKey: 2,
                        rowIndexKey: 'Numerical extrapolation',
                        value: numericalExtrapolateMethod.displayName,
                        rowValueKey: numericalExtrapolateMethod.displayName,
                    });
                    if(numericalExtrapolateMethod.value === 'CONSTANT') {
                        rows.push({
                            sectionTitle: 'Time series resampling',
                            rowSortKey: 3,
                            rowIndexKey: 'Numerical extrapolation constant',
                            value: timeseriesSampling.numericalExtrapolateConstantValue,
                            rowValueKey: timeseriesSampling.numericalExtrapolateConstantValue,
                        });
                    }
                    rows.push({
                        sectionTitle: 'Time series resampling',
                        rowSortKey: 4,
                        rowIndexKey: 'Non-numerical imputation',
                        value: categoricalImputeMethod.displayName,
                        rowValueKey: categoricalImputeMethod.displayName,
                    });
                    if(categoricalImputeMethod.value === 'CONSTANT') {
                        rows.push({
                            sectionTitle: 'Time series resampling',
                            rowSortKey: 5,
                            rowIndexKey: 'Non-numerical imputation constant',
                            value: timeseriesSampling.categoricalConstantValue,
                            rowValueKey: timeseriesSampling.categoricalConstantValue,
                        });
                    }
                    if(timeseriesSampling.numericalExtrapolateMethod !== 'NO_EXTRAPOLATION' && timeseriesSampling.startDateMode === "CUSTOM" && timeseriesSampling.customStartDate !== undefined) {
                        rows.push({
                            sectionTitle: 'Time series resampling',
                            rowSortKey: 6,
                            rowIndexKey: 'Extrapolation start date',
                            value: timeseriesSampling.customStartDate,
                            rowValueKey: timeseriesSampling.customStartDate,
                        });
                    }
                    if(timeseriesSampling.numericalExtrapolateMethod !== 'NO_EXTRAPOLATION' && timeseriesSampling.endDateMode === "CUSTOM" && timeseriesSampling.customEndDate !== undefined) {
                        rows.push({
                            sectionTitle: 'Time series resampling',
                            rowSortKey: 7,
                            rowIndexKey: 'Extrapolation end date',
                            value: timeseriesSampling.customEndDate,
                            rowValueKey: timeseriesSampling.customEndDate,
                        });
                    }

                    const duplicateTimestampsHandlingMethod = TimeseriesForecastingUtils.DUPLICATE_TIMESTAMPS_HANDLING_METHODS.find(
                        obj => obj.value === timeseriesSampling.duplicateTimestampsHandlingMethod
                    );
                    rows.push({
                        sectionTitle: 'Time series resampling',
                        rowSortKey: 8,
                        rowIndexKey: 'Duplicate timestamp handling',
                        value: duplicateTimestampsHandlingMethod.displayName,
                        rowValueKey: duplicateTimestampsHandlingMethod.displayName,
                    });
                }
            }
            return {
                columnColor: color,
                comparable: comparable,
                rows
            }
        }

        $ctrl.$onChanges = function () {
            [$ctrl.resamplingItems, $ctrl.resamplingItemsRef] = makeItemAndSetReference($ctrl.comparables, makeTimeSeriesResamplingItem);
        }
    }
});

app.component("modelComparisonRocCurveComparator", {
    templateUrl: '/templates/modelcomparisons/comparison-tabs/roc_curve.html',
    bindings: {
        comparables: '<',
        colors: '<'
    },
    controller: function(ModelComparisonHelpers) {
        const $ctrl = this;

        $ctrl.selectedClass = null;

        $ctrl.updateChart = function() {
            $ctrl.chartOptions = buildChartOptions();
        }

        function buildChartOptions() {
            const series = [];
            $ctrl.comparables.forEach((comparable, idx) => {
                const perf = comparable.details.perf;
                if(!perf) {
                    return;
                }
                let selectedData;
                // Binary classification
                if(comparable.details.coreParams.prediction_type === 'BINARY_CLASSIFICATION' && perf.rocVizData
                    && ModelComparisonHelpers.listOneVsAllClasses(comparable)[0].mainClass == $ctrl.selectedClass) {
                        selectedData = perf.rocVizData[0];
                }
                // Multiclass classification
                else if(comparable.details.coreParams.prediction_type === 'MULTICLASS' && perf.oneVsAllRocCurves) {
                    selectedData = perf.oneVsAllRocCurves[$ctrl.selectedClass];
                }

                // 'selectedData' can be null if the class doesn't exist for this model
                if(selectedData) {
                    const name = comparable.displayInfo.championedName;
                    const data = selectedData.map(d => [d.x, d.y, d.p]);
                    series.push(Object.assign({},getStdSeriesLine($ctrl.colors[idx], name, data),{
                        tooltip: {
                            formatter: (param) => {
                                const foldInfos = perf.rocVizData && perf.rocVizData.length > 1 ? '(fold 1)' : '';
                                return  `<b>${param.marker} ${sanitize(comparable.displayInfo.displayName)}</b>${sanitize(foldInfos)}<br>`
                                    + 'At p = ' + sanitize(Math.round(param.data[2] * 100)/100) + '<br>'
                                    + 'True positive: ' + sanitize(Math.round(param.data[1] * 100)) + '%<br>'
                                    + 'False positive: ' + sanitize(Math.round(param.data[0] * 100)) + '%<br>';
                            }
                        },
                        clip: true
                    }));
                }
            });

            series.push({
                name: 'Random model',
                type: 'line',
                itemStyle: {
                    color: 'red'
                },
                showSymbol:false,
                lineStyle: {
                    color: 'red',
                    width: 1,
                    type: 'dashed'
                },
                data: [[0, 0], [1, 1]]
            })

            const LEGEND_WIDTH = 180;
            return {
                legend: {
                    type: 'scroll',
                    show: true,
                    orient: 'vertical',
                    x: 'right',
                    y: 'center',
                    textStyle: {
                        lineOverflow: 'truncate',
                        overflow: 'truncate',
                        width: LEGEND_WIDTH
                    },
                    tooltip: {
                        show: true,
                        trigger: 'item',
                        formatter: (params) => {
                                return sanitize(params.name);
                        }
                    }
                },
                grid: {
                    right: `${LEGEND_WIDTH+40}px`
                },
                tooltip: {
                    confine: true,
                    trigger: 'item'
                },
                xAxis: {
                    type: 'value',
                    min: 0,
                    max: 1,
                    name: 'False positive rate',
                    nameLocation: 'middle',
                    splitNumber: 10,
                    nameGap: 30
                },
                yAxis: {
                    type: 'value',
                    name: 'True positive rate',
                    nameLocation: 'middle',
                    min: 0,
                    max: 1,
                    splitNumber: 10,
                    nameGap: 40
                },
                axisPointer: {
                    show: true,
                    snap: true,
                    lineStyle: {
                        type: "dashed"
                    },
                    triggerTooltip: false
                },
                series,
                animation: false
            };
        }

        $ctrl.$onChanges = function () {
            $ctrl.classes = ModelComparisonHelpers.listOneVsAllClassesAmongComparables($ctrl.comparables);
            if($ctrl.classes.length > 0) {
                $ctrl.selectedClass = $ctrl.classes[0].mainClass;
            }
            $ctrl.updateChart();
        }
    }
});

app.component("modelComparisonPrCurveComparator", {
    templateUrl: '/templates/modelcomparisons/comparison-tabs/pr_curve.html',
    bindings: {
        comparables: '<',
        colors: '<'
    },
    controller: function(ModelComparisonHelpers) {
        const $ctrl = this;

        $ctrl.selectedClass = null;

        $ctrl.updateChart = function() {
            $ctrl.chartOptions = buildChartOptions();
        }

        function buildChartOptions() {
            const series = [];
            $ctrl.comparables.forEach((comparable, idx) => {
                const perf = comparable.details.perf;
                if(!perf) {
                    return;
                }
                let selectedData;
                let positiveRate;
                // Binary classification
                if(comparable.details.coreParams.prediction_type === 'BINARY_CLASSIFICATION' && perf.prVizData
                    && ModelComparisonHelpers.listOneVsAllClasses(comparable)[0].mainClass == $ctrl.selectedClass) {
                    selectedData = perf.prVizData[0].bins;
                    positiveRate = perf.prVizData[0].positiveRate;
                }
                // Multiclass classification
                else if(comparable.details.coreParams.prediction_type === 'MULTICLASS' && perf.oneVsAllPrCurves && Object.keys(perf.oneVsAllPrCurves).length > 0) {
                    selectedData = perf.oneVsAllPrCurves[$ctrl.selectedClass].bins;
                    const totalRows = perf.confusion.totalRows;
                    positiveRate = perf.oneVsAllPrCurves[$ctrl.selectedClass].positiveRate;
                }

                // 'selectedData' can be null if the class doesn't exist for this model
                if(selectedData) {
                    const name = comparable.displayInfo.championedName;
                    const data = selectedData.map(d => [d.x, d.y, d.p]);
                    series.push(Object.assign({},getStdSeriesLine($ctrl.colors[idx], name, data),{
                        tooltip: {
                            formatter: (param) => {
                                const foldInfos = perf.prVizData && perf.prVizData.length > 1 ? '(fold 1)' : '';
                                return  `<b>${param.marker} ${sanitize(comparable.displayInfo.displayName)}</b>${sanitize(foldInfos)}<br>`
                                    + 'At p = ' + sanitize(Math.round(param.data[2] * 100)/100) + '<br>'
                                    + 'Precision: ' + sanitize(Math.round(param.data[1] * 100)) + '%<br>'
                                    + 'Recall: ' + sanitize(Math.round(param.data[0] * 100)) + '%<br>';
                            }
                        },
                        clip: true
                    }));
                    series.push({
                        name: 'Positive rate #' + (idx + 1),
                        type: 'line',
                        itemStyle: {
                            color: $ctrl.colors[idx]
                        },
                        showSymbol: false,
                        lineStyle: {
                            color: $ctrl.colors[idx],
                            width: 1,
                            type: 'dashed'
                        },
                        data: [[0, positiveRate], [1, positiveRate]]
                    })
                }
            });

            const LEGEND_WIDTH = 180;
            return {
                legend: {
                    type: 'scroll',
                    show: true,
                    orient: 'vertical',
                    x: 'right',
                    y: 'center',
                    textStyle: {
                        lineOverflow: 'truncate',
                        overflow: 'truncate',
                        width: LEGEND_WIDTH
                    },
                    tooltip: {
                        show: true,
                        trigger: 'item',
                        formatter: (params) => {
                            return sanitize(params.name);
                        }
                    }
                },
                grid: {
                    right: `${LEGEND_WIDTH+40}px`
                },
                tooltip: {
                    confine: true,
                    trigger: 'item'
                },
                xAxis: {
                    type: 'value',
                    min: 0,
                    max: 1,
                    name: 'Recall',
                    nameLocation: 'middle',
                    splitNumber: 10,
                    nameGap: 30
                },
                yAxis: {
                    type: 'value',
                    name: 'Precision',
                    nameLocation: 'middle',
                    min: 0,
                    max: 1,
                    splitNumber: 10,
                    nameGap: 40
                },
                axisPointer: {
                    show: true,
                    snap: true,
                    lineStyle: {
                        type: "dashed"
                    },
                    triggerTooltip: false
                },
                series,
                animation: false
            };
        }

        $ctrl.$onChanges = function () {
            $ctrl.classes = ModelComparisonHelpers.listOneVsAllClassesAmongComparables($ctrl.comparables);
            if($ctrl.classes.length > 0) {
                $ctrl.selectedClass = $ctrl.classes[0].mainClass;
            }
            $ctrl.updateChart();
        }
    }
});

app.component("modelComparisonFeatureImportanceComparator", {
    templateUrl: '/templates/modelcomparisons/comparison-tabs/feature_importance.html',
    bindings: {
        comparables: '<',
        pinnedMetrics: '<'
    },
    controller: function(ModelComparisonHelpers) {
        const $ctrl = this;

        const DKU_SPACING = 8;
        const MAX_FEATURES = 20;
        const FONT_SIZE = 12;
        const ITEM_GAP = DKU_SPACING / 2;
        const SYMBOL_SIZE = 22;
        const MARGIN_TOP = DKU_SPACING * 4;
        const SYNTHETIC_SEGMENT_COLOR = '#3B99FC'; // @digital-blue-base
        const MISSING_FI_NODE_COLOR = '#999999'; // @grey-lighten-4
        const MISSING_FI_FEATURE_NAME = '__dku_missing_ufi__';

        $ctrl.showEmptyState = false;

        function comparableToTableItem(comparable, color) {
            const modelFeatureRanks = $ctrl.featureRanks[comparable.ref.fullId];
            const rows = [];
            ModelComparisonHelpers.addPinnedMetricsInComparisonTables(comparable, $ctrl.pinnedMetrics, rows);
            Array.from($ctrl.features).forEach(featureName => {
                rows.push({
                    rowIndexKey: featureName,
                    rowValueKey: modelFeatureRanks && modelFeatureRanks[featureName],
                    sectionTitle: 'Feature Ranking',
                    value: modelFeatureRanks && modelFeatureRanks[featureName] && '#' + modelFeatureRanks[featureName]
                });
            });

            return {
                comparable,
                rows,
                columnColor: color
            }
        }

        function buildChartOptions(displayInfo, nbFeatures, refModel, sortedFeatures, refIsChampion) {

            const containerColumnNb = $ctrl.tableItems.length + 1; // We divide the chart in nbModels + 1 columns. It is useful to align axis with %
            function getFeatureLegendRank(name) {
                return displayInfo[refModel.comparable.ref.fullId].features.find(r => r.rowIndexKey === name).rowValueKey;
            }

            function buildXAxis() {
                return {
                    type: 'category',
                    data: Object.keys(displayInfo).map((modelFmi) => {
                        const championPrefix = refIsChampion && modelFmi === refModel.comparable.ref.fullId ? ModelComparisonHelpers.getChampionSymbol() : '';
                        const modelName = displayInfo[modelFmi].displayName
                        return modelName.length >= 20 ? `${championPrefix} ${modelName.slice(0, 17)}` + '...' : `${championPrefix} ${modelName}`;
                    }),
                    boundaryGap: false,
                    position: 'top',
                    axisLine: { show: false },
                    axisTick: { show: false }
                }
            }

            function buildYAxis() {
                return {
                    type: 'category',
                    data: _.range(nbFeatures),
                    inverse: true,
                    show: false,
                }
            }

            function createFeatureNodesLinks(nodes) {
                const edges = [];
                nodes.slice(0, -1).forEach((currentNode, nodeIdx) => {
                    const nextNode = nodes[nodeIdx + 1];
                    if (nextNode.value[0] - currentNode.value[0] === 1) { // Adjacent models so we draw an edge
                        edges.push({
                            source: nodeIdx,
                            target: nodeIdx + 1
                        });
                    }
                });
                return edges
            }

            function buildSeries() {
                const nodesByFeature = {};
                Object.entries(Object.entries(displayInfo).map(([modelFmi, info], modelIdx) => {
                    if (info.features.length === 0) {
                        if (!nodesByFeature.hasOwnProperty(MISSING_FI_FEATURE_NAME)) {
                            nodesByFeature[MISSING_FI_FEATURE_NAME] = [];
                        }
                        nodesByFeature[MISSING_FI_FEATURE_NAME].push({
                            value: [modelIdx, (nbFeatures / 2.) - 1],
                            itemStyle: {
                                color: MISSING_FI_NODE_COLOR,
                            },
                            tooltip: {
                                formatter: "Feature importance ranking is not available for this model or was not computed.",
                                borderColor: MISSING_FI_NODE_COLOR
                            }
                        });
                    }
                    info.features.forEach(feature => {
                        const featureName = feature.rowIndexKey;
                        if (!nodesByFeature.hasOwnProperty(featureName)) {
                            nodesByFeature[featureName] = [];
                        }
                        nodesByFeature[featureName].push({
                            value: [modelIdx, feature.rowValueKey - 1],
                            itemStyle: { color: info.columnColor },
                        });
                    });
                }))

                return Object.entries(nodesByFeature).map(([feature, nodes]) => {
                    return {
                        name: feature,
                        type: 'graph',
                        coordinateSystem: 'cartesian2d',
                        symbolSize: SYMBOL_SIZE,
                        label: {
                            show: true,
                            formatter: feature === MISSING_FI_FEATURE_NAME
                                ? '?'
                                : feature.slice(0, 2),
                            color: 'white',
                            fontSize: FONT_SIZE,
                            fontFamily: 'SourceSansPro'
                        },
                        emphasis: { focus: 'series', scale: 1.2 },
                        lineStyle: {
                            color: SYNTHETIC_SEGMENT_COLOR,
                            width: 1,
                            opacity: 1,
                        },
                        blur: {
                            itemStyle: { opacity: 1 },
                            lineStyle: { opacity: 0.3 },
                            label: { opacity: 1 }
                        },
                        tooltip: {
                            borderColor: SYNTHETIC_SEGMENT_COLOR
                        },
                        nodes: nodes,
                        links: feature === MISSING_FI_FEATURE_NAME ? [] : createFeatureNodesLinks(nodes)
                    }
                })
            }


            return {
                xAxis: buildXAxis(),
                yAxis: buildYAxis(),
                tooltip: {
                    formatter: series => series.seriesName,
                },
                grid: {
                    left: `${(3 / 2) * (100 / containerColumnNb)}%`, // Align first parallel axis with the first model column -> 1,5 columns = 3/2
                    top: MARGIN_TOP,
                    height: ((nbFeatures) * (SYMBOL_SIZE + ITEM_GAP)), // Since item gap is used in the legend it is important to use it here so that this aligns correctly
                    right: `${(1 / 2) * (100 / containerColumnNb)}%`,
                },
                textStyle: {
                    fontFamily: 'SourceSansPro'
                },
                series: buildSeries(),
                legend: {
                    data: sortedFeatures,
                    textStyle: {
                        rich: {
                            label: {
                                padding: [0, 0, 0, 8] // To display legend description (feature name) right of the ranking circle
                            },
                            rank: {
                                width: SYMBOL_SIZE,
                                height: SYMBOL_SIZE,
                                padding: 0,
                                color: '#fff',
                                backgroundColor: refModel.columnColor,
                                borderRadius: 100,
                                align: 'center',
                                fontSize: FONT_SIZE
                            },
                            itemStyle: {
                                backgroundColor: refModel.columnColor
                            }
                        },
                    },
                    formatter: (name) => {
                        // First we display the rank as colored circle
                        // Then we display the feature name
                        return `{rank|${getFeatureLegendRank(name)}}{label|${name}}`
                    },
                    icon: 'none', // Hide default legend icon
                    orient: 'vertical',
                    itemGap: ITEM_GAP, // Space between legend elements.
                    padding: 0,
                    left: `${(1 / 4) * (100 / containerColumnNb)}%`,
                    top: MARGIN_TOP + 2, // The + 2 is a bit of magic here. There is a misalignment with the series otherwise.
                    tooltip: {
                        show: true,
                        borderColor: SYNTHETIC_SEGMENT_COLOR
                    },
                },
            }
        }

        function itemHasFeatureImportance(item) {
            return item.rows.some((r) => r.sectionTitle === "Feature Ranking" && !!r.value);
        }

        $ctrl.$onChanges = function () {
            const features = new Set();
            const featureRanks = {};

            $ctrl.comparables.forEach(comparable => {
                if (!comparable.details.globalExplanationsAbsoluteImportance) {
                    return;
                }
                const currentModelFeaturesRanks = {};
                Object.entries(comparable.details.globalExplanationsAbsoluteImportance.absoluteImportance).sort((a, b) => {
                    return b[1] - a[1];
                }).forEach((v, idx) => {
                    const feature = v[0];
                    currentModelFeaturesRanks[feature] = idx + 1;
                    features.add(feature);
                });
                featureRanks[comparable.ref.fullId] = currentModelFeaturesRanks;
            });

            $ctrl.features = features;
            $ctrl.featureRanks = featureRanks;

            [$ctrl.tableItems, $ctrl.tableItemsRef] = makeItemAndSetReference($ctrl.comparables, comparableToTableItem);

            let refModel;
            if ($ctrl.tableItemsRef && itemHasFeatureImportance($ctrl.tableItemsRef)) {
                refModel = $ctrl.tableItemsRef;
            } else {
                refModel = $ctrl.tableItems.find(item => itemHasFeatureImportance(item));
            }

            if (!refModel || !$ctrl.features.size) {
                $ctrl.showEmptyState = true;
                return;
            }

            const refIsChampion = $ctrl.tableItemsRef === refModel;

            const sortedFeatures = refModel.rows
                .filter((r) => r.sectionTitle === "Feature Ranking" && !!r.value) // If there is no value, the feature is not in the model
                .sort((first, second) => first.rowValueKey - second.rowValueKey)
                .slice(0, MAX_FEATURES) // We display at most 20 features
                .map(row => row.rowIndexKey);

            const displayInfo = {};
            let maxFeatures = 0;
            $ctrl.tableItems.forEach((item, _) => {
                displayInfo[item.comparable.ref.fullId] = {
                    // We filter rows to only display ones that have a value and we sort them in decreasing order to obtain rankings
                    features: item.rows.filter((r) => r.sectionTitle === "Feature Ranking" && !!r.value).sort((a, b) => a.rowValueKey - b.rowValueKey),
                    columnColor: item.columnColor,
                    displayName: item.comparable.displayInfo.displayName
                };
                maxFeatures = Math.max(maxFeatures, displayInfo[item.comparable.ref.fullId].features.length);
            });
            maxFeatures = Math.min(maxFeatures, MAX_FEATURES);

            $ctrl.containerHeight = (MARGIN_TOP + (SYMBOL_SIZE + ITEM_GAP) * maxFeatures) + 'px';
            $ctrl.chartOptions = buildChartOptions(displayInfo, maxFeatures, refModel, sortedFeatures, refIsChampion);
        }
    }
})

app.component("modelComparisonDensityChartComparator", {
    templateUrl: '/templates/modelcomparisons/comparison-tabs/density_chart.html',
    bindings: {
        comparables: '<'
    },
    controller: function(ModelComparisonHelpers, $filter) {
        const $ctrl = this;

        $ctrl.selectedClass = null;

        $ctrl.updateChart = function() {
            $ctrl.chartOptions = buildChartOptions();
        }

        function buildChartOptions() {
            const HEIGHT = 250;
            const HEIGHT_MARGIN = 50;

            const series = [];
            const xAxis = [];
            const yAxis = [];
            const grid = [];
            const displayedComparables = [];

            for(const comparable of $ctrl.comparables) {
                const perf = comparable.details.perf;
                if(perf && perf.densityData && perf.densityData[$ctrl.selectedClass]) {
                    const densityData = perf.densityData[$ctrl.selectedClass];
                    const oneVsAllClass = ModelComparisonHelpers.listOneVsAllClassesAmongComparables($ctrl.comparables)
                        .find(c => c.mainClass == $ctrl.selectedClass);
                    const yMax = Math.max(...densityData.actualIsThisClass,...densityData.actualIsNotThisClass);
                    series.push({
                        animation: false,
                        type: 'line',
                        smooth: true,
                        name: oneVsAllClass.mainClass,
                        xAxisIndex: grid.length,
                        yAxisIndex: grid.length,
                        lineStyle: {
                            width: LINE_GRAPH_LINE_WIDTH,
                            color: '#ff7f0e'
                        },
                        showSymbol: false,
                        symbol: LINE_GRAPH_SYMBOL,
                        areaStyle: {
                            opacity: 0.6,
                            color: '#ff7f0e'
                        },
                        emphasis: {
                            focus: 'series'
                        },
                        data: _.zip(_.range(0, 1, 1.0/densityData.actualIsThisClass.length), densityData.actualIsThisClass)
                    });

                    series.push({
                        animation: false,
                        type:'line',
                        name: oneVsAllClass.otherClasses.join(', '),
                        smooth: true,
                        xAxisIndex: grid.length,
                        yAxisIndex: grid.length,
                        lineStyle: {
                            width: LINE_GRAPH_LINE_WIDTH,
                            color: '#1f77b4'
                        },
                        showSymbol: false,
                        symbol: LINE_GRAPH_SYMBOL,
                        areaStyle: {
                            opacity: 0.6,
                            color: '#1f77b4'
                        },
                        emphasis: {
                            focus: 'series'
                        },
                        data: _.zip(_.range(0, 1, 1.0/densityData.actualIsNotThisClass.length), densityData.actualIsNotThisClass)
                    });

                    xAxis.push({
                        type: 'value',
                        scale: true,
                        gridIndex: grid.length,
                        nameLocation: 'middle',
                        nameGap: 20,
                        axisLabel: {
                            formatter: function(value) {
                                return sanitize($filter('nicePrecision')(value,2));
                            }
                        },
                        min: 0,
                        max: 0.99,
                        name: 'Predicted probability'
                    })

                    yAxis.push({
                        type: 'value',
                        axisLabel: {
                            formatter: function(value) {
                                return sanitize($filter('nicePrecision')(value,2));
                            }
                        },
                        min: 0,
                        max: yMax,
                        gridIndex: grid.length,
                        nameLocation: 'middle',
                        name: 'Probability density',
                        nameGap: 35,
                    })

                    grid.push({
                        left: '50px',
                        top: grid.length * (HEIGHT + HEIGHT_MARGIN) + 40 + 'px',
                        height: HEIGHT,
                        right: '100px',
                        containLabel: true
                    });

                    displayedComparables.push(comparable);
                }
            }

            return {
                // Not an echart option - but this is used to set the widget's height
                chartHeight: grid.length * (HEIGHT + HEIGHT_MARGIN) + 'px',
                grid,
                color: ['#ff7f0e', '#1f77b4'],
                legend: {
                    type: 'scroll',
                    show: true,
                    orient: 'vertical',
                    x: 'right',
                    y: 'top',
                    textStyle: {
                        lineOverflow: 'truncate'
                    }
                },
                tooltip: {
                    trigger: 'item',
                    showContent: true,
                    formatter: (params) => {
                        const currentSeries = series[params.seriesIndex];
                        let ret = `<b>Predicted probability: ${sanitize($filter('nicePrecision')(params.value[0],3))}</b><br/>`;
                        const currentMetricSeries = series.filter(s => s.xAxisIndex === currentSeries.xAxisIndex);
                        const curXIdx = series[params.seriesIndex].data.map(x => x[0]).indexOf(params.value[0]);
                        for (const cms of currentMetricSeries) {
                            ret += `${params.marker.replace(/#[a-f0-9]{6}/i, sanitize(cms.lineStyle.color))} ${sanitize(cms.name)}: ${sanitize($filter('nicePrecision')(cms.data[curXIdx][1],4))}<br/>`;
                        }
                        return ret;
                    },
                    rich: {
                        bold: {
                            fontWeight: 'bold'
                        }
                    },
                    confine: true
                },
                axisPointer: {
                    show: true,
                    link: [
                        {
                            xAxisIndex: 'all'
                        },
                        {
                            yAxisIndex: 'all'
                        }
                    ],
                    snap: true,
                    lineStyle: {
                        type: "dashed"
                    },
                    triggerTooltip: false
                },
                xAxis,
                yAxis,
                series,
                title: displayedComparables.map((dc,idx) => {
                    const title = dc.displayInfo.championedName;
                    return {
                        left: 'center',
                        top: idx * (HEIGHT + HEIGHT_MARGIN) + 10 + 'px',
                        text: title,
                        textStyle: {
                            lineOverflow: 'truncate',
                            overflow: 'truncate',
                            width: 500
                        },
                    };
                })
            };
        }

        $ctrl.$onChanges = function () {
            $ctrl.classes = ModelComparisonHelpers.listOneVsAllClassesAmongComparables($ctrl.comparables);
            if($ctrl.classes.length > 0) {
                $ctrl.selectedClass = $ctrl.classes[0].mainClass;
            }
            $ctrl.updateChart();
        }
    }
});

app.component("modelComparisonScatterPlotComparator", {
    templateUrl: '/templates/modelcomparisons/comparison-tabs/scatter_plot.html',
    bindings: {
        comparables: '<',
        colors: '<'
    },
    controller: function($filter) {
        const $ctrl = this;

        $ctrl.$onInit = function () {
            $ctrl.chartOptions = buildChartOptions();
        }

        $ctrl.$onChanges = function () {
            $ctrl.chartOptions = buildChartOptions();
        }
        function buildChartOptions() {
            const series = $ctrl.comparables.map((comparable, idx) => {
                const scatterData = comparable.details.perf && comparable.details.perf.scatterPlotData;

                return {
                    name: comparable.displayInfo.championedName,
                    type: 'scatter',
                    symbolSize: 4,
                    large: true,
                    color: $ctrl.colors[idx],
                    data: scatterData ? _.zip(scatterData.x, scatterData.y) : []
                }
            });

            const max = Math.max(...series.flatMap(s => s.data).flatMap(v => v));
            series.forEach(s =>
                s.markLine = {
                    animation: false,
                    silent: true,
                    lineStyle: {
                        type: 'dotted',
                        color: 'lightcoral',
                        width: 2
                    },
                    tooltip: {
                        formatter: 'Ideal'
                    },
                    data: [[{
                        coord: [0, 0],
                        symbol: 'none'
                    }, {
                        coord: [max, max],
                        symbol: 'none'
                    }]]
                }
            );

            const LEGEND_WIDTH = 180;
            return {
                color: $ctrl.colors,
                grid: {
                    left: '3%',
                    right: `${LEGEND_WIDTH+40}px`,
                    bottom: '7%',
                    containLabel: true
                },

                legend: {
                    type: 'scroll',
                    show: true,
                    orient: 'vertical',
                    icon: LINE_GRAPH_SYMBOL,
                    x: 'right',
                    y: 'center',
                    textStyle: {
                        lineOverflow: 'truncate',
                        overflow: 'truncate',
                        width: LEGEND_WIDTH
                    },
                    tooltip: {
                        show: true,
                        trigger: 'item',
                        formatter: (params) => {
                            return sanitize(params.name);
                        }
                    }
                },
                tooltip: {
                    confine: true,
                    trigger: 'item',
                    axisPointer: {
                        type: 'cross'
                    },
                    formatter: (param) => {
                        const actualValue = param.value[0];
                        const predictedValue = param.value[1];
                        const error = Math.abs(actualValue - predictedValue);
                        const errorPct = (error/actualValue)*100;
                        return  `${param.marker}<b>${sanitize(param.seriesName)}</b><br>`
                            + `Actual value: ${sanitize($filter('nicePrecision')(actualValue,4))}<br>`
                            + `Predicted value: ${sanitize($filter('nicePrecision')(predictedValue,4))}<br>`
                            + `Error: ${sanitize($filter('nicePrecision')(error,4))} (${sanitize($filter('nicePrecision')(errorPct,4))}%)`;
                    }
                },
                axisPointer: {
                    show: true,
                    link: [
                        {
                            xAxisIndex: 'all'
                        },
                        {
                            yAxisIndex: [0,1,2]
                        }
                    ],
                    snap: true,
                    lineStyle: {
                        type: "dashed"
                    },
                    triggerTooltip: false
                },
                xAxis: [
                    {
                        type: 'value',
                        scale: true,
                        min:0,
                        max,
                        splitLine: {
                            show: false
                        },
                        name: 'Actual values',
                        nameLocation: 'middle',
                        nameGap: 20
                    }
                ],
                yAxis: [
                    {
                        type: 'value',
                        min:0,
                        max,
                        scale: true,
                        splitLine: {
                            show: false
                        },
                        name: 'Predicted values',
                        nameLocation: 'middle',
                        nameGap: 30
                    }
                ],
                series
            };
        }
    }
});

app.component("modelComparisonUpliftComparator", {
    templateUrl: '/templates/modelcomparisons/comparison-tabs/uplift_charts.html',
    bindings: {
        comparables: '<',
        colors: '<'
    },
    controller: ['$scope', '$filter', 'StateUtils', function($scope, $filter, StateUtils) {
        const $ctrl = this;
        $scope.StateUtils = StateUtils;
        $ctrl.modelsWithNullTestAte = [];

        function getCausalPerf(comparable) {
            if (!comparable.details || !comparable.details.perf) return;
            return comparable.details.perf.causalPerf;
        }

        $ctrl.getTestAte = function(comparable) {
            const causalPerf = getCausalPerf(comparable) || {};
            return causalPerf.testATE;
        }

        $ctrl.$onChanges = function () {
            if ($ctrl.comparables && $ctrl.comparables.length && $ctrl.colors && $ctrl.colors.length) {
                $ctrl.chartOptions = {
                    legend: {
                        type: 'scroll',
                        show: true,
                        orient: 'vertical',
                        x: 'right',
                        y: 'center',
                        textStyle: {
                            lineOverflow: 'truncate',
                            overflow: 'truncate',
                            width: 160
                        },
                        itemStyle: { opacity: 0 },
                        tooltip: {
                            show: true,
                            trigger: 'item',
                            formatter: (params) => {
                                return sanitize(params.name);
                            }
                        },
                    },
                    tooltip: {
                        trigger: 'axis',
                        textStyle: { fontSize: 13 },
                        formatter: function(modelData) {
                            let tooltipContent = `<b>${toFixed(modelData[0].data[0], 4)}% of the population</b><br/>`;
                            modelData.forEach(function(data) {
                                tooltipContent += tooltipContentForSeries(data);
                            });
                            return tooltipContent;
                        }
                    },
                    textStyle: { fontFamily: 'SourceSansPro' },
                    grid: [],
                    xAxis: [],
                    yAxis: [],
                    series: [],
                    title: []
                };
                buildChartOptions("upliftGainCurve", "Cumulative uplift");
                buildChartOptions("qiniCurve", "Qini");
                $ctrl.modelsWithNullTestAte = $ctrl.comparables.filter(cp => !$ctrl.getTestAte(cp));
            } else {
                $ctrl.chartOptions = null;
                $ctrl.modelsWithNullTestAte = [];
            }
        }

        const toFixed = (number, precision) => parseFloat(number.toFixed(precision));
        function tooltipContentForSeries(series) {
            return `<i class='icon-circle mright8' style='color: ${series.color}'></i>
                ${series.seriesName}: ${toFixed(series.data[1], 3)}<br/>`;
        };

        function buildChartOptions(curveKey, curveName) {
            if (!$ctrl.comparables || !$ctrl.comparables.length) return;

            const axisIndex = $ctrl.chartOptions.xAxis.length;
            const modelSeries = [];
            $ctrl.comparables.forEach((comparable, idx) => {
                const causalPerf = getCausalPerf(comparable);
                const data = causalPerf && causalPerf[curveKey];
                if (!data) return;

                const testATE = causalPerf.testATE;
                if (!testATE) return;

                modelSeries.push({
                    ate: testATE,
                    type: 'line',
                    symbol: 'none',
                    data: data.map(dataPoint => [ dataPoint.x, dataPoint.y / Math.abs(testATE) ]),
                    name: comparable.displayInfo.championedName,
                    color: $ctrl.colors[idx],
                    xAxisIndex: axisIndex,
                    yAxisIndex: axisIndex,
                });
            });

            $ctrl.chartOptions.title.push({
                textAlign: "left",
                text: `${curveName} curve`,
                textStyle: { fontWeight: 'bold', fontSize: 16 },
                top: $ctrl.chartOptions.grid.length * 400 + 16,
                bottom: 24,
            });

            $ctrl.chartOptions.xAxis.push({
                axisLine: { onZero: false },
                axisLabel: {
                    rotate: 45,
                    formatter: val => toFixed(val, 3)
                },
                name: "Fraction of the test observations, sorted by decreasing predicted individual effect\n(% of total test observations)",
                nameGap: 24,
                nameLocation: "middle",
                gridIndex: $ctrl.chartOptions.grid.length
            });

            $ctrl.chartOptions.yAxis.push({
                axisLabel: { formatter: val => toFixed(val, 3) },
                scale: true,
                name: "Cumulative effect\n(ratio of theAverage Treatment Effect on the test set)",
                nameGap: 40,
                nameLocation: "middle",
                type: "value",
                gridIndex: $ctrl.chartOptions.grid.length
            });

            $ctrl.chartOptions.grid.push({
                containLabel: true,
                height: 320,
                top: ($ctrl.chartOptions.grid.length * 400) + 48,
                left: 40,
                right: 200,
            });

            $ctrl.chartOptions.series = $ctrl.chartOptions.series.concat(modelSeries);
            const firstSeriesWithPositiveAte = modelSeries.find(series => series.ate > 0);
            const firstSeriesWithNegativeAte = modelSeries.find(series => series.ate < 0);
            if (firstSeriesWithPositiveAte) {
                $ctrl.chartOptions.series.push({
                    color: "#999",  // @grey-lighten-4
                    data: firstSeriesWithPositiveAte.data.map(data => [data[0], data[0] / 100]),
                    lineStyle: { type: 'dashed' },
                    name: "Random assignment (positive test ATE)",
                    symbol: 'none',
                    type: 'line',
                    xAxisIndex: axisIndex,
                    yAxisIndex: axisIndex,
                });
            }
            if (firstSeriesWithNegativeAte) {
                $ctrl.chartOptions.series.push({
                    color: "black",
                    data: firstSeriesWithNegativeAte.data.map(data => [data[0], - data[0] / 100]),
                    lineStyle: { type: 'dashed' },
                    name: "Random assignment (negative test ATE)",
                    symbol: 'none',
                    type: 'line',
                    xAxisIndex: axisIndex,
                    yAxisIndex: axisIndex,
                });
            }
        }

        $scope.differentNetUpliftPoints = function() {
            if (!$ctrl.comparables || !$ctrl.comparables.length) return false;

            const netUpliftPoints = $ctrl.comparables.map(cp => cp.metricParams.netUpliftPoint);
            const firstNetUpliftPoint = netUpliftPoints[0];
            return netUpliftPoints.some(netUpliftPoint => netUpliftPoint !== firstNetUpliftPoint);
        };
    }]
});



app.component("modelComparisonFeatureComparator", {
    templateUrl: '/templates/modelcomparisons/comparison-tabs/features.html',
    bindings: {
        comparables: '<',
        pinnedMetrics: '<',
        colors: '<'
    },
    controller: function(ModelComparisonHelpers) {
        const $ctrl = this;

        function comparableToTableItem(comparable, color) {
            const rows = [];

            ModelComparisonHelpers.addPinnedMetricsInComparisonTables(comparable, $ctrl.pinnedMetrics, rows);

            if(comparable
                && comparable.details
                && comparable.details.preprocessing
                && comparable.details.preprocessing.per_feature) {
                for(let featureName in comparable.details.preprocessing.per_feature) {
                    const handling = comparable.details.preprocessing.per_feature[featureName];
                    const row = {
                        rowIndexKey: featureName,
                        sectionTitle: 'Feature handling',
                        rowValueKey: handling && [handling.role, handling.category_handling, handling.type, handling.numerical_handling, handling.rescaling, handling.missing_handling],
                        handling: handling
                    };

                    // For causal models
                    if (comparable.details.coreParams) {
                        const coreParams = comparable.details.coreParams;
                        if (handling.role === "TARGET") {
                            if (coreParams.prediction_type === 'CAUSAL_BINARY_CLASSIFICATION') {
                                row.positiveClass = coreParams.positive_class;
                            }
                        }
                        if (handling.role === "TREATMENT") {
                            row.controlValue = coreParams.control_value;
                        }
                    }

                    rows.push(row);
                }
            }

            if(comparable
                && comparable.details
                && comparable.details.preprocessingReport
                && comparable.details.preprocessing.feature_generation) {
                const preprocessingReport = comparable.details.preprocessingReport;
                const feature_generation = comparable.details.preprocessing.feature_generation;

                if(preprocessingReport.pairwise_linear) {
                    rows.push({
                        rowIndexKey: 'Linear combinations',
                        sectionTitle: 'Feature generation',
                        rowValueKey: preprocessingReport.pairwise_linear,
                        value: `Generated ${preprocessingReport.pairwise_linear.built_features} new features from ${preprocessingReport.pairwise_linear.input_features} input features`
                    });
                }

                if(preprocessingReport.polynomial_interactions) {
                    rows.push({
                        rowIndexKey: 'Polynomial combinations',
                        sectionTitle: 'Feature generation',
                        rowValueKey: preprocessingReport.polynomial_interactions,
                        value: `Generated ${preprocessingReport.polynomial_interactions.built_features} new features from ${preprocessingReport.polynomial_interactions.input_features} input features`
                    });
                }

                if(feature_generation.manual_interactions) {
                    const manualInteractions = _.sortBy(feature_generation.manual_interactions.interactions || [], ['column_1', 'column_2'])
                        .map(interaction => `${interaction.column_1} \u00d7 ${interaction.column_2}`);

                    if(manualInteractions.length > 0) {
                        rows.push({
                            rowIndexKey: 'Feature interactions',
                            sectionTitle: 'Feature generation',
                            rowValueKey: manualInteractions,
                            values: manualInteractions
                        });
                    }
                }
            }

            return {
                columnColor: color,
                comparable: comparable,
                rows
            }
        }

        $ctrl.$onChanges = function () {
            [$ctrl.tableItems, $ctrl.tableItemsRef] = makeItemAndSetReference($ctrl.comparables, comparableToTableItem);
        }
    }
});

app.controller("modelComparisonCompositionController", function($scope, TopNav) {
    TopNav.setLocation(TopNav.TOP_MODEL_COMPARISONS, TopNav.LEFT_MODEL_COMPARISONS, TopNav.TABS_MODEL_COMPARISON, "composition");

    $scope.tabNotAvailableText = (tabName) => {
        if (!$scope.uiState.comparables || !$scope.uiState.comparables.length) {
            return null;
        }

        if (tabName === "training_information" && $scope.isLLM) {
                return "Not available for LLM comparisons";
        }

        if (tabName === "training_information" && $scope.uiState.comparables.every(
            cmp => cmp && (cmp.details.modeling || {}).algorithm === "VIRTUAL_MLFLOW_PYFUNC")) {
                return "Not available for MLflow models";
        }

        if (tabName === "training_information" && $scope.uiState.comparables.every(
            cmp => cmp && (cmp.details.modeling || {}).algorithm === "VIRTUAL_PROXY_MODEL")) {
                return "Not available for External Models";
        }
        return null;
    }

    $scope.$on("$stateChangeSuccess", function(event, toState) {
        const stateName = toState.name.split(".").pop();
        $scope.uiState.comparisonPane = stateName;
        $scope.$applyAsync();
    });
});

app.controller("ModelEvaluationsComparatorSummaryController", function($scope, $filter, ModelEvaluationUtils,
            PMLFilteringService, ComparablesService, $stateParams) {

    $scope.onDelete = function(modelEvaluationsToDelete) {
        const idxsToDelete = []
        for(const modelEvaluationToDelete of modelEvaluationsToDelete) {
            idxsToDelete.push($scope.uiState.refs.indexOf(modelEvaluationToDelete));
        }
        idxsToDelete.sort((a,b) => b-a);
        for(const idxToDelete of idxsToDelete) {
            $scope.modelComparison.comparedModels.splice(idxToDelete,1);
            // also splice displayed data, to avoid flickering effect
            if ($scope.uiState.allComparables) $scope.uiState.allComparables.splice(idxToDelete,1);
        }
    }

    $scope.onReverseDisplay = function(comparedItems) {
        const idxsToDisplay = []
        for(const comparedItemToDisplay of comparedItems) {
            idxsToDisplay.push($scope.uiState.refs.indexOf(comparedItemToDisplay));
        }
        for(const idxToDisplay of idxsToDisplay) {
            $scope.modelComparison.comparedModels[idxToDisplay].isSelected =
                !$scope.modelComparison.comparedModels[idxToDisplay].isSelected;
        }
    }

    $scope.clearChampion = function() {
        $scope.modelComparison.comparedModels.forEach(cm => {cm.isChampion = false;});
        $scope.onModelComparisonChange();
    }

    $scope.onSetChampion = function(toSet) {
        $scope.modelComparison.comparedModels.forEach(cm => {
            cm.isChampion = toSet.ref.fullId == cm.refId;
        });
        $scope.onModelComparisonChange();
    }

    const deregistrationFn = $scope.$watch("$root.projectSummary.canWriteProjectContent", (nv, ov) => {
        if (nv) {
            deregistrationFn();
            $scope.$watch("modelComparison.displayParams", function(nv2, ov2) { if (nv2 && ov2 && !angular.equals(nv2,ov2)) $scope.save() }, true);
        }
    });
});

app.controller("ModelEvaluationsComparatorPageRightColumnActions", function($controller, $scope, $state, $rootScope, $timeout, DataikuAPI, $stateParams, ActiveProjectKey, CreateModalFromTemplate, ActivityIndicator) {

    $controller('_TaggableObjectPageRightColumnActions', {$scope: $scope});

    $scope.selection = {};


    $scope.renameMEC = function() {
        CreateModalFromTemplate("/templates/taggable-objects/rename-modal.html", $scope, null, function(newScope) {
            const currentName = $rootScope.topNav.item.data.name;
            newScope.objectName = currentName;
            newScope.uiState = { newName: currentName };

            newScope.go = function() {
                $scope.modelComparison.displayName = newScope.uiState.newName;
                DataikuAPI.modelcomparisons.save($scope.modelComparison, {summaryOnly: false}).success(() => {
                    ActivityIndicator.success("Saved");
                    $state.reload();
                }).error(setErrorInScope.bind($scope));
                newScope.dismiss();
            }
        });
    };

    DataikuAPI.modelcomparisons.getFullInfo(ActiveProjectKey.get(), $stateParams.modelComparisonId).success((data) => {
        data.nodeType = 'MODEL_COMPARISON';
        data.id = data.modelComparison.id;
        data.name = data.modelComparison.displayName;

        $scope.selection = {
            selectedObject : data,
            confirmedItem : data,
        };
    }).error(setErrorInScope.bind($scope));

    function updateUserInterests() {
        DataikuAPI.interests.getForObject($rootScope.appConfig.login, "MODEL_COMPARISON", $stateParams.projectKey, $stateParams.modelComparisonId).success(function(data) {

            $scope.selection.selectedObject.interest = data;
            $scope.modelComparisonData.interest = data;

        }).error(setErrorInScope.bind($scope));
    }

    $scope.isOnStoreObjectPage = function() {
        return $state.includes('projects.project.modelcomparisons.modelcomparison');
    }

    const interestsListener = $rootScope.$on('userInterestsUpdated', updateUserInterests);
    $scope.$on("$destroy", interestsListener);
});


app.directive('modelComparisonRightColumnSummary', function($controller, $state, $stateParams, $rootScope, FlowGraphSelection,
    DataikuAPI, QuickView, ActiveProjectKey, ActivityIndicator) {

    return {
        templateUrl :'/templates/modelcomparisons/right-column-summary.html',
        link : function(scope, element, attrs) {
            $controller('_TaggableObjectsMassActions', {$scope: scope});


            scope.$stateParams = $stateParams;
            scope.QuickView = QuickView;

            scope.getSmartName = function (projectKey, name) {
                if (projectKey == ActiveProjectKey.get()) {
                    return name;
                } else {
                    return projectKey + '.' + name;
                }
            }

            scope.refreshData = function() {
                var projectKey = scope.selection.selectedObject.projectKey;
                var name = scope.selection.selectedObject.id || scope.selection.selectedObject.name;
                DataikuAPI.modelcomparisons.getFullInfo(ActiveProjectKey.get(), scope.getSmartName(projectKey, name)).success(function(data){
                    if (!scope.selection.selectedObject || scope.selection.selectedObject.projectKey != projectKey ||
                        (scope.selection.selectedObject.id != name) && (scope.selection.selectedObject.name != name)) {
                        return; // too late!
                    }
                    scope.modelComparisonData = data;
                    scope.modelComparison = data.modelComparison;
                    scope.hideDetails = scope.$eval(attrs.hideDetails);
                    if (!scope.hideDetails) {
                        DataikuAPI.modelcomparisons.getModelsDetails(data.modelComparison.comparedModels.map(cm => cm.refId))
                            .success(dataModels => {
                                scope.modelComparison.modelNames = dataModels.map(cm => cm?cm.userMeta.name:"(unavailable)");
                            })
                            .error(setErrorInScope.bind(scope));
                        }
                }).error(setErrorInScope.bind(scope));
            };

            scope.$on("objectSummaryEdited", function() {
                DataikuAPI.modelcomparisons.save(scope.modelComparison, {summaryOnly: true})
                .success(function() {
                    ActivityIndicator.success("Saved");
                }).error(setErrorInScope.bind(scope));
            });

            scope.$watch("selection.selectedObject",function() {
                if(scope.selection.selectedObject != scope.selection.confirmedItem) {
                    scope.modelComparison = null;
                    scope.objectTimeline = null;
                }
            });

            scope.$watch("selection.confirmedItem", function(nv) {
                if (!nv) {
                    return;
                }
                if (!nv.projectKey) {
                    nv.projectKey = ActiveProjectKey.get();
                }
                scope.refreshData();
            });

            scope.zoomToOtherZoneNode = function(zoneId) {
                let otherNodeId = scope.selection.selectedObject.id.replace(/zone__.+?__saved/, "zone__" + zoneId + "__saved");
                if ($stateParams.zoneId) {
                    $state.go('projects.project.flow', Object.assign({}, $stateParams, { zoneId: zoneId }))
                }
                else {
                    scope.zoomGraph(otherNodeId);
                    FlowGraphSelection.clearSelection();
                    FlowGraphSelection.onItemClick(scope.nodesGraph.nodes[otherNodeId], null);
                }
            }

            scope.editCustomFields = function() {
                if (!scope.selection.selectedObject) {
                    return;
                }
            };

            const customFieldsListener = $rootScope.$on('customFieldsSaved', scope.refreshData);
            scope.$on("$destroy", customFieldsListener);
        }
    }
});


app.controller("modelComparisonListController", function($scope, $controller, $stateParams, DataikuAPI, CreateModalFromTemplate, $state, TopNav) {
    $controller('_TaggableObjectsListPageCommon', {$scope: $scope});

    $scope.sortBy = [
        { value: 'name', label: 'Name' },
        { value: '-lastModifiedOn', label: 'Last modified'}
    ];

    $scope.selection = $.extend({
        filterQuery: {
            userQuery: '',
            tags: [],
            interest: {
                starred: '',
            },
            inputDatasetSmartName: []
        },
        filterParams: {
            userQueryTargets: ["name", "tags"],
            propertyRules: {tag: 'tags'},
        },
        orderQuery: "-lastModifiedOn",
        orderReversed: false
    }, $scope.selection || {});

    $scope.maxItems = 20;

    $scope.list = function() {
        DataikuAPI.modelcomparisons.listHeads($stateParams.projectKey).success(function(data) {
            $scope.listItems = data;
            $scope.restoreOriginalSelection();
        }).error(setErrorInScope.bind($scope));
    };

    TopNav.setLocation(TopNav.TOP_MODEL_COMPARISONS, TopNav.LEFT_MODEL_COMPARISONS, TopNav.TABS_NONE, null);
    TopNav.setNoItem();
    $scope.list();

    /* Tags handling */

    $scope.$on('selectedIndex', function(){
        // an index has been selected, we unselect the multiselect
        $scope.$broadcast('clearMultiSelect');
    });

    /* Specific actions */
    $scope.goToItem = function(data) {
        $state.go("projects.project.modelcomparisons.modelcomparison.summary", {projectKey : $stateParams.projectKey, modelComparisonId : data.id});
    }

    $scope.newModelComparison = function() {
        CreateModalFromTemplate("/templates/modelcomparisons/modals/new-model-evaluation-comparator-modal.html", $scope);
    }

});

app.controller("newModelComparisonController", function($scope, $state, $rootScope, DataikuAPI, WT1, $stateParams, PMLSettings, LLMEvaluationMetrics) {
    $scope.modelTaskTypes = PMLSettings.task.predictionTypes.filter(type => type.classical || type.forecast || type.causal);
    if ($rootScope.appConfig.licensedFeatures.advancedLLMMeshAllowed) {
        $scope.modelTaskTypes = $scope.modelTaskTypes.concat(LLMEvaluationMetrics.llmModelTaskTypes);
    }
    $scope.newModelComparison = {
        name : null,
        modelTaskType : $scope.modelTaskTypes[0].type
    };

    $scope.create = function(){
        resetErrorInScope($scope);
        WT1.event("model-evaluation-comparison-created", { from: 'mc-list', predictionType: $scope.newModelComparison.modelTaskType, nbItems: 0 }); // predictionType name is kept for retro-compatibility
        DataikuAPI.modelcomparisons.create($stateParams.projectKey, $scope.newModelComparison.name, $scope.newModelComparison.modelTaskType).success(function(data) {
            $scope.dismiss();
            $state.go("projects.project.modelcomparisons.modelcomparison.summary", {modelComparisonId: data.id})
        }).error(setErrorInScope.bind($scope));
    }
});

app.component("modelComparisonLiftComparator", {
    bindings: {
        comparables: '<',
        colors: '<'
    },
    templateUrl: '/templates/modelcomparisons/comparison-tabs/lift_comparator.html',
    controller: ['$scope', '$filter', 'StateUtils',
        function ctrlLiftComparator($scope, $filter, StateUtils) {
            const ctrl = this;
            $scope.ctrl = ctrl;
            $scope.StateUtils = StateUtils;

            function refreshEchartsData() {
                if (ctrl.comparables && ctrl.comparables.length
                    && ctrl.colors && ctrl.colors.length) {
                    ctrl.chartOptions = {
                        legend: {
                            type: 'scroll',
                            show: true,
                            orient: 'vertical',
                            x: 'right',
                            y: 'center',
                            textStyle: {
                                lineOverflow: 'truncate',
                                overflow: 'truncate',
                                width: 160
                            },
                            tooltip: {
                                show: true,
                                trigger: 'item',
                                formatter: (params) => {
                                    return sanitize(params.name);
                                }
                            },
                            itemStyle: { opacity: 0 },
                        },
                        textStyle: { fontFamily: 'SourceSansPro' },
                        animation: false,
                        tooltip: {
                            confine: true,
                            trigger: 'item',
                            formatter: (params) => {
                                let ret = `<b>${sanitize($filter('nicePrecision')(params.value[0],2)*100)}%</b><br/>`
                                ret += `${params.marker}${sanitize(params.seriesName)}: <b>${sanitize($filter('nicePrecision')(params.value[1],2)*100)}%</b>`;
                                return ret;
                            },
                            rich: {
                                bold: { fontWeight: 'bold' }
                            },
                        },
                        grid: [],
                        xAxis: [],
                        yAxis: [],
                        series: [],
                        title: []
                    };

                    filterLiftVizComparables();
                    computeLineChartOptions();
                    computeBarChartOptions();
                } else {
                    ctrl.chartOptions = null;
                }
            }
            ctrl.$onChanges = refreshEchartsData;

            $scope.uiState = {
                liftLessModelsMsg: null
            };

            function filterLiftVizComparables() {
                $scope.comparablesWithLift = ctrl.comparables.filter(cp => cp.details.perf && cp.details.perf.liftVizData);
                if ($scope.comparablesWithLift.length != ctrl.comparables.length) {
                    const filteredCount = ctrl.comparables.length - $scope.comparablesWithLift.length;
                    $scope.uiState.liftLessModelsMsg = `${filteredCount} model${(filteredCount>1)?'s have ':' has '} no lift data and ${(filteredCount>1)?'were ':'was '} filtered out`;
                } else {
                    $scope.uiState.liftLessModelsMsg = null;
                }
            }

            function computeLineChartOptions() {
                const series = $scope.comparablesWithLift.map(
                    (cp,idx) => {
                        let data;
                        const name = cp.displayInfo.championedName;
                        if (cp.details.perf) {
                            if (!cp.details.perf.liftVizData.folds) {
                                data = [[0, 0]].concat(cp.details.perf.liftVizData.bins.map(b => [b.cum_size,b.cum_lift]));
                            } else {
                                data = [[0, 0]].concat(cp.details.perf.liftVizData.folds[0].map(b => [b.cum_size,b.cum_lift, 0]));
                            }
                        } else {
                            data = [];
                        }
                        return getStdSeriesLine(ctrl.colors[idx], name, data);
                    }
                ).flat();

                series.push({
                    name: 'Random',
                    type: 'line',
                    symbol: 'none',
                    smooth: false,
                    lineStyle: {
                        type: 'dashed'
                    },
                    color: '#444444',
                    data: series.length?series[0].data.map(xy => [xy[0], xy[0]]):[...Array(10).keys()].map(x => [x/10, x/10]),
                    xAxisIndex: ctrl.chartOptions.grid.length,
                    yAxisIndex: ctrl.chartOptions.grid.length,
                });

                ctrl.chartOptions.xAxis.push({
                    type: 'value',
                    splitNumber: 10,
                    name: 'Observations by decreasing probability',
                    nameLocation: 'middle',
                    nameGap: 30,
                    gridIndex: ctrl.chartOptions.grid.length,
                    axisPointer: {
                        show: true,
                        snap: true,
                        lineStyle: {
                            type: "dashed"
                        },
                        triggerTooltip: false
                    },
                });

                ctrl.chartOptions.yAxis.push({
                    type: 'value',
                    axisLabel: {
                        formatter: function(value) {
                            const ret = sanitize(`${value*100} %`);
                            if ((0 == value) || (1 == value)) {
                                return `{bold|${ret}}`;
                            } else {
                                return ret;
                            }
                        },
                        rich: {
                            bold: {
                                fontWeight: 'bold'
                            }
                        }
                    },
                    splitNumber: 10,
                    name: 'Positive cases captured',
                    nameLocation: 'middle',
                    nameGap: 50,
                    gridIndex: ctrl.chartOptions.grid.length,
                    axisPointer: {
                        show: true,
                        snap: true,
                        lineStyle: { type: "dashed" },
                        triggerTooltip: false
                    },
                });

                ctrl.chartOptions.series = ctrl.chartOptions.series.concat(series);

                ctrl.chartOptions.grid.push({
                    height: 360,
                    top: 40,
                    left: 80,
                    right: 180,
                });
            }

            function computeBarChartOptions() {
                const fullNamesMap = new Map();
                const series = $scope.comparablesWithLift.map(
                    (cp,idx) => {
                        const name = cp.displayInfo.championedName;
                        return {
                            id: cp.ref.fullId,
                            name,
                            type: 'bar',
                            barGap: 0,
                            color: ctrl.colors[idx],
                            data: cp.details.perf?cp.details.perf.liftVizData.bins.map(b => b.bin_lift):[],
                            xAxisIndex: ctrl.chartOptions.grid.length,
                            yAxisIndex: ctrl.chartOptions.grid.length,
                        }
                    }
                ).flat();

                series.forEach(s =>
                    s.markLine = {
                        lineStyle: {
                            color: '#ff0000',
                            width: 1,
                            type: 'dashed'
                        },
                        symbol: 'none',
                        data: [
                            {
                                yAxis: 1
                            },
                        ],
                        label: {
                            normal: {
                                show: false
                            }
                        },
                        tooltip: {
                            show: false
                        }
                    }
                );

                ctrl.chartOptions.xAxis.push({
                    type: 'category',
                    data: series.length?Array.from(Array(series[0].data.length).keys()):[],
                    name: 'Observations by decreasing probability decile',
                    nameLocation: 'middle',
                    nameGap: 30,
                    axisLabel: {
                        formatter: function(value) {
                            return sanitize(parseInt(value)+1);
                        }
                    },
                    splitLine: {
                        show: true,
                        lineStyle: {
                            type: 'dashed'
                        }
                    },
                    gridIndex: ctrl.chartOptions.grid.length
                });

                ctrl.chartOptions.xAxis.push({
                    type: 'value',
                    name: 'Observations for random model',
                    show: false,
                    gridIndex: ctrl.chartOptions.grid.length
                });

                ctrl.chartOptions.yAxis.push({
                    type: 'value',
                    name: 'Lift on bin',
                    nameLocation: 'middle',
                    nameGap: 50,
                    gridIndex: ctrl.chartOptions.grid.length
                });

                ctrl.chartOptions.series = ctrl.chartOptions.series.concat(series);

                ctrl.chartOptions.grid.push({
                    height: 300,
                    top: 400 * ctrl.chartOptions.grid.length + 56,
                    left: 80,
                    right: 180,
                    tooltip: {
                        trigger: 'item',
                        formatter: (params) => {
                            return `${params.marker} ${sanitize(params.seriesName)}, bin ${sanitize(parseInt(params.name)+1)}: ${sanitize($filter('nicePrecision')(params.value,4))}`;
                        }
                    }
                });
            };

            $scope.differentLiftPoints = function() {
                if (!ctrl.comparables || !ctrl.comparables.length) return false;

                const liftPoints = ctrl.comparables.map(cp => cp.metricParams.liftPoint);
                const firstLiftPoint = liftPoints[0];
                return liftPoints.some(liftPoint => liftPoint !== firstLiftPoint);
            };
        }
    ]
});

app.controller("EditComparedModelsSelectionController", function($scope, StateUtils, ModelComparisonHelpers, LLMEvaluationMetrics, CreateModalFromTemplate) {
    $scope.colors = window.dkuColorPalettes.discrete.find(palette => palette.id === "dku_font").colors;
    $scope.StateUtils = StateUtils;
    $scope.uiState.comparedModels = [];
    $scope.uiState.displayParams = null;
    $scope.uiState.missingRefs = [];
    $scope.init = function(modelTaskType, modelTaskTypeName, comparedModels, displayParams, comparables, missings, refs) {
        $scope.uiState.additionalComparedModelsRefIds = new Set();
        $scope.uiState.modelTaskType = modelTaskType;
        $scope.uiState.modelTaskTypeName = modelTaskTypeName;
        $scope.uiState.isLLM = LLMEvaluationMetrics.llmModelTaskTypes.map(t => t.type).includes(modelTaskType);
        $scope.uiState.comparedModels = angular.copy(comparedModels);
        $scope.uiState.displayParams = angular.copy(displayParams);
        if (!missings || !missings.length) {
            $scope.uiState.missingRefs = [];
        } else {
            $scope.uiState.missingRefs = missings.map(m => m.refId);
        }
        $scope.uiState.comparedModels.forEach((cm, idx) => {
            cm.championSymbol = ModelComparisonHelpers.getChampionSymbol();
            cm.displayName = comparables[idx]?comparables[idx].userMeta.name:("Deleted " + cm.refId);
            const ref = $scope.uiState.refs.find(r => r.ref.fullId === cm.refId);
            if (ref) {
                cm.taskType = ref.taskType;
                cm.modelTaskType = ref.modelTaskType;
                cm.proxyModelProtocol = ref.proxyModelProtocol;
                cm.savedModelType = ref.savedModelType;
                cm.backendType = ref.backendType;
            }
        });
    }

    $scope.comparableNotFound = function(toCheck) {
        return $scope.uiState.missingRefs.includes(toCheck.refId);
    }

    $scope.resetColors = function() {
        const palette = ModelComparisonHelpers.getPalette();
        $scope.uiState.comparedModels.forEach((cm, idx) => {
            cm.color = palette[idx];
        });
    }

    $scope.massDisplay = function(items) {
        items.forEach(i => i.isSelected = true);
    }

    $scope.massHide = function(items) {
        items.forEach(i => i.isSelected = false);
    }

    $scope.massRemove = function(items) {
        const refIds = items.map(i => i.refId);
        refIds.forEach(refId => $scope.uiState.additionalComparedModelsRefIds.delete(refId));
        $scope.uiState.comparedModels = $scope.uiState.comparedModels.filter(cm => !refIds.includes(cm.refId));
    }

    $scope.addMELikes = function() {
        CreateModalFromTemplate("/templates/modelcomparisons/modals/add-melikes-modal/add-melikes-modal.html", $scope, "AddMELikesToModelComparisonController", function(newScope) {
            newScope.init($scope.uiState.comparedModels, $scope.uiState.modelTaskType, $scope.uiState.modelTaskTypeName);
        }).then(function(additionalComparedModels) {
            for (let addedCM of additionalComparedModels) {
                $scope.uiState.additionalComparedModelsRefIds.add(addedCM.refId);
            }
            $scope.uiState.comparedModels.push(...additionalComparedModels);
            ModelComparisonHelpers.adjustComparableModelsColors($scope.uiState.comparedModels);
        });
    }

    $scope.apply = function() {
        const uiStateSelectedModels = $scope.uiState.comparedModels.filter(comparable => comparable.isSelected)
        const mcSelectedModels = $scope.modelComparison.comparedModels.filter(comparable => comparable.isSelected)
        if (!_.isEqual(new Set(uiStateSelectedModels.map(model => model.displayName)), new Set(mcSelectedModels.map(model=> model.displayName)))) {
            // Trye when the lengths are different (added, hidden or deleted model) or when a model has been renamed
            $scope.modelComparison.$forceSampleRefresh = true;
        }

        const newModels = $scope.uiState.comparedModels.filter(cm => $scope.uiState.additionalComparedModelsRefIds.has(cm.refId));
        $scope.resolveModal([$scope.uiState.comparedModels, newModels, $scope.uiState.displayParams]);
    }

    $scope.reorderRowCallback = function(draggedLine, hoveredLine, rowId, referenceRowId) {
        let rowOldPosition;
        let rowNewPosition;
        const comparableIds = $scope.uiState.comparedModels.map(cm => cm.refId);
        rowOldPosition = comparableIds.indexOf(rowId);
        rowNewPosition = comparableIds.indexOf(referenceRowId);
        const curItem = $scope.uiState.comparedModels[rowOldPosition];

        if (rowOldPosition < 0 || rowNewPosition < 0) {
            return;
        }

        $scope.uiState.comparedModels.splice(rowOldPosition, 1);
        if (rowNewPosition === 0) {
            $scope.uiState.comparedModels.splice(0, 0, curItem);
        } else if (rowNewPosition === $scope.uiState.comparedModels.length) {
            $scope.uiState.comparedModels.push(curItem);
        } else {
            $scope.uiState.comparedModels.splice(rowNewPosition, 0, curItem);
        }
        $scope.$digest();
    }

    $scope.remove = function(item) {
        $scope.uiState.additionalComparedModelsRefIds.delete(item.refId);
        $scope.uiState.comparedModels = $scope.uiState.comparedModels.filter(cm => cm.refId !== item.refId);
    }

    $scope.clearChampion = function(item) {
        $scope.uiState.comparedModels.forEach(cm => {
            cm.isChampion = false;
        });
    }

    $scope.setChampion = function(item) {
        $scope.uiState.comparedModels.forEach(cm => {
            cm.isChampion = cm.refId === item.refId;
        });
    }

    $scope.startEditName = function(item) {
        item.originalName = item.displayName;
    }

    $scope.revertEditName = function(item) {
        item.displayName = item.originalName;
        item.originalName = undefined;
    }

    $scope.getOriginStyle = function(item) {
        return ModelComparisonHelpers.getOriginStyle(item);
    }
});

const MODEL_ADDITION_STATES = {
    // Select type of ME is handled on a specific autonomous component, thus the absence of fragment
    SELECT_TYPE: {
        name: "SELECT_TYPE",
        back: null,
        initial: true,
        title: 'Select Model Evaluations Category',
        hideSubmitButton: true
    },
    FROM_SAVED_MODELS: {
        name: "FROM_SAVED_MODELS",
        final: true,
        fragment: '/templates/modelcomparisons/modals/add-melikes-modal/fragments/add-melikes-saved-models.html',
        title: 'Select Evaluations of Saved Models deployed in the Flow',
        titleIcon: 'icon universe-color saved-model dku-icon-machine-learning-regression-24'
    },
    FROM_ANALYSIS: {
        name: "FROM_ANALYSIS",
        final: true,
        fragment: '/templates/modelcomparisons/modals/add-melikes-modal/fragments/add-melikes-analysis.html',
        title: 'Select Evaluations of Models from Visual Analyses',
        titleIcon: 'icon universe-color analysis dku-icon-ml-analysis-24',
    },
    FROM_MES: {
        name: "FROM_MES",
        final: true,
        fragment: '/templates/modelcomparisons/modals/add-melikes-modal/fragments/add-melikes-mes.html',
        title: 'Select Evaluations from Evaluation Stores',
        titleIcon: 'icon universe-color model-evaluation-store dku-icon-model-evaluation-store-24',
    }
};

MODEL_ADDITION_STATES.FROM_SAVED_MODELS.back = MODEL_ADDITION_STATES.SELECT_TYPE.name;
MODEL_ADDITION_STATES.FROM_ANALYSIS.back = MODEL_ADDITION_STATES.SELECT_TYPE.name;
MODEL_ADDITION_STATES.FROM_MES.back = MODEL_ADDITION_STATES.SELECT_TYPE.name;

MODEL_ADDITION_STATES.FROM_MES.relatedModelsStateName = MODEL_ADDITION_STATES.FROM_SAVED_MODELS.name;
MODEL_ADDITION_STATES.FROM_SAVED_MODELS.relatedModelsStateName = MODEL_ADDITION_STATES.FROM_MES.name;

const MELIKES_CATEGORIES = {
    SAVED_MODELS: {
        label: 'Select Saved Model Versions from the Flow',
        icon: 'icon universe-color saved-model dku-icon-machine-learning-regression-48',
        state: MODEL_ADDITION_STATES.FROM_SAVED_MODELS,
        modelTaskTypes: [ "BINARY_CLASSIFICATION", "REGRESSION", "MULTICLASS", "TIMESERIES_FORECAST", "CAUSAL_BINARY_CLASSIFICATION", "CAUSAL_REGRESSION" ]
    },
    ANALYSES: {
        label: 'Select Lab Models',
        icon: 'icon universe-color analysis dku-icon-ml-analysis-48 ',
        state: MODEL_ADDITION_STATES.FROM_ANALYSIS,
        modelTaskTypes: [ "BINARY_CLASSIFICATION", "REGRESSION", "MULTICLASS", "TIMESERIES_FORECAST", "CAUSAL_BINARY_CLASSIFICATION", "CAUSAL_REGRESSION" ]
    },
    MES: {
        label: 'Select Evaluations from Stores',
        icon: 'icon universe-color dku-icon-model-evaluation-store-48',
        state: MODEL_ADDITION_STATES.FROM_MES
    }
};

function melCategoriesFormodelTaskType(modelTaskType) {
    const categories = {};
    for (const key in MELIKES_CATEGORIES) {
        const category = MELIKES_CATEGORIES[key];
        if (!category.modelTaskTypes || category.modelTaskTypes.includes(modelTaskType)) {
            categories[key] = category;
        }
    }
    return categories;
}



// use createOrAppendMELikeToModelComparisonModalDirective when calling CreateModalFromComponent
app.component("createOrAppendMELikeToModelComparisonModal", {
    bindings: {
        modalControl: '<',
        fullIds: '<',
        modelTaskType: '<',
        allowImportOfRelatedEvaluations: '<',
        allowImportOfRelatedVersions: '<',
        suggestedMCName: '<',
        projectKey: '<',
        trackFrom: '<', // WT1 tracking
        areAllModelsExternal: '<'
    },
    templateUrl: '/templates/modelcomparisons/modals/create-or-append-to-mc-modal.html',
    controller: function($scope, DataikuAPI, $q, $state, $stateParams, ModelComparisonHelpers, CustomMetricIDService, PMLSettings, LLMEvaluationMetrics, WT1) {
        const $ctrl = this;
        $ctrl.$onInit = function () {
            $ctrl.compatibleModelComparisons = []; // List of existing ModelComparison to which these ME-like can be appended
            $ctrl.selectedModelComparison = null; // Selected ModelComparison (in APPEND mode)
            $ctrl.mode = 'CREATE'; // CREATE or APPEND
            $ctrl.importRelatedEvaluations = false;
            $ctrl.importRelatedVersions = false;
            $ctrl.saving = false; // Whether the ModelComparison is being updated
            $ctrl.mcName = $ctrl.suggestedMCName;
            $ctrl.isLLM = LLMEvaluationMetrics.llmModelTaskTypes.map(t => t.type).includes($ctrl.modelTaskType);
            listCompatibleModelComparisons();
        }

        function listCompatibleModelComparisons() {
            DataikuAPI.modelcomparisons.list($stateParams.projectKey).success(function(data) {
                $ctrl.compatibleModelComparisons = data.filter(data => data.modelTaskType == $ctrl.modelTaskType);
            });
        }

        function appendToModelComparison(fullIds) {
            if(!$ctrl.selectedModelComparison) {
                return;
            }

            for(const fullId of fullIds) {
                const alreadyAdded = $ctrl.selectedModelComparison.comparedModels.find(cm => cm.refId == fullId);

                if(!alreadyAdded) {
                    $ctrl.selectedModelComparison.comparedModels.push({
                        refId: fullId,
                        isSelected: true
                    });
                }
            }

            DataikuAPI.modelcomparisons.getModelsDetails(fullIds).then((resp) => {
                const modelsDetails = resp.data;
                if (!$ctrl.isLLM) {
                    const evaluationMetrics = _.uniq(modelsDetails.map(md => md.metricParams.evaluationMetric));
                    evaluationMetrics.forEach((evaluationMetric) => {
                        if (!$ctrl.selectedModelComparison.displayParams.pinnedMetrics.includes(evaluationMetric)) {
                            $ctrl.selectedModelComparison.displayParams.pinnedMetrics.push(evaluationMetric);
                        }
                    });
                }
                WT1.event("model-evaluation-comparison-appended", { from: $ctrl.trackFrom, nbItems: fullIds.length });
                return DataikuAPI.modelcomparisons.save($ctrl.selectedModelComparison)
                .then(()=> {
                    $ctrl.modalControl.resolve();
                    $state.go("projects.project.modelcomparisons.modelcomparison.summary", {modelComparisonId: $ctrl.selectedModelComparison.id})
                })
                .catch(setErrorInScope.bind($scope));
            });
        }

        function createModelComparison(fullIds) {
            WT1.event("model-evaluation-comparison-created", { from: $ctrl.trackFrom, predictionType: $ctrl.modelTaskType, nbItems: fullIds.length }); // predictionType name is kept for retro-compatibility
            return DataikuAPI.modelcomparisons.create($ctrl.projectKey, $ctrl.mcName, $ctrl.modelTaskType).then(function({data}) {
                const uniqFullIds = _.uniq(fullIds);
                data.comparedModels = uniqFullIds.map(fullId => ({
                    refId: fullId,
                    isSelected: true
                }))

                DataikuAPI.modelcomparisons.getModelsDetails(uniqFullIds).then((resp) => {
                    const modelsDetails = resp.data;
                        if (!$ctrl.isLLM) {
                            data.displayParams.pinnedMetrics = _.uniq(modelsDetails.map(md => {
                                if (md.metricParams) {
                                    if (md.metricParams.evaluationMetric === "CUSTOM" && md.metricParams.customEvaluationMetricName) {
                                        return CustomMetricIDService.getCustomMetricId(md.metricParams.customEvaluationMetricName);
                                    } else {
                                        return md.metricParams.evaluationMetric;
                                    }
                                } else {
                                    return null;
                                }
                            }
                            )).filter(m => m !== null);
                        }
                    return DataikuAPI.modelcomparisons.save(data)
                    .then(()=> {
                        $ctrl.modalControl.resolve();
                        $state.go("projects.project.modelcomparisons.modelcomparison.summary", {modelComparisonId: data.id})
                    })
                });
            });
        }

        $ctrl.compare = function() {
            $ctrl.saving = true;
            const expandedFullIds = [$q.when($ctrl.fullIds)];

            if (!$ctrl.isLLM) {
                if($ctrl.allowImportOfRelatedEvaluations && $ctrl.importRelatedEvaluations) {
                    expandedFullIds.push(
                        ModelComparisonHelpers.lookupEvaluationsOfSavedModelVersions($ctrl.projectKey, $ctrl.modelTaskType, $ctrl.fullIds)
                    );
                }

                if($ctrl.allowImportOfRelatedVersions && $ctrl.importRelatedVersions) {
                    expandedFullIds.push(
                        ModelComparisonHelpers.lookupSavedModelVersionsFromTheirEvaluations($ctrl.fullIds)
                    );
                }
            }

            $q.all(expandedFullIds)
                .then(fullIds => fullIds.flat())
                .then(fullIds => {
                    if($ctrl.mode == 'CREATE') {
                        createModelComparison(fullIds);
                    }
                    if($ctrl.mode == 'APPEND') {
                        appendToModelComparison(fullIds);
                    }
                })
                .catch(setErrorInScope.bind($scope)) // TODO check me
                .finally(()=> $ctrl.saving = false);
        }

        $ctrl.canCompare = function() {
            return !$ctrl.saving && ($ctrl.mode == 'CREATE' || ($ctrl.mode == 'APPEND' && $ctrl.selectedModelComparison));
        }

        $ctrl.switchToAppendModeIfPossible = function() {
            if($ctrl.compatibleModelComparisons && $ctrl.compatibleModelComparisons.length > 0){
                $ctrl.mode = 'APPEND';
            }
        }


        $scope.getAddSMLabelTooltip = function() {
            if($ctrl.areAllModelsExternal) {
                return "Unavailable for evaluations not backed by a DSS model.";
            }
            return "";
        }

    }
});

app.controller("AddMELikesToModelComparisonController", function($scope, StateUtils, DataikuAPI,
    $stateParams, TreeViewSelectorUtils, WT1, ModelComparisonHelpers, LLMEvaluationMetrics, ComparablesService) {
    $scope.StateUtils = StateUtils;
    $scope.meLikesCategories = [];
    $scope.statesStack = [];
    $scope.uiState.currentComparedModels = [];
    $scope.uiState.proposedMELikes = [];
    $scope.uiState.selectedLabels = [];
    $scope.uiState.availableLabels = [];
    $scope.uiState.selectedLabels = [];

    // TODO: make this configurable
    $scope.LABELS_TO_DISPLAY = ["id", "model:algorithm", "model:date", "evaluation:date", "evaluationDataset:dataset-name"];

    $scope.init = function(currentComparedModels, modelTaskType, modelTaskTypeName, state, comparables) {
        if (!state) {
            state = Object.entries(MODEL_ADDITION_STATES).filter(([_, value]) => value.initial).map(([key,_]) => key);
            $scope.statesStack = [MODEL_ADDITION_STATES[state]];
        } else {
            $scope.statesStack = [state];
        }
        $scope.uiState.currentComparedModels = angular.copy(currentComparedModels);
        $scope.uiState.selectedCategory = null;
        $scope.uiState.modelTaskType = modelTaskType;
        $scope.meLikesCategories = melCategoriesFormodelTaskType(modelTaskType);

        $scope.uiState.modelTaskTypeName = modelTaskTypeName;
        $scope.uiState.isLLM = LLMEvaluationMetrics.llmModelTaskTypes.map(t => t.type).includes(modelTaskType);
        $scope.uiState.addMEsRelatedToSMs = true;
        if (!comparables){
            $scope.comparables = {};
            ComparablesService.fetchComparables($stateParams.projectKey, modelTaskType)
                .then(ret => {
                    $scope.comparables = ret.comparableItems;
                    $scope.infoMessages = ret.infoMessages;
                })
        } else {
            $scope.comparables = comparables;
        }
    }

    $scope.currentState = function() {
        return $scope.statesStack[$scope.statesStack.length -1];
    }

    $scope.canGoBack = function() {
        return $scope.statesStack.length > 1;
    }

    $scope.back = function() {
        $scope.statesStack.pop();
    }

    $scope.selectCategory = function(categ) {
        $scope.uiState.selectedCategory = categ;
        $scope.submit();
    }

    $scope.canSubmit = function() {
        switch ($scope.currentState().name) {
            case MODEL_ADDITION_STATES.SELECT_TYPE.name:
                return $scope.uiState.selectedCategory;
            case MODEL_ADDITION_STATES.FROM_ANALYSIS.name:
            case MODEL_ADDITION_STATES.FROM_SAVED_MODELS.name:
            case MODEL_ADDITION_STATES.FROM_MES.name:
                return $scope.hasNewSelectedTreeNodes();
            default:
                return false;
        }
    }

    $scope.hasNewSelectedTreeNodes = function() {
        return $scope.uiState.selectedLeafs && $scope.uiState.initiallySelectedLeafs
            && $scope.uiState.selectedLeafs.length > $scope.uiState.initiallySelectedLeafs.length;
    }

    $scope.prepareAddedModels = function() {
        const addedTreeNodes = _.difference($scope.uiState.selectedLeafs, $scope.uiState.initiallySelectedLeafs);
        return addedTreeNodes.map(atn => {
            return {
                refId: atn.payload.mli.fullId,
                isSelected: true,
                isChampion: false,
                evaluatedSavedModelVersion: atn.payload.evaluatedSavedModelVersion?atn.payload.evaluatedSavedModelVersion.fullId:atn.payload.mli.fullId,
                evaluationMetric: atn.payload.evaluationMetric,
                displayName: atn.data[0],
                championSymbol: ModelComparisonHelpers.getChampionSymbol()
            }
        });
    }

    $scope.submit = function() {
        if ($scope.currentState().final) {
            let addedModels = $scope.prepareAddedModels();

            WT1.event("model-evaluation-comparison-appended", { from: 'model-comparison-add-modal', nbItems: addedModels.length });

            if ($scope.currentState().relatedModelsStateName && $scope.uiState.addMEsRelatedToSMs) {
                DataikuAPI.modelcomparisons.browseComparables($stateParams.projectKey, $scope.currentState().relatedModelsStateName, $scope.uiState.modelTaskType,
                    // evaluatedSavedModelVersion is the SMV fullid if am is a saved model version, the full id of the evaluated SMV is am is a ME
                    addedModels.map(am => am.evaluatedSavedModelVersion))
                        .success(ret => {
                            const relatedModels = ret.comparableModelItems;
                            const smMesMap = new Map();
                            for (const relatedModel of relatedModels) {
                                const id = relatedModel.evaluatedSavedModelVersion?relatedModel.evaluatedSavedModelVersion.fullId:relatedModel.mli.fullId;
                                if (!smMesMap.has(id)) {
                                    smMesMap.set(id, []);
                                }
                                smMesMap.get(id).push(relatedModel);
                            }
                            let idsAlreadyInModelComparison = new Set($scope.uiState.currentComparedModels.map(cp => cp.refId));
                            const initiallyAddedModels = angular.copy(addedModels);
                            for (const addedModel of initiallyAddedModels) {
                                if (smMesMap.has(addedModel.refId)) {
                                    for (const meToAdd of smMesMap.get(addedModel.refId)) {
                                        if (!idsAlreadyInModelComparison.has(meToAdd.mli.fullId)) {
                                            addedModels.push({
                                                refId: meToAdd.mli.fullId,
                                                isSelected: true,
                                                evaluatedSavedModelVersion: meToAdd.evaluatedSavedModelVersion,
                                                evaluationMetric: meToAdd.evaluationMetric,
                                                displayName: meToAdd.evaluationName,
                                                championSymbol: ModelComparisonHelpers.getChampionSymbol()
                                            });
                                            idsAlreadyInModelComparison.add(meToAdd.mli.fullId);
                                        }
                                    }
                                } else if (smMesMap.has(addedModel.evaluatedSavedModelVersion)) {
                                    for (const meToAdd of smMesMap.get(addedModel.evaluatedSavedModelVersion)) {
                                        if (!idsAlreadyInModelComparison.has(meToAdd.mli.fullId)) {
                                            addedModels.push({
                                                refId: meToAdd.mli.fullId,
                                                isSelected: true,
                                                evaluatedSavedModelVersion:  meToAdd.mli.fullId,
                                                evaluationMetric: meToAdd.evaluationMetric,
                                                displayName: meToAdd.savedModelVersionName,
                                                championSymbol: ModelComparisonHelpers.getChampionSymbol()
                                            });
                                            idsAlreadyInModelComparison.add(meToAdd.mli.fullId);
                                        }
                                    }
                                }
                            }
                            $scope.resolveModal(addedModels);
                        });
            } else {
                $scope.resolveModal(addedModels);
            }
        }
        if ($scope.currentState() == MODEL_ADDITION_STATES.SELECT_TYPE) {
            $scope.statesStack.push($scope.uiState.selectedCategory.state);
        }
    }

    function transformMELikesIntoTree(comparables, labelsToDisplay, alreadyDisplayedIds) {
        const comparablesWithGroups = comparables.map(comparable => {
            let groups = [];
            if(comparable.type == 'ANALYSIS_TRAINED_MODEL') {
                groups = [comparable.analysisName, comparable.mlTaskName, comparable.mli.sessionId];
            }
            if(comparable.type.includes('MODEL_EVALUATION')) { // tabular or llm ME
                groups = [comparable.storeName];
            }
            if(comparable.type == 'SAVED_MODEL_VERSION') {
                groups = [comparable.savedModelName];
            }
            return {
                payload: comparable,
                groups: groups
            };
        })

        const sortedComparablesWithGroups = _.orderBy(comparablesWithGroups, cwg => cwg.payload.createdOn, 'desc');

        return TreeViewSelectorUtils.treeify(sortedComparablesWithGroups, (comparable) => {
            const selected = alreadyDisplayedIds.includes(comparable.mli.fullId);
            let message;
            if (comparable.isPartitioned) {
                message = "Item is partitioned and can not be added to a comparison";
            } else if (comparable.isEnsembled) {
                message = "Item is ensembled and can not be added to a comparison";
            } else if (selected) {
                message = "Item is already in comparison";
            }
            const node = {
                type: 'leaf',
                data: [],
                payload: comparable,
                disabled: selected || comparable.isPartitioned || comparable.isEnsembled,
                preselected: selected,
                message
            };
            for (const label of labelsToDisplay) {
                if ("id" == label) {
                    node.data.push(comparable.displayName);
                } else {
                    node.data.push(comparable.labels[label]);
                }
            }
            return node;
        });
    }

    $scope.onEnterState = function() {
        switch ($scope.currentState().name) {
            case MODEL_ADDITION_STATES.FROM_ANALYSIS.name:
            case MODEL_ADDITION_STATES.FROM_SAVED_MODELS.name:
            case MODEL_ADDITION_STATES.FROM_MES.name:
                $scope.uiState.initiallySelectedLeafs = [];
                $scope.uiState.selectedLeafs = [];
                $scope.uiState.expandedNodes = [];
                $scope.uiState.proposedMELikesTree = undefined;
                if ($scope.comparables && Object.keys($scope.comparables).length > 0){
                    const models = $scope.comparables[$scope.currentState().name]
                    $scope.uiState.availableLabels = ["id"].concat(_.uniq(models.flatMap(m => Object.keys(m.labels))));
                    $scope.uiState.selectedLabels = $scope.LABELS_TO_DISPLAY.filter(l => $scope.uiState.availableLabels.includes(l));
                    const alreadyDisplayedIds = $scope.uiState.currentComparedModels.map(cp => cp.refId);
                    [$scope.uiState.proposedMELikesTree, $scope.uiState.initiallySelectedLeafs] = transformMELikesIntoTree(models, $scope.uiState.selectedLabels, alreadyDisplayedIds);
                    $scope.uiState.selectedLeafs = $scope.uiState.initiallySelectedLeafs;
                }
                break;
        }
    }

    $scope.onStatesStackChange = function(nv) {
        if (nv) {
            $scope.onEnterState();
        }
    }

    $scope.allSelectedEvaluationsAreExternal = function() {
        if($scope.uiState.selectedLeafs.length === 0){
            return false;
        }
        return $scope.uiState.selectedLeafs
        .map((c) => c.payload)
        .every(p => isExternalModelType(p) === true)
    }

    $scope.getAddSMLabelTooltip = function() {
        if($scope.allSelectedEvaluationsAreExternal()) {
            return "Unavailable for evaluations not backed by a DSS model.";
        }
        return "";
    }

    $scope.$watchCollection("statesStack", $scope.onStatesStackChange);
});

app.component("modelComparisonDecisionComparator", {
    bindings: {
        comparables: '<',
        colors: '<',
        showConfidenceIntervals: '<'
    },
    templateUrl: '/templates/modelcomparisons/comparison-tabs/decision_comparator.html',
    controller: ['$scope', '$filter', 'StateUtils','ModelComparisonHelpers',
        function ctrlDecisionComparator($scope, $filter, StateUtils) {
            const ctrl = this;
            $scope.refreshEchartsData = function() {
                if (ctrl.comparables && ctrl.comparables.length
                    && ctrl.colors && ctrl.colors.length) {
                    let grid = [];
                    let series = [];
                    let xAxis = [];
                    let yAxis = [];
                    const fullNamesMap = new Map();

                    const HEIGHT = 180;
                    const HEIGHT_MARGIN = 30;
                    const TOP_MARGIN = 20;
                    const LEGEND_WIDTH = 140;

                    const metrics = ['precision', 'recall', 'f1', 'cmg'];
                    const stdmetrics = metrics.map(m => m + 'std');
                    const metricNames = ['Precision', 'Recall', 'F1-score', 'Cost Matrix Gain'];

                    metrics.forEach((_, metricIdx) =>{
                        grid.push({
                            left: '3%',
                            top: grid.length * (HEIGHT + HEIGHT_MARGIN) + TOP_MARGIN + 'px',
                            height: HEIGHT,
                            right: `${LEGEND_WIDTH+40}px`,
                            containLabel: true,
                            });

                        xAxis.push({
                            type: 'value',
                            scale: true,
                            gridIndex: metricIdx,
                            nameLocation: 'middle',
                            nameGap: 20,
                            name: 'Cut-off'
                        });

                        yAxis.push({
                            type: 'value',
                            gridIndex: metricIdx,
                            splitNumber: 5,
                            nameLocation: 'middle',
                            nameGap: 40,
                            name: metricNames[metricIdx]
                        });
                    });

                    const seriesWeighted = [];

                    ctrl.comparables.forEach((comparable, comparableIdx) => {
                        const name = comparable.displayInfo.championedName;
                        const fullName = `${comparable.displayInfo.displayName}`;
                        fullNamesMap.set(name, fullName);
                        const pcd = comparable.details.perf && comparable.details.perf.perCutData;
                        if (!pcd) {
                            return;
                        }

                        metrics.forEach((metric, metricIdx) =>{
                            if (!ctrl.showConfidenceIntervals || !pcd[stdmetrics[metricIdx]]) {
                                seriesWeighted.push(isComparableWeighted(comparable)?' (Weighted)':'');
                                const data = _.zip(pcd.cut, pcd[metric]);
                                let serieLine = Object.assign({},
                                        getStdSeriesLine(ctrl.colors[comparableIdx], name, data),
                                        {
                                            xAxisIndex: metricIdx,
                                            yAxisIndex: metricIdx

                                        }
                                    );
                                if(isExternalModelType(ctrl.comparables[comparableIdx])){
                                    serieLine.lineStyle.type = 'dashed';
                                }
                                series.push(serieLine);
                            } else {
                                // we plot dotted lines for higher and upper bounds of confidence interval
                                // because, as of the current version, echarts does not correctly plot
                                // confidence bands in complex graphs
                                // There is a ticket on this topic on echarts, but closed because stale for
                                // several years https://github.com/apache/echarts/issues/4645
                                const std = pcd[stdmetrics[metricIdx]];
                                const val = pcd[metric];
                                const lower = val.map((v, idx) => v - 2*std[idx]);
                                const upper = val.map((v, idx) => v + 2*std[idx]);
                                [lower, upper, val].forEach((currentValues) => {
                                    seriesWeighted.push(isComparableWeighted(comparable)?' (Weighted)':'');
                                    let lineStyle;
                                    if (currentValues != val) {
                                        lineStyle = {
                                            color: ctrl.colors[comparableIdx],
                                            opacity: .5,
                                            type: 'dotted'

                                        }
                                    } else {
                                        lineStyle = {
                                            color: ctrl.colors[comparableIdx],
                                            width: 1
                                        }
                                    }
                                    const data = _.zip(pcd.cut, currentValues);
                                    series.push(Object.assign({}, getStdSeriesLine(ctrl.colors[comparableIdx], name, data), {
                                        lineStyle,
                                        xAxisIndex: metricIdx,
                                        yAxisIndex: metricIdx
                                    }));
                                });
                                series[series.length-1].lowerUpper= [lower, upper];
                                series[series.length-2].band = true;
                                series[series.length-3].band = true;
                            }
                        });
                    });
                    yAxis.forEach((axis,idx) => {
                        axis.min = 0;
                        axis.max = _.max(series.filter((_,seriesIdx) => idx == (seriesIdx % metrics.length))
                            .flatMap(s => s.data).map(d => d[1]));
                        axis.axisLabel = {
                                formatter: function(value) {
                                    const ret = sanitize($filter('nicePrecision')(value,2));
                                    if ((axis.min == value) || (axis.max == value)) {
                                        return `{bold|${ret}}`;
                                    } else {
                                        return ret;
                                    }
                                },
                                rich: {
                                    bold: {
                                        fontWeight: 'bold'
                                    }
                                }
                        };
                    });

                    ctrl.chartOptions = {
                        grid,
                        tooltip: {
                            trigger: 'item',
                            showContent: true,
                            formatter: (params) => {
                                // As we are in 'item' trigger mode, echarts passes as params the series
                                // of the currenlty hovered item, along with its x and y values
                                // We get from series the values for all the series corresponding to the current
                                // metric, filtering series corresponding to the lower/upper confidence band,
                                // when available
                                const currentSeries = series[params.seriesIndex];
                                let ret = `<b>Cut: ${sanitize($filter('nicePrecision')(params.value[0],3))}</b><br/>`;
                                const currentMetricSeries = series.filter(s => s.xAxisIndex === currentSeries.xAxisIndex);
                                ret += `<b>${sanitize(metricNames[currentSeries.xAxisIndex])}</b><br/>`;
                                const curCutIdx = series[params.seriesIndex].data.map(x => x[0]).indexOf(params.value[0]);
                                for (const [idxCurMetricSeries, cms] of currentMetricSeries.entries()) {
                                    if (cms.band) {
                                        continue;
                                    }
                                    if (cms.lowerUpper) {
                                        const lowerBound = $filter('nicePrecision')(cms.lowerUpper[0][curCutIdx],4);
                                        const upperBound = $filter('nicePrecision')(cms.lowerUpper[1][curCutIdx],4);
                                        ret += `${params.marker.replace(/#[a-f0-9]{6}/i, sanitize(cms.lineStyle.color))} ${sanitize(cms.name)}: ${sanitize($filter('nicePrecision')(cms.data[curCutIdx][1],4))} [${sanitize(lowerBound)}-${sanitize(upperBound)}]${sanitize(seriesWeighted[idxCurMetricSeries])}<br/>`;
                                    } else {
                                        ret += `${params.marker.replace(/#[a-f0-9]{6}/i, sanitize(cms.lineStyle.color))} ${sanitize(cms.name)}: ${sanitize($filter('nicePrecision')(cms.data[curCutIdx][1],4))}${sanitize(seriesWeighted[idxCurMetricSeries])}`;
                                        if(isExternalModelType(ctrl.comparables[idxCurMetricSeries])) {
                                            ret += " (External Model: Unknown cut-off)";
                                        }
                                        ret += "<br/>";
                                    }
                                }
                                return ret;
                            },
                            rich: {
                                bold: {
                                    fontWeight: 'bold'
                                }
                            },
                            confine: true
                        },
                        axisPointer: {
                            show: true,
                            link: [
                                {
                                    xAxisIndex: 'all'
                                },
                                {
                                    yAxisIndex: [0,1,2]
                                }
                            ],
                            snap: true,
                            lineStyle: {
                                type: "dashed"
                            },
                            triggerTooltip: false
                        },
                        xAxis,
                        yAxis,
                        animation: false,
                        series,
                        legend: {
                            type: 'scroll',
                            show: true,
                            orient: 'vertical',
                            x: 'right',
                            y: 'center',
                            textStyle: {
                                lineOverflow: 'truncate',
                                overflow: 'truncate',
                                width: LEGEND_WIDTH
                            },
                            tooltip: {
                                show: true,
                                trigger: 'item',
                                formatter: (params) => {
                                    return sanitize(fullNamesMap.get(params.name));
                                }
                            }
                        },
                        color: ctrl.colors
                    };
                } else {
                    ctrl.chartOptions = null;
                }
            }
            ctrl.$onChanges = function() {
                $scope.refreshEchartsData();
            }
            $scope.ctrl = ctrl;
            $scope.StateUtils = StateUtils;


            $scope.differentCmgMatrices = function() {
                if (!ctrl.comparables || !ctrl.comparables.length) {
                    return false;
                }
                const allCostMatrixWeights = ctrl.comparables.map(cp => cp.metricParams.costMatrixWeights);
                const firstCostMatrixWeights = allCostMatrixWeights[0];
                return allCostMatrixWeights.some(function(costMatrixWeights) {
                    return costMatrixWeights.tpGain !== firstCostMatrixWeights.tpGain
                        || costMatrixWeights.tnGain !== firstCostMatrixWeights.tnGain
                        || costMatrixWeights.fpGain !== firstCostMatrixWeights.fpGain
                        || costMatrixWeights.fnGain !== firstCostMatrixWeights.fnGain;
                });
            }
        }
    ]
});

app.component("addMelikesSelectType", {
    bindings: {
        projectKey: '<',
        modelTaskType: '<',
        meLikesCategories : '<',
        selectCategory : '<',
        selectedCategory: '<',
        comparables : '<'
    },
    templateUrl: '/templates/modelcomparisons/modals/add-melikes-modal/fragments/add-melikes-select-type.html',
    controller: function(){
        const ctrl = this;

        ctrl.disabledCategory = {}

        ctrl.$onChanges = () => {
            ctrl.comparables && disableSelectCategories(ctrl.comparables);
        }

        const disableSelectCategories = newComparables => {
            Object.entries(newComparables).forEach(elem => {
                const category = elem[0];
                const comparables = elem[1];
                if (["CAUSAL_BINARY_CLASSIFICATION", "CAUSAL_REGRESSION"].includes(ctrl.modelTaskType) && category === 'FROM_MES') {
                    ctrl.disabledCategory[category] = "Model evaluations are not available for causal models";
                    return;
                }
                if (comparables.length === 0) {
                    switch (category) {
                        case "FROM_SAVED_MODELS":
                            ctrl.disabledCategory[category] = "There are no compatible saved model versions in this project.";
                            break;
                        case "FROM_ANALYSIS":
                            ctrl.disabledCategory[category] = "There are no compatible Lab models in this project.";
                            break;
                        case "FROM_MES":
                            ctrl.disabledCategory[category] = "There are no compatible model evaluations in this project.";
                            break;
                        default:
                            break;
                    }
                }
            });
        }
    }
});



app.service("ComparablesService", function(DataikuAPI, $q) {

    const categories = {
        "FROM_SAVED_MODELS": ["SAVED_MODEL_VERSION"],
        "FROM_ANALYSIS": ["ANALYSIS_TRAINED_MODEL"],
        "FROM_MES": ["MODEL_EVALUATION", "LLM_MODEL_EVALUATION"]
    };

    function fetchComparables(projectKey, modelTaskType){
        return DataikuAPI.modelcomparisons.browseComparables(projectKey, null, modelTaskType).
            then(function(resp) {
                const ret = {
                    comparableItems: {}
                };
                Object.entries(categories).forEach(([reqType,objType]) => {
                    ret.comparableItems[reqType] = resp.data.comparableModelItems.filter(c => objType.includes(c.type));
                });
                ret.infoMessages = resp.data.infoMessages;
                return ret;
            });
    }

    return { fetchComparables }
})

app.component("modelComparisonRowByRowAnalysisComparator", {
    templateUrl: '/templates/modelcomparisons/comparison-tabs/row_by_row_analysis.html',
    bindings: {
        comparables: '<',
        colors: '<',
        modelEvaluationInfos: '<',
        modelComparison: '<'
    },
    controller: function($scope) {
        $scope.$watch("$ctrl.comparables", (nv) => {
            $scope.comparables = nv;
        })
        $scope.$watch("$ctrl.colors", (nv) => {
            $scope.colors = nv;
        })
        $scope.$watch("$ctrl.modelComparison", (nv) => {
            $scope.modelComparison = nv;
        })
    }
});


app.directive("llmComparisonRowByRowAnalysis", function() {
    return {
        scope: true,
        controller: function($scope, $controller, WT1, $rootScope, $stateParams, $filter, DataikuAPI, MonoFuture, FutureWatcher, CustomMetricIDService, MetricsUtils, PMLFilteringService, LLMEvaluationMetrics) {
            WT1.event("llmcomparison-row-by-row-analysis-open");

            $scope.shakerWithSteps = false;
            $scope.additionalColumns = [];
            //let fme = FullModelLikeIdUtils.buildModelEvaluationFmeFromComponents(ActiveProjectKey.get(), $stateParams.mesId, $stateParams.evaluationId)
            $scope.canWriteProject = () => $rootScope.topNav.isProjectAnalystRW;
            $scope.isLocalSave = true;
            $scope.translate = $rootScope.translate;

            $scope.getFmeIds = (comparables) => comparables.map(comparable => comparable.ref.fullId);


            // Nothing to save
            $scope.shakerHooks.saveForAuto = function() {
                return Promise.resolve({});
            }

            const monoFuturizedRefresh = MonoFuture($scope).wrap(DataikuAPI.modelcomparisons.sampleRefreshTable);

            /*
            Since we are plugged into the shaker table, we send a request with a list of selected columns. We therefore don't receive the full row. Since we need it for the bottom part of the
            comparison screen, we made the backend call return both the tableWithSelectedColumns and the fullTable. We hack the monoFuturizedRefresh and extract the fullTable and passing to the shaker
            only the tableWithSelectedColumns
             */
            $scope.fullChunks = [];
            $scope.shakerHooks.getRefreshTablePromise = function(filtersOnly, filterRequest) {
                function transform(future) {
                    if (future.status === 200 && future.data.result) {
                        $scope.fullTable = future.data.result.fullTable;
                        $scope.fullChunks = [$scope.fullTable.initialChunk];
                        future.data.result = {...future.data.result.tableWithSelectedColumns};
                    }
                    return future;
                }

                const initialPromise = monoFuturizedRefresh($stateParams.projectKey, $stateParams.modelComparisonId, $scope.getFmeIds($scope.comparables), $scope.shaker, filterRequest);
                const newPromise = initialPromise.then(transform, null, transform);
                FutureWatcher.enrichPromise({promise : newPromise});
                return newPromise;
            }

            $scope.shakerHooks.shakerForQuery = function(){
                return angular.copy($scope.shaker);
            }

            $scope.shakerHooks.updateColumnWidth = function(name, width) {
                $scope.shaker.columnWidthsByName[name] = width;
                $scope.refreshTable(false);
            };

            $scope.shakerHooks.afterTableRefresh = function() {
                const content = $scope.table.initialChunk.content;
                if (!content || !content.length) {
                    return;
                }
                const cols = $scope.table.initialChunk.nbCols;
                let row = [];
                for (let i = 0; i < cols; i++) {
                    row.push(content[i]);
                }
                $scope.shakerState.selectedRow = 0;
                $scope.lastSelectedRow = row;

                // Trying to apply the saved column widths to the newly selected columns width
                const shakerColumnsWithWidth = Object.keys($scope.shaker.columnWidthsByName);
                if (shakerColumnsWithWidth.length == 1
                        && shakerColumnsWithWidth.includes('Input')
                        && $scope.modelComparison.displayParams
                        && $scope.modelComparison.displayParams.rowByRowShakerScript) {
                    const rowByRowShakerScript = JSON.parse($scope.modelComparison.displayParams.rowByRowShakerScript);
                    const savedColumnsWithWidth = Object.keys(rowByRowShakerScript.columnWidthsByName);
                    const shakerAvailableColumns = new Set($scope.shaker.columnsSelection.list.map(col => col.name))
                    for (const savedColumnName of savedColumnsWithWidth) {
                        if (!shakerAvailableColumns.has(savedColumnName)) {
                            delete rowByRowShakerScript.columnWidthsByName[savedColumnName];
                        }
                    }
                    $scope.shaker.columnWidthsByName = rowByRowShakerScript.columnWidthsByName;
                }

                $scope.modelComparison.displayParams.rowByRowShakerScript = JSON.stringify($scope.shaker);
                $scope.$emit("saveIfDirty");
                setTimeout(() => {
                    $rootScope.$broadcast("compareColumnValuesChangeRow", 'previous');
                    // Signal to Puppeteer that the content of the element has been loaded and is thus available for content extraction
                    $scope.puppeteerHook_elementContentLoaded = true;
                });
            }

            $scope.shakerHooks.getTableChunk = function(firstRow, nbRows, firstCol, nbCols, filterRequest) {
                function transform(result) {
                    if (result.status === 200) {
                        if (!$scope.fullChunks.map(fc => fc.firstRow).includes(result.data.fullChunk.firstRow)) {
                            $scope.fullChunks.push(result.data.fullChunk);
                        }
                        result.data = result.data.chunkWithSelectedColumns;
                    }
                    return result;
                }

                const initialPromise = DataikuAPI.modelcomparisons.sampleGetTableChunk($stateParams.projectKey, $stateParams.modelComparisonId, $scope.getFmeIds($scope.comparables),
                    $scope.shaker, firstRow, nbRows, firstCol, nbCols, filterRequest);
                const newPromise = initialPromise.then(transform, null, transform);
                FutureWatcher.enrichPromise({promise : newPromise});
                return newPromise;
            }

            $scope.shakerReadOnlyActions = true;
            $scope.shakerWritable = false;
            $scope.shakerCellClickable = true;
            $scope.isCompareCellAvailable = false;

            $scope.setSpinnerPosition = function() { };

            let folded = {};
            $scope.isFolded = function(key) {
                return folded[key];
            }
            $scope.foldOrUnfold = function(key) {
                if (folded[key]) {
                    delete folded[key];
                } else {
                    folded[key] = true;
                }
            }


            $scope.$on('shakerCellClick', function(event, rowPromise, colId, fatId) {
                // no filtering on the source table, we never show two comparisons together
                rowPromise.then(setNewRow);
            });

            $scope.shakerHooks.fetchDetailedAnalysis = function(setAnalysis, handleError, columnName, alphanumMaxResults, fullSamplePartitionId, withFullSampleStatistics) {
                // withFullSampleStatistics, fullSamplePartitionId are not relevant in this context
                DataikuAPI.modelcomparisons.sampleDetailedColumnAnalysis($stateParams.projectKey, $stateParams.modelComparisonId, $scope.getFmeIds($scope.comparables),
                                                                         $scope.shakerHooks.shakerForQuery(), columnName, alphanumMaxResults).success(function(data){
                    setAnalysis(data);
                }).error(function(a, b, c) {
                    if (handleError) {
                        handleError(a, b, c);
                    }
                    setErrorInScope.bind($scope)(a, b, c);
                });
            };

            // Load shaker
            $scope.$watch('comparables', (nv ,ov) => {
                if (!$scope.comparables || angular.equals(nv, ov)) {
                    return;
                }

                if (!$scope.modelComparison.displayParams.rowByRowShakerScript || $scope.modelComparison.$forceSampleRefresh) { // Only initialize the columns for the first time
                    $scope.modelComparison.$forceSampleRefresh = false;
                    resetShaker(false);
                } else {
                    $scope.autoSaveForceRefresh();
                }
            })

            $scope.$watch('evaluationDetails', (nv, ov) => {
                $scope.lastSelectedRow = null;
                $scope.searchableDataset = false;
                $scope.baseInit();
                if ($scope.modelComparison.displayParams.rowByRowShakerScript) {
                    $scope.shaker = JSON.parse($scope.modelComparison.displayParams.rowByRowShakerScript);
                    $scope.shakerState.filtersExplicitlyAllowed = true;
                    $scope.autoSaveForceRefresh();
                } else {
                    $scope.shaker = {};
                    resetShaker(true);
                }
            });

            function resetShaker(isInitialCall) {
                let defaultMetric = null;

                if ($scope.comparables && $scope.comparables.length > 0) {
                    const commonMetrics = _.intersection(...$scope.comparables.map(comparable => Object.keys(comparable.metrics)));
                    if (commonMetrics.length > 0) {
                        defaultMetric = commonMetrics[0];
                    }
                }

                DataikuAPI.modelcomparisons.getSampleSchema($stateParams.projectKey, $scope.getFmeIds($scope.comparables))
                    .then(function({data}) {
                        const allNewColumns = data.columns.map(column => column.name);
                        // We reset the additional columns select but we keep those that are still selectable
                        const additionalSelectableColumns = allNewColumns
                            .filter(fullColumnName => "Input" !== fullColumnName)
                            .map(fullColumnName => extractParts(fullColumnName)[0])
                            .filter(columnName => !["Output", "Ground truth", "Context"].includes(columnName))

                        $scope.modelComparison.displayParams.rowByRowAdditionalColumns = _.intersection($scope.modelComparison.displayParams.rowByRowAdditionalColumns, additionalSelectableColumns);

                        // Same for sorting and exploration filters : we keep them on still existing columns
                        const sorting = !$scope.shaker.sorting ? [] : $scope.shaker.sorting.filter(sorting => "Input" === sorting.column || (sorting.column.startsWith(defaultMetric) && allNewColumns.includes(sorting.column)));
                        const explorationFilters = !$scope.shaker.explorationFilters ? [] : $scope.shaker.explorationFilters.filter(filter => "Input" === filter.column || (filter.column.startsWith(defaultMetric) && allNewColumns.includes(filter.column)));
                        $scope.shaker = {
                            steps: [],
                            globalSearchQuery: "",
                            columnWidthsByName: {Input: 600},
                            "$headerOptions": {
                                showName: true,
                                showMeaning: false,
                                showDescription: false,
                                showCustomFields: false,
                                showProgressBar: false,
                                disableHeaderMenu: false,
                            },
                            sorting: sorting,
                            explorationFilters: explorationFilters,
                            columnsSelection: {
                                list: data.columns.map(column => {
                                    if (column.name === "Input") {
                                        return { "name": "Input", "d": true }
                                    }
                                    if (defaultMetric !== null && column.name.startsWith(defaultMetric)) {
                                        return { "name": column.name, "d": true }
                                    } else {
                                        return { "name": column.name, "d": false }
                                    }
                                }),
                                mode: "SELECT"
                            }
                        };

                        if (isInitialCall) {
                            $scope.shakerState.filtersExplicitlyAllowed = true;
                            $scope.originalShaker = angular.copy($scope.shaker);
                            $scope.fixupShaker();
                            $scope.refreshTable(false);
                        } else {
                            $scope.autoSaveForceRefresh();
                        }
                    }).catch(setErrorInScope.bind($scope));
            }

            function extractParts(fullColName) {
                const match = fullColName.match(/(.+?)\s+for\s+(.+)/);
                return match ? [match[1], match[2]] : [null, null];
            }

            function setNewRow(row) {
                if (!$scope.table) {
                    return;
                }
                const rowId = row[0].rowId;
                $scope.isLastRow = rowId === $scope.table.totalRows - 1;
                $scope.isFirstRow = rowId === 0;


                const fullSelectedRow = {}
                $scope.comparables.forEach(comparable => {
                    const meName = comparable.userMeta.name;
                    const labels = comparable.userMeta.labels;
                    fullSelectedRow[meName] = {labels, values : {}}
                })
                const fullChunk = $scope.fullChunks.find(fc => fc.firstRow <= rowId && fc.firstRow+fc.nbRows > rowId);
                $scope.lastSelectedRow = row;
                const startIdx = (rowId - fullChunk.firstRow) * $scope.fullTable.initialCols;

                for (let i = 0; i < $scope.fullTable.initialCols; i++) {
                    const fullColumnName = $scope.fullTable.headers[i].name;
                    const value = fullChunk.content[startIdx + i];
                    if (fullColumnName === "Input") {
                        $scope.selectedInput = value;
                    } else {
                        const [column, meName] = extractParts(fullColumnName);
                        if (meName != null && meName in fullSelectedRow) {
                            fullSelectedRow[meName].values[column] = value;
                        }
                    }
                }

                $scope.additionalColumns = _.uniq(...Object.values(fullSelectedRow).map(me => Object.keys(me.values)))
                    .filter(column => !['Ground truth', 'Context', 'Output'].includes(column));

                $scope.additionalColumnValues = $scope.additionalColumns.reduce((acc, cur) => ({...acc, [cur] : []}), {});

                const meNameAndColors = [];
                const outputs = [];
                const contexts = [];
                const groundTruths = [];
                const allMetricsToDisplay = [];

                $scope.comparables.forEach((comparable, idx) => {
                    const meName = comparable.userMeta.name;
                    meNameAndColors.push({
                        championedName: comparable.displayInfo.championedName,
                        color: $scope.colors[idx],
                        ref: comparable.displayInfo.href
                    })
                    outputs.push(fullSelectedRow[meName].values["Output"])
                    try {
                        contexts.push(JSON.parse(fullSelectedRow[meName].values["Context"]))
                    } catch {
                        contexts.push(fullSelectedRow[meName].values["Context"]) // Before we did JSON.dumps in the python code
                    }

                    groundTruths.push(fullSelectedRow[meName].values["Ground truth"])

                    // e.g. (ANSWER_RELEVANCY, Answer Relevancy, answerRelevancy)
                    let keyAndNameByField = Object.keys(LLMEvaluationMetrics.nameByMetricCode)
                        .reduce((res, key) => {
                            res[PMLFilteringService.metricMap[key]] = [key, PMLFilteringService.getEvaluationMetricName(key)];
                            return res;
                          }, {});
                    let metricsToDisplay = []
                    for (let field of Object.keys(comparable.metrics)) {
                        const keyAndName = keyAndNameByField[field]
                        if (keyAndName) {
                            const metricKey = keyAndName[0];
                            const metric = LLMEvaluationMetrics.getMetricByCode(metricKey);
                            const formattedValue = MetricsUtils.getMetricValue(comparable.metrics, metricKey, 3);

                            metricsToDisplay.push({
                                metricName: metric ? (metric.labelRowByRow ? metric.labelRowByRow : keyAndName[1]) : keyAndName[1],
                                formattedValue: formattedValue,
                                description: metric ? (metric.descriptionRowByRow ? metric.descriptionRowByRow : metric.shortDescription) : undefined,
                            })
                        }
                    }

                    if(comparable.metrics.customMetricsResults) {
                        for (let customMetric of comparable.metrics.customMetricsResults) {
                            const metricName =  customMetric.metric.name;
                            const formattedValue = $filter('mlMetricFormat')(customMetric.value, metricName, 3);
                            metricsToDisplay.push({
                                metricName: metricName,
                                formattedValue: formattedValue,
                                description: customMetric.metric.description,
                                customMetric: customMetric.metric
                            })
                        }
                    }

                    allMetricsToDisplay.push(metricsToDisplay.sort((a, b) => a.metricName.localeCompare(b.metricName)))

                    $scope.additionalColumns.forEach(column => {
                        $scope.additionalColumnValues[column].push(fullSelectedRow[meName].values[column])
                    })
                });

                $scope.displayMes = meNameAndColors;
                $scope.outputs = outputs;
                $scope.hasOutput = $scope.outputs.some(elem => elem !== undefined)

                $scope.contexts = contexts;
                $scope.hasContext = $scope.contexts.some(elem => elem !== undefined)

                $scope.groundTruths = groundTruths;
                $scope.hasGroundTruth = $scope.groundTruths.some(elem => elem !== undefined)

                $scope.allMetricsToDisplay = allMetricsToDisplay;

                $scope.$apply();
            }

            $scope.isMultimodalContext = context => {
                return context && Array.isArray(context) && context.some(source => source && typeof(source) === "object" && source.excerpt && source.excerpt.type === "IMAGE_REF")
            }


            $rootScope.$on("compareCellValueNewRowSelected", (_event, rowPromise, fatId) => {
                // no filtering on the source table, we never show two comparisons together
                rowPromise.then(setNewRow);
            });


            $scope.onChangeRow = function(direction) {
                setTimeout(() => {
                    $rootScope.$broadcast("compareColumnValuesChangeRow", direction);
                    // Signal to Puppeteer that the content of the element has been loaded and is thus available for content extraction
                    $scope.puppeteerHook_elementContentLoaded = true;
                });
            }

            $scope.saveSelectedColumns = function() {
                $scope.$emit("saveIfDirty");
            }
        }
    }
});

app.directive("llmComparisonRowByRowTextOverflowHandler", function($timeout) {
    return {
        scope: {
            isFolded: "&",
            foldKey: "="
        },
        link: function(scope, element) {
            function eventHandler() {
                $timeout(function () {
                    const FOLDING_HEIGHT_THRESHOLD = 28;

                    var overflowTitle = element.find(".row-by-row--overflow");
                    var noOverflowTitle = element.find(".row-by-row--no-overflow");
                    var unfoldedText = element.find(".row-by-row--unfolded");
                    var foldedText = element.find(".row-by-row--folded");
                    var contextImages = element.find(".row-by-row--images");

                    // We ensure the unfolded text is displayed since we need
                    // it to check for text overflow and hide it again at the end
                    unfoldedText.css("display", "block")
                    // We temporarily hide one of the row headers because having
                    // both for a split second can disturb our clientHeight logic
                    noOverflowTitle.css("display", "none")

                    // Check if any text in the row is overflowing
                    scope.isOverflowing = false;

                    // If there are images displayed then the row is considered as overflowing
                    if (contextImages.length > 0) {
                        scope.isOverflowing = true;
                    }
                    else {
                        // Folded: check clientWidth
                        if (scope.isFolded()) {
                            for (let i=0; i < foldedText.length; i++) {
                                let fText = foldedText[i];
                                let rawText = fText.textContent;
                                let ufText = unfoldedText[i];
                                // clientWidth can vary if one or several models are compared
                                // therefore we use the folded width as a comparison
                                // if there are any linebreak characters it should be considered as overflowing anyways
                                if (ufText.clientWidth > fText.clientWidth || /\r?\n/.test(rawText)) {
                                    scope.isOverflowing = true;
                                    break;
                                }
                            }
                        }
                        // Unfolded: check clientHeight
                        else {
                            for (let ufText of unfoldedText) {
                                if (ufText.clientHeight > FOLDING_HEIGHT_THRESHOLD) {
                                    scope.isOverflowing = true;
                                    break;
                                }
                            }
                        }
                    }
                    // We modify elements depending on text overflow
                    if (scope.isOverflowing) {
                        if (scope.foldKey !== "input") {
                            overflowTitle.css("display", "table-cell");
                        }
                        else {
                            overflowTitle.css("display", "flex");
                        }
                        noOverflowTitle.css("display", "none");
                        foldedText.css("cursor", "pointer");
                        foldedText.css("pointer-events", "");

                        // Replace text after first linebreak with "..."
                        // use thin spaces to match css ellipsis appearance
                        for (let i=0; i < foldedText.length; i++) {
                            let rawText = foldedText[i].textContent;
                            if (/\r?\n/.test(rawText)) {
                                var splitText = rawText.split(/\r?\n/)[0] + ".  .  .";
                                foldedText[i].textContent = splitText;
                            }
                        }
                    }
                    else {
                        overflowTitle.css("display", "none");
                        if (scope.foldKey !== "input") {
                            noOverflowTitle.css("display", "table-cell");
                        }
                        else {
                            noOverflowTitle.css("display", "flex");
                        }
                        noOverflowTitle.css("cursor", "default");
                        foldedText.css("cursor", "default");
                        foldedText.css("pointer-events", "none");
                    }
                    if (scope.isFolded()) {
                        unfoldedText.css("display", "none")
                    }
                })
            }

            function foldHandler() {
                $timeout(function () {
                    var unfoldedText = element.find(".row-by-row--unfolded");
                    unfoldedText.css("display", "block");
                    if(scope.isFolded()) {
                        unfoldedText.css("display", "none");
                    }
                })
            }

            eventHandler(); // Run the overflow check once on init

            var unbindChangeRow = scope.$on("compareColumnValuesChangeRow", eventHandler)
            var unbindRowSelected = scope.$on("compareCellValueNewRowSelected", eventHandler)
            var unbindShakerCell = scope.$on("shakerCellClick", eventHandler)
            var unbindFoldUnfold = scope.$watch(function() {return scope.isFolded()}, foldHandler)

            // Clean up the event listeners on directive destruction
            scope.$on("$destroy", function() {
                unbindChangeRow();
                unbindRowSelected();
                unbindShakerCell();
                unbindFoldUnfold();
            });
        }
    };
});

app.directive("llmComparisonRowByRowStickyHeader", function($interval) {
    return {
        link: function(scope, element) {
            const intervalPromise = $interval(() => {

                // Try to initialize the HTML elements
                const tableContainer = element[0].querySelector(".table-container");
                const tableElem = element[0].querySelector(".row-by-row-llm-table");
                const tableHead = tableElem ? tableElem.querySelector("thead") : undefined;
                const tableHeadLeftCorner = tableHead ? tableHead.querySelector(".tabular-data-comparator__top-left-corner") : undefined;

                // If all are properly set, cancel the interval and run the logic
                if (tableContainer && tableHead && tableHeadLeftCorner) {

                    $interval.cancel(intervalPromise);
                    let stickyEnabled = false;
                    let scrollValue = null;

                    element.on("scroll compareColumnValuesChangeRow compareCellValueNewRowSelected shakerCellClick", function() {
                        const tableHeadTop = tableHead.getBoundingClientRect().top;
                        const containerTop = element[0].getBoundingClientRect().top;
                        const offset = tableHeadTop - containerTop;

                        if (stickyEnabled === false) {
                            if (offset < 0) {
                                angular.element(tableHead).addClass("header--fixed");
                                angular.element(tableHeadLeftCorner).addClass("header__top-left-corner--fixed");
                                angular.element(tableHead).css("width", tableContainer.getBoundingClientRect().width + "px");
                                tableHead.scrollLeft = tableContainer.scrollLeft;
                                stickyEnabled = true;
                                scrollValue = element[0].scrollTop + offset;
                            }
                        }
                        else {
                            if (element[0].scrollTop < scrollValue) {
                                angular.element(tableHead).removeClass("header--fixed");
                                angular.element(tableHeadLeftCorner).removeClass("header__top-left-corner--fixed");
                                stickyEnabled = false;
                            }
                        }
                    });

                    tableContainer.addEventListener("scroll", () => {
                        if (stickyEnabled) {
                            tableHead.scrollLeft = tableContainer.scrollLeft;
                        }
                    });

                    tableHead.addEventListener("scroll", () => {
                        if (stickyEnabled) {
                            tableContainer.scrollLeft = tableHead.scrollLeft;
                        }
                    });

                    // Make sure the table and header still match sizes on window resize
                    window.addEventListener("resize", () => {
                        if (stickyEnabled) {
                            angular.element(tableHead).css("width", tableContainer.getBoundingClientRect().width + "px");
                        }
                    });

                }
            }, 200);
        }
    };
});

app.component("llmEvalRowByRowSources", {
    templateUrl: '/templates/modelcomparisons/llm-eval-row-by-row-sources.html',
    bindings: {
        sources: '<'
    },
    controller: function() {
        const $ctrl = this;

        function displayImages() {
            if ($ctrl.sources) {
                $ctrl.displayedSources = $ctrl.sources.map(source => {
                    const displayedSource = angular.copy(source);

                    if (source.excerpt && source.excerpt.type === 'IMAGE_REF') {
                        if (source.excerpt.images && source.excerpt.images.length) {
                            // assume excerpt uses only one folder
                            const fullFolderId = source.excerpt.images[0].fullFolderId;
                            const folder = fullFolderId.split('.');
                            displayedSource.excerpt.fullFolderId = fullFolderId;
                            displayedSource.excerpt.projectKey = folder[0];
                            displayedSource.excerpt.folderId = folder[1];
                            displayedSource.excerpt.imagePaths = source.excerpt.images.map(image => image['path']);
                        }
                    }

                    return displayedSource;
                });
            }
            else { // sources are not available anymore, emptying displayedSources
                $ctrl.displayedSources = [];
            }
        }

        $ctrl.$onInit = displayImages;
        $ctrl.$onChanges = displayImages;

        $ctrl.isExpanded = {};
        $ctrl.toggleExpansion = function(index) {
            $ctrl.isExpanded[index] = !$ctrl.isExpanded[index];
        }

    }
});

})();
