(function () {
    'use strict';

    var app = angular.module('dataiku.projects.actions', []);

    app.directive("scrollyMenu", function () {
        return {
            scope: true,
            link: function (scope, element, attrs) {

                scope.submenuAlign = attrs.submenuAlign;
                // defines rules to positions a submenu when the main menu opens downward
                scope.posnMenu = function(evt) {
                    var $menuItem = $(evt.currentTarget);
                    var $submenuWrapper = $('> .scrolly-wrapper', $menuItem);
                    if ($submenuWrapper.length==0) return;

                    // grab the menu item's position relative to its positioned parent (position=relative)
                    var menuItemPos = $menuItem.position();
                    const globalMenuItemPos = $menuItem.offset();
                    const globalMenuTop = globalMenuItemPos.top
                    const styles = window.getComputedStyle($menuItem[0]);
                    const bottomMargin = Number.parseInt(styles.getPropertyValue('--dropdown-menu-bottom-margin'), 10) || 8;

                    var subT = menuItemPos.top;
                    var subH = $submenuWrapper.height();
                    var vH = $(window).height()

                    const menuRealEstate = subH + globalMenuTop

                    if (menuRealEstate + bottomMargin > vH) {
                        const diff = vH - menuRealEstate - bottomMargin;
                        subT = Math.max(Math.round(diff), 0); //shift up submenu to fit on screen

                        if (diff < 0) {
                            // If there is not enough space below to fit the list, we make the list shorter
                            const $subList = $('> ul', $submenuWrapper);
                            if ($subList.length) {
                                $subList.css({
                                    maxHeight: vH - globalMenuTop - bottomMargin + 'px'
                                })
                            }
                        }
                    }

                    // We add a small margin between the 2 dropdowns to have space for the dropdown shadow
                    const borderMargin = 0;

                    //By default display the submenu on the left
                    let subL = menuItemPos.left - $submenuWrapper.outerWidth() - borderMargin;
                    if (scope.submenuAlign === 'right') {
                        subL = menuItemPos.left + $menuItem.outerWidth() + borderMargin;
                    }

                    // place submenu relative to the positioned parent
                    $submenuWrapper.css({
                        top: subT,
                        left: subL
                    });

                }

                // defines rules to positions a submenu when the main menu opens upward
                scope.posnMenuUp = function (evt) {
                    var $menuItem = $(evt.currentTarget);
                    var $submenuWrapper = $('> .scrolly-wrapper', $menuItem);
                    if ($submenuWrapper.length == 0) return;

                    // grab the menu item's position relative to its positioned parent (position=relative)
                    var menuItemPos = $menuItem.position();

                    var subT = menuItemPos.top;

                    // the bottom of the submenu is aligned with top bottom of the menu item
                    var subH = $submenuWrapper.height();
                    var vH = $(window).height()
                    var subB = subT + $menuItem.height();

                    //By default display the submenu on the left
                    let subL = menuItemPos.left - $submenuWrapper.outerWidth();
                    if (scope.submenuAlign === 'right') {
                        subL = menuItemPos.left + $menuItem.outerWidth();
                    }

                    if ($menuItem.parent().height() - menuItemPos.top + $menuItem.height() + subH > 0.6 * vH) {
                        // if menu item position + submenu height is too high, apply special rules to avoid overflow
                        if (subH > $menuItem.parent().height() - menuItemPos.top + $menuItem.height()) {
                            // aligns submenu bottom edge with main menu bottom edge
                            $submenuWrapper.css({
                                bottom: - $menuItem.parent().height(),
                                left: subL,
                            });
                        } else {
                            // aligns submenu top edge with main menu item top edge
                            $submenuWrapper.css({
                                top: menuItemPos.top,
                                left: subL
                            });
                        }
                    } else {
                        // default case: we align submenu bottom edge with item bottom edge
                        $submenuWrapper.css({
                            bottom: -subB,
                            left: subL
                        });
                    }
                }
            }
        }
    });

    app.directive("newRecipeMenuRecipePage", function (GlobalProjectActions) {
        return {
            templateUrl: '/templates/recipes/new-recipe-menu-recipe-page.html',
            scope: true,
            link: function (scope, element, attrs) {
                scope.title = attrs.title;
                scope.displayedItems = GlobalProjectActions.getAllRecipesBySection(scope);

                scope.do = function (cb) {
                    cb();
                }
            }
        }
    });
    app.component("recipeSelectionMenu", {
        templateUrl: '/templates/recipes/recipe-selection-menu.html',
        bindings: {
            title : '@',
            onSelect: '<',
            menuItems: '<'
        },
        controller: function(){
            const $ctrl = this;

            $ctrl.select = function(item){
                $ctrl.selectedRecipe = item;
                $ctrl.onSelect(item);
            }
        }
    });

    app.directive("newStreamingRecipeMenu", function (GlobalProjectActions) {
        return {
            templateUrl: '/templates/recipes/new-streaming-recipe-menu.html',
            scope: true,
            link: function (scope, element, attrs) {
                scope.title = attrs.title;
                scope.displayedItems = GlobalProjectActions.getStreamingRecipesBySection(scope);

                scope.do = function (cb) {
                    cb();
                }
            }
        }
    });

    app.directive("zoneSelectionMenu", function (GlobalProjectActions, DataikuAPI, $rootScope, $stateParams, $state) {
        return {
            templateUrl: '/templates/recipes/zone-selection-menu.html',
            scope: true,
            link: function (scope, element, attrs) {
                scope.selectedZone = $stateParams.zoneId;
                scope.$watch(attrs.color, function(nv, ov) {
                    scope.bgColor = nv;
                    scope.fgColor = '#' + getContrastYIQ(stripNumberSign(nv));
                });
                scope.zones = [];
                scope.changeZone = () => {
                    $state.go('projects.project.flow', Object.assign({}, $stateParams, { zoneId: scope.selectedZone, id: null }));
                };

                scope.listZones = () => {
                    DataikuAPI.flow.zones.list($stateParams.projectKey).then(data => {
                        scope.zones = data.data;
                    });
                };
                scope.listZones();

                const zonesListChangedListener = $rootScope.$on("zonesListChanged", scope.listZones);
                scope.$on('$destroy', zonesListChangedListener);

                function getContrastYIQ(hexcolor) {
                    var r = parseInt(hexcolor.substr(0,2),16);
                    var g = parseInt(hexcolor.substr(2,2),16);
                    var b = parseInt(hexcolor.substr(4,2),16);
                    var yiq = ((r*299)+(g*587)+(b*114))/1000;
                    return (yiq >= 128) ? '000000' : 'FFFFFF';
                }

                function stripNumberSign(color) {
                    if(color[0] === "#") {
                        color = color.substring(1, color.length);
                    }
                    return color;
                }
            }
        }
    });

    app.directive("newStreamingEndpointMenu", function (GlobalProjectActions) {
        return {
            templateUrl: '/templates/streaming-endpoints/new-streaming-endpoint-menu.html',
            scope: true,
            link: function (scope, element, attrs) {
                scope.title = attrs.title;
                scope.displayedItems = GlobalProjectActions.getAllStreamingEndpointsBySection(scope);

                scope.do = function (cb) {
                    cb();
                }
            }
        }
    });

    app.controller("MassImportConnectionSelectionController", function ($scope, $state, $stateParams, DataikuAPI, TopNav) {
        $scope.connection = null;
        $scope.connections = [];
        DataikuAPI.connections.listMassImportSources($stateParams.projectKey).success(function (data) {
            $scope.connections = data.sources;
            $scope.hiveError = data.hiveError;
        }).error(setErrorInScope.bind($scope));
        $scope.isValid = function () {
            return $scope.connection != null;
        };
        $scope.massImport = function () {
            $state.go("projects.project.datacatalog.database-explorer", {
                projectKey: $stateParams.projectKey,
                connectionName: $scope.connection.name
            });
            $scope.dismiss();
        };
    });

    app.controller("TablesImportProjectSelectionModalController", function ($scope, $state, $stateParams, DataikuAPI) {
        DataikuAPI.projects.list().success(function (data) {
            $scope.projects = data;
        }).error(setErrorInScope.bind($scope));

        $scope.clickImport = function () {
            $scope.dismiss();
            $state.go('projects.project.tablesimport', {
                    projectKey: $scope.project,
                    importData: JSON.stringify($scope.getImportData())
            });
        };
    });

    app.controller("MassImportTablesFromCatalogModalController", function ($scope, $state, $stateParams, DataikuAPI) {
         DataikuAPI.projects.list().success(function (data) {
            $scope.projects = data;
            if ($scope.projects.length === 1) {
                $scope.project = $scope.projects[0].projectKey;
            }
        }).error(setErrorInScope.bind($scope));

        $scope.clickImport = function () {
            $scope.dismiss();
            $state.go('projects.project.tablesimport', {
                    projectKey: $scope.project,
                    importData: JSON.stringify($scope.getImportData())
            });
        }
    });


    app.factory("GlobalProjectActions", function ($stateParams, $rootScope, $filter, $state,  $translate, translate, Assert, CreateModalFromTemplate, DatasetUtils, Logger, Dialogs,
        DataikuAPI, ComputablesService, TaggableObjectsService, RecipeDescService, uiCustomizationService, SmartId, SavedModelsService, DatasetDetailsUtils,
        FlowBuildService, AnyLoc, FeatureFlagsService) {

        function ok(details) {
            return {ok: true, details: details}
        }

        function nok(reason) {
            return {ok: false, reason: reason}
        }

        function makePluginSection(pluginId, items) {
            var plugin = Array.dkuFindFn($rootScope.appConfig.loadedPlugins, function (n) {
                return n.id == pluginId
            });
            if (plugin == null || plugin.hideComponents) return null; // could have been deleted on disk or the plugin components visibility settings
            items.forEach(function (dtype) {
                if (!dtype.icon) dtype.icon = plugin.icon;
            });
            // add an item to point to the info
            items.splice(0, 0, {isInfo: true, pluginId: plugin.id});
            return {
                isSection: true,
                id: "plugin_" + plugin.id,
                icon: plugin.icon,
                label: plugin.label || plugin.id,
                items: items
            };
        }

        function mostFrequentIcon(pluginList) {
            let max = 0, last = 'icon-puzzle-piece', items = {};

            pluginList.some(function(plugin) {
                if (!plugin.icon) return false;

                items[plugin.icon] = !items[plugin.icon] ? 1 : items[plugin.icon] + 1;

                if (max < items[plugin.icon]) {
                    last = plugin.icon;
                    max = items[plugin.icon];

                    if (max > pluginList.length / 2) {
                        return true;
                    }
                }
                return false;
            });

            return last;
        }


        /* **************************************************
         * Recipes that have license restrictions
         *
         * - When you have a CE, we show as enabled and show an upgrade CTA
         * - When you have a real EE, we don't want to show a CTA so we disable the icon
         * if not configured.
         * - The "configured but not licensed mode" should be fairly rare
         */

        function notLicensedCE() {
            return {
                usable: false,
                iconEnabled: true,
                enableStatus: "NOT_LICENSED_CE"
            }
        }

        function notConfigured(reason) {
            return {
                usable: false,
                iconEnabled: false,
                enableStatus: "NOT_CONFIGURED",
                iconDisabledReason: reason
            }
        }

        function notLicensedEE() {
            return {
                usable: false,
                iconEnabled: true,
                enableStatus: "NOT_LICENSED_EE"
            }
        }

        function usable() {
            return {
                usable: true,
                iconEnabled: true,
                enableStatus: "OK"
            }
        }

        function notDataScientist() {
            return {
                usable: false,
                iconEnabled: false,
                enableStatus: "NOT_LICENSED_EE",
                iconDisabledReason: translate("PROJECT.PERMISSIONS.PROFILE_NOT_ALLOWED_RECIPE_ERROR", "Your user profile does not allow you to create this kind of recipe")
            }
        }

        function noUnsafeCode() {
            return {
                usable: false,
                iconEnabled: false,
                enableStatus: "NOT_LICENSED_EE",
                iconDisabledReason: translate("PROJECT.PERMISSIONS.WRITE_UNISOLATED_CODE_ERROR", "You may not write unisolated code")
            }
        }

        function noSafeCode() {
            return {
                usable: false,
                iconEnabled: false,
                enableStatus: "NOT_LICENSED_EE",
                iconDisabledReason: translate("PROJECT.PERMISSIONS.WRITE_ISOLATED_CODE_ERROR", "You may not write isolated code")
            }
        }


        function getHiveStatus() {
            if ($rootScope.appConfig.communityEdition) {
                return notLicensedCE();
            } else {
                if (!$rootScope.appConfig.hiveEnabled) {
                    return notConfigured(translate("PROJECT.PERMISSIONS.HIVE_NOT_CONFIGURED", "Hive not configured on this DSS instance"));
                }
                if (!$rootScope.addLicInfo.hiveLicensed) {
                    return notLicensedEE();
                }
                return usable();
            }
        }

        function getImpalaStatus() {
            if ($rootScope.appConfig.communityEdition) {
                return notLicensedCE();
            } else {
                if (!$rootScope.appConfig.impalaEnabled) {
                    return notConfigured(translate("PROJECT.PERMISSIONS.IMPALA_NOT_CONFIGURED", "Impala not configured on this DSS instance"));
                }
                if (!$rootScope.addLicInfo.impalaLicensed) {
                    return notLicensedEE();
                }
                return usable();
            }
        }

        function getPigStatus() {
            if ($rootScope.appConfig.communityEdition) {
                return notLicensedCE();
            } else {
                if (!$rootScope.appConfig.pigEnabled) {
                    return notConfigured(translate("PROJECT.PERMISSIONS.PIG_NOT_CONFIGURED", "Pig not configured on this DSS instance"));
                }
                if (!$rootScope.addLicInfo.pigLicensed) {
                    return notLicensedEE();
                }
                return usable();
            }
        }

        function getSparkStatus() {
            if ($rootScope.appConfig.communityEdition) {
                return notLicensedCE();
            } else {
                if (!$rootScope.appConfig.sparkEnabled) {
                    return notConfigured(translate("PROJECT.PERMISSIONS.SPARK_NOT_CONFIGURED", "Spark not configured on this DSS instance"));
                }
                if (!$rootScope.addLicInfo.sparkLicensed) {
                    return notLicensedEE();
                }
                return usable();
            }
        }

        function retrieveDatasetStatusForRecipe(datasetStatus, recipeType) {
            if (datasetStatus && datasetStatus.recipes && Object.prototype.hasOwnProperty.call(datasetStatus.recipes, recipeType)) {
                return {
                    usable: datasetStatus.recipes[recipeType].ok,
                    iconEnabled: datasetStatus.recipes[recipeType].ok,
                    enableStatus: datasetStatus.recipes[recipeType].ok ? "OK" : datasetStatus.recipes[recipeType].reason.enableStatus,
                    iconDisabledReason: datasetStatus.recipes[recipeType].reason ? datasetStatus.recipes[recipeType].reason.iconDisabledReason : ""
                }
            }
            return usable();
        }

        const svc = {
            /**
             * Returns [
             *   { isSection : false, ... }
             *   { isSection : true, id, label: icon: item : []}}}
             * ]
             */
            getAllRecipesBySection: function (scope) {
                var ret = [];

                const visualRecipes = {
                    isSection: true,
                    id: "visual",
                    label: translate('PROJECT.ACTIONS.RECIPES.VISUAL', "Visual"),
                    icon: "icon-eye",
                    items: [
                        {
                            type: "shaker", label: translate("PROJECT.ACTIONS.RECIPES.PREPARE", "Data preparation"),
                            fn: function () {
                                scope.showCreateShakerModal(undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                            }
                        },
                        {
                            type: "sync", label: translate("PROJECT.ACTIONS.RECIPES.SYNC", "Sync"),
                            fn: function () {
                                scope.showCreateSyncModal(undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                            }
                        },
                        {
                            type: "filter", label: translate("PROJECT.ACTIONS.RECIPES.SAMPLE", "Sample / Filter"),
                            fn: function () {
                                scope.showCreateSamplingModal(undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                            }
                        },
                        {
                            type: "grouping", label: translate("PROJECT.ACTIONS.RECIPES.GROUP", "Group"),
                            fn: function () {
                                scope.showCreateGroupingModal(undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                            }
                        },
                        {
                            type : "distinct", label : translate("PROJECT.ACTIONS.RECIPES.DISTINCT", "Distinct"),
                            fn : function(){ scope.showCreateDistinctModal(undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId)); }
                        },
                        {
                            type: "window", label: translate("PROJECT.ACTIONS.RECIPES.WINDOW", "Window"),
                            fn: function () {
                                scope.showCreateWindowRecipeModal(undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                            }
                        },
                        {
                            type: "join", label: translate("PROJECT.ACTIONS.RECIPES.JOIN", "Join"),
                            fn: function () {
                                scope.showCreateJoinModal(undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                            }
                        },
                        {
                            type: "fuzzyjoin", label: translate("PROJECT.ACTIONS.RECIPES.FUZZY_JOIN", "Fuzzy join"),
                            fn: function () {
                                scope.showCreateFuzzyJoinModal(undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                            }
                        },
                        {
                            type: "geojoin", label: translate("PROJECT.ACTIONS.RECIPES.GEO_JOIN", "Geo join"),
                            fn: function () {
                                scope.showCreateGeoJoinModal(undefined, scope.$stateParams.zoneId);
                            }
                        },
                        {
                            type: "split", label: translate("PROJECT.ACTIONS.RECIPES.SPLIT", "Split"),
                            fn: function () {
                                scope.showCreateSplitModal(undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                            }
                        },
                        {
                            type: "topn", label : translate("PROJECT.ACTIONS.RECIPES.TOPN", "Top N"),
                            fn : function(){ scope.showCreateTopNModal(undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId)); }
                        },
                        {
                            type: "sort", label : translate("PROJECT.ACTIONS.RECIPES.SORT", "Sort"),
                            fn : function(){ scope.showCreateSortModal(undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId)); }
                        },
                        {
                            type: "pivot", label : translate("PROJECT.ACTIONS.RECIPES.PIVOT", "Pivot"),
                            fn : function(){ scope.showCreatePivotModal(undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId)); }
                        },
                        {
                            type: "vstack", label: translate("PROJECT.ACTIONS.RECIPES.STACK", "Stack vertically"),
                            fn: function () {
                                scope.showCreateVStackModal(undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                            }
                        },
                        {
                            type: "generate_features", label: translate("PROJECT.ACTIONS.RECIPES.GENERATE_FEATURES", "Generate features"),
                            fn: function () {
                                scope.showCreateAfgModal(undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                            }
                        },
                        {
                            type: "labeling", label: translate("PROJECT.ACTIONS.RECIPES.LABELING", "Labeling"),
                            fn: function () {
                                scope.showCreateLabelingTaskModal(undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                            }
                        },
                        {
                            type: "merge_folder", label: translate("PROJECT.ACTIONS.RECIPES.MERGE_FOLDERS", "Merge folders"),
                            fn: function () {
                                scope.showCreateMergeFolderModal(undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                            }
                        },
                        {
                            type: "list_folder_contents", label: translate("PROJECT.ACTIONS.RECIPES.LIST_FOLDER_CONTENTS", "List Folder Contents"),
                            fn: function () {
                                scope.showCreateListFolderContentsModal(undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                            }
                        },
                        {
                            type: "list_access", label: translate("PROJECT.ACTIONS.RECIPES.LIST_ACCESS", "List Access"),
                            fn: function() {
                                scope.showCreateListAccessModal(undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                            }
                        },
                        {
                            type: "embed_documents", label: translate("PROJECT.ACTIONS.RECIPES.EMBED_DOCUMENTS", "Embed documents"),
                            fn: function () {
                                scope.showCreateEmbedDocumentsModal(undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                            }
                        },
                        {
                            type: "update", label: translate("PROJECT.ACTIONS.RECIPES.PUSH_TO_EDITABLE", "Push to Editable"),
                            fn: function () {
                                scope.showCreateUpdateModal(undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                            }
                        },
                        {
                            type: "export", label: translate("PROJECT.ACTIONS.RECIPES.EXPORT", "Export"),
                            fn: function () {
                                scope.showCreateExportModal(undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                            }
                        },
                        {
                            type: "extract_failed_rows", label: translate("PROJECT.DATASET.RIGHT_PANEL.ACTIONS.EXTRACT_FAILED_ROWS", "Extract failed rows"),
                            fn: function () {
                                scope.showExtractFailedRowsModal(undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                            }
                        },
                        {
                            type : "download", label : translate("PROJECT.ACTIONS.RECIPES.DOWNLOAD", "Download"),
                            fn : function(){ scope.showCreateDownloadModal(undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId)); }
                        }
                    ]
                };

                if($rootScope.featureFlagEnabled('upsert')) {
                    const upsertRecipe = {
                                                type: "upsert", label: translate("PROJECT.ACTIONS.RECIPES.UPSERT", "Upsert"),
                                                fn: function () {
                                                    scope.showCreateUpsertModal(undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                                                }
                                            };
                    // right after grouping
                    const index = visualRecipes.items.findIndex(item => item.type === "grouping");
                    visualRecipes.items.splice(index + 1, 0, upsertRecipe);
                }

                const edaRecipeStatus = scope.appConfig.userProfile.mayEDA ?
                    usable() : notDataScientist();

                const statisticsRecipe = {
                    type: "statistics",
                    label: translate("PROJECT.ACTIONS.RECIPES.STATISTICS", "Generate statistics"),
                    icon: "dku-icon-recipe-eda-16",
                    disabled: !edaRecipeStatus.iconEnabled,
                    reason: edaRecipeStatus.iconDisabledReason,
                    fn: function() {
                        const relevantZoneId = scope.getRelevantZoneId(scope.$stateParams.zoneId);
                        scope.showCreateEdaRecipeModal(undefined, relevantZoneId);
                    }
                };

                // insert the "statistics" recipe just after the "generate features" recipe
                const index = visualRecipes.items.findIndex(item => item.type === "generate_features");
                visualRecipes.items.splice(index + 1, 0, statisticsRecipe);

                // register the visual recipes
                ret.push(visualRecipes);

                const llmRecipes = {
                    isSection: true,
                    id: "nlp",
                    label: translate('PROJECT.ACTIONS.RECIPES.LLM_RECIPES', "GenAI"),
                    icon: "dku-icon-text-case-sensitive-16",
                    items: [
                        {
                            type: "prompt", label: translate("PROJECT.ACTIONS.RECIPES.PROMPT", "Prompt"),
                            fn: function () {
                                scope.showCreatePromptModal(undefined, undefined, undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                            }
                        },
                        {
                            type: "nlp_llm_user_provided_classification", label: translate("PROJECT.ACTIONS.RECIPES.CLASSIFY_TEXT", "Classify text"),
                            fn: function () {
                                scope.showCreateNLPClassificationRecipeSuperModal(undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                            }
                        },
                        {
                            type: "nlp_llm_summarization", label: translate("PROJECT.ACTIONS.RECIPES.SUMMARIZE_TEXT", "Summarize text"),
                            fn: function () {
                                scope.showCreateNLPLLMSummarizationRecipeModal(undefined, undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                            }
                        },
                        {
                            type: "nlp_llm_rag_embedding", label: translate("PROJECT.ACTIONS.RECIPES.EMBED_DATASET", "Embed dataset"),
                            fn: function () {
                                scope.showCreateNLPRAGEmbeddingRecipeModal(undefined, undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                            }
                        }
                    ]
                };

                const fullLLMMeshStatus = scope.appConfig.userProfile.mayFullLLMMesh ? usable() : notDataScientist();

                llmRecipes.items.push({
                    type: "nlp_llm_finetuning",
                    label: translate("PROJECT.ACTIONS.RECIPES.FINE_TUNE", "Fine tune"),
                    disabled: !fullLLMMeshStatus.iconEnabled,
                    reason: fullLLMMeshStatus.iconDisabledReason,
                    fn: function () {
                        scope.showCreateNLPFineTuningRecipeModal(undefined, undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                    }
                });

                llmRecipes.items.push({
                    type: "nlp_llm_evaluation",
                    label: translate("PROJECT.DATASET.RIGHT_PANEL.ACTIONS.EVALUATE_LLM", "Evaluate LLM"),
                    disabled: !fullLLMMeshStatus.iconEnabled,
                    reason: fullLLMMeshStatus.iconDisabledReason,
                    fn: function () {
                        scope.showCreateNLPLLMEvaluationRecipeModal(undefined, undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                    }
                });

                ret.push(llmRecipes);

                var pyScientistStatus = scope.appConfig.userProfile.mayRegularCode ? usable() : notDataScientist();
                var rScientistStatus = scope.appConfig.userProfile.mayRegularCode ? usable() : notDataScientist();
                var jlScientistStatus = scope.appConfig.userProfile.mayRegularCode ? usable() : notDataScientist();
                var scalaScientistStatus = scope.appConfig.userProfile.mayRegularCode ? usable() : notDataScientist();
                var sqlScientistStatus = scope.appConfig.userProfile.maySQL ? usable() : notDataScientist();

                var unsafeStatus = usable();
                if (!$rootScope.appConfig.globalPermissions || !$rootScope.appConfig.globalPermissions.mayWriteUnsafeCode) {
                    unsafeStatus = noUnsafeCode();
                }
                var safeStatus = usable();
                if ($rootScope.appConfig.impersonationEnabled && (!$rootScope.appConfig.globalPermissions || !$rootScope.appConfig.globalPermissions.mayWriteSafeCode)) {
                    safeStatus = noSafeCode();
                }

                var safeCodeStatus = $rootScope.appConfig.impersonationEnabled ? safeStatus : unsafeStatus;

                const codeRecipeItems = []
                codeRecipeItems.push({
                    type: "python",
                    label: "Python",
                    disabled: !pyScientistStatus.iconEnabled || !safeCodeStatus.iconEnabled,
                    reason: pyScientistStatus.iconDisabledReason || safeCodeStatus.iconDisabledReason,
                    fn: function () {
                        scope.showCreateCodeBasedModal("python", undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                    }
                });
                if($rootScope.appConfig.uiCustomization.showR) {
                    codeRecipeItems.push({
                        type: "r",
                        label: "R",
                        disabled: !rScientistStatus.iconEnabled || !safeCodeStatus.iconEnabled || !$rootScope.appConfig.rEnabled,
                        reason: (rScientistStatus.iconDisabledReason || safeCodeStatus.iconDisabledReason) ? (rScientistStatus.iconDisabledReason || safeCodeStatus.iconDisabledReason) : ( $rootScope.appConfig.rEnabled ? null : "R not configured on your DSS instance"),
                        fn: function () {
                            scope.showCreateCodeBasedModal("r", undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId))
                        }
                    });
                }
                if($rootScope.featureFlagEnabled('julia')) {
                    codeRecipeItems.push({
                        type: "julia",
                        label: "Julia",
                        disabled: !jlScientistStatus.iconEnabled || !safeCodeStatus.iconEnabled,
                        reason: (jlScientistStatus.iconDisabledReason || safeCodeStatus.iconDisabledReason) ? (jlScientistStatus.iconDisabledReason || safeCodeStatus.iconDisabledReason) : ( $rootScope.featureFlagEnabled('julia') ? null : "Julia plugin not installed"),
                        fn: function () {
                            scope.showCreateCodeBasedModal("julia", undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId))
                        }
                    });
                }

                codeRecipeItems.push({
                    type: "sql_query",
                    label: "SQL",
                    disabled: !sqlScientistStatus.iconEnabled || !safeCodeStatus.iconEnabled,
                    reason: sqlScientistStatus.iconDisabledReason || safeCodeStatus.iconDisabledReason,
                    fn: function () {
                        scope.showSQLRecipeModal(undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId))
                    }
                });
                codeRecipeItems.push({
                    type: "shell",
                    label: "Shell",
                    disabled: !pyScientistStatus.iconEnabled || !safeCodeStatus.iconEnabled,
                    reason: pyScientistStatus.iconDisabledReason || safeCodeStatus.iconDisabledReason,
                    fn: function () {
                        scope.showCreateCodeBasedModal("shell", undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                    }
                })

                ret.push({
                    isSection: true,
                    id: "code",
                    label: translate('PROJECT.ACTIONS.RECIPES.CODE', "Code"),
                    icon: "icon-code",
                    items: codeRecipeItems,
                });

                let sparkStatus = getSparkStatus();
                var hiveStatus = getHiveStatus();
                var impalaStatus = getImpalaStatus();
                var pigStatus = getPigStatus();

                const hadoopSparkRecipeItems = [];
                if($rootScope.appConfig.uiCustomization.showTraditionalHadoop) {
                    hadoopSparkRecipeItems.push({
                        type: "hive",
                        icon: "icon-code_hive_recipe",
                        label: "Hive",
                        disabled: !hiveStatus.iconEnabled,
                        reason: hiveStatus.iconDisabledReason,
                        fn: function () {
                            if (hiveStatus.enableStatus == "OK") {
                                scope.showCreateCodeBasedModal("hive", undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId))
                            } else if (hiveStatus.enableStatus == "NOT_LICENSED_CE") {
                                scope.showCERestrictionModal("Hadoop / Hive is")
                            }
                        }
                    });

                    hadoopSparkRecipeItems.push({
                        type: "impala",
                        icon: "icon-code_impala_recipe",
                        label: "Impala",
                        disabled: !impalaStatus.iconEnabled,
                        reason: impalaStatus.iconDisabledReason,
                        fn: function () {
                            if (impalaStatus.enableStatus == "OK") {
                                scope.showCreateCodeBasedModal("impala", undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId))
                            } else if (impalaStatus.enableStatus == "NOT_LICENSED_CE") {
                                scope.showCERestrictionModal("Hadoop / Impala is")
                            }
                        }
                    });

                    hadoopSparkRecipeItems.push({
                        type: "pig",
                        label: "Pig",
                        disabled: !pigStatus.iconEnabled,
                        reason: pigStatus.iconDisabledReason,
                        fn: function () {
                            if (pigStatus.enableStatus == "OK") {
                                scope.showCreateCodeBasedModal("pig", undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId))
                            } else if (pigStatus.enableStatus == "NOT_LICENSED_CE") {
                                scope.showCERestrictionModal("Hadoop / Pig is")
                            }
                        }
                    });

                    hadoopSparkRecipeItems.push({divider: true});
                }

                hadoopSparkRecipeItems.push({
                    type: "spark_sql_query",
                    label: "Spark SQL",
                    disabled: !sqlScientistStatus.iconEnabled || !sparkStatus.iconEnabled,
                    reason: sqlScientistStatus.iconDisabledReason || sparkStatus.iconDisabledReason,
                    fn: function () {
                        if (sparkStatus.enableStatus == "OK") {
                            scope.showCreateCodeBasedModal("spark_sql_query", undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId))
                        } else if (sparkStatus.enableStatus == "NOT_LICENSED_EE") {
                            scope.showSparkNotLicensedModal();
                        } else if (sparkStatus.enableStatus == "NOT_LICENSED_CE") {
                            scope.showCERestrictionModal("Spark is")
                        }
                    }
                });

                if($rootScope.appConfig.uiCustomization.showScala) {
                    hadoopSparkRecipeItems.push({
                        type: "spark_scala",
                        label: "Spark Scala",
                        disabled: !sparkStatus.iconEnabled || !scalaScientistStatus.iconEnabled || !safeCodeStatus.iconEnabled,
                        reason: sparkStatus.iconDisabledReason || scalaScientistStatus.iconDisabledReason || safeCodeStatus.iconDisabledReason,
                        fn: function () {
                            if (sparkStatus.enableStatus == "OK") {
                                scope.showCreateCodeBasedModal("spark_scala", undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId))
                            } else if (sparkStatus.enableStatus == "NOT_LICENSED_EE") {
                                scope.showSparkNotLicensedModal();
                            } else if (sparkStatus.enableStatus == "NOT_LICENSED_CE") {
                                scope.showCERestrictionModal("Spark is")
                            }
                        }
                    });
                }

                hadoopSparkRecipeItems.push({
                    type: "pyspark",
                    label: "PySpark",
                    disabled: !sparkStatus.iconEnabled || !pyScientistStatus.iconEnabled || !safeCodeStatus.iconEnabled,
                    reason: sparkStatus.iconDisabledReason || pyScientistStatus.iconDisabledReason || safeCodeStatus.iconDisabledReason,
                    fn: function () {
                        if (sparkStatus.enableStatus == "OK") {
                            scope.showCreateCodeBasedModal("pyspark", undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId))
                        } else if (sparkStatus.enableStatus == "NOT_LICENSED_EE") {
                            scope.showSparkNotLicensedModal();
                        } else if (sparkStatus.enableStatus == "NOT_LICENSED_CE") {
                            scope.showCERestrictionModal("Spark is")
                        }
                    }
                });

                if($rootScope.appConfig.uiCustomization.showR) {
                    hadoopSparkRecipeItems.push({
                        type: "sparkr",
                        label: "SparkR",
                        disabled: !sparkStatus.iconEnabled || !rScientistStatus.iconEnabled || !safeCodeStatus.iconEnabled,
                        reason: sparkStatus.iconDisabledReason || rScientistStatus.iconDisabledReason || safeCodeStatus.iconDisabledReason,
                        fn: function () {
                            if (sparkStatus.enableStatus == "OK") {
                                scope.showCreateCodeBasedModal("sparkr", undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId))
                            } else if (sparkStatus.enableStatus == "NOT_LICENSED_EE") {
                                scope.showSparkNotLicensedModal();
                            } else if (sparkStatus.enableStatus == "NOT_LICENSED_CE") {
                                scope.showCERestrictionModal("Spark is")
                            }
                        }
                    });
                }

                ret.push({
                    isSection: true,
                    id: "code",
                    label: "Hadoop & Spark",
                    icon: "icon-HDFS",
                    items: hadoopSparkRecipeItems,
                });

                let pluginsByCategory = {};
                let recipeCategories = {};
                const pluginById = $rootScope.appConfig.loadedPlugins.reduce(function(map, obj) {
                    map[obj.id] = obj;
                    return map;
                }, {});

                $rootScope.appConfig.customCodeRecipes.forEach(function (x) {
                    const plugin = pluginById[x.ownerPluginId];
                    if (angular.isUndefined(plugin) || plugin.hideComponents) return; // could have been deleted on disk or the plugin components visibility settings

                    let category = angular.isDefined(plugin.category) ? plugin.category : 'Misc';

                    if (category.toLowerCase() in recipeCategories) {
                        category = recipeCategories[category.toLowerCase()];
                    } else {
                        recipeCategories[category.toLowerCase()] = category = $filter('capitalize')(category);
                    }

                    pluginsByCategory[category] = pluginsByCategory[category] || {};
                    pluginsByCategory[category][plugin.id] = {
                        label: plugin.label ? plugin.label : plugin.id,
                        icon: plugin.icon,
                        fn: () => scope.showCreateRecipeFromPlugin(plugin.id, null, scope.getRelevantZoneId(scope.$stateParams.zoneId))
                    }
                });

                let appsByCategory = {};
                let appCategories = {};
                $rootScope.appConfig.appRecipes.forEach(function (x) {
                    if (x.hideComponent) return; // Do not show depending on plugin component visibility settings

                    let category = x.category || translate("PROJECT.ACTIONS.RECIPES.APPLICATIONS", "Applications");
                    if (category.toLowerCase() in appCategories) {
                        category = appCategories[category.toLowerCase()];
                    } else {
                        appCategories[category.toLowerCase()] = category = $filter('capitalize')(category);
                    }

                    appsByCategory[category] = appsByCategory[category] || {};
                    appsByCategory[category][x.recipeType] = {
                        label: x.label,
                        icon: x.icon,
                        fn: () => scope.showCreateAppRecipeModal(x.recipeType)
                    }
                });

                let displayedAppSections = [];

                $.each(appsByCategory, function (category, apps) {
                    let appItems = Object.values(apps).filter(item => item.label);
                    appItems.sort(function(a, b) { return a.label.toLowerCase().localeCompare(b.label.toLowerCase()); });

                    displayedAppSections.push({
                        isSection: true,
                        id: "tag_" + category,
                        icon: mostFrequentIcon(appItems),
                        label: category,
                        items: appItems
                    });
                });

                displayedAppSections.sort(function(a, b) { return a.label.toLowerCase().localeCompare(b.label.toLowerCase()); });
                if (displayedAppSections.length > 0) {
                    ret.push({ divider: true });
                    Array.prototype.push.apply(ret, displayedAppSections);
                }

                let displayedPluginSections = [];
                let miscPlugins = null;

                $.each(pluginsByCategory, function (category, plugins) {
                    let pluginItems = Object.values(plugins).filter(item => item.label);
                    pluginItems.sort(function(a, b) { return a.label.toLowerCase().localeCompare(b.label.toLowerCase()); });

                    if (category === 'Misc') {
                        miscPlugins = pluginItems;
                    } else {
                        displayedPluginSections.push({
                            isSection: true,
                            id: "tag_" + category,
                            icon: mostFrequentIcon(pluginItems),
                            label: category,
                            items: pluginItems
                        });
                    }
                });

                if ((displayedPluginSections && displayedPluginSections.length > 0) || (miscPlugins && miscPlugins.length > 0)) {
                    ret.push({ divider: true });
                    ret.push({ header: true, label: translate('PROJECT.ACTIONS.RECIPES.PLUGINS', 'Plugins') });

                    if (displayedPluginSections && displayedPluginSections.length > 0) {
                        displayedPluginSections.sort(function (a, b) { return a.label.toLowerCase().localeCompare(b.label.toLowerCase()); });
                        Array.prototype.push.apply(ret, displayedPluginSections);
                    }

                    if (miscPlugins && miscPlugins.length > 0) {
                        Array.prototype.push.apply(ret, miscPlugins);
                    }
                }

                var getIconFromRecipeType = $filter("recipeTypeToIcon");

                ret.forEach(function (item) {
                    if (item.isSection) {
                        item.items.forEach(function (sectionItem) {
                            if (!sectionItem.icon) sectionItem.icon = getIconFromRecipeType(sectionItem.type, 16);
                        });

                    } else {
                        if (!item.icon) item.icon = getIconFromRecipeType(item.type, 16);
                    }
                });
                return ret;
            },

            /**
             * Similar to getAllRecipesBySection
             * It returns recipes that can be inserted between a given dataset and a recipe
             * @param {Object} scope The scope must be prepared with modal triggers which are defined in `_CreateRecipesBehavior` controller
             * @param {String} datasetSmartName smart name of the dataset to which will be hooked the inserted recipe
             * @param {String} targetZoneId ID of the zone where the insertion should happen
             * @param {Object} datasetStatus the dataset status object containing all information about the dataset usability
             * @returns An array of recipes, each is flagged whether it is possible to be inserted between the dataset and the recipe and
             * with a reason if not. Every recipe has a `fn` property that triggers the recipe creation modal.
             */
            getInsertableRecipesBySection: function (scope, datasetSmartName, targetZoneId, datasetStatus) {
                var ret = [];

                ret.push({
                    isSection: true,
                    id: "visual",
                    label: translate("PROJECT.ACTIONS.RECIPES.VISUAL", "Visual"),
                    icon: "icon-eye",
                    items: [
                        {
                            type: "shaker", label: translate("PROJECT.ACTIONS.RECIPES.PREPARE", "Data preparation"),
                            fn: function (recipeAdditionalParams) {
                                scope.showCreateShakerModal(datasetSmartName, targetZoneId, recipeAdditionalParams);
                            }
                        },
                        {
                            type: "sync", label: translate("PROJECT.ACTIONS.RECIPES.SYNC", "Sync"),
                            fn: function (recipeAdditionalParams) {
                                scope.showCreateSyncModal(datasetSmartName, targetZoneId, recipeAdditionalParams);
                            }
                        },
                        {
                            type: "filter", label: translate("PROJECT.ACTIONS.RECIPES.SAMPLE", "Sample / Filter"),
                            fn: function (recipeAdditionalParams) {
                                scope.showCreateSamplingModal(datasetSmartName, targetZoneId, recipeAdditionalParams);
                            }
                        },
                        {
                            type : "distinct", label : translate("PROJECT.ACTIONS.RECIPES.DISTINCT", "Distinct"),
                            fn : function (recipeAdditionalParams) {
                                scope.showCreateDistinctModal(datasetSmartName, targetZoneId, recipeAdditionalParams);
                            }
                        },
                        {
                            type: "window", label: translate("PROJECT.ACTIONS.RECIPES.WINDOW", "Window"),
                            fn: function (recipeAdditionalParams) {
                                scope.showCreateWindowRecipeModal(datasetSmartName, targetZoneId, recipeAdditionalParams);
                            }
                        },
                        {
                            type: "join", label: translate("PROJECT.ACTIONS.RECIPES.JOIN", "Join"),
                            fn: function (recipeAdditionalParams) {
                                scope.showCreateJoinModal([datasetSmartName], targetZoneId, recipeAdditionalParams);
                            }
                        },
                        {
                            type: "fuzzyjoin", label: translate("PROJECT.ACTIONS.RECIPES.FUZZY_JOIN", "Fuzzy join"),
                            fn: function (recipeAdditionalParams) {
                                scope.showCreateFuzzyJoinModal([datasetSmartName], targetZoneId, recipeAdditionalParams);
                            }
                        },
                        {
                            type: "geojoin", label: translate("PROJECT.ACTIONS.RECIPES.GEO_JOIN", "Geo join"),
                            fn: function (recipeAdditionalParams) {
                                scope.showCreateGeoJoinModal([datasetSmartName], scope.$stateParams.zoneId, recipeAdditionalParams);
                            }
                        },
                        {
                            type: "split", label: translate("PROJECT.ACTIONS.RECIPES.SPLIT", "Split"),
                            fn: function (recipeAdditionalParams) {
                                scope.showCreateSplitModal(datasetSmartName, targetZoneId, recipeAdditionalParams);
                            }
                        },
                        {
                            type: "topn", label : translate("PROJECT.ACTIONS.RECIPES.TOPN", "Top N"),
                            fn : function (recipeAdditionalParams) {
                                scope.showCreateTopNModal(datasetSmartName, targetZoneId, recipeAdditionalParams);
                            }
                        },
                        {
                            type: "sort", label : translate("PROJECT.ACTIONS.RECIPES.SORT", "Sort"),
                            fn : function (recipeAdditionalParams){
                                scope.showCreateSortModal(datasetSmartName, targetZoneId, recipeAdditionalParams);
                            }
                        },
                        {
                            type: "vstack", label : translate("PROJECT.ACTIONS.RECIPES.STACK", "Stack vertically"),
                            fn : function (recipeAdditionalParams){
                                scope.showCreateVStackModal([datasetSmartName], targetZoneId, recipeAdditionalParams);
                            }
                        }
                    ]
                });

                const codeRecipeItems = []

                const pythonStatus = retrieveDatasetStatusForRecipe(datasetStatus, "python");
                codeRecipeItems.push({
                    type: "python",
                    label: "Python",
                    disabled: !pythonStatus.iconEnabled,
                    reason: pythonStatus.iconDisabledReason,
                    fn: function (recipeAdditionalParams) {
                        scope.showCreateCodeBasedModal("python", [datasetSmartName], targetZoneId, null, recipeAdditionalParams);
                    }
                });
                if($rootScope.appConfig.uiCustomization.showR) {
                    const rStatus = retrieveDatasetStatusForRecipe(datasetStatus, "r");
                    codeRecipeItems.push({
                        type: "r",
                        label: "R",
                        disabled: !rStatus.iconEnabled,
                        reason: rStatus.iconDisabledReason,
                        fn: function (recipeAdditionalParams) {
                            scope.showCreateCodeBasedModal("r", [datasetSmartName], targetZoneId, null, recipeAdditionalParams);
                        }
                    });
                }
                if($rootScope.featureFlagEnabled('julia')) {
                    const juliaStatus = retrieveDatasetStatusForRecipe(datasetStatus, "julia");
                    codeRecipeItems.push({
                        type: "julia",
                        label: "Julia",
                        disabled: !juliaStatus.iconEnabled,
                        reason: juliaStatus.iconDisabledReason,
                        fn: function (recipeAdditionalParams) {
                            scope.showCreateCodeBasedModal("julia", [datasetSmartName], targetZoneId, null, recipeAdditionalParams);
                        }
                    });
                }

                const sqlQueryStatus = retrieveDatasetStatusForRecipe(datasetStatus, "sql_query");
                codeRecipeItems.push({
                    type: "sql_query",
                    label: "SQL",
                    disabled: !sqlQueryStatus.iconEnabled,
                    reason: sqlQueryStatus.iconDisabledReason,
                    fn: function (recipeAdditionalParams) {
                        scope.showSQLRecipeModal([datasetSmartName], targetZoneId, recipeAdditionalParams);
                    }
                });
                const shellStatus = retrieveDatasetStatusForRecipe(datasetStatus, "shell");
                codeRecipeItems.push({
                    type: "shell",
                    label: "Shell",
                    disabled: !shellStatus.iconEnabled,
                    reason: shellStatus.iconDisabledReason,
                    fn: function (recipeAdditionalParams) {
                        scope.showCreateCodeBasedModal("shell", [datasetSmartName], targetZoneId, null, recipeAdditionalParams);
                    }
                })

                ret.push({
                    isSection: true,
                    id: "code",
                    label: translate("PROJECT.ACTIONS.RECIPES.CODE", "Code"),
                    icon: "icon-code",
                    items: codeRecipeItems,
                });

                const hadoopSparkRecipeItems = [];

                const sparkSqlStatus = retrieveDatasetStatusForRecipe(datasetStatus, "spark_sql_query");
                hadoopSparkRecipeItems.push({
                    type: "spark_sql_query",
                    label: "Spark SQL",
                    disabled: !sparkSqlStatus.iconEnabled,
                    reason: sparkSqlStatus.iconDisabledReason,
                    fn: function (recipeAdditionalParams) {
                        if (sparkSqlStatus.enableStatus === "OK") {
                            scope.showCreateCodeBasedModal("spark_sql_query", [datasetSmartName], targetZoneId, null, recipeAdditionalParams);
                        } else if (sparkSqlStatus.enableStatus === "NOT_LICENSED_EE") {
                            scope.showSparkNotLicensedModal();
                        } else if (sparkSqlStatus.enableStatus === "NOT_LICENSED_CE") {
                            scope.showCERestrictionModal("Spark is")
                        }
                    }
                });

                if($rootScope.appConfig.uiCustomization.showScala) {
                    const sparkScalaStatus = retrieveDatasetStatusForRecipe(datasetStatus, "spark_scala");
                    hadoopSparkRecipeItems.push({
                        type: "spark_scala",
                        label: "Spark Scala",
                        disabled: !sparkScalaStatus.iconEnabled,
                        reason: sparkScalaStatus.iconDisabledReason,
                        fn: function (recipeAdditionalParams) {
                            if (sparkScalaStatus.enableStatus === "OK") {
                                scope.showCreateCodeBasedModal("spark_scala", [datasetSmartName], targetZoneId, null, recipeAdditionalParams);
                            } else if (sparkScalaStatus.enableStatus === "NOT_LICENSED_EE") {
                                scope.showSparkNotLicensedModal();
                            } else if (sparkScalaStatus.enableStatus === "NOT_LICENSED_CE") {
                                scope.showCERestrictionModal("Spark is")
                            }
                        }
                    });
                }

                const pysparkStatus = retrieveDatasetStatusForRecipe(datasetStatus, "pyspark");
                hadoopSparkRecipeItems.push({
                    type: "pyspark",
                    label: "PySpark",
                    disabled: !pysparkStatus.iconEnabled,
                    reason: pysparkStatus.iconDisabledReason,
                    fn: function (recipeAdditionalParams) {
                        if (pysparkStatus.enableStatus === "OK") {
                            scope.showCreateCodeBasedModal("pyspark", [datasetSmartName], targetZoneId, null, recipeAdditionalParams);
                        } else if (pysparkStatus.enableStatus === "NOT_LICENSED_EE") {
                            scope.showSparkNotLicensedModal();
                        } else if (pysparkStatus.enableStatus === "NOT_LICENSED_CE") {
                            scope.showCERestrictionModal("Spark is")
                        }
                    }
                });

                if($rootScope.appConfig.uiCustomization.showR) {
                    const sparkrStatus = retrieveDatasetStatusForRecipe(datasetStatus, "sparkr");
                    hadoopSparkRecipeItems.push({
                        type: "sparkr",
                        label: "SparkR",
                        disabled: !sparkrStatus.iconEnabled,
                        reason: sparkrStatus.iconDisabledReason,
                        fn: function (recipeAdditionalParams) {
                            if (sparkrStatus.enableStatus === "OK") {
                                scope.showCreateCodeBasedModal("sparkr", [datasetSmartName], targetZoneId, null, recipeAdditionalParams);
                            } else if (sparkrStatus.enableStatus === "NOT_LICENSED_EE") {
                                scope.showSparkNotLicensedModal();
                            } else if (sparkrStatus.enableStatus === "NOT_LICENSED_CE") {
                                scope.showCERestrictionModal("Spark is")
                            }
                        }
                    });
                }

                ret.push({
                    isSection: true,
                    id: "code",
                    label: "Hadoop & Spark",
                    icon: "icon-HDFS",
                    items: hadoopSparkRecipeItems,
                });

                var getIconFromRecipeType = $filter("recipeTypeToIcon");

                ret.forEach(function (item) {
                    if (item.isSection) {
                        item.items.forEach(function (sectionItem) {
                            if (!sectionItem.icon) sectionItem.icon = getIconFromRecipeType(sectionItem.type);
                        });

                    } else {
                        if (!item.icon) item.icon = getIconFromRecipeType(item.type);
                    }
                });

                return ret;
            },
            /**
             * Returns [
             *   { isSection : false, type, label, icon, fn?, licenseStatus, disabled?, em? }
             *   { isSection : true, id, label: icon: item : []}
             * ]
             */
            getAllDatasetsBySection: function (scope) {
                var ret = [];

                ret.push({
                    type: 'search_and_import', label: translate('PROJECT.ACTIONS.DATASETS.SEARCH_AND_IMPORT', "Search and import\u2026"), icon: 'icon-mail-forward',
                    em: true, fn: () => $state.go('projects.project.datacatalog.home', {zoneId: scope.getRelevantZoneId($stateParams.zoneId)})
                });

                if($rootScope.appConfig.alationSettings.enabled) {
                    ret.push({
                        type: 'import_from_alation', label: translate("PROJECT.ACTIONS.DATASETS.IMPORT_FROM_ALATION", "Import from Alation\u2026"), icon: 'icon-book',
                        em: true, fn: () => scope.importFromAlation()
                    });
                }
                ret.push({ divider: true });

                // type, label, disabled, reason, icon
                ret.push({
                    type: "UploadedFiles", label: translate('PROJECT.ACTIONS.DATASETS.UPLOAD_YOUR_FILES', "Upload your files")
                });
                ret.push({
                    type: "Filesystem", label: translate('PROJECT.ACTIONS.DATASETS.FILESYSTEM', "Filesystem")
                });

                ret.push({
                    isSection : true,
                    id : "network",
                    label : translate('PROJECT.ACTIONS.DATASETS.NETWORK', "Network"),
                    icon : "icon-FTP-HTTP-SSH",
                    items : [
                        { type: "FTP", label: " FTP" },
                        { type: "SFTP", label: " SFTP" },
                        { type: "SCP", label: " SCP" },
                        { type: "HTTP", label: "HTTP" },
                        { type: "CachedHTTP", label: translate("PROJECT.ACTIONS.RECIPES.HTTP_WITH_CACHE", "HTTP (with cache)"), fn: () => scope.showCreateUrlDownloadToFolderDataset($stateParams.projectKey)}
                    ]
                });

                ret.push({
                    type: "HDFS", label: "HDFS",
                    disabled: !$rootScope.appConfig.hadoopEnabled,
                    reason: $rootScope.appConfig.hadoopEnabled ? null : translate("PROJECT.PERMISSIONS.HADOOP_NOT_CONFIGURED", "Hadoop not configured on your DSS instance")
                });

                ret.push({divider: true});

                ret.push({
                    type: "hiveserver2", label: "Hive",
                    disabled: !$rootScope.appConfig.hadoopEnabled,
                    reason: $rootScope.appConfig.hadoopEnabled ? null : translate("PROJECT.PERMISSIONS.HADOOP_NOT_CONFIGURED", "Hadoop not configured on your DSS instance")
                });

                var sql = {
                    isSection: true,
                    id: "sql",
                    label: translate('PROJECT.ACTIONS.DATASETS.SQL_DATABASES', "SQL databases"),
                    icon: "icon-database",
                    items: [
                        {
                            type: "Snowflake", label: "Snowflake", section: "SQL Databases"
                        },
                        {
                            type: "Databricks", label: "Databricks", section: "SQL Databases"
                        },
                        {
                            type: "Redshift", label: "Amazon Redshift", section: "SQL Databases"
                        },
                        {
                            type: "Synapse", label: "Azure Synapse", section: "SQL Databases"
                        },
                        {
                            type: "FabricWarehouse", label: "MS Fabric Warehouse", section: "SQL Databases"
                        },
                        {
                            type: "BigQuery", label: "Google BigQuery", section: "SQL Databases"
                        },
                        {
                            type: "PostgreSQL", label: "PostgreSQL", section: "SQL Databases"
                        },
                        {
                            type: "MySQL", label: "MySQL", section: "SQL Databases"
                        },
                        {
                            type: "SQLServer", label: "MS SQL Server", section: "SQL Databases"
                        },
                        {
                            type: "Oracle", label: "Oracle", section: "SQL Databases"
                        },
                        {
                            type: "Teradata", label: "Teradata", section: "SQL Databases"
                        },
                        {
                            type: "Greenplum", label: "Greenplum", section: "SQL Databases"
                        },
                        {
                            type: "AlloyDB", label: "Google AlloyDB", section: "SQL Databases"
                        },
                        {
                            type: "Athena", label: "Athena", section: "SQL Databases"
                        },
                        {
                            type: "Trino", label: "Trino", section: "SQL Databases"
                        },
                        {
                            type: "Denodo", label: "Denodo", section: "SQL Databases"
                        },
                        {
                            type: "TreasureData", label: "Treasure Data", section: "SQL Databases"
                        },
                        {
                            type: "Vertica", label: "Vertica", section: "SQL Databases"
                        },
                        {
                            type: "SAPHANA", label: "SAP HANA", section: "SQL Databases"
                        },
                        {
                            type: "Netezza", label: "IBM Netezza", section: "SQL Databases"
                        }
                    ]
                };
                if ($rootScope.featureFlagEnabled("lakebase")) {
                    sql.items.splice(12, 0, {
                        type: "DatabricksLakebase", label: "Databricks Lakebase", section: "SQL Databases"
                    })
                }

                if ($rootScope.featureFlagEnabled("kdbplus")) {
                    sql.items.push({
                        type: "KDBPlus", label: "KDB+", section: "SQL Databases"
                    })
                }

                sql.items.push({
                    type: "JDBC", label: translate('PROJECT.ACTIONS.DATASETS.OTHER_SQL_DATABASES', "Other SQL databases"), section: "SQL Databases"
                });
                ret.push(sql);

                ret.push({
                    isSection: true,
                    id: "cloud",
                    label: translate('PROJECT.ACTIONS.DATASETS.CLOUD_STORAGES_AND_SOCIAL', "Cloud storages & Social"),
                    icon: "icon-cloud",
                    items: [
                        {
                            type: "S3", label: "Amazon S3"
                        },
                        {
                            type: "Azure", label: "Azure Blob Storage"
                        },
                        {
                            type: "GCS", label: "Google Cloud Storage"
                        },
                        {
                            type: "SharePointOnline", label: "SharePoint Document"
                        },
                        {
                            type: "SharePointOnlineList", label: "SharePoint List"
                        }
                    ]
                })

                let noSQLSection = {
                    isSection: true,
                    id: "nosql",
                    label: translate('PROJECT.ACTIONS.DATASETS.NOSQL', "NoSQL"),
                    icon: "icon-database",
                    items: [
                        {
                            type: "MongoDB", label: "MongoDB"
                        }, {
                            type: "Cassandra", label: "Cassandra"
                        }, {
                            type: "ElasticSearch", label: "ElasticSearch"
                        }
                    ]
                };
                if ($rootScope.featureFlagEnabled('DynamoDB')) {
                    noSQLSection.items.push({
                        type: "DynamoDB", label: "DynamoDB"
                    });
                }
                ret.push(noSQLSection);

                ret.push({divider: true});
                ret.push({
                    isSection: true,
                    id: "dataikuManaged",
                    label: translate('PROJECT.ACTIONS.DATASETS.DATAIKU_MANAGED', "Dataiku-Managed"),
                    icon: "icon-dataiku",
                    items: [
                        {
                            type: "Sample",
                            label: translate('PROJECT.ACTIONS.DATASETS.SAMPLES', "Samples"),
                            icon: "dku-icon-test-tube-filled-16",
                            fn: scope.newSample
                        },
                        {
                            type: "managed_folder", label: translate("PROJECT.ACTIONS.RECIPES.FOLDER", "Folder"), icon: "icon-box", fn: scope.newManagedFolder
                        },
                        {
                            type: "FilesInFolder", label: translate('PROJECT.ACTIONS.DATASETS.FILES_IN_FOLDER', "Files in folder")
                        },
                        {
                            type: "Inline", label: translate('PROJECT.ACTIONS.DATASETS.EDITABLE', "Editable")
                        },
                        {
                            type: "managed",
                            label: translate("PROJECT.ACTIONS.RECIPES.MANAGED_DATASET", "Managed dataset"),
                            icon: "icon-beaker",
                            fn: scope.newManagedDataset,
                        }
                    ]
                });

                ret.push({
                    isSection: true,
                    id: "other",
                    label: translate('ADD_WORKFLOW.DROPDOWN.SECOND.OTHER', "Other"),
                    icon: "icon-tasks",
                    items: [
                        {
                            type: "model_evaluation_store",
                            label: translate("PROJECT.ACTIONS.RECIPES.MODEL_EVALUATION_STORE", "Evaluation store"),
                            icon: "icon-model-evaluation-store",
                            fn: scope.newModelEvaluationStore
                        },
                        {
                            type: "import_from_feature_store",
                            label: translate("PROJECT.ACTIONS.RECIPES.FEATURE_GROUP", "Feature Group"),
                            icon: "icon-dku-feature-store",
                            id: "qa_flow_use-feature-group",
                            fn: () => $state.go('projects.project.featurestore', { zoneId: scope.getRelevantZoneId($stateParams.zoneId) })
                        },
                        {
                            type: "JobsDB", label: translate("PROJECT.ACTIONS.RECIPES.METRICS", "Metrics")
                        },
                        {
                            type: "StatsDB", label: translate("PROJECT.ACTIONS.RECIPES.INTERNAL_STATS", "Internal stats")
                        },
                        {
                            type: "ExperimentsDB", label: translate("PROJECT.ACTIONS.RECIPES.EXPERIMENTS", "Experiments")
                        },
                        {
                            type: "ForeignDataset", label: translate('PROJECT.ACTIONS.DATASETS.FOREIGN_DATASET', "Dataset from another project"), icon: "icon-dku-share",
                            fn: () => $state.go('projects.project.datacatalog.datasources', {selectedTab: 'datasets', zoneId: scope.getRelevantZoneId($stateParams.zoneId)})
                        }
                    ]
                });


                let pluginsByCategory = {};
                let recipeCategories = {};
                const pluginById = $rootScope.appConfig.loadedPlugins.reduce(function (map, obj) {
                    map[obj.id] = obj;
                    return map;
                }, {});

                const orderPluginInCategory = function(dataset, getPluginType) {
                    const plugin = pluginById[dataset.ownerPluginId];
                    if (angular.isUndefined(plugin) || plugin.hideComponents) return; // could have been deleted on disk or plugin components visibility settings require it be hidden

                    let category = angular.isDefined(plugin.category) ? plugin.category : 'Misc';

                    if (category.toLowerCase() in recipeCategories) {
                        category = recipeCategories[category.toLowerCase()];
                    } else {
                        recipeCategories[category.toLowerCase()] = category = $filter('capitalize')(category);
                    }

                    pluginsByCategory[category] = pluginsByCategory[category] || {};
                    pluginsByCategory[category][plugin.id] = {
                        pluginId: dataset.ownerPluginId,
                        type : getPluginType(dataset),
                        label : plugin.label ? plugin.label : plugin.id,
                        icon : plugin.icon,
                        fn: () => scope.showCreateDatasetFromPlugin(dataset.ownerPluginId)
                    }
                };

                $rootScope.appConfig.customFSProviders.forEach(plugin => orderPluginInCategory(plugin, x => x.fsProviderType));
                $rootScope.appConfig.customDatasets.forEach(plugin => orderPluginInCategory(plugin, x => x.datasetType));

                let displayedPluginSections = [];
                let miscPlugins = null;

                $.each(pluginsByCategory, function (category, plugins) {
                    let pluginItems = Object.values(plugins);
                    pluginItems.sort(function(a, b) { return a.label.toLowerCase().localeCompare(b.label.toLowerCase()); });

                    if (category === 'Misc') {
                        miscPlugins = pluginItems;
                    } else {
                        displayedPluginSections.push({
                            isSection: true,
                            isPluginSection: true,
                            id: "tag_" + category,
                            icon: mostFrequentIcon(pluginItems),
                            label: category,
                            items: pluginItems
                        });
                    }
                });

                if ((displayedPluginSections && displayedPluginSections.length > 0) || (miscPlugins && miscPlugins.length > 0)) {
                    ret.push({divider: true});
                    ret.push({header: true, label: translate('PROJECT.ACTIONS.RECIPES.PLUGINS', 'Plugins')});

                    if (displayedPluginSections && displayedPluginSections.length > 0) {
                        displayedPluginSections.sort(function(a, b) { return a.label.toLowerCase().localeCompare(b.label.toLowerCase()); });
                        Array.prototype.push.apply(ret, displayedPluginSections);
                    }

                    if (miscPlugins && miscPlugins.length > 0) {
                        Array.prototype.push.apply(ret, miscPlugins);
                    }
                }

                var getIconFromType = $filter("datasetTypeToIcon");


                /**
                 * Enrich the info with items icons, licence status and filter hidden datasets.
                 */
                const postTreat = (items, computeStatus) => {
                    let haveItemBeforeCurrent = false;
                    return items
                        .map((item) => {
                            if(item.isSection) {
                                item.items = postTreat(item.items, computeStatus);
                                return item;
                            } else {
                                if (item.type) {
                                    item.licenseStatus = svc.getDatasetLicenseStatus(item.type);
                                    if (item.licenseStatus.status === 'NOT_LICENSED_EE') {
                                        item.disabled = true;
                                        item.reason = translate("PROJECT.PERMISSIONS.DATASET_NOT_LICENSED", "This dataset type is not licensed");
                                    }
                                }
                                if (!item.icon) item.icon = getIconFromType(item.type);
                                return item;
                            }
                        })
                        .filter(item => !( // remove hidden dataset type and empty sections (plugins are always shown)
                            item.isSection && item.items.length === 0 ||
                            item.type && item.pluginId === undefined && computeStatus(item.type) !== uiCustomizationService.datasetTypeStatus.SHOW
                        ))
                        .filter((item, index, arr) => { // remove useless dividers
                            if(item.divider && (
                                !haveItemBeforeCurrent ||                   // remove divider if there is not at least one item before
                                index === arr.length - 1 ||                 // remove divider if in last position
                                arr[index + 1] && arr[index + 1].divider    // remove divider if following item is also a divider
                            )) return false;
                            haveItemBeforeCurrent = true;
                            return true;
                        });
                }

                return uiCustomizationService.getComputeDatasetTypesStatus(scope, $stateParams.projectKey)
                    .then((computeStatus) => postTreat(ret, computeStatus));
            },

            /**
             * Raw unfiltered version of getAllDatasetsByTiles, used internally, but also for options.
             * Neat trick: calling this function with and undefined scope will work if you only want the list, but some clickCallback will be broken.
             * @returns {{title: string, icon: string, types: {type: string, label: string, disabledReason?:string}[]}[]} the list of dataset options, by tiles.
             */
            getAllDatasetByTilesNoFilter: function (scope) {
                const zoneId = scope && scope.getRelevantZoneId($stateParams.zoneId); // scope can be undefined
                const hadoopDisabledMessage = $rootScope.appConfig.hadoopEnabled ? undefined : `Hadoop connection is not enabled on your ${$rootScope.wl.productShortName} instance.` + ($rootScope.isDSSAdmin() ? '' : ` Please contact your administrator`);

                const tiles = [{
                    title: translate('PROJECT.ACTIONS.DATASETS.FILES', "Files"),
                    icon: 'icon-Filesystem',
                    types: [
                        {type: 'UploadedFiles', label: translate('PROJECT.ACTIONS.DATASETS.UPLOAD_YOUR_FILES', "Upload your files")},
                        {type: 'Filesystem'},
                    ]
                }, {
                    title: 'Hadoop',
                    icon: 'icon-elephant',
                    types: [
                        {type: 'HDFS', label: 'HDFS', disabledReason: hadoopDisabledMessage},
                        {type: 'hiveserver2', label: 'Hive', disabledReason: hadoopDisabledMessage},
                    ],
                }, {
                    title: translate('PROJECT.ACTIONS.DATASETS.SQL', "SQL"),
                    icon: 'icon-sql marker-database',
                    types: [
                        {type: 'Snowflake'},
                        {type: 'Databricks'},
                        {type: 'Redshift'},
                        {type: 'Synapse'},
                        {type: 'BigQuery'},
                        {type: 'PostgreSQL'},
                        {type: 'MySQL'},
                        {type: 'SQLServer'},
                        {type: 'FabricWarehouse'},
                        {type: 'Oracle'},
                        {type: 'Teradata'},
                        {type: 'Greenplum'},
                        {type: 'AlloyDB'},
                        {type: 'Athena'},
                        {type: 'Trino'},
                        {type: 'Denodo'},
                        {type: 'TreasureData'},
                        {type: 'Vertica'},
                        {type: 'SAPHANA'},
                        {type: 'Netezza'},
                        {type: 'JDBC', label:'Other SQL'},
                    ]
                }, {
                    title: translate('PROJECT.ACTIONS.DATASETS.CLOUD_STORAGES', "Cloud Storages"),
                    icon: 'icon-cloud',
                    types: [
                        {type: 'S3'},
                        {type: 'Azure'},
                        {type: 'GCS'},
                        {type: 'SharePointOnline'},
                        {type: 'SharePointOnlineList'},
                    ],
                    types2: [
                        {type: 'FTP'},
                        {type: 'SFTP'},
                        {type: 'SCP'},
                        {type: 'HTTP'},
                        {type: 'CachedHTTP', clickCallback: () => scope.showCreateUrlDownloadToFolderDataset($stateParams.projectKey)},
                    ]
                }, {
                    title: translate('PROJECT.ACTIONS.DATASETS.NOSQL', "NoSQL"),
                    icon: 'icon-signal',
                    types: [
                        {type: 'MongoDB'},
                        {type: 'Cassandra'},
                        {type: 'ElasticSearch'},
                        $rootScope.featureFlagEnabled('DynamoDB') ? {type: 'DynamoDB'} : undefined,
                    ]
                }, {
                    title: translate('PROJECT.ACTIONS.DATASETS.SOCIAL', "Social"),
                    icon: 'icon-twitter',
                    types: [
                        {type: 'Twitter'},
                    ]
                }, {
                    title: $rootScope.wl.productShortName,
                    icon: 'icon-beaker',
                    types: [
                        {type: 'FilesInFolder', label: translate('PROJECT.ACTIONS.DATASETS.FILES_IN_FOLDER', "Files in folder")},
                        {type: 'managed', label: translate("PROJECT.ACTIONS.RECIPES.MANAGED_DATASET", "Managed dataset"), clickCallback: () => scope.newManagedDataset()},
                        {type: 'managed_folder', label: translate("PROJECT.ACTIONS.RECIPES.FOLDER", "Folder"), clickCallback: () => scope.newManagedFolder()},
                        {type: 'model_evaluation_store', label: translate("PROJECT.ACTIONS.RECIPES.MODEL_EVALUATION_STORE", "Evaluation store"), clickCallback: () => scope.newModelEvaluationStore()},
                        {type: 'Sample', label: translate("PROJECT.ACTIONS.DATASETS.SAMPLES", "Samples"), clickCallback: () => scope.newSample()},
                        {type: 'JobsDB'},
                        {type: 'StatsDB'},
                        {type: 'Inline'},
                        {type: 'ExperimentsDB'}
                    ]
                }, {
                    title: translate('PROJECT.ACTIONS.DATASETS.IMPORT_EXISTING', "Import existing"),
                    icon: 'icon-cloud-download',
                    types: [
                        {type: 'import_from_data_collection', label:translate('PROJECT.ACTIONS.DATASETS.IMPORT_FROM_DATA_COLLECTION', "Import from Data Collection"), clickCallback: () => $state.go('projects.project.datacatalog.datacollections.home', {zoneId})},
                        {type: 'ForeignDataset', label:translate('PROJECT.ACTIONS.DATASETS.IMPORT_FROM_ANOTHER_PROJECT', "Import dataset from another project"), clickCallback: () => $state.go('projects.project.datacatalog.datasources', {selectedTab: 'datasets', zoneId})},
                        {type: 'import_from_catalog', label:translate('PROJECT.ACTIONS.DATASETS.IMPORT_INDEXED_TABLE', "Import indexed table"), clickCallback: () => $state.go('projects.project.datacatalog.datasources', {selectedTab: 'external-tables', zoneId})},
                        {type: 'connection_explorer', label:translate('PROJECT.ACTIONS.DATASETS.IMPORT_FROM_CONNECTION', "Import from connection"), clickCallback: () => $state.go('projects.project.datacatalog.database-explorer', {zoneId})},
                        {type: 'import_from_feature_store', label:translate('PROJECT.ACTIONS.DATASETS.IMPORT_FROM_FEATURE_STORE', "Import from feature store"), id: 'qa_dataset_type_import-from-feature-store', clickCallback: () => $state.go('projects.project.featurestore', {zoneId})},
                        {type: 'import_dss_item', label:translate('PROJECT.ACTIONS.DATASETS.IMPORT_DSS_ITEM_FROM_OTHER_PROJECT', "Import DSS item from another project"), clickCallback: () => $state.go('projects.project.catalog.items', {zoneId})},
                        $rootScope.appConfig.alationSettings.enabled ? {type: 'import_from_alation', label:translate('PROJECT.ACTIONS.DATASETS.IMPORT_FROM_ALATION', "Import from Alation..."), clickCallback: () => scope.importFromAlation()} : undefined,
                    ]
                }];

                return tiles;
            },

            /**
             * Similar to getAllDatasetsBySection, but order according to the new dataset page tiles
             * Used by the new dataset page
             *
             * returns Tile[]
             *
             * Tile = {
             *  title: string,
             *  icon: string,
             *  types: DatasetType[] (8 max)
             *  types2: DatasetType[] (second column)
             * }
             *
             * DatasetType = {
             *  type: string,
             *  label: string,
             *  clickCallback: function
             *  status: 'OK' | 'NOT_LICENCIED_CD' | 'NOT_LICENCIED_EE' | 'NO_CONNECTION' | 'HIDDEN'
             *  tooltip: string, what to display when the type is hoovered
             * }
             *
             */
            getAllDatasetsByTiles: function (scope) {
                return uiCustomizationService.getComputeDatasetTypesStatus(scope, $stateParams.projectKey).then((computeStatus) => {
                    const tiles = svc.getAllDatasetByTilesNoFilter(scope);

                    /**
                     * Finds out how the dataset should be displayed in the new dataset page by merging the uiCustomization effect and the license status
                     * @param {string} type
                     * @returns {string} 'OK', 'HIDDEN', 'NO_CONNECTION', 'NOT_LICENSED_EE' or 'NOT_LICENSED_CE'
                     */
                    const computeStatusWithLicense = (datasetType) => {
                        const uiCustomizationStatus = computeStatus(datasetType.type);
                        const licenseStatus = svc.getDatasetLicenseStatus(datasetType.type).status;

                        if(uiCustomizationStatus === uiCustomizationService.datasetTypeStatus.HIDE) {
                            return uiCustomizationStatus;
                        } else if(licenseStatus !== 'OK') {
                            return licenseStatus;
                        } else if(datasetType.disabledReason !== undefined) {
                            // when Hadoop is disabled, Hive & HDFS should appear disabled with a specific message. We reuse the NO_CONNECTION status
                            return uiCustomizationService.datasetTypeStatus.NO_CONNECTION;
                        } else {
                            return uiCustomizationStatus;
                        }
                    }

                    const datasetTypeToName = $filter('datasetTypeToName');
                    const enrichAndFilterTypes = (types) => types
                        .filter((t) => t) // remove undefined (feature flags not enabled)
                        .map((datasetType) => {
                            const status = computeStatusWithLicense(datasetType);
                            return {
                                type: datasetType.type,
                                label: datasetType.label || datasetTypeToName(datasetType.type),
                                status,
                                tooltip: status === 'NOT_LICENSED_EE'
                                    ? translate('PROJECT.ACTIONS.DATASETS.NOT_AUTHORIZED_BY_LICENCE', "This dataset is not authorized by your license")
                                    : status === 'NO_CONNECTION'
                                        ? datasetType.disabledReason || translate('PROJECT.ACTIONS.DATASETS.UNAVAILABLE_CONNECTION', "This dataset requires a connection that is not available. It may be because no such connection has been created by your administrator or because you don't have sufficient authorization")
                                        : undefined,
                                clickCallback: datasetType.clickCallback,
                                id: datasetType.id
                            };
                        })
                        .filter((dt) => dt.status !== uiCustomizationService.datasetTypeStatus.HIDE);

                    const enrichAndFilterTile = (tile) => ({
                        ...tile,
                        types: enrichAndFilterTypes(tile.types),
                        types2: tile.types2 && enrichAndFilterTypes(tile.types2),
                    });

                    return tiles
                        .map(tile => {
                            const tmp = enrichAndFilterTile(tile);
                            if(tmp.types.length > 8) tmp.types2 = tmp.types.splice(8);
                            if(tmp.types2 && tmp.types.length === 0) {
                                tmp.types = tmp.types2; // if first column is empty but 2nd is defined, move the second column in first place
                                tmp.types2 = undefined;
                            }
                            return tmp;
                        })
                        .filter(tile => tile.types.length > 0); // remove empty tiles

                })

            },

            getHighlightedDatasets(scope) {
                return uiCustomizationService.getComputeDatasetTypesStatus(scope, $stateParams.projectKey).then((computeStatus) => {
                    const highlightedTypes = $rootScope.appConfig.uiCustomization.highlightedDatasets;

                    const tiles = svc.getAllDatasetByTilesNoFilter(scope);
                    const flatDatasetInfos = tiles
                        .reduce(
                            (acc, tile) => acc.concat(tile.types).concat(tile.types2 || []),
                            []
                        ).filter(info => info); // remove feature-flags disabled

                    // special case : the search and import option is not in the 'classic' tiles, but can been highlighted
                    flatDatasetInfos.push({
                        type: 'search_and_import',
                        label: translate('PROJECT.ACTIONS.DATASETS.SEARCH_AND_IMPORT', 'Search and import\u2026'),
                        clickCallback: () => $state.go('projects.project.datacatalog.home', {zoneId: scope.getRelevantZoneId($stateParams.zoneId)})
                    });

                    const datasetTypeToName = $filter('datasetTypeToName');
                    const getIconFromType = $filter("datasetTypeToIcon");
                    const forcedIcons = {
                        import_from_alation: 'icon-book',
                        managed_folder: 'icon-box',
                        model_evaluation_store: 'icon-model-evaluation-store',
                        import_from_feature_store: 'icon-dku-feature-store',
                        managed: 'icon-beaker',
                        ForeignDataset: 'icon-dku-share',
                        import_from_data_collection: 'dku-icon-stacked-32',
                    };
                    const highlightedIcons = {
                        UploadedFiles: 'promoted-upload-files.png',
                        import_from_catalog: 'promoted-search-catalog.png',
                        import_dss_item: 'promoted-search-catalog.png',
                        connection_explorer: 'promoted-search-connections.png',
                        search_and_import: 'promoted-search-catalog.png',
                        ForeignDataset: 'promoted-dataset-project.png',
                    };

                    const highlightedDatasets = highlightedTypes.map((type => {
                        const info = flatDatasetInfos.find(i => i.type === type);
                        if(!info) return; // this could happen if a type is promoted and then the associated feature-flag is set to false.

                        info.status = computeStatus(type);
                        info.label = info.label || datasetTypeToName(type);
                        info.highlightedIcon = highlightedIcons[type];
                        if(!info.highlightedIcon) {
                            info.icon = forcedIcons[type] || getIconFromType(type);
                        }
                        return info;
                    })).filter(info => info
                        && info.status !== uiCustomizationService.datasetTypeStatus.HIDE
                        && info.status !== uiCustomizationService.datasetTypeStatus.NO_CONNECTION
                    );

                    return highlightedDatasets;
                })
            },

            /**
             * Execute the action to initiate dataset creation from the new dataset page (can be a modal or a page change)
             * @param type dataset type description, as produced by getAllDatasetsByTiles or getHighlightedDatasets
             */
            startDatasetCreation(type) {
                if(type.status === 'NOT_LICENSED_CE') {
                    $rootScope.showCERestrictionModal(type.label + ' dataset');
                } else if(type.status === 'SHOW') {
                    if(type.clickCallback) {
                        type.clickCallback();
                    } else {
                        $state.go('projects.project.datasets.new_with_type.settings', {
                            type: type.type,
                            zoneId: $stateParams.zoneId
                        });
                    }
                }
            },

            /**
             * Returns [
             *   { isSection : false, ... }
             *   { isSection : true, id, label: icon: item : []}}}
             * ]
             */
            getAllStreamingEndpointsBySection: function (scope) {
                var ret = [];

                // type, label, disabled, reason, icon
                ret.push({
                    type: "kafka", label: "Kafka", icon: "icon-kafka",
                    fn: () => scope.showCreateStreamingEndpointModal("kafka")
                });
                ret.push({
                    type: "sqs", label: "SQS", icon: "icon-sqs",
                    fn: () => scope.showCreateStreamingEndpointModal("sqs")
                });
                ret.push({
                    label: "HTTP Server-Sent-Events", type:"httpsse", icon: "icon-httpsse",
                    fn: () => scope.showCreateStreamingEndpointModal("httpsse")
                });
                if ($rootScope.featureFlagEnabled('kdbplus')) {
                    ret.push({
                        label: translate("PROJECT.ACTIONS.OTHER.STREAMING_ENDPOINTS.KDB_TICKER_PLANT", "KDB+Tick ticker plant"), type:"kdbplustick", icon: "icon-httpsse",
                        fn: () => scope.showCreateStreamingEndpointModal("kdbplustick")
                    });
                }
                return ret;
            },

            getStreamingRecipesBySection: function(scope) {
                var pyScientistStatus = usable();
                if (!$rootScope.appConfig.userProfile.mayRegularCode) {
                    pyScientistStatus = notDataScientist();
                }

                var unsafeStatus = usable();
                if (!$rootScope.appConfig.globalPermissions || !$rootScope.appConfig.globalPermissions.mayWriteUnsafeCode) {
                    unsafeStatus = noUnsafeCode();
                }
                var safeStatus = usable();
                if ($rootScope.appConfig.impersonationEnabled && (!$rootScope.appConfig.globalPermissions || !$rootScope.appConfig.globalPermissions.mayWriteSafeCode)) {
                    safeStatus = noSafeCode();
                }

                var safeCodeStatus = $rootScope.appConfig.impersonationEnabled ? safeStatus : unsafeStatus;

                var ret = [
                        {
                            type: "cpython",
                            label: translate("PROJECT.ACTIONS.RECIPES.PYTHON_STREAMING", "Python (streaming)"),
                            icon: "icon-continuous_python_recipe",
                            disabled: !pyScientistStatus.iconEnabled || !safeCodeStatus.iconEnabled,
                            reason: pyScientistStatus.iconDisabledReason || safeCodeStatus.iconDisabledReason,
                            fn: function () {
                                scope.showCreateCodeBasedModal("cpython", undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                            }
                        },
                        {
                            type: "csync",
                            label: translate("PROJECT.ACTIONS.RECIPES.CSYNC_STREAMING", "CSync (streaming)"),
                            icon: 'icon-continuous_sync_recipe',
                            fn: function () {
                                scope.showCreateCodeBasedModal("csync", undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                            }
                        },
                    ];

                if (FeatureFlagsService.featureFlagEnabled('ignoreKsqlDeprecation')) {
                    ret.push(
                        {
                            type: "ksql",
                            label: translate("PROJECT.ACTIONS.RECIPES.KSQL_STREAMING", "KSQL (streaming)"),
                            icon: "icon-continuous_ksql_recipe",
                            disabled: !pyScientistStatus.iconEnabled || !safeCodeStatus.iconEnabled,
                            reason: pyScientistStatus.iconDisabledReason || safeCodeStatus.iconDisabledReason,
                            fn: function () {
                                scope.showCreateCodeBasedModal("ksql", undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                            }
                        }
                    );
                }

                if($rootScope.appConfig.uiCustomization.showScala) {
                    ret.push(                        {
                        type: "streaming_spark_scala",
                        label: translate("PROJECT.ACTIONS.RECIPES.SPARK_SCALA_STREAMING", "Spark Scala (streaming)"),
                        icon: "icon-continuous_spark_scala_recipe",
                        disabled: !pyScientistStatus.iconEnabled || !safeCodeStatus.iconEnabled,
                        reason: pyScientistStatus.iconDisabledReason || safeCodeStatus.iconDisabledReason,
                        fn: function () {
                            scope.showCreateCodeBasedModal("streaming_spark_scala", undefined, scope.getRelevantZoneId(scope.$stateParams.zoneId));
                        }
                    });
                }
                return ret;
            },

            getDatasetLicenseStatus: function (type) {
                /* !! Make sure this is synchronized with the backend !! */
                if (['managed_folder', 'managed', 'ForeignDataset', 'import_from_alation', 'search_and_import', 'model_evaluation_store', 'connection_explorer', 'import_from_feature_store', 'import_from_catalog', 'import_from_datacatalog', 'import_dss_item', 'import_from_data_collection'].includes(type)) {
                    return {
                        status: "OK"
                    };
                }
                /* Always allowed */
                const isPluginDataset = ['CustomPython_', 'CustomJava_', 'fsprovider_']
                    .some((it) => type.startsWith(it));

                if (isPluginDataset) {
                    return {
                        status: "OK",
                    };
                }

                // Core minimum for DSS to work properly :)
                if (["Filesystem", "UploadedFiles", "Inline", "FilesInFolder"].indexOf(type) >= 0) {
                    return {
                        status: "OK"
                    };
                }
                // Always included
                if (["MySQL", "GCS", "PostgreSQL", "S3", "Azure", "FTP", "SFTP", "SCP", "HTTP", "CachedHTTP", "SharePointOnline", "SharePointOnlineList", "Sample"].indexOf(type) >= 0) {
                    return {
                        status: "OK"
                    };
                }
                // Nothing else in CE
                if ($rootScope.appConfig.communityEdition) {
                    return {
                        status: "NOT_LICENSED_CE"
                    };
                }
                var adt = $rootScope.appConfig.licensedFeatures.allowedDatasetTypes;
                if (adt != null && adt.length > 0) {
                    if (adt.indexOf(type) < 0) {
                        return {status: "NOT_LICENSED_EE"};
                    } else {
                        return {status: "OK"};
                    }
                } else {
                    // All allowed
                    return {
                        status: "OK"
                    }
                }
            },

            getAllStatusForDataset: function (dataset) {
                Assert.trueish(dataset, 'no dataset');
                Assert.trueish(dataset.type, 'no dataset type');

                const allRecipeTypes = $.map(RecipeDescService.getDescriptors(), (desc, type) => type).filter(type => !type.startsWith('Custom') && !type.startsWith('App_'));
                const allThings = ["sql", "hive", "impala", "pig", "sql99"];

                var ret = {recipes: {}, things: {}};
                allRecipeTypes.forEach(function (type) {
                    ret.recipes[type] = svc.recipeMaybePossibleFromDataset(dataset, type);
                });
                allThings.forEach(function (type) {
                    ret.things[type] = svc.specialThingMaybePossibleFromDataset(dataset, type);
                });

                return ret;
            },

            getAllStatusForStreamingEndpoint: function (streamingEndpoint) {
                Assert.trueish(streamingEndpoint, 'no streaming endpoint');
                Assert.trueish(streamingEndpoint.type, 'no streaming endpoint type');

                const allRecipeTypes = $.map(RecipeDescService.getDescriptors(), (desc, type) => type).filter(type => !type.startsWith('Custom'));
                const allThings = ["sql", "hive", "impala", "pig", "sql99"];

                var ret = {recipes: {}, things: {}};
                allRecipeTypes.forEach(function (type) {
                    ret.recipes[type] = svc.recipeMaybePossibleFromStreamingEndpoint(streamingEndpoint, type);
                });
                allThings.forEach(function (type) {
                    ret.things[type] = svc.specialThingMaybePossibleFromStreamingEndpoint(streamingEndpoint, type);
                });

                return ret;
            },

            /**
             * Returns whether this recipe *looks* possible from this dataset.
             * Note that additional restrictions might apply and must be checked later.
             *
             * Returns { ok : boolean, reason : String (only if !ok)}
             */
            recipeMaybePossibleFromDataset: function (dataset, recipeType) {
                var pyScientistStatus = usable();
                if (!$rootScope.appConfig.userProfile.mayRegularCode) {
                    pyScientistStatus = notDataScientist();
                }
                var rScientistStatus = usable();
                if (!$rootScope.appConfig.userProfile.mayRegularCode) {
                    rScientistStatus = notDataScientist();
                }
                var jlScientistStatus = usable();
                if (!$rootScope.appConfig.userProfile.mayRegularCode) {
                    jlScientistStatus = notDataScientist();
                }
                var scalaScientistStatus = usable();
                if (!$rootScope.appConfig.userProfile.mayRegularCode) {
                    scalaScientistStatus = notDataScientist();
                }
                var sqlScientistStatus = usable();
                if (!$rootScope.appConfig.userProfile.maySQL) {
                    sqlScientistStatus = notDataScientist();
                }
                var fullLLMMeshStatus = usable();
                if (!$rootScope.appConfig.userProfile.mayFullLLMMesh) {
                    fullLLMMeshStatus = notDataScientist();
                }

                var unsafeStatus = usable();
                if (!$rootScope.appConfig.globalPermissions || !$rootScope.appConfig.globalPermissions.mayWriteUnsafeCode) {
                    unsafeStatus = noUnsafeCode();
                }
                var safeStatus = usable();
                if ($rootScope.appConfig.impersonationEnabled && (!$rootScope.appConfig.globalPermissions || !$rootScope.appConfig.globalPermissions.mayWriteSafeCode)) {
                    safeStatus = noSafeCode();
                }

                var safeCodeStatus = $rootScope.appConfig.impersonationEnabled ? safeStatus : unsafeStatus;
                let sparkStatus;
                let isSQLTable;
                let isPartitioned;
                switch (recipeType) {
                    case "hive": {
                        var hiveStatus = getHiveStatus();
                        if (dataset.type != "HDFS" && dataset.type != "hiveserver2") {
                            return nok(translate("PROJECT.ACTIONS.RECIPES.DATASET_NOT_HDFS_HIVE", "Dataset is not on HDFS or Hive"));
                        }
                        if (hiveStatus.iconEnabled) return ok(hiveStatus);
                        else return nok(hiveStatus.iconDisabledReason);
                    }
                    case "impala": {
                        var impalaStatus = getImpalaStatus();
                        if (dataset.type != "HDFS" && dataset.type != "hiveserver2") {
                            return nok(translate("PROJECT.ACTIONS.RECIPES.DATASET_NOT_HDFS_HIVE", "Dataset is not on HDFS or Hive"));
                        }
                        if (impalaStatus.iconEnabled) return ok(impalaStatus);
                        else return nok(impalaStatus.iconDisabledReason);
                    }
                    case "pig" : {
                        var pigStatus = getPigStatus();
                        if (dataset.type != "HDFS") {
                            return nok(translate("PROJECT.ACTIONS.RECIPES.DATASET_NOT_HDFS", "Dataset is not on HDFS"));
                        }
                        if (pigStatus.iconEnabled) return ok(pigStatus);
                        else return nok(pigStatus.iconDisabledReason);
                    }
                    case "spark_sql_query":
                        sparkStatus = getSparkStatus();
                        if (!sparkStatus.iconEnabled) return nok(sparkStatus);
                        if (!sqlScientistStatus.iconEnabled) return nok(sqlScientistStatus);
                        else return ok(sparkStatus);

                    case "pyspark":
                        sparkStatus = getSparkStatus();
                        if (!sparkStatus.iconEnabled) return nok(sparkStatus);
                        if (!pyScientistStatus.iconEnabled) return nok(pyScientistStatus);
                        if (!safeCodeStatus.iconEnabled) return nok(safeCodeStatus);
                        else return ok(sparkStatus)
                    case "sparkr":
                        sparkStatus = getSparkStatus();
                        if (!sparkStatus.iconEnabled) return nok(sparkStatus);
                        if (!rScientistStatus.iconEnabled) return nok(rScientistStatus);
                        if (!safeCodeStatus.iconEnabled) return nok(safeCodeStatus);
                        else return ok(sparkStatus)
                    case "spark_scala":
                    case "streaming_spark_scala":
                        sparkStatus = getSparkStatus();
                        if (!sparkStatus.iconEnabled) return nok(sparkStatus);
                        if (!scalaScientistStatus.iconEnabled) return nok(scalaScientistStatus);
                        if (!safeCodeStatus.iconEnabled) return nok(safeCodeStatus);
                        else return ok(sparkStatus);

                    case "sql_query":
                    case "sql_script":
                        if (!sqlScientistStatus.iconEnabled) return nok(sqlScientistStatus);
                        if (DatasetUtils.isSQLQueryAble(dataset)) {
                            return ok();
                        } else {
                            return nok({"iconDisabledReason": translate("PROJECT.ACTIONS.RECIPES.DATASET_NOT_SQL_TABLE", "Dataset is not a SQL Table")});
                        }
                    case "sql99":
                        if (getSparkStatus().usable) {
                            return ok();
                        }
                        if (dataset.type == "HDFS") {
                            if ($rootScope.appConfig.hiveEnabled) {
                                return ok();
                            } else {
                                return nok(translate("PROJECT.ACTIONS.RECIPES.HIVE_SPARK_CONFIG_REQUIRED", "Hive or Spark configuration required"));
                            }
                        }
                        if (DatasetUtils.isSQLTable(dataset) && dataset.type != "MySQL") {
                            return ok();
                        }
                        return ok();
                    //return nok("Requires an SQL99-compliant database or Spark");
                    case "python":
                    case "cpython":
                    case "shell":
                        if (!pyScientistStatus.iconEnabled) return nok(pyScientistStatus);
                        if (!safeCodeStatus.iconEnabled) return nok(safeCodeStatus);
                        else return ok();
                    case "r":
                        if (!rScientistStatus.iconEnabled) return nok(rScientistStatus);
                        if (!safeCodeStatus.iconEnabled) return nok(safeCodeStatus);
                        else return ok();
                    case "julia":
                        if (!jlScientistStatus.iconEnabled) return nok(jlScientistStatus);
                        if (!safeCodeStatus.iconEnabled) return nok(safeCodeStatus);
                        else return ok();
                    case 'generate_features':
                        isSQLTable = DatasetUtils.isSQL(dataset);
                        isPartitioned = DatasetDetailsUtils.isDatasetPartitioned(dataset);
                        if (!getSparkStatus().usable) {
                            if (!isSQLTable && isPartitioned) return nok(translate("PROJECT.ACTIONS.RECIPES.DATASET_NOT_SQL_NOT_PARTITIONED", "Dataset is not a SQL table and is partitioned"))
                            if (!isSQLTable) return nok(translate("PROJECT.ACTIONS.RECIPES.DATASET_NOT_SQL_TABLE", "Dataset is not a SQL Table"));
                        }
                        if (isPartitioned) return nok(translate("PROJECT.ACTIONS.RECIPES.DATASET_NOT_PARTITIONED", "Dataset is partitioned"));
                        else return ok();
                    case 'update':
                    case 'vstack':
                    case 'sort':
                    case 'download':
                    case 'merge_folder':
                    case 'list_folder_contents':
                    case 'list_access':
                    case 'clustering_cluster':
                    case 'prediction_training':
                    case 'evaluation':
                    case 'standalone_evaluation':
                    case 'distinct':
                    case 'clustering_training':
                    case 'grouping':
                    case 'upsert':
                    case 'sync':
                    case 'sampling':
                    case 'export':
                    case 'topn':
                    case 'clustering_scoring':
                    case 'join':
                    case 'fuzzyjoin':
                    case 'geojoin':
                    case 'shaker':
                    case 'window':
                    case 'split':
                    case 'prediction_scoring':
                    case 'pivot':
                    case 'prompt':
                    case 'nlp_llm_model_provided_classification':
                    case 'nlp_llm_user_provided_classification':
                    case 'nlp_llm_summarization':
                    case 'nlp_llm_rag_embedding':
                    case 'extract_failed_rows':
                        return ok();
                    case 'embed_documents':
                        return nok(translate("PROJECT.ACTIONS.RECIPES.CAN_ONLY_RUN_MANAGED_FOLDER", "Can only run from a managed folder"))
                    case 'nlp_llm_finetuning':
                    case 'nlp_llm_evaluation':
                        if (!fullLLMMeshStatus.iconEnabled) return nok(fullLLMMeshStatus);
                        else return ok();
                    case 'eda_pca':
                    case 'eda_stats':
                    case 'eda_univariate':
                        return $rootScope.appConfig.userProfile.mayEDA ?
                            ok() : nok(notDataScientist());
                    case "ksql":
                        return nok(translate("PROJECT.ACTIONS.RECIPES.CAN_ONLY_RUN_CONTINUOUSLY_STREAMING_POINT", "Can only run continuously from a streaming endpoint"));
                    case 'csync':
                        return nok(translate("PROJECT.ACTIONS.RECIPES.CAN_ONLY_SYNC_CONTINUOUSLY_STREAMING_POINT", "Can only sync continuously from a streaming endpoint"));
                    default:
                        Logger.error("Recipe usability not implemented for type:", recipeType)
                }
                return ok();
            },

            specialThingMaybePossibleFromDataset: function (dataset, thing) {
                var sqlScientistStatus = usable();
                if (!$rootScope.appConfig.userProfile.maySQL) {
                    sqlScientistStatus = notDataScientist();
                }
                switch (thing) {
                    case "sql":
                        if (!sqlScientistStatus.iconEnabled) return nok(sqlScientistStatus);
                        if (DatasetUtils.isSQLQueryAble(dataset)) {
                            return ok();
                        } else {
                            return nok({"iconDisabledReason": translate("PROJECT.ACTIONS.RECIPES.DATASET_NOT_SQL_TABLE", "Dataset is not a SQL Table")});
                        }
                    case "hive":
                    case "impala":
                    case "pig":
                        return svc.recipeMaybePossibleFromDataset(dataset, thing);
                    case "sql99":
                        if (getSparkStatus().usable) {
                            return ok(); // In addition a configuration enabling HiveContext is necessary but we don't check it
                        }
                        if (dataset.type == "HDFS") {
                            if ($rootScope.appConfig.hiveEnabled) {
                                return ok();
                            } else {
                                return nok(translate("PROJECT.ACTIONS.RECIPES.HIVE_SPARK_CONFIG_REQUIRED", "Hive or Spark configuration required"));
                            }
                        }
                        if (DatasetUtils.isSQLTable(dataset) && dataset.type != "MySQL") {
                            return ok();
                        }
                        return ok();
                    //return nok("Requires an SQL99-compliant database or Spark");
                }
                Logger.warn("Unknown thing type", thing);
                return ok();
            },

            /**
             * Returns whether this recipe *looks* possible from this dataset.
             * Note that additional restrictions might apply and must be checked later.
             *
             * Returns { ok : boolean, reason : String (only if !ok)}
             */
            recipeMaybePossibleFromStreamingEndpoint: function (streamingEndpoint, recipeType) {
                var pyScientistStatus = usable();
                if (!$rootScope.appConfig.userProfile.mayRegularCode) {
                    pyScientistStatus = notDataScientist();
                }
                var rScientistStatus = usable();
                if (!$rootScope.appConfig.userProfile.mayRegularCode) {
                    rScientistStatus = notDataScientist();
                }
                var jlScientistStatus = usable();
                if (!$rootScope.appConfig.userProfile.mayRegularCode) {
                    jlScientistStatus = notDataScientist();
                }
                var scalaScientistStatus = usable();
                if (!$rootScope.appConfig.userProfile.mayRegularCode) {
                    scalaScientistStatus = notDataScientist();
                }

                var unsafeStatus = usable();
                if (!$rootScope.appConfig.globalPermissions || !$rootScope.appConfig.globalPermissions.mayWriteUnsafeCode) {
                    unsafeStatus = noUnsafeCode();
                }
                var safeStatus = usable();
                if ($rootScope.appConfig.impersonationEnabled && (!$rootScope.appConfig.globalPermissions || !$rootScope.appConfig.globalPermissions.mayWriteSafeCode)) {
                    safeStatus = noSafeCode();
                }

                var safeCodeStatus = $rootScope.appConfig.impersonationEnabled ? safeStatus : unsafeStatus;

                switch (recipeType) {
                    case "streaming_spark_scala": {
                        let sparkStatus = getSparkStatus();
                        if (!sparkStatus.iconEnabled) return nok(sparkStatus);
                        if (!scalaScientistStatus.iconEnabled) return nok(scalaScientistStatus);
                        if (!safeCodeStatus.iconEnabled) return nok(safeCodeStatus);
                        else return ok(sparkStatus);
                    }
                    case 'ksql':
                        if (streamingEndpoint.type != 'kafka') return nok(translate("PROJECT.ACTIONS.RECIPES.ONLY_APPLICABLE_KAKFA_ENDPOINTS", "Only applicable to Kafka endpoints"));
                        return ok();
                    case 'cpython':
                        return ok();
                    case 'csync':
                        return ok();
                    default:
                        return nok(translate("PROJECT.ACTIONS.RECIPES.ONLY_ACCEPT_CONTINUOUS_RECIPES", "Only accept continuous recipes"));
                }
                return ok(); // NOSONAR: OK to have a fallback instruction
            },

            specialThingMaybePossibleFromStreamingEndpoint: function (streamingEndpoint, thing) {
                Logger.warn("Unknown thing type", thing);
                return ok();
            },

            newPredictionScoringRecipeFromSMID: function (scope, smProjectKey, smId, dsProjectKey, dsName, zone) {
                scope.zone = zone;
                CreateModalFromTemplate("/templates/savedmodels/new-prediction-scoring-recipe.html", scope, null, function (newScope) {
                    SavedModelsService.listPredictionModels($stateParams.projectKey).then(function(savedModels) {
                        newScope.savedModels = savedModels;
                        newScope.smId = (smProjectKey === $stateParams.projectKey ? '' : smProjectKey + '.') + smId;
                    }, setErrorInScope.bind(newScope));
                    if (dsName) {
                        newScope.$broadcast('preselectInputDataset', SmartId.create(dsName, dsProjectKey));
                    }
                });
            },
            newPredictionScoringRecipeFromDataset: function (scope, dsName, zone) {
                scope.zone = zone;
                CreateModalFromTemplate("/templates/savedmodels/new-prediction-scoring-recipe.html", scope, null, function (newScope) {
                    SavedModelsService.listPredictionModels($stateParams.projectKey).then(function(savedModels) {
                        newScope.savedModels = savedModels;
                    }, setErrorInScope.bind(newScope));
                    newScope.$broadcast('preselectInputDataset', dsName);
                });
            },
            newEvaluationRecipeFromDataset: function (scope, dsName, zone) {
                scope.zone = zone;
                CreateModalFromTemplate("/templates/savedmodels/new-evaluation-recipe.html", scope, null, function (newScope) {
                    newScope.recipeParams = {
                        inputDs: "",
                        smId: "",
                        managedFolderSmartId: null,
                    };
                    SavedModelsService.listEvaluablePredictionModels($stateParams.projectKey).then(function(savedModels) {
                        newScope.savedModels = savedModels;
                    }, setErrorInScope.bind(newScope));
                    newScope.$broadcast('preselectInputDataset', dsName);
                });
            },
            newEvaluationRecipeFromSMID: function (scope, smProjectKey, smId, dsProjectKey, dsName, zone) {
                scope.zone = zone;
                CreateModalFromTemplate("/templates/savedmodels/new-evaluation-recipe.html", scope, null, function (newScope) {
                    newScope.recipeParams = {
                        inputDs: "",
                        managedFolderSmartId: null,
                    };
                    SavedModelsService.listEvaluablePredictionModels($stateParams.projectKey).then(function(savedModels) {
                        newScope.savedModels = savedModels;
                        newScope.recipeParams.smId = (smProjectKey === $stateParams.projectKey ? '' : smProjectKey + '.') + smId;
                    }, setErrorInScope.bind(newScope));
                    if (dsName) {
                        newScope.$broadcast('preselectInputDataset', SmartId.create(dsName, dsProjectKey));
                    }
                });
            },
            newStandaloneEvaluationRecipeFromDataset: function (scope, dsName, zone) {
                scope.zone = zone;
                CreateModalFromTemplate("/templates/savedmodels/new-standalone-evaluation-recipe.html", scope, null, function (newScope) {
                    newScope.projectKey = $stateParams.projectKey;
                    newScope.$broadcast('preselectInputDataset', dsName);
                });
            },
            newStandaloneEvaluationRecipeFromTwoDataset: function (scope, evaluationDSName, referenceDSName, zone) {
                scope.zone = zone;
                CreateModalFromTemplate("/templates/savedmodels/new-standalone-evaluation-recipe.html", scope, null, function (newScope) {
                    newScope.projectKey = $stateParams.projectKey;
                    newScope.$broadcast('preselectInputDataset', evaluationDSName);
                    newScope.$broadcast('preselectReferenceDataset', referenceDSName);
                });
            },
            newClusteringScoringRecipeFromSMID: function (scope, smProjectKey, smId, dsProjectKey, dsName, zone) {
                scope.zone = zone;
                CreateModalFromTemplate("/templates/savedmodels/new-clustering-scoring-recipe.html", scope, null, function (newScope) {
                    SavedModelsService.listClusteringModels($stateParams.projectKey).then(function(savedModels) {
                        newScope.savedModels = savedModels;
                        newScope.smId = (smProjectKey === $stateParams.projectKey ? '' : smProjectKey + '.') + smId;
                    }, setErrorInScope.bind(newScope));
                    if (dsName) {
                        newScope.$broadcast('preselectInputDataset', SmartId.create(dsName, dsProjectKey));
                    }
                });
            },

            newClusteringScoringRecipeFromDataset: function (scope, dsName, zone) {
                scope.zone = zone;
                CreateModalFromTemplate("/templates/savedmodels/new-clustering-scoring-recipe.html", scope, null, function (newScope) {
                    SavedModelsService.listClusteringModels($stateParams.projectKey).then(function(savedModels) {
                        newScope.savedModels = savedModels;
                    }, setErrorInScope.bind(newScope));
                    newScope.$broadcast('preselectInputDataset', dsName);
                });
            },

            smartNewAnalysis: function smartNewAnalysis(scope, datasetSmartName, forMLTask) {
                CreateModalFromTemplate("/templates/analysis/new-analysis-on-dataset-modal.html", scope, null, function (newScope) {
                    newScope.forMLTask = !!forMLTask;
                    newScope.datasetSmartName = datasetSmartName;
                })
            },

            trainSavedModel: function (scope, id) {
                FlowBuildService.openSingleComputableBuildModalFromObjectTypeAndLoc(scope, "SAVED_MODEL",
                            AnyLoc.makeLoc($stateParams.projectKey, id));
            },

            deleteTaggableObject: function(scope, taggableType, id, displayName) {
                var deletionRequests = [{
                    type: taggableType,
                    projectKey: $stateParams.projectKey,
                    id: id,
                    displayName: displayName
                }];
                return TaggableObjectsService.delete(deletionRequests)
                    .then(function() {
                        if ($rootScope.topNav.item.id == id) {
                            if (taggableType == 'CODE_STUDIO') {
                                $state.go("projects.project.code-studios.list");
                            } else {
                                $state.go("projects.project.flow");
                            }
                        } else if (typeof scope.list == "function") {
                            scope.list();
                        }
                    }, setErrorInScope.bind(scope));
            },

            clearDataset: function (scope, id) {
                const taggableItems = [{
                    type: 'DATASET',
                    projectKey: $stateParams.projectKey,
                    id: id,
                    displayName: id
                }];
                return ComputablesService.clear(scope, taggableItems);
            },

            clearManagedFolder: function (scope, id, name) {
                const taggableItems = [{
                    type: 'MANAGED_FOLDER',
                    projectKey: $stateParams.projectKey,
                    id: id,
                    displayName: name
                }];
                return ComputablesService.clear(scope, taggableItems);
            },

            buildManagedFolder: function (scope, id) {
                FlowBuildService.openSingleComputableBuildModalFromObjectTypeAndLoc(scope, "MANAGED_FOLDER",
                            AnyLoc.makeLoc($stateParams.projectKey, id));
            },

            buildModelEvaluationStore: function (scope, id) {
                FlowBuildService.openSingleComputableBuildModalFromObjectTypeAndLoc(scope, "MODEL_EVALUATION_STORE",
                            AnyLoc.makeLoc($stateParams.projectKey, id));
            },

            buildRetrievableKnowledge: function (scope, id) {
                FlowBuildService.openSingleComputableBuildModalFromObjectTypeAndLoc(scope, "RETRIEVABLE_KNOWLEDGE",
                            AnyLoc.makeLoc($stateParams.projectKey, id));
            },
        }


        svc['getFlowOtherItemsBySection'] = function(scope) {
            var ret = [];

            const genaiItems = [];

            var pyScientistStatus = usable();
            if (!scope.appConfig.userProfile.mayRegularCode) {
                pyScientistStatus = notDataScientist();
            }

            var unsafeStatus = usable();
            if (!$rootScope.appConfig.globalPermissions || !$rootScope.appConfig.globalPermissions.mayWriteUnsafeCode) {
                unsafeStatus = noUnsafeCode();
            }
            var safeStatus = usable();
            if ($rootScope.appConfig.impersonationEnabled && (!$rootScope.appConfig.globalPermissions || !$rootScope.appConfig.globalPermissions.mayWriteSafeCode)) {
                safeStatus = noSafeCode();
            }

            var safeCodeStatus = $rootScope.appConfig.impersonationEnabled ? safeStatus : unsafeStatus;

            genaiItems.push({
                type: "It does not matter",
                // TODO @agents Fix translation
                label: translate("PROJECT.ACTIONS.OTHER.GENAI.AGENT", "Code Agent"),
                icon: "dku-icon-ai-agent-code-16",
                disabled: !pyScientistStatus.iconEnabled || !safeCodeStatus.iconEnabled,
                reason: pyScientistStatus.iconDisabledReason || safeCodeStatus.iconDisabledReason,
                fn: function () {
                    SavedModelsService.newCodeAgentPrompt(scope.getRelevantZoneId(scope.$stateParams.zoneId), 'flow')
                },
            });


            genaiItems.push({
                type: "It does not matter",
                // TODO @agents Fix translation
                label: translate("PROJECT.ACTIONS.OTHER.GENAI.AGENT", "Visual Agent"),
                icon: "dku-icon-ai-agent-visual-16",
                disabled: !pyScientistStatus.iconEnabled || !safeCodeStatus.iconEnabled,
                reason: pyScientistStatus.iconDisabledReason || safeCodeStatus.iconDisabledReason,
                fn: function () {
                    SavedModelsService.newVisualAgentPrompt(scope.getRelevantZoneId(scope.$stateParams.zoneId), 'flow')
                },
            });

            if (pyScientistStatus.iconEnabled) {
                let pluginAgents = SavedModelsService.getAllPluginAgents()
                for (let agent of pluginAgents) {
                    genaiItems.push({
                        type: "It does not matter",
                        label: agent.label,
                        icon: 'dku-icon-ai-agent-plugin-16',
                        disabled: false,
                        reason: null,
                        fn: function (){
                            SavedModelsService.newPluginAgentPrompt(agent.label, agent.agentId, scope.getRelevantZoneId(scope.$stateParams.zoneId), 'flow')
                        }
                    });
                }
            }

            genaiItems.push({
                label: translate('PROJECT.ACTIONS.OTHER.GENAI.RETRIEVAL_AUGMENTED_LLM', 'Retrieval-Augmented LLM'),
                icon: 'dku-icon-llm-augmented-16',
                disabled: !pyScientistStatus.iconEnabled || !safeCodeStatus.iconEnabled,
                reason: pyScientistStatus.iconDisabledReason || safeCodeStatus.iconDisabledReason,
                fn: function () {
                    CreateModalFromTemplate('/templates/savedmodels/retrieval-augmented-llm/create-retrieval-augmented-llm-modal.html', scope, 'CreateRetrievalAugmentedLLMModalController', function (newScope) {
                        newScope.input.preselectedInput = null;
                    });
                },
            });

            ret.push({
                isSection: true,
                id: "genai",
                label: translate("PROJECT.ACTIONS.OTHER.GENAI", "Generative AI"),
                icon: "dku-icon-stars-16",
                items: genaiItems
            });

            let streamingEndpoints = {
                isSection: true,
                id: "streaming-endpoints",
                label: translate("PROJECT.ACTIONS.OTHER.STREAMING_ENDPOINTS", "Streaming endpoints"),
                icon: "icon-double-angle-right",
                items: [
                    {
                        type: "kafka", label: "Kafka", icon: "icon-kafka",
                        fn: () => scope.showCreateStreamingEndpointModal("kafka")
                    },
                    {
                        type: "sqs", label: "SQS", icon: "icon-sqs",
                        fn: () => scope.showCreateStreamingEndpointModal("sqs")
                    },
                    {
                        label: "HTTP Server-Sent-Events", type:"httpsse", icon: "icon-httpsse",
                        fn: () => scope.showCreateStreamingEndpointModal("httpsse")
                    }
                ]
            };
            if ($rootScope.featureFlagEnabled("kdbplus")) {
                streamingEndpoints.items.push({
                    label: translate("PROJECT.ACTIONS.OTHER.STREAMING_ENDPOINTS.KDB_TICKER_PLANT", "KDB+Tick ticker plant"), type:"kdbplustick", icon: "icon-httpsse",
                    fn: () => scope.showCreateStreamingEndpointModal("kdbplustick")
                });
            }
            if ($rootScope.appConfig.streamingEnabled) {
                ret.push(streamingEndpoints);
            }

            if ($rootScope.appConfig.streamingEnabled) {
                ret.push({
                    isSection: true,
                    id: "streaming-recipes",
                    label: translate("PROJECT.ACTIONS.OTHER.STREAMING_RECIPES", "Streaming recipes"),
                    icon: "icon-double-angle-right",
                    items: svc.getStreamingRecipesBySection(scope)
                });
            }

            return ret;
        }

        return svc;
    });



})();
