(function() {
'use strict';

const app = angular.module('dataiku.controllers');

app.controller('OnboardingQuestionnaireController', function($scope, $location, $timeout, $interval, DataikuAPI, ProjectFolderService, $rootScope, CreateModalFromTemplate, MonoFuture, $state, OpalsService, $log, WT1, TopbarDrawersService, TOPBAR_DRAWER_IDS, PageSpecificTourService, QuestionnaireService, AvailableLanguages, UserSettingsService, translate, ActivityIndicator) {
        $location.search('fullScreen', true);
        DataikuAPI.profile.getQuestionnaire().then(function(response) {
            $scope.loadQuestionnaire(response.data);
            // after loading the questionnaire, hide the opals drawer
            TopbarDrawersService?.getDrawer(TOPBAR_DRAWER_IDS.OPALS_HELP)?.hide();
        }).catch(function(error) {
            $log.warn("There was an error loading the Onboarding Questionnaire: ", error);
            closeOnboarding();
        });

        function closeOnboarding() {
            $location.search('fullScreen', null);
            $rootScope.appConfig.onboardingPending = false;
        }

        $scope.currentOnboardingStep = 0;

        $scope.onboardingState = {
            skippedQuestionnaire: false, // Questionnaire part was skipped or bypassed automatically in the past (bypassOnboarding() or an upgrade)
            finishedQuestionnaire: false, // finished questionnaire part of the onboarding - either done it, skipped or bypassed
            finishedOnboardingChoice: false, // finished final step of onboarding with tutorial choice - either done it, skipped or bypassed
        };

        // This should be kept in sync with the backend equivalent enum in OnboardingQuestionnaire.java
        const answerTypes = {
            singleChoice: 'SINGLE_CHOICE',
            multipleChoice: 'MULTIPLE_CHOICE',
            singleChoiceOrFreeText: 'SINGLE_CHOICE_OR_FREE_TEXT',
        };

        $scope.questions = [
            {
                id: 'experience',
                label: 'What would you like to do with Dataiku?',
                translationKey: 'ONBOARDING_QUESTIONNAIRE.WHAT_TO_DO_DATAIKU',
                template: '/templates/onboarding-questionnaire/question_experience.html',
                animationFile: "/static/dataiku/images/onboarding-questionnaire/Q1.json",
                answerType: answerTypes.singleChoice,
                noChoiceFadeoutAnimationSegment: [0, 48],
                choices: [
                    { id: 'analysis', label: 'Analytics - Build data pipelines, insights and dashboards', translationKey: 'ONBOARDING_QUESTIONNAIRE.ANALYTICS', fadeinAnimationSegment: [48, 96], fadeoutAnimationSegment: [96, 144]},
                    { id: 'ml', label: 'Models - Use autoML or develop models in code', translationKey: 'ONBOARDING_QUESTIONNAIRE.MODELS', fadeinAnimationSegment: [240, 288], fadeoutAnimationSegment: [288, 336]},
                    { id: 'gen_ai', label: 'GenAI - Leverage LLMs to structure data, build RAG or agents', translationKey: 'ONBOARDING_QUESTIONNAIRE.AGENTS', fadeinAnimationSegment: [432, 480], fadeoutAnimationSegment: [480, 528]},
                    { id: 'new', label: 'Learn the basics - I am new to this', translationKey: 'ONBOARDING_QUESTIONNAIRE.BASICS', fadeinAnimationSegment: [336, 384], fadeoutAnimationSegment: [384, 432]},
                ],
                hasCategories: false,
            },
            {
                id: 'area',
                label: 'Which domain are you involved in?',
                translationKey: 'ONBOARDING_QUESTIONNAIRE.WHICH_DOMAIN',
                template: '/templates/onboarding-questionnaire/question_area.html',
                animationFile: "/static/dataiku/images/onboarding-questionnaire/Q2.json",
                answerType: answerTypes.singleChoiceOrFreeText,
                choices: [
                    { id: 'sales_marketing', label: 'Sales/Marketing', translationKey: 'ONBOARDING_QUESTIONNAIRE.SALES_MARKETING', freeField: false, fadeinAnimationSegment: [0, 48], fadeoutAnimationSegment: [48, 96]},
                    { id: 'manufacturing', label: 'Manufacturing', translationKey: 'ONBOARDING_QUESTIONNAIRE.MANUFACTURING', freeField: false, fadeinAnimationSegment: [96, 144], fadeoutAnimationSegment: [144, 192]},
                    { id: 'supply_chain', label: 'Supply chain', translationKey: 'ONBOARDING_QUESTIONNAIRE.SUPPLY_CHAIN', freeField: false, fadeinAnimationSegment: [192, 240], fadeoutAnimationSegment: [240, 288]},
                    { id: 'finance', label: 'Finance', translationKey: 'ONBOARDING_QUESTIONNAIRE.FINANCE', freeField: false, fadeinAnimationSegment: [288, 336], fadeoutAnimationSegment: [336, 384]},
                    { id: 'other', label: 'Other', translationKey: 'ONBOARDING_QUESTIONNAIRE.OTHER', freeField: true, fadeinAnimationSegment: [384, 432], fadeoutAnimationSegment: [432, 480]},
                ],
                hasCategories: false,
            },
            {
                id: 'tools',
                label: 'What tools are you already familiar with?',
                translationKey: 'ONBOARDING_QUESTIONNAIRE.WHAT_TOOLS_ARE_YOU_ALREADY_FAMILIAR_WITH',
                template: '/templates/onboarding-questionnaire/question_tools.html',
                animationFile: "/static/dataiku/images/onboarding-questionnaire/Q3.json",
                answerType: answerTypes.multipleChoice,
                choices: [
                    {id: 'dataiku', label: 'Dataiku', category: 'data_ai'},
                    {id: 'excel', label: 'Excel', category: 'data_ai'},
                    {id: 'alteryx', label: 'Alteryx', category: 'data_ai'},
                    {id: 'sas', label: 'SAS', category: 'data_ai'},
                    {id: 'knime', label: 'Knime', category: 'data_ai'},
                    {id: 'h2o', label: 'H20.ai', category: 'data_ai'},
                    {id: 'dbt', label: 'dbt', category: 'ingestion_etl'},
                    {id: 'snowflake_ml', label: 'Snowflake ML', category: 'ml_gen_ai_platforms'},
                    {id: 'vertex', label: 'Vertex', category: 'ml_gen_ai_platforms'},
                    {id: 'datarobot', label: 'Datarobot', category: 'ml_gen_ai_platforms'},
                    {id: 'sagemaker', label: 'Sagemaker', category: 'ml_gen_ai_platforms'},
                    {id: 'azure_ml', label: 'Azure ML', category: 'ml_gen_ai_platforms'},
                    {id: 'databricks', label: 'Databricks', category: 'ml_gen_ai_platforms'},
                    {id: 'looker', label: 'Looker', category: 'visualization_dashboarding'},
                    {id: 'qlik', label: 'Qlik', category: 'visualization_dashboarding'},
                    {id: 'power_bi', label: 'Power BI', category: 'visualization_dashboarding'},
                    {id: 'tableau', label: 'Tableau', category: 'visualization_dashboarding'},
                ],
                hasCategories: true,
                categories: [
                    {id: 'data_ai', label: 'Data & AI Tools', translationKey: 'ONBOARDING_QUESTIONNAIRE.DATA_AI_TOOLS', fadeinAnimationSegment: [0, 48], fadeoutAnimationSegment: [50, 72]},
                    {id: 'ingestion_etl', label: 'Ingestion & ETL', translationKey: 'ONBOARDING_QUESTIONNAIRE.INGESTION_ETL', fadeinAnimationSegment: [72, 120], fadeoutAnimationSegment: [122, 144]},
                    {id: 'ml_gen_ai_platforms', label: 'ML & Gen AI Platforms', translationKey: 'ONBOARDING_QUESTIONNAIRE.CLOUD_BASED_ML_PLATFORMS', fadeinAnimationSegment: [144, 192], fadeoutAnimationSegment: [192, 216]},
                    {id: 'visualization_dashboarding', label: 'Visualisation & Dashboarding', translationKey: 'ONBOARDING_QUESTIONNAIRE.VISUALISATION_DASHBOARDING', fadeinAnimationSegment: [216, 264], fadeoutAnimationSegment: [264, 288]},
                ],
            }
        ];

        var paths = [
            {
                id: "quick-start-data-preparation",
                tutorialId: "QS_DATA_PREP",
                opalsPage: OpalsService.PAGES.QUICKSTART_DATA_PREPARATION,
                icon: "dku-icon-dataset-48",
                title: "Explore and analyze a dataset",
                titleTranslationKey: "ONBOARDING_QUESTIONNAIRE.EXPLORE_AND_ANALYZE",
                subtitle: "Discover how to clean and analyze your data",
                subtitleTranslationKey: "ONBOARDING_QUESTIONNAIRE.DISCOVER_CLEAN_ANALYZE",
                fadeinAnimationSegment: [0, 120],
                fadeoutAnimationSegment: [120, 200],
            },
            {
                id: "quick-start-machine-learning",
                tutorialId: "QS_ML",
                opalsPage: OpalsService.PAGES.QUICKSTART_ML,
                icon: "dku-icon-modelize-48",
                title: "Build a model",
                titleTranslationKey: "ONBOARDING_QUESTIONNAIRE.BUILD_A_MODEL",
                subtitle: "Learn how to perform machine learning tasks in Dataiku",
                subtitleTranslationKey: "ONBOARDING_QUESTIONNAIRE.LEARN_MACHINE_LEARNING",
                fadeinAnimationSegment: [200, 330],
                fadeoutAnimationSegment: [330, 370],
            }
        ];
        
        

        if ($rootScope.appConfig.globalPermissions.mayCreateCodeEnvs) {
            paths.push({
                id: "quick-start-developer-guide",
                tutorialId: "QS_DEVELOPERS",
                opalsPage: OpalsService.PAGES.QUICKSTART_DEVELOPER,
                icon: "dku-icon-code-48",
                title: "Code in a notebook",
                titleTranslationKey: "ONBOARDING_QUESTIONNAIRE.CODE_IN_A_NOTEBOOK",
                subtitle: "See how to write and execute your own code in Dataiku",
                subtitleTranslationKey: "ONBOARDING_QUESTIONNAIRE.WRITE_EXECUTE_CODE",
                fadeinAnimationSegment: [370, 452],
                fadeoutAnimationSegment: [452, 504],
            });
        } else {
            paths.push({
                id: "quick-start-code-notebooks-and-recipes",
                tutorialId: "DKU_TUT_CODE_NOTEBOOKS",
                opalsPage: OpalsService.PAGES.QUICKSTART_CODE_NOTEBOOKS,
                icon: "dku-icon-code-48",
                title: "Code in a notebook",
                titleTranslationKey: "ONBOARDING_QUESTIONNAIRE.CODE_IN_A_NOTEBOOK",
                subtitle: "See how to write and execute your own code in Dataiku",
                subtitleTranslationKey: "ONBOARDING_QUESTIONNAIRE.WRITE_EXECUTE_CODE",
                fadeinAnimationSegment: [370, 452],
                fadeoutAnimationSegment: [452, 504],
            });
        }

        // Check that the user has a profile that allows basic LLM Mesh functionalities (writing project content is checked before)
        if ($rootScope.appConfig.userProfile.mayBasicLLMMesh) {
             paths.push({
                 id: "quick-start-gen-ai",
                 tutorialId: "QS_GEN_AI",
                 opalsPage: OpalsService.PAGES.QUICKSTART_GEN_AI,
                 icon: "dku-icon-llm-48",
                 title: "Use Generative AI",
                 titleTranslationKey: "ONBOARDING_QUESTIONNAIRE.GEN_AI",
                 subtitle: "Learn how to leverage large language models in Dataiku",
                 subtitleTranslationKey: "ONBOARDING_QUESTIONNAIRE.USE_GEN_AI_FEATURES",
                 fadeinAnimationSegment: [504, 616],
                 fadeoutAnimationSegment: [616, 670],
             });
        }
        
        $scope.recommendationStep = {
            template: '/templates/onboarding-questionnaire/recommendation.html',
            animationFile: "/static/dataiku/images/onboarding-questionnaire/Q4.json",
            paths: paths
        };


        $scope.installationStep = {
            template: '/templates/onboarding-questionnaire/installation.html',
            animationFile: "/static/dataiku/images/onboarding-questionnaire/installation.json",
        };

        $scope.initAnswerMap = function() {
            const answerMap = {};
            let answerObj = null;
            for (let question of $scope.questions) {
                switch(question.answerType) {
                    case answerTypes.multipleChoice: 
                        answerObj = {
                            selection: []
                        }
                        break;
                    case answerTypes.singleChoiceOrFreeText:
                        answerObj = {
                            selection: null,
                            freeText: null
                        }
                        break;
                    case answerTypes.singleChoice: 
                    default:
                        answerObj = {
                            selection: null
                        };
                }
                answerMap[question.id] = answerObj;
            }
            return answerMap;
        }

        /* A map of question id to answer objects to hold the users answers (build from the question data)
           The shape of object depends on the answer type - see initAnswerMap
         **/
        $scope.answers = $scope.initAnswerMap();

        $scope.getQuestion = function(questionId) {
            return $scope.questions.find((question) => question.id === questionId);
        }

        $scope.getTranslatedQuestion = function(questionId) {
            const question = $scope.getQuestion(questionId);
            return translate(question.translationKey, question.label);
        }

        $scope.getChoiceData = function(question, answerId) {
            return question.choices.find((choice) => choice.id === answerId);
        }

        $scope.getCategoryData = function(question, categoryId) {
            if (!question.hasCategories) {
                return;
            }
            return question.categories.find((category) => category.id === categoryId);
        }

        $scope.getTutorialPathData = function(pathId) {
            return $scope.recommendationStep.paths.find((path) => path.id === pathId);
        }

        // Questions (set of ids) the user has seen and clicked next for (this is what we considered "responded to")
        // Note - this set never shrinks, and if a user submits something and goes back we continue to submit the current state of the answer 
        // (even if they have changed it then only clicked "back")
        $scope.questionsRespondedTo = new Set();

        // To replace JS modulo which can give negative values
        $scope.modulo = function(n, m) {
            return ((n % m) + m) % m;
        }

        $scope.getNextElement = function(array, condition, movement = 1) {
            const currentElementIndex = array.findIndex((element) => condition(element));
            const sizeArray = array.length;
            const targetElementIndex = $scope.modulo(currentElementIndex + movement, sizeArray);
            return array[targetElementIndex];
        }

        $scope.getTargetChoiceId = function(questionId, choiceId, arrowKey) {
            const question = $scope.getQuestion(questionId);
            const choiceData = $scope.getChoiceData(question, choiceId);

            const movement = ['ArrowUp', 'ArrowLeft'].includes(arrowKey) ? -1 : 1;

            if (question.hasCategories && ['ArrowUp', 'ArrowDown'].includes(arrowKey)) {
                const category = choiceData.category;
                const choiceIndexInCategory = question.choices.filter((choice) => choice.category === category).findIndex((choice) => choice.id === choiceId);
                const nextCategory = $scope.getNextElement(question.categories, (cat) => cat.id === category, movement);
                const lengthNextCategory = question.choices.filter((choice) => choice.category === nextCategory.id).length;
                return question.choices.filter((choice) => choice.category === nextCategory.id)[Math.min(choiceIndexInCategory, lengthNextCategory - 1)]?.id;
            }
            return $scope.getNextElement(question.choices, (choice) => choice.id === choiceId, movement)?.id;
        }

        $scope.languageSelectorKeyboardInteraction = function(event, languageId) {
            if (event.key === 'Enter' || event.key == ' ') {
                event.preventDefault();
                $scope.onLanguageChange(languageId);
                return;
            }
            if (['ArrowUp', 'ArrowDown', 'ArrowLeft', 'ArrowRight'].includes(event.key)) {
                const movement = ['ArrowUp', 'ArrowLeft'].includes(event.key) ? -1 : 1;
                const nextLanguageId = $scope.getNextElement($scope.availableLanguages, (lang) => lang.id === languageId, movement)?.id;
                $timeout(function() {
                    document.getElementById(nextLanguageId).focus();
                });
            }
        }

        $scope.selectAnswerOnEnter = function(event, questionId, choiceId) {
            if (event.key === 'Enter' || event.key == ' ') {
                event.preventDefault();
                $scope.selectAnswer(questionId, choiceId);
                return;
            }

            if (['ArrowUp', 'ArrowDown', 'ArrowLeft', 'ArrowRight'].includes(event.key)) {
                const targetChoiceId = $scope.getTargetChoiceId(questionId, choiceId, event.key);
                $timeout(function() {
                  document.querySelector(`label[for="${targetChoiceId}"]`).focus();
                });

                event.preventDefault();
            }
        }

        $scope.selectAnswer = function(questionId, choiceId) {
            if (!$scope.isQuestionStep($scope.currentOnboardingStep)) {
                return;
            }
            const question = $scope.getQuestion(questionId);
            const choiceData = $scope.getChoiceData(question, choiceId);
            switch(question.answerType) {
                case answerTypes.multipleChoice:
                    $scope.toggleSelection($scope.answers[questionId].selection, choiceId);
                    break;
                case answerTypes.singleChoiceOrFreeText:
                    if (choiceData.freeField) {
                        $timeout(function() {
                            document.getElementById(`freeField_${choiceId}`).focus();
                        });
                    };
                    $scope.answers[questionId].selection = choiceId;
                    break;
                default:
                    $scope.answers[questionId].selection = choiceId;
            }
        }

        $scope.toggleSelection = function(choiceList, value) {
            const indexValue = choiceList.indexOf(value);
            if (indexValue >= 0) {
                choiceList.splice(indexValue, 1);
            } else {
                choiceList.push(value);
            }
        }

        $scope.selectTutorialOnEnter = function(event, path) {
            if (!(event.key === 'Enter' || event.key == ' ')) {
                return;
            }
            event.preventDefault();
            $scope.selectQuickStartPath(path);
        }

        $scope.isSelected = function(toolId) {
            return $scope.answers.tools.selection.includes(toolId);
        }

        $scope.filterByCategory = function(choices, categoryId) {
            return choices.filter(choice => choice.category === categoryId);
        }

        $scope.loadQuestionnaire = function(questionnaire) {
            if (!questionnaire || !questionnaire.answers) {
                // No data was found for the user - we can just proceed with questionnaire startup using default state 
                return;
            }

            if (questionnaire.finishedOnboardingChoice) {
                // If the onboarding was actually done but we ended up on this page somehow
                closeOnboarding();
                return;
            }

            // Set answer states from what was saved
            // Note - we need to do this even if we have skipped, or we will loose that data when we skip/start the tutorial and it submits
            for (let savedAnswer of questionnaire.answers) {
                $scope.questionsRespondedTo.add(savedAnswer.questionId);
                const question = $scope.getQuestion(savedAnswer.questionId);
                const answerState = $scope.answers[savedAnswer.questionId];

                if (question.answerType === answerTypes.multipleChoice) {
                    answerState.selection = savedAnswer.answers;
                } else {
                    answerState.selection = savedAnswer.answerId;
                    if (question.answerType === answerTypes.singleChoiceOrFreeText) {
                        answerState.freeText = savedAnswer.freeTextAnswer;
                    }
                }
            }

            // Set current step to the index of the first question they haven't answered (if they have answered them all, just beyond the last question)
            $scope.currentOnboardingStep = questionnaire.answers.length;

            // Has already finished questionnaire (skipped or completed)
            if (questionnaire.finishedQuestionnaire) {
                $scope.progressToRecommendation({skip: questionnaire.skippedQuestionnaire, shouldSubmit: false});
            }
        }

        $scope.isQuestionStep = (step) => step < $scope.questions.length;

        $scope.isRecommendationStep = (step) => step === $scope.questions.length;

        $scope.isInstallationStep = (step) => step === $scope.questions.length + 1;

        $scope.getCurrentOnboardingPane = function () {
            if ($scope.isQuestionStep($scope.currentOnboardingStep)) {
                return $scope.questions[$scope.currentOnboardingStep];
            }
            return $scope.recommendationStep;
        }

        $scope.buildQuestionnaireState = function () {
            const answers = [];
            for (const question of $scope.questions) {
                const answerState = $scope.answers[question.id];
                if (!$scope.questionsRespondedTo.has(question.id) || !answerState.selection) {
                    continue;
                }
                const answer = {
                    questionId: question.id,
                    answerType: question.answerType
                };
                if (question.answerType === answerTypes.multipleChoice) {
                    answer.answers = answerState.selection;
                } else {
                    answer.answerId = answerState.selection;
                    if (question.answerType === answerTypes.singleChoiceOrFreeText) {
                        const choiceInfo = question.choices.find((choice) => choice.id === answerState.selection);
                        if (choiceInfo?.freeField) {
                            answer.freeTextAnswer = answerState.freeText;
                        }
                    }
                }
                answers.push(answer);
            }

            return {
                skippedQuestionnaire: $scope.onboardingState.skippedQuestionnaire,
                finishedQuestionnaire: $scope.onboardingState.finishedQuestionnaire,
                finishedOnboardingChoice: $scope.onboardingState.finishedOnboardingChoice,
                answers: answers,
            };
        }

        $scope.submitQuestionnaire = function () {
            const questionnaireContent = $scope.buildQuestionnaireState();
            DataikuAPI.profile.setQuestionnaire(questionnaireContent).catch(function(error) {
                $log.warn("There was an error saving the Onboarding Questionnaire state: ", error);
            });
            OpalsService.setLocalStorage('onboardingQuestionnaire', JSON.stringify(questionnaireContent));
        }

        $scope.resetKeyboardFocus = function () {
            $timeout(function() {
                document.getElementById('focus-reset-anchor')?.focus();
            });
        }

        $scope.goToNextQuestion = function () {
            $scope.questionsRespondedTo.add($scope.questions[$scope.currentOnboardingStep].id);
            $scope.resetKeyboardFocus();
            if ($scope.isQuestionStep($scope.currentOnboardingStep + 1))
            {
                $scope.submitQuestionnaire();
                $scope.currentOnboardingStep += 1;
            }
            else
            {
               $scope.progressToRecommendation({skip: false, shouldSubmit: true});
            }
        }

        $scope.goToPreviousQuestion = function () {
            if ($scope.currentOnboardingStep > 0)
            {
                $scope.resetKeyboardFocus();
                $scope.currentOnboardingStep -= 1;
            }
        }

        // Finish onboarding recording it is done at the backend and raising any WT1 events needed
        // In some cases we neither skip nor start tutorial as we don't offer it - hence the two params
        $scope.finishOnboarding = function({skippingTutorial, startingTutorial}) {
            $scope.onboardingState.finishedOnboardingChoice = true;
            $scope.submitQuestionnaire();
            if (skippingTutorial) {
                WT1.event("onboarding-tutorial-skip");
            }
            closeOnboarding();
        }

        // What the UI validation considers the question has a valid answer - for multichoice, as no choice (empty array) is a valid answer it is always true
        $scope.hasAnswered = function(questionNumber) {
            const question = $scope.questions[questionNumber];
            const answer = $scope.answers[question?.id];
            if (!answer) {
                // Should not happen
                $log.warn("No answer object for question number " + questionNumber);
                return false;
            }

            if (question.answerType === answerTypes.multipleChoice) {
                return true;
            }
            if (!answer.selection) {
                return false;
            }
            const choiceData = $scope.getChoiceData(question, answer.selection);
            if (question.answerType === answerTypes.singleChoiceOrFreeText && choiceData.freeField && !(answer.freeText)) {
                return false;
            }
            return true;
        }

        $scope.startAreaEditing = function() {
            $timeout(function() {
                const inputElement = document.getElementById('otherInput');
                if (inputElement) {
                  inputElement.focus();
                }
          }, 0);
        }

        $scope.isDotActive = function(dotIndex) {
            return dotIndex <= $scope.currentOnboardingStep;
        }

        /* ********************* Tutorial Recommendation ******************* */

        $scope.progressToRecommendation = function({skip, shouldSubmit}) {
            $scope.onboardingState.skippedQuestionnaire = skip;
            const skippedQuestionIndex = skip && $scope.isQuestionStep($scope.currentOnboardingStep) ? $scope.currentOnboardingStep : null;
            $scope.currentOnboardingStep = $scope.questions.length;
            $scope.onboardingState.finishedQuestionnaire = true;

            if ($scope.answers?.tools?.selection?.includes('dataiku')) {
                $scope.finishOnboarding({skippingTutorial: false, startingTutorial: false});
                ActivityIndicator.success(translate("ONBOARDING_QUESTIONNAIRE.THANK_YOU_FOR_COMPLETING", "Thank you for completing the questionnaire. Welcome back!"));
            }

            $scope.quickStartPaths = $scope.getQuickStartPaths();
            if (!skip) {
                $scope.selectQuickStartPath($scope.quickStartPaths[0]);
            }
            if (shouldSubmit) {
                // Don't submit this stuff when we're just loading the page
                $scope.submitQuestionnaire();
                WT1.tryEvent(skip ? "onboarding-questionnaire-skip" : "onboarding-questionnaire-complete", () => $scope.buildWt1Payload(skippedQuestionIndex));
            }
        }

        // skippedQuestionIndex - index of question skipped to include, or nullish to not include one
        $scope.buildWt1Payload = function(skippedQuestionIndex) {
            // The format is to add properties like this
            // "onboardingQuestionnaire__experience__question"
            // "onboardingQuestionnaire__experience__answer"
            // "onboardingQuestionnaire__experience__answerId"
            // i.e. this pattern
            // "onboardingQuestionnaire__<questionId>__<part>"

            // It's a flattened format mixpanel can cope with well (a list of complex objects would not be)
            // We could have added a nested JSON object (with question ids as keys) and done unnesting in the analytics flow.
            // The result would be the same (for mixpanel) - we use `__` to be consistent with such unnested events

            const payload = {};
            const separator = '__';
            const paramKey = (question, part) => ["onboardingQuestionnaire", question.id, part].join(separator);

            if (skippedQuestionIndex != null) {
                const skipQuestionData = $scope.questions[skippedQuestionIndex];
                payload["onboardingQuestionnaireSkipQuestionId"] = skipQuestionData.id;
                payload["onboardingQuestionnaireSkipQuestion"] = skipQuestionData.label;
            }

            for (const question of $scope.questions) {
                const answerState = $scope.answers[question.id];
                if (!$scope.questionsRespondedTo.has(question.id) || !answerState.selection) {
                    continue;
                }
                const questionKey = paramKey(question, "question");
                const answerKey = paramKey(question, "answer");
                payload[questionKey] = question.label;

                if (question.answerType === answerTypes.multipleChoice) {
                    const selectedChoices = new Set(answerState.selection);
                    const answerIdList = question.choices.filter((choice) => selectedChoices.has(choice.id)).map((choice) => choice.id);
                    // We just give a list of id for multipleChoice questions - we do not send the choice labels
                    payload[answerKey] = answerIdList;
                } else {
                    const choiceInfo = question.choices.find((choice) => choice.id === answerState.selection);
                    let answer;
                    if (choiceInfo?.freeField && question.answerType === answerTypes.singleChoiceOrFreeText) {
                        answer = $scope.answers[question.id].freeText || "";
                    } else {
                        answer = choiceInfo.label;
                    }
                    payload[answerKey] = answer;
                    payload[paramKey(question, "answerId")] = choiceInfo.id;
                }
            }
            return payload;
        }

        $scope.showRecommendedBanner = function() {
            return !$scope.onboardingState.skippedQuestionnaire;
        }

        $scope.getQuickStartPaths = function() {
            // Order the list of quick start paths depending on user answers, since the paths are displayed by their position in the list
            const experienceSelection = $scope.answers.experience.selection;
            const areaSelection = $scope.answers.area.selection;

            const experienceToPathId = {
                "data_preparation": "quick-start-data-preparation",
                "new": "quick-start-data-preparation",
                "ml": "quick-start-machine-learning",
                "gen_ai": "quick-start-gen-ai"
            };

            let prioritizedPathId = null;
            if (experienceSelection === "gen_ai") {
                prioritizedPathId = experienceToPathId["gen_ai"];
            } else if (areaSelection === "manufacturing") {
                prioritizedPathId = experienceToPathId["data_preparation"];
                // The data preparation path is updated in-place to open the manufacturing tutorial instead
                const dataPrepPath = $scope.recommendationStep.paths.find(p => p.id === "quick-start-data-preparation");
                dataPrepPath.opalsPage =  OpalsService.PAGES.QUICKSTART_MANUFACTURING_DATA_PREPARATION;
                dataPrepPath.tutorialId = "QS_MANUF_DATA_PREP_VIZ";
            } else if (experienceSelection && experienceToPathId[experienceSelection]) {
                prioritizedPathId = experienceToPathId[experienceSelection];
            }

            if (prioritizedPathId) {
                $scope.recommendationStep.paths.sort((a, b) => {
                    if (a.id === prioritizedPathId) return -1;
                    if (b.id === prioritizedPathId) return 1;
                    return 0;
                });
            }
            return $scope.recommendationStep.paths;
        };

        $scope.selectedQuickStartPath = null;

        $scope.selectQuickStartPath = function(path) {
            $scope.selectedQuickStartPath = path.id;
            $scope.tutorialId = path.tutorialId;
            $scope.opalsPage = path.opalsPage;
            $scope.tutorialDescription = path.title;
            
            $scope.tutorialStartingStep = 2;
            if (path.opalsPage === OpalsService.PAGES.QUICKSTART_DEVELOPER) {
                $scope.tutorialStartingStep = 1;
            }
        };

        $scope.getRecommendationMessage = function () {
            if ($scope.showRecommendedBanner()) {
                return "💡" + translate("ONBOARDING_QUESTIONNAIRE.SUGGEST_YOU_BEGIN", "Based on your answers, we suggest you begin with this short tutorial");
            }
            return "💡" + translate("ONBOARDING_QUESTIONNAIRE.FOLLOW_THESE_HANDS_ON", "Follow these hands-on tutorials to discover Dataiku's core features");
        }

        $scope.getTitleRecommendation = function () {
            if ($scope.showRecommendedBanner()) {
                return translate("ONBOARDING_QUESTIONNAIRE.HOW_DO_YOU_WANT_TO_GET_STARTED", "How do you want to get started?");
            }
            return translate("ONBOARDING_QUESTIONNAIRE.GET_STARTED_WITH_DATAIKU_STEP_BY_STEP", "Get started with Dataiku step by step");
        }

        /* ********************* Tutorial installation ******************* */


        // @TODO : needs refactoring
        $scope.startTutorial = function () {
            $scope.currentOnboardingStep = $scope.questions.length + 1;
            $scope.installationState = "PENDING";

            ProjectFolderService.getDefaultFolderForNewProject()
                .then(function (defaultFolder) {
                    $scope.defaultFolder = defaultFolder;
                    $scope.installationProgress = 0;

                    MonoFuture($scope).wrap(DataikuAPI.projects.createTutorial)($scope.tutorialId, "TUTORIAL", $scope.defaultFolder.id)
                        .success(function (data) {
                            if (!data.result.success) {
                                $scope.installationState = "ERROR";
                                $scope.failure = {
                                    aborted : data.aborted,
                                    message : data.result.installationError.detailedMessage
                                }
                            } else {
                                $scope.installationState = "SUCCESS";
                                const projectKey = data.result.projectKey;

                                setTimeout(function() {
                                    $state.go("projects.project.flow", { projectKey }).then(() => {
                                        QuestionnaireService.setIsFromQuestionnaire(true);
                                        QuestionnaireService.setOpalsPage($scope.opalsPage); 
                                        QuestionnaireService.setTutorialStartingStep($scope.tutorialStartingStep);
                                
                                        // wait for the flow to be rendered - If the Flow tour is gonna be displayed,
                                        // do not open the Help Center as it will be closed right after - else open the relevant tutorial
                                        const unregisterListener = $rootScope.$on('graphRendered', function() {
                                            PageSpecificTourService.checkFlowTourConditions().then((shouldShowFlowTour) => {
                                                if (!shouldShowFlowTour) {
                                                    OpalsService.navigateToAndShowDrawer(
                                                        $scope.opalsPage,
                                                        { number: $scope.tutorialStartingStep, numCompletedTasks: $scope.tutorialStartingStep - 1 },
                                                    );
                                                };
                                            });
                                            unregisterListener(); // remove listener
                                        });
                                
                                        $scope.finishOnboarding({ skippingTutorial: false, startingTutorial: true });
                                    // Catches error with the state.go
                                    }).catch((error) => {
                                        $scope.installationState = "ERROR";
                                        $scope.failure = {
                                            message : error
                                        }
                                    });
                                // wait for the animation to finish before redirecting (3.3s)
                                }, 3300);
                                
                                

                            }
                        })
                        .update(function (data) {
                            $scope.installationState = "RUNNING";
                            if (data.progress && data.progress.states && data.progress.states.length > 0) {
                                $scope.installationProgress = Math.max($scope.installationProgress, data.progress.states[0]?.cur || 0);
                            } 
                            $scope.installingFuture = data;
                        })
                        // Catches error with tutorial download
                        .error(function (data, status, headers) {
                            $scope.installationState = "ERROR";
                            if ( data.aborted) {
                                $scope.failure = {
                                    aborted : data.aborted,
                                    message : "Aborted"
                                }
                            } else if (data.hasResult) {
                                $scope.failure = {
                                    aborted : data.aborted,
                                    message : data.result.errorMessage
                                }
                            } else {
                                $scope.failure = {
                                    aborted : data.aborted,
                                    message : data.detailedMessage
                                }
                            }
                            $scope.installingFuture = null;
                        });
                })
                // Catches error with the getDefaultFolder
                .catch(function (error) {
                    $scope.installationState = "ERROR";
                    $scope.failure = {
                        message : error
                    }
                });
        };

        $scope.retryTutorialInstallation = function() {
            $scope.installationState = "PENDING";
            $scope.installationProgress = 0;
            $scope.startTutorial();
        };

        $scope.abortStartTutorial = function () {
            if ($scope.installingFuture && $scope.installingFuture.jobId) {
                DataikuAPI.futures.abort($scope.installingFuture.jobId, false);
            }
            $scope.currentOnboardingStep -= 1;
            $scope.installationState = "CANCELED";
            $scope.installationProgress = 0;
            $scope.resetKeyboardFocus();
        };

        $scope.getLoadingText = function() {
            if ($scope.installationState === 'ERROR') {
                return {
                    key: 'ONBOARDING_QUESTIONNAIRE.ERROR_LAUNCHING_TUTORIAL.PLEASE_TRY_AGAIN',
                    fallback: 'Error launching tutorial. Please try again.'
                };
            } else if ($scope.installationState === 'SUCCESS') {
                return {
                    key: 'ONBOARDING_QUESTIONNAIRE.SUCCESS_LOADING_TUTORIAL',
                    fallback: 'Tutorial loaded successfully. Launching tutorial.'
                };
            } else {
                return {
                    key: 'ONBOARDING_QUESTIONNAIRE.LOADING_TUTORIAL',
                    fallback: 'Loading tutorial...'
                };
            }
        };
        
        


        /* ********************* Animations ******************* */
        // The multichoice animation is in sections all the same length
        const MC_ANIMATION_LENGTH_SECONDS = 48;

        $scope.initAnimationObject = (animationInstance) => {
            $scope.animationObject = animationInstance;
            $scope.initAnimation();

            $scope.animationObject.addEventListener("complete", () => {
                if ($scope.nextAnimation) {
                    $scope.animationObject.playSegments($scope.nextAnimation, true);
                    $scope.nextAnimation = null;
                }
            });
        }

        $scope.nextAnimation = null;

        $scope.initAnimation = function() {
            if (!$scope.isQuestionStep($scope.currentOnboardingStep)) {
                if (!$scope.selectedQuickStartPath) {
                    return;
                }
                const selectedPathData = $scope.getTutorialPathData($scope.selectedQuickStartPath);
                $scope.animationObject.playSegments(selectedPathData.fadeinAnimationSegment, true);
                return;
            }

            const currentQuestion = $scope.questions[$scope.currentOnboardingStep];
            const currentAnswer = $scope.answers[currentQuestion.id];

            let animationLastFrame;
            switch (currentQuestion.answerType) {
                case answerTypes.multipleChoice:
                    if (currentAnswer.selection.length === 0) {
                        return;
                    }
                    animationLastFrame = (currentAnswer.selection.length) * MC_ANIMATION_LENGTH_SECONDS;
                    break;
                default:
                    if (!currentAnswer.selection) {
                        return;
                    }
                    animationLastFrame = $scope.getChoiceData(currentQuestion, currentAnswer.selection).fadeinAnimationSegment[1];
                    break;
            }
            $scope.animationObject.playSegments([animationLastFrame, animationLastFrame + 1], true);
            $scope.animationObject.stop();
        }

        $scope.playSelectionAnimation = function (question, previousAnswerSelection, newAnswerSelection) {
            let fadeinAnimationSegment;
            let fadeoutAnimationSegment = question.noChoiceFadeoutAnimationSegment;

            if (question.answerType == answerTypes.multipleChoice) {
                // For multichoice we fade between images which reflect the number of options selected,
                // going from n-1 -> n or n+1 -> n (n is the number of options selected now)
                fadeinAnimationSegment = [MC_ANIMATION_LENGTH_SECONDS * previousAnswerSelection.length, MC_ANIMATION_LENGTH_SECONDS * newAnswerSelection.length];
            } else {
                const newChoiceData = $scope.getChoiceData(question, newAnswerSelection)
                fadeinAnimationSegment = newChoiceData.fadeinAnimationSegment;
                if (previousAnswerSelection) {
                    const previousChoiceData = $scope.getChoiceData(question, previousAnswerSelection)
                    fadeoutAnimationSegment = previousChoiceData.fadeoutAnimationSegment;
                }
            }

            if (fadeoutAnimationSegment) {
                $scope.animationObject.playSegments(fadeoutAnimationSegment, true);
                $scope.nextAnimation = fadeinAnimationSegment;
            } else {
                $scope.animationObject.playSegments(fadeinAnimationSegment, true);
            }
        }

        // Animate the recommendation step
        $scope.$watch('selectedQuickStartPath', function(selectedPath, previouslySelectedPath) {
            if (!$scope.animationObject || !selectedPath) {
                return;
            }
            const selectedPathData = $scope.getTutorialPathData(selectedPath);
            if (previouslySelectedPath)  {
                const previouslySelectedPathData = $scope.getTutorialPathData(previouslySelectedPath);
                $scope.animationObject.playSegments(previouslySelectedPathData.fadeoutAnimationSegment, true);
                $scope.nextAnimation = selectedPathData.fadeinAnimationSegment;
            } else {
                $scope.animationObject.playSegments(selectedPathData.fadeinAnimationSegment, true);
            }
        }, false);

        // Animate questions
        $scope.$watch('answers', function(newAnswers, previousAnswers) {
            if (!$scope.isQuestionStep($scope.currentOnboardingStep)) {
                return;
            }

            const currentQuestion = $scope.questions[$scope.currentOnboardingStep];
            if (previousAnswers[currentQuestion.id].selection === newAnswers[currentQuestion.id].selection) {
                return;
            }
            if (currentQuestion.answerType === answerTypes.multipleChoice && previousAnswers[currentQuestion.id].selection.length === newAnswers[currentQuestion.id].selection.length) {
                return;
            }

            $scope.playSelectionAnimation($scope.getQuestion(currentQuestion.id), previousAnswers[currentQuestion.id].selection, newAnswers[currentQuestion.id].selection);
        }, true);

        /* ********************* i18n ******************* */
        $scope.translate = translate;

        $scope.availableLanguages = AvailableLanguages;
        $scope.selectedLanguage = $rootScope.appConfig.userSettings.uiLanguage;

        $scope.onLanguageChange = function(selectedLanguage) {
            let userSettings = angular.copy($rootScope.appConfig.userSettings);
            userSettings.uiLanguage = selectedLanguage;
            UserSettingsService.saveUserSettings(userSettings).catch(setErrorInScope.bind($scope));
        }

    });

    app.service('QuestionnaireService', function () {
        var isFromQuestionnaire = false;
        var opalsPage;
        var tutorialStartingStep;
        return {
            isFromQuestionnaire: function () {
                return isFromQuestionnaire;
            },

            setIsFromQuestionnaire: function(value) {
                isFromQuestionnaire = value;
            },

            getOpalsPage: function () {
                return opalsPage;
            },

            setOpalsPage: function(page) {
                opalsPage = page;
            },

            setTutorialStartingStep: function(step) {
                tutorialStartingStep = step;
            },

            getTutorialStartingStep: function() {
                return tutorialStartingStep;
            }
        }
    })

}());
