 /* Shaker support functions for processors library */

function StepIAE(message) {
    this.message = message;
    this.name = "StepIAE";
}
StepIAE.prototype = new Error;

(function(){
'use strict';

if (!String.prototype.format) {
    String.prototype.format = function() {
        var formatted = this;
        for(var arg in arguments) {
            formatted = formatted.replace("{" + arg + "}", arguments[arg]);
        }
        return formatted;
    };
}

function truncate(str, len) {
    /**
     * Truncate a string to make sure it takes at most
     * n characters.
     * Whenever possible truncates on special chars.
     *
     * If str is not a string, returns str unchanged.
     */
    if ((typeof str !== "string") || (str.length <= len)) {
        return str;
    }

    // eslint-disable-next-line no-useless-escape
    var cutOn = /[ ,\.,;\-\\\"\n\?\!\|]/g
    var truncated = str.substring(0, len-1);
    var lastSeparatorIdx = regexLastIndexOf(cutOn, truncated);

    // we don't want to cut more too much.
    if (lastSeparatorIdx > len / 2) {
        truncated = str.substring(0, lastSeparatorIdx);
    }
    return truncated + '…';
}

function hasNonEmptyParam(params, key){
    switch (typeof params[key]) {
        case 'undefined': return false;
        case 'string':    return params[key].length > 0;
        default:          return true;
    }
}

function hasAll(params, keys) {
    for (var index in keys) {
        if (!hasNonEmptyParam(params, keys[index])) return false;
    }
    return true;
}

function inColName(value) {
    return "<span class=\"input-column-name\">" + sanitize(value) + "</span>";
}
function outColName(value) {
    return "<span class=\"output-column-name\">" + sanitize(value) + "</span>";
}
function numLiteral(value) {
    return "<span class=\"num-literal\">" + sanitize(value) + "</span>";
}
function anumLiteral(value) {
    if (value == null || value.length === 0) {
        return '<span class="alphanum-literal">\'\'</span>';
    } else {
        return "<span class=\"alphanum-literal\">" + sanitize(value) + "</span>";
    }
}
function actionVerb(value) {
    return "<span class=\"action-verb\">" + sanitize(value) + "</span>";
}
function meaningName(value) {
    return "<span class=\"meaning-label\">" + sanitize(value) + "</span>";
}

// strong param value
// Please use with caution, it breaks the right panel view.
function strongify(value) {
    return "<strong>" + value + "</strong>";
}

// Boolean param value
function toBoolean(value) {
    return value == true || value == "true";
}

function isBlank(x) {
    return x == null|| x.length === 0;
}

function checkDate(d, name) {
    var match = /^(?:[1-9]\d{3}-(?:(?:0[1-9]|1[0-2])-(?:0[1-9]|1\d|2[0-8])|(?:0[13-9]|1[0-2])-(?:29|30)|(?:0[13578]|1[02])-31)|(?:[1-9]\d(?:0[48]|[2468][048]|[13579][26])|(?:[2468][048]|[13579][26])00)-02-29)(T(?:[01]\d|2[0-3])(:[0-5]\d:[0-5]\d(\.\d{3})?)?(?:Z|[+-][01]\d:[0-5]\d)?)?$/
    if (!(match.exec(d))) {
        throw new StepIAE("Invalid " + name);
    }
}

const booleanLiteralsMapping = {
    "true":true,
    "false":false,
    "yes": true,
    "no": false,
    "1": true,
    "0": false,
    "y": true,
    "n": false,
    "t": true,
    "f": false,
    "o": true
};

function isNumber(type) {
    return ["tinyint", "smallint", "int", "bigint", "float", "double"].indexOf(type) > -1;
}

function isBool(value) {
    return Object.keys(booleanLiteralsMapping).indexOf(value.toLowerCase()) > -1;
}

function parseBoolean(value) {
    return booleanLiteralsMapping[value.toLowerCase()] || false;
}

function filterDescEqualityOperator(type, value) {
    if (isNumber(type)) {
        return "== [number]";
    } else if(type === "date" && !isNaN(new Date(Date.parse(value)))) {
        return "== [date]";
    } else if(type === "boolean" && isBool(value)) {
        return parseBoolean(value).toString();
    } else if(type === "array" && "[]" === value) {
        return "empty array";
    }
    return "== [string]";
}

function parseNumber(type, value) {
    switch (type) {
        case "tinyint":
        case "smallint":
        case "int":
        case "bigint":
            return Number.parseInt(value);
        case "float":
        case "double":
            return Number.parseFloat(value);
        default:
            return value;
        }
}

function flagCheckValid(params) {
    if (!params["flagColumn"]) throw new StepIAE("Flag output column not selected");
}

function visualIfParams(input, type, value, operator) {
    let condition = {
        "input": input,
        "operator": operator
    };
    if(value) {
        if(isNumber(type)) {
            condition["num"] = parseNumber(type, value);
        } else if (type === "date") {
            const dateTime = new Date(Date.parse(value));
            if(!isNaN(dateTime)) {
                const dateTimeUTC = moment(dateTime).utc();
                const date = dateTimeUTC.format('YYYY-MM-DD');
                const time = dateTimeUTC.format('HH:mm');
                condition["date"] = date;
                condition["time"] = time;
                condition["unit"] = "minutes";
            } else {
                condition["string"] = value;
            }
        } else if (type === "boolean") {
            //nothing to do here
        } else {
            condition["string"] = value;
        }
    }
    return {
        "visualIfDesc": {
            "elseIfThens": [],
            "elseActions": [],
            "ifThen": {
                "filter": {
                    "uiData": {
                        "mode": "&&",
                        "conditions": [
                            condition
                        ]
                    },
                    "distinct": true,
                    "enabled": true
                },
                "actions": [
                    {
                        "outputColumnName": "",
                        "formula": "",
                        "value": "",
                        "operator": "ASSIGN_VALUE"
                    }
                ]
            }
        }
    };
}

var app = angular.module('dataiku.shaker.library', ['dataiku.filters', 'platypus.utils']);

app.factory("ShakerProcessorsInfo", function($filter, Fn, Assert, $translate, translate, DateUtilsService, ChartFilterUtils) {

    function isValidIfHasColumn(params, columnName, translateID, translateDefaultValue, {throwError}) {
        const isValid = hasNonEmptyParam(params, columnName);
        if (!isValid && throwError) {
            throw new StepIAE(translate(translateID, translateDefaultValue));
        }
        return isValid;
    }

    function defaultIsValidIfHasColumn(params, columnName, {throwError}) {
         return isValidIfHasColumn(params,
                                   columnName,
                                   "SHAKER.PROCESSORS.ERROR.COLUMN_NOT_SPECIFIED", "Column not specified",
                                   {throwError});
    }

    function isCurrencyConverterProcessorValid(params, {throwError}) {
        if (params["isCurrencyFromColumn"] && !hasNonEmptyParam(params, "refCurrencyColumn")) {
            if (throwError) {
                throw new StepIAE(translate('SHAKER.PROCESSOR.CurrencyConverter.FORM.ERROR.MISSING_CURRENT_CODE', "'Currency code' column expected"));
            }
            return false;
        }
        if (params["dateInput"] === "COLUMN" && !hasNonEmptyParam(params, "refDateColumn")) {
            if (throwError) {
                throw new StepIAE(translate('SHAKER.PROCESSOR.CurrencyConverter.FORM.ERROR.MISSING_DATE_COLUMN', "'Date parsed' column expected"));
            }
            return false;
        }
        if (params["dateInput"] === "CUSTOM" && !hasNonEmptyParam(params, "refDateCustom")) {
            if (throwError) {
                throw new StepIAE(translate('SHAKER.PROCESSOR.CurrencyConverter.FORM.ERROR.MISSING_DATE_CUSTOM', "'Custom date' column expected"));
            }
            return false;
        }
        // No need to throw an error here, the UI display is already correct on some basic inputs missing
        return hasAll(params, ["inputColumn", "inputCurrency", "outputCurrency"]);
    }

    function isColumnRenamerValid(params, {throwError}) {
        if (Array.isArray(params["renamings"])) {
            for (const elt of params["renamings"]) {
                if (!hasAll(elt, ["from", "to"])) {
                    if (throwError) {
                        throw new StepIAE(translate('SHAKER.PROCESSOR.ColumnRenamer.ERROR.MISSING_RENAMINGS_SRC', "Missing parameter: Renamings column"));
                    }
                    return false;
                }
            }
        }
        // No need to throw an error here, the UI display is already correct on some basic inputs missing
        return hasNonEmptyParam(params, "renamings");
    }

    function isDateComponentsExtractorValid(params, {throwError}) {
        if ((params["timezone_id"] === "extract_from_column" || params["timezone_id"] === "extract_from_ip") && !hasNonEmptyParam(params, "timezone_src")) {
            if (throwError) {
                throw new StepIAE(translate('SHAKER.PROCESSOR.DateComponentsExtractor.ERROR.MISSING_TZ_SOURCE', "Missing parameter: Source column"));
            }
            return false;
        }
        // No need to throw an error here, the UI display is already correct on some basic inputs missing
        return hasAll(params, ["column", "timezone_id"]);
    }

    function isHolidaysComputerValid(params, {throwError}) {
        if (params["calendar_id"] === "extract_from_column" && !hasNonEmptyParam(params, "calendar_src")) {
            if (throwError) {
                throw new StepIAE(translate('SHAKER.PROCESSOR.HolidaysComputer.ERROR.MISSING_CALENDAR_SRC', "Missing parameter: Country code source column"));
            }
            return false;
        }
        if ((params["timezone_id"] === "extract_from_column" || params["timezone_id"] === "extract_from_ip") && !hasNonEmptyParam(params, "timezone_src")) {
            if (throwError) {
                throw new StepIAE(translate('SHAKER.PROCESSOR.HolidaysComputer.ERROR.MISSING_TIMEZONE_SRC', "Missing parameter: Timezone source column"));
            }
            return false;
        }
        // No need to throw an error here, the UI display is already correct on some basic inputs missing
        return hasAll(params, ["inCol", "calendar_id", "timezone_id"]);
    }

    function appliesToCheckValid(params) {
        if (!params["appliesTo"]) throw new StepIAE(translate('SHAKER.PROCESSORS.ERROR.APPLIES_TO_NOT_SELECTED', "'Applies to' mode not selected"));

        switch (params["appliesTo"]) {
            case "SINGLE_COLUMN":
                if (!params.columns || params.columns.length === 0 || params.columns[0] == null || params.columns[0].length === 0) {
                    throw new StepIAE(translate('SHAKER.PROCESSORS.ERROR.COLUMN_NOT_SPECIFIED', "Column not specified"));
                }
                break;
            case "COLUMNS":
                if(!params.columns || params.columns.length === 0){
                    throw new StepIAE(translate('SHAKER.PROCESSORS.ERROR.COLUMN_NOT_SPECIFIED', "Column not specified"));
                }
                params.columns.forEach(colName => {
                    if (!colName) {throw new StepIAE(translate('SHAKER.PROCESSORS.ERROR.EMPTY_COLUMN_NAME', "Empty column name"));}
                })
                break;
            case "PATTERN":
                if (params.appliesToPattern == null || params.appliesToPattern.length === 0) {
                    throw new StepIAE(translate('SHAKER.PROCESSORS.ERROR.PATTERN_NOT_SPECIFIED', "Pattern not specified"));
                }
                break;
            default:
                break;
        }
    }

    function appliesToDescription(params) {
        switch (params["appliesTo"]) {
            case "SINGLE_COLUMN":
                return inColName(params["columns"][0]);
            case "COLUMNS":
                if (params["columns"].length > 4) {
                    return translate('SHAKER.PROCESSORS.DESCRIPTION.APPLIES_TO.COLUMNS.MANY', "{{count}} columns", {
                        count: anumLiteral(params["columns"].length)
                    });
                } else {
                    return translate('SHAKER.PROCESSORS.DESCRIPTION.APPLIES_TO.COLUMNS.FEW', '{{count > 1 ? "columns" : "column"}} {{columns}}', {
                        count: params["columns"].length,
                        columns: params["columns"].map(inColName).join(", ")
                    });
                }
            case "PATTERN":
                return translate('SHAKER.PROCESSORS.DESCRIPTION.APPLIES_TO.PATTERN', "columns matching /{{pattern}}/", {
                    pattern: anumLiteral(params["appliesToPattern"])
                });
            case "ALL":
                if (params["booleanMode"] === "OR") {
                    return translate('SHAKER.PROCESSORS.DESCRIPTION.APPLIES_TO.AT_LEAST_ONE', strongify("at least") + " one column");
                } else {
                    return translate('SHAKER.PROCESSORS.DESCRIPTION.APPLIES_TO.ALL', strongify("all") + " columns");
                }
        }
    }

    function comaList(items, lastSeparator) {
        if (items.length === 1) {
            return items[0]
        }
        return items.slice(0, items.length - 1).join(', ') + " " + lastSeparator + " " + items[items.length - 1]
    }

    function filterAndFlagSummary(params, conditionFunction, columns) {
        const columnsDesc = columns === undefined ? appliesToDescription(params) : columns;
        const multipleColumns = !(columns !== undefined
            || params["appliesTo"] === "SINGLE_COLUMN"
            || (params["appliesTo"] === "COLUMNS" && params["columns"].length === 1)
            || (params["appliesTo"] === "ALL" && params["booleanMode"] === "OR"));
        switch (params["action"]) {
            case "KEEP_ROW":
                return translate('SHAKER.PROCESSOR.FilterAndFlag.SUMMARY.KEEP_ROW', actionVerb("Keep") + " rows where {{condition}}",
                    { condition: conditionFunction(params, columnsDesc, multipleColumns) });
            case "REMOVE_ROW":
                return translate('SHAKER.PROCESSOR.FilterAndFlag.SUMMARY.REMOVE_ROW', actionVerb("Remove") + " rows where {{condition}}",
                    { condition: conditionFunction(params, columnsDesc, multipleColumns) });
            case "FLAG":
                return translate('SHAKER.PROCESSOR.FilterAndFlag.SUMMARY.FLAG', actionVerb("Flag") + " rows where {{condition}}",
                    { condition: conditionFunction(params, columnsDesc, multipleColumns) });
            case "CLEAR_CELL":
                return translate('SHAKER.PROCESSOR.FilterAndFlag.SUMMARY.CLEAR_CELL', actionVerb("Clear") + " cells in {{columns}} if {{condition}}",
                    { columns: columnsDesc, condition: conditionFunction(params, "value", false) });
            case "DONTCLEAR_CELL":
                return translate('SHAKER.PROCESSOR.FilterAndFlag.SUMMARY.DONTCLEAR_CELL', actionVerb("Keep") + " cells in {{columns}} only if {{condition}}",
                    { columns: columnsDesc, condition: conditionFunction(params, "value", false) });
            default:
                return false;
        }
    }

    function filterAndFlagOnNumericRangeCondition(params, columns) {
        if (params["min"] && params["max"]) {
            return translate("SHAKER.PROCESSOR.FilterAndFlag.SUMMARY.CONDITION.BETWEEN", "{{min}} &le; {{columns}} &le; {{max}}",
                { min: numLiteral(params["min"]), max: numLiteral(params["max"]), columns: columns });
        } else if (params["min"]) {
            return translate("SHAKER.PROCESSOR.FilterAndFlag.SUMMARY.CONDITION.MIN", "{{columns}} &ge; {{min}}",
                { min: numLiteral(params["min"]), columns: columns });
        } else {
            return translate("SHAKER.PROCESSOR.FilterAndFlag.SUMMARY.CONDITION.MAX", "{{columns}} &le; {{max}}",
                { max: numLiteral(params["max"]), columns: columns });
        }
    }

    function filterAndFlagOnValueCondition(params, columns, multipleColumns) {
        const p = {
            literal: anumLiteral(truncate(comaList(params["values"], translate('SHAKER.PROCESSOR.COMMA_LIST_OR', 'or')), 60)),
            columns: columns,
            multipleColumns: multipleColumns
        }
        const mode = params["matchingMode"] || "FULL_STRING";
        const exclude = params["exclude"] || false;
        if (exclude) {
            if (mode === "FULL_STRING") {
                return translate("SHAKER.PROCESSOR.FilterAndFlag.SUMMARY.CONDITION.FULL_STRING.EXCLUDE", "{{columns}} {{multipleColumns?'are':'is'}} not {{literal}}", p);
            } else if (mode === "SUBSTRING") {
                return translate("SHAKER.PROCESSOR.FilterAndFlag.SUMMARY.CONDITION.SUBSTRING.EXCLUDE", "{{columns}} {{multipleColumns?'do':'does'}} not contain {{literal}}", p);
            } else {
                return translate("SHAKER.PROCESSOR.FilterAndFlag.SUMMARY.CONDITION.MATCHES.EXCLUDE", "{{columns}} {{multipleColumns? 'do' : 'does'}} not match {{literal}}", p);
            }
        } else {
            if (mode === "FULL_STRING") {
                return translate("SHAKER.PROCESSOR.FilterAndFlag.SUMMARY.CONDITION.FULL_STRING", "{{columns}} {{multipleColumns?'are':'is'}} {{literal}}", p);
            } else if (mode === "SUBSTRING") {
                return translate("SHAKER.PROCESSOR.FilterAndFlag.SUMMARY.CONDITION.SUBSTRING", "{{columns}} {{multipleColumns?'contain':'contains'}} {{literal}}", p);
            } else {
                return translate("SHAKER.PROCESSOR.FilterAndFlag.SUMMARY.CONDITION.MATCHES", "{{columns}} {{multipleColumns?'match':'matches'}} {{literal}}", p);
            }
        }
    }

    function filterAndFlagImpactVerb(params){
        if (params.action === "REMOVE_ROW" || params.action === "KEEP_ROW") {
            return "deleted";
        } else {
            return "modified";
        }
    }

    function filterAndFlagOnDateSummary(type, params) {
        const datePartLabels = {
            YEAR: translate("GLOBAL.DATE.YEAR", "Year"),
            QUARTER_OF_YEAR: translate("GLOBAL.DATE.QUARTER", "Quarter"),
            WEEK_OF_YEAR: translate("GLOBAL.DATE.WEEK", "Week"),
            MONTH_OF_YEAR: translate("GLOBAL.DATE.MONTH", "Month"),
            DAY_OF_MONTH: translate("GLOBAL.DATE.DAY_OF_MONTH", "Day of month"),
            DAY_OF_WEEK: translate("GLOBAL.DATE.DAY_OF_WEEK", "Day of week"),
            HOUR_OF_DAY: translate("GLOBAL.DATE.HOUR", "Hour"),
            INDIVIDUAL: translate("GLOBAL.DATE_PART.INDIVIDUAL", "Individual dates")
        };
        const relativeDatePartsLabels = {
            YEAR: translate("SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RELATIVE_PART.YEAR", "year"),
            QUARTER_OF_YEAR: translate("SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RELATIVE_PART.QUARTER_OF_YEAR", "quarter"),
            WEEK_OF_YEAR: translate("SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RELATIVE_PART.WEEK_OF_YEAR", "week"),
            MONTH_OF_YEAR: translate("SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RELATIVE_PART.MONTH_OF_YEAR", "month"),
            DAY_OF_MONTH: translate("SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RELATIVE_PART.DAY_OF_MONTH", "day"),
            HOUR_OF_DAY: translate("SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RELATIVE_PART.HOUR_OF_DAY", "hour")
        };

        function relativeDatePartsLabelPlural(part) {
            const translationID = "SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RELATIVE_PART." + part;
            return translate(translationID + ".PLURAL", translate(translationID) + "s");
        }

        function dateRangeCondition(params, columns, multipleColumns) {
            if (!params["min"] && params["max"]) {
                return translate('SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RANGE.MAX', "{{columns}} is before {{dateMax}}", {
                    columns: columns, multipleColumns: multipleColumns,
                    dateMax: numLiteral(DateUtilsService.convertDateFromTimezone(new Date(params["max"]), params["timezone_id"]).toISOString())
                });
            } else if (params["min"] && !params["max"]) {
                return translate('SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RANGE.MIN', "{{columns}} is after {{dateMin}}", {
                    columns: columns, multipleColumns: multipleColumns,
                    dateMin: numLiteral(DateUtilsService.convertDateFromTimezone(new Date(params["min"]), params["timezone_id"]).toISOString())
                });
            } else if (params["min"] && params["max"]) {
                return translate('SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RANGE.BETWEEN', "{{columns}} is between {{dateMin}} and {{dateMax}}", {
                    columns: columns, multipleColumns: multipleColumns,
                    dateMin: numLiteral(DateUtilsService.convertDateFromTimezone(new Date(params["min"]), params["timezone_id"]).toISOString()),
                    dateMax: numLiteral(DateUtilsService.convertDateFromTimezone(new Date(params["max"]), params["timezone_id"]).toISOString())
                });
            } else {
                return translate('SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.MISSING_BOUNDS', "{{columns}} {{multipleColumns?'are':'is'}} not empty", {
                    columns: columns, multipleColumns: multipleColumns
                });
            }
        }
        function dateRelativeCondition(params, columns, multipleColumns) {
            if (params["part"] == null  || params["option"] == null || !ChartFilterUtils.isRelativeDateFilterEffective(params["part"], params["option"])) {
                return translate('SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.MISSING_BOUNDS', "{{columns}} {{multipleColumns?'are':'is'}} not empty", {
                    columns: columns, multipleColumns: multipleColumns
                });
            }
            const translateIdParts = [];
            const defaultValueParts = [];
            if (params["option"].last) {
                if (params["option"].last > 1) {
                    translateIdParts.push('.LAST.MULTIPLE');
                    defaultValueParts.push('{{columns}} is within the last {{last}} {{parts}}')

                } else {
                    translateIdParts.push('.LAST');
                    defaultValueParts.push('{{columns}} is within the last {{part}}')
                }
            }
            if (params["option"].containsCurrentDatePart) {
                if (params["option"].isUntilNow) {
                    translateIdParts.push('.THIS.TO');
                    defaultValueParts.push('{{columns}} is between the beginning of this {{part}} and now')
                    return translate(`SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RELATIVE${translateIdParts.join('')}`, defaultValueParts.join(', '), { 
                        columns, 
                        multipleColumns, 
                        part: relativeDatePartsLabels[params["part"]], 
                        parts: relativeDatePartsLabelPlural(params["part"]),
                        last: params["option"].last,
                        next: params["option"].next
                    });
                } else {
                    translateIdParts.push('.THIS');
                    defaultValueParts.push('{{columns}} is in this {{part}}')
                }
                if (!params["option"].last && !params["option"].next) {
                    if (params["part"] === 'HOUR_OF_DAY') {
                        return translate('SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RELATIVE.THIS.HOUR_OF_DAY', "{{columns}} is today within this hour", {
                            columns, 
                            multipleColumns
                        });
                    } else if (params["part"] === 'DAY_OF_MONTH') {
                        return translate('SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RELATIVE.THIS.DAY_OF_MONTH', "{{columns}} is today", {
                            columns, 
                            multipleColumns 
                        });
                    } else {
                        return translate('SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RELATIVE.THIS.OTHER', "{{columns}} is in this {{part}}", { 
                            columns,
                            multipleColumns, 
                            part: relativeDatePartsLabels[params["part"]]
                        });
                    }
                }
            }
            
            if (params["option"].next) {
                if (params["option"].next > 1) {
                    translateIdParts.push('.NEXT.MULTIPLE');
                    defaultValueParts.push('{{columns}} is within the next {{next}} {{parts}}')
                } else {
                    translateIdParts.push('.NEXT');
                    defaultValueParts.push('{{columns}} is within the next {{part}}')
                }
            }

            return translate(`SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RELATIVE${translateIdParts.join('')}`, defaultValueParts.join(', '), { 
                columns, 
                multipleColumns, 
                part: relativeDatePartsLabels[params["part"]], 
                parts: relativeDatePartsLabelPlural(params["part"]),
                last: params["option"].last,
                next: params["option"].next
            });
        }

        function datePartCondition(params, columns, multipleColumns) {
            if (params["part"] && params["values"] && params["values"].length > 0) {
                if (params["part"] === 'INDIVIDUAL') {
                    return translate('SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.PART.INDIVIDUAL', "{{columns}} with {{part}}", { columns: columns, multipleColumns: multipleColumns, part: datePartLabels[params["part"]] });
                } else {
                    return translate('SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.PART.MULTIPLE', "{{columns}} with {{part}} in {{values}}", { columns: columns, multipleColumns: multipleColumns, part: datePartLabels[params["part"]], values: params["values"] });
                }
            }
            return translate('SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.MISSING_BOUNDS', "{{columns}} {{multipleColumns?'are':'is'}} not empty", { columns: columns, multipleColumns: multipleColumns });
        }

        switch (params['filterType']) {
            case 'RANGE':
                return filterAndFlagSummary(params, dateRangeCondition);
            case 'RELATIVE':
                return filterAndFlagSummary(params, dateRelativeCondition);
            case 'PART':
                return filterAndFlagSummary(params, datePartCondition);
        }
    }

    function filterAndFlagOnDateRangeSummary(type, params){
        return filterAndFlagSummary(params, function(params, columns, multipleColumns) {
            const min = params["min"];
            const max = params["max"];
            if (!min && max) {
                return translate('SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RANGE.MAX', "{{columns}} {{multipleColumns?'are':'is'}} before {{dateMax}}", {
                    columns: columns,
                    multipleColumns: multipleColumns,
                    dateMax: numLiteral(max)
                });
            } else if (min && !max) {
                return translate('SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RANGE.MIN', "{{columns}} {{multipleColumns?'are':'is'}} after {{dateMin}}", {
                    columns: columns,
                    multipleColumns: multipleColumns,
                    dateMin: numLiteral(min)
                });
            } else if (min && max) {
                return translate('SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.RANGE.BETWEEN', "{{columns}} {{multipleColumns?'are':'is'}} between {{dateMin}} and {{dateMax}}", {
                    columns: columns,
                    multipleColumns: multipleColumns,
                    min: numLiteral(min),
                    dateMax: numLiteral(max)
                });
            } else {
                return translate('SHAKER.PROCESSOR.FilterAndFlagOnDate.SUMMARY.MISSING_BOUNDS', "{{columns}} {{multipleColumns?'are':'is'}} not empty", {
                    columns: columns, multipleColumns: multipleColumns
                });
            }
        });
    }

    function geoDistanceCheckValid(params){
        if (!params["input2"] && params["compareTo"] === "COLUMN") {
            throw new StepIAE(translate('SHAKER.PROCESSOR.GeoDistance.ERROR.MISSING_PARAM_OTHER_COLUMN', "Missing parameter: Other column"));
        }
        if (params["compareTo"] === "GEOPOINT") {
            if (!params["refLatitude"]) throw new StepIAE(translate('SHAKER.PROCESSOR.GeoDistance.ERROR.MISSING_PARAM_LATITUDE', "Missing parameter:  Geopoint latitude"));
            if (!params["refLongitude"]) throw new StepIAE(translate('SHAKER.PROCESSOR.GeoDistance.ERROR.MISSING_PARAM_LONGITUDE', "Missing parameter: Geopoint longitude"));
        }
        if (!params["refGeometry"] && params["compareTo"] === "GEOMETRY") {
            throw new StepIAE(translate('SHAKER.PROCESSOR.GeoDistance.ERROR.MISSING_PARAM_GEOMETRY', "Missing parameter: Geometry"));
        }
    }

    function switchCaseCheckValid(params){
        if (!params["inputColumn"] || params.inputColumn  === null || params.inputColumn.length === 0) {
            throw new StepIAE(translate('SHAKER.PROCESSOR.SwitchCase.ERROR.MISSING_INPUT_COLUMN', "Input column not specified"));
        }
        if (!params["outputColumn"] || params.outputColumn==="" || params.outputColumn===null) {
            throw new StepIAE(translate('SHAKER.PROCESSOR.SwitchCase.ERROR.MISSING_OUTPUT_COLUMN', "Output column not specified"));
        }
    }

    function visualIfRuleCheckValid(params) {
        if (params.visualIfDesc.ifThen.filter.uiData.conditions.length===0) {
            throw new StepIAE(translate("SHAKER.PROCESSOR.VisualIfRule.ERROR.MISSING_IF_CONDITION", "Missing if condition"));
        }
        if (params.visualIfDesc.ifThen.actions.length===0) {
            throw new StepIAE(translate("SHAKER.PROCESSOR.VisualIfRule.ERROR.MISSING_THEN_ACTION", "Missing then action"));
        }

        params.visualIfDesc.elseIfThens.forEach((element) => {
            if (element.filter.uiData.conditions.length===0) {
                throw new StepIAE(translate("SHAKER.PROCESSOR.VisualIfRule.ERROR.MISSING_IF_CONDITION", "Missing if condition"));

            } else if (element.actions.length===0) {
                throw new StepIAE(translate("SHAKER.PROCESSOR.VisualIfRule.ERROR.MISSING_ELSE_IF_ACTION", "Missing else if action"));
            }
            element.actions.forEach((action) => {
                if(!("outputColumnName" in action) || !action.outputColumnName || action.outputColumnName==="") {
                    throw new StepIAE(translate("SHAKER.PROCESSOR.VisualIfRule.ERROR.MISSING_OUTPUT_COLUMN_NAME", "Missing output column name"));
                }
            });
        });

        params.visualIfDesc.ifThen.actions.forEach((action) => {
            if (!("outputColumnName" in action) || !action.outputColumnName || action.outputColumnName==="") {
                throw new StepIAE(translate("SHAKER.PROCESSOR.VisualIfRule.ERROR.MISSING_OUTPUT_COLUMN_NAME", "Missing output column name"));
            }
        });

        params.visualIfDesc.elseActions.forEach((action) => {
            if (!("outputColumnName" in action) || !action.outputColumnName || action.outputColumnName==="") {
                throw new StepIAE(translate("SHAKER.PROCESSOR.VisualIfRule.ERROR.MISSING_OUTPUT_COLUMN_NAME", "Missing output column name"));
            }
        });
    }

    function fmtMeaningLabel(x){
        return meaningName($filter("meaningLabel")(x));
    }

    var svc = {};

    svc.map = {
    "CurrencySplitter": {
        description: function(type, params){
            if (!hasAll(params, ["inCol"])) return null;
            return translate("SHAKER.PROCESSOR.CurrencySplitter.SUMMARY", actionVerb("Split") + " {{column}} between currency and amount", { column: inColName(params.inCol)});
        },
        icon: 'dku-icon-scissors'
    },
    "ColumnSplitter": {
        description: function(type, params){
            if (!hasAll(params, ["inCol", "separator"])) return null;
            return translate("SHAKER.PROCESSOR.ColumnSplitter.SUMMARY", actionVerb("Split") + " {{column}} on {{separator}}", { column: inColName(params.inCol), separator: anumLiteral(params.separator)});
        },
        icon: 'dku-icon-scissors'
    },
    "FindReplace": {
        checkValid: appliesToCheckValid,
        description: function(type, params){
            if (!hasAll(params, ["matching", "normalization"])) return null;
            if (!params.mapping) {
                params.mapping = [];
            }
            const nbReplaces = Object.keys(params.mapping).length;
            if (nbReplaces === 1) {
                return translate("SHAKER.PROCESSOR.FindReplace.SUMMARY.SINGLE", actionVerb("Replace") + " {{key}} by {{value}} in {{columns}}",
                    { key: anumLiteral(params.mapping[0].from), value: anumLiteral(params.mapping[0].to), columns: appliesToDescription(params)}
                );
            } else {
                return translate("SHAKER.PROCESSOR.FindReplace.SUMMARY.MULTIPLE", actionVerb("Replace") + " {{count}} values in {{columns}}",
                    { rawCount: nbReplaces, count: numLiteral(nbReplaces), columns: appliesToDescription(params)});
            }
        },
        icon: 'dku-icon-edit-note',
    },
    "SwitchCase": {
        checkValid: switchCaseCheckValid,
        description: function(type, params) {
            if (!hasAll(params, ["normalization"])) return null;
            if (!params.mapping) {
                params.mapping = [];
            }
            const nbRules = Object.keys(params.mapping).length;
            return translate("SHAKER.PROCESSOR.SwitchCase.SUMMARY", actionVerb("Apply") + " {{count}} {{rawCount === 1 ? 'rule':'rules'}} from {{inCol}} to {{outCol}}",
                { rawCount: nbRules, count: numLiteral(nbRules), inCol: inColName(params.inputColumn), outCol: inColName(params.outputColumn)});
        },
        icon: 'dku-icon-brackets-curly',
    },
    "VisualIfRule" : {
        checkValid: visualIfRuleCheckValid,
        description: function(type, params) {
            if (params.visualIfDesc.ifThen.filter.uiData.conditions.length===0){
                return actionVerb("Create ") +  "if, then, else statements";
            }
            let outputColumnNames = params.visualIfDesc.ifThen.actions.map(action => action.outputColumnName);
            if (params.visualIfDesc.elseIfThens.length !== 0){
                for (let elseIfThen of params.visualIfDesc.elseIfThens) {
                    outputColumnNames = outputColumnNames.concat(elseIfThen.actions.map(action => action.outputColumnName));
                }
            }
            if (params.visualIfDesc.elseActions.length!==0){
                outputColumnNames = outputColumnNames.concat(params.visualIfDesc.elseActions.map(action => action.outputColumnName));
            }
            outputColumnNames = Array.from(new Set(outputColumnNames));
            if (outputColumnNames.length === 1) {
                return translate("SHAKER.PROCESSOR.VisualIfRule.SUMMARY.SINGLE_OUTPUT", actionVerb("Create") + " column {{column}} from if, then, else statements", {
                    column: outColName(outputColumnNames[0])
                });
            } else if (outputColumnNames.length < 5) {
                return translate("SHAKER.PROCESSOR.VisualIfRule.SUMMARY.UP_TO_FIVE_OUTPUTS", actionVerb("Create") + " columns {{columns}} from if, then, else statements", {
                    columns: outColName(outputColumnNames.join(', '))
                });
            } else {
                return translate("SHAKER.PROCESSOR.VisualIfRule.SUMMARY.MULTIPLE_OUTPUTS", actionVerb("Create") + " {{count}} columns from if, then, else statements", {
                    count: numLiteral(outputColumnNames.length)
                });
            }
        },
        icon: 'dku-icon-brackets-curly'
    },
    "MatchCounter": {
        description: function(type, params){
            if (!hasAll(params, ["inCol", "outCol", "pattern", "normalizationMode", "matchingMode"])) return null;
            return translate("SHAKER.PROCESSOR.MatchCounter.SUMMARY", actionVerb("Count") + " number of occurrences of {{pattern}} in {{column}}", {
                pattern: (params.matchMode === 2) ? ("/" + params.pattern + "/") : params.pattern,
                column: inColName(params.inCol)
            });
        },
        icon: 'dku-icon-search',
    },
    "MeaningTranslate": {
        checkValid: appliesToCheckValid,
        description: function(type, params){
            if (!hasAll(params, ["meaningId"])) return null;
            return translate("SHAKER.PROCESSOR.MeaningTranslate.SUMMARY", actionVerb("Replace") + " values in {{columns}} using meaning {{meaning}}", {
                columns: appliesToDescription(params),
                meaning: strongify(params.meaningId)
            });
        },
        icon: 'dku-icon-edit-note',
    },
    "ColumnReorder": {
        checkValid: appliesToCheckValid,
        description: function(type, params){
            const p = {
                columns: appliesToDescription(params),
                referenceColumn: inColName(isBlank(params.referenceColumn) ? translate('SHAKER.PROCESSOR.ColumnReorder.DESCRIPTION.MISSING', "'missing'") : params.referenceColumn)
            }
            switch (params.reorderAction) {
                case "AT_START":
                    return translate("SHAKER.PROCESSOR.ColumnReorder.DESCRIPTION.AT_START", actionVerb("Move") + " {{columns}} at beginning", p);
                case "AT_END":
                    return translate("SHAKER.PROCESSOR.ColumnReorder.DESCRIPTION.AT_END", actionVerb("Move") + " {{columns}} at end", p);
                case "BEFORE_COLUMN":
                    return translate("SHAKER.PROCESSOR.ColumnReorder.DESCRIPTION.BEFORE_COLUMN", actionVerb("Move") + " {{columns}} before {{referenceColumn}}", p);
                case "AFTER_COLUMN":
                    return translate("SHAKER.PROCESSOR.ColumnReorder.DESCRIPTION.AFTER_COLUMN", actionVerb("Move") + " {{columns}} after {{referenceColumn}}", p);
            }
        },
        icon: 'dku-icon-text-align-justified',
    },
    "ColumnRenamer": {
        checkValid: (params) => isColumnRenamerValid(params, {throwError: true}),
        description: function(type, params) {
            if (!isColumnRenamerValid(params, {throwError: false})) {
                return null;
            }
            if (params["renamings"].length === 1) {
                const inCol = params["renamings"][0].from;
                const outCol = params["renamings"][0].to;
                return translate('SHAKER.PROCESSOR.ColumnRenamer.SUMMARY.SINGLE', actionVerb("Rename") + " column '{{inCol}}' to '{{outCol}}'", {
                    inCol: inColName(inCol),
                    outCol: inColName(outCol)
                });
            } else {
                const columnCount = numLiteral(params["renamings"].length);
                return translate('SHAKER.PROCESSOR.ColumnRenamer.SUMMARY.MULTIPLE', actionVerb("Rename") + " {{columnCount}} columns", {
                    columnCount: columnCount
                });
            }
        },
        paramsMatcher: function(query, params) {
            query = query.toLowerCase();
            const { renamings } = params;
            // If the step is empty, it should not match anything
            if (!renamings) return false;
            return renamings.some(({from, to}) => (from && from.toLowerCase().includes(query)) || (to && to.toLowerCase().includes(query)));
        },
        icon: 'dku-icon-edit-note'
    },

    "TextSimplifierProcessor": {
        description: function(type, params){
            if (!hasAll(params, ["inCol"])) return null;
            return translate('SHAKER.PROCESSOR.TextSimplifierProcessor.SUMMARY', '<span class="action-verb">Simplify</span> text in {{column}}', {
                column:inColName(params.inCol)
            });
        },
        icon: 'dku-icon-edit-note',
    },
    "Tokenizer": {
        description: function(type, params){
            if (!hasAll(params, ["inCol"])) return null;
            const inCol = inColName(params["inCol"]);
            if (params.operation === "TO_JSON") {
                return translate('SHAKER.PROCESSOR.Tokenizer.SUMMARY.TO_JSON', actionVerb("Tokenize") + " column {{column}} into JSON", { column: inCol });
            } else if (params.operation === "FOLD") {
                return translate('SHAKER.PROCESSOR.Tokenizer.SUMMARY.FOLD', actionVerb("Tokenize") + " column {{column}} and fold tokens (one per row)", { column: inCol });
            } else if (params.operation === "SPLIT") {
                return translate('SHAKER.PROCESSOR.Tokenizer.SUMMARY.SPLIT', actionVerb("Tokenize") + " column {{column}} and split tokens (one per column)", { column: inCol });
            }
        },
        icon: 'dku-icon-scissors',
    },
    "SplitIntoChunks": {
        description: function(type, params){
            if (!hasAll(params, ["inCol"])) return null;
            const inCol = inColName(params["inCol"]);
            return translate('SHAKER.PROCESSOR.SplitIntoChunks.SUMMARY', actionVerb("Split") + " column {{column}} into chunks", { column: inCol });
        },
        icon: 'dku-icon-scissors-horizontal',
    },
    "NGramExtract": {
        description: function(type, params){
            if (!hasAll(params, ["inCol"])) return null;
            const inCol = inColName(params["inCol"]);
            const op = params["operation"];
            if (op === "TO_JSON") {
                return translate('SHAKER.PROCESSOR.NGramExtract.SUMMARY.TO_JSON', actionVerb("Extract ngrams") + " from column {{column}} into JSON", { column: inCol });
            } else if (op === "FOLD") {
                return translate('SHAKER.PROCESSOR.NGramExtract.SUMMARY.FOLD', actionVerb("Extract ngrams") + " from column {{column}} and fold them (one per row)", { column: inCol });
            } else if (op === "SPLIT") {
                return translate('SHAKER.PROCESSOR.NGramExtract.SUMMARY.SPLIT', actionVerb("Extract ngrams") + " from column {{column}} and split them (one per column)", { column: inCol });
            }
        },
        icon: 'dku-icon-beaker',
    },
    "ExtractNumbers": {
        description: function(type, params){
            if (!hasAll(params, ["input"])) return null;
            const input = inColName(params.input);
            if (toBoolean(params["multipleValues"])) {
                return translate('SHAKER.PROCESSOR.ExtractNumbers.SUMMARY.MULTIPLE', actionVerb("Extract numbers") + " from {{column}}", { column: input });
            } else {
                return translate('SHAKER.PROCESSOR.ExtractNumbers.SUMMARY.SINGLE',  actionVerb("Extract a number") + " from {{column}}", { column: input });
            }
        },
        icon: 'dku-icon-beaker',
    },
    "NumericalFormatConverter": {
        checkValid: appliesToCheckValid,
        description: function(type, params){
            if (!hasAll(params, ["inFormat", "outFormat"])) return null;
            const formats = {
                RAW: translate('SHAKER.PROCESSOR.NumericalFormatConverter.SUMMARY.RAW_FORMAT', "Raw"),
                EN: translate('GLOBAL.LANGUAGE.ENGLISH', "English"),
                FR: translate('GLOBAL.LANGUAGE.FRENCH', "French"),
                IT: translate('GLOBAL.LANGUAGE.ITALIAN', "Italian"),
                GE: translate('GLOBAL.LANGUAGE.GERMAN', "German")
            };
            return translate('SHAKER.PROCESSOR.NumericalFormatConverter.SUMMARY',
                actionVerb("Convert number formats") + " from {{inFormat}} to {{outFormat}} in {{columns}}",
                { inFormat: formats[params['inFormat']], outFormat: formats[params['outFormat']], columns: appliesToDescription(params) });
        },
        icon: 'dku-icon-text-superscript',
    },
    "TypeSetter": {
        description: function(type, params){
            if (!hasAll(params, ["column", "type"])) return null;
            const column = inColName(params["column"]);
            const typeloc = sanitize(params["type"]);
            const typestr = meaningName(typeloc);
            return translate('SHAKER.PROCESSOR.TypeSetter.SUMMARY', "Set meaning of {{column}} to {{meaning}}", { column: column, meaning: typestr });
        },
        icon: 'dku-icon-edit-note',
        postLinkFn : function(scope, element) {
            if (angular.isUndefined(scope.step.params.type)) {
                scope.step.params.type = "Text";
            }
        }
    },
    "StringTransformer" : {
        checkValid : function(params) {
            appliesToCheckValid(params);
            if (!params.mode) {
                throw new StepIAE(translate('SHAKER.PROCESSOR.StringTransformer.ERROR.MODE_NOT_SPECIFIED', "Mode not specified"));
            }
        },
        description : function(type, params) {
            if (!hasAll(params, ["mode"])) return null;
            const mode = sanitize(params["mode"]).toLowerCase();
            return translate('SHAKER.PROCESSOR.StringTransformer.SUMMARY', 'Perform <span class="action-verb">{{mode}}</span> on {{columns}}',
                { mode: mode, columns: appliesToDescription(params) });
        },
        postLinkFn : function(scope, element) {
            if (angular.isUndefined(scope.step.params.mode)) {
                scope.step.params.mode = "TO_LOWER";
            }
        },
        icon: 'dku-icon-edit-note'
    },
    "ColumnsConcat" : {
        description: function(type, params) {
            if (!hasAll(params, ["outputColumn"])) return null;
            return translate('SHAKER.PROCESSOR.ColumnsConcat.SUMMARY', actionVerb("Concatenate") + " columns in {{outCol}}", { outCol: outColName(params.outputColumn) });
        },
        icon: 'dku-icon-arrow-expand-exit'
    },
    "ArrayExtractProcessor" : {
        checkValid: (params) => defaultIsValidIfHasColumn(params, "input", {throwError: true}),
        description: function(type, params) {
            if (!defaultIsValidIfHasColumn(params, "input", {throwError: false})) {
                return null;
            }
            if (params.mode === "INDEX") {
                return translate('SHAKER.PROCESSOR.ArrayExtractProcessor.SUMMARY.INDEX', actionVerb("Extract") + " elt {{index}} from {{column}}",
                    { index: numLiteral(params.index), column: inColName(params.input) });
            } else {
                return translate('SHAKER.PROCESSOR.ArrayExtractProcessor.RANGE', actionVerb("Extract") + " elts {{begin}}-{{end}} from {{column}}",
                    { begin: numLiteral(params.begin), end: numLiteral(params.end), column: inColName(params.input) });
            }
        },
        icon: 'dku-icon-arrow-external-link'
    },
    "ArraySortProcessor" : {
        checkValid: (params) => defaultIsValidIfHasColumn(params, "input", {throwError: true}),
        description: function(type, params) {
            if (!defaultIsValidIfHasColumn(params, "input", {throwError: false})) {
                return null;
            }
            return translate('SHAKER.PROCESSOR.ArraySortProcessor.SUMMARY', actionVerb("Sort") + " array in {{column}}", { column: inColName(params.input) });
        },
        icon: 'dku-icon-sort-ascending'
    },
    "ColumnsSelector": {
        checkValid: appliesToCheckValid,
        description: function(type, params) {
            return toBoolean(params["keep"]) ?
                translate('SHAKER.PROCESSOR.ColumnsSelector.FORM.KEEP', '<span class="action-verb">Keep only</span> {{columns}}', { columns: appliesToDescription(params) }) :
                translate('SHAKER.PROCESSOR.ColumnsSelector.FORM.REMOVE', '<span class="action-verb">Remove</span> {{columns}}', { columns: appliesToDescription(params) });
        },
        icon: 'dku-icon-filter',
    },
    "FilterOnValue": {
        checkValid: appliesToCheckValid,
        description: function(type, params){
            if (!hasAll(params, ["matchingMode", "values"]) || params["values"].length === 0) return null;
            return filterAndFlagSummary(params, filterAndFlagOnValueCondition);
        },
        impactVerb : filterAndFlagImpactVerb,
        icon: 'dku-icon-trash',
    },
    "FlagOnValue": {
        checkValid : function(params) {
            appliesToCheckValid(params);
            flagCheckValid(params);
        },
        description: function(type, params){
            if (!hasAll(params, ["matchingMode", "values", "flagColumn"]) || params["values"].length === 0) return null;
            return filterAndFlagSummary(params, filterAndFlagOnValueCondition);
        },
        impactVerb : filterAndFlagImpactVerb,
        icon: 'dku-icon-flag',
    },
    "FilterOnNumericalRange": {
        checkValid: function(params) {
            appliesToCheckValid(params);
            if (params["min"] === undefined && params["max"] === undefined) {
                throw new StepIAE(translate('SHAKER.PROCESSOR.FilterAndFlagOnNumericalRange.ERROR.BOUNDS_NOT_DEFINED', "Bounds are not defined"));
            }
        },
        description: function(type, params) {
            return filterAndFlagSummary(params, filterAndFlagOnNumericRangeCondition);
        },
        impactVerb: filterAndFlagImpactVerb,
        icon: 'dku-icon-trash',
    },
    "FlagOnNumericalRange": {
        checkValid: function(params) {
            appliesToCheckValid(params)
            flagCheckValid(params);
            if (params["min"] === undefined && params["max"] === undefined) {
                throw new StepIAE(translate('SHAKER.PROCESSOR.FilterAndFlagOnNumericalRange.ERROR.BOUNDS_NOT_DEFINED', "Bounds are not defined"));
            }
        },
        description: function(type, params) {
            return filterAndFlagSummary(params, filterAndFlagOnNumericRangeCondition);
        },
        impactVerb : filterAndFlagImpactVerb,
        icon: 'dku-icon-flag',
    },
    "FilterOnDate": {
        checkValid: appliesToCheckValid,
        description: filterAndFlagOnDateSummary,
        impactVerb: filterAndFlagImpactVerb,
        icon: 'dku-icon-trash',
    },
    "FlagOnDate": {
        checkValid: function(params) {
            appliesToCheckValid(params);
            flagCheckValid(params);
        },
        description: filterAndFlagOnDateSummary,
        impactVerb: filterAndFlagImpactVerb,
        icon: 'dku-icon-flag',
    },
    "FilterOnDateRange": {
        checkValid : function(params) {
            appliesToCheckValid(params);
            if (params["max"]) checkDate(params["max"], "upper-bound date");
            if (params["min"]) checkDate(params["min"], "lower-bound date");
            if (!(params["max"] || params["min"])) {
                throw new StepIAE("Input at least one date bound.");
            }
        },
        description: filterAndFlagOnDateRangeSummary,
        impactVerb : filterAndFlagImpactVerb,
        icon: 'dku-icon-trash',
    },
    "FlagOnDateRange": {
        checkValid : function(params) {
            appliesToCheckValid(params);
            flagCheckValid(params);
            if (!(params["max"] || params["min"])) {
                throw new StepIAE("Input at least one date bound.");
            }
            if (params["max"]) checkDate(params["max"], "upper-bound date");
            if (params["min"]) checkDate(params["min"], "lower-bound date");
        },
        description: filterAndFlagOnDateRangeSummary,
        impactVerb : filterAndFlagImpactVerb,
        icon: 'dku-icon-flag',
    },
    "FilterOnCustomFormula" :{
        checkValid : function(params) {
            if (isBlank(params.expression)) throw new StepIAE(translate('SHAKER.PROCESSOR.FilterAndFlagOnCustomFormula.ERROR.EXPRESSION_NOT_SPECIFIED', "Expression not specified"));
            if (params.action === "CLEAR_CELL" || params.action === "DONTCLEAR_CELL") {
                if (isBlank(params.clearColumn)) {
                    throw new StepIAE(translate('SHAKER.PROCESSOR.FilterAndFlagOnCustomFormula.ERROR.COLUMN_TO_CLEAR_NOT_SPECIFIED', "Column to clear not specified"))
                }
            }
        },
        description : function(type, params) {
            const conditionFunction = function(params) {
                return params.expression.length > 20 ? translate('SHAKER.PROCESSOR.FilterAndFlagOnCustomFormula.SUMMARY.FORMULA_IS_TRUE', "formula is true") : anumLiteral(params.expression)
            };
            return filterAndFlagSummary(params, conditionFunction, params.clearColumn);
        },
        shouldExpandFormula: function (params) {
            return params.expression.length > 20;
        },
        impactVerb : filterAndFlagImpactVerb,
        icon : "dku-icon-trash"
    },
    "FlagOnCustomFormula" : {
        checkValid : function(params) {
            if (isBlank(params.expression)) throw new StepIAE(translate('SHAKER.PROCESSOR.FilterAndFlagOnCustomFormula.ERROR.EXPRESSION_NOT_SPECIFIED', "Expression not specified"));
            if (isBlank(params.flagColumn)) {
                throw new StepIAE(translate('SHAKER.PROCESSOR.FilterAndFlagOnCustomFormula.ERROR.COLUMN_TO_FLAG_NOT_SPECIFIED', "Column to flag not specified"))
            }
        },
        description : function(type, params) {
            const conditionFunction = function(params) {
                return params.expression.length > 20 ? translate('SHAKER.PROCESSOR.FilterAndFlagOnCustomFormula.SUMMARY.FORMULA_IS_TRUE', "formula is true") : anumLiteral(params.expression)
            };
            return filterAndFlagSummary(params, conditionFunction, params.clearColumn);
        },
        shouldExpandFormula: function(params) {
            return params.expression.length > 20;
        },
        impactVerb : filterAndFlagImpactVerb,
        icon : "dku-icon-flag"
    },

    /* Cleansing */
    "FilterOnBadType": {
        checkValid: appliesToCheckValid,
        description: function(type, params){
            if (!hasAll(params, ["type"])) return null;
            const conditionFunction = function(params, columns, multipleColumns) {
                return translate('SHAKER.PROCESSOR.FilterAndFlagOnBadType.SUMMARY.CONDITION', "{{columns}} {{multipleColumns?'are':'is'}} not a valid {{meaning}}",
                    {columns: columns, multipleColumns:multipleColumns, meaning:fmtMeaningLabel(params["type"])});
            };
            return filterAndFlagSummary(params, conditionFunction);
        },
        icon: 'dku-icon-trash',
        impactVerb : filterAndFlagImpactVerb,
        postLinkFn : function(scope, element) {
            if (angular.isUndefined(scope.step.params.type)) {
                scope.step.params.type = "Text";
            }
        }
    },
    "FlagOnBadType": {
        checkValid : function(params){
            appliesToCheckValid(params);
            flagCheckValid(params);
        },
        description: function(type, params){
            if (!hasAll(params, ["type", "flagColumn"])) return null;
            const conditionFunction = function(params, columns, multipleColumns) {
                return translate('SHAKER.PROCESSOR.FilterAndFlagOnBadType.SUMMARY.CONDITION', "{{columns}} {{multipleColumns?'are':'is'}} not a valid {{meaning}}",
                    {columns: columns, multipleColumns:multipleColumns, meaning:fmtMeaningLabel(params["type"])});
            };
            return filterAndFlagSummary(params, conditionFunction);
        },
        icon: 'dku-icon-flag',
        impactVerb : filterAndFlagImpactVerb,
        postLinkFn : function(scope, element) {
            if (angular.isUndefined(scope.step.params.type)) {
                scope.step.params.type = "Text";
            }
        }
    },
    "RemoveRowsOnEmpty": {
        checkValid: appliesToCheckValid,
        description: function(type, params){
            const applyToAllOrPattern = params["appliesTo"] === "ALL" || params["appliesTo"] === "PATTERN";
            const p = { columns: appliesToDescription(params), hasMultipleColumns: (params["appliesTo"] != "SINGLE_COLUMN" && params["columns"] && params["columns"].length > 1) || applyToAllOrPattern };
            if (params["keep"]) {
                return translate('SHAKER.PROCESSOR.RemoveRowsOnEmpty.SUMMARY.KEEP', actionVerb("Keep") + " rows with empty values in {{hasMultipleColumns ? 'at least one of ' : '' }}{{columns}}", p);
            } else {
                return translate('SHAKER.PROCESSOR.RemoveRowsOnEmpty.SUMMARY.REMOVE', actionVerb("Remove") + " rows with empty values in {{hasMultipleColumns ? 'at least one of ' : '' }}{{columns}}", p);
            }
        },
        impactVerb : Fn.cst("deleted"),
        icon: 'dku-icon-trash',
    },
    "FillEmptyWithValue": {
        checkValid: appliesToCheckValid,
        description: function(type, params){
            if (!hasAll(params, ["value"])) return null;
            return translate('SHAKER.PROCESSOR.FillEmptyWithValue.SUMMARY', actionVerb("Fill empty") +" cells of {{columns}} with '{{value}}'", {
                columns: appliesToDescription(params),
                value: anumLiteral(truncate(params["value"], 60))
            });
        },
        impactVerb : Fn.cst("filled"),
        icon: 'dku-icon-circle-fill',
    },
    "FillEmptyWithComputedValue": {
        checkValid: appliesToCheckValid,
        description: function(type, params){
            if (!hasAll(params, ["mode"])) return null;
            return translate('SHAKER.PROCESSOR.FillEmptyWithComputedValue.SUMMARY', actionVerb("Impute missing values") + " of {{columns}} with {{value}}", {
                columns: appliesToDescription(params),
                value: translate('SHAKER.PROCESSOR.FillEmptyWithComputedValue.SUMMARY.' + params["mode"], strongify(params["mode"]).toLowerCase())
            });
        },
        impactVerb : Fn.cst("filled"),
        icon: 'dku-icon-circle-fill',
    },
    "SplitInvalidCells" : {
        description : function(type, params) {
            if (!hasAll(params, ["column"])) return null;
            return translate('SHAKER.PROCESSOR.SplitInvalidCells.SUMMARY', actionVerb("Move invalid") + " cells of {{column}} to {{invalidColumn}}", {
                column: inColName(params["column"]),
                invalidColumn: outColName(params["invalidColumn"])
            });
        },
        impactVerb : Fn.cst("splitted"),
        icon : 'dku-icon-scissors'
    },
    "UpDownFiller" : {
        description : function(type, params) {
            if (!params.columns || params.columns.length < 1) return null;
            return translate('SHAKER.PROCESSOR.UpDownFiller.SUMMARY', actionVerb("Fill empty") +" cells of {{columns}} with {{up?'next':'previous'}} value", {
                columns: inColName(params["columns"]),
                up: toBoolean(params["up"])
            });
        },
        impactVerb : Fn.cst("filled"),
        icon : 'dku-icon-circle-fill'
    },
    "LongTailGrouper" : {
        description : function(type, params) {
            if (!hasAll(params, ["column"])) return null;
            return translate('SHAKER.PROCESSOR.LongTailGrouper.SUMMARY', actionVerb("Merge") + " long-tail values in {{column}}", {
                column: inColName(params["column"]),
            });
        },
        impactVerb : Fn.cst("merged"),
        icon: 'dku-icon-arrow-expand-exit'
    },
    "ComputeNTile" : {
        checkValid: appliesToCheckValid,
        description: function(type, params){
            if (!hasAll(params, ["n"])) return null;
            return translate('SHAKER.PROCESSOR.ComputeNTile.SUMMARY', actionVerb("Compute quantile") +" of {{columns}} with {{binCount}} bins", {
                columns: appliesToDescription(params),
                binCount: strongify(params["n"])
            });
        },
        impactVerb : Fn.cst("filled"),
        icon: 'dku-icon-circle-fill'
    },
    "MergeLongTailValues" : {
        checkValid: appliesToCheckValid,
        description: function(type, params){
            if (!hasAll(params, ["thresholdMode", "replacementValue"])) return null;
            return translate('SHAKER.PROCESSOR.MergeLongTailValues.SUMMARY', actionVerb("Replace long tail") + " of {{columns}} with {{value}}", {
                columns: appliesToDescription(params),
                value: strongify(params["replacementValue"])
            });
        },
        impactVerb : Fn.cst("filled"),
        icon: 'dku-icon-circle-fill'
    },

    /* Types */
    "QueryStringSplitter": {
        description: function(type, params){
            if (!hasAll(params, ["column"])) return null;
            return translate('SHAKER.PROCESSOR.QueryStringSplitter.SUMMARY', actionVerb("Split") + " HTTP Query string in {{column}}", {
                column: inColName(params["column"])
            });
        },
        icon: 'dku-icon-scissors',
    },
    "URLSplitter": {
        description: function(type, params){
            if (!hasAll(params, ["column"])) return null;
            return translate('SHAKER.PROCESSOR.URLSplitter.SUMMARY', actionVerb("Split") + " URL in {{column}}", {
                column: inColName(params["column"])
            });
        },
        icon: 'dku-icon-scissors',
    },
    "EmailSplitter": {
        description: function(type, params){
            if (!hasAll(params, ["column"])) return null;
            return translate('SHAKER.PROCESSOR.EmailSplitter.SUMMARY', actionVerb("Split") + " email in {{column}}", {
                column: inColName(params["column"])
            });
        },
        icon: 'dku-icon-scissors',
    },
    "VisitorIdGenerator" : {
        description : function(type, params) {
            if (!hasAll(params, ["outputColumn"])) return null;
            return translate('SHAKER.PROCESSOR.VisitorIdGenerator.SUMMARY', actionVerb("Generate") +" visitor-id in {{column}}", {
                column: outColName(params["outputColumn"])
            });
        },
        icon : 'dku-icon-beaker'
    },
    "RegexpExtractor": {
        description: function(type, params){
            if (!hasAll(params, ["column","pattern"])) return null;
            return translate("SHAKER.PROCESSOR.RegexpExtractor.SUMMARY", actionVerb("Extract") +" from {{column}} with {{pattern}}", {
                column: inColName(params.column),
                pattern: anumLiteral(params.pattern)
            });
        },
        icon: 'dku-icon-beaker',
    },
    "UserAgentClassifier": {
        description: function(type, params){
            if (!hasAll(params, ["column"])) return null;
            return translate("SHAKER.PROCESSOR.UserAgentClassifier.SUMMARY", actionVerb("Classify") +" User-Agent in {{column}}", {
                column: inColName(params.column)
            });
        },
        icon: 'dku-icon-laptop',
    },
    "UseRowAsHeader": {
        description: function(type, params){
            if (!hasAll(params, ["rowIdx"])) return null;
            return translate("SHAKER.PROCESSOR.UseRowAsHeader.SUMMARY", "Use row {{index}}'s values as column names", {
                index: strongify(sanitize(params["rowIdx"]))
            });
        },
        icon: 'dku-icon-list-numbered',
    },

    /* Basic */
    "JSONPathExtractor" : {
        description: function(type, params) {
            if (!hasAll(params, ["inCol"])) return null;
            return translate("SHAKER.PROCESSOR.JSONPathExtractor.SUMMARY", actionVerb("Extract") + " from {{column}} with JSONPath {{expression}}", {
                column: inColName(params.inCol),
                expression: anumLiteral(truncate(params.expression, 60))
            });
        },
        icon: 'dku-icon-beaker',
    },

    /* Time */
    "DateParser": {
        checkValid: appliesToCheckValid,
        description: function(type, params){
            if (!hasAll(params, ["formats"])) return null;
            return translate('SHAKER.PROCESSOR.DateParser.SUMMARY', actionVerb("Parse date") + " in {{columns}}", {columns: appliesToDescription(params)});
        },
        icon: 'dku-icon-calendar',
    },
    "DateIncrement": {
        checkValid : function(params) {
            if (params["incrementBy"] !== 'STATIC' && params["valueCol"] === undefined) {
                throw new StepIAE("Value column is not defined");
            }
        },
        description: function (type, params) {
            if (!hasAll(params, ["inCol"])) return null;
            function datePartsLabelSingular(part) {
                return translate("SHAKER.PROCESSOR.DateIncrement.PART." + part, part.toLowerCase());
            }
            function datePartsLabelPlural(part) {
                const translationID = "SHAKER.PROCESSOR.DateIncrement.SUMMARY.PART." + part;
                return translate(translationID + ".PLURAL", datePartsLabelSingular(part) + "s");
            }
            function datePartsLabel(part, count) {
                return count > 1 ? datePartsLabelPlural(part) : datePartsLabelSingular(part);
            }
            const p = {
                column: inColName(params["inCol"]),
            };
            if (params["incrementBy"] === 'STATIC') {
                p.increment = params["increment"] > -1;
                p.count = Math.abs(params["increment"]);
                p.part = datePartsLabel(params["datePart"], Math.abs(params["increment"]));
            } else if (params["valueCol"] === undefined) {
                return null;
            } else {
                p.increment = true;
                p.count = inColName(params["valueCol"]);
                p.part = datePartsLabelPlural(params["datePart"]);
            }
            return translate("SHAKER.PROCESSOR.DateIncrement.SUMMARY", "<span class=\"action-verb\">{{increment?'Increment':'Decrement'}} date</span> in {{column}} by {{count}} {{part}}", p);
        },
        icon: "dku-icon-calendar",
    },
    "DateTruncate": {
        description: function (type, params) {
            if (!hasAll(params, ["inCol", "datePart"])) return null;
            return translate("SHAKER.PROCESSOR.DateTruncate.SUMMARY", actionVerb("Truncate") + " {{column}} on {{part}}", {
                column: inColName(params["inCol"]),
                part: anumLiteral(translate('SHAKER.PROCESSOR.DateTruncate.PART.' + params["datePart"], params["datePart"].toLowerCase()))
            });
        },
        icon: "dku-icon-calendar",
    },
    "DateFormatter": {
        description: function(type, params){
            if (!hasAll(params, ["inCol"])) return null;
            return translate("SHAKER.PROCESSOR.DateFormatter.SUMMARY", actionVerb("Format date") + " in {{column}}", {
                column: inColName(params["inCol"])
            });
        },
        icon: 'dku-icon-calendar',
    },

    "DateDifference": {
        checkValid : function(params) {
            if(params.compareTo === "DATE"){
                if(!params.refDate){
                    throw new StepIAE(translate("SHAKER.PROCESSOR.DateDifference.ERROR.MISSING_REF_DATE", "No date specified"));
                } else {
                     checkDate(params.refDate)
                }
            } else if (params.compareTo === "COLUMN") {
                if (!params.input2) {
                    throw new StepIAE(translate("SHAKER.PROCESSOR.DateDifference.ERROR.MISSING_INPUT2", "No column specified"))
                }
            }
             if (params.excludeHolidays === true || params.excludeWeekends === true) {
                 isHolidaysComputerValid(params, {throwError: true});
             }
         },
        description: function(type, params){
            if (!hasAll(params, ["input1", "output", "outputUnit", "compareTo"])) return null;
            const in1 = inColName(params["input1"]);
            const in2 = inColName(params["input2"]);
            if (params.compareTo === "NOW") {
                return translate("SHAKER.PROCESSOR.DateDifference.SUMMARY.NOW", actionVerb("Compute time difference") + " between {{date}} and now", {
                    date: in1
                });
            } else if (params.compareTo === "COLUMN") {
                return translate("SHAKER.PROCESSOR.DateDifference.SUMMARY.BETWEEN", actionVerb("Compute time difference") + " between {{date1}} and {{date2}}", {
                    date1: in1,
                    date2: in2
                });
            } else if (params.compareTo === "DATE") {
                return translate("SHAKER.PROCESSOR.DateDifference.SUMMARY.REFERENCE", actionVerb("Compute time difference") + " between {{date}} and a reference", {
                    date: in1
                });
            }
        },
        icon: 'dku-icon-calendar',
    },
    "UNIXTimestampParser": {
        description : function(type, params) {
            if (!hasAll(params, ["inCol"])) return null;
            return translate("SHAKER.PROCESSOR.UNIXTimestampParser.SUMMARY", actionVerb("Convert UNIX") + " timestamp in {{column}}", {
                column: inColName(params["inCol"])
            });
        },
        icon: 'dku-icon-calendar'
    },
    "JSONFlattener": {
        description : function(type, params) {
            if (!hasAll(params, ["inCol"])) return null;
            return translate("SHAKER.PROCESSOR.JSONFlattener.SUMMARY", actionVerb("Unnest") + " object in {{column}}", {
                column: inColName(params["inCol"])
            });
        },
        icon: 'dku-icon-scissors'
    },
    "DateComponentsExtractor": {
        checkValid: (params) => isDateComponentsExtractorValid(params, {throwError: true}),
        description: function(type, params) {
            if (!isDateComponentsExtractorValid(params, {throwError: false})) {
                return null;
            }
            return translate("SHAKER.PROCESSOR.DateComponentsExtractor.SUMMARY", actionVerb("Extract date") + " components from {{column}}", {
                column: inColName(params["column"])
            });
        },
        icon: 'dku-icon-calendar'
    },
    "HolidaysComputer": {
        checkValid: (params) => isHolidaysComputerValid(params, {throwError: true}),
        description: function(type, params) {
            if (!isHolidaysComputerValid(params, {throwError: false})) {
                return null;
            }
            if (params["calendar_id"] !== "extract_from_column") {
                return translate("SHAKER.PROCESSOR.HolidaysComputer.SUMMARY.CALENDAR_ID", actionVerb("Extract holidays") + " from {{column}} using calendar {{calendar}}", {
                    column: inColName(params["inCol"]),
                    calendar: anumLiteral(params["calendar_id"])
                });
            } else {
                return translate("SHAKER.PROCESSOR.HolidaysComputer.SUMMARY.CALENDAR_SRC", actionVerb("Extract holidays") + " from {{column}} using the country in {{calendar}}", {
                    column: inColName(params["inCol"]),
                    calendar: anumLiteral(params["calendar_src"])
                });
            }
        },
        icon: 'dku-icon-calendar'
    },

    /* Numbers */
    "BinnerProcessor": {
        checkValid: (params) => defaultIsValidIfHasColumn(params, "input", {throwError: true}),
        description: function(type, params) {
            if (!defaultIsValidIfHasColumn(params, "input", {throwError: false})) {
                return null;
            }
            return translate("SHAKER.PROCESSOR.BinnerProcessor.SUMMARY.CALENDAR_SRC", actionVerb("Bin") + " values in {{column}}", {
                column: inColName(params["input"])
            });
        },
        icon: 'dku-icon-scissors',
    },
    "NumericalCombinator": {
        checkValid : function(params) {
            appliesToCheckValid(params);
            const allOps = ["add", "sub", "mul", "div"];
            let hasOp = false;
            for (let opidx in allOps) {
                if (toBoolean(params[allOps[opidx]])) {
                    hasOp = true;
                }
            }
            if (!hasOp) {
                throw new StepIAE(translate("SHAKER.PROCESSOR.NumericalCombinator.ERROR.MISSING_OPERATION", "No selected operation"));
            }
        },
        description: function(type, params){
            const ops = [],
                allOps = ["add", "sub", "mul", "div"],
                opDesc = ["+", "-", "×", "÷"];
            for (let opidx in allOps) {
                if (toBoolean(params[allOps[opidx]])) {
                    ops.push(opDesc[opidx]);
                }
            }
            if (!ops.length) { return null; }
            const p = {
                columns: appliesToDescription(params),
                opsCount: ops.length,
                ops: anumLiteral(ops.join(", ")),
            }
            if (hasNonEmptyParam(params, "prefix")) {
                p.prefix = anumLiteral(params.prefix);
                return translate("SHAKER.PROCESSOR.NumericalCombinator.SUMMARY.PREFIX", actionVerb("Combine values") +" in {{columns}} with {{opsCount>1?'operations':'operation'}} {{ops}} into {{prefix}}*", p);
            } else {
                return translate("SHAKER.PROCESSOR.NumericalCombinator.SUMMARY", actionVerb("Combine values") +" in {{columns}} with {{opsCount>1?'operations':'operation'}} {{ops}}", p);
            }
        },
        icon: 'dku-icon-plus',
    },
    "RoundProcessor": {
        checkValid: appliesToCheckValid,
        description: function(type, params){
            return translate("SHAKER.PROCESSOR.RoundProcessor.SUMMARY", actionVerb("Round") + " values in {{columns}}", { columns: appliesToDescription(params) });
        },
        icon: 'dku-icon-text-superscript',
    },
    "MinMaxProcessor": {
        description: function(type, params){
            if (!hasAll(params, ["columns"])) return null;
            const min = parseFloat(params["lowerBound"]);
            const max = parseFloat(params["upperBound"]);
            const hasMin = !isNaN(min);
            const hasMax = !isNaN(max);
            if ((!hasMin && !hasMax) || min > max) return null;
            const p = {
                clear: params['clear'],
                lowerBound: numLiteral(params["lowerBound"]),
                upperBound: numLiteral(params["upperBound"]),
                column: inColName(params["columns"])
            }
            if (hasMin && hasMax) {
                return translate("SHAKER.PROCESSOR.MinMaxProcessor.SUMMARY.MIN_MAX",
                    "<span class=\"action-verb\">{{clear?'Clear':'Clip'}}</span> values outside [{{lowerBound}},{{upperBound}}] in {{column}}", p);
            } else if (hasMin) {
                return translate("SHAKER.PROCESSOR.MinMaxProcessor.SUMMARY.MIN",
                    "<span class=\"action-verb\">{{clear?'Clear':'Clip'}}</span> values &lt;&nbsp;{{lowerBound}} in {{column}}", p);
            } else {
                return translate("SHAKER.PROCESSOR.MinMaxProcessor.SUMMARY.MAX",
                    "<span class=\"action-verb\">{{clear?'Clear':'Clip'}}</span> values &gt;&nbsp;{{upperBound}} in {{column}}", p);
            }
        },
        icon: 'dku-icon-beaker',
    },
    "MeanProcessor": {
        checkValid: appliesToCheckValid,
        description: function(type, params){
            if (!hasAll(params, ["columns"])) return null;
            return translate("SHAKER.PROCESSOR.MeanProcessor.SUMMARY", actionVerb('Compute') + " mean of {{columns}}", {
                columns: appliesToDescription(params)
            });
        },
        icon: 'dku-icon-text-superscript',
    },
    "CurrencyConverterProcessor": {
        checkValid: (params) => isCurrencyConverterProcessorValid(params, {throwError: true}),
        description: function(type, params) {
            if (!isCurrencyConverterProcessorValid(params, {throwError: false})) {
                return null;
            }
            return translate("SHAKER.PROCESSOR.CurrencyConverterProcessor.SUMMARY", actionVerb("Convert") + " {{column}} to {{currency}}", {
                column: inColName(params["inputColumn"]),
                currency: anumLiteral(params["outputCurrency"])
            });
        },
        icon: 'dku-icon-edit-note'
    },

    /* Geo */
    "GeoIPResolver": {
        description: function(type, params){
            if (!hasAll(params, ["inCol"])) return null;
            return translate("SHAKER.PROCESSOR.GeoIPResolver.SUMMARY", actionVerb("Geo-locate IP") + " in {{column}}", {
                column: inColName(params["inCol"])
            });
        },
        icon: 'dku-icon-globe',
    },
    "GeoPointExtractor": {
        description: function(type, params){
            if (!hasAll(params, ["column"])) return null;
            return translate("SHAKER.PROCESSOR.GeoPointExtractor.SUMMARY", actionVerb("Extract") + " latitude/longitude from {{column}}", {
                column: inColName(params["column"])
            });
        },
        icon: 'dku-icon-globe',
    },
    "GeoPointCreator": {
        description: function(type, params){
            if (!hasAll(params, ["out_column","lat_column","lon_column"])) return null;
            return translate("SHAKER.PROCESSOR.GeoPointCreator.SUMMARY", actionVerb("Create GeoPoint") + " from {{latitude}} & {{longitude}}", {
                latitude: inColName(params["lat_column"]),
                longitude: inColName(params["lon_column"])
            });
        },
        icon: 'dku-icon-globe',
    },

    "NearestNeighbourGeoJoiner" : {
        description : function(type, params) {
            return translate("SHAKER.PROCESSOR.NearestNeighbourGeoJoiner.SUMMARY", actionVerb("Geo-join") + " from dataset {{rightInput}}", {
                rightInput: inColName(params["rightInput"])
            });
        },
        icon : 'dku-icon-globe'
    },
    "Geocoder" : {
        description : function(type, params) {
            if (!hasAll(params, ["inCol","api", "prefixOutCol", "apiKey"])) return null;
            return translate("SHAKER.PROCESSOR.Geocoder.SUMMARY", actionVerb("Geocode") + " from address {{column}}", {
                column: inColName(params["inCol"])
            });
        },
        icon : 'dku-icon-globe'
    },
    "CityLevelReverseGeocoder" : {
        description : function(type, params) {
            if (!hasAll(params, ["inputCol"])) return null;
            return translate("SHAKER.PROCESSOR.CityLevelReverseGeocoder.SUMMARY", actionVerb("Reverse-geocode") + " location {{column}}", {
                column: inColName(params["inputCol"])
            });
        },
         icon : 'dku-icon-globe'
    },
    "ChangeCRSProcessor" : {
        description : function(type, params) {
            if (!hasAll(params, ["geomCol"])) return null;
            return translate("SHAKER.PROCESSOR.ChangeCRSProcessor.SUMMARY", actionVerb("Change CRS") + " in {{column}}", {
                column: inColName(params["geomCol"])
            });
        },
         icon : 'dku-icon-globe'
    },
    "ZipCodeGeocoder" : {
        description : function(type, params) {
            if (!hasAll(params, ["zipCodeCol"])) return null;
            return translate("SHAKER.PROCESSOR.ZipCodeGeocoder.SUMMARY", actionVerb("Geocode zipcode") + " in {{column}}", {
                column: inColName(params["zipCodeCol"])
            });
        },
        icon : 'dku-icon-globe'
    },
    "GeometryInfoExtractor" : {
        description : function(type, params) {
            if (!hasAll(params, ["inputCol"])) return null;
            return translate("SHAKER.PROCESSOR.GeometryInfoExtractor.SUMMARY", actionVerb("Extract geo info") + " from {{column}}", {
                column: inColName(params["inputCol"])
            });
        },
        icon : 'dku-icon-globe'
    },
    "GeoDistanceProcessor" : {
        checkValid: geoDistanceCheckValid,
        description : function(type, params) {
            if (!hasAll(params, ["input1", "output", "outputUnit", "compareTo"])) return null;
            const p = {
                in1: inColName(params["input1"]),
                in2: inColName(params["input2"])
            }
            if (params.compareTo === "COLUMN"){
                return translate("SHAKER.PROCESSOR.GeoDistanceProcessor.SUMMARY", actionVerb("Compute distance") + " between {{in1}} and {{in2}}", p);
            } else if(params.compareTo === "GEOPOINT"  || params.compareTo === "GEOMETRY"){
                return translate("SHAKER.PROCESSOR.GeoDistanceProcessor.SUMMARY.REFERENCE", actionVerb("Compute distance") + " between {{in1}} and a reference", p);
            }
        },
        icon : 'dku-icon-globe'
    },
    "GeoPointBufferProcessor" : {
        description: function(type, params) {
            if (!params["inputColumn"] || params["inputColumn"].length === 0) {
                return translate("SHAKER.PROCESSOR.GeoPointBufferProcessor.SUMMARY.ERROR.MISSING_INPUT_COLUMN","Choose an input geopoint column.");
            }
            switch (params["shapeMode"]) {
                case "RECTANGLE":
                    return translate('SHAKER.PROCESSOR.GeoPointBufferProcessor.SUMMARY.RECTANGLE',
                        actionVerb("Generate a rectangle") + " centered on {{column}} with dimensions {{width}} x {{height}} {{unit==='KILOMETERS'?'km':'mi'}}", {
                            column: inColName(params["inputColumn"]),
                            width: sanitize(params["width"]),
                            height: sanitize(params["height"]),
                            unit: params["unitMode"]
                    });
                case "CIRCLE":
                    return translate('SHAKER.PROCESSOR.GeoPointBufferProcessor.SUMMARY.CIRCLE',
                        actionVerb("Generate a circle") + " centered on {{column}} with a {{radius}} {{unit==='KILOMETERS'?'km':'mi'}} radius", {
                            column: inColName(params["inputColumn"]),
                            radius: sanitize(params["radius"]),
                            unit: params["unitMode"]
                        });
                default:
                    return translate('SHAKER.PROCESSOR.GeoPointBufferProcessor.SUMMARY.POLYGON', actionVerb("Generate a polygon") + " centered on {{column}}", {
                        column: inColName(params["inputColumn"])
                    });
            }
        },
        icon: 'dku-icon-globe'
    },

    /* Open data */
    "EnrichFrenchPostcode" : {
        description : function(type, params) {
            if (!hasAll(params, ["column"])) return null;
            return translate("SHAKER.PROCESSOR.EnrichFrenchPostcode.SUMMARY", actionVerb("Enrich French postcode") + " info from {{column}}", {
                column: inColName(params["column"])
            });
        },
        icon : 'dku-icon-user-group'
    },
    "EnrichFrenchDepartement" : {
        description : function(type, params) {
            if (!hasAll(params, ["column"])) return null;
            return translate("SHAKER.PROCESSOR.EnrichFrenchDepartment.SUMMARY", actionVerb("Enrich French department") + " info from {{column}}", {
                column: inColName(params["column"])
            });
        },
        icon : 'dku-icon-user-group'
    },

    /* Reshaping */
    "Unfold" : {
        description : function(type, params) {
            if (!hasAll(params, ["column"])) return null;
            return translate("SHAKER.PROCESSOR.Unfold.SUMMARY", actionVerb("Create dummy") + " columns from values of {{column}}", {
                column: inColName(params["column"])
            });
        },
        icon : 'dku-icon-level-up'
    },
    "ArrayUnfold" : {
    description : function(type, params) {
        if (!hasAll(params, ["column"])) return null;
        return translate("SHAKER.PROCESSOR.ArrayUnfold.SUMMARY", actionVerb("Unfold") + " columns from values of {{column}}", {
            column: inColName(params["column"])
        });
    },
    icon : 'dku-icon-level-up'
    },
    "SplitUnfold" : {
        description : function(type, params) {
            if (!hasAll(params, ["column", "separator"])) return null;
            return translate("SHAKER.PROCESSOR.SplitUnfold.SUMMARY", actionVerb("Create dummy") + " columns by splitting {{column}} on {{separator}}", {
                column: inColName(params["column"]),
                separator: anumLiteral(params.separator)
            });
        },
        icon : 'dku-icon-level-up'
    },
    "Pivot" : {
        description : function(type, params) {
            if (!hasAll(params, ["indexColumn", "labelsColumn", "valuesColumn"])) return null;
            return translate("SHAKER.PROCESSOR.Pivot.SUMMARY", actionVerb("Pivot") + " around {{indexColumn}}, labels from {{labelsColumn}}, values from {{valuesColumn}}", {
                indexColumn: inColName(params["indexColumn"]),
                labelsColumn: inColName(params["labelsColumn"]),
                valuesColumn: inColName(params["valuesColumn"])
            });
        },
        icon : 'dku-icon-level-up'
    },
    "ZipArrays" : {
        description : function(type, params) {
            if (!hasAll(params, ["inputColumns", "outputColumn"])) return null;
            return translate("SHAKER.PROCESSOR.ZipArrays.SUMMARY", actionVerb("Zip arrays") + " from {{columns}} to arrays of objects in {{outputColumn}}", {
                columns: inColName(params["inputColumns"].join(", ")),
                outputColumn: outColName(params["outputColumn"])
            });
        },
        icon : 'dku-icon-level-up'
    },
    "ConcatArrays" : {
        description : function(type, params) {
            if (!hasAll(params, ["inputColumns", "outputColumn"])) return null;
            return translate("SHAKER.PROCESSOR.ConcatArrays.SUMMARY", actionVerb("Concatenate arrays") + " from {{inputColumns}} in {{outputColumn}}", {
                inputColumns: inColName(params["inputColumns"].join(", ")),
                outputColumn: outColName(params["outputColumn"])
            });
        },
        icon : 'dku-icon-arrow-expand-exit'
    },
    "SplitFold" : {
        description : function(type, params) {
            if (!hasAll(params, ["column", "separator"])) return null;
            return translate("SHAKER.PROCESSOR.SplitFold.SUMMARY", actionVerb("Split") + " values of {{column}} on {{separator}} and " + actionVerb("fold") + " to new rows", {
                column: inColName(params["column"]),
                separator: anumLiteral(params["separator"])
            });
        },
        icon : 'dku-icon-level-down'
    },
    "ArrayFold" : {
        description : function(type, params) {
            if (!hasAll(params, ["column"])) return null;
            return translate("SHAKER.PROCESSOR.ArrayFold.SUMMARY", actionVerb("Fold array values") + " of {{column}} to new rows", {
                column: inColName(params["column"])
            });
        },
        icon : 'dku-icon-level-down'
    },
    "ObjectFoldProcessor" : {
        description : function(type, params) {
            if (!hasAll(params, ["column"])) return null;
            return translate("SHAKER.PROCESSOR.ObjectFoldProcessor.SUMMARY", actionVerb("Fold keys/values") + " of {{column}} to new rows", {
                column: inColName(params["column"])
            });

        },
        icon : 'dku-icon-level-down'
    },
    "MultiColumnFold" : {
         description : function(type, params) {
            if (!hasAll(params, ["foldNameColumn", "foldValueColumn"])) return null;
             return translate("SHAKER.PROCESSOR.MultiColumnFold.SUMMARY", actionVerb("Fold multiple columns") + " into {{foldNameColumn}} / {{foldValueColumn}}", {
                 foldNameColumn: outColName(params["foldNameColumn"]),
                 foldValueColumn: outColName(params["foldValueColumn"])
             });
        },
        icon : 'dku-icon-level-down'
    },
    "RepeatableUnfold" : {
        description : function(type, params) {
            if (!hasAll(params, ["keyColumn", "foldColumn", "foldTrigger", "dataColumn"])) return null;
            return translate("SHAKER.PROCESSOR.RepeatableUnfold.SUMMARY", "Foreach {{keyColumn}}, create columns for values of {{dataColumn}} between each occurrence of '{{foldColumn}}' in {{foldTrigger}}", {
                keyColumn: inColName(params["keyColumn"]),
                dataColumn: inColName(params["dataColumn"]),
                foldColumn: inColName(params["foldColumn"]),
                foldTrigger: anumLiteral(params["foldTrigger"])
            });
        },
        icon : 'dku-icon-level-up'
    },
    "MultiColumnByPrefixFold" : {
        description : function(type, params) {
           if (!hasAll(params, ["columnNamePattern", "columnNameColumn", "columnContentColumn"])) return null;
            return translate("SHAKER.PROCESSOR.MultiColumnByPrefixFold.SUMMARY", actionVerb("Fold columns") + " matching {{columnNamePattern}} into {{columnNameColumn}} / {{columnContentColumn}}", {
                columnNamePattern: anumLiteral(params["columnNamePattern"]),
                columnNameColumn: outColName(params["columnNameColumn"]),
                columnContentColumn: outColName(params["columnContentColumn"])
            });
       },
       icon : 'dku-icon-level-down'
    },
    "NestProcessor" : {
        checkValid : function(params) {
            appliesToCheckValid(params);
            if (params["outputColumn"] === undefined) {
                throw new StepIAE(translate('SHAKER.PROCESSOR.SwitchCase.ERROR.MISSING_OUTPUT_COLUMN', "Output column not specified"));
            }
        },
        description : function(type, params) {
            return translate("SHAKER.PROCESSOR.NestProcessor.SUMMARY", actionVerb("Nest") + " {{columns}} into {{outputColumn}}", {
                columns: appliesToDescription(params),
                outputColumn: outColName(params["outputColumn"])
            });
        },
        icon : 'dku-icon-arrow-expand-exit'
    },

    /* Join */
    "MemoryEquiJoiner": {
        description : function(type, params){
            if (!hasAll(params, ["leftCol", "rightInput", "rightCol"])) return null;
            return translate("SHAKER.PROCESSOR.MemoryEquiJoiner.SUMMARY", actionVerb("Join") + " column {{leftCol}} with column {{rightCol}} of dataset {{rightInput}}", {
                leftCol: inColName(params["leftCol"]),
                rightCol: inColName(params["rightCol"]),
                rightInput: inColName(params["rightInput"])
            });
        },
        icon : 'dku-icon-arrow-expand-exit'
    },
    "MemoryEquiJoinerFuzzy": {
        description : function(type, params){
            if (!hasAll(params, ["leftCol", "rightInput", "rightCol"])) return null;
            return translate("SHAKER.PROCESSOR.MemoryEquiJoinerFuzzy.SUMMARY", actionVerb("Fuzzy-join") + " column {{leftCol}} with column {{rightCol}} of dataset {{rightInput}}", {
                leftCol: inColName(params["leftCol"]),
                rightCol: inColName(params["rightCol"]),
                rightInput: inColName(params["rightInput"])
            });
        },
        icon : 'dku-icon-arrow-expand-exit'
    },

    /* GREL */
    "CreateColumnWithGREL" : {
        description: function(type, params){
            if (!hasAll(params, ["column", "expression"])) return null;
            if (params.expression.length > 30) {
                return translate('SHAKER.PROCESSOR.CreateColumnWithGREL.DESCRIPTION.LONG', actionVerb("Create column") + " {{column}} with formula", {
                    column: outColName(params["column"])
                });
            } else {
                return translate('SHAKER.PROCESSOR.CreateColumnWithGREL.DESCRIPTION.SHORT', actionVerb("Create column") + " {{column}} with formula {{formula}}", {
                    column: outColName(params["column"]),
                    formula: anumLiteral(params.expression)
                });
            }
        },
        shouldExpandFormula: function(params) {
            if (!hasAll(params, ["expression"])) return false;
            return params.expression.length > 30;
        },
        icon : 'dku-icon-beaker'
    },

    /* Custom */
    "PythonUDF" : {
        checkValid: function(params) {
            if (params.mode === "CELL") {
                defaultIsValidIfHasColumn(params, "column", {throwError: true})
            }
        },
        description: function(type, params) {
            if (params.mode === "CELL") {
                if (!defaultIsValidIfHasColumn(params, "column", {throwError: false})) {
                    return null;
                }
                return translate("SHAKER.PROCESSOR.PythonUDF.SUMMARY.CELL", actionVerb("Create column") + "{{column}} with Python code", {
                    column: outColName(params.column)
                });
            } else if (params.mode === "ROW") {
                return translate("SHAKER.PROCESSOR.PythonUDF.SUMMARY.ROW", actionVerb("Modify row") + " with Python code");
            } else {
                return translate("SHAKER.PROCESSOR.PythonUDF.SUMMARY.DATA", actionVerb("Modify data") + " with Python code");
            }
        },
        shouldExpandFormula: function(params) {
            return params.pythonSourceCode;
        },
        icon : 'dku-icon-beaker'
    },
    "GenerateBigData" : {
        description: function(type, params) {
            return translate("SHAKER.PROCESSOR.GenerateBigData.SUMMARY.CELL", actionVerb("Generate Data") + " {{expansionFactor}} times bigger", {
                expansionFactor: numLiteral(params["expansionFactor"])
            });
        },
        icon : 'dku-icon-beaker'

    },
    "ColumnCopier" : {
        description: function(type, params) {
            if (!hasAll(params, ["inputColumn", "outputColumn"])) {
                return null;
            }
            return translate("SHAKER.PROCESSOR.ColumnCopier.SUMMARY.CELL", actionVerb("Copy") + " column {{inputColumn}} to {{outputColumn}}", {
                inputColumn: inColName(params["inputColumn"]),
                outputColumn: outColName(params["outputColumn"])
            });
        },
        icon : 'dku-icon-copy-step'
    },
    "EnrichWithRecordContextProcessor" : {
        description: function() {
            return translate("SHAKER.PROCESSOR.EnrichWithRecordContextProcessor.SUMMARY.CELL", actionVerb("Enrich") + " records with files info");
        },
        icon : 'dku-icon-file'
    },
    "EnrichWithBuildContextProcessor" : {
        description: function() {
            return translate("SHAKER.PROCESSOR.EnrichWithBuildContextProcessor.SUMMARY.CELL", actionVerb("Enrich") + " records with build info");
        },
        icon : 'dku-icon-play-outline'
    },
    "FastPredict" : {
        description: function(type, params) {
            return translate("SHAKER.PROCESSOR.FastPredict.SUMMARY.CELL", actionVerb("Predict missing") + " values on {{targetColumn}}", {
                inputColumn: strongify(sanitize(params["targetColumn"])),
                outputColumn: outColName(params["outputColumn"])
            });
        },
        icon : 'dku-icon-beaker'

    },
    "BooleanNot" : {
        description: function(type, params) {
            return translate("SHAKER.PROCESSOR.BooleanNot.SUMMARY.CELL", actionVerb("Negate") + " boolean {{column}}", {
                column: inColName(params["column"])
            });
        },
        icon : 'dku-icon-beaker'
    },
    "FillColumn" : {
        description: function(type, params) {
            return translate("SHAKER.PROCESSOR.FillColumn.SUMMARY.CELL", actionVerb("Fill") + " column{{hasColumn?(' '+column):''}}{{hasValue?(' with '+value):''}}", {
                hasColumn: !!params["column"],
                hasValue:!!params["value"],
                column: outColName(params["column"]),
                value: anumLiteral(params["value"])
            });
        },
        icon : 'dku-icon-beaker'
    },
    "ColumnPseudonymization": {
        checkValid: appliesToCheckValid,
        description: function(_, params) {
            return translate("SHAKER.PROCESSOR.ColumnPseudonymization.DESCRIPTION", "{{verbStylised}} {{columns}} {{saltColumn ? (' with column {{saltColumnStylised}} for salting') : ''}} {{pepper ? (' with {{pepperStylised}} as pepper') : ''}}", {
                 verbStylised: actionVerb("Pseudonymize"),
                 columns: appliesToDescription(params),
                 saltColumn: params["saltColumn"],
                 saltColumnStylised: inColName(params["saltColumn"]),
                 pepper: params["pepper"],
                 pepperStylised: anumLiteral(params["pepper"]),
             });
        },
        icon: 'dku-icon-edit-note'
    },
    "MeasureNormalize" : {
        description: function(type, params) {
            return translate("SHAKER.PROCESSOR.MeasureNormalize.SUMMARY.CELL", actionVerb("Normalize") + " measure {{column}} to SI units", {
                column: inColName(params["column"])
            });
        },
        icon : 'dku-icon-beaker'
    },
    "Transpose": {
        description: function (type, params) {
            return translate("SHAKER.PROCESSOR.Transpose.SUMMARY.CELL", actionVerb("Transpose") + " rows into columns around {{column}}", {
                column: inColName(params["column"])
            });
        },
        icon: 'dku-icon-arrow-clockwise'
    },
    "GrokProcessor" : {
        description: function(type, params) {
            return translate("SHAKER.PROCESSOR.GrokProcessor.SUMMARY.CELL", actionVerb("Extract") + " {{column}} with grok.", {
                column: inColName(params["sourceColumn"])
            });
        },
        shouldExpandFormula: function(params) {
            return params.grokPattern;
        },
        icon : 'dku-icon-beaker'
    },
    }

    function jythonEntry(processor, loadedDesc){
        return {
            description : function(type, params) {
                return loadedDesc.desc.meta.label;
            },
            icon : loadedDesc.desc.meta.icon || 'dku-icon-puzzle-piece',
            checkValid : function(params) {
                Object.keys(params.customConfig).forEach(function(k) {
                    let v = params.customConfig[k];
                    let param = loadedDesc.desc.params.find(x => x.name === k);
                    if (param != null && param.mandatory && (v ==null || v.length === 0)) {
                        throw new StepIAE("No value for parameter " + param.name);
                    }
                })
            }
        }
    }

    svc.get = function(processor) {
        if (processor && processor.indexOf("jython-processor") === 0) {
            let loadedDesc = window.dkuAppConfig.customJythonProcessors.find(x => x.elementType === processor);
            Assert.trueish(loadedDesc, 'processor desc not loaded');
            return jythonEntry(processor, loadedDesc);
        } else {
            return svc.map[processor];
        }
    };

    return svc;
});

app.factory("ShakerProcessorsUtils", function($filter, ShakerProcessorsInfo, translate) {

    var svc = {}

    // Wonderfully copy/pasted from script.js, with some additional try/catch to avoid some exceptions
    /* Perform JS validation of the step. Does not set frontError */
    svc.validateStep = function(step, processors) {
        if (step.metaType == "GROUP") {
            if (step.steps != null) {
                for (let i = 0; i < step.steps.length; i++) {
                    var subValidationResult = svc.validateStep(step.steps[i], processors);
                    if (subValidationResult) {
                        return subValidationResult;
                    }
                }
            }
        } else {
            try {
                var processorType = $filter('processorByType')(processors, step.type);
                /* If we have some stepParams, then check using them */
                if (processorType.params) {
                    for (var paramIdx in processorType.params) {
                        var param = processorType.params[paramIdx];
                        var value = step.params[param.name];
                        if (param.mandatory && !param.canBeEmpty && (value == null || value.length === 0)) {
                            // eslint-disable-next-line no-undef
                            return new StepIAE("Missing parameter: " + (param.label || param.name));
                        }
                    }
                }
                /* Then also play the specific validation of each step */
                let processorInfo = ShakerProcessorsInfo.get(step.type);
                if (processorInfo.checkValid){
                    try {
                        processorInfo.checkValid(step.params);
                    } catch (e) {
                        return e;
                    }
                }
            } catch(e) { // happens for a plugin processor if plugin has been uninstalled
                // eslint-disable-next-line no-undef
                return new StepIAE("Unknown processor: " + step.type);
            }
        }
        return null;
    };

    /**
     * The purpose of this method is to manipulate the provided steps to display them in the UI.
     * Requirements to display steps are:
     * - A single array of steps with all standalone steps, group steps and their sub-steps at the same level.
     * - Only enabled steps are displayed.
     * - In case a step has a validation error, it should be displayed with its validation error message.
     * - Group sub-steps must be distinguishable from standalone steps.
     * The method returns a dictionary containing:
     * - an array with the flattened enabled steps with the enriched information for each step
     * - an array with all the provided steps with the enriched information for each step
     */
    svc.getStepsWithSubSteps = function(steps, processors) {
        // compute recursive steps
        let flattenedEnabledSteps = [];
        steps.forEach(step => {
            step["mainStep"] = true; // Normal step => No padding
            step["errorMsg"] = svc.validateStep(step, processors);
            if(!step.disabled) {
                if (step.metaType === "GROUP") {
                    let enabledSubSteps = [];
                    step.steps.forEach(subStep => {
                        subStep["mainStep"] = false; // Step inside a group => need padding
                        subStep["errorMsg"] = svc.validateStep(subStep, processors);
                        if(!subStep.disabled) {
                            enabledSubSteps.push(subStep);
                        }
                    });
                    if (enabledSubSteps.length > 0) {
                        step.steps = enabledSubSteps;
                        flattenedEnabledSteps = [...flattenedEnabledSteps, step, ...enabledSubSteps];
                    }
                } else {
                    flattenedEnabledSteps.push(step);
                }
            }
        });
        return { flattenedEnabledSteps, enrichedSteps: steps };
    }

    svc.getGroupName = function(step, steps) {
        if (step.metaType == 'GROUP') {
            if (step.name && step.name.length > 0) {
                return step.name;
            } else {
                const groupIndex = steps.filter(function(s) { return s.metaType === 'GROUP'; })
                                        .indexOf(step);
                return 'GROUP ' + (groupIndex < 0 ? '' : groupIndex + 1);
            }
        } else {
            return "Invalid group";
        }
    }

    svc.getStepDescription = function(processor, type, params) {
        const prefix = (processor && processor.deprecated) ? "<span class=\"deprecation-tag\">"+ translate("SHAKER.PROCESSORS.DESCRIPTION.DEPRECATED", "[DEPRECATED]") +"</span> " : "";
        const e = ShakerProcessorsInfo.get(type);
        if (!e || !e.description) {
            return prefix + type + " " + translate("SHAKER.PROCESSORS.DESCRIPTION.UNKNOWN", "(UNKNOWN)");
        }
        if (e.checkValid) {
            try {
                e.checkValid(params);
            } catch (e) {
                const invalidDecorator = " " + translate("SHAKER.PROCESSORS.DESCRIPTION.INVALID", "(invalid)")
                if (processor) {
                    return processor.enDescription + invalidDecorator;  // no need for prefix since already handled in CachedAPICalls with enDescription
                } else {
                    return type + invalidDecorator; // same as above
                }
            }
        }
        const desc = e.description(type, params);
        if (!desc) {
            const invalidNoDescDecorator = " " + translate("SHAKER.PROCESSORS.DESCRIPTION.INVALID_NO_DESC", "(invalid - no desc)")
            if (processor) {
                return processor.enDescription + invalidNoDescDecorator; // same as above
            } else {
                return type + invalidNoDescDecorator; // same as above
            }
        } else {
            return prefix + desc;
        }
    };

    // Match a query to the step params. A step belongs to a processor
    // Every processor is registered in `ShakerProcessorInfo.svc.map`
    // Define a `paramsMatcher` on the processor that matches the step params with the given query
    svc.matchStepParams = function(step, query) {
        if (!step || !query || !step.params) {
            return false;
        }
        function findMatchInObject(object, query) {
            if (['string', 'bigint', 'number'].includes(typeof object)) {
                return normalizeForSearch(object).includes(query);
            }
            if (typeof object === 'boolean') {
                return object ? query === 'true' : query === 'false';
            }
            if (Array.isArray(object)) {
                return object.some(value => findMatchInObject(value, query));
            }
            if (typeof object === 'object') {
                return Object.values(object).some(value => findMatchInObject(value, query));
            }
            return false;
        }


        const processorInfo = ShakerProcessorsInfo.get(step.type);
        if (processorInfo) {
            if (processorInfo.paramsMatcher) {
                return processorInfo.paramsMatcher(query, step.params);
            }
            return findMatchInObject(step.params, query);
        }
        // If the processor is unknown or the step does not have any params
        // It cannot match with anything
        return false;
    }

    svc.getHtmlStepDescriptionOrError = function(step, processors) {
        try {
            var processor = $filter('processorByType')(processors, step.type);
        } catch(e) { // may happen for a processor from a deleted plugin
            return "<span class=\"text-error\">Unknown processor: " + sanitize(step.type) + "</span>"
        }
        if (step.errorMsg) {
            return "<span class=\"text-error\">" + processor.enDescription + ": " + step.errorMsg.message + "</span>";
        } else {
            return svc.getStepDescription(processor, step.type, step.params);
        }
    }

    svc.getStepIcon = function(type, params, size) {
        const e = ShakerProcessorsInfo.get(type);
        const icon = e && e.icon && (typeof e.icon == 'function' ? e.icon(type, params) : e.icon) || 'dku-icon-warning-fill';

        if(icon.startsWith('dku-icon-')) {
            // new icons require a size
            return `${icon}-${size}`;
        } else {
            // not new icons (from plugins) don't
            return icon
        }
    };

    svc.getStepImpactVerb = function(type, params) {
        const e = ShakerProcessorsInfo.get(type);
        if (e && e.impactVerb) {
            return e.impactVerb(params);
        }
        return translate("SHAKER.PROCESSORS.IMPACT.MODIFIED", "modified");
    };

    return svc;
});

app.factory("ShakerSuggestionsEngine", function(ShakerProcessorsUtils, CreateModalFromDOMElement, Dialogs, PluginsService, $filter, $rootScope, $q, translate) {
var svc = {};

const categoryFilterData = translate('SHAKER.SUGGESTIONS.CATEGORIES.FILTER_DATA', "Filter data");
const categoryNumbers = translate('SHAKER.SUGGESTIONS.CATEGORIES.NUMBERS', "Numbers");
const categoryGeography = translate('SHAKER.SUGGESTIONS.CATEGORIES.GEOGRAPHY', "Geography");
const categoryWeb = translate('SHAKER.SUGGESTIONS.CATEGORIES.WEB', "Web");
const categoryEmail = translate('SHAKER.SUGGESTIONS.CATEGORIES.EMAIL', "Email");
const categoryBoolean = translate('SHAKER.SUGGESTIONS.CATEGORIES.BOOLEAN', "Boolean");
const categoryMeasure = translate('SHAKER.SUGGESTIONS.CATEGORIES.MEASURE', "Measure");
const categoryDate = translate('SHAKER.SUGGESTIONS.CATEGORIES.DATE', "Date");
const categoryTransformations = translate('SHAKER.SUGGESTIONS.CATEGORIES.TRANSFORMATIONS', "Transformations");
const categoryKeepDelete = translate('SHAKER.SUGGESTIONS.CATEGORIES.KEEP_DELETE', "Delete/Keep");
const categoryDataCleansing = translate('SHAKER.SUGGESTIONS.CATEGORIES.DATA_CLEANSING', "Data cleansing");
const categorySmartPatternBuilder = translate('SHAKER.SUGGESTIONS.CATEGORIES.SMART_PATTERN_BUILDER', "Smart Pattern Builder");
const categoryText = translate('SHAKER.SUGGESTIONS.CATEGORIES.TEXT', "Text");

function addCustomSuggestion(map, category, id, func, text, icon) {
    (category in map ?
         map[category] :
         (map[category] = []))
        .push({id: id, text: text, action: func, icon});
}

svc.getCategoryPriority = function (category) {
    // default category is 0, unspecified categories will remain in insertion order
    return {
        [categorySmartPatternBuilder]: -1
    } [category] ?? 0;
}

/* Cell suggestions
 * MUST NOT include column-level suggestions
 */
svc.computeCellSuggestions =  function(column, value, validity) {
    const truncateFilter = $filter("gentleTruncate");

    function addStepBasedSuggestion(map, category, id, type, params, text) {
        addCustomSuggestion(map, category, id, function(shakerScope) {
            window.WT1SVC.event("shaker-use-cell-sugg", {
                "colMeaning" : column.selectedType ? column.selectedType.name : "unk" ,
                "cellValid" : validity,
                "processor" : type,
                "suggText" : text,
            });
            shakerScope.addStepNoPreview(type, params);
            if (type == "FilterOnValue") {
            	shakerScope.mergeLastDeleteRows();
            }
            shakerScope.autoSaveAutoRefresh();
        }, text, ShakerProcessorsUtils.getStepIcon(type, params, '16')
        );
    }

    function addUnconfiguredStepBasedSuggestion(map, category, id, type, params, text) {
        addCustomSuggestion(map, category, id, function(shakerScope) {
            window.WT1SVC.event("shaker-use-cell-sugg", {
                "colMeaning" : column.selectedType ? column.selectedType.name : "unk" ,
                "cellValid" : validity,
                "processor" : type,
                "suggText" : text,
            });
            shakerScope.addUnconfiguredStep(type, params);
            if (type === "FilterOnValue") {
            	shakerScope.mergeLastDeleteRows();
            }
        }, text, ShakerProcessorsUtils.getStepIcon(type, params, '16')
        );
    }


    function visualIfSuggestionTitle(type, columnName, value) {
        const prefix = actionVerb("Create") +  " If Then Else rule based on " +  inColName(truncateFilter(columnName, 20));
        if(!value) {
            return prefix + " " + anumLiteral("is not defined");
        } else {
            if(type === "boolean") {
                return prefix + " is " + anumLiteral(parseBoolean(value));
            } else if(type === "array" && "[]" === value) {
                return prefix + " is " + anumLiteral("an empty array");
            } else {
                const operator = isNumber(type) ? " == " : " equals "
                return prefix + operator + anumLiteral(truncateFilter(value, 20));
            }
        }
    }

    var map = Object();

    if (value) {
        addStepBasedSuggestion(map, categoryFilterData, "keep", "FilterOnValue", {
            'appliesTo': "SINGLE_COLUMN",
            'action': "REMOVE_ROW",
            'columns': [column.name],
            'values': [value],
            'matchingMode' : "FULL_STRING",
            'normalizationMode' : "EXACT"
        }, actionVerb("Remove") + " rows equal to " + anumLiteral(truncateFilter(value, 35)));

        addStepBasedSuggestion(map, categoryFilterData, "remove", "FilterOnValue", {
            'appliesTo': "SINGLE_COLUMN",
            'action': "KEEP_ROW",
            'columns': [column.name],
            'values': [value],
            'matchingMode' : "FULL_STRING",
            'normalizationMode' : "EXACT"
        }, actionVerb("Keep") + " only rows equal to " + anumLiteral(truncateFilter(value, 35)));

        addStepBasedSuggestion(map, categoryFilterData, "clear", "FilterOnValue", {
            'appliesTo': "SINGLE_COLUMN",
            'action': "CLEAR_CELL",
            'columns': [column.name],
            'values': [value],
            'matchingMode' : "FULL_STRING",
            'normalizationMode' : "EXACT"
        }, actionVerb("Clear") + " cells equal to " + anumLiteral(truncateFilter(value, 35) ));
    }

    const type = column.recipeSchemaColumn && column.recipeSchemaColumn.column
            ? column.recipeSchemaColumn.column.type
            : column.datasetSchemaColumn
                ? column.datasetSchemaColumn.type
                : "string";
    const operator = value ? filterDescEqualityOperator(type, value) : "is empty";

    if(value) {
        addUnconfiguredStepBasedSuggestion(map, categoryFilterData, "visual_if", "VisualIfRule",
            visualIfParams(column.name, type, value, operator), visualIfSuggestionTitle(type, column.name, value));
    }


    if (value && validity && validity.indexOf('I') === 0) {
        addStepBasedSuggestion(map, categoryDataCleansing, "c1", "FilterOnBadType", {
            'appliesTo': "SINGLE_COLUMN",
            'action': 'REMOVE_ROW',
            'columns': [column.name],
            'type': column.selectedType.name,
        }, actionVerb("Remove invalid") + " rows for meaning");
        addStepBasedSuggestion(map, categoryDataCleansing, "c1", "FilterOnBadType", {
            'appliesTo': "SINGLE_COLUMN",
            'action': 'CLEAR_CELL',
            'columns': [column.name],
            'type': column.selectedType.name,
        }, actionVerb("Clear invalid") + " cells for meaning");
    }

    if (!value) {
        addStepBasedSuggestion(map, categoryDataCleansing, "c1", "RemoveRowsOnEmpty", {
            'columns': [column.name],
            appliesTo : 'SINGLE_COLUMN',
            'keep' : false
        }, actionVerb("Remove") +" rows where cell is empty");
        addStepBasedSuggestion(map, categoryDataCleansing, "c1", "RemoveRowsOnEmpty", {
            'columns': [column.name],
            appliesTo : 'SINGLE_COLUMN',
            'keep' : true
        }, actionVerb("Keep only") +  " rows where cell is empty");
        addCustomSuggestion(map, categoryDataCleansing, "fill_empty_withval", function(scope) {
            CreateModalFromDOMElement("#fill-empty-with-value-box", scope, "FillEmptyWithValueController",
                function(newScope) { newScope.$apply(function() {
                        newScope.setColumns([column.name]);
                        newScope.isNumericOnly = ["LongMeaning", "DoubleMeaning"]
                            .indexOf(column.selectedType && column.selectedType.name) > -1;
                }); });
        }, actionVerb("Fill") +  " empty rows with...", ShakerProcessorsUtils.getStepIcon("FillEmptyWithValue", {}, '16'));
        addUnconfiguredStepBasedSuggestion(map, categoryDataCleansing, "visual_if", "VisualIfRule",
            visualIfParams(column.name, type, value, operator), visualIfSuggestionTitle(type, column.name, value));
    }
    return map;
};

svc.computeContentSuggestions = function(column, cellValue, value, validity, CreateModalFromTemplate, selectionStartOffset, selectionEndOffset) {
    const truncateFilter = $filter("gentleTruncate");
    function addStepBasedSuggestion(map, category, id, type, params, text) {
        addCustomSuggestion(map, category, id, function(shakerScope) {
            window.WT1SVC.event("shaker-use-content-sugg", {
                "colMeaning" : column.selectedType ? column.selectedType.name : "unk" ,
                "cellValid" : validity,
                "value" : value, "startOff" : selectionStartOffset, "endOff" : selectionEndOffset,
                "processor" : type,
                "suggText" : text,
            });
            shakerScope.addStepAndRefresh(type, params);
        }, text, ShakerProcessorsUtils.getStepIcon(type, params, '16'));
    }

    function addUnconfiguredStepBasedSuggestion(map, category, id, type, params, text) {
        addCustomSuggestion(map, category, id, function(shakerScope) {
            window.WT1SVC.event("shaker-use-content-sugg", {
                "colMeaning" : column.selectedType ? column.selectedType.name : "unk" ,
                "cellValid" : validity,
                "value" : value,
                "startOff" : selectionStartOffset,
                "endOff" : selectionEndOffset,
                "processor" : type,
                "suggText" : text,
            });
            shakerScope.addUnconfiguredStep(type, params);
        }, text, ShakerProcessorsUtils.getStepIcon(type, params, '16')
        );
    }


    var map = Object();
    const type = column.recipeSchemaColumn && column.recipeSchemaColumn.column
            ? column.recipeSchemaColumn.column.type
            : column.datasetSchemaColumn
                ? column.datasetSchemaColumn.type
                : "string";

    addStepBasedSuggestion(map, categoryTransformations, "c1", "ColumnSplitter", {
            'inCol': column.name,
            'separator': value,
            'outColPrefix' : column.name + "_",
            'target' : "COLUMNS"
        }, actionVerb("Split") +  " column on " + anumLiteral(value) + "");
    addStepBasedSuggestion(map, categoryTransformations, "c1", "FindReplace", {
            'appliesTo' :"SINGLE_COLUMN",
            "columns" : [column.name],
            'matching' : "SUBSTRING",
            'normalization' : "EXACT",
            "mapping" : [{"from":value,"to":""}],
        }, actionVerb("Replace") +  " " + anumLiteral(value) + " by ...");

    addCustomSuggestion(map, categorySmartPatternBuilder, "extractor",
        function(scope) {
            var deferred = $q.defer();
            CreateModalFromTemplate("/templates/shaker/regexbuilder-box.html", scope, "RegexBuilderController",
                function(newScope) {
                    newScope.deferred = deferred;
                    newScope.$apply(function() {
                        newScope.columnName = column.name;
                        newScope.firstSentence = cellValue;
                        newScope.addSelection(cellValue, selectionStartOffset, selectionEndOffset);
                        newScope.calledFrom = "highlight-extract_like";
                    });
                    deferred.promise.then(function(newPattern) {
                        var params = {
                            "column" : column.name,
                            "prefix": column.name + '_extracted_',
                            "pattern": newPattern.regex,
                            "extractAllOccurrences": newPattern.hasMultiOccurrences,
                        };
                        scope.addStepAndRefresh("RegexpExtractor", params);
                    });
                },
                "sd-modal");
        },
        actionVerb("Extract") + " text like " + anumLiteral(value) + "...",
        ShakerProcessorsUtils.getStepIcon('RegexpExtractor', {}, '16')
    );

    addCustomSuggestion(map, categorySmartPatternBuilder, "filter",
        function(scope) {
            var deferred = $q.defer();
            CreateModalFromTemplate("/templates/shaker/regexbuilder-box.html", scope, "RegexBuilderController",
                function(newScope) {
                    newScope.deferred = deferred;
                    newScope.$apply(function() {
                        newScope.columnName = column.name;
                        newScope.firstSentence = cellValue;
                        newScope.addSelection(cellValue, selectionStartOffset, selectionEndOffset);
                        newScope.calledFrom = "highlight-remove_like";
                    });
                    deferred.promise.then(function(newPattern) {
                        var params = {
                            'action': "REMOVE_ROW",
                            'appliesTo' : "SINGLE_COLUMN",
                            'columns': [column.name],
                            'values': [newPattern.regex],
                            'matchingMode' : "PATTERN",
                            'normalizationMode' : "EXACT"
                        };
                        scope.addStepAndRefresh("FilterOnValue", params);
                    });
                },
                "sd-modal");
        },
        actionVerb("Remove") + " rows containing text like " + anumLiteral(value) + "...",
        ShakerProcessorsUtils.getStepIcon('FilterOnValue', {}, '16')
    );

    addCustomSuggestion(map, categorySmartPatternBuilder, "keep",
        function(scope) {
            var deferred = $q.defer();
            CreateModalFromTemplate("/templates/shaker/regexbuilder-box.html", scope, "RegexBuilderController",
                function(newScope) {
                    newScope.deferred = deferred;
                    newScope.$apply(function() {
                        newScope.columnName = column.name;
                        newScope.firstSentence = cellValue;
                        newScope.addSelection(cellValue, selectionStartOffset, selectionEndOffset);
                        newScope.calledFrom = "highlight-filter_like";
                    });
                    deferred.promise.then(function(newPattern) {
                        var params = {
                            'action': "KEEP_ROW",
                            'appliesTo' : "SINGLE_COLUMN",
                            'columns': [column.name],
                            'values': [newPattern.regex],
                            'matchingMode' : "PATTERN",
                            'normalizationMode' : "EXACT"
                        };
                        scope.addStepAndRefresh("FilterOnValue", params);
                    });
                },
                "sd-modal");
        },
        actionVerb("Keep") + " rows containing text like " + anumLiteral(value) + "...",
        ShakerProcessorsUtils.getStepIcon("FilterOnValue", {}, '16')
    );

    addCustomSuggestion(map, categorySmartPatternBuilder, "flag",
        function(scope) {
            var deferred = $q.defer();
            CreateModalFromTemplate("/templates/shaker/regexbuilder-box.html", scope, "RegexBuilderController",
                function(newScope) {
                    newScope.deferred = deferred;
                    newScope.$apply(function() {
                        newScope.columnName = column.name;
                        newScope.firstSentence = cellValue;
                        newScope.addSelection(cellValue, selectionStartOffset, selectionEndOffset);
                        newScope.calledFrom = "highlight-flag_like";
                    });
                    deferred.promise.then(function(newPattern) {
                        var params = {
                            "flagColumn" : column.name + "_flagged",
                            'action': "KEEP_ROW",
                            'appliesTo' : "SINGLE_COLUMN",
                            'columns': [column.name],
                            'values': [newPattern.regex],
                            'matchingMode' : "PATTERN",
                            'normalizationMode' : "EXACT"
                        };
                        scope.addStepAndRefresh("FlagOnValue", params);
                    });
                },
                "sd-modal");
        },
        actionVerb("Flag") + " rows containing text like " + anumLiteral(value) + "...",
        ShakerProcessorsUtils.getStepIcon("FlagOnValue", {}, '16')
    );

    addStepBasedSuggestion(map, categoryFilterData, "c1", "FilterOnValue", {
            'action': "REMOVE_ROW",
            'appliesTo' : "SINGLE_COLUMN",
            'columns': [column.name],
            'values': [value],
            'matchingMode' : "SUBSTRING",
            'normalizationMode' : "EXACT"
        }, actionVerb("Remove") +  " rows containing " + anumLiteral(value) + "");

    addStepBasedSuggestion(map, categoryFilterData, "c1", "FilterOnValue", {
            'action': "KEEP_ROW",
            'appliesTo' : "SINGLE_COLUMN",
            'columns': [column.name],
            'values': [value],
            'matchingMode' : "SUBSTRING",
            'normalizationMode' : "EXACT"
        }, actionVerb("Keep only") + " rows containing " + anumLiteral(value) + "");

    if (type === 'string') {
        addUnconfiguredStepBasedSuggestion(map, categoryFilterData, "visual_if", "VisualIfRule",
            visualIfParams(column.name, type, value, 'contains'),
            actionVerb("Create") +  " If Then Else rule based on " +  inColName(truncateFilter(column.name, 20)) + " contains " + anumLiteral(truncateFilter(value, 20))
        );
    }

    return map;

};


/* Column suggestions are handled fully in Javascript based on the data in the column header, to avoid useless
 * calls.
 *
 * Returns an array of two maps of <category name, array[]>
 *   - id (string)
 *   - text (string)
 *   - action (function)
 * The first map contains "important" suggestions, the second one more suggestions.
 * We ensure that there is always at least one non-additional suggestion per column
 */
svc.computeColumnSuggestions = function(column, CreateModalFromDOMElement, CreateModalFromTemplate, forCell, forInvalidCell, appConfig) {
    // UGLY
    var meaningLabelFilter = angular.element("body").injector().get("$filter")("meaningLabel");

    var COLUMN_NAME_PH = '__DKU_COLUMN_NAME__', CNRE = new RegExp(COLUMN_NAME_PH, 'g'),
        TYPE_NAME_PH   = '__DKU_COLUMN_TYPE__', TNRE = new RegExp(TYPE_NAME_PH, 'g');

    function addStepBasedSuggestion(map, category, id, type, params, text, incomplete, oneStep) {
        params = JSON.stringify(params);
        var reportEvent = report("shaker-use-col-sugg", type, text),
            addStepFnName = incomplete ? 'addUnconfiguredStep' : multi ? 'addStepNoPreview' : 'addStep',
            addStepFn = function (c) { this[addStepFnName](type, JSON.parse(params
                    .replace(CNRE, c.name).replace(TNRE, c.selectedType && c.selectedType.name || ''))); };
        addCustomSuggestion(map, category, id, function(shakerScope) {
            reportEvent();
            if (oneStep) {
            	addStepFn.call(shakerScope, columns[0]);
            } else {
            	columns.forEach(addStepFn, shakerScope);
            }
            if (type == "FindReplace") {
            	shakerScope.mergeLastFindReplaces();
            }
            shakerScope.autoSaveAutoRefresh();

        }, text, ShakerProcessorsUtils.getStepIcon(type, params, '16'));
    }

    function addAppliesToSuggestion(map, category, id, type, params, text, incomplete) {
        if (columnNames.length === 1) {
            params["appliesTo"] = "SINGLE_COLUMN";
        } else {
            params["appliesTo"] = "COLUMNS";
        }
        params["columns"] = columnNames;

        addStepBasedSuggestion(map, category, id, type, params, text, incomplete, true);
    }

    var map = {}, moreMap = {}, target,
        columns = [column];
    if (Array.isArray(column)) {
        columns = column;
        column = column.length === 1 ? column = column[0] : { name: COLUMN_NAME_PH };
    }

    var types = columns.reduce(function(ts, c){
            if (c.selectedType && ts.indexOf(c.selectedType.name) === -1) {
                ts.push(c.selectedType.name);
            } return ts; }, []),
        multi = columns.length > 1,
        hasNOK   = columns.some(function(c) { return c.selectedType && c.selectedType.nbNOK   > 0; }),
        hasEmpty = columns.some(function(c) { return c.selectedType && c.selectedType.nbEmpty > 0; }),
        typesAllIn = types.every.bind(types, function(t) { return this.indexOf(t) !== -1; }),
        noTypeIn   = types.every.bind(types, function(t) { return this.indexOf(t) === -1; }),
        typeIs     = function(t) { return types.length === 1 && types[0] === t; },
        typeName = types.length === 1 ? types[0] : TYPE_NAME_PH,
        columnName = column.name,
        columnNames = columns.map(function(c){ return c.name; }),
        eventBase = multi ? {
            colMeaning: column.selectedType ? column.selectedType.name    : "unk" ,
            nbOK:       column.selectedType ? column.selectedType.nbOK    : -1,
            nbNOK:      column.selectedType ? column.selectedType.nbNOK   : -1,
            nbEmpty:    column.selectedType ? column.selectedType.nbEmpty : -1
        } : {
            colMeanings: types.join(','),
            colCount: columns.length,
            hasNOK: hasNOK,
            hasEmpty: hasEmpty
        },
        report = function(evt, proc, text) {
            var ep = { processor: proc, text: text }, i;
            for (i in eventBase) {
                ep[i] = eventBase[i];
            }
            return window.WT1SVC.event.bind(window.WT1SVC, evt, ep);
        };

    /* Step 1 : add type-based main suggestions */
    if (typeIs("IPAddress")) {
        addStepBasedSuggestion(map, categoryGeography, "geoip", "GeoIPResolver", {
            'inCol': columnName,
            'outColPrefix': columnName+'_',
            'extract_country' : true,
            'extract_countrycode' : true,
            'extract_region' : true,
            'extract_city' : true,
            'extract_geopoint' : true
        }, translate('SHAKER.SUGGESTIONS.COLUMN.IPAddress.RESOLVE', actionVerb("Resolve") + " GeoIP"));
    } else if (typeIs("GeoPoint")) {
        addStepBasedSuggestion(map, categoryGeography, "extractlatlon", "GeoPointExtractor", {
            'column': columnName,
            'lat_col': columnName+'_latitude',
            'lon_col' : columnName+'_longitude'
        }, translate('SHAKER.SUGGESTIONS.COLUMN.GeoPoint.EXTRACT_COORDINATES', actionVerb("Extract") + " latitude/longitude"));
        if (PluginsService.isPluginLoaded('geoadmin')) {
            addStepBasedSuggestion(map, categoryGeography, "geoadmin", "CityLevelReverseGeocoder", {
                'inputCol': columnName
            }, translate('SHAKER.SUGGESTIONS.COLUMN.GeoPoint.REVERSE_GEOCODE', actionVerb("Reverse-geocode") + " location"), true);
        } else {
            addCustomSuggestion(map, categoryGeography, "geoadmin", function(shakerScope) {
                Dialogs.ack(shakerScope, "Plugin required",
                    translate("SHAKER.SUGGESTIONS.COLUMN.GeoPoint.REVERSE_GEOCODE.PLUGIN_REQUIRED",
                        "The <a href='https://doc.dataiku.com/dss/latest/preparation/geo_processors.html' target='_blank'>Reverse Geocoding</a> plugin is required for this action.<br><br><a href='/plugins-explore/store/' target='_blank'>Go to plugins</a> to install or request it.",
                        { pluginStoreUrl: "https://doc.dataiku.com/dss/latest/preparation/geo_processors.html", pluginPageUrl: "/plugins-explore/store/" })
                );
            }, translate('SHAKER.SUGGESTIONS.COLUMN.GeoPoint.REVERSE_GEOCODE', actionVerb("Reverse-geocode") + " location"),
            'dku-icon-globe-16');
        }
    } else if (typeIs("QueryString")) {
        addStepBasedSuggestion(map, categoryWeb, "querystring", "QueryStringSplitter", {
            'column': columnName,
        }, translate('SHAKER.SUGGESTIONS.COLUMN.QueryString.SPLIT', actionVerb("Split query string") + " elements"));
    } else if (typeIs("URL")) {
        addStepBasedSuggestion(map, categoryWeb, "urlsplit", "URLSplitter", {
            'column': columnName,
            extractPath:true,
            extractQueryString:true,
            extractPort:true,
            extractAnchor:true,
            extractScheme:true,
            extractHost:true
        }, translate('SHAKER.SUGGESTIONS.COLUMN.URL.SPLIT', actionVerb("Split URL") + " into host, port ..."));
    } else if (typeIs("Email")) {
        addStepBasedSuggestion(map, categoryEmail, "emailsplit", "EmailSplitter", {
            'column': columnName,
        }, translate('SHAKER.SUGGESTIONS.COLUMN.Email.SPLIT', actionVerb("Split email") + " address"));
    } else if (typeIs("Boolean")) {
        addStepBasedSuggestion(map, categoryBoolean, "booleannot", "BooleanNot", {
            'column': columnName
        }, translate('SHAKER.SUGGESTIONS.COLUMN.Boolean.NEGATE', actionVerb("Negate") + " boolean"));
    } else if (typeIs("Measure")) {
         addStepBasedSuggestion(map, categoryMeasure, "measurenormalize", "MeasureNormalize", {
             'column': columnName
         }, translate('SHAKER.SUGGESTIONS.COLUMN.Measure.NORMALIZE', actionVerb("Normalize") + " to SI units"));
    } else if (typeIs("DateSource") && !multi) {
        addCustomSuggestion(map, categoryDate, "date_stuff", function(scope) {
            report("shaker-use-col-sugg", "SmartDate", "Parse date...")();
            var deferred = $q.defer();
            CreateModalFromTemplate("/templates/shaker/smartdate-box.html", scope, "SmartDateController",
                function(newScope) { newScope.$apply(function() {
                        newScope.deferred = deferred;
                        newScope.setColumn(columnName);
                }); }, "sd-modal");
            deferred.promise.then(function([newFormat, newFormatLanguage]) {
                scope.addStepAndRefresh("DateParser", {
                    "appliesTo" : "SINGLE_COLUMN",
                    "columns" : [columnName],
                    "outCol" : columnName + "_parsed", "formats" : [newFormat],
                    "lang" : newFormatLanguage || "en_US", "timezone_id" : "UTC"});
            });
        }, translate('SHAKER.SUGGESTIONS.COLUMN.DateSource.PARSE', actionVerb("Parse") + " date..."), "dku-icon-calendar-16");
    } else if (typeIs("DateSource")) {
        addAppliesToSuggestion(map, categoryDate, null, "DateParser", {
            "lang" : "en_US", "timezone_id" : "UTC"
        },  translate('SHAKER.SUGGESTIONS.COLUMN.DateSource.PARSE', actionVerb("Parse") + " date..."), true);
    }
    else if (typeIs("Date") || typeIs("DatetimeNoTz") || typeIs("DateOnly") ) {
        addStepBasedSuggestion(map, categoryDate, "date_difference", "DateDifference", {
            "input1" : columnName,
            "output" : "since_" + columnName + "_days",
            "compareTo" : "NOW",
            "outputUnit":  "DAYS"
        }, translate('SHAKER.SUGGESTIONS.COLUMN.DateSource.COMPUTE_TIME_SINCE', actionVerb("Compute time") + " since"));
        addStepBasedSuggestion(map, categoryDate, "extract_components", "DateComponentsExtractor", {
                "column" : columnName,
                "timezone_id" : "UTC",
                "outYearColumn" : columnName + "_year",
                "outMonthColumn" : columnName + "_month",
                "outDayColumn" : columnName + "_day"
        }, translate('SHAKER.SUGGESTIONS.COLUMN.DateSource.EXTRACT_DATE_COMPONENTS', actionVerb("Extract") + " date components"));
        addStepBasedSuggestion(map, categoryDate, "filterOnDate", "FilterOnDate", {
                "appliesTo": 'SINGLE_COLUMN',
                "columns" : [columnName],
                "action": 'KEEP_ROW',
                "filterType": 'RANGE',
                "timezone_id": "UTC",
                "part": 'YEAR',
                "option": {
                    containsCurrentDatePart: true,
                    isUntilNow: false,
                    last: 0,
                    next: 0
                },
        }, translate('SHAKER.SUGGESTIONS.COLUMN.DateSource.FILTER_ON_DATE', actionVerb("Filter") + " on date"), true);
    } else if (typeIs("UserAgent")) {
         addStepBasedSuggestion(map, categoryWeb, "useragent", "UserAgentClassifier", {
            'column': columnName,
        }, translate('SHAKER.SUGGESTIONS.COLUMN.UserAgent.CLASSIFY', actionVerb("Classify") + " User-Agent"));
    } else if (typeIs("JSONObjectMeaning")) {
        addStepBasedSuggestion(map, categoryTransformations, null, "JSONFlattener", {
            "inCol" : columnName,
            "maxDepth" : 1,
            'prefixOutputs' : true,
            'nullAsEmpty' : true,
            'separator' : '_'
        }, translate('SHAKER.SUGGESTIONS.COLUMN.JSONObject.UNNEST', actionVerb("Unnest") + " object"));
    } else if (typeIs("FreeText")) {
         addStepBasedSuggestion(map, categoryText, "null", "Tokenizer", {
             'inCol': columnName,
             'operation' : 'TO_JSON',
             'language' : 'english'
         }, translate('SHAKER.SUGGESTIONS.COLUMN.FreeText.TOKENIZE', actionVerb("Split") + " in words (tokenize)"));
         addStepBasedSuggestion(map, categoryText, "null", "SplitIntoChunks", {
             'inCol': columnName,
             'outCol': columnName + '_chunked',
             "separators": [
                 {
                     value: "\\n\\n",
                     isDefault: true,
                     description: "Double new lines",
                     enabled: true
                 },
                 {
                     value: "\\n",
                     isDefault: true,
                     description: "New Lines",
                     enabled: true
                 },
                 {
                     value: " ",
                     isDefault: true,
                     description: "Spaces",
                     enabled: true
                 },
                 {
                     value: "",
                     isDefault: true,
                     description: "Each character",
                     enabled: true
                 }
             ],
             "chunkSize": 4000,
             "chunkOverlap": 200,
             "keepSeparator": true,
             "isRegex": false,
             "stripWhitespace": true,
         }, translate('SHAKER.SUGGESTIONS.COLUMN.FreeText.SPLIT_INTO_CHUNKS', actionVerb("Split") + " into chunks"));
         addStepBasedSuggestion(map, categoryText, "null", "TextSimplifierProcessor", {
             'inCol': columnName,
             'normalize' : true,
             'language' : 'english'
         }, translate('SHAKER.SUGGESTIONS.COLUMN.FreeText.SIMPLIFY', actionVerb("Simplify") + " text (normalize, stem, clear stop words)"));
     } else if (typeIs("BagOfWordsMeaning")) {
        addStepBasedSuggestion(map, categoryText, null, "ArrayFold", {
             "column" : columnName
        }, translate('SHAKER.SUGGESTIONS.COLUMN.Text.FOLD', actionVerb("Fold") + " to one word per line"));
        addStepBasedSuggestion(map, categoryText, null, "ArrayUnfold", {
             "column" : columnName,
             "prefix": columnName + "_",
             'countVal': true,
             'limit': 100,
             'overflowAction': 'ERROR'
        }, translate('SHAKER.SUGGESTIONS.COLUMN.Text.UNFOLD', actionVerb("Unfold") + " to several columns"));
    } else if (typeIs("JSONArrayMeaning")) {
        addStepBasedSuggestion(map, categoryText, null, "ArrayFold", {
             "column" : columnName
        }, translate('SHAKER.SUGGESTIONS.COLUMN.JSONArray.FOLD', actionVerb("Fold") + " to one element per line"));
        addStepBasedSuggestion(map, categoryText, null, "ArrayUnfold", {
             "column" : columnName,
             "prefix": columnName + "_",
             'countVal': true,
             'limit': 100,
             'overflowAction': 'ERROR'
        }, translate('SHAKER.SUGGESTIONS.COLUMN.JSONArray.UNFOLD', actionVerb("Unfold") + " to several columns"));
    } else if (typeIs("DoubleMeaning")) {
        addAppliesToSuggestion(map, categoryNumbers, null, "RoundProcessor", {
            "mode": "ROUND",
            "precision": 0,
            "places": 0
        }, translate('SHAKER.SUGGESTIONS.COLUMN.Double.ROUND', actionVerb("Round") + " to integer"), false);
    } else if (typeIs("FrenchDoubleMeaning")) {
        addAppliesToSuggestion(map, categoryNumbers, null, "NumericalFormatConverter", {
            inFormat: 'FR', outFormat: 'RAW'
        }, translate('SHAKER.SUGGESTIONS.COLUMN.FrenchDouble.CONVERT', actionVerb("Convert") + " French format to regular decimal"), true);
    } else if (typeIs("CurrencyAmountMeaning")) {
        addStepBasedSuggestion(map, categoryTransformations, "c1", "CurrencySplitter", {
            'inCol': column.name,
            'outColCurrencyCode' : column.name + "_currency_code",
            'outColAmount' : column.name + "_amount",
            'pristineAmount': false,
        }, translate('SHAKER.SUGGESTIONS.COLUMN.CurrencyAmount.SPLIT', actionVerb("Split") + " currency and amount"));
    }

    /* Suggestions for bad data */
    if (hasNOK) {
        if (!(forCell && forInvalidCell)) {
            addAppliesToSuggestion(map, categoryDataCleansing, "remove_badtype", "FilterOnBadType", {
                "action": "REMOVE_ROW", "type" : typeName, "booleanMode" : "AND"
            }, types.length === 1
                ? translate('SHAKER.SUGGESTIONS.COLUMN.RemoveBadType.REMOVE.SINGLE', actionVerb("Remove invalid") + " rows for meaning {{meaning}}", { meaning: meaningName(meaningLabelFilter(typeName))})
                : translate('SHAKER.SUGGESTIONS.COLUMN.RemoveBadType.REMOVE.MULTIPLE', actionVerb("Remove invalid") + " rows for meaning")
            );
            addAppliesToSuggestion(map, categoryDataCleansing, "clear_badtype", "FilterOnBadType", {
                "action": "CLEAR_CELL", "type" : typeName
            }, types.length === 1
                ? translate('SHAKER.SUGGESTIONS.COLUMN.RemoveBadType.CLEAR.SINGLE', actionVerb("Clear invalid") + " cells for meaning {{meaning}}", { meaning: meaningName(meaningLabelFilter(typeName))})
                : translate('SHAKER.SUGGESTIONS.COLUMN.RemoveBadType.CLEAR.MULTIPLE', actionVerb("Clear invalid") + " cells for meaning")
            );
        }
        addStepBasedSuggestion(moreMap, categoryDataCleansing, "split_badtype", "SplitInvalidCells", {
            "column" : columnName, "type" : typeName,
            "invalidColumn" : columnName + "_invalid"
        }, types.length === 1
            ? translate('SHAKER.SUGGESTIONS.COLUMN.SplitBadType.SPLIT.SINGLE', "Move invalid cells for meaning {{meaning}} to <em>{{multi ? '&lt;column name&gt;' : columnName}}_invalid</em>", { meaning: meaningName(meaningLabelFilter(typeName)), multi: multi, columnName: columnName })
            : translate('SHAKER.SUGGESTIONS.COLUMN.SplitBadType.SPLIT.MULTIPLE', "Move invalid cells for meaning to <em>{{multi ? '&lt;column name&gt;' : columnName}}_invalid</em>", { multi: multi, columnName: columnName })
        );
    }

    if (types.length === 1 && appConfig.meanings.categories.filter(cat => cat.label === "User-defined")[0].meanings.filter(mapping => mapping.id === typeName && mapping.type === 'VALUES_MAPPING').length) {
        addAppliesToSuggestion(map, categoryTransformations, "translate_meaning", "MeaningTranslate", { "meaningId": typeName },
            translate('SHAKER.SUGGESTIONS.COLUMN.MeaningTranslate.TRANSLATE', actionVerb("Translate") + " using meaning {{meaning}}", { meaning: meaningName(typeName) }), false);
    }

    /* Now, if at this point we have 0 items in the main suggestions,
     * we add some that would normally go to more */

    if (hasEmpty) {
        target = Object.keys(map).length > 0 ? moreMap : map;
        addAppliesToSuggestion(target, categoryDataCleansing, "rm_empty", "RemoveRowsOnEmpty", {"keep": false},
            translate('SHAKER.SUGGESTIONS.COLUMN.RemoveRowsOnEmpty.REMOVE', actionVerb("Remove") + " rows with no value"), false);

        addCustomSuggestion(target, categoryDataCleansing, "fill_empty_withval", function(scope) {
            CreateModalFromDOMElement("#fill-empty-with-value-box", scope, "FillEmptyWithValueController",
                function(newScope) { newScope.$apply(function() {
                        newScope.setColumns(columnNames);
                        newScope.isNumericOnly = ["LongMeaning", "DoubleMeaning"]
                            .indexOf(column.selectedType && column.selectedType.name) > -1;
                }); });
        }, translate('SHAKER.SUGGESTIONS.COLUMN.FillEmptyWithVal.FILL', actionVerb("Fill") + " empty rows with..."),
        ShakerProcessorsUtils.getStepIcon('FillEmptyWithValue', {}, '16'));

        addStepBasedSuggestion(moreMap, categoryDataCleansing, "fill_empty_prev", "UpDownFiller", {
            "columns" : [columnName],
            "up" : false
        }, translate('SHAKER.SUGGESTIONS.COLUMN.FillEmptyWithVal.FILL_WITH_PREVIOUS', actionVerb("Fill") + " empty rows with previous value"), true);
    }

    if (typesAllIn(["LongMeaning", "DoubleMeaning"])) {
        target = Object.keys(map).length > 2 ? moreMap : map;
        if (multi) {
            addCustomSuggestion(target, categoryKeepDelete, "numerical_range_selector", function(scope) {
                report("shaker-use-col-sugg", "FilterOnNumericalRange", "Delete/Keep on number range...")();
                CreateModalFromDOMElement("#value-range-filter-box", scope, "MultiRangeController",
                    function(newScope) { newScope.$apply(function() {
                            newScope.setColumns(columnNames);
                    }); });
            }, translate('SHAKER.SUGGESTIONS.COLUMN.FilterOnNumericalRange.FILTER', actionVerb("Filter") + " with numerical range..."),
            ShakerProcessorsUtils.getStepIcon('FilterOnNumericalRange', {}, '16')
        );
        } else {
            addStepBasedSuggestion(target, categoryKeepDelete, null, "FilterOnNumericalRange", {
                    "appliesTo" : "SINGLE_COLUMN",
                    "columns": [columnName],
                    "min": undefined,
                    "max" : undefined,
                    "action" : "KEEP_ROW",
                }, translate('SHAKER.SUGGESTIONS.COLUMN.FilterOnNumericalRange.FILTER', actionVerb("Filter") + " with numerical range..."), true);
        }
    }

    if (noTypeIn(["LongMeaning", "DoubleMeaning", "Date"])) {
        target = Object.keys(map).length > 0 ? moreMap : map;

        addAppliesToSuggestion(target, categoryTransformations, null, "StringTransformer", {
                "mode" : "TO_LOWER"
        }, translate('SHAKER.SUGGESTIONS.COLUMN.StringTransformer.TO_LOWER', actionVerb("Convert") + " to lowercase"));
        addAppliesToSuggestion(target, categoryTransformations, null, "StringTransformer", {
                "mode" : "TO_UPPER"
        }, translate('SHAKER.SUGGESTIONS.COLUMN.StringTransformer.TO_UPPER', actionVerb("Convert") + " to uppercase"));
        addAppliesToSuggestion(target, categoryTransformations, null, "StringTransformer", {
                "mode": "TO_LOWER"
        }, translate('SHAKER.SUGGESTIONS.COLUMN.StringTransformer.TRANSFORM', actionVerb("Transform") + " string"), true);

    }
    /* At this point, we should always have something in the main map */

    /* Add the rest of type-based tranformations to the more map */

    if (typeIs("Date")) {
        addStepBasedSuggestion(moreMap, categoryDate, "extract_ts", "DateComponentsExtractor", {
                "column" : columnName,
                "timezone_id" : "UTC",
                "outTimestampColumn" : columnName + "_timestamp",
        }, translate('SHAKER.SUGGESTIONS.COLUMN.Date.CONVERT_TO_UNIX', actionVerb("Convert") + " to UNIX timestamp"));
        addStepBasedSuggestion(moreMap, categoryDate, "holidays", "HolidaysComputer", {
            "inCol" : columnName,
            "outColPrefix" : columnName+'_holiday_',
            "flagBankHolidays":true,
            "calendar_id":"FR",
            "timezone_id":"use_preferred_timezone",
            "flagWeekends":true,
            "flagSchoolHolidays":true
        }, translate('SHAKER.SUGGESTIONS.COLUMN.Date.FLAG_HOLIDAYS', actionVerb("Flag") + " holidays"));
        addStepBasedSuggestion(moreMap, categoryDate, "date_format", "DateFormatter", {
            "inCol": columnName,
            "outCol": columnName + "_formatted",
            "lang" : "en_US",
            "timezone_id" : "UTC",
            "format": "yyyy-MM-dd HH:mm:ss"
        }, translate('SHAKER.SUGGESTIONS.COLUMN.Date.REFORMAT', actionVerb("Reformat") + " date"));
    }

    /* Text also gets some text handling, but in more Map unlike FreeText */
    if (typeIs("Text")) {
        addStepBasedSuggestion(moreMap, categoryText, null, "Tokenizer", {
             'inCol': columnName,
             'operation' : 'TO_JSON',
             'language' : 'english'
         }, translate('SHAKER.SUGGESTIONS.COLUMN.Text.TOKENIZE', actionVerb("Split") + " in words (tokenize)"));
         addStepBasedSuggestion(moreMap, categoryText, null, "TextSimplifierProcessor", {
             'inCol': columnName,
             'normalize':true,
             'language' : 'english'
         }, translate('SHAKER.SUGGESTIONS.COLUMN.Text.NORMALIZE', actionVerb("Normalize") + " text"));
    }

    addStepBasedSuggestion(moreMap, categoryTransformations, null, "ColumnCopier", {
        "inputColumn": columnName,
        "outputColumn": columnName+"_copy"
    }, translate('SHAKER.SUGGESTIONS.COLUMN.DUPLICATE', actionVerb("Duplicate") + " column"));

    if (!multi) {
        addAppliesToSuggestion(moreMap, categoryTransformations, null, "FindReplace", {
            "output": "",
            "mapping": [],
            "matching" : "FULL_STRING",
            "normalization" : "EXACT"
        }, translate('SHAKER.SUGGESTIONS.COLUMN.FIND_AND_REPLACE', actionVerb("Find") + " and " + actionVerb("replace") + "..."), true);
        addStepBasedSuggestion(
            ["DoubleMeaning", "LongMeaning"].indexOf(typeName) === -1 ? moreMap : map,
            categoryNumbers, null, "CreateColumnWithGREL",{
                "column" : columnName,
                "expression" : columnName.match(/^[a-z0-9_]+$/i) ? columnName : `val("${columnName}")`
            }, translate('SHAKER.SUGGESTIONS.COLUMN.PROCESS_WITH_FORMULA', actionVerb("Process") + " with formula..."));
    } else {
        addCustomSuggestion(moreMap, categoryTransformations, null, function(shakerScope) {
            report("shaker-use-col-sugg", "ColumnsConcat", "Concatenate columns")();
            shakerScope.addUnconfiguredStep("ColumnsConcat",
                { columns: columnNames, join: ',' });
            shakerScope.autoSave();
        }, translate('SHAKER.SUGGESTIONS.COLUMN.CONCAT_COLUMNS', actionVerb("Concatenate") + " columns"),
        ShakerProcessorsUtils.getStepIcon('ColumnsConcat', {}, '16'));
        addCustomSuggestion(moreMap, categoryTransformations, null, function(shakerScope) {
            report("shaker-use-col-sugg", "NestProcessor", "Nest columns")();
            shakerScope.addUnconfiguredStep("NestProcessor",
                { columns: columnNames, join: ',', appliesTo: 'COLUMNS' });
            shakerScope.autoSave();
        }, translate('SHAKER.SUGGESTIONS.COLUMN.NEST_TO_OBJECT', actionVerb("Nest") + " columns to object"),
        ShakerProcessorsUtils.getStepIcon('FilterOnNumericalRange', {}, '16'));
        addCustomSuggestion(moreMap, categoryTransformations, null, function(shakerScope) {
            report("shaker-use-col-sugg", "MultiColumnFold", "Fold columns")();
            shakerScope.addUnconfiguredStep("MultiColumnFold",
                { columns: columnNames, join: ',' });
            shakerScope.autoSave();
        }, translate('SHAKER.SUGGESTIONS.COLUMN.FOLD_COLUMNS', actionVerb("Fold ") + " columns"),
        ShakerProcessorsUtils.getStepIcon('MultiColumnFold', {}, '16'));
        addCustomSuggestion(moreMap, categoryTransformations, null, function(shakerScope) {
            report("shaker-use-col-sugg", "FindReplace", "Find & replace")();
            shakerScope.addUnconfiguredStep("FindReplace",
                { columns: columnNames, join: ',', appliesTo: 'COLUMNS'});
            shakerScope.autoSave();
        }, translate('SHAKER.SUGGESTIONS.COLUMN.FIND_AND_REPLACE_IN_COLUMNS', actionVerb("Find") + " and " + actionVerb("replace") + " in columns"),
        ShakerProcessorsUtils.getStepIcon('FindReplace', {}, '16'));
    }

    if (Object.keys(map).length === 0) { // shift
        map = moreMap;
        moreMap = {};
    }
    return [map, moreMap, Object.keys(moreMap).length];
};

return svc;
});

})();

;
(function() {
'use strict';

/* Base directives for the "exploration-only" part of shaker */

// Base_explore is the first loaded and creates the module.
const app = angular.module('dataiku.shaker', ['dataiku.filters', 'dataiku.shared', 'platypus.utils']);

app.directive("shakerExploreBase", function(Logger, $filter, $rootScope, $translate, translate) {
    return {
        scope: true,
        priority : 100,
        controller : function($scope, $rootScope, $stateParams, $state, DataikuAPI, CachedAPICalls, $filter, CreateModalFromTemplate,
                              WT1, ActivityIndicator, $timeout, $q, Debounce, MonoFuture, GraphZoomTrackerService, ActiveProjectKey,
                              Dialogs, FutureProgressModal, computeColumnWidths, SmartId, SamplingData, ChartsStaticData, RatingFeedbackParams, 
                              ConditionalFormattingEditorService, ClipboardUtils) {
            $scope.isRecipe = false;
            let lastValidatedSteps;

            $scope.ratingFeedbackParams = RatingFeedbackParams;

            //This is necessary to make sure  the banner disappears when we route to another page (otherwise, it appears again after we open a recipe)
            $scope.$on('$stateChangeStart', (event, toState, toParams, fromState) => {
                if (fromState.name !== 'projects.project.recipes.recipe') {
                    // Only alter the rating feedback banner if what we are leaving is NOT a recipe (mainly because the prepare recipe also use this directive)
                    // For the recipes the alteration is done accordingly in the RecipeEditorController
                    $scope.ratingFeedbackParams.showRatingFeedback = false;
                }
            });

            if (!$stateParams.fromFlow) {
                // Do not change the focus item zoom if coming from flow
                GraphZoomTrackerService.setFocusItemByName("dataset", $stateParams.datasetName);
            }

            $scope.RIGHT_PANE_VIEW = {
                NONE: 0,
                QUICK_COLUMNS_VIEW: 1,
                QUICK_CONDITIONAL_FORMATTING_VIEW: 2
            };

            $scope.isQuickColumnsViewOpened = function () {
                return $scope.shakerState.rightPaneView === $scope.RIGHT_PANE_VIEW.QUICK_COLUMNS_VIEW;
            }
            $scope.isQuickConditionalFormattingViewOpened = function () {
                return $scope.shakerState.rightPaneView === $scope.RIGHT_PANE_VIEW.QUICK_CONDITIONAL_FORMATTING_VIEW;
            }
            // rightPaneView is a value from the "enum" $scope.RIGHT_PANE_VIEW
            $scope.openRightPaneWith = function (rightPaneView) {
                $scope.shakerState.rightPaneView = rightPaneView;
            }

            $scope.conditionalFormattingState = {
                columns: [],
                isWholeData: false,
                openedFromColumnHeader: null,
                tableHeaders: [],
            };

            $scope.shakerState = {
                activeView: 'table',
                // For now only 1 view can be opened on the right panel
                rightPaneView: $scope.RIGHT_PANE_VIEW.NONE,

                lockedHighlighting : [],
                // this is a row selected, now used for the compare feature, the row is highlighted in blue
                selectedRow: null,
            };

            updateColoringStates();

            // Real controller inserts its hooks here
            $scope.shakerHooks = {
                isMonoFuturizedRefreshActive: function(){}, // NOSONAR: OK to have an empty function

                // Returns a promise when save is done
                saveForAuto: undefined,
                // Returns a promise that resolves with a future for the refresh
                getRefreshTablePromise: undefined,

                // Sets the meaning of a column
                setColumnMeaning : undefined,

                // Sets the storage type of a column
                getSetColumnStorageTypeImpact : undefined,
                setColumnStorageType : undefined,

                // Should open a box to edit the details of the column
                // (meaning, storage type, description)
                editColumnDetails : undefined,

                // Hook called in parallel to the table refresh
                onTableRefresh : function(){}, // NOSONAR: OK to have an empty function that does nothing by default
                // Hook called after the table refresh
                afterTableRefresh : function(){}, // NOSONAR: OK to have an empty function that does nothing by default

                // analysis modal :
                // - fetch the detailed analysis of a column
                fetchDetailedAnalysis : undefined,
                // - get clusters
                fetchClusters : undefined,
                // - compute text analysis
                fetchTextAnalysis : undefined
            };

            Mousetrap.bind("r s", function(){
                DataikuAPI.shakers.randomizeColors();
                $scope.refreshTable(false);
            })

            Mousetrap.bind("alt+a", function(){
                $scope.$apply(function(){
                    $scope.shaker.exploreUIParams.autoRefresh = !$scope.shaker.exploreUIParams.autoRefresh;
                    ActivityIndicator.success("Auto refresh is now " +
                        ($scope.shaker.exploreUIParams.autoRefresh ? "enabled" : "disabled"));
                    if ($scope.shaker.exploreUIParams.autoRefresh) {
                        $scope.autoSaveAutoRefresh();
                    }
                });
            })

            $scope.$on("$destroy", function(){
                Mousetrap.unbind("r s");
                Mousetrap.unbind("alt+a")
            })

            function id(dataset) {
                //if the current dataset is foreign, force the use of full dataset names (equi-joiner for example requires it)
                if ($scope.inputDatasetProjectKey != $stateParams.projectKey) {
                    return dataset.projectKey + '.' + dataset.name;
                } else {
                    return dataset.smartName;
                }
            }

            $scope.datasetHref = function() {
                if (!$scope.dataset) {return ''}
                return $state.href('projects.project.datasets.dataset.explore', {datasetName: $scope.dataset.name});
            }
            /** Called by the real controller to fetch required data once context has been set */
            $scope.baseInit = function(displaySpinner = true) {
                if ($scope.inputDatasetName) {
                    if ($rootScope.topNav.isProjectAnalystRO) {
                        DataikuAPI.datasets.getFullInfo($stateParams.projectKey, $scope.inputDatasetProjectKey, $scope.inputDatasetName).spinner(displaySpinner).success(function(data){
                            $scope.datasetFullInfo = data;
                        }).error(setErrorInScope.bind($scope));

                        DataikuAPI.datasets.get($scope.inputDatasetProjectKey, $scope.inputDatasetName, $stateParams.projectKey).spinner(displaySpinner)
                        .success(function(data) {
                            $scope.dataset = data;
                        }).error(setErrorInScope.bind($scope));
                        var opts = {
                            datasetsOnly : true
                        };
                        DataikuAPI.flow.listUsableComputables($stateParams.projectKey, opts).spinner(displaySpinner).success(function(computables) {
                            $scope.datasetNames = $.map(computables, function(val) {
                                return id(val);
                            });
                        }).error(setErrorInScope.bind($scope));

                        CachedAPICalls.datasetTypes.then(function(datasetTypes) {
                            $scope.dataset_types = datasetTypes;
                        }).catch(setErrorInScope.bind($scope));
                        $scope.datasetColumns = {};
                        $scope.getDatasetColumns = function(datasetId) { // datasetId is something id() would return
                            // only for input datasets. Only once (we don't care if the schema is changing while we edit the shaker)
                            if ($scope.datasetNames && $scope.datasetNames.indexOf(datasetId) >= 0 && !(datasetId in $scope.datasetColumns)) {
                                let resolvedSmartId = SmartId.resolve(datasetId, $stateParams.projectKey);
                                $scope.datasetColumns[datasetId] = [];
                                DataikuAPI.datasets.get(resolvedSmartId.projectKey, resolvedSmartId.id, $stateParams.projectKey).spinner(displaySpinner).success(function(dataset){
                                    $scope.datasetColumns[datasetId] = $.map(dataset.schema.columns, function(el) {
                                        return el.name;
                                    });
                                    // and let the digest update the UI...
                                }).error(setErrorInScope.bind($scope));
                            }
                            return $scope.datasetColumns[datasetId];
                        };
                    }
                }

                CachedAPICalls.processorsLibrary.success(function(processors){
                    $scope.processors = processors;
                }).error(setErrorInScope.bind($scope));

            }


            /** Real handler calls this once $scope.shaker is set to fixup incomplete scripts */

            $scope.fixupShaker = function(shaker) {
                shaker = shaker || $scope.shaker;
                if (shaker.exploreUIParams == null) {
                    shaker.exploreUIParams = {};
                }
                if (shaker.exploreUIParams.autoRefresh == null) {
                    shaker.exploreUIParams.autoRefresh = true;
                }
                if (shaker.explorationFilters == null) {
                    shaker.explorationFilters = [];
                }
            }

            $scope.$watch("shaker.steps", function(nv, ov){
                if (!nv) return;
                 function _addChange(s) {
                    if (s.metaType == "GROUP") {
                        if (s.steps) {
                            s.steps.forEach(_addChange);
                        }
                    }
                    if (s.$stepState == null) {
                        s.$stepState = {}
                    }
                }
                $scope.shaker.steps.forEach(_addChange);
            }, true);

            $scope.invalidScriptError = {};

            $scope.forgetSample = function() {
                $scope.requestedSampleId = null;
            };

            $scope.computeFullRowCount = function() {
                // Only force recomputing the whole row count if we are sure we are not on a partitioned dataset.
                // On a partitioned dataset, we only want to recompute the partitions that may have been updated.
                const forceRecompute = Boolean($scope.datasetFullInfo && !$scope.datasetFullInfo.partitioned);
                DataikuAPI.datasets.getRefreshedSummaryStatus($scope.inputDatasetProjectKey, $scope.inputDatasetName, true, forceRecompute).success(function(data) {
                    FutureProgressModal.show($scope, data, "Computing row count").then(function(result){
                        if ($scope.datasetFullInfo) {
                            $scope.datasetFullInfo.dataset.status = result;
                        }
                        if (result) {
                            Dialogs.infoMessagesDisplayOnly($scope, "Row count updated", result.messages);
                            if (result.records && result.records.hasData && !result.records.incomplete && $scope.table && $scope.table.sampleMetadata) {
                                $scope.table.sampleMetadata.datasetRecordCount = result.records.totalValue;
                                $scope.table.sampleMetadata.recordCountIsObsolete = false;
                                $scope.table.sampleMetadata.recordCountIsApproximate = false;
                                $rootScope.$broadcast('rightPanelSummary.triggerFullInfoUpdate'); // Ensure that the right panel updates itself with the latest information
                            }
                        }
                    });
                }).error(setErrorInScope.bind($scope));
            };

            $scope.showComputeFullRowCountButton = function() {
                return $scope.table
                    && $scope.table.sampleMetadata
                    && !$scope.table.sampleMetadata.sampleIsWholeDataset
                    && ($scope.table.sampleMetadata.datasetRecordCount === -1
                        || $scope.table.sampleMetadata.recordCountIsApproximate
                        || $scope.table.sampleMetadata.recordCountIsObsolete)
                    && (ActiveProjectKey.get() === $scope.inputDatasetProjectKey); // No refresh button on foreign datasets because user might not have the permissions to compute metrics on this dataset
            }

            $scope.saveOnly = function() {
                $scope.shakerHooks.saveForAuto().then(function(){
                    ActivityIndicator.success("Changes saved");
                });
            };

            $scope.getSampleConfigDesc = function() {
                if ($scope.shaker && $scope.shaker.explorationSampling && $scope.shaker.explorationSampling.selection) {
                    const selection = $scope.shaker.explorationSampling.selection;
                    return SamplingData.formatSamplingConfig(selection.samplingMethod, selection.maxRecords, selection.targetRatio);
                }
                return "-";
            }

            $scope.switchToSamplingParamsTab = function() {
                $rootScope.$broadcast('tabSelect', 'sample-settings')
            }

            $scope.getSampleDesc = function() {
                function formatSamplingRowCountRestrictions(sampleMetadata) {
                    const spacer = " ";
                    if (sampleMetadata.partitionCount > 0) {
                        const params = { partitionCount: sampleMetadata.partitionCount };
                        if (sampleMetadata.hasFilter) {
                            return translate("SHAKER.SAMPLING.ACTUAL.FILTERED_WITH_PARTITION", "(filtered & {{partitionCount}} {{partitionCount == 1 ? 'partition' : 'partitions'}})", params) + spacer;
                        } else {
                            return translate("SHAKER.SAMPLING.ACTUAL.WITH_PARTITION", "({{partitionCount}} {{partitionCount == 1 ? 'partition' : 'partitions'}})", params) + spacer;
                        }
                    } else if (sampleMetadata.hasFilter) {
                        return translate("SHAKER.SAMPLING.ACTUAL.FILTERED", "(filtered)") + spacer;
                    }
                    return "";
                }

                const table = $scope.table;
                if (!table) {
                    return "";
                }

                const sampleMetadata = table.sampleMetadata;
                let strRow = 'row' + (table.initialRows === 1 ? '': 's');
                if (sampleMetadata && !sampleMetadata.sampleIsWholeDataset) {
                    const params = {
                        rowCount: table.initialRows,
                        smartRowCount: $filter('longSmartNumber')(table.initialRows),
                        rowCountRestrictions: formatSamplingRowCountRestrictions(sampleMetadata),
                        totalRowCount: sampleMetadata.datasetRecordCount === -1
                            ? translate('SHAKER.SAMPLING.ACTUAL.TOTAL_NOT_COMPUTED', "not computed")
                            : $filter('longSmartNumber')(Math.max(table.initialRows, sampleMetadata.datasetRecordCount)),
                        totalRowCountEstimated: sampleMetadata.recordCountIsApproximate
                            ? (" " + translate('SHAKER.SAMPLING.ACTUAL.TOTAL_ESTIMATED', "(estimated)"))
                            : ""
                    };
                    switch (sampleMetadata.samplingMethod) {
                        case "HEAD_SEQUENTIAL":
                            return translate("SHAKER.SAMPLING.ACTUAL.HEAD_SEQUENTIAL", `First <strong>{{smartRowCount}}</strong> {{rowCount == 1 ? 'row': 'rows'}} {{rowCountRestrictions}}out of <strong>{{totalRowCount}}</strong>{{totalRowCountEstimated}}`, params);
                        case "TAIL_SEQUENTIAL":
                            return translate("SHAKER.SAMPLING.ACTUAL.TAIL_SEQUENTIAL", `Last <strong>{{smartRowCount}}</strong> {{rowCount == 1 ? 'row': 'rows'}} {{rowCountRestrictions}}out of <strong>{{totalRowCount}}</strong>{{totalRowCountEstimated}}`, params);
                        case "RANDOM_FIXED_NB":
                        case "COLUMN_BASED":
                        case "STRATIFIED_TARGET_NB_EXACT":
                        case "CLASS_REBALANCE_TARGET_NB_APPROX":
                        case "RANDOM_FIXED_NB_EXACT":
                            return translate("SHAKER.SAMPLING.ACTUAL.RANDOM", `<strong>{{smartRowCount}}</strong> random {{rowCount == 1 ? 'row': 'rows'}} {{rowCountRestrictions}}out of <strong>{{totalRowCount}}</strong>{{totalRowCountEstimated}}`, params);
                        case "RANDOM_FIXED_RATIO":
                        case "STRATIFIED_TARGET_RATIO_EXACT":
                        case "CLASS_REBALANCE_TARGET_RATIO_APPROX":
                        case "RANDOM_FIXED_RATIO_EXACT":
                            return translate("SHAKER.SAMPLING.ACTUAL.RANDOM", `<strong>{{smartRowCount}}</strong> random {{rowCount == 1 ? 'row': 'rows'}} {{rowCountRestrictions}}out of <strong>{{totalRowCount}}</strong>{{totalRowCountEstimated}}`, params);
                        case "FULL":
                        default:
                            return translate("SHAKER.SAMPLING.ACTUAL.FULL", `<strong>{{smartRowCount}}</strong> {{rowCount == 1 ? 'row': 'rows'}} {{rowCountRestrictions}}out of <strong>{{totalRowCount}}</strong>{{totalRowCountEstimated}}`, params);
                    }
                } else {
                    const params = {
                        rowCount: table.initialRows,
                        smartRowCount: $filter('longSmartNumber')(table.initialRows),
                        rowCountRestrictions: formatSamplingRowCountRestrictions(sampleMetadata)
                    };
                    return translate("SHAKER.SAMPLING.ACTUAL.WHOLE", `<strong>{{smartRowCount}}</strong> {{rowCount == 1 ? 'row': 'rows'}} {{rowCountRestrictions}}`, params);
                }
            }

            $scope.getSampleTooltip = function() {
                const table = $scope.table;
                if (!table) {
                    return "";
                }
                let result = $scope.getSampleDesc();
                if (table.sampleMetadata && table.sampleMetadata.memoryLimitReached) {
                    result += `<br><strong>${table.sampleMetadata.memoryLimitInMB}</strong> MB memory limit reached`
                }
                return result;
            }

            $scope.getSampleDescSimple = function() {
                const table = $scope.table;
                if (!table) {
                    return "";
                }
                return SamplingData.formatSamplingConfig("FULL", table.totalRows);
            }

            $scope.$on('refresh-table',function() {
                $scope.autoSaveForceRefresh();
            });

            /* Save if auto-save is enabled and force a refresh */
            $scope.autoSaveForceRefresh = function() {
                if (!$scope.isRecipe && ($scope.canWriteProject() || $scope.isLocalSave)) {
                    $scope.shakerHooks.saveForAuto().catch(error => {
                        Logger.error("Could not save", { error });
                    });
                }
                $scope.refreshTable(false);
            };

            $scope.autoSave = function() {
                if (!$scope.isRecipe && ($scope.canWriteProject() || $scope.isLocalSave)) {
                    $scope.shakerHooks.saveForAuto();
                }
            };

            // returns relevant shaker data, a fat-free data only object
            // without the change information.
            $scope.getShakerData = function() {
                // get only own property stuff.
                if ($scope.shaker == undefined)  {
                    return undefined;
                }

                function clearOne(step) {
                    if (step.metaType == "GROUP") {
                        step.steps.forEach(clearOne);
                    } else {
                        delete step.$stepState;
                        delete step.$$hashKey;
                    }
                }

                var shakerData = JSON.parse(JSON.stringify($scope.shaker));
                shakerData.steps.forEach(clearOne);
                return shakerData;
            };

            // formerShakerData is supposed to hold the last shaker state for which we updated
            // the table.
            $scope.setFormerShakerData = function() {
                $scope.formerShakerData = $scope.getShakerData();
            }

            /* Save if auto-save is enabled and refresh if auto-refresh is enabled */
            $scope.autoSaveAutoRefresh = function() {
                var shakerData = $scope.getShakerData();

                if (angular.equals(shakerData, $scope.formerShakerData)) {
                    // nothing has changed, we don't have to do this.
                    return;
                }

                $scope.autoRefreshDirty = true;
                if ($scope.isRecipe){
                    if ($scope.shaker.exploreUIParams.autoRefresh && $scope.recipeOutputSchema) {
                        // Only call refreshTable when recipeOutputSchema is set.
                        // If it is not set yet, it will be set later when the DataikuAPI.datasets.get() call in recipe.js will succeed
                        // and refreshTable will called then.
                        $scope.refreshTable(false);
                    }
                } else {
                    if ($scope.shaker.exploreUIParams.autoRefresh) {
                        $scope.shakerHooks.saveForAuto();
                        $scope.refreshTable(false);
                    } else {
                        $scope.saveOnly();
                        $scope.autoRefreshDirty = true;
                        $scope.setFormerShakerData();
                    }
                }
            };

            $scope.$on("overrideTableUpdated", function(){
                $scope.autoSaveAutoRefresh();
            });

            function clearBackendErrors(step) {
                step.$stepState.change = null;
                step.$stepState.backendError = null;
                if (step.metaType == "GROUP") {
                    step.steps.forEach(clearBackendErrors);
                }
            }

            function mergeChanges(step, change) {
                step.$stepState.change = change;
                step.designTimeReport = change.recordedReport;
                if (step.metaType == "GROUP") {
                    step.steps.forEach(function(substep, i){
                        if (change.groupStepsChanges && change.groupStepsChanges[i]){
                            mergeChanges(substep, change.groupStepsChanges[i]);
                        } else {
                            substep.$stepState.change = null;
                            step.designTimeReport = null;
                        }
                    });
                }
            }
            function mergeBackendErrors(step, errHolder) {
                if (errHolder.error) {
                    step.$stepState.backendError = errHolder.error;
                } else {
                    step.$stepState.backendError = null;
                }
                if (step.metaType === "GROUP") {
                    step.steps.forEach(function(substep, i) {
                        if (errHolder.children && errHolder.children.length > i && errHolder.children[i] != null) {
                            mergeBackendErrors(substep, errHolder.children[i]);
                            step.$stepState.backendError = substep.$stepState.backendError;
                        } else {
                            substep.$stepState.backendError = null;
                        }
                    });
                }
            }

            $scope.onRefreshFutureDone = function(filtersOnly) {
                $scope.shakerState.runError = null;
                $scope.shakerState.initialRefreshDone = true;
                $scope.requestedSampleId = $scope.future.result.usedSampleId;
                $scope.invalidScriptError = {};

                $scope.shakerState.lockedHighlighting = [];
                $scope.shakerState.selectedRow = null;

                $scope.table = $scope.future.result;

                $scope.setSpinnerPosition(undefined);
                $scope.lastRefreshCallTime = (new Date().getTime()-$scope.refreshCallBeg);
                if ($scope.updateFacetData) {
                    $scope.updateFacetData();
                }

                $scope.shaker.columnsSelection = $scope.table.newColumnsSelection;

                $scope.shaker.steps.forEach(function(step, i){
                    if ($scope.table.scriptChange.groupStepsChanges[i] != null) {
                        mergeChanges(step, $scope.table.scriptChange.groupStepsChanges[i]);
                    }
                })

                $scope.shakerState.hasAnyComment = false;
                $scope.shakerState.hasAnyCustomFields = false;

                var getNoFakeExtremeDoubleDecimalPercentage = function(numerator, denominator) {
                    var result = numerator * 10000 / denominator;
                    switch (Math.round(result)) {
                        case 0:
                            result = result == 0 ? 0 : 1;
                            break;
                        case 10000:
                            result = result == 10000 ? 10000 : 9999;
                            break
                        default:
                            result = Math.round(result);
                    }
                    return result / 100;
                }

                $scope.columns = $.map($scope.table.headers, function(header) {
                    if (header.selectedType) {
                        header.selectedType.totalCount = (header.selectedType.nbOK + header.selectedType.nbNOK + header.selectedType.nbEmpty);
                        header.okPercentage = getNoFakeExtremeDoubleDecimalPercentage(header.selectedType.nbOK, header.selectedType.totalCount);
                        header.emptyPercentage = !header.selectedType.nbEmpty ? 0 : getNoFakeExtremeDoubleDecimalPercentage(header.selectedType.nbEmpty, header.selectedType.totalCount);
                        header.nonemptyPercentage = header.selectedType.nbEmpty == null ? 0 : getNoFakeExtremeDoubleDecimalPercentage(header.selectedType.totalCount - header.selectedType.nbEmpty, header.selectedType.totalCount);
                        header.nokPercentage = !header.selectedType.nbNOK ? 0 : getNoFakeExtremeDoubleDecimalPercentage(header.selectedType.nbNOK, header.selectedType.totalCount);

                        if (header.deletedMeaningName) {
                            header.meaningLabel = header.deletedMeaningName + ' (deleted)';
                        } else {
                            header.meaningLabel = $filter('meaningLabel')(header.selectedType.name);
                        }
                    }

                    /* Check if this column has a comment */
                    if (header.recipeSchemaColumn && header.recipeSchemaColumn.column.comment) {
                        $scope.shakerState.hasAnyComment = true;
                        header.comment = header.recipeSchemaColumn.column.comment
                    }
                    if (header.datasetSchemaColumn && header.datasetSchemaColumn.comment) {
                        $scope.shakerState.hasAnyComment = true;
                        header.comment = header.datasetSchemaColumn.comment
                    }
                    if ($scope.shaker.origin == "ANALYSIS" &&
                       $scope.shaker.analysisColumnData[header.name] &&
                       $scope.shaker.analysisColumnData[header.name].comment) {
                        $scope.shakerState.hasAnyComment = true;
                        header.comment = $scope.shaker.analysisColumnData[header.name].comment;
                    }

                    // This part propagates the column comment coming from the input dataset
                    if (header.recipeSchemaColumn?.column != null && !header.recipeSchemaColumn.column.isColumnEdited) {
                        const columns = $scope.dataset?.schema?.columns || null;
                        if (columns) {
                            const columnFromInput = columns.find(col => col.name === header.name);
                            const newComment = columnFromInput?.comment;
                            if (newComment && header.comment !== newComment) {
                                $scope.shakerState.hasAnyComment = true;
                                header.comment = newComment;
                                header.recipeSchemaColumn.column.comment = newComment;
                                $scope.schemaDirtiness.dirty = true;
                            }
                        }
                    }

                    /* Check if this column has preview custom fields */
                    function addCustomFieldsPreviews(customFields) {
                        const ret = [];
                        const customFieldsMap = $rootScope.appConfig.customFieldsMap['COLUMN'];
                        for (let i = 0; i < customFieldsMap.length; i++) {
                            const selectCFList = (customFieldsMap[i].customFields || []).filter(cf => cf.type == 'SELECT');
                            for (let j = 0; j < selectCFList.length; j++) {
                                const cfDef = selectCFList[j];
                                const value = (cfDef.selectChoices || []).find(choice => choice.value == (customFields && customFields[cfDef.name] || cfDef.defaultValue));
                                if (value && value.showInColumnPreview) {
                                    ret.push({definition: cfDef, value: value});
                                }
                            }
                        }
                        return ret;
                    }
                    $scope.customFieldsMap = $rootScope.appConfig.customFieldsMap['COLUMN'];
                    if (header.recipeSchemaColumn) {
                        header.customFields = header.recipeSchemaColumn.column.customFields;
                    }
                    if (header.datasetSchemaColumn) {
                        header.customFields = header.datasetSchemaColumn.customFields;
                    }
                    if ($scope.shaker.origin == "ANALYSIS" &&
                        $scope.shaker.analysisColumnData[header.name]) {
                        header.customFields = $scope.shaker.analysisColumnData[header.name].customFields;
                    }
                    const cfPreviews = addCustomFieldsPreviews(header.customFields);
                    if (cfPreviews.length > 0) {
                        $scope.shakerState.hasAnyCustomFields = true;
                        header.customFieldsPreview = cfPreviews;
                    }

                    return header.name;
                });
                if ($scope.shakerState.activeView === 'table') {
                    $scope.setQuickColumns();
                    $scope.clearQuickColumnsCache();
                }
                if ($scope.isRecipe && $scope.table.newRecipeSchema) {
                    $scope.recipeOutputSchema = $scope.table.newRecipeSchema;
                }
                $scope.$broadcast("shakerTableChanged");
                $rootScope.$broadcast("shakerTableChangedGlobal", $scope.shaker);

                getDigestTime($scope, function(time) {
                    $scope.lastRefreshDigestTime = time;
                    updateColoringStates();
                    $scope.$broadcast("reflow");
                    WT1.event("shaker-table-refreshed", {
                        "activeFFs" : $scope.shaker.explorationFilters.length,
                        "backendTime" : $scope.lastRefreshCallTime,
                        "digestTime" : time,
                        "numCols" : $scope.table.headers.length,
                        "totalKeptRows" : $scope.table.totalKeptRows,
                        "totalRows" : $scope.table.totalRows
                    });
                });
            };
            $scope.onRefreshFutureFailed = function(data, status, headers) {
                $scope.shakerState.runError = null;
                $scope.shakerState.initialRefreshDone = true;
                $scope.setSpinnerPosition(undefined);
                if(data && data.hasResult && data.aborted) {
                    $rootScope.$broadcast('shakerTableChangeAbortedGlobal');
                    return; // Abortion is not an error
                }
                var apiErr = getErrorDetails(data, status, headers);
                $scope.shakerState.runError = apiErr;

                if (apiErr.errorType == "ApplicativeException" && apiErr.code == "STEP_RUN_EXCEPTION" && apiErr.payload) {
                    $scope.shaker.steps.forEach(function(step, i){
                        if (apiErr.payload.children[i] != null) {
                            mergeBackendErrors(step, apiErr.payload.children[i]);
                        }
                    })
                }
                $rootScope.$broadcast('shakerTableChangeFailedGlobal');
            };

            $scope.showWarningsDetails = function(){
                CreateModalFromTemplate("/templates/widgets/warnings-details.html", $scope, null, function($newScope) {
                    if ($scope.table && $scope.table.warnings && $scope.table.warnings.warnings) {
                        $newScope.warnings = $scope.table.warnings.warnings
                    }
                });
            }

            $scope.markSoftDisabled = function(){
                function _mark(s, isAfterPreview) {
                    if (isAfterPreview) {
                        s.$stepState.softDisabled = true;
                    }
                    if (s.metaType == "GROUP") {
                        if (s.steps) {
                            for (var i = 0; i < s.steps.length; i++) {
                                isAfterPreview = _mark(s.steps[i], isAfterPreview);
                            }
                        }
                    }
                    if (s.preview) {
                        $scope.stepBeingPreviewed = s;
                        return true;
                    }
                    return isAfterPreview
                }
                $scope.stepBeingPreviewed = null;
                var isAfterPreview = false;
                for (var i = 0; i < $scope.shaker.steps.length; i++) {
                    isAfterPreview = _mark($scope.shaker.steps[i], isAfterPreview);
                }
            }

            $scope.hasAnySoftDisabled = function(){
                var hasAny = false;
                function _visit(s) {
                    if (s.metaType == "GROUP") {
                        s.steps.forEach(_visit);
                    }
                    if (s.$stepState.softDisabled) hasAny = true;
                }
                $scope.shaker.steps.forEach(_visit);
                return hasAny;
            }

            // Make sure that every step after a preview is marked soft-disabled
            $scope.fixPreview = function() {
                $scope.markSoftDisabled();
                // var disable = false;
                // for (var i = 0; i < $scope.shaker.steps.length; i++) {
                //     if(disable) {
                //         $scope.shaker.steps[i].disabled = true;
                //     }
                //     if($scope.shaker.steps[i].preview) {
                //         disable=true;
                //     }
                // }
                // #2459
                if ($scope.dataset && $scope.dataset.partitioning && $scope.dataset.partitioning.dimensions){
                    if (!$scope.dataset.partitioning.dimensions.length && $scope.shaker.explorationSampling.selection.partitionSelectionMethod != "ALL") {
                        Logger.warn("Partition-based sampling requested on non partitioned dataset. Force non-partitioned sample.")
                        $scope.shaker.explorationSampling.selection.partitionSelectionMethod = "ALL";
                        delete $scope.shaker.explorationSampling.selection.selectedPartitions;
                    }
                }

            };

            /**
            * Refreshes the whole table
            * Set "filtersOnly" to true if this refresh is only for a change of filters / facets
            */
            $scope.refreshTable = function(filtersOnly) {
                const refreshArguments = arguments;
                return CachedAPICalls.processorsLibrary.success(function(){
                    if ($scope.validateScript){
                        if (!$scope.validateScript()) {
                            Logger.info("Aborted refresh: script is invalid");
                            ActivityIndicator.error("Not refreshing: script is invalid !");
                            return;
                        }
                    }
                    lastValidatedSteps = angular.copy($scope.shaker.steps);
                    // Mark the table as longer relevant until the new & refreshed table is displayed.
                    // This allows to stop displaying some info about the table while it is being refreshed.
                    if ($scope.table) {
                        $scope.table.$invalidated = true;
                    }
                    $scope.refreshTable_.apply(this, refreshArguments);
                });
            };

            const refreshDebounce = Debounce();
            $scope.refreshTable_ = refreshDebounce
                .withDelay(200, 500) // delay should be long enough to debounce dataset preview (single-clicking a dataset) when the user wants to explore a dataset (double-clicking a dataset)
                .withSpinner(!$scope.refreshNoSpinner)
                .withScope($scope)
                .wrap(function(filtersOnly){
                    if (!angular.isDefined(filtersOnly)) throw new Error();

                    // Because of the delay, it might happen that the parameters from the steps changed
                    // between $scope.validateScript() and $scope.refreshTable_().
                    // That leads to the fact that the steps could not be validated anymore when sending data to the server.
                    // In order to avoid this problem, we check that the validated data in steps did not change during the delay
                    if (!angular.equals(lastValidatedSteps, $scope.shaker.steps)) {
                        return;
                    }
                    $scope.fixPreview();
                    var filterRequest = $scope.buildFilterRequest ? $scope.buildFilterRequest($scope.shaker.explorationFilters) : [];
                    $scope.setFormerShakerData();

                    $scope.shaker.steps.forEach(clearBackendErrors);

                    $scope.$broadcast("scrollToLine", 0);

                    $scope.refreshCallBeg  = new Date().getTime();
                    $scope.future = null;

                    $scope.shakerHooks.onTableRefresh();
                    // Offer a chance to the right panel to be informed of the new state of the shaker
                    $rootScope.$broadcast("shakerTableRefresh", $scope.shaker);

                    $scope.shakerHooks.getRefreshTablePromise(filtersOnly, {"elements": filterRequest})
                    .update(function(future) {
                        $scope.autoRefreshDirty = true;
                        $scope.future = future;
                    }).success(function(future) {
                        $scope.autoRefreshDirty = false;
                        Logger.info("Got table data");
                        $scope.future = future;
                        $scope.onRefreshFutureDone(filtersOnly);
                        $scope.shakerHooks.afterTableRefresh();
                    }).error(function(data,status,headers) {
                        $scope.future = null;
                        $scope.onRefreshFutureFailed(data,status,headers);
                        
                        if ($scope.setDashboardTileError) {
                            $scope.setDashboardTileError(data, status, headers);
                        }

                        $scope.shakerHooks.afterTableRefresh();
                    });
            });

            /**
             * Checks weather there is a pending debounced refresh. and if the MonoFuturizedRefresh has an empty refresh queue
             */
            $scope.allRefreshesDone = function() {
                return !refreshDebounce.active() && !$scope.shakerHooks.isMonoFuturizedRefreshActive();
            };

            /**
            * Waits for all RefreshTable calls to be resolved. Returns a promise.
            */
            $scope.waitAllRefreshesDone = function () {
                const deferred = $q.defer();
                const inter = setInterval(
                    function () {
                        if ($scope.allRefreshesDone()) {
                            clearInterval(inter);
                            deferred.resolve();
                        }
                    }, 25);
                return deferred.promise;
            }

            /**
            * Fetches a chunk of the table. Returns a promise.
            * Out of bound is NOT handled, and will throw.
            */
            $scope.getTableChunk = function(firstRow, nbRows, firstCol, nbCols) {
                var deferred = $q.defer();
                var filterRequest = $scope.buildFilterRequest ? $scope.buildFilterRequest($scope.shaker.explorationFilters) : [];
                $scope.shakerHooks.getTableChunk(firstRow, nbRows, firstCol, nbCols,
                        {"elements":filterRequest})
                .then(({data}) => deferred.resolve(data))
                .catch(err => {
                    deferred.reject();
                    setErrorInScope.bind($scope)(err)
                });
                return deferred.promise;
            };

            $scope.analyseColumn = function(column, columns) {
                CreateModalFromTemplate("/templates/shaker/analyse-box.html",
                    $scope, "ColumnAnalysisController", function(newScope) {
                        newScope.setColumn(column, columns || $scope.table.headers);
                    }, "analyse-box");
            };

            $scope.copyColumnName = function(column) {
                if (!column?.name) {
                    return;
                }
                ClipboardUtils.copyToClipboard(column?.name, translate("SHAKER.HEADER.CONTEXTUAL_MENU.COPY_COLUMN_NAME.NOTIF", `Copied "${column?.name}" to clipboard`, {column: column?.name}));
            }

            $scope.editColumnDetails = function(column) {
                CreateModalFromTemplate("/templates/shaker/modals/shaker-edit-column.html",
                    $scope, null, function(newScope) {
                        newScope.setColumn(column);
                    });
            }

            $scope.hideColumn = function(columnName) {
                if (columnName) {
                    if ($scope.shaker.columnsSelection && $scope.shaker.columnsSelection.mode === "SELECT" && $scope.shaker.columnsSelection.list) {
                        let displayedColumn = $scope.shaker.columnsSelection.list.find(column => column.name === columnName);
                        if (displayedColumn) {
                            displayedColumn.d = false;
                        } else {
                            $scope.shaker.columnsSelection.list.push({name: columnName, d:false});
                        }
                    } else {
                        $scope.shaker.columnsSelection = {
                            mode: "SELECT",
                            list: $scope.columns.map(colName => ({ name: colName, d: (colName !== columnName) }))
                        }
                    }
                }
                $scope.autoSaveForceRefresh();
            }

            $scope.scrollToColumn = $scope.$broadcast.bind($scope, 'scrollToColumn'); // broadcast to child fattable
            $scope.$watch('shakerState.activeView', function(nv) {
                if ($scope.shakerState.activeView === 'table') {
                    $scope.setQuickColumns();
                }
            });
            $scope.setQuickColumns = function(qc) {
                $scope.quickColumns = qc || ($scope.table && $scope.table.headers || []);
            };
            $scope.clearQuickColumnsCache = function () {
                $scope.quickColumnsCache = {};
            };
            $scope.quickColumns = [];
            $scope.quickColumnsCache = {};

            /**
             * Returns true if coloring.scheme is MEANING_AND_STATUS otherwise false
             */
            $scope.isColoringSchemeByMeaning = function () {
                return (
                    $scope.shaker && $scope.shaker.coloring && $scope.shaker.coloring.scheme === "MEANING_AND_STATUS"
                );
            };
            /**
             * Returns true if coloring.scheme is ALL_COLUMNS_VALUES otherwise false
             */
            $scope.isColoringSchemeByScale = function () {
                return (
                    $scope.shaker && $scope.shaker.coloring && $scope.shaker.coloring.scheme === "ALL_COLUMNS_VALUES"
                );
            };
            /**
             * Returns true if coloring.scheme is COLORING_GROUPS or legacy (XXX_COLUMNS_RULES or XXX_COLUMNS_VALUES) otherwise false
             */
            $scope.isColoringSchemeByRules = function () {
                return (
                    $scope.shaker &&
                    $scope.shaker.coloring &&
                    ($scope.shaker.coloring.scheme === "COLORING_GROUPS" ||
                        $scope.shaker.coloring.scheme === "INDIVIDUAL_COLUMNS_RULES" ||
                        $scope.shaker.coloring.scheme === "SINGLE_COLUMN_RULES" ||
                        $scope.shaker.coloring.scheme === "INDIVIDUAL_COLUMNS_VALUES" ||
                        $scope.shaker.coloring.scheme === "SINGLE_COLUMN_VALUES")
                );
            };

            function getColoringModeString(visible, text) {
                return (
                    '<i class="' +
                    (visible ? "dku-icon-checkmark-12 mright4" : "mright16") +
                    '"></i><a>' +
                    text +
                    "</a>"
                );
            }

            /**
             * Updates the coloring display string applied and the Conditional Formatting pane
             */
            function updateColoringStates() {
                let isColoringSchemeByMeaning = false;
                let isColoringSchemeByRules = false;
                let isColoringSchemeByScale = false;

                if ($scope.shaker && $scope.shaker.coloring) {
                    if ($scope.isColoringSchemeByScale()) {
                        isColoringSchemeByScale = true;
                        $scope.coloredByText = translate("FLOW.DATASET.DISPLAY_MENU.COLORED_BY.SCALE", "scale");
                    } else if ($scope.isColoringSchemeByRules()) {
                        const table = $scope.table;
                        isColoringSchemeByRules = true;
                        $scope.conditionalFormattingState.columns = $scope.shaker.columnsSelection.list;
                        $scope.conditionalFormattingState.isWholeData = table.sampleMetadata && table.sampleMetadata.sampleIsWholeDataset;
                        $scope.conditionalFormattingState.tableHeaders = table.headers;
                        $scope.setColoredByRules();
                    } else {
                        // Nothing to display for default
                        isColoringSchemeByMeaning = true;
                        $scope.coloredByText = "";
                    }
                } else {
                    // Nothing to display for default
                    isColoringSchemeByMeaning = true;
                    $scope.coloredByText = "";
                }

                // Updates DISPLAY menus
                $scope.displayColoringMeaning = getColoringModeString(
                    isColoringSchemeByMeaning,
                    translate("FLOW.DATASET.DISPLAY_MENU.COLOR.BY_MEANING_VALIDITY", "Meaning validity (all columns)")
                );
                $scope.displayColoringRules = getColoringModeString(
                    isColoringSchemeByRules,
                    translate("FLOW.DATASET.DISPLAY_MENU.COLOR.BY_RULES", "Conditional formatting (can be slow)")
                );
                $scope.displayColoringScale = getColoringModeString(
                    isColoringSchemeByScale,
                    translate("FLOW.DATASET.DISPLAY_MENU.COLOR.SCALE", "Color scale (all columns - can be slow)")
                );
            }

            /**
             * Sets the coloring display string by rules
             */
            $scope.setColoredByRules = function () {
                $scope.coloredByText = translate('FLOW.DATASET.DISPLAY_MENU.COLORED_BY.RULES', 'rules');
            };

            /**
             * Closes the right panel if the Conditional formatting panel is opened
             */
            $scope.closeRightPaneIfConditionalFormattingOpened = function () {
                if ($scope.isQuickConditionalFormattingViewOpened() && typeof $scope.closeRightPane === "function") {
                    $scope.shakerState.rightPaneView = $scope.RIGHT_PANE_VIEW.NONE;
                    $scope.closeRightPane();
                }
            };

            $scope.openConditionalFormattingFromColumnHeader = function (columnName) {
                $scope.conditionalFormattingState.openedFromColumnHeader = columnName;
                $scope.openRightPaneWith($scope.RIGHT_PANE_VIEW.QUICK_CONDITIONAL_FORMATTING_VIEW);
            };

            /**
             * Switches to "display color by meaning on all columns" with the table coloring scheme MEANING_AND_STATUS.
             */
            $scope.onDisplayAllColumnsColorByMeaning = function () {
                if (!$scope.shaker || !$scope.shaker.coloring || $scope.isColoringSchemeByMeaning()) {
                    return;
                }

                WT1.event("dataset-change-display-mode", { displayMode: "meaning", appliesTo: "all-columns" });

                ConditionalFormattingEditorService.prepareMigrationToColoringGroups(
                    $scope.shaker.coloring,
                    $scope.conditionalFormattingState.tableHeaders
                );

                // No migration, only updating the table coloring scheme
                $scope.shaker.coloring.scheme = "MEANING_AND_STATUS";
                updateColoringStates();

                $scope.autoSaveForceRefresh();
            };

            /**
             * Switches to "display color by rules on all columns" with the table coloring scheme COLORING_GROUPS. 
             * Opens the Conditional formatting right pane if the user has the write access 
             * or migrates to COLORING_GROUPS scheme only.
             */
            $scope.onDisplayAllColumnsColorByRules = function () {
                if (!$scope.shaker || !$scope.shaker.coloring) return;

                if (!$scope.isColoringSchemeByRules()) {
                    WT1.event("dataset-change-display-mode", { displayMode: "rules", appliesTo: "all-columns" });
                }

                if (
                    $scope.shakerState.writeAccess &&
                    $scope.table &&
                    $scope.table.headers &&
                    $scope.table.headers.length
                ) {
                    $scope.openRightPaneWith($scope.RIGHT_PANE_VIEW.QUICK_CONDITIONAL_FORMATTING_VIEW);
                } else {
                    if (
                        ConditionalFormattingEditorService.migrateToColoringGroups(
                            $scope.shaker.coloring,
                            $scope.conditionalFormattingState.tableHeaders
                        )
                    ) {
                        $scope.setColoredByRules();
                        $scope.autoSaveForceRefresh();
                    }
                }
            };

            /**
             * Switches to "display color scale on all columns" with the table coloring scheme ALL_COLUMNS_VALUES.
             */
            $scope.onDisplayAllColumnsColorScale = function () {
                if (!$scope.shaker || !$scope.shaker.coloring || $scope.isColoringSchemeByScale()) {
                    return;
                }

                WT1.event("dataset-change-display-mode", { displayMode: "colorscale", appliesTo: "all-columns" });

                ConditionalFormattingEditorService.prepareMigrationToColoringGroups(
                    $scope.shaker.coloring,
                    $scope.conditionalFormattingState.tableHeaders
                );

                // No migration, only updating the table coloring scheme
                $scope.shaker.coloring.scheme = "ALL_COLUMNS_VALUES";
                updateColoringStates();

                $scope.autoSaveForceRefresh();
            };

            $scope.toggleScientificNotation = function(columnName) {
                $scope.shaker.columnUseScientificNotationByName[columnName] = $scope.shaker.columnUseScientificNotationByName[columnName] === true ? false : true;
                $scope.autoSaveForceRefresh();
            }
            $scope.sortDirection = function(column) {
                var sortElem = ($scope.shaker.sorting || []).filter(function(e) {return e.column == column;})[0];
                return sortElem == null ? null : sortElem.ascending;
            };
            $scope.toggleSort = function(column) {
                if ($scope.shaker.sorting == null) {
                    $scope.shaker.sorting = [];
                }
                var sorting = $scope.shaker.sorting;
                if (sorting.length == 1 && sorting[0].column == column) {
                    sorting[0].ascending = !sorting[0].ascending;
                } else {
                    $scope.shaker.sorting = [{column:column, ascending:true}];
                }
                $scope.autoSaveForceRefresh();
            }
            $scope.addSort = function(column) {
                if ($scope.shaker.sorting == null) {
                    $scope.shaker.sorting = [];
                }
                var sorting = $scope.shaker.sorting;
                var matching = sorting.filter(function(s) {return s.column == column;});
                if (matching.length > 0) {
                    matching[0].ascending = !matching[0].ascending;
                } else {
                    $scope.shaker.sorting.push({column:column, ascending:true});
                }
                $scope.autoSaveForceRefresh();
            }

            // Callback called when dropping a column while reordering (see fatDraggable directive) - for the explore view
            $scope.reorderColumnCallback = function(draggedColumn, hoveredColumn, columnName, referenceColumnName) {

                let movement = {};

                let columnOldPosition = $scope.columns.indexOf(columnName);
                let columnNewPosition = $scope.columns.indexOf(referenceColumnName);

                if (columnOldPosition < 0 || columnNewPosition < 0) {
                    return;
                }

                if (columnNewPosition === 0) {
                    movement.reorderAction = "AT_START";
                } else if (columnNewPosition === $scope.columns.length - 1) {
                    movement.reorderAction = "AT_END";
                } else if (columnOldPosition > columnNewPosition) {
                    movement.reorderAction = "BEFORE_COLUMN";
                } else {
                    movement.reorderAction = "AFTER_COLUMN";
                }

                movement.movedColumn = $scope.columns[columnOldPosition];

                if (movement.reorderAction === "BEFORE_COLUMN" || movement.reorderAction === "AFTER_COLUMN") {
                    movement.referenceColumn = $scope.columns[columnNewPosition];
                }

                if ($scope.shaker) {
                   if (!$scope.shaker.columnOrder) {
                       $scope.shaker.columnOrder = [movement];
                   } else {
                       $scope.shaker.columnOrder.push(movement);
                   }
                }

                $scope.autoSaveForceRefresh();
            };

            $scope.openColumnsSelectionModal = function(){
                CreateModalFromTemplate("/templates/shaker/select-columns-modal.html", $scope);
            }
            $scope.openSortSelectionModal = function(){
                CreateModalFromTemplate("/templates/shaker/select-sort-modal.html", $scope);
            }
            $scope.clearSort = function(column) {
                if (column && $scope.shaker.sorting) {
                    var sorting = $scope.shaker.sorting;
                    var matching = sorting.filter(function(s) {return s.column == column;});
                    if (matching.length > 0) {
                        sorting.splice(sorting.indexOf(matching[0]), 1);
                    }
                } else {
                    $scope.shaker.sorting = [];
                }
                $scope.autoSaveForceRefresh();
            }

            $scope.clearResize = function() {
                const minColumnWidth = 100;
                $scope.shaker.columnWidthsByName = computeColumnWidths($scope.table.initialChunk, $scope.table.headers, minColumnWidth, $scope.hasAnyFilterOnColumn, $scope.shaker.columnWidthsByName, $scope.shaker.columnUseScientificNotationByName, true)[1];
                $scope.autoSaveAutoRefresh();
            }

            $scope.clearColumnOrder = function() {
                $scope.shaker.columnOrder = [];
                $scope.autoSaveAutoRefresh();
            }

            this.$scope = $scope; // fugly
    
            $scope.$on('datasetSchemaChanged', (event, data) => {
                $scope.refreshTable(false);
            });
        }
    }
});

/**
 * Finds the coloring group ColoringGroup that is applied to a column.
 * 
 * Iterates over the list of coloring groups (reverse order, like a stack) to find the coloring group
 * which is enabled and targets the given column.
 * If no coloring group is found for the column, returns null.
 * 
 * If optional filters `{ filterScope, filterScheme }` are provided, returns the first coloring group targeting 
 * the column which has the provided scope and/or scheme.
 */
app.service("findAppliedColoringGroup", function () {
    return function (coloringGroups, columnName, { filterScope, filterScheme } = { filterScope: null, filterScheme: null }) {
        for (let i = coloringGroups.length - 1; i >= 0; i--) {
            const coloringGroup = coloringGroups[i];
            if (!coloringGroup.enabled) {
                continue;
            }

            // Skips coloring groups that do not respect the optional filters on scope and/or scheme
            if ((filterScope && filterScope != coloringGroup.scope) || (filterScheme && filterScheme != coloringGroup.scheme)) {
                continue;
            }

            // Targeted by an "all columns" logic
            if (coloringGroup.scope === "ALL_COLUMNS_BASED_ON_ANOTHER_COLUMN") {
                return coloringGroup;
            }

            // Specifically targeted
            if (coloringGroup.scope === "COLUMNS") {
                const isTargeted = coloringGroup.targetedColumnNames.some(targeted => columnName === targeted);
                if (isTargeted) {
                    return coloringGroup;
                }
            }
        }

        return null;
    };
});

app.directive('quickColumnsView', function(DataikuAPI, Fn, Debounce, MonoFuture, $filter, $stateParams) {
    var COLUMN_CHUNK = 50,
        dateFmt = Fn(function(d){ return new Date(d); }, d3.time.format('%Y-%m-%d %H:%M')),
        numFmt = $filter('smartNumber');
    return {
        scope: true,
        require: '^shakerExploreBase',
        templateUrl: '/templates/shaker/quick-columns-view.html',
        link: function(scope, element, attrs, exploreCtrl) {
            var monoLoad = [];
            scope.$watch('shakerState.rightPaneView', function(newVal) {
                if (newVal === scope.RIGHT_PANE_VIEW.QUICK_COLUMNS_VIEW) {
                    scope.openRightPane();
                }
            });
            scope.onClose = function() {
                scope.shakerState.rightPaneView = scope.RIGHT_PANE_VIEW.NONE;
                scope.closeRightPane();
            }
            scope.initColumnScope = function(cScope) {
                if (!cScope.col) return;
                cScope.activateColBar = scope.activateBar.bind(null, cScope);
            };
            scope.quickColumnsChanged = function() {
                scope.quickColumnsFilterChanged(scope.quickColumnsFilter);
            };
            scope.quickColumnCacheCleared = function() {
                monoLoad.forEach(function(m){ if (m.running) m.abort(); });
            };

            scope.quickColumnsFilter = '';
            scope.quickColumnsFilterChanged = function(newVal, oldVal) {
                // $watch triggers this listener a first time to initialize the watcher.
                // But it needs to be stopped to avoid a first call to quickColumnCacheCleared
                // which aborts our initial running mono futures (in particular multiColumnAnalysis).
                if (newVal === oldVal) {
                    return;
                } 

                scope.quickColumnCacheCleared();
                scope.quickColumnsFiltered = !newVal ? scope.quickColumns : scope.quickColumns.filter(
                    function(c) { return c.name.toLowerCase().indexOf(this) >= 0; }, newVal.toLowerCase());
                // append MonoFuture at will
                for (var i = monoLoad.length; i < Math.ceil(scope.quickColumnsFiltered.length / COLUMN_CHUNK); i++) {
                    monoLoad[i] = MonoFuture(scope);
                }
            };
            // Can’t use PagedAsyncTableModel because of divergent invalidation policy:
            // cache is kept when closing QCV or filtering columns,
            // but reset when editing shaker steps
            scope.tableModel = function() {
                var model = new fattable.TableModel();
                model.hasCell = Fn.cst(true); // always drawable
                model.getCell = function(i, j, cb) {
                    if (scope.shakerState.rightPaneView !== scope.RIGHT_PANE_VIEW.QUICK_COLUMNS_VIEW) return;
                    var page = Math.floor(i / COLUMN_CHUNK);
                    // Initiate block fetch...
                    loadQuickColumns(page, cb);
                    // ...but render immediately (name, type, validity)
                    cb(scope.quickColumnsFiltered[i]);
                };
                return model;
            };
            function loadQuickColumns(page, cb) {
                if (monoLoad[page].running) return;
                var uncached = scope.quickColumnsFiltered
                    .slice(page * COLUMN_CHUNK, (page + 1) * COLUMN_CHUNK)
                    .map(Fn.prop('name'))
                    .filter(Fn.not(Fn.dict(scope.quickColumnsCache)));
                if (!uncached.length) return;
                monoLoad[page].running = true;
                monoLoad[page].exec(
                    DataikuAPI.shakers.multiColumnAnalysis(
                        $stateParams.projectKey,
                        scope.inputDatasetProjectKey, scope.inputDatasetName, scope.inputStreamingEndpointId,
                        scope.shakerHooks.shakerForQuery(),
                        scope.requestedSampleId, uncached, '*', 40))
                .success(function(data){
                    monoLoad[page].running = false;
                    if (!data.hasResult) return;
                    data = data.result;
                    // Update quick column using data from scope table
                    scope.setQuickColumns();
                    for (var k in data) {
                        if (data[k].facets) {
                            scope.quickColumnsCache[k] = {
                                values: data[k].facets.counts,
                                labels: data[k].facets.values
                            };
                        } else {
                            scope.quickColumnsCache[k] = { values: data[k].histogram };
                            var col = scope.quickColumns.filter(Fn(Fn.prop("name"), Fn.eq(k)))[0],
                                fmt = col && col.selectedType && ['Date', 'DateOnly', 'DatetimeNoTz'].indexOf(col.selectedType.name) >= 0 ? dateFmt : numFmt;
                            scope.quickColumnsCache[k].labels =
                              data[k].histogramLowerBounds.map(fmt).map(
                                function(lb, i) { return lb + " - " + this[i]; },
                                data[k].histogramUpperBounds.map(fmt))
                        }
                    }
                }).error(function() {
                    monoLoad[page].running = false;
                    setErrorInScope.apply(exploreCtrl.$scope, arguments);
                });
            }
            scope.$watch('quickColumnsCache', function(newVal, oldVal) {
                scope.quickColumnCacheCleared();
            });
            scope.$watch('quickColumns', function(newVal, oldVal) {
                scope.quickColumnsChanged();
            }, true);
            scope.$watch('quickColumnsFilter',
                function(newVal,oldVal) {
                    Debounce().withDelay(150,300).withScope(scope).wrap(scope.quickColumnsFilterChanged)(newVal, oldVal);
                });
            scope.activateBar = function(colScope, value, i) {
                colScope.setLabels(value !== null ? {
                    pop: value.toFixed(0),
                    label: scope.quickColumnsCache[colScope.col.name].labels[i],
                    part: (colScope.col.selectedType ? (value * 100 / colScope.col.selectedType.totalCount).toFixed(1) + '&nbsp;%' : '')
                } : null);
            };
            scope.defaultAction = !scope.scrollToColumn ? null :
                function(column) { scope.scrollToColumn(column.name); };
        }
    };
});

app.directive('quickConditionalFormattingView', function($q, $stateParams, DataikuAPI, WT1, Logger) {
    return {
        scope: true,
        require: '^shakerExploreBase',
        templateUrl: '/templates/shaker/quick-conditional-formatting-view.html',
        link: function(scope) {
            scope.$watch("shakerState.rightPaneView", function (newVal) {
                if (newVal === scope.RIGHT_PANE_VIEW.QUICK_CONDITIONAL_FORMATTING_VIEW) {
                    const coloring = scope.shaker.coloring;
                    const nbRules = coloring ? (coloring.coloringGroups ? coloring.coloringGroups.length : 0) : 0;
                    WT1.event("conditional-formatting-pane-open", {
                        nbRules: nbRules,
                    });

                    scope.setColoredByRules();
                    scope.conditionalFormattingState.columns = scope.$parent.shaker.columnsSelection.list;
                    scope.conditionalFormattingState.tableHeaders = scope.$parent.table.headers;
                    scope.openRightPane();
                }
            });

            scope.onClose = function () {
                scope.shakerState.rightPaneView = scope.RIGHT_PANE_VIEW.NONE;
                scope.closeRightPane();
            };

            scope.saveAndRefresh = function () {
                scope.autoSaveForceRefresh();
            };

            // Returns a promise which resolves when saving the explore config is done
            scope.savePromise = function () {
                // Same checks done in autoSaveForceRefresh
                if (!scope.isRecipe && (scope.canWriteProject() || scope.isLocalSave)) {
                    return scope.shakerHooks.saveForAuto();
                }

                // Returns an empty promise
                return $q.when();
            };

            // Returns a promise
            scope.setScaleMinMaxFromAnalysis = function (columnName, colorScaleDef) {
                if (columnName && colorScaleDef) {
                    DataikuAPI.shakers
                        .detailedColumnAnalysis(
                            $stateParams.projectKey,
                            scope.inputDatasetProjectKey,
                            scope.inputDatasetName,
                            scope.shakerHooks.shakerForQuery(),
                            scope.requestedSampleId,
                            columnName,
                            0,
                            null,
                            false
                        )
                        .success(function (data) {
                            if (data.hasOwnProperty("numericalAnalysis")) {
                                colorScaleDef.max = data.numericalAnalysis.max;
                                colorScaleDef.min = data.numericalAnalysis.min;
                            }
                        })
                        .error(function (a, b, c) {
                            Logger.error("setScaleMinMaxFromAnalysis error", a, b, c);
                        });
                }
            };
        },
    };
});

/**
 * Base directive for all instances where a shaker table is made on a dataset
 * (explore, analysis script, prepare recipe).
 * (Counter examples: predicted data)
 */
app.directive("shakerOnDataset", function() {
    return {
        priority: 50,
        scope: true,
        controller  : function ($rootScope, $scope, $state, $stateParams, DataikuAPI, MonoFuture) {
            const monoFuture = MonoFuture($scope);
            const monoFuturizedRefresh = monoFuture.wrap(DataikuAPI.shakers.refreshTable);

            $scope.shakerState.onDataset = true;

            $scope.shakerHooks.isMonoFuturizedRefreshActive = monoFuture.active;

            $scope.shakerHooks.shakerForQuery = function(){
                var queryObj = angular.copy($scope.shaker);
                if ($scope.isRecipe) {
                    queryObj.recipeSchema = $scope.recipeOutputSchema;
                }
                queryObj.contextProjectKey = $stateParams.projectKey; // quick 'n' dirty, but there are too many call to bother passing the projectKey through them
                return queryObj;
            }

            $scope.shakerHooks.updateColumnWidth = function(name, width) {
                $scope.shaker.columnWidthsByName[name] = width;
                $scope.autoSaveAutoRefresh();
            };

            $scope.shakerHooks.getRefreshTablePromise = function(filtersOnly, filterRequest) {
                const ret = monoFuturizedRefresh($stateParams.projectKey, $scope.inputDatasetProjectKey, $scope.inputDatasetName,
                    $scope.shakerHooks.shakerForQuery(), $scope.requestedSampleId, filtersOnly, filterRequest, false);

                return $scope.refreshNoSpinner ? ret.noSpinner() : ret;
            };

            /**
            * Fetches a chunk of the table. Returns a promise.
            * Out of bound is NOT handled, and will throw.
            */
            $scope.shakerHooks.getTableChunk = function(firstRow, nbRows, firstCol, nbCols, filterRequest) {
                return DataikuAPI.shakers.getTableChunk(
                    $stateParams.projectKey,
                    $scope.inputDatasetProjectKey,
                    $scope.inputDatasetName,
                    $scope.shakerHooks.shakerForQuery(),
                    $scope.requestedSampleId,
                    firstRow,
                    nbRows,
                    firstCol,
                    nbCols,
                    filterRequest);
            }

            $scope.shakerHooks.fetchDetailedAnalysis = function(setAnalysis, handleError, columnName, alphanumMaxResults, fullSamplePartitionId, withFullSampleStatistics) {
                DataikuAPI.shakers.detailedColumnAnalysis($stateParams.projectKey, $scope.inputDatasetProjectKey, $scope.inputDatasetName,
                        $scope.shakerHooks.shakerForQuery(), $scope.requestedSampleId, columnName, alphanumMaxResults, fullSamplePartitionId, withFullSampleStatistics).success(function(data){
                            setAnalysis(data);
                }).error(function(a, b, c) {
                    if (handleError) {
                        handleError(a, b, c);
                    }
                    setErrorInScope.bind($scope)(a, b, c);
                });
            };
            $scope.shakerHooks.fetchClusters = function(setClusters, columnName, setBased, radius, timeOut, blockSize) {
                DataikuAPI.shakers.getClusters($stateParams.projectKey,
                    $scope.inputDatasetProjectKey, $scope.inputDatasetName,
                        $scope.shakerHooks.shakerForQuery(), $scope.requestedSampleId,
                        columnName, setBased, radius, timeOut, blockSize
                    ).success(function(data) {
                        setClusters(data);
                    }).error(setErrorInScope.bind($scope));
            };
            $scope.shakerHooks.fetchTextAnalysis = function(setTextAnalysis, columnName, textSettings) {
                DataikuAPI.shakers.textAnalysis(
                        $stateParams.projectKey,
                        $scope.inputDatasetProjectKey, $scope.inputDatasetName,
                        $scope.shakerHooks.shakerForQuery(), $scope.requestedSampleId,
                        columnName, textSettings)
                    .success(function(data){setTextAnalysis(data);})
                    .error(setErrorInScope.bind($scope));
            };

        }
    }
});

app.directive("shakerForPreview", function(Assert, DataikuAPI, DatasetErrorCta, MonoFuture) {
    return {
        scope: true,
        controller: function ($scope) {
            Assert.inScope($scope, 'shakerHooks');
            $scope.setSpinnerPosition = () => {};
            const monoFuture = MonoFuture($scope, !$scope.refreshNoSpinner);
            const monoFuturizedRefresh = monoFuture.wrap(DataikuAPI.shakers.refreshTable);
            function isDefined(variable) {
                return !(variable === null || variable === undefined);
            }
            $scope.shaker = {
                steps: [],
                explorationFilters: [],
                explorationSampling: {
                    selection: {
                        samplingMethod: "HEAD_SEQUENTIAL",
                        maxRecords: 50,
                        maxStoredBytes: 1 * 50 * 1024 * 1024 // i.e. max 1 MB RAM per record
                    }
                },
                coloring: {
                    scheme: isDefined($scope.shakerColoringScheme) ? $scope.shakerColoringScheme : "MEANING_AND_STATUS"
                },
                origin: isDefined($scope.shakerOrigin) ? $scope.shakerOrigin : "DATASET_EXPLORE",
                contextProjectKey: $scope.contextProjectKey ? $scope.contextProjectKey : $scope.projectKey,
                $headerOptions: {
                    showName: isDefined($scope.showName) ? $scope.showName : true,
                    showMeaning: isDefined($scope.showMeaning) ? $scope.showMeaning : true,
                    showStorageType: isDefined($scope.showStorageType) ? $scope.showStorageType : true,
                    showDescription: isDefined($scope.showDescription) ? $scope.showDescription : true,
                    showCustomFields: isDefined($scope.showCustomFields) ? $scope.showCustomFields : true,
                    showProgressBar: isDefined($scope.showProgressBar) ? $scope.showProgressBar : true,
                    showHeaderSeparator: isDefined($scope.showHeaderSeparator) ? $scope.showHeaderSeparator : false,
                    disableHeaderMenu: isDefined($scope.disableHeaderMenu) ? $scope.disableHeaderMenu : true,
                }
            };

            /** Error Management **/
            $scope.updateUiState = DatasetErrorCta.getupdateUiStateFunc($scope);
            $scope.$watch("shakerState", _ => $scope.updateUiState($scope.shakerState.runError), true);
            $scope.$watch("table", _ => $scope.updateUiState($scope.shakerState.runError), true);

            /** Refresh table **/
            $scope.shakerHooks.getTableChunk = function(firstRow, nbRows, firstCol, nbCols, filterRequest) {
                return DataikuAPI.shakers.getTableChunk(
                    $scope.contextProjectKey ? $scope.contextProjectKey : null,
                    $scope.projectKey,
                    $scope.datasetName,
                    $scope.shaker,
                    $scope.requestedSampleId,
                    firstRow,
                    nbRows,
                    firstCol,
                    nbCols,
                    filterRequest);
            }

            refreshPreviewTable(true);

            $scope.$on('refresh-preview-table', () => {
                refreshPreviewTable(true);
            });

            $scope.$on('refresh-preview-table-without-cache', () => {
                refreshPreviewTable(false);
            });

            function refreshPreviewTable(useCache) {
                $scope.shakerHooks.getRefreshTablePromise = function(filtersOnly, filterRequest) {
                    // Either do not send any sample ID (so the backend computes it itself from the sample settings), or send a dummy one to force cache invalidation
                    const requestedSampleId = useCache ? null : -1;
                    const contextProjectKey = $scope.contextProjectKey ? $scope.contextProjectKey : null;
                    var ret = monoFuturizedRefresh(contextProjectKey, $scope.projectKey, $scope.datasetName, $scope.shaker, requestedSampleId, filtersOnly, null, true);
                    return $scope.refreshNoSpinner ? ret.noSpinner() : ret;
                };
                $scope.refreshTable(false);
            }
        }
    }
});

/**
 * Base directive for all instances where a shaker table is made on a streaming endpoint
 */
app.directive("shakerOnStreamingEndpoint", function() {
    return {
        scope: true,
        controller  : function ($scope, $state, $stateParams, DataikuAPI, MonoFuture, WT1) {
            const monoFuture = MonoFuture($scope);
            const monoFuturizedRefresh = monoFuture.wrap(DataikuAPI.shakers.refreshCapture);

            $scope.shakerState.onDataset = false;

            $scope.shakerHooks.isMonoFuturizedRefreshActive = monoFuturizedRefresh.active;

            $scope.shakerHooks.shakerForQuery = function(){
                var queryObj = angular.copy($scope.shaker);
                if ($scope.isRecipe) {
                    queryObj.recipeSchema = $scope.recipeOutputSchema;
                }
                queryObj.contextProjectKey = $stateParams.projectKey; // quick 'n' dirty, but there are too many call to bother passing the projectKey through them
                return queryObj;
            }

            $scope.shakerHooks.updateColumnWidth = function(name, width) {
                $scope.shaker.columnWidthsByName[name] = width;
                $scope.autoSaveAutoRefresh();
            };

            $scope.shakerHooks.getRefreshTablePromise = function(filtersOnly, filterRequest) {
                WT1.event("streaming-refresh-explore")

                var ret = monoFuturizedRefresh($stateParams.projectKey, $scope.inputDatasetProjectKey, $scope.inputStreamingEndpointId,
                        $scope.shakerHooks.shakerForQuery(), $scope.requestedSampleId, filtersOnly, filterRequest);

                return $scope.refreshNoSpinner ? ret.noSpinner() : ret;
            };

            /**
            * Fetches a chunk of the table. Returns a promise.
            * Out of bound is NOT handled, and will throw.
            */
            $scope.shakerHooks.getTableChunk = function(firstRow, nbRows, firstCol, nbCols, filterRequest) {
                return DataikuAPI.shakers.getCaptureChunk(
                    $stateParams.projectKey,
                    $scope.inputDatasetProjectKey,
                    $scope.inputStreamingEndpointId,
                    $scope.shakerHooks.shakerForQuery(),
                    $scope.requestedSampleId,
                    firstRow,
                    nbRows,
                    firstCol,
                    nbCols,
                    filterRequest);
            }

            $scope.shakerHooks.fetchDetailedAnalysis = function(setAnalysis, handleError, columnName, alphanumMaxResults, fullSamplePartitionId, withFullSampleStatistics) {
                DataikuAPI.shakers.detailedStreamingColumnAnalysis($stateParams.projectKey, $scope.inputDatasetProjectKey, $scope.inputStreamingEndpointId,
                        $scope.shakerHooks.shakerForQuery(), $scope.requestedSampleId, columnName, alphanumMaxResults, fullSamplePartitionId, withFullSampleStatistics).success(function(data){
                            setAnalysis(data);
                }).error(function(a, b, c) {
                    if (handleError) {
                        handleError(a, b, c);
                    }
                    setErrorInScope.bind($scope)(a, b, c);
                });
            };
            $scope.shakerHooks.fetchClusters = function(setClusters, columnName, setBased, radius, timeOut, blockSize) {
                // Do nothing
            };
            $scope.shakerHooks.fetchTextAnalysis = function(setTextAnalysis, columnName, textSettings) {
                // Do nothing
            };
        }
    }
});

app.service("DatasetChartsUtils", function(SamplingData){
    var svc = {
        makeSelectionFromScript: function(script) {
            return {
                 selection : SamplingData.makeStreamableFromMem(script.explorationSampling.selection)
            }
        }
    }
    return svc;
})

app.controller("_ChartOnDatasetSamplingEditorBase", function($scope, $stateParams, Logger, DatasetChartsUtils,
                                                                    DataikuAPI, CreateModalFromTemplate, SamplingData){
    $scope.getPartitionsList = function() {
        return DataikuAPI.datasets.listPartitions($scope.dataset)
                .error(setErrorInScope.bind($scope))
                .then(function(ret) { return ret.data });
    };

    $scope.$watch("chart.copySelectionFromScript", function(nv, ov) {
        if ($scope.canCopySelectionFromScript) {
            if ($scope.chart.copySelectionFromScript === false && !$scope.chart.refreshableSelection) {
                $scope.chart.refreshableSelection = DatasetChartsUtils.makeSelectionFromScript($scope.script);
            }
        } else {
            Logger.warn("Can't copy selection from script");
        }
    })

    $scope.showFilterModal = function() {
        var newScope = $scope.$new();
        newScope.updateFilter = (filter) => $scope.chart.refreshableSelection.selection.filter = filter;
        DataikuAPI.datasets.get($scope.dataset.projectKey, $scope.dataset.name, $stateParams.projectKey)
        .success(function(data){
            newScope.dataset = data;
            newScope.schema = data.schema;
            newScope.filter = angular.copy($scope.chart.refreshableSelection.selection.filter);
            CreateModalFromTemplate('/static/dataiku/nested-filters/input-filter-block/filter-modal.component.html', newScope, undefined, false, false, 'static');
        }).error(setErrorInScope.bind($scope));
    }

    $scope.SamplingData = SamplingData;
});

app.controller("ChartsCommonController", function ($scope, $timeout, ChartLegendsWrapper, DefaultDSSVisualizationTheme) {
    $scope.$on("listeningToForceExecuteChart", function() {
        $scope.canForceExecuteChart = true;
    });

    /**
     * Broadcast for a forceToExecute() call only if it's sure someone is already listening to such a broadcast.
     * Otherwise recheck every 100ms until some directive has told it it was listening or that 3s have passed (at which point broadcast will be made).
     */
    $scope.forceExecuteChartOrWait = function(){
        let nbTimeouts = 0;

        // Inner function does the job to isolate nbTimeouts
        function inner() {
            if ($scope.canForceExecuteChart || nbTimeouts > 30) {
                $scope.$broadcast("forceExecuteChart");
            } else {
                nbTimeouts++;
                $scope.forceExecuteChartTimeout = $timeout(inner,100);
            }
        }
        inner();
    };

    //avoid two concurrent timeouts if two calls were made to forceExecuteChartOrWait()
    $scope.$watch('forceExecuteChartTimeout', function(nv, ov) {
        if (ov!= null) {
            $timeout.cancel(ov);
        }
    });

    // We pass the scope in params to be able to hide existing props
    // in the top scope, without touching the props in child scope.
    $scope.initChartCommonScopeConfig = function(scope, chart) {
        if (!scope) {
            scope = $scope;
        }
        
        scope.legendsWrapper = new ChartLegendsWrapper();
        scope.animation = {};
        scope.tooltips = {};
        scope.displayedGeometries = [];
        scope.chartSpecific = {};
        scope.chartPicker = {};

        scope.getChartTheme = () => {
            return scope.chart && scope.chart.theme;
        };
        
        if (chart) {
            scope.chart = chart;
        }
    };
});

app.controller("ShakerChartsCommonController", function ($scope, $sce, translate, $timeout, $controller, $state, WT1, $stateParams, Logger, CreateModalFromTemplate, MonoFuture, ContextualMenu, Dialogs, ChartCustomMeasures, ChartRBNDs, ChartUsableColumns, ColumnAvailability, ChartsContext, DataikuAPI, ChartLabels, DKUPivotCharts, ChartDimension, ChartFeatures, ChartTypeChangeHandler, ChartRequestComputer, DefaultDSSVisualizationTheme) {
    $controller("ChartsCommonController", {$scope:$scope});

    $scope.summary = {};
    $scope.currentChart = {index: 0};
    $scope.chartBottomOffset = 30;
    $scope.filterState = {
        chartFilter: ''
    };

    const getHashCodedDatasetNameFromContext = () => {
        const context = $scope.analysisCoreParams || $scope.insight || $scope.dataset || { name: 'unknown' };
        return context.name.dkuHashCode();
    };

    $scope.addChart = function (params = { chart: {}, datasetName: undefined, replace: false, copyOfName: false, index: undefined, wt1Event: 'chart-create' }) {
        const { chart, datasetName, replace, index, copyOfName, wt1Event } = params;
        const newChart = {
            ...$scope.getDefaultNewChart(),
            ...angular.copy(chart || {})
        };

        if (copyOfName) {
            newChart.def.name = `Copy of ${newChart.def.name}`;
            newChart.def.userEditedName = true;
        } else {
            newChart.def.name = ChartLabels.NEW_CHART_LABEL;
        }

        const isInInsight = $state.current.name.indexOf('insight') != -1;

        if (replace) {
            if (isInInsight) {
                $scope.chart = newChart;
            } else {
                const chartIndex = index === undefined ? $scope.currentChart.index : index;
                $scope.charts[chartIndex] = newChart;
            }
        } else {
            // if copied, put the new chart just after the current one, otherwise put it at the end
            const targetIdx = index === undefined ? $scope.charts.length : index + 1;
            $scope.charts.splice(targetIdx, 0, newChart);
            $scope.currentChart.index = targetIdx;
        }

        if (typeof $scope.fetchColumnsSummaryForCurrentChart === 'function') {
            $scope.fetchColumnsSummaryForCurrentChart();
        } else if (typeof $scope.fetchColumnsSummary  === 'function') {
            $scope.fetchColumnsSummary().then(() => $scope.forceExecuteChartOrWait());
        }

        const hashCodedDatasetname = datasetName ? datasetName.dkuHashCode() : getHashCodedDatasetNameFromContext();

        WT1.event(wt1Event, {
            chartId: `${$stateParams.projectKey.dkuHashCode()}.${hashCodedDatasetname}.${newChart.def.name.dkuHashCode()}`,
            chartType: newChart.def.type,
            chartVariant: newChart.def.variant
        });

        if (isInInsight) {
            $scope.insight.params.def = newChart.def;
            $scope.saveChart();
        } else {
            $scope.saveShaker();
        }
    };

    $scope.pageSortOptions = {
        helper: 'clone',
        axis: 'x',
        cursor: 'move',
        update: onSortUpdated,
        handle: '.thumbnail',
        items: '> a.chart',
        delay: 100,
        'ui-floating': true
    };

    function onSortUpdated(evt, ui) {
        var prevIdx = ui.item.sortable.index, newIdx = ui.item.sortable.dropindex;
        if (prevIdx == $scope.currentChart.index) {
            $scope.currentChart.index = ui.item.sortable.dropindex;
        } else if (prevIdx < $scope.currentChart.index && newIdx >= $scope.currentChart.index) {
            $scope.currentChart.index--;
        } else if (prevIdx > $scope.currentChart.index && newIdx <= $scope.currentChart.index) {
            $scope.currentChart.index++;
        }

        $timeout($scope.saveShaker);
    }

    $scope.pasteChart = function(params, index) {
        $scope.addChart({
            chart: {
                ...params.chartDef,
                def: {
                    ...params.chartDef.def,
                    id: window.crypto.getRandomValues(new Uint32Array(1))[0].toString(16)
                }
            },
            index,
            copyOfName: true,
            replace: !params.pasteAfter,
            wt1Event: 'chart-paste'
        });
    };

    $scope.duplicateChart = function(index) {
        $scope.addChart({
            chart: {
                ...$scope.charts[index],
                def: {
                    ...$scope.charts[index].def,
                    id: window.crypto.getRandomValues(new Uint32Array(1))[0].toString(16)
                }
            },
            index,
            copyOfName: true,
            wt1Event: 'chart-duplicate'
        });
    };

    $scope.deleteCurrentChart = function(index) {
        $scope.deleteChart(index);
    };

    $scope.getCurrentChartFromContext = function() {
        const charts = getChartsFromContext();

        if ($scope.currentChart && charts && charts.length > $scope.currentChart.index) {
            return charts[$scope.currentChart.index];
        }
        return undefined;
    }

    $scope.deleteChart = function(idx, datasetName) {
        const hashCodedDatasetname = datasetName ? datasetName.dkuHashCode() : getHashCodedDatasetNameFromContext();

        WT1.event('chart-delete', {
            chartId: `${$stateParams.projectKey.dkuHashCode()}.${hashCodedDatasetname}.${$scope.charts[idx].def.name.dkuHashCode()}`,
            chartType: $scope.charts[idx].def.type,
            chartVariant: $scope.charts[idx].def.variant
        });

        $scope.charts.splice(idx, 1);
        if ($scope.currentChart.index >= $scope.charts.length) {
            $scope.currentChart.index = $scope.charts.length - 1;
        }
        if ($scope.charts.length == 0) {
            $scope.addChart();
        }
        $scope.saveShaker();
    };

    $scope.makeUsableColumns = function(data) {
        $scope.usableColumns = ChartUsableColumns.makeUsableColumns(data, getChartsCacheableContext())
    };

    $scope.onPercentileChoice = (measure, event) => {
        measure.isCustomPercentile = event.isCustomPercentile;
        measure.percentile = event.percentile;
    };

    $scope.chartEntityMenu = new ContextualMenu({
        template: "/templates/shaker/chart-entity-edition-menu.html",
        cssClass: "chart-dataset-edition-menu",
        scope: $scope,
        contextual: false,
        enableClick: true
    });

    $scope.RBNDMenu = new ContextualMenu({
        template: "/templates/shaker/rbnd-edition-menu.html",
        cssClass: "chart-dataset-edition-menu",
        scope: $scope,
        contextual: false,
        enableClick: true
    });
    
    $scope.$on("craftBinningFormChart", function($event, params) {
        $scope.openReusableDimensionPanel( { ...params, fromChart: true });
    });

    $scope.editEntity = (entity, type = 'CUSTOM_MEASURE') => {
        $scope.hideChartEntityMenu();
        switch(type) {
            case 'RBND':
                $scope.openModalRBND({ dimensionRef: entity, dimension: angular.copy(entity), isEditMode: true, fromChart: false });
                break;
            case 'CUSTOM_MEASURE':
                $scope.openModalCustomMeasure(entity).then(data => {
                    if (entity.isDefaultMeasure && data) {
                        ChartCustomMeasures.setImpactedCharts(getChartsFromContext(), entity);
                        ChartCustomMeasures.onItemEdit(getChartsFromContext(), data.newCustomMeasure);
                    }
                });
                break;
        }
    };

    $scope.duplicateEntity = (entity, type) => {
        $scope.hideChartEntityMenu();
        const copy = { ...angular.copy(entity), name: computeUniqueTitle(entity.name, type) };
        switch(type) {
            case 'RBND':
                modifyReusableDimensionInContext({ dimension: copy });
                break;
            case 'CUSTOM_MEASURE':
                modifyMeasuresInContext(undefined, { ...copy, formula: copy.function });
                break;
        }
    };

    $scope.openReusableDimensionPanel = (params) => {
        if (!params.fromChart) {
            WT1.event('reusable-dimension-create', {});
        }

        $scope.hideRBNDMenu();
        $scope.openModalRBND(params);
    };

    
    $scope.getCurrentChartsContext = () => {
        if ($scope.explore) {
            return ChartsContext.EXPLORE;
        } else if($scope.insight) {
            return ChartsContext.INSIGHT;
        } else if ($scope.acp) {
            return ChartsContext.ACP;
        } else if ($scope.mlTaskDesign) {
            return ChartsContext.ML;
        }

        Logger.error('Could not find current context in scope');
    };
    
    $scope.datasetColumnsFilter = (datasetColumn) => {
        const filterText = $scope.filterState.chartFilter?.trim().toLowerCase();
        if (!filterText) {
            return true;
        }
        
        if (datasetColumn.column && datasetColumn.column.toLowerCase().includes(filterText) ||
            datasetColumn.type && datasetColumn.type.toLowerCase().includes(filterText)) {
            return true;
        }

        const reusableDimensions = $scope.getReusableDimensions(datasetColumn);
        if (reusableDimensions?.length) {
            for (let i = 0; i < reusableDimensions.length; i++) {
                if (reusableDimensions[i].name && reusableDimensions[i].name.toLowerCase().includes(filterText)) {
                    return true;
                }
            }
        }
    
        return false;
    };
    
    /**
     * Get the charts list reference from the correct context:
     * Analysis, Insight or Explorer
     */
    const getChartsFromContext = () => {
        return $scope.explore && $scope.explore.charts
            || $scope.insight && $scope.insight.params && [$scope.insight.params]
            || $scope.acp && $scope.acp.charts
            || $scope.mlTaskDesign && $scope.mlTaskDesign.predictionDisplayCharts;
    };

    const getChartsCacheableContext = () => {
        const context = $scope.getCurrentChartsContext();
        const dataSpec = $scope.getDataSpec();

        return {
            datasetProjectKey: dataSpec.datasetProjectKey,
            datasetName: dataSpec.datasetName,
            context
        }
    };

    const saveCorrectContext = () => {
        $scope.saveShaker && $scope.saveShaker() || $scope.saveChart && $scope.saveChart();
    };

    /**
     * Fetch current measures context & remove/add measure
     * according to function parameters. Presence of `oldMeasure` will remove it
     * from context and presence of `newMeasure` will add it to it.
     */
    const modifyMeasuresInContext = (oldMeasure, newMeasure) => {
        const measures = getCustomMeasuresFromContext();
        oldMeasure && ChartCustomMeasures.removeByNameInPlace(measures, oldMeasure.name);
        newMeasure && measures.push(newMeasure);
        const customMeasuresLike = $scope.addCustomMeasuresToScopeAndCache(measures);

        ColumnAvailability.updateAvailableColumns(undefined, undefined, customMeasuresLike);
        saveCorrectContext();
    };

    const modifyReusableDimensionInContext = (params) => {

        const { dimensionRef: oldDimension, dimension: newDimension, isEditMode } = params;
        const applyChange = () => {
            const reusableDimensions = getReusableDimensionsFromContext();
            oldDimension && ChartRBNDs.removeByEqualityInPlace(reusableDimensions, oldDimension);
            newDimension && reusableDimensions.push(newDimension);
            $scope.addBinnedDimensionToScopeAndCache(reusableDimensions);
            saveCorrectContext();
        }

        if (isEditMode) {
            ChartRBNDs.setImpactedCharts(getChartsFromContext(), oldDimension);
            if (!Object.keys(ChartRBNDs.impactedCharts).length) {
                applyChange();
            } else {
                Dialogs.confirm($scope, translate('CHARTS.REUSABLE_DIMENSION.CONFIRM_EDIT', 'Confirm edit'), translate('CHARTS.REUSABLE_DIMENSION.CONFIRM_EDIT_WARNING_MESSAGE', 'This reusable dimension is used in at least one chart. Editing it will affect charts using it.'))
                .then(() => {
                    ChartRBNDs.onItemEdit(getChartsFromContext(), newDimension);
                    applyChange();
                }, () => {
                    $scope.openModalRBND(params);
                });
            }
        } else {
            applyChange();
        }     
    };

    const computeUniqueTitle = (name, type) => {
        const entities = type === 'CUSTOM_MEASURE' ? getCustomMeasuresFromContext() : getReusableDimensionsFromContext();
        const regex = /\s\(\d+\)$/;
        const baseTitle = name.replace(regex, '');
        let title = `${baseTitle} (2)`;
        let index = 2;
        while (entities.some(m => m.name === title)) {
            index++;
            title = `${baseTitle} (${index})`;
        }
    
        return title;
    };

    const getCustomMeasuresFromContext = () => {
        const cmc = getChartsCacheableContext();
        return ChartCustomMeasures.getItems(cmc.datasetProjectKey, cmc.datasetName, cmc.context);
    };

    const getReusableDimensionsFromContext = () => {
        const cmc = getChartsCacheableContext();
        return ChartRBNDs.getItems(cmc.datasetProjectKey, cmc.datasetName, cmc.context);
    };

    $scope.deleteEntity = (entity, type) => {
        switch(type) {
            case 'CUSTOM_MEASURE':
                ChartCustomMeasures.setImpactedCharts(getChartsFromContext(), entity);
                if (Object.keys(ChartCustomMeasures.impactedCharts).length) {
                    Dialogs.confirm($scope, "Confirm deletion", "This custom measure is used in at least one chart. Deleting it will remove it from the charts using it.").then(() => {
                        ChartCustomMeasures.onItemDelete(getChartsFromContext());
                        modifyMeasuresInContext(entity);
                    }, () => {});
                } else {
                    modifyMeasuresInContext(entity);
                }
                break;
            
            case 'RBND':
                ChartRBNDs.setImpactedCharts(getChartsFromContext(), entity);
                if (Object.keys(ChartRBNDs.impactedCharts).length) {
                    Dialogs.confirm($scope, "Confirm deletion", "This reusable dimension is used in at least one chart. Deleting it will remove it from the charts using it.").then(() => {
                        ChartRBNDs.onItemDelete(getChartsFromContext());
                        modifyReusableDimensionInContext({ dimensionRef: entity });
                    }, () => {});
                } else {
                    modifyReusableDimensionInContext({ dimensionRef: entity });
                }
                break;

        }
       
        $scope.hideChartEntityMenu();
    };

    $scope.addOrUpdateNewMeasure = (newMeasure, oldMeasure) => {
        if (oldMeasure) {
            ChartCustomMeasures.setImpactedCharts(getChartsFromContext(), oldMeasure);
            if (!Object.keys(ChartCustomMeasures.impactedCharts).length) {
                modifyMeasuresInContext(oldMeasure, newMeasure);
            } else {
                Dialogs.confirm($scope, translate("CHARTS.CUSTOM_MEASURES_EDITION.EDIT_AGGREGATION.CONFIRM", "Confirm edit"), translate("CHARTS.CUSTOM_MEASURES_EDITION.EDIT_AGGREGATION.BODY", "This custom measure is used in at least one chart. Editing it will affect charts using it."))
                .then(() => {
                    ChartCustomMeasures.onItemEdit(getChartsFromContext(), newMeasure);
                    modifyMeasuresInContext(oldMeasure, newMeasure);
                }, () => {});
            }
        } else {
            modifyMeasuresInContext(undefined, newMeasure);
        }
    };

    $scope.showChartEntityMenu = (entity, type, $event) => {
        $scope.chartEntityMenu.scope.type = type;
        $scope.chartEntityMenu.scope.entity = entity;
        $scope.chartEntityMenu.openAtXY($event.pageX, $event.pageY);
    };

    $scope.hideChartEntityMenu = () => {
        $scope.chartEntityMenu.closeAny();
    };

    $scope.showRBNDMenu = (datasetColumn, $event) => {
        $scope.datasetColumn = angular.copy(datasetColumn);
        $scope.RBNDMenu.openAtXY($event.pageX, $event.pageY);
    };

    $scope.hideRBNDMenu = () => {
        $scope.RBNDMenu.closeAny();
    };

    $scope.addCustomMeasuresToScopeAndCache = function(data) {
        const cmc = getChartsCacheableContext();
        ChartCustomMeasures.setItems(cmc.datasetProjectKey, cmc.datasetName, cmc.context, data);
        const measures = ChartCustomMeasures.getMeasuresLikeCustomMeasures(cmc.datasetProjectKey, cmc.datasetName, cmc.context);
        // read only for html templates
        $scope.customMeasures = measures;
        return measures;
    };

    $scope.addBinnedDimensionToScopeAndCache = function(data) {
        const cmc = getChartsCacheableContext();
        ChartRBNDs.setItems(cmc.datasetProjectKey, cmc.datasetName, cmc.context, data);
        $scope.reusableDimensions = ChartRBNDs.getItems(cmc.datasetProjectKey, cmc.datasetName, cmc.context);
    };

    $scope.openModalCustomMeasure = (oldMeasure) => {
        const customMeasures = getCustomMeasuresFromContext();
        return $scope._openModalCustomMeasure(oldMeasure, customMeasures.map(c => c.name));
    }

    $scope.openModalRBND = (params) => {
        let existingRBNDs = getReusableDimensionsFromContext().map(c => ({ name: c.name, column: c.column }));
        if (params.isEditMode) {
            existingRBNDs = existingRBNDs.filter(rbnd => !(rbnd.column === params.dimension.column && rbnd.name === params.dimension.name));
        }
        params = {
            ...params,
            existingRBNDs
        };
        return $scope._openModalRBND(params);
    }

    $scope.getReusableDimensions = (column) => {
        return $scope.reusableDimensions && $scope.reusableDimensions.filter(cb => cb.column === column.column && cb.type === column.type);
    }

    const getComplexity = formula => formula ? formula.length : 0;

    $scope._openModalCustomMeasure = (oldMeasure, allMeasureNames) => {
        $scope.sections = [null, null];

        const doclink = `<doclink page="/visualization/custom-aggregations" title="${translate('CHARTS.CUSTOM_MEASURES_EDITION.EXAMPLE_AND_DOCUMENTATION.DOCUMENTATION_TITLE','Documentation')}" show-icon></doclink>`
        
        $scope.exampleAndDocumentationBody = $sce.trustAsHtml(
            translate('CHARTS.CUSTOM_MEASURES_EDITION.EXAMPLE_AND_DOCUMENTATION.BODY', 'You can use your dataset columns to create new aggregations. You can combine existing columns, apply specific calculations, and define your own unique aggregations based on the requirements of your analysis. For more details about custom aggregation please visit the {{ documentationLink }}.', { documentationLink: doclink })
        );

        return CreateModalFromTemplate("/templates/shaker/custom-measures-edition.html", $scope, null, function(newScope) {

            const toColumn = (chartColumn) => {
                let type = chartColumn.type ? chartColumn.type.toLowerCase() : 'unspecified';
                if (chartColumn.type === 'ALPHANUM') {
                    type = 'string';
                }
                if (chartColumn.type === 'NUMERICAL') {
                    type = 'double'
                }
                return { name: chartColumn.column, type };
            }

            newScope.toggle = () => {
                newScope.sections.forEach(section => section.scope().foldableToggle());
            };
            newScope.inFullScreen = true;
            newScope.modalCanBeClosed = true;
            newScope.grelExpressionError = false;
            newScope.grelExpressionValid = false;
            newScope.grelExpressionEmpty = true;
            newScope.changeInProgress = false;
            newScope.oldMeasure = oldMeasure;
            newScope.newCustomMeasureNameIsValid = true;
            newScope.allMeasureNames = allMeasureNames;
            newScope.newCustomMeasure = oldMeasure ? { name: oldMeasure.column, formula: oldMeasure.function } : { name: '', formula: '' };
            newScope.formulaValid = !!oldMeasure;
            newScope.availableColumns = newScope.usableColumns.filter(c => c.column !== "__COUNT__").map(c => toColumn(c));
            newScope.isEditMode = oldMeasure && !oldMeasure.isDefaultMeasure;

            const validateExpressionMonoFuture = MonoFuture($scope, false, 250).wrap($scope.getExecutePromise);
            newScope.expressionValidator = (expression) => {
                // test the formula on a small dataset sample
                const dataSpec = angular.copy(newScope.getDataSpec());
                dataSpec.copySelectionFromScript = false;
                dataSpec.sampleSettings = {
                    selection: {
                        maxRecords: 10,
                        samplingMethod: 'HEAD_SEQUENTIAL',
                    }
                }

                return validateExpressionMonoFuture(
                    {
                        type: 'NO_PIVOT_AGGREGATED',
                        aggregations: [{
                            column: '',
                            function: 'CUSTOM',
                            type: 'CUSTOM',
                            customFunction: expression
                        }],
                        formulaValidation: true
                    },
                    false,
                    true,
                    newScope.requiredSampleId,
                    dataSpec
                );
            };

            newScope.onExpressionChange = () => {
                newScope.changeInProgress = true;
            };

            newScope.onCustomMeasureNameChange = () => {
                newScope.newCustomMeasureNameIsValid = !newScope.allMeasureNames.filter(n => n !== (oldMeasure && oldMeasure.name)).find(n => n === newScope.newCustomMeasure.name);
                newScope.measureForm.name.$setValidity('name', newScope.newCustomMeasureNameIsValid);
            };

            newScope.onEscape = (state) => {
                newScope.modalCanBeClosed = !state.completion;
            }

            newScope.canCloseModal = () => {
                return newScope.modalCanBeClosed;
            }

            newScope.canSaveOrEdit = () => {
                return newScope.formulaValid() && newScope.newCustomMeasure.name && newScope.newCustomMeasureNameIsValid;
            };

            newScope.formulaValid = () => {
                return newScope.newCustomMeasure.formula && !newScope.changeInProgress && newScope.grelExpressionValid;
            };

            newScope.onValidate = (result) => {
                newScope.changeInProgress = false;
                newScope.grelExpressionValid = result.valid !== false;
                newScope.grelExpressionError = result.error;
                newScope.grelExpressionEmpty = result.inputExpr.trim().length == 0;
                newScope.requiredSampleId = result.data.updatedSampleId;
                const pivotResponse = result.data.pivotResponse;
                if (pivotResponse) {
                    newScope.grelExpressionNeedFixup = pivotResponse.hasExpressionPlusWithNotNumeric;
                    newScope.newCustomMeasure.inferredType = pivotResponse.aggregations[0].type;
                }
            };

            newScope.fixupFormula = (newFormula, fixName = "plus") => {
                newScope.fixExpression(newFormula, fixName).then(data => {
                    newScope.newCustomMeasure.formula = data.data;
                });
            };


            if (!newScope.expression) {
                newScope.fixExpression = (newFormula, fixName = "plus") => {
                    const spec = $scope.getDataSpec();
                    return DataikuAPI.shakers.fixExpression($stateParams.projectKey, spec.datasetProjectKey, spec.datasetName, { ...spec.script, contextProjectKey: $stateParams.projectKey, origin: 'DATASET_EXPLORE' }, $scope.requestedSampleId, newFormula, fixName, -1, -1, -1, spec.copySelectionFromScript, true);
                };
            }

            newScope.onError = (data) => {
                newScope.changeInProgress = false;
                newScope.grelExpressionNeedFixup = false;
                newScope.grelExpressionValid = false;
                newScope.grelExpressionEmpty = newScope.newCustomMeasure.formula.trim().length == 0;
                newScope.grelExpressionError = data.message || 'Unexpected error.';
                if (newScope.getDataSpec().engineType === 'SQL' && (newScope.grelExpressionError.startsWith('ERROR: operator does not exist: text +') || newScope.grelExpressionError.match(/ERROR: operator does not exist: .+ \+ text/))) {
                    newScope.grelExpressionNeedFixup = true;
                }
            };

            newScope.formulaLabelClick = () => {
                newScope.$broadcast('codemirror-focus-input');
            }

            newScope.createOrUpdateMeasure = () => {
                $scope.addOrUpdateNewMeasure(newScope.newCustomMeasure, newScope.oldMeasure);
                newScope.resolveModal({ newCustomMeasure: newScope.newCustomMeasure, oldMeasure: newScope.oldMeasure });
            }

            if (newScope.newCustomMeasure.name) {
                newScope.onCustomMeasureNameChange();
            }
        }).then((data) => {
            const wt1Event = data.oldMeasure ? 'custom-measure-edited' : 'custom-measure-saved';
            const measureComplexity = getComplexity(data.newCustomMeasure && data.newCustomMeasure.formula);
            const oldMeasureComplexity = getComplexity(data.oldMeasure && data.oldMeasure.function);
            WT1.event(wt1Event, { measureComplexity, oldMeasureComplexity });
            return data;
        }, () => {
            WT1.event('custom-measure-cancel');
        });
    };

    /**
     * Opens a modal for configuring a RBND.
     * 
     * @typedef {Object} RBNDParams
     * @property {DimensionDef} dimension
     * @property {DimensionDef} dimensionRef
     * @property {boolean} fromChart
     * @property {boolean} isEditMode
     * @property {DimensionDef[]} existingRBNDs
     * 
     * @param {RBNDParams} params
     * @returns {void}
     */
    $scope._openModalRBND = (params) => { 
        return CreateModalFromTemplate("/templates/shaker/rbnd-edition.html", $scope, null, function(newScope, element) {

            const fillManualBins = (dimension) => {
                newScope.manualBinning = { values: (dimension && dimension.numParams && dimension.numParams.customBinValues || []).map((d) => ({ value: d })) };
            };

            const openNameModal = (existingRBNDs, applyChangeToDimensionRef) => {
                const options = { btnConfirm: translate('GLOBAL.DIALOGS.CREATE', 'Create') };
                promptRBNDName(existingRBNDs, translate('CHARTS.REUSABLE_DIMENSION.NAME_PROMPT_TITLE', 'Add a name to your Reusable Dimension'), '', options, applyChangeToDimensionRef);
            };

            const promptRBNDName = (existingRBNDs, title, name, options, applyChangeToDimensionRef) => {
                Dialogs.prompt($scope, title, translate('CHARTS.REUSABLE_DIMENSION.NAME', 'Name'), name, options).then((name) => {
                    if (existingRBNDs.some(d => d.name === name && d.column === newScope.dimension.column)) {
                        promptRBNDName(existingRBNDs, title, name, {...options, warningMessageContent: newScope.alreadyUsedNameWarning }, applyChangeToDimensionRef);
                        return;
                    } else {
                        newScope.propagateNewDimension(name, applyChangeToDimensionRef);
                    }
                }, () => {
                    $scope._openModalRBND( {...params, dimension: newScope.dimension });
                });
            };

            const onChartError = (message) => {
                newScope.response = null;
                newScope.requestError = true;
                newScope.showChartPreview = true;
                newScope.lastErrorMessage = message;
            };

            const getChart = () => {
                let dimension = {
                    column: newScope.dimension.column,
                    type: newScope.dimension.type,
                    isA: 'dimension',
                    numParams: {
                      mode: 'FIXED_NB',
                      nbBins: 35,
                      emptyBinsMode: 'ZEROS',
                      niceBounds: true
                    }
                };

                if (!shouldDisplayDistribution()) {
                    dimension = newScope.dimension;
                }

                const chart = { 
                    def: {
                        $isInReusableDimensionPreview: true,
                        type: 'grouped_columns',
                        variant: 'normal',
                        genericMeasures: [{
                            displayAxis: 'axis1',
                            column: newScope.dimension.column,
                            type: newScope.dimension.type,
                            function: 'COUNT',
                            isA: 'measure'
                        }],
                        genericDimension0: [dimension]
                    },
                    theme: {
                        ...newScope.chartTheme,
                    }
                };

                return chart;
            };

            let distributionModePivotResponse;

            // set scope context
            Object.assign(newScope, params)
            newScope.DKUPivotCharts = DKUPivotCharts;
            newScope.ChartFeatures = ChartFeatures;
            newScope.ChartDimension = ChartDimension;
            newScope.newRBNDNameValid = true;
            newScope.isRBND = true;
            newScope.noClickableAxisLabels = true;
            newScope.dirty = false;
            newScope.canResetCustomBins = false;
            newScope.showChartPreview = false;
            newScope.toggleIsApplyBinningMode = false;
            newScope.requestError = false;
            newScope.alreadyUsedNameWarning = translate('CHARTS.REUSABLE_DIMENSION.ALREADY_USED_NAME_WARNING', 'This name is already taken by another reusable dimension');
            newScope.chartTheme = newScope.getCurrentChartFromContext()?.theme || DefaultDSSVisualizationTheme;
            
            // fill values and configuration
            fillManualBins(newScope.dimension);
            ChartTypeChangeHandler.autocompleteGenericDimension({ type: 'dummy', variant: 'dummy' }, newScope.dimension);
            let pivotReponse = {};
            const axesDef =  { x: 0 };

            let debounce;
            newScope.$watch('dimension', function(nv, ov) {
                newScope.dirty = !angular.equals(newScope.dimensionRef, newScope.dimension) && newScope.newRBNDNameValid;
                newScope.canResetCustomBins = newScope.isEditMode && !angular.equals(newScope.dimension.numParams.customBinValues, newScope.dimensionRef.numParams.customBinValues);
                clearTimeout(debounce);
                debounce = setTimeout(() => setTimeout(() => drawChart(false, ov, nv), 200), 500);
            }, true);

            newScope.onTogglePreviewChange = (mode) => {
                if (newScope.toggleIsApplyBinningMode !== mode) {
                    newScope.toggleIsApplyBinningMode = mode;
                    drawChart(true);
                }
            };

            newScope.onRBNDNameChange = () => {
                newScope.newRBNDNameValid = !newScope.existingRBNDs.find(rbnd => rbnd.name === newScope.dimension.name && rbnd.column === newScope.dimension.column);
                newScope.rbndForm.name.$setValidity('name', newScope.newRBNDNameValid);
            };
            
            const shouldDisplayDistribution = () => {
                return !newScope.toggleIsApplyBinningMode || (!!newScope.dimension.numParams.customBinValues && newScope.dimension.numParams.customBinValues.length === 0 && newScope.dimension.numParams.mode === 'CUSTOM');
            };
            
            newScope.showMe = () => {
                drawChart(true);
            };

            const needsCustomBinningRedraw = (oldDimension, newDimension) => {
                if (!oldDimension?.numParams || !newDimension?.numParams) {
                    return false;
                }
                const oldMode = oldDimension.numParams.mode;
                const newMode = newDimension.numParams.mode;
                const oldCustomValues = oldDimension.numParams.customBinValues;
                const newCustomValues = newDimension.numParams.customBinValues;
                const modeChanged = oldMode !== newMode;
                const oneModeIsCustom = oldMode === 'CUSTOM' || newMode === 'CUSTOM';
                const customValuesChanged = newMode === 'CUSTOM' && !angular.equals(oldCustomValues, newCustomValues);
                return (modeChanged && oneModeIsCustom) || customValuesChanged;
            }

            const drawChart = (forceRedraw = false, oldDimension, newDimension) => {
                if (!forceRedraw && !newScope.showChartPreview) return;

                const displayDistribution = shouldDisplayDistribution();
                const customBinningRequiresRedraw = needsCustomBinningRedraw(oldDimension, newDimension);
                const shouldRedraw = forceRedraw || !displayDistribution || customBinningRequiresRedraw;

                if (!shouldRedraw) return;

                const isCustomBinningMode = newScope.dimension.numParams.mode === 'CUSTOM';
                const withRefLines = !newScope.toggleIsApplyBinningMode && displayDistribution && isCustomBinningMode;
                
                const chart = getChart();
                chart.def.referenceLines = [];
                withRefLines && (newScope.dimension.numParams.customBinValues || []).forEach(element => {
                    chart.def.referenceLines.push({
                        sourceType: 'Constant',
                        constantValue: element,
                        displayValue: false,
                        lineFormatting: {
                        color: 'black',
                        type: 'DASHED',
                            size: 2
                        },
                        axis: {
                            type: 'X_AXIS'
                        }
                    })
                });

                const renderChart = () => {
                    ChartTypeChangeHandler.fixupChart(chart.def, chart.theme);
                    ChartTypeChangeHandler.fixupSpec(chart);
                    const rootElement = element.find('.pivot-charts').css('display', '');
                    if (rootElement && chart.theme && chart.theme.generalFormatting && chart.theme.generalFormatting.fontFamily) {
                        element.css('--visualization-font-family', chart.theme.generalFormatting.fontFamily);
                    }
                    DKUPivotCharts.GroupedColumnsChart(rootElement, chart.def, newScope, axesDef, pivotReponse);
                };

                const onSucess = (data) => {
                    newScope.response = data;
                    pivotReponse = data.result.pivotResponse;
                    newScope.showChartPreview = true;
                    newScope.requestError = false;
                };

                const useCachedDistributionData = displayDistribution && distributionModePivotResponse;
                if (useCachedDistributionData) {
                    onSucess(distributionModePivotResponse);
                    renderChart();
                } else {
                    newScope.initChartCommonScopeConfig(newScope, chart);
                    ChartTypeChangeHandler.fixupChart(chart.def, chart.theme);
                    ChartTypeChangeHandler.fixupSpec(chart);
                    const request = ChartRequestComputer.compute(chart.def, 10, 10, newScope.chartSpecific);
                    const executePivotRequest = MonoFuture(newScope).wrap(newScope.getExecutePromise);
                    executePivotRequest(request, false, false)
                        .update((data) => {
                            newScope.response = data;
                        })
                        .success((data) => {
                            onSucess(data);
                            renderChart();
                            if (displayDistribution) {
                                distributionModePivotResponse = data;
                            }
                        }).error((error) => {
                            onChartError(error?.message ||  translate('CHARTS.REUSABLE_DIMENSION.SOMETHING_WENT_WRONG', 'Something went wrong'));
                        });
                }
            }

            newScope.createOrUpdateBinning = (applyChangeToDimensionRef = false) => {
                newScope.resolveModal({ dimension: newScope.dimension, isEditMode: newScope.isEditMode, fromChart: newScope.fromChart, changeAppliedToSourceReusableDimension: applyChangeToDimensionRef });
                if (newScope.isEditMode) {
                    newScope.propagateNewDimension(newScope.dimension.name);
                } else {
                    openNameModal(newScope.existingRBNDs, applyChangeToDimensionRef);
                }
            };

            newScope.propagateNewDimension = (name, applyChangeToDimensionRef = false) => {
                newScope.dimension.name = name;
                newScope.dimension.isRBND = true;
                if (applyChangeToDimensionRef) {
                    Object.assign(newScope.dimensionRef, newScope.dimension);
                }
                const params = {
                    dimensionRef: newScope.dimensionRef,
                    dimension: newScope.dimension,
                    isEditMode: newScope.isEditMode,
                    fromChart: newScope.fromChart,
                }
                modifyReusableDimensionInContext(params);
            };

            newScope.applyBinningToChart = () => {
                newScope.resolveModal({ dimension: newScope.dimension, isEditMode: newScope.isEditMode, fromChart: newScope.fromChart, changeAppliedToSourceReusableDimension: false });
                Object.assign(newScope.dimensionRef, angular.copy(newScope.dimension));
            };

            newScope.reset = () => {
                newScope.dimension.numParams.customBinValues = angular.copy(newScope.dimensionRef.numParams.customBinValues);
                fillManualBins(newScope.dimension);
            };

            newScope.computeBins = () => {
                const customBinValues = newScope.manualBinning.values.map(input => parseFloat(input.value)).filter(value => !isNaN(value));
                if (!angular.equals(newScope.dimension.numParams.customBinValues, customBinValues)) {
                    newScope.dimension.numParams.customBinValues = customBinValues;
                }
                return ChartDimension.getCustomBins(customBinValues);
            };
        }).then((params) => {
            WT1.event("reusable-dimension-panel-save", 
                { 
                    isReusableDimension: !!params.dimension.isRBND, 
                    mode: params.dimension?.numParams?.mode,
                    fromChart: !!params.fromChart,
                    isEditMode: !!params.isEditMode,
                    changeAppliedToSourceReusableDimension: !!params.changeAppliedToSourceReusableDimension
                });
        });
    };

    if ($stateParams.tabSelect === undefined) {
        $timeout(function() {
            $scope.$broadcast("tabSelect", "columns");
        }, 0);
    }
});

// Chart management for Analyses & Datasets
app.directive("shakerChartsCommon", function() {
    return {
        scope: true,
        priority : 100,
        controller  : 'ShakerChartsCommonController'
    }
});

})();

;
(function(){
    'use strict';
    
    const app = angular.module('dataiku.shaker');

    app.component('imageView', {
        bindings: {
            requestedSampleId: '<',
            inputDatasetProjectKey: "<",
            inputDatasetName: "<",
            imageViewSettings: '<',
            nbColumnsInDataset: "<",
            shakerHooks: "=",
            filterRequestBuilderFn: "<"
        },
        controller: function(){
            const $ctrl = this;
            
            function setScriptAndFilters() {
                $ctrl.resolvedScript = $ctrl.shakerHooks.shakerForQuery();
                $ctrl.filters = {'elements': $ctrl.filterRequestBuilderFn($ctrl.resolvedScript.explorationFilters)};
            }

            this.$onInit = () => {
                setScriptAndFilters();
                // We make sure to update script & table when table has been refreshed on the table view, 
                // for instance if the search was changed
                $ctrl.shakerHooks.onTableRefresh = function() {
                    setScriptAndFilters();
                }
            }
        },
        templateUrl: "/templates/shaker/image-view-component.html"
    });

    app.component('imageViewSettings', {
        bindings: {
            settings: "=",
            columns: "<"
        },
        controller: function($scope, DataikuAPI, FutureProgressModal, $stateParams) {
            const $ctrl = this;

            $ctrl.autoDetectCategoriesForImageView = function() {
                DataikuAPI.shakers.autoDetectCategoriesForImageView($stateParams.projectKey, $stateParams.datasetName, $ctrl.settings).success(function(data) {
                    const newScope = $scope.$new();
                    FutureProgressModal.show(newScope, data, "Auto detecting categories for image view").then(function(newCategories) {
                        if (newCategories) { // will be undefined if computation was aborted
                            $ctrl.settings.annotationParams.annotationCategories = newCategories;
                        }
                    });
                }).error(setErrorInScope.bind($scope));
            };

            $ctrl.autoDetectButtonDisabledMessage = function() {
                if (!$ctrl.settings.annotationParams || !$ctrl.settings.annotationParams.enabled) { // Should not happen
                    return "Annotation params must be enabled";
                }
                if (!$ctrl.settings.annotationParams.annotationColumn) {
                    return "Annotation column must be defined";
                }
                if (!$ctrl.settings.annotationParams.annotationType) {
                    return "Annotation type must be defined";
                }
                return null;
            }
        },
        templateUrl: "/templates/shaker/image-view-settings-component.html"
    });

    app.component('filterAdder', {
        bindings: {
            explorationFilters: "<",
            tableHeaders: "<",
            addColumnFilterCallback: "<"
        },
        templateUrl: "/templates/shaker/filter-adder-component.html",
        controller: function() {
            const $ctrl = this;

            // This component "hacks" the `dku-bs-select` directive by just making it a selector and not really an input. As soon as a
            // column is selected, we put back the placeholder value to be able to filter other columns.
            $ctrl.addColumnFilter = function() {
                const column = $ctrl.tableHeaders.find(h => h.name === $ctrl.tmpFilterToAdd);
                $ctrl.addColumnFilterCallback(column.name, {}, 'full_string', column.selectedType.name, column.isDouble);
                $ctrl.tmpFilterToAdd = null; // Resetting the tmp filter to have the widget displaying placeholder again
            };

            $ctrl.allColumnsFiltered = function() {
                return $ctrl.filteredColumnNames && $ctrl.allColumnNames && $ctrl.filteredColumnNames.length === $ctrl.allColumnNames.length;
            };

            $ctrl.removeFilteredColumns = function (columnName) {
                return $ctrl.filteredColumnNames.indexOf(columnName) < 0;
            };

            function updateColumnNames(filteredColumnNames) {
                $ctrl.filteredColumnNames = filteredColumnNames;
                $ctrl.allColumnNames = $ctrl.tableHeaders.map(h => h.name);
            }
            
            let currentFilteredColumnNames = [];
            this.$onInit = () => {
                updateColumnNames(currentFilteredColumnNames);
                $ctrl.tmpFilterToAdd = null;
            }

            this.$doCheck = () => {
                if (!angular.equals(currentFilteredColumnNames, $ctrl.explorationFilters.map(f => f.column))) {
                    currentFilteredColumnNames = $ctrl.explorationFilters.map(f => f.column);
                    updateColumnNames(currentFilteredColumnNames);
                }
            }
        }
    });

    app.component('imagePreviewComponent', {
        bindings: {
            "imagePath": "<",
            "imageViewSettings": "<",
            "popupContainer": "<"
        },
        template: `
            <ng2-dataset-view-image-preview [image-path]="$ctrl.imagePath" [image-view-settings]="$ctrl.imageViewSettings"></ng2-dataset-view-image-preview>`,

        controller: function(DataikuAPI, $scope, $stateParams) {
            const $ctrl = this;

            // bindings are not properly set now so we CAN'T just do: $ctrl.$postLink = $ctrl.popupContainer.onPopupCreated
            $ctrl.$postLink = () => $ctrl.popupContainer.onPopupCreated();
        }
    })
})();
;
(function(){
'use strict';

const app = angular.module('dataiku.shaker');


app.directive("shakerExplorePristine", function($timeout, $q, Assert, DataikuAPI, WT1, ActivityIndicator, TopNav, DKUtils, DatasetErrorCta, PageSpecificTourService, OpalsService, OpalsMessageService) {
    return {
        scope: true,
        controller: function ($rootScope, $scope, $stateParams, $state) {

            /* ********************* Callbacks for shakerExploreBase ******************* */
            function handleError(deferred, ...args) {
                setErrorInScope.bind($scope)(...args);
                deferred.reject($scope.fatalAPIError);
            }

            $scope.shakerHooks.saveForAuto = function () {
                var deferred = $q.defer();
                resetErrorInScope($scope);
                var shakerData = $scope.getShakerData();

                if ($scope.isRecipe) {
                    throw "Should not call this for a recipe";
                } else {
                    if (!$scope.canWriteProject || $scope.canWriteProject()) {
                        DataikuAPI.explores
                            .saveScript($stateParams.projectKey, $stateParams.datasetName, shakerData)
                            .then(function (data) {
                                $scope.originalShaker = shakerData;
                                deferred.resolve();
                            })
                            .catch((...args) => handleError(deferred, ...args));
                    } else {
                        deferred.resolve();
                    }
                }
                return deferred.promise;
            };

            $scope.shakerHooks.setColumnMeaning = function(column, newMeaning){
                DataikuAPI.explores.setColumnMeaning($stateParams.projectKey, $stateParams.datasetName,
                    column.name, newMeaning).success(function(data){
                    $scope.refreshTable(false);
                }).error(setErrorInScope.bind($scope));
            };

            $scope.shakerHooks.getSetColumnStorageTypeImpact = function(column, newType){
                return DataikuAPI.explores.getSetColumnStorageTypeImpact($stateParams.projectKey, $stateParams.datasetName, column.name, newType);
            };

            $scope.shakerHooks.setColumnStorageType = function(column, newType, actions){
                DataikuAPI.explores.setColumnStorageType($stateParams.projectKey, $stateParams.datasetName,
                    column.name, newType, actions).success(function(data){
                        $scope.refreshTable(false);
                        if (data.reload) {
                            DKUtils.reloadState();
                        } else if (data.refreshSample) {
                            $scope.shaker.explorationSampling._refreshTrigger = new Date().getTime();
                            $scope.forgetSample();
                            $scope.autoSaveForceRefresh();
                        } else {
                            ActivityIndicator.success("Dataset schema saved - You might need to refresh the sample", 4000);
                        }
                }).error(function(a,b,c) {
                    ActivityIndicator.error("Failed to change column name, check sampling pane", 4000);
                    setErrorInScope.bind($scope)(a,b,c)
                });
            };

            $scope.shakerHooks.updateColumnDetails = function(column) {
                Assert.trueish(column, 'cannot update column with null');
                DataikuAPI.explores.updateColumn($stateParams.projectKey, $stateParams.datasetName, column).success(function(data){
                    // So the RHP knows the change without having to refresh the page
                    $rootScope.$broadcast('columnChanged', { column });
                    $scope.refreshTable(false);
                    ActivityIndicator.success("Dataset schema saved - You might need to refresh the sample", 4000);
                }).error(setErrorInScope.bind($scope));
            };
 
            /* ********************* Main ******************* */

            // Set base context and call baseInit
            Assert.inScope($scope, 'shakerHooks');

            TopNav.setLocation(TopNav.TOP_FLOW, 'datasets', TopNav.TABS_DATASET, "explore")

            $scope.table = null;
            $scope.scriptId = "__pristine__";
            $scope.shakerWithSteps = false;
            $scope.shakerReadOnlyActions = true;
            $scope.shakerWritable = false;
            $scope.isCompareCellAvailable = true;
            $scope.inputDatasetProjectKey = $stateParams.projectKey;
            $scope.inputDatasetName = $stateParams.datasetName;
            $scope.inputDatasetSmartName = $stateParams.datasetName;

            WT1.event("shaker-explore-open");

            $scope.$watch("projectSummary", function(nv, ov) {
                $scope.shakerState.writeAccess = $scope.isProjectAnalystRW();
            });

            //For datasetErrorCTA directive (CTA in case of error while loading dataset sample)

            $scope.updateUiState = DatasetErrorCta.getupdateUiStateFunc($scope);

            $scope.$watch("datasetFullInfo", _ => $scope.updateUiState($scope.shakerState.runError), true);
            $scope.$watch("shakerState", _ => $scope.updateUiState($scope.shakerState.runError), true);
            $scope.$watch("table", _ => $scope.updateUiState($scope.shakerState.runError));

            // Load shaker, set the necessary stuff in scope and call the initial refresh
            DataikuAPI.explores.getScript($stateParams.projectKey, $stateParams.datasetName).success(function(shaker) {
                $scope.shaker = shaker;
                $scope.shaker.origin = "DATASET_EXPLORE";
                $scope.fixupShaker();
                $scope.requestedSampleId = null;
                $scope.refreshTable(false, "SkipDebounceAndRunImmediately");

            }).error(setErrorInScope.bind($scope));

            $timeout(function() { $scope.$broadcast("tabSelect", "Filters") });

            // Load stuff for "edit last analysis"
            DataikuAPI.analysis.listOnDataset($stateParams.projectKey, $stateParams.datasetName).success(function(data) {
                data.sort(function(a, b) {
                    return b.lastModifiedOn - a.lastModifiedOn;
                });
                if (data.length) {
                    Mousetrap.bind("g l a", $state.go.bind($state,
                        "projects.project.analyses.analysis.script", {analysisId: data[0].id}));
                    $scope.$on("$destroy", function(){
                        Mousetrap.unbind("g l a")
                    });
                }
            }).error(setErrorInScope.bind($scope));

            // Fetch image view settings
            DataikuAPI.datasets.getPublicInfo($stateParams.projectKey, $scope.inputDatasetProjectKey, $scope.inputDatasetName).success(function(data) {
                $scope.imageViewSettings = data.dataset.imageViewSettings;
                $scope.nbColumnsInDataset = data.dataset.schema.columns.length;
            }).error(setErrorInScope.bind($scope));

            let imageViewDefaultTabSet = false;
            $scope.shakerHooks.afterTableRefresh = function() {
                if (imageViewDefaultTabSet) {
                    return;
                }
                const imageViewEnabled = $scope.imageViewSettings && $scope.imageViewSettings.enabled;
                const defaultImageView = $scope.imageViewSettings && $scope.imageViewSettings.defaultViewOnExplore;
                // Note that this is a simplified logic to not display the image view if dataset is empty or being built, as there
                // are already nice call to actions in the table view.
                const tableHasRows = $scope.table && $scope.table.initialRows > 0;
                if ($scope.shakerState && imageViewEnabled && defaultImageView && tableHasRows) {
                    $scope.shakerState.activeView = 'images';
                }
                imageViewDefaultTabSet = true;
                if (PageSpecificTourService.canStartExploreTour($scope)) {
                    PageSpecificTourService.startExploreTour({ scope: $scope, fromContext: 'dataset-explore' });
                    OpalsService.sendPageSpecificTourRecommendation(OpalsMessageService.PAGE_SPECIFIC_TOURS_RECOMMENDATIONS.EXPLORE);
                } else {
                    OpalsService.sendPageSpecificTourRecommendation(null);
                }
            }

            $scope.imageViewDisabledMessage = function() {
                if (!$scope.imageViewSettings) {
                    return "Image view settings are not properly loaded";
                }
                if (!$scope.imageViewSettings.managedFolderSmartId) {
                    return "Managed folder not properly set";
                }
                if (!$scope.imageViewSettings.pathColumn) {
                    return "Path column not properly set";
                }
                if (!$scope.table || !$scope.table.headers || !$scope.table.headers.length) {
                    return "Cannot fetch data from dataset";
                }
                if (!$scope.table.allColumnNames || !$scope.table.allColumnNames.includes($scope.imageViewSettings.pathColumn)) {
                    return "Path column not in dataset columns";
                }
                const selectColumnsMode = $scope.table.newColumnsSelection && $scope.table.newColumnsSelection.mode === "SELECT";
                const pathColumnNotSelected = selectColumnsMode && ($scope.table.newColumnsSelection.list &&
                                                                   !$scope.table.newColumnsSelection.list.some(c => c.d && c.name === $scope.imageViewSettings.pathColumn));
                if (pathColumnNotSelected) {
                    return "Path column not selected";
                }

                const hasAnnotation = $scope.imageViewSettings.annotationParams && $scope.imageViewSettings.annotationParams.enabled;

                if (hasAnnotation && !$scope.imageViewSettings.annotationParams.annotationColumn) {
                    return "Annotation column not set";
                }

                const annotationNotSelected = hasAnnotation && selectColumnsMode && ($scope.table.newColumnsSelection.list &&
                                                                                     !$scope.table.newColumnsSelection.list.some(c => c.d && c.name === $scope.imageViewSettings.annotationParams.annotationColumn));

                if (annotationNotSelected) {
                    return "Annotation column not selected";
                }
                return null;
            }

            const unregisterExploreTourListener = $rootScope.$on('startExploreTour', function() {
                PageSpecificTourService.startExploreTour({ scope: $scope, fromContext: 'opals' });
            });
            $scope.$on("$destroy", function() {
                unregisterExploreTourListener();
            });
        }
    }
});

app.directive("shakerExploreStreamingEndpoint", function($timeout, $q, Assert, DataikuAPI, WT1, ActivityIndicator, TopNav, DKUtils, DatasetErrorCta) {
    return {
        scope: true,
        controller: function ($scope, $stateParams, $state) {

            /* ********************* Callbacks for shakerExploreBase ******************* */

            $scope.shakerHooks.saveForAuto = function() {
                var deferred = $q.defer();
                resetErrorInScope($scope);
                var shakerData = $scope.getShakerData();

                if ($scope.isRecipe) {
                    throw "Should not call this for a recipe";
                } else {
                    DataikuAPI.explores.saveCaptureScript($stateParams.projectKey, $stateParams.streamingEndpointId,
                        shakerData).success(function(data){
                        $scope.originalShaker = shakerData;
                        deferred.resolve();
                    }).error(setErrorInScope.bind($scope));
                }
                return deferred.promise;
            };

            $scope.shakerHooks.setColumnMeaning = function(column, newMeaning){
            };

            $scope.shakerHooks.getSetColumnStorageTypeImpact = function(column, newType){
                return null;
            };

            $scope.shakerHooks.setColumnStorageType = function(column, newType, actions){
            };

            $scope.shakerHooks.updateColumnDetails = function(column) {
            };

            /* ********************* Main ******************* */

            // Set base context and call baseInit
            Assert.inScope($scope, 'shakerHooks');

            $scope.table = null;
            $scope.scriptId = "__pristine__";
            $scope.shakerWithSteps = false;
            $scope.shakerWritable = false;
            $scope.inputDatasetProjectKey = $stateParams.projectKey;
            $scope.inputStreamingEndpointId = $stateParams.streamingEndpointId;

            WT1.event("shaker-explore-open");

            $scope.$watch("projectSummary", function(nv, ov) {
                $scope.shakerState.writeAccess = $scope.isProjectAnalystRW();
            });

            //For datasetErrorCTA directive (CTA in case of error while loading dataset sample)

            $scope.updateUiState = DatasetErrorCta.getupdateUiStateFunc($scope);

            $scope.$watch("streamingEndpoint", _ => $scope.updateUiState($scope.shakerState.runError), true);
            $scope.$watch("shakerState", _ => $scope.updateUiState($scope.shakerState.runError), true);
            $scope.$watch("table", _ => $scope.updateUiState($scope.shakerState.runError));

            // Load shaker, set the necessary stuff in scope and call the initial refresh
            DataikuAPI.explores.getCaptureScript($stateParams.projectKey, $stateParams.streamingEndpointId).success(function(shaker) {
                $scope.shaker = shaker;
                $scope.shaker.origin = "DATASET_EXPLORE";
                if ($scope.shaker.explorationSampling && $scope.shaker.explorationSampling.selection && $scope.shaker.explorationSampling.selection.timeout < 0) {
                    $scope.shaker.explorationSampling.selection.timeout = 10;
                }
                if ($scope.shaker.vizSampling && $scope.shaker.vizSampling.selection && $scope.shaker.vizSampling.selection.timeout < 0) {
                    $scope.shaker.vizSampling.selection.timeout = 10;
                }
                $scope.fixupShaker();
                $scope.requestedSampleId = null;
                $scope.refreshTable(false);

            }).error(setErrorInScope.bind($scope));

            $timeout(function() { $scope.$broadcast("tabSelect", "Filters") });
        }
    }
});


}());

;
(function(){
'use strict';

const app = angular.module('dataiku.shaker');

app.component('partitionSummary', {
    templateUrl: "/templates/datasets/fragments/partition-summary.html",
    bindings: {
        selection: "<",
        partitioned: "<",
        projectKey: "<",
        datasetName: "<",
    }
});

app.service("ElasticSearchQueryUtils", function() {
    return {
        escapeColumn: (columnName) => {
            // Taken from https://www.elastic.co/guide/en/elasticsearch/reference/current/query-dsl-query-string-query.html#_reserved_characters
            // Selected text is quoted, so only '"' is escaped. Special character in field names need to be escaped
            return columnName.replace(/[&|=-\s+<>!^\\~*?:()[\]{}"]/g, m => `\\${m}`);
        },
        getShakerHeaderOptions: () => {
            return {
                showName: true,
                showStorageType: false,
                showMeaning: false,
                showProgressBar: false,
                showHeaderSeparator: true,
            };
        },
        setShakerCommonParameters: (scope) => {
            scope.table = null;
            scope.scriptId = "__pristine__";
            scope.shakerWithSteps = false;
            scope.shakerReadOnlyActions = true;
            scope.searchableDataset = true;
            scope.shakerWritable = false;
            scope.setPageHeightWidthFromChunk = true;
            scope.highlightTagName = "MARK";
        }
    };
});


app.component('savedQuery', {
    templateUrl: "/templates/datasets/fragments/saved-es-query.html",
    bindings: {
        searchQuery: "<",
        projectKey: "<",
        inputDatasetSmartName: "<",
        partitioned: "<",
        replaceQuery: "<",
        canWriteProjectContent: "<",
        inDashboard: "<"
    },
    require: {
        apiErrorContext: "^apiErrorContext"
    },
    controller: function($scope, DataikuAPI, CreateModalFromTemplate, Dialogs) {
        const ctrl = this;
        ctrl.keys = Object.keys;

        ctrl.$onInit = () => {
            ctrl.refreshList();
            ctrl.selectedSavedQuery = null;
        };

        ctrl.$doCheck = () => {
            if (ctrl.selectedSavedQuery !== null && ctrl.searchQuery && !angular.equals(ctrl.selectedSavedQuery.query, ctrl.searchQuery)) {
                ctrl.selectedSavedQuery = null;  // reset selected query if search was modified
            }
        }

        ctrl.filteredSavedQueries = () => {
            if (ctrl.selectedSavedQuery === null) {
                return ctrl.savedQueries;
            }
            return ctrl.savedQueries.filter(q => q.name !== ctrl.selectedSavedQuery.name);
        }

        ctrl.refreshList = () => {
            DataikuAPI.explores.getInteractiveSearchSavedQueries(ctrl.projectKey, ctrl.inputDatasetSmartName)
            .then(({data}) => {
                ctrl.savedQueries = data;
            })
            .catch(ctrl.apiErrorContext.setError);
        }

        ctrl.deleteQuery = (savedQuery, clearSelected) => {
            Dialogs.confirmInfoMessages($scope, 'Confirm query deletion', null, `Are you sure you want to delete query '${sanitize(savedQuery.name)}'?`, false).then(function() {
                DataikuAPI.explores.deleteInteractiveSearchSavedQuery(ctrl.projectKey, ctrl.inputDatasetSmartName, savedQuery.name)
                    .then(() => {
                        ctrl.refreshList();
                        // Unselect selected query if it was the one deleted
                        if (ctrl.selectedSavedQuery !== null && savedQuery.name === ctrl.selectedSavedQuery.name) {
                            ctrl.selectedSavedQuery = null;
                        }
                    })
                    .catch(ctrl.apiErrorContext.setError);
            });
        };

        ctrl.replaceAndUpdateQuery = (savedQuery) => {
            ctrl.selectedSavedQuery = savedQuery;
            ctrl.replaceQuery(angular.copy(savedQuery.query));  // copy here to prevent modifying the saved query
        }

        ctrl.openSaveQueryModal = (queryToEdit) => {
            CreateModalFromTemplate("/templates/datasets/fragments/save-es-query-modal.html", $scope, null, function(newScope) {
                let oldQueryName = null;
                if (queryToEdit) {
                    newScope.modalTitle = "Edit query";
                    newScope.data = angular.copy(ctrl.selectedSavedQuery);
                    oldQueryName = queryToEdit.name;
                } else {
                    newScope.modalTitle = "Save query";
                    newScope.data = {name: "", description: "", query: angular.copy(ctrl.searchQuery)};
                }
                newScope.partitioned = ctrl.partitioned;

                newScope.isQueryNameUnique = function (v) {
                    if (v === null){
                        return true;
                    }
                    for (const query of ctrl.savedQueries) {
                        if (query.name === v && oldQueryName !== query.name) {
                            return false;
                        }
                    }
                    return true;
                };

                newScope.saveQuery = function() {
                    DataikuAPI.explores.saveInteractiveSearchQuery(ctrl.projectKey, ctrl.inputDatasetSmartName, newScope.data, oldQueryName)
                    .then(() => {
                        newScope.dismiss();
                        ctrl.refreshList();
                    }).then(() => {
                        ctrl.selectedSavedQuery = newScope.data;
                    })
                    .catch(setErrorInScope.bind(newScope));
                }
            });
        }
    }
});


app.controller("_ShakerExploreCommonElasticController", function($scope, ElasticSearchQueryUtils, Assert) {
    Assert.inScope($scope, 'shakerHooks');
    $scope.shakerHooks.setColumnMeaning = function(column, newMeaning){
    };

    $scope.shakerHooks.getSetColumnStorageTypeImpact = function(column, newType){
        return null;
    };

    $scope.shakerHooks.setColumnStorageType = function(column, newType, actions){
    };

    $scope.shakerHooks.updateColumnDetails = function(column) {
    };

    /* ********************* Main ******************* */

    // Set base context and call baseInit
    ElasticSearchQueryUtils.setShakerCommonParameters($scope);
});


app.controller("_InteractiveSearchController", function($scope, $stateParams, MonoFuture, WT1, ElasticSearchQueryUtils) {
    $scope.interactiveSearchQuery = {
        queryString: $stateParams.queryString || "",
        datasetSelection: { selectedPartitions: [], latestPartitionsN: 1, partitionSelectionMethod: "ALL" }
    };

    $scope.isDirty = function() {
        return !angular.equals($scope.previousInteractiveSearchQuery, $scope.interactiveSearchQuery);
    }

    $scope.resetSearch = function() {
        $scope.interactiveSearchQuery.queryString = "";
        $scope.searchInteractive()
    }

    $scope.replaceQuery = function(query) {
        $scope.interactiveSearchQuery = query;
        $scope.searchInteractive()
    }

    // If editing interactively from a recipe, consider we have input a query and enable search
    $scope.previousInteractiveSearchQuery = $stateParams.queryString ? {} : angular.copy($scope.interactiveSearchQuery);

    $scope.searchInteractive = function(wt1Origin) {
        $scope.previousInteractiveSearchQuery = angular.copy($scope.interactiveSearchQuery);

        $scope.refreshTable(true);
        if (wt1Origin) {
            WT1.event($scope.searchWT1EventName, {origin: wt1Origin});
        }
    }

    const quoteContent = function(string) {
        // Content is quoted to prevent problems with special characters, so we need to escape '"' in the text
        return '"' + string.replace(/"/g, "\\\"") + '"';
    }

    const searchInteractiveFilterNoEscapeContent = function(columnName, content, wt1Origin) {
        columnName = ElasticSearchQueryUtils.escapeColumn(columnName);
        if ($scope.interactiveSearchQuery.queryString) {
            $scope.interactiveSearchQuery.queryString = "(" + $scope.interactiveSearchQuery.queryString + ") AND " + columnName + ":"  + content;
        } else {
            $scope.interactiveSearchQuery.queryString = columnName + ":" + content
        }
        $scope.searchInteractive(wt1Origin);
    }

    $scope.searchInteractiveFilter = function(columnName, content) {
        content = quoteContent(content);
        searchInteractiveFilterNoEscapeContent(columnName, content, "selection");
    }

    // Filtering
    $scope.addDateRangeToInteractiveFilter = function(columnName, fromDate, toDate) {
        let content;
        if (!fromDate && !toDate) {
            return;
        }
        if (fromDate && !toDate) {
            content = `[${fromDate.toISOString()} TO *]`;
        } else if (!fromDate && toDate) {
            content = `[* TO ${toDate.toISOString()}]`;
        } else {
            content = `[${fromDate.toISOString()} TO ${toDate.toISOString()}]`;
        }

        searchInteractiveFilterNoEscapeContent(columnName, content, "date-filter");
    }

    $scope.appendTextValuesToInteractiveFilter = function (columnName, textValues) {
        textValues = textValues.filter(c => c.length > 0);
        if (textValues.length == 0) {
            return;
        }

        textValues = textValues.map(quoteContent);
        let content = textValues.join(" OR ");
        if (textValues.length > 1) {
            content = "(" + content + ")";
        }
        searchInteractiveFilterNoEscapeContent(columnName, content, "text-filter");
    }

    // Sorting
    // override default sortDirection from the shaker to easily stay compatible
    $scope.sortDirection = function(column) {
        if (!$scope.interactiveSearchQuery.sortParameters || $scope.interactiveSearchQuery.sortParameters[0].column !== column) {
            return null;
        }
        return $scope.interactiveSearchQuery.sortParameters[0].order === "asc";
    }

    $scope.isColumnTypeSortable = function(type) {
        return ["date", "dateonly", "datetimenotz", "tinyint", "smallint", "int", "bigint", "float", "double", "long", "boolean"].includes(type);
    }

    // override default addSort from the shaker to easily stay compatible
    $scope.addSort = function(column) {
        if (!$scope.interactiveSearchQuery.sortParameters || $scope.interactiveSearchQuery.sortParameters[0].column !== column) {
            $scope.interactiveSearchQuery.sortParameters = [{ "column": column, "order": "asc"}];
        } else if ($scope.interactiveSearchQuery.sortParameters[0].order === "asc") {
            $scope.interactiveSearchQuery.sortParameters[0].order = "desc";
        } else {
            $scope.interactiveSearchQuery.sortParameters[0].order = "asc";
        }
        $scope.searchInteractive();
    }

    $scope.clearSortInteractive = () => {
        $scope.interactiveSearchQuery.sortParameters = null;
        $scope.searchInteractive();
    }

    $scope.checkQueryInput = (event) => {
        if (event.originalEvent.code === "Enter") {
            event.preventDefault();
            event.target.blur();
            $scope.searchInteractive("click");
        }
    }

    $scope.resetTextareaSize = (textareaInput) => {
        if (!textareaInput) {
            textareaInput = document.getElementById("interactive-search-input");
        }
        textareaInput.style.height = "";
        textareaInput.style.overflowY = "hidden";  // Do not display scroll bar for a single line
    }

    let textareaInputOriginalHeight = null;
    let textareaInput = null;
    $scope.checkTextareaSize = () => {
        if (textareaInputOriginalHeight === null) {
            textareaInput = document.getElementById("interactive-search-input");
            textareaInputOriginalHeight = textareaInput.scrollHeight;
        }
        textareaInput.style.overflowY = "hidden";  // Do not display scroll bar for a single line
        textareaInput.style.height = "1px";
        if (textareaInput.scrollHeight > textareaInputOriginalHeight * 1.5) {  // More than one line of query
            textareaInput.style.height = Math.min(textareaInput.scrollHeight, 100) + "px";  // Recompute height based on input
            textareaInput.style.overflowY = "auto";
        } else {
            $scope.resetTextareaSize(textareaInput);
        }
    }

    const monoFuture = MonoFuture($scope);

    $scope.monoFuturizedSearch = monoFuture.wrap($scope.searchAPI);

    $scope.shakerState.onDataset = true;

    $scope.shakerHooks.isMonoFuturizedRefreshActive = monoFuture.active;

    $scope.shakerHooks.shakerForQuery = function(){
        const queryObj = angular.copy($scope.shaker);
        if ($scope.isRecipe) {
            queryObj.recipeSchema = $scope.recipeOutputSchema;
        }
        queryObj.contextProjectKey = $stateParams.projectKey; // quick 'n' dirty, but there are too many call to bother passing the projectKey through them
        return queryObj;
    }

    $scope.shakerHooks.updateColumnWidth = function(name, width) {
        $scope.shaker.columnWidthsByName[name] = width;
        $scope.autoSaveAutoRefresh();
    };

    $scope.getQuerySyntaxError = function() {
        if (!$scope.table || $scope.table.warnings.totalCount == 0 || !('INPUT_ELASTICSEARCH_BAD_QUERY' in $scope.table.warnings.warnings)) {
            return null;
        }
        // Only return first message
        return $scope.table.warnings.warnings['INPUT_ELASTICSEARCH_BAD_QUERY']['stored'][0]['message'];
    }

    $scope.shakerHooks.fetchDetailedAnalysis = function(setAnalysis, handleError, columnName, alphanumMaxResults, fullSamplePartitionId, withFullSampleStatistics) {
        // Do nothing
    };

    $scope.shakerHooks.fetchClusters = function(setClusters, columnName, setBased, radius, timeOut, blockSize) {
        // Do nothing
    };

    $scope.shakerHooks.fetchTextAnalysis = function(setTextAnalysis, columnName, textSettings) {
        // Do nothing
    };

    $scope.toSuccessAPI = ({result}) => {
        $scope.table.warnings = result.warnings; // bubble up refresh warnings
        return {data: result}; // mimic http response, in a data field;
    };
});


app.directive("shakerOnDatasetElastic", function() {
    return {
        scope: true,
        controller: function ($scope, $stateParams, DataikuAPI, FutureWatcher, $controller, $q, WT1, TopNav, DatasetErrorCta, DatasetUtils, ElasticSearchQueryUtils, SmartId) {
            $scope.searchAPI = DataikuAPI.shakers.searchElasticDataset;
            $controller("_InteractiveSearchController", {$scope: $scope});

            const batchSize = 128;  // number of rows to fetch in a single "refresh" call
            $scope.shakerHooks.getRefreshTablePromise = function(filtersOnly, filterRequest) {
                const ret = $scope.monoFuturizedSearch($stateParams.projectKey, $scope.inputDatasetProjectKey, $scope.inputDatasetName,
                        $scope.shakerHooks.shakerForQuery(), $scope.interactiveSearchQuery, 0, batchSize, batchSize);
                return $scope.refreshNoSpinner ? ret.noSpinner() : ret;
            };

            $scope.shakerHooks.getTableChunk = function(firstRow, nbRows, firstCol, nbCols, filterRequest) {
                return DataikuAPI.shakers.searchElasticDataset($stateParams.projectKey, $scope.inputDatasetProjectKey, $scope.inputDatasetName,
                                                       $scope.shakerHooks.shakerForQuery(), $scope.interactiveSearchQuery,
                                                       firstRow, $scope.table.totalKeptRows, batchSize)
                .then(({data}) => {
                    if (data.hasResult) {
                        return $scope.toSuccessAPI(data);
                    }
                    return FutureWatcher.watchJobId(data.jobId).then(({data}) => $scope.toSuccessAPI(data));
                });
            }


            $controller("_ShakerExploreCommonElasticController", {$scope: $scope});
            $scope.projectKey = $stateParams.projectKey;

            if ($scope.insight) { // On a dashboard
                /* ********************* Callbacks for shakerExploreBase ******************* */
                $scope.shakerHooks.saveForAuto = function() {
                    return $q((resolve, reject) => {
                        try {
                            $scope.insight.params.shakerScript = $scope.getShakerData();
                            if ($scope.hook && $scope.hook.onSave) {
                                $scope.hook.onSave($scope.insight.params);
                            }

                            resolve();
                        } catch (error) {
                            reject(error);
                        }
                    });
                };

                $scope.setSpinnerPosition = function() {}

                /* ********************* Main ******************* */

                // Set base context and call baseInit

                WT1.event("dashboard-dataset-search-open");
                $scope.searchWT1EventName = "dashboard-dataset-search-action-search";
                $scope.refreshNoSpinner = false;  // Display spinner for initial search

                const resolvedDataset = SmartId.resolve($scope.insight.params.datasetSmartName);
                $scope.inputDatasetProjectKey = resolvedDataset.projectKey;
                $scope.inputDatasetName = resolvedDataset.id;
                $scope.inputDatasetSmartName = $scope.insight.params.datasetSmartName;


                $scope.shaker = $scope.insight.params.shakerScript;
                $scope.shaker.origin = "DATASET_EXPLORE";
                if ($scope.origInsight) {
                    $scope.origInsight.params.shakerScript.origin = "DATASET_EXPLORE";
                }

                $scope.shakerState.writeAccess = true;
                if ($scope.insight.params.shakerState) {
                    Object.entries($scope.insight.params.shakerState).forEach(([key, value]) => $scope.shakerState[key] = value);
                }

                $scope.shaker.$headerOptions = ElasticSearchQueryUtils.getShakerHeaderOptions();


                $scope.shaker.coloring.highlightSearchMatches = true;

                $scope.fixupShaker();
                if ($scope.origInsight) {
                    $scope.fixupShaker($scope.origInsight.params.shakerScript);
                }

                $scope.insight.$filteringStatus = {};

                $scope.$watch('shakerState.activeView', (nv, ov) => {
                    if (nv === 'table' && ov) {
                        $scope.refreshTable(false);
                    }
                });
            } else { // On a dataset
                /* ********************* Callbacks for shakerExploreBase ******************* */
                $scope.shakerHooks.saveForAuto = function() {
                    var deferred = $q.defer();
                    resetErrorInScope($scope);
                    var shakerData = $scope.getShakerData();

                    if ($scope.isRecipe) {
                        throw "Should not call this for a recipe";
                    } else {
                        DataikuAPI.explores.saveScript($stateParams.projectKey, $scope.inputDatasetSmartName,
                            shakerData).success(function(data){
                            $scope.originalShaker = shakerData;
                            deferred.resolve();
                        }).error(setErrorInScope.bind($scope));
                    }
                    return deferred.promise;
                };


                TopNav.setLocation(TopNav.TOP_FLOW, 'datasets', TopNav.TABS_DATASET, "search");

                if ($stateParams.datasetFullName) {  // On a foreign (exposed) dataset
                    const loc = DatasetUtils.getLocFromFull($stateParams.datasetFullName);
                    $scope.inputDatasetProjectKey = loc.projectKey;
                    $scope.inputDatasetName = loc.name;
                    $scope.inputDatasetSmartName = $stateParams.datasetFullName;
                } else {  // dataset local to project
                    $scope.inputDatasetProjectKey = $stateParams.projectKey;
                    $scope.inputDatasetName = $stateParams.datasetName;
                    $scope.inputDatasetSmartName = $stateParams.datasetName;
                }

                WT1.event("dataset-search-open");
                $scope.searchWT1EventName = "dataset-search-action-search";

                //For datasetErrorCTA directive (CTA in case of error while loading dataset sample)
                $scope.updateUiState = DatasetErrorCta.getupdateUiStateFunc($scope);
                $scope.$watch("datasetFullInfo", _ => $scope.updateUiState($scope.shakerState.runError), true);
                $scope.$watch("shakerState", _ => $scope.updateUiState($scope.shakerState.runError), true);
                $scope.$watch("table", _ => $scope.updateUiState($scope.shakerState.runError));


                // Load shaker, set the necessary stuff in scope and call the initial refresh
                DataikuAPI.explores.getScript($stateParams.projectKey, $scope.inputDatasetSmartName).success(function(shaker) {
                    $scope.shaker = shaker;
                    $scope.shaker.$headerOptions = ElasticSearchQueryUtils.getShakerHeaderOptions();

                    $scope.shaker.coloring.highlightSearchMatches = true;
                    $scope.shaker.origin = "DATASET_EXPLORE";
                    $scope.fixupShaker();
                    $scope.requestedSampleId = null;
                    $scope.refreshTable(false);
                }).error(setErrorInScope.bind($scope));
            }
        }
    }
});

app.directive("shakerOnSearchNotebook", function() {
    return {
        scope: true,
        controller: function ($scope, $stateParams, DataikuAPI, FutureWatcher, $controller, ElasticSearchQueryUtils, DatasetErrorCta, WT1, $q, $rootScope) {
            $scope.searchAPI = DataikuAPI.searchNotebooks.search;
            $controller("_InteractiveSearchController", {$scope: $scope});

            const batchSize = 128;  // number of rows to fetch in a single "refresh" call
            $scope.shakerHooks.getRefreshTablePromise = function(filtersOnly, filterRequest) {
                const index = $scope.notebookCell.$resolvedIndex;
                $scope.notebookCell.query = $scope.interactiveSearchQuery.queryString;
                $scope.notebookCell.lastExecutionTimestamp = Date.now();
                const ret = $scope.monoFuturizedSearch($stateParams.projectKey, $scope.notebookConnection, index, $scope.searchNotebookSchema,
                                                       $scope.interactiveSearchQuery, 0, batchSize, batchSize)
                return $scope.refreshNoSpinner ? ret.noSpinner() : ret;
            };

            $scope.shakerHooks.getTableChunk = function(firstRow, nbRows, firstCol, nbCols, filterRequest) {
                const index = $scope.notebookCell.$resolvedIndex;
                return DataikuAPI.searchNotebooks.search($stateParams.projectKey, $scope.notebookConnection, index, $scope.searchNotebookSchema,
                                                         $scope.interactiveSearchQuery, firstRow, $scope.table.totalKeptRows, batchSize)
                .then(({data}) => {
                    if (data.hasResult) {
                        return $scope.toSuccessAPI(data);
                    }
                    return FutureWatcher.watchJobId(data.jobId).then(({data}) => $scope.toSuccessAPI(data));
                });
            }

            $scope.$watch("selectedField", (nv, ov) => {
                if (!nv || !nv.name) {
                    return;
                }
                const columnName = ElasticSearchQueryUtils.escapeColumn(nv.name);

                if ($scope.interactiveSearchQuery.queryString) {
                    $scope.interactiveSearchQuery.queryString = "(" + $scope.interactiveSearchQuery.queryString + ") AND " + columnName + ":";
                } else {
                    $scope.interactiveSearchQuery.queryString = columnName + ":";
                }
            });

            $scope.$watch("notebookCell.searchScope", (nv, ov) => {
                if (!nv) {
                    return;
                }
                $scope.interactiveSearchQuery.queryString = $scope.notebookCell.query || "";
                $scope.searchInteractive("load-notebook-cell");
            });



            $controller("_ShakerExploreCommonElasticController", {$scope: $scope});
            $scope.setSpinnerPosition = function(position){
                $rootScope.spinnerPosition = position;
            };

            $scope.shakerHooks.saveForAuto = function() {
                var deferred = $q.defer();
                resetErrorInScope($scope);
                deferred.resolve();
                return deferred.promise;
            };

            $scope.projectKey = $stateParams.projectKey;

            WT1.event("search-notebook-open");
            $scope.searchWT1EventName = "search-notebook-action-search";

            //For datasetErrorCTA directive (CTA in case of error while loading dataset sample)

            $scope.updateUiState = DatasetErrorCta.getupdateUiStateFunc($scope);
            $scope.$watch("shakerState", _ => $scope.updateUiState($scope.shakerState.runError), true);
            $scope.$watch("table", _ => $scope.updateUiState($scope.shakerState.runError));


            // Load shaker, set the necessary stuff in scope and call the initial refresh
            $scope.shaker = {
                steps: [],
                coloring: {}
            };

            $scope.shaker.$headerOptions = ElasticSearchQueryUtils.getShakerHeaderOptions();

            $scope.shaker.$isColumnMenuDisabled = (columnName) => {
                return columnName === "__index"; // added on search notebooks
            }

            $scope.shaker.coloring.highlightSearchMatches = true;
            $scope.shaker.origin = "SEARCH_NOTEBOOK";
            $scope.fixupShaker();
            $scope.requestedSampleId = null;
            $scope.refreshTable(false);
        }
    }
});

}());

;
(function() {
'use strict';

const app = angular.module('dataiku.shaker');

app.directive("shakerFacets", function($rootScope, $timeout, $filter, Assert, WT1, DateUtilsService, ChartFilterUtils) {
    return {
        scope: true,
        priority: 99,
        controller: function($scope, $stateParams, $state) {
            /* Actions by filter type */
            let filterTypes = {
                facet: {
                    computeFilter : function(ff, active) {
                        let ret = {
                            "column" : ff.column,
                            params : {},
                            isAGlobalFilter : ff.isAGlobalFilter,
                            columnType : ff.columnType
                        };
                        ret.type = ff.currentMode + "_FACET";
                        ret.active = active;
                        if (!ff.selectedValues && !ff.excludedValues) {
                            ff.selectedValues = {};
                        }
                        if (ff.currentMode === "ALPHANUM") {
                            if (ff.selectedValues) {
                                ret.selectedValues = Object.keys(ff.selectedValues);
                                ret.effective = !!ret.selectedValues.length;
                            }
                            if (ff.excludedValues) {
                                ret.excludedValues = Object.keys(ff.excludedValues);
                                ret.effective = ret.effective || !!ret.excludedValues.length;
                            }
                            ret.canBecomeStep = (ret.selectedValues || []).length >= 1;
                        } else if (ff.currentMode === "BOUNDING_BOX") {
                            ret.selectedValues = Object.keys(ff.selectedValues);
                            ret.effective = !!ret.selectedValues.length;
                            ret.canBecomeStep = false;
                        } else if (ff.currentMode === "NUMERICAL") {
                            ret.minValue = ff.minValue;
                            ret.maxValue = ff.maxValue;
                            ret.effective = ff.minValue != null || ff.maxValue != null;
                            ret.canBecomeStep = ret.effective;
                            ret.includeEmptyValues = ff.includeEmptyValues != null ? ff.includeEmptyValues : !ret.effective;
                        } else if (ff.currentMode === "DATE") {
                            ret.dateFilterType = ff.dateFilterType;
                            if (ff.dateFilterType === "RANGE") {
                                ret.minValue = ff.minValue;
                                ret.maxValue = ff.maxValue;
                                ret.timezone = ff.timezone;
                                ret.effective = ff.minValue != null || ff.maxValue != null;
                                ret.canBecomeStep = ret.effective;
                                ret.includeEmptyValues = ff.includeEmptyValues != null ? ff.includeEmptyValues : !ret.effective;
                            } else if (ff.dateFilterType === "RELATIVE") {
                                ret.dateFilterPart = ff.dateFilterPart;
                                ret.dateFilterOption = ff.dateFilterRelativeOption;
                                ret.dateFilterRelativePreset = ff.dateFilterRelativePreset;
                                ret.effective = ChartFilterUtils.isRelativeDateFilterEffective(ff.dateFilterPart, ff.dateFilterRelativeOption, ff.includeEmptyValues);
                                ret.canBecomeStep = ret.effective;
                                ret.includeEmptyValues = ff.includeEmptyValues != null ? ff.includeEmptyValues : !ret.effective;
                            } else {
                                ret.dateFilterPart = ff.dateFilterPart;
                                if (ff.selectedValues) {
                                    ret.selectedValues = Object.keys(ff.selectedValues);
                                    ret.effective = ret.selectedValues.length;
                                }
                                if (ff.excludedValues) {
                                    ret.excludedValues = Object.keys(ff.excludedValues);
                                    ret.effective = ret.effective || ret.excludedValues.length;
                                }
                                ret.canBecomeStep = ret.effective;
                            }
                        } else if (ff.currentMode === "EXPLICIT") {
                            if (ff.explicitConditions) {
                                ret.explicitConditions = ff.explicitConditions;
                            }
                            if (ff.explicitExclude) {
                                ret.explicitExclude = ff.explicitExclude;
                            }
                        }
                        return ret;
                    },
                    clearFilter : function(ff) {
                        delete ff.excludedValues;
                        ff.selectedValues = {};
                        ff.minValue = undefined;
                        ff.maxValue = undefined;
                        ff.timezone = 'UTC';
                        ff.dateFilterRelativePreset = undefined;
                        ff.dateFilterRelativeOption = undefined;
                        ff.dateFilterPart = "YEAR";
                    },
                    addSteps : function(ff) {
                        if (ff.currentMode === 'ALPHANUM') {
                            $scope.addStepAndRefresh('FilterOnValue', {
                                appliesTo: 'SINGLE_COLUMN',
                                columns: [ff.column],
                                action: 'KEEP_ROW',
                                // Steps are added from dataset explore view where only `selectedValues` is used.
                                values: Object.keys(ff.selectedValues),
                                matchingMode: 'FULL_STRING',
                                normalizationMode: 'EXACT'
                            });
                        } else if (ff.currentMode === 'NUMERICAL') {
                            $scope.addStepAndRefresh('FilterOnNumericalRange', {
                                appliesTo: 'SINGLE_COLUMN',
                                columns: [ff.column],
                                action: 'KEEP_ROW',
                                min: ff.minValue,
                                max: ff.maxValue
                            });
                        } else if (ff.currentMode === 'DATE') {
                            if (ff.dateFilterType === 'RANGE') {
                                $scope.addStepAndRefresh('FilterOnDate', {
                                    appliesTo: 'SINGLE_COLUMN',
                                    columns: [ff.column],
                                    action: 'KEEP_ROW',
                                    filterType: 'RANGE',
                                    // The processor is expecting - in min & max - a string in ISO 8601 format without the time zone part (ex: "2020-01-01T18:00:00.000")
                                    min: ff.minValue ?  DateUtilsService.formatDateToISOLocalDateTime(DateUtilsService.convertDateToTimezone(new Date(ff.minValue), ff.timezone)) : '',
                                    max: ff.maxValue ?  DateUtilsService.formatDateToISOLocalDateTime(DateUtilsService.convertDateToTimezone(new Date(ff.maxValue), ff.timezone)) : '',
                                    timezone_id: ff.timezone,
                                    part: 'YEAR',
                                    option:  {last: 0, next: 0, isUntilNow: false, containsCurrentDatePart: true},
                                });
                            } else if(ff.dateFilterType === 'RELATIVE') {
                                $scope.addStepAndRefresh('FilterOnDate', {
                                    appliesTo: 'SINGLE_COLUMN',
                                    columns: [ff.column],
                                    action: 'KEEP_ROW',
                                    filterType: 'RELATIVE',
                                    option: ff.dateFilterRelativeOption,
                                    part: ff.dateFilterPart,
                                    relativePreset: ff.dateFilterRelativePreset,
                                    timezone_id: 'UTC'
                                });
                            } else {
                                let selectedValues = Object.keys(ff.selectedValues);
                                if (ff.dateFilterPart === 'INDIVIDUAL') {
                                    selectedValues = selectedValues.map(v =>  DateUtilsService.formatDateToISOLocalDate(new Date(v * 1000)));
                                } else if (['QUARTER_OF_YEAR', 'DAY_OF_WEEK', 'DAY_OF_MONTH', 'MONTH_OF_YEAR', 'WEEK_OF_YEAR'].includes(ff.dateFilterPart)) {
                                    selectedValues = selectedValues.map(v => parseInt(v) + 1);
                                }
                                $scope.addStepAndRefresh('FilterOnDate', {
                                    appliesTo: 'SINGLE_COLUMN',
                                    columns: [ff.column],
                                    action: 'KEEP_ROW',
                                    filterType: 'PART',
                                    part: ff.dateFilterPart,
                                    values: selectedValues,
                                    option: ff.dateFilterRelativeOption,
                                    timezone_id: 'UTC',
                                });
                            }
                        }
                    }
                },
                alphanum : {
                    computeFilter : function(ff, active) {
                        return {
                            "column" : ff.column,
                            "type" : "ALPHANUM",
                            selectedValues : ff.selectedValues,
                            excludedValues: ff.excludedValues,
                            effective : (ff.selectedValues || ff.excludedValues || []).length,
                            params : ff.params,
                            active: active
                        }
                    },
                    clearFilter : function(ff) {
                        Assert.trueish(false, 'cannot call alphanum');
                    }
                },
                validity : {
                    computeFilter : function(ff, active) {
                        return {
                            "column" : ff.column,
                            "type" : "VALIDITY",
                            "params" : ff.params,
                            effective : !ff.params.empty || !ff.params.nok || !ff.params.ok,
                            active: active
                        };
                    },
                    clearFilter : function(ff) {
                        Assert.trueish(false, 'cannot call validity');
                    }
                },
                global_search : {
                    computeFilter : function(ff) {
                        return {
                            "type" : "GLOBAL_SEARCH",
                            selectedValues : [ff.filter],
                            effective : ff.filter && !!ff.filter.length
                        }
                    },
                    clearFilter : function(ff) {
                        Assert.trueish(false, 'cannot call global_search');
                    }
                }
            };

            $scope.dateFilterTypes = ChartFilterUtils.getDateFilterTypes();
            $scope.dateFilterParts = ChartFilterUtils.getDateFilterParts();
            $scope.dateRelativeFilterParts = ChartFilterUtils.getCustomDateRelativeFilterParts();

            /* This removes the *filters* and clears the built-in filter of a facet, but does not remove the filters */
            $scope.removeAllFiltersOnColumn = function(column) {
                let newFFs = [];
                for (let i in $scope.shaker.explorationFilters) {
                    let fi = $scope.shaker.explorationFilters[i];
                    if (fi.type == "facet" && fi.column == column) {
                        filterTypes[fi.type].clearFilter(fi);
                        newFFs.push(fi);
                    } else if (fi.column != column) {
                        newFFs.push(fi);
                    }
                }
                $scope.shaker.explorationFilters = newFFs;
            };

            $scope.viewAllFilter = false;
            $scope.toggleFilterView = function() {
            	$scope.viewAllFilter = !$scope.viewAllFilter;
            	if (!$scope.viewAllFilter) {
            	    $scope.setMustBeVisibleFilter('');
            	    $rootScope.$broadcast("reflow");
            	}
            };

            $scope.mustBeVisibleFilter = {column: ''};
            $scope.isMustBeVisibleFilter = function(column) {
                return column == $scope.mustBeVisibleFilter.column;
            };
            $scope.setMustBeVisibleFilter = function(column) {
                $scope.mustBeVisibleFilter.column = column;
            };

            $scope.removeAllFilters = function() {
                $scope.shaker.explorationFilters.splice(1);
                $scope.shaker.globalSearchQuery = "";
            };

            $scope.clearFilter = function(filter) {
                filterTypes[filter.type].clearFilter(filter);
            };

            $scope.removeFFByColumn = function(columnName) {
                let newFFs = [];
                for (let i in $scope.shaker.explorationFilters) {
                    if ($scope.shaker.explorationFilters[i].column != columnName) {
                        newFFs.push($scope.shaker.explorationFilters[i]);
                    }
                }
                $scope.shaker.explorationFilters = newFFs;
            };

            $scope.buildFilterRequest = function(explorationFilters) {
                if ($scope.shaker == null) return [];
                let filterRequest = [];
                for (let ffidx in explorationFilters) {
                    let ffi = explorationFilters[ffidx];
                    let fList = getFiltersList(ffi);
                    for (let fidx in fList) {
                        let fi = fList[fidx];
                        let requestElt =filterTypes[fi.type].computeFilter(fi, ffi.active);
                        if (requestElt != null) {
                            filterRequest.push(requestElt);
                        }
                    }
                }
                if (typeof($scope.shaker.globalSearchQuery)!=='undefined' && $scope.shaker.globalSearchQuery.length > 0) {
                    let globalFilter = {
                        type : "global_search",
                        filter: $scope.shaker.globalSearchQuery
                    };
                    filterRequest.push(filterTypes[globalFilter.type].computeFilter(globalFilter, true));
                }
                return filterRequest;
            };

            $scope.hasAnyFilter = function() {
                if(!$scope.shaker) return false;
                let ret = false;
                for (let ffidx in $scope.shaker.explorationFilters) {
                    let ffi = $scope.shaker.explorationFilters[ffidx];
                    let fList = getFiltersList(ffi);
                    for (let fidx in fList) {
                        let fi = fList[fidx];
                        if (filterTypes[fi.type].computeFilter(fi).effective) {
                            ret = true;
                            break;
                        }
                    }
                }
                // UGLY ! But as we use the tabs directive, we don't have an easy access to the filters tab title ...
                if (ret) {
                    $(".leftPane .tabbable li:eq(2)").addClass("filter-active");
                } else {
                    $(".leftPane .tabbable li:eq(2)").removeClass("filter-active");
                }
                return ret;
            };

            $scope.hasAnyFilterOnColumn = function(column, uneffectiveFilterCount) {
                if(!$scope.shaker || $scope.tile) return false;
                for (let ffidx in $scope.shaker.explorationFilters) {
                    let ffi = $scope.shaker.explorationFilters[ffidx];
                    let fList = getFiltersList(ffi);
                    for (let fidx in fList) {
                        let fi = fList[fidx];
                        if (fi.column && fi.column == column) {
                            if (filterTypes[fi.type].computeFilter(fi).effective || uneffectiveFilterCount) {
                                return true;
                            }
                        }
                    }
                }
                return false;
            };

            $scope.filterIsEffective = function(filter) {
                let fList = getFiltersList(filter);
                for (let fidx in fList) {
                    let fi = fList[fidx];
                    if (filterTypes[fi.type].computeFilter(fi).effective) {
                        return true;
                    }
                }
                return false
            };

            $scope.filterCanBecomeStep = function(filter) {
                let fList = getFiltersList(filter);
                for (let fidx in fList) {
                    let fi = fList[fidx];
                    if (filterTypes[fi.type].computeFilter(fi).canBecomeStep) {
                        return true;
                    }
                }
                return false;
            };

            $scope.addStepsFromFilter = function(filter) {
                return filterTypes[filter.type].addSteps(filter);
            };

            $scope.isBoundingBoxColumn = function(column) {
                // For now we only rely on image view settings, might evolve to rely on meaning if we add a dedicated meaning at some point
                if (!$scope.imageViewSettings || !$scope.imageViewSettings.annotationParams || $scope.imageViewSettings.annotationParams.annotationType !== "OBJECT_DETECTION") {
                    return false;
                }
                return column === $scope.imageViewSettings.annotationParams.annotationColumn;
            }

            $scope.addColumnFilter = function(column, selectedValues, matchingMode, columnType, isDouble) {
                if (!$scope.hasAnyFilterOnColumn(column, true)) {
                    WT1.event("anum-facet-add");
                    let facetType, currentMode;
                    if ($scope.isBoundingBoxColumn(column)) {
                        facetType = "BOUNDING_BOX";
                        currentMode = "BOUNDING_BOX";
                    } else {
                        facetType = ['Date', 'DateOnly', 'DatetimeNoTz'].indexOf(columnType) >= 0 ? 'DATE' : (isDouble ? 'NUMERICAL' : 'ALPHANUM');
                        currentMode = selectedValues && Object.keys(selectedValues).length ? 'ALPHANUM' : facetType;
                    }
                    let columnFilter = {
                        column: column,
                        type: 'columnFilter',
                        currentMode: (selectedValues && Object.keys(selectedValues).length && matchingMode === 'substring')
                            ? 'SIMPLE_ALPHANUM'
                            : 'FACET',
                        active: true,
                        facet: {
                            type: "facet",
                            column: column,
                            columnType : facetType,
                            currentMode : currentMode,
                            sort:"count",
                            minValue : null,
                            maxValue : null,
                            selectedValues: selectedValues
                        },
                        alphanumFilter: {
                            type : "alphanum",
                            column : column,
                            selectedValues : Object.keys(selectedValues),
                            params : { mode : matchingMode, normalization : "exact"}
                        },
                        validityFilter : {
                            type : "validity",
                            column : column,
                            params : {
                                type : columnType,
                                ok : true,
                                nok : true,
                                empty : true
                            }
                        }
                    };
                    if (facetType === "DATE") {
                        columnFilter.facet.timezone = "UTC";
                        columnFilter.facet.dateFilterType = "RELATIVE";
                        columnFilter.facet.dateFilterPart = "YEAR";
                        columnFilter.facet.dateFilterRelativeOption = undefined;
                        columnFilter.facet.dateFilterRelativePreset = undefined;
                        columnFilter.facet.minValue = undefined; // undefined <=> Reset the bound to the smallest value
                        columnFilter.facet.maxValue = undefined; // undefined <=> Reset the bound to the largest value
                    }
                    if (!$scope.viewAllFilter) {
                    	$scope.openFacetContextualMenuAtAnimationEnd(column);
                    }
                    $scope.shaker.explorationFilters.push(columnFilter);
                    if ($scope.viewAllFilter) {
					    $timeout(function() {
					    	$scope.$apply(function() {
					    		$scope.setMustBeVisibleFilter(column);
					    	})
					    }, 0, false);
					}
                } else {
                	if (!$scope.viewAllFilter) {
                		$scope.openFacetContextualMenuAtAnimationEnd(column);
                        $scope.$broadcast('slideToId', '.facetsFilters', '.filters-slider' , $scope.getFFGroupIdByColumn(column));
                	} else {
                		$timeout(function() {
					    	$scope.$apply(function() {
					    		$scope.setMustBeVisibleFilter(column);
					    	})
					    }, 0, false);
                	}
                }
            };

            $scope.getFFGroupIdByColumn = function(column) {
                return 'facet-' + column;
            };

            $scope.openFacetContextualMenuAtAnimationEnd = function(column) {
            	var off = $('[dku-arrow-slider]')?.scope()?.$on('DKU_ARROW_SLIDER:animation_over',function() {
            		 $scope.$broadcast('openFilterFacetContextualMenu', column);
            		 off(); //to unregister the listener set with $on
            	});
            };

            /*
             * If ff is a column filter, returns all its active filters
             * Else, returns a list only containing ff (the filter passed in parameter)
             */
            var getFiltersList = function(ff) {
                let ffList = [];
                if (ff.type === "columnFilter") {
                    if (ff.currentMode === "FACET") {
                        ffList.push(ff.facet);
                    } else if (ff.currentMode === "SIMPLE_ALPHANUM") {
                        ffList.push(ff.alphanumFilter);
                    }
                    ff.validityFilter && ffList.push(ff.validityFilter);
                } else {
                    ffList.push(ff);
                }
                return ffList;
            };

            $scope.isDateRangeFilter = ChartFilterUtils.isDateRangeFilter.bind(ChartFilterUtils);
            $scope.isRelativeDateFilter = ChartFilterUtils.isRelativeDateFilter.bind(ChartFilterUtils);
            $scope.isDatePartFilter = ChartFilterUtils.isDatePartFilter.bind(ChartFilterUtils);

            $scope.resetFilter = function(filter) {
                filterTypes[filter.type].resetFilter(filter);
            };

            $scope.getFilterByColumn = function(column) {
            	for (let ffIdx = 0; ffIdx<$scope.shaker.explorationFilters.length; ffIdx++) {
            		var ff = $scope.shaker.explorationFilters[ffIdx];
            		if (ff.type === "columnFilter" && ff.column == column) {
            			return ff;
            		}
            	}
            	return undefined;
            };

            $scope.updateFacetData = function() {
                if ($scope.filterTmpDataWatchDeregister) {
                    $scope.filterTmpDataWatchDeregister();
                }
                $scope.filterTmpData = {};
                /* Build tmpData */
                for (let fIdx = 0; fIdx < $scope.table.filterFacets.length; fIdx++) {
                    let responseFacet = $scope.table.filterFacets[fIdx];
                    let column = responseFacet.column;
                    let type = responseFacet.type;
                	type = type.replace('_FACET', '');
                    let filter = $scope.getFilterByColumn(column);

                    if (filter) {
                        let tmpData =  $scope.filterTmpData[column] ;
                        if (!tmpData) {
                            tmpData = {};
                            $scope.filterTmpData[column] = tmpData;
                        }

                        if (type === 'VALIDITY') {
                            for (let v = 0 ; v < responseFacet.values.length; v++) {
                                let facetVal = responseFacet.values[v];
                                if (facetVal.id === 'ok') {
                                    tmpData.nbOk = facetVal.count;
                                } else if (facetVal.id === 'nok') {
                                    tmpData.nbNok = facetVal.count;
                                } else if (facetVal.id === 'empty') {
                                    tmpData.nbEmpty = facetVal.count;
                                }
                            }
                            let total = tmpData.nbOk + tmpData.nbNok + tmpData.nbEmpty;
                            tmpData.okPercentageStr = total > 0 ? $filter("smartPercentage")(tmpData.nbOk/total) : 'none';
                            tmpData.nokPercentageStr = total > 0 ? $filter("smartPercentage")(tmpData.nbNok/total) : 'none';
                            tmpData.emptyPercentageStr = total > 0 ? $filter("smartPercentage")(tmpData.nbEmpty/total) : 'none';
                            tmpData.nonemptyPercentageStr = total > 0 ? $filter("smartPercentage")((total-tmpData.nbEmpty)/total) : 'none';
                            tmpData.okPercentage = total > 0 ? tmpData.nbOk*100 / total : 'none';
                            tmpData.nokPercentage = total > 0 ? tmpData.nbNok*100 / total : 'none';
                            tmpData.emptyPercentage = total > 0 ? tmpData.nbEmpty*100 / total : 'none';
                            tmpData.nonemptyPercentage = total > 0 ? (total - tmpData.nbEmpty)*100 / total : 'none';
                        } else {
                            const valuesLength = responseFacet.values.length;
                            tmpData.values = [];
                            tmpData.type = type;
                            tmpData.isRange = responseFacet.isRange;
                            if (type === 'ALPHANUM' || type === "BOUNDING_BOX" || (type === 'DATE' && !responseFacet.isRange)) {
                                for (let v = 0 ; v < valuesLength; v++) {
                                    let facetVal = responseFacet.values[v];
                                    let included;
                                    if (filter.facet.selectedValues) {
                                        included = filter.facet.selectedValues[facetVal.id] !== undefined && filter.facet.selectedValues[facetVal.id] ;
                                    } else if (filter.facet.excludedValues) {
                                        included = filter.facet.excludedValues[facetVal.id] === undefined;
                                    }
                                    tmpData.values.push({
                                        id : facetVal.id,
                                        label : facetVal.label,
                                        count : facetVal.count,
                                        rowCount: facetVal.rowCount,
                                        included
                                    });
                                }
                            } else if (type === 'DATE' && responseFacet.isRange) {
                                tmpData.response = responseFacet;
                                if (filter.facet.timezone != null) {
                                    tmpData.timezone = filter.facet.timezone;
                                }
                                // For dates, we use the following convention to improve user experience:
                                // - valid number => Use the value
                                // - undefined    => Take the smallest/largest date found in the sample
                                // - null         => Leave as it is (it will display the default date placeholder in the UI)
                                tmpData.minValue = filter.facet.minValue !== undefined ? filter.facet.minValue : responseFacet.minValue;
                                tmpData.maxValue = filter.facet.maxValue !== undefined ? filter.facet.maxValue : responseFacet.maxValue;
                            } else if (type === 'NUMERICAL') {
                                tmpData.response = responseFacet;
                                tmpData.minValue = filter.facet.minValue != null ? filter.facet.minValue : responseFacet.minValue;
                                tmpData.maxValue = filter.facet.maxValue != null ? filter.facet.maxValue : responseFacet.maxValue;
                            }

                            tmpData.uniqueRowCount = responseFacet.count;
                        }
                    }
                }

                /* Triggered by slide end */
                $scope.filterTmpDataWatchDeregister =  $scope.$watch("filterTmpData", function(nv, ov) {
                    for (let column in $scope.filterTmpData) {
                        let filter = $scope.getFilterByColumn(column);
                        let tmpData = $scope.filterTmpData[column];

                        if (tmpData.type === "ALPHANUM" || tmpData.type === "BOUNDING_BOX" || (tmpData.type === 'DATE' && !tmpData.isRange)) {
                            if (filter.facet.selectedValues) {
                                filter.facet.selectedValues = {};
                                for (let i in tmpData.values) {
                                    if (tmpData.values[i].included) {
                                        filter.facet.selectedValues[tmpData.values[i].id] = true;
                                    }
                                }
                            } else if (filter.facet.excludedValues) {
                                filter.facet.excludedValues = {};
                                for (let i in tmpData.values) {
                                    if (!tmpData.values[i].included) {
                                        filter.facet.excludedValues[tmpData.values[i].id] = true;
                                    }
                                }
                            }
                        } else if (tmpData.type  === "NUMERICAL" || (tmpData.type === 'DATE' && tmpData.isRange)) {
                            // Detect when the entered value is the same as the lower or upper bound, and replace it with an undefined value
                            // to say that we don't want to filter using this bound.

                            // To prevent bounds to be lost in case of a second refresh, reopening an active filter
                            if(filter.facet.minValue && !tmpData.minValue && tmpData.response.minValue) {
                                tmpData.minValue = filter.facet.minValue;
                            }
                            if(filter.facet.maxValue && !tmpData.maxValue && tmpData.response.maxValue) {
                                tmpData.maxValue = filter.facet.maxValue;
                            }

                            filter.facet.minValue = tmpData.minValue !== tmpData.response.minValue ? tmpData.minValue : undefined;
                            filter.facet.maxValue = tmpData.maxValue !== tmpData.response.maxValue ? tmpData.maxValue : undefined;
                            filter.facet.timezone = tmpData.timezone;
                        }
                    }
                }, true);
            };

            var filterChanged = function(nv, ov) {
                if (nv  == null || ov == null) return;
                if ($scope.isRecipe) {
                    // Skip when on a recipe and recipeOutputSchema was not yet set due to concurrent calls, refreshTable() need the schema here
                    if ($scope.recipeOutputSchema) {
                        $scope.refreshTable(true);
                    }
                } else {
                    $scope.refreshTable(true);
                    /* Don't save synchronously, we want optimal performance here */
                    $timeout($scope.shakerHooks.saveForAuto, 100);
                }
            };

            $scope.$watch("shaker.explorationFilters", function(nv, ov) {
                if (!$scope.tile) { // No refresh when it's a dashboard filters change (which uses logic from shakerExploreInsight)
                    filterChanged(nv, ov);
                }
            }, true);

            $scope.$watch("shaker.globalSearchQuery", function(nv, ov) {
                if (!$scope.tile) { // No refresh when it's a dashboard filters change (which uses logic from shakerExploreInsight)
                    filterChanged(nv, ov);
                }
            }, true)
        }
    }
});

/*
 * Directive grouping a shakerFacet, a simpleAlphanumFilter, and a validityFilter in order to display them all into one single contextual menu
 */
app.directive('columnFilter', ['$filter', 'ContextualMenu', '$window', 'ChartFilterUtils', 'DateUtilsService', 'translate', function($filter, ContextualMenu, $window, ChartFilterUtils, DateUtilsService, translate) {
    return {
        scope: true,
        restrict : 'AE',
        link : function($scope, element, attrs) {

            /*
             * Filter panel visibility
             */

            $scope.isFilterPanelVisible = false;
            let numFmt = $filter('smartNumber');

            $scope.menu = new ContextualMenu({
                template: "/templates/shaker/column-filter-panel.html",
                cssClass : "ff-contextual-menu",
                scope: $scope,
                contextual: false,
                handleKeyboard: false,
                onOpen: function() {
                    $scope.isFilterPanelVisible = true;
                },
                onClose: function() {
                    $scope.isFilterPanelVisible = false;
                },
                enableClick: true
            });

            $scope.showMenu = function() {
                let openAtX = $(element).offset().left;
            	if (openAtX > $($window).width()/2) {
            		openAtX += $(element).outerWidth();
            	}
                $scope.menu.openAtXY(openAtX, $(element).offset().top + $(element).height(), function() {}, false, true); // NOSONAR: OK to have empty method
            };

            $scope.hideMenu = function() {
                $scope.menu.closeAny();
            };

            $scope.toggleMenu = function() {
                if ($scope.isFilterPanelVisible) {
                    $scope.hideMenu();
                } else {
                    $scope.showMenu();
                }
            };

            $scope.$on("openFilterFacetContextualMenu", function(event, column) {
                if ($scope.ffGroup.column == column) {
                    $scope.showMenu();
                }
            });

            /*
             * Switching filter mode
             */

            $scope.switchToFacetNumerical = function() {
                $scope.ffGroup.currentMode = "FACET";
                $scope.ffGroup.facet.currentMode = "NUMERICAL";
            };
            $scope.switchToFacetAlphanum = function() {
                $scope.ffGroup.currentMode = "FACET";
                $scope.ffGroup.facet.currentMode = "ALPHANUM";
            };
            $scope.switchToFacetBoundingBox = function() {
                $scope.ffGroup.currentMode = "FACET";
                $scope.ffGroup.facet.currentMode = "BOUNDING_BOX";
            };
            $scope.switchToSimpleAlphanum = function() {
                $scope.ffGroup.currentMode = "SIMPLE_ALPHANUM";
            };
            $scope.switchDateFilterType = function(filterType) {
                $scope.ffGroup.facet.dateFilterType= filterType;
            };

            $scope.isFacet = function() {
                return $scope.ffGroup.currentMode === "FACET";
            };

            $scope.isFacetNumerical = function() {
                return $scope.isFacet() && $scope.ffGroup.facet.currentMode === "NUMERICAL";
            };

            $scope.isFacetAlphanum = function() {
                return $scope.isFacet() && $scope.ffGroup.facet.currentMode === "ALPHANUM";
            };

            $scope.isFacetBoundingBox = function() {
                return $scope.isFacet() && $scope.ffGroup.facet.currentMode === "BOUNDING_BOX";
            }

            $scope.isSimpleAlphanum = function() {
                return $scope.ffGroup.currentMode === "SIMPLE_ALPHANUM";
            };

            $scope.isFacetDate = function() {
                return $scope.isFacet() && $scope.ffGroup.facet.currentMode === "DATE";
            };

            $scope.isFacetDateRange = function() {
                return $scope.isFacet() && $scope.isFacetDate() && $scope.ffGroup.facet.dateFilterType === "RANGE";
            };

            $scope.isFacetDateRelativeRange = function() {
                return $scope.isFacet() && $scope.isFacetDate() && $scope.ffGroup.facet.dateFilterType === "RELATIVE";
            };

            $scope.isFacetDatePart = function() {
                return $scope.isFacet() && $scope.isFacetDate() && $scope.ffGroup.facet.dateFilterType === "PART";
            };

            $scope.isEffective = function() {
            	if ($scope.filterIsEffective($scope.ffGroup.validityFilter)) {
            		return true;
            	} else if ($scope.isFacet()) {
                    return $scope.filterIsEffective($scope.ffGroup.facet);
                } else if ($scope.isSimpleAlphanum()) {
                    return $scope.filterIsEffective($scope.ffGroup.alphanumFilter);
                }
                return false;
            };

            $scope.$watch('ffGroup.facet.dateFilterType', () => {
                if($scope.ffGroup.facet.dateFilterType !== 'PART' && $scope.ffGroup.facet.dateFilterPart === 'INDIVIDUAL') {
                    $scope.ffGroup.facet.dateFilterPart = "YEAR"
                }
            })

            $scope.getFilterChipInfo = function() {
                if (!$scope.isEffective()) {
                    return translate('SHAKER.FILTER_CONTENT.VALIDITY.ALL', 'All');
                }
                //if validity filter is the only one filtering
                if ($scope.filterIsEffective($scope.ffGroup.validityFilter) && !$scope.filterIsEffective($scope.ffGroup.facet) && !$scope.filterIsEffective($scope.ffGroup.alphanumFilter)) {
                    let validityChipInfo = '';
                	if ($scope.ffGroup.validityFilter.params.ok) {
                		validityChipInfo += 'OK';
                	}
                	if ($scope.ffGroup.validityFilter.params.nok) {
                		validityChipInfo += validityChipInfo.length ? ' & NOK' : 'NOK';
                	}
                	if ($scope.ffGroup.validityFilter.params.empty) {
                		validityChipInfo += validityChipInfo.length ? ' & ∅' : '∅';
                	}
                	return validityChipInfo;
                }
                //otherwise we compute info relatively to "more important filters"
                if ($scope.isFacetNumerical() || $scope.isFacetDateRange()) {
                    let formatedMinValue;
                    let formatedMaxValue;
                    if ($scope.isFacetNumerical()) {
                        if (typeof($scope.ffGroup.facet.minValue) !== 'undefined' && $scope.ffGroup.facet.minValue) {
                            formatedMinValue = numFmt($scope.ffGroup.facet.minValue);
                        }
                        if (typeof($scope.ffGroup.facet.maxValue) !== 'undefined' && $scope.ffGroup.facet.maxValue) {
                            formatedMaxValue = numFmt($scope.ffGroup.facet.maxValue);
                        }
                    } else {
                        if (typeof($scope.ffGroup.facet.minValue) !== 'undefined' && $scope.ffGroup.facet.minValue) {
                            formatedMinValue = $scope.ffGroup.facet.timezone ?
                                               DateUtilsService.formatDateToISOLocalDate(DateUtilsService.convertDateToTimezone(new Date($scope.ffGroup.facet.minValue), $scope.ffGroup.facet.timezone)) :
                                                $filter('date')($scope.ffGroup.facet.minValue, 'yyyy-MM-dd');
                        }
                        if (typeof($scope.ffGroup.facet.maxValue) !== 'undefined' && $scope.ffGroup.facet.maxValue) {
                            formatedMaxValue = $scope.ffGroup.facet.timezone ?
                                               DateUtilsService.formatDateToISOLocalDate(DateUtilsService.convertDateToTimezone(new Date($scope.ffGroup.facet.maxValue), $scope.ffGroup.facet.timezone)) :
                                                $filter('date')($scope.ffGroup.facet.maxValue, 'yyyy-MM-dd');
                        }
                    }
                    if (typeof(formatedMinValue) === 'undefined' && typeof(formatedMaxValue) === 'undefined') {
                        return translate('SHAKER.FILTER_CONTENT.VALIDITY.ALL', 'All');
                    } else if (typeof(formatedMinValue) === 'undefined') {
                        return ' ≤ ' + formatedMaxValue;
                    } else if (typeof(formatedMaxValue) === 'undefined') {
                        return ' ≥ ' + formatedMinValue;
                    } else {
                        return formatedMinValue + ' to ' + formatedMaxValue;
                    }
                } else if ($scope.isFacetDateRelativeRange()) {
                    const { dateFilterRelativeOption, dateFilterPart, includeEmptyValues} = $scope.ffGroup.facet;
                    return ChartFilterUtils.computeRelativeDateLabel(dateFilterRelativeOption, dateFilterPart, includeEmptyValues);
                } else if ($scope.isFacetAlphanum() || $scope.isFacetBoundingBox() || $scope.isFacetDatePart()) {
                    let nbValues = 0;
                    for (let v in $scope.ffGroup.facet.selectedValues) { // NOSONAR
                        nbValues++;
                    }
                    return nbValues === 1 ? nbValues + ' value' : nbValues + ' values';
                } else if ($scope.isSimpleAlphanum()) {
                    let nbValues = $scope.ffGroup.alphanumFilter.selectedValues.length;
                    return nbValues === 1 ? nbValues + ' value' : nbValues + ' values';
                }
            }
        }
    };
}]);


app.directive('shakerFacet', [ '$timeout', 'Logger', 'DateUtilsService', function($timeout, Logger, DateUtilsService) {
    return {
        templateUrl : '/templates/shaker/facet.html',
        replace:true,
        scope:true,
        restrict : 'E',
        link : function($scope, element, attrs) {
            $(element).find(".accordion-body").addClass("in");
            $scope.dateRelativeFilterPartsLabel = "Year";
            $scope.dateRelativeFilterComputedStart = '-';
            $scope.dateRelativeFilterComputedEnd = '-';

            $scope.$watch("filterTmpData", function(nv, ov) {
                if (nv == null) return;
                if (!$scope.filterTmpData[$scope.facet.column] || $scope.filterTmpData[$scope.facet.column].type !== $scope.facet.currentMode) return;
                $scope.facetUiState = $scope.facetUiState || {};

                let minValue = $scope.filterTmpData[$scope.facet.column].minValue;
                let maxValue = $scope.filterTmpData[$scope.facet.column].maxValue;
                if ($scope.facet.currentMode === "DATE") {
                    const timezone = $scope.filterTmpData[$scope.facet.column].timezone || 'UTC';
                    $scope.facetUiState.timezoneDateRangeModel = timezone;
                    // For dates, we use the following convention to improve user experience:
                    // - valid number => Use the value
                    // - undefined    => Take the smallest/largest date found in the sample
                    // - null         => Leave as it is (it will display the default date placeholder in the UI)
                    if (minValue === undefined) {
                        minValue = $scope.facetUiState.sliderModelMin;
                    }
                    if (maxValue === undefined) {
                        maxValue = $scope.facetUiState.sliderModelMax;
                    }
                    $scope.facetUiState.fromDateRangeModel = minValue != null ? DateUtilsService.convertDateToTimezone(new Date(minValue), timezone) : null;
                    $scope.facetUiState.toDateRangeModel = maxValue != null ? DateUtilsService.convertDateToTimezone(new Date(maxValue), timezone) : null;
                } else {
                    $scope.facetUiState.sliderModelMin = minValue != null ? minValue : $scope.facetUiState.sliderModelMin;
                    $scope.facetUiState.sliderModelMax = maxValue != null ? maxValue : $scope.facetUiState.sliderModelMax;
                }

                // 10000 ticks
                let sliderSpan = $scope.facetUiState.sliderModelMax - $scope.facetUiState.sliderModelMin;
                if (sliderSpan > 0.00001) {
                    $scope.sliderStep = Math.round(10000*sliderSpan)/100000000;
                } else {
                    $scope.sliderStep = sliderSpan / 10000;
                }

                // Handle min=max
                if ($scope.sliderStep === 0) {
                    $scope.sliderStep = 1;
                }
                // handle scientific notation to get the # of decimal places
                $scope.sliderDecimals = 0;
                if ($scope.sliderStep < 1e-14) {
                    // no point in getting the # of decimal places, we'll end up below the precision of 64bit doubles
                    $scope.sliderDecimals = 14;
                } else {
                    let dec = 1;
                    while (dec > $scope.sliderStep) {
                        dec /= 10;
                        $scope.sliderDecimals++;
                    }
                }

                if ($scope.facet.currentMode === "NUMERICAL") {
                    let selector = $(element).find("div.ministogram-container").get(0);
                    let response = $scope.filterTmpData[$scope.facet.column].response;
                    $scope.isChart = response.histogramBars.length > 0;
                    $scope.isRangeSlider = $scope.isChart;
                    if ($scope.isChart) {
                    	d3.select(selector).selectAll("svg").remove();
                        let height = 100;
                        let width = $(selector).parent().width() !== 0 ? $(selector).parent().width() : 300;
                        let svg = d3.select(selector).append("svg").style("height", height).style("width", width).append("g");

                        let maxCount = 0;
                        for (let i = 0; i < response.histogramBars.length; i++) {
                            maxCount = Math.max(maxCount, response.histogramBars[i].count);
                        }
                        let xscale = d3.scale.linear().domain([response.minValue, response.maxValue]).range([0, width]);
                        let yscale = d3.scale.linear().domain([0, maxCount]).range([0, height]);

                        /* Each data is [lb, hb, value]*/
                        let barWidth = width / response.histogramBars.length;

                        let tooltip = d3.select("body").append("div")
                        .attr("class", "histogramtooltip")
                        .style("left", "0").style("top", "0")
                        .style("opacity", 0);

                        svg.selectAll("rect").data(response.histogramBars).enter().append("rect")
                        .attr("class", "histogrambar")
                        .attr("x", function(d) { return xscale(d.minValue) + 2; })
                        .attr("y", function(d) { return height - yscale(d.count);})
                        .attr("min", function(d) { return d.minValue;})
                        .attr("max", function(d) { return d.maxValue;})
                        .attr("count", function(d) { return d.count;})
                        .attr("width", barWidth-4)
                        .attr("height", function(d) { return yscale(d.count);})
                        .on("mouseover", function(d) {
                            tooltip.transition()
                            .duration(400)
                            .style("opacity", 1);
                            tooltip.html("[{0} - {1}] - {2} records".format(d.minValue.toFixed(2),
                                     d.maxValue.toFixed(2), Math.round(d.count)))
                            .style("left", (d3.event.pageX) + "px")
                            .style("top", (d3.event.pageY - 28) + "px");
                        }).on("mouseout", function(d) {
                            tooltip.transition()
                            .duration(500)
                            .style("opacity", 0);
                        });
                        svg.append("line").attr("x1", 0).attr("x2", width).attr("y1", height).attr("y2", height)
                        .style("stroke", "#ccc");
                    }
                }

                if ($scope.isDateRangeFilter($scope.facet)) {
                	$scope.isChart = false;
                }
            }, true);

            $scope.dateRangeChange = function() {
                if ($scope.facetUiState) {
                    const from = $scope.facetUiState.fromDateRangeModel;
                    const to = $scope.facetUiState.toDateRangeModel;
                    const tz = $scope.facetUiState.timezoneDateRangeModel;

                    $scope.filterTmpData[$scope.facet.column].timezone = tz;
                    $scope.filterTmpData[$scope.facet.column].minValue = from != null ? DateUtilsService.convertDateFromTimezone(from, tz).getTime() : null;
                    $scope.filterTmpData[$scope.facet.column].maxValue = to != null ? DateUtilsService.convertDateFromTimezone(to, tz).getTime() : null;
                }
            };

            $scope.relativeDateChange = function(facet) {
                const { datePart, relativeOption, relativePreset } = facet;
                $scope.facet.dateFilterPart = datePart;
                $scope.facet.dateFilterRelativePreset = relativePreset;
                $scope.facet.dateFilterRelativeOption = relativeOption;
            };

            $scope.slideEnd = function() {
                $timeout(function() {
                    Logger.info("slideEnd event", $scope.facetUiState);
                    $scope.filterTmpData[$scope.facet.column].minValue = $scope.facetUiState.sliderModelMin;
                    $scope.filterTmpData[$scope.facet.column].maxValue = $scope.facetUiState.sliderModelMax;
                	$scope.$apply();
                }, 0);
            };

            $scope.switchToNumerical = function() {
                $scope.facet.currentMode = "NUMERICAL";
            };
            $scope.switchToAlphanum = function() {
                $scope.facet.currentMode = "ALPHANUM";
            };

            $scope.resetThisFilter = function() {
                $scope.clearFilter($scope.facet);
            };

        	$scope.isSpinner = function() {
        		return !$scope.filterTmpData || !$scope.filterTmpData[$scope.facet.column] || (!$scope.filterTmpData[$scope.facet.column].values && !$scope.filterTmpData[$scope.facet.column].response);
        	}
        }
    };
}]);


app.directive('simpleAlphanumFilter', function(translate) {
    return {
        templateUrl : '/templates/shaker/simple-alphanum-filter.html',
        replace:true,
        scope:true,
        restrict : 'E',
        link : function(scope, element, attrs) {
            scope.filterModes = [
                ["full_string", translate("SHAKER.FILTER_CONTENT.FACET_FILTER_MODES.FULL_STRING", "Full string")],
                ["substring", translate("SHAKER.FILTER_CONTENT.FACET_FILTER_MODES.SUBSTRING", "Substring")],
                [ "pattern", translate("SHAKER.FILTER_CONTENT.FACET_FILTER_MODES.REG_EXP", "Regular expression")]
            ];
            scope.filterNormalizations = [
                ["exact", translate("SHAKER.FILTER_CONTENT.FACET_FILTER_NORMALIZATIONS.CASE_SENSITIVE","Case-sensitive")],
                ["lowercase", translate("SHAKER.FILTER_CONTENT.FACET_FILTER_NORMALIZATIONS.IGNORE_CASE", "Ignore case")],
                ["normalized", translate("SHAKER.FILTER_CONTENT.FACET_FILTER_NORMALIZATIONS.NORMALISED", "Normalized")]
            ];

            if (angular.isUndefined(scope.filter.params.mode)) {
                scope.filter.params.mode = "full_string";
            }
            $(element).find(".accordion-body").addClass("in");
            scope.onSmartChange = function() {
                const isFiltered = (angular.isDefined(scope.filter.selectedValues[0]) && scope.filter.selectedValues[0].length > 0) || (angular.isDefined(scope.filter.excludedValues[0]) && scope.filter.excludedValues[0].length > 0)
                if (isFiltered) {
                    scope.refreshTable(true);
                }
            };
            scope.changeNormModeIfRegexp = function() {
            	if (scope.filter.params.mode === "pattern") {
            		scope.filter.params.normalization = "exact";
            	}
            };

            scope.resetFilter = function(ff) {
                scope.filter.selectedValues = [];
                scope.filter.excludedValues = [];
                scope.filter.params.mode = "full_string";
                scope.filter.params.normalization = "exact";
            }

            scope.filterCanBeReset = function() {
                return (scope.filter.selectedValues || scope.filter.excludedValues || []).length
                    || scope.filter.params.mode !== "full_string"
                    || scope.filter.params.normalization !== "exact";
            };

            scope.filterCanBecomeStep = function() {
                // Steps are added from dataset explore view where only `selectedValues` is used.
                return (scope.filter.selectedValues || []).length;
            };

            scope.addStepsFromFilter = function() {
                scope.addStepAndRefresh('FilterOnValue', {
                    appliesTo: 'SINGLE_COLUMN',
                    columns: [scope.filter.column],
                    action: 'KEEP_ROW',
                    // Steps are added from dataset explore view where only `selectedValues` is used.
                    values: scope.filter.selectedValues || [],
                    matchingMode: (scope.filter.params && scope.filter.params.mode)
                        ? scope.filter.params.mode.toUpperCase()
                        : 'FULL_STRING',
                    normalizationMode: (scope.filter.params && scope.filter.params.normalization)
                        ? scope.filter.params.normalization.toUpperCase()
                        : 'EXACT'
                });
            };
        }
    };
});

app.directive('validityFilter', function() {
    return {
        templateUrl : '/templates/shaker/validity-filter.html',
        replace:true,
        scope:true,
        restrict : 'E',
        link : function(scope, element, attrs) {
            $(element).find(".accordion-body").addClass("in");

            scope.toggleValidityFacet = function(facetValue) {
            	return !facetValue;
            };

            scope.isAll = function() {
                return scope.filter.params.ok && scope.filter.params.nok && scope.filter.params.empty;
            };

            function goBackToAllIfNeeded(){
                if (!scope.filter.params.ok && !scope.filter.params.nok && !scope.filter.params.empty) {
                    scope.filter.params.ok = scope.filter.params.nok = scope.filter.params.empty = true;
                }
            }

            scope.toggleAll = function() {
                if (!scope.isAll()) {
                    scope.filter.params.ok = true;
                    scope.filter.params.nok = true;
                    scope.filter.params.empty = true;
                }
            };

            scope.toggleOk = function() {
                if (scope.isAll()) {
                    scope.filter.params.ok = true;
                    scope.filter.params.nok = false;
                    scope.filter.params.empty = false;
                } else {
                    scope.filter.params.ok = !scope.filter.params.ok;
                }
                goBackToAllIfNeeded();
            };

            scope.toggleNok = function() {
                if (scope.isAll()) {
                    scope.filter.params.ok = false;
                    scope.filter.params.nok = true;
                    scope.filter.params.empty = false;
                } else {
                    scope.filter.params.nok = !scope.filter.params.nok;
                }
                goBackToAllIfNeeded();

            };

            scope.toggleEmpty = function() {
                if (scope.isAll()) {
                    scope.filter.params.ok = false;
                    scope.filter.params.nok = false;
                    scope.filter.params.empty = true;
                } else {
                    scope.filter.params.empty = !scope.filter.params.empty;
                }
                goBackToAllIfNeeded();
            };

            scope.displayValue = function (value) {
                if (isNaN(value)) {
                    return value;
                } else {
                    return scope.roundForDisplay(value) + '%';
                }
            };

            scope.roundForDisplay = function (value) {
                let rounded = Math.floor(value);
                if (rounded == 0 && value > 0) {
                    return 1;
                }
                return rounded;
            };

            scope.isNaN = function(value) {
                return isNaN(value);
            }
        }
    };
});

})();

;
(function(){
'use strict';

var app = angular.module('dataiku.shaker');

function oneWayCompare(small,big) {
    if(small==big) {
        return true;
    } else if(Array.isArray(small)) {
        if(!Array.isArray(big)) {
            return false;
        }
        if(small.length!=big.length) {
           return false;
        }
        for(var i = 0 ; i < small.length; i++) {
            if(!oneWayCompare(small[i],big[i])) {
                return false;
            }
        }
        return true;
    } else if(typeof small=='object'){
        if(typeof big!='object') {
            return false;
        }
        for(var k in small) {
            if(!k.startsWith('$') && !oneWayCompare(small[k], big[k])) {
                return false;
            }
        }
        return true;
    }
}


app.controller("ShakerRecipeCreationController", function($scope, Fn, $stateParams, DataikuAPI, $controller) {
    $scope.recipeType = "shaker";
    $controller("SingleOutputDatasetRecipeCreationController", {$scope:$scope});

    $scope.autosetName = function() {
        if ($scope.io.inputDataset) {
            var niceInputName = $scope.io.inputDataset.replace(/[A-Z]*\./,"");
            $scope.maybeSetNewDatasetName(niceInputName + "_prepared");
        }
    };
});


app.directive("shakerRecipe", function($rootScope, $filter, $timeout, $q, Assert, DataikuAPI, WT1, TopNav, PartitionDeps, RecipesUtils, StateUtils, AnyLoc, Dialogs, Logger, ComputableSchemaRecipeSave, computeColumnWidths) {
    return {
        scope: true,
        controller: function($scope, $stateParams, $state, $controller) {
            $controller("_RecipeWithEngineBehavior", {$scope});

            TopNav.setTab(StateUtils.defaultTab("code"));

            WT1.event("shaker-script-open");

            $scope.hooks.getShaker = function() {
                return $scope.shaker;
            };

            $scope.hooks.onRecipeLoaded = function(){
                Logger.info("On Recipe Loaded");
                $scope.hooks.updateRecipeStatus();
            };

            $scope.hooks.updateRecipeStatus = function(forceUpdate, exactPlan) {
                var deferred = $q.defer();
                var payload = $scope.hooks.getPayloadData();
                var outputSchema = {columns:[]};
                if ($scope.table && $scope.table.headers) {
                	$scope.table.headers.forEach(function(h) {
                		if (h.recipeSchemaColumn && h.recipeSchemaColumn.column) {
                			var c = angular.copy(h.recipeSchemaColumn.column)
                			if (!c.name) {
                				c.name = h.name;
                			}
                			outputSchema.columns.push(c);
                		}
                	});
                }
                $scope.updateRecipeStatusBase(exactPlan, payload, {reallyNeedsExecutionPlan: exactPlan, outputSchema: outputSchema}).then(function() {
                    // $scope.recipeStatus should have been set by updateRecipeStatusBase
                    if (!$scope.recipeStatus) return deferred.reject();
                    deferred.resolve($scope.recipeStatus);
                    $scope.updateStepTranslatabilities();
                });
                return deferred.promise;
            };

            $scope.hooks.getPayloadData = function() {
                return JSON.stringify($scope.hooks.getShaker());
            };

            $scope.hooks.save = function() {
                var deferred = $q.defer();

                $scope.fixPreview();

                if ($scope.hasAnySoftDisabled()){
                    Dialogs.error($scope, "Cannot save", "Cannot save this prepare recipe: please disable Step preview");
                    deferred.reject();
                    return deferred.promise;
                }

                /* Complete the partition deps from the "fixedup" version */
                var recipeSerialized = angular.copy($scope.recipe);
                PartitionDeps.prepareRecipeForSerialize(recipeSerialized);

                var shaker = $scope.hooks.getShaker();

                ComputableSchemaRecipeSave.handleSaveShaker($scope, recipeSerialized, shaker, $scope.recipeOutputSchema, deferred);
                return deferred.promise;
            };

            $scope.hooks.recipeIsDirty = function() {
                if (!$scope.recipe) return false;
                if ($scope.creation) {
                    return true;
                } else {
                    var dirty = !angular.equals($scope.recipe, $scope.origRecipe);
                    dirty = dirty || $scope.schemaDirtiness.dirty;
                    var shaker = $scope.hooks.getShaker();
                    dirty = dirty || !oneWayCompare($scope.origShaker.steps,shaker.steps);
                    // FIXME That is ugly. oneWayCompare is used to ignore "stepStep" on steps,
                    // but we do want to notice when override table changes
                    if (!dirty) {
                        for(var i in $scope.origShaker.steps) {
                            var oldS = $scope.origShaker.steps[i];
                            var newS = shaker.steps[i];
                            dirty = dirty || !angular.equals(oldS.overrideTable, newS.overrideTable);
                            dirty = dirty || !angular.equals(oldS.comment, newS.comment);
                        }
                    }
                    dirty = dirty || !angular.equals($scope.origShaker.explorationFilters, shaker.explorationFilters)
                    dirty = dirty || !angular.equals($scope.origShaker.explorationSampling, shaker.explorationSampling)
                    return dirty;
                }
            };

            $scope.hooks.recipeContainsUnsavedFormulaChanges = function() {
                return $scope.recipe && 
                    $scope.hooks.getShaker().steps.some(
                        (step) => step.$stepState && step.$stepState.containsUnsavedFormulaChanges
                    );
            };

            $scope.shakerHooks.setColumnMeaning = function(column, newMeaning) {
                Assert.inScope($scope, 'shaker');
                Assert.inScope($scope, 'recipeOutputSchema');

                var colData = $scope.recipeOutputSchema.columns[column.name];
                if (!colData){
                    Logger.warn("Column " + column.name + " not found");
                    return;
                }
                colData.column.meaning = newMeaning;
                $scope.schemaDirtiness.dirty = true;

                $scope.refreshTable(false);
            };

            $scope.shakerHooks.getSetColumnStorageTypeImpact = function(column, newType){
                var deferred = $q.defer();
                deferred.resolve({justDoIt:true});
                return deferred.promise;
            };
            $scope.shakerHooks.setColumnStorageType = function(column, newType, actionId){
                var colData = $scope.recipeOutputSchema.columns[column.name];
                if (!colData){
                    Logger.warn("Column " + column.name + " not found");
                    return;
                }
                colData.column.type = newType;
                colData.persistent = true;
                $scope.schemaDirtiness.dirty = true;
                $scope.refreshTable(true);
            };

            $scope.shakerHooks.updateColumnDetails = function(column) {
                var colData = $scope.recipeOutputSchema.columns[column.name];
                if (!colData){
                    Logger.warn("Column " + column.name + " not found");
                    return;
                }
                colData.column = column;
                colData.persistent = true;
                $scope.schemaDirtiness.dirty = true;
                $scope.refreshTable(true);
            };

            $scope.shakerHooks.updateColumnWidth = function(name, width) {
                Assert.inScope($scope, 'shaker');
                Assert.trueish($scope.shaker.columnWidthsByName, 'columnWidthsByName is null');

                $scope.shaker.columnWidthsByName[name] = width;
                $scope.schemaDirtiness.dirty = true;
                $scope.refreshTable(false);
            };

            $scope.clearResize = function() {
                Assert.inScope($scope, 'shaker');
                Assert.trueish($scope.shaker.columnWidthsByName, 'columnWidthsByName is null');

                const minColumnWidth = 100;
                $scope.shaker.columnWidthsByName = computeColumnWidths($scope.table.initialChunk, $scope.table.headers, minColumnWidth, $scope.hasAnyFilterOnColumn, $scope.shaker.columnWidthsByName, $scope.shaker.columnUseScientificNotationByName, true)[1];
                $scope.schemaDirtiness.dirty = true;
                $scope.refreshTable(false);
            }

            $scope.isRecipe = true;
            $scope.table = undefined;
            $scope.processors = undefined;
            $scope.scriptId = "I don't need a script id"
            $scope.shakerWithSteps = true;
            $scope.shakerWritable = $scope.isProjectAnalystRW();

            $scope.schemaDirtiness = { dirty : false};

            var input = RecipesUtils.getSingleInput($scope.recipe, "main").ref;
            if (input.indexOf(".") > -1) {
                $scope.inputDatasetProjectKey = input.split(".")[0];
                $scope.inputDatasetName = input.split(".")[1];
            } else {
                $scope.inputDatasetProjectKey = $stateParams.projectKey;
                $scope.inputDatasetName = input;
            }

            $scope.shaker = JSON.parse($scope.script.data);
            $scope.shaker.origin = "PREPARE_RECIPE";
            $scope.origShaker = angular.copy($scope.shaker);
            $scope.fixupShaker();
            $scope.requestedSampleId = null;

            $scope.shakerState.writeAccess = $scope.isProjectAnalystRW();
            $scope.$watch("projectSummary", function(nv, ov) {
                $scope.shakerWritable = $scope.isProjectAnalystRW();
                $scope.shakerState.writeAccess = $scope.isProjectAnalystRW();
            });
            $scope.shakerState.withSteps = true;

            $scope.shakerHooks.onTableRefresh = function() {
                $scope.updateRecipeStatusLater();
            }
            $scope.shakerHooks.afterTableRefresh = function() {
            	// for steps with a report, because the report comes back from the table report
                $scope.updateRecipeStatusLater();
            }

            $scope.updateStepTranslatabilities = function() {
            	if (!$scope.recipeStatus) return;
            	if (!$scope.shaker.steps) return;
            	var flattenEnabledSteps = function(steps) {
            		steps.forEach(function(s) {delete s.$translatability;});
            		var flatList = [];
            		return steps.map(function(s) {
                		if (!s.disabled) {
                			if (s.metaType == 'GROUP') {
                				return flattenEnabledSteps(s.steps);
                			} else {
                				return [s];
                			}
                		} else {
                			return [];
                		}
                	}).reduce(function(acc, a) {return acc.concat(a);}, []);
            	};
            	var flatStepList = flattenEnabledSteps($scope.shaker.steps);
            	if (!$scope.recipeStatus.translatabilities) return; // do it here so that the translabilites are reset if the status is failed
            	if (flatStepList.length == $scope.recipeStatus.translatabilities.length) {
            		flatStepList.forEach(function(s, i) {s.$translatability = $scope.recipeStatus.translatabilities[i];});
            	}
            };

            var outputRef = RecipesUtils.getSingleOutput($scope.recipe, "main").ref;
            var outputLoc = AnyLoc.getLocFromSmart($stateParams.projectKey, outputRef);

            /* Set the initial dataset output schema as current recipe output schema */
            DataikuAPI.datasets.get(outputLoc.projectKey, outputLoc.localId, $stateParams.projectKey)
            .success(function(outputDataset) {
                $scope.recipeOutputSchema = { columns : {}, columnsOrder : [], outputDatasetType : outputDataset.type }
                angular.forEach(outputDataset.schema.columns, function(col) {
                    $scope.recipeOutputSchema.columns[col.name] = {
                        column: col,
                        persistent : true
                    };
                    $scope.recipeOutputSchema.columnsOrder.push(col.name);
                });
                $scope.refreshTable(false);
                $scope.baseInit();
            }).error(setErrorInScope.bind($scope));

            $scope.enableAutoFixup();

            /* When the "running job" alert is shown or removed, we need to force the
             * fat table to redraw itself */
            $scope.$watch("startedJob.jobId", function(){
                Logger.info("Forcing shaker table resize");
                $rootScope.$broadcast("forcedShakerTableResizing");
            });

            //TODO @recipes32 remove?
            $scope.$watch("recipe.params.engine", function(nv, ov) {
                if (nv == "SPARK" && !$scope.recipe.params.sparkConfig) {
                    $scope.recipe.params.sparkConfig = {}
                }
            });
            // params is not in the same place
            $scope.$watch("recipe.params.engineParams", $scope.updateRecipeStatusLater, true);
        }
    }
});
})();

;
(function(){
'use strict';

const app = angular.module('dataiku.shaker.analyse', ['dataiku.filters', 'platypus.utils']);

    function mesure(attr, def) {
        if (!attr) { return def; }
        var m = parseInt(attr);
        return m && !isNaN(m) ? m : def;
    }
    function svgElement(selector, width, height) {
        return d3.select(selector).style({width: "100%", "max-width": width + "px"})
            .append("div").classed('d3-container', true) // fixed aspect ratio padding trick:
                .style({ position: "relative", "padding-top": (height * 100 / width) + "%" })
            .append("svg").attr("viewBox", "0 0 " + width + " " + height)
                .attr("preserveAspectRatio", "xMinYMin meet");
    }
    function dateFormatter(asDate) {
        return (d) => {
            try {
                return (new Date(d)).toISOString().replace(/([^T]+)T(\d\d:\d\d):(\d\d)(\.\d+)?Z(.*)/, asDate);
            }
            catch (e) {
                return d3.format("s")(d);
            }
        }
    }

    app.directive("histogram", function($filter, NumberFormatter) {
        return {
            restrict: 'ECA',
            replace: true,
            template: '<div class="d3graph" />',
            scope: { data: '=', isDate: '=' },
            link: function(scope, element, attrs){
                var selector = $(element).get(0),
                    width  = mesure(attrs.width,  560),
                    height = mesure(attrs.height, 120);

                scope.$watch("data", function() {
                    d3.select(selector).selectAll('.d3-container').remove();
                    if (!scope.data) {
                        return;
                    }

                    var min = scope.data.min,
                        max = scope.data.max,
                        asDate = scope.isDate,
                        bottom = 40;
                    for(var i in scope.data.chistogram) {
                        var val = scope.data.chistogram[i];
                        min = Math.min(min, val[0]);
                        max = Math.max(max, val[1]);
                    }

                    if (asDate) {
                        var days = (max - min) / 86400000;
                        if (days / 24 <= 1) { asDate = '$2:$3'; bottom = 50; }
                        else if (days <= 2) { asDate = '$2'; }
                        else { asDate = '$1'; bottom = 50; }
                    }

                    var svg = svgElement(selector, width, height + bottom).append("g"),
                        xscale = d3.scale.linear().domain([min, max]).range([0, width]),
                        yscale = d3.scale.linear().domain([0, scope.data.longestHistogramBar]).range([0, height]);

                    var barWidth = width / scope.data.chistogram.length;

                    var tooltip = d3.select("body").append("div")
                            .attr("class", "histogramtooltip")
                            .style("opacity", 0).style("top", "0");

                    /* Each datum is [lb, hb, value]*/
                    var entry = svg.selectAll("g.histogramentry").data(scope.data.chistogram).enter()
                        .append("g").attr("class", "histogramentry")
                        .on("mouseover", function(d, i) {
                            tooltip.transition().duration(400).style("opacity", 1);
                            var lowerBracket = i === 0 ? "[" : "(";
                            var lowerValue = scope.isDate ? $filter('utcDate')(d[0], 'YYYY-MM-DD HH:mm:ss') : d[0].toFixed(2);
                            var upperValue = scope.isDate ? $filter('utcDate')(d[1], 'YYYY-MM-DD HH:mm:ss') : d[1].toFixed(2);
                            tooltip.html(lowerBracket + " {0} , {1} ] - {2} rows".format(lowerValue,
                                         upperValue, Math.round(d[2])))
                                .style("left", (d3.event.pageX) + "px")
                                .style("top", (d3.event.pageY - 28) + "px");
                        })
                        .on("mouseout", function(d) {
                            tooltip.transition().duration(500).style("opacity", 0);
                        });
                    entry.append("rect").attr("class", "histogrambar")
                        .attr("x", function(d) { return xscale(d[0]); })
                        .attr("y", function(d) { return height - yscale(d[2]);})
                        .attr("width", barWidth-1)
                        .attr("height", function(d) { return yscale(d[2]);});
                    entry.append("rect").attr("class", "histogramhover")
                        .attr("x", function(d) { return xscale(d[0]); })
                        .attr("y", 0)
                        .attr("width", barWidth-1)
                        .attr("height", height);

                    const axisFormatter = asDate ? dateFormatter(asDate) : NumberFormatter.getForRange(min, max, 10, false, false);

                    var drawnAxis = svg.append("g")
                        .attr("class", "x axis")
                        .style('fill', '#999')
                        .style('stroke', '#999')
                        .attr("transform", "translate(0," + height + ")")
                        .call(d3.svg.axis().scale(xscale).orient("bottom")
                                .tickFormat(axisFormatter));
                    drawnAxis.selectAll("text")
                        .style('stroke', 'none')
                        .style("text-anchor", "end")
                        .attr("dx", "-.8em")
                        .attr("dy", ".15em")
                        .attr("transform", "rotate(-35)");
                });
            }
        };
    });

    app.directive("miniHistogram", function() {
        return {
            restrict: 'ECA',
            replace: true,
            template: '<div class="d3graph" />',
            scope: { values: '=', activateBar: '=?', range: '=?' },
            link: function(scope, element, attrs) {
                const selector = $(element).get(0),
                    width  = mesure(attrs.width,  500),
                    height = mesure(attrs.height, 180),
                    range = scope.range || null;

                scope.$watch("values", function() {
                    d3.select(selector).selectAll('.d3-container').remove();
                    if (!scope.values) return;

                    const nBars = Math.max(scope.values.length, 10), // ensure max bar width
                        barWidth = width / nBars,
                        min = Math.min.apply(Math, scope.values),
                        max = Math.max.apply(Math, scope.values);

                    const xscale = d3.scale.linear().domain([0, nBars]).range([0, width]),
                        yscale = d3.scale.linear().domain([Math.min(0, min), max]).range([0, height]);

                    let svg;
                    if (range) {
                        svg = svgElement(selector, width + 24, height + 16)
                            .append("g").attr("transform", "translate(4, 0)");
                    } else {
                        svg = svgElement(selector, width, height).append("g");
                    }

                    /* Each datum is the value */
                    const entry = svg.selectAll("g.histogramentry").data(scope.values).enter()
                        .append("g").attr("class", "histogramentry");

                    entry.append("rect").attr("class", "histogrambar")
                        .attr("x", function(d, i) { return xscale(i); })
                        .attr("y", function(d) { return height - yscale(d);})
                        .attr("width", barWidth - 1)
                        .attr("height", yscale);

                    if (range) {
                        svg.append("g").classed("x axis", true)
                            .attr("transform", "translate(0," + height + ")")
                            .call(d3.svg.axis().tickValues([0, nBars]).tickFormat((d, i) => range[i])
                                .scale(xscale).orient("bottom")
                            )
                            .selectAll("text").attr("dy", ".6em");
                    }

                    if (scope.activateBar) {
                        entry.on("mouseover", scope.activateBar).on("mouseout", scope.activateBar.bind(null, null));

                        entry.append("rect").attr("class", "histogramhover")
                            .attr("x", function(d, i) { return xscale(i); })
                            .attr("y", 0)
                            .attr("width", barWidth - 1)
                            .attr("height", height);
                    }
                });
            }
        };
    });

    app.directive("barChart", function() {
        return {
            restrict: 'ECA',
            replace: true,
            template: '<div class="d3graph" />',
            scope: {
                data: '=data',
                count: '=count'
            },
            link: function(scope, element, attrs){
                var selector = $(element).get(0),
                    width =  mesure(attrs.width,  500),
                    baseHeight = mesure(attrs.height, 180);

                scope.$watch("data", function() {
                    if (!scope.data || !scope.data.percentages || !scope.data.percentages.length) {
                        return;
                    }
                    d3.select(selector).selectAll('.d3-container').remove();
                    const count = Math.min(scope.data.percentages.length, scope.count || 7),
                        height = count * baseHeight / scope.count,
                        svg = svgElement(selector, width, height + 40).append("g"),
                        max = Math.max.apply(Math, scope.data.percentages.slice(0, count)),
                        xscale = d3.scale.linear().domain([0, max]).range([0, width]),
                        yscale = d3.scale.linear().domain([0, count]).range([0, height]),
                        perCent = d3.format(".1%"),
                        barHeight = height / count,
                        ti = 0;

                    svg.selectAll("rect").data(scope.data.percentages.slice(0, count)).enter().append("rect")
                        .attr("class", "histogrambar")
                        .attr("x", 0)
                        .attr("y", function(d, i) { return yscale(i); })
                        .attr("width", function(d) { return xscale(d); })
                        .attr("height", barHeight - 1);

                    svg.append("g").attr("transform", "translate(10, 3)")
                        .selectAll("text").data(scope.data.percentages.slice(0, count)).enter().append("text")
                        .text(function(d, i) { return [scope.data.values[i],
                            " (", perCent(scope.data.percentages[i]), ")"].join(""); })
                        .attr("x", 0)
                        .attr("y", function(d, i) { return yscale(i) + barHeight / 2; });

                    const drawnAxis = svg.append("g")
                        .attr("class", "x axis")
                        .style('fill', '#999')
                        .style('stroke', '#999')
                        .attr("transform", "translate(0," + height + ")")
                        .call(d3.svg.axis().scale(xscale).orient("bottom")
                            // eslint-disable-next-line no-undef
                            .tickFormat(perCent));
                    drawnAxis.selectAll("text")
                        .style('stroke', 'none')
                        .style("text-anchor", "end")
                        .attr("dx", "-.8em")
                        .attr("dy", ".15em")
                        .attr("transform", "rotate(-35)");
                });
            }
        };
    });

    app.directive('boxPlot', function() {
        return {
            restrict: 'ECA',
            replace: true,
            template: '<div class="d3graph" />',
            scope : {
                data : '=data',
            },
            link: function(scope, element, attrs){
                var selector = $(element).get(0),
                    height = mesure(attrs.height, 25),
                    width  = mesure(attrs.width,  560),
                    fill = '#C4E0FE',   // digital-blue-lighten-4
                    stroke = '#000';

                scope.$watch("data", function() {
                    if (!scope.data) {
                        return;
                    }

                    d3.select(selector).selectAll('.d3-container').remove();

                    var svg = svgElement(selector, width, height).append("g");

                    var x1 = d3.scale.linear()
                    .domain([scope.data.min, scope.data.max])
                    .range([0, width]);

                    var center = svg.selectAll("line.center")
                    .data([scope.data])
                    .enter().insert("svg:line", "rect")
                    .attr("class", "center")
                    .attr("x1", function(d) { return x1(d.lowWhisker); })
                    .attr("y1", height/2)
                    .attr("x2", function(d) { return x1(d.highWhisker); })
                    .attr("y2", height / 2)
                    .style("opacity", 1)
                    .style("stroke", stroke);

                    var box = svg.selectAll("rect.box").data([scope.data])
                    .enter().append("svg:rect")
                    .attr("class", "box")
                    .attr("x", function(d) { return x1(d.quartiles[0]); })
                    .attr("y", 0) // Avoid margin issues
                    .attr("width", function(d) { return x1(d.quartiles[2]) - x1(d.quartiles[0]);})
                    .attr("height", height) // Avoid margin issues
                    .attr("fill", fill)
                    //.attr("stroke", fill)
                    .style("opacity", "1");

                    var median = svg.selectAll("line.median").data([scope.data])
                    .enter().append("svg:line")
                    .attr("class", "median")
                    .attr("y1", 0)
                    .attr("x1", function(d) { return x1(d.median); })
                    .attr("y2", height)
                    .attr("x2", function(d) { return x1(d.median); })
                    .style("stroke", stroke);

                    var whiskers = svg.selectAll("line.whisker").data([scope.data.lowWhisker, scope.data.highWhisker])
                    .enter().append("svg:line")
                    .attr("class", "whisker")
                    .attr("y1", height * 0.3)
                    .attr("x1", function(d) { return x1(d); })
                    .attr("y2", height * 0.7)
                    .attr("x2", function(d) { return x1(d); })
                    .style("stroke", stroke);

                    svg.selectAll("text.whisker").data([scope.data.lowWhisker, scope.data.highWhisker])
                    .enter().append("svg:text")
                    .attr("class", "whisker")
                    .attr("dy", ".3em")
                    .attr("dx", 6)
                    .attr("x", width)
                    .attr("y", x1).style("font-size", "12px")
                    .text(function(d) { return d.toPrecision(3);});

                    svg.selectAll("text.box").data(scope.data.quartiles)
                    .enter().append("svg:text")
                    .attr("class", "box")
                    .attr("dy", ".3em")
                    .attr("dx", function(d, i) { return i & 1 ? 6 : -6; })
                    .attr("x", function(d, i) { return i & 1 ? width : 0; })
                    .attr("text-anchor", function(d, i) { return i & 1 ? "start" : "end"; })
                    .attr("y", x1).style("font-size", "12px")
                    .text(function(d) { return d.toPrecision(3);});
                });
            }
        };
    });

app.directive('analyseFullSampleToggle', function($stateParams, DataikuAPI, CreateModalFromTemplate, FutureWatcher, FutureProgressModal, translate) {
    return {
        scope: false,
        restrict: 'A',
        templateUrl: "/templates/shaker/analyse-full-sample-toggle.html",
        link: function($scope, element, attrs) {
            function generateSampleModes() {
                function makeMode(label, partitionId) {
                    return {
                        useFullSampleStatistics: true,
                        label: label,
                        partitionId: partitionId
                    };
                }
                const modes = [{
                    useFullSampleStatistics:false,
                    label: translate("SHAKER.ANALYSE.SAMPLE_MODE.SAMPLE","Sample")
                }];

                if ($scope.datasetFullInfo && $scope.datasetFullInfo.partitioned) {
                    if ($scope.shaker && $scope.shaker.explorationSampling && $scope.shaker.explorationSampling.selection) {
                        const selection = $scope.shaker.explorationSampling.selection;
                        if (selection.partitionSelectionMethod == 'ALL') {
                            modes.push(makeMode(translate("SHAKER.ANALYSE.SAMPLE_MODE.WHOLE_DATA", "Whole data"), "ALL"));
                        } else if (selection.partitionSelectionMethod == 'LATEST_N') {
                            // todo : get the list of the latest n partitions in the front
                            modes.push(makeMode(translate("SHAKER.ANALYSE.SAMPLE_MODE.WHOLE_DATA", "Whole data"), "ALL"));
                        } else {
                            selection.selectedPartitions.forEach(function(partitionId) {
                                modes.push(makeMode(translate("SHAKER.ANALYSE.SAMPLE_MODE.WHOLE", "Whole") + " " + partitionId, partitionId));
                            });
                        }
                    } else {
                        modes.push(makeMode(translate("SHAKER.ANALYSE.SAMPLE_MODE.WHOLE_DATA", "Whole data"), "ALL"));
                    }
                } else {
                    modes.push(makeMode(translate("SHAKER.ANALYSE.SAMPLE_MODE.WHOLE_DATA", "Whole data"), "NP"));
                }
                $scope.sampleModes = modes;
                const old = $scope.sampleMode;
                $scope.sampleMode = modes.filter(function(m) {
                    return old && m.useFullSampleStatistics == old.useFullSampleStatistics && m.partitionId == old.partitionId;
                })[0];
                if ($scope.sampleMode == null) {
                    // use the sample as default
                    $scope.sampleMode = $scope.sampleModes[0];
                }
            }
            $scope.sampleModes = [];
            $scope.sampleMode = null;
            generateSampleModes();
            // prepare the data for the partition selection of the full sample pane (if partitioned)
            var updateSampleMode = function() {
                $scope.uiState.useFullSampleStatistics = $scope.sampleMode ? $scope.sampleMode.useFullSampleStatistics : false;
                $scope.uiState.fullPartitionId = $scope.sampleMode ? $scope.sampleMode.partitionId : null;
            };
            updateSampleMode();
            $scope.$watch('sampleMode', updateSampleMode);

            $scope.prefix = attrs.prefix;

            $scope.configureFullSampleStatistics = function(initial) {
                var origFullSampleStatistics = initial ? null : angular.copy($scope.shaker.fullSampleStatistics);
                CreateModalFromTemplate("/templates/shaker/analyze-full-sample-config.html", $scope, "AnalyzeFullSampleConfigController").then(function(decision) {
                    if (decision && decision.save) {
                        if (!angular.equals(origFullSampleStatistics, $scope.shaker.fullSampleStatistics)) {
                            $scope.autoSaveForceRefresh();
                        }
                    }
                    if (decision && decision.compute) {
                        $scope.doComputeFullMetrics($scope.columnName, false); // no need to wait for the shaker refresh, we send the $scope.shaker.fullSampleStatistics in the compute call
                    }
                });
            };
            if ($scope.uiState) {
                // put it also in the uiState to share with links in the modal
                $scope.uiState.configureFullSampleStatistics = $scope.configureFullSampleStatistics;
            }
            $scope.doComputeFullMetrics = function(columnName, forceRefresh) {
                // columnName null means 'do all columns'
                $scope.fatalAPIError = null;
                DataikuAPI.datasets.computeDetailedColumnMetrics($stateParams.projectKey, $scope.inputDatasetName, columnName, $scope.shaker.fullSampleStatistics, $scope.uiState.fullPartitionId, forceRefresh).success(function(data) {
                    $scope.computingFullMetrics = data;
                    $scope.computingModalHandle = FutureProgressModal.reopenableModal($scope, $scope.computingFullMetrics, translate("SHAKER.ANALYSE.COMPUTING_METRICS", "Computing metrics..."));
                    $scope.computingModalHandle.promise.then(function(result) {
                        // success
                        $scope.computingFullMetrics = null;
                        $scope.$eval(attrs.callback)();
                        const errorRuns = result && result.runs && result.runs.filter(_ => _.error);
                        if (errorRuns && errorRuns.length) {
                            $scope.lastComputeResult = {runs: errorRuns, startTime: result.startTime, endTime: result.endTime};
                        } else {
                            $scope.lastComputeResult = null;
                        }
                    }, function(data) {
                        $scope.computingFullMetrics = null;
                    });
                    $scope.showProgressModal();
                }).error(setErrorInScope.bind($scope));
            };

            $scope.showProgressModal = function (jobId) {
                if ($scope.computingModalHandle && $scope.computingModalHandle.open) {
                    $scope.computingModalHandle.open();
                }
            };

            $scope.abortComputingFullMetrics = function() {
                DataikuAPI.futures.abort($scope.computingFullMetrics.jobId).error(setErrorInScope.bind($scope));
            };

            var updateUseFullSampleStatistics = function() {
                if ($scope.uiState.useFullSampleStatistics && (($scope.shaker && $scope.shaker.fullSampleStatistics == null) || ($scope.analysis && $scope.analysis.fullSampleStatistics == null))) {
                    var doConfigure = function() {
                        DataikuAPI.datasets.getFullSampleStatisticsConfig($stateParams.projectKey, $scope.inputDatasetProjectKey, $scope.inputDatasetName).success(function(data) {
                            if ($scope.shaker) {
                                $scope.shaker.fullSampleStatistics = data;
                            }
                            if ($scope.analysis) {
                                $scope.analysis.fullSampleStatistics = data;
                            }
                            $scope.configureFullSampleStatistics(data); // will do the save
                        }).error(setErrorInScope.bind($scope));
                    };
                
                    // first time activating statistics on the full dataset for this dataset => 
                    if ($scope.columnFilter) {
                        // columns-view mode: all ready, we have the multi-column analysis already
                        doConfigure();
                    } else {
                        // single-column-header mode (ie the modal)
                        // 1) fetch with full-sample statistics
                        $scope.refreshAnalysis().then(function(data) {
                            // 2) if still all empty, ask for the configuration then compute
                            var stillNoGood = false; // whether we have the count of values serves as a check that something was ever computed
                            if (data.fullSampleAnalysis == null) {
                                stillNoGood = true;
                            } else if (data.fullSampleAnalysis.categorical == null) {
                                stillNoGood = true;
                            } else if (data.fullSampleAnalysis.categorical.count == null) {
                                stillNoGood = true;
                            } else if (data.fullSampleAnalysis.categorical.count.value == null) {
                                stillNoGood = true;
                            }
                            if (stillNoGood) {
                                doConfigure();
                            }
                        });
                    }
                }
            };
            $scope.$watch('uiState.useFullSampleStatistics', updateUseFullSampleStatistics);

            $scope.$watch('shaker.explorationSampling.selection', generateSampleModes, true);
        }
    }
});


app.controller("AnalyzeFullSampleConfigController", function($scope, DataikuAPI, $stateParams, $timeout, $filter,
        TableChangePropagator, WT1, LoggerProvider, Fn, CreateModalFromTemplate) {
    WT1.event("analyse-full-sample-configuration-open");

    $scope.uiState = {
        tab: 'METRICS'
    };
});


app.controller("ColumnAnalysisController", function($scope, DataikuAPI, $stateParams, $timeout, $filter,
        Assert, TableChangePropagator, WT1, LoggerProvider, Fn, $q) {

    WT1.event("analyse-open");

    var Logger = LoggerProvider.getLogger('ColumnAnalysisController');

    $scope.tcp = TableChangePropagator;
    $scope.uiState = {
        activeTab: "categorical",
        useFullSampleStatistics: false,
        fullPartitionId: null
    };
    $scope.$watch('uiState.fullPartitionId', function(nv, ov) {
        // note: (null == undefined) = true but (null === undefined) = false
        if (nv != null && ov != null && !angular.equals(ov, nv)) {
            $scope.refreshAnalysis();
        }
    });

    /* Main initialization + column change function */
    $scope.setColumn  = function(column, columns) {
        var changed = $scope.columnName !== column.name;
        $scope.column = column;
        $scope.columnName = column.name;
        $scope.showNumerical = column.isDouble;
        $scope.isDate  = ['Date', 'DateOnly', 'DatetimeNoTz'].indexOf(column.selectedType.name) >= 0;
        $scope.showArray = column.selectedType.name === 'JSONArrayMeaning';
        $scope.showText  = column.selectedType.name === 'FreeText' && $scope.shakerHooks && $scope.shakerHooks.fetchTextAnalysis;
        $scope.showClusters = $scope.shakerHooks && $scope.shakerHooks.fetchClusters;

        if (columns) {  // optional columns list for previous/next
            $scope.columns = columns;
        }
        if ($scope.columns && $scope.columns.length) {
            var columnIndex = $scope.columns.indexOf(column);
            if (columnIndex >= 0) {
                $scope.nextColumn = $scope.columns[columnIndex + 1];
                $scope.prevColumn = $scope.columns[columnIndex - 1];
            }
        }

        if (changed) { // Force the auto-select of analysis type to retrigger
            $scope.analysis = null;
        }
        $scope.textAnalysis = null;
        $scope.refreshAnalysis();
    };
    $scope.updateColumn = function(name) {
        var col = null;
        $scope.table.headers.some(function(h) {
            if (h.name === name) {
                col = h;
                return true;
            }
            return false;
        });
        return col;
    };

    $scope.getFullCHistogram = function() {
        if (!$scope.analysis || !$scope.analysis.fullSampleAnalysis || !$scope.analysis.fullSampleAnalysis.numeric) return null;
        var numeric = $scope.analysis.fullSampleAnalysis.numeric;
        if (numeric.histogram && numeric.histogram.value != null) {
            if (numeric.histogramData) return numeric.histogramData; // cache for $watch
            var histogram = numeric.histogram.value;
            var longestHistogramBar = 0;
            histogram.forEach(function(bin) {longestHistogramBar = Math.max(longestHistogramBar, bin[2]);});
            numeric.histogramData = {min:numeric.min.value, max:numeric.max.value, chistogram:histogram, longestHistogramBar:longestHistogramBar};
            return numeric.histogramData;
        } else {
            return null;
        }
    };
    $scope.getFullBoxplot = function() {
        if (!$scope.analysis || !$scope.analysis.fullSampleAnalysis || !$scope.analysis.fullSampleAnalysis.numeric) return null;
        var numeric = $scope.analysis.fullSampleAnalysis.numeric;
        if ($scope.hasNumeric('median') && $scope.hasNumeric('min') && $scope.hasNumeric('max') && $scope.hasNumeric('p75') && $scope.hasNumeric('p25')) {
            if (numeric.boxplotData) return numeric.boxplotData; // cache for $watch
            var min = numeric.min.value;
            var median = numeric.median.value;
            var max = numeric.max.value;
            var p25 = numeric.p25.value;
            var p75 = numeric.p75.value;
            var iqr = p75 - p25;
            var lowWhisker = Math.max(min, p25 - iqr * 1.5);
            var highWhisker = Math.min(max, p75 + iqr * 1.5);
            numeric.boxplotData = {min:numeric.min.value, max:numeric.max.value, median:median, quartiles:[p25,median, p75], lowWhisker:lowWhisker, highWhisker:highWhisker};
            return numeric.boxplotData;
        } else {
            return null;
        }
    };

    $scope.getFullTop10WithCounts = function() {
        var categorical = $scope.analysis.fullSampleAnalysis.categorical;
        if (categorical.top10WithCounts && categorical.top10WithCounts.value) {
            if (categorical.top10WithCountsData) return categorical.top10WithCountsData; // cache for $watch
            var top10WithCounts = categorical.top10WithCounts.value;
            var total = categorical.count != null && categorical.countMissing != null ? (categorical.count.value + categorical.countMissing.value) : null;
            var top10WithCountsData = [];
            var cum = 0;
            var maxCount = top10WithCounts.length > 0 ? top10WithCounts[0][1] : 0;
            top10WithCounts.forEach(function(pair) {
                var value = pair[0], count = pair[1];
                cum += count;
                var top10WithCountsPoint = {value: value, count:count, cum:cum, maxCount:maxCount};
                if (total != null && total > 0) {
                    top10WithCountsPoint.percent = 100.0 * count / total;
                    top10WithCountsPoint.cumPercent = 100.0 * cum / total;
                }
                top10WithCountsData.push(top10WithCountsPoint);
            });
            categorical.top10WithCountsData = top10WithCountsData;
            return categorical.top10WithCountsData;
        } else {
            return null;
        }
    };

    $scope.numericNeedsRecompute = function(p) {
        if (!$scope.analysis || !$scope.analysis.fullSampleAnalysis || !$scope.analysis.fullSampleAnalysis.numeric) return false;
        if (p in $scope.analysis.fullSampleAnalysis.numeric) {
            var v = $scope.analysis.fullSampleAnalysis.numeric[p];
            return !v.current || (v.value == null && v.reason == null); // no 'reason' field == no reason for missing
        } else {
            return false;
        }
    };
    $scope.hasNumeric = function(p) {
        if (!$scope.analysis || !$scope.analysis.fullSampleAnalysis || !$scope.analysis.fullSampleAnalysis.numeric) return false;
        if (p in $scope.analysis.fullSampleAnalysis.numeric) {
            var v = $scope.analysis.fullSampleAnalysis.numeric[p];
            return v.value != null;
        } else {
            return false;
        }
    };
    $scope.numericsNeedRecompute = function(ps) {
        var need = false;
        ps.forEach(function(p) {need |= $scope.numericNeedsRecompute(p);});
        return need;
    };
    $scope.categoricalNeedsRecompute = function(p) {
        if (!$scope.analysis || !$scope.analysis.fullSampleAnalysis || !$scope.analysis.fullSampleAnalysis.categorical) return false;
        if (p in $scope.analysis.fullSampleAnalysis.categorical) {
            var v = $scope.analysis.fullSampleAnalysis.categorical[p];
            return !v.current || (v.value == null && v.reason == null); // no 'reason' field == no reason for missing
        } else {
            return false;
        }
    };
    $scope.hasCategorical = function(p) {
        if (!$scope.analysis || !$scope.analysis.fullSampleAnalysis || !$scope.analysis.fullSampleAnalysis.categorical) return false;
        if (p in $scope.analysis.fullSampleAnalysis.categorical) {
            var v = $scope.analysis.fullSampleAnalysis.categorical[p];
            return v.value != null;
        } else {
            return false;
        }
    };
    $scope.categoricalsNeedRecompute = function(ps) {
        var need = false;
        ps.forEach(function(p) {need |= $scope.categoricalNeedsRecompute(p);});
        return need;
    };

    $scope.updateUseFullSampleStatistics = function() {
        // leave tabs that are not available in full sample
        if ($scope.uiState.useFullSampleStatistics) {
            if ($scope.uiState.activeTab != 'categorical' && $scope.uiState.activeTab != 'numerical') {
                $scope.uiState.activeTab = 'categorical';
            }
        }
    };
    $scope.$watch('uiState.useFullSampleStatistics', $scope.updateUseFullSampleStatistics);

    $scope.$on("$destroy", $scope.$parent.$on("shakerTableChanged", function() {
        // $scope.column[s] stats may be stale but names should not be -> remap
        $scope.setColumn($scope.updateColumn($scope.columnName),
            !$scope.columns ? null :
                $scope.columns.map(Fn(Fn.prop('name'), $scope.updateColumn)).filter(Fn.SELF));
    }));

    $scope.roundNumerical = function(d, long) {
        return long ? d : $filter('nicePrecision')(d, 5);
    }

    $scope.numData = function(d, interval, long) {
        if ($scope.isDate) {
            if (interval === true) {
                return $filter('friendlyDuration' + (long ? '' : 'Short'))(d);
            } else if (long) {
                return moment(d).toISOString().substring(0, 19) + 'Z'; // drop the milliseconds
            } else {
                return $filter('utcDate')(d, 'YYYY-MM-DD HH:mm:ss');
            }
        }
        return $scope.roundNumerical(d, long);
    };

    $scope.initializeClusterer = function(){
        Assert.inScope($scope, 'analysis');

        var getNumberOfSpaces = function(str){
            return str.split(" ").filter(function(v){return v !== '';}).length - 1;
        }

        var facet = $scope.analysis.alphanumFacet;

        var lengthTotal = 0;
        var spacesTotal = 0;
        for (var i in facet.values) {
            spacesTotal += getNumberOfSpaces(facet.values[i]);
            lengthTotal += facet.values[i].length;
        }

        // Parameters for the clusterer
        $scope.cp.meanLength = Math.round(lengthTotal / facet.values.length);
        $scope.cp.meanSpaces = Math.round(spacesTotal / facet.values.length);
        var nValues = facet.totalNbValues;
        // if average word length is high, clustering is slower

        if (!$scope.cp.initialized){
            if (nValues >= 1500) {
                $scope.cp.blockSize = SpeedLevel.FAST;
            } else {//if (nValues >= 200) {
                $scope.cp.blockSize = SpeedLevel.MID;
            } /*else {
                $scope.cp.blockSize = SpeedLevel.SLOW;
            }*/
        }

        if($scope.cp.meanSpaces < 1){
            $scope.cp.setBased = "false";
        } else {
            $scope.cp.setBased = "true";
        }
    }

    $scope.refreshAnalysis = function() {
        let deferred = $q.defer();
        var first = $scope.analysis == null;
        var setAnalysis = function(data) {
            $scope.analysis = data;

            if (first) {
                if($scope.showNumerical && data.alphanumFacet.totalNbValues > 15) {
                    $scope.uiState.activeTab = 'numerical';
                } else {
                    $scope.uiState.activeTab = 'categorical';
                }
            }

            if ($scope.analysis.numericalAnalysis){
                var na = $scope.analysis.numericalAnalysis;
                $scope.removeBounds = {
                    "1.5 IQR" : [
                        Math.max((na.quartiles[0] - na.iqr * 1.5), na.min),
                        Math.min((na.quartiles[2] + na.iqr * 1.5), na.max)
                    ],
                    "5 IQR" : [
                        Math.max((na.quartiles[0] - na.iqr * 5), na.min),
                        Math.min((na.quartiles[2] + na.iqr * 5), na.max)
                    ]
                };
            }

            data.alphanumFacet.selected = data.alphanumFacet.values.map(Fn.cst(false));
            data.alphanumFacet.maxRatio = data.alphanumFacet.percentages[0];

            if (data.arrayFacet) {
                data.arrayFacet.selected = [];
                data.arrayFacet.values.forEach(function(){
                    data.arrayFacet.selected.push(false);
                })

            }

            $scope.initializeClusterer();
            deferred.resolve(data);
        };
        
        var failAnalysis = function() {
            deferred.reject("failed");
        };
        
        if ( $scope.shakerHooks.fetchDetailedAnalysis ) {
            $scope.shakerHooks.fetchDetailedAnalysis(setAnalysis, failAnalysis, $scope.columnName, 50, $scope.uiState.fullPartitionId, $scope.uiState.useFullSampleStatistics);
        } else {
            deferred.resolve({});
        }
        return deferred.promise;
    };

    $scope.deleteColumn = function() {
        var goner = $scope.columnName,
            col = $scope.nextColumn || $scope.prevColumn;
        if (col) {
            $scope.setColumn(col);
        }

        $scope.addStepNoPreview("ColumnsSelector", { keep: false, columns: [goner], appliesTo: "SINGLE_COLUMN" }, true);
        $scope.mergeLastColumnDeleters();
        $scope.autoSaveForceRefresh();

        if (!col && $scope.dismiss) { //$scope.dismiss is available in the context of a modal
            $scope.dismiss();
        }
    };

    /*******************************************************
     * Generic "current transform" handling
     *******************************************************/

    $scope.currentTransform = null;
    $scope.cancelTransform = function() {
        $scope.currentTransform = null;
        $scope.merge = null;
        $scope.editRow = null;
    };
    // Cancel current transform at each refresh
    $scope.$watch("analysis", $scope.cancelTransform);

    /*******************************************************
     * 'Edit' category view actions
     *******************************************************/

    $scope.startEditValue = function(rowId, objectScope) {
        if($scope.editRow !== rowId) {
            $scope.editRow = rowId;
            objectScope.newValue = $scope.analysis.alphanumFacet.values[$scope.editRow];
            window.setTimeout(function() {
                document.getElementById('analyseCatEdit' + rowId).focus();
            }, 0);
        }
    };
    $scope.handleKey = function($event) {
        switch ($event.keyCode) {
        case 27:
            $scope.editRow = null;
            $event.stopPropagation();
            return;
        case 13:
            $scope.doneEditing($event.delegateTarget.value);
            $event.preventDefault();
            $event.stopPropagation();
            return;
        }
    };
    $scope.doneEditing = function(newValue) {
        if (!newValue) {
            $scope.editRow = null;
            return;
        }
        if ($scope.editRow === null) return;
        Assert.inScope($scope, 'analysis');

        var facets = $scope.analysis.alphanumFacet.values,
            oldValue = facets[$scope.editRow];

        if (newValue !== oldValue) {
            facets[$scope.editRow] = newValue;
            $scope.editRow = null;

            if(oldValue) {
                $scope.addStepNoPreview("FindReplace", {
                    appliesTo: 'SINGLE_COLUMN',
                    columns: [$scope.columnName],
                    mapping: [{from: oldValue, to: newValue}],
                    normalization: 'EXACT',
                    matching: 'FULL_STRING'
                }, true);
            } else {
                $scope.addStepNoPreview("FillEmptyWithValue", {
                    appliesTo: 'SINGLE_COLUMN',
                    columns: [$scope.columnName],
                    value: newValue
                }, true);
            }
            $scope.mergeLastFindReplaces();
            $scope.autoSaveForceRefresh();
            $scope.cancelTransform();
            WT1.event("analyse-category-merge", {mergedVals: 1});
        } else {
            $scope.editRow = null;
        }
    }

    /*******************************************************
     * Regular category view actions
     *******************************************************/

    $scope.nbSelected = function() {
        return $scope.getSelectedValues().length;
    };
    $scope.getSelectedValues = function() {
        if (!$scope.analysis) return [];
        return $scope.analysis.alphanumFacet.values.filter(Fn.from($scope.analysis.alphanumFacet.selected, 1));
    };
    $scope.selectAllValues = function(sel) {
        if (!$scope.analysis) return;
        $scope.analysis.alphanumFacet.selected.forEach(
            function(s, i) { $scope.analysis.alphanumFacet.selected[i] = sel; });
    };

    /* Merging */
    $scope.merge = null;
    $scope.mergeSelected = function(revert) {
        var vals = $scope.getSelectedValues(),
            target = $scope.merge ? $scope.merge.value : (revert ? "Others" : vals[0]),
            hasVals = revert ? vals.length < $scope.analysis.alphanumFacet.totalNbValues : vals.length > 0;
        if (!hasVals) {
            $scope.cancelTransform();
            return;
        }
        $scope.merge = {
            count: revert ? $scope.analysis.alphanumFacet.totalNbValues - vals.length : vals.length,
            index: null, // selection
            revert: !!revert,
            empty: vals.indexOf('') >= 0,
            value: target
        };
        $scope.currentTransform = "merge";
    };
    $scope.mergeTail = function(index) {
        if (typeof index !== 'number') {
            return;
        }
        $scope.merge = {
            count: $scope.analysis.alphanumFacet.totalNbValues - index - 1,
            index: index,
            revert: false,
            empty: $scope.analysis.alphanumFacet.values.slice(0, index + 1).indexOf('') < 0,
            value: "Others"
        };
        $scope.currentTransform = "merge";
    };
    $scope.execMerge = function() {
        Assert.trueish($scope.merge && $scope.merge.value, 'no merge value');
        var vals;
        function filter(v) {
            return v && v !== $scope.merge.value;
        }

        if (typeof $scope.merge.index === 'number') {  // long tail after index
            vals = $scope.analysis.alphanumFacet.values.slice(0, $scope.merge.index + 1).filter(filter);
            if (!$scope.merge.empty) {
                vals.push('');
            }
            if (!vals.length) return;
            $scope.addStepAndRefresh("LongTailGrouper", {
                column: $scope.columnName,
                replace: $scope.merge.value,
                toKeep: vals
            }, true);
            WT1.event("analyse-category-longtailgroup", {keptVals: vals.length, type: 'below'});

        } else if ($scope.merge.revert) {    // long tail (merge unselected)
            vals = $scope.getSelectedValues().filter(filter);
            if (!$scope.merge.empty) {
                vals.push('');
            }
            if (!vals.length) return;
            $scope.addStepAndRefresh("LongTailGrouper", {
                column: $scope.columnName,
                replace: $scope.merge.value,
                toKeep: vals
            }, true);
            WT1.event("analyse-category-longtailgroup", {keptVals: vals.length, type: 'selection'});

        } else {    // merge selected
            vals = $scope.getSelectedValues().filter(filter);
            if (vals.length) {
                $scope.addStepNoPreview("FindReplace", {
                    appliesTo: 'SINGLE_COLUMN',
                    columns: [$scope.columnName],
                    mapping: vals.map(function(v) { return {from: v, to: $scope.merge.value}; }),
                    normalization: 'EXACT',
                    matching: 'FULL_STRING'
                }, true);
                $scope.mergeLastFindReplaces();
            }
            if ($scope.merge.empty) {
                $scope.addStepNoPreview("FillEmptyWithValue", {
                    appliesTo: 'SINGLE_COLUMN',
                    columns: [$scope.columnName],
                    value: $scope.merge.value
                }, true);
            }
            WT1.event("analyse-category-merge", {mergedVals: vals.length + ($scope.merge.empty ? 1 : 0)});
        }
        $scope.autoSaveForceRefresh();
        $scope.cancelTransform();
    };

    /* Removing, Keeping, Clearing */
    function flagValues(action, values) {
        var touched = 0;
        if (values.indexOf('') >= 0) {
            $scope.addStepNoPreview("RemoveRowsOnEmpty",{
                appliesTo: 'SINGLE_COLUMN',
                columns: [$scope.columnName],
                keep: action === 'KEEP_ROW'
            }, true);
            values = values.filter(Fn.SELF);
            touched++;
        }
        if (values.length) {
            $scope.addStepNoPreview("FilterOnValue", {
                appliesTo: 'SINGLE_COLUMN',
                columns: [$scope.columnName],
                action: action,
                matchingMode: 'FULL_STRING',
                normalizationMode: 'EXACT',
                values: values
            }, true);
            touched += values.length;
        }
        if (!touched) return 0;
        if (['REMOVE_ROW', 'KEEP_ROW', 'CLEAR_CELL'].includes(action)) {
            $scope.mergeLastDeleteRows();
        }
        $scope.autoSaveForceRefresh();
        return touched;
    }
    $scope.keepOnlyRowsOnSelection = function() {
        var n = flagValues('KEEP_ROW', $scope.getSelectedValues(), true);
        WT1.event("analyse-category-keeponlyselected", {keptVals: n});
    }
    $scope.removeRowsOnSelection = function() {
        var n = flagValues('REMOVE_ROW', $scope.getSelectedValues());
        WT1.event("analyse-category-removeselected", {removedVals: n});
    }
    $scope.removeValue = function(index) {
        flagValues('REMOVE_ROW', [$scope.analysis.alphanumFacet.values[index]]);
        WT1.event("analyse-category-removeone");
    }
    $scope.keepValue = function(index) {
        flagValues('KEEP_ROW', [$scope.analysis.alphanumFacet.values[index]]);
        WT1.event("analyse-category-keepone");
    };
    $scope.clearValue = function(index) {
        flagValues('CLEAR_CELL', [$scope.analysis.alphanumFacet.values[index]]);
        WT1.event("analyse-category-clearone");
    };
    $scope.clearCellsOnSelection = function() {
        var n = flagValues('CLEAR_CELL', $scope.getSelectedValues().filter(Fn.SELF));
        WT1.event("analyse-category-clearselected", {clearedVals: n});
    };
    $scope.removeEmpty = function() { flagValues('REMOVE_ROW', ['']); };
    $scope.emptyCellsInSelection = function() {
        return $scope.getSelectedValues().indexOf('') >= 0;
    };
    $scope.onlyEmptyCellsInSelection = function() {
        return $scope.getSelectedValues().indexOf('') >= 0 && $scope.getSelectedValues().length === 1;
    };

    /* Filtering */
    $scope.filterViewOnSelection = function() {
        $scope.addColumnFilter($scope.columnName,
            // transform ['a', 'b'] into {a: true, b: true} for facets
            $scope.getSelectedValues().reduce(function(o, k) { o[k] = true; return o; }, {}),
            "full_string", $scope.column.selectedType.name, $scope.column.isDouble);
        $scope.dismiss();
        $scope.autoSaveForceRefresh();
    };
    $scope.handleInvalids = function(action) {  // e.g. REMOVE_ROW or CLEAR_CELL
        $scope.addStepNoPreview("FilterOnBadType", {
            appliesTo: 'SINGLE_COLUMN',
            columns: [$scope.columnName],
            action: action,
            type: $scope.column.selectedType.name
        }, true);
        if (action === 'REMOVE_ROW' || action === 'KEEP_ROW') {
            $scope.mergeLastDeleteRows();
        }
        $scope.autoSaveForceRefresh();
    };

    /* **************************************************************************
     * Categorical clusterer actions
     * **************************************************************************/

    var SpeedLevel = {FAST: 0, MID: 1, SLOW: 2};

    // cp : clustering parameters
    $scope.cp = {blockSize : SpeedLevel.MID, meanLength : 0, meanSpaces : 0,
        fuzziness : 0, nowComputing : false,
        initialized : false, setBased : false, radius : 0,
        timeOut : 15, clusters : [], mergeValues : [],
        allSelected : false, selected : [], hasTimedOut : false};

    $scope.clustersSelectAll = function() {
        $scope.cp.selected = $scope.cp.selected.map(Fn.cst($scope.cp.allSelected));
    };
    $scope.nbClustersSelected = function() {
        return $scope.cp.selected.filter(Fn.SELF).length;
    };
    $scope.refreshClusterer = function(recur) {
        var lastRecur = true;
        $scope.cp.nowComputing = true;

        var blockSize = 0,
            setBased = $scope.cp.setBased === 'true';
        $scope.cp.selected = [];
        $scope.cp.allSelected = false;
        $scope.cp.mergeValues = [];

        if (setBased){
            switch (+$scope.cp.fuzziness) {
                case 0:  $scope.cp.radius = 0.8;   break; // 4 words out of 5
                case 1:  $scope.cp.radius = 0.775; break;
                case 2:  $scope.cp.radius = 0.75;  break; // 3 words out of 4
            }
        } else {
            // we define slightly fuzzy as 0.5 mistake per word, very fuzzy as 1.5
            var nWords = $scope.cp.meanSpaces + 1;
            switch (+$scope.cp.fuzziness) {
                case 0:  $scope.cp.radius = 0.5; break;
                case 1:  $scope.cp.radius = 1.0; break;
                case 2:  $scope.cp.radius = 1.5; break;
            }
            $scope.cp.radius = Math.max(1, Math.round($scope.cp.radius * nWords));

            // a high blocksize => less calculations => relatively faster
            // we are slowed by a high number of distinct values, not by sample size
            if($scope.cp.meanLength >= 40){ // Usually, a bad idea to compute edit distance
                blockSize = 10;
            } else {
                // 1-9 => 2, 10-19 => 3, 20-29 => 4, 30-39 => 5
                var lowBlockSize = Math.floor($scope.cp.meanLength / 10) + 1;
                Logger.log("Low block size" + lowBlockSize);
                if ($scope.cp.blockSize == SpeedLevel.MID){
                    blockSize = lowBlockSize ;
                } else if ($scope.cp.blockSize == SpeedLevel.FAST){
                    blockSize = lowBlockSize * 2;
                }
            } // blockSize should be between 1 & 10
        }

        var setClusters = function(data) {
            $scope.cp.hasTimedOut = data.timedOut;
            $scope.cp.clusters = data.values;
            $scope.cp.initialized = true;
            $scope.cp.mergeValues = data.values.map(Fn.prop(0));
            $scope.cp.selected = data.values.map(Fn.cst(false));
            $scope.cp.nowComputing = false;
        };
        if ( $scope.shakerHooks.fetchClusters ) {
            $scope.shakerHooks.fetchClusters(setClusters, $scope.columnName, setBased, $scope.cp.radius, $scope.cp.timeOut, blockSize);
        }
    };
    $scope.mergeSelectedClusters = function() {
        var mapping = [],
            index = {},
            mergeCount = 0;
        // we have to go backwards to process smallest clusters first
        // so that the rules are effective in case of nested clusters
        for (var cluster = $scope.cp.selected.length - 1; cluster >= 0; cluster--) {
            if (!$scope.cp.selected[cluster]) continue;
            $scope.cp.clusters[cluster].forEach(function(toMerge) {
                if(!toMerge) {
                    $scope.addStepNoPreview("FillEmptyWithValue", {
                        appliesTo: 'SINGLE_COLUMN',
                        columns: [$scope.columnName],
                        value: $scope.cp.mergeValues[cluster]
                    }, true);
                    mergeCount++;
                } else if (toMerge !== $scope.cp.mergeValues[cluster]) {
                    if (toMerge in index){ // remove
                        mapping.splice(index[toMerge], 1);
                        // updating indexes: reduce by 1 all the indexes that are above
                        // the removed index in mapping array
                        for (var toMergeTmp in index) {
                            if (index[toMergeTmp] > index[toMerge]) {
                                index[toMergeTmp] -= 1;
                            }
                        }
                    }
                    index[toMerge] = mapping.push({from: toMerge, to: $scope.cp.mergeValues[cluster]}) - 1;
                    mergeCount++;
                }
            });
        }

        WT1.event("analyse-category-merge", {mergedVals: mergeCount});
        $scope.addStepNoPreview("FindReplace", {
            appliesTo: 'SINGLE_COLUMN',
            columns: [$scope.columnName],
            mapping: mapping,
            matching: 'FULL_STRING',
            normalization: 'EXACT'
        });
        $scope.mergeLastFindReplaces();
        $scope.autoSaveForceRefresh();
        $scope.cancelTransform();
        $scope.dismiss();
    }

     /*******************************************************
     * Text analysis actions
     *******************************************************/

     $scope.textSettings = {
        normalize: true,
        stem: false,
        clearStopWords: true,
        language: 'english'
    };
    var setTextAnalysis = function(data) {$scope.textAnalysis = data;};
    $scope.computeTextAnalysis = function() {
        if ($scope.shakerHooks.fetchTextAnalysis) {
            $scope.shakerHooks.fetchTextAnalysis(setTextAnalysis, $scope.columnName, $scope.textSettings);
        }
    }
    /*******************************************************
     * Numerical analysis actions
     *******************************************************/

    function niceToPrecision(val, p) {
        if (Math.abs(val) < Math.pow(10, p)) {
            return val.toPrecision(p);
        } else {
            return val.toFixed(0);
        }
    }
    $scope.removeOutliers = function(iqrRatio) {
        var na = $scope.analysis.numericalAnalysis;
        Assert.trueish(na, 'no numericalAnalysis');
        var min = Math.max((na.quartiles[0] - na.iqr * iqrRatio), na.min);
        var max = Math.min((na.quartiles[2] + na.iqr * iqrRatio), na.max);
        WT1.event("analyse-numerical-rmoutliers", {iqrRatio: iqrRatio});

        if ($scope.isDate) {
            $scope.addStepNoPreview("FilterOnDate", {
                appliesTo: 'SINGLE_COLUMN',
                columns: [$scope.column],
                action: 'KEEP_ROW',
                filterType: 'RANGE',
                min: $scope.numData(min, false, true),
                max: $scope.numData(max, false, true),
                timezone_id: "UTC",
                part: 'YEAR',
                option: {
                    containsCurrentDatePart: true,
                    isUntilNow: false,
                    last: 0,
                    next: 0
                },
            }, true);
        } else {
            $scope.addStepNoPreview("FilterOnNumericalRange", {
                appliesTo: 'SINGLE_COLUMN',
                columns: [$scope.columnName],
                action: 'KEEP_ROW',
                min: niceToPrecision(min, 4),
                max: niceToPrecision(max, 4)
            }, true);
        }

        $scope.autoSaveForceRefresh();
    };
    $scope.clipOutliers = function(iqrRatio, clear) {
        var na = $scope.analysis.numericalAnalysis;
        Assert.trueish(na, 'no numericalAnalysis');
        var min = Math.max((na.quartiles[0] - na.iqr * iqrRatio), na.min);
        var max = Math.min((na.quartiles[2] + na.iqr * iqrRatio), na.max);
        WT1.event("analyse-numerical-clipoutliers", {iqrRatio: iqrRatio, clear: !!clear});

        if ($scope.isDate) {
            $scope.addStepNoPreview("FilterOnDate", {
                appliesTo: 'SINGLE_COLUMN',
                columns: [$scope.column],
                action: 'KEEP_ROW',
                filterType: 'RANGE',
                min: $scope.numData(min, false, true),
                max: $scope.numData(max, false, true),
                timezone_id: "UTC",
                part: 'YEAR',
                option: {
                    containsCurrentDatePart: true,
                    isUntilNow: false,
                    last: 0,
                    next: 0
                }
            }, true);
        } else {
            $scope.addStepNoPreview("MinMaxProcessor", {
                columns: [$scope.columnName],
                clear: !!clear,
                lowerBound: niceToPrecision(min, 4),
                upperBound: niceToPrecision(max, 4)
            }, true);
        }

        $scope.autoSaveForceRefresh();
    };

    /*******************************************************
     * Array view action
     *******************************************************/

    $scope.arraySelectAll = function(sel) {
        $scope.analysis.arrayFacet.selected = $scope.analysis.arrayFacet.values.map(Fn.cst(sel));
    };
    $scope.arrayNbSelected = function() {
        if (!$scope.analysis) return 0;
        return $scope.analysis.arrayFacet.selected.filter(Fn.SELF).length;
    };
    $scope.getArraySelectedValues = function() {
        if (!$scope.analysis) return [];
        return $scope.analysis.arrayFacet.values.filter(Fn.from($scope.analysis.arrayFacet.selected, 1));
    };
    /* Removing */
    $scope.arrayRemoveSelectedRows = function(keep) {
        var vals = $scope.getArraySelectedValues();
        Assert.trueish(vals.length > 0, 'no selected values');

        vals.forEach(function(val) {
            $scope.addStepNoPreview("FilterOnCustomFormula", {
                appliesTo: 'SINGLE_COLUMN',
                columns: [$scope.columnName],
                action: keep ? 'KEEP_ROW' : 'REMOVE_ROW',
                expression: 'arrayContains(' + $scope.columnName + ', "' + val.replace(/"/g,"\\\"") + '")'
           }, true);
        })
        WT1.event("analyse-array-removeselected", {removedVals: vals.length});
        $scope.mergeLastDeleteRows();
        $scope.autoSaveAutoRefresh();
    };

});
})();

;
(function() {
    'use strict';

    var app = angular.module('dataiku.shaker.table', ['dataiku.filters', 'platypus.utils', 'dataiku.constants']);

    const BIN_UNCOLORED = 0;
    const BIN_EMPTY = -1;
    //const BIN_HAPAX = -2;
    //const BIN_INVALID = -3;
    const EVEN = " even";
    const ODD = " odd";

    const binCache = new Map([
        ["CA1-1", 1],
        ["CA1-2", 2],
        ["CA1-3", 3],
        ["CA1-4", 4],
        ["CA1-5", 5],
        ["CA1-6", 6],
        ["CA1-7", 7],
        ["CA1-8", 8],
        ["CA1-9", 9],
        ["CA1-10", 10],
        ["CA1-11", 11],
        ["CA1--1", -1],
        ["CA1--2", -2],
        ["CB-1", 1],
        ["CB-2", 2],
        ["CB--3", -3],
        ["CN-1", 1],
        ["CN-2", 2],
        ["CN-3", 3],
        ["CN-4", 4],
        ["CN-5", 5],
        ["CN--1", -1],
        ["CN--3", -3],
        ["CG-1", 1],
        ["CG-2", 2],
        ["CG--3", -3],
    ]);

    app.service('ShakerTableModel', function() {
        return function(tableData, scope, maxPageWidth, maxPageHeight) {
            var PAGE_WIDTH = !maxPageWidth ? Math.pow(2,5) : maxPageWidth;
            var PAGE_HEIGHT = !maxPageHeight ? Math.pow(2,6) : maxPageHeight;

            var pageFromData = function(I,J,data) {
                return function(i,j)  {
                    var offset = (i-I)*data.nbCols + (j-J);
                    return {
                        content: data.content[offset],
                        highlightedIndices: data.highlightedIndices && data.highlightedIndices[offset],
                        status: data.status[offset],
                        colorBin : data.colorBin ? data.colorBin[offset] : null,
                        origRowIdx : data.origRowIdx[i-I],
                        rowId: i,
                        colId: j,
                    };
                };
            };

            tableData = $.extend(
                new fattable.PagedAsyncTableModel(),
                tableData,
            {
                getHeader: function(j, cb) {
                    // Here we fork the scope for each header to append
                    // a new header property.
                    var newScope = scope.$new(false);
                    newScope.header = tableData.headers[j];
                    cb(newScope);
                }
            ,
                searchHeader: function(q) {
                    q = q.toLowerCase();
                    var results = [];
                    for (let i = 0; i < tableData.headers.length; i++) {
                        let header = tableData.headers[i];
                        if (header.name.toLowerCase() == q) {
                            results.push(i);
                        }
                    }
                    for (let i = 0; i < tableData.headers.length; i++) {
                        let header = tableData.headers[i];
                        if ((header.name.toLowerCase().indexOf(q) != -1) && (results.indexOf(i) == -1) ) {
                            results.push(i);
                        }
                    }
                    return results;

                }
            ,
                hasHeader:  function() {
                    return true;  // we are synchronous for headers.
                }
            ,
                fetchCellPage: function(pageName, cb) {
                    var coords =  JSON.parse(pageName);
                    var I = coords[0];
                    var J = coords[1];
                    var nbRequestedRows = Math.min(this.totalKeptRows - I, PAGE_HEIGHT);
                    var nbRequestedCols = Math.min(this.allColumnNames.length - J, PAGE_WIDTH);
                    var promise = scope.getTableChunk(I, nbRequestedRows, J, nbRequestedCols);
                    promise.then(function(resp) {
                        var page = pageFromData(I,J,resp);
                        cb(page);
                    });
                }
            ,
                cellPageName: function(i,j) {
                    return JSON.stringify([Math.floor(i / PAGE_HEIGHT) * PAGE_HEIGHT, Math.floor(j / PAGE_WIDTH) * PAGE_WIDTH]);
                }
            });

            // populate the page cache with the initial data.
            var initialPage = pageFromData(0,0, tableData.initialChunk);
            var initialPageName = tableData.cellPageName(0,0);
            tableData.pageCache.set(initialPageName, initialPage);
            tableData.PAGE_WIDTH = PAGE_WIDTH;
            tableData.PAGE_HEIGHT = PAGE_HEIGHT;
            return tableData;
        }
    });


    app.service('computeColumnWidths', function() {
        return function(sampleData, headers, minColumnWidth, hasAnyFilterOnColumn, columnWidthsByName, columnUseScientificNotationByName, reset = false) {
            // Upper bounds for a cell/col containing only capital M: { header = 99, body = 95 }
            // Lower bound wih only small l: {header = 2.9, body = 2.6 }

            // Seems reasonable: 7 / 7.5
            const CELL_LETTER_WIDTH = 7;
            const HEADER_LETTER_WIDTH = 7.5;

            const CELL_MARGIN = 15;
            const HEADER_MARGIN = 15;
            const MAX_WIDTH = 300;
            const FILTER_FLAG_WIDTH = 20;

            let columnWidthsByIndex = [];
            const nbCols = headers.length;

            for (var colId = 0; colId < nbCols; colId++) {
                const header = headers[colId];
                const columnName = header.name;
                let columnWidth;

                if (!reset) {
                    columnWidth = columnWidthsByName[columnName];
                }

                if (!(Number.isInteger(columnWidth)) && columnUseScientificNotationByName[columnName] !== true) {
                    // when the column contains doubles, we rewrite the server-computed width in order to contain full numbers (no scientific notation) from 1e-15 to 9.999e15
                    let ncharsToShow = header.selectedType && ['DoubleMeaning', 'FrenchDoubleMeaning'].includes(header.selectedType.name) ? Math.max(17, header.ncharsToShow) : header.ncharsToShow;
                    let cellColumnWidth =  Math.ceil(ncharsToShow * CELL_LETTER_WIDTH + CELL_MARGIN);
                    let colColumnWidth =  Math.ceil(header.name.length * HEADER_LETTER_WIDTH + HEADER_MARGIN);
                    columnWidth = Math.max(colColumnWidth, cellColumnWidth);
                    columnWidth = fattable.bound(columnWidth, minColumnWidth, MAX_WIDTH);

                    if ((hasAnyFilterOnColumn === undefined) || hasAnyFilterOnColumn(header.name)) {
                        columnWidth += FILTER_FLAG_WIDTH;
                    }

                    columnWidthsByName[columnName] = columnWidth;
                }

                columnWidthsByIndex.push(columnWidth);
            }
            return [ columnWidthsByIndex, columnWidthsByName ];
        };
    });

    app.directive('fattable', function(ShakerTableModel, computeColumnWidths, ColorUtils, ContextualMenu, CreateModalFromDOMElement, CreateModalFromTemplate,
            $filter, $templateCache, $q, $http, $timeout, $rootScope, $stateParams, $compile,Debounce, ShakerProcessorsUtils, ShakerSuggestionsEngine, Logger, WT1, FatTouchableService, FatDraggableService, FatResizableService,
            ClipboardUtils, PrettyPrintDoubleService, ChartFilters, DashboardFilters, DashboardUtils, DKU_NO_VALUE, FeatureFlagsService, findAppliedColoringGroup, ConditionalFormattingEditorService) {
        const insertHighlightTags = (content, indices) => {
            const contents = []
            let sliceStart = 0;
            for (let i = 0; i < indices.length; i++) {
                const start = indices[i][0];
                const end = indices[i][1];
                if (sliceStart < start) {
                    contents.push(sanitize(content.slice(sliceStart, start)));
                }

                contents.push("<mark>");
                contents.push(sanitize(content.slice(start, end)));
                contents.push("</mark>");

                // there is another highlighted match after that one, get its start to slice
                const sliceEnd = i < indices.length - 1 ? indices[i + 1][0] : content.length;
                contents.push(sanitize(content.slice(end, sliceEnd)));
                sliceStart = sliceEnd;
            }
            return contents.join("");
        }


        // Fattable delegates filling cells / columns header
        // with content to this object.
        function ShakerTablePainter(scope) {

            const hasHighlights = (cell) => {
                return scope.shaker.coloring.highlightSearchMatches && cell.highlightedIndices;
            }

            const applyHighlights = (contentWithColor) => {
                let contentWithColorAndHighlight = contentWithColor;

                if (scope.shaker.coloring.highlightWhitespaces) {
                    contentWithColorAndHighlight = contentWithColorAndHighlight
                            .replace(/^(\s*)/, "<span class='ls'>$1</span>")
                            .replace(/(\s*)$/, "<span class='ts'>$1</span>")
                            .replace(/(\s\s+)/g, "<span class='ms'>$1</span>");
                }

                return contentWithColorAndHighlight;
            }

            const isValidHTTPUrl = (potentialUrl) => {
                let url;
                
                try {
                  url = new URL(potentialUrl);
                } catch (_) {
                  return false;  
                }
              
                return url.protocol === "http:" || url.protocol === "https:";
              }

            const getElForLink = (el, content, cell) => {
                const linkEl = document.createElement('a');
                const sanitizedContent = sanitize(content);
                linkEl.href = content;
                linkEl.target = '_blank';
                linkEl.rel = 'noopener noreferrer';

                if (scope.shaker.coloring) {
                    let contentWithColor = hasHighlights(cell) ? insertHighlightTags(content, cell.highlightedIndices) : sanitizedContent;
                    let contentWithColorAndHighlight = applyHighlights(contentWithColor);
                    linkEl.innerHTML = contentWithColorAndHighlight;
                    $(el).html(linkEl);
                } else {
                    linkEl.textContent = sanitizedContent;
                    el.appendChild(linkEl);
                }

                return el;
            }

            const getElForText = (el, content, cell) => {
                el.textContent = content;
                if (scope.shaker.coloring) {
                    let contentWithColor = hasHighlights(cell) ? insertHighlightTags(content, cell.highlightedIndices) : sanitize(content);
                    let contentWithColorAndHighlight = applyHighlights(contentWithColor);

                    $(el).html(contentWithColorAndHighlight);
                }
                return el;
            }

            return $.extend(new fattable.Painter(), {

                setupHeader: function(el) {
                    el.setAttribute("column-header", "header");
                    el.setAttribute("data-page-tour", "column-header");
                    el.setAttribute("ng-class", "{'columnHeader': true, 'filtered': hasAnyFilterOnColumn(column.name)}");
                 }
            ,
                fillHeader: function(el, headerScope)  {
                    var $el = $(el);
                    this.destroyFormerHeaderScope(el);
                    el.scopeToDestroy = headerScope;
                    $el.empty();
                    $compile($el)(headerScope);
                }
            ,
                destroyFormerHeaderScope: function(el) {
                    if (el.scopeToDestroy !== undefined) {
                        el.scopeToDestroy.$destroy();
                        el.scopeToDestroy = undefined;
                    }
                }
            ,
                fillCellPending: function(el,cell) {
                    el.textContent = "Wait...";
                    el.className = "PENDING"
                }
            ,
                fillCell: function(el, cell)  {
                    const coloring = scope.shaker.coloring;

                    const MAX_TITLE_LENGTH = 980;
                    el.dataset.rowId = cell.rowId;
                    // `cell.colId` represents the id of the column after removing unselected columns
                    el.dataset.colId = cell.colId;

                    const columnLabel = scope.tableModel.headers[cell.colId].name;
                    const viewMoreContentLabel = `...\n\nShift + v to ${scope.isCellKO(cell.status) || !scope.columnSupportsSpecialPreview(columnLabel) ? "view complete cell value" : "preview" }.`;


                    // optionally rewrite the cell content when meaning is double (only use scientific notation if exponent is >15)
                    if(scope.tableModel.headers[cell.colId].selectedType && ['DoubleMeaning', 'FrenchDoubleMeaning'].includes(scope.tableModel.headers[cell.colId].selectedType.name) &&
                    scope.shaker.columnUseScientificNotationByName[columnLabel] !== true) {
                        cell.content = PrettyPrintDoubleService.avoidScientificNotation(cell.content);
                    }

                    el.title = cell.content && cell.content.length > MAX_TITLE_LENGTH ? cell.content.slice(0, MAX_TITLE_LENGTH) + viewMoreContentLabel : cell.content;

                    if (cell.colorBin !== null) {
                        let coloringGroup = null;
                        if (coloring.scheme === "COLORING_GROUPS") {
                            coloringGroup = findAppliedColoringGroup(coloring.coloringGroups, columnLabel);
                        }

                        // Also checks if the based on column has not been deleted
                        let isBasedOnAnotherColumnScope =
                            coloringGroup && coloringGroup.scope === "ALL_COLUMNS_BASED_ON_ANOTHER_COLUMN";

                        let isColoredByRules =
                            coloring.scheme === "INDIVIDUAL_COLUMNS_RULES" ||
                            coloring.scheme === "SINGLE_COLUMN_RULES" ||
                            (coloringGroup && coloringGroup.scheme === "RULES");

                        if (isColoredByRules) {
                            // Alternates cell style every row even if no conditional formatting rules applies
                            el.className = cell.status + [EVEN, ODD][cell.rowId % 2];
                            // Resets the element style to avoid side effects from a previous fillCell
                            el.style.backgroundColor = "";
                            el.style.color = "";

                            if (cell.colorBin !== undefined && cell.colorBin !== BIN_EMPTY) {
                                // Collects the filter descriptions from the group of rules associated to the current column
                                let rulesGroupFilterDescs = null;

                                if (coloring.scheme === "COLORING_GROUPS") {
                                    if (
                                        (coloringGroup.scope === "COLUMNS" || isBasedOnAnotherColumnScope) &&
                                        coloringGroup.rulesGroup.filterDescs.length > 0
                                    ) {
                                        rulesGroupFilterDescs = coloringGroup.rulesGroup.filterDescs;
                                    }
                                }
                                // Legacy logic
                                else if (coloring.scheme === "INDIVIDUAL_COLUMNS_RULES") {
                                    const columnRules = angular.copy(
                                        scope.shaker.coloring.individualColumnsRules.filter(
                                            r => r.column === columnLabel
                                        )
                                    );

                                    if (columnRules[0] && columnRules[0].rulesDesc) {
                                        rulesGroupFilterDescs = columnRules[0].rulesDesc;
                                    }
                                }
                                // Legacy logic
                                else if (coloring.scheme === "SINGLE_COLUMN_RULES") {
                                    const columnRules = angular.copy(
                                        scope.shaker.coloring.individualColumnsRules.filter(
                                            r => r.column === coloring.singleColumn
                                        )
                                    );

                                    if (columnRules[0] && columnRules[0].rulesDesc) {
                                        rulesGroupFilterDescs = columnRules[0].rulesDesc;
                                    }
                                }

                                if (rulesGroupFilterDescs) {
                                    let rules = rulesGroupFilterDescs;
                                    if (
                                        cell.colorBin >= 0 &&
                                        cell.colorBin < rules.length &&
                                        "uiData" in rules[cell.colorBin]
                                    ) {
                                        if ("color" in rules[cell.colorBin].uiData) {
                                            if (rules[cell.colorBin].enabled === true) {
                                                el.className =
                                                    el.className + " " + rules[cell.colorBin].uiData.color.styleClass;
                                            }
                                            if (el.className.split(" ").includes("shaker-color-rule--custom")) {
                                                // An empty string represents a "none" color.
                                                // Because the css class above is set to even/odd, there is nothing else
                                                // to do when the color is "none".
                                                el.style.backgroundColor =
                                                    rules[cell.colorBin].uiData.color.styleCustomBackgroundColor;
                                                el.style.color = rules[cell.colorBin].uiData.color.styleCustomFontColor;
                                            }
                                        }
                                    }
                                }
                            }
                        } else {
                            // Reset styles to avoid unexpected behavior with previous cells colors
                            el.style.color = "";
                            el.style.backgroundColor = "";

                            const colorBinFloor = Math.floor(cell.colorBin);

                            const isColorScale = coloringGroup && coloringGroup.scheme === "COLOR_SCALE";
                            const hasValidColorScale =
                                isColorScale &&
                                coloringGroup.colorScaleDef &&
                                coloringGroup.colorScaleDef.sample &&
                                colorBinFloor > 0;

                            if (hasValidColorScale) {
                                if (colorBinFloor <= coloringGroup.colorScaleDef.sample.length) {
                                    // Interpolates cell colors for numbers only
                                    if (
                                        cell.status &&
                                        cell.status.indexOf("CN") !== -1 &&
                                        colorBinFloor < coloringGroup.colorScaleDef.sample.length
                                    ) {
                                        el.style.backgroundColor = ColorUtils.getLerpRGB(
                                            coloringGroup.colorScaleDef.sample[colorBinFloor - 1],
                                            coloringGroup.colorScaleDef.sample[colorBinFloor],
                                            cell.colorBin - colorBinFloor
                                        );
                                    } else {
                                        el.style.backgroundColor =
                                            coloringGroup.colorScaleDef.sample[colorBinFloor - 1];
                                    }
                                    el.style.color = ColorUtils.getFontContrastColor(el.style.backgroundColor);

                                    el.className = cell.status + "-" + colorBinFloor + " shaker-color-scale";
                                } else {
                                    // Alternates white/grey
                                    el.className = cell.status + " VU " + [EVEN, ODD][cell.rowId % 2];
                                }
                            } else {
                                if (colorBinFloor === BIN_UNCOLORED) {
                                    el.className = cell.status + [EVEN, ODD][cell.rowId % 2];
                                } else if (colorBinFloor > 0 && cell.status && cell.status.indexOf("CN") !== -1) {
                                    // Interpolates cell colors for numbers only
                                    const sample = ConditionalFormattingEditorService.getScalePaletteForNumber();
                                    el.style.backgroundColor = ColorUtils.getLerpRGB(
                                        sample[colorBinFloor - 1],
                                        sample[colorBinFloor],
                                        cell.colorBin - colorBinFloor
                                    );
                                    el.style.color = ColorUtils.getFontContrastColor(el.style.backgroundColor);

                                    el.className = cell.status + "-" + colorBinFloor + " shaker-color-scale";
                                } else {
                                    el.className = cell.status + "-" + colorBinFloor + [EVEN, ODD][cell.rowId % 2];
                                }
                            }
                        }
                    } else {
                        el.className = cell.status + [EVEN, ODD][cell.rowId % 2];
                    }
                    if (scope.searchableDataset) {
                        if (!el.className.split(" ").includes("VU")) { // does not include class name if already here
                            el.className += " VU";  // Valid unit when hovering cell, not set in the backend
                        }
                        const isColumnMenuDisabledFn = scope.shaker.$isColumnMenuDisabled;
                        if (isColumnMenuDisabledFn && isColumnMenuDisabledFn(columnLabel)) {
                            el.className += " IG"; // IG = ignore, not part of the schema
                        }
                    }

                    if (scope.shakerState.lockedHighlighting.indexOf(cell.rowId) >=0) {
                        el.className += " FH";
                    }
                    if (scope.shakerState.selectedRow === cell.rowId) {
                        el.className += " SELECTED";
                    }
                    if(!cell.content){
                        el.textContent = "";
                        return;
                    }

                    // highlight selections
                    var lastDisplayedRow = scope.shakerTable.firstVisibleRow + scope.shakerTable.nbRowsVisible;
                    const content = cell.content.replace(/(\r\n|\n)/g, "¶");
                    const shouldDisplayLink = $rootScope.appConfig.dataTableLinksEnabled && scope.shaker.origin !== "PREPARE_RECIPE" && scope.tableModel.headers[cell.colId].meaningLabel === 'URL' && isValidHTTPUrl(content);

                    if (shouldDisplayLink) {
                        el = getElForLink(el, content, cell);
                    } else {
                        el = getElForText(el, content, cell);
                    }

                    el.appendChild(document.createElement('div'));
                }
            ,
                setupCell: function(el) {
                    el.oncontextmenu = function(evt) {
                        var row = el.dataset.rowId;
                        var col = el.dataset.colId;
                        scope.showCellPopup(el, row, col, evt);
                        return false;
                    };
                }
            ,
                cleanUpCell: function(cellDiv) {
                    $(cellDiv).remove();
                }
            ,
                cleanUpheader: function(headerDiv) {
                    this.destroyFormerHeaderScope(headerDiv);
                    $(headerDiv).remove();
                }
            });
        }
        

        return {
            restrict: 'A',
            scope: true,
            link: function(scope, element, attrs) {

                const getSelectedCell = function(hasHighlightMatches, elt) {
                    if (hasHighlightMatches && elt.tagName === scope.highlightTagName) {
                        // on a searchable dataset with and a highlighted word, so get parent
                        return elt.parentElement;
                    }
                    return elt;
                }

                var currentMousePos;

                $(element).mousemove(function(evt) {
                    currentMousePos = {
                        x : evt.clientX,
                        y: evt.clientY
                    }
                });

                var tableDataExpr = attrs.fattableData;

                { // bind "c" to "scroll to column"

                    var shown = false;

                    //<input type="text" class="form-control" ng-model="selectedState" ng-options="state for state in states" placeholder="Enter state" bs-typeahead>
                    scope.openSearchBox = function() {
                        shown=true;
                        const newScope = scope.$new();
                        const controller = function() {
                            newScope.searchHeaderName = function(query) {
                                const columnIds = scope.tableModel.searchHeader(query);
                                return columnIds.map(i => scope.tableModel.headers[i].name);
                            }
                            newScope.move = function(query) {
                                const columnIds = scope.tableModel.searchHeader(query);
                                if (columnIds.length > 0) {
                                    const columnSelected = columnIds[0];
                                    scope.shakerTable.goTo(undefined, columnSelected);
                                }
                            }
                            $("body").addClass("fattable-searchbox-modal");
                            $("#fattable-search").focus();
                        };
                        CreateModalFromTemplate('/templates/shaker/search-column.html', newScope, controller, function(modalScope) {
                            $(".modal").one("hide", function() {
                                shown = false;
                                $("body").removeClass("fattable-searchbox-modal");
                            });
                            modalScope.onSubmit = function(e) {
                                modalScope.dismiss();
                            }
                            modalScope.$on('typeahead-updated', modalScope.onSubmit);
                        });
                    };

                    var $window = $(window);

                    var keyCodes = {
                        tab: 9,
                        pageup: 33,
                        pagedown: 34,
                        left: 37,
                        up: 38,
                        right: 39,
                        down: 40
                    };

                    if (!DashboardUtils.isInDashboard()) {
                        Mousetrap.bind("c", function() {
                            if (!shown) {
                                scope.hidePopups();
                                scope.openSearchBox();
                            }
                        });
                    }

                    $window.on("keydown.fattable", function(e){
                        if (["INPUT", "SELECT", "TEXTAREA"].indexOf(e.target.tagName) == -1) {
                            var move = function(dx,dy) {
                                var scrollBar = scope.shakerTable.scroll;
                                var x = scrollBar.scrollLeft + dx;
                                var y = scrollBar.scrollTop + dy;
                                scrollBar.setScrollXY(x,y);
                            };

                            var smallJump = 20;
                            var bigJump = smallJump * 7;
                            switch(e.keyCode) {
                                case keyCodes.up:
                                    move(0, -smallJump);
                                    break;
                                case keyCodes.down:
                                    move(0, smallJump);
                                    break;
                                case keyCodes.left:
                                    move(-smallJump, 0);
                                    break;
                                case keyCodes.right:
                                    move(smallJump, 0);
                                    break;
                                case keyCodes.pagedown:
                                    move(0, bigJump);
                                    break;
                                case keyCodes.pageup:
                                    move(0, -bigJump);
                                    break;
                            }
                        }
                    });

                    scope.$on('scrollToColumn', function(e, columnName) {
                        var c = scope.tableModel.searchHeader(columnName)[0];
                        if (c >= 0) {
                            scope.shakerTable.goTo(undefined, c);
                        }
                    });

                    scope.$on('$destroy', function() {
                        $(window).off("keydown.fattable");
                    });

                }


                // binding cell click events.
                {
                    var $el = $(element);
                    var $currentSelectCell = null;
                    $el.off(".shakerTable");
                    var getRow = function($el) {
                        var rowId = $el[0].dataset.rowId;
                        return $el.siblings("[data-row-id='"+ rowId+ "']");
                    };

                    if (scope.shakerCellClickable) {
                        $el.on("click.shakerTable", ".fattable-body-container > div > div", function(evt) {
                            const rowIdx = parseInt(scope.shakerState.hoverredRow);
                            const colIdx = parseInt(scope.shakerState.hoverredCol);
                            scope.selectRow(rowIdx);
                            $rootScope.$broadcast('shakerCellClick', scope.getRowCells(rowIdx), colIdx, attrs.fatId);
                        });
                    }

                    $el.on("mousedown.shakerTable", ".fattable-body-container > div > div", function(evt) {
                        // Prevent a bit selection of more than one cell.
                        if ($currentSelectCell != null) {
                            $currentSelectCell.parent().find(".selectable").removeClass("selectable");
                            $currentSelectCell.parent().removeClass("inselection");
                        }
                        $currentSelectCell = $(getSelectedCell(scope.searchableDataset, $(evt.target)[0]));
                        $currentSelectCell.addClass("selectable");
                        $currentSelectCell.parent().addClass("inselection");
                    });

                    $el.on("mouseup.shakerTable", ".fattable-body-container > div", function(evt) {
                        if (evt.button != 1) {
                            if (!scope.isCellPopupVisible() ) {
                                var target = evt.target;
                                if ($currentSelectCell != null) {
                                    target = getSelectedCell(scope.searchableDataset, target);
                                    var row = target?.dataset?.rowId;
                                    var col = target?.dataset?.colId;
                                    if (row && col && $currentSelectCell[0] == target) {
                                        scope.showCellPopup(target, row, col, evt);
                                        // If the event bubbles up to body,
                                        // it will trigger hidePopup.
                                        evt.stopPropagation();
                                    }
                                }
                                $currentSelectCell = null;
                            }
                        }
                    });

                    const hoverRow = (target) => {
                        getRow(target).addClass('H');
                        scope.shakerState.hoverredRow = target[0].dataset.rowId;
                        scope.shakerState.hoverredCol = target[0].dataset.colId;
                    }

                    const blurRow = (target) => {
                        getRow(target).removeClass('H');
                        scope.shakerState.hoverredRow = null;
                        scope.shakerState.hoverredCol = null;
                    }

                    $el.on("mouseenter.shakerTable", ".fattable-body-container > div > div", function(evt) {
                        hoverRow($(evt.target));
                    });

                    $el.on("mouseleave.shakerTable", ".fattable-body-container > div > div", function(evt) {
                        blurRow($(evt.target));
                    });

                    $el.on("mouseenter.shakerTable", ".fattable-body-container  > div > div > a", function(evt) {
                        hoverRow($(evt.target).parent());
                    });

                    $el.on("mouseleave.shakerTable", ".fattable-body-container  > div > div > a", function(evt) {
                        blurRow($(evt.target).parent());
                    });
                }

                const CELL_PREVIEW_TYPE = {
                    CELL_VALUE_PREVIEW : "CELL_VALUE_PREVIEW",
                    GEO_PREVIEW : "GEO_PREVIEW",
                    IMAGE_PREVIEW : "IMAGE_PREVIEW"
                };

                // setuping the cell popup.
                const popupContent = $('<div><div class="popover-content shaker-cell-popover"></div></div>');
                $("body").append(popupContent);

                const cvPopupContent = $('<div><div class="popover-content"></div></div>');
                cvPopupContent.canClose = true;
                cvPopupContent.setPopupClosable = () => { cvPopupContent.canClose = true; };
                cvPopupContent.setPopupNotClosable = () => { cvPopupContent.canClose = false; };
                $("body").append(cvPopupContent);

                function showPopupUsing(template, customScope, cellPreviewType) {
                    var newDOMElt = $('<div>');
                    newDOMElt.html(template);
                    $timeout(function() {
                        $compile(newDOMElt)(customScope);
                        cvPopupContent.find(".popover-content").empty().append(newDOMElt);
                        $timeout(function() {
                            cvPopupContent.css("display", "block");
                            cvPopupContent[0].className = "shaker-cell-popover popover ";
                            switch(cellPreviewType) {
                                case CELL_PREVIEW_TYPE.GEO_PREVIEW:
                                    cvPopupContent[0].className += "shaker-cell-popover--geo-preview ";
                                    break;
                                case CELL_PREVIEW_TYPE.IMAGE_PREVIEW:
                                    cvPopupContent[0].className += "shaker-cell-popover--image-preview ";
                            }
                            cvPopupContent.on('click', function(e){
                                if(! $(e.target).closest('a,input,button,select,textarea').length){
                                    e.stopPropagation();
                                }
                            });
                        });
                    });
                }

                var $doc = $(document);

                scope.isCellPopupVisible = function() {
                    return popupContent.css("display") != "none";
                };

                scope.hidePopup = function(popup, className) {
                    if(popup.canClose === undefined || popup.canClose) {
                        var formerPopupScope = popup.find(`.${className} > div`).first().scope();
                        if ((formerPopupScope != undefined) && (formerPopupScope !== scope)) {
                            formerPopupScope.$destroy();
                        }
                        popup.css("display", "none");
                    }
                };

                scope.hidePopups = function(){
                    scope.hidePopup(popupContent, "popover-content");
                    scope.hidePopup(cvPopupContent, "popover-content")
                    $doc.unbind("click.shaker.cellPopup");
                };

                scope.isCellKO = function(cellStatus) {
                    if(cellStatus === null) return false; // in case of wrong status allow the option (column coloring for instance)
                    return cellStatus !== undefined
                        && (cellStatus.startsWith("I") || cellStatus.startsWith("E") || cellStatus.startsWith("F"));
                };

                function columnSupportsGeoPreview(columnLabel) {
                    const columnHeaders = scope.table.headers.filter(column => column.name === columnLabel);
                    if(columnHeaders.length > 1 || columnHeaders.length === 0) {
                        return false;
                    }

                    const columnHeader = columnHeaders[0];
                    if (!columnHeader.selectedType) return false;
                    const meaning = columnHeader.selectedType.name;

                    return meaning === "GeometryMeaning" || meaning === "GeoPoint";
                };

                function columnSupportsImagePreview(columnLabel) {
                    if (!scope.imageViewSettings || !scope.imageViewSettings.enabled) {
                        return false;
                    }
                    if (scope.imageViewSettings.pathColumn === columnLabel) {
                        return true;
                    }
                    return false;
                };

                function getPreviewType(columnName) {
                    if (columnSupportsGeoPreview(columnName)) {
                        return CELL_PREVIEW_TYPE.GEO_PREVIEW;
                    } else if (columnSupportsImagePreview(columnName)) {
                        return CELL_PREVIEW_TYPE.IMAGE_PREVIEW;
                    }
                    return CELL_PREVIEW_TYPE.CELL_VALUE_PREVIEW;
                }

                scope.columnSupportsSpecialPreview = function(columnName) {
                    return getPreviewType(columnName) !== CELL_PREVIEW_TYPE.CELL_VALUE_PREVIEW;
                }

                scope.toggleRowHighlight = function(rowIdx) {
                    var arr = scope.shakerState.lockedHighlighting;
                    if (arr.indexOf(rowIdx) >=0){
                        arr.splice(arr.indexOf(rowIdx), 1);
                    } else {
                        arr.push(rowIdx);
                    }
                    scope.shakerTable.refreshAllContent(true);
                }

                /**
                 * only one row can be selected at time, null is used to deselect the current row
                 * @param {*} rowIdx number | null
                 */
                scope.selectRow = function(rowIdx) {
                    scope.shakerState.selectedRow = rowIdx;
                    scope.shakerTable.refreshAllContent(true);
                }

                scope.changeSelectedRow = function(direction) {
                    let currentlySelectedRow = scope.shakerState.selectedRow;
                    if (!isNaN(currentlySelectedRow)) {
                      if (direction === "previous" && currentlySelectedRow > 0) {
                        currentlySelectedRow = currentlySelectedRow - 1;
                      } else if (direction === "next" && currentlySelectedRow < scope.shakerTable.nbRows - 1) {
                        currentlySelectedRow = currentlySelectedRow + 1;
                      }
                      // if we do not apply the whole table will rerender with a blink.
                      scope.$apply(function () {
                        scope.selectRow(currentlySelectedRow);
                      });
                      scope.scrollToLine(currentlySelectedRow);
                      $rootScope.$broadcast("compareCellValueNewRowSelected", scope.getRowCells(currentlySelectedRow), attrs.fatId);
                    }
                }

                scope.copyRowAsJSON = async function(rowIdx) {
                    function getColumnSchema(column) {
                        if (scope.shaker.origin === "DATASET_EXPLORE") {
                            return column.datasetSchemaColumn;
                        } else if (scope.shaker.origin === "PREPARE_RECIPE" && column.recipeSchemaColumn) {
                            return column.recipeSchemaColumn.column;
                        }
                    }

                    function getCellPromise(rowIdx, colIdx) {
                        return new Promise((resolve) => {
                            scope.tableModel.getCell(rowIdx, colIdx, resolve);
                        });
                    }

                    function smartCast(colType, colValue) {
                        switch (colType) {
                            case "tinyint":
                            case "smallint":
                            case "int":
                            case "bigint":
                                return Number.parseInt(colValue);
                            case "float":
                            case "double":
                                return Number.parseFloat(colValue);
                            default:
                                return colValue;
                            }
                    }



                    const colTypes = scope.table.headers.reduce((obj, column) => {
                        const colSchema = getColumnSchema(column);
                        obj[column.name] = colSchema ? colSchema.type : null;
                        return obj;
                      }, {});

                    const columnNames = scope.tableModel.allColumnNames;
                    const columnIndices = [...Array(columnNames.length).keys()];
                    const row = {};

                    await Promise.all(columnIndices.map(colIdx => getCellPromise(rowIdx, colIdx)))
                        .then((cells) => {
                            for (const [index, cell] of cells.entries()) {
                                const columnName = columnNames[index];
                                row[columnName] = smartCast(colTypes[columnName], cell.content);
                            }
                        });

                    ClipboardUtils.copyToClipboard(JSON.stringify(row, null, 2), `Row copied to clipboard.`);
                };

                scope.getRowCells = async function(rowIdx) {
                    function getCellPromise(rowIdx, colIdx) {
                        return new Promise((resolve) => {
                            scope.tableModel.getCell(rowIdx, colIdx, resolve);
                        });
                    }

                    const columnNames = scope.tableModel.allColumnNames;
                    const columnIndices = [...Array(columnNames.length).keys()];

                    return Promise.all(columnIndices.map(colIdx => getCellPromise(rowIdx, colIdx)));
                }

                if (!DashboardUtils.isInDashboard()) {
                    Mousetrap.bind("shift+h", function(){
                        var rowIdx = scope.shakerState.hoverredRow;
                        if (!rowIdx) return;
                        rowIdx = parseInt(rowIdx);
                        scope.$apply(function(){
                            scope.toggleRowHighlight(rowIdx);
                        });
                    }, 'keyup');
    
                     Mousetrap.bind("shift+v", function(){
                        if (!scope.shakerState.hoverredRow) return;
                        const rowIdx = parseInt(scope.shakerState.hoverredRow);
                        const colIdx = parseInt(scope.shakerState.hoverredCol);
                        scope.$apply(function(){
                            scope.hidePopups();
                            scope.showCVCellPopup(rowIdx, colIdx);
                        });
                    }, 'keyup');
    
                    Mousetrap.bind("shift+j", function(){
                        var rowIdx = scope.shakerState.hoverredRow;
                        if (!rowIdx) return;
                        rowIdx = parseInt(rowIdx);
                        scope.$apply(function(){
                            scope.copyRowAsJSON(rowIdx);
                        });
                    }, 'keyup');
    
                    Mousetrap.bind("shift+down", function (){
                        scope.changeSelectedRow('next');
                        return false;
                    });
    
                    Mousetrap.bind("shift+up", function (){
                        scope.changeSelectedRow('previous');
                        return false;
                    });
                }

                scope.$on('$destroy', function() {
                    scope.hidePopups();
                    $(document).off('.shaker');
                    popupContent.remove();
                    Mousetrap.unbind('c');
                    Mousetrap.unbind('shift+h', 'keyup');
                    Mousetrap.unbind('shift+v', 'keyup');
                    Mousetrap.unbind('shift+j', 'keyup');
                    Mousetrap.unbind('shift+down');
                    Mousetrap.unbind('shift+up');
                });

                scope.showCVCellPopup2 = function(cellValue, column, sanitizedHTMLContent, cellPreviewType, mousePosition) {
                    ContextualMenu.prototype.closeAny();

                    // no popup displayed for empty cells
                    if(!cellValue || !column) return;

                    // computing new custom scope
                    const newScope = scope.$new();
                    newScope.cellValue = cellValue;
                    const placePopup = function () {
                        $timeout(() => {
                            const placement = getPlacementForMouse(mousePosition, cvPopupContent, mousePosition.left, mousePosition.top);
                            cvPopupContent.css(placement.css);
                            cvPopupContent[0].className += placement.clazzes.join(" ");
                            });
                        };
                    cvPopupContent.onPopupCreated = placePopup;
                    newScope.popup = cvPopupContent;
                    placePopup(); // first placement to avoid lag
                    let template;
                    switch(cellPreviewType) {
                        case CELL_PREVIEW_TYPE.CELL_VALUE_PREVIEW : {
                            newScope.sanitizedHtmlContent = sanitizedHTMLContent;
                            newScope.column = column;
                            if (scope.searchableDataset) {
                                newScope.highlightTagName = scope.highlightTagName;
                                newScope.searchValue = (content) => {
                                    scope.searchInteractiveFilter(column.name, content);
                                };
                            }
                            template = `
                                <cell-value-popup-component 
                                    cell-value="cellValue"
                                    sanitized-html-content="sanitizedHtmlContent"
                                    search-value="searchValue"
                                    column-meaning="column.selectedType.name"
                                    popup-container="popup">
                                </cell-value-popup-component>`;
                                break;
                            }
                        case CELL_PREVIEW_TYPE.GEO_PREVIEW : {
                            template = `
                                <geo-preview-map-content 
                                    cell-value="cellValue" 
                                    popup-container="popup">
                                </geo-preview-map-content>`;
                                break;
                            }
                        case CELL_PREVIEW_TYPE.IMAGE_PREVIEW:
                            template = `
                              <image-preview-component
                                image-path="cellValue"
                                image-view-settings="imageViewSettings"
                                popup-container="popup">
                              </image-preview-component>`;
                            break;
                        default :
                            throw new Error(`Unknown value for CELL_PREVIEW_TYPE: ${cellPreviewType}`);
                        }
                        showPopupUsing(template, newScope, cellPreviewType);
                }

                scope.showCVCellPopup = function(row, col) {
                    scope.tableModel.getCell(row, col, function(cellData) {
                        const currentMousePosition = {
                            top: currentMousePos.y,
                            left: currentMousePos.x
                        }
                        const columnName = scope.table.headers[col].name;
                        const cellValue = cellData.content;
                        const cellPreviewType = scope.isCellKO(cellData.status) ? CELL_PREVIEW_TYPE.CELL_VALUE_PREVIEW : getPreviewType(columnName);
                        const hasHighlight = scope.shaker.coloring.highlightSearchMatches && cellData.highlightedIndices;
                        const sanitizedHTMLContent = hasHighlight ? insertHighlightTags(cellValue, cellData.highlightedIndices) : sanitize(cellValue);  // html content has already been sanitized in insertHighlightTags()
                        scope.showCVCellPopup2(cellValue, scope.table.headers[col], sanitizedHTMLContent, cellPreviewType, currentMousePosition);
                    });
                }



                scope.showCellPopup = function(elt, row, col, evt) {
                    const rowIndex = Number.parseInt(row);
                    const colIndex = Number.parseInt(col);
                    if (isNaN(rowIndex) || isNaN(colIndex)) {
                        Logger.warn("Invalid cell index while showing cell popup")
                        return;
                    }
                    ContextualMenu.prototype.closeAny();
                    scope.hidePopups();
                    // TODO eventually get rid of this.
                    // terrible monkey patching
                    {
                        var parent = popupContent.parent();
                        if (popupContent.parent().length == 0) {
                            // why is not under body anymore!?
                            $("body").append(popupContent);
                            WT1.event("shaker-cell-popup-content-disappear");
                            Logger.error("POPUP CONTENT DISAPPEARED. MONKEY PATCH AT WORK");
                        }
                    }
                    // end of terrible monkey...
                    // explicit activation of popup menu
                    // if neither shakerWritable nor shakerReadOnlyActions are set, do not show the popup
                    if (!scope.shakerWritable && !scope.shakerReadOnlyActions) return;
                    scope.tableModel.getCell(rowIndex, colIndex, function(cellData) {
                        var cellValue = cellData.content;
                        var req = {
                            cellValue: cellValue,
                            type: "cell",
                            row: rowIndex,
                            column: scope.table.headers[colIndex].name,
                            cellStatus: cellData.status
                        };
                        const currentMousePosition = {
                            top: currentMousePos.y,
                            left: currentMousePos.x
                        };
                        const hasHighlight = scope.shaker.coloring && scope.shaker.coloring.highlightSearchMatches && cellData.highlightedIndices;
                        elt = getSelectedCell(hasHighlight, elt);
                        let highlightTagName = null;
                        if (hasHighlight) {
                            highlightTagName = scope.highlightTagName;  // search highlight (ElasticSearch)
                        } else if (scope.shaker.coloring && scope.shaker.coloring.highlightWhitespaces) {
                            highlightTagName = "span"; // tag inserted to highlight whitespace: <span class="ms">...</span>
                        }
                        const selection = getSelectionInElement(elt, highlightTagName);
                        if (selection != null) {
                            req.type = "content";
                            req.content = selection.content;
                            req.startOffset = selection.startOffset;
                            req.endOffset = selection.endOffset;
                        }

                        var templateUrl = "/templates/shaker/suggestions-popup.html";
                        $q.when($templateCache.get(templateUrl) || $http.get(templateUrl, {
                            cache: true
                        })).then(function(template) {
                            if(angular.isArray(template)) {
                                template = template[1];
                            } else if(angular.isObject(template)) {
                                template = template.data;
                            }
                            var newDOMElt = $('<div>');
                            newDOMElt.html(template);
                            $timeout(function() {
                                var newScope = scope.$new();
                                newScope.req = req;
                                if(cellValue == null){
                                    cellValue = "";
                                }

                                const cellSuggestions = ShakerSuggestionsEngine.computeCellSuggestions(scope.table.headers[col], cellValue, cellData.status);
                                const contentSuggestions = newScope.shakerState.withSteps && newScope.shakerWritable && req.type == "content"
                                    ? ShakerSuggestionsEngine.computeContentSuggestions(scope.table.headers[col], cellValue, req.content, cellData.status, CreateModalFromTemplate, req.startOffset, req.endOffset)
                                    : {} ;

                                newScope.allSuggestions = [
                                    ...Object.entries(contentSuggestions),
                                    ...Object.entries(cellSuggestions)
                                ].map(([name, suggestions]) => ({name, suggestions}))
                                .sort((a, b) => {
                                    // Sort is stable (part of EMCA specs), which means content suggestions will stay on top unless otherwise specified even if using the same category
                                    return ShakerSuggestionsEngine.getCategoryPriority(b.name) - ShakerSuggestionsEngine.getCategoryPriority(a.name);
                                });

                                newScope.executeSuggestion = function(sugg) {
                                    sugg.action(scope);
                                };
                                newScope.showCellValue = function(){
                                    const sanitizedHTMLContent = hasHighlight ? elt.innerHTML : null;  // html content has already been sanitized in insertHighlightTags()
                                    scope.showCVCellPopup2(cellValue, scope.table.headers[col], sanitizedHTMLContent, CELL_PREVIEW_TYPE.CELL_VALUE_PREVIEW, currentMousePosition);
                                }
                                newScope.showSpecialPreview = function() {
                                    scope.showCVCellPopup2(cellValue, scope.table.headers[col], null, getPreviewType(scope.table.headers[col].name), currentMousePosition);
                                }
                                newScope.getPreviewIcon = function(columnName) {
                                    switch(getPreviewType(columnName)) {
                                        case CELL_PREVIEW_TYPE.GEO_PREVIEW:
                                            return "dku-icon-globe-16";
                                        case CELL_PREVIEW_TYPE.IMAGE_PREVIEW:
                                            return "dku-icon-image-16";
                                        default:
                                            return "";
                                    }
                                }

                                newScope.triggerCompareColumnValues = function(rowIdx) {
                                    scope.selectRow(rowIdx);
                                    $rootScope.$broadcast('triggerCompareColumnValues', scope.getRowCells(rowIdx), cellData.colId);
                                }

                                newScope.copyCellValue = function()  {
                                    ClipboardUtils.copyToClipboard(cellValue, `Copied to clipboard.`);
                                }
                                newScope.searchValue = function(content) {
                                    scope.searchInteractiveFilter(req.column, content);
                                }
                                newScope.getStepDescription = function(a,b) {
                                    return ShakerProcessorsUtils.getStepDescription(null, a,b);
                                };
                                newScope.filter = function(val, matchingMode) {
                                    if(!val) {
                                        val = '';
                                    }
                                    var v = {};
                                    v[val] = true;
                                    if (scope.table.headers[col].selectedType) {
                                        scope.addColumnFilter(scope.table.headers[col].name, v, matchingMode,
                                            scope.table.headers[col].selectedType.name, scope.table.headers[col].isDouble);
                                    }
                                };
                                newScope.enableFilter = !DashboardUtils.isInDashboard() && !scope.searchableDataset;
                                newScope.canCrossFilter = function(value) {
                                    if (DashboardUtils.isInDashboard() && DashboardFilters.canCrossFilter($stateParams.pageId)) {
                                        const isDouble = scope.table.headers[col].isDouble;
                                        const columnType = ['Date', 'DateOnly', 'DatetimeNoTz'].indexOf(scope.table.headers[col].selectedType.name) >= 0 ? 'DATE' : (isDouble ? 'NUMERICAL' : 'ALPHANUM');
                                        // In case of an empty cell in a date or numerical column we do not allow the include only
                                        // - On SQL, we currently do not support alphanum filter for numeric columns
                                        // - TODO For date, build a date part filter, filtering on 'No value'
                                        return !(value == null && (columnType === 'DATE' || columnType === 'NUMERICAL'));
                                    }
                                    return false;
                                };

                                function computeCrossFilterData(value) {
                                    const isDouble = scope.table.headers[col].isDouble;
                                    const columnType = ['Date', 'DateOnly', 'DatetimeNoTz'].indexOf(scope.table.headers[col].selectedType.name) >= 0 ? 'DATE' : (isDouble ? 'NUMERICAL' : 'ALPHANUM');
                                    let axisElt = { label: value };

                                    if (value == null) {
                                        axisElt = { label: DKU_NO_VALUE };
                                    } else if (columnType === 'DATE') {
                                        // The getTime method always uses UTC for time representation
                                        const timestamp = new Date(value).getTime();
                                        if (isNaN(timestamp)) {
                                            Logger.error("Invalid timestamp");
                                            return;
                                        }
                                        axisElt = { min: timestamp, max: timestamp, tsValue: timestamp, label: value };
                                    } else if (columnType === 'NUMERICAL') {
                                        const numericalValue = Number(value);
                                        axisElt = { min: numericalValue, max: numericalValue, sortValue: numericalValue, label: value };
                                    }
                                    const column = scope.table.headers[col].name;

                                    return {axisElt, column, columnType};
                                }

                                newScope.includeOnly = function(value) {
                                    const {axisElt, column, columnType} = computeCrossFilterData(value);
                                    const filter = ChartFilters.createFilter({
                                        column,
                                        columnType,
                                        isAGlobalFilter: true,
                                        excludeOtherValues: true,
                                        useMinimalUi: true,
                                        includeEmptyValues: false
                                    }, {
                                        axisElt
                                    });
                                    $rootScope.$emit('crossFiltersAdded', {
                                        filters: [filter],
                                        wt1Args: {
                                            from: 'dataset_table',
                                            action: 'contextual_menu',
                                            filterType: 'include'
                                        }
                                    });
                                };
                                newScope.exclude = function(value) {
                                    const {axisElt, column, columnType} = computeCrossFilterData(value);
                                    const filter = ChartFilters.getExcludeNDFilter([{
                                        dimension: {column, type: columnType },
                                        axisElt,
                                        value: axisElt.label
                                    }]);
                                    filter.isAGlobalFilter = true;
                                    $rootScope.$emit('crossFiltersAdded', {
                                        filters: [filter],
                                        wt1Args: {
                                            from: 'dataset_table',
                                            action: 'contextual_menu',
                                            filterType: 'exclude'
                                        }
                                    });
                                };

                                $compile(newDOMElt)(newScope);
                                popupContent.find(".popover-content").empty().append(newDOMElt);
                                $timeout(function() {
                                    var placement = getPlacement2(elt, popupContent, evt);
                                    newScope.popupPlacement = placement;

                                    popupContent.css("display", "block");
                                    popupContent.css(placement.css);
                                    var popupClassNames = "shaker-cell-popover popover ";
                                    popupClassNames += placement.clazzes.join(" ");
                                    popupContent[0].className = popupClassNames;
                                    popupContent.on('click', function(e){
                                        if(! $(e.target).closest('a,input,button,select,textarea').length){
                                            e.stopPropagation();
                                        }
                                    });
                                }, 0);
                            });
                        });
                    });
                };

                scope.shakerTable = null;

                var ratioX = 0;
                var ratioY = 0;
                scope.setNewTable = function(tableData) {
                    if (scope.shakerTable) {
                        if (scope.shakerTable.scroll) {
                            ratioX = scope.shakerTable.scroll.scrollLeft / scope.shakerTable.W;
                            ratioY = scope.shakerTable.scroll.scrollTop  / scope.shakerTable.H;
                        } else {
                            // we're in the middle of refreshing, so use the last saved values of ratioX and ratioY
                            // and anyway scrollLeft and scrollTop haven't been regenerated yet
                        }
                        scope.shakerTable.cleanUp();
                    } else {
                        ratioX = 0;
                        ratioY = 0;
                    }
                    if (scope.tableScope) {
                        scope.tableScope.$destroy();
                        scope.shakerTable.onScroll = null;
                    }
                    scope.tableScope = scope.$new();
                    let maxPageHeight = null;
                    let maxPageWidth = null;
                    if (scope.setPageHeightWidthFromChunk) {
                        maxPageHeight = tableData.initialChunk.nbRows;
                        maxPageWidth = tableData.initialChunk.nbCols;
                    }
                    scope.tableModel = ShakerTableModel(tableData, scope.tableScope, maxPageWidth, maxPageHeight);

                    // Absolute minimum for "Decimal (FR format)"
                    var minColumnWidth = 100;
                    var headerHeight = 63;
                    /* Space for schema */
                    if (!scope.shaker.$headerOptions) {
                        if (scope.shaker.origin == "PREPARE_RECIPE" || scope.shaker.origin == "DATASET_EXPLORE") {
                            headerHeight += 19;
                        }
                        if (scope.shakerState.hasAnyComment) {
                            headerHeight += 15;
                        }
                        if (scope.shakerState.hasAnyCustomFields) {
                            headerHeight += 19;
                        }
                    } else {
                        headerHeight += scope.shakerState.hasAnyComment ? -4 : 3;

                        if (!scope.shaker.$headerOptions.showName) {
                            headerHeight -= scope.shakerState.hasAnyComment ? 28 : 34;
                        }

                        if (!scope.shaker.$headerOptions.showStorageType) {
                            headerHeight -= 19;
                        }

                        if (scope.shaker.$headerOptions.showMeaning) {
                            headerHeight += 19;
                        }

                        if (scope.shakerState.hasAnyComment && scope.shaker.$headerOptions.showDescription) {
                            headerHeight += 19;
                        }

                        if (scope.shakerState.hasAnyCustomFields && scope.shaker.$headerOptions.showCustomFields) {
                            headerHeight += 19;
                        }

                        if (!scope.shaker.$headerOptions.showProgressBar) {
                            headerHeight -= 11;
                        }

                        var unwatch = scope.$watch("shaker.$headerOptions", function(nv, ov) {
                            if (!nv || nv == ov) return;
                            unwatch();
                            scope.setNewTable(tableData);
                        }, true);
                    }

                    var ROW_HEIGHT = 27;

                    scope.shaker.columnWidthsByName = scope.shaker.columnWidthsByName || {};
                    scope.shaker.columnUseScientificNotationByName = scope.shaker.columnUseScientificNotationByName || {};
                    [ tableData.columnWidthsByIndex, scope.shaker.columnWidthsByName ] = computeColumnWidths(scope.tableModel.initialChunk, scope.tableModel.headers, minColumnWidth, scope.hasAnyFilterOnColumn, scope.shaker.columnWidthsByName, scope.shaker.columnUseScientificNotationByName);

                    scope.shakerTable = fattable({
                        "container": element[0],
                        "model": scope.tableModel,
                        "nbRows": scope.tableModel.totalKeptRows,
                        "headerHeight": headerHeight,
                        "rowHeight":  ROW_HEIGHT,
                        "columnWidths": tableData.columnWidthsByIndex,
                        "painter": ShakerTablePainter(scope.tableScope),
                        "autoSetup": false
                    });

                    scope.shakerTable.onScroll = function(x,y) {
                        scope.hidePopups();
                    }

                    // we save the scroll state, as a ratio.
                    // we scroll back to the position we were at.
                    var newX = (scope.shakerTable.W *ratioX) | 0;
                    var newY = (scope.shakerTable.H *ratioY) | 0;


                    var leftTopCorner = scope.shakerTable.leftTopCornerFromXY(newX, newY);
                    var I = leftTopCorner[0];
                    var J = leftTopCorner[1];

                    var requested = 0;
                    var shakerTable = scope.shakerTable;
                    // A lib like async would have been nice here.
                    // only draw the table if all the
                    // pages are ready.
                    var everythingDone = function() {
                        if (requested == 0) {
                            // we check that the shaker has not
                            // been replaced.
                            if (shakerTable === scope.shakerTable) {
                                scope.shakerTable.setup();
                                scope.shakerTable.scroll.setScrollXY(newX, newY);
                                if (typeof scope.setDashboardTileLoaded === 'function') {
                                    scope.setDashboardTileLoaded();
                                }
                            }
                        }

                        if (isTouchDevice()) {
                            if (typeof(scope.unsetTouchable) === "function") {
                                scope.unsetTouchable();
                            }
                            scope.unsetTouchable = FatTouchableService.setTouchable(scope, element, scope.shakerTable);
                        }

                        if (attrs.fatDraggable !== undefined) {
                            scope.isDraggable = true;

                            // fatDraggable callback for placeholder shaping : use the whole table height instead of header only
                            scope.onPlaceholderUpdate = function(dimensions) {
                                let table = scope.shakerTable.container;
                                if (table) {
                                    dimensions.height = table.getBoundingClientRect().height;
                                }
                            };

                            FatDraggableService.setDraggable({
                                element: scope.shakerTable.container,
                                onDrop: scope.reorderColumnCallback,
                                onPlaceholderUpdate: scope.onPlaceholderUpdate,
                                scrollBar: scope.shakerTable.scroll,
                                classNamesToIgnore: ['icon-sort-by-attributes', 'sort-indication', 'filter-indication', 'pull-right', 'fat-resizable__handler']
                            })
                        }

                        if (attrs.fatResizable !== undefined) {
                            scope.isResizable = true;

                            let table = scope.shakerTable.container;
                            let barHeight;
                            if (table) {
                                barHeight = table.getBoundingClientRect().height;
                            }

                            FatResizableService.setResizable({
                                element: scope.shakerTable.container,
                                barHeight: barHeight,
                                onDrop: function(resizeData) {
                                    tableData.columnWidthsByIndex[resizeData.index] = resizeData.width;
                                    scope.shakerHooks.updateColumnWidth(resizeData.name, Math.round(resizeData.width));
                                }
                            })
                        }
                    }
                    for (var i=I; i<I+scope.shakerTable.nbRowsVisible; i+=scope.tableModel.PAGE_HEIGHT) {
                        for (var j=J; j<J+scope.shakerTable.nbColsVisible; j+=scope.tableModel.PAGE_WIDTH){
                            if (!scope.tableModel.hasCell(i,j)) {
                                requested += 1;
                                scope.tableModel.getCell(i,j, function() {
                                    requested -= 1;
                                    everythingDone();
                                });
                            }
                        }
                    }
                    everythingDone();
                }

                // we only resize at the end of the resizing.
                // == when the user has been idle for 200ms.
                var formerScrollLeft = 0;
                var formerScrollTop = 0;
                var debouncedResizingHandler = Debounce().withScope(scope).withDelay(200,200).wrap(function() {
                    if (scope.shakerTable !== null) {
                        // check whether we really need to resize the
                        // the table. See #1851
                        var widthChanged = (scope.shakerTable.w != scope.shakerTable.container.offsetWidth);
                        var heightChanged = (scope.shakerTable.h != scope.shakerTable.container.offsetHeight - scope.shakerTable.headerHeight);
                        if (widthChanged || heightChanged) {
                            scope.shakerTable.setup();
                            scope.shakerTable.scroll.setScrollXY(formerScrollLeft, formerScrollTop);
                        }
                    }
                });
                var wrappedDebouncedResizingHandler = function() {
                    if (scope.shakerTable && scope.shakerTable.scroll) {
                        var scrollBar = scope.shakerTable.scroll;
                        formerScrollLeft = scrollBar.scrollLeft;
                        formerScrollTop = scrollBar.scrollTop;
                    } else {
                        // a table is being refreshed, keep the last known values of the scroll position
                    }
                    debouncedResizingHandler();
                };

                scope.$on('scrollToLine', function(e, lineNum) {
                    scope.scrollToLine(lineNum);
                });

                scope.scrollToLine = function(lineNum){
                    var table = scope.shakerTable;
                    if (table && table.scroll) {
                        var nbRowsVisible = table.h / table.rowHeight; // we need the float value
                        var firstVisibleRow = table.scroll.scrollTop / table.rowHeight; // we need the float value
                        var x = table.scroll.scrollLeft;
                        if (lineNum == -1) {
                            let y = table.nbRows * table.rowHeight;
                            table.scroll.setScrollXY(x, y);
                        } else if (lineNum <= firstVisibleRow) {
                            let y = Math.max(lineNum, 0) * table.rowHeight;
                            table.scroll.setScrollXY(x,y);
                        } else if (lineNum >= firstVisibleRow + nbRowsVisible - 1) {
                            let y = (Math.min(lineNum, table.nbRows) + 1) * table.rowHeight - table.h;
                            table.scroll.setScrollXY(x,y);
                        }
                    }
                }

                scope.$on("compareColumnValuesChangeRow", function (_event, direction, compareColumnId) {
                    if (compareColumnId && compareColumnId !== attrs.fatId) {
                        return; // This event is no destined to us
                    }
                    scope.changeSelectedRow(direction)
                });

                scope.$on('reflow',wrappedDebouncedResizingHandler);
                $(window).on("resize.shakerTable",wrappedDebouncedResizingHandler);
                scope.$on('resize', wrappedDebouncedResizingHandler);
                scope.$on('visualIfResize_', wrappedDebouncedResizingHandler);
                scope.$on("compareColumnValuesViewResize", (event, payload) => {
                    if (payload === "close") {
                        scope.selectRow(null);
                    }
                    wrappedDebouncedResizingHandler();
                });

                $doc.bind("click.shakerTable", scope.hidePopups);
                scope.$on("$destroy", function() {
                    scope.$broadcast("shakerIsGettingDestroyed");
                    $(window).off('.shakerTable');
                    $doc.off('.shakerTable');
                    if (scope.shakerTable) scope.shakerTable.cleanUp();
                    if (scope.tableScope) {
                        scope.tableScope.$destroy();
                    }
                    /* I'm not 100% clear on why we need this but experimentally,
                     * this helps avoid some leaks ... */
                    scope.shakerTable = undefined;
                    scope.tableModel = undefined;
                });

                scope.$on("forcedShakerTableResizing", wrappedDebouncedResizingHandler);

                scope.$watch("shaker.coloring.highlightWhitespaces", function(nv){
                    if (nv === undefined) return;
                    const tableData = scope.$eval(tableDataExpr);
                    if (tableData) {
                        scope.setNewTable(tableData);
                    }
                });

                scope.$watch(tableDataExpr, function(tableData) {

                    var curScope = undefined;
                    scope.hidePopups();
                    if (tableData) {
                        scope.setNewTable(tableData);
                    }
                });

            }
        }
    });

    app.directive('columnHeader', function($controller, CreateModalFromDOMElement, CreateModalFromTemplate, ContextualMenu, $state, DataikuAPI, WT1,
                                           ShakerSuggestionsEngine, $window, translate, $stateParams, findAppliedColoringGroup, ConditionalFormattingEditorService) {
        return {
            restrict: 'A',
            replace: false,
            templateUrl: '/templates/shaker/column_header.html',
            scope: true,
            link: function(scope, element, attrs) {

                scope.$on("shakerIsGettingDestroyed", function(){
                    /* Since fattable does not use jQuery to remove its elements,
                     * we need to use jQuery ourselves to remove our children (and
                     * ourselves).
                     * Doing that will ensure that the jQuery data cache is cleared
                     * (it's only cleared when it's jQuery that removes the element)
                     * Without that, since Angular has used jQuery().data() to retrieve
                     * some stuff in the element, the jQuery data cache will always
                     * contain the scope and ultimately the element, leading to massive
                     * DOM leaks
                     */
                    element.empty();
                    element.remove();
                })

                scope.storageTypes = [
                    ['string', 'String'],
                    ['int', 'Integer'],
                    ['double', 'Double'],
                    ['float', 'Float'],
                    ['tinyint', 'Tiny int (8 bits)'],
                    ['smallint', 'Small int (16 bits)'],
                    ['bigint', 'Big int (64 bits)'],
                    ['boolean', 'Boolean'],
                    ['date', 'Datetime with tz'],
                    ['dateonly', 'Date only'],
                    ['datetimenotz', 'Datetime no tz'],
                    ['geopoint', "Geo Point"],
                    ['geometry', "Geometry / Geography"],
                    ['array', "Array"],
                    ['object', "Complex object"],
                    ['map', "Map"]
                ];

                // We avoid using a simple bootstrap dropdown
                // because we want to avoid having the hidden menus
                // DOM polluting our DOM tree.

                scope.anyMenuShown = false;

                scope.menusState = {
                    name: false,
                    meaning: false,
                    type : false,
                    color : false
                }

                scope.menu = new ContextualMenu({
                    template: "/templates/shaker/column-header-contextual-menu.html",
                    cssClass : "column-header-dropdown-menu",
                    scope: scope,
                    contextual: false,
                     onOpen: function() {
                        scope.menusState.name = true;
                    },
                    onClose: function() {
                        scope.menusState.name = false;
                    },
                    enableClick: true
                });

                scope.meaningMenu = new ContextualMenu({
                    template: "/templates/shaker/edit-meaning-contextual-menu.html",
                    cssClass : "column-header-meanings-menu",
                    scope: scope,
                    contextual: false,
                     onOpen: function() {
                        scope.menusState.meaning = true;
                    },
                    onClose: function() {
                        scope.menusState.meaning = false;
                    }
                });

                scope.datasetStorageTypeMenu = new ContextualMenu({
                    template: "/templates/shaker/edit-storagetype-contextual-menu.html",
                    cssClass : "column-header-types-menu",
                    scope: scope,
                    contextual: false,
                     onOpen: function() {
                        scope.menusState.type = true;
                    },
                    onClose: function() {
                        scope.menusState.type = false;
                    }
                });
                scope.colorMenu = new ContextualMenu({
                    template: "/templates/shaker/column-num-color-contextual-menu.html",
                    cssClass : "column-colors-menu",
                    scope: scope,
                    contextual: false,
                     onOpen: function() {
                        scope.menusState.color = true;
                    },
                    onClose: function() {
                        scope.menusState.color = false;
                    }
                });

                scope.toggleHeaderMenu = function() {

                    if (!scope.menusState.name) {
                         element.parent().append(element); //< do not remove this!
                        // It puts the element at the end, and put the menu
                        // over the siblings
                        // The former z-index machinery is broken by the use of css transform.
                        scope.menu.openAlignedWithElement(element.find(".name"), function() {}, true, true);
                    } else {
                        scope.menu.closeAny();
                    }
                };

                scope.toggleMeaningMenu = function() {
                    if (!scope.menusState.meaning) {
                        element.parent().append(element); //< do not remove this!
                        scope.meaningMenu.openAlignedWithElement(element.find(".meaning"), function() {}, true, true);
                    } else {
                        scope.meaningMenu.closeAny();
                    }
                };
                scope.toggleStorageTypeMenu = function() {
                    if (!scope.menusState.type) {
                        element.parent().append(element); //< do not remove this!
                        scope.datasetStorageTypeMenu.openAlignedWithElement(element.find(".storage-type"), function() {}, true, true);
                    } else {
                        scope.datasetStorageTypeMenu.closeAny();
                    }
                };
                 scope.toggleColorMenu = function() {
                    if (!scope.menusState.color) {
                        element.parent().append(element); //< do not remove this!
                        scope.colorMenu.openAlignedWithElement(element.find(".progress:visible"), function() {}, true, true);
                    } else {
                        scope.colorMenu.closeAny();
                    }
                };

                /**
                 * Returns the rules (filter descriptions) targeting a column. If none are found, returns null.
                 */
                function getRulesGroupFilterDescs (columnName) {
                    const coloring = scope.shaker.coloring;

                    if (coloring.scheme === "COLORING_GROUPS") {
                        const coloringGroup = findAppliedColoringGroup(coloring.coloringGroups, columnName);
                        if (coloringGroup && coloringGroup.scheme === "RULES") {
                            if (coloringGroup.scope === "COLUMNS" || coloringGroup.scope === "ALL_COLUMNS_BASED_ON_ANOTHER_COLUMN") {
                                return coloringGroup.rulesGroup.filterDescs;
                            }
                        }
                    }
                    // Legacy
                    else if (coloring.scheme === "INDIVIDUAL_COLUMNS_RULES" || coloring.scheme === "SINGLE_COLUMN_RULES") {
                        const columnRules = angular.copy(coloring.individualColumnsRules.filter(columnRule => columnRule.column === columnName));
                        // Takes the first one, and it is the list of FilterDesc `rulesDesc` that is returned as the "column rules"
                        if (columnRules && columnRules[0] && columnRules[0].rulesDesc) {
                            return columnRules[0].rulesDesc;
                        }
                    }

                    return null;
                };

                scope.ruleEnabledWithColorInfo = function(rules, ruleIndex) {
                    return rules && ruleIndex !== -1
                        && ruleIndex < rules.length
                        && rules[ruleIndex]
                        && rules[ruleIndex].enabled === true
                        && rules[ruleIndex].uiData
                        && rules[ruleIndex].uiData.color
                        && rules[ruleIndex].uiData.color.styleClass;
                }

                const isDefaultFontColor = function(color) {
                    return color === "#000000" || color === "#FFFFFF";
                }

                const isDefaultBackgroundColor = function(color) {
                    return color !== undefined && (color === "#F2F2F2" || color === "rgb(242,242,242)");
                }

                const isNoneColor = function(color) {
                    return color === "";
                }

                /**
                 * Computes the style for a header progress bar chunk of a column on which conditional formatting rules are applied
                 * for a specific rule.
                 * If the rule is not defined with a custom color, returns an empty style.
                 */
                scope.getProgressBarChunkColorStyle = function (columnName, ruleIndex) {
                    const rules = getRulesGroupFilterDescs(columnName);

                    if (
                        scope.ruleEnabledWithColorInfo(rules, ruleIndex) &&
                        "shaker-color-rule--custom" === rules[ruleIndex].uiData.color.styleClass
                    ) {
                        const color = rules[ruleIndex].uiData.color;

                        if (isDefaultFontColor(color.styleCustomFontColor) || isNoneColor(color.styleCustomFontColor)) {
                            if (isDefaultBackgroundColor(color.styleCustomBackgroundColor) || isNoneColor(color.styleCustomBackgroundColor)) {
                                return {
                                    "background-color": "none",
                                };
                            }

                            // Only displays the non default/none-color (background) color
                            return {
                                "background-color": color.styleCustomBackgroundColor,
                            };
                        } else if (isDefaultBackgroundColor(color.styleCustomBackgroundColor) || isNoneColor(color.styleCustomBackgroundColor)) {
                            // Only displays the non default/none-color (font) color
                            return { "background-color": color.styleCustomFontColor };
                        } else {
                            // Displays the background (top) and font (bottom) colors
                            return {
                                "background-color": color.styleCustomBackgroundColor,
                                "border-bottom": "5px solid " + color.styleCustomFontColor,
                            };
                        }
                    }

                    return {};
                };

                function groupBasedOnAnotherColumn(group, columnName) {
                    return (
                        group.scope === "ALL_COLUMNS_BASED_ON_ANOTHER_COLUMN" && group.basedOnColumnName === columnName
                    );
                }

                function extractBin(cls) {
                    if (binCache.has(cls)) {
                        // Check if bin is already in cache
                        return binCache.get(cls);
                    }
                    // Compute the bin and store it in cache
                    // Matches the possible bins. Bins are defined in TableColoringService.getColorBin
                    const matches = cls.match(/-?\d+/g);
                    if (!matches) return null;
                    let lastMatch = matches[matches.length - 1];

                    if (!cls.includes("--")) {
                        const parts = lastMatch.split("-"); // Splits around '-'
                        lastMatch = parts[parts.length - 1]; // Takes the part after the last '-'
                    }
                    binCache.set(cls, parseInt(lastMatch, 10));
                    return lastMatch;
                }

                scope.hasLegacyColorScheme = function () {
                    return scope.shaker.coloring.scheme !== "COLORING_GROUPS";
                };

                /**
                 * Computes the style for a header progress bar chunk of a column on which conditional formatting scale is applied
                 * If the scale is not defined (should never happen), returns an empty style.
                 */
                scope.getProgressBarChunkColorStyleForScale = function (columnName, cls) {
                    if (scope.hasLegacyColorScheme()) {
                        if (cls.indexOf("CN") !== -1) {
                            const bin = extractBin(cls);
                            const scalePalette = ConditionalFormattingEditorService.getScalePaletteForNumber();
                            if (0 < bin && bin < scalePalette.length) {
                                return {
                                    background:
                                        "linear-gradient(to right," +
                                        scalePalette[bin - 1] +
                                        "," +
                                        scalePalette[bin] +
                                        ")",
                                };
                            }
                        }
                        return "";
                    }
                    const coloringGroup = scope.shaker.coloring.coloringGroups.findLast(
                        group =>
                            group.enabled &&
                            (group.targetedColumnNames.includes(columnName) ||
                                groupBasedOnAnotherColumn(group, columnName))
                    );

                    if (
                        coloringGroup &&
                        coloringGroup.scheme === "COLOR_SCALE" &&
                        coloringGroup.colorScaleDef &&
                        coloringGroup.colorScaleDef.sample
                    ) {
                        const bin = extractBin(cls);
                        if (0 < bin && bin <= coloringGroup.colorScaleDef.sample.length) {
                            if (
                                cls.indexOf("CN") !== -1 &&
                                bin < coloringGroup.colorScaleDef.sample.length
                            ) {
                                return {
                                    background:
                                        "linear-gradient(to right," +
                                        coloringGroup.colorScaleDef.sample[bin - 1] +
                                        "," +
                                        coloringGroup.colorScaleDef.sample[bin] +
                                        ")",
                                };
                            } else {
                                return { "background-color": coloringGroup.colorScaleDef.sample[bin - 1] };
                            }
                        }
                    }
                    return { "background-color": "none" };
                };

                /**
                 * Computes the class name for a header progress bar chunk of a column on which conditional formatting rules are applied
                 * for a specific rule.
                 */
                scope.getProgressBarChunkColorClass = function (columnName, ruleIndex) {
                    const rules = getRulesGroupFilterDescs(columnName);

                    if (scope.ruleEnabledWithColorInfo(rules, ruleIndex)) {
                        // Non-custom color
                        if (
                            rules[ruleIndex].uiData.color.styleClass.includes("text") ||
                            rules[ruleIndex].uiData.color.styleClass.includes("background")
                        ) {
                            return rules[ruleIndex].uiData.color.styleClass + "__progress-bar";
                        }

                        // Custom color
                        return rules[ruleIndex].uiData.color.styleClass;
                    }

                    return "";
                };

                scope.column = scope.header;
                scope.columnIndex = scope.columns.indexOf(scope.column.name);

                scope.isType = function(x) {
                    return this.column.selectedType && this.column.selectedType.name == x;
                };

                scope.possibleMeanings = $.map(scope.column.possibleTypes, function(t) {
                    return t.name;
                });


                    // scope.unprobableTypes = [];
                    // for (var tIdx in scope.types) {
                    //     if ($.inArray(scope.types[tIdx], scope.possibleTypes) == -1) {
                    //         scope.unprobableTypes.push(scope.types[tIdx]);
                    //     }
                    // }

                    // Column have changed, need to update layout -
                    // We only do it for the last column of the layout
                    if (scope.header && scope.table && scope.table.headers && scope.table.headers.length &&
                        scope.column  === scope.table.headers[scope.table.headers.length - 1]) {
                        scope.$emit('updateFixedTableColumns');
                    }
                // });

                if (scope.shakerWritable) {
                    var s = ShakerSuggestionsEngine.computeColumnSuggestions(scope.column, CreateModalFromDOMElement, CreateModalFromTemplate,
                                undefined, undefined, scope.appConfig);
                    scope.suggestions = s[0];
                    scope.moreSuggestions = s[1];
                } else {
                    scope.suggestions = [];
                }

                if (scope.isRecipe){
                    scope.setStorageType = function(newType) {
                        scope.recipeOutputSchema.columns[scope.column.name].column.type = newType;
                        scope.recipeOutputSchema.columns[scope.column.name].persistent = true;
                        scope.schemaDirtiness.dirty = true;
                    };
                }

                scope.executeSuggestion = function(sugg) {
                    sugg.action(scope);
                };
                scope.hasSuggestions = function() {
                    return Object.keys(scope.suggestions).length > 0;
                };
                scope.hasMoreSuggestions = function() {
                    return Object.keys(scope.moreSuggestions).length > 0;
                };

                scope.hasInvalidData = function() {
                    return scope.column.selectedType && scope.column.selectedType.nbNOK > 0;
                };
                scope.hasEmptyData = function() {
                    return scope.column.selectedType && scope.column.selectedType.nbEmpty > 0;
                };

                scope.setColumnMeaning = function(newMeaning) {
                    scope.shakerHooks.setColumnMeaning(scope.column, newMeaning);
                };

                scope.editColumnUDM = function(){
                    CreateModalFromTemplate("/templates/meanings/column-edit-udm.html", scope, null, function(newScope){
                        newScope.initModal(scope.column.name, scope.setColumnMeaning);
                    });
                }

                scope.setColumnStorageType = function(newType){
                    var schemaColumn = null;

                    if (scope.shaker.origin == "DATASET_EXPLORE") {
                        schemaColumn = scope.column.datasetSchemaColumn;
                    } else if (scope.shaker.origin == "PREPARE_RECIPE") {
                        if (scope.column.recipeSchemaColumn) {
                            schemaColumn = scope.column.recipeSchemaColumn.column;
                        } else {
                            return; // ghost column, added by a stray filter for ex
                        }
                    } else {
                        throw Error("Can't set storage type here origin=" + scope.shaker.origin);
                    }
                    var impact = scope.shakerHooks.getSetColumnStorageTypeImpact(scope.column, newType);
                    if (impact != null) {
                        var doSetStorageType = function(data) {
                            if (data.justDoIt) {
                                scope.shakerHooks.setColumnStorageType(scope.column, newType, null);
                            } else {
                                CreateModalFromTemplate("/templates/shaker/storage-type-change-warning-modal.html", scope, null, function(newScope){
                                    newScope.ok = function() {
                                        newScope.dismiss();
                                        scope.shakerHooks.setColumnStorageType(scope.column, newType, newScope.extraActions.filter(function(a) {return a.selected;}).map(function(a) {return a.id;}));
                                    };
                                    newScope.warnings = data.warnings;
                                    newScope.extraActions = data.extraActions;
                                });
                            }
                        };
                        if (impact.success) {
                            impact.success(doSetStorageType).error(setErrorInScope.bind(scope));
                        } else {
                            impact.then(doSetStorageType);
                        }
                    }
                }
                scope.editThisColumnDetails = function() {
                    var schemaColumn = null;

                    if (scope.shaker.origin == "DATASET_EXPLORE") {
                        schemaColumn = scope.column.datasetSchemaColumn;
                    } else if (scope.shaker.origin == "PREPARE_RECIPE") {
                        if (scope.column.recipeSchemaColumn) {
                            schemaColumn = scope.column.recipeSchemaColumn.column;
                        } else {
                            return; // ghost column, added by a stray filter for ex
                        }
                    } else {
                        schemaColumn = angular.extend({}, scope.shaker.analysisColumnData[scope.column.name], {name: scope.column.name});
                        if (!schemaColumn) {
                            schemaColumn = {name: scope.column.name}
                            scope.shaker.analysisColumnData[scope.column.name] = schemaColumn;
                        }
                    }
                    scope.editColumnDetails(schemaColumn);
                }

                scope.goToImageView = function() {
                    scope.shakerState.activeView = 'images';
                };

                scope.isImageViewSpecialColumn = function() {
                    if (!scope.imageViewSettings || !scope.imageViewSettings.enabled) {
                        return false;
                    }

                    if (scope.imageViewSettings.pathColumn === scope.header.name) {
                        return true;
                    }

                    if (scope.imageViewSettings.annotationParams && scope.imageViewSettings.annotationParams.enabled && scope.imageViewSettings.annotationParams.annotationColumn === scope.header.name) {
                        return true;
                    }

                    return false;
                };

                scope.getModelType = function() {
                    if (scope.imageViewSettings && scope.imageViewSettings.enabled && scope.imageViewSettings.annotationParams && scope.imageViewSettings.annotationParams.enabled && scope.imageViewSettings.annotationParams.annotationColumn === scope.header.name) {
                        return scope.imageViewSettings.annotationParams.annotationType;
                    }
                    if (['Date', 'DateOnly', 'DatetimeNoTz'].indexOf(scope.header.selectedType.name) >= 0) {
                        return 'Temporal';
                    }
                    return 'Default';
                };


                scope.setFilterEmpty = function() {
                    if (!scope.column.selectedType) return;
                    scope.addValidityFilter(scope.column.name, scope.column.selectedType.name, "empty");
                };
                scope.setFilterOK = function() {
                    if (!scope.column.selectedType) return;
                    scope.addValidityFilter(scope.column.name, scope.column.selectedType.name, "ok");
                };
                scope.setFilterNOK = function() {
                    if (!scope.column.selectedType) return;
                    scope.addValidityFilter(scope.column.name, scope.column.selectedType.name, "nok");
                };

                scope.createColumnFilter = function() {
                    if (!scope.column.selectedType) return;
                    scope.addColumnFilter(scope.column.name, {}, 'full_string', scope.column.selectedType.name, scope.column.isDouble);
                };

                scope.deleteColumn = function() {
                    scope.addStepNoPreview("ColumnsSelector", {
                        "keep": false,
                        "appliesTo": "SINGLE_COLUMN",
                        "columns": [ scope.column.name ]
                    });
                    scope.mergeLastColumnDeleters();
                    scope.autoSaveForceRefresh();
                };

                scope.renameColumn = function() {
                    CreateModalFromDOMElement("#rename-column-box", scope, "RenameColumnController", function(newScope) {
                        newScope.setColumn(scope.column.name);
                    });
                };

                scope.datasetInsightLoaded = false;
                scope.callbackDatasetLoaded = function() {
                    if (typeof scope.setDashboardTileLoaded === 'function') {
                        scope.setDashboardTileLoaded();
                    }
                    scope.datasetInsightLoaded = true;
                }

                scope.moveColumn = function() {
                    CreateModalFromDOMElement("#move-column-box", scope, "MoveColumnController", function(newScope) {
                        newScope.setColumn(scope.column.name);
                    });
                };



                scope.createPredictionModelOnColumn = function(column, datasetName, mode = "auto", predictionType = null) {
                    if (scope.analysisCoreParams){ // In an analysis, we do not create a new analysis to create the ML task
                        $controller('AnalysisNewMLTaskController', { $scope: scope });
                    } else { // otherwise we create a new analysis
                        $controller('DatasetLabController', { $scope: scope});
                    }

                    if(mode === "deephub-computer-vision") {
                        scope.newPrediction(column, datasetName, mode, predictionType, scope.imageViewSettings.managedFolderSmartId);
                    } else {
                        scope.newPrediction(column, datasetName, mode, predictionType);
                    }
                };

                scope.inWorkspace = function() {
                    return $stateParams.workspaceKey !== undefined;
                }

                const getOutputDatasetOrSmartName = (scope, datasetSmartName) => {
                    if (scope && scope.recipe && scope.recipe.outputs && scope.recipe.outputs.main &&
                        Array.isArray(scope.recipe.outputs.main.items) && scope.recipe.outputs.main.items.length > 0) {
                        
                        return scope.recipe.outputs.main.items[0].ref;
                    }
                    return datasetSmartName;
                }

                scope.buildDataLineageRoute = function(column, datasetSmartName) {
                    // Use output dataset for prepare recipes
                    return $state.href('datalineage.graph', {contextProjectKey: $stateParams.projectKey, smartName: getOutputDatasetOrSmartName(scope, datasetSmartName), columnName: column});
                };

                scope.selectedShakerOrigin = function(shakerOrigin) {
                    switch (shakerOrigin) {
                        case "DATASET_EXPLORE":
                            return "dataset"
                        case "PREPARE_RECIPE":
                            return "prepare-recipe"
                        case "ANALYSIS":
                            return "analysis"
                        default:
                            return undefined
                    }
                }

                scope.buildDataLineageRouteWT1Event = function(datasetSmartName) {
                    WT1.tryEvent("data-lineage-access", () => ({
                        from: `${scope.selectedShakerOrigin(scope.shaker.origin)}-column-header-menu`,
                        dataseth: md5($stateParams.projectKey + "." + getOutputDatasetOrSmartName(scope, datasetSmartName))
                    }));
                };

                // For interactive search filter menu
                let isInteractiveSearchFilterMenuVisible = false;
                scope.interactiveSearchFilterMenu = new ContextualMenu({
                    template: "/templates/shaker/column-interactive-search-filter-panel.html",
                    cssClass : "ff-contextual-menu",
                    scope: scope,
                    contextual: false,
                    handleKeyboard: false,
                    onOpen: function() {
                        isInteractiveSearchFilterMenuVisible = true;
                    },
                    onClose: function() {
                        isInteractiveSearchFilterMenuVisible = false;
                    },
                    enableClick: true
                });

                scope.showInteractiveSearchFilterMenu = function() {
                    let openAtX = $(element).offset().left;
                    if (openAtX > $($window).width()/2) {
                        openAtX += $(element).outerWidth();
                    }
                    scope.interactiveSearchFilterMenu.openAtXY(openAtX, $(element).offset().top + $(element).height(), function() {}, false, true); // NOSONAR: OK to have empty method
                };

                scope.toggleInteractiveSearchFilterMenu = function() {
                    if (scope.isInteractiveSearchFilterMenuVisible) {
                        scope.hideInteractiveSearchFilterMenu();
                    } else {
                        scope.showInteractiveSearchFilterMenu();
                    }
                };

                scope.hideInteractiveSearchFilterMenu = function() {
                    scope.interactiveSearchFilterMenu.closeAny();
                };
            }
        };
    });

app.controller("InteractiveSearchFilterPanelController", function($scope, DateUtilsService) {
    $scope.facetUiState = {timezoneDateRangeModel: "UTC", textFilters: []};

    $scope.submitDateRange = function(columnName) {
        const from = $scope.facetUiState.fromDateRangeModel;
        const to = $scope.facetUiState.toDateRangeModel;
        const tz = $scope.facetUiState.timezoneDateRangeModel;

        const fromDate = from != null ? DateUtilsService.convertDateFromTimezone(from, tz) : null;
        const toDate = to != null ? DateUtilsService.convertDateFromTimezone(to, tz) : null;
        $scope.addDateRangeToInteractiveFilter(columnName, fromDate, toDate);
        $scope.hideInteractiveSearchFilterMenu();
    }

    $scope.submitTextFilter = function(columnName) {
        $scope.appendTextValuesToInteractiveFilter(columnName, $scope.facetUiState.textFilters);
        $scope.hideInteractiveSearchFilterMenu();
    }

    $scope.isValidDateRange = function() {
        if (!$scope.facetUiState.toDateRangeModel && !$scope.facetUiState.fromDateRangeModel) {
            return false;
        }
        if ($scope.facetUiState.toDateRangeModel && $scope.facetUiState.fromDateRangeModel && $scope.facetUiState.fromDateRangeModel > $scope.facetUiState.toDateRangeModel) {
            return false;
        }

        return true;
    }
});

app.controller("ShakerEditColumnDetailsController", function($scope, $controller, DataikuAPI, $state, Debounce, $stateParams, categoricalPalette, ContextualMenu, CreateModalFromTemplate){
    $scope.column = null;

    $scope.uiState = {};

    $scope.setColumn = function(column) {
        $scope.column = column;
    }

    $scope.save = function() {
        if ($scope.column.customFields && Object.keys($scope.column.customFields).length == 0) {
            delete $scope.column.customFields;
        }
        const prevColumnComment = $scope.dataset?.schema?.columns?.find(item => item.name === $scope.column.name)?.comment || null;
        if (areStringsMeaningfullyDifferent(prevColumnComment, $scope.column.comment)) {
            $scope.column.isColumnEdited = true;
        }
        $scope.shakerHooks.updateColumnDetails($scope.column);
        $scope.dismiss();
    };


    $scope.openMeaningMenu = function($event, column) {
            $scope.meaningMenu.openAtXY($event.pageX, $event.pageY);
            $scope.meaningColumn = column;
    };

    $scope.setColumnMeaning = function(meaningId) {
        $scope.meaningColumn.meaning = meaningId;
        $(".code-edit-schema-box").css("display", "block");
    };

    $scope.editColumnUDM = function() {
        CreateModalFromTemplate("/templates/meanings/column-edit-udm.html", $scope, null, function(newScope) {
            newScope.initModal($scope.meaningColumn.name, $scope.setColumnMeaning);
        });
    };

    $scope.meaningMenu = new ContextualMenu({
        template: "/templates/shaker/edit-meaning-contextual-menu.html",
        cssClass : "column-header-meanings-menu pull-right",
        scope: $scope,
        contextual: false,
        onOpen: function() {},
        onClose: function() {}
    });

    //Here the values "", null & undefined are treated in the same way as non-meaningful
    function areStringsMeaningfullyDifferent(str1, str2) {
        // If both are null, undefined, or empty return false
        if ((str1 ?? "") === "" && (str2 ?? "") === "") {
            return false;
        }
        return str1 !== str2;
    }
});

})();

;
(function() {
'use strict';

const app = angular.module('dataiku.shaker');


app.constant("NORMALIZATION_MODES", [
    ["EXACT", "Exact"],
    ["LOWERCASE", "Ignore case"],
    ["NORMALIZED", "Normalize (ignore accents)"]
]);

app.controller('ShakerSelectColumnsController', function($scope) {
    $scope.uiState = {
        columnsSelectionMode: "ALL"
    };

    const refreshColumns = () => {
        if ($scope.shaker && $scope.shaker.columnsSelection && $scope.shaker.columnsSelection.list) {
            // get mode and all columns from the saved shaker
            $scope.uiState.columnsSelectionMode = $scope.shaker.columnsSelection.mode;
            $scope.allColumnNames = angular.copy($scope.shaker.columnsSelection.list.map((x) => {
                return { name : x.name, $selected : x.d };
            }));
        } else if ($scope.table) {
            // we don't have the saved selection from shaker yet, get all columns from the table
            $scope.allColumnNames = angular.copy($scope.table.allColumnNames.map((x) => {
                return { name : x, $selected : true };
            }));
        }
    }

    $scope.$watch("shaker", () => {
        if ($scope.shaker) {
            refreshColumns();
        }
    });

    $scope.$watch("table", () => {
        if ($scope.table) {
            refreshColumns();
        }
    });

    $scope.ok = () => {
        if ($scope.shaker) {
            $scope.shaker.columnsSelection.mode = $scope.uiState.columnsSelectionMode;
            if ($scope.shaker.columnsSelection.mode == "ALL") {
                $scope.shaker.columnsSelection.list = null;
            } else if ($scope.selection && $scope.selection.allObjects) {
                $scope.shaker.columnsSelection.list = angular.copy($scope.selection.allObjects.map((x) => {
                    return { name : x.name, d : x.$selected };
                }));
            }
        }
        $scope.dismiss();
        $scope.autoSaveForceRefresh();
    }
});


app.controller('ShakerSelectSortController', function($scope) {
    $scope.uiState = {
        query : ''
    };

    // init the 2 lists
    $scope.choicesLeft = [];
    $scope.choicesMade = [];
    
    $scope.$watch("shaker", () => {
        $scope.choicesMade = $scope.shaker && $scope.shaker.sorting ? angular.copy($scope.shaker.sorting) : [];
        const selectedColumnNames = new Set($scope.choicesMade.map(c => c.column));
        if ($scope.shaker && $scope.shaker.columnsSelection && $scope.shaker.columnsSelection.list) {
            $scope.choicesLeft = $scope.shaker.columnsSelection.list
                .filter(col => col.d) // only keep selected/displayed columns
                .filter(col => !selectedColumnNames.has(col.name)) // remove already selected
                .map(item => ({column: item.name, ascending:true}));
        } else if ($scope.table && $scope.table.headers) {
            $scope.choicesLeft = $scope.table.headers
                .filter(header => !selectedColumnNames.has(header.name)) // remove already selected
                .map(header => ({column: header.name, ascending:true}));
        }
    });

    // utils
    $scope.hasSearch = function() {
        return $scope.uiState.query;
    };
    $scope.resetSearch = function() {
        $scope.uiState.query = null;
    };
    $scope.toggle = function(column) {
        column.ascending = !column.ascending;
    };

    // list operations
    $scope.removeAll = function() {
        $scope.choicesLeft = [...$scope.choicesLeft, ...$scope.choicesMade];
        $scope.choicesMade = [];
    };
    $scope.add = function(column) {
        const i = $scope.choicesLeft.indexOf(column);
        if (i >= 0) {
            $scope.choicesLeft.splice(i, 1);
            $scope.choicesMade.push(column);
        }
    };
    $scope.remove = function(column) {
        const i = $scope.choicesMade.indexOf(column);
        if (i >= 0) {
            $scope.choicesMade.splice(i, 1);
            $scope.choicesLeft.push(column);
        }
    };

    $scope.ok = function() {
        if ($scope.shaker) {
            $scope.shaker.sorting = angular.copy($scope.choicesMade);
        }
        $scope.dismiss();
        $scope.autoSaveForceRefresh();
    }
});

app.controller('RegexBuilderController', function ($scope, $stateParams, DataikuAPI, FutureWatcher, SpinnerService, $filter, WT1) {
    $scope.uiState = {};
    $scope.customRegexError = "";
    $scope.firstSentence = "";
    $scope.sentences = [];
    $scope.selectionPositions = []; // changes when new selections or when new sentences
    $scope.selections = [];
    $scope.excludedSentences = [];
    $scope.patterns = [];
    $scope.columnName = "";
    $scope.onColumnNames = false;
    $scope.selectedPattern = null;
    $scope.wrapLines = false;
    $scope.warningMessage = "";
    $scope.lastRequestNumber = 0;
    const MULTI_OCCURRENCES_THRESHOLD = 0.1;

    $scope.createCustomPattern = function (regex) {
        return {
            oldRegex: regex || "",
            regex: regex || "",
            nbOK: -1,
            nbNOK: -1,
            extractions: [],
            errors: [],
        }
    };

    $scope.customPattern = $scope.createCustomPattern();

    $scope.removeNextStepsFromShaker = function (shaker, step) {
        const stepId = $scope.findStepId(step);
        if (typeof (stepId) !== 'undefined') {
            if (stepId.depth === 0) {
                shaker.steps = shaker.steps.slice(0, stepId.id);
            } else if (stepId.depth === 1) {
                shaker.steps[stepId.id].steps = shaker.steps[stepId.id].steps.slice(0, stepId.subId);
                shaker.steps = shaker.steps.slice(0, stepId.id + 1);
            }
        }
    };

    const findSelections = function (sentence, selections) {
        const foundSelections = [];
        for (const sel of selections) {
            if (sentence === sel.before + sel.selection + sel.after) {
                foundSelections.push({
                    start: sel.before.length,
                    end: sel.before.length + sel.selection.length,
                });
            }
        }
        return foundSelections;
    }

    const computeSelectionPositions = function (sentences, selections) {
        const selectionPositions = [];
        for (const sentence of sentences) {
            selectionPositions.push(findSelections(sentence, selections));
        }
        return selectionPositions;
    }

    const computeErrorPositionsOneRow = function (selectionPositions, extractionPositions, isExcluded) {
        extractionPositions = extractionPositions || [];
        if (isExcluded) {
            return extractionPositions;
        }
        if (selectionPositions.length == 0) {
            return [];
        }
        const errorPositions = [];
        for (const selection of selectionPositions) {
            if (!extractionPositions.some(extr => extr.start === selection.start && extr.end === selection.end)) {
                errorPositions.push(selection);
            }
        }
        for (const extraction of extractionPositions) {
            if (!selectionPositions.some(sel => sel.start === extraction.start && sel.end === extraction.end)) {
                errorPositions.push(extraction);
            }
        }
        return errorPositions;
    }

    $scope.computeErrorPositions = function (pattern) {
        const errorPositions = [];
        for (const [index, sentence] of $scope.sentences.entries()) {
            errorPositions.push(computeErrorPositionsOneRow($scope.selectionPositions[index], pattern.extractions[index], $scope.isUsedAsExclusion(sentence)));
        }
        return errorPositions;
    }

    $scope.selectPattern = function (pattern) {
        $scope.selectedPattern = pattern;
    }

    $scope.setDefaultSelectedPattern = function (onlyCustomComputed) {
        if (onlyCustomComputed) {
            if ($scope.selectedPattern.category !== 'userCustomPattern') {
                // if the custom pattern was not selected before , we should not force select it
                const selectedIndex = $scope.patterns.findIndex(p => p.regex == $scope.selectedPattern.regex);
                if (selectedIndex >= 0) {
                    $scope.selectPattern($scope.patterns[selectedIndex]);
                    return;
                }
            }
            $scope.selectPattern($scope.customPattern);
            return;
        }
        //select a default pattern
        if ($scope.patterns.length >= 1) {
            $scope.selectPattern($scope.patterns[0]);
        } else {
            $scope.selectPattern($scope.customPattern);
        }
    }

    $scope.filterRequestParameter = function () {
        return { "elements": $scope.buildFilterRequest($scope.shaker.explorationFilters) };
    }

    $scope.computePatterns = function (computeOnlyCustom) {
        $scope.lastRequestNumber++;
        const currentRequestNumber = $scope.lastRequestNumber;
        const shakerForQuery = $scope.shakerHooks.shakerForQuery();
        // Remove currently edited step and next steps from query
        if ($scope.editStep) {
            $scope.removeNextStepsFromShaker(shakerForQuery, $scope.editStep);
        }
        let selections = [];
        let excludedSentences = [];
        const customRegex = $scope.customPattern.regex;
        if (!computeOnlyCustom) {
            selections = $scope.selections;
            excludedSentences = $scope.excludedSentences;
        }
        DataikuAPI.shakers.smartExtractor($stateParams.projectKey, $scope.inputDatasetProjectKey, $scope.inputDatasetName, shakerForQuery, $scope.requestedSampleId,
            $scope.columnName,
            selections,
            excludedSentences,
            customRegex,
            $scope.onColumnNames,
            $scope.firstSentence,
            $scope.filterRequestParameter()
        ).success(function (initialResponse) {
            SpinnerService.lockOnPromise(FutureWatcher.watchJobId(initialResponse.jobId).success(function (data) {
                if (currentRequestNumber !== $scope.lastRequestNumber) return;
                if (!computeOnlyCustom) {
                    $scope.patterns = data.result.categories
                        .filter(c => c.name !== "userCustomPattern")
                        .map(c => c.propositions)
                        .flat();
                }
                $scope.customRegexError = data.result.customRegexError;
                const oldRegex = $scope.customPattern.regex
                const customPatternCategory = data.result.categories.find(c => c.name === "userCustomPattern");
                if (customPatternCategory && customPatternCategory.propositions.length >= 1) {
                    $scope.customPattern = customPatternCategory.propositions[0];
                }
                $scope.customPattern.oldRegex = oldRegex;
                $scope.sentences = data.result.sentences;
                $scope.selectionPositions = computeSelectionPositions($scope.sentences, $scope.selections);
                $scope.patterns = $scope.patterns.map(p => {
                    return {
                        ...p,
                        errors: $scope.computeErrorPositions(p),
                    }
                });
                $scope.customPattern.errors = $scope.computeErrorPositions($scope.customPattern)
                if (!$scope.customPattern) {
                    $scope.customPatterns = $scope.createCustomPattern();
                }
                $scope.setDefaultSelectedPattern(computeOnlyCustom);
            }));
        }).error(setErrorInScope.bind($scope));
    };

    $scope.addSelection = function (sentence, startOff, endOff) {
        if ($scope.isUsedAsExclusion(sentence)) {
            $scope.removeSentenceSelectionExcluded(sentence);
        }
        while (sentence[startOff] === " " && startOff <= sentence.length) {
            startOff++;
        }
        while (sentence[endOff - 1] === " " && endOff >= 0) {
            endOff--;
        }
        if (endOff <= startOff) {
            return;
        }
        const sel = {
            before: sentence.substring(0, startOff),
            selection: sentence.substring(startOff, endOff),
            after: sentence.substring(endOff),
        };
        if ($scope.isExistingSelection(sel)) return;
        $scope.selections.push(sel);
        $scope.computePatterns(false);
    };

    $scope.removeSelection = function (idx) {
        $scope.selections.splice(idx, 1);
        $scope.computePatterns(false);
    };

    $scope.removeSentenceSelectionExcluded = function (sentence) {
        $scope.selections = $scope.selections.filter(sel => sel.before + sel.selection + sel.after !== sentence);
        $scope.excludedSentences = $scope.excludedSentences.filter(excl => excl !== sentence);
    };

    $scope.ignoreErrors = function (sentence) {
        $scope.removeSentenceSelectionExcluded(sentence);
        $scope.computePatterns(false);
    };

    $scope.addExcludedSentence = function (sentence) {
        $scope.removeSentenceSelectionExcluded(sentence);
        $scope.excludedSentences.push(sentence);
        $scope.computePatterns(false);
    };

    $scope.removeExcludedSentence = function (idx) {
        $scope.excludedSentences.splice(idx, 1);
        $scope.computePatterns(false);
    };

    $scope.isExistingSelection = function (selection) {
        return $scope.selections.some(sel => sel.before === selection.before && sel.selection === selection.selection && sel.after === selection.after);
    };

    $scope.isUsedAsExclusion = function (sentence) {
        if ($scope.excludedSentences.includes(sentence)) {
            return true;
        }
        return false;
    };

    $scope.findStartOffset = function (nodes, selection) {
        //selection is not a string, it's a https://developer.mozilla.org/en-US/docs/Web/API/Selection
        //nodes are DOM elements
        let startOff = 0;
        for (const node of nodes) {
            if (node.nodeType == 1 && !selection.containsNode(node, true)) {
                startOff += node.textContent.length;
            }
            if (node.nodeType == 1 && selection.containsNode(node, true)) {
                // the selection is between anchorNode and focusNode, but they can either be at the begining or the end of the selection depending on the selection direction (to the right or to the left)
                const isAnchor = node.isSameNode(selection.anchorNode.parentElement);
                const isFocus = node.isSameNode(selection.focusNode.parentElement);
                if (isAnchor && isFocus) {
                    return startOff + Math.min(selection.anchorOffset, selection.focusOffset);
                }
                if (isAnchor) {
                    return startOff + selection.anchorOffset;
                }
                if (isFocus) {
                    return startOff + selection.focusOffset;
                }
            }
        }
    };

    $scope.isSingleNodeSelection = function (nodes, selection) {
        let containsAnchor = false;
        let containsFocus = false;
        for (const node of nodes) {
            if (node.isSameNode(selection.anchorNode.parentElement)) {
                containsAnchor = true;
            }
            if (node.isSameNode(selection.focusNode.parentElement)) {
                containsFocus = true;
            }
        }
        return containsAnchor && containsFocus;
    };

    $scope.onSelection = function (evt, sentence) {
        evt.stopPropagation();
        var userSelection;
        if (window.getSelection) {
            userSelection = window.getSelection();
        } else if (document.selection) {
            userSelection = document.selection.createRange();
        }
        const rowNodes = evt.currentTarget.childNodes;
        if (!$scope.isSingleNodeSelection(rowNodes, userSelection)) {
            return;
        }
        var selectedText = userSelection + "";
        if (userSelection.text) {
            selectedText = userSelection.text;
        }
        if (selectedText) {
            const startOff = $scope.findStartOffset(rowNodes, userSelection);
            $scope.addSelection(sentence, startOff, startOff + selectedText.length);
        }
    };

    $scope.onCustomRegexChange = function () {
        const pattern = $scope.customPattern;
        if (pattern.regex !== pattern.oldRegex) {
            $scope.computePatterns(true);
        }
    }

    $scope.getWT1Stats = function (action) {
        let stats = {};
        stats.action = action || '';
        stats.nbSelections = $scope.selections.length;
        if ($scope.selectedPattern != null) {
            stats.category = $scope.selectedPattern.category;
            stats.matchOK = $scope.selectedPattern.nbOK;
            stats.matchNOK = $scope.selectedPattern.nbNOK;
        }
        stats.calledFrom = $scope.calledFrom || '';
        return stats;
    };

    $scope.save = function () {
        const WT1stats = $scope.getWT1Stats("save")
        WT1.event("patternbuilder", WT1stats);
        if ($scope.selectedPattern != null) {
            if ($scope.deferred) {
                const extractingLines = $scope.selectedPattern.extractions.filter(extractions => extractions.length > 0).length;
                const multiExtractingLines = $scope.selectedPattern.extractions.filter(extractions => extractions.length > 1).length;
                let multiOccurenceRatio = 0;
                if (extractingLines >= 1) {
                    multiOccurenceRatio = multiExtractingLines / extractingLines;
                }
                const pattern = {
                    regex: $scope.selectedPattern.regex,
                    hasMultiOccurrences: multiOccurenceRatio > MULTI_OCCURRENCES_THRESHOLD,
                };
                $scope.deferred.resolve(pattern);
            }
        }
        $scope.dismiss();
    };

    $scope.cancel = function () {
        const WT1stats = $scope.getWT1Stats("cancel");
        WT1.event("patternbuilder", WT1stats);
        if ($scope.deferred) {
            $scope.deferred.reject();
        }
        $scope.dismiss();
    };

});

app.controller(
    "GrokDebuggerController",
    function ($scope, $stateParams, DataikuAPI) {
        $scope.sentences = [];
        $scope.captureOffsets = [];
        $scope.matchOffsets = [];
        const shakerForQuery = $scope.shakerHooks.shakerForQuery();

        $scope.removeNextStepsFromShaker = function (shaker, step) {
            const stepId = $scope.findStepId(step);
            if (typeof stepId !== "undefined") {
                if (stepId.depth === 0) {
                    shaker.steps = shaker.steps.slice(0, stepId.id);
                } else if (stepId.depth === 1) {
                    shaker.steps[stepId.id].steps = shaker.steps[stepId.id].steps.slice(
                        0,
                        stepId.subId
                    );
                    shaker.steps = shaker.steps.slice(0, stepId.id + 1);
                }
            }
        };

        $scope.loadGrokExpressionSample = function () {
            if ($scope.editStep) {
                $scope.removeNextStepsFromShaker(shakerForQuery, $scope.editStep);
            }
            const sourceColumnIndex = $scope.step.$stepState.change.columnsBeforeStep.indexOf($scope.step.params.sourceColumn);
            resetErrorInScope($scope);
            DataikuAPI.shakers
                .getGrokExpressionSample(
                    $stateParams.projectKey,
                    $scope.inputDatasetProjectKey,
                    $scope.inputDatasetName,
                    shakerForQuery,
                    $scope.requestedSampleId,
                    0,
                    100,
                    sourceColumnIndex,
                    1, // number of columns (always one)
                    $scope.grokPattern || $scope.step.params.grokPattern
                )
                .success(function (response) {
                    $scope.sentences = response.table.content;
                    $scope.matchOffsets = response.matchOffsets.map((value) => [value]);
                    $scope.captureOffsets = response.captureOffsets;
                    $scope.matchedLines = response.matchedLines;
                    $scope.processedLines = response.processedLines;
                })
                .error(setErrorInScope.bind($scope));
        };

        $scope.handleChange = function () {
            $scope.loadGrokExpressionSample();
        };

        $scope.save = function () {
            if ($scope.deferred) {
                $scope.deferred.resolve($scope.grokPattern);
            }
            $scope.dismiss();
        };

        $scope.cancel = function () {
            if ($scope.deferred) {
                $scope.deferred.reject();
            }
            $scope.dismiss();
        };
    }
);
  

app.directive('overlapUnderlineHighlight', function () {
    const template = '<span ng-repeat="subSentence in subSentences"'
        + 'ng-class="{'
        + '\'overlap-underline-highlight--dark-green\': subSentence.darkGreen,'
        + '\'overlap-underline-highlight--light-green\': subSentence.lightGreen,'
        + '\'overlap-underline-highlight--error\': subSentence.error,'
        + '\'overlap-underline-highlight--crossed\': isCrossed}">'
        + '{{subSentence.value}}'
        + '</span>';
    return {
        template,
        restrict: 'AE',
        scope: {
            sentence: '=',
            isCrossed: '=',
            darkGreenOffsets: '=', // all offsets should look like [{start: 26, end: 41}, {start: 132, end: 138}]
            lightGreenOffsets: '=',
            errorOffsets: '=',
        },
        link: function (scope, element, attrs) {
            scope.$watch('[sentence, darkGreenOffsets, lightGreenOffsets, errorOffsets, isCrossed]', function (ov, nv) {
                if(scope.sentence === null) {
                    return;
                }
                const darkGreenOffsets = scope.darkGreenOffsets || [];
                const lightGreenOffsets = scope.lightGreenOffsets || [];
                const errorOffsets = scope.errorOffsets || [];
                scope.subSentences = [];
                let styleChangingIndexes = new Set([
                    0,
                    ...darkGreenOffsets.flatMap(o => [o.start, o.end]),
                    ...lightGreenOffsets.flatMap(o => [o.start, o.end]),
                    ...errorOffsets.flatMap(o => [o.start, o.end]),
                    scope.sentence.length,
                ]);
                const sortedIndexes = [...styleChangingIndexes].sort((a, b) => a - b);
                for (let i = 0; i < sortedIndexes.length - 1; i++) {
                    const start = sortedIndexes[i];
                    const end = sortedIndexes[i + 1];
                    let subsentence = {
                        value: scope.sentence.substring(start, end),
                        darkGreen: darkGreenOffsets.some(o => o.start <= start && o.end >= end),
                        lightGreen: lightGreenOffsets.some(o => o.start <= start && o.end >= end),
                        error: errorOffsets.some(o => o.start <= start && o.end >= end),
                    };
                    scope.subSentences.push(subsentence);
                };
            }, true)
        }
    }
});

app.controller('SmartDateController', function($scope, $stateParams, $timeout, DataikuAPI, WT1) {
    $scope.uiState = {
        customFormatInput: ""
    };

    $scope.removeNextStepsFromShaker = function(shaker, step) {
        const stepId = $scope.findStepId(step);
        if (typeof(stepId)!=='undefined') {
            if (stepId.depth == 0) {
                shaker.steps = shaker.steps.slice(0, stepId.id);
            } else if (stepId.depth == 1) {
                shaker.steps[stepId.id].steps = shaker.steps[stepId.id].steps.slice(0, stepId.subId);
                shaker.steps = shaker.steps.slice(0, stepId.id + 1);
            }
        }
    };

    $scope.setColumn  = function(name) {
        $scope.columnName = name;
        // Copy of the original shaker for the query
        const shakerForQuery = $scope.shakerHooks.shakerForQuery();
        // Remove currently edited step and next steps from query
        if ($scope.editStep) {
            $scope.removeNextStepsFromShaker(shakerForQuery, $scope.editStep);
        }

        DataikuAPI.shakers.smartDateGuess($stateParams.projectKey,
            $scope.inputDatasetProjectKey, $scope.inputDatasetName, shakerForQuery, $scope.requestedSampleId, $scope.columnName).success(function(data) {
            $scope.autodetected = data.formats;
            WT1.event("smartdate-guessed", {"nbGuessed" : $scope.autodetected.length});
            let detectedFormatIndex;
            for (const i in $scope.autodetected) {
                let fmt = $scope.autodetected[i];
                fmt.parsesPercentage = 100*fmt.nbOK/(fmt.nbOK + fmt.nbNOK + fmt.nbPartial);
                fmt.partialPercentage = 100*fmt.nbPartial/(fmt.nbOK + fmt.nbNOK + fmt.nbPartial);
                fmt.failsPercentage = 100 - fmt.parsesPercentage - fmt.partialPercentage;
                if ($scope.editFormat == fmt.format) {
                    detectedFormatIndex = i;
                }
            }

            let selectCustom = false
            if (!$scope.editFormat) {
                // New smart date or no format to edit => select first format if found
                if ($scope.autodetected.length > 0) {
                    $scope.selectFormat(0);
                } else { // select the custom format
                    selectCustom = true;
                }
            } else if (detectedFormatIndex >= 0) {
                // Found format in the guess list => select this one
                $scope.selectFormat(detectedFormatIndex);
            } else {
                selectCustom = true;
            }
            // need to validate the empty custom so that we can display examples (they come from backend and depend on the format)
            $scope.validateCustom(selectCustom);
        }).error(setErrorInScope.bind($scope));
    };

    WT1.event("smartdate-open");

    $scope.selectFormat = function(idx) {
        $scope.selectedFormat = $scope.autodetected[idx];
    };

    $scope.validateSeqId = 0;

    $scope.onCustomFormatClick = function() {
        $scope.selectedFormat = $scope.customFormat;
    };

    $scope.validateCustom = function(isSelected) {
        $scope.validateSeqId++;
        const seqId = $scope.validateSeqId;
        WT1.event("smartdate-validate-custom", {"format" : $scope.uiState.customFormatInput});
        // Get a copy of the current shaker for the query
        let shakerForQuery = $scope.shakerHooks.shakerForQuery();
        // Remove currently edited step and next steps from query
        if ($scope.editStep) {
            $scope.removeNextStepsFromShaker(shakerForQuery, $scope.editStep);
        }
        DataikuAPI.shakers.smartDateValidate($stateParams.projectKey,
                $scope.inputDatasetProjectKey, $scope.inputDatasetName, shakerForQuery, $scope.requestedSampleId,
                $scope.columnName, $scope.uiState.customFormatInput == null ? "" : $scope.uiState.customFormatInput).success(function(data) {
            if (seqId != $scope.validateSeqId) return;
            data.parsesPercentage = 100*data.nbOK/(data.nbOK + data.nbNOK + data.nbPartial);
            data.partialPercentage = 100*data.nbPartial/(data.nbOK + data.nbNOK + data.nbPartial);
            data.failsPercentage = 100 - data.parsesPercentage - data.partialPercentage;
            if (isSelected) {
                $scope.selectedFormat = data;
            }
            $scope.customFormat = data;
        }).error(setErrorInScope.bind($scope));
    };

    $scope.save = function() {
        if ($scope.selectedFormat != null && $scope.selectedFormat.validFormat) {
            WT1.event("smartdate-accept", {"format" : $scope.selectedFormat.format});
            if ($scope.deferred) {
                $scope.deferred.resolve([$scope.selectedFormat.format, $scope.selectedFormat.language]);
            }
            $scope.dismiss();
        }
    };

    $scope.cancel = function() {
        if ($scope.deferred) {
            $scope.deferred.reject();
        }
        $scope.dismiss();
    };

    $scope.$watch("uiState.customFormatInput", function(ov, nv) {
        if (ov != nv) {
            $timeout(function () {$scope.validateCustom(true);}, 200);
        }
    });
});

app.controller('DateParserController', function($scope, $q, $element, WT1, CreateModalFromTemplate) {
    if (!$scope.step.params.formats) $scope.step.params.formats = [''];
    $scope.formatItemTemplate = {format: ''};
    $scope.formatItems = $scope.step.params.formats.map(function(f) { return {format: f}; });
    $scope.formatsChanged = function() {
        [].splice.apply($scope.step.params.formats,
            [0, $scope.step.params.formats.length].concat($scope.formatItems.map(function(fi) { return fi.format; })));
        $scope.checkAndRefresh();
    };
    $scope.validateFormatList = function(it, itemIndex) {
        if (!it || !it.format || it.format.length == 0) return false;
        for (var i in $scope.formatItems) {
            if ($scope.formatItems[i].format == it.format) return false;
        }
        return true;
    };
    $scope.showSmartDateTool = function() {
        return $scope.columns && $scope.step.params.columns && $scope.step.params.columns[0] && $scope.columns.indexOf($scope.step.params.columns[0]) >= 0;
    };
    $scope.openSmartDateTool = function() {
        if (!$scope.step.params.columns || !$scope.step.params.columns[0] || $scope.columns.indexOf($scope.step.params.columns[0]) == -1) return;
        
        WT1.event("shaker-parsedate-edit-smartdate");
        var deferred = $q.defer();
        CreateModalFromTemplate("/templates/shaker/smartdate-box.html", $scope, "SmartDateController",
            function(newScope) { newScope.$apply(function() {
                    newScope.deferred = deferred;
                    newScope.editStep = $scope.step;
                    newScope.editFormat = "";
                    newScope.setColumn($scope.step.params.columns[0]);
            }); }, "sd-modal");
        deferred.promise.then(function([newFormat, newFormatLanguage]) {
            if (!newFormat || newFormat.length == 0) return;
            // Checks if the new format is already present in the list and retrieve the old format index
            let isNewFormatThere, newFormatIdx;
            for (let i in $scope.formatItems) {
                if ($scope.formatItems[i].format == newFormat) {
                    isNewFormatThere = true;
                    newFormatIdx = i;
                    break;
                }
            }
            // Edit the new format if not present, otherwise focus on the existing input containing the new format
            if (!isNewFormatThere) {
                if ($scope.formatItems.length > 0) {
                    const lastItem = $scope.formatItems[$scope.formatItems.length - 1];
                    if (lastItem.format == '') {
                        $scope.formatItems.pop();
                    }
                }
                $scope.formatItems = [...$scope.formatItems, { format: newFormat }];
                newFormatIdx = $scope.formatItems.length - 1;
                $scope.formatsChanged();
            }
            if (newFormatLanguage) {
                $scope.step.params.lang = newFormatLanguage;
            }
            setTimeout(() => {
                // element not yet created ion the dom when checking
                $element.find(".dateFormatInput")[newFormatIdx].focus();
            }, 100);
        });
    };
    $scope.storageTemporalTypes = [
        [{name:'foo', type:'date'}, 'Datetime with tz'],
        [{name:'foo', type:'dateonly'}, 'Date only'],
        [{name:'foo', type:'datetimenotz'}, 'Datetime no tz']
    ];
});

app.controller('RenameColumnController', function($scope) {
    $scope.setColumn  = function(name) {
        $scope.columnName = name;
        $scope.renameTarget = name;
    }

    $scope.save = function() {
        if($scope.columnName!=$scope.renameTarget) {
            $scope.addStepNoPreviewAndRefresh("ColumnRenamer", {
                renamings : [
                    {"from" : $scope.columnName, "to" : $scope.renameTarget}
                ]
            });
            $scope.mergeLastColumnRenamers();
        }
        $scope.dismiss();
    }
});

app.controller('MoveColumnController', function($scope) {
    $scope.setColumn = function(name) {
        $scope.columnName = name;
        // Remove the column we want to move from the list of reference columns.
        let index = $scope.referenceColumnList.indexOf(name);
        $scope.referenceColumnList.splice(index, 1);
    };

    $scope.save = function() {
        $scope.addStepNoPreviewAndRefresh("ColumnReorder", {
            appliesTo : "SINGLE_COLUMN",
            columns: [$scope.columnName],
            referenceColumn: $scope.referenceColumn,
            reorderAction: $scope.reorderAction.name
        });
        $scope.mergeLastColumnReorders();
        $scope.dismiss();
    };

    $scope.reorderActions = [
        { name: "AT_START", label: "at beginning", needReferenceColumn: false },
        { name: "AT_END", label: "at end", needReferenceColumn: false },
        { name: "BEFORE_COLUMN", label: "before", needReferenceColumn: true },
        { name: "AFTER_COLUMN", label: "after", needReferenceColumn: true }
    ];

    $scope.reorderAction = $scope.reorderActions[0];
    $scope.referenceColumnList = $scope.tableModel.allColumnNames.slice();
});

app.controller('FillEmptyWithValueController', function($scope, DataikuAPI, MonoFuture, $stateParams) {
    var monoFuture = MonoFuture($scope);
    function analysis(callback) {
        monoFuture.exec(
            DataikuAPI.shakers.multiColumnAnalysis(
                $stateParams.projectKey,
                $scope.inputDatasetProjectKey, $scope.inputDatasetName, $scope.inputStreamingEndpointId, $scope.shakerHooks.shakerForQuery(),
                $scope.requestedSampleId, $scope.columns, $scope.source)
        ).success(function(data) {
            if (data.hasResult) {
                callback(data.result);
            }
        }).error(setErrorInScope.bind($scope));
    }
    $scope.source = 'constant';
    $scope.isNumericOnly = false;
    $scope.setColumns = function(cols) {
        $scope.columns = cols;
        $scope.columnName = cols.length === 1 ? cols[0] : null;
    }
    $scope.save = function() {
        var fn = $scope.columns.length === 1 ? 'addStep' : 'addStepNoPreview';
        if ($scope.source === 'constant') {
            if ($scope.columns.length == 1) {
                $scope[fn]("FillEmptyWithValue", { appliesTo : "SINGLE_COLUMN", columns: $scope.columns, value: $scope.valueToFill });
            } else {
                $scope[fn]("FillEmptyWithValue", { appliesTo : "COLUMNS", columns: $scope.columns, value: $scope.valueToFill });
            }
            $scope.autoSaveForceRefresh();
            $scope.dismiss();
        } else {
            analysis(function(data) {
                for (var c in data) {
                    $scope[fn]("FillEmptyWithValue", { appliesTo : "SINGLE_COLUMN", columns: [c], value: data[c] });
                }
                $scope.autoSaveForceRefresh();
                $scope.dismiss();
            });
        }
    }
});


app.controller('MassRenameColumnsController', function($scope, DataikuAPI) {
    var edits = {
        findReplace: function(c) {
            const pattern = $scope.find;
            if (pattern && pattern !== "") {
                if ($scope.useRegex) {
                    const re = new RegExp(pattern, $scope.ignoreCase ? "ig" : "g");
                    return c.replaceAll(re, $scope.replace);
                } else {
                    if ($scope.ignoreCase) {
                        let index = c.toLowerCase().indexOf(pattern.toLowerCase());
                        if (index >= 0) {
                            let newName = c.substring(0, index) + $scope.replace;
                            let remaining = c.substring(index + pattern.length);
                            while (remaining.length > 0) {
                                index = remaining.toLowerCase().indexOf(pattern.toLowerCase());
                                if (index >= 0) {
                                    newName += remaining.substring(0, index) + $scope.replace;
                                    remaining = remaining.substring(index + pattern.length);
                                } else {
                                    newName += remaining;
                                    remaining = "";
                                }
                            }
                            return newName;
                        }
                    } else {
                        return c.replaceAll(pattern, $scope.replace);
                    }
                }
            }
            return c;
        },
        addPrefix: c => $scope.prefix ? $scope.prefix + c : c,
        addSuffix: c => $scope.suffix ? c + $scope.suffix : c,
        removePrefix: c => {
            const prefix = $scope.prefix;
            return prefix && (c.startsWith(prefix) || $scope.ignoreCase && c.toLowerCase().startsWith(prefix.toLowerCase()))
                ? c.substring(prefix.length) : c;
        },
        removeSuffix: c => {
            const suffix = $scope.suffix;
            return suffix && (c.endsWith(suffix) || $scope.ignoreCase && c.toLowerCase().endsWith(suffix.toLowerCase()))
                ? c.substring(0, c.length - suffix.length) : c;
        },
        lowercase: c => c.toLowerCase(),
        uppercase: c => c.toUpperCase()
    };
    $scope.replace = '';
    $scope.setColumns = function(cols) {
        $scope.columns = cols;
        $scope.columnName = cols.length === 1 ? cols[0] : null;
    }
    $scope.edit = 'findReplace';
    $scope.ignoreCase = true;

    $scope.validateRegex = function() {
        $scope.massRenameForm.$setValidity("regex", true);
        if ($scope.edit === 'findReplace' && $scope.useRegex) {
            try {
                // Try to compile the pattern
                new RegExp($scope.find);
            } catch (error) {
                $scope.massRenameForm.$setValidity("regex", false);
            }
        }
    }

    $scope.$watch('[edit, useRegex]', () => {
        $scope.validateRegex();
    });

    $scope.save = async function() {
        let dirty = false;
        let renamings = [];

        if ($scope.edit === "normalizeSpecialChars") {
            /* This one does an API call, so we batch it */
            const editedList = (await DataikuAPI.shakers.convertToASCII($scope.columns, true, true)).data;
            for (let i = 0; i < $scope.columns.length; i++) {
                const c = $scope.columns[i];
                const c2 = editedList[i];
                if (c2 && c2 !== c) {
                    dirty = true;
                    renamings.push({ from: c, to: c2 });
                }
            }
        } else {
            $scope.columns.forEach(function(c) {
                var c2 = edits[$scope.edit](c);
                if (c2 && c2 !== c) {
                    dirty = true;
                    renamings.push({ from: c, to: c2 });
                }
            });
        }

        if (dirty) {
            $scope.doRenameColumns(renamings);
        }
        $scope.dismiss();
    }
});


app.controller('MultiRangeController', function($scope) {
    $scope.setColumns = function(cols) {
        $scope.columns = cols;
        $scope.columnName = cols.length === 1 ? cols[0] : null;
    }

    $scope.keep = true;
    $scope.save = function() {
        $scope.addStepNoPreview("FilterOnNumericalRange", {
            appliesTo : "COLUMNS",
            columns: $scope.columns,
            min: $scope.min,
            max: $scope.max,
            action : $scope.keep ? "KEEP_ROW" : "REMOVE_ROW"
        });
        $scope.autoSaveForceRefresh();
        $scope.dismiss();
    }
});


app.controller('FindReplaceController', function($scope, translate) {
    const NORMALIZATION_MODES = [
        ["EXACT", translate("SHAKER.NORMALIZATION_MODES.EXACT", "Exact")],
        ["LOWERCASE", translate("SHAKER.NORMALIZATION_MODES.LOWERCASE", "Ignore case")],
        ["NORMALIZED", translate("SHAKER.NORMALIZATION_MODES.NORMALIZED", "Normalize (ignore accents)")]
    ]
    $scope.$watch("step.params.matching", function(nv, ov) {
        if (nv && nv === "FULL_STRING") {
            $scope.normalizationModes = NORMALIZATION_MODES;
        } else if (nv) {
            $scope.normalizationModes = NORMALIZATION_MODES.filter(function(mode) { return mode[0] !== 'NORMALIZED'; });
            if ($scope.step.params.normalization === "NORMALIZED") {
                $scope.step.params.normalization = "EXACT";
            }
        }
    })
});


app.controller('SwitchCaseController', function($scope, translate) {
    $scope.normalizationModes = [
       ["EXACT", translate("SHAKER.NORMALIZATION_MODES.EXACT", "Exact")],
       ["LOWERCASE", translate("SHAKER.NORMALIZATION_MODES.LOWERCASE", "Ignore case")],
       ["NORMALIZED", translate("SHAKER.NORMALIZATION_MODES.NORMALIZED", "Normalize (ignore accents)")]
   ];
});

app.controller('FlagOnValuesController', function($scope, translate) {
    $scope.normalizationModes = [
        ["EXACT", translate("SHAKER.NORMALIZATION_MODES.EXACT", "Exact")],
        ["LOWERCASE", translate("SHAKER.NORMALIZATION_MODES.LOWERCASE", "Ignore case")],
        ["NORMALIZED", translate("SHAKER.NORMALIZATION_MODES.NORMALIZED", "Normalize (ignore accents)")]
    ];
});

app.controller('VisualIfRuleController', function($scope, $controller, CreateCustomElementFromTemplate, DataikuAPI, $stateParams, ShakerPopupRegistry, $rootScope){
    $scope.modalShown = false;
    $scope.parentDismiss = function() {
    	$rootScope.$broadcast("dismissModalInternal_");
    	$scope.modalShown = false;
    }

    $scope.editRule = function(){
        $scope.editRule_(angular.copy($scope.step.params.visualIfDesc));
    }

    $scope.columns = [];
    const updateColumns = () => {
        if ($scope.step.$stepState.change) {
            $scope.columns = $scope.step.$stepState.change.columnsBeforeStep;
        }
    }
    updateColumns();

    const stopWatchingForColumnsCompute = $scope.$watch('[step.preview, shaker.steps]', () => {
        updateColumns();
    });

    $scope.editRule_ = function(visualIfDesc) {
        if ($scope.modalShown) {
            $scope.parentDismiss();
        } else {
            ShakerPopupRegistry.dismissAllAndRegister($scope.parentDismiss);
            $scope.modalShown = true;
            DataikuAPI.datasets.get($scope.inputDatasetProjectKey, $scope.inputDatasetName, $stateParams.projectKey)
                .success(function(data){
                    CreateCustomElementFromTemplate("/static/dataiku/processors/visual-if/visual-if-editor/visual-if-rule.html", $scope, null, function(newScope) {
                        $scope.editing = {};
                        $scope.editing.visualIfDesc = visualIfDesc;
                        $scope.step.columnsCache = $scope.columns;
                    }, $scope.customFormulaEdition.displayCustomFormula);
            }).error(setErrorInScope.bind($scope));
        }
    }

    if ($scope.step.isNew === true) {
        $scope.editRule();
    }

    $scope.$on("resetModalShownInternal_", function() {
        $scope.modalShown = false;
    });

    $scope.$on("$destroy", stopWatchingForColumnsCompute);

});

app.controller('MeaningTranslateController', function($scope) {
    $scope.meanings = $scope.appConfig.meanings.categories
        .filter(function(cat) { return cat.label === "User-defined"; })[0]
        .meanings.filter(function(meaning) { return meaning.type === 'VALUES_MAPPING'; });
});


app.controller('CurrencySplitterController', function($scope) {
    $scope.$watch("step.params.inCol", function(nv, ov) {
        if (angular.isDefined(nv) && nv.length > 0 && $scope.step.isNew) {
          $scope.step.params.outColCurrencyCode = $scope.step.params.inCol + "_currency_code";
          $scope.step.params.outColAmount = $scope.step.params.inCol + "_amount";
        }
    }, true);

    /* TODO: This should rather be done by a "default values" section ... */
});


app.controller('ColumnSplitterController', function($scope) {
    $scope.$watch("step.params.inCol", function(nv, ov) {
        if (angular.isDefined(nv) && nv.length > 0 && $scope.step.isNew) {
            $scope.step.params.outColPrefix = $scope.step.params.inCol + "_";
        }
    }, true);

    /* TODO: This should rather be done by a "default values" section ... */
    $scope.$watch("step.params.limitOutput", function(nv, ov) {
        if ($scope.step.params.limitOutput && !$scope.step.params.limit) {
            $scope.step.params.limit = 1;
            $scope.step.params.startFrom = "beginning";
        }
    }, true);
});

app.controller('SplitIntoChunksController', function($scope) {
    $scope.$watch("step.params.inCol", function(nv, ov) {
        if (angular.isDefined(nv) && nv.length > 0 && $scope.step.isNew) {
            $scope.step.params.outCol = $scope.step.params.inCol + "_chunked";
        }
    }, true);
    $scope.getTemplate = function() {
        return {
            value: "",
            isDefault: false,
            enabled: true
        }
    }
    $scope.disableRemove = function(it) {
        return it.isDefault
    }
});

app.controller('CurrencyConverterController', function($scope, DataikuAPI, translate) {

    $scope.currencies = [
        ["AED", "AED (United Arab Emirates Dirham)"],
        ["AUD", "AUD (Australian Dollar)"],
        ["BGN", "BGN (Bulgarian Lev)"],
        ["BND", "BND (Brunei Dollar)"],
        ["BRL", "BRL (Brazilian Real)"],
        ["BWP", "BWP (Botswana Pula)"],
        ["CAD", "CAD (Canadian Dollar)"],
        ["CHF", "CHF (Swiss Franc)"],
        ["CLP", "CLP (Chilean Peso)"],
        ["CNY", "CNY (Chinese Yuan)"],
        ["CYP", "CYP (Cypriot Pound)"],
        ["CZK", "CZK (Czech Koruna)"],
        ["DKK", "DKK (Danish Krone)"],
        ["DZD", "DZD (Algerian Dinar)"],
        ["EEK", "EEK (Estonian Kroon)"],
        ["EUR", "EUR (Euro)"],
        ["GBP", "GBP (British Pound Sterling)"],
        ["HKD", "HKD (Hong Kong Dollar)"],
        ["HRK", "HRK (Croatian Kuna)"],
        ["HUF", "HUF (Hungarian Forint)"],
        ["IDR", "IDR (Indonesian Rupiah)"],
        ["ILS", "ILS (Israeli Shekel)"],
        ["INR", "INR (Indian Rupee)"],
        ["ISK", "ISK (Icelandic Krona)"],
        ["JPY", "JPY (Japanese Yen)"],
        ["KRW", "KRW (South Korean Won)"],
        ["KWD", "KWD (Kuwaiti Dinar)"],
        ["LTL", "LTL (Lithuanian Litas)"],
        ["LVL", "LVL (Latvian Lats)"],
        ["MTL", "MTL (Maltese Lira)"],
        ["MUR", "MUR (Mauritian Rupee)"],
        ["MXN", "MXN (Mexican Peso)"],
        ["MYR", "MYR (Malaysian Ringgit)"],
        ["NOK", "NOK (Norwegian Krone)"],
        ["NZD", "NZD (New Zealand Dollar)"],
        ["OMR", "OMR (Omani Rial)"],
        ["PEN", "PEN (Peruvian Sol)"],
        ["PHP", "PHP (Philippine Peso)"],
        ["PLN", "PLN (Polish Zloty)"],
        ["QAR", "QAR (Qatari Riyal)"],
        ["ROL", "ROL (Romanian Leu (Old))"],
        ["RON", "RON (Romanian Leu (New))"],
        ["RUB", "RUB (Russian Ruble)"],
        ["SAR", "SAR (Saudi Riyal)"],
        ["SEK", "SEK (Swedish Krona)"],
        ["SGD", "SGD (Singapore Dollar)"],
        ["SIT", "SIT (Slovenian Tolar)"],
        ["SKK", "SKK (Slovak Koruna)"],
        ["THB", "THB (Thai Baht)"],
        ["TRL", "TRL (Turkish Lira (Old))"],
        ["TRY", "TRY (Turkish Lira (New))"],
        ["TTD", "TTD (Trinidad and Tobago Dollar)"],
        ["USD", "USD (United States Dollar)"],
        ["UYU", "UYU (Uruguayan Peso)"],
        ["ZAR", "ZAR (South African Rand)"]
    ]

    $scope.dateInputs = [];
    DataikuAPI.shakers.getLastKnownCurrencyRateDate().success(function(data) {
        $scope.dateInputs = buildDateInputsList(data);
    }).error(function() {
        $scope.dateInputs = buildDateInputsList(translate("SHAKER.PROCESSOR.CurrencyConverter.ERROR.UNKNOWN_DATE", "unknown date"));
    });

    function buildDateInputsList(lastKnownRateDate) {
        return [
            ["LATEST", translate("SHAKER.PROCESSOR.CurrencyConverter.DATE.LATEST", "Last known rates ({{date}})", {date: lastKnownRateDate})],
            ["COLUMN", translate("SHAKER.PROCESSOR.CurrencyConverter.DATE.COLUMN", "From Column (Date)")],
            ["CUSTOM", translate("SHAKER.PROCESSOR.CurrencyConverter.DATE.CUSTOM", "Custom input")]
        ];
    }

    function isColumnValid(col, meaning) {
        if(!$scope.table) return true;
        return $scope.table.headers.some(function(h) { return h.name === col && h.selectedType.name === meaning; });
    }

    $scope.$watch('step.params.inputColumn', function(nv, ov) {
        if (angular.isDefined(nv) && nv.length > 0 && $scope.step.isNew) {
            $scope.step.params.outputColumn = $scope.step.params.inputColumn + "_";
        }
    });

    $scope.$watch("step.params.refDateColumn", function(nv, ov) {
        $scope.dateColumnIsValid = isColumnValid(nv, "Date");
        if (nv != ov)
            $scope.processorForm.dateReferenceColumn.$setValidity('columnTypeInvalid', $scope.dateColumnIsValid);
    });

    $scope.$watch("step.params.refCurrencyColumn", function(nv, ov) {
        $scope.currencyColumnIsValid = isColumnValid(nv, "CurrencyMeaning");
        if (nv != ov)
            $scope.processorForm.currencyReferenceColumn.$setValidity('columnTypeInvalid', $scope.currencyColumnIsValid);
    });

    $scope.$watch("step.params.refDateCustom", function(nv, ov) {
        if (nv != ov) {
            var minDate = new Date("1999-01-04");
            /*
             * Date() constructor accepts yyyy or yyyy-MM or yyyy-MM-dd
             * We want to restrict user's entry to yyyy-MM-dd which explains
             * the Regex, checking XXXX-XX-XX, with X any number.
             * Date() constructor will check if it's a valid date
             * If it's not, then "date" will be false
             */
            var dateFormat = new RegExp("^[0-9]{4}(-[0-9]{2}){2}$").test(nv);
            var date = new Date(nv);
            $scope.processorForm.dateReferenceCustom.$setValidity('Date type invalid', dateFormat && date && date > minDate);
            $scope.outOfDateReference = (dateFormat && date && date < minDate);
        }
    });
});

app.controller('DateDifferenceController', function($scope) {
    // [Legacy Compatibility] DateDifference step parameters created before v14 are missing `calendar_id` and `timezone_id`.
    // This fix set the default values ("FR" for calendar_id and "use_preferred_timezone" for timezone_id) to match what is done in the backend
    if ($scope.step.params.calendar_id === undefined) {
        $scope.step.params.calendar_id = "FR";
    }
    if ($scope.step.params.timezone_id === undefined) {
        $scope.step.params.timezone_id = "use_preferred_timezone";
    }
});


app.controller('RegexpExtractorController', function($scope, $q, CreateModalFromTemplate) {
    $scope.validColumn = function() {
        return $scope.step.$stepState.change && $scope.step.$stepState.change.columnsBeforeStep.indexOf($scope.step.params.column) !== -1;
    };
    $scope.openSmartRegexBuilder = function() {
        let deferred = $q.defer();
        CreateModalFromTemplate("/templates/shaker/regexbuilder-box.html", $scope, "RegexBuilderController",
            function(newScope) {
                newScope.$apply(function() {
                    newScope.deferred = deferred;
                    newScope.columnName = $scope.step.params.column;
                    newScope.editStep = $scope.step;
                    newScope.calledFrom = "shaker_recipe-regexpextractor";
                    newScope.customPattern = newScope.createCustomPattern($scope.step.params.pattern);
                    newScope.computePatterns(false);
                });
                deferred.promise.then(function(newPattern) {
                    $scope.step.params.pattern = newPattern.regex;
                    $scope.step.params.extractAllOccurrences = newPattern.hasMultiOccurrences;
                    $scope.checkAndRefresh();
                });
            },
            "sd-modal");
    };
});

app.controller('GrokProcessorController', function($scope, $q, CreateModalFromTemplate) {
    $scope.validColumn = function() {
        return $scope.step.$stepState.change && $scope.step.$stepState.change.columnsBeforeStep.indexOf($scope.step.params.sourceColumn) !== -1;
    };
    $scope.openGrokDebugger = function() {
        let deferred = $q.defer();
        CreateModalFromTemplate("/templates/shaker/grokdebugger-box.html", $scope, "GrokDebuggerController",
            function(newScope) {
                newScope.$apply(function() {
                    newScope.deferred = deferred;
                    newScope.columnName = $scope.step.params.sourceColumn;
                    newScope.editStep = $scope.step;
                    newScope.calledFrom = "shaker_recipe-grokprocessor";
                    newScope.grokPattern = $scope.step.params.grokPattern || '';
                    newScope.debugColumns =  {...$scope.columns};
                    newScope.loadGrokExpressionSample();
                });
                deferred.promise.then(function(newGrokPattern) {
                    $scope.step.params.grokPattern = newGrokPattern;
                    $scope.checkAndRefresh();
                });
            },
            "sd-modal");
    };
});

app.controller('MultiColumnByPrefixFoldController', function($scope, $q, CreateModalFromTemplate) {
    $scope.openSmartRegexBuilder = function() {
        var deferred = $q.defer();
        CreateModalFromTemplate("/templates/shaker/regexbuilder-box.html", $scope, "RegexBuilderController",
            function(newScope) {
                newScope.deferred = deferred;
                newScope.$apply(function() {
                    newScope.onColumnNames = true;
                    newScope.editStep = $scope.step;
                    newScope.calledFrom = "shaker_recipe-multi_column_by_prefix_fold";
                    newScope.customPattern = newScope.createCustomPattern($scope.step.params.columnNamePattern);
                    newScope.computePatterns(false);
                });
                deferred.promise.then(function(newPattern) {
                    let columnNamePattern = newPattern.regex;
                    if (!columnNamePattern.startsWith('.*?')) {
                        columnNamePattern = ".*?" + columnNamePattern;
                    }
                    if (!columnNamePattern.endsWith('.*')) {
                        columnNamePattern = columnNamePattern + ".*";
                    }
                    $scope.step.params.columnNamePattern = columnNamePattern;
                    $scope.checkAndRefresh();
                });
            },
            "sd-modal");
    };
});


app.controller('PythonUDFController', function(
       $scope, $timeout, $rootScope, $q, $stateParams,
       CreateModalFromTemplate, CreateCustomElementFromTemplate, DataikuAPI, ShakerPopupRegistry, CodeMirrorSettingService) {

    var pythonSourceCodes = {

"CELL_true": "\
# Modify the process function to fit your needs\n\
import pandas as pd\n\
def process(rows):\n\
    # In 'cell' mode, the process function must return\n\
    # a single Pandas Series for each block of rows,\n\
    # which will be affected to a new column.\n\
    # The 'rows' argument is a dictionary of columns in the\n\
    # block of rows, with values in the dictionary being\n\
    # Pandas Series, which additionally holds an 'index'\n\
    # field.\n\
    return pd.Series(len(rows), index=rows.index)\n"
,
"ROW_true": "\
# Modify the process function to fit your needs\n\
import pandas as pd\n\
def process(rows):\n\
    # In 'row' mode, the process function \n\
    # must return the full rows.\n\
    # The 'rows' argument is a dictionary of columns in the\n\
    # block of rows, with values in the dictionary being\n\
    # Pandas Series, which additionally holds an 'index'\n\
    # field.\n\
    # You may modify the 'rows' in place to\n\
    # keep the previous values of the row.\n\
    # Here, we simply add two new columns.\n\
    rows[\"rowLength\"] = pd.Series(len(rows), index=rows.index)\n\
    rows[\"static_value\"] = pd.Series(42, index=rows.index)\n\
    return rows\n"
,
"MULTI_ROWS_true": "\
# Modify the process function to fit your needs\n\
import numpy as np, pandas as pd\n\
def process(rows):\n\
    # In 'multi rows' mode, the process function\n\
    # must return an indexed dictionary of vectors,\n\
    # either built by modifying the 'rows' \n\
    # parameter, or by returning a pandas DataFrame.\n\
    # To get an input dataframe, use\n\
    # rows.get_dataframe([col_name1, ...])\n\
    input_index = rows.index\n\
    # the values in the output index indicate which\n\
    # row of the input is used as base for the \n\
    # output rows. -1 signals that the new row comes\n\
    # from a blank base\n\
    new_index = np.concatenate([input_index, -1 * np.ones(input_index.shape[0])])\n\
    rows.index = new_index\n\
    # input columns are passed as pandas Series\n\
    existing_column_name = rows.columns[0]\n\
    existing_column = rows[existing_column_name]\n\
    rows[existing_column_name] = pd.concat([existing_column, existing_column])\n\
    # new columns can be numpy arrays\n\
    rows[\"static_value\"] = 42 * np.ones(2 * input_index.shape[0])\n\
    # the index field of the 'rows' parameter\n\
    # is a numpy array\n\
    rows[\"index\"] = np.concatenate([input_index, input_index])\n\
    return rows"
,
"CELL_false": "\
# Modify the process function to fit your needs\n\
def process(row):\n\
    # In 'cell' mode, the process function must return\n\
    # a single cell value for each row,\n\
    # which will be affected to a new column.\n\
    # The 'row' argument is a dictionary of columns of the row\n\
    return len(row)\n"
,
"ROW_false": "\
# Modify the process function to fit your needs\n\
def process(row):\n\
    # In 'row' mode, the process function \n\
    # must return the full row.\n\
    # The 'row' argument is a dictionary of columns of the row\n\
    # You may modify the 'row' in place to\n\
    # keep the previous values of the row.\n\
    # Here, we simply add two new columns.\n\
    row[\"rowLength\"] = len(row)\n\
    row[\"static_value\"] = 42\n\
    return row\n"
,
"MULTI_ROWS_false": "\
# Modify the process function to fit your needs\n\
def process(row):\n\
    # In 'multi rows' mode, the process function\n\
    # must return an iterable list of rows.\n\
    ret = []\n\
    # Here we append a new row with only one column\n\
    newrow1 = { \"previous_row_length\" : len(row) }\n\
    ret.append(newrow1)\n\
    # We can also modify the original row and reappend it\n\
    row[\"i\"] = 3\n\
    ret.append(row)\n\
    \n\
    return ret"
}

    $scope.editorOptions = CodeMirrorSettingService.get('text/x-python', {onLoad: function(cm) {$scope.codeMirror = cm}});

    $scope.$watch("[step.params.mode,step.params.vectorize, step.params.useKernel]", function(nv, ov) {
        if (!nv) return;
        // put defaults if they're not here already
        if ($scope.step.params.vectorize == null) {
            $scope.step.params.vectorize = false;
        }
        if ($scope.step.params.vectorSize == null) {
            $scope.step.params.vectorSize = 256;
        }
        const oldVectorized = ov[1] && ov[2];
        const newVectorized = nv[1] && nv[2];
        const oldDefaultPythonSourceCode = pythonSourceCodes[ov[0] + '_' + oldVectorized];
        const newDefaultPythonSourceCode = pythonSourceCodes[nv[0] + '_' + newVectorized];

        let oldPythonSource = $scope.step.params.pythonSourceCode;

        /* If we have already some code but it was the example code of the previous mode,
        then override it */
        if (oldPythonSource == oldDefaultPythonSourceCode) {
            oldPythonSource = null;
        }

        if ( (!oldPythonSource) || oldPythonSource.trim().length == 0) {
            $scope.step.params.pythonSourceCode = newDefaultPythonSourceCode;
        }
    }, true);

    $scope.parentDismiss = function() {
    	$rootScope.$broadcast("dismissModalInternal_");
    	$scope.modalShown = false;
    }
    $scope.$on("dismissModals", $scope.parentDismiss);

    $scope.hooks= {
            ok : function(){
                throw Error("not implemented");
            },
		    apply : function(){
		        throw Error("not implemented");
		    }
        };

    $scope.editPythonSource = function() {
        if ($scope.modalShown) {
            $scope.parentDismiss();
        } else {
            ShakerPopupRegistry.dismissAllAndRegister($scope.parentDismiss);
            $scope.modalShown = true;
            CreateCustomElementFromTemplate("/templates/shaker/pythonedit-box.html", $scope, null, function(newScope) {
                var pythonSource = $scope.step.params.pythonSourceCode;
                var mode = $scope.step.params.mode;
                var vectorize = $scope.step.params.vectorize;
                newScope.modified = false;
                if ( (!pythonSource) || pythonSource.trim().length == 0) {
                    newScope.pythonSource = pythonSourceCodes[mode + '_' + vectorize];
                    newScope.modified = true;
                }
                else {
                    newScope.pythonSource = $scope.step.params.pythonSourceCode;
                }

                $scope.hooks.apply = function() {
                	if ( newScope.modified ) {
                		$scope.step.params.pythonSourceCode = newScope.pythonSource;
                		$scope.checkAndRefresh();
                	}
                };
                $scope.hooks.ok = function() {
                	$scope.hooks.apply();
                	$scope.parentDismiss();
                };

                newScope.checkSyntax = function(pythonCode) {
                	var stepPosition = $scope.findStepId($scope.step);
                	DataikuAPI.shakers.validateUdf($stateParams.projectKey, $scope.inputDatasetProjectKey, $scope.inputDatasetName, $scope.shakerHooks.shakerForQuery()
                			, $scope.requestedSampleId, pythonCode, stepPosition.id, stepPosition.subId || 0, stepPosition.depth).success(function(data) {
                				newScope.validationError = data;
                			}).error(setErrorInScope.bind($scope));
                };

                $scope.uiState = { codeSamplesSelectorVisible: false };
                var insertCode = function (codeToInsert) {
                 	//timeout to make sure of an angular safe apply
                  	$timeout(function() {
                   		$scope.codeMirror.replaceSelection(codeToInsert + '\n\n', "around");
                   	});
                   	$scope.codeMirror.focus();
                };
                $scope.insertCodeSnippet = function(snippet) {
                    insertCode(snippet.code);
                };
            } , $scope.customFormulaEdition.displayCustomFormula);
        }
    };
});

/**
 * Formula function hinter directive
 *  @param {string}     name        - Name of the function
 *  @param {boolean}    applied     - Is function applied to column
 *  @param {string}     arguments   - Argument list as string, separated by ","
 *  @param {string}     description - Function description
 *  @param {string}     deprecated  - Is function deprecated
 *  @param {string}     deprecationNotice - Optional additional message displayed when function is deprecated
 *  @param {number}     left        - Absolute CSS position left
 *  @param {number}     top         - Absolute CSS position top
 *  @param {number}     cursor      - Cursor position in the current line
 *  @param {string}     line        - Line of code user actually typing into
 *  @param {number}     start       - Position of the function token in the code line
 */
app.directive('formulaFunctionHinter', function($compile, translate) {
    return {
        template: ``,
        restrict :'E',
        scope : {
            name: "<",
            applied: "<",
            arguments: "<",
            deprecated: "<",
            deprecationNotice: "<",
            description: "<",
            examples: "<",
            left: "<",
            top: "<",
            cursor: '<',
            line: '<',
            start: '<'
        },
        link: function($scope) {
            $scope.translate = translate;
            const removeAnchor = () => {
                const prevAnchor = document.getElementById('formula-function-hinter');
                if(prevAnchor) {
                    prevAnchor.parentNode.removeChild(prevAnchor);
                }
            }

            removeAnchor();
            $scope.$on('$destroy', removeAnchor);

            const formatExamples = function(name, examples) {
                if (!examples) {
                    return "";
                } 
                return examples.map((example) =>
                        `<div class="formula-tooltip__example">
                            <code>${name}(${example.args.join(", ")})</code> returns <code>${example.result}</code>
                        </div>`).join("");
            }

            // Element need to be placed at root level to avoid overflowing issue due to editor container
            $scope.htmlExamples = formatExamples($scope.name, $scope.examples);
            const body = angular.element(document.body);
            const anchor = angular.element(`
                <div class="formula-tooltip" id="formula-function-hinter" ng-style="getStyle()">
                    <strong>{{name}}</strong>(<span ng-bind-html="argsString"></span>)
                    <div ng-if="deprecated" class="formula-tooltip__deprecated">
                        {{::translate('FORMULA.DEPRECATED', '[DEPRECATED]')}}
                        <span ng-if="deprecationNotice" ng-bind-html="deprecationNotice"></span>
                    </div>
                    <p class="formula-tooltip__description" ng-bind-html="description"></p>
                    <span ng-bind-html="htmlExamples"></span>
                </div>
            `);
            body.append(anchor);
            $compile(anchor)($scope);
            
            $scope.getStyle = () => ({
                top: $scope.top + 'px',
                left: $scope.left + 'px'
            });

            const getCurrentArgumentIndex = () => {
                let part = $scope.line.substr($scope.start);
                let pos = 0;
                let parentheses = 0;
                let brackets = 0;
                let quotes = 0;
                let sglQuotes = 0;
                let index = 0;
                while(pos + $scope.start < $scope.cursor) {
                    const char = part.substr(pos, 1);
                    if(char === '\\') {
                        // Escaping character, so we should also skip next one
                        pos += 2;
                        continue;
                    }
                    
                    if(char === ',' && parentheses === 0 && brackets === 0 && sglQuotes % 2 === 0 && quotes % 2 === 0) {
                        // We are not between parentheses or quotes, we can increment the count
                        index ++;
                    } else if(char === '(') {
                        parentheses ++;
                    } else if(char === ')') {
                        parentheses --;
                    } else if(char === '[') {
                        brackets ++;
                    } else if(char === ']') {
                        brackets --;
                    } else if(char === '"' && sglQuotes % 2 === 0) {
                        quotes ++;
                    } else if(char === "'" && quotes % 2 === 0) {
                        sglQuotes --;
                    }
                    pos ++;
                }
                return index;
            };

            $scope.argsString = '';

            $scope.$watch('[arguments,cursor,line,start]', () => {
                const index = getCurrentArgumentIndex();
                $scope.argsString = $scope.arguments.split(', ')
                    .filter((_, id) => id > 0 || !$scope.applied)
                    .map((arg, id) => id === index ? `<strong>${arg}</strong>` : arg)
                    .join(', ');
            });

            $scope.$watch('examples', nv => {
                $scope.htmlExamples = formatExamples($scope.name, nv);
            })
        }       
    };
});

/**
 * Grel formula editor
 *  @param {string}    expression          - Current formula expression
 *  @param {array}     columns             - Array of column objects, formatted as { name: string, type: string, meaning?: string, comment?: string }
 *  @param {object}    recipe              - The recipe object the editor is associated with, if any
 *  @param {object}    scopeVariables      - If set, this will replace the variables coming from API. Must be a key-value pair dictionary.
 *  @param {function}  validator           - Function validating the expression (must return a MonoFuture promise-like object with a success method)
 *  @param {function}  onValidate          - Event fired after complete validation
 *  @param {function}  onExpressionChange  - Event fired when expression changes
 */
app.directive('grelEditor', function($timeout, $stateParams, $filter, $sanitize, DataikuAPI, CachedAPICalls, Debounce, Logger, translate) {
    return {
        template: `<div class="editor-tooltip-anchor h100">
                        <formula-function-hinter
                            ng-if="tooltip.shown"
                            description="tooltip.description"
                            applied="tooltip.applied"
                            deprecated="tooltip.deprecated"
                            deprecation-notice="tooltip.deprecationNotice"
                            left="tooltip.left"
                            top="tooltip.top"
                            arguments="tooltip.arguments"
                            name="tooltip.name"
                            examples="tooltip.examples"
                            line="tooltip.line"
                            cursor="tooltip.cursor"
                            start="tooltip.tokenStart" />
                        <textarea class="h100"></textarea>
                    </div>`,
        restrict :'E',
        replace: true,
        scope : {
            expression: "=",
            columns: "<",
            recipe: "<?",
            scopeVariables: "=?",
            validator: "<",
            onValidate: "<",
            onExpressionChange: "<",
            onError: "<",
            generationsTracker: "<",
            autoFocus: "<?",
            mode: "@",
            onEscape: "&"
        },
        link: function($scope, element) {
            const getRecipeVariables = () => {
                if ($scope.recipe) {
                    return DataikuAPI.flow.recipes.generic.getVariables($scope.recipe);
                }
                return DataikuAPI.flow.recipes.generic.getVariablesWithProjectKey($stateParams.projectKey);
            };

            $scope.tooltip = {
                shown: false,
                applied: false,
                left: 0,
                top: 0,
                arguments: '',
                deprecated: false,
                deprecationNotice: null,
                description: '',
                examples: '',
                name: '',
                line: '',
                cursor: 0,
                tokenStart: 0
            }

            $scope.translate = translate;

            $scope.$watch('scopeVariables', () => {
                if($scope.scopeVariables && Object.keys($scope.scopeVariables).length > 0) {
                    const newVars = {};
                    Object.keys($scope.scopeVariables).forEach(key => {
                        newVars[`variables["${key}"]`] = { value: $scope.scopeVariables[key] };
                        newVars[`\${${key}}`] = { value: $scope.scopeVariables[key] };
                    });
                    $scope.variables = newVars;
                }
            });

            $scope.variables = [];
            getRecipeVariables().success(data => {
                const newVars = {};
                Object.keys(data).forEach(key => {
                    newVars[`variables["${key}"]`] = { value: data[key] };
                    newVars[`\${${key}}`] = { value: data[key] };
                });
                $scope.variables = {
                    ...newVars,
                    ...$scope.variables,
                };
            });

            let helpers = [];
            const formulasReference = $scope.mode === 'udaf' ? CachedAPICalls.udafCustomFormulasReference : CachedAPICalls.customFormulasReference;
            formulasReference.then(data => helpers = data);

            const columnTypeMapping = {
                string: ['String', 'icon-dku-string', 'cm-string'],
                int: ['Integer', 'icon-dku-hexa_view', 'cm-number'],
                double: ['Double', 'icon-dku-hexa_view', 'cm-number'],
                float: ['Float', 'icon-dku-hexa_view', 'cm-number'],
                tinyint: ['Tiny int (8 bits)', 'icon-dku-hexa_view', 'cm-number'],
                smallint: ['Small int (16 bits)', 'icon-dku-hexa_view', 'cm-number'],
                bigint: ['Big int (64 bits)', 'icon-dku-hexa_view', 'cm-number'],
                boolean: ['Boolean', 'icon-dku-true-false', 'cm-bool'],
                date: ['Datetime with tz', 'icon-calendar', 'cm-date'],
                dateonly: ['Date only', 'icon-calendar', 'cm-date'],
                datetimenotz: ['Datetime no tz', 'icon-calendar', 'cm-date'],
                geopoint: ['Geo Point', 'icon-globe', 'cm-date'],
                geometry: ['Geometry/Geography', 'icon-globe', 'cm-date'],
                array: ['Array', 'icon-dku-array', 'cm-date'],
                object: ['Complex object', 'icon-dku-object', 'cm-date'],
                map: ['Map', 'icon-dku-map', 'cm-date'],
                unspecified: ['', 'icon-dku-column', 'cm-table']
            };

            const getColumnNames = () => {
                return $scope.columns.map(c => c.name || c);
            };

            const helperScrollHandler = (e) => {
                const elems = document.getElementsByClassName('helper-display');
                for(let i = 0; i < elems.length; i++) {
                    elems[i].style.top = e.target.scrollTop + 'px';
                }
            };

            const escapeHtml = (html) => {
                return $filter('escapeHtml')(html);
            }
            const sanitizeHtml = (html) => {
                return $sanitize(html);
            }
        
            const hintRenderer = (hint, value, index, type) => {
                return (elt) => {
                    if(!hint || hint === '') {
                        return;
                    }

                    let icon = 'icon-dku-function cm-function';
        
                    const helperElement = document.createElement('div');
                    helperElement.className = 'helper-display';


                    const helper = helpers.find(h => h.name === hint);
                    const helperFooter = '<div class="helper-tip">' + $scope.translate('FORMULA.TAB_TO_COMPLETE', 'Hit tab to complete') + '<div style="float: right;">' + $scope.translate('FORMULA.ESC_TO_HIDE', 'Hit esc to hide') + '</div></div>';
                    if(helper && type==='function') {
                        let htmlTitle = `<div class="helper-title"><strong>${escapeHtml(helper.name)}(${escapeHtml(helper.params)}) ${escapeHtml(helper.returns) || ''}</strong></div>`
                        let htmlDescription = `<div class="helper-description"><p>${sanitizeHtml(helper.description)}</p>`;
                        if (helper.examples) {
                            htmlDescription += helper.examples.map((example) =>
                                `<div class="helper-example">
                                        <code>${escapeHtml(helper.name)}(${escapeHtml(example.args.join(", "))})</code>${$scope.translate('FORMULA.RETURNS', ' returns ')}<code>${escapeHtml(example.result)}</code>
                                    </div>`).join("");
                        }
                        htmlDescription += "</div>";
                        helperElement.innerHTML = htmlTitle + htmlDescription + helperFooter;
                    } else if(Object.keys($scope.variables).includes(hint) && type === 'variable') {
                        const value = escapeHtml(typeof $scope.variables[hint].value === 'object' ?
                                        JSON.stringify($scope.variables[hint].value, null, 2) :
                                        $scope.variables[hint].value);
                        helperElement.innerHTML = `<div class="helper-title">`
                                                        + `<strong>${value}</strong>`
                                                        + `</div>`
                                                        + helperFooter;
                        icon = 'icon-dku-variable cm-variable';
                    } else if(getColumnNames().includes(hint) && type === 'column') {
                        const col = $scope.columns.find(c => c.name === hint || c === hint);
                        const ref = columnTypeMapping[col && col.type ? col.type : 'unspecified'] || ['icon-dku-string', 'cm-default'];
                        if(col) {
                            helperElement.innerHTML = `<div class="helper-title">`
                                                        + (ref[0] ? `${ $scope.translate('FORMULA.TYPE', 'Type: ') }<strong>${ref[0]}</strong>` : '')
                                                        + (col.meaning ? `<br />${ $scope.translate('FORMULA.MEANING', 'Meaning: ') }<strong>${escapeHtml(col.meaning)}</strong>` : '')
                                                        + (col.comment ? `<p>${escapeHtml(col.comment)}</p>` : '<p class="text-weak">' + $scope.translate('FORMULA.NO_DESC', 'No description provided.') + '</p>')
                                                        + `</div>`
                                                        + helperFooter;
                        }
                        icon = `${ref[1]} ${ref[2]}`;
                    }
        
                    elt.innerHTML = `<i class="${icon}"></i>&nbsp;<strong>${hint.substr(0, value.length)}</strong>${hint.substr(value.length)}`;
                    elt.appendChild(helperElement);
        
                    if(index === 0) {
                        elt.parentElement.id = "qa_formula_auto_complete";
                        elt.parentElement.removeEventListener('scroll', helperScrollHandler);
                        elt.parentElement.addEventListener('scroll', helperScrollHandler);
                    }
                }
            }

            const isDeprecated = (hint) => {
                const helper = helpers.find(h => h.name === hint);
                return helper && helper.deprecated;
            }

            const autoCompleteHandler = (hint) => {
                const isColumn = hint.type === 'column' && getColumnNames().includes(hint.text);
                const isFunction = hint.type === 'function' && !Object.keys($scope.variables).includes(hint.text);
                
                // When column has a complex name, the autocompletion should be wrapped inside val('')
                if (isColumn && !hint.text.match(/^[a-z0-9_]+$/i)) {
                    const doc = $scope.cm.getDoc();
                    const cursor = doc.getCursor();
                    const leftPart = cm.getLine(cursor.line).substr(0, cursor.ch - hint.text.length);
                    if(leftPart.endsWith('val(')) {
                        doc.setSelection({line: cursor.line, ch: cursor.ch - hint.text.length});
                        doc.replaceSelection('"');
                        doc.setSelection({line: cursor.line, ch: cursor.ch + 1 });
                        doc.replaceSelection('"');
                    } else if(leftPart.endsWith('val("') || leftPart.endsWith('val(\'')) {
                        doc.setSelection({line: cursor.line, ch: cursor.ch - hint.text.length});
                        doc.setSelection({line: cursor.line, ch: cursor.ch });
                    } else {
                        doc.setSelection({line: cursor.line, ch: cursor.ch - hint.text.length});
                        doc.replaceSelection('val("');
                        doc.setSelection({line: cursor.line, ch: cursor.ch + 5 });
                        doc.replaceSelection('")');
                    }
                }
                else if(isFunction) {
                    const doc = $scope.cm.getDoc();
                    const cursor = doc.getCursor();
                    const line = cm.getLine(cursor.line);
                    if (line.length < cursor.ch || line[cursor.ch] !== '(') {
                        doc.setSelection({ line: cursor.line, ch: cursor.ch });
                        doc.replaceSelection('()');
                    }
                    cursor.ch = cursor.ch + 1;
                    doc.setCursor(cursor);
                }
                else if (!isColumn && !isFunction && hint.text.endsWith('}')) {
                    const doc = $scope.cm.getDoc();
                    const cursor = doc.getCursor();
                    const nextChar = cm.getLine(cursor.line).substr(cursor.ch, 1);
                    if(nextChar === '}') {
                        doc.replaceRange('', { ...cursor }, { line: cursor.line, ch: cursor.ch + 1 });
                    }
                }

                $scope.validateGRELExpression();
            }

            const getLineQuoteState = (line, pos) => {
                let quoteOpened = false;
                let doubleQuoteOpened = false;
                for(let i = 0; i < pos - 1; i++) {
                    if(line[i] == "\\") {
                        i ++;
                    } else if(line[i] === "'" && !doubleQuoteOpened) {
                        quoteOpened = !quoteOpened;
                    } else if(line[i] === '"' && !quoteOpened) {
                        doubleQuoteOpened = !doubleQuoteOpened;
                    }
                }
                return { quoteOpened, doubleQuoteOpened };
            }

            const autoClosingPairs = {'(': ')', '"': '"', "'": "'", '[': ']', '{': '}'};
            const autoCloseCharacter = (cm, key, event) => {
                const doc = cm.getDoc();
                const cursor = doc.getCursor();
                const line = cm.getLine(cursor.line);
                const nextChar = line[cursor.ch];
                const selection = doc.getSelection();
        
                if(nextChar === key) {
                    doc.replaceRange('', { ...cursor }, { line: cursor.line, ch: cursor.ch + 1 });
                    return;
                }

                if(selection.length === 0 && (key === '"' || key === "'") && ![undefined, ']', ')', '}'].includes(nextChar)) {
                    return;
                }

                // Check if we are not currently inside a string definition
                const quoteStatus = getLineQuoteState(line, cursor.ch);
                if(!quoteStatus.doubleQuoteOpened && !quoteStatus.quoteOpened) {
                    if(selection.length > 0) {
                        const endCursor = doc.getCursor(false);
                        doc.replaceSelection(key + selection + autoClosingPairs[key]);
                        doc.setCursor({...endCursor, ch: endCursor.ch + 2});
                    } else {
                        doc.replaceSelection(key + autoClosingPairs[key]);
                        doc.setCursor({...cursor, ch: cursor.ch + 1});
                    }
                    event.preventDefault();
                    event.stopPropagation();
                }
            }
        
            const autoCloseCharacterRemover = (cm) => {
                const doc = cm.getDoc();
                const cursor = doc.getCursor();
                const line = cm.getLine(cursor.line);
                const deletedChar = line[cursor.ch -1];
                const nextChar = line[cursor.ch];
        
                // Check if we are not currently inside a string definition
                const quoteStatus = getLineQuoteState(line, cursor.ch);
                if(quoteStatus.doubleQuoteOpened || quoteStatus.quoteOpened) {
                    return;
                }
        
                if(autoClosingPairs[deletedChar] && autoClosingPairs[deletedChar] === nextChar) {
                    doc.replaceRange('', { ...cursor }, { line: cursor.line, ch: cursor.ch + 1 });
                }
            }

            const getTokenBeforeCursor = (cm) => {
                const cursor = cm.doc.getCursor();
                const tokens = cm.getLineTokens(cursor.line);
                let token = [];
                let parenthese = [];
                let isAppliedToColumn = false;
                for(let i = 0; i < tokens.length; i++) {
                    if(tokens[i].end < cursor.ch + 1) {
                        if(tokens[i].type === 'builtin') {
                            token.push(tokens[i]);
                            if(i > 0 && tokens[i-1].string === '.') {
                                isAppliedToColumn = true;
                            } else {
                                isAppliedToColumn = false;
                            }
                        } else if(tokens[i].string === '(') {
                            parenthese.push(tokens[i].end);
                        } else if(tokens[i].string === ')') {
                            token.pop();
                            parenthese.pop();
                        }
                    }
                }
                if(token.length > 0 && parenthese.length > token.length - 1) {
                    $scope.tooltip.tokenStart = parenthese[token.length - 1];
                    return {...token[token.length - 1], isAppliedToColumn };
                }
                return  null;
            };

            const mode = angular.isDefined($scope.mode) ? $scope.mode : 'grel';
            const editorOptions = {
                value: $scope.expression || "",
                mode: `text/${mode}`,
                theme:'elegant',
                variables: getColumnNames,
                lineNumbers : false,
                lineWrapping : true,
                autofocus: angular.isDefined($scope.autoFocus) ? $scope.autoFocus : true,
                hintOptions: {
                    hint: (editor) => {
                        const grelWordHints = CodeMirror.hint[mode]($scope.cm, { columns: getColumnNames, variables: Object.keys($scope.variables), completeSingle: false });
                        const words = grelWordHints.list;

                        let cursor = editor.getCursor();
                        let curLine = editor.getLine(cursor.line);
                        let start = cursor.ch;
                        let end = start;
                        while (end < curLine.length && /[\w\p{L}$]/u.test(curLine.charAt(end))) ++end;
                        while (start && (/[\w\p{L}.$]/u.test(curLine.charAt(start - 1)) || curLine.charAt(start - 1) == '{')) --start;
                        let curWord = start !== end ? curLine.slice(start, end) : '';
                        // The dot should only be considered a token separator when outside of the ${...} variable syntax
                        const firstDot = curWord.indexOf('.');
                        if(!curWord.startsWith('$') && firstDot > -1) {
                            curWord = curWord.substr(firstDot + 1);
                            start += firstDot + 1;
                        }

                        const list = (!curWord ? words : words.filter(word => {
                            return word.text.toLowerCase().startsWith(curWord.toLowerCase());
                        }))
                        list.sort((a, b) => {
                            const malusA = Object.keys($scope.variables).includes(a.text) ? 2 : getColumnNames().includes(a.text) ? 1 : 0;
                            const malusB = Object.keys($scope.variables).includes(b.text) ? 2 : getColumnNames().includes(b.text) ? 1 : 0;
                            if(malusA === malusB)
                                return a.text - b.text;
                            return malusA - malusB;
                         });

                        const data = {
                            list: list.filter(item => !isDeprecated(item.text)).map((item, index) => ({ text: item.text, type: item.type, render: hintRenderer(item.text, curWord, index, item.type)})),
                            from: CodeMirror.Pos(cursor.line, start),
                            to: CodeMirror.Pos(cursor.line, end)
                        };

                        CodeMirror.on(data, 'pick', autoCompleteHandler);
                            
                        return data;
                    },
                    completeSingle: false
                }
            };

            const prevCursor = { line: 0, ch: 0 };
            const textarea = element.find('textarea')[0];
            const cm = CodeMirror.fromTextArea(textarea, editorOptions);

            $scope.$watch('expression', (newValue, oldValue) => {
                const cursor = cm.doc.getCursor();
                cm.setValue(newValue || '');
                cm.setCursor(cursor);
            });

            $scope.cm = cm;
            cm.on("keydown", function(cm, evt) {
                if(evt.key === "Backspace") { // Backspace
                    autoCloseCharacterRemover(cm);
                }
                else if(Object.keys(autoClosingPairs).includes(evt.key)) {
                    autoCloseCharacter(cm, evt.key, evt);
                }
                if (evt.key === 'Escape' && $scope.onEscape) {
                    $scope.onEscape({state: {completion: cm.state.completionActive && cm.state.completionActive.widget}});
                }
            });

            cm.on("keyup", function(cm, evt) {
                /* Ignore tab, esc, and navigation/arrow keys */
                if (evt.key === "Tab" || evt.key === 'Escape' || (evt.keyCode>= 33 && evt.keyCode <= 40)) {
                    if(evt.key === "Tab") {
                        // We force the expression validation on tab, as autocompletion does not trigger smart change
                        $scope.validateGRELExpression();
                    }
                    CodeMirror.signal(cm, "endCompletion", cm);
                }
                else if(evt.key !== "Enter") {
                    CodeMirror.commands.autocomplete(cm, null, {
                        columns: $scope.columns,
                        completeSingle: false
                    });
                }
            });

            cm.on('cursorActivity', () => {
                const parent = cm.getTextArea().closest('.editor-tooltip-anchor');
                if(!parent) {
                    return;
                }

                const coords = cm.cursorCoords();
                $scope.tooltip.left = coords.left;
                $scope.tooltip.top = coords.top + 16;
                $scope.tooltip.shown = false;

                const doc = cm.getDoc();
                const cursor = doc.getCursor();
                const token = getTokenBeforeCursor(cm);
                const leftPart = cm.getLine(cursor.line).substr(0, cursor.ch);
                if(token && !leftPart.endsWith('val("') && !leftPart.endsWith('val(\'')) {
                    const helper = helpers.find(h => h.name === token.string);
                    if(helper) {
                        $scope.tooltip.shown = true;
                        $scope.tooltip.name = helper.name;
                        $scope.tooltip.arguments = helper.params;
                        $scope.tooltip.deprecated = helper.deprecated;
                        $scope.tooltip.deprecationNotice = helper.deprecationNotice;
                        $scope.tooltip.description = helper.description;
                        $scope.tooltip.examples = helper.examples;
                        $scope.tooltip.line = cm.getLine(cursor.line);
                        $scope.tooltip.cursor = cursor.ch;
                        $scope.tooltip.applied = token.isAppliedToColumn;
                    }
                }

                // We do not want to auto display the autocomplete hinter if the user is just moving to the next line
                // Or if the completion has just ended
                if (cursor.line === prevCursor.line && cm.state.completionActive !== null) {
                    CodeMirror.commands.autocomplete(cm, null, {
                        columns: $scope.columns,
                        completeSingle: false
                    });
                }
                safeApply($scope);

                prevCursor.ch = cursor.ch;
                prevCursor.line = cursor.line;
            });
        
            let errorMarker = null;
            const highlightOffset = (offset) => {
                let current = 0;
                const lineCount = $scope.cm.lineCount();
                for(let i = 0; i < lineCount; i++) {
                    const line = $scope.cm.doc.getLine(i);
                    if(current + line.length > offset) {
                        errorMarker = $scope.cm.doc.markText(
                            { line: i, ch: offset - current },
                            { line: i, ch: offset - current + 1 },
                            { className: 'error-hint', clearOnEnter: true }
                        );
                        return;
                    }
                    current += line.length + 1; // CrLf is counted as 1 offset
                }
            }

            $scope.fireExpressionChange = () => {
                if($scope.onExpressionChange) {
                    $scope.onExpressionChange($scope.expression);
                }
                $scope.validateGRELExpression($scope.expression);
            };
            const debouncedFireExpressionChange = Debounce().withDelay(0, 500).wrap($scope.fireExpressionChange);
        
            $scope.validateGRELExpression = () => {
                const validateStart = performance.now();
                const expr = cm.getValue();
                const contentGenerationBeingValidated = $scope.generationsTracker ? $scope.generationsTracker.currentContentGeneration : 0;
                Logger.info("Starting to validate expression at generation ", contentGenerationBeingValidated);

                $scope.examples = null;

                if($scope.validator) {

                    // prevent from calling the backend if the formula is not complete
                    if (expr.trim().length == 0) {
                        $scope.onValidate({ valid: false, error: false, data: {}, inputExpr: expr });
                        return;
                    }

                    $scope.validator(expr).success((futureResult) => {
                        const data = futureResult.result;

                        const afterBackend = performance.now();
                        Logger.info("Timing: grelEditor.validateBackend: " + (afterBackend-validateStart) + "ms");

                        if($scope.onValidate) {
                            $scope.onValidate({ valid: data.ok, error: data.message, data, inputExpr: expr });
                        }
    
                        if(errorMarker != null) {
                            errorMarker.clear();
                        }
            
                        const matches = (data.message || '').match(/at offset (\d+)/i);
                        if(matches != null) {
                            highlightOffset(matches[1])
                        }

                        const afterApply = performance.now();
                        Logger.info("Timing: grelEditor.totalPerceivedValidationLatency: " + (afterApply-lastCodeMirrorChange) + "ms");

                        if ($scope.generationsTracker && data.ok) {
                            $scope.generationsTracker.validatedContentGeneration = contentGenerationBeingValidated;
                        }

                    }).error($scope.onError ? $scope.onError : setErrorInScope.bind($scope));
                }
            };

            $scope.$on('codemirror-focus-input', () => {
                cm.focus();
            });

            let lastCodeMirrorChange = 0;

            cm.on('change', () => {
                if ($scope.generationsTracker) {
                    $scope.generationsTracker.currentContentGeneration++;
                }
                lastCodeMirrorChange = performance.now();
                $scope.expression = cm.getValue();
                debouncedFireExpressionChange();
            });
        }
    };
});



app.controller('FormulaAwareProcessorController', function($scope, $stateParams, $rootScope, DataikuAPI,
               CreateCustomElementFromTemplate, ShakerPopupRegistry, MonoFuture, DeducedMonoFuture, PrettyPrintDoubleService) {

    $scope.editing = {}
    $scope.columns = [];

    const canGetColumnDetails = () => {
        const stepPosition = $scope.shaker.steps.indexOf($scope.step);

        // If the current step is the last one or if it is in preview mode or if the steps after it are disabled.
        return $scope.shaker && $scope.shaker.steps && $scope.shaker.steps[$scope.shaker.steps.length - 1] === $scope.step 
            || $scope.step.preview === true
            || (!$scope.step.disabled && $scope.shaker.steps.slice(stepPosition + 1).every(step => step.disabled))
    } 

    const computeColumns = () => {
        if (canGetColumnDetails()) {
            $scope.columns = $scope.quickColumns.map(c => ({
                ...(c.recipeSchemaColumn ? c.recipeSchemaColumn.column : {}),
                name: c.name,
                meaning: c.meaningLabel || '',
                comment: c.comment || ''
            }));
        } else {
            $scope.columns = $scope.step.$stepState.change ? $scope.step.$stepState.change.columnsBeforeStep : [];
        }
    }

    const stopWatchingForColumnsCompute = $scope.$watch('[quickColumns, step.preview, shaker.steps]', () => {
        computeColumns();
    });

    let validateExpressionMonoFuture = DeducedMonoFuture($scope, false, 250).wrap(DataikuAPI.shakers.validateExpression);
    $scope.expressionValidator = (expression) => {
    	const stepPosition = $scope.findStepId($scope.step);
        return validateExpressionMonoFuture(
            $stateParams.projectKey,
            $scope.inputDatasetProjectKey,
            $scope.inputDatasetName,
            $scope.shakerHooks.shakerForQuery(),
            $scope.requestedSampleId,
            expression,
            stepPosition.id,
            stepPosition.subId || 0,
            stepPosition.depth,
            true,
            false
        );
    };

    $scope.generationsTracker = {
        currentContentGeneration: 0,
        validatedContentGeneration: -1
    }

    $scope.fixupFormula = (expression, fixName = "plus") => {
        const stepPosition = $scope.findStepId($scope.step);
        return DataikuAPI.shakers.fixExpression(
            $stateParams.projectKey,
            $scope.inputDatasetProjectKey,
            $scope.inputDatasetName,
            $scope.shakerHooks.shakerForQuery(),
            $scope.requestedSampleId,
            expression,
            fixName,
            stepPosition.id,
            stepPosition.subId || 0,
            stepPosition.depth).then(data => {
                $scope.editing.expression = data.data
            }, setErrorInScope.bind($scope));
    }

    $scope.onValidate = (result) => {
        $scope.changeInProgress = false;
        $scope.grelExpressionValid = result.valid;
        $scope.grelExpressionError = result.error;        
        $scope.grelExpressionEmpty = result.inputExpr.trim().length == 0;

        if (result.valid) {
            $scope.grelExpressionData = result.data;
        }

        if($scope.grelExpressionData && $scope.grelExpressionData.table) {
            PrettyPrintDoubleService.patchFormulaPreview(
                $scope.grelExpressionData.table,
                'double', // true output type is unknown but we want the patch to happen for scientific notations values
                $scope.columns.map(c => ({...c, type: c.meaning === 'Decimal' ? 'double' : ''}))
            )
        }
    };

    const checkUnsavedFormulaChanges = () => {
        if ($scope.modalShown && $scope.step.$stepState) {
            $scope.step.$stepState.containsUnsavedFormulaChanges = !angular.equals($scope.step.params.expression, $scope.editing.expression) || !angular.equals($scope.step.params.column, $scope.editing.outputColumnName);
        }
    };

    $scope.onFormulaColumnChange = () => {
        checkUnsavedFormulaChanges();
    };

    $scope.onExpressionChange = (expression) => {
        $scope.changeInProgress = true;
        checkUnsavedFormulaChanges();
    };

    $scope.parentDismiss = function() {
        $rootScope.$broadcast("dismissModalInternal_");
        $scope.modalShown = false;
        if ($scope.step.$stepState) {
            delete $scope.step.$stepState.containsUnsavedFormulaChanges;
        }
    }
    $scope.$on("dismissModals", $scope.parentDismiss);
    $scope.grelExpressionError = false;
    $scope.grelExpressionValid = false;    
    $scope.grelExpressionEmpty = true;
    $scope.changeInProgress = false;

    $scope._edit = function(expression, outputColumnName = null) {
        if ($scope.modalShown) {
            $scope.parentDismiss();
        } else {
            ShakerPopupRegistry.dismissAllAndRegister($scope.parentDismiss);
            $scope.modalShown = true;
            CreateCustomElementFromTemplate("/templates/shaker/formula-editor.html", $scope, null, function(newScope) {
                $scope.editing.expression = expression;
                $scope.editing.outputColumnName = outputColumnName;
                $scope.editing.showColumnNameEditor = outputColumnName !== null; // not all processors have an output column name concept
            }, $scope.customFormulaEdition.displayCustomFormula);
        }
    }


    $scope.hooks= {
        ok : function(){
            throw Error("not implemented");
        }
    }

    $scope.$on("$destroy", stopWatchingForColumnsCompute);
});


app.controller('CreateColumnWithGRELController', function($scope, $controller) {
    $controller("FormulaAwareProcessorController", {$scope:$scope});

    if (angular.isUndefined($scope.step.params.column)) {
        $scope.step.params.column = "formula_result";
    }
    if (angular.isUndefined($scope.step.params.expression)) {
        $scope.step.params.expression = "";
    }
    if (angular.isUndefined($scope.step.params.errorColumn)) {
        $scope.step.params.errorColumn = "";
    }
    $scope.mode = "create";

    $scope.hooks.ok = function() {
        $scope.step.params.expression = $scope.editing.expression;
        $scope.step.params.column = $scope.editing.outputColumnName || '';
        $scope.editing.expression = null;
        $scope.checkAndRefresh();
        $scope.parentDismiss();
    }

    $scope.edit = function() {
        $scope._edit(angular.copy($scope.step.params.expression), $scope.step.params.column);
    }

    $scope.storageTypes = [
    	[null, 'None'],
        [{name:'foo', type:'string'}, 'String'],
        [{name:'foo', type:'int'}, 'Integer'],
        [{name:'foo', type:'double'}, 'Double'],
        [{name:'foo', type:'float'}, 'Float'],
        [{name:'foo', type:'tinyint'}, 'Tiny int (8 bits)'],
        [{name:'foo', type:'smallint'}, 'Small int (16 bits)'],
        [{name:'foo', type:'bigint'}, 'Big int (64 bits)'],
        [{name:'foo', type:'boolean'}, 'Boolean'],
        [{name:'foo', type:'date'}, 'Datetime with tz'],
        [{name:'foo', type:'dateonly'}, 'Date only'],
        [{name:'foo', type:'datetimenotz'}, 'Datetime no tz'],
        [{name:'foo', type:'geopoint'}, "Geo Point"],
        [{name:'foo', type:'geometry'}, "Geometry/Geography"]
    ];

    // automatically open the panel on step creation
    if ($scope.step.$stepState.justCreated === true) {
        $scope.edit();
    }
});

app.controller("CustomJythonProcessorController", function($scope, Assert, DataikuAPI, WT1, $stateParams, TopNav, PluginConfigUtils, Logger){
    $scope.loadedDesc = $scope.appConfig.customJythonProcessors.find(x => x.elementType == $scope.step.type);
    Assert.inScope($scope, 'loadedDesc');
    $scope.pluginDesc = $scope.appConfig.loadedPlugins.find(x => x.id == $scope.loadedDesc.ownerPluginId);
    Assert.inScope($scope, 'pluginDesc');

    // Make a copy to play with roles
    $scope.desc = angular.copy($scope.loadedDesc.desc);
    $scope.desc.params.forEach(function(param) {
        if (param.type == "COLUMNS" || param.type == "COLUMN") {
            param.columnRole = "main";
        }
    });

    if (!$scope.step.params) {
        $scope.step.params = {}
    }
    if (!$scope.step.params.customConfig){
        $scope.step.params.customConfig = {}
    }

    var getCurrentColumns = function() {
        // step info are loaded asynchronously
        if ($scope.step.$stepState.change) {
            return $scope.step.$stepState.change.columnsBeforeStep.map(colName => ({name: colName, type: 'STRING'}));
        } else {
            return [];
        }
    }

    $scope.columnsPerInputRole = {
        "main" : []
    };

    $scope.$watch("step", function(nv, ov) {
        if (nv && nv.$stepState) {
            $scope.columnsPerInputRole = {
                main: getCurrentColumns()
            };
        }
    }, true);

    PluginConfigUtils.setDefaultValues($scope.desc.params, $scope.step.params.customConfig);
});


app.controller('FilterOnCustomFormulaController', function($scope, $controller, $stateParams, $rootScope, DataikuAPI, CreateCustomElementFromTemplate, ShakerPopupRegistry) {
    $controller("FormulaAwareProcessorController", {$scope:$scope});

    $scope.hooks.ok = function() {
        $scope.step.params.expression = $scope.editing.expression;
        $scope.editing.expression = null;
        $scope.checkAndRefresh();
        $scope.parentDismiss();
    }

    $scope.edit = function(){
        $scope._edit(angular.copy($scope.step.params.expression));
    }

    // automatically open the panel on step creation
    if ($scope.step.$stepState.justCreated === true) {
        $scope.edit();
    }
});


app.controller('ClearOnCustomFormulaController', function($scope, $controller, $stateParams, $rootScope,
               DataikuAPI, CreateCustomElementFromTemplate, ShakerPopupRegistry) {
    $controller("FormulaAwareProcessorController", {$scope:$scope});

    $scope.hooks.ok = function() {
        $scope.step.params.expression = $scope.editing.expression;
        $scope.editing.expression = null;
        $scope.checkAndRefresh();
        $scope.parentDismiss();
    }

    $scope.edit = function(){
        $scope._edit(angular.copy($scope.step.params.expression));
    }

    // automatically open the panel on step creation
    if ($scope.step.$stepState.justCreated === true) {
        $scope.edit();
    }
});


app.controller('FlagOnCustomFormulaController', function($scope, $controller, $stateParams, $rootScope,
               DataikuAPI, CreateCustomElementFromTemplate, ShakerPopupRegistry) {
    $controller("FormulaAwareProcessorController", {$scope:$scope});

    $scope.hooks.ok = function() {
        $scope.step.params.expression = $scope.editing.expression;
        $scope.editing.expression = null;
        $scope.checkAndRefresh();
        $scope.parentDismiss();
    }

    $scope.edit = function(){
        $scope._edit(angular.copy($scope.step.params.expression));
    }

    // automatically open the panel on step creation
    if ($scope.step.$stepState.justCreated === true) {
        $scope.edit();
    }
});


app.controller('FilterAndFlagProcessorController', function($scope) {
    // don't show the generic input for clearColumn if the processor already declares a parameter called clearColumn
    $scope.showClearColumn = !$scope.processor.params.some(function(param) { return param.name == 'clearColumn'; });

    $scope.onActionChange = function() {
        // if (['FLAG', 'KEEP_ROW'].indexOf($scope.step.params['action']) != -1) {
        //     $scope.step.params.appliesTo = 'SINGLE_COLUMN';
        //     $scope.$parent.$parent.$parent.appliesToDisabled = true;
        // } else {
        //     $scope.$parent.$parent.$parent.appliesToDisabled = false;
        // }

        $scope.checkAndRefresh();
    }

    if ($scope.processor.filterAndFlagMode == "FLAG") {
        $scope.step.params.action = "FLAG";
    }
});


app.controller('AppliesToProcessorController', function($scope) {
    // Indicate the position of the mouse relative to the add button
    // The flag is immediately set to true when mouse is over the button
    $scope.isMouseOverAddButton = false;

    $scope.clearEmptyColumns = function() {
        if ($scope.step.params.columns.length == 1 && $scope.step.params.columns[0] == '') {
            $scope.step.params.columns = [];
        }
    }

    /**
     * Handle field blur event of the value list
     * @param {function} handler - function that will be triggered when the event occurs
     * @param {boolean} doesTakeIntoAccountOfMouseOverAddButton - set to true to take into account the mouse over event on the value list add button
     */
    $scope.handleValueListFieldBlur = function (event, handler, doesTakeIntoAccountOfMouseOverAddButton) {
        if (
            event.relatedTarget &&
            event.relatedTarget.offsetParent &&
            event.relatedTarget.offsetParent.classList &&
            event.relatedTarget.offsetParent.classList.contains('mat-autocomplete-panel')
        ) return; // story 99882 - prevents shaker refresh table on column selection
        if (!doesTakeIntoAccountOfMouseOverAddButton) {
            handler();
            return;
        }
        if (!$scope.isMouseOverAddButton) {
            // Only handle the blur event when the mouse was not over the add button
            handler();
        }
    }

    /**
     * Handle mouse over event of the value list's add button
     * Immediately switch the scope's isMouseOverAddButton to 'true'
     */
    $scope.handleValueListMouseOverAddButton = function () {
        $scope.isMouseOverAddButton = true;
    }

    /**
     * Handle mouse leave event of the value list's add button
     */
    $scope.handleValueListMouseLeaveAddButton = function () {
        $scope.isMouseOverAddButton = false;
    }
});


app.controller('ConfigureSamplingController', function($scope, DataikuAPI, $stateParams, $timeout,
               WT1, CreateModalFromTemplate, SamplingData, DatasetUtils) {
    $scope.getPartitionsList = function() {
        return DataikuAPI.datasets.listPartitionsWithName($scope.inputDatasetProjectKey, $scope.inputDatasetName)
            .error(setErrorInScope.bind($scope))
            .then(function(ret) { return ret.data; })
    };

    $scope.SamplingData = SamplingData;

    $scope.showFilterModal = function() {
        var newScope = $scope.$new();
        newScope.updateFilter = (filter) => $scope.shaker.explorationSampling.selection.filter = filter;
        if ($scope.inputDatasetName) {
            DataikuAPI.datasets.get($scope.inputDatasetProjectKey, $scope.inputDatasetName, $stateParams.projectKey)
            .success(function(data){
                newScope.dataset = data;
                newScope.schema = data.schema;
                newScope.filter = angular.copy($scope.shaker.explorationSampling.selection.filter);
                CreateModalFromTemplate('/static/dataiku/nested-filters/input-filter-block/filter-modal.component.html', newScope, undefined, false, false, 'static');
            }).error(setErrorInScope.bind($scope));
        } else if ($scope.inputStreamingEndpointId) {
            DataikuAPI.streamingEndpoints.get($scope.inputDatasetProjectKey, $scope.inputStreamingEndpointId)
            .success(function(data){
                newScope.dataset = data;
                newScope.schema = data.schema;
                newScope.filter = angular.copy($scope.shaker.explorationSampling.selection.filter);
                CreateModalFromTemplate('/static/dataiku/nested-filters/input-filter-block/filter-modal.component.html', newScope, undefined, false, false, 'static');
            }).error(setErrorInScope.bind($scope));
        }
    };

    $scope.datasetIsSQL = function() {
        return $scope.dataset_types && $scope.dataset && $scope.dataset_types[$scope.dataset.type] && $scope.dataset_types[$scope.dataset.type].sql;
    };

    $scope.datasetIsSQLTable = function() {
        return $scope.datasetFullInfo && DatasetUtils.isSQLTable($scope.datasetFullInfo.dataset);
    };

    $scope.datasetSupportsReadOrdering = function() {
        return $scope.datasetFullInfo && DatasetUtils.supportsReadOrdering($scope.datasetFullInfo.dataset);
    };

    $scope.save = function() {
        let evt = {
            analysis: $scope.shaker.origin === 'ANALYSIS',
            samplingMethod: $scope.shaker.explorationSampling.selection.samplingMethod,
            recordsNumber: $scope.shaker.explorationSampling.selection.maxRecords,
            targetRatio: $scope.shaker.explorationSampling.selection.targetRatio,
            filtersNumber: $scope.shaker.explorationSampling.selection.filter &&
                           $scope.shaker.explorationSampling.selection.filter.enabled &&
                           $scope.shaker.explorationSampling.selection.filter.uiData &&
                           $scope.shaker.explorationSampling.selection.filter.uiData.conditions
                ? $scope.shaker.explorationSampling.selection.filter.uiData.conditions.length
                : 0,
            sortingKeyNumber: $scope.shaker.explorationSampling.selection.ordering &&
                              $scope.shaker.explorationSampling.selection.ordering.enabled &&
                              $scope.shaker.explorationSampling.selection.ordering
                ? $scope.shaker.explorationSampling.selection.ordering.rules.length
                : 0,
            maxMemory: $scope.shaker.explorationSampling.selection.maxStoredBytes !== -1 ? $scope.shaker.explorationSampling.selection.maxStoredBytes : null,
            autoRefresh: $scope.shaker.explorationSampling.autoRefreshSample
        };
        if ($scope.inputDatasetName) {
            evt["datasetId"] = `${$scope.inputDatasetProjectKey.dkuHashCode()}.${$scope.inputDatasetName.dkuHashCode()}`;
        }
        if ($scope.inputStreamingEndpointId) {
            evt["streamingEndpointId"] = `${$scope.inputDatasetProjectKey.dkuHashCode()}.${$scope.inputStreamingEndpointId.dkuHashCode()}`;
        }
        WT1.event('dataset-sample-settings-update', evt);
        $scope.shaker.explorationSampling._refreshTrigger = new Date().getTime();
        $scope.forgetSample();
        $scope.autoSaveForceRefresh();
    };
});

app.controller('DateRangeShakerController', function($scope, $timeout, DataikuAPI, Debounce, DateUtilsService, translate, ChartFilterUtils) {
    $scope.step.params.part = $scope.step.params.part != null ? $scope.step.params.part : "YEAR";
    $scope.dateRelativeFilterComputedStart = '-';
    $scope.dateRelativeFilterComputedEnd = '';
    $scope.displayed = { values: [] };
    $scope.valuesCount = 0;
    // Firefox always displays the milliseconds in the time picker, so give more space to the time picker
    $scope.timeInputStyle = { 'width': (navigator.userAgent.includes("Gecko/") ? '144px' : '104px') };

    if($scope.step.params.values) {
        $scope.displayed.values = [...$scope.step.params.values];
    }

    if ($scope.step.params.min) {
        // If date was written as UTC (old format), convert it to the corresponding time zone
        if ($scope.step.params.min.slice(-1).toUpperCase() === "Z") {
            $scope.step.params.min = DateUtilsService.formatDateToISOLocalDateTime(DateUtilsService.convertDateToTimezone(new Date($scope.step.params.min), $scope.step.params.timezone_id));
        }
        $scope.displayed.min = new Date($scope.step.params.min);
        $scope.displayed.min.setSeconds($scope.displayed.min.getSeconds(), 0);
    }

    if ($scope.step.params.max) {
        // If date was written as UTC (old format), convert it to the corresponding time zone
        if ($scope.step.params.max.slice(-1).toUpperCase() === "Z") {
            $scope.step.params.max = DateUtilsService.formatDateToISOLocalDateTime(DateUtilsService.convertDateToTimezone(new Date($scope.step.params.max), $scope.step.params.timezone_id));
        }
        $scope.displayed.max = new Date($scope.step.params.max);
        $scope.displayed.max.setSeconds($scope.displayed.max.getSeconds(), 0);
    }

    // This is a fix for firefox, not firing blur event if input cleared with cross icon
    $scope.updateBoundariesWithDelay = function(boundary) {
        setTimeout($scope.updateBoundaries, 200, boundary);
    }

    $scope.updateBoundaries = function(boundary) {
        if (boundary === 'min') {
            if ($scope.displayed.min) {
                $scope.step.params.min = DateUtilsService.formatDateToISOLocalDateTime($scope.displayed.min);
            } else {
                $scope.step.params.min = null;
            }
        } else if (boundary === 'max') {
            if ($scope.displayed.max) {
                $scope.step.params.max = DateUtilsService.formatDateToISOLocalDateTime($scope.displayed.max);
            } else {
                $scope.step.params.max = null;
            }
        }
        $scope.checkAndRefresh();
    };

    const computeRelativeDateIntervalDebounced = Debounce().withDelay(100,100).withScope($scope).wrap(function() {
        DataikuAPI.shakers.computeRelativeDateInterval({
            part: $scope.step.params.part,
            option: $scope.step.params.option,
        }).success(function(interval) {
            $scope.dateRelativeFilterComputedStart = interval.start;
            $scope.dateRelativeFilterComputedEnd = interval.end;
        }).error(function() {
            $scope.dateRelativeFilterComputedStart = '-';
            $scope.dateRelativeFilterComputedEnd = '-';
        });
    });
    const refreshRelativeIntervalHint = function() {
        $scope.isRelativeFilterEffective = ChartFilterUtils.isRelativeDateFilterEffective($scope.step.params.part, $scope.step.params.option);
        if ($scope.step.params.filterType === 'RELATIVE' && $scope.isRelativeFilterEffective) {
            computeRelativeDateIntervalDebounced();
        }
    }

    $scope.$watchGroup(["step.params.filterType", "step.params.part"], refreshRelativeIntervalHint);

    $scope.handleRelativeDateOptionChange = (relativeOption) => {
        $scope.step.params.option = relativeOption;
        refreshRelativeIntervalHint();
        $scope.updateBoundaries()
    }

    $scope.$watch("displayed.values", function() {
        $scope.step.params.values = [...$scope.displayed.values];
    });

    $scope.$watch("step.params.timezone_id", $scope.updateBoundaries);

    $scope.$watch("step.params.filterType", function(nv) {
        if (nv === "RELATIVE") {
            if (["INDIVIDUAL", "DAY_OF_WEEK"].includes($scope.step.params.part)) {
                $scope.step.params.part = "YEAR";
            }
        }
    });
});

app.controller('UnfoldController', function($scope, translate){

    $scope.overflowActions = [
        ["KEEP", translate("SHAKER.PROCESSORS.Unfold.FORM.OVERFLOW_ACTION.KEEP", "Keep all columns")],
        ["WARNING", translate("SHAKER.PROCESSORS.Unfold.FORM.OVERFLOW_ACTION.WARNING", "Add warnings")],
        ["ERROR", translate("SHAKER.PROCESSORS.Unfold.FORM.OVERFLOW_ACTION.ERROR", "Raise an error")],
        ["CLIP", translate("SHAKER.PROCESSORS.Unfold.FORM.OVERFLOW_ACTION.CLIP", "Clip further data")],
    ];

    $scope.overflowActionsDesc = [
        [translate("SHAKER.PROCESSORS.Unfold.FORM.OVERFLOW_ACTION.KEEP.DESC", "Will keep all the created columns. Warning, this may create a huge amount of columns and slow the whole computation.")],
        [translate("SHAKER.PROCESSORS.Unfold.FORM.OVERFLOW_ACTION.WARNING.DESC", "Will raise warning during the computation but continues to process all the columns. It may create a huge amount of columns and slow the whole computation.")],
        [translate("SHAKER.PROCESSORS.Unfold.FORM.OVERFLOW_ACTION.ERROR.DESC", "Will raise an error and make the computation fail as soon as the maximum number of created columns is exceeded.")],
        [translate("SHAKER.PROCESSORS.Unfold.FORM.OVERFLOW_ACTION.CLIP.DESC", "Will silently drop the remaining columns when the maximum number of columns is reached.")]
    ];

    $scope.modifyOverflowAction = function() {
        if ($scope.step.params.limit === 0) {
            $scope.step.params.overflowAction = "KEEP";
        }
    };

});

app.controller('ColumnRenamerController', function(
    $scope, $timeout, $rootScope, $q, $stateParams, CreateModalFromTemplate) {

    $scope.massRenameColumns = function() {
        CreateModalFromTemplate('/templates/shaker/modals/shaker-rename-columns.html', $scope, 'MassRenameColumnsController', function(newScope) {
            let columnNames = [...computeColumns()];
            $scope.step.params.renamings.forEach(function(renaming) {
                let index = columnNames.indexOf(renaming.from);
                if (index >= 0) {
                    columnNames[index] = renaming.to;
                }
            });

            newScope.setColumns(columnNames);
            newScope.doRenameColumns = function(renamings) {
                $scope.step.params.renamings = $scope.step.params.renamings.concat(renamings);
            };
        });
    };

    const canGetColumnDetails = () => {
        const stepPosition = $scope.shaker.steps.indexOf($scope.step);

        // If the current step is the last one or if it is in preview mode or if the steps after it are disabled.
        return $scope.shaker && $scope.shaker.steps && $scope.shaker.steps[$scope.shaker.steps.length - 1] === $scope.step
            || $scope.step.preview === true
            || (!$scope.step.disabled && $scope.shaker.steps.slice(stepPosition + 1).every(step => step.disabled))
    }

    const computeColumns = () => {
        if (canGetColumnDetails()) {
            return $scope.quickColumns
                .filter(c => c.recipeSchemaColumn ? !c.recipeSchemaColumn.deleted : true)
                .map(c => ({
                    ...(c.recipeSchemaColumn ? c.recipeSchemaColumn.column : {}),
                    name: c.name
                }))
                .map(c => c.name);
        } else {
            return $scope.step.$stepState.change ? $scope.step.$stepState.change.columnsBeforeStep : [];
        }
    }
});

}());

;
(function(){
    'use strict';

    /* Misc additional directives for Shaker */

    var app = angular.module('dataiku.shaker.misc', ['dataiku.filters', 'platypus.utils']);

    /**
     * Directive using editableList with bootstrap typeahead on the inputs   
     * @param {Array}           ngModel                         - The list to bind to display.
     * @param {Function}        [onChange]                      - Callback called on change.
     * @param {addLabel}        [onAdd]                         - Label of the add button.
     * @param {boolean}         [typeAhead]                     - Text to display in the add button; Optional if disableAdd is true
     * @param {boolean}         [noChangeOnAdd=false]           - True to prevent the callback onChange to be called when an item is added.
    */
    app.directive('dkuListTypeaheadV2', function(){
        return {
            templateUrl: '/templates/widgets/dku-list-typeahead-v2.html',
            scope: {
                model: '=ngModel',
                onChange: '&',
                addLabel: '@',
                typeAhead: '=',
                noChangeOnAdd: '<'
            },
            link: function(scope) {
                scope.richModel = {}

                scope.$watch("model", function(nv){
                    if (!nv) return;
                    scope.richModel = scope.model.map(function(x){
                        return { value  : x }
                    });
                });

                if (scope.onChange) {
                    scope.callback = function(){
                        // Update in place instead of replacing, important because
                        // we don't want this to trigger another watch cycle in the
                        // editableList, which watches items non-deeply
                        scope.model.length = 0;
                        scope.richModel.forEach(function(x){
                            scope.model.push(x.value);
                        });
                        scope.onChange.bind(scope)({model : scope.model});
                    }
                }
                if (scope.typeAhead) {
                    scope.$watch("model", function() {
                        scope.remainingSuggests = listDifference(scope.typeAhead, scope.model);
                    }, true);
                }
            }
        };
    });
    
    app.directive('meaningSelect', function(ContextualMenu) {
    	return {
    		restrict: 'A',
    		template: '<div class="select-button">'
    					+'<button ng-click="openMeaningMenu($event)" class="btn  dku-select-button btn--secondary">'
                            +'<span class="filter-option pull-left">{{ngModel|meaningLabel}}</span>'
							+'&nbsp;'
							+'<span class="caret"></span>'
    					+'</button>'
    				 +'</div>',
            scope: {
                ngModel: '=',
                appConfig: '='
            },
            link: function($scope, element, attrs) {
    			$scope.menuState = {};
                $scope.meaningMenu = new ContextualMenu({
                    template: "/templates/shaker/select-meaning-contextual-menu.html",
                    cssClass : "column-header-meanings-menu",
                    scope: $scope,
                    contextual: false,
                    onOpen: function() {
                        $scope.menuState.meaning = true;
                    },
                    onClose: function() {
                        $scope.menuState.meaning = false;
                    }
                });
                $scope.openMeaningMenu = function($event) {
                    $scope.meaningMenu.openAtXY($event.pageX, $event.pageY);
                };

	            $scope.setMeaning = function(meaningId) {
                    $scope.ngModel = meaningId;
                    $(element).trigger('change');
	            };
    		}
    	}
    });

    app.directive('nextOnEnter', function() {
        return {
            score: true,
            priority: 90,
            restrict: 'A',
            link: function(scope, el, attrs) {
                var form = el[0].form;
                $(el).keyup(function (e) {
                    if (e.keyCode === 13) {
                        // on enter, we behave like for tab.
                        // and focus the next element of the form.
                        var tabbables = $(form).find(":tabbable");
                        var elId = tabbables.index(el);
                        if ( (elId >= 0) && (elId < tabbables.length -1) ) {
                            tabbables[elId+1].focus();
                        }
                        else {
                            // reached the last element... Just blur.
                            el.blur();
                        }
                    }
                });
            }
        };
    });
    app.directive('blurOnEnter', function() {
        return {
            score: true,
            priority: 90,
            restrict: 'A',
            link: function(scope, el, attrs) {
                var form = el[0].form;
                $(el).keyup(function (e) {
                    if (e.keyCode === 13) {
                            el.blur();
                    }
                });
            }
        };
    });
    app.directive('blurOnEnterAndEsc', function() {
        return {
            score: true,
            priority: 90,
            restrict: 'A',
            link: function(scope, el, attrs) {
                var form = el[0].form;
                $(el).keyup(function (e) {
                    if (e.keyCode === 13 || e.keyCode === 27) {
                            el.blur();
                    }
                });
            }
        };
    });

    app.directive('shakerProcessorStep', function($filter, CachedAPICalls, ShakerProcessorsInfo, ShakerProcessorsUtils, RecipesUtils){
        return {
            templateUrl: '/templates/shaker/processor-step.html',
            replace: true,
            /* We have to use prototypal inheritance scope instead of isolate scope because of bug
            https://github.com/angular/angular.js/issues/1941
            *
            * And also because we access a lot of the scope
            *
            * Requires in scope:
            *  - step
            *  - columns (array[string])
            *  - $index
            */
            scope:true,
            link: function(scope, element, attrs){
                // TODO: also linked to the isolate scope issue

                // at instantiation, always scroll to element
                $(element).find(".content").get(0).scrollIntoView(true);

                scope.remove = function(step) {
                    $('.processor-help-popover').popover('hide');//hide any displayed help window
                    scope.removeStep(step.step);
                    keepInputsInSync();
                };
                
                scope.deleteHelper = function(obj, key) {
                    delete obj[key];
                };

                scope.isStepActive = function() {
                	return scope.step == scope.currentStep;
                }

                const keepInputsInSync = () => {
                    if (!scope.recipe || !scope.shaker) return;
                    RecipesUtils.removeInputsForRole(scope.recipe, "scriptDeps");
                    RecipesUtils.removeInputsForRole(scope.recipe, "reference");

                    const addInputsFromSteps = (steps) => {
                        steps.forEach((step) => {
                            if (step.metaType === 'GROUP') {
                                addInputsFromSteps(step.steps);
                            } else {
                                // Keep reference datasets in sync with recipe config
                                if (step.params && step.params.useDatasetForMapping && step.params.mappingDatasetRef && RecipesUtils.getInput(scope.recipe, "reference", step.params.mappingDatasetRef) === null) {
                                    RecipesUtils.addInput(scope.recipe, "reference", step.params.mappingDatasetRef);
                                }
                                // Keep joined datasets in sync with recipe config
                                if (step.params && step.params.rightInput && RecipesUtils.getInput(scope.recipe, "scriptDeps", step.params.rightInput) === null) {
                                    RecipesUtils.addInput(scope.recipe, "scriptDeps", step.params.rightInput);
                                }
                            }
                        });
                    }
                    addInputsFromSteps(scope.shaker.steps);
                }

                /**
                 * This is the method called by all forms when a value is changed by the user.
                 * It triggers validation of the step, and, if the step is valid, the refresh.
                 *
                 * This handles a special case: processors that are "new", i.e. that have never been valid.
                 * For them, we don't display their 'in-error' state while they have not been valid at least
                 * once
                 */
                scope.checkAndRefresh = function() {
                    if (!scope.step.$stepState) {
                        scope.step.$stepState = {};
                    }
                    const state = scope.step.$stepState;

                    state.frontError = scope.validateStep(scope.step);

                    if (state.frontError && state.isNew){
                        // Don't do anything for a new processor that is still invalid
                    } else if (!state.frontError && state.isNew) {
                        // No error in new processor -> so it's not new anymore, and we can refresh
                        state.isNew = false;
                        scope.autoSaveAutoRefresh();
                        keepInputsInSync();
                    } else if (state.frontError && !state.isNew) {
                        // Error in non-new processor: Don't refresh
                    } else if (!state.frontError && !state.isNew) {
                        // No error in non-new processor -> the 'normal' case
                        scope.autoSaveAutoRefresh();
                        keepInputsInSync();
                    }
                };

                CachedAPICalls.processorsLibrary.success(function(processors){
                    scope.processors = processors;
                    scope.processor = $filter('processorByType')(scope.processors, scope.step.type);

                    var e = ShakerProcessorsInfo.get(scope.step.type);
                    if (angular.isDefined(e) && angular.isDefined(e.postLinkFn)){
                        e.postLinkFn(scope, element);
                    }

                    scope.$watch("step", function(step, ov) {
                        if (!step.$stepState) {
                            step.$stepState = {};
                        }

                        step.$stepState.frontError = scope.validateStep(scope.step);

                        scope.description = ShakerProcessorsUtils.getStepDescription(scope.processor, step.type, step.params);
                        scope.icon = ShakerProcessorsUtils.getStepIcon(step.type, step.params, 16); // I don't think it's used anywhere, in doubt default to 16px
                    }, true);
                });

                scope.types = Object.keys(scope.appConfig.meanings.labelsMap);

                scope.isMandatory = function(paramName) {
                    const param = scope.processor.params.find((param) => param.name === paramName);
                    return param && param.mandatory;
                };
            }
        };
    });

    app.directive('shakerGroupStep', function($filter, CachedAPICalls, Fn, $timeout){
        return {
            templateUrl: '/templates/shaker/group-step.html',
            replace: true,
            /*
            * Requires in scope:
            *  - step
            *  - columns (array[string])
            *  - $index
            */
            scope:true,
            link: function(scope, element, attrs){
                scope.remove = function(step) {
                    $('.processor-help-popover').popover('hide');//hide any displayed help window
                    scope.removeStep(step.step);
                };

                scope.hasMatchingSteps = function() {
                    return scope.step.steps.filter(Fn.prop('match')).length > 0;
                }

                scope.isGroupActive = function() {
                   return !scope.isCollapsed() || (scope.hasMatchingSteps() && !scope.step.closeOnMatch);
                }

                scope.toggleGroup = function() {
                    if (scope.isCollapsed() && scope.isGroupActive()) {
                        scope.step.closeOnMatch = !scope.step.closeOnMatch;
                    } else {
                        scope.toggle();
                    }
                }

                scope.$on('openShakerGroup', function(e, step) {
                    if (scope.step === step && scope.isCollapsed()) {
                        scope.toggle();        
                    }
                });

                scope.$watch('groupChanged.addedStepsTo', function() {
                    if (scope.groupChanged.addedStepsTo === scope.step) {
                        if (!scope.isGroupActive()) {
                            scope.toggleGroup();
                        }
                        scrollToStep();
                    } else if (scope.groupChanged.removedStepsFrom.indexOf(scope.step) > -1 && scope.step.steps.length === 0) {
                        if (scope.isGroupActive()) {
                            scope.toggleGroup();
                        }
                    }
                });

                scope.$watch("step.forceOpenOnCreation", function(nv, ov) {
                    if (nv === true) {
                        scope.toggleGroup();
                    }
                })


                scrollToStep();

                function scrollToStep() {
                    // at instantiation, always scroll to element and start editing
                    $timeout(function() {
                        $(element).get(0).scrollIntoView({
                            behavior: 'auto',
                            block: 'center',
                            inline: 'center'
                        });
                    });
                }

                function hasActuallyChanged(newSteps, oldSteps) {
                    if (!newSteps && !oldSteps) {
                        return false;
                    }

                    if ((newSteps && !oldSteps) || (!newSteps && oldSteps)) {
                        return true;
                    }

                    if (newSteps.length !== oldSteps.length) {
                        return true;
                    }

                    for (let i = 0; i < newSteps.length; i++) {
                        const newStep = newSteps[i];
                        const oldStep = oldSteps[i];
                        if (newStep.type !== oldStep.type
                            || newStep.metaType !== oldStep.metaType
                            || JSON.stringify(newStep.params) !== JSON.stringify(oldStep.params)) {
                                return true;
                            }
                    }
                    return false;
                }

            }
        };
    });


    app.directive('shaker', function($timeout) {
        return {
            restrict: 'C',
            link: function(scope, element, attrs){
                scope.$watch('shaker.explorationFilters.length', function(nv, ov){
                    if (nv && nv > 1) {
                        scope.$broadcast('tabSelect', 'filters');
                    }
                });
                scope.$watch('shaker.steps.length', function(nv, ov){
                    scope.$broadcast('tabSelect', 'script');
                    if (nv > ov) {
                        let ul = $(element).find('ul.steps.accordion');
                        let items = ul.children();
                        let scrollIndex = scope.pasting ? findFirstCopy(scope.shaker.steps) : items.length - 1;

                        $timeout(function() {
                            let addedElement = ul.children().get(scrollIndex);
                            // scroll to element
                            if (addedElement) {
                                addedElement.scrollIntoView({ 'block': 'center' });
                            }
                        });
                    }
                });

                function findFirstCopy(steps) {
                    return steps.findIndex(_ => {
                        return (_.$stepState && _.$stepState.isNewCopy) || (_.steps && findFirstCopy(_.steps) >= 0);
                    });
                }

                scope.clearFFS = function(){
                    scope.ffs = [];
                };
            }
        };
    });

    // small directive meant to replace "-1" by "not set", since -1 is used as a marker of "no limit" in the backend
    // Also handles megabytes
    app.directive('optionalMaxSizeMb', function() { // Warning : this directive cannot handle nulls -> ng-if above it
        return {
            scope: true,
            restrict: 'A',
            link: function($scope, el, attrs) {
                $scope.$optionalState = {};
                var initSize = $scope.$eval(attrs.optionalMaxSizeMb);
                $scope.$optionalState.hasMaxSize = initSize >= 0;
                if ($scope.$optionalState.hasMaxSize) {
                    $scope.$optionalState.maxSize = initSize / (1024 * 1024);
                }
                $scope.$watch('$optionalState.hasMaxSize', function(nv, ov) {
                    if (!$scope.$optionalState.hasMaxSize) {
                        $scope.$eval(attrs.optionalMaxSizeMb + " = -1");
                    } else {
                        /* Put a sane default value */
                        if ($scope.$optionalState.maxSize === undefined || $scope.$optionalState.maxSize < 0) {
                            $scope.$optionalState.maxSize = 1;
                        }
                        $scope.$eval(attrs.optionalMaxSizeMb + " = " + ($scope.$optionalState.maxSize * 1024 * 1024));
                    }
                });
                $scope.$watch('$optionalState.maxSize', function(nv, ov) {
                    if (nv === undefined) return;
                    $scope.$eval(attrs.optionalMaxSizeMb + " = " + ($scope.$optionalState.maxSize * 1024 * 1024));
                });
            }
        };
    });


    var services = angular.module('dataiku.services');

    services.factory('ShakerPopupRegistry', function(Logger) {
        var callbacks = [];
        function register(dismissFunction) {
            callbacks.push(dismissFunction);
        }
        function dismissAll() {
            callbacks.forEach(function(f) {
                try {
                    f();
                } catch (e) {
                    Logger.warn("failed to dismiss shaker popup", e);
                }
            });
            callbacks = [];
        }

        function dismissAllAndRegister(dismissFunction) {
            dismissAll();
            register(dismissFunction);
        }

        return {
            register: register,
            dismissAll: dismissAll,
            dismissAllAndRegister: dismissAllAndRegister
        }
    });

    // to put on the element in which the custom formula editor is supposed to be shown. It provides a function
    // that can be passed to CreateCustomElementFromTemplate in order to insert the formula editor in the DOM,
    // instead of the usual mechanism (which is: append to <body>). This directive sets a boolean in the scope
    // to indicate the formula editor is open (so that you can hide other stuff while it's open, for example)
    app.directive('customFormulaZone', function($rootScope) {
        return {
            scope: true,
            restrict: 'A',
            link: function($scope, el, attrs) {
            	var type = attrs.customFormulaZone || 'replace';
            	$scope.customFormulaEdition.editing = 0;

            	$scope.customFormulaEdition.displayCustomFormula = function(formulaElement) {
            		$scope.customFormulaEdition.editing += 1;

                	$(formulaElement).on("remove", function() {
                		$scope.customFormulaEdition.editing -= 1;
                		if ($scope.customFormulaEdition.editing == 0 ) {
                			if ( type == 'replace' ) {
                				$(el).removeClass("replaced-by-formula");
                			}
                		}
                		$scope.customFormulaEdition.reflowStuff();
                	});

                	if (type == 'replace') {
                		$(el).after(formulaElement);
                		if ( $scope.customFormulaEdition.editing == 1 ) {
                			$(el).addClass("replaced-by-formula");
                		}
                	} else {
                		$(el).append(formulaElement);
                	}
            	};
            }
        };
    });

    app.directive('resizable', ['$document', function($document) {
        return {
            restrict: 'A',
            link: function(scope, element, attrs) {
                    const resizeDirection = attrs.resizable || 'both';
                    const { minHeight, maxHeight, minWidth, maxWidth } = attrs;
                    if (minHeight) {
                        element.css('min-height', minHeight);
                    }
                    if (maxHeight) {
                        element.css('max-height', maxHeight);
                    }
                    if (minWidth) {
                        element.css('min-width', minWidth);
                    }
                    if (maxWidth) {
                        element.css('max-width', maxWidth);
                    }
                    let icon = 'dku-icon-arrow-expand-16';
                    switch (resizeDirection) {
                        case 'vertical':
                            icon = 'dku-icon-arrow-double-vertical-16';
                            break;
                        case 'horizontal':
                            icon = 'dku-icon-arrow-double-horizontal-16';
                            break;
                        default:
                            break;
                    }

                    const resizeHandle = angular.element(`
                        <div class="resize-handle">
                            <i class="${icon}"></i>
                        </div>
                    `);
                    element.append(resizeHandle);

                    let startX, startY, startWidth, startHeight;

                    resizeHandle.on('mousedown', function(event) {
                        event.preventDefault();
                        event.stopPropagation();
                        startX = event.pageX;
                        startY = event.pageY;
                        startWidth = element[0].offsetWidth;
                        startHeight = element[0].offsetHeight;
                        $document.on('mousemove', mousemove);
                        $document.on('mouseup', mouseup);
                    });

                    function mousemove(event) {
                        const dx = event.pageX - startX;
                        const dy = event.pageY - startY;

                        if (resizeDirection === 'horizontal' || resizeDirection === 'both') {
                            element.css('width', (startWidth + dx) + 'px');
                        }
                        if (resizeDirection === 'vertical' || resizeDirection === 'both') {
                            element.css('height', (startHeight + dy) + 'px');
                        }
                    }

                    function mouseup() {
                        $document.off('mousemove', mousemove);
                        $document.off('mouseup', mouseup);
                    }
                }        
            };
    }]);

    // dumb directive to put somewhere above the element providing the custom formula, and the element receiving it.
    // Its purpose is to bridge the scopes of the step using the formula editor and the place where the formula
    // editor is shown (they're most likely in different panes on the screen)
    app.directive('hasCustomFormulaZone', function() {
        return {
            scope: true,
            restrict: 'A',
            link: function($scope, el, attrs) {
            	$scope.customFormulaEdition = {
            		reflowStuff : function() {$scope.$broadcast("reflow");} // reflow just inside the shaker screen, not the entire dss
            	};
            }
        };
    });

})();

;
(function () {
    "use strict";

    const app = angular.module('dataiku.shaker');


    app.directive('datasetChartsBase', function ($rootScope, Assert, ChartTypeChangeHandler, Logger, CreateModalFromTemplate,
        DatasetUtils, WT1, TopNav, DataikuAPI, $timeout, ActivityIndicator, $state, $stateParams, $q, DatasetChartsUtils, ChartSetErrorInScope, ChartCustomMeasures, ChartsContext, DatasetErrorCta, ATSurveyService, DSSVisualizationThemeUtils, DefaultDSSVisualizationTheme, ChartNavigationService) {
        return {
            priority: 2,
            scope: true,
            controller: function ($scope, $stateParams) {
                ChartSetErrorInScope.defineInScope($scope);
                $scope.onLoad = function (projectKey, datasetName, contextProjectKey, datasetSmartName) {
                    //For datasetErrorCTA directive (CTA in case of error while loading dataset sample)
                    $scope.errorCTA = {};

                    $scope.updateUiState = DatasetErrorCta.getupdateUiStateFunc($scope);

                    $scope.$watch("datasetFullInfo", () => $scope.updateUiState($scope.errorCTA.error), true);
                    $scope.$watch("errorCTA", () => $scope.updateUiState($scope.errorCTA.error), true);

                    /* ********************* Execute Callbacks for chartsCommon ******************* */

                    $scope.getDataSpec = function () {
                        var currentChart = $scope.charts[$scope.currentChart.index];
                        Assert.trueish(currentChart, 'no currentChart');

                        var dataSpec = {
                            datasetProjectKey: projectKey,
                            datasetName: datasetName,
                            script: angular.copy($scope.shaker),
                            copySelectionFromScript: currentChart.copySelectionFromScript,
                            sampleSettings: currentChart.refreshableSelection,
                            engineType: currentChart.engineType
                        };
                        dataSpec.script.origin = "DATASET_EXPLORE";
                        return dataSpec;
                    }

                    $scope.getExecutePromise = function (request, saveShaker = true, noSpinner = false, requiredSampleId = undefined, dataSpec = $scope.getDataSpec()) {
                        var currentChart = $scope.charts[$scope.currentChart.index];
                        Assert.trueish(currentChart.summary, "Current chart summary is not ready");
                        if (requiredSampleId === undefined) {
                            requiredSampleId = currentChart.summary.requiredSampleId;
                        }
                        (saveShaker !== false) && $scope.saveShaker();
                        if (request) {
                            let promise = DataikuAPI.shakers.charts.getPivotResponse(
                                contextProjectKey ? contextProjectKey : projectKey,
                                dataSpec,
                                request,
                                requiredSampleId
                            );

                            if (noSpinner === true) {
                                promise = promise.noSpinner();
                            }

                            return promise;
                        }
                    };

                    $rootScope.$on("chartSamplingChanged", function (event, opts) {
                        if (angular.equals($scope.charts[$scope.currentChart.index], opts.chart)) {
                            $scope.clearCachedSummaries();
                            $scope.fetchColumnsSummaryForCurrentChart().then(function () {
                                Logger.info("Sample reloaded, executing chart");
                                $scope.$broadcast("forceExecuteChart");
                            });
                        }
                    });

                    $scope.getDefaultNewChart = function () {
                        var newChart = null;
                        const defaultTheme = DSSVisualizationThemeUtils.getThemeOrDefault($rootScope.appConfig.selectedDSSVisualizationTheme);
                        if ($scope.charts.length > 0) {
                            // Copy to retrieve the same sample, copySample and engine settings
                            newChart = angular.copy($scope.charts[$scope.charts.length - 1]);
                            newChart.theme = defaultTheme;
                            newChart.def = ChartTypeChangeHandler.defaultNewChart(newChart.theme);
                        } else {
                            newChart = {
                                def: ChartTypeChangeHandler.defaultNewChart(defaultTheme),
                                copySelectionFromScript: true,
                                engineType: "LINO",
                                theme: defaultTheme
                            }
                        }
                        return newChart;
                    }

                    function exploreIsDirty(ignoreThumbnailChanges) {
                        try {
                            var savedExplore2 = angular.copy(savedExplore);
                            var explore = angular.copy($scope.explore);

                            if (ignoreThumbnailChanges) {
                                if (explore) {
                                    explore.charts.forEach(function (chart) {
                                        chart.def.thumbnailData = null;
                                    });
                                }
                                if (savedExplore2) {
                                    savedExplore2.charts.forEach(function (chart) {
                                        chart.def.thumbnailData = null;
                                    });
                                }
                            }
                            return !angular.equals(explore, savedExplore2);
                        } catch (e) {
                            Logger.error(e);
                            return true;
                        }
                    }

                    $scope.saveShaker = function (isThumbnail = false) {
                        Logger.info("Saving shaker");
                        var ignoreThumbnailChanges = !$scope.isProjectAnalystRW();
                        if (!exploreIsDirty(ignoreThumbnailChanges)) {
                            Logger.info("No changes: don't save explore");
                            return;
                        }

                        if ($scope.isProjectAnalystRW()) {
                            DataikuAPI.explores.save(contextProjectKey, datasetSmartName, $scope.explore)
                            .success(function (data) {
                                if (!isThumbnail && !ActivityIndicator.isDisplayed()) {
                                    ActivityIndicator.success("Charts saved");
                                }
                            }).error(setErrorInScope.bind($scope)).noSpinner();
                        } else {
                            ActivityIndicator.warning("You don't have write access - not saving");
                        }
                    };

                    $scope.saveChart = $scope.saveShaker;

                    /* ********************* Load callback ******************* */

                    var cachedColumnSummaries = {};

                    $scope.clearCachedSummaries = function () {
                        $scope.charts.forEach(function (x) {
                            x.summary = null;
                        });
                        cachedColumnSummaries = {};
                    }

                    $scope.fetchColumnsSummaryForCurrentChart = function (forceRefresh) {
                        var currentChart = $scope.charts[$scope.currentChart.index];
                        var dataSpec = $scope.getDataSpec();
                        var cacheKey = JSON.stringify(dataSpec).dkuHashCode();

                        var promise = null;
                        if (cachedColumnSummaries[cacheKey] != null && !forceRefresh) {
                            Logger.info("Already cached for", dataSpec);
                            promise = $q.when(cachedColumnSummaries[cacheKey]);
                        } else {
                            Logger.info("No cache for", dataSpec);
                            promise = DataikuAPI.shakers.charts.getColumnsSummary(contextProjectKey ? contextProjectKey : projectKey, dataSpec)
                                .error($scope.chartSetErrorInScope)
                                .then(function (response) {
                                    cachedColumnSummaries[cacheKey] = response.data;
                                    return response.data;
                                })
                        }

                        return promise.then(
                            function (data) {
                                currentChart.summary = data;
                                $scope.makeUsableColumns(data);
                                if ($scope.errorCTA) {
                                    $scope.errorCTA.error = null;
                                }
                            },
                            function (attr) {
                                if ($scope.errorCTA) {
                                    $scope.errorCTA.error = getErrorDetails(attr.data, attr.status, attr.headers, attr.statusText);
                                }
                            }
                        );
                    };

                    $scope.overrideFormattingWithTheme = function (theme) {
                        const currentChart = $scope.charts[$scope.currentChart.index];
                        const currentChartCopy = angular.copy(currentChart);
                        const formerTheme = currentChart.theme;
                        currentChart.theme = theme;

                        DSSVisualizationThemeUtils.applyToChart({ chart: currentChart.def, theme, formerTheme });
                        DSSVisualizationThemeUtils.showThemeAppliedSnackbar(currentChart, currentChartCopy);
                    };

                    $scope.$on('$destroy', () => DSSVisualizationThemeUtils.hideThemeAppliedSnackbar());

                    $scope.createAndPinInsight = function () {
                        const insights = $scope.charts.map(chart => {
                            let insight = {
                                type: 'chart',
                                projectKey: contextProjectKey,
                                name: chart.def.name + ' on ' + datasetSmartName,
                                params: {
                                    datasetSmartName: datasetSmartName,
                                    engineType: chart.engineType,
                                    refreshableSelection: chart.refreshableSelection,
                                    copySelectionFromScript: chart.copySelectionFromScript,
                                    def: chart.def,
                                    summary: chart.summary,
                                    theme: chart.theme || DefaultDSSVisualizationTheme,
                                    customMeasures: $scope.explore.customMeasures,
                                    reusableDimensions: $scope.explore.reusableDimensions
                                }
                            };
                            if (insight.params.refreshableSelection == null) {
                                insight.params.refreshableSelection = DatasetChartsUtils.makeSelectionFromScript($scope.shaker);
                            }

                            return insight;
                        });

                        CreateModalFromTemplate("/templates/dashboards/insights/create-and-pin-insights-modal.html", $scope, "CreateAndPinInsightsModalController", function (newScope) {
                            let selectedCharts = angular.copy($scope.charts);
                            selectedCharts.forEach(_ => _.selected = false);
                            selectedCharts[$scope.currentChart.index].selected = true;

                            newScope.insightData = {
                                items: selectedCharts,
                                type: 'chart'
                            }

                            newScope.init(insights);
                        });
                    };

                    /* ********************* Main ******************* */

                    var savedExplore;
                    var main = function () {
                        WT1.event("dataset-charts-open");
                        // Call ATSurveyService to trigger survey if needed
                        ATSurveyService.updateCounter('ChartsOpen');
                        TopNav.setLocation(TopNav.TOP_FLOW, 'datasets', TopNav.TABS_DATASET, "visualize");

                        ChartNavigationService.bindCurrentChartWithUrl();

                        $scope.$on('$destroy', () => {
                            ChartNavigationService.unbindCurrentChartWithUrl()
                        });

                        $scope.$watchGroup(
                            ChartNavigationService.getCurrentChartWatchers($scope),
                            function () {
                                ChartNavigationService.updateCurrentChart($scope.charts, $scope.currentChart.index)
                            }
                        );

                        DataikuAPI.explores.get(contextProjectKey, datasetSmartName).success(function (data) {
                            $scope.explore = data;
                            $scope.shaker = data.script;
                            $scope.charts = data.charts;
                            $scope.currentChart.index = ChartNavigationService.getChartIndexFromId($scope.charts.map(chart => chart.def), $stateParams.chartId);
                            savedExplore = angular.copy($scope.savedExplore);

                            DataikuAPI.datasets.get(projectKey, datasetName, $stateParams.projectKey).success(function (data) {
                                $scope.dataset = data;
                            }).error(setErrorInScope.bind($scope));

                            if ($scope.charts.length == 0) {
                                $scope.addChart({ datasetName });
                            }

                            $scope.addCustomMeasuresToScopeAndCache($scope.explore.customMeasures);
                            $scope.addBinnedDimensionToScopeAndCache($scope.explore.reusableDimensions);


                            Logger.info("Explore loaded, get summary");

                            $scope.$watch("charts[currentChart.index]", function (nv) {
                                Logger.info("Chart changed, fetching summary and executing");
                                if (nv) {
                                    $scope.fetchColumnsSummaryForCurrentChart()
                                        .then(function () {
                                            // Fixes a race condition that used to happen sometimes when explores.get returned before the
                                            // event listeners in chart_logic.js were properly set up, causing the forceExecuteChart to be missed
                                            // and nothing to be drawn.
                                            $scope.forceExecuteChartOrWait();
                                        })
                                }
                            });
                            if ($scope.errorCTA) {
                                $scope.errorCTA.error = null;
                            }
                        }).error(function (data, status, headers, config, statusText, xhrStatus) {
                            setErrorInScope.bind($scope)(data, status, headers, config, statusText, xhrStatus);
                            if ($scope.errorCTA) {
                                $scope.errorCTA.error = getErrorDetails(data, status, headers, statusText);
                            }
                        });
                    };

                    main();
                };
            }
        }
    });

    app.directive('datasetCharts', function () {
        return {
            scope: true,
            controller: function ($scope, $stateParams) {
                $scope.onLoad($stateParams.projectKey, $stateParams.datasetName, $stateParams.projectKey, $stateParams.datasetName);
            }
        }
    });

    app.directive('foreignDatasetCharts', function (Logger, DatasetUtils) {
        return {
            scope: true,
            controller: function ($scope, $stateParams) {
                var loc = DatasetUtils.getLocFromFull($stateParams.datasetFullName);
                $scope.onLoad(loc.projectKey, loc.name, $stateParams.projectKey, $stateParams.datasetFullName);
            }
        }
    });

    app.directive("datasetChartSamplingEditor", function (DatasetUtils, ChartFeatures, $rootScope, WT1, translate) {
        return {
            scope: {
                dataset: '=',
                chart: '=',
                script: '=',
                canCopySelectionFromScript: '=',
                useUpperButtons: '<?',
                initialChart: '=?', // reference data to check dirty state
            },
            templateUrl: "/templates/simple_report/dataset-chart-sampling-editor.html",
            controller: function ($scope, $controller) {
                $controller("_ChartOnDatasetSamplingEditorBase", { $scope: $scope });

                function makeEnginesStatus(script, chartSpec) {
                    var engines = [
                        ["LINO", $rootScope.wl.productShortName, true, ""]
                    ]
                    var sqlEngine = [
                        "SQL",
                        translate("CHARTS.SAMPLING.IN_DATABASE", "In-database"),
                        false,
                        ""
                    ];
                    if (!DatasetUtils.canUseSQL($scope.dataset)) {
                        sqlEngine[3] = translate("CHARTS.SAMPLING.DATASET_IS_NOT_SQL", "Dataset is not SQL");
                    } else if (script != null && script.steps.length) {
                        sqlEngine[3] = translate("CHARTS.SAMPLING.SCRIPT_CONTAINS_STEPS", "Script contains steps");
                    } else if (!ChartFeatures.canUseSQL(chartSpec)) {
                        sqlEngine[3] = translate("CHARTS.SAMPLING.CHART_INCOMPATIBLE_WITH_IN_DATABASE", "This chart is not compatible with in-database");
                    } else {
                        sqlEngine[2] = true;
                    }
                    engines.push(sqlEngine);
                    if ($rootScope.appConfig.interactiveSparkEngine != null) {
                        var sparksqlEngine = ["SPARKSQL", "SparkSQL", false, ""];
                        if (!DatasetUtils.canUseSparkSQL($scope.dataset)) {
                            sqlEngine[3] = translate("CHARTS.SAMPLING.DATASET_IS_SQL", "Dataset is SQL, use in-database engine");
                        } else if (script != null && script.steps.length) {
                            sparksqlEngine[3] = translate("CHARTS.SAMPLING.SCRIPT_CONTAINS_STEPS", "Script contains steps");
                        } else if (!ChartFeatures.canUseSQL(chartSpec)) {
                            sparksqlEngine[3] = translate("CHARTS.SAMPLING.CHART_INCOMPATIBLE_WITH_SPARK", "This chart is not compatible with SparkSQL");
                        } else {
                            sparksqlEngine[2] = true;
                        }
                        engines.push(sparksqlEngine);
                    }
                    return engines;
                }

                const useApplyCancelButtons = $scope.useUpperButtons && $scope.initialChart;

                $scope.objectName = translate("CHARTS.SAMPLING.OBJECT_NAME", "charts");

                $scope.$watch("chart", function () {
                    $scope.availableEngines = makeEnginesStatus($scope.script, $scope.chart.def);
                });

                /* Auto-revert to compatible settings */
                $scope.$watch("chart.engineType", function (nv, ov) {
                    if (!nv || !ov) return;

                    if ((nv == "SQL" || nv == "SPARKSQL") && !$scope.chart.refreshableSelection) {
                        $scope.chart.refreshableSelection = {
                            selection: {
                                samplingMethod: "FULL",
                                partitionSelectionMethod: "ALL"
                            }
                        }
                    }
                });

                $scope.save = function () {
                    if ($scope.chart.refreshableSelection != null) {
                        if (!$scope.chart.copySelectionFromScript) {
                            WT1.event("chart-sample-setting-update", {
                                chartId: `${$scope.dataset.projectKey.dkuHashCode()}.${$scope.dataset.name.dkuHashCode()}.${$scope.chart.def.name.dkuHashCode()}`,
                                analysis: false,
                                engineType: $scope.chart.engineType,
                                reuseFromExplore: false,
                                samplingMethod: $scope.chart.refreshableSelection.selection.samplingMethod,
                                recordsNumber: $scope.chart.refreshableSelection.selection.maxRecords,
                                targetRatio: $scope.chart.refreshableSelection.selection.targetRatio,
                                filtersNumber:
                                    $scope.chart.refreshableSelection.selection.filter &&
                                        $scope.chart.refreshableSelection.selection.filter.enabled &&
                                        $scope.chart.refreshableSelection.selection.filter.uiData &&
                                        $scope.chart.refreshableSelection.selection.filter.uiData.conditions
                                        ? $scope.chart.refreshableSelection.selection.filter.uiData.conditions.length
                                        : 0
                            });
                        } else {
                            WT1.event('chart-sample-setting-update', {
                                chartId: `${$scope.dataset.projectKey.dkuHashCode()}.${$scope.dataset.name.dkuHashCode()}.${$scope.chart.def.name.dkuHashCode()}`,
                                analysis: false,
                                engineType: $scope.chart.engineType,
                                reuseFromExplore: true
                            });
                        }
                        $scope.chart.refreshableSelection._refreshTrigger = new Date().getTime();
                    }

                    $rootScope.$emit("chartSamplingChanged", { chart: $scope.chart });

                    if (useApplyCancelButtons) {
                        $scope.initialChart = angular.copy($scope.chart); // new reference
                        $scope.samplingForm.$setPristine();
                    }
                };

                $scope.saveNoRefresh = function () {
                    $rootScope.$emit("chartSamplingChanged", { chart: $scope.chart });
                };

                if (useApplyCancelButtons) {

                    $scope.isDirty = function () {
                        return !angular.equals($scope.chart, $scope.initialChart);
                    };

                    $scope.cancel = function () {
                        $scope.chart = angular.copy($scope.initialChart);
                        $scope.samplingForm.$setPristine();
                    };
                }

            }
        }
    });

})();

;
(function() {
'use strict';

const app = angular.module('dataiku.shaker');


app.directive("shakerWithLibrary", function(PageSpecificTourService, OpalsService, OpalsMessageService) {
    return {
        scope:true,
        controller : function($scope, $rootScope, $timeout, $filter, ListFilter, ShakerPopupRegistry) {

            $scope.$on("paneSelected", function(e, pane) {
            	if ($scope.uiState) {
            		$scope.uiState.shakerLeftPane = pane.slug;
            	}
            });

            /* ******************* Processors library management *************** */

            Mousetrap.bind("esc", function() {
                if ($scope.shakerUIState.showProcessorsLibrary &&
                    !$(".library-search-input").is(":focus")) {
                    $scope.shakerUIState.showProcessorsLibrary = false;
                    $scope.$apply();
                }
            });
            $scope.$on("$destroy", function() {Mousetrap.unbind("esc")});

            $scope.displayProcessor = function(p) {
                $scope.shakerUIState.displayedProcessor = p;
            }

            $scope.displayProcessorByType = function(t) {
                const p = $scope.processors.processors.find(p => p.type == t);
                if (p) {
                    $scope.displayProcessor(p);
                }
            }

            $scope.toggleLibrary = function(show) {
                if (show === undefined) {
                    show = !$scope.shakerUIState.showProcessorsLibrary;
                }
                if (show) {
                    ShakerPopupRegistry.dismissAllAndRegister(function(){$scope.toggleLibrary(false);})
                    setupLibraryPopup();
                }
                $scope.shakerUIState.showProcessorsLibrary = show;
            }

            /**
             * Closes the right panel if the Conditional formatting panel is opened
             */
            $scope.closeRightPaneIfConditionalFormattingOpened = function () {
                if ($scope.isQuickConditionalFormattingViewOpened() && typeof $scope.closeRightPane === "function") {
                    $scope.shakerState.rightPaneView = $scope.RIGHT_PANE_VIEW.NONE;
                    $scope.closeRightPane();
                }
            };

            function setupLibraryPopup() {
                $timeout(function(){$(".library-search-input").focus()}, 0);

                $(".library-search-input").on("keyup", function(e) {
                    if (e.which == 27) {
                        $scope.toggleLibrary();
                        $scope.$apply();
                    }
                });

                $(".library-search-input").off("keyup").on("keyup", function(e) {
                    var s = $scope.shakerUIState.displayedProcessor;
                    var i = -1;

                    if (e.which === 27) {
                        $scope.shakerUIState.showProcessorsLibrary = false;
                        $scope.$apply();
                    }

                    if (s) {
                        i = $scope.filteredProcessors.indexOf(s);
                    }
                    if (e.which == 13 && s) {
                        $scope.toggleLibrary();
                        $scope.addUnconfiguredStep(s.type);
                        e.preventDefault();
                        e.stopPropagation();
                        $scope.$apply();
                        return false;
                    } else if (e.which == 40 && $scope.filteredProcessors.length) {
                        if (i == -1) {
                            i = 0;
                        } else if (i < $scope.filteredProcessors.length - 1) {
                            i++;
                        }
                        $scope.shakerUIState.displayedProcessor = $scope.filteredProcessors[i];
                    } else if (e.which == 38 && $scope.filteredProcessors.length) {
                        if (i >= 1) {
                            i--;
                            $scope.shakerUIState.displayedProcessor = $scope.filteredProcessors[i];
                        }
                    }
                    $scope.$apply();
                });
            }

            $scope.selectTag = function(tag) {
                if (tag.selected) {
                    $scope.processors.tags.forEach(function(x){x.selected=false});
                } else {
                    $scope.processors.tags.forEach(function(x){x.selected=false});
                    tag.selected = true;
                }
                $scope.refreshLibrarySearch();
            }

            $scope.refreshLibrarySearch = function () {
                if (!$scope.processors) return;

                let processors = $.grep(angular.copy($scope.processors.processors), function (processor) {
                    if (processor.plugin) {
                        var plugin = Array.dkuFindFn($rootScope.appConfig.loadedPlugins, function (n) {
                            return n.id == processor.plugin;
                        });

                        return plugin != null && !plugin.hideComponents && processor.displayInLibrary && !processor.disabledByAdmin;
                    }

                    return processor.displayInLibrary && !processor.disabledByAdmin;
                });
                if ($scope.shakerUIState.libraryQuery) {
                    processors = ListFilter.filter(processors, $scope.shakerUIState.libraryQuery);
                }

                const selectedTags = $scope.processors.tags.filter(function (tag) {
                    return tag.selected;
                }).map(function (tag) {
                    return tag.id;
                });

                // Facet
                $scope.shakerUIState.tagsCount = {};
                angular.forEach($scope.processors.tags, function (tag) {
                    $scope.shakerUIState.tagsCount[tag.id] = 0;
                    tag.selected = selectedTags.indexOf(tag.id) >= 0;
                });
                angular.forEach(processors, function (processor) {
                    angular.forEach(processor.tags, function (tag) {
                        $scope.shakerUIState.tagsCount[tag]++;
                    });
                });

                // Filter on tags
                if (selectedTags.length) {
                    angular.forEach(selectedTags, function (tag) {
                        processors = $.grep(processors, function (processor) {
                            return processor.tags && processor.tags.indexOf(tag) >= 0;
                        });
                    });
                }

                // Sort the list to show first the most relevant ones (i.e. the query matches the title)
                if ($scope.shakerUIState.libraryQuery) {
                    const queryString = $scope.shakerUIState.libraryQuery;
                    processors.forEach(p => { p.score = ListFilter.computeMatchScore(p.enDescription, queryString) })
                    processors.sort((a, b) => b.score - a.score);
                }

                // Remove displayed processor if not in the filtered results
                if ($scope.shakerUIState.displayedProcessor && processors.map(function (p) { return p.type; }).indexOf($scope.shakerUIState.displayedProcessor.type) < 0) {
                    delete $scope.shakerUIState.displayedProcessor;
                }

                $scope.filteredProcessors = processors;
            };

            Mousetrap.bind("a", function() {
                $scope.toggleLibrary();
                $scope.$apply();
            })
            $scope.$on("$destroy", function() {
                Mousetrap.unbind("a");
            });

            $scope.$watch("shakerUIState.libraryQuery", $scope.refreshLibrarySearch);
            $scope.$watch("processors", function(nv, ov) {
                if (nv) $scope.refreshLibrarySearch();
            });
            $scope.$watch("table", function() {
                if (PageSpecificTourService.canStartPrepareTour($scope)) {
                    PageSpecificTourService.startPrepareTour({ scope: $scope, fromContext: 'prepare-recipe' });
                    OpalsService.sendPageSpecificTourRecommendation(OpalsMessageService.PAGE_SPECIFIC_TOURS_RECOMMENDATIONS.PREPARE);
                } else {
                    OpalsService.sendPageSpecificTourRecommendation(null);
                }
            });

            const unregisterPrepareTourListener = $rootScope.$on('startPrepareTour', function() {
                PageSpecificTourService.startPrepareTour({ scope: $scope, fromContext: 'opals' });
            });
            $scope.$on("$destroy", function() {
                unregisterPrepareTourListener();
            });
        }
    }
});


app.directive("shakerWithProcessors", function($rootScope, translate, Assert, CreateModalFromTemplate, ShakerProcessorsInfo, ShakerProcessorsUtils, Logger, Dialogs, SpinnerService, FutureWatcher, DataikuAPI) {
    return {
        scope: true,
        controller: function($scope, $stateParams, $state, CachedAPICalls, $filter, TableChangePropagator, WT1, $timeout,$q, Fn, openDkuPopin, ClipboardUtils, ActivityIndicator, GenerateStepsService, RatingFeedbackParams){

            $scope.shakerUIState  = { selectedTags : [] };
            $scope.shakerState.withSteps = true;
            $scope.groupChanged = {justCreated: false, addedStepsTo: null, removedStepsFrom: []};

            // you're going to need them
            CachedAPICalls.processorsLibrary.success(function(processors){
                $scope.processors = processors;
            }).error(setErrorInScope.bind($scope));

            /*
             * Adding Step
             */

            /* When you add a step, the previous ones are not new anymore */
            function clearNewState(){
                function clearNewState_(step) {
                    if (step.$stepState) {
                        step.$stepState.isNew = false;
                        step.$stepState.isNewCopy = false;
                    }
                    if (step.metaType == "GROUP") {
                        step.steps.forEach(clearNewState_);
                    }
                }
                $scope.shaker.steps.forEach(clearNewState_);
            }

            const WT1StepEvents = {
                Add : 'shaker-step-add',
                Remove : 'shaker-step-remove',
                Enabled : 'shaker-step-enabled',
                Disabled : 'shaker-step-disabled',
            }

            function sendWT1EventOnStepChange(processorType, WT1StepEvent) {
                const datasetId = ($scope.dataset && $scope.dataset.name) ? md5($scope.dataset.name) : null;

                // We don't want to send any incomplete event
                if (!processorType || !datasetId) {
                    return;
                }
                WT1.tryEvent(WT1StepEvent, () => ({
                    processorType,
                    datasetId,
                    stepsCountAfter: ($scope.shaker && $scope.shaker.steps) ? $scope.getNumberOfSteps($scope.shaker.steps) : 0
                }));
            }

            $scope.addStep = function(processor, params, keepClosed, onOpenCallback) {
                clearNewState();
                if (angular.isString(processor)) {
                    processor = $filter('processorByType')($scope.processors, processor)
                }
                $scope.stopPreview(true);
                const processorType = processor.type;
                const step = {
                    type: processorType,
                    preview: true,
                    params: params
                };
                if (!keepClosed) {
                    $scope.openStep(step, onOpenCallback);
                }
                $scope.shaker.steps.push(step);
                sendWT1EventOnStepChange(processorType, WT1StepEvents.Add);
            }


            $scope.addGeneratedSteps = function (steps, queryPrompt) {
                clearNewState();
                $scope.stopPreview(true);
            
                if (!steps || steps.length === 0) {
                    return;
                }
            
                if (steps.length === 1) {
                    // If there is only one step, add it directly
                    steps[0].preview = true;
                    steps[0].comment = `Step generated using prompt : ${queryPrompt}`;
                    $scope.shaker.steps.push(steps[0]);
                } else if (steps.length > 1)  {
                    // If there are multiple steps, create a group
                    const groupStep = {
                        metaType: "GROUP",
                        steps: steps, 
                        forceOpenOnCreation: true,
                        preview: true,
                        comment: `Steps generated using prompt : ${queryPrompt}`
                    };
                    $scope.shaker.steps.push(groupStep);
                }

                $scope.generateStepsData.invalidGeneration = steps.some(step => step.disabled === true);
                
            };
            

            $scope.addStepAndRefresh = function(processor, params, keepClosed) {
                clearNewState();
                $scope.addStep(processor, params, keepClosed);
                $scope.autoSaveForceRefresh();
            }

            $scope.addStepNoPreview = function(processor, params, keepClosed) {
                clearNewState();
                $scope.addStep(processor, params, keepClosed);
                $scope.shaker.steps[$scope.shaker.steps.length-1].preview = false;
            }

            $scope.addStepNoPreviewAndRefresh = function(processor, params, keepClosed) {
                clearNewState();
                $scope.addStep(processor, params, keepClosed);
                $scope.shaker.steps[$scope.shaker.steps.length-1].preview = false;
                $scope.autoSaveForceRefresh();
            }

            $scope.addUnconfiguredStep = function(type, params, idx, preview = true) {
                clearNewState();
                var processor = $filter('processorByType')($scope.processors, type);
                if (angular.isUndefined(params)) {
                    if (processor.defaultParams) {
                        params = angular.copy(processor.defaultParams);
                    } else {
                        params = {}
                    }
                    angular.forEach(processor.params, function(pparam){
                        if (pparam.defaultValue) {
                            params[pparam.name] = angular.copy(pparam.defaultValue);
                        }
                    });
                }
                $scope.stopPreview(true);

                const processorType = processor.type;
                const step = {
                    type: processorType,
                    preview: preview,
                    isNew : true,
                    params: params,

                    $stepState: {
                        justCreated: true, // marks a step that has been created very recently - for the step controller to make a specific action on step creation if required
                        isNew : true,
                        change: {
                           columnsBeforeStep: $scope.columns
                        }
                    }
                };
                if (idx) {
                    $scope.shaker.steps.splice(idx, 0, step);
                } else {
                    $scope.shaker.steps.push(step);
                }
                $scope.openStep(step);

                setTimeout(() => {
                    // fully scroll the new step into view (as close as the top as possible, which will show as much as the deployed step as possible, and even the new step / group buttons if there is enough space)
                    const firstMatchDomIndex = $scope.findStepFlattenIndex(step);
                    $('.processor')[firstMatchDomIndex].scrollIntoView();

                    step.$stepState.justCreated = false; // step controller is now fully initialized
                }, 100) // we need a small delay for the step to be fully rendered
                sendWT1EventOnStepChange(processorType, WT1StepEvents.Add);
                return step;
            }

            $scope.duplicateStep = function(step){
                $scope.disablePreviewOnAllSteps();
                var newStep = angular.copy(step);
                if (typeof(newStep.name)!=='undefined' && newStep.name.length > 0) {
                    var suffix = ' (copy)';
                    if (newStep.name.indexOf(suffix, newStep.name.length - suffix.length) === -1) {
                        newStep.name += ' (copy)';
                    }
                }
                var stepId = $scope.findStepId(step);
                if (stepId.depth == 1) {
                	var group = $scope.shaker.steps[stepId.id];
                	group.steps.splice(stepId.subIndex + 1, 0, newStep);
                } else {
                	$scope.shaker.steps.splice(stepId.id + 1, 0, newStep);
                }
                $scope.currentStep = newStep;
                $scope.autoSaveForceRefresh();
                sendWT1EventOnStepChange(step.type, WT1StepEvents.Add);
            }

            $scope.appendGroup = function(){
                $scope.stopPreview(true);
                var group = {
                    metaType : "GROUP",
                    steps : []
                }
                $scope.shaker.steps.push(group);
                if (!$scope.isRecipe){
                	$scope.saveOnly();
                }
                $scope.groupChanged.justCreated = true;
            }

            //TODO: to remove ?
            $scope.addStepToPrevGroup = function(step){
                var lastGroup = null, stepIdx = -1;
                for (let i = 0; i < $scope.shaker.steps.length; i++) {
                    if ($scope.shaker.steps[i].metaType == 'GROUP') {
                        lastGroup = $scope.shaker.steps[i];
                    }
                    if ($scope.shaker.steps[i] == step) {
                        stepIdx = i;
                        break;
                    }
                }
                if (!lastGroup) {
                    Logger.error("No group before step!");
                } else {
                    lastGroup.steps.push(step);
                    $scope.shaker.steps.splice(stepIdx, 1);
                }
                sendWT1EventOnStepChange(step.type, WT1StepEvents.Add);
            }

            /*
             * Removing Step
             */

            var removeStepNoRefresh = function(step, doesWT1EventTriggered = true) {
                //removing step from shaker.steps
                var stepId = $scope.findStepId(step);
                if (typeof(stepId)!=='undefined') {
                    if (stepId.depth == 0) {
                        $scope.shaker.steps.splice(stepId.id, 1);
                    } else if (stepId.depth == 1) {
                        $scope.shaker.steps[stepId.id].steps.splice(stepId.subId, 1);
                    }

                    if(doesWT1EventTriggered) {
                        sendWT1EventOnStepChange(step.type, WT1StepEvents.Remove);
                    }
                }

                // Group case : Sending WT1 events for each steps
                if(step.metaType === 'GROUP' && step.steps.length > 0 && doesWT1EventTriggered) {
                    step.steps.forEach(s => sendWT1EventOnStepChange(s.type, WT1StepEvents.Remove));
                }
            }

            $scope.removeStep = function(step, saveAndRefresh) {
                removeStepNoRefresh(step);
                $scope.autoSaveForceRefresh();
            };

            /*
             * Generate steps in Prepare (AI Completion)
             */
            
            $scope.ratingFeedbackParams = RatingFeedbackParams;
            
            $scope.resetGenerateStepsData = function (previousRequestId = null) {
                $scope.generateStepsData = {
                    generateStepQuery: "",
                    jobId: null,
                    response: null,
                    warning: null,
                    error: null,
                    errorMessage: "",
                    isFetchingQueryResult: false,
                    isQueryIncorrect: false,
                    previousRequestId: previousRequestId
                };
            }

            $scope.resetGenerateStepsData();


            $scope.openGenerateStepsContainer = function () {
              WT1.event("prepare-generate-steps-open", {
                aiServer: $rootScope.appConfig.isUsingLocalAiAssitant && $rootScope.appConfig.isUsingLocalAiAssitant.prepareAICompletion ? "webapp" : "default"
              });
                $scope.generatingSteps = true;
                $scope.scrollDownWrapper();
                $scope.focusGenerateStepsTextArea();                
            }

            $scope.closeGenerateStepsContainer = function () {
              WT1.event("prepare-generate-steps-close", {
                aiServer: $rootScope.appConfig.isUsingLocalAiAssitant && $rootScope.appConfig.isUsingLocalAiAssitant.prepareAICompletion ? "webapp" : "default"
              });
                if ($scope.generateStepsData.isFetchingQueryResult) {
                    $scope.abortGenerateSteps();
                }
                $scope.resetGenerateStepsData($scope.generateStepsData.previousRequestId);

                $scope.generatingSteps = false;

                if ($scope.generateStepsData.previousRequestId && !$rootScope.appConfig.isUsingLocalAiAssitant?.prepareAICompletion) {
                    $scope.ratingFeedbackParams.requestIdForFeedback = $scope.generateStepsData.previousRequestId;
                    $scope.ratingFeedbackParams.featureRated = "generateSteps";
                    $scope.ratingFeedbackParams.showRatingFeedback = true;
                }
            };

            

            $scope.onGenerateStepsKeydown = function(event) {
                if (event.ctrlKey || event.metaKey) {
                    if (event.key === "Enter") {
                        event.preventDefault();
                        
                        if (!$scope.generateStepsData.generateStepQuery ||
                            $scope.generateStepsData.isQueryIncorrect ||
                            $scope.generateStepsData.isFetchingQueryResult) {
                            return;
                        }
        
                        $scope.startGenerateSteps();
                    }
                }
            };

            $scope.clearGenerateStepsError = function () {
                $scope.generateStepsData.error = false;
                $scope.generateStepsData.errorMessage = "";
            }

            $scope.startGenerateSteps = function () {
              WT1.event("prepare-generate-steps-request-start", {
                aiServer: $rootScope.appConfig.isUsingLocalAiAssitant && $rootScope.appConfig.isUsingLocalAiAssitant.prepareAICompletion ? "webapp" : "default"
              });
                $scope.generateStepsLastQueryPrompt = $scope.generateStepsData.generateStepQuery;
                const query = $scope.generateStepsData.generateStepQuery
                const shakerForQuery = $scope.shakerHooks.shakerForQuery();
                $scope.generateStepsData.isFetchingQueryResult = true;
                $scope.ratingFeedbackParams.showRatingFeedback = false;


                GenerateStepsService.complete($scope.inputDatasetProjectKey, $scope.inputDatasetName, shakerForQuery, $scope.requestedSampleId, query, '').success(function (initialResponse) {
                    $scope.generateStepsData.jobId = initialResponse.jobId;
                    SpinnerService.lockOnPromise(
                        FutureWatcher.watchJobId($scope.generateStepsData.jobId)
                            .success((data) => {
                                Logger.info(data);

                                if (data.aborted) return;

                                if (data.result.ok) {
                                  WT1.event("prepare-generate-steps-request-successful", {
                                    aiServer: $rootScope.appConfig.isUsingLocalAiAssitant && $rootScope.appConfig.isUsingLocalAiAssitant.prepareAICompletion ? "webapp" : "default"
                                  });
                                    $scope.generateStepsData.response = data.result;
                                    $scope.generateStepsData.error = false;
                                    $scope.generateStepsData.generateStepQuery = '';
                                    const steps = $scope.generateStepsData.response.steps;
                                    $scope.generateStepsData.warning = steps.some(step => step.disabled);
                                    $scope.generateStepsData.previousRequestId = $scope.generateStepsData.response.requestId;
                                    $scope.addGeneratedSteps(steps, query)
                                    $scope.autoSaveAutoRefresh();
                                    $scope.focusGenerateStepsTextArea();
                                } else {
                                  WT1.event("prepare-generate-steps-request-failed", {
                                    aiServer: $rootScope.appConfig.isUsingLocalAiAssitant && $rootScope.appConfig.isUsingLocalAiAssitant.prepareAICompletion ? "webapp" : "default"
                                  });
                                    $scope.generateStepsData.error = true;
                                    if( !data.result.reason || data.result.reason === "AI completion was not able to suggest valid actions") {
                                        data.result.reason = translate("SHAKER.AI_PREPARE.ERROR", "We couldn't find what you meant. Please rewrite your prompt.");
                                    }
                                    $scope.generateStepsData.errorMessage = data.result.reason;
                                    $scope.generateStepsData.isQueryIncorrect = true;
                                }
                            })
                            .finally(() => {
                                $scope.generateStepsData.isFetchingQueryResult = false;
                                $scope.scrollDownWrapper();
                            })
                    );
                }).error(setErrorInScope.bind($scope));
            }

            $scope.abortGenerateSteps = function () {
              WT1.event("prepare-generate-steps-request-aborted", {
                aiServer: $rootScope.appConfig.isUsingLocalAiAssitant && $rootScope.appConfig.isUsingLocalAiAssitant.prepareAICompletion ? "webapp" : "default"
              });
                GenerateStepsService.abortGenerateSteps($scope.generateStepsData.jobId).error(setErrorInScope.bind($scope));
            }

            $scope.applyGenerateStepsBoilerplate = function (query) {
                $scope.generateStepsData.generateStepQuery = query;
                $scope.focusGenerateStepsTextArea();
            };

            $scope.applyGenerateStepsLastPrompt = function () {
                $scope.generateStepsData.generateStepQuery = $scope.generateStepsLastQueryPrompt;
                $scope.focusGenerateStepsTextArea();
            };

            $scope.focusGenerateStepsTextArea = function () {
                $timeout(function() {
                    const textarea = document.getElementById('generateStepsTextarea');
                    if (textarea) {
                        textarea.focus();
                        const length = textarea.value.length;
                        textarea.setSelectionRange(length, length);
                    }
                }, 10);
            }

            // to scroll down the shaker tab after clicking the generate steps button
            $scope.scrollDownWrapper = function () {
                setTimeout(() => {
                    const element = document.querySelector('.steps-wrapper');
                    if (element) {
                        element.scrollTop = element.scrollHeight;
                    }
                }, 10);
            }

            $scope.onGenerateStepQueryChange = function () {
                $scope.generateStepsData.isQueryIncorrect = false;
            }

            /*
             * Reordering Steps
             */

            $scope.afterStepMove = function(){
                $scope.stopPreview(true);
                $scope.autoSaveAutoRefresh();
            }

            $scope.treeOptions = {
                dropped: $scope.afterStepMove,
                accept: function(sourceNodeScope, destNodesScope, destIndex) {
                    return destNodesScope.depth() == 0 || sourceNodeScope.$modelValue.metaType != 'GROUP';
                }
            }

            /*
             * Disabling steps
             */

            $scope.toggleDisable = function(step) {
                toggleDisableNoRefresh(step);
                $scope.isManualToggleDisable = true;
                $scope.autoSaveForceRefresh();
            }

            var toggleDisableNoRefresh = function(step) {
                step.disabled = !step.disabled;
                onDisableChange(step);
            }

            var enableStepNoRefresh = function(step) {
                step.disabled = false;
                onDisableChange(step);

            }

            var disableStepNoRefresh = function(step) {
                step.disabled = true;
                onDisableChange(step);
            }

            $scope.isAllStepsDisabled = function() {
            	return typeof($scope.shaker) === 'undefined' || typeof($scope.shaker.steps) === 'undefined' || isAllStepsInArrayDisabled($scope.shaker.steps);
            }

            var isAllStepsInArrayDisabled = function(steps) {
    			for (var id = 0; id < steps.length; id ++) {
    				var step = steps[id];
            		if (!step.disabled) {
            			return false;
            		}
            		if (step.metaType == 'GROUP') {
            			if (!isAllStepsInArrayDisabled(step.steps)) {
            				return false;
            			}
            		}
    			}
    			return true;
            }

            var onDisableChange = function(step) {
                if (step.disabled) {
                    /* This step was enabled, also disable preview on it */
                    step.preview = false;
                    //if it's a group all nested processor are disabled too
                    if (step.metaType === 'GROUP') {
                        for (let i = 0; i<step.steps.length; i++) {
                            step.steps[i].disabled = true;
                            step.steps[i].preview = false;
                            sendWT1EventOnStepChange(step.steps[i].type, WT1StepEvents.Disabled);
                        }
                    }
                    else {
                        sendWT1EventOnStepChange(step.type, WT1StepEvents.Disabled);
                    }
                } else {
                    if (step.metaType === 'GROUP') {
                        for (let i = 0; i<step.steps.length; i++) {
                            step.steps[i].disabled = false;
                            sendWT1EventOnStepChange(step.steps[i].type, WT1StepEvents.Enabled);
                        }
                    } else {
                        var stepId = $scope.findStepId(step);
                        if (stepId.depth === 1) {
                            $scope.shaker.steps[stepId.id].disabled = false;

                        }
                        sendWT1EventOnStepChange(step.type, WT1StepEvents.Enabled);
                    }
                }
            }

            /*
             * Previewing steps
             */

            $scope.togglePreview = function(step) {
                if (step.preview) {
                    /* Disable preview : disable it everywhere */
                    $scope.stopPreview(true);
                } else {
                    $scope.stopPreview(true);

                    /* Enable it here */
                    step.preview = true;
                    /* And mark further steps as softdisabled */
                    $scope.markSoftDisabled();
                }
                $scope.autoSaveForceRefresh();
            }

            $scope.stopPreview = function(norefresh){
                function _disablePreviewOnStep(s) {
                    if (s.metaType == "GROUP") {
                        if (s.steps) {
                            s.steps.forEach(_disablePreviewOnStep);
                        }
                        s.preview = false;
                        if (s.$stepState) s.$stepState.softDisabled=false;
                    } else {
                        s.preview = false;
                        if (s.$stepState) s.$stepState.softDisabled=false;
                    }
                }
                /* Disable preview everywhere */
                $scope.shaker.steps.forEach(_disablePreviewOnStep);

                $scope.stepBeingPreviewed = null;

                if (!norefresh){
                    $scope.autoSaveForceRefresh();
                }
            }

            $scope.getStepBeingPreviewedDescription =function(){
                Assert.inScope($scope, 'stepBeingPreviewed');

                var processor = {
                    enDescription: "UNKNOWN",
                    deprecated: $scope.processors.deprecatedTypes && $scope.processors.deprecatedTypes.includes($scope.stepBeingPreviewed.type)  // so deprecation message is handled correctly by getStepDescription
                }
                if ($scope.stepBeingPreviewed.metaType == "GROUP") {
                    return $scope.getGroupName($scope.stepBeingPreviewed);
                } else {
                    return ShakerProcessorsUtils.getStepDescription(processor, $scope.stepBeingPreviewed.type, $scope.stepBeingPreviewed.params);
                }
            }
            $scope.getStepBeingPreviewedImpactVerb =function(){
                Assert.inScope($scope, 'stepBeingPreviewed');
                return ShakerProcessorsUtils.getStepImpactVerb($scope.stepBeingPreviewed.type, $scope.stepBeingPreviewed.params);
            }

            $scope.disablePreviewOnAllSteps = function() {
                for (let i = 0; i < $scope.shaker.steps.length; i++) {
                    $scope.shaker.steps[i].preview = false;
                    if ($scope.shaker.steps[i].metaType == 'GROUP' && $scope.shaker.steps[i].steps && $scope.shaker.steps[i].length > 0) {
                        for (var j=0; j<$scope.shaker.steps[i].steps.length; j++) {
                            $scope.shaker.steps[i].steps[j].preview = false;
                        }
                    }
                }
            }

            /*
             * Copy/Paste steps
             */
            let copyType = 'shaker-steps';

            function sanitizeSteps(data) {
                let steps = data;
                // ensure steps are in order they appear in the shaker
                // list so order is preserved when pasting
                steps = sortSteps(steps);
                // if selecting a group, ensure that the substeps
                // aren't included twice in the data
                steps = removeExtraChildren(steps);

                return steps;
            }

            /*
                Copy JSON of steps to clipboard
            */
            $scope.copyData = function(data) {
                let copy = {
                    "type": copyType,
                    "version": $scope.appConfig.version.product_version,
                    "steps": sanitizeSteps(data)
                };

                // this removes all instances of the keys, including substeps
                const dataStr = JSON.stringify(copy, (key, value) => {
                    let keysToRemove = ['$$hashKey', '$stepState', '$translatability'];

                    return keysToRemove.includes(key) ? undefined : value;
                }, 2);
                const stepCount = $scope.getNumberOfSteps(copy.steps);
                const plural = stepCount > 1 ? 's' : '';

                ClipboardUtils.copyToClipboard(dataStr, `Copied ${stepCount} step${plural} to clipboard.`);
            }

            // steps: list of existing steps describing where
            // to insert the new steps
            $scope.openPasteModalFromStep = function(steps) {
                let newScope = $scope.$new();
                // ensure existing steps are in the correct order so
                // we know where to insert the pasted steps
                steps = sortSteps(steps);
                $scope.insertAfter = steps[steps.length - 1];

                CreateModalFromTemplate("/templates/shaker/paste-steps-modal.html", newScope, 'PasteModalController', function(modalScope) {
                    modalScope.copyType = copyType;
                    modalScope.formatData = $scope.formatStepData;
                    modalScope.itemKey = 'steps';
                    modalScope.pasteItems = $scope.pasteSteps;
                });
            };

            $scope.formatStepData = function(steps) {
                if ($scope.insertAfter) {
                    const stepId = $scope.findStepId($scope.insertAfter);
                    
                    if (stepId.depth === 1) {
                        // flatten any groups so we don't have groups within groups
                        steps = steps.reduce((acc, c) => acc.concat(c.metaType === 'GROUP' ? c.steps : c), []);
                    }
                }

                steps.forEach(_ => {
                    const name = _.name;
                    if (typeof name !== 'undefined' && name.length > 0) {
                        const suffix = ' (copy)';
                        if (name.indexOf(suffix, name.length - suffix.length) === -1) {
                            _.name += ' (copy)';
                        }
                    }

                    _.$stepState = _.$stepState || {
                        isNewCopy: true
                    };
                    _.selected = true;
                    _.preview = false;
                });

                return steps;
            };

            $scope.pasteSteps = function(steps) {
                let insertAt = $scope.shaker.steps.length;
                let addTo = $scope.shaker.steps;
                
                if ($scope.insertAfter) {
                    const stepId = $scope.findStepId($scope.insertAfter);
                    insertAt = stepId.id + 1;

                    if (stepId.depth === 1) {
                        insertAt = stepId.subId + 1;
                        addTo = $scope.shaker.steps[stepId.id].steps;
                    }
                }

                if (steps && steps.length) {
                    $scope.pasting = true;
                    $scope.stopPreview();
                    $scope.unselectSteps();
                    clearNewState();

                    addTo.splice(insertAt, 0, ...steps);
                    
                    const stepCount = steps.length;
                    const stepText = stepCount + ' step' + stepCount > 1 ? 's' : '';
                    ActivityIndicator.success(`Pasted ${stepText} successfully.`, 5000);
                    
                    $scope.autoSaveAutoRefresh();
                    $scope.insertAfter = null;
                    $timeout(() => $scope.pasting = false);

                    steps.forEach(function(step){
                        if(step.metaType && step.metaType === "GROUP") {
                            step.steps.forEach(function(groupStep) {
                                sendWT1EventOnStepChange(groupStep.type, WT1StepEvents.Add);
                            })
                        }
                        else sendWT1EventOnStepChange(step.type, WT1StepEvents.Add);
                    });
                }
            };

            /*
                Called when user uses ctrl + v from within
                the shaker step list (not in the modal)

                Immediately show preview modal since we've already pasted
            */
            $scope.openPasteModalFromKeydown = function(data) {
                try {
                    data = JSON.parse(data);
                } catch(e) { /* Nothing for now */ }

                if (data && data.steps && data.steps.length && data.type === copyType) {
                    CreateModalFromTemplate("/templates/shaker/paste-steps-modal.html", $scope, 'PasteModalController', function(modalScope) {
                        modalScope.uiState.editMode = false;
                        modalScope.uiState.items = data.steps;
                        modalScope.uiState.type = data.type;
                        modalScope.pasteItems = $scope.pasteSteps;
                    });
                }
            }

            /*
                Called when user uses ctrl + c from within
                the shaker step list (not in the modal)
            */
            $scope.keydownCopy = function(event) {
                let selectedSteps = $scope.getSelectedSteps();
                        
                if (selectedSteps.length) {
                    $scope.copyData(selectedSteps);
                }
                
                event.currentTarget.focus();
            }

            /*
             * Displaying info to user
             */

            $scope.getGroupName = function(step) {
                if (step.metaType == 'GROUP') {
                    return step.name && step.name.length>0 ? step.name : 'GROUP ' + $scope.findGroupIndex(step);
                }
            }

            $scope.getGroupNameInputId = function(step) {
                if (step.metaType == 'GROUP') {
                    return "group_name_input_" + $scope.findGroupIndex(step);
                }
            }


            $scope.getScriptDesc = function() {
                const nbSteps = $scope.shaker && $scope.shaker.steps ? $scope.shaker.steps.length : 0;
                if (nbSteps === 0) {
                    return translate('SHAKER.SCRIPT_DESCRIPTION.NO_STEPS', "no steps");
                } else {
                    return translate('SHAKER.SCRIPT_DESCRIPTION.STEPS', "<strong>{{count}}</strong> {{count === 1 ? 'step' : 'steps'}}", {
                        count: nbSteps
                    });
                }
            };

            /**
             * Some messages are not relevant for all engines.
             * Here we filter the messages displayed based on the currently selected engine.
             */
            function getFilteredMessagesByEngine(step) {
                if(!step.$stepState.change || !step.$stepState.change.messages || !Array.isArray(step.$stepState.change.messages)) {
                    return [];
                }
                const selectedEngine = $scope.recipeStatus && $scope.recipeStatus.selectedEngine && $scope.recipeStatus.selectedEngine.type;
                if(!selectedEngine) { // engine is undefined in analysis => we don't filter anything
                    return step.$stepState.change.messages;
                } else {
                    return step.$stepState.change.messages.filter(m => (
                        !(m.code === 'WARN_DELETED_COLUMN_MAY_BE_USED' && selectedEngine !== 'DSS') // show WARN_DELETED_COLUMN_MAY_BE_USED only for DSS engine
                    ));
                }
            }

            $scope.isStepInWarning = function(step) {
                return getFilteredMessagesByEngine(step).length > 0;
            };

            $scope.getWarningMessage = function(step) {
                var message = "";
                if (step.metaType == "GROUP") {
                    var warningList = "";
                    getFilteredMessagesByEngine(step).forEach(function(e) {
                        if (warningList.indexOf(e.title) == -1) {
                            if (warningList.length > 0) {
                                warningList +=", ";
                            }
                            warningList +="<b>" + e.title + "</b>";
                        }
                    });
                    message = "<h5>" + "Inner warning(s)" + "</h5>" + "<p>" + "Some inner step(s) have warning(s) (" + warningList + "), open group for more information." + "</p>";
                } else {
                    getFilteredMessagesByEngine(step).forEach(function(m) {
                        message += "<h5>" + m.title + "</h5>" + "<p>" + m.details + "</p>";
                    });
                }
                return message;
            }

            /*
             * Group Utils: used to situate a step is the steps tree
             */

            // Given a step object, returns its id in the script or undefined if it not in the list.
            // N.B.: the function is 'public' because the formula processors need it to send the position of the
            // step they're validating to the backend.
            $scope.findStepId = function(step) {
                var steps = $scope.shaker.steps;
                for (var stepId=0; stepId <steps.length; stepId++) {
                    if (steps[stepId] === step) {
                        return {'id':stepId, 'subId':undefined, 'depth':0};
                    }
                    if (steps[stepId].metaType == "GROUP") {
                        for (var subStepId=0; subStepId<steps[stepId].steps.length; subStepId++) {
                            var subStep = steps[stepId].steps[subStepId];
                            if (step == subStep) {
                                return {'id':stepId, 'subId':subStepId, 'depth':1};
                            }
                        }
                    }
                }
                return undefined;
            };

            $scope.findStepFlattenIndex = function(step) {
                let counter = 0;
                
                var findStepFlattenIndexInArray = function(arr, step) {
                    for (let i = 0; i<arr.length; i++) {
                        var currStep = arr[i];
                        if (currStep==step) {
                            return counter;
                        } else {
                            counter++;
                            if (currStep.metaType == 'GROUP') {
                                var recRet = findStepFlattenIndexInArray(currStep.steps, step);
                                if (recRet != -1) {
                                    return recRet;
                                }
                            }
                        }
                    }
                    return -1;
                }
                return findStepFlattenIndexInArray($scope.shaker.steps, step);
            }

            $scope.recursiveStepsFilter = function(filteringProp) {
                var recursiveStepsFilterInArray = function(arr, filteringProp) {
                    var filteredList = arr.filter(Fn.prop(filteringProp));
                    var groups = arr.filter(function(s) { return s.metaType === 'GROUP'; })
                    for (let i = 0; i < groups.length; i++) {
                        filteredList = filteredList.concat(recursiveStepsFilterInArray(groups[i].steps, filteringProp));
                    }
                    return filteredList;
                }
                return recursiveStepsFilterInArray($scope.shaker.steps, filteringProp);
            }

            $scope.findGroupIndex = null;
            $scope.$watch('shaker.steps', function(nv, ov) {
                if (nv) {
                    var tmpFindGroupIndex = Array.prototype.indexOf.bind($scope.shaker.steps.filter(function(s) { return s.metaType === 'GROUP'; }));
                    $scope.findGroupIndex = function(step) {
                        const groupIndex = tmpFindGroupIndex(step);
                        return groupIndex < 0 ? '' : groupIndex + 1;
                    }
                    if (nv.length > (ov ? ov.length : 0)) {
                        if ($scope.generateStepsData.isFetchingQueryResult) {
                            $scope.abortGenerateSteps();
                        }
                    }
                }
            }, true);

            var isGroupFatherOfStep = function (group, step) {
                if (group.metaType != 'GROUP' || group.steps.length == 0 || step.metaType != 'PROCESSOR') {
                    return false;
                }
                for (var subId = 0; subId < group.steps.length; subId++) {
                    if (group.steps[subId] == step) {
                        return true;
                    }
                }
                return false;
            }

            // when selecting a group, if its children are also selected, make sure they aren't in both
            // the group's steps and in the main steps
            function removeExtraChildren(steps) {
                steps = angular.copy(steps);

                let groups = steps.filter(_ => _.metaType === 'GROUP');
              
                groups.forEach(group => {
                    const stepCount = group.steps.length;
                    const intersection = steps.filter(step => group.steps.indexOf(step) !== -1);

                    // if some substeps but not all are selected, only keep those
                    if (intersection.length && intersection.length !== stepCount) {
                        group.steps = intersection;
                    }
                });

                // if any group already includes the step, remove it
                return steps.filter(step => !groups.some(group => group.steps.includes(step)));
            }

            /*
                A group counts as 1 step, regardless if its children
                are selected or not
            */
            $scope.getNumberOfSteps = function(steps) {
                steps = removeExtraChildren(steps);
                const groups = steps.filter(_ => _.metaType === 'GROUP' && _.steps && _.steps.length);
                // don't include the group itself in the step count (but include its children)
                return groups.reduce((acc, _) => acc + _.steps.length, steps.length - groups.length);
            }
            
            // returns a sorted subset of steps based on the entire set of shaker steps
            function sortSteps(steps) {
                let indices = steps.map(_ => $scope.findStepFlattenIndex(_));

                return indices
                    .map((_, i) => i) // create array of numbers from 0 to steps.length
                    .sort((a, b) => indices[a] - indices[b]) // sort indices array 
                    .map(_ => steps[_]); // sort steps based on the indices
            }

            /*
             * Selecting
             */

            $scope.openStep = function(step, onOpenCallback) {
                $scope.currentStep = step;
                window.setTimeout(function() {
                	if (onOpenCallback && typeof(onOpenCallback) == "function") {
                		onOpenCallback(step);
                	} else {
                		$("ul.steps").find("div.active input[type='text']").first().focus();
                	}
                }, 0);
            }

            $scope.toggleStep = function(step) {
                if ($scope.currentStep != step) {
                    $scope.openStep(step);
                } else {
                    $scope.currentStep = null;
                }
            }

            $scope.toggleStepSelection = function(step, $event) {
                if (typeof $scope.activeShakerMenu === 'function') {
                    $scope.activeShakerMenu();
                }

                var selectedSteps = $scope.getSelectedSteps();
                if ($event.shiftKey && selectedSteps.length > 0) {
                    var range1 = getRangeStep(step, selectedSteps[0]);
                    var range2 = getRangeStep(step, selectedSteps[selectedSteps.length-1]);
                    var isAllStepInRange1Selected = range1.filter(Fn.prop('selected')).length == range1.length;
                    var isAllStepInRange2Selected = range2.filter(Fn.prop('selected')).length == range2.length;
                    var rangeToSelect;
                    if (isAllStepInRange1Selected && isAllStepInRange2Selected) {
                        rangeToSelect = range2.length < range1.length ? range2 : range1;
                        for (let i = 0; i<rangeToSelect.length; i++) {
                            rangeToSelect[i].selected = false;
                        }
                        step.selected = true;
                    } else {
                        rangeToSelect = range2.length > range1.length ? range2 : range1;
                        $scope.unselectSteps();
                        for (let i = 0; i<rangeToSelect.length; i++) {
                            rangeToSelect[i].selected = true;
                        }
                    }
                } else {
                    step.selected = !step.selected;
                    if (!step.selected && step.metaType == "GROUP") {
                        // unselecting a group => unselect its contents as well, otherwise you could not notice they're still selected (if the group is folded)
                        // in the other direction (selecting) it's fine
                        step.steps.forEach(function(subStep) {subStep.selected = false;});
                    }
                }
            }

            var getRangeStep = function(fromStep, toStep) {
                //Return next step in group, null if stepId is the last step in group
                var getNextStepInGroup = function(stepId, group) {
                    return stepId.depth!=0 && stepId.subId < group.steps.length - 1 ? group.steps[stepId.subId+1] : null;
                }
                //Return next step in level 0, null if stepId is last the last step in level 0
                var getNextStep = function(stepId) {
                    return stepId.depth==0 && stepId.id < $scope.shaker.steps.length - 1 ? $scope.shaker.steps[stepId.id + 1] : null;
                }
                // Return the next visual step
                var getNextStepMultipleLevel = function(step) {
                    var stepId = $scope.findStepId(step);
                    if (stepId.depth == 1) {
                        var group = $scope.shaker.steps[stepId.id];
                        return getNextStepInGroup(stepId, group)!=null ? getNextStepInGroup(stepId, group) : getNextStep({depth: 0, id : stepId.id, subId : undefined});
                    } else {
                        if (step.metaType != "GROUP") {
                            return getNextStep(stepId);
                        } else {
                            if (step.steps.length > 0) {
                                return step.steps[0];
                            } else {
                                return getNextStep(stepId);
                            }
                        }
                    }
                }
                // Returns range of step between toSteps and fromSteps included. fromStep must be before toStep (ie: fromStep's id must be inferior to toStep's id)
                // Returns null if toStep was never found while iterating (ie: toStep does not exist or toStep is before fromStep)
                var getRange = function(fromStep, toStep) {
                   var range = [];
                   var nextStep = fromStep;
                   while (nextStep!=toStep && nextStep!=null) {
                       range.push(nextStep);
                       nextStep = getNextStepMultipleLevel(nextStep);
                   }
                   range.push(nextStep);
                   return nextStep ? range : null;
                }
                // compare fromStep's id and toStep's id and call getRange
                var c = compareStepId($scope.findStepId(fromStep), $scope.findStepId(toStep));
                if (c == 0) {
                    return [fromStep];
                } else if (c < 0 ) {
                    return getRange(fromStep, toStep);
                }  else {
                    return getRange(toStep, fromStep);
                }
            }

            var compareStepId = function(stepId1, stepId2) {
                if (stepId1.id != stepId2.id) {
                    return stepId1.id > stepId2.id ? 1 : -1;
                }
                var subId1 = typeof(stepId1.subId)!=='undefined' ? stepId1.subId : -1;
                var subId2 = typeof(stepId2.subId)!=='undefined' ? stepId2.subId : -1;
                if (subId1 == subId2) {
                    return 0;
                }
                return subId1 > subId2 ? 1 : -1;
            }

            /*
             * Search
             */
            $scope.query = {'val' : ''};

            $scope.searchSteps = function() {

                var searchStepArray = function(arr, processors, query) {
                    query = normalizeForSearch(query);
                    for (var id = 0; id < arr.length; id ++) {
                        var step = arr[id];
                        var stepDescription;
                        if (step.metaType == 'GROUP') {
                            stepDescription = $scope.getGroupName(step);
                            searchStepArray(step.steps, processors, query);
                        } else {
                            stepDescription = $scope.getStepDescription(step, processors);
                        }
                        stepDescription = stepDescription.toLowerCase();

                        step.match = stepDescription.indexOf(query) !== -1 || ShakerProcessorsUtils.matchStepParams(step, query);
                    }
                }

                var removeCloseOnMatchFlag = function() {
                    var stepsToUnflag = $scope.recursiveStepsFilter('closeOnMatch');
                    stepsToUnflag.forEach(function(el) {
                       delete el.closeOnMatch;
                    });
                }

                removeCloseOnMatchFlag();
                var query = $scope.query.val;
                if (query.length > 0) {
                    searchStepArray($scope.shaker.steps, $scope.processors, query);
                    var matchs = $scope.recursiveStepsFilter('match');
                    if (matchs.length > 0) {
                        var firstMatch = matchs[0];
                        var firstMatchDomIndex = $scope.findStepFlattenIndex(firstMatch);
                        $('.processor')[firstMatchDomIndex].scrollIntoView();
                    }
                } else {
                    $scope.unmatchSteps();
                }
            }

            $scope.getStepDescription = function(step, processors) {
                var processor = $filter('processorByType')($scope.processors, step.type);
                return ShakerProcessorsUtils.getStepDescription(processor, step.type, step.params);
            }

            $scope.unmatchSteps = function() {
                var unmatchStepsArray = function(arr) {
                    for (var id = 0; id < arr.length; id++) {
                        var step = arr[id];
                        step.match = false;
                        if (step.metaType == 'GROUP') {
                            unmatchStepsArray(step.steps);
                        }
                    }
                }
                unmatchStepsArray($scope.shaker.steps);
            };

            /*
             * Selecting
             */
            $scope.getSelectedSteps = function() {
                return $scope.recursiveStepsFilter('selected');
            }

            $scope.isAllStepsSelected = function() {
                var isAllStepsSelectedInArray = function(arr) {
                    for (let i = 0; i<arr.length; i++) {
                        var step = arr[i];
                        if (!step.selected) {
                            return false;
                        }
                        if (step.metaType == "GROUP" && !isAllStepsSelectedInArray(step.steps)) {
                            return false;
                        }
                    }
                    return true;
                }
                return typeof($scope.shaker) !== 'undefined' && isAllStepsSelectedInArray($scope.shaker.steps);
            }

            $scope.isNoStepSelected = function() {
                var isNoStepSelectedInArray = function(arr) {
                    for (let i = 0; i<arr.length; i++) {
                        var step = arr[i];
                        if (step.selected) {
                            return false;
                        }
                        if (step.metaType == "GROUP" && !isNoStepSelectedInArray(step.steps)) {
                            return false;
                        }
                    }
                    return true;
                }
                return typeof($scope.shaker) !== 'undefined' && isNoStepSelectedInArray($scope.shaker.steps);
            }

            $scope.selectAllSteps = function() {
                // on initial selection, only select steps currently filtered by search box
                const steps = $scope.recursiveStepsFilter('match').length && $scope.isNoStepSelected() ? $scope.recursiveStepsFilter('match') : $scope.shaker.steps;
                const selectAllStepsInArray = function(arr) {
                    for (let i = 0; i<arr.length; i++) {
                        var step = arr[i];
                        step.selected = true;
                        if (step.metaType == "GROUP") {
                            selectAllStepsInArray(step.steps);
                        }
                    }
                }
                selectAllStepsInArray(steps);
            }

            $scope.unselectSteps = function() {
                var unselectStepsInArray = function(arr) {
                    for (let i = 0; i<arr.length; i++) {
                        var step = arr[i];
                        step.selected = false;
                        if (step.metaType == "GROUP") {
                            unselectStepsInArray(step.steps);
                        }
                    }
                }
                unselectStepsInArray($scope.shaker.steps);
            }

            /*
             * Grouping
             */
            $scope.canGroupSelectedSteps = function() {
                var selectedSteps = $scope.getSelectedSteps();
                for (let i = 0; i < selectedSteps.length; i++) {
                    var step = selectedSteps[i];
                    if (step.metaType == 'GROUP' || $scope.findStepId(step).depth == 1) {
                        return false;
                    }
                }
                return selectedSteps.length > 1;
            }

            /*
             * Add to an existing group
             * 
             * Optional to pass a list of steps, otherwise just use what is selected
             */
            $scope.canAddMoreStepsToGroup = function(steps) {
                for (let i = 0; i < steps.length; i++) {
                    var step = steps[i];
                    if (step.metaType == 'GROUP') {
                        return false;
                    }
                }

                return steps.length && $scope.shaker.steps.filter(function(s) { return s.metaType === 'GROUP'; }).length;
            }

            $scope.addMoreStepsToGroup = function(group, steps) {
                if ($scope.canAddMoreStepsToGroup(steps)) {
                    const newSteps = steps.map(_ => {
                        _.selected = false
                        return _;
                    });                    
                    
                    //removing steps to be grouped from shaker's steps list
                    const removedGroups = [...new Set(steps.map(_ => $scope.findStepId(_)).map(_ => $scope.shaker.steps[_.id]))]
                    $scope.groupChanged.removedStepsFrom = removedGroups;
                    steps.forEach((_) => removeStepNoRefresh(_));

                    group.steps = group.steps.concat(newSteps);
                    $scope.groupChanged.addedStepsTo = group;

                    $scope.autoSaveForceRefresh();
                }
            };

            $scope.groupSelectedSteps = function() {
                var selectedSteps = $scope.getSelectedSteps();
                if ($scope.canGroupSelectedSteps()) {
                    // creating group
                    var group = {
                        metaType : "GROUP",
                        steps : []
                    }
                    //prepopulating its steps list in some extra array
                    var groupSteps = [];
                    for (let i = 0; i<selectedSteps.length; i++) {
                        let step = selectedSteps[i];
                        step.selected = false;
                        groupSteps.push(step);
                    }
                    //placing it in shaker's steps list
                    var groupId = $scope.findStepId(selectedSteps[0]).id;
                    $scope.shaker.steps[groupId] = group;
                    //removing steps to be grouped from shaker's steps list
                    for (let i = 0; i<selectedSteps.length; i++) {
                        let step = selectedSteps[i];
                        removeStepNoRefresh(step, false);
                    }
                    //finally setting new group's steps list
                    group.steps = groupSteps;
                    //saving
                    $scope.autoSaveForceRefresh();
                    $scope.groupChanged.justCreated = true;
                }
            }

            $scope.canUngroupSelectedSteps = function() {
                var selectedSteps = $scope.getSelectedSteps();
                for (let i = 0; i < selectedSteps.length; i++) {
                    var step = selectedSteps[i];
                    if (step.metaType != 'GROUP' && $scope.findStepId(step).depth == 0) {
                        return false;
                    }
                }
                return selectedSteps.length > 0;
            }

            $scope.ungroupSelectedSteps = function() {
                var selectedSteps = $scope.getSelectedSteps();
                var selectedProcessors = selectedSteps.filter(function(el) {
                    return el.metaType != 'GROUP';
                });
                var selectedGroups = selectedSteps.filter(function(el) {
                    return el.metaType == 'GROUP';
                });
                var unpopedGroups = [];
                for (let i = 0; i<selectedProcessors.length; i++) {
                    var step = selectedProcessors[i];
                    var stepId = $scope.findStepId(step);
                    let group = $scope.shaker.steps[stepId.id];
                    //is step is not among list of groups to ungroup we take it out of its group
                    if (selectedGroups.indexOf(group)==-1) {
                        group.steps.splice(stepId.subId, 1);
                        $scope.shaker.steps.splice(stepId.id,0,step);
                        //later we'll check if the group the current step used to belong to is now empty
                        if (unpopedGroups.indexOf(group)==-1) {
                            unpopedGroups.push(group);
                        }
                    }
                }
                //going through all groups unpopulated during previous loop and deleting theme if empty
                for (let i = 0; i<unpopedGroups.length; i++) {
                    let group = unpopedGroups[i];
                    if (group.steps.length == 0) {
                        var id = $scope.shaker.steps.indexOf(group);
                        $scope.shaker.steps.splice(id, 1);
                    }
                }
                for (let i = 0; i < selectedGroups.length; i++) {
                    $scope.ungroup(selectedGroups[i], true);
                }
                $scope.autoSaveForceRefresh();
            }

            /*
             * Deletes a group and puts all its steps at its previous index in the same order they used to be in the group
             */
            $scope.ungroup = function(step, noRefresh) {
                if (step.metaType == "GROUP") {
                    var groupIndex = $scope.findStepId(step).id;
                    var spliceArgs = [groupIndex, 1].concat(step.steps);
                    Array.prototype.splice.apply($scope.shaker.steps, spliceArgs);
                    if (!noRefresh) {
                        $scope.autoSaveForceRefresh();
                    }
                }
            }

            /*
             * Disabling
             */
            $scope.toggleDisableSelectedSteps = function() {
                var selectedSteps = $scope.getSelectedSteps();
                var allStepsDisabled = true;
                for (let i = 0; i<selectedSteps.length; i++) {
                    if (!selectedSteps[i].disabled) {
                        allStepsDisabled = false;
                        break;
                    }
                }
                for (let i = 0; i<selectedSteps.length; i++) {
                    if (allStepsDisabled) {
                        enableStepNoRefresh(selectedSteps[i]);
                    } else {
                        disableStepNoRefresh(selectedSteps[i]);
                    }
                }
                $scope.autoSaveForceRefresh();
            }

            /*
             * Deleting
             */
            var inModal = false;
            $scope.deleteSelectedSteps = function(evt) {
                // Delete selected steps, or all if
                // no step is selected.
                if (inModal)
                    return;
                if (evt.type=="keydown") {
                    var $focusedEl = $("input:focus, textarea:focus");
                    if ($focusedEl.length > 0) {
                        return;
                    }
                }

                // TODO prompt the user here.
                var stepsToDelete = $scope.getSelectedSteps();
                if (stepsToDelete.length == 0) {
                    stepsToDelete = $scope.shaker.steps;
                }
                if (stepsToDelete.length > 0) {
                    stepsToDelete = stepsToDelete.slice(0);
                    var dialogScope = $scope.$new();
                    dialogScope.stepsToDelete = stepsToDelete;
                    dialogScope.cancel = function() {
                        inModal = false;
                    }
                    dialogScope.perform = function() {
                        inModal = false;
                        for (let i = 0; i<stepsToDelete.length; i++) {
                            var step = stepsToDelete[i];
                            removeStepNoRefresh(step);
                        }
                        $scope.autoSaveForceRefresh();
                    }
                    inModal = true;
                    CreateModalFromTemplate("/templates/widgets/delete-step-dialog.html", dialogScope);
                }
            }


            $scope.remove = function(step) {
                $('.processor-help-popover').popover('hide');//hide any displayed help window
                $scope.removeStep(step.step);
            };

            /*
             * Coloring
             */
            $scope.uncolorStep = function(step) {
                delete step.mainColor;
                delete step.secondaryColor;

                if (!$scope.isRecipe){
                	$scope.saveOnly();
                }
            }

            $scope.colorStep = function(step, main, secondary) {
                step.mainColor = main;
                step.secondaryColor = secondary;

                if (!$scope.isRecipe){
                	$scope.saveOnly();
                }
            }

            $scope.uncolorSelectedSteps = function() {
                var selectedSteps = $scope.getSelectedSteps();
                for (let i = 0; i<selectedSteps.length; i++) {
                    var step = selectedSteps[i];
                    delete step.mainColor;
                    delete step.secondaryColor;
                }
                if (!$scope.isRecipe){
                	$scope.saveOnly();
                }
            }

            $scope.colorSelectedSteps = function(main, secondary) {
                var selectedSteps = $scope.getSelectedSteps();
                for (let i = 0; i<selectedSteps.length; i++) {
                    var step = selectedSteps[i];
                    step.mainColor = main;
                    step.secondaryColor = secondary;
                }
                if (!$scope.isRecipe){
                	$scope.saveOnly();
                }
            }

            /*
             * Validating scipt
             */

            /**
             * Performs JS validation of the whole script.
             * Sets frontError on all invalid steps.
             *
             * Returns true if script is ok, false if script is NOK
             * Does not count disabled step as invalid for the function result, but still set their frontError value if they are
             */
            $scope.validateScript = function() {
                var nbBadProc = 0;
                function validateProcessor(proc) {
                    if (!proc.$stepState) proc.$stepState = {}
                    proc.$stepState.frontError = $scope.validateStep(proc);
                    if (!proc.disabled && proc.$stepState.frontError) {
                        Logger.info("Bad processor: " + JSON.stringify(proc))
                        ++nbBadProc;
                    }
                }
                $scope.shaker.steps.forEach(function(step) {
                    if (step.metaType == "GROUP") {
                        step.steps.forEach(validateProcessor);
                    } else {
                        validateProcessor(step);
                    }
                });
                if (nbBadProc > 0) {
                    return false;
                } else {
                    return true;
                }
            }


            /* Perform JS validation of the step. Does not set frontError */
            $scope.validateStep = function(step) {
                if (step.metaType == "GROUP") {
                    if (step.steps != null) {
                        for (let i = 0; i < step.steps; i++) {
                            var subvalidationResult = $scope.validateStep(step.steps[i]);
                            if (subvalidationResult) return subvalidationResult;
                        }
                    }
                } else {
                    var processorType = $filter('processorByType')($scope.processors, step.type);
                    /* If we have some stepParams, then check using them */
                    if (processorType.params) {
                        for (var paramIdx in processorType.params) {
                            var param = processorType.params[paramIdx];
                            var value = step.params[param.name];
                            if (param.mandatory && !param.canBeEmpty && (value == null || value.length === 0)) {
                                // StepIAE is in library.js
                                // eslint-disable-next-line no-undef
                                return new StepIAE("Missing parameter: " + (param.label || param.name));
                            }
                        }
                    }
                    /* Then also play the specific validation of each step */
                    if (ShakerProcessorsInfo.get(step.type).checkValid){
                        try {
                            ShakerProcessorsInfo.get(step.type).checkValid(step.params);
                        } catch (e) {
                            return e;
                        }
                    }
                }
                return null;
            };

            /*
             * Factorising script steps
             */

            $scope.mergeLastColumnDeleters = function() {
                var deletedColumns = [];
                var deletedFromIndex = $scope.shaker.steps.length;
                for(var i = $scope.shaker.steps.length-1 ; i >= 0; i--) {
                    var step = $scope.shaker.steps[i];
                    if(step.type=='ColumnsSelector' && (step.params.appliesTo === "SINGLE_COLUMN" || step.params.appliesTo === "COLUMNS") && (step.params.keep=="false" || step.params.keep==false)) {
                        deletedColumns = deletedColumns.concat(step.params.columns);
                        deletedFromIndex = i;
                    } else {
                        break;
                    }
                }

                if(deletedColumns.length>0 && deletedFromIndex != $scope.shaker.steps.length) {
                    $scope.shaker.steps.splice(deletedFromIndex,$scope.shaker.steps.length-deletedFromIndex);
                    $scope.addStepNoPreview("ColumnsSelector", {
                        "appliesTo": deletedColumns.length > 1 ? "COLUMNS" : "SINGLE_COLUMN",
                        "keep": false,
                        "columns": deletedColumns
                    });
                }
            }

            $scope.mergeLastColumnRenamers = function() {
                var renamedColumns = [];
                var renamedFromIndex = $scope.shaker.steps.length;
                for(var i = $scope.shaker.steps.length-1 ; i >= 0; i--) {
                    var step = $scope.shaker.steps[i];
                    if(step.type=='ColumnRenamer') {
                        renamedColumns = step.params.renamings.concat(renamedColumns);
                        renamedFromIndex = i;
                    } else {
                        break;
                    }
                }

                if(renamedColumns.length>0 && renamedFromIndex != $scope.shaker.steps.length) {
                    $scope.shaker.steps.splice(renamedFromIndex,$scope.shaker.steps.length-renamedFromIndex);
                    $scope.addStepNoPreview("ColumnRenamer", {
                        "renamings": renamedColumns
                    });
                }
            }

            $scope.mergeLastColumnReorders = function() {
                // We'll only look at the last step and the step before...
                let stepCount = $scope.shaker.steps.length;
                if (stepCount < 2) {
                    return;
                }
                let lastStep = $scope.shaker.steps[stepCount - 1]; // last step
                let penultimateStep = $scope.shaker.steps[stepCount - 2]; // step before last step
                if (lastStep.type !== "ColumnReorder" || penultimateStep.type !== "ColumnReorder") {
                    return;
                }
                if ((lastStep.params.appliesTo !== "SINGLE_COLUMN" && lastStep.params.appliesTo !== "COLUMNS") ||
                    (penultimateStep.params.appliesTo !== "SINGLE_COLUMN" && penultimateStep.params.appliesTo !== "COLUMNS")) {
                    return;
                }
                // At this point the last two steps are ColumnReorder steps dealing with specific columns. Let's merge them if possible.

                // If the new step operates on a column that is already present in the penultimate step,
                // we remove this column from the penultimate step.
                let lastColumns = lastStep.params.columns;
                let lastAction = lastStep.params.reorderAction;
                let lastRefColumn = lastStep.params.referenceColumn;
                let penultimateColumns = penultimateStep.params.columns;
                let penultimateAction = penultimateStep.params.reorderAction;
                let penultimateRefColumn = penultimateStep.params.referenceColumn;

                penultimateColumns = penultimateColumns.filter(col => !lastColumns.includes(col));
                if (penultimateColumns.length === 0) {
                    // Penultimate step is now empty, remove it.
                    $scope.shaker.steps.splice(stepCount - 2, 2);
                    $scope.addStepNoPreview("ColumnReorder", lastStep.params);
                }

                // Merge the 2 steps if they both move the columns at start/end or before/after the same reference column.
                else if ((lastAction === "AT_END" && penultimateAction === "AT_END") ||
                        (lastAction === "BEFORE_COLUMN" && penultimateAction === "BEFORE_COLUMN" && lastRefColumn === penultimateRefColumn) ||
                        (lastAction === "AFTER_COLUMN" && penultimateAction === "AFTER_COLUMN" && lastRefColumn === penultimateRefColumn)) {
                    $scope.shaker.steps.splice(stepCount - 2, 2);
                    penultimateStep.params.columns = penultimateColumns.concat(lastColumns);
                    penultimateStep.params.appliesTo = "COLUMNS";
                    $scope.addStepNoPreview("ColumnReorder", penultimateStep.params);
                }
                else if ((lastAction === "AT_START" && penultimateAction === "AT_START")) {
                    $scope.shaker.steps.splice(stepCount - 2, 2);
                    penultimateStep.params.columns = lastColumns.concat(penultimateColumns);
                    penultimateStep.params.appliesTo = "COLUMNS";
                    $scope.addStepNoPreview("ColumnReorder", penultimateStep.params);
                }

                // Merge the 2 steps if the last step uses - as reference column - a column that is moved by the penultimate step.
                // (but not if one of the columns moved in the last steps are not a reference column in the penultimate step)
                else if ((lastAction === "BEFORE_COLUMN" || lastAction === "AFTER_COLUMN") && penultimateColumns.includes(lastRefColumn)
                        && !((penultimateAction === "BEFORE_COLUMN" || penultimateAction === "AFTER_COLUMN") && lastColumns.includes(penultimateRefColumn))) {
                    let columnIndex = penultimateColumns.indexOf(lastRefColumn);
                    if (lastAction === "AFTER_COLUMN") {
                        columnIndex++;
                    }
                    for (let i = 0; i < lastColumns.length; i++) {
                        let column = lastColumns[i];
                        penultimateColumns.splice(columnIndex + i, 0, column);
                    }

                    $scope.shaker.steps.splice(stepCount - 2, 2);
                    penultimateStep.params.columns = penultimateColumns;
                    penultimateStep.params.appliesTo = "COLUMNS";
                    $scope.addStepNoPreview("ColumnReorder", penultimateStep.params);
                }
            };

            $scope.mergeLastDeleteRows = function() {
                var firstVRProcessorIdx = $scope.shaker.steps.length;
                var relatedColumn = null, relatedAction = null;
                var defaults = {
                    appliesTo: 'SINGLE_COLUMN',
                    normalizationMode: 'EXACT',
                    matchingMode: 'FULL_STRING'
                };

                for(var i = $scope.shaker.steps.length - 1; i >= 0; i--) {
                    var step = $scope.shaker.steps[i];
                    if (step.type === 'FilterOnValue'
                            && step.params.appliesTo         === defaults.appliesTo
                            && step.params.matchingMode      === defaults.matchingMode
                            && step.params.normalizationMode === defaults.normalizationMode
                            && (relatedAction === null || step.params.action === relatedAction)
                            && step.params.columns && step.params.columns.length === 1 && step.params.columns[0]
                            && (relatedColumn === null || step.params.columns[0] === relatedColumn)) {
                        firstVRProcessorIdx = i;
                        relatedColumn = step.params.columns[0];
                        relatedAction = step.params.action;
                    } else {
                        break;
                    }
                }

                // Not enough processors to trigger a merge
                if($scope.shaker.steps.length - firstVRProcessorIdx - 1 < 1) {
                    return; // Not enough processors to trigger a merge
                }

                var valuesTotal = $scope.shaker.steps.slice(firstVRProcessorIdx).reduce(function (arr, step) {
                        return arr.concat(step.params.values);
                    }, []);
                // Remove previous processors
                $scope.shaker.steps.splice(firstVRProcessorIdx, $scope.shaker.steps.length - firstVRProcessorIdx);

                if (valuesTotal.length > 0) {
                    defaults.action = relatedAction;
                    defaults.columns = [relatedColumn];
                    defaults.values = valuesTotal;
                    $scope.addStep("FilterOnValue", defaults);
                }
            };

            $scope.mergeLastFindReplaces = function() {
                var firstVRProcessorIdx = $scope.shaker.steps.length;
                var relatedColumn = null;
                var defaults = {
                    appliesTo: 'SINGLE_COLUMN',
                    normalization: 'EXACT',
                    matching: 'FULL_STRING'
                };

                for (var i = $scope.shaker.steps.length - 1; i >= 0; i--) {
                    var step = $scope.shaker.steps[i];
                    if (step.type === 'FindReplace'
                            && step.params.appliesTo     === defaults.appliesTo
                            && step.params.matching      === defaults.matching
                            && step.params.normalization === defaults.normalization
                            && !step.params.output  // in-place only
                            && step.params.columns && step.params.columns.length === 1 && step.params.columns[0]
                            && (relatedColumn === null || step.params.columns[0] === relatedColumn)) {
                        firstVRProcessorIdx = i;
                        relatedColumn = step.params.columns[0];
                    } else {
                        break;
                    }
                }

                if($scope.shaker.steps.length - firstVRProcessorIdx - 1 < 1) {
                    return; // Not enough processors to trigger a merge
                }

                var mapping = [];
                // Mapping builder & merger
                function addMapping(add) {
                    if (add.from === null || add.from === undefined) return;
                    var updated = false;
                    // Apply transitivity
                    for (let i = 0; i < mapping.length; i++ ) {
                        let map = mapping[i];
                        if (map.to === add.from) {
                            map.to = add.to;
                        }
                    }
                    // Edit existing mapping for this input
                    for (let i = 0; i < mapping.length; i++ ) {
                        let map = mapping[i];
                        if(map.from === add.from) {
                            map.to = add.to;
                            updated = true;
                            break;
                        }
                    }
                    if (!updated) {
                        mapping.push(add);
                    }
                }

                // Build internal mapping
                for(let i = firstVRProcessorIdx; i < $scope.shaker.steps.length ; i++) {
                    $scope.shaker.steps[i].params.mapping.forEach(addMapping);
                }

                // Remove previous processors
                $scope.shaker.steps.splice(firstVRProcessorIdx,$scope.shaker.steps.length - firstVRProcessorIdx);

                if (mapping.length > 0) {
                    defaults.columns = [relatedColumn];
                    defaults.mapping = angular.copy(mapping);
                    defaults.mapping.push({ from: '', to: '' });
                    $scope.addStep("FindReplace", defaults, false, function(step) {
                        const inputs = $(".steps .active .editable-list__input");
                    	if (inputs.length > 1) {
                    	   $(inputs[inputs.length - 2]).focus();
                    	}
                    });
                }
            };

            /*
             * Column Reordering
             */

            // Callback called when dropping a column while reordering (see fatDraggable directive)
            $scope.reorderColumnCallback = function(draggedColumn, hoveredColumn, columnName, referenceColumnName) {
                let columnOldPosition;
                let columnNewPosition;
                let options = {};

                columnOldPosition = $scope.columns.indexOf(columnName);
                columnNewPosition = $scope.columns.indexOf(referenceColumnName);

                if (columnOldPosition < 0 || columnNewPosition < 0) {
                    return;
                }

                if (columnNewPosition === 0) {
                    options.reorderAction = "AT_START";
                } else if (columnNewPosition === $scope.columns.length - 1) {
                    options.reorderAction = "AT_END";
                } else if (columnOldPosition > columnNewPosition) {
                    options.reorderAction = "BEFORE_COLUMN";
                } else {
                    options.reorderAction = "AFTER_COLUMN";
                }

                options.appliesTo = "SINGLE_COLUMN";
                options.columns = [$scope.columns[columnOldPosition]];

                if (options.reorderAction === "BEFORE_COLUMN" || options.reorderAction === "AFTER_COLUMN") {
                    options.referenceColumn = $scope.columns[columnNewPosition];
                }

                $scope.addStepNoPreviewAndRefresh("ColumnReorder", options);
                $scope.mergeLastColumnReorders();
            };

            /*************************** OTHER ************************************/

            $scope.$watch("shaker.exploreUIParams.autoRefresh", function(nv, ov) {
                // tracking usage of the autorefresh button.
                if ((ov !== undefined) && (ov !== null) && (ov !== nv)) {
                    WT1.event("auto-refresh-set", {
                        "ov": ov,
                        "nv": nv
                    });
                }
            });

            /*
                Menu
            */

           $scope.previewTitle = function(step) {
            return step.preview ? translate("SHAKER.STEP.ACTIONS.STOP_PREVIEW", "Stop viewing impact") : translate("SHAKER.STEP.ACTIONS.START_PREVIEW", "View impact");
            };
            
            $scope.disableTitle = function(step) {
                return step.disabled ? translate("SHAKER.STEP.ACTIONS.ENABLE", "Enable step") : translate("SHAKER.STEP.ACTIONS.DISABLE", "Disable step");
            };

            $scope.openShakerMenu = function($event, step) {
                // only open menu if we aren't right clicking an input field
                if (!($event.target && ($event.target.tagName === 'INPUT' || $event.target.tagName === 'TEXTAREA'))) {
                    const selectedSteps = $scope.getSelectedSteps();
                    if (selectedSteps.length > 1 && step.selected) {
                        $scope.openActionsMenu($event);
                    } else {
                        $scope.openStepMenu($event, step, true);
                    }
                    
                    $event.preventDefault();
                    $event.stopPropagation();
                }
            }
 
            $scope.openStepMenu = function($event, step, showFullMenu) {
                // dismiss existing menu
                if (typeof $scope.activeShakerMenu === 'function') {
                    $scope.activeShakerMenu();
                }
                
                function isElsewhere() {
                    return true;
                }

                let newScope = $scope.$new();
                newScope.step = step;
                newScope.showFullMenu = showFullMenu;
                newScope.toggleComment = function($event) {
                    if (step.metaType === 'GROUP') {
                        $rootScope.$broadcast('openShakerGroup', step);
                    } else {
                        $scope.openStep(step);
                    }
                    $rootScope.$broadcast('toggleEditingComment', $event, step);
                };

                const template = `
                    <ul class="dropdown-menu" processor-footer>
                        <li ng-if="showFullMenu">
                            <a class="previewbutton"  ng-click="togglePreview(step);"
                                title="{{ previewTitle(step) }}" ng-if="!step.disabled" ng-class="{'previewActive': step.preview}">
                                <i alt="{{translate('SHAKER.STEP.ACTIONS.PREVIEW.ALT','Preview')}}" class="dku-icon-eye-16 dibvab"/> {{ previewTitle(step) }}
                            </a>
                        </li>
                        <!-- disable -->
                        <li ng-if="showFullMenu">
                            <a class="disablebutton" ng-click="toggleDisable(step);"
                                title="{{ disableTitle(step) }}">
                                <i alt="{{translate('SHAKER.STEP.ACTIONS.DISABLE.ALT','Disable')}}" class="dku-icon-power-16 dibvab" /> {{ disableTitle(step) }}
                            </a>
                        </li>
                        <li class="dropdown-submenu">
                            <a ng-if="canAddMoreStepsToGroup([step])"><i class="dku-icon-plus-16 dibvab"></i>&nbsp; <span translate="SHAKER.STEP.ACTIONS.ADD_TO_GROUP">Add to Group</span></a>
                            <ul class="dropdown-menu step-add-to-group-panel">
                                <li ng-repeat="group in shaker.steps | filter: { metaType: 'GROUP' }">
                                    <a ng-click="addMoreStepsToGroup(group, [step])">{{getGroupName(group)}}</a>
                                </li>
                            </ul>
                        </li>
                        <li>
                            <a class="previewbutton" id="qa_prepare_copy-single" ng-click="copyData([step]);"
                                title="Copy step">
                                <i alt="Copy step" class="dku-icon-copy-step-16 dibvab"/> <span translate="SHAKER.STEP.ACTIONS.COPY" translate-value-type="{{step.metaType}}">Copy this {{ step.metaType === 'GROUP' ? 'group' : 'step' }}</span> 
                            </a>
                        </li>
                        <li>
                            <a class="previewbutton" id="qa_prepare_open-paste-modal-single" ng-click="openPasteModalFromStep([step]);"
                                title="Paste after" >
                                <i alt="Paste after" class="dku-icon-copy-paste-step-16 dibvab"/> <span translate="SHAKER.STEP.ACTIONS.PASTE_AFTER" translate-value-type="{{step.metaType}}">Paste after this {{ step.metaType === 'GROUP' ? 'group' : 'step' }}</span>
                            </a>
                        </li>
                        <li>
                            <a title="comment" ng-click="toggleComment($event)">
                                <i class="dku-icon-info-circle-fill-16 dibvab"/> <span translate="SHAKER.STEP.ACTIONS.COMMENT">Comment</span>
                            </a>
                        </li>
                        <li class="dropup dropdown-submenu step-color-pannel" step-color-picker>
                            <a><i class="dku-icon-color-palette-16 dibvab"></i> <span translate="SHAKER.STEP.ACTIONS.COLOR">Color</span></a>
                            <ul class="dropdown-menu">
                                <li ng-click="uncolorStep(step)"><div class="color"></div></li>
                                <li ng-repeat="color in colors" ng-click="colorStep(step, color.main, color.secondary)">
                                    <div class="color" style="background-color:{{color.secondary}};border-color:{{color.main}}"></div>
                                </li>
                            </ul>
                        </li>
                        <li>
                            <a title="Duplicate step" ng-click="duplicateStep(step)">
                                <i class="dku-icon-copy-16 dibvab"/> <span translate="SHAKER.STEP.ACTIONS.DUPLICATE">Duplicate step</span></a>
                            </a>
                        </li>
                        <!-- delete -->
                        <li ng-if="showFullMenu">
                            <a ng-click="remove({step:step});" title="Delete step">
                                <i class="dku-icon-trash-16 dibvab"></i> <span translate="SHAKER.STEP.ACTIONS.DELETE">Delete step</span></a>
                            </a>
                        </li>
                    </ul>
                `
  
                let dkuPopinOptions = {
                    template: template,
                    isElsewhere: isElsewhere,
                    callback: null,
                    popinPosition: 'CLICK',
                    onDismiss: () => {
                        $scope.activeShakerMenu = null;
                        $scope.activeMenuType = null;
                    }
                };

                $scope.activeShakerMenu = openDkuPopin(newScope, $event, dkuPopinOptions);
            }

            $scope.openActionsMenu = function($event, menuType = 'CLICK') {
                // dismiss existing menu
                if (typeof $scope.activeShakerMenu === 'function') {
                    const previousMenuType = $scope.activeShakerMenuType;
                    $scope.activeShakerMenu();

                    // close actions dropdown if we clicked on it again
                    if (previousMenuType === menuType) {
                        return;
                    }
                }

                $scope.activeShakerMenuType = menuType;
                
                function isElsewhere() {
                    return true;
                }
                
                let newScope = $scope.$new();
                newScope.selectedSteps = $scope.getSelectedSteps();

                const template = `
                    <ul class="dropdown-menu shaker-column-row-popup">
                        <li class="dropdown-submenu">
                            <a ng-if="canAddMoreStepsToGroup(getSelectedSteps())"><i class="dku-icon-plus-16 dibvab"></i> {{::translate('SHAKER.STEPS.ACTIONS.ADD_STEPS_TO_GROUP', 'Add to Group')}}</a>
                            <ul class="dropdown-menu step-add-to-group-panel">
                                <li ng-repeat="group in shaker.steps | filter: { metaType: 'GROUP' }">
                                    <a ng-click="addMoreStepsToGroup(group, getSelectedSteps())">{{getGroupName(group)}}</a>
                                </li>
                            </ul>
                        </li>
                        <li><a ng-if="canGroupSelectedSteps()" ng-click="groupSelectedSteps()"><i class="dku-icon-folder-closed-16 dibvab"></i> {{::translate('SHAKER.STEPS.ACTIONS.GROUP', 'Group')}}</a></li>
                        <li><a ng-if="canUngroupSelectedSteps()" ng-click="ungroupSelectedSteps()"><i class="dku-icon-folder-open-16 dibvab"></i> {{::translate('SHAKER.STEPS.ACTIONS.UNGROUP', 'Ungroup')}}</a></li>
                        <li><a id="qa_prepare_copy-selection" ng-click="copyData(selectedSteps)"><i class="dku-icon-copy-step-16 dibvab" /> {{translate('SHAKER.STEPS.ACTIONS.COPY_STEPS', "Copy \\{\\{count\\}\\} \\{\\{count > 1 ? 'steps' : 'step'\\}\\}", {count:getNumberOfSteps(selectedSteps)})}}</a></li>
                        <li><a id="qa_prepare_open-paste-modal-selection" ng-click="openPasteModalFromStep(selectedSteps)"><i class="dku-icon-copy-paste-step-16 dibvab" /> {{::translate('SHAKER.STEPS.ACTIONS.PASTE_AFTER_SELECTION', 'Paste after selection')}}</a></li>
                        <li><a ng-click="toggleDisableSelectedSteps()"><i class="dku-icon-power-16 dibvab" /> {{::translate('SHAKER.STEPS.ACTIONS.TOGGLE_ENABLE_DISABLE', 'Toggle enable/disable')}}</a></li>
                        <li><a ng-click="deleteSelectedSteps($event)"><i class="dku-icon-trash-16 dibvab" /> {{::translate('SHAKER.STEPS.ACTIONS.DELETE_STEPS', 'Delete')}}</a></li>
                        <li class="dropup dropdown-submenu step-color-pannel" step-color-picker>
                            <a><i class="dku-icon-color-palette-16 dibvab"></i> {{::translate('SHAKER.STEPS.ACTIONS.COLOR', 'Color')}}</a>
                            <ul class="dropdown-menu">
                                <li ng-click="uncolorSelectedSteps()"><div class="color"></div></li>
                                <li ng-repeat="color in colors" ng-click="colorSelectedSteps(color.main, color.secondary)">
                                    <div class="color" style="background-color:{{color.secondary}};border-color:{{color.main}}"></div>
                                </li>
                            </ul>
                        </li>
                    </ul>
                `

                let dkuPopinOptions = {
                    template: template,
                    isElsewhere: isElsewhere,
                    callback: null,
                    popinPosition: menuType,
                    onDismiss: () => {
                        $scope.activeShakerMenu = null;
                        $scope.activeMenuType = null;
                    }
                };

                $scope.activeShakerMenu = openDkuPopin(newScope, $event, dkuPopinOptions);
            }

        }
    }
});


app.directive('groupNameEditor', [ '$timeout', function($timeout) {
    return {
        scope: true,
        restrict: 'A',
        link : function($scope, element, attrs) {
            $scope.showGroupNameForm = false;

            $scope.toggleGroupNameForm = function($event, inputId) {
                $scope.showGroupNameForm = !$scope.showGroupNameForm;
                if ($scope.showGroupNameForm) {
                    $timeout(function() {
                        document.getElementById(inputId).focus();
                    }, false);
                }
            }

            if ($scope.groupChanged.justCreated) {
                $timeout(function() {
                    angular.element(element).find('.show-group').triggerHandler('click');
                });
                $scope.groupChanged.justCreated = false;
            }
        }
    };
}]);


app.directive('processorFooter', [ '$timeout', function($timeout) {
    return {
        scope: true,
        restrict : 'A',
        link : function($scope, element, attrs) {

            //flag for edition state
            $scope.editingComment = false;

            /*
             * Display/Hide methods
             */

            $scope.showFooter = function (expanded) {
                return expanded || $scope.showComment(expanded) || $scope.showCommentEditor(expanded);
            }

            $scope.showComment = function(expanded) {
                return $scope.hasComment() && ($scope.step.alwaysShowComment || expanded) && !$scope.showCommentEditor(expanded);
            }

            $scope.showCommentEditor = function(expanded) {
                return $scope.editingComment && expanded;
            }

            /*
             * Display/Hide utils
             */

            $scope.hasComment = function() {
                return typeof($scope.step.comment) !== 'undefined' && $scope.step.comment.length > 0;
            }

            /*
             * Comment editor utils
             */

            $scope.toggleEditingComment = function ($event) {
                $scope.editingComment = !$scope.editingComment;
                if (!$scope.editingComment) {
                    $scope.saveComment();
                }
            }
            
            $scope.$on('toggleEditingComment', (e, $event, step) => {
                if ($scope.step === step) {
                    $scope.toggleEditingComment($event);
                }
            });

            $scope.saveComment = function() {
                $scope.editingComment = false;
                if (!$scope.isRecipe){
                	$scope.saveOnly();
                }
            }

            $scope.deleteComment = function() {
                $scope.step.comment = undefined;
                $scope.editingComment = false;
                if (!$scope.isRecipe){
                	$scope.saveOnly();
                }
            }


        }
    }
}]);
app.directive('stepColorPicker', [ 'ContextualMenu', function(ContextualMenu) {
   return {
       scope: true,
       restrict : 'A',
       link : function($scope, element, attrs) {
           $scope.colors = [
                {
                    main: '#ff9c00',
                    secondary: '#f4e0c1'
                },
                {
                    main: '#ffdc00',
                    secondary: '#f4edc1'
                },
                {
                    main: '#30c2ff',
                    secondary: '#cae8f4'
                },

                {
                    main: '#61c1b0',
                    secondary: '#d4e7e4'
                },
                {
                    main: '#90d931',
                    secondary: '#deeccb'
                },

           ];

           $scope.colorMenu = new ContextualMenu({
               template: "/templates/shaker/step-color-picker.html",
               cssClass : "step-color-picker",
               scope: $scope,
               contextual: false,
               onClose: function() {
                   $scope.stepToColor = undefined;
               }
           });

           $scope.openColorPicker = function(step, $event) {
               $scope.colorMenu.openAtXY($($event.target).offset().left, $($event.target).offset().top + $($event.target).height(), function() {}, true, false);
               $scope.stepToColor = step;
           }

           $scope.setStepColor = function(main, secondary) {
               $scope.stepToColor.mainColor = main;
               $scope.stepToColor.secondaryColor = secondary;
               if (!$scope.isRecipe){
               	$scope.saveOnly();
               }
           }

           $scope.removeStepColor = function() {
               delete $scope.stepToColor.mainColor;
               delete $scope.stepToColor.secondaryColor;
               if (!$scope.isRecipe){
               	$scope.saveOnly();
               }
           }
       }
   }
}]);

app.service('GenerateStepsService', function(DataikuAPI) {
    this.abortGenerateSteps = function(generateStepsJobId) {
        return DataikuAPI.futures.abort(generateStepsJobId);
    }
    this.complete = function(projectKey, datasetName, data, requestedSampleId, query, previousRequestId) {
        return DataikuAPI.shakers.generateSteps(projectKey, datasetName, data, requestedSampleId, query, previousRequestId);
    }
});

app.service('ShakerService', function() {
    this.findStepId = function($scope, inputStep) {
        const steps = $scope.shaker.steps;
        if (!steps) {
            throw new Error("No steps found in scope");
        }
        let returnStep;
        steps.forEach((step, i) => {
            if (step === inputStep) {
                returnStep = {'id':i, 'subId':undefined, 'depth':0};
            }
            if (step.metaType === "GROUP") {
                step.steps.forEach((subStep, j) => {
                    if (inputStep === subStep) {
                        returnStep = {'id':i, 'subId':j, 'depth':1};
                    }
                });
            }
        });
        return returnStep;
    }

});

})();
