(function() {
    'use strict';

    angular.module('dataiku.charts').service('D3ChartAxes', D3ChartAxes);

    const SCALE_TYPES = {
        ORDINAL: 'ORDINAL',
        LINEAR: 'LINEAR',
        LOG: 'LOG',
        TIME: 'TIME'
    };

    /**
     * A set of helpers to create and enhance d3 svg axes
     */
    function D3ChartAxes(d3Utils, CHART_AXIS_TYPES, CHART_TYPES, CHART_MODES, ChartDimension, ChartUADimension, ChartAxesUtils, AxisTicksConfigMode, ChartFeatures, ChartFormatting, ChartsStaticData, AxisTicksConfiguration, ChartYAxisPosition, ChartFormattingPaneSections, GridlinesAxisType, ChartDataUtils, ChartLabels) {

        const addFormatterToBinnedAxis = (axis, tickExtents, formattingOptions) => {
            axis.tickFormat(ChartFormatting.getForBinnedAxis(tickExtents, formattingOptions));
        };

        const addFormatterToCustomBinnedAxis = (axis, tickExtents, formattingOptions) => {
            axis.tickFormat(ChartFormatting.getForCustomBinnedAxis(tickExtents, formattingOptions));
        };

        const addFormatterToOrdinalAxis = (axis, tickExtents) => {
            axis.tickFormat((_d, i) => {
                const value = tickExtents[i];
                return ChartFormatting.getForOrdinalAxis(value);
            });
        };

        const addFormatterToPercentageAxis = (axis, formattingOptions = {}) => {
            const scale = (axis.scale() instanceof Function) ? axis.scale() : axis.scale;
            const minValue = (Math.min(...scale.domain()) || 0) * 100;
            const maxValue = (Math.max(...scale.domain()) || 0) * 100;
            const numValues = axis.tickValues() ? axis.tickValues().length : axis.ticks()[0];
            axis.tickFormat(ChartFormatting.getForAxis(minValue, maxValue, numValues, formattingOptions, true));
        };

        // filtering out the tick that overlaps with the axis (if present)
        const getTicksForGridlines = (axisG, axisToCheck, chartG) => {
            const ticks = axisG.selectAll('g.tick')[0];
            return ticks && ticks.filter(tick => {
                const tickTranslate = tick.getAttribute('transform');
                const tickTranslateValue = tickTranslate.match(/translate\(([^,]+),([^,]+)\)/);
                if (axisToCheck === 'x') {
                    const axisG = chartG.select('g.x.axis')[0][0];
                    const axisTranslate = axisG && axisG.getAttribute('transform');
                    //no transform if we have all negative values
                    const axisTranslateValue = axisTranslate ? axisTranslate.match(/translate\(([^,]+),([^,]+)\)/) : '0';
                    return tickTranslateValue[2] !== axisTranslateValue[2];
                } else if (axisToCheck === 'y') {
                    //the y axis always has x translate of 0
                    return tickTranslateValue[1] !== '0';
                }
            });
        };

        const adjustAxisPadding = function(axis, extent, axisSize, valuesFontSize, addPaddingToStart, addPaddingToEnd) {
            if (!addPaddingToStart && !addPaddingToEnd) {
                return;
            }

            const diff = extent[1] - extent[0];
            const domainPadding = valuesFontSize / ((axisSize - valuesFontSize) / diff);

            const paddedDomain = [
                extent[0] - (addPaddingToStart ? domainPadding : 0),
                extent[1] + (addPaddingToEnd ? domainPadding : 0)
            ];

            axis.scale().domain(paddedDomain);
        };

        const computeTextHeightFromFontSize = (svgs) => (fontSize) => {
            let result = fontSize;
            const textElement = d3.select(svgs.get(0))
                .append('text')
                .attr('class', 'measureInChartTempText')
                .attr('opacity', 0)
                .attr('font-size', `${fontSize}px`)
                .text('0');
            if (textElement){
                const textHeight = textElement.node()?.getBBox().height;
                textElement.remove();
                if (textHeight){
                    result = textHeight;
                }
            }
            return result;
        };

        const svc = {

            scaleTypes: Object.freeze(SCALE_TYPES),

            /**
             * Create a svg axis for the given axisSpec
             * @param {ChartTensorDataWrapper}     chartData
             * @param {AxisSpec}                   axisSpec
             * @param {boolean}                    isPercentScale
             * @param {boolean }                   isLogScale
             * @param {boolean}                    includeZero          force inclusion of zero in domain
             * @param {NumberFormattingOptions}    formattingOptions
             * @param {ChartDef.java}              chartDef
             * @returns {d3 axis}
             */
            createAxis: function(chartData, axisSpec, isPercentScale, isLogScale, includeZero, formattingOptions, chartDef, ignoreLabels, otherAxesIndexes = []) {
                if (!axisSpec || !formattingOptions) {
                    return null;
                }

                let axis;
                switch (axisSpec.type) {
                    case CHART_AXIS_TYPES.DIMENSION:
                    case CHART_AXIS_TYPES.UNAGGREGATED:
                        axis = svc.createDimensionAxis(chartData, axisSpec, isLogScale, includeZero, formattingOptions, chartDef, ignoreLabels);
                        break;
                    case CHART_AXIS_TYPES.MEASURE:
                        axis = svc.createMeasureAxis(chartData, axisSpec, isPercentScale, isLogScale, includeZero, formattingOptions, otherAxesIndexes);
                        break;
                    default:
                        throw new Error('Unknown axis type: ' + axisSpec.type);
                }

                if (axis) {
                    axis.type = axisSpec.type;
                    axis.position = axisSpec.position;
                    axis.id = axisSpec.id;
                }

                return axis;
            },


            /**
             * Create a svg axis for a UNAGGREGATED / DIMENSION column
             * @param {ChartTensorDataWrapper}     chartData
             * @param {AxisSpec}                   axisSpec
             * @param {Boolean}                    isLogScale
             * @param {Boolean}                    includeZero
             * @param {NumberFormattingOptions}    formattingOptions
             * @param {ChartDef.java}              chartDef
             * @returns {d3 axis}
             */
            createDimensionAxis: function(chartData, axisSpec, isLogScale, includeZero, formattingOptions, chartDef, ignoreLabels) {
                const isNumerical = ChartAxesUtils.isNumerical(axisSpec);
                let extent = ChartAxesUtils.getDimensionExtent(chartData, axisSpec, ignoreLabels);

                if (!isLogScale && !ChartAxesUtils.isManualMode(axisSpec.customExtent)) {
                    extent = ChartAxesUtils.fixUnbinnedNumericalExtent(axisSpec, extent);
                }

                if (includeZero && isNumerical) {
                    extent = ChartAxesUtils.includeZero(extent);
                }

                const linearScale = d3.scale.linear().domain([extent.min, extent.max]);
                const ordinalScale = d3.scale.ordinal().domain(extent.values.map(function(d, i) {
                    return i;
                }));
                let timeScale;
                let logScale;
                let axis;

                if (isLogScale && isNumerical) {
                    extent = ChartAxesUtils.fixNumericalLogScaleExtent(axisSpec, extent, includeZero);
                    logScale = d3.scale.log().domain([extent.min, extent.max]);
                }

                const createNumericAxis = () => {
                    let axis;
                    if (logScale) {
                        axis = d3.svg.axis().scale(logScale);
                        axis.scaleType = SCALE_TYPES.LOG;
                        svc.setLogAxisTicks(axis, extent.min, extent.max);
                    } else {
                        axis = d3.svg.axis().scale(linearScale);
                        axis.scaleType = SCALE_TYPES.LINEAR;
                    }
                    svc.addNumberFormatterToAxis(axis, formattingOptions);
                    return axis;
                };

                const createOrdinalAxis = (isGroupedNumerical = false, isCustomBinning = false) => {
                    const axis = d3.svg.axis().scale(ordinalScale);
                    if (isCustomBinning) {
                        addFormatterToCustomBinnedAxis(axis, extent.values, formattingOptions);
                    } else if (isGroupedNumerical) {
                        addFormatterToBinnedAxis(axis, extent.values, formattingOptions);
                    } else {
                        addFormatterToOrdinalAxis(axis, extent.values);
                    }
                    axis.scaleType = SCALE_TYPES.ORDINAL;
                    return axis;
                };

                const createTimeAxis = () => {
                    timeScale = d3.time.scale.utc().domain([extent.min, extent.max]);
                    const timeAxis = d3.svg.axis().scale(timeScale);
                    timeAxis.scaleType = SCALE_TYPES.TIME;
                    return timeAxis;
                };

                if (chartDef.type === CHART_TYPES.BOXPLOTS) {
                    axis = createOrdinalAxis(ChartDimension.isGroupedNumerical(axisSpec.dimension));
                } else {
                    // Choose which scale to display on the axis
                    if (ChartDimension.isTimeline(axisSpec.dimension) || (axisSpec.type === CHART_AXIS_TYPES.UNAGGREGATED && ChartUADimension.isDate(axisSpec.dimension))) {
                        // In case we have one value, don't create timescale but use the linearScale else we start at the beginning and not on the middle of the axis
                        axis = (extent.min !== extent.max ? createTimeAxis() : createNumericAxis()).tickFormat((d) => ChartFormatting.getForDate()(d));
                    } else if (ChartDimension.isUngroupedNumerical(axisSpec.dimension)) {
                        axis = createNumericAxis();
                    } else if (ChartDimension.isGroupedNumerical(axisSpec.dimension) || (axisSpec.type === CHART_AXIS_TYPES.UNAGGREGATED && ChartUADimension.isTrueNumerical(axisSpec.dimension))) {
                        if (ChartDimension.hasOneTickPerBin(axisSpec.dimension) && ChartFeatures.chartSupportOneTickPerBin(chartDef)) {
                            axis = createOrdinalAxis(true);
                        } else {
                            axis = createNumericAxis();
                        }
                    } else {
                        axis = createOrdinalAxis();
                    }
                }

                axis.ordinalScale = ordinalScale;
                axis.linearScale = linearScale;

                axis.setScaleRange = function(range) {
                    (logScale || timeScale || linearScale).range(range);

                    const numColumns = ordinalScale.domain().length;
                    const step = range[range.length - 1] / numColumns;
                    const padding = svc.getColumnPadding(numColumns);
                    const axisPadding = axisSpec.padding ?? 0.5;

                    switch (axisSpec.mode) {
                        case CHART_MODES.POINTS:
                            if (step >= 1) {
                                ordinalScale.rangeRoundPoints(range, axisPadding, 0);
                            } else {
                                // 'rangeRoundPoints` can't be used when band widths are not large enough to prevent rounding to zero
                                ordinalScale.rangePoints(range, axisPadding, 0);
                            }
                            break;
                        case CHART_MODES.COLUMNS: {
                            if (step >= 1) {
                                ordinalScale.rangeRoundBands(range, padding, padding / 2);
                            } else {
                                // 'rangeRoundBands` can't be used when band widths are not large enough to prevent rounding to zero
                                ordinalScale.rangeBands(range, padding, padding / 2);
                            }
                            break;
                        }
                        default:
                            throw new Error('Unknown scale type: ' + axisSpec.mode);
                    }
                    return axis;
                };

                svc.fixUpAxis(axis);

                axis.dimension = axisSpec.dimension;

                return axis;
            },


            /**
             * Returns the padding between columns based on the number of columns
             * @param numColumns
             * @returns {number}
             */
            getColumnPadding: function(numColumns) {
                if (numColumns > 20) {
                    return 0.1;
                } else {
                    return 0.45 - numColumns / 20 * 0.35;
                }
            },

            /**
             * Create a svg axis for a MEASURE axisSpec
             *
             * @param {ChartTensorDataWrapper}                                                                                               chartData
             * @param {AxisSpec}                                                                                                             axisSpec
             * @param {boolean}                                                                                                              isPercentScale
             * @param {boolean}                                                                                                              isLogScale
             * @param {boolean}                                                                                                              includeZero
             * @param {{multiplier: keyof ChartsStaticData.availableMultipliers, decimalPlaces: number, prefix: string, suffix: string }}    formattingOptions
             * @returns {d3 axis}
             */
            createMeasureAxis: function(chartData, axisSpec, isPercentScale, isLogScale, includeZero, formattingOptions, otherAxesIndexes = []) {
                axisSpec.extent = ChartAxesUtils.getMeasureExtent(chartData, axisSpec, isLogScale, includeZero, true, otherAxesIndexes);

                if (axisSpec.extent === null) {
                    return null;
                }

                const scale = isLogScale ? d3.scale.log() : d3.scale.linear(),
                    axis = d3.svg.axis().scale(scale).orient('left');

                scale.domain(axisSpec.extent);

                if (isPercentScale || axisSpec.isPercentScale) {
                    addFormatterToPercentageAxis(axis, formattingOptions);
                } else {
                    svc.addNumberFormatterToAxis(axis, formattingOptions);
                }

                if (isLogScale) {
                    svc.setLogAxisTicks(axis, axisSpec.extent[0], axisSpec.extent[1]);
                }

                axis.setScaleRange = function(range) {
                    scale.range(range);
                };

                svc.fixUpAxis(axis);

                axis.measure = axisSpec.measure;
                return axis;
            },

            /**
             * Add hand-picked axis ticks for log scales
             * @param {d3 axis} axis
             * @param {number} minVal: the minimum value of the axis
             * @param {number} maxVal: the maximum value of the axis
             */
            setLogAxisTicks: function(axis, minVal, maxVal) {
                const maxValLog = Math.floor(log10(maxVal));
                const minValLog = Math.ceil(log10(minVal));
                const arr = [];
                for (let i = minValLog; i <= maxValLog; i++) {
                    arr.push(Math.pow(10, i));
                }
                axis.tickValues(arr);
            },

            /**
             * Algorithm finding overlapping and having an identical label ticks. Guaranteeing to keep the first & last tick of the sorted tick list
             * The first pass identifies the overlapping ticks from left to right, excluding the last tick.
             * The second and last pass removes the ticks overlapping the last tick.
             * @param {Array} Array of sorted ticks
             */
            sanitizeTicksDisplay: function(sortedTicks, forceLastTickDisplay = false, onlyRemoveLabel = false) {
                // Remove overlapping labels from left to right (most left and right values excluded)
                let left = 0;
                let right = 1;
                const overlappingTicks = [];
                const length = forceLastTickDisplay ? sortedTicks.length : sortedTicks.length - 1;

                while (right < length) {
                    if (sortedTicks[left].textContent === sortedTicks[right].textContent
                        || areBboxOverlapping(sortedTicks[left].getBoundingClientRect(), sortedTicks[right].getBoundingClientRect())) {
                        overlappingTicks.push(sortedTicks[right]);
                        right++;
                    } else {
                        left = right++;
                    }
                }

                // Check rightmost label overlapping with its left neighbors
                while (!forceLastTickDisplay && left >= 0 && areBboxOverlapping(sortedTicks[left].getBoundingClientRect(), sortedTicks[sortedTicks.length - 1].getBoundingClientRect())) {
                    overlappingTicks.push(sortedTicks[left--]);
                }

                overlappingTicks.forEach(tick => {
                    if (onlyRemoveLabel) {
                        const labelDom = tick.querySelector('text');
                        labelDom && labelDom.remove();
                    } else {
                        tick.remove();
                    }
                });

                return overlappingTicks;
            },

            /**
             * Extra treatment on d3 axis to handle some edge cases
             * @param {d3.axis} axis
             */
            fixUpAxis: function(axis) {
                const scale = axis.scale();

                // d3 axis and scales don't really behave well on empty domains
                if (scale.domain().length == 2 && scale.domain()[0] == scale.domain()[1]) {
                    axis.tickValues([scale.domain()[0]]);
                    scale.domain([scale.domain()[0] - 1, scale.domain()[0] + 1]);
                }
            },

            /**
             * Adjust the bottom margin to make room for the x-axis, and the angle of the tick labels if they need to rotate
             * @param {{top: number, bottom: number, right: number, left: number}} margins
             * @param {jQuery selection} $svg
             * @param {d3 axis} xAxis
             * @param {number} forceRotation
             * @returns {{top: number, bottom: number, right: number, left: number}} the updated margins object
             */
            adjustBottomMargin: function(axisTicksFormatting, margins, $svg, xAxis, forceRotation = 0) {
                const chartHeight = $svg.height(),
                    chartWidth = $svg.width(),
                    svg = d3.select($svg.get(0));

                let labels, usedBand;

                if (xAxis.type === CHART_AXIS_TYPES.MEASURE || xAxis.scaleType === SCALE_TYPES.LINEAR || xAxis.scaleType === SCALE_TYPES.TIME) {
                    const ticks = xAxis.tickValues() || xAxis.scale().ticks();
                    labels = xAxis.tickFormat() ? ticks.map(xAxis.tickFormat()) : ticks;
                    usedBand = (chartWidth - margins.left - margins.right) / (labels.length + 1);
                } else {
                    const ticks = xAxis.tickValues() || xAxis.ordinalScale.domain();
                    labels = xAxis.tickFormat() ? ticks.map(xAxis.tickFormat()) : ticks;
                    if (xAxis.ordinalScale.rangeBand() > 0) {
                        usedBand = xAxis.ordinalScale.rangeBand();
                    } else {
                        usedBand = (chartWidth - margins.left - margins.right) / (labels.length + 1);
                    }
                }

                if (labels.length == 0) { // Nothing to do ...
                    return margins;
                }

                svg.selectAll('.tempText').data(labels)
                    .enter()
                    .append('text').attr('class', 'tempText')
                    .text(function(d) {
                        return ChartFormatting.getForOrdinalAxis(d);
                    });


                const maxLabelWidth = d3.max(svg.selectAll('.tempText')[0].map(function(itm) {
                    return itm.getBoundingClientRect().width;
                }));

                const labelHeight = axisTicksFormatting.fontSize;
                const hasLongLabels = !svg.selectAll('.tempText').filter(function() {
                    return this.getBoundingClientRect().width > usedBand;
                }).empty();

                svg.selectAll('.tempText').remove();

                if (forceRotation) {
                    xAxis.labelAngle = forceRotation;
                } else {
                    xAxis.labelAngle = hasLongLabels ? Math.atan((labelHeight * 2) / usedBand) : 0;
                }

                if (xAxis.labelAngle > Math.PI / 3) {
                    xAxis.labelAngle = Math.PI / 2;
                }

                // Prevent the xAxis from taking more than a quarter of the height of the chart
                margins.bottom = Math.min(chartHeight / 4, margins.bottom + Math.sin(xAxis.labelAngle) * maxLabelWidth + Math.cos(xAxis.labelAngle) * labelHeight);

                return margins;
            },


            /**
             * Get the chart's horizontal margins leaving room for the y-axis labels
             * @param {{top: number, bottom: number, right: number, left: number}} margins
             * @param {jQuery selection} $svg
             * @param {ChartDef.java} chartDef
             * @param {array of d3 axis} yAxes
             * @returns {{top: number, bottom: number, right: number, left: number}} the margins object
             */
            getHorizontalMarginsAndAxesWidth: function(margins, $svg, chartDef, yAxes) {
                const axesWidth = [];
                margins[ChartYAxisPosition.LEFT] = ChartsStaticData.CHART_BASE_MARGIN;
                margins[ChartYAxisPosition.RIGHT] = ChartsStaticData.CHART_BASE_MARGIN;
                yAxes.forEach(axis => {
                    const tempMargins = {
                        [ChartYAxisPosition.LEFT]: 0,
                        [ChartYAxisPosition.RIGHT]: 0
                    };
                    const side = axis.position;
                    const axisFormatting = ChartAxesUtils.getFormattingForYAxis(chartDef.yAxesFormatting, axis.id);
                    if (axis && axisFormatting.displayAxis) {
                        let labels;
                        if (axis.type === CHART_AXIS_TYPES.MEASURE || axis.scaleType === SCALE_TYPES.LINEAR || axis.scaleType === SCALE_TYPES.TIME) {
                            labels = axis.scale().ticks();
                        } else {
                            labels = axis.ordinalScale.domain();
                        }

                        // LOG scale or Number of ticks feature set.
                        const manualTicks = axis.tickValues();
                        if (manualTicks) {
                            labels = manualTicks;
                        }

                        labels = axis.tickFormat() ? labels.map(axis.tickFormat()) : labels;
                        const svg = d3.select($svg.get(0));

                        svg.selectAll('.tempText').data(labels).enter()
                            .append('text').attr('class', 'tempText')
                            .attr('font-size', `${axisFormatting.axisValuesFormatting.axisTicksFormatting.fontSize}px`)
                            .text(function(d) {
                                return d;
                            });

                        const maxLabelWidth = d3.max(svg.selectAll('.tempText')[0].map(function(itm) {
                            return itm.getBoundingClientRect().width;
                        })) || 0;

                        tempMargins[side] += maxLabelWidth + ChartsStaticData.AXIS_WIDTH;
                        svg.selectAll('.tempText').remove();
                    }

                    const canHaveAxisTitle = ((!_.isNil(axisFormatting.axisTitle) && axisFormatting.axisTitle.length > 0)
                    || (axis && (axis.dimension !== undefined || axis.measure !== undefined))
                    || ChartAxesUtils.getMeasuresDisplayedOnAxis(axis.position, chartDef.genericMeasures).length >= 1);

                    if (canHaveAxisTitle && axisFormatting.showAxisTitle) {
                        tempMargins[side] += ChartsStaticData.AXIS_MARGIN + axisFormatting.axisTitleFormatting.fontSize;
                    }

                    margins.left += tempMargins.left;
                    margins.right += tempMargins.right;
                    axesWidth.push({ id: axis.id, width: tempMargins[side] });
                });

                const leftYAxesNb = yAxes.filter(axis => ChartAxesUtils.isLeftYAxis(axis)).length;
                if (leftYAxesNb > 1) {
                    // Adding 16px spacing between each y axis
                    margins.left += (leftYAxesNb - 1) * ChartsStaticData.AXIS_Y_SPACING;
                }

                return { updatedMargins: margins, axesWidth };
            },

            getMarginsAndAxesWidth: function(chartHandler, chartDef, $svg, xAxis, yAxes) {
                let margins = { top: ChartsStaticData.CHART_BASE_MARGIN, bottom: ChartsStaticData.CHART_BASE_MARGIN };
                let yAxesWidth;
                if (!chartHandler.noXAxis && ChartFeatures.canDisplayAxes(chartDef.type)) {
                    if (chartDef.xAxisFormatting.displayAxis) {
                        margins.bottom += ChartsStaticData.AXIS_WIDTH;
                    }

                    const canHaveAxisTitle = ChartFeatures.canShowXAxisTitle(chartDef) && ((!_.isNil(chartDef.xAxisFormatting.axisTitle) && chartDef.xAxisFormatting.axisTitle.length > 0)
                    || (xAxis && (xAxis.dimension !== undefined || xAxis.measure !== undefined))
                    || chartDef.genericDimension0.length === 1);

                    if (canHaveAxisTitle && chartDef.xAxisFormatting.showAxisTitle) {
                        margins.bottom += ChartsStaticData.AXIS_MARGIN + chartDef.xAxisFormatting.axisTitleFormatting.fontSize;
                    }
                }

                if (chartHandler.noYAxis) {
                    margins.left = ChartsStaticData.CHART_BASE_MARGIN;
                    margins.right = ChartsStaticData.CHART_BASE_MARGIN;
                } else {
                    const { updatedMargins, axesWidth } = svc.getHorizontalMarginsAndAxesWidth(margins, $svg, chartDef, yAxes);
                    margins = updatedMargins;
                    yAxesWidth = axesWidth;
                }

                return { margins, yAxesWidth };
            },

            /**
             * - Align 0 on both axis if there is 2 axes with negatives and positives
             * - Set the correct number of ticks to 1 or both axes if ticksConfig.number is set.
             */
            alignAndSetYTickValues: function(axes, axisOptions, yAxesFormatting) {
                const axesMins = [];
                const availableAxesMins = [];
                const axesMaxes = [];
                const availableAxesMaxes = [];
                const axesRes = [];

                const isAxisCompatible = (axis) => {
                    const isAxisLogScale = (yAxesFormatting.find(v => v.id === axis.id) || {}).isLogScale;
                    const [axisMin] = (axis && svc.getCurrentAxisExtent(axis)) || [];
                    const isZeroInRange = axisMin <= 0;
                    return (!axis.dimension || ChartAxesUtils.isNumerical({ dimension: axis.dimension })) && !isAxisLogScale && isZeroInRange;
                };

                const availableAxes = axes.filter(axis => isAxisCompatible(axis));
                axes.forEach(axis => {
                    const [axisMin, axisMax] = (axis && svc.getCurrentAxisExtent(axis)) || [];
                    axesMins.push(axisMin);
                    axesMaxes.push(axisMax);
                    if (isAxisCompatible(axis)) {
                        availableAxesMins.push(axisMin);
                        availableAxesMaxes.push(axisMax);
                    }
                    axesRes.push({ min: axisMin, max: axisMax });
                });


                if (availableAxes.length > 1 && availableAxesMins.some(v => v < 0) && availableAxesMaxes.some(v => v > 0)) {
                    const minFactors = [];
                    const maxFactors = [];
                    const axesAbsolutes = [];
                    availableAxes.forEach(axis => {
                        const index = axes.indexOf(axis);
                        const axisAbsolute = (Math.max(Math.abs(axesMins[index]), Math.abs(axesMaxes[index])));
                        axesAbsolutes.push(axisAbsolute);
                        minFactors.push(Math.abs(Math.min(0, axesMins[index]) / axisAbsolute));
                        maxFactors.push(Math.abs(Math.max(0, axesMaxes[index]) / axisAbsolute));

                    });


                    const maxAxisFactor = Math.max(...maxFactors);
                    const minAxisFactor = Math.max(...minFactors);

                    availableAxes.forEach((axis, i) => {
                        const index = axes.indexOf(axis);
                        axesRes[index].min = -axesAbsolutes[i] * minAxisFactor;
                        axesRes[index].max = axesAbsolutes[i] * maxAxisFactor;
                        axis.scale().domain([axesRes[index].min, axesRes[index].max]);

                    });
                }

                axes.forEach((axis, i) => {
                    const axisFormatting = yAxesFormatting.find(v => v.id === axis.id);
                    // only align axes and return if there is ticks configuration number set
                    if (_.isNil(axisFormatting.ticksConfig.number)) {
                        axis.tickValues(axisOptions[i].tickValues);
                        axis.tickFormat(axisOptions[i].tickFormat);
                    } else {
                        svc.setNumberOfTicks(axis, axesRes[i], axisFormatting.ticksConfig, axisFormatting.numberFormatting);
                    }
                });

            },

            /**
             * Set the correct number of ticks to axis if ticksConfig.number is set.
             */
            alignAndSetXTickValues: function(xAxis, axisOptions, xAxisFormatting) {
                const [axisMin, axisMax] = (xAxis && svc.getCurrentAxisExtent(xAxis)) || [];
                const axisRes = { min: axisMin, max: axisMax };

                // only align axes and return if there is ticks configuration number set
                if (xAxisFormatting.ticksConfig.number == null) {
                    xAxis.tickValues(axisOptions.tickValues);
                    xAxis.tickFormat(axisOptions.tickFormat);
                    return;
                }

                svc.setNumberOfTicks(xAxis, axisRes, xAxisFormatting.ticksConfig, xAxisFormatting.numberFormatting);
            },

            setNumberOfTicks: (axis, resAxis, ticksConfig, numberFormatting) => {
                const maxTicksDisplayed = 500;
                if (svc.canSetNumberOfTicks(axis)) {
                    const axisTicks = Math.min(maxTicksDisplayed, ticksConfig.mode === AxisTicksConfigMode.NUMBER ? Math.floor(ticksConfig.number + 1) : (resAxis.max - resAxis.min) / ticksConfig.number);
                    resAxis.interval = (resAxis.max - resAxis.min) / axisTicks;
                    const values = Array.from({ length: axisTicks + 1 }, (_, i) => resAxis.min + i * resAxis.interval);
                    axis.tickValues(values);
                    axis.tickFormat(ChartFormatting.getForAxis(resAxis.min, resAxis.max, values, numberFormatting));
                }
            },

            /*
             * Interval division might produce arbitrary-precision arithmetic issues (IEEE 754)
             * formatter should round axis origin to '0' (as it might be a value like 4e-14)
             */
            canSetNumberOfTicks: (axis) => {
                if (!axis) {
                    return false;
                } else if (axis.scaleType) {
                    return [SCALE_TYPES.LINEAR, SCALE_TYPES.LOG].includes(axis.scaleType);
                } else {
                    return axis.type === CHART_AXIS_TYPES.MEASURE;
                }
            },

            /**
             * Adjust the domain of two given axis so that they have the same scales
             * @param chartDef
             * @param {d3 scale} xScale
             * @param {d3 scale} yScale
             */
            equalizeScales: function(chartDef, xScale, yScale) {

                const compatibleAxes = ChartUADimension.areAllNumericalOrDate(chartDef);
                if (compatibleAxes) {

                    const axisMin = Math.min(xScale.domain()[0], yScale.domain()[0]);
                    const axisMax = Math.max(xScale.domain()[1], yScale.domain()[1]);

                    const extent = function(v) {
                        return Math.abs(v[1] - v[0]);
                    };

                    // Add a gap to Y axis so that it reaches origin. X axis is always correct because it starts from it.
                    const diff = Math.abs(extent(yScale.range()) - extent(xScale.range()));
                    if (extent(yScale.range()) > extent(xScale.range())) {
                        yScale.range([xScale.range()[1] + diff, xScale.range()[0] + diff]);
                    } else {
                        xScale.range([yScale.range()[1], yScale.range()[0]]);
                    }

                    xScale.domain([axisMin, axisMax]);
                    yScale.domain([axisMin, axisMax]);
                }

                chartDef.compatibleAxes = compatibleAxes;
            },

            clearAxes: function(g) {
                g.selectAll('.x.axis').remove();
                g.selectAll('.y.axis').remove();
            },

            drawGridlines: function(axisG, axisName, formattingOptions, position, chartG) {
                const isYAxis = axisName === 'y';
                const ticks = getTicksForGridlines(axisG, isYAxis ? 'x' : 'y', chartG);
                svc.appendGridlines(d3.selectAll(ticks), position, formattingOptions, isYAxis ? 'hline' : 'vline');
            },

            appendGridlines(el, position, formattingOptions, className) {
                const { x1, x2, y1, y2 } = position;
                const isDashed = formattingOptions.type === 'DASHED';
                el.append('line')
                    .attr('class', className)
                    .attr('stroke', formattingOptions.color)
                    .attr('stroke-width', formattingOptions.size)
                    .attr('stroke-dasharray', isDashed ? 12 : 0)
                    .attr('x1', x1)
                    .attr('x2', x2)
                    .attr('y1', y1)
                    .attr('y2', y2);
            },

            shouldDrawGridlinesForAxis: (chartType, axis, displayAxis) => {
                if (!ChartFeatures.canSelectGridlinesAxis(chartType)) {
                    return true;
                }
                switch (displayAxis.type) {
                    case GridlinesAxisType.CUSTOM_Y_AXIS:
                        return axis.id === displayAxis.axisId;
                    case GridlinesAxisType.LEFT_Y_AXIS:
                        return axis.position === ChartYAxisPosition.LEFT;
                    case GridlinesAxisType.RIGHT_Y_AXIS:
                        return axis.position === ChartYAxisPosition.RIGHT;
                    default:
                        return true;
                }
            },

            /**
             * Draw the given axes for all the given svg(s)
             * @param {d3DrawContext}: an object, with the following properties:
             *      - $svgs {jQuery} $svgs,
             *      - chartDef {ChartDef.java}
             *      - chartHandler {$scope}
             *      - ySpecs {AxisSpec}: y axis specs
             *      - xAxis {d3 axis} (nullable)
             *      - yAxes {d3 axis} (nullable)
             *      - axisOptions {AxisOptions}: (nullable)
             *      - handleZoom {boolean}: (nullable) Applied for lines only if it is zoomable
             *      - yAxesColors {[specID: string]: string}: colors of the yAxes (black for all the charts except scattersMP)
             */
            drawAxes: function(d3DrawContext) {
                if (!d3DrawContext) {
                    return;
                }
                const { $svgs, chartDef, chartHandler, ySpecs, xAxis, yAxes, axisOptions, handleZoom, yAxesColors } = d3DrawContext;
                // Align double axes and set tick values
                if (ChartFeatures.canDisplayAxes(chartDef.type)) {
                    const xOptions = axisOptions.x;
                    const yOptions = [...axisOptions.y];

                    let xTicksConfig = chartDef.xAxisFormatting.ticksConfig;
                    let yAxesFormatting = chartDef.yAxesFormatting.map(v => {
                        return { id: v.id, ticksConfig: v.ticksConfig, numberFormatting: v.axisValuesFormatting.numberFormatting, isLogScale: v.isLogScale };
                    });

                    if (ChartFeatures.isScatterZoomed(chartDef.type, chartDef.$zoomControlInstanceId)) {
                        xTicksConfig = { ...xTicksConfig, number: null };
                        yAxesFormatting = yAxesFormatting.map(v => {
                            return { ...v, ticksConfig: { ...v.ticksConfig, number: null } };
                        });
                    }

                    const yAxesOneTickPerBin = Object.keys(ySpecs).some(key => ChartFeatures.hasOneTickPerBinSelected(chartDef && chartDef.$chartStoreId, key));

                    !handleZoom && !ChartFeatures.hasOneTickPerBinSelected(chartDef && chartDef.$chartStoreId, 'x') && svc.alignAndSetXTickValues(xAxis, xOptions, { ticksConfig: xTicksConfig, numberFormatting: chartDef.xAxisFormatting.axisValuesFormatting.numberFormatting });
                    !yAxesOneTickPerBin && svc.alignAndSetYTickValues(yAxes, yOptions, yAxesFormatting);
                }

                // Initial margins
                const $svg = $svgs.eq(0),
                    width = $svg.width(),
                    height = $svg.height();
                let { margins, yAxesWidth } = svc.getMarginsAndAxesWidth(chartHandler, chartDef, $svg, xAxis, yAxes);

                // Update x scale based on horizontal margins
                const vizWidth = width - margins.left - margins.right;
                if (xAxis) {
                    xAxis.setScaleRange([0, vizWidth]);
                }

                // Adjust bottom margin for x axis
                if (xAxis && !chartHandler.noXAxis && chartDef.xAxisFormatting.displayAxis) {
                    margins = svc.adjustBottomMargin(chartDef.xAxisFormatting.axisValuesFormatting.axisTicksFormatting, margins, $svg, xAxis, chartHandler.forceRotation);
                }

                let allNegative = true;
                let allPositive = true;
                yAxes.forEach(yAxis => {
                    const currentYExtent = svc.getCurrentAxisExtent(yAxis);
                    if (currentYExtent && currentYExtent[1] >= 0) {
                        allNegative = false;
                    }
                    if (currentYExtent && currentYExtent[0] < 0) {
                        allPositive = false;
                    }
                });

                if (xAxis && chartDef.singleXAxis && chartDef.facetDimension.length) {
                    // Override bottom margins when we don't actually need space in the svgs for the axis
                    margins.axisHeight = margins.bottom;
                    margins.bottom = allPositive ? 0 : 0.2 * height;
                    margins.top = allNegative ? 0 : 0.2 * height;
                }

                if (chartDef.xAxisFormatting.isLogScale && ChartFeatures.isUnaggregated(chartDef.type) && !ChartFeatures.isScatterZoomed(chartDef.type, chartDef.$zoomControlInstanceId)) {
                    svc.adjustScatterPlotAxisPadding(xAxis, vizWidth);
                }

                // Update y scales accordingly
                let vizHeight = height - margins.top - margins.bottom;
                yAxes.forEach((axis, i) => {
                    if (i === 0 && ChartAxesUtils.isLeftYAxis(axis)) {
                        if (ySpecs[axis.id].ascendingDown) {
                            axis.setScaleRange([0, vizHeight]);
                        } else {
                            axis.setScaleRange([vizHeight, 0]);
                        }

                        // Enforce minRangeBand (eg) for horizontal bars
                        if (chartDef.facetDimension.length === 0 && axis.type === 'DIMENSION' && ySpecs[axis.id].minRangeBand > 0 && axis.ordinalScale.rangeBand() < ySpecs[axis.id].minRangeBand) {
                            const numLabels = axis.ordinalScale.domain().length;
                            const padding = svc.getColumnPadding(numLabels);
                            const range = d3Utils.getRangeForGivenRangeBand(ySpecs[axis.id].minRangeBand, numLabels, padding, padding / 2);
                            if (ySpecs[axis.id].ascendingDown) {
                                axis.setScaleRange([0, range]);
                            } else {
                                axis.setScaleRange([range, 0]);
                            }
                            vizHeight = range;
                            const svgHeight = vizHeight + margins.top + margins.bottom;
                            $svgs.height(svgHeight);
                        }
                    } else {
                        axis.setScaleRange([vizHeight, 0]);
                    }

                    if (ChartAxesUtils.isYAxisLogScale(chartDef.yAxesFormatting, axis.id) && ChartFeatures.isUnaggregated(chartDef.type) && !ChartFeatures.isScatterZoomed(chartDef.type, chartDef.$zoomControlInstanceId)) {
                        svc.adjustScatterPlotAxisPadding(axis, vizHeight);
                    }

                    if (chartDef.valuesInChartDisplayOptions.displayValues && chartDef.type === CHART_TYPES.GROUPED_COLUMNS) {
                        svc.adjustVerticalBarsYAxisPadding(axis, vizHeight, ChartLabels.getMeasureValueInChartMaxHeight(chartDef, true, false, 2, computeTextHeightFromFontSize($svgs)), true);
                    }

                    if (chartDef.valuesInChartDisplayOptions.displayValues && ChartFeatures.canDisplayTotalValues(chartDef)) {
                        svc.adjustVerticalBarsYAxisPadding(axis, vizHeight, ChartLabels.getMeasureValueInChartMaxHeight(chartDef, false, true, 5, computeTextHeightFromFontSize($svgs)), true);
                    }

                    if (chartDef.valuesInChartDisplayOptions.displayValues && chartDef.type === CHART_TYPES.LINES) {
                        svc.adjustLinesYAxisPadding(axis, vizHeight, ChartLabels.getMeasureValueInChartMaxHeight(chartDef, true, false, 12, computeTextHeightFromFontSize($svgs)));
                    }

                    if (chartDef.valuesInChartDisplayOptions.displayValues && chartDef.type === CHART_TYPES.MULTI_COLUMNS_LINES) {
                        svc.adjustVerticalBarsYAxisPadding(axis, vizHeight, ChartLabels.getMeasureValueInChartMaxHeight(chartDef, true, false, 5, computeTextHeightFromFontSize($svgs)), false);
                    }
                });

                // Equalize x and y to same scale if needed
                if (chartDef.type === CHART_TYPES.SCATTER && chartDef.scatterOptions && chartDef.scatterOptions.equalScales) {
                    svc.equalizeScales(chartDef, xAxis.scale(), yAxes[0].scale());
                }

                const updatedMargins = _.cloneDeep(margins);

                let $xAxisSvgs = $svgs;

                // If chart is facetted and 'singleXAxis' is enabled, we put the axis in a separate svg, fixed at the bottom of the screen
                if (xAxis && chartDef.singleXAxis && chartDef.facetDimension.length) {
                    $xAxisSvgs = $('<svg class="x-axis-svg" xmlns="http://www.w3.org/2000/svg">');
                    $('<div class="x-axis noflex">').css('height', updatedMargins.axisHeight).append($xAxisSvgs).appendTo($svgs.eq(0).closest('.mainzone'));
                    d3.select($xAxisSvgs.get(0)).append('g').attr('class', 'chart').attr('transform', 'translate(' + ($svgs.closest('.chart').find('h2').outerWidth() + updatedMargins.left) + ', -1)');
                }

                // Create a g.chart in every svg
                $svgs.each(function() {
                    const svg = d3.select(this);
                    const g = svg.select('g.chart');
                    //  If g.chart already exists, we don't need to recreate it (useful for zoom)
                    if (!g[0][0]) {
                        svg.append('g').attr('class', 'chart');
                    }
                });

                if (!chartHandler.noXAxis && xAxis) {
                    xAxis.orient('bottom');

                    $xAxisSvgs.each(function() {
                        const g = d3.select(this).select('g');

                        const xAxisG = g.append('g')
                            .attr('class', 'x axis qa_charts_x-axis-column-label-text');

                        if (!chartDef.singleXAxis || !chartDef.facetDimension.length) {
                            if (!allNegative) {
                                xAxisG.attr('transform', 'translate(0,' + vizHeight + ')');
                                xAxis.orient('bottom');
                            } else {
                                xAxis.orient('top');
                                const bottomMargin = updatedMargins.bottom;
                                updatedMargins.bottom = updatedMargins.top;
                                updatedMargins.top = bottomMargin;
                            }
                        }

                        if (chartDef.xAxisFormatting.displayAxis) {
                            xAxisG.call(xAxis);

                            const axisScale = yAxes[0].scale()(0);
                            if (!allNegative && !allPositive && xAxis.type !== CHART_AXIS_TYPES.UNAGGREGATED && axisScale) {
                                xAxisG.select('path.domain').attr('transform', 'translate(0,' + (axisScale - vizHeight) + ')');
                            }

                            if (xAxis.labelAngle) {
                                if (!allNegative) {
                                    xAxisG.selectAll('text')
                                        .attr('transform', (xAxis.labelAngle == Math.PI / 2 ? 'translate(-13, 9)' : 'translate(-10, 0)') + ' rotate(' + xAxis.labelAngle * -180 / Math.PI + ', 0, 0)')
                                        .style('text-anchor', 'end');
                                } else {
                                    xAxisG.selectAll('text')
                                        .attr('transform', 'translate(10, 0) rotate(' + xAxis.labelAngle * -180 / Math.PI + ', 0, 0)')
                                        .style('text-anchor', 'start');
                                }
                            }

                            if (xAxisG) {
                                const ticksFormatting = chartDef.xAxisFormatting.axisValuesFormatting.axisTicksFormatting;
                                ticksFormatting && xAxisG.selectAll('.tick text')
                                    .attr('fill', ticksFormatting.fontColor)
                                    .attr('font-size', `${ticksFormatting.fontSize}px`);

                                // sanitize ticks if a specific configuration is set
                                if (ChartFeatures.shouldRemoveOverlappingTicks(chartDef.xAxisFormatting, chartDef, 'x')) {
                                    xAxisG.selectAll('.tick').forEach(ticks => {
                                        ticks.sort((a, b) => a.__data__ - b.__data__);
                                        const overlappingTicks = svc.sanitizeTicksDisplay(ticks, true, true);
                                        AxisTicksConfiguration.setXTicksAreOverlapping(overlappingTicks.length > 0);
                                    });
                                } else {
                                    AxisTicksConfiguration.setXTicksAreOverlapping(false);
                                }
                            }

                            if (!xAxis.labelAngle) {
                                // The last tick might be at the very end of the axis and overflow the SVG
                                const lastTick = xAxisG.select('.tick:last-of-type');
                                const domainWidth = xAxisG.select('.domain').node()?.getBBox()?.width || 0;
                                const lastTickTransform = lastTick.attr('transform');
                                const lastTickText = lastTick.select('text');

                                if (lastTickTransform) {
                                    const translateMatch = lastTickTransform.match(/translate\(([^,]+),([^)]+)\)/);
                                    const lastTickX = translateMatch ? parseFloat(translateMatch[1]) : 0;
                                    const lastTextWidth = lastTickText.node()?.getBBox()?.width || 0;
                                    if (lastTickX + lastTextWidth / 2 > domainWidth + updatedMargins.right) {
                                        lastTickText.attr('x', -(lastTickX + (lastTextWidth / 2) - domainWidth));
                                    }
                                }
                            }
                        }

                        svc.addTitleToXAxis(xAxisG, xAxis, chartDef, chartHandler, updatedMargins, vizWidth);
                    });
                }

                $svgs.each(function() {
                    const g = d3.select(this).select('g');

                    g.attr('transform', 'translate(' + updatedMargins.left + ',' + updatedMargins.top + ')');

                    if (xAxis && chartDef.gridlinesOptions.vertical.show && ChartFeatures.canHaveVerticalGridlines(chartDef) && !chartHandler.noXAxis) {
                        const gridlinesPosition = { x1: 0, x2: 0, y1: 0, y2: allNegative ? vizHeight : -vizHeight };
                        svc.drawGridlines(g.select('g.x.axis'), 'x', chartDef.gridlinesOptions.vertical.lineFormatting, gridlinesPosition, g);
                    }

                    if (!chartHandler.noYAxis) {
                        yAxes.forEach((axis, index) => {
                            const axisColor = yAxesColors[axis.id] || '#000';
                            const axisFormatting = ChartAxesUtils.getFormattingForYAxis(chartDef.yAxesFormatting, axis.id);

                            axis.orient(axis.position);

                            if (vizHeight < 300) {
                                axis.ticks(svc.optimizeTicksNumber(vizHeight));
                            }

                            const yAxisG = g.append('g').attr('class', `y y${index + 1} axis`);
                            let remainingAxesWidth = 0;
                            if (ChartAxesUtils.isRightYAxis(axis)) {
                                yAxisG.attr('transform', 'translate(' + vizWidth + ',0)');
                            } else {
                                remainingAxesWidth = yAxesWidth.slice(index + 1).reduce((acc, val) => {
                                    const isLeftAxis = ChartAxesUtils.isLeftYAxis(yAxes.find(axis => axis.id === val.id));
                                    if (isLeftAxis && val.width > 0) {
                                        acc += val.width + ChartsStaticData.AXIS_Y_SPACING;
                                    }
                                    return acc;
                                }, 0);
                                yAxisG.attr('transform', `translate(${-remainingAxesWidth},0)`);
                            }
                            if (axisFormatting.displayAxis) {
                                yAxisG.call(axis);

                                if (chartDef.gridlinesOptions.horizontal.show && svc.shouldDrawGridlinesForAxis(chartDef.type, axis, chartDef.gridlinesOptions.horizontal.displayAxis)) {
                                    const gridlinesPosition = { x1: remainingAxesWidth, x2: ChartAxesUtils.isRightYAxis(axis) ? -vizWidth : vizWidth, y1: 0, y2: 0 };
                                    svc.drawGridlines(yAxisG, 'y', chartDef.gridlinesOptions.horizontal.lineFormatting, gridlinesPosition, g);
                                }


                                if (xAxis && chartDef.singleXAxis && chartDef.facetDimension.length && !index) {
                                    yAxisG.select('.domain').attr('d', 'M0,-100V1000'); // TODO @charts dirty, use vizHeight+margins.top+margins.bottom instead of 1000?
                                    if (allPositive && (axis.type === CHART_AXIS_TYPES.MEASURE || axis.scaleType === SCALE_TYPES.LINEAR)) {
                                        yAxisG.select('.tick').remove(); // remove tick for '0'
                                    }
                                }

                                const axisG = g.select(`.y${index + 1}`);
                                const ticksFormatting = axisFormatting.axisValuesFormatting.axisTicksFormatting;

                                axisG.selectAll('.tick text')
                                    .attr('fill', ChartFeatures.canSetAxisTextColor(chartDef.type) ? ticksFormatting.fontColor : axisColor)
                                    .attr('font-size', `${ticksFormatting.fontSize}px`);
                            }

                            const axisG = g.select(`.y${index + 1}`);
                            svc.addTitleToYAxis(axisG, axis, chartDef, chartHandler, margins, vizHeight, yAxesWidth[index].width, axisColor);


                            // sanitize ticks if a specific configuration is set
                            if (ChartFeatures.shouldRemoveOverlappingTicks(axisFormatting, chartDef, axisFormatting.id)) {
                                g.selectAll(`g.y${index + 1}.axis`).selectAll('g.tick').forEach(ticks => {
                                    ticks.sort((a, b) => a.__data__ - b.__data__);
                                    const overlappingTicks = svc.sanitizeTicksDisplay(ticks, true, true);
                                    AxisTicksConfiguration.setYTicksAreOverlapping(overlappingTicks.length > 0);
                                });
                            } else {
                                AxisTicksConfiguration.setYTicksAreOverlapping(false);
                            }
                            const domain = yAxisG.selectAll('.domain, .tick');
                            domain.style('stroke', axisColor);
                        });
                    }
                });

                return { margins, updatedMargins, vizWidth, vizHeight };
            },


            /**
             * Crop the title if it exceeds the length limit
             * @param {d3 selection}labelElement : d3 selection of the text element containing the title
             * @param {string} title: title to display
             * @param {number} maxTextWidth: maximum width for the graphical text element
             */
            formatAxisTitle: function(labelElement, title, maxTextWidth){
                if (maxTextWidth > 0) {
                    let textLength = labelElement.node().getComputedTextLength();
                    let displayedText = title;
                    while (textLength > 0 && displayedText && displayedText.length && textLength > maxTextWidth) {
                        // crop the title text
                        displayedText = displayedText.substring(0, displayedText.length - 1);
                        labelElement.text(`${displayedText}...`);
                        textLength = labelElement.node().getComputedTextLength();
                    }
                }
            },

            /**
             * When adding a title to an axis corresponding to a DATE dimension, we might need to add the date as a suffix to the title
             * @param {string} title: title to display
             * @param {d3 axis}axis : d3 axis for which we want to display the title
             */
            addDateToTitle: function(title, axis){
                let finalTitle = title;
                if (title && axis.dimension && ChartUADimension.isDate(axis.dimension) && axis.scaleType === SCALE_TYPES.TIME) {
                // specific case for a date dimension, if the current domain is contained within a single day, we add this date at the end of the title
                    const extent = svc.getCurrentAxisExtent(axis);
                    const computedDateDisplayUnit = ChartDataUtils.computeDateDisplayUnit(extent[0], extent[1]);
                    if (computedDateDisplayUnit.formattedMainDate) {
                        finalTitle += ` (${computedDateDisplayUnit.formattedMainDate})`;
                    }
                }

                return finalTitle;
            },

            /**
             * Add the <text> title to the X axis' <g>
             * @param {SVG:g} axisG
             * @param {d3 axis} xAxis
             * @param {ChartDef.java} chartDef
             * @param {$scope} chartHandler
             * @param {Object {top: top, left: left, right: right, bottom: bottom}} margins
             * @param {number} chartWidth
             */
            addTitleToXAxis: function(axisG, xAxis, chartDef, chartHandler, margins, chartWidth) {
                let titleText = ChartAxesUtils.getXAxisTitle(xAxis, chartDef);
                titleText = svc.addDateToTitle(titleText, xAxis);

                const titleStyle = chartDef.xAxisFormatting.axisTitleFormatting;
                const axisTitleMaxWidth = chartWidth - margins.right;


                if (titleText && chartDef.xAxisFormatting.showAxisTitle) {
                    const rect = axisG.append('rect');

                    const labelElement = axisG.append('text')
                        .attr('x', chartWidth / 2)
                        .attr('y', xAxis.orient() == 'bottom' ? margins.bottom - titleStyle.fontSize - ChartsStaticData.AXIS_MARGIN : titleStyle.fontSize + ChartsStaticData.AXIS_MARGIN - margins.top)
                        .attr('text-anchor', 'middle')
                        .attr('dominant-baseline', 'middle')
                        .attr('fill', titleStyle.fontColor)
                        .attr('class', 'axis-title x qa_charts_x-axis-title')
                        .style('font-size', `${titleStyle.fontSize}px`)
                        .text(titleText);

                    this.formatAxisTitle(labelElement, titleText, axisTitleMaxWidth);

                    if (!chartHandler.noClickableAxisLabels) {
                        labelElement.attr('no-global-contextual-menu-close', true)
                            .on('click', function() {
                                chartHandler.openSection(ChartFormattingPaneSections.X_AXIS);
                            });
                    }

                    const bbox = labelElement.node().getBoundingClientRect();

                    rect.attr('x', chartWidth / 2 - bbox.width / 2)
                        .attr('y', xAxis.orient() == 'bottom' ? margins.bottom - 15 - bbox.height / 2 : 15 - margins.top - bbox.height / 2)
                        .attr('width', bbox.width)
                        .attr('height', bbox.height)
                        .attr('fill', 'none')
                        .attr('class', 'chart-wrapper__x-axis-title')
                        .attr('stroke', 'none');
                }
            },

            /**
             * @param {d3 axis} axis
             * @returns {[Float, Float]} [extentMin, extentMax]
             */
            getCurrentAxisExtent(axis) {
                const extent = axis.scale().domain();
                return [Math.min(...extent), Math.max(...extent)];
            },


            /**
             * Add the <text> title to the Y axis' <g>
             * @param {SVG:g} axisG
             * @param {d3 axis} yAxis
             * @param {ChartDef.java} chartDef
             * @param {$scope} chartHandler
             * @param {{top: number, bottom: number, left: number, right: number}} margins
             * @param {number} chartHeight
             * @param {number} axisWidth
             * @param {number} axisColor black for all charts apart from scatters MP
             */
            addTitleToYAxis: function(axisG, yAxis, chartDef, chartHandler, margins, chartHeight, axisWidth, axisColor) {
                const axisFormatting = ChartAxesUtils.getFormattingForYAxis(chartDef.yAxesFormatting, yAxis.id);
                const titleStyle = axisFormatting.axisTitleFormatting;
                const xPosition = ChartAxesUtils.isLeftYAxis(yAxis) ? -axisWidth : axisWidth;
                const axisTitleMaxWidth = chartHeight - margins.top;
                let titleText = ChartAxesUtils.getYAxisTitle(yAxis, chartDef, axisFormatting);
                titleText = svc.addDateToTitle(titleText, yAxis);

                if (titleText && axisFormatting.showAxisTitle) {
                    const labelElement = axisG.append('text')
                        .attr('x', xPosition)
                        .attr('y', (chartHeight - margins.top) / 2)
                        .attr('text-anchor', 'middle')
                        .attr('dominant-baseline', ChartAxesUtils.isLeftYAxis(yAxis) ? 'middle' : 'text-top')
                        .attr('class', 'axis-title y qa_charts_y-axis-title')
                        .attr('fill', ChartFeatures.canSetAxisTextColor(chartDef.type) ? titleStyle.fontColor : axisColor)
                        .style('font-size', `${titleStyle.fontSize}px`)
                        .attr('transform', 'rotate(-90, ' + xPosition + ',' + (chartHeight - margins.top) / 2 + ')')
                        .text(titleText);

                    this.formatAxisTitle(labelElement, titleText, axisTitleMaxWidth);

                    if (!chartHandler.noClickableAxisLabels) {
                        labelElement.attr('no-global-contextual-menu-close', true)
                            .on('click', function() {
                                chartHandler.openSection(ChartFormattingPaneSections.Y_AXIS);
                            });
                    }
                }
            },

            /**
             * Add padding to a Scatter Plot axis, because it cannot be computed at axis creation because we need the axis height to compute it (to handle the log scale).
             * Mutates the given axis.
             * @param {Record<string, unknown>} axis a d3 axis object.
             * @param {number} axisHeight the height of the axis in pixels.
             * @param {number} [paddingPct=0.05] - a value between 0 and 1 corresponding to the percentage of padding that should be applied.
             * @returns {Record<string, unknown>} the updated d3 axis object.
             */
            adjustScatterPlotAxisPadding: (axis, axisHeight, paddingPct = 0.05) => {
                const extent = axis.originalDomain || svc.getCurrentAxisExtent(axis);
                const logMin = Math.log(extent[0]);
                const logMax = Math.log(extent[1]);
                const logDifference = logMax - logMin;
                const paddingPx = axisHeight * paddingPct;
                const axisHeightMinusPadding = axisHeight - (paddingPx * 2);
                const logPadding = logDifference / axisHeightMinusPadding * paddingPx;

                const paddedDomain = [
                    Math.exp(logMin - logPadding),
                    Math.exp(logMax + logPadding)
                ];

                axis.originalDomain = axis.originalDomain || [...axis.scale().domain()];
                axis.scale().domain(paddedDomain);

                return axis;
            },

            /**
             * Add padding to a Vertical bars y axis, because it cannot be computed at axis creation because we need the axis height to compute it (to add space between the bar and the x axis to display values).
             * Mutates the given axis.
             * @param {Record<string, unknown>} axis a d3 axis object.
             * @param {number} axisHeight the height of the axis in pixels.
             * @param {number} valuesFontSize - the font size of the values displayed in chart
             * @returns {Record<string, unknown>} the updated d3 axis object.
             */
            adjustVerticalBarsYAxisPadding: (axis, axisHeight, valuesFontSize, addNegativeExtent = true) => {
                const extent = svc.getCurrentAxisExtent(axis);
                adjustAxisPadding(axis, extent, axisHeight, valuesFontSize, extent[0] < 0 && addNegativeExtent, extent[1] > 0);
                return axis;
            },

            /**
             * Add padding to a Line chart y axis, because it cannot be computed at axis creation because we need the axis height to compute it (to add space between the bar and the x axis to display values).
             * Mutates the given axis.
             * @param {Record<string, unknown>} axis a d3 axis object.
             * @param {number} axisHeight the height of the axis in pixels.
             * @param {number} valuesFontSize - the font size of the values displayed in chart
             * @returns {Record<string, unknown>} the updated d3 axis object.
             */
            adjustLinesYAxisPadding: (axis, axisHeight, valuesFontSize) => {
                const extent = svc.getCurrentAxisExtent(axis);
                adjustAxisPadding(axis, extent, axisHeight, valuesFontSize, true, true);

                return axis;
            },

            /**
             * Computes the optimum number of ticks to display on an axis
             * @param {number} axisSize (in pixel)
             * @returns {number} the optimum ticks number
             */
            optimizeTicksNumber: function(axisSize){
                return Math.floor(Math.max(axisSize / 30, 2));
            },

            /**
             * @param { d3.axis }                   axis
             * @param { FormattingOptions }         [formattingOptions = {}]     The number formatting options. Can be omitted to use the default formatting.
             */
            addNumberFormatterToAxis(axis, formattingOptions = {}) {
                const scale = (axis.scale() instanceof Function) ? axis.scale() : axis.scale;
                const minValue = Math.min(...scale.domain());
                const maxValue = Math.max(...scale.domain());
                const numValues = axis.tickValues() ? axis.tickValues().length : axis.ticks()[0];
                axis.tickFormat(ChartFormatting.getForAxis(minValue, maxValue, numValues, formattingOptions));
            }
        };

        return svc;
    }
})();
