(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    /**
     * (!) This service previously was in static/dataiku/js/simple_report/column-bars/multiplot.js
     */
    app.factory('MultiplotChart', function(ChartManager, ChartDataWrapperFactory, GroupedColumnsDrawer, GroupedColumnsUtils, LinesDrawer, LinesUtils, ChartDataUtils, ReferenceLines, ChartUsableColumns, ChartYAxisPosition, ChartAxesUtils, CHART_LABELS, MultiplotUtils, ChartCustomMeasures, SVGUtils, ChartDimension) {
        return function($container, chartDef, chartHandler, axesDef, data) {

            const chartData = ChartDataWrapperFactory.chartTensorDataWrapper(data, axesDef);
            const ignoreLabels = new Set([CHART_LABELS.SUBTOTAL_BIN_LABEL]);
            const includeEmptyBins = LinesUtils.hasEmptyBinsToIncludeAsZero(chartDef, chartData);
            const subtotals = Array.from({ length: chartData.numAxes });
            const isDisplayedAsColumn = MultiplotUtils.isColumn;
            const hasSubtotals = MultiplotUtils.shouldSubtotalsBeCalculated(chartDef);
            let getRefLineValuesOverride;

            if (hasSubtotals) {
                subtotals[chartData.axesDef.color] = chartData.subtotalCoords[chartData.axesDef.color];
                getRefLineValuesOverride = (measure, measureIdx) => {
                    const dataIndices = MultiplotUtils.isColumn(measure) ?
                        ChartDataUtils.getValuesForColumns(subtotals, chartData, ignoreLabels) :
                        ChartDataUtils.getValuesForLines(subtotals, chartData, ignoreLabels);
                    return ChartDataUtils.retrieveAggrData(chartData, dataIndices, measureIdx);
                };
            }

            const leftYAxisID = ChartAxesUtils.computeYAxisID(ChartYAxisPosition.LEFT);
            const rightYAxisID = ChartAxesUtils.computeYAxisID(ChartYAxisPosition.RIGHT);
            const facetLabels = chartData.getAxisLabels('facet', ignoreLabels) || [{ $tensorIndex: 0 }]; // We'll through the next loop only once if the chart is not facetted


            const dataSpec = chartHandler.getDataSpec();
            const xSpec = { type: 'DIMENSION', mode: 'COLUMNS', dimension: chartDef.genericDimension0[0], name: 'x', customExtent: chartDef.xAxisFormatting.customExtent };
            const customMeasures = ChartCustomMeasures.getMeasuresLikeCustomMeasures(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext());
            const allMeasures = ChartUsableColumns.getUsableColumns(dataSpec.datasetProjectKey, dataSpec.datasetName, chartHandler.getCurrentChartsContext()).filter(m => ['NUMERICAL', 'ALPHANUM', 'DATE'].includes(m.type));

            // If we are not with a color dimension, use the old system it should be faster (just a loop on the tensors)
            const yExtents = hasSubtotals ? ChartDataUtils.getMeasureExtentsForMixColored(chartDef, chartData, includeEmptyBins, ignoreLabels, subtotals, isDisplayedAsColumn) : ChartDataUtils.getMeasureExtents(chartDef, data, undefined, includeEmptyBins);

            const displayedReferenceLines = ReferenceLines.getDisplayedReferenceLines(chartDef.referenceLines, xSpec, undefined),
                referenceLinesValues = ReferenceLines.getReferenceLinesValues(displayedReferenceLines, chartData, allMeasures, chartDef.genericMeasures, customMeasures, true, getRefLineValuesOverride),
                referenceLinesExtents = ReferenceLines.getReferenceLinesExtents(displayedReferenceLines, referenceLinesValues, { [leftYAxisID]: { isPercentScale: yExtents[leftYAxisID].onlyPercent }, [rightYAxisID]: { isPercentScale: yExtents[rightYAxisID].onlyPercent } });

            const leftYExtent = ReferenceLines.getExtentWithReferenceLines(yExtents[leftYAxisID].extent, referenceLinesExtents[leftYAxisID]),
                rightYExtent = ReferenceLines.getExtentWithReferenceLines(yExtents[rightYAxisID].extent, referenceLinesExtents[rightYAxisID]);

            ReferenceLines.mutateDimensionSpecForReferenceLine(ChartDataUtils.getAxisExtent(chartData, xSpec.name, xSpec.dimension), referenceLinesExtents.x, xSpec);

            const columnsData = GroupedColumnsUtils.prepareData(chartDef, chartData, isDisplayedAsColumn, ignoreLabels),
                linesData = LinesUtils.prepareData(chartDef, chartData, measure => !isDisplayedAsColumn(measure), ignoreLabels, false);
            const hasColumns = chartDef.genericMeasures.some(isDisplayedAsColumn);

            const drawFrame = (frameIdx, chartBase) => {
                chartData.fixAxis('animation', frameIdx);
                facetLabels.forEach((facetLabel, f) => {
                    const g = d3.select(chartBase.$svgs.eq(f).find('g.chart').get(0));
                    // initialize labels collision detection, common to columns and lines
                    const labelCollisionDetectionHandler = SVGUtils.initLabelCollisionDetection(chartBase);
                    //  Reference lines will be drawn with both drawers, deactivating it for columns, lines drawer will do the job.
                    GroupedColumnsDrawer(g, chartDef, chartHandler, chartData.fixAxis('facet', facetLabel.$tensorIndex), chartBase, columnsData, facetLabel.$tensorIndex, labelCollisionDetectionHandler, false);
                    // priority to the columns labels, which are diplayed first
                    LinesDrawer(g, chartDef, chartHandler, chartData.fixAxis('facet', facetLabel.$tensorIndex), chartBase, linesData, facetLabel.$tensorIndex, false, false, labelCollisionDetectionHandler, getRefLineValuesOverride, hasColumns ? g.select('.group-wrapper') : null);
                });
            };

            const leftYCustomExtent = ChartAxesUtils.getYAxisCustomExtent(chartDef.yAxesFormatting, leftYAxisID);
            const rightYCustomExtent = ChartAxesUtils.getYAxisCustomExtent(chartDef.yAxesFormatting, rightYAxisID);

            const ySpecs = {
                [leftYAxisID]: { id: leftYAxisID, type: 'MEASURE', extent: leftYExtent, customExtent: leftYCustomExtent, position: ChartYAxisPosition.LEFT },
                [rightYAxisID]: { id: rightYAxisID, type: 'MEASURE', extent: rightYExtent, customExtent: rightYCustomExtent, position: ChartYAxisPosition.RIGHT }
            };
            const availableAxes = ReferenceLines.getAvailableAxesForReferenceLines(chartDef);
            const colorSpec = hasSubtotals ? {
                type: 'CUSTOM',
                name: 'color',
                handle: (measure) => !isDisplayedAsColumn(measure),
                buildGenericMeasures: (measures, colorLabels) => {
                    const trimmeredColorlabels = colorLabels.filter(label => !ignoreLabels.has(label.label));
                    const linesMeasures = measures.filter(measure => !isDisplayedAsColumn(measure));
                    return [...trimmeredColorlabels, ...linesMeasures];
                },
                ignoreColorDim: (index, measuresOrColors) => measuresOrColors[index].isA === 'measure'
            } : {
                type: 'DIMENSION',
                name: 'color'
            };

            const axisSpecs = { x: xSpec, ...ySpecs };
            ChartAxesUtils.setNumberOfBinsToDimensions(chartData, chartDef, axisSpecs);
            ReferenceLines.updateAvailableAxisOptions([
                { axis: 'LEFT_Y_AXIS', isDisplayed: availableAxes['LEFT_Y_AXIS'], isNumerical: true, isPercentScale: ySpecs[leftYAxisID].isPercentScale },
                { axis: 'RIGHT_Y_AXIS', isDisplayed: availableAxes['RIGHT_Y_AXIS'], isNumerical: true, isPercentScale: ySpecs[rightYAxisID].isPercentScale },
                { axis: 'X_AXIS', isDisplayed: true, isNumerical: ChartAxesUtils.isNumerical(xSpec) && !ChartDimension.hasOneTickPerBin(xSpec.dimension), isContinuousDate: ChartAxesUtils.isContinuousDate(xSpec) }
            ]);
            ChartManager.initChart(chartDef, chartHandler, chartData, $container, drawFrame,
                axisSpecs,
                colorSpec, undefined, undefined, ignoreLabels);
        };
    });
})();
