(function() {
    'use strict';

    /**
     * <workspace-activity-feed 
     *      workspace="workspace"
     *      workspaceTimeline="workspaceTimeline">
     * </workspace-activity-feed>
     * 
     * Reuses the timeline directive with some CSS overrides.
     * 
     * @param { Object }    workspace             
     * @param { Object }    workspaceTimeline     
     * 
     */
    const workspaceActivityFeed = {
        templateUrl: '/static/dataiku/workspaces/workspace-activity-feed/workspace-activity-feed.component.html',
        bindings: {
            workspace: '<',
            workspaceTimeline: '<',
        },
        controller: function WorkspaceActivityFeedController($scope, $rootScope, DataikuAPI, WorkspaceDisplayService) {
            const ctrl = this;

            $rootScope.$on('workspaceUpdated', (_event, workspace) => {
                if (workspace.workspaceKey === ctrl.workspace.workspaceKey) {
                    DataikuAPI.workspaces.timeline(workspace.workspaceKey)
                        .success(timeline => {
                            this.workspaceTimeline = WorkspaceDisplayService.transformTimeline(timeline);
                        })
                        .error(setErrorInScope.bind($scope));
                }
            });

            // Used to override the timeline logic

            ctrl.humanReadableObjectType = humanReadableObjectTypeFunc => objectType => {
                if (!objectType) {
                    return '';
                }
                switch(objectType) {
                case "APP":
                    return "application";
                case "WORKSPACE_LINK":
                    return "link";
                case "WORKSPACE_STORY":
                    return "story";
                default:
                    return humanReadableObjectTypeFunc(objectType);
                }
            }

            ctrl.isLinkableDssObject = isLinkableDssObjectFunc => item => {
                return ['WORKSPACE_LINK', 'APP', 'WORKSPACE_STORY'].includes(item.objectType) || isLinkableDssObjectFunc(item);
            }

            ctrl.isExternalLink = isExternalLinkFunc => item => {
                return ['WORKSPACE_LINK', 'WORKSPACE_STORY'].includes(item.objectType) || isExternalLinkFunc(item);
            }

            ctrl.objectLink = () => item => {
                let object = {};
                switch (item.objectType) {
                    case 'WORKSPACE_LINK':
                        object = { htmlLink: { url: item.objectId } };
                        break;
                    case 'PROJECT':
                    case 'APP':
                        object = { appId: 'PROJECT_' + item.objectId };
                        break;
                    case 'WORKSPACE_STORY':
                        object = { story: {id: item.objectId} };
                        break;
                    default:
                        object = { reference: { type: item.objectType, id: item.objectId, projectKey: item.projectKey } };
                }
                const link = WorkspaceDisplayService.getObjectLink(item.workspaceKey, object);
                return link ? link.href : undefined;
            }
        }
    };

    angular.module('dataiku.workspaces').component('workspaceActivityFeed', workspaceActivityFeed);
}());
