import os, json

class CodeStudioTemplate(object):
    """The base interface for a Python CodeStudio"""

    def __init__(self, config, plugin_config):
        """
        :param config: the dict of the configuration of the object
        :param plugin_config: contains the plugin settings
        """
        self.config = config
        self.plugin_config = plugin_config

    def build_spec(self, env):  # NOSONAR - not static, will be overridden
        """
        Compute the spec (for mode=SPEC_COMPUTER). See 
        
        :param env: the build env, as a dict of:
                    - buildDir: the directory passed to `docker build ...`. The dokerfile will be written
                      to that directory
                    - workingDir: a parent of buildDir, for use by the building process. Logs are written
                      in this directory
                    - globalCodeEnvsExtraSettings: settings related to building of code envs, defined in
                      Administration > Settings. Notably this dict contains:
                        * virtualenvCreateExtraOptions: extra options to pass to "python -m venv ..." commands, as a list of str
                        * pipInstallExtraOptions: extra options to pass to "pip install..." commands, as a list of str

        :returns: the spec, ie a dict with a field 'dockerfile'
        """
        raise Exception("unimplemented")

    def build_launch(self, env):  # NOSONAR - not static, will be overridden
        """
        Compute the spec (for mode=SPEC_COMPUTER)
        
        :param env: the launch env, as a dict of:
                    - workingDir: a folder where the CodeStudio run files and logs are written to
                    - projectKey: project in which the CodeStudio is located
                    - executionId: an unique identifier for the CodeStudio on the K8S cluster
                    - apiTicket: a ticket to get access to DSS APIs
                    - templateImage: the docker image built for the CodeStudio template. Note that
                      the image repository URL isn't included in this field and should be fetched
                      from the containerConfig
                    - baseUrl: url to the CodeStudio view inside DSS
                    - containerConfig: container configuration used to run the CodeStudio
                    - labels: a dict of labels to add on deployments, pods, etc...
                    - annotations: a dict of annotations to add on deployments, pods, etc...

        :returns: the spec, as a dict of:
                  - yaml: the yaml to pass to kubectl to create the CodeStudio in the cluster
                  - k8sHeadObjectType and k8sHeadObjectName: type and name of object created by the yaml
                    that DSS should associate with the CodeStudio. 
                  - selector: selectors to use to find which pods DSS should associate with the CodeStudio.
                    This means retrieving the logs of the pods and setting up port-forwards to them
                  - entrypoints: a list of scripts to run in the pods. DSS passes these script names in the initial
                    command given to the pod(s). Order matters, the first exposed thing is the "main" one
                  - exposedPorts: array of dicts, one for each port of the pod(s) that DSS should setup
                    port-forwards for. Each port is defined by a dict of
                      * port: port in the pod of what should be shown to the user
                      * proxiedUrlSuffix: suffix for the proxy_pass in DSS's nginx conf associated to this
                        port-forward
                      * label: user-friendly name of what the port exposes
                      * exposeHtml (bool): Should we show this port/suffix in the UI/IFrame
                  - readinessProbeUrl: url to probe for readiness
                  - readinessProbeHeaders: a list of (header:value)
                  - useTcpReadiness (bool): should we use Tcp to probe for readiness
                  - initialSyncTimeoutInMinutes (int): maximum time for the initial sync of the zones in minutes.
                  - syncedZones: list of zones that should be synchronized from DSS to some location in the container.
                    Each zone is a dict of:
                      * zone: one of {code_studio, user, project_lib}x{versioned, resources} or recipes. Zones
                        named xxxx_versioned are stored in the git of DSS, while zones names xxx_resources are
                        stored outside datadir/config. Zones named user_yyy are shared between all CodeStudios of the 
                        user, zones named project_lib_yyy are shared between all CodeStudios of the project, and zones
                        named code_studio_yyy are specific to each CodeStudio instance
                      * pathInZone: optional, limits the synchronization to only a subpart of the zone
                      * pathInContainer: where the files go in the container
                      * oneWay: set to True to sync only from DSS to the container
                  - excludedFromSync [optional] : list of blobs for file paths, used to exclude files from being
                    sent back from the container to the DSS libraries
                  - fileAdjustmentScript [optional] : path to a script to run on a folder to adapt files
                    sent from DSS to the code studio, for example for adjusting notebook kernels
                  - customCommands: array of dicts, one for each command that you may want to send to the pod.
                    Each command is defined by a dict of
                      * name
                      * label
                      * command
                      * icon
                      * description
                      * behavior = one of : FIRE_AND_FORGET, WAIT_FOR_RETURN_VALUE, WAIT_FOR_RETURN_VALUE_AND_REFRESH_VIEW, WAIT_FOR_RETURN_VALUE_IN_BACKGROUND
                  - extraLabels: a dict of labels to add on deployments, pods, etc...
                  - extraAnnotations: a dict of annotations to add on deployments, pods, etc...
        """
        raise Exception("unimplemented")

    def build_creation(self, env):  # NOSONAR - not static, will be overridden
        """
        Compute the spec (for mode=SPEC_COMPUTER)
        
        :param env: the creation env, as a dict of:
                    - tmpDir: a folder where resources files can be written to, and later placed in their
                              target locations when the CodeStudio is effectively created
                    - CodeStudio: the CodeStudio object being created

        :returns: the spec, as a dict of:
                  - versionedFiles : a list of {'key': ..., 'value': ....} where key is a path 
                                     inside the CodeStudio's versioned files and value is a path
                                     inside the env's tmpDir
                  - userDataFiles : a list of {'key': ..., 'value': ....} where key is a path 
                                    inside the user data files and value is a path inside
                                    the env's tmpDir. Note that files are effectively added 
                                    only if no file pre-exists at the target location
                                    
        """
        raise Exception("unimplemented")

    def build_block_list(self, build_env, launch_env, creation_env):  # NOSONAR - not static, will be overridden
        """
        Compute the blocks (for mode=BLOCK_LIST_PROVIDER)
        
        :param build_env: the build env (None if computing the launch spec). See :func:`build_spec`
        :param launch_env: the launch env (None if computing the build spec). See :func:`build_launch`
        :param creation_env: the creation env (None if computing the creation spec). See :func:`build_creation`

        :returns: a list of CodeStudio blocks defining the template, as {"blocks": [...]}
        """
        raise Exception("unimplemented")