import abc
import enum
import json
from typing import TYPE_CHECKING, Optional, List, Generator, Any

if TYPE_CHECKING:
    from langchain_core.documents import Document

class RetrievalSource(enum.Enum):
    EMBEDDING = "EMBEDDING"
    MULTIMODAL = "MULTIMODAL"

DKU_TEXT_EMBEDDING_COLUMN = "DKU_TEXT_EMBEDDING_COLUMN"

CONTENT_KEY = "Content"

class BaseRagHandler(abc.ABC):
    def __init__(self, retrieval_columns: Optional[List[str]] = None):
        self.retrieval_columns = retrieval_columns

    def get_content(self, doc: "Document") -> str:
        if self.retrieval_columns is None or len(self.retrieval_columns) == 0:
            return doc.page_content
        content = {}
        for col in self.retrieval_columns:
            if col == DKU_TEXT_EMBEDDING_COLUMN:
                content["Content"] = doc.page_content
            else:
                key = col
                if key == CONTENT_KEY:
                    # To avoid overriding the content field if a column was named "Content"
                    key = col + " (from metadata)"
                content[key] = doc.metadata.get(col)

        if len(content) == 1:
            # We keep it as string if we only have one value
            return list(content.values())[0]
        else:
            return json.dumps(content)