CREATE FUNCTION artifacts_children_to_autogovern(parentArtifactId TEXT)
RETURNS TABLE (
    artifact_id TEXT
)
LANGUAGE plpgsql AS
    $func$
BEGIN
    RETURN QUERY

WITH RECURSIVE artifact_graph AS (
    SELECT
        rr.reference_relationship_artifact_id_from AS artifact_id,
        rr.reference_relationship_artifact_id_to AS parent_artifact_id,
        a.artifact_blueprint_id,
        rr.reference_relationship_json,
        ARRAY[rr.reference_relationship_artifact_id_from] AS path, -- to detect cycles (even if should not happen on system objects)
		stop_check_rr.reference_relationship_artifact_id_to AS governed_by,
		-- stop recursion if
        (
			-- the type has no children
			a.artifact_blueprint_id IN ('bp.system.dataiku_bundle', 'bp.system.dataiku_saved_model_version') 
			OR 
			-- the item is not governed, since we will search again for its children after (and if) it has been governed
			stop_check_rr.reference_relationship_artifact_id_from IS NULL
		) AS stop_recursion_at_this_node
    FROM
        reference_relationships rr
    INNER JOIN
        artifacts a ON a.artifact_id = rr.reference_relationship_artifact_id_from
    LEFT JOIN
        reference_relationships stop_check_rr 
		ON stop_check_rr.reference_relationship_artifact_id_from = a.artifact_id
        AND stop_check_rr.reference_relationship_json -> 'relationshipData' ->> 'fieldId' = 'governed_by'
    WHERE
        -- starting parent artifact ID
        rr.reference_relationship_artifact_id_to = parentArtifactId
        AND (
			(a.artifact_blueprint_id IN ('bp.system.dataiku_saved_model_version') AND rr.reference_relationship_json -> 'relationshipData' ->> 'fieldId' = 'dataiku_saved_model')
			OR
			(a.artifact_blueprint_id IN ('bp.system.dataiku_saved_model', 'bp.system.dataiku_bundle') AND rr.reference_relationship_json -> 'relationshipData' ->> 'fieldId' = 'dataiku_project')
		)


    UNION

    -- Recursive Member: This part finds the next level of children.
    -- It will only execute if the stop condition on the parent was NOT met.
    SELECT
        child_rr.reference_relationship_artifact_id_from AS artifact_id,
        child_rr.reference_relationship_artifact_id_to AS parent_artifact_id,
        child_a.artifact_blueprint_id,
        child_rr.reference_relationship_json,
        parent_node.path || child_rr.reference_relationship_artifact_id_from,
		child_stop_check_rr.reference_relationship_artifact_id_to AS governed_by,

        -- evaluate the same stop condition on the child
        (
			child_a.artifact_blueprint_id IN ('bp.system.dataiku_bundle', 'bp.system.dataiku_saved_model_version') 
			OR 
			child_stop_check_rr.reference_relationship_artifact_id_from IS NULL
		) AS stop_recursion_at_this_node

    FROM
        reference_relationships child_rr
    INNER JOIN
        artifact_graph parent_node ON child_rr.reference_relationship_artifact_id_to = parent_node.artifact_id
    INNER JOIN
        artifacts child_a ON child_a.artifact_id = child_rr.reference_relationship_artifact_id_from
    LEFT JOIN
        reference_relationships child_stop_check_rr ON child_stop_check_rr.reference_relationship_artifact_id_from = child_a.artifact_id
                                                    AND child_stop_check_rr.reference_relationship_json -> 'relationshipData' ->> 'fieldId' = 'governed_by'
    WHERE
        -- only continue traversing if the parent node didn't meet the stop condition
        parent_node.stop_recursion_at_this_node = FALSE
        -- and ensure we are not cycling
        AND NOT (child_rr.reference_relationship_artifact_id_from = ANY(parent_node.path))
)
SELECT
    artifact_graph.artifact_id
FROM
    artifact_graph
WHERE governed_by IS NULL;


END;
$func$;