import os, json
from dataiku.code_studio import CodeStudioBlock, get_dataiku_user_uid

from block_utils import LibLocationPathReplacer, infer_base_url_per_port

class RStudioCodeStudioBlock(CodeStudioBlock):
    def __init__(self, config, plugin_config):
        self.config = config
        self.plugin_config = plugin_config
        
    def _get_entrypoint_path(self):
        entrypoint_path = self.config.get("startScript", "/opt/dataiku")
        if entrypoint_path.endswith("/") or not entrypoint_path.endswith(".sh"):
            entrypoint_path = os.path.join(entrypoint_path, "rstudio-entrypoint.sh")
        return entrypoint_path

    def _get_port(self):
        return self.config.get("port", 9292)
        
    def build_spec(self, spec, env):
        port = self._get_port()
        entrypoint_path = self._get_entrypoint_path()
        settings_path = self.config.get("settingsPath", "/home/dataiku/.config")
        open_in_path = self.config.get("openInPath", "/home/dataiku/workspace")

        # replace the lib locations in settings_path and open_in_path
        replacer = LibLocationPathReplacer(spec)
        settings_path = replacer.replace_variable_by_path(settings_path)
        open_in_path = replacer.replace_variable_by_path(open_in_path)

        # add the entrypoint script in the buildir
        entrypoint_script = """
#! /bin/bash

USER=dataiku
HOME=/home/dataiku

export XDG_CONFIG_HOME={settings_path}

mkdir -p XDG_CONFIG_HOME

# Initial config set-up. Should only run once, on the Code Studio first start. Afterwards, we'll get it from the config sync.
if [ ! -d "{settings_path}" ]; then
    # these settings are user-specific, we expect users to have their own
    mkdir -p $XDG_CONFIG_HOME/rstudio
    cat << 'EOF' > $XDG_CONFIG_HOME/rstudio/rstudio-prefs.json
{{
    "initial_working_directory": "{open_in_path}",
    "posix_terminal_shell": "bash"
}}
EOF
fi

# Add all the code envs to R_LIBS (so brace yourself for bugs if there are many)
# no need to insert dataiku lib into R_LIBS_USER, it has already been done by in the base image DockerFile (see build-images.py)
if [ -d /opt/dataiku/r-code-envs ]; then
    for i in $(ls -r /opt/dataiku/r-code-envs); do 
        echo "Adding code env $i"
        export R_LIBS_USER=/opt/dataiku/r-code-envs/$i/R.lib:$R_LIBS_USER
    done
fi

# Add a default writable (by dataiku) library path
HOME_LIB_USER=/home/dataiku/R.lib
mkdir -p $HOME_LIB_USER
export R_LIBS_USER=$HOME_LIB_USER:$R_LIBS_USER

# Pass env vars to RStudio. We could also have used /usr/lib64/R/etc/Renviron.site which have a highest precedence
env | egrep "DKU_|DIP|R_LIBS_USER|XDG_" > /home/dataiku/.Renviron
echo "TMP=$TMPDIR" >> /home/dataiku/.Renviron

export BASE_URL=$(eval echo "$DKU_CODE_STUDIO_BROWSER_PATH_{port}")

if [ $DKU_CODE_STUDIO_IS_PUBLIC_PORT_{port} = "1" ]; then
    BIND_ADDR=0.0.0.0
else
    BIND_ADDR=127.0.0.1
fi

if [ "$UID" -lt "{uid}" ]; then
  echo "Runtime UID is lower than UID set at build-time. RStudio may not work";
fi
USER=dataiku /usr/lib/rstudio-server/bin/rserver --server-daemonize=0 --auth-none=1 --www-address $BIND_ADDR --www-port {port} --auth-minimum-user-id {uid} --www-frame-origin=* --www-root-path $BASE_URL --server-user=dataiku --server-data-dir /home/dataiku/.rstudio-server-run --server-pid-file /home/dataiku/.rstudio-server.pid
""".format(settings_path=settings_path, open_in_path=open_in_path, port=port, uid=get_dataiku_user_uid())
        with open(os.path.join(env["buildDir"], "rstudio-entrypoint.sh"), "wb") as f:
            f.write(entrypoint_script.encode("utf8"))

        default_rpm = r"https://download2.rstudio.org/server/rhel$(rpm -E %{rhel})/x86_64/rstudio-server-rhel-2025.05.0-496-x86_64.rpm"
        # the dockerfile addition
        pkgurl = self.config.get("rstudioServerUrl", default_rpm).strip()
        if not pkgurl:
            pkgurl = default_rpm
        pkg = pkgurl.split("/")[-1]

        spec["dockerfile"] = spec.get("dockerfile", "") + """

##### RSTUDIO BLOCK #####

USER root
WORKDIR /opt/dataiku
        
RUN yum install epel-release -y \
    && curl -O {pkgurl} \
    && yum install {pkg} -y \
    && rm {pkg} \
    && yum -y autoremove && yum clean all 

RUN mkdir -p /usr/share/doc/R
RUN mkdir -p /home/dataiku/.rstudio-server-run/db
RUN chown -R dataiku: /home/dataiku/.rstudio-server-run
# the rserver.conf is in there
RUN chown -R dataiku: /etc/rstudio
# permissions on stuff vscode can have created
RUN find /home/dataiku/ -name .cache | while read line; do chown -R dataiku: $line; done

# Not sure _all_ of them are needed
RUN echo "rsession-which-r=/opt/dataiku/R/bin/R" >> /etc/rstudio/rserver.conf \
 && echo "session-default-working-dir={open_in_path}" >> /etc/rstudio/rsession.conf \
 && echo "session-default-new-project-dir={open_in_path}" >> /etc/rstudio/rsession.conf \
 && echo "[*]" >> /etc/rstudio/logging.conf\
 && echo "log-level=info" >> /etc/rstudio/logging.conf\
 && echo "logger-type=stderr" >> /etc/rstudio/logging.conf\
 && echo -e "provider=sqlite\\ndirectory=/home/dataiku/.rstudio-server-run/db" > /etc/rstudio/database.conf

COPY rstudio-entrypoint.sh {entrypoint_path}
RUN chown dataiku:root {entrypoint_path} && chmod +x {entrypoint_path}
RUN chgrp -R 0 /home/dataiku && chmod -R 775 /home/dataiku

# USER dataiku
USER {uid}
WORKDIR /home/dataiku
""".format(open_in_path=open_in_path, entrypoint_path=entrypoint_path, pkgurl=pkgurl, pkg=pkg, uid=get_dataiku_user_uid())
        return spec

    def build_launch(self, spec, env):
        if env['launchedFrom'] == 'WEBAPP' and not self.config.get("useInWebapps", False):
            return spec
        port = self._get_port()
        spec['entrypoints'] = spec.get('entrypoints', []) + [self._get_entrypoint_path()]
        readiness_probe_url = "http://localhost:" + str(port) + "/images/favicon.ico" # the favicon is not on the subpath
        if spec.get('readinessProbeUrl', "") == "":
            spec['readinessProbeUrl'] = readiness_probe_url
        exposed_port = {"label": "RStudio", "proxiedUrlSuffix": "/", "exposeHtml": True, "port": port, "readinessProbeUrl":readiness_probe_url}
        spec['exposedPorts'] = spec.get('exposedPorts', []) + [exposed_port]
        return spec

    def build_creation(self, spec, env):
        return spec
        
