import asyncio
from collections import defaultdict
from collections import deque
from dataclasses import dataclass
from typing import Awaitable
from typing import Callable
from typing import Dict
from typing import Generic
from typing import Hashable
from typing import List
from typing import TypeVar
from typing import Union
from typing import Deque
from typing import Set

ItemType = TypeVar("ItemType")
ResultType = TypeVar("ResultType")


@dataclass
class _WorkItem(Generic[ItemType]):
    future: asyncio.Future
    item: ItemType


class Batcher(Generic[ItemType, ResultType]):
    """
    A batcher that groups items by a key and processes them in batches of a given size. Only one batch is
    processed at a time. If a batch is not filled within a given timeout, it is processed as is even if it is not
    full.
    """

    _batch_size: int
    _timeout: float
    _process_batch: Callable[[List[ItemType]], Awaitable[List[ResultType]]]
    _group_by: Callable[[ItemType], Hashable]
    _batches: Dict[Hashable, Deque[_WorkItem[ItemType]]]
    _batch_lock: Union[None, asyncio.Lock]
    _background_tasks: Set[asyncio.Task]
    _pending_timeout: Dict[Hashable, asyncio.Task]
    _scheduled: Set[Hashable]

    def __init__(
        self,
        batch_size: int,
        timeout: float,
        group_by: Callable[[ItemType], Hashable],
        process_batch: Callable[[List[ItemType]], Awaitable[List[ResultType]]],
    ):
        self._batch_size = batch_size
        self._timeout = timeout
        self._process_batch = process_batch
        self._group_by = group_by
        self._batches = defaultdict(deque)
        self._batch_lock = None
        self._background_tasks = set()
        self._pending_timeout = {}
        self._scheduled = set()

    async def process(self, item: ItemType) -> ResultType:
        group_id = self._group_by(item)
        future = asyncio.Future()

        self._batches[group_id].append(_WorkItem(future=future, item=item))

        if group_id not in self._scheduled:
            if len(self._batches[group_id]) >= self._batch_size:
                self._cancel_timeout_if_needed(group_id)
                self._schedule_batch_soon(group_id)
            elif group_id not in self._pending_timeout:
                self._schedule_batch_after_timeout(group_id)

        try:
            return await future
        except asyncio.CancelledError:
            if not future.cancelled():
                future.cancel()
            raise

    async def _trigger_timeout(self, group_id):
        await asyncio.sleep(self._timeout)
        if self._batches[group_id] and (group_id not in self._scheduled):
            self._schedule_batch_soon(group_id)
        else:
            del self._batches[group_id]
        if group_id in self._pending_timeout:
            del self._pending_timeout[group_id]

    def _create_task(self, coro):
        task = asyncio.create_task(coro)
        self._background_tasks.add(task)
        task.add_done_callback(self._background_tasks.discard)
        return task

    def _schedule_batch_soon(self, group_id):
        self._scheduled.add(group_id)
        self._create_task(self._process_batch_and_clear(group_id))

    def _schedule_batch_after_timeout(self, group_id):
        task = self._create_task(self._trigger_timeout(group_id))
        self._pending_timeout[group_id] = task

    def _cancel_timeout_if_needed(self, group_id):
        if group_id in self._pending_timeout:
            self._pending_timeout[group_id].cancel()
            del self._pending_timeout[group_id]

    async def _process_batch_and_clear(self, group_id):
        if self._batch_lock is None:
            # Create the lock lazily to allow the batcher to be created before the event loop is running
            self._batch_lock = asyncio.Lock()
        async with self._batch_lock:
            batch = self._batches[group_id]

            batch_items = []
            while batch and len(batch_items) < self._batch_size:
                entry = batch.popleft()
                if not entry.future.cancelled():
                    batch_items.append(entry)

            if batch_items:
                items_to_process = [entry.item for entry in batch_items]
                futures_to_settle = [entry.future for entry in batch_items]

                try:
                    results = await self._process_batch(items_to_process)
                    for future, result in zip(futures_to_settle, results):
                        if not future.done():
                            future.set_result(result)
                except BaseException as e:
                    for future in futures_to_settle:
                        if not future.done():
                            future.set_exception(e)

            if len(batch) >= self._batch_size:
                self._cancel_timeout_if_needed(group_id)
                self._schedule_batch_soon(group_id)
            else:
                self._scheduled.discard(group_id)
                if batch and (group_id not in self._pending_timeout):
                    self._schedule_batch_after_timeout(group_id)
                else:
                    del self._batches[group_id]
