import os
import sys
import logging
import shutil
import tempfile
import uuid
from dataiku.base.utils import is_os_windows
import time

logger = logging.getLogger(__name__)

class SafeFile(object):

    def __init__(self, file_path, mode):
        """
        Context manager to safely and atomically write a file to `file_path`.
        Yields a file like object that will be written to `file_path` upon exit/close.

        If multiple threads / process try to concurrently write to the same `file_path`, this should not fail and the
        last to close will have its content saved in `file_path`

        NB: almost entirely inspired from: https://stackoverflow.com/a/12007885

        :param file_path: str
        :param mode: str
        """
        file_dir, file_name = os.path.split(file_path)
        # the utility is responsible for deleting the file upon exit or close
        random_filename = os.path.join(file_dir, file_name + str(uuid.uuid4()))
        self.temp_file = open(random_filename, mode=mode)
        self.file_path = file_path
        self._is_windows = is_os_windows()

    def __enter__(self):
        self.temp_file.__enter__()
        return self

    def __getattr__(self, attr):
        return getattr(self.temp_file, attr)

    def close(self):
        # no need to catch errors here, if something bad happened, we don't want to go further
        self.temp_file.close()
        self._rename()

    def _rename(self):
        if sys.version_info >= (3, 3):
            renaming_func = os.replace
        else:
            renaming_func = os.rename

        if not self._is_windows:
            renaming_func(self.temp_file.name, self.file_path)
            return

        # On windows there can be a race condition where we get a permissin error
        # if we try to modify the file while it's being read. The easiest workaround
        # is to retry after a short period of time.
        exception = None
        for _ in range(100):
            try:
                renaming_func(self.temp_file.name, self.file_path)
                return
            except Exception as e:
                exception = e
                time.sleep(0.03)
        raise exception

    def __exit__(self, exc_type, exc_val, exc_tb):
        if exc_type is None:
            result = self.temp_file.__exit__(exc_type, exc_val, exc_tb)
            self._rename()
        else:
            result = self.temp_file.__exit__(exc_type, exc_val, exc_tb)
            if os.path.isfile(self.temp_file.name):
                os.remove(self.temp_file.name)

        return result
