(function(){
'use strict';

var app = angular.module('dataiku.ml.core', []);

app.constant("worstPrefixName", "Worst ");
/**
 * Filtering of prediction models
 */

app.factory("_MLFilteringServicePrototype", function(Fn) {
    const computeMetric = (model, prop, currentMetric, customMetrics, currentMetricIsCustom, parentService) => {
        var modelProp = Fn.prop(prop)(model);
        if (currentMetricIsCustom) {
            modelProp.mainMetric = null;
            modelProp.mainMetricStd = null;
            modelProp.sortMainMetric = -1 * Number.MAX_VALUE;

            if (modelProp.customMetricsResults) {
                const foundCustomMetricResult = modelProp.customMetricsResults.filter(cmr => cmr.metric.name === currentMetric)[0];
                if (foundCustomMetricResult) {
                    modelProp.mainMetric = foundCustomMetricResult.value;
                    const lib = !foundCustomMetricResult.metric.greaterIsBetter ? -1 : 1;
                    if (modelProp.mainMetric) {
                        modelProp.sortMainMetric = lib * modelProp.mainMetric;
                    }
                    modelProp.mainMetricStd = foundCustomMetricResult.valuestd;
                }
            }
        } else {
            const lib = parentService.MLSettings.sort.lowerIsBetter(currentMetric, customMetrics) ? -1 : 1;
            modelProp.mainMetric = modelProp[parentService.metricMap[currentMetric]];
            modelProp.sortMainMetric = modelProp.mainMetric ? lib * modelProp.mainMetric : -1 * Number.MAX_VALUE;
            if (parentService.getMetricStdFromSnippet) {
                modelProp.mainMetricStd = parentService.getMetricStdFromSnippet(modelProp, currentMetric);
            }
        }

        if (modelProp.partitions && modelProp.partitions.summaries) {
            Object.values(modelProp.partitions.summaries)
                .forEach(summary => summary.snippet && computeMetric(summary.snippet, [], currentMetric, customMetrics, currentMetricIsCustom, parentService));
        }
    };
    return {
        /**
         *
         * @param {array} modelsList - array of models
         * @param {string[]} rootProp - property path to the model metric
         * @param {string} currentMetric - current selected metric
         * @param {array[]} customMetrics - custom metrics array from the modeling params
         * @param {boolean} currentMetricIsCustom - current metric is custom
         */
        setMainMetric: function(modelsList, rootProp, currentMetric, customMetrics, currentMetricIsCustom=false)  {
            if (!rootProp) rootProp = [];
            modelsList.forEach((m) => {
                computeMetric(m, rootProp, currentMetric, customMetrics, currentMetricIsCustom, this);
            });
        },
        getMetricFromSnippet: function(model, metric) {
            return model[this.metricMap[metric]];
        }
    };
});

app.factory("AlgorithmsSettingsService", function() {
    function getCustomAlgorithmSettings(mlTaskDesign, algorithmKey) {
        if (algorithmKey.startsWith('custom_python_')) {
            return mlTaskDesign.modeling.custom_python[algorithmKey.slice(14)];
        } else if (algorithmKey.startsWith('custom_mllib_')) {
            return mlTaskDesign.modeling.custom_mllib[algorithmKey.slice(13)];
        }
    }

    function getPluginAlgorithmSettings(mlTaskDesign, algorithmKey) {
        return mlTaskDesign.modeling.plugin_python[algorithmKey] || {};
    }

    function getAlgorithmSettings(mlTaskDesign, algorithm) {
        if (algorithm.isCustom) {
            return getCustomAlgorithmSettings(mlTaskDesign, algorithm.algKey)
        } else if (algorithm.algKey.startsWith("CustomPyPredAlgo_")) {
            return getPluginAlgorithmSettings(mlTaskDesign, algorithm.algKey);
        } else {
            return mlTaskDesign.modeling[algorithm.hpSpaceName || algorithm.algKey];
        }
    }

    function getDefaultAlgorithm(mlTaskDesign, algorithms) {
        const filteredAlgorithms = algorithms.filter(function(o){ return (!o.condition || o.condition()) });

        return (
            filteredAlgorithms.find(algorithm => getAlgorithmSettings(mlTaskDesign, algorithm).enabled)
            ||
            filteredAlgorithms[0]
        ).algKey;
    }

    function addCustomAlgorithmsToBaseAlgorithms(baseAlgorithms, customAlgorithms, isPythonBackend) {
        (customAlgorithms || []).forEach(function(customAlgo, idx) {
            const newAlgo = {
                name: customAlgo.name || (isPythonBackend ? "Custom python model" : "Custom mllib model"),
                algKey: (isPythonBackend ? "custom_python_" : "custom_mllib_") + idx,
                isCustom: true,
            };
            if (isPythonBackend) newAlgo.supportedCausalMethod = 'META_LEARNER';
            baseAlgorithms.push(newAlgo)
        });
    }

    return {
        getAlgorithmSettings,
        getCustomAlgorithmSettings,
        getPluginAlgorithmSettings,
        getDefaultAlgorithm,
        addCustomAlgorithmsToBaseAlgorithms,
    }
})

app.factory("CustomMetricIDService", function(worstPrefixName) {
    return {
        checkMetricIsCustom: function(metricId) {
            if (!metricId) return "";
            return metricId.substring(0,2) === "!!" || metricId.substring(0, 8) === "WORST_!!";
        },
        checkMetricIsWorst : function(metricId) {
            if (!metricId) return "";
            return metricId.substring(0, 5) === "WORST";
        },
        getCustomMetricId: function(metricName) {
            if (!metricName) return "";
            return "!!" + metricName;
        },
        getCustomMetricName: function(metricId) {
            if (!metricId) return "";
            if (metricId.substring(0,2) === "!!"){
                return metricId.substring(2);
            } else if (metricId.substring(0, 8) === "WORST_!!") {
                return worstPrefixName + metricId.substring(8)
            }
        },
        getCustomMetricBaseName : function(metricId) {
            if (!metricId) return "";
            if (metricId.substring(0,2) === "!!"){
                return metricId.substring(2);
            } else if (metricId.substring(0, 8) === "WORST_!!") {
                return metricId.substring(8)
            }
        }
    }
});

app.factory("FinetuningUtilsService", function() {
    function supportsModelDeployment(llmType) {
        return ['BEDROCK', 'AZURE_OPENAI_MODEL', 'SAVED_MODEL_FINETUNED_AZURE_OPENAI', 'SAVED_MODEL_FINETUNED_BEDROCK'].includes(llmType);
    }
    return {
        supportsModelDeployment,
        getNewDeploymentMessage: function(llmType, checked,) {
            const suffix = (checked ? "will" : "would") + " be created.";
            const count = supportsModelDeployment(llmType) ? 1 : 0;
            return `${count === 0 ? 'No' : count} ${count === 1 ? 'deployment' : 'deployments'} ${suffix}`
        },
        getDeletedDeploymentsCountMessage: function(deployments, savedModel, newVersionId, manualActivation, checked) {
            const suffix = (checked ? "will" : "would") + " be deleted.";
            let count;
            if (manualActivation) { // Make active button is used
                count = deployments.length - !!deployments.find(d => d.versionId === newVersionId);
            } else { // Recipe run context. SM settings taken into account
                const deploymentsVersionIds = deployments.map((d => d.versionId));
                count = (deployments.length - (savedModel.publishPolicy === 'MANUAL' && deploymentsVersionIds.includes(savedModel.activeVersion)));
            }

            return `${count === 0 ? 'No' : count} ${count === 1 ? 'deployment' : 'deployments'} ${suffix}`
        }
    }
})

app.factory("DeepHubMetricsService", function() {

    // There should be only one 'default' per predictionType, in sync with what is set in the backend
    // in `DeepHubMetricParams` and `DeepHubPredictionGuesser`
    const allMetricsInfo = {
        DEEP_HUB_IMAGE_OBJECT_DETECTION: [
            {
                name: "AVERAGE_PRECISION_IOU50",
                default: true,
                fieldName: "averagePrecisionIOU50",
                description: "Average Precision (IoU=0.5)",
                explanation: "Average Precision for IoU set to 0.5" // to be refined
            },
            {
                name: "AVERAGE_PRECISION_IOU75",
                fieldName: "averagePrecisionIOU75",
                description: "Average Precision (IoU=0.75)",
                explanation: "Average Precision for IoU set to 0.75" // to be refined
            },
            {
                name: "AVERAGE_PRECISION_ALL_IOU",
                fieldName: "averagePrecisionAllIOU",
                description: "Average Precision (all IoUs)",
                explanation: "Average Precision, averaged over all IoUs between 0.5 and 0.95 (with a 0.05 step)" // to be refined
            }
        ],
        DEEP_HUB_IMAGE_CLASSIFICATION: [  // TODO @deephub: merge with multiclass metrics ?
            {
                name: "ROC_AUC",
                default: true,
                fieldName: "auc",
                description: "ROC AUC",
                perfFieldName: "mrocAUC" /* We need this additional field for AUC metric because modelData.perf.metrics doesn't expect the same
                field name than snippetData (mrocAUC vs auc). This is to be consistent with the classical doctor in multiclass case,
                see getMetricValueFromModel for more details. */
            },
            {
                name: "AVERAGE_PRECISION",
                fieldName: "averagePrecision",
                description: "Average Precision"
            },
            {
                name: "PRECISION",
                fieldName: "precision",
                description: "Precision"
            },
             {
                name: "RECALL",
                fieldName: "recall",
                description: "Recall"
            },
            {
                name: "F1",
                fieldName: "f1",
                description: "F1-score"
            },
             {
                name: "ACCURACY",
                fieldName: "accuracy",
                description: "Accuracy"
            },
             {
                name: "LOG_LOSS",
                fieldName: "logLoss",
                description: "Log Loss"
            }
        ]
    }
    const confidenceScoreThresholdOptimMetrics = [
        {
            name: "F1",
            description: "F1 Score"
        },
        {
            name: "PRECISION",
            description: "Precision",
        },
        {
            name: "RECALL",
            description: "Recall"
        }
    ];

    return {
        objectDetectionMetricsInfo: () => {
            return allMetricsInfo.DEEP_HUB_IMAGE_OBJECT_DETECTION;
        },
        metricNameToFieldNameMap: () => {
            const map = {};
            for (const metricsForPredictionType of Object.values(allMetricsInfo)) {
                for (const metric of metricsForPredictionType) {
                    map[metric.name] = metric.fieldName;
                }
            }
            return map;
        },
        metricNameToPerfFieldNameMap: () => {
            const map = {};
            for (const metricsForPredictionType of Object.values(allMetricsInfo)) {
                for (const metric of metricsForPredictionType) {
                    map[metric.name] = metric.perfFieldName? metric.perfFieldName: metric.fieldName;
                }
            }
            return map;
        },
        metricNameToDescriptionMap: () => {
            const map = {};
            for (const metricsForPredictionType of Object.values(allMetricsInfo)) {
                for (const metric of metricsForPredictionType) {
                    map[metric.name] = metric.description;
                }
            }
            return map;
        },
        getPossibleMetricsNames: (predictionType) => {
            if (!(predictionType in allMetricsInfo)) {
                throw new Error("Unsupported prediction type: " + predictionType);
            }
            return allMetricsInfo[predictionType].map(metricsForPredictionType => metricsForPredictionType.name);
        },
        metricNameToDescriptionMapPerPrediction: (predictionType) => {
            if (!(predictionType in allMetricsInfo)) {
                throw new Error("Unsupported prediction type: " + predictionType);
            }

            const map = {};
                for (const metric of allMetricsInfo[predictionType]) {
                    map[metric.name] = metric.description;
                }
            return map;
        },
        getDefaultMetricName: (predictionType) => {
            if (!(predictionType in allMetricsInfo)) {
                throw new Error("Unsupported prediction type: " + predictionType);
            }
            const defaultMetric = allMetricsInfo[predictionType].find(m => m.default);
            if (defaultMetric !== undefined) {
                return defaultMetric.name;
            }
            throw new Error("Prediction type '" + predictionType + "' must have a default metric");
        },
        confidenceScoreThresholdOptimMetricNameToDescription: () => {
            return confidenceScoreThresholdOptimMetrics.map(e => [e.name, e.description]);
        }
    }
})

app.factory("PMLFilteringService", function(_MLFilteringServicePrototype, Assert, PMLSettings, BinaryClassificationModelsService, DeepHubMetricsService, CustomMetricIDService, $filter) {
    var svc = Object.create(_MLFilteringServicePrototype);
    svc.MLSettings = PMLSettings;
    svc.metricMap = {
        ACCURACY: 'accuracy',
        PRECISION: 'precision',
        RECALL: 'recall',
        F1: 'f1',
        CALIBRATION_LOSS: 'calibrationLoss',
        COST_MATRIX: 'costMatrixGain',
        CUMULATIVE_LIFT: 'lift',
        LOG_LOSS: 'logLoss',
        ROC_AUC: 'auc',
        AVERAGE_PRECISION: 'averagePrecision',
        EVS: 'evs',
        MAPE: 'mape',
        MAE: 'mae',
        MSE: 'mse',
        RMSE: 'rmse',
        RMSLE: 'rmsle',
        R2: 'r2',
        PEARSON: 'pearson',
        CUSTOM: 'customScore',
        DATA_DRIFT: 'dataDrift',
        DATA_DRIFT_PVALUE: 'dataDriftPValue',
        AUUC: "auuc",
        QINI: "qini",
        NET_UPLIFT: "netUplift",
        MASE: 'mase',
        SMAPE: 'smape',
        MEAN_ABSOLUTE_QUANTILE_LOSS: 'meanAbsoluteQuantileLoss',
        MEAN_WEIGHTED_QUANTILE_LOSS: 'meanWeightedQuantileLoss',
        MSIS: 'msis',
        ND: 'nd',
        WORST_MASE: 'worstMase',
        WORST_MAPE: 'worstMape',
        WORST_SMAPE: 'worstSmape',
        WORST_MSE: 'worstMse',
        WORST_MSIS: 'worstMsis',
        WORST_MAE: 'worstMae',
        MIN_KS: 'minKs',
        MIN_CHISQUARE: 'minChiSquare',
        MAX_PSI: 'maxPsi',
        PREDICTION_DRIFT_KS: 'predictionDrift_KS',
        PREDICTION_DRIFT_PSI: 'predictionDrift_PSI',
        PREDICTION_DRIFT_CHISQUARE: 'predictionDrift_ChiSquare',
        FAITHFULNESS: 'faithfulness',
        MULTIMODAL_FAITHFULNESS: 'multimodalFaithfulness',
        ANSWER_RELEVANCY: 'answerRelevancy',
        MULTIMODAL_RELEVANCY: 'multimodalRelevancy',
        ANSWER_CORRECTNESS: 'answerCorrectness',
        ANSWER_SIMILARITY: 'answerSimilarity',
        CONTEXT_RECALL: 'contextRecall',
        CONTEXT_PRECISION: 'contextPrecision',
        BERT_SCORE_PRECISION: 'bertScorePrecision',
        BERT_SCORE_RECALL: 'bertScoreRecall',
        BERT_SCORE_F1: 'bertScoreF1',
        BLEU: 'bleu',
        ROUGE_1_PRECISION: 'rouge1Precision',
        ROUGE_1_RECALL: 'rouge1Recall',
        ROUGE_1_F1: 'rouge1F1',
        ROUGE_2_PRECISION: 'rouge2Precision',
        ROUGE_2_RECALL: 'rouge2Recall',
        ROUGE_2_F1: 'rouge2F1',
        ROUGE_L_PRECISION: 'rougeLPrecision',
        ROUGE_L_RECALL: 'rougeLRecall',
        ROUGE_L_F1: 'rougeLF1',
        INPUT_TOKENS_PER_ROW: 'inputTokensPerRow',
        OUTPUT_TOKENS_PER_ROW: 'outputTokensPerRow',
        AVERAGE_TOOL_EXECUTIONS_PER_ROW: 'averageToolExecutionsPerRow',
        AVERAGE_FAILED_TOOL_EXECUTIONS_PER_ROW: 'averageFailedToolExecutionsPerRow',
        AVERAGE_TOOL_EXECUTION_TIME_SECONDS_PER_ROW: 'averageToolExecutionTimeSecondsPerRow',
        P95_TOTAL_AGENT_CALL_EXECUTION_TIME_SECONDS_PER_ROW: 'p95TotalAgentCallExecutionTimeSecondsPerRow',
        SAMPLE_ROW_COUNT: 'sampleRowCount',
        TOOL_CALL_EXACT_MATCH: 'toolCallExactMatch',
        TOOL_CALL_PARTIAL_MATCH: 'toolCallPartialMatch',
        TOOL_CALL_PRECISION: 'toolCallPrecision',
        TOOL_CALL_RECALL: 'toolCallRecall',
        TOOL_CALL_F1: 'toolCallF1',
        AGENT_GOAL_ACCURACY_WITH_REFERENCE: 'agentGoalAccuracyWithReference',
        AGENT_GOAL_ACCURACY_WITHOUT_REFERENCE: 'agentGoalAccuracyWithoutReference'
    };

    // Adding Deephub metrics
    Object.assign(svc.metricMap, DeepHubMetricsService.metricNameToFieldNameMap());

    svc.getPossibleMetrics = function(mlTask) {
        const ret = [];
        switch(mlTask.predictionType) {
            case 'BINARY_CLASSIFICATION':
                ret.push('ACCURACY', 'PRECISION', 'RECALL', 'F1', 'COST_MATRIX', 'ROC_AUC', 'CUMULATIVE_LIFT', 'AVERAGE_PRECISION', 'LOG_LOSS', 'CALIBRATION_LOSS');
                break;
            case 'MULTICLASS':
                ret.push('ACCURACY', 'PRECISION', 'RECALL', 'F1', 'ROC_AUC', 'AVERAGE_PRECISION', 'LOG_LOSS', 'CALIBRATION_LOSS');
                break;
            case 'REGRESSION':
                ret.push('EVS', 'MAPE', 'MAE', 'MSE', 'RMSE', 'RMSLE', 'R2', 'PEARSON');
                break;
            case 'TIMESERIES_FORECAST':
                ret.push('MASE', 'MAPE', 'SMAPE', 'MAE', 'MEAN_ABSOLUTE_QUANTILE_LOSS', 'MEAN_WEIGHTED_QUANTILE_LOSS', 'MSE', 'RMSE', 'MSIS', 'ND');
                break;
            case 'DEEP_HUB_IMAGE_OBJECT_DETECTION':
            case 'DEEP_HUB_IMAGE_CLASSIFICATION':
                ret.push(...DeepHubMetricsService.getPossibleMetricsNames(mlTask.predictionType));
                break;
            case 'CAUSAL_BINARY_CLASSIFICATION':
            case 'CAUSAL_REGRESSION':
                ret.push("AUUC", "QINI", "NET_UPLIFT");
                break;
        }
        return ret.map(function(m) { return [m, PMLSettings.names.evaluationMetrics[m]] });
    };

    // This will return the metric object for the most recently used version of each custom metric (done by name)
    // E.g., if a model is trained with custom metric 'CM1'.greaterIsBetter = true, then trained again with 'CM1'.greaterIsBetter = false,
    // the second (newer) custom metric object will be returned
    svc.getPossibleCustomMetrics = function(modelSnippets) {
        const metricMap = new Map();
        for (const[_, snippet] of Object.entries(modelSnippets)) {
            if(snippet && snippet.customMetricsResults) {
                const snippetDate = snippet.trainDate;
                snippet.customMetricsResults.forEach(customMetricResult => {
                    const metricName = customMetricResult.metric.name;
                    if (!metricMap.has(metricName) || snippetDate > metricMap.get(metricName)[0]) {
                        metricMap.set(metricName, [snippetDate, customMetricResult.metric]);
                    }
                });
            }
        }

        const ret = [];
        metricMap.forEach(function(data) {
            const metric = data[1];
            metric.id = CustomMetricIDService.getCustomMetricId(metric.name);
            ret.push(metric);
        });
        return ret;
    }

    svc.getMetricStdFromSnippet = function(model, metric) {
        switch (metric) {
            case "LOG_LOSS" : return model.logLossstd;
            case "CUSTOM" : return model.customScorestd;
            default:
                return model[svc.metricMap[metric] + "std"];
        }
    }

    svc.getMetricValueFromModelWithCurrentCut = function(modelData, metric) {
        const pcd = BinaryClassificationModelsService.findCutData(modelData.perf, modelData.userMeta.activeClassifierThreshold);
        return svc.getMetricValueFromModel(modelData, metric, pcd);
    }

    svc.getMetricValueFromModel = function(modelData, metric, currentCutData) {
        function getFromTIMetrics() {
            if (!(modelData.perf && modelData.perf.tiMetrics)) {
                return '<No metric available>';
            }
            if (!modelData.perf.tiMetrics[svc.metricMap[metric]]) {
                return '<'+metric+' not available>';
            }
            return modelData.perf.tiMetrics[svc.metricMap[metric]];
        };

        function getFromCurrentCutData(key) {
            Assert.trueish(currentCutData, 'no currentCutData');
            return currentCutData[key];
        };

        function extractFromCustomMetrics(customMetricName, modelData) {
            const customMetric = modelData.modeling.metrics.customMetrics.find(customMetric => customMetric.name === customMetricName);
            let customMetricsResults;
            if (customMetric.needsProbability) {
                customMetricsResults = modelData.perf.tiMetrics.customMetricsResults
            } else {
                Assert.trueish(currentCutData, 'no currentCutData');
                customMetricsResults = currentCutData.customMetricsResults
            }
            return customMetricsResults.filter(item => item.metric.name === customMetric.name)[0].value;
        }
        function getMetricForNonBinaryClassifPredictions(metrics, metricMap, metricField) {
            Assert.trueish(metricField, 'metric not specified');
            const metricName = metricMap[metricField];
            Assert.trueish(metricName, 'cannot display metric ' + metricField);

            if (!metrics) return '<No metric available>';
            if (!(metricName in metrics) || metrics[metricName] == undefined) {
                return '<' + metricName + ' not available>';
            }
            return metrics[metricName];
        };

        const metricIsCustom = CustomMetricIDService.checkMetricIsCustom(metric);

        // This not really a metric (like the others) but we want to treat it as such
        if(metric === "DATA_DRIFT" && modelData.dataEvaluationMetrics && modelData.dataEvaluationMetrics.driftModelAccuracy) {
            return modelData.dataEvaluationMetrics.driftModelAccuracy.value;
        }
        Assert.trueish(modelData, 'no modelData');
        Assert.trueish(modelData.coreParams, 'no coreParams');
        switch (modelData.coreParams.prediction_type) {
            case 'BINARY_CLASSIFICATION':
                Assert.trueish(metric, 'metric not specified');

                if (metricIsCustom) {
                    return extractFromCustomMetrics(CustomMetricIDService.getCustomMetricName(metric), modelData);
                }
                switch (metric) {
                    case "F1":
                        Assert.trueish(currentCutData, 'no currentCutData');
                        return getFromCurrentCutData("F1-Score");
                    case "RECALL":
                        Assert.trueish(currentCutData, 'no currentCutData');
                        return getFromCurrentCutData("Recall");
                    case "PRECISION":
                        Assert.trueish(currentCutData, 'no currentCutData');
                        return getFromCurrentCutData("Precision")
                    case "ACCURACY":
                        Assert.trueish(currentCutData, 'no currentCutData');
                        return getFromCurrentCutData("Accuracy")
                    case "COST_MATRIX":
                        Assert.trueish(currentCutData, 'no currentCutData');
                        return getFromCurrentCutData("cmg");
                    case "LOG_LOSS":
                    case "ROC_AUC":
                    case "AVERAGE_PRECISION":
                    case "CUMULATIVE_LIFT":
                        return getFromTIMetrics();
                    case "CUSTOM":
                        return extractFromCustomMetrics(modelData.modeling.metrics.customEvaluationMetricName, modelData);
                }
                Assert.fail('Unknown metric ' + metric);
            case "MULTICLASS":
            {
                if (metricIsCustom) {
                    const metricName = CustomMetricIDService.getCustomMetricName(metric);
                    return modelData.perf && modelData.perf.metrics && modelData.perf.metrics.customMetricsResults.filter(item => item.metric.name === metricName)[0].value;
                } else {
                    const multiclassMetricMap = {...svc.metricMap, ROC_AUC: "mrocAUC"};
                    return getMetricForNonBinaryClassifPredictions(
                        modelData.perf && modelData.perf.metrics, multiclassMetricMap, metric
                    );
                }
            }
            case "DEEP_HUB_IMAGE_OBJECT_DETECTION":
            case "DEEP_HUB_IMAGE_CLASSIFICATION":
                return getMetricForNonBinaryClassifPredictions(
                    modelData.perf && modelData.perf.metrics,
                    DeepHubMetricsService.metricNameToPerfFieldNameMap(), metric
                );
            case "REGRESSION": {
                if (metricIsCustom) {
                    const metricName = CustomMetricIDService.getCustomMetricName(metric);
                    return modelData.perf && modelData.perf.metrics && modelData.perf.metrics.customMetricsResults.filter(item => item.metric.name === metricName)[0].value;
                } else {
                    return getMetricForNonBinaryClassifPredictions(
                        modelData.perf && modelData.perf.metrics, svc.metricMap, metric
                    );
                }
            }
            case "CAUSAL_BINARY_CLASSIFICATION":
            case "CAUSAL_REGRESSION":
                return modelData.perf.causalPerf.normalized[svc.metricMap[metric]];
            case "TIMESERIES_FORECAST":
                return getMetricForNonBinaryClassifPredictions(
                    modelData.perf && modelData.perf.aggregatedMetrics, svc.metricMap, metric
                );
        }
    };

    const imageMetricToSuffixMap = [
        { name: 'meanRedPerFeature',        suffix: '_Mean_Red_KS' },
        { name: 'meanGreenPerFeature',      suffix: '_Mean_Green_KS' },
        { name: 'meanBluePerFeature',       suffix: '_Mean_Blue_KS' },
        { name: 'meanSaturationPerFeature', suffix: '_Mean_Saturation_KS' },
        { name: 'rmsContrastPerFeature',    suffix: '_Contrast_RMS_KS' },
        { name: 'laplacianVarPerFeature',   suffix: '_Sharpness_Laplacian_KS' },
        { name: 'tenengradPerFeature',      suffix: '_Sharpness_Tenengrad_KS' },
        { name: 'entropyPerFeature',        suffix: '_Entropy_Complexity_KS' },
        { name: 'edgeDensityPerFeature',    suffix: '_Edge_Density_KS' },
        { name: 'areaPerFeature',           suffix: '_Area_KS' },
        { name: 'aspectRatioPerFeature',    suffix: '_Aspect_Ratio_KS' },
    ];

    svc.isEmbeddingMetric = function(metric) {
        return (metric.endsWith('_ED')) || (metric.endsWith('_CS')) || (metric.endsWith('_Classifier_gini'));
    }

    svc.isImageMetric = function(metric) {
        return imageMetricToSuffixMap.some(config => metric.endsWith(config.suffix));
    }

    svc.isUnivariateMetric = function(metric) {
        return (metric.endsWith('_KS') && metric !== "MIN_KS" && metric !== "PREDICTION_DRIFT_KS" && metric !== "predictionDrift_KS"
                && ! svc.isImageMetric(metric))
            || (metric.endsWith('_PSI') && metric !== "MAX_PSI" && metric !== "PREDICTION_DRIFT_PSI" && metric !== "predictionDrift_PSI")
            || (metric.endsWith('_Chi-square') && metric !== "predictionDrift_ChiSquare");
    }

    svc.univariateMetrics = ["ksPerFeature", "psiPerFeature", "chiSquarePerFeature"];
    svc.embeddingMetrics  = ["euclidianDistancePerFeature", "cosineSimilarityPerFeature", "classifierGiniPerFeature"]
    svc.imageMetrics      = imageMetricToSuffixMap.map(config => config.name)
    svc.dataDriftMetrics  = ["DATA_DRIFT", "DATA_DRIFT_PVALUE", "MIN_KS", "MIN_CHISQUARE", "MAX_PSI", "dataDrift",
                            "dataDriftPValue", "minKs", "minChiSquare", "maxPsi", "dataDriftDeviation",
                            ...svc.univariateMetrics, ...svc.embeddingMetrics, ...svc.imageMetrics];
    svc.predictionDriftMetrics = ["PREDICTION_DRIFT_KS", "PREDICTION_DRIFT_CHISQUARE", "PREDICTION_DRIFT_PSI",
                                  "predictionDrift_KS", "predictionDrift_ChiSquare", "predictionDrift_PSI"];
    svc.driftMetrics = svc.dataDriftMetrics.concat(svc.predictionDriftMetrics);

    svc.isDataDriftMetric = metric => svc.dataDriftMetrics.includes(metric) || svc.isUnivariateMetric(metric)
                                      || svc.isEmbeddingMetric(metric) || svc.isImageMetric(metric) ;
    svc.isPredictionDriftMetric = metric => svc.predictionDriftMetrics.includes(metric);
    svc.isDriftMetric = metric => svc.isDataDriftMetric(metric) || svc.isPredictionDriftMetric(metric);

    svc.univariateMetric = function(metric, metrics) {
        if (metric.endsWith('_KS')){
            return metrics.ksPerFeature[metric.split('_KS')[0]]
        }
        if (metric.endsWith('_PSI')){
            return metrics.psiPerFeature[metric.split('_PSI')[0]]
        }
        if (metric.endsWith('_Chi-square')){
            return metrics.chiSquarePerFeature[metric.split('_Chi-square')[0]]
        }
        return undefined;
    }

    svc.embeddingMetric = function(metric, metrics) {
        if (metric.endsWith('_ED')) {
            return metrics.euclidianDistancePerFeature[metric.split('_ED')[0]]
        }
        if (metric.endsWith('_CS')) {
            return metrics.cosineSimilarityPerFeature[metric.split('_CS')[0]]
        }
        if (metric.endsWith('_Classifier_gini')) {
            return metrics.classifierGiniPerFeature[metric.split('_Classifier_gini')[0]]
        }
        return undefined;
    }

    svc.imageMetric = function(metric, metrics) {
        const config = imageMetricToSuffixMap.find(c => metric.endsWith(c.suffix));

        if (config) {
            const baseKey = metric.split(config.suffix)[0];
            return metrics[config.name][baseKey];
        }
        return undefined;
    }

    svc.getDataDriftWithDeviation = function(metrics, metric, precision){
        let ret = $filter('mlMetricFormat')(metrics[svc.metricMap[metric]], metric, precision);
        if (metrics['dataDriftDeviation']){
            ret = ret.toString() + ' ± (' + $filter('mlMetricFormat')(metrics['dataDriftDeviation'], metric, precision) + ')';
        }
        return ret;
    }

    svc.getEvaluationMetricName = function(metric) {
        if ('DATA_DRIFT' === metric) {
            return 'Data drift';
        } else if (svc.isUnivariateMetric(metric) || svc.isEmbeddingMetric(metric) || svc.isImageMetric(metric)) {
            return metric;
        } else if (CustomMetricIDService.checkMetricIsCustom(metric)) {
            return CustomMetricIDService.getCustomMetricName(metric)
        } else {
            return PMLSettings.names.evaluationMetrics[metric];
        }
    }

    return svc;
});

app.factory("MLContainerInfoService", function(DataikuAPI) {

    function inContainer($scope, projectKey) {

        // Retrieving list of containers to know if computation will occur on a container or not
        let listContainersWithDefault = null;
        DataikuAPI.containers.listNamesWithDefault(projectKey, null, "USER_CODE").success(function(data) {
                    listContainersWithDefault = data;
        }).error(setErrorInScope.bind($scope));

        return function(selectedContainer) {
            if (selectedContainer.containerMode === "NONE" || listContainersWithDefault === null) {
                return false;
            } else if (selectedContainer.containerMode === "INHERIT") {
                return listContainersWithDefault.resolvedInheritValue != null;
            } else {
                return true;
            }
        };
    }

    return {
        inContainer
    }

})

app.factory("CMLFilteringService", function(_MLFilteringServicePrototype, Assert, CMLSettings, Fn) {
    var svc = Object.create(_MLFilteringServicePrototype);
    svc.MLSettings = CMLSettings;
    svc.metricMap = { SILHOUETTE: 'silhouette', INERTIA: 'inertia', NB_CLUSTERS: 'nbClusters' };
    svc.getPossibleMetrics = Fn.cst(CMLSettings.task.evaluationMetrics);
    // Custom Metrics are not available when clustering
    svc.getPossibleCustomMetrics = Fn.cst([]);

    svc.getMetricNameFromModel = function(modelData) {
        Assert.trueish(modelData, 'no modelData');
        return modelData.actualParams.resolved.metrics.evaluationMetric;
    };

    svc.getMetricValueFromModel = function(modelData) {
        Assert.trueish(modelData, 'no modelData');
        let metricName = modelData.actualParams.resolved.metrics.evaluationMetric;
        metricName = svc.metricMap[metricName];
        let metricValue = modelData.perf.metrics[metricName];
        return metricValue;
    };

    return svc;
});

app.service("MLExportService", function(DataikuAPI, WT1, $stateParams, Dialogs, CreateModalFromTemplate, SpinnerService, FutureWatcher, FullModelLikeIdUtils, SavedModelsService) {
    this.downloadFile = (scope, generateFile, getUrl) => {
        generateFile()
            .success(data => {
                SpinnerService.lockOnPromise(FutureWatcher.watchJobId(data.jobId)
                    .success(data => {
                        downloadURL(getUrl(data.result.exportId));
                        scope.dismiss();
                    }).error(error => {
                        Dialogs.error(scope, "An error occured while exporting file", error.message);
                    }));
               })
            .error(error => {
                Dialogs.error(scope, "An error occured while exporting file", error.message);
            });
    };

    this.showExportModel = function(appConfig) {
        if (!appConfig.licensedFeatures) {
            return false;
        }
        else {
            return appConfig.licensedFeatures.modelsExport
        }
    };

    this.downloadDocForbiddenReason = function(appConfig, model){
        if (!appConfig.graphicsExportsEnabled){
            return "Graphics export must be enabled in order to use this feature. Your administrator needs to intervene to fix the issue."
        }
        if (!model) return null;
        if (model.backendType === 'KERAS') {
            return "Deep learning models are not compatible with documentation export";
        }
        if (model.backendType === 'DEEP_HUB') {
            return "Computer vision models are not compatible with documentation export";
        }
        if(["CAUSAL_REGRESSION", "CAUSAL_BINARY_CLASSIFICATION"].includes(model.coreParams.prediction_type)) {
            return "Causal prediction models are not compatible with documentation export";
        }
        if (model && model.modeling.algorithm === 'VIRTUAL_PROXY_MODEL') {
            return "External Models are not compatible with documentation export";
        }
        if (model.modeling.algorithm.endsWith('_ENSEMBLE')){
            return "Ensemble models are not compatible with documentation export.";
        }
        return null;
    }

    this.downloadDoc = function(scope) {
        CreateModalFromTemplate("/templates/analysis/prediction/model/download-documentation-modal.html",
            scope, "DownloadModelDocumentationController", undefined, undefined, 'static');
    }

    this.mayExportModel = function(model, type) {
        switch(type){
            case 'pmml': return model && model.pmmlCompatibility.compatible;
            case 'python': return model && model.pythonCompatibility.compatible;
            case 'jar': return model && model.javaExportCompatibility.compatible;
            case 'snowflake': return model && model.javaExportCompatibility.compatible;
            default: return model && (model.pmmlCompatibility.compatible || model.pythonCompatibility.compatible || model.javaExportCompatibility.compatible);
        }
    }

    this.exportModelModal = function(scope, model, cantExportToSnowflake) {
        scope.exportType = model.javaExportCompatibility.compatible ? (cantExportToSnowflake ? "jar" : "snowflake") : (model.pythonCompatibility.compatible ? "python" : null);
        CreateModalFromTemplate("/templates/analysis/prediction/model/export-model-modal.html", scope, "ExportModelController", function(modalScope){
            modalScope.cantExportToSnowflake = cantExportToSnowflake
            modalScope.model = model;
            modalScope.exportType = scope.exportType
            DataikuAPI.connections.getNames("Snowflake").success(function (connectionNames) {
                scope.snowflakeConnectionNames = connectionNames;
            }).error(setErrorInScope.bind(scope));
            DataikuAPI.connections.getNames("DatabricksModelDeployment").success(function (connectionNames) {
                scope.databricksConnectionNames = connectionNames;
            }).error(setErrorInScope.bind(scope));
        })
    }

    this.disableExportModelModalReason = function(model){
        if (model && model.backendType === "KERAS"){
            return "Export of Deep Learning models is not supported"
        }
        if (model && model.backendType === "DEEP_HUB"){
            return "Export of Computer vision models is not supported"
        }
        if (model && model.coreParams && model.coreParams.prediction_type === "TIMESERIES_FORECAST"){
            return "Export of Time series forecasting models is not supported"
        }
        if (model && model.coreParams && ["CAUSAL_BINARY_CLASSIFICATION", "CAUSAL_REGRESSION"].includes(model.coreParams.prediction_type)) {
            return "Export of Causal prediction models is not supported"
        }
        if (SavedModelsService.isProxyModel(model)) {
            return "Export of External Models is not supported";
        }
        return null;
    }

    //TO DO sc-86783 : Delete, saved for now to keep the logic about partitioned model exports
    this.exportModel = function(scope, model, type, partitionName) {
        if (type == 'docgen') {
            if (scope.appConfig.graphicsExportsEnabled) {
                if (['KERAS_CODE', 'DEEP_HUB'].includes(model.backendType)) {
                    Dialogs.error(scope, "Model is not compatible", "Model backend not supported: " + model.backendType);
                } else if (model.modeling.algorithm === 'VIRTUAL_MLFLOW_PYFUNC') {
                    Dialogs.error(scope, "Model is not compatible", "Documentation export of MLflow imported models is not supported");
                } else if (model.modeling.algorithm === 'VIRTUAL_PROXY_MODEL') {
                    Dialogs.error(scope, "Model is not compatible", "Documentation export of External Models is not supported");
                } else if (!model.modeling.algorithm.endsWith('_ENSEMBLE')) {
                    CreateModalFromTemplate("/templates/analysis/prediction/model/download-documentation-modal.html",
                        scope, "DownloadModelDocumentationController", undefined, undefined, 'static');
                } else {
                    Dialogs.error(scope, "Model is not compatible", "Documentation export of ensembled models is not supported");
                }
            } else {
                CreateModalFromTemplate("/templates/exports/graphics-export-disabled-modal.html", scope);
            }

        } else if (type === 'jar-thin' || type === 'jar-fat') {
            if (model.javaExportCompatibility.compatible) {
                const downloadPrompt = (id) => {
                    Dialogs.prompt(
                        scope,
                        "Download model JAR",
                        "Fully-qualified class name for the model",
                        "com.company.project.Model",
                        { pattern: "^((?:[a-z]\\w*\\.)*)?([A-Z]\\w*)$" }
                    ).then( name => {
                        WT1.event("model-export", {exportType: type});
                        this.downloadFile(scope, () => DataikuAPI.ml.prediction.createScoringModelFile(type, id, "&fullClassName=" + encodeURIComponent(name)),
                            (exportId) => DataikuAPI.ml.prediction.getScoringModelDownloadURL(type, exportId));
                    });
                };
                if (! partitionName) {
                    downloadPrompt(model.fullModelId);
                } else {
                    const choices = [
                        {
                            id: FullModelLikeIdUtils.getBase(model.fullModelId),
                            title: "Full model, with all partitions"
                        }, {
                            id: model.fullModelId,
                            title: "Single model partition",
                            desc: partitionName
                        }
                    ];
                    Dialogs.select(
                        scope,
                        "Export partitioned model",
                        "Do you want to export the full model or just the current partition?",
                        choices,
                        choices[0]
                    ).then((selected) => { downloadPrompt(selected.id); });
                }
            } else {
                Dialogs.error(scope, "Model is not compatible with Java export", model.javaExportCompatibility.reason);
            }
        } else if (type === 'pmml' && ! model.pmmlCompatibility.compatible) {
            Dialogs.error(scope, "Model is not compatible with PMML export", model.pmmlCompatibility.reason);
        } else if (type === 'python' && ! model.pythonCompatibility.compatible) {
            Dialogs.error(scope, "Model is not compatible with Python export", model.pythonCompatibility.reason);
        } else if (type === 'mlflow' && ! model.pythonCompatibility.compatible) {
            Dialogs.error(scope, "Model is not compatible with MLflow export", model.pythonCompatibility.reason);
        } else {
            WT1.event("model-export", {exportType: type});
            this.downloadFile(scope, () => DataikuAPI.ml.prediction.createScoringModelFile(type, model.fullModelId),
             (exportId) => DataikuAPI.ml.prediction.getScoringModelDownloadURL(type, exportId));
        }
    }
});

app.factory("ExportModelDatasetService", function(DataikuAPI, $stateParams, $state, Dialogs, CreateExportModal, FutureProgressModal, FullModelLikeIdUtils) {
    const service = {
        exportTrainTestSets: exportTrainTestSets,
        exportTrainTestSetsForbiddenReason: exportTrainTestSetsForbiddenReason,
        exportPredictedData: exportPredictedData,
        exportPredictedDataForbiddenReason: exportPredictedDataForbiddenReason,
    };
    return service;

    ////////////////

    function exportTrainTestSets(scope, fullModelId) {
        const datasetDialog = {
            title : "Export the train and test sets"
        };
        const features = {  // only allow export to dataset
            hideDestinationTabs: true,
            hideAdvancedParameters: true
        };
        CreateExportModal(scope, datasetDialog, features, { destinationType: 'DATASET' }).then(function(params) {
            DataikuAPI.ml.exportModelDataset(fullModelId, params).then(function(response) {
                showProgressDialog(scope, response.data.futureResponse, params.destinationDatasetName, "train and test sets");
            }).catch(function(response) {
                Dialogs.error(scope, "An error occurred while exporting the train and test sets", response.data.detailedMessageHTML);
            });
        });
    }

    function exportPredictedData(scope, fullModelId) {
        const datasetDialog = {
            title: "Export the predicted data"
        };
        const features = {  // only allow export to dataset
            hideDestinationTabs: true,
            hideAdvancedParameters: true
        };
        CreateExportModal(scope, datasetDialog, features, {destinationType: 'DATASET'}).then(function (params) {
            DataikuAPI.ml.exportModelPredictedData(fullModelId, params).then(function (response) {
                showProgressDialog(scope, response.data.futureResponse, params.destinationDatasetName, "predicted data");
            }).catch(function (response) {
                Dialogs.error(scope, "An error occurred while exporting the predicted data", response.data.detailedMessageHTML);
            });
        });
    }

    function showProgressDialog(scope, futureResponse, destinationDatasetName, dataTypeName, warningMessage) {
        FutureProgressModal.show(scope, futureResponse, "Exporting the " + dataTypeName).then(function(result) {
            if (result) {
                const datasetPage = "projects.project.datasets.dataset.explore";
                const datasetPageParams = {projectKey: $stateParams.projectKey, datasetName: destinationDatasetName};
                const datasetLink = `<a href="` + $state.href(datasetPage, datasetPageParams) + `">` + sanitize(destinationDatasetName) + "</a>"
                let message;
                if (warningMessage) {
                    message = `<p class="alert alert-warning">Dataset ` + datasetLink + ` has been created.<br />` + sanitize(warningMessage) + `</p>`;
                } else {
                    message = `<p class="alert alert-success">Dataset ` + datasetLink + ` has been created successfully.</p>`;
                }
                Dialogs.confirm(scope,
                    "Exported the " + dataTypeName, message,
                    {
                        btnCancel: "Close",
                        btnConfirm: "View dataset",
                        positive: true,
                    }).then(function() {
                        $state.go(datasetPage, datasetPageParams);
                    }, function() {
                        // Dialog closed
                    });
            }
        });
    }

    function exportTrainTestSetsForbiddenReason(model, canWriteProject) {
        if (!model) {
            return null;
        }
        if (model.coreParams.prediction_type === 'TIMESERIES_FORECAST') {
            return "Not available for time series forecasting. Export the resampled data from the Predicted data tab.";
        }
        if(model.trainInfo.kfold) {
            return "Export of train and test sets not supported for models evaluated using kfold";
        }
        const isPartitionedModel = model.coreParams.partitionedModel && model.coreParams.partitionedModel.enabled;
        const isIndividualPartitionedModel = !!FullModelLikeIdUtils.parse(model.fullModelId).partitionName;
        if(isPartitionedModel && !isIndividualPartitionedModel) {
            return "Export of train and test sets for overall model is not supported (only available for individual partitions)";
        }
        if (model.modeling && model.modeling.algorithm && model.modeling.algorithm == "VIRTUAL_MLFLOW_PYFUNC") {
            return "Models imported from mlflow do not have train and test data to export";
        }
        if (!(model.splitDesc && model.splitDesc.testPath && model.splitDesc.trainPath)) {
            return "No train and test sets to export for this model";
        }
        if (typeof canWriteProject !== 'undefined' && !canWriteProject){
            return "You don't have write permissions for this project";
        }
    }

    function exportPredictedDataForbiddenReason(model) {
        if (!model) {
            return null;
        }
        if(model.coreParams.partitionedModel && model.coreParams.partitionedModel.enabled &&
            model.splitDesc && model.splitDesc.params && model.splitDesc.params.ssdSelection && model.splitDesc.params.ssdSelection.partitionSelectionMethod == "ALL") {
            return "Export of predicted data for overall model is not supported (only available for individual partitions)";
        }
    }
});

app.filter("gridParameter", function(){
    return function(par){
        if (par.vals) {
             return par.vals.join(", ");
        } else if (par.val) {
            return par.val;
        } else if (par.cnt) {
            return par.cnt + " value(s)";
        } else if (par.min && par.max) {
            return "(" + par.min + ", " + par.max + ")";
        }
    };
});

app.directive("gridDescription", function(){
    return {
        scope: { desc: '=', trained: '=', gridsearchData: "=", filterPostTrainParams: "=" },
        restrict: 'A',
        templateUrl: '/templates/ml/model-snippet-grid-description.html',
        link: function(scope) {
            scope.filterPostTrainParams = scope.filterPostTrainParams || (param => true);
        }
    };
});

app.directive("modelSnippet", function($state, $rootScope, $filter, PMLSettings, MLModelsUIRouterStates, ModelDataUtils, CustomMetricIDService, FullModelLikeIdUtils, CreateModalFromTemplate, DataikuAPI, $stateParams, AnyLoc, PromptUtils) {
    return {
        scope: { snippetData: '=', snippetSource : '@', taskType : '=', smData : '=', makeActive : '=', deleteModel : '=', hideSelectors: '@', currentMetric: '='},
        templateUrl: '/templates/ml/model-snippet.html',
        link: function($scope) {
            $scope.$state = $state;
            $scope.appConfig = $rootScope.appConfig;
            $scope.out = $scope.$parent;
            $scope.ModelDataUtils = ModelDataUtils;
            let defaultTab =  '.report';
            if (["TOOLS_USING_AGENT", "PLUGIN_AGENT", "PYTHON_AGENT"].includes($scope.snippetData.savedModelType)) {
                defaultTab = '.design';
            }
            $scope.baseRef = $scope.out.sRefPrefix + ($scope.snippetSource === 'SAVED' ? '' : '.model') + defaultTab;

            $scope.displayMainMetric = function(snippetData, currentMetric) {
                if (!currentMetric) return "";
                return $scope.out.isModelDone(snippetData) && (snippetData.mainMetric || CustomMetricIDService.checkMetricIsCustom(currentMetric));
            }

            $scope.getDeploymentStatus = function(snippetData, deployments) {
                if (!snippetData.deployment || !deployments) return;
                const deployment = deployments.find(d => d.versionId == snippetData.deployment.versionId)
                return deployment && deployment.status;
            }

            $scope.getTabForDiagnostic = () => {
                if ($scope.taskType === "CLUSTERING") {
                    return "train";
                }
                const isDeephubPrediction = $scope.snippetData.backendType === 'DEEPHUB';
                if ($scope.snippetData.partitionedModelEnabled) {
                    return MLModelsUIRouterStates.getPredictionReportSummaryTab(isDeephubPrediction, false)
                }
                return MLModelsUIRouterStates.getPredictionReportTrainTab(isDeephubPrediction)
            }
            $scope.getTabForOverrideMetrics = () => MLModelsUIRouterStates.getPredictionReportOverridesMetricsTab();
            if ($scope.snippetData.predictionType === "TIMESERIES_FORECAST") {
                $scope.algosWithoutQuantiles = PMLSettings.algorithmCategories("TIMESERIES_FORECAST")["Baseline Models"];
                $scope.$watch('snippetData.forecasts', function() {
                    if (!$scope.snippetData.forecasts) return;
                    $scope.$selectedTimeseries = Object.keys($scope.snippetData.forecasts.perTimeseries)[0];

                    $scope.filterImportantParamPerTs = function(param) {
                        return !param.id || param.id && param.id === $scope.$selectedTimeseries;
                    };
                });
            }

            $scope.openExternalModelSetupMonitoringModal = function(proxyModelEndpointInfo, versionId) {
                //This method is only used for proxy models
                return CreateModalFromTemplate("/templates/savedmodels/external-model-setup-monitoring.html", $scope, 'ExternalModelSetupMonitoringModalController', function(modalScope) {
                    modalScope.createMes = true;
                    modalScope.createOutputDataset = true;
                    modalScope.predictionLogsUri = proxyModelEndpointInfo.predictionLogsUri;
                    modalScope.versionId = versionId;
                });
            }

            $scope.getSummariesAsArray = function(snippetData) {
                if (snippetData && snippetData.partitions && snippetData.partitions.summaries) {
                    return Object.entries(snippetData.partitions.summaries).map(s => ({name: s[0], summary: s[1]}));
                } else {
                    return [];
                }
            };

            $scope.getLLMIcon = function(llmType, size) {
                return PromptUtils.getLLMIcon(llmType, size);
            }

            $scope.uiState = $scope.uiState || {}

            if ($scope.snippetData) {
                $scope.uiState.parsedFMI = FullModelLikeIdUtils.parse($scope.snippetData.fullModelId);

                if ($scope.snippetData.proxyModelConfiguration && $scope.snippetData.proxyModelEndpointInfo) {
                    switch ($scope.snippetData.proxyModelConfiguration.protocol) {
                        case 'vertex-ai':
                            $scope.uiState.proxyModelConfigurationUIInfo = {
                                protocol: $scope.snippetData.proxyModelConfiguration.protocol,
                                project_id: $scope.snippetData.proxyModelConfiguration.project_id,
                                region: $scope.snippetData.proxyModelConfiguration.region,
                                endpoint_id: $scope.snippetData.proxyModelEndpointInfo.name ? $scope.snippetData.proxyModelEndpointInfo.name.split('/').pop() : '',
                                prediction_type: $filter('mlModelType')($scope.snippetData.predictionType)
                            }
                            break;
                        case 'sagemaker':
                            $scope.uiState.proxyModelConfigurationUIInfo = {
                                protocol: $scope.snippetData.proxyModelConfiguration.protocol,
                                region: $scope.snippetData.proxyModelConfiguration.region,
                                endpoint_name: $scope.snippetData.proxyModelEndpointInfo.arn ? $scope.snippetData.proxyModelEndpointInfo.arn.split('/').pop() : '',
                                prediction_type: $filter('mlModelType')($scope.snippetData.predictionType)
                            }
                            break;
                        case 'azure-ml':
                            $scope.uiState.proxyModelConfigurationUIInfo = {
                                protocol: $scope.snippetData.proxyModelConfiguration.protocol,
                                subscription_id: $scope.snippetData.proxyModelConfiguration.subscription_id,
                                resource_group: $scope.snippetData.proxyModelConfiguration.resource_group,
                                workspace: $scope.snippetData.proxyModelConfiguration.workspace,
                                endpoint_name: $scope.snippetData.proxyModelEndpointInfo.endpointInfo &&
                                     $scope.snippetData.proxyModelEndpointInfo.endpointInfo.id ? $scope.snippetData.proxyModelEndpointInfo.endpointInfo.id.split('/').pop() : '',
                                prediction_type: $filter('mlModelType')($scope.snippetData.predictionType)
                            }
                            break;
                        case 'databricks':
                            $scope.uiState.proxyModelConfigurationUIInfo = {
                                protocol: $scope.snippetData.proxyModelConfiguration.protocol,
                                endpoint_name: $scope.snippetData.proxyModelEndpointInfo.endpointName,
                                prediction_type: $filter('mlModelType')($scope.snippetData.predictionType)
                            }
                            break;
                        }
                }

                if ($scope.snippetData.savedModelType && $scope.snippetData.savedModelType === "TOOLS_USING_AGENT") {
                    let allUsedToolRefs = $scope.snippetData.toolsUsingAgentSettings.tools
                        .filter(tool => tool.toolRef)
                        .map(tool => AnyLoc.getLocFromSmart($stateParams.projectKey, tool.toolRef).fullId);
                    DataikuAPI.agentTools.listAvailable($stateParams.projectKey).success(function(data) {
                        const toolLookup = {};
                        data.forEach(item => {
                            toolLookup[`${item.projectKey}.${item.id}`] = item;
                        });

                        $scope.snippetData.fullTools = allUsedToolRefs
                            .map(toolRefId => {
                                const item = toolLookup[toolRefId];
                                if (item) {
                                    const isForeign = item.projectKey !== $stateParams.projectKey;

                                    return angular.extend({}, item, {
                                        foreign: isForeign,
                                        name: isForeign ? `${item.projectKey}.${item.name}` : item.name
                                    });
                                } else {
                                    // Can happen for deleted or unshared tools
                                    return null;
                                }
                            })
                            .filter(item => item !== null);
                    }).error(setErrorInScope.bind($scope));

                    DataikuAPI.pretrainedModels.listAvailableLLMs($stateParams.projectKey, "GENERIC_COMPLETION").success(function(data) {
                        $scope.snippetData.llm = data.identifiers.find(llm => llm.id === $scope.snippetData.toolsUsingAgentSettings.llmId);
                    }).error(setErrorInScope.bind($scope));
                }
            } else {
                $scope.uiState.parsedFMI = null;
            }
        }
    };
});


app.directive("externalEndpointSync", function(DataikuAPI, FutureProgressModal, InfoMessagesModal) {
    return {
        bindings: {
            projectKey: '<',
            smId: '<',
            versionId: '<'
        },
        scope: {
            projectKey: '<',
            smId: '<',
            versionId: '<'
        },
        replace: false,
        template: `
            <a ng-click="checkEndpoint()" toggle="tooltip" title="{{uiState.endpointCheckProgress}}">
                <i class="{{uiState.endpointStatusIcon}}" style="vertical-align: {{uiState.verticalAlign}};"></i> Check Endpoint
            </a>
        `,
        link: function ($scope) {
            $scope.uiState = {
                endpointCheckProgress: "Sync with remote endpoint not checked",
                endpointStatusIcon: "dku-icon-question-circle-outline-16",
                endpointStatusBtn: null,
                verticalAlign: "bottom"
            };
            $scope.checkEndpoint = function () {
                $scope.uiState.endpointStatusIcon = "dku-icon-arrow-clockwise-dashes-12 icon-spin";
                $scope.uiState.endpointStatusBtn = null;
                $scope.uiState.verticalAlign = "middle";

                DataikuAPI.savedmodels.prediction.checkProxyModelVersionEndpoint($scope.projectKey, $scope.smId, $scope.versionId).then(function (resp) {
                    FutureProgressModal.show($scope, resp.data, "Checking endpoint").then(function (result) {
                        $scope.uiState.verticalAlign = "bottom";
                        if (result) {
                            let subHeader = "Differences were found between the current endpoint and the endpoint when" +
                                " this Saved Model Version was created. If those differences impact the model" +
                                " predictions, we recommend creating a new Saved Model Version to materialize" +
                                " this change."
                            if (result.maxSeverity == "INFO") {
                                $scope.uiState.endpointStatusIcon = "dku-icon-checkmark-circle-outline-16";
                                $scope.uiState.endpointCheckProgress = "Online endpoint matches active version";
                                subHeader = null;
                            } else if (result.maxSeverity == "WARNING") {
                                $scope.uiState.endpointStatusIcon = "dku-icon-arrow-circular-strike-16";
                                $scope.uiState.endpointCheckProgress = "Online endpoint does not fully match this version";
                            } else {
                                $scope.uiState.endpointStatusIcon = "dku-icon-error-circle-outline-16";
                                $scope.uiState.endpointCheckProgress = "Online endpoint differs from this version";
                                $scope.uiState.endpointStatusBtn = "btn--danger";
                            }
                            InfoMessagesModal.showIfNeeded($scope, result, "Endpoint check result", subHeader);
                        } else {
                            $scope.uiState.endpointCheckProgress = "Sync with remote endpoint not checked";
                            $scope.uiState.endpointStatusIcon = "dku-icon-question-circle-outline-16";
                            $scope.uiState.endpointStatusBtn = null;
                        }
                    });
                }, setErrorInScope.bind($scope));
            }
        }
    }
});

app.filter('safeTemplateUrl', function(LoggerProvider) {
    const logger = LoggerProvider.getLogger('ml.core');
    const safeTemplateUrlRegex = /^\/(templates|static\/dataiku)(\/[-\w]+)+\.html$/;
    return path => {
        if (safeTemplateUrlRegex.test(path)) {
            return path;
        } else {
            logger.warn("Invalid template url: " + path);
            return 'invalid_template_url';
        }
    };
});

app.filter('niceModelState', function ($filter) {
    return function (state, source) {
        if (!state) {
            return '-';
        }
        if (state.startsWith('REUSED_')) {
            return 'Re-used';
        } else if (state === 'DONE') {
            return 'Trained';
        }

        return $filter('niceConst')(state);
    }
});

app.directive("modelState", function($state, MLModelsUIRouterStates) {
    return {
        scope: { state: '=', model: '=', sRefPrefix: '=', displayDiagnostics: '='},
        templateUrl: '/templates/ml/model-snippet-state.html',
        link: function($scope) {
            $scope.$state = $state;
            $scope.getTrainTab = () => {
                return MLModelsUIRouterStates.getPredictionReportTrainTab($scope.model.backendType === 'DEEPHUB');
            }
        }
    };
});


app.directive("modelsTable", function($state, DataikuAPI){
    return {
        scope:true,
        link : function($scope, element) {
            $scope.saveMeta = function(snippetData) {
                DataikuAPI.ml.saveModelUserMeta(snippetData.fullModelId, snippetData.userMeta)
                            .error(setErrorInScope.bind($scope.$parent));
            }
        }
    }
});


app.filter('mlMetricFormat', function() {
    // probably 0 is suspicious in any metric but for already identified cases, we do not display 0
    var ignoreZerosForMetrics = [
        'INERTIA', // Inertia is missing for all but K-means, don't display it
        'RMSLE', // RMSLE cannot be computed sometimes (negative values in log)
    ];

    var usePercentageForMetrics = [
        'MAPE', 'SMAPE'
    ];

    return function(metricValue, metricName, precision, sigma, exp) {
        if ( (metricValue === undefined) || (metricValue === null) || ('<No metric available>' == metricValue) || (metricValue == '<'+metricName+' not available>')) {
            return "-";
        }
        if (ignoreZerosForMetrics.indexOf(metricName) >= 0 && metricValue == 0) {
            return "-";
        }

        var percent = usePercentageForMetrics.indexOf(metricName) >= 0;
        var sigmaPrecision;
        var pc = '';

        if(typeof precision === 'number') {
            sigmaPrecision = precision;
        } else {
            precision = 4;
            sigmaPrecision = 2;
        }

        if (percent) {
            pc = '%';
            metricValue *= 100;
            sigma = (sigma || 0) * 100;
            precision = Math.max(1, precision - 2);
            sigmaPrecision = Math.max(1, sigmaPrecision - 2);
            exp = false;
        }

        var abs = Math.abs(metricValue);
        if (isNaN(abs)) {
            return "-";
        } else if (abs >= 10000 && !percent) { // big numbers in exp notation
            exp = true;
        } else if (abs >= 100) { // medium numbers w/o decimals
            precision = 0;
            exp = false;
        }
        // 95% of values fall within two standard deviations. If another confidence interval is chosen, make sure to modify the display in Govern too.
        return (metricValue || 0)[exp ? 'toPrecision' : 'toFixed'](precision) + pc +
            (sigma ? ' <span class="sigma">(\u00b1\u00a0' + (2 * sigma)[exp ? 'toPrecision' : 'toFixed'](sigmaPrecision) + pc + ')</span>' : '');
    };
});

app.filter("mlMetricName", function($filter, PMLSettings, CMLSettings, Fn) {
    var allMetrics = angular.extend({},
            PMLSettings.names.evaluationMetrics,
            CMLSettings.names.evaluationMetrics);
    return function(input, snippetData){
        if (input == "CUMULATIVE_LIFT") {
            return $.isNumeric(snippetData.liftPoint) ? "Lift at " + Math.round(snippetData.liftPoint* 100) + "%" : "Lift";
        } else if (input == "NET_UPLIFT") {
            return $.isNumeric(snippetData.netUpliftPoint) ? "Net Uplift at " + Math.round(snippetData.netUpliftPoint* 100) + "%" : "Net Uplift";
        }
        return allMetrics[input] || $filter("niceConst")(input);
    }
});

app.filter('mlScoreAssess', function() {
    var SCORES = { // all lowercase and attached for easy access
        auc: [ ["...too good to be true?", 1], ["excellent", .9], ["very good", .8], ["good", .7],
               ["fair", .6], ["not very good...", .5], ["worse than random guess :("] ],
        pearson: [ ["...too good to be true?", 1], ["very good", .8], ["good", .7], ["fair", .5], ["not very good..."] ],
        pvalue: [ ["\u2605\u2605\u2605", .001], ["\u2605\u2605\u2606", .01],
                  ["\u2605\u2606\u2606", .05 ], ["\u2606\u2606\u2606", 1] ]
    }, LOWER_BETTER = ['pvalue'];

    function test(k) {
        if (k.length === 1 || (this.lt ? this.score <= k[1] : this.score >= k[1])) {
            this.ret = k[0];
            return true;
        }
        return false;
    }

    return function(score, metric) {
        metric = metric.toLowerCase().split('_').join(''); // accomodate constants
        if (! metric in SCORES) { throw "Unkown metric: " + metric; }
        var ctx = { score: +score, grades: SCORES[metric], ret: null,
                lt: LOWER_BETTER.indexOf(metric) !== -1 };
        return ctx.grades.some(test, ctx) ? ctx.ret : "";
    };
});



app.filter('mlFeature', function($filter, Fn, FeatureNameUtils) {
    return function(input, asHtml) {
        if(asHtml){
            return FeatureNameUtils.getAsHtmlString(input);
        } else {
            return FeatureNameUtils.getAsText(input);
        }
    };
});

app.filter('mlModelType', function() {
    return function(predictionType) {
        if (!predictionType) {
            return '-';
        }
        const predictionTypeToLabel = {
            'BINARY_CLASSIFICATION': 'Two-class classification',
            'MULTICLASS': 'Multiclass classification',
            'REGRESSION': 'Regression',
            'CLUSTERING': 'Clustering',
            'DEEP_HUB_IMAGE_CLASSIFICATION': 'Image classification',
            'DEEP_HUB_IMAGE_OBJECT_DETECTION': 'Object detection',
            'TIMESERIES_FORECAST': 'Time Series Forecasting',
            'CAUSAL_BINARY_CLASSIFICATION': 'Causal classification',
            'CAUSAL_REGRESSION': 'Causal regression'
        };
        return predictionTypeToLabel[predictionType] || predictionType;
    };
});

    app.filter('weightMethod', function() {
        return function(weightMethod) {
            if (!weightMethod) {
                return '-';
            }
            const weightMethodToLabel = {
                'NO_WEIGHTING': 'No weighting',
                'SAMPLE_WEIGHT': 'Sample weights',
                'CLASS_WEIGHT': 'Class weights',
                'CLASS_AND_SAMPLE_WEIGHT': 'Class and sample weights',
            };
            return weightMethodToLabel[weightMethod] || weightMethod;
        };
    });

// FMI parsing methods should be equivalent to the ones defined in com.dataiku.dip.analysis.ml.FullModelId
app.service("FullModelLikeIdUtils", function($stateParams){
    function parseAnalysisModel(fullIdString) {
        const analysisPattern = /^A-(\w+)-(\w+)-(\w+)-(s[0-9]+)-(pp[0-9]+(?:-part-(\w+)|-base)?)-(m[0-9]+)$/;
        const matchingResult = fullIdString.match(analysisPattern);
        if (!matchingResult) {
            throw new Error("Invalid analysis model id: " + fullIdString);
        } else {
            const [_, projectKey, analysisId, mlTaskId, sessionId, ppsId, partitionName, modelId] = matchingResult;
            return { projectKey, analysisId, mlTaskId, sessionId, ppsId, partitionName, modelId,
                partitionedBase: ppsId.endsWith('-base')};
        }
    }

    function parseSavedModel(fullIdString) {
        const savedModelPattern = /^S-(\w+)-(\w+)-(\w+)(?:-part-(\w+)-(v?\d+))?$/;
        const matchingResult = fullIdString.match(savedModelPattern);
        if (!matchingResult) {
            throw new Error("Invalid saved model id: " + fullIdString);
        } else {
            const [_, projectKey, savedModelId, versionId, partitionName, partitionVersion] = matchingResult;
            return { projectKey, savedModelId, versionId, partitionName, partitionVersion };
        }
    }

    function parseModelEvaluation(fullIdString) {
        const modelEvaluationPattern = /^ME-(\w+)-(\w+)-(\w+)$/;
        const matchingResult = fullIdString.match(modelEvaluationPattern);
        if (!matchingResult) {
            throw new Error("Invalid model evaluation id: " + fullIdString);
        } else {
            const [_, projectKey, id, evaluationId] = matchingResult;
            return { projectKey, id, evaluationId };
        }
    }

    function buildAnalysisModelFmi(fmiComponents) {
        // ppsId holds potential partition info
        return "A-{0}-{1}-{2}-{3}-{4}-{5}".format(fmiComponents.projectKey, fmiComponents.analysisId,
                                                 fmiComponents.mlTaskId, fmiComponents.sessionId,
                                                 fmiComponents.ppsId, fmiComponents.modelId);
    }

    function buildSavedModelFmi(fmiComponents) {
        let fmi;
        const idComponents = fmiComponents.savedModelId.split(".");
        if (idComponents.length > 2) {
            throw new Error("Invalid savedModelID");
        } else if (idComponents.length == 2) {
            // foreign / shared model
            fmi = "S-{0}-{1}-{2}".format(idComponents[0], idComponents[1], fmiComponents.versionId);
        } else {
            fmi = "S-{0}-{1}-{2}".format(fmiComponents.projectKey, fmiComponents.savedModelId, fmiComponents.versionId);
        }
        if (fmiComponents.partitionName) {
            fmi += "-part-{0}-{1}".format(fmiComponents.partitionName, fmiComponents.partitionVersion);
        }
        return fmi;
    }

    function buildModelEvaluationFmeFromComponents(projectKey, id, evaluationId) {
        const idComponents = id.split(".");
        if (idComponents.length > 2) {
            throw new Error("Invalid FME ID");
        } else if (idComponents.length == 2) {
            // foreign / shared model evaluation
            return "ME-{0}-{1}-{2}".format(idComponents[0], idComponents[1], evaluationId);
        } else {
            return "ME-{0}-{1}-{2}".format(projectKey, id, evaluationId);
        }
    }

    function buildModelEvaluationFme(fmeComponents) {
        return buildModelEvaluationFmeFromComponents(fmeComponents.projectKey, fmeComponents.id, fmeComponents.evaluationId);
    }

    function isAnalysis(fullId) {
        return fullId.startsWith("A");
    }

    function isSavedModel(fullId) {
        return fullId.startsWith("S");
    }

    function isModelEvaluation(fullId) {
        return fullId.startsWith("ME");
    }

    function parse(fullIdString) {
        if (isAnalysis(fullIdString)) {
            return parseAnalysisModel(fullIdString);
        } else if (isSavedModel(fullIdString)) {
            return parseSavedModel(fullIdString);
        } else if (isModelEvaluation(fullIdString)) {
            return parseModelEvaluation(fullIdString);
        } else {
            throw new Error("Invalid id: " + fullIdString);
        }
    }

    // Enforcing projectKey to be current Project and not the one hard coded in fullModelId
    // to prevent from breaking when changing projectKey of analysis (e.g. importing project
    // and changing projectKey)
    // See FullModelId.buildFmiWithEnforcedProjectKey() in the backend
    function parseWithEnforcedProjectKey(fmi, projectKey) {
        const elements = parse(fmi);
        elements.projectKey = projectKey;
        return {elements, fullModelId: buildAnalysisModelFmi(elements)};
    }

    function getFmi(scope) {
        let fmi = $stateParams.fullModelId; // Saved models & Doctor --> FullModelLikeId is in $stateParams
        if (scope.evaluation) { // MES --> FullModelLikeId is not in $stateParams
            fmi = scope.evaluation.evaluation.fullId || fmi;
        }
        return fmi || scope.fullModelId; // Dashboards --> FullModelLikeId is in the scope
    }

    return {
        parse: parse,
        getBase: function(fullModelId) {
            if (isAnalysis(fullModelId)) {
                const fmiComponents = parseAnalysisModel(fullModelId);
                fmiComponents.ppsId = fmiComponents.ppsId.replace(/-part-(\w+)/, "-base")
                return buildAnalysisModelFmi(fmiComponents);
            } else if (isSavedModel(fullModelId)) {
                const fmiComponents = parseSavedModel(fullModelId);
                delete fmiComponents.partitionName;
                return buildSavedModelFmi(fmiComponents);
            } else {
                throw new Error("Invalid model id: " + fullModelId);
            }
        },
        buildAnalysisModelFmi,
        buildSavedModelFmi,
        buildModelEvaluationFme,
        buildModelEvaluationFmeFromComponents,
        parseWithEnforcedProjectKey,
        isAnalysisPartitionBaseModel: function (fullModelId) {
            if (isAnalysis(fullModelId)) {
                const parts = parse(fullModelId);
                return parts.partitionedBase;
            }
            return false;
        },
        isPartition: function (fullModelId) {
            const parts = parse(fullModelId);
            return !angular.isUndefined(parts.partitionName);
        },
        getFmi,
        isAnalysis,
        isSavedModel,
        isModelEvaluation
    }
});

app.service('ModelDataUtils', function(FullModelLikeIdUtils) {
    function areMetricsWeighted(modelData) {
        return modelData
            && modelData.coreParams
            && !!modelData.coreParams.weight
            && (modelData.coreParams.weight.weightMethod === "SAMPLE_WEIGHT"
                || modelData.coreParams.weight.weightMethod === "CLASS_AND_SAMPLE_WEIGHT");
    }
    function getAlgorithm(modelData) {
        if (modelData
            && modelData.actualParams
            && modelData.actualParams.resolved
            && modelData.actualParams.resolved.algorithm) {
            return modelData.actualParams.resolved.algorithm;
        } else if (modelData
            && modelData.modeling
            && modelData.modeling.algorithm) {
            return modelData.modeling.algorithm;
        }
    }
    function hasCalibration(modelData) {
        return modelData
            && modelData.coreParams
            && modelData.coreParams.calibration
            && modelData.coreParams.calibration.calibrationMethod !== 'NO_CALIBRATION';
    }
    function isPartitionedBaseModel(modelData) {
        return modelData
            && modelData.coreParams
            && modelData.coreParams.partitionedModel
            && modelData.coreParams.partitionedModel.enabled
            && !FullModelLikeIdUtils.isPartition(modelData.fullModelId);
    }
    function isBinaryClassification(modelData) {
        return modelData
            && modelData.coreParams
            && modelData.coreParams.prediction_type === 'BINARY_CLASSIFICATION';
    }
    function isMulticlass(modelData) {
        return modelData
            && modelData.coreParams
            && ['MULTICLASS', 'DEEP_HUB_IMAGE_CLASSIFICATION'].includes(modelData.coreParams.prediction_type);
    }
    function isClassification(modelData) {
        return isBinaryClassification(modelData) || isMulticlass(modelData);
    }
    function isRegression(modelData) {
        return modelData
            && modelData.coreParams
            && modelData.coreParams.prediction_type === 'REGRESSION';
    }
    function countOverrides(modelData) {
        if (!modelData || !modelData.overridesParams || !modelData.overridesParams.overrides) {
            return 0;
        }
        return modelData.overridesParams.overrides.length;
    }
    function getUncertaintySettings(modelData) {
        return modelData && modelData.coreParams && modelData.coreParams.uncertainty;
    }
    function hasOverrides(modelData) {
        return countOverrides(modelData) > 0;
    }
    function isEnsemble(modelData) {
        const algorithm = getAlgorithm(modelData);
        return algorithm && algorithm.endsWith('_ENSEMBLE');
    }
    function hasProbas(modelData) {
        const isProbaAware = modelData
            && modelData.iperf
            && modelData.iperf.probaAware;
        const isDeephubImageClassif = modelData
            && modelData.modeling
            && modelData.modeling.type
            && modelData.modeling.type === "DEEP_HUB_IMAGE_CLASSIFICATION";
        return !!(isProbaAware || isDeephubImageClassif);
    }
    function createThresholdUpdateChecker() {
        const ret = {
            _oldThreshold: null,
            executeIfUpdated: undefined
        };
        ret.executeIfUpdated = (modelData, callback) => {
            const threshold = modelData.userMeta.activeClassifierThreshold;
            if (threshold !== ret._oldThreshold) {
                ret._oldThreshold = threshold;
                callback();
            }
        };
        return ret;
    }
    function getPredictionType(modelData) {
        if (modelData &&
            modelData.coreParams &&
            modelData.coreParams.prediction_type) {
            return modelData.coreParams.prediction_type;
        }
    }
    function isPartitionedModel(modelData) {
         return modelData
             && modelData.coreParams
             && modelData.coreParams.partitionedModel
             && modelData.coreParams.partitionedModel.enabled;
     }
    function getTargetVariable(modelData) {
        if (modelData &&
            modelData.coreParams &&
            modelData.coreParams.target_variable) {
            return modelData.coreParams.target_variable;
        }
    }
    function isFromModelEvaluation(modelData) {
        return modelData && modelData.modelEvaluation;
    }
    function computeCmgForGivenPerf(modelData, perf) {
        if (!perf || !perf.perCutData) return;
        return perf.perCutData.cut.map((_, i) => {
            const fn = perf.perCutData.fn[i];
            const tn = perf.perCutData.tn[i];
            const fp = perf.perCutData.fp[i];
            const tp = perf.perCutData.tp[i];
            const weights = modelData.headTaskCMW;
            return (weights.fnGain * fn
                + weights.tnGain * tn
                + weights.fpGain * fp
                + weights.tpGain * tp) / (fn + tn + fp + tp);
        });
    }

    return {
        areMetricsWeighted,
        getAlgorithm,
        hasCalibration,
        isPartitionedBaseModel,
        isEnsemble,
        hasProbas,
        isBinaryClassification,
        isMulticlass,
        isClassification,
        isRegression,
        hasOverrides,
        countOverrides,
        getUncertaintySettings,
        createThresholdUpdateChecker,
        getPredictionType,
        isPartitionedModel,
        getTargetVariable,
        isFromModelEvaluation,
        computeCmgForGivenPerf,
    };
});

app.service("MLTaskDesignUtils", function() {
    function isBinaryClassification(mltaskDesign) {
        return mltaskDesign && mltaskDesign.predictionType === "BINARY_CLASSIFICATION";
    }

    function isMulticlass(mltaskDesign) {
        return mltaskDesign
            && ['MULTICLASS', 'DEEP_HUB_IMAGE_CLASSIFICATION'].includes(mltaskDesign.predictionType);
    }

    function isClassification(mltaskDesign) {
        return isBinaryClassification(mltaskDesign) || isMulticlass(mltaskDesign);
    }

    function getUncertaintySettings(mltaskDesign) {
        return mltaskDesign && mltaskDesign.uncertainty;
    }

    return {
        isBinaryClassification,
        isMulticlass,
        isClassification,
        getUncertaintySettings
    }
});

app.controller("_ModelUtilsController", function($scope, $q, ModelDataUtils, DataikuCloudService) {

    $scope.getPredictionType = () => ModelDataUtils.getPredictionType($scope.modelData);
    $scope.isPartitionedModel = () => ModelDataUtils.isPartitionedModel($scope.modelData);
    $scope.getTargetVariable =  () => ModelDataUtils.getTargetVariable($scope.modelData);
    $scope.getAlgorithm = () => ModelDataUtils.getAlgorithm($scope.modelData)

    $scope.getTreatmentVariable = function() {
        return $scope.modelData &&
            $scope.modelData.coreParams &&
            $scope.modelData.coreParams.treatment_variable;
    };

    $scope.fetchCloudInfo = () => {
        if ($scope.cloudInfo) return $q.when(null);
        else {
            return DataikuCloudService.getCloudInfo().then(cloudInfo => {
                $scope.cloudInfo = cloudInfo;
            });
        }
    };

    $scope.isNotStaticModelSkin = skin => !$scope.staticModelSkins.some(modelSkin => skin.id === modelSkin.id);
});

app.controller("_ModelReportControllerBase", function($scope, $rootScope, $controller, PluginsService) {

    $controller("_ModelUtilsController", {$scope:$scope});

    $scope.hooks = {};
    $scope.staticModelSkins = [];

    $scope.fetchCloudInfo().then(() => {
        $scope.addModelViewsLabel = 'Add views';
        $scope.pluginStoreLink = PluginsService.getPluginStoreLink($scope.cloudInfo);

        if ($scope.cloudInfo.isDataikuCloud) {
            $scope.addModelViewTooltipText = ($scope.cloudInfo.isSpaceAdmin ? 'You can install plugins on the Launchpad' : 'Ask your administrator to install plugins') + ' to have more model views';

            if (!$scope.cloudInfo.isSpaceAdmin) {
                $scope.addModelViewsLabel = 'See views';
            }
        } else {
            $scope.addModelViewTooltipText = `You can ${$rootScope.appConfig.admin ? 'install' : 'request' } plugins from the store to have more model views`;
        }
    });
});

app.controller('MLReportPreparationController', function($scope, Assert, ShakerProcessorsUtils) {
    Assert.inScope($scope, 'modelData');

    /*
     * Displaying info to user
     */
    var tmpFindGroupIndex = Array.prototype.indexOf.bind($scope.modelData.trainedWithScript.steps.filter(function(s) { return s.metaType === 'GROUP'; }));
    $scope.findGroupIndex = function(step) {
        const groupIndex = tmpFindGroupIndex(step);
        return groupIndex < 0 ? '' : groupIndex + 1;
    }
    $scope.getGroupName = function(step) {
        if (step.metaType == 'GROUP') {
            return step.name && step.name.length > 0 ? step.name : 'GROUP ' + $scope.findGroupIndex(step);
        } else {
            return "Invalid group";
        }
    }

    // compute recursive steps
    var stepsWithSubSteps = [];
    $scope.modelData.trainedWithScript.steps.forEach(function (step, index) {
        step["mainStep"] = true; // Normal step => No padding
        stepsWithSubSteps.push(step);
        if (step.metaType === "GROUP") {
            step.steps.forEach(function (subStep, index) {
                subStep["mainStep"] = false; // Step inside a group => need padding
                stepsWithSubSteps.push(subStep);
            });
        }

    });
    $scope.steps = stepsWithSubSteps;
    $scope.getStepDescription = ShakerProcessorsUtils.getStepDescription;
    $scope.getStepIcon        = ShakerProcessorsUtils.getStepIcon;

    // Signal to Puppeteer that the content of the element has been loaded and is thus available for content extraction
    $scope.puppeteerHook_elementContentLoaded = true;
});

app.controller("_MLReportSummaryController", function($scope) {
    $scope.editState = {
        editing : false,
    }
    $scope.userMeta = function() {
        if ($scope.evaluationDetails) {
            return $scope.evaluationDetails.evaluation.userMeta;
        } else {
            return $scope.modelData.userMeta;
        }
    }
    $scope.startEdit = function(){
        $scope.editState.editing = true;
        $scope.editState.name = $scope.userMeta().name;
        $scope.editState.description = $scope.userMeta().description;
    }
    $scope.cancelEdit = function(){
        $scope.editState.editing = false;
    }
    $scope.validateEdit = function() {
        $scope.userMeta().name = $scope.editState.name;
        $scope.userMeta().description = $scope.editState.description;
        $scope.editState.editing = false;
    }
});

app.controller("_ModelViewsController", function($scope, $rootScope, $state, $stateParams, $controller, DataikuAPI, PluginsService) {
    $scope.uiState = {};
    $scope.appConfig = $rootScope.appConfig;
    $scope.$state = $state;

    $scope.webAppConfig = {};
    $scope.webAppType = null;
    $scope.runningWebAppId = null;

    $scope.getBackendLogURL = function(projectKey, webAppId) {
        return DataikuAPI.webapps.getBackendLogURL(projectKey, webAppId);
    };

    if ($stateParams.analysisId) {
        $controller("TrainedModelSkinsController", {$scope});
    } else {
        $controller("SavedModelVersionSkinsController", {$scope});
    }

    if ($scope.modelViewsSkin) {
        // the ui state modification is handled in the skin controllers : TrainedModelSkinsController or SavedModelVersionSkinsController
        $scope.uiState.skin = $scope.modelViewsSkin;
    }

    $scope.fetchCloudInfo().then(() => {
        $scope.canAddModelView = !$scope.cloudInfo.isDataikuCloud || $scope.cloudInfo.isSpaceAdmin;
        $scope.pluginUrl = PluginsService.getPluginStoreLink($scope.cloudInfo, ($scope.uiState.skin || {}).ownerPluginId);
    });
});


app.controller("EvaluationLabelUtils", function($scope) {
    const DATASET_RE = /^[^:]*dataset:/i;
    const MODEL_RE = /^[^:]*model:/i;
    const EVALUATION_RE = /^[^:]*evaluation:/i;

    const DOMAIN_MAPPING = [
        {re: DATASET_RE, icon: "icon-dataset universe-color dataset"},
        {re: MODEL_RE, icon: "icon-machine_learning_regression universe-color saved_model"},
        {re: EVALUATION_RE, icon: "icon-model-evaluation-store universe-color saved_model"},
    ];

    $scope.setIcon = function(label) {
        if (label.key) {
            for (const mapping of DOMAIN_MAPPING) {
                if (mapping.re.test(label.key)) {
                    return mapping.icon;
                }
            }
        }
        return "icon-dku-edit";
    }
});

app.controller("MLBaseRouteReportController", function($scope) {
    $scope.uiState = $scope.uiState || {};
    // The skinId parameter is used for model views
    $scope.setSettingsPane = (pane, skinId) => {
        $scope.uiState.settingsPane = skinId ? pane + "-" + skinId : pane;
        $scope.uiState.skinId = skinId;
    }
});

app.controller("_PMLReportSummaryController", function($scope, $controller, DataikuAPI, PMLFilteringService, PerformanceMetricsDataComposer, MetricsUtils,
                                                       $stateParams, ActiveProjectKey, $filter, FullModelLikeIdUtils) {

    $controller("_MLReportSummaryController", {$scope:$scope});
    $controller("EvaluationLabelUtils", {$scope:$scope});
    
    const fullModelId = $stateParams.fullModelId || $scope.fullModelId;
    const predictionType = $scope.modelData.coreParams.prediction_type;
    
    var metricsInfo;
    switch (predictionType) {
        case "BINARY_CLASSIFICATION":
            metricsInfo = [
                ...PerformanceMetricsDataComposer.getThresholdIndependentBinaryClassifMetricsData($scope.modelData).rows,
                ...PerformanceMetricsDataComposer.getThresholdDependentBinaryClassifMetricsData($scope.modelData).rows
            ];
            break;
        case "MULTICLASS":
            metricsInfo = PerformanceMetricsDataComposer.getMulticlassMetricsData($scope.modelData).rows;
            break;
        case "REGRESSION":
            metricsInfo = PerformanceMetricsDataComposer.getRegressionMetricsData($scope.modelData).rows;
            break;
        case "TIMESERIES_FORECAST":
            metricsInfo = PerformanceMetricsDataComposer.getTimeSeriesMetricsModelEvaluationData($scope.modelData).rows;
            break;
        default:
            metricsInfo = null;
            break;
    }

    $scope.refreshCurrentMetricNames = function() {
        if ($scope.uiState.currentMetrics) {
            $scope.uiState.currentFormattedNames = $scope.uiState.currentMetrics
                .filter(m => m).map(cur => {
                return {
                    key: PMLFilteringService.metricMap[cur],
                    label: $scope.possibleMetrics.find(x => x[0] === cur)[1],
                    code: cur,
                    short_description: metricsInfo?.find(x => x['name'] === cur)?.info,
                    isEvaluationMetric: $scope.modelData.modeling.metrics.evaluationMetric === cur ? true : false,
                    isThresholdOptimizationMetric: $scope.modelData.modeling.metrics.thresholdOptimizationMetric === cur ? true : false
                };
            });
        } else {
            $scope.uiState.currentFormattedNames = [];
        }
        $scope.refreshMetricsValues();
    }

    $scope.refreshMetricsValues = function() {
        let metrics = ($scope.evaluationDetails&&$scope.evaluationDetails.metrics)?$scope.evaluationDetails.metrics:null;
        $scope.uiState.$formattedMetrics = {};
        if (!metrics || !Object.keys(metrics).length) {
            $scope.uiState.noperf = true;
            return;
        }
        $scope.uiState.noperf = false;

        for (let metricCode of $scope.uiState.currentMetrics) {
            $scope.uiState.$formattedMetrics[metricCode] = MetricsUtils.getMetricValue(metrics, metricCode, 3);
        }
    }

    $scope.getMetricValueFromModel = PMLFilteringService.getMetricValueFromModel.bind(PMLFilteringService);


    if ($scope.versionsContext) {
        $scope.$watch("versionsContext.activeMetric", function() {
            $scope.activeMetric = $scope.versionsContext.activeMetric;
        });
        $scope.activeMetric = $scope.versionsContext.activeMetric;
    } else if ($scope.mlTasksContext) {
        $scope.$watch("mlTasksContext.activeMetric", function() {
            $scope.activeMetric = $scope.mlTasksContext.activeMetric;
        });
        $scope.activeMetric = $scope.mlTasksContext.activeMetric;
    }
 })


app.controller("PMLReportSummaryController", function($scope, $controller, SmartId, StateUtils, Debounce, TimeseriesForecastingUtils,
                                                        PMLSettings, Assert, $stateParams, $state, PMLFilteringService, MetricsUtils, worstPrefixName) {
    $controller("_PMLReportSummaryController", {$scope:$scope});
    $scope.FilteringService = PMLFilteringService;
    $scope.SmartId = SmartId;
    $scope.StateUtils = StateUtils;
    $scope.prettyTimeSteps = TimeseriesForecastingUtils.prettyTimeSteps;

    $scope.$watch("modelData", () => {
        if($scope.evaluationDetails && $scope.evaluationDetails.evaluation) {
            $scope.refreshMetrics($scope.evaluationDetails.evaluation.predictionType);
        } else if ($scope.modelData) {
            $scope.refreshMetrics($scope.modelData.coreParams.prediction_type);
        }
        // Signal to Puppeteer that the content of the element has been loaded and is thus available for content extraction
        $scope.puppeteerHook_elementContentLoaded = true;
    });

    $scope.$watch("currentPartitionedModel", () => {
        if($scope.currentPartitionedModel) {
            $scope.possibleCustomMetrics = $scope.FilteringService.getPossibleCustomMetrics([$scope.currentPartitionedModel]);
            if ($scope.uiState.display) {
                // metrics used in subpopulation chart
                $scope.uiState.display.customMetrics = $scope.possibleCustomMetrics.map(item => {
                    item.displayed = false;
                    item.getMetricFromPerf = function(perf) {
                        const foundValues = perf.singleMetrics.customMetricsResults.filter(result => result.metric.name === item.name);
                        if (foundValues) {
                            return foundValues[0].value;
                        } else {
                            return null;
                        }
                    }
                    return item;
                });
            }
        }
    })

    $scope.refreshMetrics = function(predictionType) {
        $scope.possibleMetrics = MetricsUtils.getDefaultMesMetrics(predictionType, $scope.evaluationDetails) || [];
        if ($scope.uiState.currentMetric && $scope.possibleMetrics.filter(_ => _[0] == $scope.uiState.currentMetric).length == 0) {
            // old selected metric isn't possible anymore
            $scope.uiState.currentMetric = null;
        }
        if ($scope.uiState.currentMetric == null) {
            if ('BINARY_CLASSIFICATION' === predictionType) {
                $scope.uiState.currentMetric = 'ROC_AUC';
            }
            if ('MULTICLASS' === predictionType) {
                $scope.uiState.currentMetric = 'ROC_AUC';
            }
            if ('REGRESSION' === predictionType) {
                $scope.uiState.currentMetric = 'R2';
            }
            if ('TIMESERIES_FORECAST' === predictionType) {
                $scope.uiState.currentMetric = 'MASE';
            }
        }
        $scope.uiState.currentMetrics = $scope.possibleMetrics.map(pm => pm[0]).filter(x => x);
        if ($scope.modelData && $scope.modelData.modelEvaluation){
            if (!$scope.modelData.modelEvaluation.hasModel){
                $scope.uiState.currentMetrics = $scope.uiState.currentMetrics.filter(PMLFilteringService.isDataDriftMetric)
            }
            else if ($scope.modelData.modelEvaluation.evaluateRecipeParams && $scope.modelData.modelEvaluation.evaluateRecipeParams.dontComputePerformance){
                $scope.uiState.currentMetrics = $scope.uiState.currentMetrics.filter(PMLFilteringService.isDriftMetric)
            }
        }
        $scope.refreshCurrentMetricNames();
    }

    $scope.getSimpleCustomMetricsResults = function() {
        const customMetricsResults = [];

        if($scope.evaluationDetails && $scope.evaluationDetails.metrics && $scope.evaluationDetails.metrics.customMetricsResults) {
            $scope.evaluationDetails.metrics.customMetricsResults.forEach(item => {
                customMetricsResults.push({
                    name:item.metric.name,
                    value:item.value,
                    description:item.metric.description,
                    displayName:item.metric.name,
                    code:item.metric.metricCode,
                    isEvaluationMetric: $scope.modelData.modeling.metrics.customEvaluationMetricName === item.metric.name ? true : false,
                });
                if (item.worstValue) {
                    customMetricsResults.push({
                        name:worstPrefixName + item.metric.name,
                        value:item.worstValue,
                        description: `Worst value of ${item.metric.name} across all time series.`,
                        displayName:worstPrefixName + item.metric.name,
                        code:item.metric.metricCode,
                        isEvaluationMetric: false
                    });
                }
            });
        }
        return customMetricsResults;
    }

    $scope.getGoToExperimentRun = function() {
        Assert.trueish($scope.modelData, 'no model data');
        Assert.trueish($scope.modelData.mlflowOrigin, 'no MLflow origin');
        Assert.trueish($scope.modelData.mlflowOrigin.type === 'EXPERIMENT_TRACKING_RUN', 'MLflow origin not from experiment/run');
        Assert.trueish($scope.modelData.mlflowOrigin.runId, 'MLflow origin malformed (no runId)');

        const origin = $scope.modelData.mlflowOrigin;
        const params = {
            projectKey: $stateParams.projectKey,
            runId: origin.runId
        };
        return $state.href("projects.project.experiment-tracking.run-details", params);
    }

    $scope.getGoToExperimentRunArtifact = function() {
        Assert.trueish($scope.modelData, 'no model data');
        Assert.trueish($scope.modelData.mlflowOrigin, 'no MLflow origin');
        Assert.trueish($scope.modelData.mlflowOrigin.type === 'EXPERIMENT_TRACKING_RUN', 'MLflow origin not from experiment/run');
        Assert.trueish($scope.modelData.mlflowOrigin.runId, 'MLflow origin malformed (no runId)');
        Assert.trueish($scope.modelData.mlflowOrigin.modelSubfolder, 'MLflow origin malformed (no modelSubfolder)');

        const origin = $scope.modelData.mlflowOrigin;
        const params = {
            projectKey: $stateParams.projectKey,
            runId: origin.runId,
            subfolder: origin.modelSubfolder
        };
        return $state.href("projects.project.experiment-tracking.run-artifacts", params);
    }

    $scope.getGoToExperiment = function() {
        Assert.trueish($scope.modelData, 'no model data');
        Assert.trueish($scope.modelData.mlflowOrigin, 'no MLflow origin');
        Assert.trueish($scope.modelData.mlflowOrigin.type === 'EXPERIMENT_TRACKING_RUN', 'MLflow origin not from experiment/run');
        Assert.trueish($scope.modelData.mlflowOrigin.experimentId, 'MLflow origin malformed (no experimentId)');

        const origin = $scope.modelData.mlflowOrigin;
        const params = {
            projectKey: $stateParams.projectKey,
            experimentIds: origin.experimentId
        };
        return $state.href("projects.project.experiment-tracking.runs-list", params);
    }
});

app.component("displayJsonModal", {
    bindings: {
        modalControl: '<',
        title: '<',
        json: '<'
    },
    template: `
    <div>
        <div dku-modal-header-with-totem modal-title="{{$ctrl.title}}"/>
        <form name="displayJson" class="dkuform-modal-horizontal dkuform-modal-wrapper">
            <div class="modal-body">
                <pre>{{$ctrl.json|prettyjson}}</pre>
            </div>
            <div class="modal-footer">
            <button type="button" class="btn btn--secondary" ng-click="$ctrl.copyToClipboard()">Copy</button>
            <button type="button" class="btn btn--primary" ng-click="$ctrl.modalControl.dismiss()">Close</button>
            </div>
        </form>
    </div>`,
    controller: function(ClipboardUtils) {
        const $ctrl = this;
        $ctrl.copyToClipboard = function() {
            ClipboardUtils.copyToClipboard(JSON.stringify($ctrl.json, null, 4));
            $ctrl.modalControl.dismiss();
        }
    }
});

app.component("displayTextModal", {
    bindings: {
        modalControl: '<',
        title: '<',
        text: '<'
    },
    template: `
    <div>
        <div dku-modal-header-with-totem modal-title="{{$ctrl.title}}"/>
        <form name="displayText" class="dkuform-modal-horizontal dkuform-modal-wrapper">
            <div class="modal-body">
                <pre>{{$ctrl.text}}</pre>
            </div>
            <div class="modal-footer">
            <button type="button" class="btn btn--secondary" ng-click="$ctrl.copyToClipboard()">Copy</button>
            <button type="button" class="btn btn--primary" ng-click="$ctrl.modalControl.dismiss()">Close</button>
            </div>
        </form>
    </div>`,
    controller: function(ClipboardUtils) {
        const $ctrl = this;
        $ctrl.copyToClipboard = function() {
            ClipboardUtils.copyToClipboard($ctrl.text);
            $ctrl.modalControl.dismiss();
        }
    }
});


app.component("showEllipsedTextAndCopy", {
    bindings: {
        text: '<',
        tooltip: '<?',
        maxWidthPx: '<?'
    },
    template: `
    <div show-tooltip-on-text-overflow
        text-tooltip="$ctrl.tooltip?$ctrl.tooltip:$ctrl.text" style="display:inline-block; max-width: {{$ctrl.maxWidthPx?$ctrl.maxWidthPx:330}}px;">
    </div>
    <a ng-click="$ctrl.copyToClipboard()"><i class="dku-icon-copy-16"/></a>`,
    controller: function(ClipboardUtils) {
        const $ctrl = this;
        $ctrl.copyToClipboard = function() {
            ClipboardUtils.copyToClipboard($ctrl.text);
        }
    }

});

app.component("externalEndpointInfo", {
    bindings: {
        fullModelId: '<',
        proxyModelConfiguration: '<',
        proxyModelEndpointInfo: '<'
    },
    templateUrl: '/templates/ml/prediction-model/external_endpoint_info.html',
    controller: function(CreateModalFromComponent, displayJsonModalDirective, ClipboardUtils, FullModelLikeIdUtils, RemoteResourcesLinksUtils) {
        const $ctrl = this;
        $ctrl.uiState = {
            parsedFMI: null
        }

        $ctrl.showOpenAPIJson = function() {
            CreateModalFromComponent(displayJsonModalDirective, {
                title: "OpenAPI definition retrieved at version creation",
                json: JSON.parse($ctrl.proxyModelEndpointInfo.openAPI)
            }, ['modal-wide']);
        }

        $ctrl.showInputParameters = function() {
            CreateModalFromComponent(displayJsonModalDirective, {
                title: "Input parameters",
                json: $ctrl.proxyModelEndpointInfo.input
            }, ['modal-wide']);
        }

        $ctrl.$onChanges = function() {
            if ($ctrl.proxyModelConfiguration && $ctrl.proxyModelConfiguration.protocol === 'vertex-ai') {
                $ctrl.deploymentCount = $ctrl.proxyModelEndpointInfo.models.length;
            } else if ($ctrl.proxyModelConfiguration && $ctrl.proxyModelConfiguration.protocol === 'azure-ml'
                && $ctrl.proxyModelEndpointInfo && $ctrl.proxyModelEndpointInfo.modelByDeployment) {
                $ctrl.deploymentCount = Object.keys($ctrl.proxyModelEndpointInfo.modelByDeployment).length;
            } else {
                $ctrl.deploymentCount = 0;
            }
            if ($ctrl.fullModelId) {
                $ctrl.uiState.parsedFMI = FullModelLikeIdUtils.parse($ctrl.fullModelId);
            } else {
                $ctrl.uiState.parsedFMI = null;
            }
        }

        $ctrl.copyInputToClipboard = function() {
            ClipboardUtils.copyToClipboard(JSON.stringify($ctrl.proxyModelEndpointInfo.input, undefined, 4));
        }

        $ctrl.extractVertexAIEndpointId = function(fullEndpointName) {
            return fullEndpointName.split('/').pop();
        }

        $ctrl.tryGetResourceLink = function () {
            try {
                return $ctrl.getResourceLink();
            } catch (e) {
                return null;
            }
        }

        $ctrl.getResourceLink = function () {
            if ($ctrl.proxyModelConfiguration && $ctrl.proxyModelConfiguration.protocol === "sagemaker") {
                return RemoteResourcesLinksUtils.getSageMakerResourceLink(
                    "endpoints", $ctrl.proxyModelConfiguration.region, $ctrl.proxyModelEndpointInfo.endpointName
                );
            }
            if ($ctrl.proxyModelConfiguration && $ctrl.proxyModelConfiguration.protocol === "vertex-ai") {
                return RemoteResourcesLinksUtils.getVertexAIResourceLink(
                    "endpoints",
                    $ctrl.proxyModelConfiguration.project_id,
                    $ctrl.proxyModelConfiguration.region,
                    $ctrl.extractVertexAIEndpointId($ctrl.proxyModelEndpointInfo.name)
                );
            }
            if ($ctrl.proxyModelConfiguration && $ctrl.proxyModelConfiguration.protocol === "azure-ml") {
                const endpointInfo = {
                    azWorkspace: $ctrl.proxyModelEndpointInfo.endpointInfo.workspace,
                    azResourceGroup: $ctrl.proxyModelEndpointInfo.endpointInfo.resourceGroup,
                    azSubscription: $ctrl.proxyModelEndpointInfo.endpointInfo.subscription,
                    azTenantId: $ctrl.proxyModelEndpointInfo.retrievedByTenantId,
                    azOnlineEndpointName: $ctrl.proxyModelEndpointInfo.endpointInfo.name
                }
                return RemoteResourcesLinksUtils.getAzureMLOnlineEndpointLink(endpointInfo);
            }
            return null;
        }
    }
});

app.controller("PartPMLReportSummaryController", function($scope, $controller, $stateParams, DataikuAPI) {
    $controller("PMLReportSummaryController", {$scope: $scope});
    $controller('_SubpopTableUtilsController', {$scope: $scope});

    $scope.dimensionsList = function() {
        return $scope.modelData.coreParams.partitionedModel.dimensionNames
            .map(dim => `<b>${sanitize(dim)}</b>`)
            .join(' and ');
    };

    $scope.getCurrentFeatureData = () => {
        return $scope.partitionsPerf;
    };

    const mergeSnippetsWithModalities = (data, snippets) => {
        data.modalities.forEach((modality) => {
            let snippet;
            if (modality.value && modality.value in snippets.partitions.summaries) {
                snippet = snippets.partitions.summaries[modality.value].snippet;
                // 'status' corresponds:
                //  * for ANALYSIS, to trainInfo.state (ModelTrainState)
                //  * for SAVED, to trainInfo.state if partition was trained or REUSED_... if partition was reused (PartitionState)
                snippet["status"] = snippets.partitions.summaries[modality.value].state;
                modality.snippet= snippet;
            }
        });

        data.allDatasetModality.snippet = snippets;
    };

    const getThreshold = (snippets) => (mod) => {
        if (mod.value && mod.value in snippets.partitions.summaries) {
            const partSnippet = snippets.partitions.summaries[mod.value].snippet;
            if (partSnippet && partSnippet.userMeta) {
                return partSnippet.userMeta.activeClassifierThreshold;
            }
        }

        return snippets.baseModel;
    };

    const preparePartitionsPerf = (partSnippets) => {
        DataikuAPI.ml.prediction.getPartitionsPerf(partSnippets.fullModelId)
            .then(resp => {
                $scope.formatTableResults(resp.data, resp.data.allDatasetPerf, getThreshold(partSnippets));
                mergeSnippetsWithModalities(resp.data, partSnippets);

                $scope.partitionsPerf = resp.data;
            }, setErrorInScope.bind($scope));
    };

    $scope.$watch('partitionedModelSnippets', function(partSnippets) {
        if (!partSnippets) {
            return;
        }

        preparePartitionsPerf(partSnippets);
    });
});

app.controller("PMLReportDriftController", function($scope, $controller, $stateParams, DataikuAPI, ActiveProjectKey,
                                                    FutureProgressModal, ModelEvaluationUtils) {
    $scope.fullModelId = $stateParams.fullModelId || $scope.fullModelId;
    if ($stateParams.mesId) {
        $scope.mesId = $stateParams.mesId;
        $scope.evaluationId = $stateParams.evaluationId;
    }
    $scope.uiState = $scope.uiState || {
        driftState: {
            selectedReference: null
        },
        driftParams: angular.copy(ModelEvaluationUtils.defaultDriftParams),
        refProbabilityDensities: null,
        pdd: null,
        currentClass: null,
        refPredValueCount: null,
        curPredValueCount: null,
        pdfs: null
    };

    $scope.wt1Properties = ModelEvaluationUtils.wt1Properties;

    $scope.colors = window.dkuColorPalettes.discrete[0].colors.filter((_,idx) => idx%2 === 0);

    $scope.isPvalueRejected = function(pvalue) {
        const confidenceLevel = $scope.uiState.driftState.driftParamsOfResult && $scope.uiState.driftState.driftParamsOfResult.confidenceLevel;
        if(confidenceLevel == null) {
            return false;
        }
        const significanceLevel = 1 - confidenceLevel;
        if (pvalue != null) {
            return pvalue <= significanceLevel;
        }
        return false;
    }

    $scope.hasNewValues = function(newValuesPercentage) {
        return newValuesPercentage && newValuesPercentage !== 0;
    }

    $scope.isPSIAboveThreshold = function(psi) {
        const threshold = $scope.uiState.driftState.driftParamsOfResult && $scope.uiState.driftState.driftParamsOfResult.psiThreshold;
        if ((threshold != null) && (psi != null)) {
            return psi > threshold;
        }
        return false;
    }

    $scope.showUnivariateMetric = function(metric) {
        if (metric === 'KS') {
            return $scope.displayParams.predictionType === 'REGRESSION';
        }
        if (metric === 'Chi-square') {
            return ['BINARY_CLASSIFICATION', 'MULTICLASS'].includes($scope.displayParams.predictionType);
        }
    }

    $scope.isDriftDetected = function() {
        return $scope.uiState.driftState.driftResult
            && $scope.uiState.driftState.driftResult.driftModelResult.driftModelAccuracy.pvalue <= (1 - $scope.uiState.driftParams.confidenceLevel);
    }


    DataikuAPI.modelevaluationstores.listWithAccessible(ActiveProjectKey.get()).success(function(data){
        $scope.storeList = data;
    }).error(setErrorInScope.bind($scope));

    DataikuAPI.savedmodels.listWithAccessible($stateParams.projectKey).success(function(data){
        $scope.modelList = data;
    }).error(setErrorInScope.bind($scope));

    if (!$scope.readOnly) {
        DataikuAPI.analysis.listHeads($stateParams.projectKey, true).success(function(data) {
            $scope.analysesWithHeads = data;
        }).error(setErrorInScope.bind($scope));
    }

    $scope.getColumnParams = function(columnName) {
        const params = $scope.uiState.driftParams.columns[columnName];
        return params ? params : { handling: 'AUTO', enabled: true };
    }

    $scope.changeColumnEnabled = function(columnName, enabled) {
        const previousColumnParams = $scope.getColumnParams(columnName);
        const newColumnParams = { ... previousColumnParams, enabled };

        $scope.changeColumnParams(columnName, newColumnParams);
    }

    $scope.changeColumnParams = function(columnName, newColumnParams) {
        $scope.uiState.driftParams.columns[columnName] = newColumnParams;
        $scope.computeDriftAndGenerateTabs({from: $scope.wt1Properties.INPUT_DATA_DRIFT_TYPE, driftType: $scope.wt1Properties.CHANGE_COLUMN_PARAMS});
    }

    $scope.changeColumnHandling = function(columnName, handling) {
        const previousColumnParams = $scope.getColumnParams(columnName);
        const newColumnParams = { ... previousColumnParams, handling };

        $scope.changeColumnParams(columnName, newColumnParams);
    }

    $scope.driftingColumns = function() {
        let driftingColumns = [];
        for (const [columnName, columnDrift] of Object.entries($scope.uiState.driftState.driftResult.univariateDriftResult.columns)) {
            if($scope.isPvalueRejected(columnDrift.chiSquareTestPvalue)
                || $scope.isPvalueRejected(columnDrift.ksTestPvalue)
                || $scope.isPSIAboveThreshold(columnDrift.populationStabilityIndex)) {
                    driftingColumns.push(columnName);
                }
        }
        return driftingColumns;
    }

    $scope.sortUnivariateValue = function(sortColumn) {
        return columnSettings => {
            const univariateResult = $scope.uiState.driftState.driftResult.univariateDriftResult.columns[columnSettings.name];
            if(univariateResult && univariateResult[sortColumn] != null) {
                return univariateResult[sortColumn];
            }
            return columnSettings[sortColumn];
        };
    }

    $scope.$watch('uiState.driftState.currentClass', function(nv, ov) {
        if (!angular.equals(nv,ov)) {
            $scope.computePddForClass(nv);
        }
    });

    $scope.perfDriftLabel = function(item) {
        return item.perfDriftLabel;
    }

    $scope.displayParams = null;
    $scope.cloneDisplayParams = function() {
        $scope.displayParams = angular.copy($scope.modelEvaluationStore.displayParams);
    }

    $scope.$watch('modelEvaluationStore.displayParams', $scope.cloneDisplayParams);


});

/**
 * @ngdoc component
 * @name embeddingDriftTable
 * @description
 * A component that displays a table of embedding drift metrics: Euclidian distance, Cosine similarity, Classifier Gini.
 *
 * @property {string} embeddingType - The type of embedding being displayed ('IMAGE' or 'TEXT').
 * @property {string} modelRefId - The ID of the reference model.
 * @property {string} modelEvaluationCurId - The ID of the current model evaluation.
 * @property {string} modelEvaluationRefId - The ID of the reference model evaluation.
 * @property {object} generalDriftResult - The main drift result object containing all metrics.
 * @property {object} embeddingDriftResult - The specific drift result object for embeddings.
 * @property {boolean} hasDrift - A flag indicating if drift was selected.
 */
app.component("embeddingDriftTable", {
    bindings: {
        embeddingType: '<',
        modelRefId: '<',
        modelEvaluationCurId: '<',
        modelEvaluationRefId: '<',
        generalDriftResult: '<',
        embeddingDriftResult: '<',
        hasDrift: '<'
    },
    templateUrl:  "/templates/ml/prediction-model/fragments/input_data_drift_embedding_drift_table.html",
    controller: function() {
        const $ctrl = this;
        $ctrl.sortEmbeddingValue = function(sortColumn, driftResultValue) {
            return columnSettings => {
                if (driftResultValue) {
                    const embeddingResult = driftResultValue.columns[columnSettings.name];
                    if (embeddingResult && embeddingResult[sortColumn] != null) {
                        return embeddingResult[sortColumn];
                    }
                }
                return columnSettings[sortColumn];
            };
        }
    }
});

/**
 * @ngdoc component
 * @name imageQualityDrift
 * @description
 * A component responsible for displaying the results of an image quality drift analysis.
 *
 * @property {string} modelRefId - The ID of the reference model.
 * @property {string} modelEvaluationCurId - The ID of the current model evaluation.
 * @property {string} modelEvaluationRefId - The ID of the reference model evaluation.
 * @property {object} generalDriftResult - The main drift result object.
 * @property {object} embeddingDriftResult - The drift result object for embeddings.
 * @property {boolean} hasDrift - A flag indicating if image drift was selected.
 * @property {number} confidenceLevel - The confidence level (e.g., 0.95) for statistical tests.
 * @property {object} colors - A list of string (color references like "#000000") for UI elements.
 */
app.component("imageQualityDrift", {
    bindings: {
        modelRefId: '<',
        modelEvaluationCurId: '<',
        modelEvaluationRefId: '<',
        generalDriftResult: '<',
        embeddingDriftResult: '<',
        hasDrift: '<',
        confidenceLevel: '<',
        colors: '<',
    },
    templateUrl:  "/templates/ml/prediction-model/fragments/input_data_drift_image_quality_drift.html",
    controller: function() {
        const $ctrl = this;

        $ctrl.sortImageQualityMetricsValue = function(sortColumn, driftResultValue) {
            return columnSettings => {
                if (sortColumn === "name")
                    return $ctrl.imageQualityDriftNameMap[driftResultValue[columnSettings][sortColumn]];
                return columnSettings[sortColumn];
            };
        }

        $ctrl.isPvalueRejected = function(pvalue) {
            if($ctrl.confidenceLevel == null) {return false;}
            const significanceLevel = 1 - $ctrl.confidenceLevel;
            if (pvalue != null) {
                return pvalue <= significanceLevel;
            }
            return false;
        }

        $ctrl.driftingColumns = function(columns) {
            let driftingColumns = [];
            for (const [columnName, columnDrift] of Object.entries(columns)) {
                if($ctrl.isPvalueRejected(columnDrift.ksTestPvalue)) {
                        driftingColumns.push(columnName);
                    }
            }
            return driftingColumns;
        }

        $ctrl.imageQualityDriftNameMap = {
            'meanRed': 'Mean Red (R)',
            'meanGreen': 'Mean Green (G)',
            'meanBlue': 'Mean Blue (B)',
            'meanSaturation': 'Mean Saturation',
            'rmsContrast': 'Contrast (RMS)',
            'laplacianVar': 'Sharpness (Laplacian)',
            'tenengrad': 'Sharpness (Tenengrad)',
            'entropy': 'Entropy (Complexity)',
            'edgeDensity': 'Edge Density',
            'area': 'Area',
            'aspectRatio': 'Aspect Ratio'
        };

        $ctrl.imageQualityDriftDescriptionMap = {
            'meanRed': 'Measures the average intensity of the red color channel',
            'meanGreen': 'Measures the average intensity of the green color channel',
            'meanBlue': 'Measures the average intensity of the blue color channel',
            'meanSaturation': 'Measures color intensity to detect if images are washed-out or vibrant',
            'rmsContrast': 'Measures the overall contrast to see if images are becoming flat or harsh',
            'laplacianVar': 'Measures sharpness to identify blurry or out-of-focus images',
            'tenengrad': 'Measures sharpness by quantifying the strength of edges in the image',
            'entropy': 'Measures the amount of texture and detail to track overall image complexity',
            'edgeDensity': 'Measures the density of edges, which relates to object presence and focus',
            'area': 'Measures the pixel size of the image',
            'aspectRatio': 'Measures the shape of the image to detect stretching or cropping'
        };

        $ctrl.imageQualityDriftCategoryMap = {
            'Colorimetry': ['meanRed', 'meanGreen', 'meanBlue', 'meanSaturation'],
            'Sharpness And Contrast': ['rmsContrast', 'laplacianVar', 'tenengrad'],
            'Texture And Geometry': ['entropy', 'edgeDensity', 'area', 'aspectRatio'],
        };
    }
});

/**
 * @ngdoc component
 * @name imageDriftGraph
 *
 * @description
 * Display a distribution graph and the most relevant images for each dataset.
 *
 * @property {string} modelRefId. The ID of the reference model.
 * @property {string} modelEvaluationCurId. The ID of the current model evaluation.
 * @property {string} modelEvaluationRefId. The ID of the reference model evaluation.
 * @property {DriftResult} generalDriftResult. Object containing general drift results.
 * @property {ImageDriftResult} embeddingDriftResult. Main data object. Contains drift metrics for image columns.
 * @property {boolean} hasDrift. A boolean that controls whether drift analysis is enabled.
 * @property {string[]} colors. An array of hexadecimal color strings for the graph.
 */
app.component("imageDriftGraph", {
    bindings: {
        modelRefId: '<',
        modelEvaluationCurId: '<',
        modelEvaluationRefId: '<',
        generalDriftResult: '<',
        embeddingDriftResult: '<',
        hasDrift: '<',
        colors: '<',
    },
    templateUrl:  "/templates/ml/prediction-model/fragments/input_data_drift_image_drift_graph.html",
    controller: function() {
        const $ctrl = this;

        $ctrl.$onChanges = function(changes) {
            const currentEmbeddingDriftResultColumns = changes.embeddingDriftResult?.currentValue?.columns;
            if (!currentEmbeddingDriftResultColumns) {
                return;
            }

            $ctrl.pdfs = {};
            $ctrl.imageSettingsRef = {};
            $ctrl.imageSettingsCur = {};
            $ctrl.resultsRef = {};
            $ctrl.resultsCur = {};
            $ctrl.combinedColors = [...$ctrl.colors.slice(0, 2), ...$ctrl.colors.slice(0, 2)];

            for (const [columnName, columnMetrics] of Object.entries(currentEmbeddingDriftResultColumns)) {
                if (!columnMetrics) return;
                setupGraphForColumn(columnMetrics, columnName)
                setupImageCardsForColumn(columnMetrics, columnName)
            }
        }

        function setupGraphForColumn(columnMetrics, columnName){
            const xs = [];
            const ys = [];
            const labels = [];

            xs.push(columnMetrics.curPredictionInfos.x);
            ys.push(columnMetrics.curPredictionInfos.pdf);
            labels.push('Current');

            xs.push(columnMetrics.refPredictionInfos.x);
            ys.push(columnMetrics.refPredictionInfos.pdf);
            labels.push('Reference');

            $ctrl.pdfs[columnName] = {
                xs: xs,
                ys: ys,
                colors: $ctrl.colors.slice(0, 2),
                labels: labels
            };
        }

        function setupImageCardsForColumn(columnMetrics, columnName){
            $ctrl.imageSettingsRef[columnName] = {
                "managedFolderSmartName" : columnMetrics.refPredictionInfos.managedFolderSmartName
            }

            $ctrl.imageSettingsCur[columnName] = {
                "managedFolderSmartName" : columnMetrics.curPredictionInfos.managedFolderSmartName
            }

            $ctrl.resultsRef[columnName] = getResultsForImageCards(columnMetrics.refPredictionInfos.top5, true);
            $ctrl.resultsCur[columnName] = getResultsForImageCards(columnMetrics.curPredictionInfos.top5, false);
            $ctrl.xMarks = [$ctrl.resultsCur[columnName]['predictions'].at(-1),
                            $ctrl.resultsRef[columnName]['predictions'].at(-1)];
            $ctrl.explanations = {};
            $ctrl.explanations['index']= [0,1,2,3,4];
        }

        function getResultsForImageCards(predictionInfos, isAscending){
            predictionInfos.sort((a, b) => {
                if (isAscending) {
                    return a.predictions - b.predictions;
                } else {
                    return b.predictions - a.predictions;
                }
            });

            const predictionInfosArrays = predictionInfos.reduce((accumulator, currentObject) => {
                for (const key in currentObject) {
                    if (!accumulator[key]) {accumulator[key] = [];}
                    accumulator[key].push(currentObject[key]);
                }
                return accumulator;
            }, {});

            const { predictions, ...observations } = predictionInfosArrays;
            return {'observations': observations, 'predictions':predictions};
        }

        $ctrl.evaluatedImageColumnName = "Most Drifted Evaluated Images";
        $ctrl.referenceImageColumnName = "Most Normal Reference Images";
    }
});

app.controller("ExportModelController", function($scope, DataikuAPI, FutureProgressModal, ActivityIndicator, WT1, MLExportService, FutureWatcher, LocalStorage) {

    $scope.mayExportModel = (type) => MLExportService.mayExportModel($scope.model, type);

    $scope.exportParams = {
        "snowflake": {functionName : ""},
        "jar": {functionName : "com.company.project.Model", type: "jar-fat"},
        "mlflow": {modelName: null, type: "zip", useUnityCatalog: true, useOriginalMLflowModel: false}
    }

    $scope.uiState = {
        ...$scope.uiState,
        fromDatabricks: {
            loadingModelList: false,
            loadingExperimentList: false
        },
        collapsedInfo: LocalStorage.get('dss.MLExportService.exportExplanation.shown') || false
    };

    $scope.resetDatabricksModel = function() {
        $scope.uiState.fromDatabricks.loadingModelList = false;
        $scope.uiState.fromDatabricks.models = null;
        $scope.exportParams.mlflow.modelName = null;
    }

    $scope.resetDatabricksParams = function() {
        $scope.resetDatabricksModel();
        $scope.uiState.fromDatabricks.loadingExperimentList = false;
        $scope.uiState.fromDatabricks.experiments = null;
        $scope.exportParams.mlflow.experimentName = null;
    }

    $scope.selectExportType = function(type){
        $scope.exportType = type;
    }

    $scope.expandOrCollapseInfo = function() {
        $scope.uiState.collapsedInfo = !$scope.uiState.collapsedInfo;
        LocalStorage.set('dss.MLExportService.exportExplanation.shown', $scope.uiState.collapsedInfo);
    };

    $scope.export = function() {
        const model = $scope.model;
        const type = $scope.exportType;
        switch($scope.exportType) {

            case "snowflake":
                DataikuAPI.ml.prediction.exportToSnowflakeFunction($scope.exportParams.snowflake.connectionName,
                            model.fullModelId, $scope.exportParams.snowflake.functionName).success(function(data){
                    FutureProgressModal.show($scope, data, "Exporting to Snowflake").then(() => {
                        ActivityIndicator.success("Successfully exported to Snowflake function")
                        $scope.dismiss();
                    });
                }).error(setErrorInScope.bind($scope));
                break;

            case "jar":
                if ($scope.exportParams.jar.type == 'jar-thin' || $scope.exportParams.jar.type == 'jar-fat'){
                    MLExportService.downloadFile($scope, () => DataikuAPI.ml.prediction.createScoringModelFile($scope.exportParams.jar.type, model.fullModelId, "&fullClassName=" + encodeURIComponent($scope.exportParams.jar.functionName)),
                        (exportId) => DataikuAPI.ml.prediction.getScoringModelDownloadURL($scope.exportParams.jar.type, exportId));
                } else {
                    MLExportService.downloadFile($scope, () => DataikuAPI.ml.prediction.createScoringModelFile($scope.exportParams.jar.type, model.fullModelId),
                     (exportId) => DataikuAPI.ml.prediction.getScoringModelDownloadURL($scope.exportParams.jar.type, exportId));
                }
                break;

            case "mlflow":
                MLExportService.downloadFile($scope, () => DataikuAPI.ml.prediction.createScoringModelFile("python", model.fullModelId, `&exportMlflow=true&useOriginalMLflowModel=${$scope.exportParams.mlflow.useOriginalMLflowModel}`),
                    (exportId) => DataikuAPI.ml.prediction.getScoringModelDownloadURL("python", exportId));
                break;

            case "databricks":
                DataikuAPI.ml.prediction.exportToDatabricksRegistry($scope.exportParams.mlflow.connectionName,
                    model.fullModelId, $scope.exportParams.mlflow.useUnityCatalog, $scope.exportParams.mlflow.modelName, $scope.exportParams.mlflow.experimentName).success(function(data){
                    FutureProgressModal.show($scope, data, "Exporting to Databricks Registry").then(() => {
                        ActivityIndicator.success("Successfully exported " + $scope.exportParams.mlflow.modelName + " to Databricks Registry")
                        $scope.dismiss();
                    });
                }).error(setErrorInScope.bind($scope));
                break;

            case "python":
                MLExportService.downloadFile($scope, () => DataikuAPI.ml.prediction.createScoringModelFile("python", model.fullModelId, "&exportMlflow=false"),
                    (exportId) => DataikuAPI.ml.prediction.getScoringModelDownloadURL("python", exportId));
                break;

            default:
                MLExportService.downloadFile($scope, () => DataikuAPI.ml.prediction.createScoringModelFile(type, model.fullModelId),
                    (exportId) => DataikuAPI.ml.prediction.getScoringModelDownloadURL(type, exportId));

        }
        WT1.event("model-export", {exportType: type});
    }

    $scope.downloadJavaScoringLibrary = function() {
        MLExportService.downloadFile($scope, () => DataikuAPI.ml.prediction.createScoringModelFile('jar-lib', $scope.model.fullModelId),
                     (exportId) => DataikuAPI.ml.prediction.getScoringModelDownloadURL($scope.exportParams.jar.type, exportId));
        WT1.event("model-export", {exportType : 'jar-lib'});
        $scope.dismiss();
    }

    $scope.getDatabricksModels = () => {
        if ( $scope.uiState.fromDatabricks.loadingModelList ) {
            return;
        }
        $scope.uiState.fromDatabricks.models = null;
        DataikuAPI.externalinfras.infos.listDatabricksRegisteredModels($scope.exportParams.mlflow.connectionName, $scope.exportParams.mlflow.useUnityCatalog)
            .success(function (resp) {
                $scope.$applyAsync(() => {
                    $scope.uiState.fromDatabricks.loadingModelList = true;
                });
                FutureWatcher.watchJobId(resp.jobId)
                    .success(function (resp2) {
                        $scope.uiState.fromDatabricks.models = resp2.result.sort((a, b) => a.name.localeCompare(b.name));;
                    })
                    .error(setErrorInScope.bind($scope))
                    .finally(() => {
                        $scope.uiState.fromDatabricks.loadingModelList = false;
                    });
                })
            .error(setErrorInScope.bind($scope))
            .finally(() => {
                $scope.uiState.fromDatabricks.loadingModelList = false;
            });
    }

    $scope.getDatabricksExperiments = () => {
        if ( $scope.uiState.fromDatabricks.loadingExperimentList ) {
            return;
        }
        $scope.uiState.fromDatabricks.experiments = null;
        DataikuAPI.externalinfras.infos.listDatabricksExperiments($scope.exportParams.mlflow.connectionName)
            .success(function (resp) {
                $scope.$applyAsync(() => {
                    $scope.uiState.fromDatabricks.loadingExperimentList = true;
                });
                FutureWatcher.watchJobId(resp.jobId)
                    .success(function (resp2) {
                        $scope.uiState.fromDatabricks.experiments = resp2.result.sort((a, b) => a.name.localeCompare(b.name));;
                    })
                    .error(setErrorInScope.bind($scope))
                    .finally(() => {
                        $scope.uiState.fromDatabricks.loadingExperimentList = false;
                    });
                })
            .error(setErrorInScope.bind($scope))
            .finally(() => {
                $scope.uiState.fromDatabricks.loadingExperimentList = false;
            });
    }
});


// Sets an attribute on the source dom element that signals to mdg that the element is ready to be 'used' (parsed/screenshotted etc)
// Requires the final step css selector (PuppeteerConfig.java) to equal the passed parameter
app.directive('puppeteerHookLoaded', function() {
    return {
        scope: false,
        restrict: 'A',
        link: function($scope, element, attributes) {
            const selectorName = attributes.puppeteerHookLoaded;
            element.attr(selectorName, true);
            // boolean value is added to scope to indicate to the puppeteer code that element content is ready for extraction
            $scope[selectorName] = true;
        }
    };
});

})();
