(function(){
'use strict';

const app = angular.module('dataiku.retrievableknowledge', []);



/* ************************************ List / Right column  *************************** */

app.directive('retrievableKnowledgeRightColumnSummary', function($controller, $state, $stateParams, $rootScope, FlowGraphSelection,
    DataikuAPI, SmartId, QuickView, ActiveProjectKey, ActivityIndicator, FlowBuildService, AnyLoc) {

    return {
        templateUrl :'/templates/retrievable-knowledge/right-column-summary.html',

        link : function(scope) {
            $controller('_TaggableObjectsMassActions', {$scope: scope});

            scope.$stateParams = $stateParams;
            scope.QuickView = QuickView;

            /* Auto save when summary is modified */
            scope.$on("objectSummaryEdited", function(){
                DataikuAPI.retrievableknowledge.save(scope.retrievableKnowledge, {summaryOnly: true}).success(function(data) {
                    ActivityIndicator.success("Saved");
                }).error(setErrorInScope.bind(scope));
            });

            scope.getSmartName = function (projectKey, name) {
                if (projectKey == ActiveProjectKey.get()) {
                    return name;
                } else {
                    return projectKey + '.' + name;
                }
            }

            scope.isOnObjectPage = function() {
                return $state.includes('projects.project.retrievableknowledges.retrievableknowledge');
            }


            scope.refreshData = function() {
                const projectKey = scope.selection.selectedObject.projectKey;
                const name = scope.selection.selectedObject.name;
                scope.canAccessObject = false;

                DataikuAPI.retrievableknowledge.getFullInfo(ActiveProjectKey.get(), SmartId.create(name, projectKey)).then(function({data}){
                    if (!scope.selection.selectedObject || scope.selection.selectedObject.projectKey != projectKey
                        || scope.selection.selectedObject.name != name) {
                        return; // too late!
                    }
                    scope.retrievableKnowledgeFullInfo = data;
                    scope.retrievableKnowledge = data.retrievableKnowledge;
                    scope.retrievableKnowledge.zone = (scope.selection.selectedObject.usedByZones || [])[0] || scope.selection.selectedObject.ownerZone;
                    scope.selection.selectedObject.interest = data.interest;
                    scope.isLocalRetrievableKnowledge = projectKey == ActiveProjectKey.get();
                    scope.objectAuthorizations = data.objectAuthorizations;
                    scope.canAccessObject = true;
                }).catch(setErrorInScope.bind(scope));
            };

            scope.$watch("selection.selectedObject",function(nv) {
                if(scope.selection.selectedObject != scope.selection.confirmedItem) {
                    scope.retrievableKnowledge = null;
                    scope.objectTimeline = null;
                }
            });

            function updateUserInterests() {
                DataikuAPI.interests.getForObject($rootScope.appConfig.login, "RETRIEVABLE_KNOWLEDGE", ActiveProjectKey.get(), scope.selection.selectedObject.name).success(function(data) {
                    scope.selection.selectedObject.interest = data;
                    scope.retrievableKnowledgeFullInfo.interest = data;
                }).error(setErrorInScope.bind(scope));
            }

            const interestsListener = $rootScope.$on('userInterestsUpdated', updateUserInterests);
            scope.$on("$destroy", interestsListener);

            scope.buildRetrievableKnowledge = function() {
                FlowBuildService.openSingleComputableBuildModalFromObjectTypeAndLoc(scope, "RETRIEVABLE_KNOWLEDGE",
                    AnyLoc.makeLoc(scope.retrievableKnowledge.projectKey, scope.retrievableKnowledge.id),
                    { downstreamBuildable: scope.retrievableKnowledgeFullInfo.downstreamBuildable });
            };

            scope.$watch("selection.confirmedItem", function(nv, ov) {
                if (!nv) {
                    return;
                }
                if (!nv.projectKey) {
                    nv.projectKey = ActiveProjectKey.get();
                }
                scope.refreshData();
            });

            scope.zoomToOtherZoneNode = function(zoneId) {
                const otherNodeId = scope.selection.selectedObject.id.replace(/zone__.+?__retrievableknowledge/, "zone__" + zoneId + "__retrievableknowledge");
                if ($stateParams.zoneId) {
                    $state.go('projects.project.flow', Object.assign({}, $stateParams, { zoneId: zoneId }))
                }
                else {
                    scope.zoomGraph(otherNodeId);
                    FlowGraphSelection.clearSelection();
                    FlowGraphSelection.onItemClick(scope.nodesGraph.nodes[otherNodeId], null);
                }
            }
        }
    }
});

app.controller("RetrievableKnowledgePageRightColumnActions", function($controller, $scope, $rootScope, DataikuAPI, $stateParams, ActiveProjectKey) {

    $controller('_TaggableObjectPageRightColumnActions', {$scope: $scope});

    $scope.selection = {};

    DataikuAPI.retrievableknowledge.get(ActiveProjectKey.get(), $stateParams.retrievableKnowledgeId).success((data) => {
        data.description = data.shortDesc;
        data.nodeType = 'LOCAL_RETRIEVABLE_KNOWLEDGE';
        data.realName = data.name;
        data.name = data.id;
        data.interest = {};

        $scope.selection = {
            selectedObject : data, confirmedItem : data,
        };
    }).error(setErrorInScope.bind($scope));

    $scope.renameObjectAndSave = function(newName) {
        $scope.selection.selectedObject.name = newName;
        return DataikuAPI.retrievableknowledge.save($scope.selection.selectedObject);
    };
});


app.controller("RetrievableKnowledgeListController", function($scope, $controller, $stateParams, DataikuAPI, $state, TopNav, WT1) {
    $controller('_TaggableObjectsListPageCommon', {$scope: $scope});

    $scope.sortBy = [
        { value: 'realName', label: 'Name' },
        { value: '-lastModifiedOn', label: 'Last modified'}
    ];

    $scope.selection = $.extend({
        filterQuery: {
            userQuery: '',
            tags: [],
            interest: {
                starred: '',
            },
            inputDatasetSmartName: []
        },
        filterParams: {
            userQueryTargets: ["realName", "tags"],
            propertyRules: {tag: 'tags'},
        },
        orderQuery: "-lastModifiedOn",
        orderReversed: false
    }, $scope.selection || {});

    $scope.maxItems = 20;

    $scope.list = function() {
        DataikuAPI.retrievableKnowledge.listHeads($stateParams.projectKey).success(function(data) {
            // dirty things to handle the discrepancy between the types of selected objects
            // which can have info displayed in the right panel
            data.forEach(mes => {
                mes.realName = mes.name;
                mes.name = mes.id;
            });
            $scope.listItems = data;
            $scope.restoreOriginalSelection();
        }).error(setErrorInScope.bind($scope));
    };

    TopNav.setLocation(TopNav.TOP_RETRIEVABLE_KNOWLEDGE, TopNav.LEFT_RETRIEVABLE_KNOWLEDGE, TopNav.TABS_NONE, null);
    TopNav.setNoItem();
    $scope.list();

    /* Tags handling */

    $scope.$on('selectedIndex', function(e, index){
        // an index has been selected, we unselect the multiselect
        $scope.$broadcast('clearMultiSelect');
    });

    /* Specific actions */
    $scope.goToItem = function(data) {
        $state.go("projects.project.retrievableknowledges.retrievableknowledge.usage", {projectKey : $stateParams.projectKey, retrievableKnowledgeId: data.id});
    }
});

app.controller("RetrievableKnowledgeController", function($scope, DataikuAPI, $stateParams,
            WT1, ActiveProjectKey, ActivityIndicator, TopNav, RetrievableKnowledgeUtils, FlowBuildService, AnyLoc) {

    $scope.uiState = {};

    DataikuAPI.retrievableknowledge.getFullInfo(ActiveProjectKey.get(), $stateParams.retrievableKnowledgeId).success(function(data){
        $scope.rkFullInfo = data;
        $scope.retrievableKnowledge = data.retrievableKnowledge;
        RetrievableKnowledgeUtils.updateIndexName($scope.retrievableKnowledge);
        $scope.savedSettings = angular.copy($scope.retrievableKnowledge);
        TopNav.setItem(TopNav.ITEM_RETRIEVABLE_KNOWLEDGE, $stateParams.retrievableKnowledgeId, {name: $scope.retrievableKnowledge.name});
        TopNav.setPageTitle($scope.retrievableKnowledge.name + " - Knowledge Bank");

        WT1.event("llm-mesh-knowledge-bank-load", {
            retrieverType: data.retrievableKnowledge.retrieverType,
            vectorStoreType: data.retrievableKnowledge.vectorStoreType,
            embeddingLLMId: data.retrievableKnowledge.embeddingLLMId,
            // Todo @augmentedllm clean WT1 llmsExposedWithCount
            //llmsExposedWithCount: data.retrievableKnowledge.llmsExposedWith.length
        });
    }).error(setErrorInScope.bind($scope));

    $scope.save = function(disableSuccessIndicator) {
        if(RetrievableKnowledgeUtils.hasConnection($scope.retrievableKnowledge) && !$scope.retrievableKnowledge.connection) {
           ActivityIndicator.error("You must select a connection");
           return;
        }

        if(RetrievableKnowledgeUtils.hasIndex($scope.retrievableKnowledge) && !$scope.retrievableKnowledge.indexName) {
            ActivityIndicator.error("You must provide an index name");
            return;
        }

        const settings = angular.copy($scope.retrievableKnowledge);
        return DataikuAPI.retrievableknowledge.save($scope.retrievableKnowledge).success(function(data) {
            $scope.savedSettings = settings;
            if (!disableSuccessIndicator) {
                ActivityIndicator.success("Saved");
            }
        }).error(setErrorInScope.bind($scope));
    };

    $scope.dirtySettings = function() {
        return !angular.equals($scope.savedSettings, $scope.retrievableKnowledge);
    };

    $scope.openBuildKbModal = function () {
      FlowBuildService.openSingleComputableBuildModalFromObjectTypeAndLoc(
        $scope,
        "RETRIEVABLE_KNOWLEDGE",
        AnyLoc.makeLoc($stateParams.projectKey, $scope.retrievableKnowledge.id),
        {
          redirectToJobPage: true,
          downstreamBuildable: $scope.rkFullInfo.downstreamBuildable,
        }
      );
    };

    const allowedTransitions = [
        'projects.project.retrievableknowledges.retrievableknowledge.search',
        'projects.project.retrievableknowledges.retrievableknowledge.usage',
        'projects.project.retrievableknowledges.retrievableknowledge.settings'
    ];

    checkChangesBeforeLeaving($scope, $scope.dirtySettings, null, allowedTransitions);
});


/* ************************************ Settings *************************** */
app.constant("DOCUMENT_SPLITTING_METHOD_MAP", {
    'CHARACTERS_BASED': 'Character count',
    'NONE': 'Do not split'
});

app.constant("VECTOR_STORE_UPDATE_METHOD_MAP", {
    'SMART_OVERWRITE': {isSmart: true, title: 'Smart Sync', description: "Insert new rows. Update modified rows. Remove rows from the Knowledge Bank that are not in the dataset."},
    'SMART_APPEND': {isSmart: true, title: 'Upsert', description: "Insert new rows. Update modified rows. No deletion is performed."},
    'OVERWRITE': {isSmart: false, title: 'Overwrite', description: "Completely rebuild the Knowledge Bank."},
    'APPEND': {isSmart: false, title: 'Append', description: "Append input rows to the Knowledge Bank. Can create duplicates."},
});

app.constant("EMBED_DOCS_VECTOR_STORE_UPDATE_METHOD_MAP", { // TODO keep in sync with above constant
    'SMART_OVERWRITE': {isSmart: true, title: 'Smart Sync', description: "Process new documents, fully reprocess modified ones, and remove those missing from the input managed folder."},
    'SMART_APPEND': {isSmart: true, title: 'Upsert', description: "Process new documents and fully reprocess modified ones without performing deletions."},
    'OVERWRITE': {isSmart: false, title: 'Overwrite', description: "Completely rebuild the Knowledge Bank."},
    'APPEND': {isSmart: false, title: 'Append', description: "Append documents to the Knowledge Bank. Can create duplicates."},
});

app.constant("EXTRACT_CONTENT_UPDATE_METHOD_MAP", { // TODO keep in sync with above constant. todo: move these methods outside of KB?
    'SMART_OVERWRITE': {isSmart: true, title: 'Smart Sync', description: "Process new documents, fully reprocess modified ones, and remove those missing from the input managed folder."},
    'SMART_APPEND': {isSmart: true, title: 'Upsert', description: "Process new documents and fully reprocess modified ones without performing deletions."},
    'OVERWRITE': {isSmart: false, title: 'Overwrite', description: "Completely rebuild the Output Dataset."},
    'APPEND': {isSmart: false, title: 'Append', description: "Append documents to the Output Dataset. Can create duplicates."},
});

app.constant("VECTOR_STORE_TYPE_MAP", {
    CHROMA: 'ChromaDB',
    FAISS: 'FAISS',
    PINECONE: 'Pinecone',
    AZURE_AI_SEARCH: 'Azure AI Search',
    VERTEX_AI_GCS_BASED: 'Vertex AI - GCS based',
    ELASTICSEARCH: 'Elasticsearch',
    QDRANT_LOCAL: 'Qdrant (local)'
});

app.constant("VECTOR_STORE_HYBRID_SUPPORT", [
    'AZURE_AI_SEARCH', 'ELASTICSEARCH'
]);

app.constant("RALLM_RETRIEVAL_SEARCH_TYPE_MAP", {
    'SIMILARITY': {title: 'Similarity score', description: "Retrieve the n closest documents by similarity score."},
    'SIMILARITY_THRESHOLD': {title: 'Similarity score with threshold', description: "Retrieve documents with a similarity score above the defined threshold."},
    'MMR': {title: 'Improve diversity of documents', description: "Helps the selected documents to cover a wider range of information while still matching the query. Uses MMR reranking."},
    'HYBRID': {title: 'Hybrid search', description: "Combine both a similarity search and a keyword search to retrieve more relevant documents."},
});


app.constant("RALLM_SEARCH_INPUT_STRATEGY_MAP", {
    'RAW_QUERY': {title: 'Raw mode', description: "Always queries the knowledge bank. For chat interactions, the input query is the full conversation history.", warnIfUsedInChat: true},
    'REWRITE_QUERY': {title: 'Smart mode', description: "Assesses first for each query if retrieving knowledge from the Knowledge Bank would be useful. If so, it reformulates the query to optimize retrieval. For chat interactions, the entire conversation history is reformulated.", warnIfUsedInChat: false},
});

app.component("printDocumentSourcesRadio", {
    bindings: {
        llmExposedWith: '='
    },
    controller: function () {
        this.$onInit = () => {
            if (this.llmExposedWith.includeContentInSources) {
                this.sourcesToPrint = "METADATA_AND_CONTENT";
            } else {
                this.sourcesToPrint = "METADATA";
            }
        };

        this.updateRagLlm = () => {
            if (this.sourcesToPrint === "METADATA") {
                this.llmExposedWith.includeContentInSources = false;
            } else if (this.sourcesToPrint === "METADATA_AND_CONTENT") {
                this.llmExposedWith.includeContentInSources = true;
            }
        };
    },
    templateUrl: "/templates/retrievable-knowledge/print-document-sources-radio.html"
});


app.component("sourceOutputFormat", {
    bindings: {
        llmExposedWith: '='
    },
    controller: function () {
        this.$onInit = () => {

            if (this.llmExposedWith?.printSources) {
                if (this.llmExposedWith.outputFormat === "TEXT") {
                    this.outputFormat = "TEXT";
                } else if (this.llmExposedWith.outputFormat === "JSON") {
                    this.outputFormat = "JSON";
                } else if (this.llmExposedWith.outputFormat === "SEPARATED") {
                    this.outputFormat = "SEPARATED";
                }
            } else {
                this.outputFormat = "NOTHING";
            }
        };

        this.updateRagLlm = () => {
            if (this.outputFormat === "NOTHING") {
                this.llmExposedWith.printSources = false;
            } else if (this.outputFormat === "TEXT") {
                this.llmExposedWith.printSources = true;
                this.llmExposedWith.outputFormat = "TEXT";
            } else if (this.outputFormat === "JSON") {
                this.llmExposedWith.printSources = true;
                this.llmExposedWith.outputFormat = "JSON";
            } else if (this.outputFormat === "SEPARATED") {
                this.llmExposedWith.printSources = true;
                this.llmExposedWith.outputFormat = "SEPARATED";
            }
        };
    },
    template: `
<label class="control-label" for="{{'sourceOutputFormat-' + $index}}">Legacy source output format</label>
<div class="controls" >
    <select id="{{'sourceOutputFormat-' + $index}}"
            ng-change="$ctrl.updateRagLlm()"
            dku-bs-select
            layout="list"
            ng-model="$ctrl.outputFormat"
            options-descriptions="[
                'Completion and sources are concatenated in a single text in the output.',
                'Completion and sources are put in a structured json in the output.',
                'Completion is put as text in the output. Sources are put in a separated field.',
                'Sources are not output.']" >
        <option value="TEXT">Plain text</option>
        <option value="JSON">JSON</option>
        <option value="SEPARATED">Separated (Recommended)</option>
        <option value="NOTHING">Do not print</option>
    </select>
</div>
`
});


app.controller("_RetrievableKnowledgeCommonController", function($scope, $stateParams, DataikuAPI, VECTOR_STORE_UPDATE_METHOD_MAP, CreateModalFromTemplate, RetrievableKnowledgeUtils, localStorageService) {
    $scope.VECTOR_STORE_UPDATE_METHOD_MAP = VECTOR_STORE_UPDATE_METHOD_MAP;
    $scope.recipeInfoLoaded = false;
    const STORAGE_KEY = `dss.retrievable-knowledge.${$stateParams.retrievableKnowledgeId}.search.settings`;
    $scope.searchParams = getInitialSearchParams();

    $scope.checkCodeEnvCompat = () => {
        if (!$scope || !$scope.retrievableKnowledge) {
            $scope.showCodeEnvVersionWarning = false;
            return;
        }

        if ($scope.retrievableKnowledge.vectorStoreType !== "PINECONE") {
            $scope.showCodeEnvVersionWarning = false;
            return;
        }

        DataikuAPI.retrievableknowledge.checkPineconeCompatibility($stateParams.projectKey, $scope.retrievableKnowledge.connection, $scope.retrievableKnowledge.envSelection).then(function({data}) {
            if (data.compatible) {
                $scope.showCodeEnvVersionWarning = false;
            } else {
                $scope.showCodeEnvVersionWarning = true;
                $scope.codeEnvVersionWarningMessage = data.reasons[0];
            }
        }).catch(setErrorInScope.bind($scope));
    }

    const deregister = $scope.$watch("retrievableKnowledge", function(nv, ov) {
        if (!nv) return;

        DataikuAPI.retrievableknowledge.getCurrentVersionInfo($stateParams.projectKey, $scope.retrievableKnowledge.id)
            .then(function({data}) {
                const { embeddingRecipeParams, rkAtVersion, isBuilt } = data;
                $scope.embeddingRecipeDesc = embeddingRecipeParams;
                $scope.rkAtEmbedding = rkAtVersion;
                $scope.isRkBuilt = isBuilt;

                if (embeddingRecipeParams) {
                    $scope.possibleSourceIDColumns = embeddingRecipeParams.metadataColumns.map((mc) => mc.column);
                } else {
                    $scope.possibleSourceIDColumns = [];
                }
                $scope.recipeInfoLoaded = true;
            })
            .catch(setErrorInScope.bind($scope));

        DataikuAPI.pretrainedModels.listAvailableLLMs($stateParams.projectKey, "GENERIC_COMPLETION").then(function({data}) {
            $scope.availableAugmentableLLMs = data.identifiers.filter(id => id.type !== "RETRIEVAL_AUGMENTED");
        }).catch(setErrorInScope.bind($scope));

        DataikuAPI.pretrainedModels.listAvailableLLMs($stateParams.projectKey, "TEXT_EMBEDDING_EXTRACTION").then(function({data}) {
            $scope.availableEmbeddingModels = data.identifiers;
        }).catch(setErrorInScope.bind($scope));

        $scope.checkCodeEnvCompat();
        deregister();
    });

    function getInitialSearchParams() {
        const defaultParams = RetrievableKnowledgeUtils.getRAGLLMSettings({
            maxDocuments: 10,
            includeScore: true,
            allowEmptyQuery: true,
            includeMultimodalContent: true,
            filter: {}
        });
        const params = localStorageService.get(STORAGE_KEY) || {};

        return Object.assign(defaultParams, params);
    }

    function setSearchParams(newParams) {
        $scope.searchParams = newParams;
        localStorageService.set(STORAGE_KEY, newParams);
    }

    $scope.openSearchSettingsModal = () => {
        CreateModalFromTemplate('/templates/retrievable-knowledge/search-settings-modal.html', $scope, null, function(modalScope) {
            modalScope.knowledgeBankSchema = {
                columns: $scope.retrievableKnowledge.metadataColumnsSchema || $scope.rkAtEmbedding?.metadataColumnsSchema || [],
            };
            modalScope.knowledgeBankSchema.knowledgeBankSchemaColumnName = modalScope.knowledgeBankSchema.columns.map((column) => column.name);
            modalScope.originalSearchParams = angular.copy($scope.searchParams);
            modalScope.saveSettings = () => {
                setSearchParams(modalScope.originalSearchParams);
                modalScope.dismiss();
            };
        });
    };
});

app.component('rrfHelpMessage', {
    bindings: {
    },
    template: `
<div class="help-inline">
    Uses Elasticsearch implementation of
    <external-link
            href="https://www.elastic.co/guide/en/elasticsearch/reference/current/rrf.html">Reciprocal Ranking Fusion (RRF)
    </external-link>
    &nbsp This feature requires an Elastic Enterprise subscription; please verify your subscription type with Elastic.
</div>
`
});

app.component('semanticRankerHelpMessage', {
    template: `
    <div class="help-inline">
        Uses Azure AI proprietary
        <external-link
                href="https://learn.microsoft.com/en-us/azure/search/semantic-search-overview">Semantic Ranker
        </external-link>
    </div>
    `
});

app.component('retrievalContentSettings', {
    bindings: {
        retrievableKnowledge: '<',
        embeddingRecipeDesc: '<',
        availableAugmentableLlms: '<',
        metadataColumns: '<',
        showContentRetrieval: '<?',
        llmExposedWith: '='
    },
    templateUrl: '/templates/retrievable-knowledge/retrieval-content-settings.html',
    controller: function() {
        const DKU_TEXT_EMBEDDING_COLUMN = 'DKU_TEXT_EMBEDDING_COLUMN';
        const DKU_GENERATED_EMBEDDING = 'DKU_GENERATED_EMBEDDING';
        const $ctrl = this;

        $ctrl.$onInit = function() {

            if ($ctrl.embeddingRecipeDesc.knowledgeColumn === DKU_GENERATED_EMBEDDING) {
                $ctrl.embeddingColumnOption = {label:"Generated embedding column", value:DKU_TEXT_EMBEDDING_COLUMN};
            }
            else {
                $ctrl.embeddingColumnOption = {label:`${$ctrl.embeddingRecipeDesc.knowledgeColumn} (Embedding column)`, value:DKU_TEXT_EMBEDDING_COLUMN};
            }

            $ctrl.retrievalColumnsOptions = [
                $ctrl.embeddingColumnOption,
                ...$ctrl.metadataColumns.filter(col => col !== $ctrl.embeddingRecipeDesc.knowledgeColumn && col !== DKU_GENERATED_EMBEDDING).map(col => {return {value:col, label:col}})
            ];
        }

        $ctrl.hasMetadataColumns = function() {
            return $ctrl.metadataColumns && $ctrl.metadataColumns.length > 0;
        };

        $ctrl.disableTextGuardrails = function (ragllmSettings) {
            ragllmSettings.ragSpecificGuardrails.faithfulnessSettings.enabled = false;
            ragllmSettings.ragSpecificGuardrails.relevancySettings.enabled = false;
        };

        $ctrl.disableMultimodalGuardrails = function (ragllmSettings) {
            ragllmSettings.ragSpecificGuardrails.multimodalFaithfulnessSettings.enabled = false;
            ragllmSettings.ragSpecificGuardrails.multimodalRelevancySettings.enabled = false;
        };

        $ctrl.getAlertRetrievalColumnsText = () => {

            if ($ctrl.llmExposedWith.retrievalSource !== 'EMBEDDING') {
                return undefined;
            }
            const columns = $ctrl.llmExposedWith.retrievalColumns || [];

            if (columns.length === 0) {
                return `No retrieval column selected, \`${$ctrl.embeddingColumnOption.label}\` will be used by default.`;
            }

            const nonAvailableColumns = columns
                .filter(col => col !== DKU_TEXT_EMBEDDING_COLUMN && !$ctrl.metadataColumns.includes(col));

            if (nonAvailableColumns.length) {
                return `The following retrieval columns are not available in the metadata: ${nonAvailableColumns.join(', ')}. They will be ignored.`;
            }
            return undefined;
        };

        $ctrl.shouldAlertMultimodalColumn = (ragllmSettings) => {
            if (ragllmSettings.retrievalSource !== "MULTIMODAL" || !$ctrl.availableAugmentableLlms) {
                return false
            }
            const selectedAugmentedLLM =  $ctrl.availableAugmentableLlms.filter(llm => llm.id == ragllmSettings.llmId);
            if(!selectedAugmentedLLM || !selectedAugmentedLLM.length){
                return false;
            }
            return !selectedAugmentedLLM[0].supportsImageInputs; // Non-vlm augmented model is selected to use the multimodal column at retrieval => can leads to errors when using it
        };

    }
});

app.component('retrievalParamsSettings', {
    bindings: {
        retrievableKnowledge: '<',
        showDocumentCostWarning: '<?',
        llmExposedWith: '='
    },
    templateUrl: '/templates/retrievable-knowledge/retrieval-params-settings.html',
    controller: function(VECTOR_STORE_HYBRID_SUPPORT, RALLM_RETRIEVAL_SEARCH_TYPE_MAP, DataikuAPI, SemanticVersionService) {
        const $ctrl = this;

        $ctrl.$onInit = () => {
            if (!$ctrl.llmExposedWith.searchType) {
                $ctrl.llmExposedWith.searchType = "SIMILARITY";
            }
            if ($ctrl.llmExposedWith.similarityThreshold === undefined) {
                $ctrl.llmExposedWith.similarityThreshold = 0.5;
            }

            $ctrl.elasticSearchData = null;
            setSearchTypeDescriptions();
        };

        $ctrl.$onChanges = (changes) => {
            if (changes.retrievableKnowledge && !(changes.retrievableKnowledge.previousValue)) {
                if ($ctrl.retrievableKnowledge && $ctrl.retrievableKnowledge.vectorStoreType === "ELASTICSEARCH" && $ctrl.retrievableKnowledge.connection) {
                    DataikuAPI.admin.connections.get($ctrl.retrievableKnowledge.connection).success(function(connectionData) {
                        DataikuAPI.admin.connections.testElasticSearch(connectionData, null).success(function(testResponse) {
                            $ctrl.elasticSearchData = testResponse;
                            setSearchTypeDescriptions();
                        });
                    }).error(setErrorInScope.bind($ctrl));
                }
            }
        };

        $ctrl.RALLM_RETRIEVAL_SEARCH_TYPE_MAP = RALLM_RETRIEVAL_SEARCH_TYPE_MAP;

        $ctrl.diversityDocsUnavailableReason = function() {
            if ($ctrl.retrievableKnowledge && $ctrl.retrievableKnowledge.vectorStoreType && ['AZURE_AI_SEARCH', 'VERTEX_AI_GCS_BASED'].includes($ctrl.retrievableKnowledge.vectorStoreType)) {
                return "This option isn't available for this vector store type";
            }
            return "";
        };

        $ctrl.hybridSearchSupported = function() {
            return $ctrl.retrievableKnowledge && VECTOR_STORE_HYBRID_SUPPORT.includes($ctrl.retrievableKnowledge.vectorStoreType);
        };

        $ctrl.hybridSearchUnavailableReason = function () {
            if (!$ctrl.hybridSearchSupported()) {
                return "Hybrid search is not supported for this vector store type";
            }
            if ($ctrl.retrievableKnowledge.vectorStoreType === "ELASTICSEARCH") {
                if (!$ctrl.elasticSearchData || !$ctrl.elasticSearchData.connectionOK) {
                    // No connection info yet, so we cannot determine if hybrid search is available
                    return "";
                }
                if ($ctrl.elasticSearchData.distributionName !== "ElasticSearch") {
                    $ctrl.llmExposedWith.useAdvancedReranking = false;
                    return "Hybrid search is only available for ElasticSearch distribution";
                }
                if (SemanticVersionService.compareVersions($ctrl.elasticSearchData.version, "8.4") <= 0) {
                    $ctrl.llmExposedWith.useAdvancedReranking = false;
                    return "Hybrid search is only available for ElasticSearch version 8.4 and above";
                }
            }
            return "";
        };

        function setSearchTypeDescriptions() {
            const similarityDescription = RALLM_RETRIEVAL_SEARCH_TYPE_MAP['SIMILARITY']['description'];
            const similarityThresholdDescription = RALLM_RETRIEVAL_SEARCH_TYPE_MAP['SIMILARITY_THRESHOLD']['description'];
            let diversityDocsDescription = RALLM_RETRIEVAL_SEARCH_TYPE_MAP['MMR']['description'];
            let hybridDescription = RALLM_RETRIEVAL_SEARCH_TYPE_MAP['HYBRID']['description'];
            const diversityDocsUnavailableReason = $ctrl.diversityDocsUnavailableReason();
            const hybridSearchUnavailableReason = $ctrl.hybridSearchUnavailableReason();
            if (diversityDocsUnavailableReason) {
                diversityDocsDescription += '<div class="alert alert-warning alert-in-dropdown" style="margin: 10px 0 0 0;"><i class="icon-dku-warning"></i>' + diversityDocsUnavailableReason + '</div>';
            }
            if (hybridSearchUnavailableReason) {
                hybridDescription += '<div class="alert alert-warning alert-in-dropdown" style="margin: 10px 0 0 0;"><i class="icon-dku-warning"></i>' + hybridSearchUnavailableReason + '</div>';
            }
            $ctrl.diversityDocsUnavailable = !!diversityDocsUnavailableReason;
            $ctrl.hybridSearchUnavailable = !!hybridSearchUnavailableReason;
            $ctrl.searchTypeDescriptions = [similarityDescription, similarityThresholdDescription, diversityDocsDescription, hybridDescription]
        };
    }
});

app.component('mmrSearchSettings', {
    bindings: {
        llmExposedWith: '=',
        vectorStoreType: '<',
    },
    templateUrl: '/templates/retrievable-knowledge/mmr-search-settings.html',
    controller: function() {
        const $ctrl = this;
    }
});

app.component('hybridSearchSettings', {
    bindings: {
        llmExposedWith: '=',
        vectorStoreType: '<',
        connection: '<',
        elasticSearchData: '<'
    },
    templateUrl: '/templates/retrievable-knowledge/hybrid-search-settings.html',
    controller: function(DataikuAPI, VECTOR_STORE_HYBRID_SUPPORT, SemanticVersionService) {
        const $ctrl = this

        $ctrl.hybridSearchSupported = function () {
            return VECTOR_STORE_HYBRID_SUPPORT.includes($ctrl.vectorStoreType);
        }

        $ctrl.$onChanges = (changes) => {
            if (!$ctrl.hybridSearchSupported()) {
                $ctrl.llmExposedWith.useAdvancedReranking = false;
            }
        };

        $ctrl.advancedRerankingUnavailableReason = function () {
            if ($ctrl.vectorStoreType === "ELASTICSEARCH") {
                if (!$ctrl.elasticSearchData || !$ctrl.elasticSearchData.connectionOK) {
                    // No connection info yet, so we cannot determine if rrf is available
                    return "";
                }
                if (SemanticVersionService.compareVersions($ctrl.elasticSearchData.version, "8.16") <= 0) {
                    $ctrl.llmExposedWith.useAdvancedReranking = false;
                    return "RRF option is only available for ElasticSearch version 8.16 and above";
                }
            }
            return "";
        }
    }
});

app.component('searchInputStrategySettings', {
    bindings: {
        llmExposedWith: '=',
        defaultRewritePrompt: '=',
    },
    templateUrl: '/templates/retrievable-knowledge/search-input-strategy-settings.html',
    controller: function(RALLM_SEARCH_INPUT_STRATEGY_MAP, DataikuAPI) {
        const $ctrl = this

        $ctrl.RALLM_SEARCH_INPUT_STRATEGY_MAP = RALLM_SEARCH_INPUT_STRATEGY_MAP;

        $ctrl.searchRewriteIsEnabled = function() {
            return $ctrl.llmExposedWith.searchInputStrategySettings?.strategy == "REWRITE_QUERY";
        };
    }
});

    app.constant("SNIPPET_FORMATS", [
        {key: 'TEXT', label: 'Text', helpText: 'The snippet is plain text'},
        {key: 'MARKDOWN', label: 'Markdown', helpText: 'The snippet is in Markdown format'},
        {key: 'HTML', label: 'HTML', helpText: 'The snippet is in HTML format'},
        {key: 'JSON', label: 'JSON', helpText: 'The snippet is in Json format'}
    ])

    app.component('metadataSelectField', {
        bindings: {
            label: '@',
            column: '=',
            format: '=',
            fieldType: '<',
            helpText: '@',
            possibleColumns: '<',
        },
        controller: function (SNIPPET_FORMATS) {
            const $ctrl = this;
            $ctrl.$onInit = function () {
                $ctrl.SNIPPET_FORMATS = SNIPPET_FORMATS;
                $ctrl.FORMAT_DESCRIPTIONS = SNIPPET_FORMATS.map(format => format.helpText);
            }

        },
        template: `
    <div class="control-group m0">
      <label class="control-label">{{ $ctrl.label }}</label>
      <div class="controls horizontal-flex" >
        <ng2-typeahead
            [(value)]="$ctrl.column"
            [suggestions]="$ctrl.possibleColumns"
            placeholder="Select metadata">
        </ng2-typeahead>
        <select
            class="mleft4"
            ng-if="$ctrl.fieldType === 'snippetMetadata'"
            layout="list"
            dku-bs-select
            ng-model="$ctrl.format"
            options-descriptions="$ctrl.FORMAT_DESCRIPTIONS" >
            <option ng-repeat="format in $ctrl.SNIPPET_FORMATS" value="{{format.key}}">{{format.label}}</option>
        </select>
      </div>
    </div>
  `
    });

    app.component('additionalInfoSourcesDefaultMetadata', {
        bindings: {
            model: '=',
            metadataColumns: '<',
        },
        controller: function () {
            const $ctrl = this;

            $ctrl.$onInit = function () {
                $ctrl.metadataItems = $ctrl.model.metadataInSources.map(col => ({name: col}))
                $ctrl.metadataToAdd = $ctrl.getMetadataToAdd();
            }

            $ctrl.updateDefaultMetadata = function () {
                $ctrl.model.metadataInSources = $ctrl.metadataItems
                    .map(item => {
                        item.name = (item.name || '').trim();
                        return item.name;
                    })
                    .filter(Boolean);
                $ctrl.metadataToAdd.splice(0, $ctrl.metadataToAdd.length, ...$ctrl.getMetadataToAdd().map(item => item.name));
                $ctrl.duplicatedMetadata = $ctrl.model.metadataInSources
                    .filter((name, i, all) => all.indexOf(name) !== i);
            }

            $ctrl.addDefault = function () {
                $ctrl.metadataItems = [...$ctrl.metadataItems, {name: ""}];
            }
            $ctrl.getMetadataToAdd = function () {
                if (!$ctrl.metadataColumns || $ctrl.metadataColumns.length === 0) {
                    return [];
                }
                const existingMetadata = new Set($ctrl.metadataItems.map(item => item.name));
                return $ctrl.metadataColumns
                    .filter(col => col !== undefined && !existingMetadata.has(col)).map(col => ({name: col}));
            }
            $ctrl.fillAllDefaultMetadata = function () {
                $ctrl.metadataItems = [...$ctrl.metadataItems, ...$ctrl.getMetadataToAdd()];
                $ctrl.updateDefaultMetadata()
            }

        },
        template: `

    <div class="control-group">
        <label class="control-label">Standard</label>
        <div class="controls">
            <editable-list ng-model="$ctrl.metadataItems"
                no-change-on-add="true"
                transcope="{
                    ctrl: $ctrl
                }"
                disable-add="true"
                template="{name: ''}"
                on-change="$ctrl.updateDefaultMetadata()"
            >
                <ng2-typeahead
                    [(value)]="it.name"
                    [suggestions]="ctrl.metadataToAdd"
                    placeholder="Select metadata">
                </ng2-typeahead>
            </editable-list>
            <div data-block="newItem" class="dropdown"">
                <button
                    type="button"
                    data-qa-add-button
                    ng-click="$ctrl.addDefault()"
                    class="btn btn--primary btn--text editable-list__add-label btn--dku-icon" >
                    <i class="dku-icon-plus-16"></i>Add Metadata
                </button>
                <span class="retrieval-augmented-llm-button_separator" ng-if="$ctrl.metadataToAdd.length > 0"></span>
                <button class="btn m0 btn--text btn--primary btn--icon btn--dku-icon" data-toggle="dropdown" dropdown-position="fixed" ng-if="$ctrl.metadataToAdd.length > 0">
                    <i class="dku-icon-caret-down-16"></i>
                </button>
                <ul class="dropdown-menu" ng-if="$ctrl.metadataToAdd.length > 0">
                    <li><a ng-click="$ctrl.fillAllDefaultMetadata()" ng-if="$ctrl.metadataToAdd.length > 0">
                            Add all metadata
                        </a>
                    </li>
                </ul>
            </div>
            <div class="alert alert-warning mtop8 mbot0 horizontal-flex" ng-if="$ctrl.duplicatedMetadata.length">
            <div class="padright16"><i class="icon-dku-warning" /></div>
              You have the following metadata {{$ctrl.duplicatedMetadata.length > 1? 's': ''}} declared more than once:
              <strong>&nbsp{{$ctrl.duplicatedMetadata.join(',&nbsp')}}</strong>
            </div>
        </div>
    </div>
    `
    });


    app.constant("METADATA_FIELDS", [
        {name: 'titleMetadata', label: 'Title', helpText: 'Title to display'},
        {name: 'urlMetadata', label: 'URL', helpText: 'URL to make a link from the source (if relevant)'},
        {
            name: 'thumbnailURLMetadata',
            label: 'Thumbnail URL',
            helpText: 'URL to an image to display a thumbnail for the source (if relevant)'
        },
        {
            name: 'snippetMetadata',
            label: 'Snippet',
            helpText: 'Snippet text to display. If left empty, the retrieval text is used instead'
        }
    ]);
    app.component('additionalInfoSourcesFormattingConfig', {
        bindings: {
            model: '=',
            metadataColumns: '<',
        },
        controller: function ($timeout, METADATA_FIELDS) {
            const $ctrl = this;

            $ctrl.$onInit = function () {
                $ctrl.METADATA_FIELDS = METADATA_FIELDS;
                $ctrl.visibleFields = METADATA_FIELDS.filter(field => $ctrl.model[field.name] !== undefined);
                if ($ctrl.model['snippetFormat'] === undefined) {
                    $ctrl.model.snippetFormat = 'TEXT'; // Default snippet format
                }
            }

            $ctrl.addField = function (field) {
                if (!$ctrl.visibleFields.some(f => f.name === field.name)) {
                    $ctrl.visibleFields = [...$ctrl.visibleFields, field];
                }
            }

            $ctrl.onDeleteField = function () {
                $ctrl.getRemainingFields().map(field => $ctrl.model[field.name] = undefined)
            }

            $ctrl.getRemainingFields = function () {
                const usedNames = new Set($ctrl.visibleFields.map(f => f.name));
                return $ctrl.METADATA_FIELDS.filter(f => !usedNames.has(f.name));
            };
        },
        template: `
    <p>Add metadata to the LLM output. Choose standard to include it as is, or select a role (e.g., Title, URL) to format it accordingly</p>
    <additional-info-sources-default-metadata
        model="$ctrl.model"
        metadata-columns="$ctrl.metadataColumns">
    </additional-info-sources-default-metadata>
    <div class="mtop16"></div>
    <div class="control-group" ng-if="$ctrl.visibleFields.length > 0">
        <label class="control-label">With role</label>
    </div>
    <editable-list
        ng-model="$ctrl.visibleFields"
        on-remove="$ctrl.onDeleteField()"
        disable-add="true"
        transcope="{
            ctrl: $ctrl
        }">
        <metadata-select-field
            label="{{it.label}}"
            column="ctrl.model[it.name]"
            format="ctrl.model.snippetFormat"
            field-type="it.name"
            help-text="{{it.helpText}}"
            possible-columns="ctrl.metadataColumns">
        </metadata-select-field>
    </editable-list>
    <div class="control-group">
        <label ng-if="$ctrl.visibleFields.length==0" class="control-label">With role</label>
        <div class="controls">
            <div data-block="newItem" class="dropdown"">
                <button
                    type="button"
                    data-qa-add-button
                    data-toggle="dropdown"
                    dropdown-position="fixed"
                    class="btn btn--primary btn--text editable-list__add-label btn--dku-icon" >
                    <i class="dku-icon-plus-16"></i>Add Role<i class="dku-icon-caret-down-16 mleft4"></i>
                </button>
                <ul class="dropdown-menu detailed-dropdown-menu">
                    <li ng-repeat="field in $ctrl.getRemainingFields()" class="detailed-dropdown-menu__item" ng-click="$ctrl.addField(field)">
                        <div class="detailed-dropdown-menu__item-info">
                            <div class="detailed-dropdown-menu__item-title">{{field.label}}</div>
                            <div class="detailed-dropdown-menu__item-description">{{field.helpText}}</div>
                        </div>
                    </li>
                    <li disabled="true" ng-repeat="field in $ctrl.visibleFields" class="detailed-dropdown-menu__item disabled">
                        <div disabled="true" class="detailed-dropdown-menu__item-info">
                            <div disabled="true" class="detailed-dropdown-menu__item-title">{{field.label}}</div>
                            <div disabled="true" class="detailed-dropdown-menu__item-description">{{field.helpText}}</div>
                        </div>
                    </li>
                </ul>
            </div>
        </div>
    </div>
    `
    });

app.controller("RetrievableKnowledgeUsageController", function($scope, $rootScope, TopNav, $controller, CreateModalFromTemplate, PluginsService, ActivityIndicator, $state, DataikuAPI, ActiveProjectKey, $stateParams) {
    $controller('_RetrievableKnowledgeCommonController', {$scope: $scope});
    $scope.uiState = $scope.uiState || {};
    $scope.uiState.searchQuery = '';

    TopNav.setLocation(TopNav.TOP_RETRIEVABLE_KNOWLEDGE, TopNav.LEFT_RETRIEVABLE_KNOWLEDGE, TopNav.TABS_RETRIEVABLE_KNOWLEDGE, "usage");

    $scope.answersDocRef = $rootScope.versionDocRoot + "generative-ai/answers.html";

    $scope.openKBSearchToolModal = function () {
        CreateModalFromTemplate('/templates/retrievable-knowledge/kb-search-agent-tool.html', $scope, 'KBSearchToolModalController');
    };

    $scope.showCreateRetrievalAugmentedLLMModal = function (preselectedInput) {
        CreateModalFromTemplate('/templates/savedmodels/retrieval-augmented-llm/create-retrieval-augmented-llm-modal.html', $scope, 'CreateRetrievalAugmentedLLMModalController', function (newScope) {
            newScope.input.preselectedInput = [{
                id: $scope.retrievableKnowledge.id,
                displayName: $scope.retrievableKnowledge.name,
                type: 'RETRIEVABLE_KNOWLEDGE',
            }];
        });
    };

    DataikuAPI.retrievableknowledge
        .getRetrievalAugmentedLLMList(ActiveProjectKey.get(), $stateParams.retrievableKnowledgeId)
        .success(function (data) {
            $scope.retrievalAugmentedLLMs = data;
        })
        .error(setErrorInScope.bind($scope));

    DataikuAPI.agentTools.list($stateParams.projectKey).then(function({ data }) {
        $scope.availableSearchTools = data?.filter(tool => tool.type === 'VectorStoreSearch' && tool.params.knowledgeBankRef === $stateParams.retrievableKnowledgeId);
    }).catch(setErrorInScope.bind($scope));

    $scope.goToRetrievalAugmentedLLMActiveVersion = function (retrievalAugmentedLLM) {
        $state.go('projects.project.savedmodels.savedmodel.retrievalaugmentedllm.design', {
            fullModelId: `S-${$stateParams.projectKey}-${retrievalAugmentedLLM.id}-${retrievalAugmentedLLM.activeVersion}`,
            smId: retrievalAugmentedLLM.id,
        });
    };

    $scope.goToSearchTool = function (searchTool) {
        $state.go('projects.project.agenttools.agenttool', {
            projectKey: $stateParams.projectKey,
            agentToolId: searchTool.id
        });
    };

    $scope.searchVectorStore = () => {
        $state.go('projects.project.retrievableknowledges.retrievableknowledge.search', {
            projectKey: $stateParams.projectKey,
            retrievableKnowledgeId: $stateParams.retrievableKnowledgeId,
            searchQuery: $scope.uiState.searchQuery,
            initiateSearch: true
        });
    };
});


app.controller("RetrievableKnowledgeSettingsController", function($scope, $rootScope, DataikuAPI, CreateModalFromTemplate, $state, $stateParams, TopNav, $controller,
    ActivityIndicator, ComputableSchemaRecipeSave, ActiveProjectKey, DOCUMENT_SPLITTING_METHOD_MAP, Dialogs) {
    $controller('_RetrievableKnowledgeCommonController', {$scope: $scope});
    TopNav.setLocation(TopNav.TOP_RETRIEVABLE_KNOWLEDGE, TopNav.LEFT_RETRIEVABLE_KNOWLEDGE, TopNav.TABS_RETRIEVABLE_KNOWLEDGE, "settings");

    $scope.showCodeEnvVersionWarning = false;
    $scope.codeEnvVersionWarningMessage = "";

    $scope.userFriendlyDocSplittingMethod = function(method) {
        return DOCUMENT_SPLITTING_METHOD_MAP[method];
    };
    $scope.clear = function() {
        $scope.save();
        return CreateModalFromTemplate("/templates/retrievable-knowledge/clear-modal.html", $scope, null, function(modalScope) {
            modalScope.confirm = function() {
                DataikuAPI.retrievableknowledge.clear(ActiveProjectKey.get(), $stateParams.retrievableKnowledgeId).then(function ({data}) {
                    modalScope.dismiss();
                    Dialogs.infoMessagesDisplayOnly($scope, "Clear result", data);
                }).catch(setErrorInScope.bind($scope));
            }
        });
    }

    DataikuAPI.pretrainedModels.listAvailableLLMs($stateParams.projectKey, "TEXT_EMBEDDING_EXTRACTION").then(function({data}) {
        $scope.availableEmbeddingLLMs = data.identifiers;
    }).catch(setErrorInScope.bind($scope));

    $scope.$watch("retrievableKnowledge.envSelection.envMode", function() {
        $scope.checkCodeEnvCompat();
    });

    $scope.$watch("retrievableKnowledge.envSelection.envName",  function() {
        $scope.checkCodeEnvCompat();
    });

});

app.controller('RetrievableKnowledgeSearchController', function($scope, $compile, $stateParams, $controller, TopNav, DataikuAPI, CreateModalFromTemplate, RetrievableKnowledgeUtils, localStorageService, $filter) {
    $controller('_RetrievableKnowledgeCommonController', {$scope: $scope});

    TopNav.setLocation(TopNav.TOP_RETRIEVABLE_KNOWLEDGE, TopNav.LEFT_RETRIEVABLE_KNOWLEDGE, TopNav.TABS_RETRIEVABLE_KNOWLEDGE, "search");

    $scope.uiState = $scope.uiState || {};
    $scope.uiState.hasSearched = false;
    $scope.uiState.searchQuery = '';
    $scope.uiState.sortColumn = 'score';
    $scope.uiState.sortDescending = true;
    $scope.metadataColumns = [];

    let isMultimodal = false;
    let hasOutline = false;
    let hasPageRange = false;

    $scope.searchVectorStore = () => {
        DataikuAPI.retrievableknowledge.search($stateParams.projectKey, $stateParams.retrievableKnowledgeId, $scope.uiState.searchQuery, $scope.searchParams).then(function({ data }) {
            $scope.previousSearch = $scope.uiState.searchQuery;
            $scope.searchResponse = data;
            const output = data?.response?.output;
            const sources = data?.response?.sources;
            const sourceItems = sources && sources[0] && sources[0].items;
            $scope.uiState.hasSearched = true;
            if (output && output.documents) {
                const documents = output.documents;
                isMultimodal = false;
                hasOutline = false;
                hasPageRange = false;
                documents.forEach((document, index) => {
                    const metadata = document.metadata;
                    if (metadata) {
                        if (metadata.DKU_MULTIMODAL_CONTENT) {
                            const content = JSON.parse(metadata.DKU_MULTIMODAL_CONTENT);
                            isMultimodal = true;
                            document.multimodal = content;
                            delete metadata.DKU_MULTIMODAL_CONTENT;
                        }
                        document.metadata = Object.fromEntries(
                            Object.entries(document.metadata).map(([key, content]) => [key, { type: 'text', content }])
                        );
                    }

                    const item = sourceItems && sourceItems[index];
                    if (item && item.type === 'FILE_BASED_DOCUMENT' && item.fileRef) {
                        document.source = {};
                        if (item.fileRef.sectionOutline) {
                            hasOutline = true;
                            document.source.sectionOutline = {
                                type: 'text',
                                content: item.fileRef.sectionOutline
                            };
                        }
                        if (item.fileRef.pageRange) {
                            hasPageRange = true;
                            document.source.pageRange = item.fileRef.pageRange.start + ' to ' + item.fileRef.pageRange.end;
                        }
                    }
                });

                $scope.documents = documents;
                prepareSearchResults();
            }
        }).catch(setErrorInScope.bind($scope));
    };

    class RetrievalCellRenderer {
        init(params) {
            this.agParams = params;
            const data = this.agParams.value;

            if (data) {
                if (data.type === 'text') {
                    this.initTextCell(data.content);
                } else if (data.type === 'images') {
                    this.initImageCell(data.content);
                }
            }
        }

        initTextCell(content) {
            const MAX_LENGTH = 500;
            const fullText = content || '';
            const isTruncated = fullText.length > MAX_LENGTH;
            const displayedText = isTruncated ? fullText.slice(0, MAX_LENGTH) + '…' : fullText;

            this.eGui = document.createElement('div');
            this.eGui.innerHTML = `
                <span ng-bind="displayedText"></span> <a ng-if="isTruncated" ng-click="toggleText()">Show {{ isExpanded ? 'less' : 'more' }}</a>
            `
            this.scope = $scope.$new(true);
            this.scope.displayedText = displayedText;
            this.scope.isTruncated = isTruncated;
            this.scope.isExpanded = false;
            this.scope.toggleText = () => {
                this.scope.isExpanded = !this.scope.isExpanded;
                this.scope.displayedText = this.scope.isExpanded ? fullText : displayedText;
                this.agParams.api.resetRowHeights();
            };

            $compile(this.eGui)(this.scope);
            this.scope.$evalAsync();
        }

        initImageCell(content) {
            this.eGui = document.createElement('div');
            this.eGui.innerHTML = `<prompt-images paths="imagePaths" folder-id="folderId" show-large-images="true" max-displayed-images="5"></prompt-images>`

            this.scope = $scope.$new(true);
            this.scope.imagePaths = content;
            this.scope.folderId = $scope.retrievableKnowledge.managedFolderId;

            $compile(this.eGui)(this.scope);
            this.scope.$evalAsync();
        }

        getGui() {
            return this.eGui;
        }

        // need to destroy scope when cell is un-rendered to avoid memory leaks
        destroy() {
            if (this.scope) {
                this.scope.$destroy();
            }
        }
    }

    if ($stateParams.initiateSearch) {
        $scope.uiState.searchQuery = $stateParams.searchQuery;
        $scope.uiState.hasSearched = true;
        $scope.searchVectorStore();
    }

    function prepareSearchResults() {
        const textOptions = {
            autoHeight: true,
            wrapText: true,
            cellStyle: {'word-break': 'break-word'},
            cellRenderer: RetrievalCellRenderer,
            valueFormatter: (params) => params.value?.content
        };
        const metadata = ($scope.retrievableKnowledge.metadataColumnsSchema || $scope.rkAtEmbedding?.metadataColumnsSchema || []).map(column => ({
            headerName: column.name,
            field: 'metadata.' + column.name,
            ...textOptions
        }));

        $scope.columnDefs = [
            ...(supportsSimilarityScore() ? [{
                field: 'score',
                comparator: (score1, score2) => score1 - score2
            }] : []),
            ...(isMultimodal ? [{
                field: 'multimodal',
                headerName: 'Content stored for retrieval',
                sortable: true,
                wrapText: true,
                autoHeight: true,
                cellStyle: {'word-break': 'break-word'},
                cellRenderer: RetrievalCellRenderer,
                valueFormatter: (params) => params.value?.content
            }] : []),
            {
                field: 'text',
                headerName: 'Chunk',
                minWidth: 200,
                ...textOptions
            },
            ...metadata,
            ...(hasOutline ? [{
                headerName: 'Section Outline',
                field: 'source.sectionOutline',
                ...textOptions
            }] : []),
            ...(hasPageRange ? [{
                headerName: 'Page Range',
                field: 'source.pageRange',
            }] : []),
        ];
        $scope.rowData = $scope.documents.map(document => {
            const row = {
                score: $filter('number')(document.score, 3),
                multimodal: document.multimodal,
                text: {
                    type: 'text',
                    content: document.text
                },
                metadata: document.metadata,
                source: document.source
            };

            return row;
        });
    }

    function supportsSimilarityScore() {
        const searchType = $scope.searchParams.searchType;
        const vectorStoreType = $scope.retrievableKnowledge?.vectorStoreType;
        return $scope.uiState.searchQuery.trim() && searchType !== 'MMR' && !(vectorStoreType === 'ELASTICSEARCH' && searchType === 'HYBRID');
    }
});

app.controller('KBSearchToolModalController', function($scope, $state, $stateParams, DataikuAPI, AgentToolService, WT1) {
    $scope.newAgentTool = {
        name: '',
        id: '',
        type: 'VectorStoreSearch',
        quickTestQuery: AgentToolService.getAgentToolQuickTestQueryForType('VectorStoreSearch')
    };

    $scope.create = function() {
        DataikuAPI.agentTools.createFromKB($stateParams.projectKey, $scope.newAgentTool, $stateParams.retrievableKnowledgeId).success(function(data) {
            WT1.event(
                'agent-tool-create', {
                    agentToolType: 'VectorStoreSearch',
                });
            $state.go("projects.project.agenttools.agenttool", {agentToolId: data.id})
            $scope.dismiss();
        }).error(setErrorInScope.bind($scope));
    };
});

app.controller('TestInPythonNotebookModalController', function ($scope, $stateParams, $state, DataikuAPI) {
    $scope.notebook = {
        name: `KB Test Notebook - ${$scope.retrievableKnowledge.name}`,
    };

    $scope.editInStudio = function () {
        const templateDesc = {
            language: 'python',
            type: 'RETRIEVABLE_KNOWLEDGE',
            id: '01-dku-test-kb-notebook',
            origin: 'BUILTIN',
        };

        DataikuAPI.jupyterNotebooks
            .newNotebookForKB($stateParams.projectKey, $scope.notebook.name, $scope.retrievableKnowledge.id, templateDesc)
            .then(function ({ data }) {
                $state.go("projects.project.notebooks.jupyter_notebook", {notebookId : data.name})
            })
            .catch(setErrorInScope.bind($scope));
    };
});

app.component('kbSearchResults', {
    bindings: {
        headers: '<',
        rows: '<'
    },
    template: `
        <div id="kb-search-results" class="kb-search-results ag-theme-alpine"></div>
    `,
    controller: function(AgGrid, $element) {
        const $ctrl = this;
        let rows;
        let headers;
        let gridApi;

        $ctrl.$onInit = function() {
            const $container = $element.find('#kb-search-results')[0];
            rows = $ctrl.rows;
            headers = $ctrl.headers;

            gridApi = AgGrid.createGrid($container, initGridOptions(headers, rows));
        };

        $ctrl.$onChanges = (changes) => {
            if (!gridApi) return;

            if (changes && changes.headers) {
                headers = $ctrl.headers;
                gridApi.setGridOption('columnDefs', headers);
            }

            if (changes && changes.rows) {
                rows = $ctrl.rows;
                gridApi.setGridOption('rowData', rows);
            }
        };

        function initGridOptions(headers, rows) {
            return {
                rowData: rows,
                columnDefs: headers,
                defaultColDef: {
                    sortable: true,
                    resizable: true,
                    filter: false
                },
                sideBar: {
                    toolPanels: [
                        {
                            id: 'columns',
                            labelDefault: 'Columns',
                            labelKey: 'columns',
                            iconKey: 'columns',
                            toolPanel: 'agColumnsToolPanel',
                            toolPanelParams: {
                                suppressRowGroups: true,
                                suppressValues: true,
                                suppressPivots: true,
                                suppressPivotMode: true,
                            }
                        }
                    ]
                },
                suppressPropertyNamesCheck: true,
                alwaysMultiSort: false,
                suppressCsvExport: true,
                suppressExcelExport: true,
                pinnedBottomRowData: [],
                tooltipShowDelay: 0,
                enableCellTextSelection: true,
                onFirstDataRendered: () => {
                    setTimeout(() => {
                        gridApi.autoSizeAllColumns();

                        const MAX_WIDTH = 600;
                        gridApi.getColumns().forEach(col => {
                            const width = col.getActualWidth();
                            if (width > MAX_WIDTH) {
                                gridApi.setColumnWidths([{
                                    key: col,
                                    newWidth: MAX_WIDTH
                                }]);
                            }
                        });
                    })
                },
            }
        }
    }
});

app.component('kbStatus', {
    bindings: {
        data: '<'
    },
    template: `
        <div class="knowledge-bank-status">
            Last build: {{ $ctrl.data.lastBuild.buildEndTime ? ($ctrl.data.lastBuild.buildEndTime | friendlyTimeDeltaShort) : 'N/A' }} <span ng-if="$ctrl.data.lastBuild.buildSuccess" class="text-success">(<span translate="PROJECT.DATASET.RIGHT_PANEL.DETAILS.SUCCESS">Success</span>)</span> 
            <span ng-if="!$ctrl.data.lastBuild.buildSuccess" class="text-error">(<span translate="PROJECT.DATASET.RIGHT_PANEL.DETAILS.FAILED">Failed</span>)</span>
            <span class="mleft16">
                Documents: {{ $ctrl.data.status.nbDocuments | stringWithFallback }}
            </span>
            <span class="mleft16">
                Chunks: {{ $ctrl.data.status.nbChunks | stringWithFallback }}
            </span>
        </div>
    `
});

app.component('vectorStoreSelector', {
    bindings: {
        retrievableKnowledge: '=',
        rkAtEmbedding: '<',
        disableIndexNaming: '<'
    },
    templateUrl: '/templates/retrievable-knowledge/vector-store-selector.html',
    controller: function ($scope, RetrievableKnowledgeUtils, VECTOR_STORE_TYPE_MAP, DataikuAPI) {
        const $ctrl = this;
        $scope.VECTOR_STORE_TYPE_MAP = VECTOR_STORE_TYPE_MAP;

        $scope.updateIndexName = RetrievableKnowledgeUtils.updateIndexName;
        $scope.hasConnection = RetrievableKnowledgeUtils.hasConnection;
        $scope.hasIndex = RetrievableKnowledgeUtils.hasIndex;

        $scope.availableVectorStoreConnectionsList = [];
        DataikuAPI.retrievableknowledge.getVectorStoreConnections().then(function({data}) {
            $scope.availableVectorStoreConnectionsList = data;
        }).catch(setErrorInScope.bind($scope));

        $scope.availableVectorStoreConnections = function () {
            if (!$scope.availableVectorStoreConnectionsList) return [];
            return $scope.availableVectorStoreConnectionsList.filter((conn) => conn.vectorStoreType === $ctrl.retrievableKnowledge.vectorStoreType);
        };
    },
});

app.service('RetrievableKnowledgeUtils', function () {
    function hasConnection(retrievableKnowledge) {
        return retrievableKnowledge && ['PINECONE', 'ELASTICSEARCH', 'AZURE_AI_SEARCH', 'VERTEX_AI_GCS_BASED'].indexOf(retrievableKnowledge.vectorStoreType) >= 0;
    }

    function hasIndex(retrievableKnowledge) {
        return retrievableKnowledge && ['ELASTICSEARCH', 'AZURE_AI_SEARCH', 'VERTEX_AI_GCS_BASED'].indexOf(retrievableKnowledge.vectorStoreType) >= 0;
    }

    function updateIndexName(retrievableKnowledge, disableIndexNaming) {
        if (hasIndex(retrievableKnowledge) && !retrievableKnowledge.indexName && !disableIndexNaming) {
            retrievableKnowledge.indexName = '${projectKey}_kb_' + retrievableKnowledge.id;
        }
    }

    function getRAGLLMSettings(newSettings = {}) {
        return Object.assign({
            maxDocuments: 4,
            searchType: "SIMILARITY",
            similarityThreshold: 0.5,
            mmrK: 20,
            mmrDiversity: .25,
            useAdvancedReranking: false,
            rrfRankConstant: 60,
            rrfRankWindowSize: 4
        }, newSettings);
    }

    return { hasConnection, hasIndex, updateIndexName, getRAGLLMSettings };
});

app.component("guardrailItem", {
    bindings: {
        "guardrailSetting": "=", // The setting object, see the FaithfulnessSettings java class for example
        "displayName": "<",
        "shortDescription": "<",
        "disabled": "<",
        "isMultimodal": "<",
    },
    template: `
<div class="control-group">
    <label class="guardrail-label control-label" for="{{ $ctrl.displayName + '-group' }}">{{ ($ctrl.isMultimodal ? 'Multimodal ' : '') + $ctrl.displayName }}
        <i class="dku-icon-question-circle-outline-16 dibvat detailed-metrics-table__metrics-info" style="color: #666666;" toggle="tooltip-right" title="{{ $ctrl.shortDescription }}"></i>
    </label>
    <div class="controls flex-row-fluid">
        <label class="dku-toggle">
            <input type="checkbox" ng-model="$ctrl.guardrailSetting.enabled" disabled-if-message="$ctrl.disabled ? 'Disabled' : null">
            <span></span>
        </label>
        <div class="mright4">
            <ng-container ng-show="$ctrl.guardrailSetting.enabled && !$ctrl.isMultimodal">
                <input id="{{ $ctrl.displayName + '-group' }}" type="range"
                    min="0" max="1" step="0.01"
                    ng-model="$ctrl.guardrailSetting.threshold">
            </ng-container>
        </div>
        <input type="number" class="input-fit-content" step="0.01" min="0" ng-show="$ctrl.guardrailSetting.enabled && !$ctrl.isMultimodal"
            max="1" ng-model="$ctrl.guardrailSetting.threshold"/>
    </div>
    <div class="controls guardrail-action-control" ng-show="$ctrl.guardrailSetting.enabled">
        <label>
            <basic-select
                items="$ctrl.handlingOptions"
                ng-model="$ctrl.guardrailSetting.handling"
                bind-value="handlingValue"
                bind-label="handlingDisplayValue">
                </basic-select>
            <div class="help-inline" ng-show="!$ctrl.isMultimodal">Action for scores below threshold</div>
            <div class="help-inline" ng-show="$ctrl.isMultimodal">{{ 'Action for low (score=0) ' + $ctrl.displayName }}</div>
        </label>
        <label>
            <textarea id="{{'answerOverwrite-' + $index}}" ng-show="$ctrl.guardrailSetting.handling == 'OVERWRITE_ANSWER'"
                        class="textarea--full-width"
                        ng-model="$ctrl.guardrailSetting.answerOverwrite"/>
        </label>
    </div>
</div>
`,
    controller: function() {
        const ctrl = this;
        ctrl.handlingOptions = [
            {
                "handlingValue": "FAIL",
                "handlingDisplayValue": "Fail"
            },
            {
                "handlingValue": "OVERWRITE_ANSWER",
                "handlingDisplayValue": "Overwrite answer"
            },
        ]
    }
});

app.component("raLlmFieldSummary", {
    bindings: {
        "field": "@",
        "value": "<",
    },
    template: `<span>{{ $ctrl.getName() }}</span>
`,
    controller: function(RALLM_RETRIEVAL_SEARCH_TYPE_MAP, RALLM_SEARCH_INPUT_STRATEGY_MAP) {
        const ctrl = this;
        const enums = {
            'search_type': RALLM_RETRIEVAL_SEARCH_TYPE_MAP,
            'search_input_strategy': RALLM_SEARCH_INPUT_STRATEGY_MAP,
        };
        ctrl.getName = function() {
            return enums[ctrl.field][ctrl.value]['title'];
        };
    }
});

})();
