(function(){
    'use strict';

    /* Misc additional directives for Shaker */

    var app = angular.module('dataiku.shaker.misc', ['dataiku.filters', 'platypus.utils']);

    /**
     * Directive using editableList with bootstrap typeahead on the inputs   
     * @param {Array}           ngModel                         - The list to bind to display.
     * @param {Function}        [onChange]                      - Callback called on change.
     * @param {addLabel}        [onAdd]                         - Label of the add button.
     * @param {boolean}         [typeAhead]                     - Text to display in the add button; Optional if disableAdd is true
     * @param {boolean}         [noChangeOnAdd=false]           - True to prevent the callback onChange to be called when an item is added.
    */
    app.directive('dkuListTypeaheadV2', function(){
        return {
            templateUrl: '/templates/widgets/dku-list-typeahead-v2.html',
            scope: {
                model: '=ngModel',
                onChange: '&',
                addLabel: '@',
                typeAhead: '=',
                noChangeOnAdd: '<'
            },
            link: function(scope) {
                scope.richModel = {}

                scope.$watch("model", function(nv){
                    if (!nv) return;
                    scope.richModel = scope.model.map(function(x){
                        return { value  : x }
                    });
                });

                if (scope.onChange) {
                    scope.callback = function(){
                        // Update in place instead of replacing, important because
                        // we don't want this to trigger another watch cycle in the
                        // editableList, which watches items non-deeply
                        scope.model.length = 0;
                        scope.richModel.forEach(function(x){
                            scope.model.push(x.value);
                        });
                        scope.onChange.bind(scope)({model : scope.model});
                    }
                }
                if (scope.typeAhead) {
                    scope.$watch("model", function() {
                        scope.remainingSuggests = listDifference(scope.typeAhead, scope.model);
                    }, true);
                }
            }
        };
    });
    
    app.directive('meaningSelect', function(ContextualMenu) {
    	return {
    		restrict: 'A',
    		template: '<div class="select-button">'
    					+'<button ng-click="openMeaningMenu($event)" class="btn  dku-select-button btn--secondary">'
                            +'<span class="filter-option pull-left">{{ngModel|meaningLabel}}</span>'
							+'&nbsp;'
							+'<span class="caret"></span>'
    					+'</button>'
    				 +'</div>',
            scope: {
                ngModel: '=',
                appConfig: '='
            },
            link: function($scope, element, attrs) {
    			$scope.menuState = {};
                $scope.meaningMenu = new ContextualMenu({
                    template: "/templates/shaker/select-meaning-contextual-menu.html",
                    cssClass : "column-header-meanings-menu",
                    scope: $scope,
                    contextual: false,
                    onOpen: function() {
                        $scope.menuState.meaning = true;
                    },
                    onClose: function() {
                        $scope.menuState.meaning = false;
                    }
                });
                $scope.openMeaningMenu = function($event) {
                    $scope.meaningMenu.openAtXY($event.pageX, $event.pageY);
                };

	            $scope.setMeaning = function(meaningId) {
                    $scope.ngModel = meaningId;
                    $(element).trigger('change');
	            };
    		}
    	}
    });

    app.directive('nextOnEnter', function() {
        return {
            score: true,
            priority: 90,
            restrict: 'A',
            link: function(scope, el, attrs) {
                var form = el[0].form;
                $(el).keyup(function (e) {
                    if (e.keyCode === 13) {
                        // on enter, we behave like for tab.
                        // and focus the next element of the form.
                        var tabbables = $(form).find(":tabbable");
                        var elId = tabbables.index(el);
                        if ( (elId >= 0) && (elId < tabbables.length -1) ) {
                            tabbables[elId+1].focus();
                        }
                        else {
                            // reached the last element... Just blur.
                            el.blur();
                        }
                    }
                });
            }
        };
    });
    app.directive('blurOnEnter', function() {
        return {
            score: true,
            priority: 90,
            restrict: 'A',
            link: function(scope, el, attrs) {
                var form = el[0].form;
                $(el).keyup(function (e) {
                    if (e.keyCode === 13) {
                            el.blur();
                    }
                });
            }
        };
    });
    app.directive('blurOnEnterAndEsc', function() {
        return {
            score: true,
            priority: 90,
            restrict: 'A',
            link: function(scope, el, attrs) {
                var form = el[0].form;
                $(el).keyup(function (e) {
                    if (e.keyCode === 13 || e.keyCode === 27) {
                            el.blur();
                    }
                });
            }
        };
    });

    app.directive('shakerProcessorStep', function($filter, CachedAPICalls, ShakerProcessorsInfo, ShakerProcessorsUtils, RecipesUtils){
        return {
            templateUrl: '/templates/shaker/processor-step.html',
            replace: true,
            /* We have to use prototypal inheritance scope instead of isolate scope because of bug
            https://github.com/angular/angular.js/issues/1941
            *
            * And also because we access a lot of the scope
            *
            * Requires in scope:
            *  - step
            *  - columns (array[string])
            *  - $index
            */
            scope:true,
            link: function(scope, element, attrs){
                // TODO: also linked to the isolate scope issue

                // at instantiation, always scroll to element
                $(element).find(".content").get(0).scrollIntoView(true);

                scope.remove = function(step) {
                    $('.processor-help-popover').popover('hide');//hide any displayed help window
                    scope.removeStep(step.step);
                    keepInputsInSync();
                };
                
                scope.deleteHelper = function(obj, key) {
                    delete obj[key];
                };

                scope.isStepActive = function() {
                	return scope.step == scope.currentStep;
                }

                const keepInputsInSync = () => {
                    if (!scope.recipe || !scope.shaker) return;
                    RecipesUtils.removeInputsForRole(scope.recipe, "scriptDeps");
                    RecipesUtils.removeInputsForRole(scope.recipe, "reference");

                    const addInputsFromSteps = (steps) => {
                        steps.forEach((step) => {
                            if (step.metaType === 'GROUP') {
                                addInputsFromSteps(step.steps);
                            } else {
                                // Keep reference datasets in sync with recipe config
                                if (step.params && step.params.useDatasetForMapping && step.params.mappingDatasetRef && RecipesUtils.getInput(scope.recipe, "reference", step.params.mappingDatasetRef) === null) {
                                    RecipesUtils.addInput(scope.recipe, "reference", step.params.mappingDatasetRef);
                                }
                                // Keep joined datasets in sync with recipe config
                                if (step.params && step.params.rightInput && RecipesUtils.getInput(scope.recipe, "scriptDeps", step.params.rightInput) === null) {
                                    RecipesUtils.addInput(scope.recipe, "scriptDeps", step.params.rightInput);
                                }
                            }
                        });
                    }
                    addInputsFromSteps(scope.shaker.steps);
                }

                /**
                 * This is the method called by all forms when a value is changed by the user.
                 * It triggers validation of the step, and, if the step is valid, the refresh.
                 *
                 * This handles a special case: processors that are "new", i.e. that have never been valid.
                 * For them, we don't display their 'in-error' state while they have not been valid at least
                 * once
                 */
                scope.checkAndRefresh = function() {
                    if (!scope.step.$stepState) {
                        scope.step.$stepState = {};
                    }
                    const state = scope.step.$stepState;

                    state.frontError = scope.validateStep(scope.step);

                    if (state.frontError && state.isNew){
                        // Don't do anything for a new processor that is still invalid
                    } else if (!state.frontError && state.isNew) {
                        // No error in new processor -> so it's not new anymore, and we can refresh
                        state.isNew = false;
                        scope.autoSaveAutoRefresh();
                        keepInputsInSync();
                    } else if (state.frontError && !state.isNew) {
                        // Error in non-new processor: Don't refresh
                    } else if (!state.frontError && !state.isNew) {
                        // No error in non-new processor -> the 'normal' case
                        scope.autoSaveAutoRefresh();
                        keepInputsInSync();
                    }
                };

                CachedAPICalls.processorsLibrary.success(function(processors){
                    scope.processors = processors;
                    scope.processor = $filter('processorByType')(scope.processors, scope.step.type);

                    var e = ShakerProcessorsInfo.get(scope.step.type);
                    if (angular.isDefined(e) && angular.isDefined(e.postLinkFn)){
                        e.postLinkFn(scope, element);
                    }

                    scope.$watch("step", function(step, ov) {
                        if (!step.$stepState) {
                            step.$stepState = {};
                        }

                        step.$stepState.frontError = scope.validateStep(scope.step);

                        scope.description = ShakerProcessorsUtils.getStepDescription(scope.processor, step.type, step.params);
                        scope.icon = ShakerProcessorsUtils.getStepIcon(step.type, step.params, 16); // I don't think it's used anywhere, in doubt default to 16px
                    }, true);
                });

                scope.types = Object.keys(scope.appConfig.meanings.labelsMap);

                scope.isMandatory = function(paramName) {
                    const param = scope.processor.params.find((param) => param.name === paramName);
                    return param && param.mandatory;
                };
            }
        };
    });

    app.directive('shakerGroupStep', function($filter, CachedAPICalls, Fn, $timeout){
        return {
            templateUrl: '/templates/shaker/group-step.html',
            replace: true,
            /*
            * Requires in scope:
            *  - step
            *  - columns (array[string])
            *  - $index
            */
            scope:true,
            link: function(scope, element, attrs){
                scope.remove = function(step) {
                    $('.processor-help-popover').popover('hide');//hide any displayed help window
                    scope.removeStep(step.step);
                };

                scope.hasMatchingSteps = function() {
                    return scope.step.steps.filter(Fn.prop('match')).length > 0;
                }

                scope.isGroupActive = function() {
                   return !scope.isCollapsed() || (scope.hasMatchingSteps() && !scope.step.closeOnMatch);
                }

                scope.toggleGroup = function() {
                    if (scope.isCollapsed() && scope.isGroupActive()) {
                        scope.step.closeOnMatch = !scope.step.closeOnMatch;
                    } else {
                        scope.toggle();
                    }
                }

                scope.$on('openShakerGroup', function(e, step) {
                    if (scope.step === step && scope.isCollapsed()) {
                        scope.toggle();        
                    }
                });

                scope.$watch('groupChanged.addedStepsTo', function() {
                    if (scope.groupChanged.addedStepsTo === scope.step) {
                        if (!scope.isGroupActive()) {
                            scope.toggleGroup();
                        }
                        scrollToStep();
                    } else if (scope.groupChanged.removedStepsFrom.indexOf(scope.step) > -1 && scope.step.steps.length === 0) {
                        if (scope.isGroupActive()) {
                            scope.toggleGroup();
                        }
                    }
                });

                scope.$watch("step.forceOpenOnCreation", function(nv, ov) {
                    if (nv === true) {
                        scope.toggleGroup();
                    }
                })


                scrollToStep();

                function scrollToStep() {
                    // at instantiation, always scroll to element and start editing
                    $timeout(function() {
                        $(element).get(0).scrollIntoView({
                            behavior: 'auto',
                            block: 'center',
                            inline: 'center'
                        });
                    });
                }

                function hasActuallyChanged(newSteps, oldSteps) {
                    if (!newSteps && !oldSteps) {
                        return false;
                    }

                    if ((newSteps && !oldSteps) || (!newSteps && oldSteps)) {
                        return true;
                    }

                    if (newSteps.length !== oldSteps.length) {
                        return true;
                    }

                    for (let i = 0; i < newSteps.length; i++) {
                        const newStep = newSteps[i];
                        const oldStep = oldSteps[i];
                        if (newStep.type !== oldStep.type
                            || newStep.metaType !== oldStep.metaType
                            || JSON.stringify(newStep.params) !== JSON.stringify(oldStep.params)) {
                                return true;
                            }
                    }
                    return false;
                }

            }
        };
    });


    app.directive('shaker', function($timeout) {
        return {
            restrict: 'C',
            link: function(scope, element, attrs){
                scope.$watch('shaker.explorationFilters.length', function(nv, ov){
                    if (nv && nv > 1) {
                        scope.$broadcast('tabSelect', 'filters');
                    }
                });
                scope.$watch('shaker.steps.length', function(nv, ov){
                    scope.$broadcast('tabSelect', 'script');
                    if (nv > ov) {
                        let ul = $(element).find('ul.steps.accordion');
                        let items = ul.children();
                        let scrollIndex = scope.pasting ? findFirstCopy(scope.shaker.steps) : items.length - 1;

                        $timeout(function() {
                            let addedElement = ul.children().get(scrollIndex);
                            // scroll to element
                            if (addedElement) {
                                addedElement.scrollIntoView({ 'block': 'center' });
                            }
                        });
                    }
                });

                function findFirstCopy(steps) {
                    return steps.findIndex(_ => {
                        return (_.$stepState && _.$stepState.isNewCopy) || (_.steps && findFirstCopy(_.steps) >= 0);
                    });
                }

                scope.clearFFS = function(){
                    scope.ffs = [];
                };
            }
        };
    });

    // small directive meant to replace "-1" by "not set", since -1 is used as a marker of "no limit" in the backend
    // Also handles megabytes
    app.directive('optionalMaxSizeMb', function() { // Warning : this directive cannot handle nulls -> ng-if above it
        return {
            scope: true,
            restrict: 'A',
            link: function($scope, el, attrs) {
                $scope.$optionalState = {};
                var initSize = $scope.$eval(attrs.optionalMaxSizeMb);
                $scope.$optionalState.hasMaxSize = initSize >= 0;
                if ($scope.$optionalState.hasMaxSize) {
                    $scope.$optionalState.maxSize = initSize / (1024 * 1024);
                }
                $scope.$watch('$optionalState.hasMaxSize', function(nv, ov) {
                    if (!$scope.$optionalState.hasMaxSize) {
                        $scope.$eval(attrs.optionalMaxSizeMb + " = -1");
                    } else {
                        /* Put a sane default value */
                        if ($scope.$optionalState.maxSize === undefined || $scope.$optionalState.maxSize < 0) {
                            $scope.$optionalState.maxSize = 1;
                        }
                        $scope.$eval(attrs.optionalMaxSizeMb + " = " + ($scope.$optionalState.maxSize * 1024 * 1024));
                    }
                });
                $scope.$watch('$optionalState.maxSize', function(nv, ov) {
                    if (nv === undefined) return;
                    $scope.$eval(attrs.optionalMaxSizeMb + " = " + ($scope.$optionalState.maxSize * 1024 * 1024));
                });
            }
        };
    });


    var services = angular.module('dataiku.services');

    services.factory('ShakerPopupRegistry', function(Logger) {
        var callbacks = [];
        function register(dismissFunction) {
            callbacks.push(dismissFunction);
        }
        function dismissAll() {
            callbacks.forEach(function(f) {
                try {
                    f();
                } catch (e) {
                    Logger.warn("failed to dismiss shaker popup", e);
                }
            });
            callbacks = [];
        }

        function dismissAllAndRegister(dismissFunction) {
            dismissAll();
            register(dismissFunction);
        }

        return {
            register: register,
            dismissAll: dismissAll,
            dismissAllAndRegister: dismissAllAndRegister
        }
    });

    // to put on the element in which the custom formula editor is supposed to be shown. It provides a function
    // that can be passed to CreateCustomElementFromTemplate in order to insert the formula editor in the DOM,
    // instead of the usual mechanism (which is: append to <body>). This directive sets a boolean in the scope
    // to indicate the formula editor is open (so that you can hide other stuff while it's open, for example)
    app.directive('customFormulaZone', function($rootScope) {
        return {
            scope: true,
            restrict: 'A',
            link: function($scope, el, attrs) {
            	var type = attrs.customFormulaZone || 'replace';
            	$scope.customFormulaEdition.editing = 0;

            	$scope.customFormulaEdition.displayCustomFormula = function(formulaElement) {
            		$scope.customFormulaEdition.editing += 1;

                	$(formulaElement).on("remove", function() {
                		$scope.customFormulaEdition.editing -= 1;
                		if ($scope.customFormulaEdition.editing == 0 ) {
                			if ( type == 'replace' ) {
                				$(el).removeClass("replaced-by-formula");
                			}
                		}
                		$scope.customFormulaEdition.reflowStuff();
                	});

                	if (type == 'replace') {
                		$(el).after(formulaElement);
                		if ( $scope.customFormulaEdition.editing == 1 ) {
                			$(el).addClass("replaced-by-formula");
                		}
                	} else {
                		$(el).append(formulaElement);
                	}
            	};
            }
        };
    });

    app.directive('resizable', ['$document', function($document) {
        return {
            restrict: 'A',
            link: function(scope, element, attrs) {
                    const resizeDirection = attrs.resizable || 'both';
                    const { minHeight, maxHeight, minWidth, maxWidth } = attrs;
                    if (minHeight) {
                        element.css('min-height', minHeight);
                    }
                    if (maxHeight) {
                        element.css('max-height', maxHeight);
                    }
                    if (minWidth) {
                        element.css('min-width', minWidth);
                    }
                    if (maxWidth) {
                        element.css('max-width', maxWidth);
                    }
                    let icon = 'dku-icon-arrow-expand-16';
                    switch (resizeDirection) {
                        case 'vertical':
                            icon = 'dku-icon-arrow-double-vertical-16';
                            break;
                        case 'horizontal':
                            icon = 'dku-icon-arrow-double-horizontal-16';
                            break;
                        default:
                            break;
                    }

                    const resizeHandle = angular.element(`
                        <div class="resize-handle">
                            <i class="${icon}"></i>
                        </div>
                    `);
                    element.append(resizeHandle);

                    let startX, startY, startWidth, startHeight;

                    resizeHandle.on('mousedown', function(event) {
                        event.preventDefault();
                        event.stopPropagation();
                        startX = event.pageX;
                        startY = event.pageY;
                        startWidth = element[0].offsetWidth;
                        startHeight = element[0].offsetHeight;
                        $document.on('mousemove', mousemove);
                        $document.on('mouseup', mouseup);
                    });

                    function mousemove(event) {
                        const dx = event.pageX - startX;
                        const dy = event.pageY - startY;

                        if (resizeDirection === 'horizontal' || resizeDirection === 'both') {
                            element.css('width', (startWidth + dx) + 'px');
                        }
                        if (resizeDirection === 'vertical' || resizeDirection === 'both') {
                            element.css('height', (startHeight + dy) + 'px');
                        }
                    }

                    function mouseup() {
                        $document.off('mousemove', mousemove);
                        $document.off('mouseup', mouseup);
                    }
                }        
            };
    }]);

    // dumb directive to put somewhere above the element providing the custom formula, and the element receiving it.
    // Its purpose is to bridge the scopes of the step using the formula editor and the place where the formula
    // editor is shown (they're most likely in different panes on the screen)
    app.directive('hasCustomFormulaZone', function() {
        return {
            scope: true,
            restrict: 'A',
            link: function($scope, el, attrs) {
            	$scope.customFormulaEdition = {
            		reflowStuff : function() {$scope.$broadcast("reflow");} // reflow just inside the shaker screen, not the entire dss
            	};
            }
        };
    });

})();
