(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    // (!) This service previously was in static/dataiku/js/simple_report/maps/administrative.js

    app.factory('AdministrativeMapChart', function($timeout, CHART_VARIANTS, ChartFormatting, ChartFeatures, _MapCharts, ChartLegendUtils, ColorUtils, ChartColorScales, ChartLabels) {
        return function($container, chartDef, data, chartHandler) {
            const geo = JSON.parse(data.geoJson);

            const aggrVals = function(aggregId) {
                let feat, featIdx;
                const arr = [];
                for (featIdx in geo.features) {
                    feat = geo.features[featIdx];
                    if (!_.isNil(feat.properties[aggregId])) {
                        arr.push(feat.properties[aggregId]);
                    }
                }
                return arr;
            };

            const aggrBounds = function(aggregId) {
                const arr = aggrVals(aggregId);
                return [d3.min(arr), d3.max(arr)];
            };

            let colorScale, singleColor;
            if (chartDef.colorMeasure.length) {
                colorScale = ChartColorScales.continuousColorScale(chartDef.colorOptions, aggrBounds('color')[0], aggrBounds('color')[1], aggrVals('color'), false, chartHandler.getChartTheme());
                colorScale.type = 'MEASURE';
            } else {
                singleColor = ColorUtils.toRgba(chartDef.colorOptions.singleColor, chartDef.colorOptions.transparency);
            }

            ChartLegendUtils.initLegend(chartDef, null, chartHandler.legendsWrapper, colorScale);

            if (colorScale) {
                chartHandler.legendsWrapper.getLegend(0).formatter = ChartFormatting.getForLegend(chartDef.colorMeasure[0], colorScale.innerScale.domain());;
            }

            ChartLegendUtils.drawLegend(chartDef, chartHandler, $container).then(function() {
                _MapCharts.adjustLegendPlacement(chartDef, $container);

                $timeout(() => {
                    const elt = _MapCharts.getOrCreateMapContainer($container);
                    const map = _MapCharts.createMapIfNeeded(elt, chartHandler.chartSpecific, chartDef);
                    if (elt.data('leaflet-data-layer')) {
                        map.removeLayer(elt.data('leaflet-data-layer'));
                    }
                    _MapCharts.repositionMap(map, elt, data);

                    const sizeScale = d3.scale.sqrt().range([chartDef.bubblesOptions.defaultRadius, chartDef.bubblesOptions.defaultRadius*5]).domain(aggrBounds('size'));

                    function onEachFeature(feature, layer) {
                        if (!ChartFeatures.shouldDisplayTooltips(chartHandler.noTooltips, chartDef.tooltipOptions)) {
                            return;
                        }

                        let html = '<h4>' + feature.properties.label+'</h4>';

                        if (!_.isNil(feature.properties.color)) {
                            html += sanitize(ChartLabels.getLongMeasureLabel(chartDef.colorMeasure[0]));
                            html += ': <strong>';
                            html += sanitize(ChartFormatting.getForIsolatedNumber(chartDef.colorMeasure[0])(feature.properties.color)) + '</strong><br />';

                        }
                        if (!_.isNil(feature.properties.size)) {
                            html += sanitize(ChartLabels.getLongMeasureLabel(chartDef.sizeMeasure[0]));
                            html += ': <strong>';
                            html += sanitize(ChartFormatting.getForIsolatedNumber(chartDef.sizeMeasure[0])(feature.properties.size)) + '</strong><br />';
                        }
                        if (!_.isNil(feature.properties.count)) {
                            html += 'Value count' + ': <strong>';
                            html += ChartFormatting.getForIsolatedNumber()(feature.properties.count) + '</strong><br />';
                        }

                        if (chartDef.tooltipMeasures.length > 0) {
                            html += '<hr/>';
                        }
                        chartDef.tooltipMeasures.forEach(function(measure, j) {
                            const tooltipFormatter = ChartFormatting.getForIsolatedNumber(measure);
                            html += sanitize(ChartLabels.getLongMeasureLabel(measure)) + ': <strong>' + sanitize(tooltipFormatter(feature.properties[j])) + '</strong><br/>';
                        });

                        //Classname is a locator for the integration tests - Leaflet API doesn't allow us to add an id
                        layer.bindPopup(html, { className: 'qa-chart-tooltip' });
                    }
                    let layer;

                    if (chartDef.variant == CHART_VARIANTS.filledMap) {
                        chartDef.sizeMeasure = [];
                        const myStyle = function(feature) {
                            return {
                                'color': singleColor || colorScale(feature.properties['color']),
                                'fillColor': singleColor || colorScale(feature.properties['color']),
                                'fillOpacity' : chartDef.fillOpacity,
                                'weight': 1,
                                'opacity': 1
                            };
                        };
                        layer = L.geoJson(geo.features, {
                            style: myStyle,
                            onEachFeature : onEachFeature
                        });
                        map.addLayer(layer);
                    } else {
                        layer = L.geoJson(geo.features, {
                            pointToLayer : function(feature, latlng) {
                                const size = feature.properties['size'] != null ? sizeScale(feature.properties['size']) : chartDef.bubblesOptions.defaultRadius;
                                const color = singleColor || (feature.properties['color'] != null ? colorScale(feature.properties['color']) : '#666');

                                return L.circleMarker(latlng, {
                                    radius : size,
                                    'color': color,
                                    'fillColor': color,
                                    'opacity' : 0.85,
                                    'fillOpacity' : 0.85
                                });
                            },
                            onEachFeature : onEachFeature
                        });
                        map.addLayer(layer);
                    }

                    elt.data('leaflet-data-layer', layer);
                });
            }).finally(function() {
                /*
                 * Signal to the callee handler that the chart has been loaded.
                 * Dashboards use it to determine when all insights are completely loaded.
                 */
                if (typeof(chartHandler.loadedCallback) === 'function') {
                    $timeout(() => {
                        chartHandler.loadedCallback();
                    });
                }
            });
        };
    });
})();
