(function() {
    'use strict';

    angular.module('dataiku.charts')
        .factory('PieChartUtils', PieChartUtils);

    // (!) This service previously was in static/dataiku/js/simple_report/pies/pie.js
    function PieChartUtils($filter) {
        const that = {
            prepareData: function(chartDef, chartData) {
                const colorLabels = chartData.getAxisLabels('color') || [null],
                    facetLabels = chartData.getAxisLabels('facet') || [null],
                    animationLabels = chartData.getAxisLabels('animation') || [null],
                    pie = d3.layout.pie().value(function(d) {
                        return d.value;
                    });

                const animationData = { frames: [] };
                animationLabels.forEach(function(animationLabel, a) {
                    chartData.fixAxis('animation', a);

                    const frameData = { facets: [] };
                    facetLabels.forEach(function(facetLabel, f) {
                        chartData.fixAxis('facet', f);

                        const facetData = { slices: [], total: 0 };
                        colorLabels.forEach(function(colorLabel, c) {
                            chartData.fixAxis('color', c);

                            chartDef.genericMeasures.forEach(function(measure, m) {
                                const d = chartData.aggr(m).get();
                                if (d < 0) {
                                    throw new ChartIAE('Cannot represent negative values on a pie chart. Please use another chart.');
                                }

                                facetData.slices.push({
                                    color: c,
                                    measure: m,
                                    facet: f,
                                    animation: a,
                                    count: chartData.getCount(),
                                    valuesInChartDisplayOptions: measure.valuesInChartDisplayOptions,
                                    value: d
                                });

                                facetData.total += d;
                            });

                            if (facetData.total > 0) {
                                facetData.slices = $filter('orderBy')(facetData.slices, 'value', true);
                                facetData.pie = pie(facetData.slices);
                            }
                        });
                        frameData.facets.push(facetData);
                    });
                    animationData.frames.push(frameData);
                });

                return animationData;
            },

            hideOverlappingLabels: function(slices, values, total) {
                if (slices.length < 2) {
                    return;
                }

                const displayedLabelSlices = [];
                const displayedLabelValues = [];
                slices.forEach(function(slice, i) {
                    if ($(slice).find('text').css('display') !== 'none') {
                        const sliceValue = values[i];
                        const nextSliceIndex = (i === slices.length - 1) ? 0 : i + 1;
                        const nextSlice = slices[nextSliceIndex];
                        if (that.slicesOverlap(slice, nextSlice)) {
                            const nextSliceValue = values[nextSliceIndex];
                            const diff = (sliceValue >= nextSliceValue) ? sliceValue - nextSliceValue : nextSliceValue - sliceValue;
                            const diffAngle = 360 * diff / total;
                            if (diffAngle < 5) {
                                $(slice).find('text').hide();
                                $(nextSlice).find('text').hide();
                            } else {
                                const smallerSlice = (sliceValue > nextSliceValue) ? nextSlice : slice;
                                $(smallerSlice).find('text').hide();
                                if (sliceValue > nextSliceValue) {
                                    displayedLabelSlices.push(slice);
                                    displayedLabelValues.push(sliceValue);
                                }
                            }
                        } else {
                            displayedLabelSlices.push(slice);
                            displayedLabelValues.push(sliceValue);
                        }
                    }
                });
                if (slices.length !== displayedLabelSlices.length) {
                    that.hideOverlappingLabels(displayedLabelSlices, displayedLabelValues);
                }
            },

            slicesOverlap: function(slice1, slice2) {
                //coordinates or first slice
                const top1 = $(slice1).find('text').offset().top;
                const left1 = $(slice1).find('text').offset().left;
                const bottom1 = top1 + $(slice1).find('text')[0].getBoundingClientRect().height;	//using getBoundingClientRect b/c jquery's height() function does not work on svg elements with FF
                const right1 = left1 + $(slice1).find('text')[0].getBoundingClientRect().width;
                //coordinates of second slice
                const top2 = $(slice2).find('text').offset().top;
                const left2 = $(slice2).find('text').offset().left;
                const bottom2 = top2 + $(slice2).find('text')[0].getBoundingClientRect().height;
                const right2 = left2 + $(slice2).find('text')[0].getBoundingClientRect().width;
                //Are slices overlapping horizontally ?
                let hOverlapping;
                if (left1 <= left2) {
                    hOverlapping = right1 >= left2;
                } else {
                    hOverlapping = right2 >= left1;
                }
                //Are slices overlapping vertically ?
                let vOverlapping;
                if (top1 <= top2) {
                    vOverlapping = bottom1 >= top2;
                } else {
                    vOverlapping = bottom2 >= top1;
                }
                //Overlapping is true if slices are overlapping horizontally and vertically
                return hOverlapping && vOverlapping;
            }
        };

        return that;
    }
})();
