/* global describe, beforeEach, module, inject, it, expect */

// See also FilesSelectionRulesTest - backend equivalent
describe("FilePatternUtils", function() {
    beforeEach(() => {
        module('dataiku.services', function() {});
    });

    describe("fileNameGlobRegExp", function() {
        it("fileNameGlobRegExp handles exact", function() {
            inject(function(FilePatternUtils) {
                expect(FilePatternUtils.fileNameGlobRegExp("hell0.csv").test("hell0.csv")).toBe(true);
                expect(FilePatternUtils.fileNameGlobRegExp("hell1.csv").test("hell0.csv")).toBe(false);
            });
        });
    
        it("fileNameGlobRegExp matches with * and ** wildcards", function() {
            inject(function(FilePatternUtils) {
                expect(FilePatternUtils.fileNameGlobRegExp("*").test("hell0.csv")).toBe(true);
                expect(FilePatternUtils.fileNameGlobRegExp("*.*").test("hell0.csv")).toBe(true);
                expect(FilePatternUtils.fileNameGlobRegExp("*.csv").test("hell0.csv")).toBe(true);
                expect(FilePatternUtils.fileNameGlobRegExp("*.csv.gz").test("hell0.csv.gz")).toBe(true);
                expect(FilePatternUtils.fileNameGlobRegExp("hel*.csv").test("hell0.csv")).toBe(true);
                expect(FilePatternUtils.fileNameGlobRegExp("h*0.c*v").test("hell0.csv")).toBe(true);
                expect(FilePatternUtils.fileNameGlobRegExp("h*0.c*v.*").test("hell0.csv.gz")).toBe(true);
                expect(FilePatternUtils.fileNameGlobRegExp("file*txt").test("file.txt")).toBe(true);
                expect(FilePatternUtils.fileNameGlobRegExp("**").test("hell0.csv")).toBe(true);
            });
        });
    
        it("fileNameGlobRegExp MISmatches with * wildcards", function() {
            inject(function(FilePatternUtils) {
                expect(FilePatternUtils.fileNameGlobRegExp("*.txt").test("file")).toBe(false);
                expect(FilePatternUtils.fileNameGlobRegExp("*.txt").test("file.csv")).toBe(false);
                expect(FilePatternUtils.fileNameGlobRegExp("*.txt").test("file.txt.gz")).toBe(false);
                expect(FilePatternUtils.fileNameGlobRegExp("h*1.c*v").test("hell0.csv")).toBe(false);
                expect(FilePatternUtils.fileNameGlobRegExp("h*1.c*v.*").test("hell0.csv.gz")).toBe(false);
                expect(FilePatternUtils.fileNameGlobRegExp("fild*txt").test("file.txt")).toBe(false);
            });
        });
    
        it("fileNameGlobRegExp handles ? wildcards", function() {
            inject(function(FilePatternUtils) {
                expect(FilePatternUtils.fileNameGlobRegExp("hell?.csv").test("hell0.csv")).toBe(true);
                expect(FilePatternUtils.fileNameGlobRegExp("hell0?csv").test("hell0.csv")).toBe(true);
                expect(FilePatternUtils.fileNameGlobRegExp("?ell0.csv").test("hell0.csv")).toBe(true);
                expect(FilePatternUtils.fileNameGlobRegExp("hell0.cs?").test("hell0.csv")).toBe(true);
                expect(FilePatternUtils.fileNameGlobRegExp("hell?.cs?").test("hell0.csv")).toBe(true);
                expect(FilePatternUtils.fileNameGlobRegExp("hel?0.csv").test("hell1.csv")).toBe(false);
            });
        });
    });
 
    describe("fullPathGlobRegExp", function() {
        it("fullPathGlobRegExp handles exact", function() {
            inject(function(FilePatternUtils) {
                expect(FilePatternUtils.fullPathGlobRegExp("hell0.csv").test("hell0.csv")).toBe(true);
                expect(FilePatternUtils.fullPathGlobRegExp("hell1.csv").test("hell0.csv")).toBe(false);
                expect(FilePatternUtils.fullPathGlobRegExp("folder1/folder2/hell0.csv").test("folder1/folder2/hell0.csv")).toBe(true);
                expect(FilePatternUtils.fullPathGlobRegExp("folder1/folder2/hell1.csv").test("folder1/folder2/hell0.csv")).toBe(false);
            });
        });

        it("fullPathGlobRegExp leading and trailing slashes", function() {
            // Note the backend FileSelectionRule.java class will normalise the input path before testing against the glob pattern
            // and the front end equivalent also does this before it calls FilePatternUtils too
            // Meaning any leading slashes in the path that is tested will have been removed - so in the UI things will behave differently as noted.
            inject(function(FilePatternUtils) {
                expect(FilePatternUtils.fullPathGlobRegExp("/hell0.csv").test("/hell0.csv")).toBe(true);   //will be false in UI (input path normalised)
                expect(FilePatternUtils.fullPathGlobRegExp("/hell0.csv").test("hell0.csv")).toBe(false);
                expect(FilePatternUtils.fullPathGlobRegExp("hell0.csv").test("/hell0.csv")).toBe(false);

                expect(FilePatternUtils.fullPathGlobRegExp("/*").test("/hell0.csv")).toBe(true); //will be false in UI (input path normalised)
                expect(FilePatternUtils.fullPathGlobRegExp("/*").test("//hell0.csv")).toBe(false); //Input path is impossible, should not come up
                expect(FilePatternUtils.fullPathGlobRegExp("/*/*").test("folder/hell0.csv")).toBe(false);
                expect(FilePatternUtils.fullPathGlobRegExp("/*").test("hell0.csv")).toBe(false);

                expect(FilePatternUtils.fullPathGlobRegExp("*").test("/hell0.csv")).toBe(false);

                expect(FilePatternUtils.fullPathGlobRegExp("/*/*").test("/folder/hell0.csv")).toBe(true); //will be false in UI (input path normalised)
                expect(FilePatternUtils.fullPathGlobRegExp("/*/*").test("folder/hell0.csv")).toBe(false);


                expect(FilePatternUtils.fullPathGlobRegExp("hell0/").test("hell0")).toBe(false);
                expect(FilePatternUtils.fullPathGlobRegExp("hell0/").test("hell0")).toBe(false);
                expect(FilePatternUtils.fullPathGlobRegExp("hell0").test("hell0/")).toBe(false); //Input path is impossible/odd anyway
                expect(FilePatternUtils.fullPathGlobRegExp("hell0/").test("hell0/")).toBe(true); //Input path is impossible/odd anyway
            });
        });

        it("fullPathGlobRegExp matches with * wildcards - just file", function() {
            inject(function(FilePatternUtils) {
            
                expect(FilePatternUtils.fullPathGlobRegExp("*").test("hell0.csv")).toBe(true);
                expect(FilePatternUtils.fullPathGlobRegExp("*.*").test("hell0.csv")).toBe(true);
                expect(FilePatternUtils.fullPathGlobRegExp("*.csv").test("hell0.csv")).toBe(true);
                expect(FilePatternUtils.fullPathGlobRegExp("*.csv.gz").test("hell0.csv.gz")).toBe(true);
                expect(FilePatternUtils.fullPathGlobRegExp("hel*.csv").test("hell0.csv")).toBe(true);
                expect(FilePatternUtils.fullPathGlobRegExp("h*0.c*v").test("hell0.csv")).toBe(true);
                expect(FilePatternUtils.fullPathGlobRegExp("h*0.c*v.*").test("hell0.csv.gz")).toBe(true);
                expect(FilePatternUtils.fullPathGlobRegExp("file*txt").test("file.txt")).toBe(true);

                expect(FilePatternUtils.fullPathGlobRegExp("**/*").test("hell0.csv")).toBe(true);
                expect(FilePatternUtils.fullPathGlobRegExp("**").test("hell0.csv")).toBe(true);
                
            });
        });

        it("fullPathGlobRegExp matches with * and ** wildcards - on paths", function() {
            inject(function(FilePatternUtils) {
            
                expect(FilePatternUtils.fullPathGlobRegExp("folder1/*/hell0.csv").test("folder1/folder2/hell0.csv")).toBe(true);
                expect(FilePatternUtils.fullPathGlobRegExp("folder1/f*2/hell0.csv").test("folder1/folder2/hell0.csv")).toBe(true);
                expect(FilePatternUtils.fullPathGlobRegExp("folder1/**/hell0.csv").test("folder1/folder2/hell0.csv")).toBe(true);
                
                expect(FilePatternUtils.fullPathGlobRegExp("folder1/**/hell0.csv").test("folder1/folder2/folder3/hell0.csv")).toBe(true);
                expect(FilePatternUtils.fullPathGlobRegExp("**/folder2/**/hell0.csv").test("folder1/folder2/folder3/hell0.csv")).toBe(true);

                expect(FilePatternUtils.fullPathGlobRegExp("a/**/b").test("a/b")).toBe(true);
                expect(FilePatternUtils.fullPathGlobRegExp("**/folder2/**/hell0.csv").test("folder2/hell0.csv")).toBe(true);
            
                expect(FilePatternUtils.fullPathGlobRegExp("dir/**").test("dir/file")).toBe(true);
                expect(FilePatternUtils.fullPathGlobRegExp("dir/**").test("dir/folder/hello.txt")).toBe(true);

                expect(FilePatternUtils.fullPathGlobRegExp("**/hell0.csv").test("folder1/folder2/hell0.csv")).toBe(true);
                expect(FilePatternUtils.fullPathGlobRegExp("**/f*2/**/hell0.csv").test("folder1/folder2/folder3/hell0.csv")).toBe(true);
                expect(FilePatternUtils.fullPathGlobRegExp("**/*.*").test("dir/file.txt")).toBe(true);

                expect(FilePatternUtils.fullPathGlobRegExp("**/*").test("folder1/folder2/hell0.csv")).toBe(true);
                expect(FilePatternUtils.fullPathGlobRegExp("**").test("folder1/folder2/hell0.csv")).toBe(true);
            });
        });


        it("fullPathGlobRegExp MISmatches with * wildcards - on paths", function() {
            inject(function(FilePatternUtils) {
                expect(FilePatternUtils.fullPathGlobRegExp("folder2/*/hell0.csv").test("folder1/folder2/hell0.csv")).toBe(false);
                expect(FilePatternUtils.fullPathGlobRegExp("folder1/*/hell0.csv").test("folder1/folder2/folder3/hell0.csv")).toBe(false);
                expect(FilePatternUtils.fullPathGlobRegExp("folder1/f*3/hell0.csv").test("folder1/folder2/hell0.csv")).toBe(false);
                expect(FilePatternUtils.fullPathGlobRegExp("folder2/**/hell0.csv").test("folder1/folder2/hell0.csv")).toBe(false);
                expect(FilePatternUtils.fullPathGlobRegExp("folder2/**/hell0.csv").test("folder1/folder2/folder3/hell0.csv")).toBe(false);
                expect(FilePatternUtils.fullPathGlobRegExp("**/hell1.csv").test("folder1/folder2/hell0.csv")).toBe(false);
                expect(FilePatternUtils.fullPathGlobRegExp("**/folder4/**/hell0.csv").test("folder1/folder2/folder3/hell0.csv")).toBe(false);
                expect(FilePatternUtils.fullPathGlobRegExp("*.*").test("dir/file.txt")).toBe(false);

                expect(FilePatternUtils.fullPathGlobRegExp("a/**/b").test("ab")).toBe(false);
                expect(FilePatternUtils.fullPathGlobRegExp("a/**/b").test("a/ab")).toBe(false);
            });
        });

        it("fullPathGlobRegExp handles ? in combinations - on paths", function() {
            inject(function(FilePatternUtils) {
                expect(FilePatternUtils.fullPathGlobRegExp("folder1/*/hell?.csv").test("folder1/folder2/hell0.csv")).toBe(true);
                expect(FilePatternUtils.fullPathGlobRegExp("folder1/*/hel?1.csv").test("folder1/folder2/hell0.csv")).toBe(false);
                expect(FilePatternUtils.fullPathGlobRegExp("folder1/*/?ell?.csv").test("folder1/folder2/hell0.csv")).toBe(true);
                expect(FilePatternUtils.fullPathGlobRegExp("folder1/*/?el?1.csv").test("folder1/folder2/hell0.csv")).toBe(false);

                expect(FilePatternUtils.fullPathGlobRegExp("**/hell?.csv").test("folder1/folder2/hell0.csv")).toBe(true);
                expect(FilePatternUtils.fullPathGlobRegExp("**/?ell?.csv").test("folder1/folder2/hell0.csv")).toBe(true);
                expect(FilePatternUtils.fullPathGlobRegExp("**/helW?.csv").test("folder1/folder2/hell0.csv")).toBe(false);
            });
        });

    });

    describe("normalizeSlashes", function() {
        it("normalizes slashes", function() {
            inject(function(FilePatternUtils) {
                expect(FilePatternUtils.normalizeSlashes("/hell0 .csv/")).toBe("hell0 .csv");
                expect(FilePatternUtils.normalizeSlashes("//hell0.csv///")).toBe("hell0.csv");
                expect(FilePatternUtils.normalizeSlashes("folder/hell0.csv/")).toBe("folder/hell0.csv");
                expect(FilePatternUtils.normalizeSlashes("/folder////hell0 .csv/")).toBe("folder/hell0 .csv");
            });
        });
    });

});
