<#
 .Synopsis
  Returns true if the given Python version is installed.

 .Description
  Checks the registry key for the current user and returns
  true if the given Python version is installed.

 .Parameter Version
  The version to check.

 .Example
   # Checks if the version 3.7 of python is installed
   Test-Python-Installation -Version 3.7
#>
function Test-PythonInstallation
{
    param (
        [Parameter(Mandatory=$true)] [string] $Version
    )

    return Test-Path "Registry::HKEY_CURRENT_USER\Software\Python\PythonCore\$Version\InstallPath"
}

<#
 .Synopsis
  Returns the path where the given Python version is installed.

 .Description
  Checks the registry key for the current user and returns
  the path where the given Python version is installed.

 .Parameter Version
  The Python version.

 .Example
   # Returns the path to the python 3.7 install directory.
   Get-PythonInstallPath -Version 3.7
#>
function Get-PythonInstallPath
{
    param (
        [Parameter(Mandatory=$true)] [string] $Version
    )

    return (Get-ItemProperty "Registry::HKEY_CURRENT_USER\Software\Python\PythonCore\$Version\InstallPath").'(default)'
}

<#
 .Synopsis
  Returns path to the given Python version executable.

 .Description
  Checks the registry key for the current user and returns
  the path to the given Python version executable.

 .Parameter Version
  The Python version.

 .Example
   # Returns the path to the python 3.7 executable.
   Get-PythonExecutablePath -Version 3.7
#>
function Get-PythonExecutablePath
{
    param (
        [Parameter(Mandatory=$true)] [string] $Version
    )

    return (Get-ItemProperty "Registry::HKEY_CURRENT_USER\Software\Python\PythonCore\$Version\InstallPath").'ExecutablePath'
}