/*
 * Decompiled with CFR 0.152.
 */
package com.dataiku.dip.server.services.catalog;

import com.codahale.metrics.Gauge;
import com.codahale.metrics.Metric;
import com.dataiku.dip.DSSMetrics;
import com.dataiku.dip.server.services.catalog.LuceneDocBuilder;
import com.dataiku.dip.server.services.catalog.LuceneMappingsAnalyzer;
import com.dataiku.dip.utils.DKULogger;
import com.dataiku.dip.utils.ExceptionUtils;
import com.dataiku.dss.shadelib.org.apache.lucene.index.IndexWriter;
import com.dataiku.dss.shadelib.org.apache.lucene.search.Query;
import java.io.IOException;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingQueue;

public class LuceneBufferedIndexer
implements AutoCloseable {
    private int uncommittedDocumentsCount = 0;
    private final int maxUncommittedDocumentCount;
    private final long sleepMs;
    private final IndexWriter indexWriter;
    private final LuceneMappingsAnalyzer luceneMappingsAnalyzer;
    private final String name;
    private final BlockingQueue<IndexAction> queue;
    private static final DKULogger logger = DKULogger.getLogger((String)LuceneBufferedIndexer.class.getName());
    private static final int NO_SPACE_LEFT_ERROR_THROTTLE_TIME_IN_MS = 30000;
    private Thread backgroundCommitThread;
    private final Thread queueThread;

    public LuceneBufferedIndexer(IndexWriter indexWriter, LuceneMappingsAnalyzer luceneMappingsAnalyzer, String indexManagerName, int maxUncommittedDocumentCount, long sleepS, int queueSize) {
        this.indexWriter = indexWriter;
        this.luceneMappingsAnalyzer = luceneMappingsAnalyzer;
        this.name = indexManagerName;
        this.maxUncommittedDocumentCount = maxUncommittedDocumentCount;
        this.sleepMs = sleepS * 1000L;
        this.queue = new LinkedBlockingQueue<IndexAction>(queueSize);
        DSSMetrics.registry().register("dku.catalog.indexing.bufferedQueue." + this.name + ".size", (Metric)((Gauge)this.queue::size));
        this.queueThread = new Thread(() -> {
            Thread.currentThread().setName("lucene-buffered-indexer-queue-" + this.name);
            int noSpaceLeftCount = 0;
            long lastNoSpaceLeftIssueLog = 0L;
            try {
                IndexAction indexAction;
                while ((indexAction = this.queue.take()) != null) {
                    try {
                        indexAction.process();
                    }
                    catch (Throwable t) {
                        if (ExceptionUtils.hasCauseWithMessage((Throwable)t, (String)"No space left")) {
                            if (lastNoSpaceLeftIssueLog == 0L) {
                                lastNoSpaceLeftIssueLog = System.currentTimeMillis();
                                logger.error((Object)("Failed to process action: " + ExceptionUtils.getMessageWithCauses((Throwable)t)));
                                continue;
                            }
                            ++noSpaceLeftCount;
                            long currentTime = System.currentTimeMillis();
                            if (currentTime - lastNoSpaceLeftIssueLog <= 30000L) continue;
                            logger.error((Object)("Failed to process action (" + noSpaceLeftCount + " occurrence(s) since last log): " + ExceptionUtils.getMessageWithCauses((Throwable)t)));
                            noSpaceLeftCount = 0;
                            lastNoSpaceLeftIssueLog = currentTime;
                            continue;
                        }
                        logger.error((Object)"Failed to process action", t);
                    }
                }
            }
            catch (Throwable t) {
                logger.warn((Object)"Fatal Lucene queue error", t);
            }
        });
        this.queueThread.setDaemon(true);
        this.queueThread.start();
    }

    private void addActionInQueue(IndexAction action) throws InterruptedException {
        if (this.queueThread.isAlive() && !this.queueThread.isInterrupted()) {
            this.queue.put(action);
        }
    }

    public void add(LuceneDocBuilder doc) throws InterruptedException {
        this.addActionInQueue(new AddAction(doc));
    }

    public void delete(Query query, int count) throws InterruptedException {
        this.addActionInQueue(new DeleteAction(query, count));
    }

    public void deleteAll() throws InterruptedException {
        this.queue.clear();
        this.addActionInQueue(new DeleteAllAction());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void commit() throws InterruptedException {
        CommitAction action = new CommitAction(true, true);
        this.addActionInQueue(action);
        LuceneBufferedIndexer luceneBufferedIndexer = this;
        synchronized (luceneBufferedIndexer) {
            while (!action.handled) {
                this.wait();
            }
        }
    }

    private void commitSynchronously() throws IOException {
        assert (Thread.holdsLock(this));
        this.indexWriter.commit();
        this.uncommittedDocumentsCount = 0;
    }

    private void startBackgroundCommitThreadIfNeeded() {
        if (this.backgroundCommitThread != null) {
            return;
        }
        this.backgroundCommitThread = new Thread(() -> {
            Thread.currentThread().setName("lucene-buffered-indexer-background-queue-" + this.name);
            try {
                Thread.sleep(this.sleepMs);
                try {
                    this.queue.put(new CommitAction(false, false));
                }
                catch (Throwable t) {
                    logger.warn((Object)"Failed to enqueue commit action", t);
                }
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
        });
        this.backgroundCommitThread.setDaemon(true);
        this.backgroundCommitThread.start();
    }

    private synchronized void increaseCountAndCommitIfNeeded(int count) throws IOException {
        this.uncommittedDocumentsCount += count;
        if (this.uncommittedDocumentsCount >= this.maxUncommittedDocumentCount) {
            this.commitSynchronously();
        }
    }

    @Override
    public void close() {
        if (this.backgroundCommitThread != null) {
            this.backgroundCommitThread.interrupt();
        }
        if (this.queueThread != null) {
            this.queueThread.interrupt();
        }
    }

    private class AddAction
    implements IndexAction {
        LuceneDocBuilder doc;

        AddAction(LuceneDocBuilder doc) {
            this.doc = doc;
        }

        @Override
        public void process() throws IOException {
            LuceneBufferedIndexer.this.startBackgroundCommitThreadIfNeeded();
            LuceneBufferedIndexer.this.luceneMappingsAnalyzer.addDocument(LuceneBufferedIndexer.this.indexWriter, this.doc);
            LuceneBufferedIndexer.this.increaseCountAndCommitIfNeeded(1);
        }
    }

    private static interface IndexAction {
        public void process() throws IOException;
    }

    private class DeleteAction
    implements IndexAction {
        Query query;
        Integer count;

        DeleteAction(Query query, Integer count) {
            this.query = query;
            this.count = count;
        }

        @Override
        public void process() throws IOException {
            LuceneBufferedIndexer.this.startBackgroundCommitThreadIfNeeded();
            LuceneBufferedIndexer.this.indexWriter.deleteDocuments(new Query[]{this.query});
            LuceneBufferedIndexer.this.increaseCountAndCommitIfNeeded(this.count);
        }
    }

    private class DeleteAllAction
    implements IndexAction {
        private DeleteAllAction() {
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void process() throws IOException {
            LuceneBufferedIndexer.this.indexWriter.deleteAll();
            LuceneBufferedIndexer luceneBufferedIndexer = LuceneBufferedIndexer.this;
            synchronized (luceneBufferedIndexer) {
                LuceneBufferedIndexer.this.commitSynchronously();
            }
        }
    }

    private class CommitAction
    implements IndexAction {
        boolean interruptBackgroundCommitThread;
        boolean notify;
        boolean handled = false;

        CommitAction(boolean interrupt, boolean notify) {
            this.interruptBackgroundCommitThread = interrupt;
            this.notify = notify;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void process() throws IOException {
            LuceneBufferedIndexer luceneBufferedIndexer = LuceneBufferedIndexer.this;
            synchronized (luceneBufferedIndexer) {
                if (this.interruptBackgroundCommitThread && LuceneBufferedIndexer.this.backgroundCommitThread != null) {
                    LuceneBufferedIndexer.this.backgroundCommitThread.interrupt();
                    LuceneBufferedIndexer.this.backgroundCommitThread = null;
                }
                LuceneBufferedIndexer.this.commitSynchronously();
                this.handled = true;
                if (this.notify) {
                    LuceneBufferedIndexer.this.notifyAll();
                }
            }
        }
    }
}

