(function() {
'use strict';


const app = angular.module('dataiku.common.nav');


    app.directive("navigatorObject", function(Navigator) {
        return {
            scope: true,
            restrict: 'A',
            controller: function($scope, $controller) {
                $controller('TaggableObjectPageMassActionsCallbacks', {$scope: $scope});
            },
            link: function(scope, element, attrs) {
                scope.Navigator = Navigator;

                Mousetrap.bind('shift+a', function() {
                    if (scope.datasetHooks && typeof scope.datasetHooks.userIsWriting == 'function' && scope.datasetHooks.userIsWriting()) {return;}
                    if (scope.isProjectAnalystRO() && (!attrs.navigatorDisabled || scope.$eval(attrs.navigatorDisabled))) {
                        Navigator.toggleForTopNav();
                    }
                });

                scope.$on('$destroy', function() {
                    Navigator.hide();
                    Mousetrap.unbind('shift+a');
                });
            }
        }
    });

    app.factory("Navigator", function ($rootScope, $stateParams, CreateCustomElementFromTemplate, DataikuAPI) {
        var navScope, removeListener;

        // Dirty hack to hide the underneath right panel when the navigator's right panel is displayed [CH40012]
        // would be better to actually replace the navigator panel by the new right panel but let's leave that for Right Panel V2
        let toggleUnderneathRightPanelIfAny = (hide) => {
            let rightPanels = document.getElementsByClassName("right-panel");
            if (rightPanels && rightPanels.length > 0) {
                rightPanels[0].style.opacity = hide ? 0 : 1;
            }
        };

        var nav = {
            show: function (projectKey, objectType, objectId) {
                navScope = true; // Initialize navScope now to keep further calls to toggleForTopNav() from calling show() again, creating several navigator elements and scopes

                DataikuAPI.flow.recipes.getGraph(projectKey, false, false).success(function (data) {
                    toggleUnderneathRightPanelIfAny(true);
                    CreateCustomElementFromTemplate("/templates/navigator.html", $rootScope, "NavigatorController", function(newScope) {
                        navScope = newScope;
                        newScope.flow = data.serializedFilteredGraph.serializedGraph;
                        newScope.focus = {projectKey: projectKey, objectType: objectType, objectId: objectId};
                        newScope.init();
                    }, function(newEl) {
                        newEl.appendTo($('body'));
                    });
                }).noSpinner();
                Mousetrap.bind('esc', nav.hide);
                removeListener = $rootScope.$on("$stateChangeStart", nav.hide);
            },

            hide: function () {
                if (navScope && navScope.dismiss) {
                    toggleUnderneathRightPanelIfAny(false);
                    navScope.dismiss();
                    navScope = null;
                    Mousetrap.unbind('esc');
                }

                if (removeListener) removeListener();
            },

            toggleForTopNav: function () {
                if (navScope) return nav.hide();
                else return nav.show($stateParams.projectKey, $rootScope.topNav.item.type, $rootScope.topNav.item.id);
            },

            showForTopNav: function () {
                return nav.show($stateParams.projectKey, $rootScope.topNav.item.type, [$stateParams.sourceProjectKey,$rootScope.topNav.item.id].filter(Boolean).join("."));
            }
        };

        return nav;
    });


    app.controller("NavigatorController", function ($scope, DataikuAPI, RecipesUtils, Navigator, QuickView) {
        $scope.Navigator = Navigator;

        $scope.$on("change-context-focus", function (evt, target) {
            $scope.focus = target;
            updateContext();
        });

        $scope.selected = {};

        $scope.QuickView = QuickView;

        var curToken = 0;
        var updateContext = function () {
            var reqToken = ++curToken;
            DataikuAPI.flow.getObjectContext($scope.focus.projectKey, $scope.focus.objectType, $scope.focus.objectId).success(function (data) {
                if (reqToken != curToken) return; // drop if not the most recent call
                $scope.context = data;
                if ($scope.focus.objectType === "RECIPE") {
                    RecipesUtils.parseScriptIfNeeded(data.nodes[data.focusNodeId]);
                }
            });
        };

        $scope.$watch("context", function (nv) {
            if (nv != null) {
                $scope.object = ($scope.context.nodes || {})[$scope.context.focusNodeId];
                $scope.node = ($scope.flow.nodes || {})[$scope.context.focusNodeId];
            }
        });

        $scope.$parent.init = updateContext;

        $scope.$on("$destroy", QuickView.hide);
    });

    app.directive("navigatorFlow", function (Fn, $filter, Navigator, StateUtils, $stateParams, getFlowNodeIcon, objectTypeFromNodeFlowType) {

        return {
            scope: {
                context: '=',
                flow: '='
            },
            restrict: 'A',
            link: function ($scope, element, attrs) {

                function endAll(transition, callback) {
                    if (transition.size() === 0) {
                        callback()
                    }
                    var n = 0;
                    transition
                        .each(function () {
                            ++n;
                        })
                        .each("end", function () {
                            if (!--n) callback.apply(this, arguments);
                        });
                }

                function translate(x, y) {
                    return 'translate(' + parseInt(x) + ',' + parseInt(y) + ')';
                }

                function parseTranslate(translate) {
                    if (!translate) return [0, 0];
                    var split = translate.split(',');
                    return [parseInt(split[0].split('(')[1]), parseInt(split[1])];
                }

                function flowNodeFromContextNode(id, contextNode) {
                    if (id.startsWith('insight_')) {
                        return {nodeType: 'INSIGHT', insightType: contextNode.insight.type, description: contextNode.insight.name, name: contextNode.insight.id, id: id};
                    } else if (id.startsWith('jupyterNotebook_')) {
                        return {nodeType: 'JUPYTER_NOTEBOOK', description: contextNode.notebook.name, name: contextNode.notebook.name, id: id};
                    }
                }

                var svg = d3.select(element[0]);

                var lineG = svg.append('g').attr('class', 'lines');
                var compG = svg.append('g').attr('class', 'computables');
                var runG = svg.append('g').attr('class', 'runnables');
                var compEls, runEls;

                var datasetInfos, recipeInfos, nodes, contextNodes, selectedNode;
                var runnables, computables, lines, drawing, flowLink;
                var centerNode, topNode;

                const defaultBlockSize = 72;
                const defaultIconSize = 48;
                const defaultPartitionOffset = -5;
                const isRecipe = nodeType => nodeType === 'RECIPE';
                const isLabelingTask = nodeType => nodeType === 'LABELING_TASK';
                const isRecipeOrLabelingTask = nodeType => isRecipe(nodeType) || isLabelingTask(nodeType);
                const isDataset = nodeType => nodeType === 'LOCAL_DATASET' || nodeType === 'FOREIGN_DATASET';
                const isManagedFolder = nodeType => nodeType === 'LOCAL_MANAGED_FOLDER' || nodeType === 'FOREIGN_MANAGED_FOLDER';
                const isSavedModel = nodeType => nodeType === 'LOCAL_SAVEDMODEL' || nodeType === 'FOREIGN_SAVEDMODEL';
                const isModelEvaluationStore = nodeType => nodeType === 'LOCAL_MODELEVALUATIONSTORE' || nodeType === 'FOREIGN_MODELEVALUATIONSTORE' || nodeType === 'LOCAL_GENAIEVALUATIONSTORE' || nodeType === 'FOREIGN_GENAIEVALUATIONSTORE';
                const isRetrievableKnowledge = nodeType => nodeType === 'LOCAL_RETRIEVABLE_KNOWLEDGE' || nodeType === 'FOREIGN_RETRIEVABLE_KNOWLEDGE';
                const isInsight = nodeType => nodeType === 'INSIGHT';
                const isSavedModelOrModelEvaluationStoreOrInsight = nodeType => isSavedModel(nodeType) || isModelEvaluationStore(nodeType) || isInsight(nodeType);
                const isCustomRecipe = recipeType => Boolean(recipeType) && recipeType.startsWith('CustomCode_'); // plugin
                const isApplicationRecipe = recipeType => Boolean(recipeType) && recipeType.startsWith('App_');
                const isCustomOrApplicationRecipe = recipeType => isCustomRecipe(recipeType) || isApplicationRecipe(recipeType);

                function draw() {
                    let blockSize = defaultBlockSize;
                    const size = d => d.center ? defaultBlockSize : blockSize;

                    drawing = true;
                    if ($scope.flow.nodes) nodes = $scope.flow.nodes;
                    if ($scope.context.nodes) contextNodes = $scope.context.nodes;

                    var showFlowLink = true;
                    var height = $(svg[0][0]).height(),
                        width = $(svg[0][0]).width();
                    var margin = 0.3;

                    for (var node in nodes) {
                        delete nodes[node].center;
                        delete nodes[node].drawn;
                        delete nodes[node].idx;
                        delete nodes[node].left;
                        delete nodes[node].runnableId;
                        delete nodes[node].up;
                    }

                    topNode = null;
                    centerNode = nodes[$scope.context.focusNodeId];
                    var objectData = contextNodes[$scope.context.focusNodeId];

                    runnables = [], computables = [], lines = [];

                    if (!centerNode) { // If the focus is on a non-flow item, we display it as topNode
                        if ($scope.context.focusNodeId.startsWith('insight')) {
                            computables.push(topNode = flowNodeFromContextNode($scope.context.focusNodeId, contextNodes[$scope.context.focusNodeId]));
                            topNode.center = true;
                            topNode.top = true;

                            centerNode = nodes[$scope.context.centerNodeId];

                            if (!centerNode) { // If the centerNode is a non-flow item as well
                                centerNode = flowNodeFromContextNode($scope.context.centerNodeId, contextNodes[$scope.context.centerNodeId]);
                                centerNode.center = true;
                                centerNode.top = false;
                                centerNode.predecessors = [];
                                centerNode.successors = [];
                                showFlowLink = false;
                            }

                            centerNode.clickable = true;
                            selectedNode = topNode;
                        } else if ($scope.context.focusNodeId.startsWith('analysis')) {
                            centerNode = nodes[objectData.datasetNodeId];
                            computables.push(topNode = {nodeType: 'ANALYSIS', description: objectData.analysis.name, center: true, top: true});
                            centerNode.clickable = true;
                            selectedNode = topNode;
                        } else if ($scope.context.focusNodeId.startsWith('jupyterNotebook')) {
                            if (objectData.datasetNodeId) {
                                centerNode = nodes[objectData.datasetNodeId];
                                computables.push(topNode = {nodeType: 'JUPYTER_NOTEBOOK', id: objectData.notebook.name, description: objectData.notebook.name, center: true, top: true});
                                centerNode.clickable = true;
                                selectedNode = topNode;
                            } else if (objectData.recipeNodeId) {
                                centerNode = nodes[objectData.recipeNodeId];
                                computables.push(topNode = {nodeType: 'JUPYTER_NOTEBOOK', id: objectData.notebook.name, description: objectData.notebook.name, center: true, top: true});
                                centerNode.clickable = true;
                                selectedNode = topNode;
                            } else {
                                showFlowLink = false;
                                centerNode = {nodeType: 'JUPYTER_NOTEBOOK', id: objectData.notebook.name, description: objectData.notebook.name, center: true, clickable: false, successors: [], predecessors: []};
                                computables.push(centerNode);
                                selectedNode = centerNode;
                            }
                        } else if ($scope.context.focusNodeId.startsWith('sqlNotebook')) {
                            if (objectData.datasetNodeId) {
                                centerNode = nodes[objectData.datasetNodeId];
                                computables.push(topNode = {nodeType: 'SQL_NOTEBOOK', id: objectData.notebook.id, description: objectData.notebook.name, center: true, top: true});
                                centerNode.clickable = true;
                                selectedNode = topNode;
                            } else {
                                showFlowLink = false;
                                centerNode = {nodeType: 'SQL_NOTEBOOK', id: objectData.notebook.id, description: objectData.notebook.name, center: true, clickable: false, successors: [], predecessors: []};
                                computables.push(centerNode);
                                selectedNode = centerNode;
                            }
                        } else if ($scope.context.focusNodeId.startsWith('searchNotebook')) {
                            showFlowLink = false;
                            centerNode = {nodeType: 'SEARCH_NOTEBOOK', id: objectData.notebook.id, description: objectData.notebook.name, center: true, clickable: false, successors: [], predecessors: []};
                            computables.push(centerNode);
                            selectedNode = centerNode;
                        }
                    } else {
                        centerNode.clickable = false;
                        selectedNode = centerNode;
                    }

                    var sides = [
                        {group: 'successors', left: 0, computables: 0, height: 0},
                        {group: 'predecessors', left: 1, computables: 0, height: 0}
                    ];

                    centerNode.center = true;
                    centerNode.drawn = true;

                    const fakeComputables = {}; // placeholders for drawing cycles and recipes without outputs (labeling tasks, Python recipes with deleted output)
                    const buildFakeComputable = (idx, left) => ({ idx, left, drawn: false });
                    const setFakeComputable = (runnableId) => (idx, left) => fakeComputables[runnableId] = buildFakeComputable(idx, left);

                    const implicitRecipes = [];
                    if (isRecipeOrLabelingTask(centerNode.nodeType)) {
                        runnables.push(centerNode);
                        sides.forEach(function (side) {
                            if (centerNode[side.group].length === 0) setFakeComputable(centerNode.id)(side.computables++, side.left);
                            centerNode[side.group].forEach(function (id) {
                                nodes[id].drawn = true;
                                nodes[id].idx = side.computables++;
                                nodes[id].left = side.left;
                                nodes[id].runnableId = centerNode.id;
                                computables.push(nodes[id]);
                                lines.push([centerNode.id, id, false]);
                            });
                        });
                    } else {
                        computables.push(centerNode);
                        sides.forEach(function (side) {
                            centerNode[side.group].forEach(function (id, i) {
                                if (isRecipeOrLabelingTask(nodes[id].nodeType)) {
                                    nodes[id].drawn = true;
                                    nodes[id].left = side.left;
                                    nodes[id].up = (i < centerNode[side.group].length / 2);
                                    runnables.push(nodes[id]);
                                    lines.push([id, centerNode.id, false]);
                                } else {
                                    var implicitRecipe = {id:'implicitRecipe_' + implicitRecipes.length, recipeType: 'aa', successors:[], predecessors:[]};
                                    implicitRecipes.push(implicitRecipe)
                                    implicitRecipe[side.group == 'successors' ? 'successors' : 'predecessors'] = [id];
                                    implicitRecipe.drawn = false;
                                    implicitRecipe.left = side.left;
                                    implicitRecipe.up = (i < centerNode[side.group].length / 2);
                                    runnables.push(implicitRecipe);
                                    lines.push([implicitRecipe.id, centerNode.id, true]);
                                }
                            });
                        });

                        runnables.forEach(function (runnable, i) {
                            if (runnable[sides[runnable.left].group].length === 0) setFakeComputable(runnable.id)(sides[runnable.left].computables++, runnable.left);
                            runnable[sides[runnable.left].group].forEach(function (id) {
                                if (nodes[id].runnableId !== undefined) setFakeComputable(nodes[id].runnableId)(nodes[id].idx, nodes[id].left);
                                nodes[id].drawn = true;
                                nodes[id].idx = sides[runnable.left].computables++;
                                nodes[id].left = runnable.left;
                                nodes[id].runnableId = runnable.id;
                                computables.push(nodes[id]);
                                lines.push([runnable.id, id, runnable.id.startsWith('implicitRecipe')]);
                            });
                        });
                    }

                    // Reduce the block size until both sides fit
                    sides.forEach(function (side) {
                        if (side.computables > 0) {
                            side.height = side.computables * blockSize + (side.computables - 1) * margin * blockSize;
                            while (blockSize >= 1 && side.height > 0.8 * $(element[0]).height()) {
                                blockSize--;
                                side.height = side.computables * blockSize + (side.computables - 1) * margin * blockSize;
                            }
                        }
                    });

                    sides.forEach(function (side) {
                        if (side.computables > 0) side.height = side.computables * blockSize + (side.computables - 1) * margin * blockSize;
                    });

                    const compTrY = d => (height - sides[d.left].height) / 2 + d.idx * size(d) * (1 + margin);

                    const formats = {
                        foreignObject: function (sel) {
                            return sel
                                .attr('x', function (d) {
                                    if (isManagedFolder(d.nodeType)) return 0;
                                    else if (isCustomOrApplicationRecipe(d.recipeType)) return size(d) * 16 / defaultBlockSize;
                                    else return size(d) * 12 / defaultBlockSize;
                                })
                                .attr('y', function (d) {
                                    if (isManagedFolder(d.nodeType)) return 0;
                                    else if (isCustomOrApplicationRecipe(d.recipeType)) return size(d) * 17 / defaultBlockSize;
                                    else return size(d) * 13 / defaultBlockSize;
                                })
                                .attr('width', function (d) {
                                    if (isCustomOrApplicationRecipe(d.recipeType)) return size(d) * 40 / defaultBlockSize;
                                    else return size(d) * defaultIconSize / defaultBlockSize;
                                })
                                .attr('height', function (d) {
                                    if (isCustomOrApplicationRecipe(d.recipeType)) return size(d) * 40 / defaultBlockSize;
                                    else return size(d) * defaultIconSize / defaultBlockSize;
                                })
                                .attr('transform', function (d) {
                                    if (isManagedFolder(d.nodeType)) return 'scale(1.5)';
                                });
                        },

                        fontIcon: function (sel) {
                            return sel
                                .attr('class', getFlowNodeIcon)
                                .attr('style', d => 'font-size: ' + (size(d) * (isCustomOrApplicationRecipe(d.recipeType) ? 24 : defaultIconSize) / defaultBlockSize) + 'px !important;');
                        },

                        partitionedRect: function(sel, partitionIndex) {
                            const rawOffset = d => defaultPartitionOffset * partitionIndex * size(d) / defaultBlockSize; // scaling offset given relative size of the partition rectangles
                            const offset = d => Math.sign(rawOffset(d)) * Math.max(defaultPartitionOffset, Math.round(Math.abs(rawOffset(d)))); // rounding offset to get a clean draw
                            return sel
                                .attr('x', d => isSavedModel(d.nodeType) ? 0 : offset(d))
                                .attr('y', d => isSavedModel(d.nodeType) ? (Math.sign(offset(d)) * Math.round(Math.sqrt(2 * Math.pow(offset(d), 2)))) : offset(d)) // good old Pythagoras for positioning a PI/4 rotated square
                                .attr('width', size)
                                .attr('height', size)
                                .attr('class', 'rect-partition-' + partitionIndex + ' fill partitioning-indicator');
                        },

                        rectMain: function (sel) {
                            return sel
                                .attr('rx', d => isSavedModelOrModelEvaluationStoreOrInsight(d.nodeType) ? size(d) / 9 : 0)
                                .attr('width', size)
                                .attr('height', size)
                                .attr('class', d => {
                                    if (isInsight(d.nodeType)) return 'rect-main universe-fill ' + $filter('insightTypeToColor')(d.insightType);
                                    else if (d.partitioned && isDataset(d.nodeType)) return 'rect-main fill main-dataset-rectangle';
                                    else return 'rect-main';
                                });
                        },

                        rectSelection: function (sel) {
                            return sel
                                .attr('width', size)
                                .attr('height', size)
                                .attr('rx', d => isSavedModelOrModelEvaluationStoreOrInsight(d.nodeType) ? size(d) / 9 : 0)
                                .attr('class', 'rect-selection');
                        },

                        rectBackground: function (sel) {
                            return sel
                                .attr('x', d => isManagedFolder(d.nodeType) ? (size(d) * 10 / defaultBlockSize) : 0)
                                .attr('y', d => isManagedFolder(d.nodeType) ? (size(d) * 10 / defaultBlockSize) : 0)
                                .attr('rx', d => isSavedModelOrModelEvaluationStoreOrInsight(d.nodeType) ? size(d) / 9 : 0)
                                .attr('width', d => isManagedFolder(d.nodeType) ? (size(d) * 52 / defaultBlockSize) : size(d))
                                .attr('height', d => isManagedFolder(d.nodeType) ? (size(d) * 52 / defaultBlockSize) : size(d))
                                .attr('class', 'rect-background');
                        },

                        circleSelection: function (sel) {
                            return sel
                                .attr('r', function (d) {
                                    return size(d) * 21 / defaultBlockSize;
                                })
                                .attr('cx', function (d) {
                                    return size(d) * 36 / defaultBlockSize;
                                })
                                .attr('cy', function (d) {
                                    return size(d) * 37 / defaultBlockSize;
                                })
                                .attr('class', 'circle-selection');
                        },

                        circleBackground: function (sel) {
                            return sel
                                .attr('r', function (d) {
                                    return size(d) * 20 / defaultBlockSize;
                                })
                                .attr('cx', function (d) {
                                    return size(d) * 36 / defaultBlockSize;
                                })
                                .attr('cy', function (d) {
                                    return size(d) * 37 / defaultBlockSize;
                                })
                                .attr('class', 'circle-background');
                        },

                        compEl: function (sel) {
                            return sel.attr('transform', function (d) {
                                if (d.top) return translate((width - size(d)) / 2, 50);
                                if (d.center) return translate((width - size(d)) / 2, (height - size(d)) / 2);
                                return translate((width - size(d)) / 2 + (d.left ? -300 : 300), (height - sides[d.left].height) / 2 + d.idx * size(d) * (1 + margin));
                            }).attr('data-size', size);
                        },

                        runEl: function (sel) {
                            return sel.attr('transform', function (d) {
                                if (d.center) return translate((width - size(d)) / 2, (height - size(d)) / 2);
                                let childrenComputables = computables.filter(c => c.runnableId === d.id);
                                if (childrenComputables.length === 0 && fakeComputables[d.id]) childrenComputables = [fakeComputables[d.id]];
                                if (childrenComputables.length === 0) return; // should not happen thanks to fake computables
                                return translate((width - size(d)) / 2 + (d.left ? -200 : 200), (compTrY(childrenComputables[0]) + compTrY(childrenComputables[childrenComputables.length - 1])) / 2);
                            }).attr('data-size', size).style('display', d => d.drawn ? 'inline-block' : 'none');
                        },

                        compTick: function (sel) {
                            return sel
                                .style("display", "none")
                                .filter(function (d) {
                                    return !d.center &&
                                        ((d.left && d.predecessors && d.predecessors.length)
                                        || (!d.left && d.successors && d.successors.length))
                                })
                                .style("display", null)
                                .attr('x1', function (d) {
                                    return d.left ? 0 : blockSize;
                                })
                                .attr('x2', function (d) {
                                    return d.left ? -15 : blockSize + 15;
                                })
                                .attr('y1', blockSize / 2)
                                .attr('y2', blockSize / 2 + 0.001)
                                .attr('stroke', function (d) {
                                    return d.left ? 'url(#grad-left-right)' : 'url(#grad-right-left)'
                                });
                        },

                        runTick: function (sel) {
                            return sel
                                .style("display", "none")
                                .filter(function (d) {
                                    return !d.center &&
                                        ((!d.left && d.predecessors && d.predecessors.length > 1)
                                        || (d.left && d.successors && d.successors.length > 1))
                                })
                                .style("display", null)
                                .style("stroke-width", "1.1px") //TODO @navigator css
                                .attr('x1', blockSize / 2)
                                .attr('x2', function (d) {
                                    return blockSize / 2 + (d.left ? blockSize / 2 : -blockSize / 2);
                                })
                                .attr('y1', blockSize / 2)
                                .attr('y2', function (d) {
                                    return blockSize / 2 + (d.up ? -blockSize / 2 : blockSize / 2);
                                })
                                .attr('stroke', function (d) {
                                    return !d.left ? 'url(#grad-left-right)' : 'url(#grad-right-left)'
                                });
                        }
                    };

                    compEls = compG.selectAll('g').data(computables, Fn.prop('id'));
                    runEls = runG.selectAll('g').data(runnables, Fn.prop('id'));
                    lines = lineG.selectAll('line.line').data(lines, function (d) {
                        return d[0] + ' ---> ' + d[1];
                    });

                    // 1. Remove exiting elements
                    compEls.exit().remove();
                    runEls.exit().remove();
                    lines.exit().remove();
                    if (datasetInfos) datasetInfos.remove();
                    if (recipeInfos) recipeInfos.remove();
                    if (flowLink) flowLink.remove();
                    svg.selectAll('line.dotted').remove();

                    // 2. Update existing elements
                    compEls.select('line.tick').style('display', 'none');
                    runEls.select('line.tick').style('display', 'none');
                    compEls.transition().call(formats.compEl).call(endAll, enter);
                    compEls.select('rect.rect-partition-2').transition().call(formats.partitionedRect, 2);
                    compEls.select('rect.rect-partition-1').transition().call(formats.partitionedRect, 1);
                    compEls.select('rect.rect-background').transition().call(formats.rectBackground);
                    compEls.select('rect.rect-main').transition().call(formats.rectMain);
                    compEls.select('foreignObject').transition().call(formats.foreignObject).select('i').call(formats.fontIcon);
                    compEls.select('rect.rect-selection').transition().call(formats.rectSelection);
                    runEls.transition().call(formats.runEl);
                    runEls.select('circle.circle-background').transition().call(formats.circleBackground);
                    runEls.select('foreignObject').transition().call(formats.foreignObject).select('i').call(formats.fontIcon);
                    runEls.select('circle.circle-selection').transition().call(formats.circleSelection);

                    runEls.order();
                    compEls.order();

                    // Update lines with a custom tween so that they stay connected to the centers of the 2 items during the transition
                    svg.selectAll('line.line').transition().tween("line", function (d) {
                        var run = runEls.filter(function (r) {
                            return r.id == d[0];
                        });
                        var comp = compEls.filter(function (c) {
                            return c.id == d[1]
                        });
                        return function () {
                            var runT = parseTranslate(run.attr('transform'));
                            var compT = parseTranslate(comp.attr('transform'));
                            var runS = run.attr('data-size');
                            var compS = comp.attr('data-size');
                            d3.select(this)
                                .attr('x1', runT[0] + runS / 2)
                                .attr('y1', runT[1] + runS / 2)
                                .attr('x2', compT[0] + compS / 2)
                                .attr('y2', compT[1] + compS / 2)
                                .attr('stroke', '#333');
                        }
                    });


                    function enter() {
                        // Create computables
                        const enteringCompEls = compEls.enter().append('g').attr('data-type', Fn.prop('nodeType'))
                            .attr("class", d => {
                                if (d.savedModelType && ['LLM_GENERIC', 'PLUGIN_AGENT', 'PYTHON_AGENT', 'TOOLS_USING_AGENT', "RETRIEVAL_AUGMENTED_LLM"].includes(d.savedModelType)) {
                                    return "fine-tuned-sm";
                                }
                            }).call(formats.compEl)
                            .on('click', function (d) {

                                if (d.center) {
                                    if (d.dblclick) {
                                        fakeClickOnLink(StateUtils.href.node(d), d3.event);
                                        Navigator.hide();
                                    }
                                    d.dblclick = true;
                                    setTimeout(function() { d.dblclick = false; }, 400);
                                }
                                d3.event.stopPropagation();

                                if (d.center && !d.clickable) return null;

                                // The objects start moving right away after the first click
                                // To make it easier to double click, any click on the svg works
                                svg.on('click.dblclick', function() {
                                    fakeClickOnLink(StateUtils.href.node(d), d3.event);
                                });
                                setTimeout(function() { svg.on('click.dblclick', null); }, 400);

                                return focusOnNode(d);
                            });

                        // Add tick
                        enteringCompEls.append('line').attr('class', 'tick');
                        const enteringCompElsPartitioned = enteringCompEls.filter(d => d.partitioned);
                        enteringCompElsPartitioned.append('rect').call(formats.partitionedRect, 2);
                        enteringCompElsPartitioned.append('rect').call(formats.partitionedRect, 1);
                        enteringCompEls.append('rect').call(formats.rectBackground);
                        enteringCompEls.append('rect').call(formats.rectMain);
                        enteringCompEls.append('svg:foreignObject').call(s => formats.foreignObject(s)).attr('class', 'nav-nodeicon')
                            .append('xhtml:div').attr('class', 'flow-tile faic jcc h100 w100')
                            .append('xhtml:i').call(formats.fontIcon);
                        enteringCompEls.append('rect').call(formats.rectSelection);

                        // Create computables legend
                        datasetInfos = compEls
                            .append('svg:foreignObject')
                            .attr('class', 'dataset-info')
                            .attr('x', function (d) {
                                if (d.center) return -130;
                                else if (d.left) return (blockSize - width) / 2 + 300;
                                else return blockSize;
                            })
                            .attr('y', function (d, i) {
                                if (d.center) return size(d);
                                else return 0;
                            })
                            .attr('height', d => size(d) + 'px')
                            .attr('width', function (d) {
                                if (d.center) return 260 + size(d) + 'px';
                                else return Math.max(0, width / 2 - 300 - blockSize / 2) + 'px';
                            });

                        datasetInfos.append('xhtml:div')
                            .classed('left', Fn.prop('left'))
                            .classed('center', Fn.prop('center'))
                            .style('height', d => size(d) + 'px')
                            .style('width', function (d) {
                                if (d.center) return 260 + size(d) + 'px';
                                else return Math.max(0, width / 2 - 300 - blockSize / 2) + 'px';
                            })
                            .append('xhtml:h6').append('span').text(Fn.prop('description'));

                        var iconGroup = function (type, icon) {
                            return function (sel) {
                                var div = sel.append('xhtml:span').attr('class', 'count ' + type).style('display', 'none');
                                div.append('xhtml:i').attr('class', 'universe-color ' + type + ' ' + icon);
                                div.append('xhtml:span');
                                return sel;
                            }
                        };

                        datasetInfos.append('div').attr('class', 'counts').append('span').each(function (d) {
                            const p = d3.select(this);
                            if (isDataset(d.nodeType)) {
                                p.call(iconGroup('analysis', 'icon-dku-nav_analysis'));
                                p.call(iconGroup('chart', 'icon-dku-nav_dashboard'));
                                p.call(iconGroup('notebook', 'icon-dku-nav_notebook'));
                            }
                        });

                        enteringCompEls.filter('g[data-type="LOCAL_SAVEDMODEL"], g[data-type="FOREIGN_SAVEDMODEL"], g[data-type="LOCAL_MODELEVALUATIONSTORE"], g[data-type="FOREIGN_MODELEVALUATIONSTORE"], g[data-type="LOCAL_GENAIEVALUATIONSTORE"], g[data-type="FOREIGN_GENAIEVALUATIONSTORE"]').selectAll('rect')
                            .attr('transform', 'rotate(45) scale(0.7071) scale(1.05)');

                        // Create runnables
                        const runnableElements = runEls.enter().append('g').attr('data-type', Fn.prop('nodeType')).attr('class', function (d) {
                            if (isLabelingTask(d.nodeType)) return 'bzicon recipeicon-labeling_task';
                            else if (isCustomRecipe(d.recipeType)) return 'bzicon recipeicon-custom-code';
                            else if (isApplicationRecipe(d.recipeType)) return 'bzicon recipeicon-app';
                            else return 'bzicon recipeicon-' + d.recipeType;
                        }).call(formats.runEl);

                        runnableElements.append('line').classed('tick', true);
                        runnableElements.append('circle').call(formats.circleBackground);
                        runnableElements
                            .append('svg:foreignObject').call(formats.foreignObject).classed('recipe-icon', true)
                            .append('xhtml:div').classed('flow-tile', true)
                            .append('xhtml:i').call(formats.fontIcon);
                        runnableElements.append('circle').call(formats.circleSelection);

                        runnableElements
                            .on('click', function (d) {
                                d3.event.stopPropagation();

                                if (d.center) {
                                    if (d.dblclick) {
                                        fakeClickOnLink(StateUtils.href.node(d), d3.event);
                                        Navigator.hide();
                                    }
                                    d.dblclick = true;
                                    setTimeout(function() { d.dblclick = false; }, 400);
                                }

                                if (d.center && !d.clickable) return null;

                                // The objects start moving right away after the first click
                                // To make it easier to double click, any click on the svg works
                                svg.on('click.dblclick', function() {
                                    fakeClickOnLink(StateUtils.href.node(d), d3.event);
                                });
                                setTimeout(function() { svg.on('click.dblclick', null); }, 400);

                                return focusOnNode(d);
                            });

                        // Create legend for center recipe
                        recipeInfos = runEls.filter(function(d) { return d.center; })
                            .append('svg:foreignObject').attr('class', 'dataset-info')
                            .attr('x', -130)
                            .attr('y', function(d) { return size(d) - 16; })
                            .attr('height', blockSize)
                            .attr('width', function (d) { return 260 + size(d); });

                        recipeInfos.append('xhtml:div').attr("class", "center")
                            .style('height', blockSize + 'px')
                            .style('width', function (d) { return 260 + size(d) + 'px'; })
                            .append('xhtml:h6').append('span').text(Fn.prop('description'));

                        lines.enter().append('line').attr('class', 'line').each(function (d) {
                            var run = runEls.filter(function (r) {
                                return r.id == d[0];
                            });
                            var comp = compEls.filter(function (c) {
                                return c.id == d[1]
                            });
                            var runT = parseTranslate(run.attr('transform'));
                            var compT = parseTranslate(comp.attr('transform'));
                            d3.select(this)
                                .attr('x1', runT[0] + run.attr('data-size') / 2)
                                .attr('y1', runT[1] + run.attr('data-size') / 2)
                                .attr('x2', compT[0] + comp.attr('data-size') / 2)
                                .attr('y2', compT[1] + comp.attr('data-size') / 2)
                                .attr('stroke', '#333')
                                .attr('stroke-dasharray', d[2] ? 4 : null);
                        });

                        if (topNode) {
                            svg.append('line').attr('class', 'vertical')
                                .attr('class', 'dotted')
                                .attr('x1', width/2)
                                .attr('x2', width/2)
                                .attr('y1', (height - defaultBlockSize) / 2 - 20)
                                .attr('y2', topNode.metrics ? 190 : 165)
                                .attr('stroke', '#AAA')
                                .attr('stroke-dasharray', 4);
                        }

                        if (showFlowLink) {
                            svg.append('line').attr('class', 'vertical')
                                .attr('class', 'dotted')
                                .attr('x1', width/2)
                                .attr('x2', width/2)
                                .attr('y1', isRecipeOrLabelingTask(centerNode.nodeType) ? (height / 2 + 60) : (height / 2 + 110))
                                .attr('y2', height-70)
                                .attr('stroke', '#AAA')
                                .attr('stroke-dasharray', 4);

                            const contextProjectKey = $scope.context && $scope.context.projectKey ? $scope.context.projectKey : centerNode.projectKey;
                            flowLink = svg.append("foreignObject")
                                .attr("class", "flowLink")
                                .attr("x", width/2 - 50)
                                .attr("width", 100)
                                .attr("height", 30)
                                .attr("y", height - 50);

                            flowLink.append("xhtml:a")
                                .attr("class", "btn btn--secondary")
                                .attr("href", StateUtils.href.flowLink(centerNode, contextProjectKey))
                                .text("View in flow");
                        }

                        compEls.selectAll('line.tick').call(formats.compTick);
                        runEls.selectAll('line.tick').call(formats.runTick);

                        writeCounts();
                        updateSelection();

                        drawing = false;
                    }
                }

                /* Keyboard navigation */
                var middle = function (arr) {
                    return arr[Math.floor((arr.length - 1) / 2)];
                };

                var mousetrap = new Mousetrap;

                mousetrap.bind('left', function () {
                    if (drawing) return;
                    if (selectedNode.top) return;
                    if (!selectedNode.center && selectedNode.left && !isRecipeOrLabelingTask(selectedNode.nodeType)) return focusOnNode(selectedNode);

                    var els = selectedNode.predecessors.map(function (id) {
                        return nodes[id];
                    }).filter(function (n) {
                        return n.drawn;
                    });
                    if (!els.length) {
                        if (selectedNode.center) return false;
                        return focusOnNode(selectedNode);
                    }

                    selectedNode = middle(els);
                    updateSelection();
                    return false;
                });

                mousetrap.bind('right', function () {
                    if (drawing) return;
                    if (selectedNode.top) return;
                    if (!selectedNode.center && !selectedNode.left && !isRecipeOrLabelingTask(selectedNode.nodeType)) return focusOnNode(selectedNode);

                    var els = selectedNode.successors.map(function (id) {
                        return nodes[id];
                    }).filter(function (n) {
                        return n.drawn;
                    });
                    if (!els.length) {
                        if (selectedNode.center) return false;
                        return focusOnNode(selectedNode);
                    }

                    selectedNode = middle(els);
                    updateSelection();
                    return false;
                });

                mousetrap.bind('up', function () {
                    if (drawing) return;
                    if (selectedNode.center) {
                        if (!selectedNode.top) {
                            if (topNode) selectedNode = topNode;
                            else return;
                        }
                    } else {
                        var siblings;
                        if (isRecipeOrLabelingTask(selectedNode.nodeType)) {
                            siblings = runnables;
                        } else {
                            siblings = computables;
                        }

                        siblings = siblings.filter(function (s) {
                            return s.left == selectedNode.left && !s.center;
                        });
                        var idx = siblings.indexOf(selectedNode);

                        if (idx == 0) return false;
                        selectedNode = siblings[idx - 1];

                    }

                    updateSelection();
                    return false;
                });

                mousetrap.bind('down', function () {
                    if (drawing) return;
                    if (selectedNode.center) {
                        if (selectedNode.top) {
                            selectedNode = centerNode;
                        }
                    } else {
                        var siblings;
                        if (isRecipeOrLabelingTask(selectedNode.nodeType)) {
                            siblings = runnables;
                        } else {
                            siblings = computables;
                        }

                        siblings = siblings.filter(function (s) {
                            return s.left == selectedNode.left && !s.center;
                        });
                        var idx = siblings.indexOf(selectedNode);

                        if (idx == siblings.length - 1) return false;
                        selectedNode = siblings[idx + 1];
                    }
                    updateSelection();
                    return false;
                });

                mousetrap.bind('enter', function (e) {
                    fakeClickOnLink(StateUtils.href.node(selectedNode), e);
                    Navigator.hide();
                    return false;
                });

                mousetrap.bind('space', function (e) {
                    if (selectedNode.center && !selectedNode.clickable) return false;
                    return focusOnNode(selectedNode);
                });

                element.on("$destroy", function () {
                    mousetrap.reset();
                });
                $scope.$on("$destroy", function () {
                    mousetrap.reset();
                });

                // TODO sucks
                var updateCount = function (type, attr) {
                    var span = compEls.select('span.count.' + type);
                    span.select('span')
                        .text(function (d) {
                            if (!$scope.context.nodes[d.id]) {
                                return null;
                            } else if (!$scope.context.nodes[d.id].hasOwnProperty(attr)) {
                                return $scope.context.nodes[d.id][attr.substr(3).toLowerCase()].length; // TODO mucho sucks!!
                            }
                            return $scope.context.nodes[d.id][attr];
                        });
                    span.style('display', function (d) {
                        if (d.center || !$scope.context.nodes[d.id]) return null;
                        return $scope.context.nodes[d.id][attr] ? null : 'none';
                    });
                };

                var updateSelection = function () {
                    compEls.classed('selected', function (d) {
                        return d === selectedNode;
                    });
                    runEls.classed('selected', function (d) {
                        return d === selectedNode;
                    });
                };

                function writeCounts() {
                    if (!$scope.context || !$scope.context.nodes) return;
                    updateCount('analysis', 'numAnalyses');
                    updateCount('chart', 'numCharts');
                    updateCount('notebook', 'numNotebooks');
                    d3.select('body').style('test', 'test');
                }

                var focusOnNode = function(node) {
                    $scope.context = {focusNodeId: node.id};
                    $scope.$apply();
                    $scope.$emit("change-context-focus", {
                        projectKey: $stateParams.projectKey || node.projectKey,
                        objectType: objectTypeFromNodeFlowType(node.nodeType),
                        objectId: node.nodeType === 'FOREIGN_DATASET' ? node.description : node.name
                    });
                    return false;
                };

                d3.select(window).on("resize.navigator", draw);
                $scope.$on('$destroy', function() {
                    d3.select(window).on("resize.navigator", null);
                });

                $scope.$watch("context", function (nv, ov) {
                    if (!nv || !nv.focusNodeId) return;
                    if (!ov || nv.focusNodeId != ov.focusNodeId) draw();
                    else writeCounts();
                });
            }
        }
    });

    // TODO FlowUtils?
    app.factory("getFlowNodeIcon", function ($filter) {
        const ICON_SIZE = 48;
        const getFlowNodeOldIcon = function (node) {
            if (!node) return;
            switch (node.nodeType) {
                case 'RECIPE':
                    return $filter('recipeTypeToIcon')(node.recipeType, ICON_SIZE);
                case 'LOCAL_DATASET':
                case 'FOREIGN_DATASET':
                    return $filter('datasetTypeToIcon')(node.datasetType, ICON_SIZE);
                case 'LOCAL_SAVEDMODEL':
                case 'FOREIGN_SAVEDMODEL': {
                    return $filter('savedModelSubtypeToIcon')(node.taskType, node.backendType, node.predictionType, node.savedModelType, node.externalSavedModelType, ICON_SIZE);
                }
                case 'LOCAL_MODELEVALUATIONSTORE':
                case 'FOREIGN_MODELEVALUATIONSTORE':
                case 'LOCAL_GENAIEVALUATIONSTORE':
                case 'FOREIGN_GENAIEVALUATIONSTORE':
                    return "icon-model-evaluation-store";
                case 'LOCAL_RETRIEVABLE_KNOWLEDGE':
                case 'FOREIGN_RETRIEVABLE_KNOWLEDGE':
                    return "dku-icon-cards-stack-48";
                case 'LABELING_TASK':
                    return 'icon-labeling';
                case 'LOCAL_MANAGED_FOLDER':
                case 'FOREIGN_MANAGED_FOLDER':
                    return 'icon-flow_dataset_folder';
                case 'LOCAL_STREAMING_ENDPOINT':
                    return $filter('datasetTypeToIcon')(node.streamingEndpointType, ICON_SIZE);

                // extra nodes in navigator
                case 'ANALYSIS':
                    return 'icon-dku-nav_analysis';
                case 'JUPYTER_NOTEBOOK':
                case 'SQL_NOTEBOOK':
                case 'SEARCH_NOTEBOOK':
                    return 'icon-dku-nav_notebook';
                case 'INSIGHT':
                    return $filter('insightTypeToIcon')(node.insightType, ICON_SIZE);

                default:
                    return;
            }
        };
        return function (node) {
            return $filter('toModernIcon')(getFlowNodeOldIcon(node), ICON_SIZE);
        };
    });

    // TODO FlowUtils?
    app.factory("objectTypeFromNodeFlowType", function() {
        var types = {
            "LOCAL_SAVEDMODEL": "SAVED_MODEL",
            "FOREIGN_SAVEDMODEL": "SAVED_MODEL",
            "LOCAL_RETRIEVABLE_KNOWLEDGE": "RETRIEVABLE_KNOWLEDGE",
            "FOREIGN_RETRIEVABLE_KNOWLEDGE": "RETRIEVABLE_KNOWLEDGE",
            "LOCAL_MODELEVALUATIONSTORE": "MODEL_EVALUATION_STORE",
            "FOREIGN_MODELEVALUATIONSTORE": "MODEL_EVALUATION_STORE",
            "LOCAL_GENAIEVALUATIONSTORE": "GENAI_EVALUATION_STORE",
            "FOREIGN_GENAIEVALUATIONSTORE": "GENAI_EVALUATION_STORE",
            "LOCAL_DATASET": "DATASET",
            "FOREIGN_DATASET": "DATASET",
            "LOCAL_MANAGED_FOLDER": "MANAGED_FOLDER",
            "FOREIGN_MANAGED_FOLDER": "MANAGED_FOLDER",
            "LOCAL_STREAMING_ENDPOINT": "STREAMING_ENDPOINT",
            "LABELING_TASK": "LABELING_TASK",
            "RECIPE": "RECIPE"
        };

        return function(nodeFlowType) {
            return types[nodeFlowType] || nodeFlowType;
        }
    });
})();
