(function() {
    'use strict';
    var app = angular.module('dataiku.recipes');

    app.controller("UpsertRecipeCreationController", function($scope, Fn, $stateParams, DataikuAPI, $controller) {
        $scope.recipeType = "upsert";
        $controller("SingleOutputDatasetRecipeCreationController", {$scope:$scope});

        $scope.autosetName = function() {
            if ($scope.io.inputDataset && $scope.io.targetVariable) {
                var niceInputName = $scope.io.inputDataset.replace(/[A-Z]*\./,"");
                var niceTargetVariable = $scope.io.targetVariable.replace(/[^\w ]+/g,"").replace(/ +/g,"_");
                $scope.maybeSetNewDatasetName(niceInputName + "_by_" + niceTargetVariable);
            }
        };

        $scope.getCreationSettings = function () {
            return {upsertKey: $scope.io.targetVariable};
        };

        var superFormIsValid = $scope.formIsValid;
        $scope.formIsValid = function() {
            return !!(superFormIsValid() && $scope.io.targetVariable !== undefined);
        };
        $scope.showOutputPane = function() {
            return !!($scope.io.inputDataset && $scope.io.targetVariable !== undefined);
        };

        $scope.$watch("io.targetVariable", Fn.doIfNv($scope.autosetName));
    });


    app.controller("UpsertRecipeController", function($scope, $stateParams, DataikuAPI, $q, Dialogs, ContextualMenu, PartitionDeps, $rootScope,
     $timeout, DKUtils, Expressions, Logger, $controller,  RecipesUtils, Fn, DatasetUtils) {
        var visualCtrl = $controller('VisualRecipeEditorController', {$scope: $scope}); //Controller inheritance
        this.visualCtrl = visualCtrl;
        let contextProjectKey = $scope.context && $scope.context.projectKey ? $scope.context.projectKey:$stateParams.projectKey;

        $scope.hooks.getPayloadData = function () {
            return angular.toJson($scope.params);
        };

        $scope.hooks.updateRecipeStatus = function(forceUpdate, exactPlan) {
            var payload = $scope.hooks.getPayloadData();
            if (!payload) return $q.reject("payload not ready");
            var deferred = $q.defer();
            $scope.updateRecipeStatusBase(forceUpdate, payload, {reallyNeedsExecutionPlan: exactPlan, exactPlan: exactPlan}).then(function() {
                // $scope.recipeStatus should have been set by updateRecipeStatusBase
                if (!$scope.recipeStatus) return deferred.reject();
                deferred.resolve($scope.recipeStatus);
            });
            return deferred.promise;
        };

        /******  filters  ******/
        function validateFilters() {
            if (!$scope.params) return;//not ready
            var inputRef = RecipesUtils.getSingleInput($scope.recipe, "main").ref
            var inputSchema = $scope.computablesMap[inputRef].dataset.schema
            validateFilter($scope.params.preFilter, inputSchema);
        }

        function validateFilter(filterDesc, schema) {
            var deferred = $q.defer();
            if (!filterDesc || !filterDesc.enabled) return;
            if (angular.isUndefined(filterDesc.expression)) return;
            Expressions.validateExpression(filterDesc.expression, schema)
                .success(function(data) {
                    if (data.ok && $scope.mustRunInDatabase && !data.fullyTranslated) {
                        data.ok = false;
                    }
                    filterDesc.$status = data;
                    filterDesc.$status.validated = true;
                    deferred.resolve(data);
                })
                .error(function(data) {
                    setErrorInScope.bind($scope);
                    deferred.reject('Error while validating filter');
                });
            return deferred.promise;
        };

        /* callback given to the filter module */
        $scope.onFilterUpdate = $scope.updateRecipeStatusLater;

        /****** computed columns ********/
        function computedColumnListUpdated(computedColumns) {
            $scope.params.computedColumns = angular.copy(computedColumns);
            resyncWithInputSchema();
            $scope.updateRecipeStatusLater();
        }

        /* callback given to the computed columns module */
        $scope.onComputedColumnListUpdate = computedColumnListUpdated;

        $scope.getColumnsWithComputed = function() {
            if (!$scope.uiState.columnsWithComputed) {
                var columns = [].concat($scope.getColumns());
                for (var i = 0; i < (($scope.params || {}).computedColumns || []).length; i++) {
                    var computedCol = $scope.params.computedColumns[i];
                    // do not add computed columns if they are blank
                    if (computedCol.name && columns.map(function(col){return col.name}).indexOf(computedCol.name) == -1) {
                        columns.push({
                            name: computedCol.name,
                            type: computedCol.type
                        });
                    }
                }
                $scope.uiState.columnsWithComputed = columns;
            }
            return $scope.uiState.columnsWithComputed;
        };

        /****** upsert keys (note: similar to distinct recipe) *****/
        $scope.selectLine = function(event, col, ignore) {
            event.preventDefault();
            var hasLastSelection = $scope.uiState.columnStatus.filter(function(c) {
                return c.ignore == ignore && !!c.$lastSelection;
            }).length > 0 ;
            if (event.shiftKey && hasLastSelection) {
                var selecting = false;
                for (var i = 0; i < $scope.uiState.columnStatus.length; i++) {
                    var c = $scope.uiState.columnStatus[i];
                    if (c.ignore != ignore) {
                        continue;
                    }
                    var bound = !!c.$lastSelection || c.name === col.name;
                    var firstBound = !selecting && bound;
                    var lastBound = !!selecting && bound;
                    if (firstBound) {
                        selecting = true;
                        c.$selected = true;
                    }
                    c.$selected = selecting;
                    if (lastBound) {
                        selecting = false;
                    }
                }
            } else {
                // refresh the last clicked item
                $scope.uiState.columnStatus
                        .filter(function(c) {
                            return c.ignore == ignore;
                        }).forEach(function(c) {
                            c.$lastSelection = c.name === col.name;
                        });
                // handle meta/ctrl click or normal click
                if (event.metaKey || event.ctrlKey) {
                    col.$selected = !col.$selected;
                } else {
                    $scope.uiState.columnStatus
                        .filter(function(c) {
                            return c.ignore == ignore;
                        }).forEach(function(c) {
                            c.$selected = c.name === col.name;
                        });
                }
            }
        };

        function assignIgnoreSelected(ignore, selected) {
            return function(col) {
                col.ignore = ignore;
                col.$selected = selected;
                col.$lastSelection = false;
            };
        }

        $scope.removeAllUpsertColumns = function() {
            if (!$scope.uiState.columnStatus) {
                return;
            }
            $scope.uiState.columnStatus.forEach(assignIgnoreSelected(true, false));
        };
        $scope.addAllUpsertColumns = function() {
            if (!$scope.uiState.columnStatus) {
                return;
            }
            $scope.uiState.columnStatus.forEach(assignIgnoreSelected(false, false));
        };
        $scope.removeUpsertColumns = function(col) {
            if (col) {
                assignIgnoreSelected(true, false)(col);
            } else if ($scope.uiState.columnStatus) {
                $scope.uiState.columnStatus
                    .filter(function(col) {
                        return !col.ignore && col.$selected;
                    }).forEach(assignIgnoreSelected(true, false));
            }
        };
        $scope.addUpsertColumns = function(col) {
            if (col) {
                assignIgnoreSelected(false, false)(col);
            } else if ($scope.uiState.columnStatus) {
                $scope.uiState.columnStatus
                    .filter(function(col) {
                        return col.ignore && col.$selected;
                    }).forEach(assignIgnoreSelected(false, false));
            }
        };

        /********  general init  ********/
        function loadParamsFromScript(scriptData) {
            if (!scriptData) return;
            $scope.params = JSON.parse(scriptData);
            $scope.params.preFilter = $scope.params.preFilter || {};
            $scope.params.computedColumns = $scope.params.computedColumns || [];

            $scope.uiState.computedColumns = angular.copy($scope.params.computedColumns);

            $scope.uiState.columnStatus = angular.copy($scope.getColumnsWithComputed());
            var keyColNames = $scope.params.keys.map(function(k){return k.column});
            $scope.uiState.columnStatus.forEach(function(col) {
                col.ignore = keyColNames.indexOf(col.name) == -1;
            });

            //keep params for dirtyness detection
            visualCtrl.saveServerParams();

            // update recipe according to current schema
            resyncWithInputSchema();
            onColumnStatusChanged();
        }

        function resyncWithInputSchema() {
            // in case the dataset schema changed since the recipe creation/last edition
            // reset the calculated columns with computed to force refresh
            $scope.uiState.columnsWithComputed = undefined;
            var inputColumnsWithComputed = $scope.getColumnsWithComputed();

            var newColumnStatus = [];
            var oldColumnStatusNames = ($scope.uiState.columnStatus || []).map(function(col){return col.name});
            inputColumnsWithComputed.forEach(function(col) {
                var oldCSNIdx = oldColumnStatusNames.indexOf(col.name);
                if (oldCSNIdx >= 0) {
                    newColumnStatus.push(angular.extend($scope.uiState.columnStatus[oldCSNIdx], col));
                } else {
                    var ncs = angular.copy(col);
                    ncs.ignore = true;
                    newColumnStatus.push(ncs);
                }
            });
            $scope.uiState.columnStatus = newColumnStatus;

            // call the callback if it exists
            if ($scope.onResyncWithInputSchema) {
                $scope.onResyncWithInputSchema();
            }
        }

        function onScriptChanged(nv, ov) {
             if (nv) {
                loadParamsFromScript($scope.script.data);
                DKUtils.reflowNext();
                DKUtils.reflowLater();
                $scope.hooks.updateRecipeStatus();
            }
        }

        $scope.uiState = {
            currentStep: 'upsert',
            computedColumns: []
        };

        $scope.hooks.onRecipeLoaded = function(){
            Logger.info("On Recipe Loaded");
            validateFilters();
            $scope.$watch("script.data", onScriptChanged, true); // this will call $scope.hooks.updateRecipeStatus when ready
            $scope.$watchCollection("recipe.inputs.main.items", function() {
                DatasetUtils.updateRecipeComputables($scope, $scope.recipe, $stateParams.projectKey, contextProjectKey)
                    .then(_ => resyncWithInputSchema());
            });
        };

        function onColumnStatusChanged() {
            if (!$scope.params) {
                return;
            }
            $scope.params.keys = ($scope.uiState.columnStatus || [])
                                    .filter(function(col){return !col.ignore})
                                    .map(function(col){return {column:col.name}});
            $scope.updateRecipeStatusLater();
        }

        $scope.$watch('topNav.tab',function(nv){
            if (nv == 'settings') {
                $timeout(function() {
                    $scope.$broadcast('redrawFatTable');
                });
            }
        });

        $scope.enableAutoFixup();
        $scope.specificControllerLoadedDeferred.resolve();
        $scope.$watch("uiState.columnStatus", onColumnStatusChanged, true);
    });
})();
