import threading
import time
import sys
import os
import re
import json
import logging
import traceback

class ProcessResourceUsageMonitor(threading.Thread):
    def __init__(self, command, pid):
        super(ProcessResourceUsageMonitor, self).__init__()
        self.pid = pid
        self.command = command
        self.stopped = False
        
    def run(self):
        logging.info("Start monitoring process %s" % self.pid)
        vm_rss_peak_mb = 0
        previous_total_cpu = 0
        previous_total_cpu_measurement_timestamp = 0
        previous_memory_sum_measurement_timestamp = 0
        HZ = 100
        log_every = 12
        updates = 0
        local_process = {'commandName':self.command, 'vmRSSTotalMBS':0}
        while not self.stopped:
            time.sleep(5)
            
            now = time.time()
            if previous_total_cpu_measurement_timestamp >= now:
                continue
            
            try:
                # read the stat of the process
                with open("/proc/%d/stat" % self.pid, "r") as fd:
                    stat_data = fd.read()
                
                stat_chunks = stat_data.split(" ")

                majflt = int(stat_chunks[11])
                cmajflt = int(stat_chunks[12])
                utime = int(stat_chunks[13])
                stime = int(stat_chunks[14])
                cutime = int(stat_chunks[15])
                cstime = int(stat_chunks[16])

                local_process["majorFaults"] = majflt
                local_process["childrenMajorFaults"] = cmajflt
                local_process["cpuUserTimeMS"] = utime * 1000 / HZ
                local_process["cpuSystemTimeMS"] = stime * 1000 / HZ
                local_process["cpuChildrenUserTimeMS"] = cutime * 1000 / HZ
                local_process["cpuChildrenSystemTimeMS"] = cstime * 1000 / HZ
                
                local_process["cpuTotalMS"] = (utime+stime+cutime+cstime) * 1000 / HZ
                
                # read the stat of the process
                with open("/proc/%d/status" % self.pid, "r") as fd:
                    status_data = fd.read()

                vm_peak = 0
                vm_size = 0
                vm_rss = 0
                vm_hwm = 0
                rss_anon = 0
                vm_data = 0
    
                for status_line in status_data.split('\n'):
                    m = re.match("VmPeak:[^0-9]*([0-9]*)", status_line)
                    if m is not None:
                        vm_peak = int(m.group(1))
                    m = re.match("VmSize:\\s+([0-9]*)", status_line)
                    if m is not None:
                        vm_size = int(m.group(1))
                    m = re.match("VmRSS:\\s+([0-9]*)", status_line)
                    if m is not None:
                        vm_rss = int(m.group(1))
                    m = re.match("VmHWM:\\s+([0-9]*)", status_line)
                    if m is not None:
                        vm_hwm = int(m.group(1))
                    m = re.match("RssAnon:\\s+([0-9]*)", status_line)
                    if m is not None:
                        rss_anon = int(m.group(1))
                    m = re.match("VmData:\\s+([0-9]*)", status_line)
                    if m is not None:
                        vm_data = int(m.group(1))
    
                local_process["vmSizePeakMB"] = vm_peak / 1024
                local_process["vmSizeMB"] = vm_size / 1024
                local_process["vmRSSMB"] = vm_rss / 1024
                local_process["vmHWMMB"] = vm_hwm / 1024
                local_process["vmRSSAnonMB"] = rss_anon / 1024
                local_process["vmDataMB"] = vm_data / 1024
                
                # Update peaks
                if local_process["vmRSSMB"] > vm_rss_peak_mb:
                    vm_rss_peak_mb = local_process["vmRSSMB"]
                local_process["vmRSSPeakMB"] = vm_rss_peak_mb
    
                # Update CPU derivative
                current_total_cpu = local_process["cpuUserTimeMS"]  + local_process["cpuSystemTimeMS"]
                if previous_total_cpu_measurement_timestamp > 0:
                    local_process["cpuCurrent"] = (current_total_cpu - previous_total_cpu) / ((now - previous_total_cpu_measurement_timestamp) * 1000.0)
                previous_total_cpu = current_total_cpu
                previous_total_cpu_measurement_timestamp = now
                
                # Update memory integral
                if previous_memory_sum_measurement_timestamp == 0:
                    previous_memory_sum_measurement_timestamp = now
                
                local_process["vmRSSTotalMBS"] += int((vm_rss / 1024) * (now - previous_memory_sum_measurement_timestamp))
                previous_memory_sum_measurement_timestamp = now
                
                if updates % log_every == 0:
                    logging.info("Local process resource usage : %s" % json.dumps(local_process))

                updates += 1
            except Exception as e:
                if "No such file or directory" in str(e):
                    logging.warning("Process not yet started, or shutting down")
                else:
                    logging.error("Unable to read process %s stats: %s" % (self.pid, str(e)))
        logging.info("Done monitoring process %s" % self.pid)
    
    def stop(self):
        logging.info("Stop monitoring process %s" % self.pid)
        self.stopped = True