CREATE TABLE govern_schema_version (
    version BIGINT NOT NULL
);

CREATE TABLE govern_configuration (
    configuration_item_id TEXT PRIMARY KEY,
    configuration_item_json JSONB
);

CREATE TABLE blueprints (
    blueprint_id TEXT PRIMARY KEY GENERATED ALWAYS AS ((blueprint_json ->> 'id')::TEXT) STORED,
    blueprint_json JSONB
);

CREATE TABLE blueprint_versions (
    blueprint_version_blueprint_id TEXT GENERATED ALWAYS AS ((blueprint_version_json -> 'id' ->> 'blueprintId')::TEXT) STORED NOT NULL REFERENCES blueprints ON DELETE CASCADE,
    blueprint_version_version_id TEXT GENERATED ALWAYS AS ((blueprint_version_json -> 'id' ->> 'versionId')::TEXT) STORED,
    blueprint_version_json JSONB,
    PRIMARY KEY (blueprint_version_blueprint_id, blueprint_version_version_id)
);

CREATE TABLE blueprint_version_traces (
    blueprint_version_trace_blueprint_id TEXT GENERATED ALWAYS AS ((blueprint_version_trace_json -> 'blueprintVersionId' ->> 'blueprintId')::TEXT) STORED NOT NULL,
    blueprint_version_trace_version_id TEXT GENERATED ALWAYS AS ((blueprint_version_trace_json -> 'blueprintVersionId' ->> 'versionId')::TEXT) STORED NOT NULL,
    blueprint_version_trace_origin_version_id TEXT GENERATED ALWAYS AS ((blueprint_version_trace_json ->> 'originVersionId')::TEXT) STORED,
    blueprint_version_trace_json JSONB,
    PRIMARY KEY (blueprint_version_trace_blueprint_id, blueprint_version_trace_version_id),
    FOREIGN KEY (blueprint_version_trace_blueprint_id, blueprint_version_trace_version_id) REFERENCES blueprint_versions (blueprint_version_blueprint_id, blueprint_version_version_id) ON DELETE CASCADE,
    FOREIGN KEY (blueprint_version_trace_blueprint_id, blueprint_version_trace_origin_version_id) REFERENCES blueprint_versions (blueprint_version_blueprint_id, blueprint_version_version_id) ON DELETE NO ACTION
);

CREATE TABLE artifacts (
    artifact_id TEXT PRIMARY KEY GENERATED ALWAYS AS ((artifact_json ->> 'id')::TEXT) STORED,
    artifact_blueprint_id TEXT GENERATED ALWAYS AS ((artifact_json -> 'blueprintVersionId' ->> 'blueprintId')::TEXT) STORED NOT NULL,
    artifact_version_id TEXT GENERATED ALWAYS AS ((artifact_json -> 'blueprintVersionId' ->> 'versionId')::TEXT) STORED NOT NULL,
    artifact_json JSONB,
    FOREIGN KEY (artifact_blueprint_id, artifact_version_id) REFERENCES blueprint_versions (blueprint_version_blueprint_id, blueprint_version_version_id) ON DELETE CASCADE
);

CREATE SEQUENCE artifacts_seq;

CREATE TABLE reference_relationships (
    reference_relationship_artifact_id_from TEXT GENERATED ALWAYS AS ((reference_relationship_json ->> 'artifactIdFrom')::TEXT) STORED NOT NULL REFERENCES artifacts ON DELETE CASCADE,
    reference_relationship_artifact_id_to TEXT GENERATED ALWAYS AS ((reference_relationship_json ->> 'artifactIdTo')::TEXT) STORED NOT NULL REFERENCES artifacts ON DELETE CASCADE,
    reference_relationship_json JSONB
);
CREATE INDEX reference_relationship_artifact_id_from_to_index ON reference_relationships (reference_relationship_artifact_id_from, reference_relationship_artifact_id_to);

CREATE TABLE uploaded_files (
    uploaded_file_id TEXT GENERATED ALWAYS AS ((uploaded_file_json ->> 'id')::TEXT) STORED PRIMARY KEY,
    uploaded_file_json JSONB,
    uploaded_file_data BYTEA
);

CREATE SEQUENCE uploaded_files_seq;

CREATE TABLE uploaded_file_relationships (
    uploaded_file_relationship_artifact_id TEXT GENERATED ALWAYS AS ((uploaded_file_relationship_json ->> 'artifactId')::TEXT) STORED NOT NULL REFERENCES artifacts ON DELETE CASCADE,
    uploaded_file_relationship_uploaded_file_id TEXT GENERATED ALWAYS AS ((uploaded_file_relationship_json ->> 'uploadedFileId')::TEXT) STORED REFERENCES uploaded_files ON DELETE CASCADE PRIMARY KEY,
    uploaded_file_relationship_json JSONB
);
CREATE INDEX uploaded_file_relationship_artifact_id_uploaded_file_id_index ON uploaded_file_relationships (uploaded_file_relationship_artifact_id, uploaded_file_relationship_uploaded_file_id);

CREATE TABLE time_series (
    time_series_id TEXT GENERATED ALWAYS AS ((time_series_json ->> 'timeSeriesId')::TEXT) STORED NOT NULL,
    time_series_timestamp TIMESTAMP(3) GENERATED ALWAYS AS (TO_TIMESTAMP(((time_series_json ->> 'timestamp')::DOUBLE PRECISION)/1000)) STORED NOT NULL,
    time_series_json JSONB,
    PRIMARY KEY (time_series_id, time_series_timestamp)
);

CREATE SEQUENCE time_series_seq;

CREATE TABLE time_series_relationships (
    time_series_relationship_artifact_id TEXT GENERATED ALWAYS AS ((time_series_relationship_json ->> 'artifactId')::TEXT) STORED NOT NULL REFERENCES artifacts ON DELETE CASCADE,
    -- time_series(time_series_id) is not unique, thus can not be used as foreign key.
    time_series_relationship_time_series_id TEXT GENERATED ALWAYS AS ((time_series_relationship_json ->> 'timeSeriesId')::TEXT) STORED PRIMARY KEY,
    time_series_relationship_json JSONB
);
-- index on artifact_id and time_series_id will be added in v11 (11000)

CREATE TABLE custom_pages (
    custom_page_id TEXT PRIMARY KEY GENERATED ALWAYS AS ((custom_page_json ->> 'id')::TEXT) STORED,
    custom_page_index INTEGER GENERATED ALWAYS AS ((custom_page_json ->> 'index')::INTEGER) STORED,
    custom_page_json JSONB
);
CREATE INDEX custom_page_index_index ON custom_pages (custom_page_index);

CREATE TABLE user_mappings (
    user_mapping_artifact_id TEXT GENERATED ALWAYS AS ((user_mapping_json ->> 'artifactId')::TEXT) STORED UNIQUE NOT NULL REFERENCES artifacts ON DELETE CASCADE,
    user_mapping_user_login TEXT GENERATED ALWAYS AS ((user_mapping_json ->> 'userLogin')::TEXT) STORED UNIQUE NOT NULL,
    user_mapping_json JSONB,
    PRIMARY KEY (user_mapping_artifact_id, user_mapping_user_login)
);

CREATE TABLE group_mappings (
    group_mapping_artifact_id TEXT GENERATED ALWAYS AS ((group_mapping_json ->> 'artifactId')::TEXT) STORED UNIQUE NOT NULL REFERENCES artifacts ON DELETE CASCADE,
    group_mapping_group_name TEXT GENERATED ALWAYS AS ((group_mapping_json ->> 'groupName')::TEXT) STORED UNIQUE NOT NULL,
    group_mapping_json JSONB,
    PRIMARY KEY (group_mapping_artifact_id, group_mapping_group_name)
);

CREATE TABLE global_api_key_mappings (
    global_api_key_mapping_artifact_id TEXT GENERATED ALWAYS AS ((global_api_key_mapping_json ->> 'artifactId')::TEXT) STORED UNIQUE NOT NULL REFERENCES artifacts ON DELETE CASCADE,
    global_api_key_mapping_api_key_id TEXT GENERATED ALWAYS AS ((global_api_key_mapping_json ->> 'apiKeyId')::TEXT) STORED UNIQUE NOT NULL,
    global_api_key_mapping_json JSONB,
    PRIMARY KEY (global_api_key_mapping_artifact_id, global_api_key_mapping_api_key_id)
);

CREATE TABLE roles (
    role_id TEXT PRIMARY KEY GENERATED ALWAYS AS ((role_json ->> 'id')::TEXT) STORED,
    role_json JSONB
);

CREATE TABLE blueprint_role_assignments (
    blueprint_role_assignments_blueprint_id TEXT PRIMARY KEY GENERATED ALWAYS AS ((blueprint_role_assignments_json ->> 'blueprintId')::TEXT) STORED NOT NULL REFERENCES blueprints ON DELETE CASCADE,
    blueprint_role_assignments_json JSONB
);

CREATE TABLE blueprint_permissions (
    blueprint_permissions_blueprint_id TEXT PRIMARY KEY GENERATED ALWAYS AS ((blueprint_permissions_json ->> 'blueprintId')::TEXT) STORED NOT NULL REFERENCES blueprints ON DELETE CASCADE,
    blueprint_permissions_json JSONB
);

-- Signoff Feature
CREATE TABLE blueprint_version_signoff_configurations (
    blueprint_version_signoff_configurations_blueprint_id TEXT GENERATED ALWAYS AS ((blueprint_version_signoff_configurations_json -> 'blueprintVersionId' ->> 'blueprintId')::TEXT) STORED NOT NULL,
    blueprint_version_signoff_configurations_version_id TEXT GENERATED ALWAYS AS ((blueprint_version_signoff_configurations_json -> 'blueprintVersionId' ->> 'versionId')::TEXT) STORED NOT NULL,
    blueprint_version_signoff_configurations_json JSONB NOT NULL,
    PRIMARY KEY (blueprint_version_signoff_configurations_blueprint_id, blueprint_version_signoff_configurations_version_id),
    FOREIGN KEY (blueprint_version_signoff_configurations_blueprint_id, blueprint_version_signoff_configurations_version_id) REFERENCES blueprint_versions (blueprint_version_blueprint_id, blueprint_version_version_id) ON DELETE CASCADE
);

CREATE TABLE artifact_signoffs (
    artifact_signoffs_artifact_id TEXT PRIMARY KEY GENERATED ALWAYS AS ((artifact_signoffs_json ->> 'artifactId')::TEXT) STORED NOT NULL REFERENCES artifacts ON DELETE CASCADE,
    artifact_signoffs_json JSONB NOT NULL
);
