/**
 * This script use dashboardExportToolbox object that expose a set of functions needed for this feature:
 * - checkLoading() to check whether a dashboard is loading.
 * - getPagesToExport() to get the indexes of pages to export (that are visible) of the current dashboard.
 * - clearDashboard() to hide fullscreen button, bottom footer and navigation arrows.
 * - goToFirstPage() to go to dashboard first page.
 * - goToNextPage() to go to dashboard next page.
 * - goToPage(pageIndex) to go to dashboard specific page.
 * - getVerticalBoxesCount(pageIndex) to retrieve the number of grid boxes of a dahsboard page.
 *
 * To access this object, a dummy span has been added with id dashboard-export-toolbox-anchor so we can access the scope.
 *
 * If you want to do modifications or understand the code better, dashboardExportToolbox object is defined in dashboardPage directive.
 *
 * Usage
 * node export.js magicHeadlessAuth scriptConfigFile
 *
 * Arguments
 * - magicHeadlessAuth (String): API Key generated by DSS that allows authentification.
 * - scriptConfigFile (String): Path to the JSON file containing the arguments of the script (see DashboardExportScriptRunner.Config Java class for details)
 */

'use strict';

const puppeteer = require('puppeteer');
const fs = require('fs');
const utils = require('./utils');
const log = require('./log');

// ========================
// Entry point
// ========================
const magicHeadlessAuth = process.argv[2].toString();

const scriptConfigFile = process.argv[3].toString();
log.info("Reading Dashboard export script configuration from " + scriptConfigFile);
const config = JSON.parse(fs.readFileSync(scriptConfigFile));
log.info(JSON.stringify(config));

const urlsRequiringAuthentication = config.urlsRequiringAuthentication;
const enforceSandboxing = config.browserSandBoxing;
const pageDefaultTimeout = config.pageDefaultTimeout;
const outputDirectory = config.outputDirectory;
const fileType = config.fileType;
const exportPageWidth = parseInt(config.width);
const exportPageHeight = parseInt(config.height);
const dashboards = config.dashboards.map(function(d) {
    return {
        id: d.id,
        url: d.url,
        slideIndex: (d.slideIndex === "ALL") ? undefined : parseInt(d.slideIndex),
        // Filters can be encoded or decoded so we make sure here to decode any possibly encoded filter.
        queryParams: { filtersBySlide: (d.filtersBySlide || []).map(filter => {
            // Any decoded non-empty filter contains at least one ":".
            const isFilterDecoded = filter.length > 0 && filter.includes(':');

            return isFilterDecoded ? filter : decodeURIComponent(filter);
        }) }
    };
});

try {
    log.info("Dashboard export script started.");

    utils.createBrowser(enforceSandboxing).then(function (browser) {
        return exportDashboards(browser, dashboards).then(function (result) {
            log.info("Closing browser");
            return browser.close();
        });
    }).then(function () {
        log.info("Done exporting dashboards");
    }).catch(function (err) {
        utils.exit(utils.ERR_GENERIC, "Error while running export script", err);
    });
} catch (err) {
    utils.exit(utils.ERR_GENERIC, "Error while running export script", err);
}

/**
 * Exports the supplied dashboards
 *
 * @return {Promise.<Void>}
 */
function exportDashboards(browser, dashboards) {
    let result = Promise.resolve();
    for (let i = 0; i < dashboards.length; i++) {
        result = result.then(function () {
            return exportDashboard(browser, dashboards[i]);
        });
    }
    return result;
}

/**
 * Export a dashboard (single or multiple slides)
 *
 * @return {Promise.<Void>}
 */
function exportDashboard(browser, dashboard) {
    const dashboardDirectory = createDashboardDirectory(outputDirectory, dashboard.id);
    return loadDashboardPage(browser, dashboard).then(function (page) {
        return getToolbox(page).then(function (toolbox) {
            if (dashboard.slideIndex === undefined || dashboard.slideIndex === null || dashboard.slideIndex === -1) {
                return exportAllDashboardSlides(page, toolbox, dashboardDirectory, dashboard.id, dashboard.queryParams);
            } else {
                return exportSingleDashboardSlide(page, toolbox, dashboardDirectory, dashboard.id, dashboard.slideIndex, dashboard.queryParams);
            }
        }).then(function () {
            // Beware: Do not modify or remove the following line as it is used by the backend to report on the progress of the script.
            log.info("Successfully exported dashboard " + dashboard.id);
        });
    });
}

/**
 * Create a new browser page and navigate to the specified dashboard
 *
 * @return {Promise.<Page>}
 */
function loadDashboardPage(browser, dashboard) {
    return utils.newBrowserPage(browser, exportPageWidth, exportPageHeight, urlsRequiringAuthentication, magicHeadlessAuth, pageDefaultTimeout).then(function(page) {
        return utils.navigateTo(page, dashboard.url).then(function () {
            return clearDashboard(page).then(function () {
                log.info("New browser page for dashboard " + dashboard.id + " created");
                return page;
            });
        });
    });
}

/**
 * Exports a single dashboard, the promise of this function indicate us if an export of a single dashboard has been done.
 *
 * @return {Promise.<Void>}
 */
function exportAllDashboardSlides(page, toolbox, dashboardDirectory, dashboardId, queryParams) {
    log.info("Exporting all slides of dashboard " + dashboardId);
    return getFirstVisibleSlideIdx(page).then(function(firstVisibleSlideIdx) {
        return goToSlide(page, firstVisibleSlideIdx, queryParams.filtersBySlide[firstVisibleSlideIdx] || '').then(function () {
            return captureDashboard(page, fileType, dashboardDirectory, queryParams).then(function () {
                log.info("Multiple pages export for dashboard " + dashboardId + " done");
                return page.close();
            })
        }).catch(function (err) {
            utils.exit(utils.ERR_GENERIC, "Multiple pages export for dashboard " + dashboardId + " failed", err);
        })
    });
}

/**
 * Exports a single slide from a dashboard
 *
 * @return {Promise.<Void>}
 */
function exportSingleDashboardSlide(page, toolbox, dashboardDirectory, dashboardId, slideIndex, queryParams) {
    log.info("Exporting slide " + slideIndex + " from dashboard " + dashboardId);
    return goToSlide(page, slideIndex, queryParams.filtersBySlide[slideIndex]).then(function () {
        return captureDashboardSlide(page, toolbox, fileType, dashboardDirectory, slideIndex).then(function () {
            log.info("Single page export for dashboard " + dashboardId + " done");
            return page.close();
        });
    }).catch(function (err) {
        utils.exit(utils.ERR_GENERIC, "Single Page export for dashboard " + dashboardId + " failed", err);
    });
}

/**
 * Iterate through all its slides of the dashboard.
 */
function captureDashboard(page, fileType, outputDirectory, queryParams) {
    return getVisibleSlideIndexes(page).then(function (indexes) {
        let result = Promise.resolve();
        if (indexes.length) {
            for (let i = 0; i < indexes.length; i++) {
                result = result.then(function () {
                    // Retrieve the toolbox here to make sure that we have the one corresponding to the current slide.
                    log.info("Exporting slide " + indexes[i]);
                    return getToolbox(page).then(function (toolbox) {
                        return captureDashboardSlide(page, toolbox, fileType, outputDirectory, indexes[i]).then(function () {
                            toolbox.dispose();
                        });
                    }).then(function () {
                        if (i < indexes.length-1) {
                            return goToSlide(page, indexes[i+1], queryParams.filtersBySlide[indexes[i+1]] || '');
                        }
                        return;
                    });
                });
            }
        } else {
            result = result.then(function () {
                log.info("Exporting empty state as there are no pages to display");
                return getToolbox(page).then(function (toolbox) {
                    return page.setViewport({width: exportPageWidth, height: exportPageHeight}).then(function () {
                        return captureScreen(page, fileType, outputDirectory, 0);
                    }).then(function () {
                        toolbox.dispose();
                    });
                });
            });
        }
        return result;
    });
}

/**
 * Iterate through all parts of a slide based on the viewport setted with initialWidth and initialHeight (= to the dimensions that the user has passed
 * as parameters to the script). Each part of a slide will be exported as a file. If a slide has a title, we add its height to the first iteration.
 */
function captureDashboardSlide(page, toolbox, fileType, outputDirectory, slideIndex) {
            return getBoundaries(page, toolbox, slideIndex)
        .then((boundaries) => {
            log.info("Retrieved boundaries > " + JSON.stringify(boundaries));

            if (boundaries.totalHeight <= exportPageHeight) {
                return page.setViewport({width: exportPageWidth, height: exportPageHeight}).then(function () {
                    return captureScreen(page, fileType, outputDirectory, (slideIndex + 1));
                });
            } else {
                // The dashboard height is larger than the pageHeight, we need to scroll and take multiple screenshots.
                // ...but we don't want to cut in the middle of a grid cell.
                return page.evaluate(function (toolbox, slideIndex) {
                    return toolbox.getVerticalBoxesCount(slideIndex);
                }, toolbox, slideIndex).then(function (verticalBoxesCount) {
                    const cellHeight = boundaries.grid.height / verticalBoxesCount;
                    const titleHeight = boundaries.title.height;
                    const gridOffset = boundaries.totalHeight - boundaries.grid.height - boundaries.title.height;
                    log.info("Computed metrics to scroll browser page > totalNumberBoxY:" + verticalBoxesCount + ", titleHeight:" + titleHeight + ", cellHeight:" + cellHeight + ", gridOffset:" + gridOffset);

                    // Compute the height for all-but first pages (might be different since we might have a title)
                    const cellsPerPage = Math.floor(exportPageHeight / cellHeight);
                    const viewportHeight = cellsPerPage * cellHeight;

                    // Compute the height for first page, which is different since we might have a title
                    const firstPageCellsPerPage = Math.floor((exportPageHeight - titleHeight) / cellHeight);
                    const firstPageViewportHeight = titleHeight + (firstPageCellsPerPage * cellHeight);

                    return captureDashboardSlidePart(page, toolbox, gridOffset, boundaries.totalHeight, firstPageViewportHeight, viewportHeight, outputDirectory, slideIndex);
                });
            }
    }).then(function () {
        // Resetting viewport to initial dimensions before going to next page
        return page.setViewport({width: exportPageWidth, height: exportPageHeight});
    });
}

function captureDashboardSlidePart(page, toolbox, initialTop, totalHeight, viewportInitialHeight, viewportHeight, outputDirectory, slideIndex) {
    // Defines the various parts and their bounds
    log.info("Capturing dashboard slide with scrolling > initialTop:" + initialTop + ", totalHeight:" + totalHeight + ", viewportInitialHeight:" + viewportInitialHeight + ", viewportHeight:" + viewportHeight);
    let top = initialTop;
    let bottom = initialTop + viewportInitialHeight;
    let partIndex = 1;
    let imageParts = [];
    while (top < totalHeight) {
        let imagePart = {top: top, bottom: bottom, partIndex: partIndex};
        imageParts.push(imagePart);
        top = bottom;
        bottom = Math.min(totalHeight, bottom + viewportHeight);
        partIndex++;
    }

    // Chain the promises to execute them in order
    if (imageParts.length === 0) {
        return Promise.all([]);
    }
    let promise = captureImagePart(page, toolbox, outputDirectory, slideIndex, imageParts[0], 2000);
    for (let i = 1; i <= imageParts.length; i++) {
        promise = promise.then(function () {
            if (i < imageParts.length) {
                return captureImagePart(page, toolbox, outputDirectory, slideIndex, imageParts[i], 2000);
            }
        });
    }
    return promise;
}

function captureImagePart(page, toolbox, outputDirectory, slideIndex, imagePart, waitTimeAfterReady = 2000) {
    log.info("Capturing image part " + imagePart.partIndex + " > top:" + imagePart.top + ", bottom:" + imagePart.bottom);
    // Width and height must be integers in `setViewport`
    return page.setViewport({width: Math.floor(exportPageWidth), height: Math.floor(imagePart.bottom - imagePart.top)}).then(function () {
        return page.evaluate(function (toolbox, top) {
            return toolbox.scroll(top);
        }, toolbox, imagePart.top).then(function () {
            return captureScreen(page, fileType, outputDirectory, (slideIndex + 1) + "_Part-" + imagePart.partIndex, waitTimeAfterReady);
        });
    });
}

/**
 * Captures the current viewport into a single PDF/PNG/JPEG file.
 *
 * @return Promise<Void>
 */
function captureScreen(page, fileType, outputDirectory, slideIndex = "", waitTimeAfterReady = 2000) {
    return utils.captureScreen(page, fileType, getToolbox, outputDirectory, 'Slide-' + slideIndex, waitTimeAfterReady);
}

/**
 * Gets the boundaries of the grid and of the title.
 *
 * @param page Browser page
 * @param toolbox Page toolbox
 * @param slideIndex 0-based index of the slide to get the boundaries of
 * @param attempt attempt number at retrieving the boundaries
 * @return Promise<Boundary>
 */
function getBoundaries(page, toolbox, slideIndex, attempt = 0) {
    // Ensure dashboard is fully loaded before we measure dimensions
    return utils.waitForPageToLoad(page, getToolbox).then(function () {
        return page.evaluate(function (toolbox, slideIndex) {
            const grid = toolbox.getGridBoundaries();
            const title = toolbox.getTitleBoundaries();
            let boundaries = {};
            boundaries.grid = {left: grid.x, top: grid.y, width: grid.width, height: grid.height};
            boundaries.title = {left: title.x, top: title.y, width: title.width, height: title.height};
            boundaries.totalHeight = boundaries.grid.height + boundaries.grid.top;
            boundaries.tileCount = toolbox.getTilesCount(slideIndex);
            return boundaries;
        }, toolbox, slideIndex).then(function(boundaries){
            if (boundaries.tileCount > 0 && boundaries.grid.height === 0) {
                // Recurse
                if (attempt < 600) {
                    log.info("Grid not loaded yet (height = 0). Try again in 100ms.");
                    return utils.timeout(100).then(function() {
                        return getBoundaries(page, toolbox, slideIndex, attempt + 1);
                    });
                } else {
                    log.info("Grid not loaded (height = 0) after 1 minute. Giving up and returning the invalid one.");
                    return Promise.resolve(boundaries);
                }
            } else {
                log.info("Grid loaded with height > 0. Returning it");
                return Promise.resolve(boundaries);
            }
        });
    });
}

function goToSlide(page, slideIndex, filtersQueryParam) {
    if (slideIndex < 0) {
        // All slides are hidden, we don't need to go to a specific slide
        return Promise.resolve();
    }
    return getToolbox(page)
        .then(function (toolbox) {
            return page.evaluate(function (toolbox, slideIndex) {
                return slideIndex === 0 ? toolbox.goToFirstPage() : toolbox.goToPage(slideIndex);
            }, toolbox, slideIndex);
        })
        .then(function () {
            // The dashboard page has changed so we need the new toolbox
            return getToolbox(page)
        })
        .then(function (toolbox) {
            return page.evaluate(function (toolbox, filtersQueryParam) {
                return toolbox.applyFilters(filtersQueryParam);
            }, toolbox, filtersQueryParam);
        }).then(function() {
            return waitForAllTilesToBeLoaded(page);
        }).then(function() {
            return waitForFiltersToBeApplied(page);
        });
}

function waitForAllTilesToBeLoaded(page) {
    return page.waitForSelector('.tile-content.loading', { hidden: true, timeout: 5 * 60 * 1000 });
}

async function waitForFiltersToBeApplied(page) {
    const filtersLoadingIndicatorSelector = '.tile-header__actions .dku-loader, .tile-header .dku-loader';
    try {
        // Wait for the filters spinner to appear
        await page.waitForSelector(filtersLoadingIndicatorSelector, { timeout: 4000 });
    } finally {
        // Wait for the filters spinner to disappear
        return page.waitForSelector(filtersLoadingIndicatorSelector, { hidden: true, timeout: 5 * 60 * 1000 });
    }
}

/**
 * Hides fullscreen button, bottom footer and navigation arrows.
 *
 * @param page Browser page
 * @return {Promise.<Void>}
 */
function clearDashboard(page) {
    return utils.waitForPageToLoad(page, getToolbox).then(function () {
        return getToolbox(page).then(function (toolbox) {
            return page.evaluate(function (toolbox) {
                return toolbox.clearDashboard();
            }, toolbox);
        });
    });
}

function getVisibleSlideIndexes(page) {
    return getToolbox(page).then(function (toolbox) {
        return page.evaluate(function (toolbox) {
            return toolbox.getPagesToExport();
        }, toolbox);
    });
}

function getFirstVisibleSlideIdx(page) {
    return getToolbox(page).then(function (toolbox) {
        return page.evaluate(function (toolbox) {
            return toolbox.getFirstVisiblePageIdx();
        }, toolbox);
    });
}

function getToolbox(page) {
    return page.evaluateHandle(function () {
        let toolboxAnchor = document.querySelector('#dashboard-export-toolbox-anchor');
        return angular.element(toolboxAnchor).scope().dashboardExportToolbox;
    });
}

/**
 * If two dashboards have the same name in a case of mass export, we use the id to differentiate them
 * (so we don't merge into a pdf two dashboards with the same name)
 */
function createDashboardDirectory(outputDirectory, dashboardId) {
    let dashboardDirectory = outputDirectory + "/" + dashboardId;
    try {
        fs.mkdirSync(dashboardDirectory);
    } catch (err) {
        utils.exit(utils.ERR_GENERIC, "Unable to create directory", err);
    }
    return dashboardDirectory;
}
