#' Saves data as a DSS static insight that can be exposed on the dashboard
#
#' @param id: Unique identifier of the insight within the project. If an insight with the same identifier
#'                 already exists, it will be replaced
#' @param payload: bytes of the data, or base64-encoded string
#' @param contentType: the MIME type of the data in payload (example: text/html or image/png)
#' @param encoding: if the payload was a base64-encoded string, this must be "base64". Else this must be NULL
#' @param label:  Optional display label for the insight. If NULL, the id will be used as label
#
#' @return Nothing
dkuSaveInsight <- function(id, payload, contentType, encoding=NULL, label = NULL) {
    library(base64enc)

    projectKey <- dkuGetCurrentProjectKey()
    payload <- dku__getEncodedPayload(payload, encoding)

    createInsightURL <- dku_intercom__get_backend_url("/insights/save-static-file-insight")
    args <- list(id=id, payload=payload,contentType=contentType, projectKey=projectKey, label=label)

    resp = POST(createInsightURL,
            body=args,
            encode="form",
            dku__get_auth_headers(), dku_intercom__get_httr_config()
        )
    dku__check_api_error(resp, "Failed to save insight")
}

#' Saves a HTML widget as a DSS static insight that can be exposed on the dashboard
#
#' @param id: Unique identifier of the insight within the project. If an insight with the same identifier
#'                 already exists, it will be replaced
#' @param payload: HTML widget
#' @param label:  Optional display label for the insight. If NULL, the id will be used as label
#
#' @return Nothing
dkuSaveHTMLInsight <- function(id, payload, label = NULL) {
    dkuSaveInsight(id=id, payload=dku__getHTMLRepr(payload), contentType="text/html", label=label)
}

#' Saves a ggvis chart as a DSS static insight that can be exposed on the dashboard
#
#' @param id: Unique identifier of the insight within the project. If an insight with the same identifier
#'                 already exists, it will be replaced
#' @param gg: the ggvis chart
#' @param label:  Optional display label for the insight. If NULL, the id will be used as label
#
#' @return Nothing
#'
#' @examples
#' \dontrun{
#' # Prepare the chart
#' chart <- mtcars %>% ggvis(~wt, ~mpg) %>% layer_points()
#'
#' # Save it as an insight
#' dkuSaveGgvisInsight("my-ggvis-plot", chart)
#' }
dkuSaveGgvisInsight <- function(id, gg, label = NULL) {
    html <- dkuGgvisToHTML(gg)
    dkuSaveInsight(id, html, "text/html", label=label)
}

#' Saves a googleViz chart as a DSS static insight that can be exposed on the dashboard
#
#' @param id: Unique identifier of the insight within the project. If an insight with the same identifier
#'                 already exists, it will be replaced
#' @param vis: The GoogleViz chart
#' @param label: Optional display label for the insight. If NULL, the id will be used as label
#
#' @return Nothing
dkuSaveGooglevisInsight <- function(id, vis, label = NULL) {
    dkuSaveInsight(id, repr::repr_text(vis), "text/html", label=label)
}

#' Saves a ggplot figure as a DSS static insight that can be exposed on the dashboard
#
#' @param id: Unique identifier of the insight within the project. If an insight with the same identifier
#'                 already exists, it will be replaced
#' @param vis: The ggplot figure. if NULL, the latest rendered ggplot figure will be used
#' @param label: Optional display label for the insight. If NULL, the id will be used as label
#
#' @return Nothing

dkuSaveGgplotInsight <- function(id, gg = last_plot(), label = NULL, scale=1, width=NA, height=NA, dpi = 300) {
    tmp <- tempfile(pattern="insight-png-save")
    tmpwithext <- paste0(tmp, ".png")
    ggsave(tmpwithext, gg, width=width, height=height, dpi = dpi, scale = scale)
    dev.off()
    data <- readBin(tmpwithext, "raw", file.info(tmpwithext)$size)
    dkuSaveInsight(id, data, "image/png", label=label)
    unlink(tmpwithext)
}

dku__getHTMLRepr <- function(val) {
    if (typeof(val) != 'character') {
        library(repr)
        return(repr::repr_html(val))
    }
    return(val)
}

dku__getEncodedPayload <- function(payload, encoding) {
    if (is.null(encoding)) {
        if (is.raw(payload)) {
            payload <- base64encode(payload)
        } else {
            payload <- base64encode(charToRaw(payload))
        }
    } else if (encoding != "base64") {
        stop('Invalid encoding: expected NULL or "base64"')
    }
    return(payload)
}