(function() {
    'use strict';

    const app = angular.module('dataiku.ml.explainability');

    const getDigitalBlueRGBA = alpha => `rgba(59, 153, 252, ${alpha})`;

    app.component('explorationConstraints', {
        bindings: {
            isClassification: '<',
            isRegression: '<',
            isMulticlass: '<',
            whatIfRouter: '<',
            fmi: '<',
            explorationLocalStore: '<',
            modelData: '<',
            storageTypes: '<',
            featuresSortingService: '<'
        },
        templateUrl: '/templates/ml/prediction-model/exploration/exploration-constraints.html',
        controller: function($scope, EmuType, FeaturesDistributionAdapter, ConstraintHelper, LoggerProvider, ExplorationWT1Service) {
            this.$onInit = function () {
                if (!this.explorationLocalStore.canUseConstraintsPage()) {
                    const logger = LoggerProvider.getLogger('ml.exploration.constraints');
                    logger.warn("Cannot open constraints: some variables are missing in the local storage");
                    this.whatIfRouter.openMainView();
                    return;
                }
                $scope.unlockedFeaturesMask = {};
                $scope.validConstraintsMask = {};
                $scope.prediction = this.explorationLocalStore.getScore().prediction;
                $scope.sortOptions = this.featuresSortingService.getSortOptions();
                $scope.selection = this.featuresSortingService.cleanSelection();
                $scope.$watch("selection.orderQuery", this.featuresSortingService.updateSortOrderAfterOptionChange.bind(this.featuresSortingService));
                $scope.featureDomains = [];
                $scope.reference = this.explorationLocalStore.getReference();
                $scope.saveTarget = target => this.explorationLocalStore.setTarget(target);
                $scope.target = this.explorationLocalStore.getTarget();

                const wt1Service = ExplorationWT1Service.init(this.isClassification, Object.keys($scope.reference).length);
    
                FeaturesDistributionAdapter.init(this.fmi, this.storageTypes).then(distributions => { // call is fast because it's been cached by `FeaturesDistributionAdapter`
                    const objFromFeatureNames = fn => Object.fromEntries(distributions.getNames().map(name => [name, fn(name)]));
                    $scope.distributionAdapters = objFromFeatureNames(distributions.getSingle);
                    $scope.trainTypes = objFromFeatureNames(name => distributions.getSingle(name).type);
                    $scope.constraintHelpers = objFromFeatureNames(name => ConstraintHelper.init($scope.distributionAdapters[name], this.storageTypes[name]));
                    // featureDomains
                    $scope.featureDomains = this.explorationLocalStore.getFeatureDomains();
                    if (!$scope.featureDomains.length) {
                        $scope.resetFeatureDomains();
                    }
                    // mask that tells if a feature is frozen or unlocked
                    $scope.unlockedFeaturesMask = Object.fromEntries($scope.featureDomains.map(feature => [feature.name, feature.type !== EmuType.FROZEN]));
                    // mask that tells if the constraint of a feature is valid or not
                    $scope.validConstraintsMask = Object.fromEntries($scope.featureDomains.map(feature => [feature.name, true]));
                    // WT1 call
                    wt1Service.emitConstraintsOpened();
                });
            }

            function setState(feature, makeActionable) {
                const actuallyMakeActionable = makeActionable && !$scope.constraintHelpers[feature.name].shouldAlwaysBeFrozen();
                $scope.unlockedFeaturesMask[feature.name] = actuallyMakeActionable;
                feature.type = actuallyMakeActionable ? $scope.constraintHelpers[feature.name].emuType : EmuType.FROZEN;
                $scope.saveFeatureDomains();
            }

            // mass action
            $scope.setStateOfAllFeatures = makeActionable => $scope.featureDomains.forEach(feature => setState(feature, makeActionable));
            // reset all
            $scope.resetFeatureDomains = () => {
                $scope.featureDomains = Object.values($scope.constraintHelpers).map(constraintHelper => constraintHelper.getDefaultConstraint());
                $scope.setStateOfAllFeatures(false);
                this.featuresSortingService.enrichWithSortingAttributes($scope.featureDomains);
                $scope.saveFeatureDomains();
            };
            // freeze/enable switch buttons
            $scope.toggleConstraint = feature => setState(feature, !$scope.unlockedFeaturesMask[feature.name]);
            // freeze feature
            $scope.freeze = feature => setState(feature, false);

            $scope.saveFeatureDomains = () => this.explorationLocalStore.setFeatureDomains($scope.featureDomains);

            $scope.nbActionableFeatures = () => Object.values($scope.unlockedFeaturesMask).reduce((a, b) => a + b, 0);
            $scope.nbFrozenFeatures = () => Object.keys($scope.unlockedFeaturesMask).length - $scope.nbActionableFeatures();
            $scope.getComputeDisabledReason = () => {
                const atLeastOneInvalidConstraint = $scope.featureDomains
                    .map(feature => feature.name)
                    .some(feature => !$scope.validConstraintsMask[feature] && $scope.unlockedFeaturesMask[feature]);
                if (atLeastOneInvalidConstraint) {
                    return 'Some constraints are invalid';
                }
                if (!$scope.nbActionableFeatures()) {
                    return 'Specify at least one actionable feature to begin exploration';
                }
                return null;
            }
        }
    });

    app.directive('constraintCard', function(ConstraintHelper) {
        return {
            scope: { distribution: '<', featureDomain: '=', reference: '<', saveFeatureDomains: '&', isValid: '=', freeze: '&', storageType: '<' },
            template: `
                <div ng-switch="type">
                    <numerical-constraint-card ng-switch-when="NUMERICAL"/>
                    <categorical-constraint-card ng-switch-when="CATEGORICAL"/>
                </div>`,
            controller: function($scope) {
                const constraintHelper = ConstraintHelper.init($scope.distribution, $scope.storageType);
                $scope.feature = $scope.featureDomain.name;
                $scope.type = constraintHelper.emuType;
                $scope.setIsValid = isValid => {
                    $scope.isValid = isValid;
                };
                // description
                $scope.getConstraintDescription = () => constraintHelper.getConstraintDescription($scope.featureDomain);
                // reset function
                $scope.reset = callback => {
                    const newConstraint = constraintHelper.getDefaultConstraint();
                    angular.extend($scope.featureDomain, newConstraint);
                    $scope.saveFeatureDomains();
                    $scope.setIsValid(true);
                    callback(); // used to execute type-specific actions
                };
                // base chart
                $scope.chart = {
                    feature: $scope.feature,
                    handle: undefined,
                    options: {
                        toolbox: {
                            show: false
                        },
                        brush: {
                            xAxisIndex: 0,
                            brushStyle: {
                                color: getDigitalBlueRGBA(0.1),
                                borderColor: getDigitalBlueRGBA(0.5)
                            }
                        },
                        grid: [
                            {
                                top: 15,
                                bottom: 20,
                                left: 24,
                                right: 24,
                            }
                        ],
                        xAxis: {
                            type: 'category',
                            data: undefined
                        },
                        yAxis: {
                            type: 'value',
                            show: false
                        },
                        series: [
                            {
                                name: $scope.feature,
                                type: 'bar',
                                data: undefined
                            }
                        ]
                    }
                };
            }
        }
    });

    app.directive('numericalConstraintCard', function($filter) {
        return {
            scope: true,
            template: `
                <constraint-card-skeleton>
                    <block-inputs class="exploration-constraints__numeric-input-box">
                        <div class="flex">
                            <label for="min-{{ feature }}" class="exploration-constraints__numeric-labels">Min</label>
                            <input id="min-{{ feature }}" type="number" max="{{ featureDomain.maxValue }}" step="any"
                                class="exploration-constraints__numeric-input" ng-model="featureDomain.minValue" ng-change="updateConstraint()">
                        </div>
                        <div>
                            <label for="max-{{ feature }}" class="exploration-constraints__numeric-labels">Max</label>
                            <input id="max-{{ feature }}" type="number" min="{{ featureDomain.minValue }}" step="any"
                                class="exploration-constraints__numeric-input" ng-model="featureDomain.maxValue" ng-change="updateConstraint()">
                        </div>
                    </block-inputs>
                </constraint-card-skeleton>`,
            controller: function($scope) {
                const { chart, distribution, featureDomain, reference } = $scope;
                const scale = distribution.scale;

                function enrichBaseChart() { // Update chart to add elements specific to numerical features
                    const distributionValues = distribution.distribution;
                    chart.options.xAxis.data = scale.map($filter('smartNumber')).slice(0, distributionValues.length);
                    const isReferenceIndex = i => scale.length === distributionValues.length
                        ? (reference === scale[i])
                        : scale[i] <= reference && reference < scale[i + 1];
                    chart.options.series[0].data = distributionValues.map((value, i) => ({
                        name: isReferenceIndex(i) ? 'ref.' : '',
                        itemStyle: {
                            color: getDigitalBlueRGBA(1),
                            decal: {
                                color: `rgba(0, 0, 0, ${isReferenceIndex(i) ? '0.3' : '0'})`,
                                dashArrayX: [1, 0],
                                dashArrayY: [4, 3],
                                rotation: -Math.PI / 4
                            }
                        },
                        value
                    }));
                    chart.options.series[0].label = {
                        show: true,
                        formatter: '{b}',
                        position: 'top'
                    };
                    chart.options.series[0].barWidth = '95%';
                    // We disable the mouse events on the series because there's no other way to disable "emphasis" on mouse
                    // hover, and the emphasis color is constant regardless of whether the brush includes the bar of not.
                    chart.options.series[0].silent = true;
                }

                enrichBaseChart();

                // This function takes as input the index of a bar and returns the position on the X axis in terms of pixels
                const pixelFromIndex = binIndex => chart.handle.convertToPixel({ seriesIndex: 0 }, [binIndex, undefined])[0];
                const indexFromPixel = x => chart.handle.convertFromPixel({ seriesIndex: 0 }, [x, undefined])[0];
                const halfDistanceBetweenTwoBars = () => (pixelFromIndex(1) - pixelFromIndex(0)) / 2;

                function updateBrushRange() {
                    if (!$scope.isValid || featureDomain.minValue > scale[scale.length - 1] || featureDomain.maxValue < scale[0]) {
                        chart.handle.dispatchAction({
                            type: 'brush', areas: [{
                                brushType: 'lineX',
                                range: [Infinity, Infinity] // put it out of the picture but don't remove it so that bars remain greyed out
                            }]
                        });
                        return;
                    }
                    const minValue = Math.max(featureDomain.minValue, scale[0]);
                    const maxValue = Math.min(featureDomain.maxValue, scale[scale.length - 1]);
                    const minValueIndex = distribution.getBinIndex(minValue);
                    const maxValueIndex = distribution.getBinIndex(maxValue) - (distribution.shouldBeRepresentedWithHistograms ? 1 : 0);
                    const minPosition = pixelFromIndex(minValueIndex) - halfDistanceBetweenTwoBars();
                    const maxPosition = pixelFromIndex(maxValueIndex) + halfDistanceBetweenTwoBars();
                    chart.handle.dispatchAction({
                        type: 'brush',
                        areas: [{
                            brushType: 'lineX',
                            range: [minPosition, maxPosition]
                        }]
                    });
                }

                $scope.onChartInit = $event => {
                    chart.handle = $event;
                    chart.handle.one('rendered', $scope.updateConstraint);
                    chart.handle.on('brushEnd', params => {
                        // The position of the highlighted range changed, we must update the constraint
                        const minValue = params.areas[0].range[0];
                        const maxValue = params.areas[0].range[1];
                        let minValueIndex = Math.max(indexFromPixel(minValue - halfDistanceBetweenTwoBars()) + 1, 0);
                        let maxValueIndex = Math.min(indexFromPixel(maxValue + halfDistanceBetweenTwoBars()) - 1, scale.length - 1);
                        if (minValueIndex > maxValueIndex) {
                            // Both values are in the same bar
                            const isMinInLeftPartOfBar = indexFromPixel(minValue + halfDistanceBetweenTwoBars()) === indexFromPixel(minValue + 2);
                            const isMaxInLeftPartOfBar = indexFromPixel(maxValue + halfDistanceBetweenTwoBars()) === indexFromPixel(maxValue - 2);
                            minValueIndex = Math.max(minValueIndex - (isMinInLeftPartOfBar ? 0 : 1), 0);
                            maxValueIndex = maxValueIndex + (isMaxInLeftPartOfBar ? 1 : 0);
                        }
                        // For histograms, even when only one bar is selected, minValue < maxValue
                        maxValueIndex = Math.min(maxValueIndex + (distribution.shouldBeRepresentedWithHistograms ? 1 : 0), scale.length - 1);
                        // Update constraint
                        featureDomain.minValue = scale[minValueIndex];
                        featureDomain.maxValue = scale[maxValueIndex];
                        // Limit the size of the brush
                        updateBrushRange();
                        // Round the value
                        const round = val => Math.round((val + Number.EPSILON) * 1000) / 1000;
                        featureDomain.minValue = round(featureDomain.minValue);
                        featureDomain.maxValue = round(featureDomain.maxValue);
                        $scope.saveFeatureDomains();
                        updateValidity();
                        $scope.$apply();
                    });
                };

                $scope.updateConstraint = () => {
                    // The value of the input just changed, we must update the position of the highlighted range
                    updateValidity();
                    updateBrushRange();
                    $scope.saveFeatureDomains();
                };

                const updateValidity = () => $scope.setIsValid(
                    featureDomain.minValue <= featureDomain.maxValue
                    && ![null, undefined].includes(featureDomain.minValue)
                    && ![null, undefined].includes(featureDomain.maxValue));
                updateValidity();
            }
        }
    });

    app.directive('categoricalConstraintCard', function(ExplorationChartsConfig) {
        return {
            scope: true,
            template: `
                <constraint-card-skeleton>
                    <block-inputs>
                        <select dku-bs-select="{selectedTextFormat:'count > 1', countSelectedText:'{0}/{1} enabled', actionsBox: true}"
                            multiple
                            ng-options="category for category in distribution.allCategories"
                            ng-model="featureDomain.categories"
                            data-count-selected-text="count"
                            data-live-search="true"
                            class="padleft16"></select>
                    </block-inputs>
                </constraint-card-skeleton>`,
            controller: function($scope) {
                const { chart, distribution, featureDomain, reference } = $scope;
                const scale = distribution.scale;

                function enrichBaseChart() { // Update chart to add elements specific to categorical features
                    const getDisplayName = category => category === ExplorationChartsConfig.OTHER_CATEGORY_INTERNAL_NAME
                        ? ExplorationChartsConfig.OTHER_CATEGORY_NAME : category;
                    chart.options.tooltip = {
                        position: 'top',
                        formatter: '{b0}',
                        backgroundColor: 'rgba(0, 0, 0, 0.9)',
                        borderWidth: 0,
                        padding: 6,
                        textStyle: { color: '#ffffff', fontSize: 12 }
                    }
                    chart.options.xAxis.data = scale.map(getDisplayName);
                    chart.options.series[0].data = distribution.distribution.map((value, i) => ({
                        value,
                        emphasis: { itemStyle: { color: getDigitalBlueRGBA(1) } },
                        itemStyle: {
                            color: '#dddddd',
                            decal: {
                                color: `rgba(0, 0, 0, ${reference === scale[i] ? '0.3' : '0'})`,
                                dashArrayX: [1, 0], // no empty space => here we deactivate dash along the X axis
                                dashArrayY: [4, 3], // 4px of line, 3px of empty space, 4px of line, 3px of empty space, ...
                                rotation: -Math.PI / 4 // horizontal lines become diagonals
                            }
                        },
                        isOtherCategory: scale[i] === ExplorationChartsConfig.OTHER_CATEGORY_INTERNAL_NAME
                    }));
                }

                enrichBaseChart();

                const isDisplayed = category => scale.includes(category);
                const isHidden = category => !isDisplayed(category);

                $scope.onChartInit = $event => {
                    chart.handle = $event;
                    chart.handle.one('rendered', () => $scope.updateConstraint());
                    chart.handle.on('click', params => {
                        const oldVal = [...featureDomain.categories];
                        if (params.data.isOtherCategory) { // clicked on the "Other" bar
                            // update the status of all hidden categories
                            const atLeastOneHiddenCategoryIsEnabled = oldVal.some(isHidden);
                            const hiddenCategories = distribution.allCategories.filter(isHidden);
                            if (atLeastOneHiddenCategoryIsEnabled) {
                                const enabledHiddenCategories = hiddenCategories.filter(category => oldVal.includes(category));
                                const enabledHiddenCategoriesIndices = enabledHiddenCategories.map(category => oldVal.indexOf(category));
                                enabledHiddenCategoriesIndices.sort((a, b) => a - b).reverse().forEach(index => {
                                    featureDomain.categories.splice(index, 1);
                                });
                            } else {
                                const disabledHiddenCategories = hiddenCategories.filter(category => !oldVal.includes(category));
                                disabledHiddenCategories.forEach(category => {
                                    featureDomain.categories.push(category);
                                });
                            }
                        } else {
                            // only toggle the status of the clicked category
                            const index = featureDomain.categories.indexOf(params.name);
                            if (index === -1) {
                                featureDomain.categories.push(params.name);
                            } else {
                                featureDomain.categories.splice(index, 1);
                            }
                        }
                        featureDomain.categories = [...featureDomain.categories]; // tell dku-bs-select that the array was modified
                        $scope.updateConstraint(oldVal);
                        $scope.$apply();
                    });
                };

                $scope.updateConstraint = oldVal => {
                    if (!chart.handle) {
                        return;
                    }
                    const newVal = featureDomain.categories;
                    if (oldVal === undefined) {
                        // If the function was called without the `oldVal` param, we consider that every single category might
                        // have been changed. So, we set `oldVal` to the exact opposite of `newVal`.
                        oldVal = distribution.allCategories.filter(category => !newVal.includes(category));
                    }
                    const addedCategories = newVal.filter(category => !oldVal.includes(category));
                    const removedCategories = oldVal.filter(category => !newVal.includes(category));
                    const displayedAddedCategories = addedCategories.filter(isDisplayed);
                    const displayedRemovedCategories = removedCategories.filter(isDisplayed);

                    function setHighlightForDisplayedCategory(actionType, category) {
                        chart.handle.dispatchAction({ type: actionType, seriesIndex: 0, name: category });
                    }

                    displayedAddedCategories.forEach(setHighlightForDisplayedCategory.bind(null, 'highlight'));
                    displayedRemovedCategories.forEach(setHighlightForDisplayedCategory.bind(null, 'downplay'));

                    if (distribution.allCategories.some(isHidden)) {
                        const setHighlightForOtherCategory = actionType => {
                            chart.handle.dispatchAction({ type: actionType, seriesIndex: 0, dataIndex: scale.length - 1 });
                        }
                        const atLeastOneHiddenCategoryIsEnabled = newVal.some(isHidden);
                        setHighlightForOtherCategory(atLeastOneHiddenCategoryIsEnabled ? 'highlight' : 'downplay');
                    }
                    $scope.saveFeatureDomains();
                    updateValidity();
                }

                const updateValidity = () => $scope.setIsValid(featureDomain.categories.length);
                updateValidity();

                $scope.$watch('featureDomain.categories', (_, oldVal) => $scope.updateConstraint(oldVal));
            }
        }
    });

    app.directive('constraintCardSkeleton', function() {
        return {
            transclude: { 'inputs': 'blockInputs' },
            templateUrl: '/templates/ml/prediction-model/exploration/exploration-constraint-card.html'
        };
    });

    app.directive('counterfactualsFeatureEnablerItem', function() {
        return {
            scope: { featureName: '<', reference: '<', trainType: '<', isDisabled: '<', isActionable: '<', toggleCallback: '&', constraintDescription: '<' },
            templateUrl: '/templates/ml/prediction-model/exploration/exploration-feature-enabler-item.html'
        };
    });

    app.component('featureTypeIcon', {
        template: `
            <div class="feature-type-icon feature-type-icon__{{ $ctrl.trainType.toLowerCase() }}-icon" ng-switch="$ctrl.trainType">
                <span ng-switch-when="NUMERIC">#</span>
                <i ng-switch-when="CATEGORY" class="icon-font"/>
                <i ng-switch-when="VECTOR" class="icon-dku-array"/>
                <i ng-switch-when="TEXT" class="icon-italic"/>
            </div>
        `,
        bindings: { trainType: '<' }
    });

    app.controller('ExplorationTargetSelectorController', function($scope, childCtrl) {
        $scope.target = childCtrl.target;
        $scope.prediction = childCtrl.prediction;
        $scope.updateTarget = target => {
            if (childCtrl.sanitizeTarget) {
                target = childCtrl.sanitizeTarget(target);
            }
            childCtrl.callback(target);
            childCtrl.target = target;
            $scope.target = target;
        }
    });

    app.component('counterfactualsTargetSelector', {
        templateUrl: '/templates/ml/prediction-model/exploration/counterfactuals-target-selector.html',
        bindings: { target: '=ngModel', callback: '=ngChange', prediction: '<', classes: '<' },
        controller: function($scope, $controller, NO_TARGET_CLASS) {
            this.$onInit = function () {
                $scope.possibleTargets = this.classes.filter(cat => cat !== this.prediction);
                angular.extend(this, $controller('ExplorationTargetSelectorController', { $scope: $scope, childCtrl: this }));
            }

            $scope.NO_TARGET_CLASS = NO_TARGET_CLASS;
        }
    });

    app.component('outcomeOptimizationTargetSelector', {
        templateUrl: '/templates/ml/prediction-model/exploration/outcome-optimization-target-selector.html',
        bindings: { target: '=ngModel', callback: '=ngChange', prediction: '<' },
        controller: function($scope, $controller, OutcomeOptimizationSpecialTarget) {
            this.$onInit = function() {
                angular.extend(this, $controller('ExplorationTargetSelectorController', { $scope: $scope, childCtrl: this }));
            }

            $scope.isTargetMinOrMax = target => Object.values(OutcomeOptimizationSpecialTarget).includes(target);
            this.sanitizeTarget = target => (!$scope.isTargetMinOrMax(target)) ? Number(target) : target; // will be called by parent controller
            $scope.OutcomeOptimizationSpecialTarget = OutcomeOptimizationSpecialTarget;
        }
    });

    app.controller('CounterfactualsConstraintsHelpController', function($scope) {
        $scope.title = 'Counterfactual explanations';
        $scope.imageUrl = '/static/dataiku/images/exploration/counterfactual-explanations.svg';
        $scope.resultDescription = 'counterfactual examples';
        $scope.descriptionParagraphs = 'Generate records similar to the reference example, striving to follow the train data distribution, that would result <strong>in a different predicted class</strong>.<br>Counterfactual samples can help interpret the model\'s decision making, as well as identify actions of recourse for achieving <strong>alternate outcomes</strong>.';
    });

    app.controller('OutcomeOptimizationConstraintsHelpController', function($scope) {
        $scope.title = 'Outcome optimization';
        $scope.imageUrl = '/static/dataiku/images/exploration/outcome-optimization.svg';
        $scope.resultDescription = 'near-optimal points';
        $scope.descriptionParagraphs = 'Generate records, striving to follow the train data distribution, that would result in either a <strong>minimal</strong>, <strong>maximal</strong>, or <strong>specific prediction</strong>.';
    });

    app.component('counterfactualsConstraintsEmptyState', {
        templateUrl: '/templates/ml/prediction-model/exploration/exploration-constraints-empty-state.html',
        controller: 'CounterfactualsConstraintsHelpController'
    });

    app.component('outcomeOptimizationConstraintsEmptyState', {
        templateUrl: '/templates/ml/prediction-model/exploration/exploration-constraints-empty-state.html',
        controller: 'OutcomeOptimizationConstraintsHelpController'
    });

    app.component('counterfactualsConstraintsPopover', {
        templateUrl: '/templates/ml/prediction-model/exploration/exploration-constraints-popover.html',
        controller: 'CounterfactualsConstraintsHelpController'
    });

    app.component('outcomeOptimizationConstraintsPopover', {
        templateUrl: '/templates/ml/prediction-model/exploration/exploration-constraints-popover.html',
        controller: 'OutcomeOptimizationConstraintsHelpController'
    });

})();
