(function() {
    'use strict';

    const app = angular.module('dataiku.charts');

    /**
     * (!) This service previously was in static/dataiku/js/simple_report/chart_view_commons.js
     */
    app.factory('CanvasUtils', function($q, $timeout) {
        const utils = {

            /**
             * Fill the entire canvas with the given color
             * @param canvas
             * @param color
             */
            fill: function(canvas, color) {
                const ctx = canvas.getContext('2d');
                ctx.fillStyle = color;
                ctx.fillRect(0, 0, canvas.getAttribute('width'), canvas.getAttribute('height'));
            },

            /**
             * Downloads the given canvas as a .png image
             * (previous implementation was window.open(canvas.toDataURL('image/png')) but that is not allowed anymore (see #7660))
             */
            downloadCanvas: function(canvas, filename) {
                const a = document.createElement('a');
                a.href = canvas.toDataURL('image/png').replace('image/png', 'image/octet-stream');
                a.download = filename;
                a.style.display = 'none';
                document.body.appendChild(a); // FF won't click if the <a> is not in the DOM
                $timeout(function() { // avoid "digest already in progress"
                    a.click();
                    a.remove();
                });
            },

            htmlToCanvas: function($el, scale) {
                const deferred = $q.defer();
                scale = angular.isDefined(scale) ? scale : 1;
                html2canvas_latest($el[0], { backgroundColor: null, scale: scale, logging: false }).then(function(canvas) {
                    deferred.resolve(canvas);
                });
                return deferred.promise;
            },

            /**
             * Create a new canvas from an SVG element using the browser ability to draw an SVG image inside a canvas
             * @param {SVGElement} svgEl                                - SVG element to draw in canvas
             * @param {number} width                                    - Width of the newly created canvas
             * @param {number} height                                   - Height of the newly created canvas
             * @param {Object} [options]                                - A set of optional parameters
             * @param {string[]} [options.filters]                      - A list of element names to remove from the SVG
             * @param {(cloneSvg:SVGElement) => void} [options.setup]   - A function called with the cloned SVG to perform some operation before converting to canvas
             * @returns {Promise<HTMLCanvasElement>} A promise resolved when the canvas is created or rejected if the SVG image can not be loaded
             */
            svgToCanvas: function(svgEl, width, height, options) {
                const bbox = svgEl.getBBox();
                const bounds = {
                    /*
                     * The svg bbox height and width can sometimes be bigger than the actual size of the element in the DOM
                     * due to manual range modifications
                     */
                    height: svgEl.clientHeight,
                    width: svgEl.clientWidth,
                    /*
                     * The svg bbox x or y can sometimes be extremely negative due to manual range modifications
                     */
                    x: bbox.x < 0 ? 0 : bbox.x,
                    y: bbox.y < 0 ? 0 : bbox.y
                };
                const clonedSvg = svgEl.cloneNode(true);

                if (options) {
                    if (options.filters) {
                        // Remove unwanted elements
                        options.filters.forEach(filter => d3.select(clonedSvg).selectAll(filter).remove());
                    }
                    if (typeof options.setup === 'function') {
                        // Setup cloned SVG
                        options.setup(clonedSvg);
                    }
                }

                clonedSvg.setAttribute('width', bounds.width + bounds.x);
                clonedSvg.setAttribute('height', bounds.height + bounds.y);

                const svgText = new XMLSerializer().serializeToString(clonedSvg);
                const blob = new Blob([svgText], { type: 'image/svg+xml;charset=utf-8' });

                return $q((resolve, reject) => {
                    const image = new Image();
                    image.onload = () => {
                        const canvas = document.createElement('canvas');
                        canvas.width = width;
                        canvas.height = height;
                        const ctx = canvas.getContext('2d');
                        const ratio = Math.min(canvas.width / bounds.width, canvas.height / bounds.height);
                        const offsetX = (canvas.width - bounds.width * ratio) / 2;
                        const offsetY = (canvas.height - bounds.height * ratio) / 2;
                        ctx.drawImage(image,
                            bounds.x, bounds.y, bounds.width, bounds.height,
                            offsetX, offsetY, bounds.width * ratio, bounds.height * ratio);
                        resolve(canvas);
                    };
                    image.onerror = reject;
                    image.src = URL.createObjectURL(blob);
                });
            },

            /**
             * Resize a canvas to the desired width and height. A new canvas is created.
             * @param {HTMLCanvasElement} canvas    - Canvas to resize
             * @param {number} width                - Target width
             * @param {number} height               - Target height
             * @param {boolean} [crop]              - (optional) Indicate that the image should not fit but should be cropped
             * @param {SVGLineElement[]} [refLines] - (optional) Reference lines
             * @param {Object} [margins]            - (optional) margins of the canvas, used to properly position refLines
             * @returns {HTMLCanvasElement} the newly created canvas
             */
            resize: function(canvas, width, height, crop, refLines, margins) {
                if (canvas) {
                    const newCanvas = document.createElement('canvas');
                    newCanvas.setAttribute('width', width);
                    newCanvas.setAttribute('height', height);
                    const ctx = newCanvas.getContext('2d');
                    if (crop) {
                        const ratio = Math.min(canvas.width / width, canvas.height / height);
                        ctx.drawImage(canvas, 0, 0, width * ratio, height * ratio, 0, 0, width, height);
                    } else {
                        ctx.drawImage(canvas, 0, 0, width, height);
                    }

                    if (refLines && refLines.length > 0) {
                        const xScale = width / canvas.width * window.devicePixelRatio;
                        const yScale = height / canvas.height * window.devicePixelRatio;

                        refLines.forEach(line => {
                            const x1 = (parseFloat(line.getAttribute('x1')) - (margins.x || 0)) * xScale;
                            const x2 = (parseFloat(line.getAttribute('x2')) - (margins.x || 0)) * xScale;
                            const y1 = (parseFloat(line.getAttribute('y1')) - (margins.y || 0)) * yScale;
                            const y2 = (parseFloat(line.getAttribute('y2')) - (margins.y || 0)) * yScale;

                            // Draw the line on the canvas
                            ctx.beginPath();
                            line.getAttribute('stroke-dasharray') ? ctx.setLineDash([8, 8]) : ctx.setLineDash([8, 0]);
                            ctx.moveTo(x1, y1);
                            ctx.lineTo(x2, y2);
                            ctx.strokeStyle = line.style.stroke;
                            ctx.lineWidth = line.style.strokeWidth / 4;
                            ctx.stroke();
                        });
                    }
                    return newCanvas;
                }

                return null;
            },

            /**
             * Add inline style attribute for all children in the given selection (that includes the style from css files)
             * Useful before using canvg, that ignores style from .css files
             * @param {d3.selection} sel
             */
            inlineAllStyles: function(sel) {
                if (sel.size() === 0) {
                    return;
                }

                let svg_style;
                for (let i = 0; i <= document.styleSheets.length - 1; i++) {
                    try {
                        svg_style = document.styleSheets[i].rules || document.styleSheets[i].cssRules || [];
                    } catch (e) {
                        // Firefox will fail with security error when attempting to access cross-domain style sheets from JS
                        if (e.name != 'SecurityError') {
                            throw e;
                        }
                    }
                    for (let j = 0; j < svg_style.length; j++) {
                        if (svg_style[j].type == 1) {
                            try {
                                sel.selectAll(svg_style[j].selectorText).style(utils._makeStyleObject(svg_style[j]));
                            } catch (e) {
                                // d3 fails to parse some of our css rules
                            }
                        }
                    }
                }
            },

            _makeStyleObject: function(rule) {
                const styleDec = rule.style;
                const output = {};
                let s;
                for (s = 0; s < styleDec.length; s++) {
                    output[styleDec[s]] = styleDec[styleDec[s]];
                    if (styleDec[styleDec[s]] === undefined) {
                        output[styleDec[s]] = styleDec.getPropertyValue(styleDec[s]);
                    }
                }
                return output;
            }
        };

        return utils;
    });

})();
