import os,sys

from contextlib import contextmanager

# We use that and not os.path.sep, because it introduce problems in Windows. (see PR #9)
DIR_SEPARATOR = "/"

def isWindows():
    return sys.platform == "win32"

def to_os_path(path, root=''):
    """Convert an API path to a filesystem path

    If given, root will be prepended to the path.
    root must be a filesystem path already.

    Jupyter ask for the path $PROJECT/($FOLDER/)$NOTEBOOK_FILE but this is a virtual filesystem.
    We need to transform this path to match the real filesystem: dss-home/config/project/$PROJECT/ipython_notebooks/($FOLDER/)$$NOTEBOOK_FILE
    """
    parts = path.strip(DIR_SEPARATOR).split(DIR_SEPARATOR)
    parts = [p for p in parts if p != '']  # remove duplicate splits
    parts.insert(1, "ipython_notebooks")

    # Here the variable "root" point to the DSS config file folder (aka dss-home/config/project)
    path = os.path.join(root, *parts)
    return path


# see https://github.com/jupyter/nbformat/blob/4.0.0/nbformat/sign.py#L72
# see https://github.com/dataiku/dip/blob/release/12/src/main/java/com/dataiku/dip/server/services/JupyterService.java#L936
@contextmanager
def remove_dss_extra(nb):
    """Context manager for operating on a notebook with its dss extra fields removed

    Used for excluding dss extra fields when computing a notebook's signature.
    """
    saved_fields = {}
    # it's ugly, a bit unstable but still more stable than a whitelist inspecting nbformat
    for field in {
        "modifiedBy",
        "tags",
        "customFields",
        "versionNumber",
        "description",
        "shortdesc",
        "associatedRecipe",
        "creationTag",
        "creator",
        "createdOn",
        "analyzedDataset",
        "automationLocal",
        "dkuGit"
    }:
        if field in nb["metadata"]:
            saved_fields[field] = nb["metadata"].pop(field)
    try:
        yield
    finally:
        for field, field_value in saved_fields.items():
            nb["metadata"][field] = field_value
