(function() {
    'use strict';

    angular.module('dataiku.dashboards').component('dashboardNavigationDrawer', {
        templateUrl: '/static/dataiku/js/dashboards/components/dashboard-navigation-drawer/dashboard-navigation-drawer.component.html',
        bindings: {
            // Inputs
            dashboard: '<', // Dashboard
            canEdit: '<', // boolean
            selectedTiles: '<', // Tile[]
            selectedInsights: '<', // Insight[],
            currentPageIndex: '<', // number
            canModerateDashboards: '<', //boolean
            variables: '<', // [key: string]: string
            // Outputs
            createPage: '&', // () => void
            navigateToPage: '&', // ({ $pageIndex: number }) => void
            openCopySlideModal: '&', // ({ $pageIndex: number }) => void
            pageIndexChange: '&', // ({ $previousIndex: number, $currentIndex }) => void
            removePage: '&' //  ({ $pageIndex: number }) => void
        },
        controller: function($timeout, DASHBOARD_SETTINGS_PANEL_VIEWS, translate, $rootScope) {
            const ctrl = this;
            const TRANSITION_DURATION_MS = 100;
            const DEFAULT_DRAWER_WIDTH = 299;
            const MIN_DRAWER_WIDTH = 149;
            const MAX_DRAWER_WIDTH = MIN_DRAWER_WIDTH * 4;
            const BASE_PAGE_SORT_OPTIONS = {
                helper: 'clone',
                axis: 'y',
                cursor: 'move',
                update: handlePageSortChange,
                handle: '.dashboard-navigation-drawer-left-bar__page-miniature-content',
                items: '> .dashboard-navigation-drawer-left-bar__page-miniature'
            };

            ctrl.translate = translate;
            ctrl.CLOSED_DRAWER_WIDTH = 49;
            ctrl.isSettingsPanelExpanded = false;
            ctrl.isDrawerExpanded = ctrl.canEdit ? false : (localStorage.getItem('dku.dashboards.drawer.expand') === 'true' || false);
            ctrl.expandedDrawerWidth = getStoredExpandedDrawerWidth();
            ctrl.activeView = DASHBOARD_SETTINGS_PANEL_VIEWS.PAGE_SETTINGS;
            ctrl.pageSortOptions = { ...BASE_PAGE_SORT_OPTIONS };
            ctrl.resizing = false;
            ctrl.toggleDrawerLabel = getToggleDrawerLabel();
            ctrl.toggleMenuLabel = getToggleMenuLabel();
            ctrl.navigationDrawerPageLabel = getNavigationDrawerPageLabel();

            ctrl.DASHBOARD_SETTINGS_PANEL_VIEWS = DASHBOARD_SETTINGS_PANEL_VIEWS;
            ctrl.pagesTitles = [];

            ctrl.$onChanges = (changes) => {
                if (changes.canEdit) {
                    handleCanEditChange(ctrl.canEdit);
                }
                if (changes.selectedTiles) {
                    handleSelectedTilesChange(ctrl.selectedTiles);
                }
                if (changes.dashboard) {
                    ctrl.visiblePages = ctrl.dashboard.pages.filter(page => page.show);
                    ctrl.navigationDrawerPageLabel = getNavigationDrawerPageLabel();
                    ctrl.pagesTitles = getPagesTitles();
                }

                if (changes.currentPageIndex) {
                    scrollToSelectedPage();
                }
            };

            ctrl.$onInit = () => {
                scrollToSelectedPage();
            };

            function scrollToSelectedPage() {
                $timeout(function() {
                    const selectedPage = document.querySelector('.dashboard-navigation-drawer-left-bar__page-miniature--current');
                    if (selectedPage) {
                        selectedPage.scrollIntoView({ block: 'nearest' });
                    }
                });
            }

            /**
             * @param {boolean} [toggle]
             */
            ctrl.toggleDrawer = (toggle = !ctrl.isDrawerExpanded) => {
                localStorage.setItem('dku.dashboards.drawer.expand', toggle);
                ctrl.isDrawerExpanded = toggle;
                ctrl.toggleDrawerLabel = getToggleDrawerLabel();
            };


            ctrl.getVisiblePages = () => {
                return ctrl.dashboard.pages.filter(page => page.show);
            };

            /**
             * @param {boolean} [toggle]
             * @param {DASHBOARD_SETTINGS_PANEL_VIEWS} [view]
             */
            ctrl.toggleEditionMenu = (toggle = !ctrl.isSettingsPanelExpanded, view = ctrl.activeView) => {
                ctrl.isSettingsPanelExpanded = toggle;
                ctrl.toggleMenuLabel = getToggleMenuLabel();
                if (toggle) {
                    ctrl.activeView = view;
                }
                $timeout(() => {
                    if (!toggle) {
                        ctrl.activeView = view;
                    }
                }, TRANSITION_DURATION_MS);
            };

            /**
             * @param {number} pageIndex
             */
            ctrl.handleMiniatureClick = (pageIndex) => {
                document.activeElement.blur();
                if (pageIndex !== ctrl.currentPageIndex) {
                    ctrl.navigateToPage({ $pageIndex: pageIndex });
                }
                if (ctrl.canEdit === true) {
                    ctrl.toggleEditionMenu(ctrl.isSettingsPanelExpanded, DASHBOARD_SETTINGS_PANEL_VIEWS.PAGE_SETTINGS);
                }
            };

            /**
             * @param {MouseEvent} event
             */
            ctrl.handleResizeStart = (event) => {
                event.preventDefault();
                event.stopPropagation();
                ctrl.resizing = true;
                document.addEventListener('mouseup', handleResizeStop, true);
                document.addEventListener('mousemove', handleResizeChange, true);
            };

            /**
             * @param {MouseEvent} event
             */
            function handleResizeStop(event) {
                event.preventDefault();
                event.stopPropagation();
                ctrl.resizing = false;
                localStorage.setItem('dku.dashboards.drawer.expanded-width', ctrl.expandedDrawerWidth);
                document.removeEventListener('mouseup', handleResizeStop, true);
                document.removeEventListener('mousemove', handleResizeChange, true);
            };

            /**
             * @param {MouseEvent} event
             */
            function handleResizeChange(event) {
                event.preventDefault();
                event.stopPropagation();
                const newWidth = event.clientX - document.body.offsetLeft;
                if (newWidth >= MIN_DRAWER_WIDTH && newWidth < MAX_DRAWER_WIDTH) {
                    $timeout(() => ctrl.expandedDrawerWidth = newWidth);
                }
            }

            function getStoredExpandedDrawerWidth() {
                const storedWidth = localStorage.getItem('dku.dashboards.drawer.expanded-width');
                if (!storedWidth) {
                    return DEFAULT_DRAWER_WIDTH;
                }
                const parsedWidth = parseInt(storedWidth);
                if (isNaN(parsedWidth)) {
                    return DEFAULT_DRAWER_WIDTH;
                }
                return parsedWidth;
            }

            function handlePageSortChange(_, ui) {
                ctrl.pageIndexChange({ $previousIndex: ui.item.sortable.index, $currentIndex: ui.item.sortable.dropindex });
                // This is needed if the user stays on the miniature after moving it, the tooltip is not updated yet
                ctrl.pagesTitles[ui.item.sortable.dropindex] = getPageTitle(ctrl.dashboard.pages[ui.item.sortable.index], ui.item.sortable.dropindex + 1);
                updatePagesTitles();
            };

            function handleSelectedTilesChange(selectedTiles) {
                if (selectedTiles == null) {
                    return;
                }
                if (selectedTiles.length === 1 && selectedTiles[0].insightType !== 'filters') {
                    ctrl.activeView = DASHBOARD_SETTINGS_PANEL_VIEWS.TILE_SETTINGS;
                } else if (selectedTiles.length > 1) {
                    ctrl.activeView = DASHBOARD_SETTINGS_PANEL_VIEWS.TILE_SETTINGS;
                    ctrl.toggleEditionMenu(true);
                } else {
                    ctrl.activeView = DASHBOARD_SETTINGS_PANEL_VIEWS.PAGE_SETTINGS;
                }
            }

            /**
             * @param {boolean} canEdit
             */
            function handleCanEditChange(canEdit) {
                ctrl.isSettingsPanelExpanded = canEdit;
                ctrl.toggleMenuLabel = getToggleMenuLabel();
                if (canEdit) {
                    ctrl.isDrawerExpanded = false;
                }
                ctrl.pageSortOptions = {
                    ...BASE_PAGE_SORT_OPTIONS,
                    disabled: !canEdit
                };
            }

            function getPageTitle(page, index) {
                return page.displayedTitle || ctrl.translate('DASHBOARD.NAVIGATION_DRAWER.PAGE_INDEX', `Page ${index}`, { index });
            }

            function getPagesTitles() {
                if (!ctrl.dashboard || !ctrl.dashboard.pages) {
                    return [];
                }
                return ctrl.dashboard.pages.map((page, index) => getPageTitle(page, index + 1));
            }

            const unregisterDashboardTitleChangedEvent = $rootScope.$on('dashboardPageTitleChanged', function(_event, newTitle) {
                ctrl.pagesTitles[newTitle.pageIndex] = newTitle.displayedTitle;
            });

            ctrl.$onDestroy = function() {
                unregisterDashboardTitleChangedEvent();
            };

            function getToggleDrawerLabel() {
                if (ctrl.isDrawerExpanded) {
                    return translate('DASHBOARD.NAVIGATION_DRAWER.HIDE_PAGE_TITLES', 'Hide page titles');
                }

                return translate('DASHBOARD.NAVIGATION_DRAWER.SHOW_PAGE_TITLES', 'Show page titles');
            }

            function getNavigationDrawerPageLabel() {
                if (!ctrl.visiblePages) {
                    return '';
                }

                if (ctrl.visiblePages.length <= 1) {
                    return translate('DASHBOARD.NAVIGATION_DRAWER.PAGE', 'PAGE', { numberOfPages: ctrl.visiblePages.length });
                }

                return translate('DASHBOARD.NAVIGATION_DRAWER.PAGES', 'PAGES', { numberOfPages: ctrl.visiblePages.length });
            }

            function getToggleMenuLabel() {
                if (ctrl.isSettingsPanelExpanded) {
                    return translate('DASHBOARD.NAVIGATION_DRAWER.CLOSE_PANEL', 'Close panel');
                }

                return translate('DASHBOARD.NAVIGATION_DRAWER.OPEN_PANEL', 'Open panel');
            }

            ctrl.addPage = function() {
                ctrl.createPage();
                updatePagesTitles();
            };

            ctrl.deletePage = function(pageIndex) {
                ctrl.removePage({ $pageIndex: pageIndex });
                updatePagesTitles();
            };

            function updatePagesTitles() {
                $timeout(function() {
                    ctrl.pagesTitles = getPagesTitles();
                });
            }
        }
    });
})();
