/* global describe, it, inject, expect, module, beforeEach, PivotTableTestUtils, ChartTestUtils */

describe('ChartContextualMenu', () => {
    let chartData, chartDef, chartStore, axisNames, element, chartContextualMenu;
    let $rootScope, $sce;

    beforeEach(module('templates'));
    beforeEach(module('dataiku.charts', $provide => {
        $provide.value('ChartDataUtils', {});
        $provide.value('ChartFeatures', {
            canDrillHierarchy: () => true
        });
        $provide.value('DRILL_UP_SOURCE', {});
        $provide.value('DRILL_DOWN_SOURCE', {});
    }));
    beforeEach(module('dataiku.shared'));
    beforeEach(module('dataiku.services'));
    beforeEach(module('dataiku.directives.widgets'));
    beforeEach(module('dataiku.mock'));
    beforeEach(module('dataiku.dashboards', $provide => {
        $provide.value('DashboardFilters', {
            canCrossFilter: () => true
        });
        $provide.value('WT1', {});
        $provide.value('$state', {});
    }));

    beforeEach(ChartTestUtils.initAngularJSTemplatingResource);
    beforeEach(inject(function(_$sce_) {
        $sce = _$sce_;
    }));

    it('is defined', function () {
        prepareChartData();
        expect(chartContextualMenu).toBeDefined();
    });

    it('should create a handler', function () {
        prepareChartData();
        let contextualMenu = chartContextualMenu.create(chartData, chartDef, chartStore);
        expect(contextualMenu).toBeDefined();
    });

    it('should display a contextual menu on specific chart coords', function () {
        prepareChartData();
        const dimensionMap = new Map();
        axisNames.forEach((axisName, index) => {
            dimensionMap.set(axisName, chartDef.yDimension[index]);
        });
        chartStore.set('dimensionDict', dimensionMap);
        
        const contextualMenu = chartContextualMenu.create(chartData, chartDef, chartStore);
        const coords =  {};
        axisNames.forEach(axisName => {
            coords[axisName] = 1;
        });
        contextualMenu.showForCoords(coords, new Event( element, { bubbles: true } ));
        $rootScope.$apply();
        expect(angular.element('[data-qa-charts-contextual-menu]').length).toBe(1);
        expect(angular.element('[data-qa-charts-contextual-menu-include-nd]').length).toBe(1);
        expect(angular.element('[data-qa-charts-contextual-menu-exclude-nd]').length).toBe(1);
        expect(angular.element('[data-qa-charts-contextual-menu-include-1d]').length).toBe(1);
        expect(angular.element('[data-qa-charts-contextual-menu-exclude-1d]').length).toBe(1);
        expect(angular.element('[data-qa-charts-contextual-menu-drilldown-hierarchy]').length).toBe(0);
        expect(angular.element('[data-qa-charts-contextual-menu-drilldown-native]').length).toBe(0);
    });

    it('should display a contextual menu on specific chart coords with native drilldown options', function () {
        prepareChartData(PivotTableTestUtils.getPivotDataWith2NumColumnDim1Measure);
        const dimensionMap = new Map();
        axisNames.forEach((axisName, index) => {
            dimensionMap.set(axisName, chartDef.xDimension[index]);
        });
        chartStore.set('dimensionDict', dimensionMap);
        
        const contextualMenu = chartContextualMenu.create(chartData, chartDef, chartStore);
        const coords =  {};
        axisNames.forEach(axisName => {
            coords[axisName] = 1;
        });
        contextualMenu.showForCoords(coords, new Event( element, { bubbles: true } ));
        $rootScope.$apply();
        expect(angular.element('[data-qa-charts-contextual-menu]').length).toBe(1);
        expect(angular.element('[data-qa-charts-contextual-menu-include-nd]').length).toBe(1);
        expect(angular.element('[data-qa-charts-contextual-menu-exclude-nd]').length).toBe(1);
        expect(angular.element('[data-qa-charts-contextual-menu-include-1d]').length).toBe(1);
        expect(angular.element('[data-qa-charts-contextual-menu-exclude-1d]').length).toBe(1);
        expect(angular.element('[data-qa-charts-contextual-menu-drilldown-hierarchy]').length).toBe(0);
        expect(angular.element('[data-qa-charts-contextual-menu-drilldown-native]').length).toBe(1);
    });

    it('should display a contextual menu on specific chart coords with hierarchy drilldown options', function() {
        prepareChartData(PivotTableTestUtils.getPivotDataWith1ColumnHierarchyDim2Measures);
        const dimensionMap = new Map();
        axisNames.forEach((axisName, index) => {
            dimensionMap.set(axisName, chartDef.xHierarchyDimension[0].dimensions[index]);
        });
        chartStore.set('dimensionDict', dimensionMap);
        
        const contextualMenu = chartContextualMenu.create(chartData, chartDef, chartStore);

        const coords =  {};
        axisNames.forEach(axisName => {
            coords[axisName] = 1;
        });
        contextualMenu.showForCoords(coords, new Event(element));
        $rootScope.$apply();
        expect(angular.element('[data-qa-charts-contextual-menu]').length).toBe(1);
        expect(angular.element('[data-qa-charts-contextual-menu-include-1d]').length).toBe(1);
        expect(angular.element('[data-qa-charts-contextual-menu-exclude-1d]').length).toBe(1);
        expect(angular.element('[data-qa-charts-contextual-menu-drilldown-hierarchy]').length).toBe(1);
    });

    it('should display background click drillup contextual menu', function () {
        prepareChartData();
        const actionName = 'drillup';
        const contextualMenu = chartContextualMenu.create(chartData, chartDef, chartStore);

        const customActions = [
            {
                label: $sce.trustAsHtml('Drill up on H1'),
                icon: 'dku-icon-copy-step-16',
                onClick: () => {
                    this.handleDrillUp(chartDef, {}, 'TOOLTIP');
                },
                name: actionName
            },
            {
                label: $sce.trustAsHtml('Drill up on H2'),
                icon: 'dku-icon-copy-step-16',
                onClick: () => {
                    this.handleDrillUp(chartDef, {}, 'TOOLTIP');
                },
                name: actionName
            },
        ];

        contextualMenu.open({ event: new Event(element), customActions, type: 'background' });

        $rootScope.$apply();
        expect(angular.element('[data-qa-charts-contextual-menu]').length).toBe(1);
        const elements = angular.element('[action-name]');
        expect(elements.length).toBe(2);
        expect(elements.eq(0).attr('action-name')).toBe(actionName);
        expect(elements.eq(1).attr('action-name')).toBe(actionName);
    });

    const prepareChartData = (getPivotData = PivotTableTestUtils.getPivotDataWith2RowDim1MeasureAggregAsRows) => {
        inject(function (ChartStoreFactory, ChartTensorDataWrapper, ChartContextualMenu) {
            $rootScope = ChartTestUtils.getScope();
            $('body').empty();
            element = angular.element(`
                <div class="chart-test" style="width: 15px; height: 15px; background-color: red;"></div>
            `);
            ChartTestUtils.renderElement(element, $rootScope);
            const pivotData = getPivotData();
            const { axesDef, data } = pivotData;
            axisNames = Object.keys(axesDef);
            chartDef = pivotData.chartDef;
            chartData = new ChartTensorDataWrapper(data, axesDef);
            const { store, id } = ChartStoreFactory.getOrCreate(chartDef.$chartStoreId);
            chartDef.$chartStoreId = id;
            chartStore = store;
            chartContextualMenu = ChartContextualMenu;
        });
    };
});
